PRO read_wrf_static, grid_info, $
                    dataroot=dataroot,filename=filename, $
                    lat=lat,lon=lon,avg=avg,$
                    use=use,ldf=ldf , $
                    lab=lab, lob=lob, lac=lac, loc=loc, $
                    std=std, env=env, zin=zin, spr=spr,cpr=cpr, $
                    mfa=mfa, mfb=mfb, mfc=mfc, cph=cph, cpv=cpv, $
                    sln=sln, slt=slt, avc=avc, mfl=mfl, stl=stl, $
                    g01=g01, g06=g06, g09=g09, g12=g12, sbl=sbl, $
                    lnd=lnd, tmp=tmp
; **********************************************************************
;  PURPOSE
;  =======
;  Provided a MOAD_DATAROOT (via environment variable or the 
;  dataroot keyword),this IDL procedure opens the
;  static.wrfsi NetCDF file and extracts the horizontal
;  grid information (map projection, dimensions, etc.).  Use 
;  of optional keywords provides return of the lat/lon arrays,
;  terrain heights, land_use, or land_fraction. 
;
;  ARGUMENTS
;  =========
;  grid_info	Output	Structure       LAPS variable name (e.g., "u3")
;  dataroot     Keyword String		Path to static (MOAD_DATAROOT)
;                  If not set, it uses GETENV
;  lat          Keyword                 Set to named variable that
;                                       will hold latitude array
;  lon          Keyword                 Set to named variable that
;                                       will hold longitude array
;  lab          Keyword                 Lats on B grid
;  lac          Keyword                 Lats on C grid
;  lob          Keyword                 Lons on B grid
;  loc          Keyword                 Lons on C gri
;  avg          Keyword                 Set to named variable that
;                                       will hold terrain heights
;  use          Keyword                 Set to named variable that
;                                       will hold land use info
;  ldf           Keyword                Set to named variable that
;                                       will hold land fraction data
;  lnd          Keyword                 Set to named variable that
;                                       will hold land-water mask (0/1)
;  g**          Keyword                 Set to named variable that
;                                       will hold monthly greenness fraction (eg.,06=June)
;  tmp          Keyword                 Set to named variable that
;                                       will hold mean annual soil temp (deg K)
;  HISTORY
;  =======
;  11 Dec 2000 - Initial Version -- Brent Shaw, NOAA/FSL/FRD/LAPB (CIRA)
;
; **********************************************************************


IF KEYWORD_SET (filename) THEN BEGIN
  datafile = filename
ENDIF ELSE BEGIN
  IF (not KEYWORD_SET(dataroot) AND not KEYWORD_SET(filename)) THEN BEGIN
    dataroot = GETENV('MOAD_DATAROOT')
    IF dataroot eq '' THEN BEGIN
      PRINT, 'No MOAD_DATAROOT environment variable set and '
      PRINT, 'no dataroot keyword set, so I am going to assume'
      PRINT, 'you are in the data directory of your domain!'
        dataroot = '.'
    ENDIF
  ENDIF
  datafile = dataroot + '/static/static.wrfsi' 
ENDELSE
wrffile = FINDFILE(datafile, COUNT=count)
IF count NE 1 THEN BEGIN
  PRINT, 'File not found:', datafile
ENDIF ELSE BEGIN 
  
  ; Open the netcdf file
  unit = NCDF_OPEN(datafile, /NOWRITE) 
  ; Get the map information
  NCDF_VARGET, unit, 'grid_type', grid_type
  grid_type = STRTRIM(STRING(grid_type),2)
  NCDF_VARGET, unit, 'Nx', nx
  NCDF_VARGET, unit, 'Ny', ny
  NCDF_VARGET, unit, 'Dx', dx
  NCDF_VARGET, unit, 'Dy', dy
  NCDF_VARGET, unit, 'LoV', orientation
  IF orientation GT 180. THEN orientation = orientation-360.
  NCDF_VARGET, unit, 'Latin1', standard_lat1
  NCDF_VARGET, unit, 'Latin2', standard_lat2
  NCDF_VARGET, unit, 'lat', lata
  NCDF_VARGET, unit, 'lon', lona
  NCDF_VARGET, unit, 'lac', latc
  NCDF_VARGET, unit, 'loc', lonc 
  ; Set up info for structure

  corners =[ latc[0,0], lonc[0,0], $
             latc[0,ny-1],lonc[0,ny-1], $
             latc[nx-1,ny-1],lonc[nx-1,ny-1],$
             latc[nx-1,0],lonc[nx-1,0] ]
  ; Get center lat/lon.  How we get this depends on whether or 
  ; not there is an even or odd number of grid points in each direction

  odd_xdir = nx MOD 2
  odd_ydir = ny MOD 2
  IF (odd_xdir EQ 1) AND (odd_ydir EQ 1) THEN BEGIN
    lat_cent = lata[(nx-1)/2,(ny-1)/2]
    lon_cent = lona[(nx-1)/2,(ny-1)/2]
  ENDIF ELSE BEGIN ; Set up for bilinear interp
    lat_temp = FLTARR(2,2)
    lon_temp = FLTARR(2,2)
    CASE 1 OF
      (odd_xdir EQ 1) AND (odd_ydir EQ 0): BEGIN
        lat_temp[0,0] = lata[(nx-1)/2,(ny/2)-1]
        lon_temp[0,0] = lona[(nx-1)/2,(ny/2)-1]
        lat_temp[0,1] = lata[(nx-1)/2,(ny/2)]
        lon_temp[0,1] = lona[(nx-1)/2,(ny/2)]
        lat_temp[1,1] = lata[(nx-1)/2,(ny/2)]
        lon_temp[1,1] = lona[(nx-1)/2,(ny/2)]
        lat_temp[1,0] = lata[(nx-1)/2,(ny/2)-1]
        lon_temp[1,0] = lona[(nx-1)/2,(ny/2)-1]
        END
      (odd_xdir EQ 0) AND (odd_ydir EQ 1):BEGIN
        lat_temp[0,0] = lata[(nx/2)-1,(ny-1)/2]
        lon_temp[0,0] = lona[(nx/2)-1,(ny-1)/2]
        lat_temp[0,1] = lata[(nx/2)-1,(ny-1)/2]
        lon_temp[0,1] = lona[(nx/2)-1,(ny-1)/2]
        lat_temp[1,1] = lata[(nx/2),(ny-1)/2]
        lon_temp[1,1] = lona[(nx/2),(ny-1)/2]
        lat_temp[1,0] = lata[(nx/2),(ny-1)/2]
        lon_temp[1,0] = lona[(nx/2),(ny-1)/2]
        END       
      (odd_xdir EQ 0) AND (odd_ydir EQ 0):BEGIN
        lat_temp[0,0] = lata[(nx/2)-1,(ny/2)-1]
        lon_temp[0,0] = lona[(nx/2)-1,(ny/2)-1]
        lat_temp[0,1] = lata[(nx/2)-1,(ny/2)]
        lon_temp[0,1] = lona[(nx/2)-1,(ny/2)]
        lat_temp[1,1] = lata[(nx/2),(ny/2)]
        lon_temp[1,1] = lona[(nx/2),(ny/2)]
        lat_temp[1,0] = lata[(nx/2),(ny/2)-1]
        lon_temp[1,0] = lona[(nx/2),(ny/2)-1]
        END
    ENDCASE          
    lat_cent_temp = BILINEAR(lat_temp, [0.5], [0.5])
    lon_cent_temp = BILINEAR(lon_temp, [0.5], [0.5])
    lat_cent = lat_cent_temp[0]
    lon_cent = lon_cent_temp[0]
  ENDELSE
  grid_info={proj:grid_type, center:[lat_cent,lon_cent], $
               corners:corners, orientation:orientation, $
               standard_lats:[standard_lat1, standard_lat2], $
               nx:nx,ny:ny, dx:dx, dy:dy }
  IF ARG_PRESENT(lat) THEN lat=lata
  IF ARG_PRESENT(lon) THEN lon=lona
  IF ARG_PRESENT(lab) THEN NCDF_VARGET,unit,'lab',lab
  IF ARG_PRESENT(lob) THEN NCDF_VARGET,unit,'lob',lob
  IF ARG_PRESENT(lac) THEN NCDF_VARGET,unit,'lac',lac 
  IF ARG_PRESENT(loc) THEN NCDF_VARGET,unit,'loc',loc 
  IF ARG_PRESENT(avg) THEN NCDF_VARGET,unit,'avg',avg
  IF ARG_PRESENT(use) THEN NCDF_VARGET,unit,'use',use
  IF ARG_PRESENT(ldf) THEN NCDF_VARGET,unit,'ldf',ldf
  IF ARG_PRESENT(spr) THEN NCDF_VARGET,unit,'spr',spr
  IF ARG_PRESENT(zin) THEN NCDF_VARGET,unit,'zin',zin 
  IF ARG_PRESENT(std) THEN NCDF_VARGET,unit,'std',std
  IF ARG_PRESENT(env) THEN NCDF_VARGET,unit,'env',env 
  IF ARG_PRESENT(cpr) THEN NCDF_VARGET,unit,'cpr',cpr 
  IF ARG_PRESENT(mfa) THEN NCDF_VARGET,unit,'mfa',mfa
  IF ARG_PRESENT(mfb) THEN NCDF_VARGET,unit,'mfb',mfb 
  IF ARG_PRESENT(mfc) THEN NCDF_VARGET,unit,'mfc',mfc
  IF ARG_PRESENT(cph) THEN NCDF_VARGET,unit,'cph',cph
  IF ARG_PRESENT(cpv) THEN NCDF_VARGET,unit,'cpv',cpv
  IF ARG_PRESENT(slt) THEN NCDF_VARGET,unit,'slt',slt
  IF ARG_PRESENT(sln) THEN NCDF_VARGET,unit,'sln',sln  
  IF ARG_PRESENT(avc) THEN NCDF_VARGET,unit,'avc',avc
  IF ARG_PRESENT(mfl) THEN NCDF_VARGET,unit,'mfl',mfl   
  IF ARG_PRESENT(stl) THEN NCDF_VARGET,unit,'stl',stl 
  IF ARG_PRESENT(sbl) THEN NCDF_VARGET,unit,'sbl',sbl               
  IF ARG_PRESENT(lnd) THEN NCDF_VARGET,unit,'lnd',lnd
  IF ARG_PRESENT(g01) THEN NCDF_VARGET,unit,'g01',g01
  IF ARG_PRESENT(g06) THEN NCDF_VARGET,unit,'g06',g06
  IF ARG_PRESENT(g09) THEN NCDF_VARGET,unit,'g09',g09
  IF ARG_PRESENT(g12) THEN NCDF_VARGET,unit,'g12',g12
  IF ARG_PRESENT(tmp) THEN NCDF_VARGET,unit,'tmp',tmp
  NCDF_CLOSE, unit
ENDELSE
RETURN
END
