#!/usr/bin/perl
#script developed for use when installing the wrfsi
#software.  J.Smart 12-00 original version
#
use English;
use strict;
umask 002;
use Getopt::Long;
use Cwd;


# Set up some default roots before the help option, so that we
# capture the use of SOURCE_ROOT, INSTALLROOT, DATAROOT, and EXT_DATAROOT
# environment variables if set
my ($defsrcroot, $definstallroot, $defdataroot,$defextdataroot, $deftemplates,
    $defgeogdataroot, $defuitempdir);

if ($ENV{SOURCE_ROOT}) {
  $defsrcroot=$ENV{SOURCE_ROOT};
}else{

  if (($0 eq "install_wrfsi.pl") or ($0 eq "./install_wrfsi.pl")) {
    $defsrcroot = cwd;
  }else{
    if ($0 =~ /S*\/install_wrfsi.pl/){
      $defsrcroot = $`;
      chdir "$defsrcroot";
      $defsrcroot = cwd;
    }else{
      die "Problem determining srcroot.  Either specify --source_root or
           run from within the source directory!\n";
    }
  }

  #$defsrcroot=cwd;
}

if ($ENV{INSTALLROOT}){
  $definstallroot=$ENV{INSTALLROOT};
}else{
  $definstallroot=$defsrcroot;
}

if ($ENV{EXT_DATAROOT}) {
  $defextdataroot=$ENV{EXT_DATAROOT};
}else{
  $defextdataroot="$definstallroot/extdata";
}

if ($ENV{DATAROOT}){
  $defdataroot=$ENV{DATAROOT};
}else{
  $defdataroot="$definstallroot/domains";
}

if ($ENV{GEOG_DATAROOT}){
  $defgeogdataroot=$ENV{GEOG_DATAROOT};
}else{
  $defgeogdataroot="$defextdataroot/GEOG";
}

if ($ENV{TEMPLATES}){
  $deftemplates=$ENV{TEMPLATES};
}else{
  $deftemplates="$definstallroot/templates";
}

if ($ENV{UI_TEMPDIR}){
  $defuitempdir=$ENV{UI_TEMPDIR};
}else{
  $defuitempdir="/tmp";
}

my($INSTALLROOT,$SRCROOT,$DATAROOT,$PATH_TO_NETCDF,$EXTDATAROOT,$MACHINE_TYPE,
   $MPICH,$PBS,$GEOGDATAROOT, $TEMPLATES,$FORTC,$install_wrfsi_gui,$UI_TEMPDIR,$helpme);
my $result = GetOptions("installroot=s" => \$INSTALLROOT,
                       "source_root=s"  => \$SRCROOT,
                       "dataroot=s" => \$DATAROOT,
                       "ext_dataroot=s" => \$EXTDATAROOT,
                       "geog_dataroot=s" => \$GEOGDATAROOT,
                       "templates=s"=> \$TEMPLATES,
                       "path_to_netcdf=s" => \$PATH_TO_NETCDF,
                       "machine=s" => \$MACHINE_TYPE,
                       "pbshome=s" => \$PBS,
                       "mpich=s" => \$MPICH,
                       "fc=s" => \$FORTC, 
                       "install_ui=s" => \$install_wrfsi_gui,
                       "ui_tempdir=s" => \$UI_TEMPDIR,
                       "help" => \$helpme, 
                       );

if ($helpme) {
  print "
WRFSI Installation Script Usage:

perl install_wrfsi.pl [options]

  Valid Options:

    --installroot=INSTALLROOT
      Top level directory to install code executable scripts/programs
      DEFAULT: $definstallroot
      (Supercedes INSTALLROOT environment variable)

    --source_root=SOURCE_ROOT   
      Source root (contains src subdirectory)
      DEFAULT: $defsrcroot
      (Supercedes SOURCE_ROOT environment variable)

    --dataroot=DATAROOT
      Area where MOAD_DATAROOTs will be created (large data volume)
      DEFAULT: $defdataroot
      (Supercedes DATAROOT environment variable)

    --ext_dataroot=EXT_DATAROOT
      Area containing config and output from grib_prep program
      DEFAULT: $defextdataroot
      (Supercedes EXT_DATAROOT environment variable)

    --geog_dataroot=GEOG_DATAROOT
      Top-level directory containing global landuse/topography data directories
      DEFAULT: $defgeogdataroot
      (Supercedes GEOG_DATAROOT environment variable)

    --templates=TEMPLATES
      Directory containing template subdirectories for window_domain script
      DEFAULT: $deftemplates

    --path_to_netcdf=NETCDF
      Set to top-level netCDF installation directory (usually something like
      /usr/local/netcdf).  If this switch is not used, the code looks for
      a NETCDF environment variable.  If that is not found, the script will
      abort, as the NETCDF path is mandatory.

    --machine=machine_type
      Set to the machine type that you are building on.  The machine type must
      have a corresponding makefile_{machine_type}.inc.in file in 
      SOURCE_ROOT/include.  If not set here, the script will attempt to 
      automatically determine the type. If you need to customize an intallation,
      copy the most representative makefile_{mach}.inc.in to your own custom
      machine name file, edit it, and use this switch to specify it at
      installation.
 
    --pbshome=PBSHOME
      Top-level directory where the Portable Batch System (OpenPBS) is
      installed.
      This is not required, and is provided for convenience for those who have
      and use PBS on a large shared computer.  This path is usually something
      like /usr/local/pbs.

    --mpich=MPICH
      Path to MPI libraries.  Not required unless you are running the MPI 
      version of WRF and wish to use the WRFSI WRF run scripts.  Usually, this
      path is something like /usr/local/mpich.

    --fc=g95
      Fortran Compiler override option.  Set to 'g95' for GNU Fortran 95 on Linux 
      systems, only. 
      Note: NetCDF must also be built with g95! See wrfsi/etc/install_netcdf_g95.sh. 
      (Otherwise, the Fortran Compiler defaults to be one of the following values 
      depending on your system architecture: pgf90, xlf, fort or f90.)

    --install_ui=[y|n]
      Do you want to install the WRF SI graphical user interface? [y|n]
      If flag is not set, then user will be asked to respond at the command line.

    --ui_tempdir=UI_TEMPDIR
      Scratch directory for files created by the user interface. 
      DEFAULT: $defuitempdir
      (Supercedes UI_TEMPDIR environment variable)

}
exit;


Typical Installation:

 perl install_wrfsi.pl --path_to_netcdf=/usr/local/netcdf

This will install the code in the bin and etc subdirectories of the 
source directory created when you untarred the file.  DATAROOT and
EXTDATAROOT will also be in that same directory, and a template subdirectory
will be created there as well.\n";
exit;
}

  
# If it turns out we are on an IBM with GPFS file system,
# we will need a MODDIR variable for the makefile.inc
my $MODDIR;

print "Routine: Install_wrfsi \n\n";
my $PATH_TO_PERL=$EXECUTABLE_NAME;
chomp $PATH_TO_PERL;
if(length($PATH_TO_PERL) <=4 ){
   $PATH_TO_PERL = `which 'perl'|tail -1`;
   chomp $PATH_TO_PERL;
}
print "Path to perl: $PATH_TO_PERL\n\n";

# If srcroot was not specfied, then try and determine source root
# based on the knowledge that the install_wrfsi.pl script is usually
# found in the SRCROOT.

if (! $SRCROOT) { 
  $SRCROOT = $defsrcroot;
  $ENV{SOURCE_ROOT} = $SRCROOT;
} 

if (! $INSTALLROOT)   { 
  $INSTALLROOT = $definstallroot;
  $ENV{INSTALLROOT}=$INSTALLROOT;
}

if (! $DATAROOT) {
  if (! $ENV{DATAROOT}) {
    $DATAROOT="$INSTALLROOT/domains";
    $ENV{DATAROOT}=$DATAROOT;
  }else{
    $DATAROOT=$ENV{DATAROOT};
  }
}
if (! $EXTDATAROOT) { 
  if(! $ENV{EXT_DATAROOT}) {
    $EXTDATAROOT="$INSTALLROOT/extdata";
    $ENV{EXT_DATAROOT}=$EXTDATAROOT;
  }else{
    $EXTDATAROOT=$ENV{EXT_DATAROOT};
  }
}
if (! $GEOGDATAROOT) {
  if (! $ENV{GEOG_DATAROOT}) {
    $GEOGDATAROOT="$EXTDATAROOT/GEOG";
    $ENV{GEOG_DATAROOT} = $GEOGDATAROOT;
  }else{
    $GEOGDATAROOT=$ENV{GEOG_DATAROOT};
  }
}

if (! $TEMPLATES ) {
  if (! $ENV{TEMPLATES}){
    $TEMPLATES = "$INSTALLROOT/templates";
    $ENV{TEMPLATES}=$TEMPLATES;
  }else{
    $TEMPLATES=$ENV{TEMPLATES};
  }
}

if (! $PBS) {
  if ($ENV{PBS}) {
    $PBS = $ENV{PBS};
  }else{
    $PBS = `which qsub`; chomp $PBS;
    if (! $PBS) {
      print "No path to PBS queuing system found!\n"
    }else{
      if ($PBS =~ /(\S{1,})\/bin\/qsub$/) {
        $PBS = $1;
        $ENV{PBS} = $PBS;
      }
    }
  }
}else{
  $ENV{PBS} = $PBS;
}


if (! $MPICH) { 
  if ($ENV{MPICH}) {
    $MPICH = $ENV{MPICH};
  }else{
    $MPICH = `which mpirun`; chomp $MPICH;
    if (! $MPICH) {
      print "No path to MPICH system found!\n"
    }else{
      if ($MPICH =~ /(\S{1,})\/bin\/mpirun$/) {
        $MPICH = $1;
        $ENV{MPICH} = $MPICH;
      }
    }
  }
}else{
  $ENV{MPICH} = $MPICH;
}

# Try and determine path to netcdf if not specified

if (! $PATH_TO_NETCDF){
  print "--path_to_netcdf not specified, attempting to determine...\n";
  if ($ENV{NETCDF}){
    $PATH_TO_NETCDF = $ENV{NETCDF};
    print "netCDF path found from environment variable.\n";
  }else{
    die "No path to netcdf found.  Specify netcdf installation root by
setting --path_to_netcdf or NETCDF environment variable! \n";
  }
}else{
  $ENV{NETCDF} = $PATH_TO_NETCDF;
}
    

# Query "Do you want to install the WRF SI GUI?"
# ------------------------------------------------------
if (!$install_wrfsi_gui) {
   print "\nDo you want to install the WRF SI graphical user interface? [y|n]: ";
   my $answer = getc;
   if ($answer =~ /^y/i) {
      $install_wrfsi_gui='y';
      print "Okay... WRF SI GUI will be installed!\n\n";
   } else {
      $install_wrfsi_gui='n';
      print "Okay... WRF SI GUI will NOT be installed!\n\n";
   }
}

# If machine type not specified, then try and determine
# Currently supported are:
#  IBM with AIX, Intel with Linux, Compaq Alpha w/Linux, Sun Solaris
if (! $MACHINE_TYPE){
  print "--machine_type not specified, attempting to determine...\n";
  my $os = $^O;
  if ($os eq "aix"){
    # See if the source root is on a GPFS file system 
    print "IBM system detected...checking for GPFS file system.\n";
    my ($notgpfs, $dfres, $fspart);
    $dfres=`df $SRCROOT | tail -1`;
    if ($dfres =~ /^(\S*)/) {
      $fspart = $1;
      print "Partition used: $fspart\n";
      $notgpfs = system("(mount | grep $fspart) | grep mmfs > /dev/null");
      if (! $notgpfs){
         print "GPFS file system detected, using machine type ibmgpfs.\n";
         $MACHINE_TYPE="ibmgpfs";
         # We need to create a directory on local disk to put
         # mod files for F90 modules
         my $userid = `whoami` ; chomp $userid;
         $MODDIR="/tmp/$userid"; 
         if (! -d "$MODDIR") {
           mkdir "$MODDIR", 0777 || die
            "Cannot mkdir $MODDIR for F90 modules.\n";     
         }
      }else{
         print "Normal file system detected, using machine type ibm.\n";
         $MACHINE_TYPE="ibm";
      }
    }else{
      die "Cannot determine file system partition.\n";
    }
  }elsif ($os eq "solaris") {
    $MACHINE_TYPE = "sun";
  }elsif ($os eq "irix") {
    $MACHINE_TYPE = "sgi";
  }elsif ($os eq "dec_osf") {
    $MACHINE_TYPE = "alpha"; 
  }elsif ($os eq "linux") {
    my $arch;
    $arch = `uname -m`;
    if ($arch =~ /(\S*)/){
      $arch = $1;
    }else{
      die "Cannot determine Linux machine type.  Please use --machine_type option.\n";
    }
    if ($arch eq "i686"){
       if ($FORTC eq "g95"){
          $MACHINE_TYPE = "pcg95";
       } else {
          $MACHINE_TYPE = "pc";
       }
    }else{
       $MACHINE_TYPE = "$arch"."linux";
    }
  }else{
    $MACHINE_TYPE = `uname -m`;
  }
}
if (! $MACHINE_TYPE) { die "Undetermined machine/OS combination.  You need to specify
the machine type using the --machine_type=MACHINE option, and there must be a corresponding
makefile_MACHINE.inc.in in $SRCROOT/src/include \n"; }  

print "Compiling for $MACHINE_TYPE \n";  
print "Directory setup:\n";
print "SRCROOT = $SRCROOT \n";
print "INSTALLROOT = $INSTALLROOT \n";
print "DATAROOT = $DATAROOT \n";
print "EXT_DATAROOT = $EXTDATAROOT\n";
print "GEOG_DATAROOT = $GEOGDATAROOT\n";
print "TEMPLATES = $TEMPLATES\n";
print "NETCDF = $PATH_TO_NETCDF \n";

chdir "$SRCROOT/src" || die "Can't change to directory $SRCROOT/src: $!\n";
# modify makefile.inc.in (for all binaries)
# ------------------------------------------
# check if ijet ... remove  "-Bstatic" compile switch
my $ijet;
if( $ENV{HOSTNAME}){
    if ($ENV{HOSTNAME} =~ m/^(g|v)0.*(0|1|2|3)\.fsl\.noaa\.gov$/) {
        $ijet=1;
    }
}
chdir "include" || die "Can't change to directory src/include $!\n";
open(MF,"makefile\_$MACHINE_TYPE.inc.in") || 
     die "$0: Can't open makefile\_$MACHINE_TYPE.inc.in $! \n";
my @flines=<MF>;
my $fline;
close(MF);
open(MFOUT,">makefile.inc");
foreach $fline (@flines){
   $fline =~ s/\@top_srcdir\@/$SRCROOT/;
   $fline =~ s/\@prefix\@/$INSTALLROOT/;
   $fline =~ s/\@dataroot\@/$DATAROOT/;
   $fline =~ s/\@PERL\@/$PATH_TO_PERL/;
   $fline =~ s/\@NETCDF\@/$PATH_TO_NETCDF/;
   $fline =~ s/\@EXTDATAROOT\@/$EXTDATAROOT/;
   $fline =~ s/\@MODDIR\@/$MODDIR/;
   if($ijet==1 && $fline =~ /Bstatic/){
      $fline =~ s/\-Bstatic//;
   }
   print MFOUT $fline;
}
close(MFOUT);

# Switch on or off the trigd module.
#if($MACHINE_TYPE eq "sun"){
if($FORTC eq "g95"){
   open(TRGD,"trigd.inc");
   @flines=<TRGD>;
   close(TRGD);
   open(TRGD,">trigd.inc");
   foreach (@flines){
      if(/s*use trigd/){s/^[cC]/ /;}
      print TRGD $_;
   }
   close(TRGD);
}

#make the binaries
#make directories in installroot if necessary
#--------------------------------------------
if(!-d "$INSTALLROOT"     ){mkdir "$INSTALLROOT", 0777 
               or die "Can't mkdir $INSTALLROOT $!\n";}
if(!-d "$INSTALLROOT/bin" ){mkdir "$INSTALLROOT/bin", 0777 
               or die "Can't mkdir $INSTALLROOT/bin $!\n";}
if(!-d "$INSTALLROOT/etc" ){mkdir "$INSTALLROOT/etc", 0777 
               or die "Can't mkdir $INSTALLROOT/etc $!\n";}
if(!-d "$INSTALLROOT/data"){mkdir "$INSTALLROOT/data", 0777 
               or die "Can't mkdir $INSTALLROOT/data $!\n";}
if(!-d "$INSTALLROOT/data/log") {mkdir "$INSTALLROOT/data/log",0777 
                      or die "Can't mkdir $INSTALLROOT/data/log $!\n";}
if(!-d "$INSTALLROOT/data/siprd") {mkdir "$INSTALLROOT/data/siprd",0777 
                      or die "Can't mkdir $INSTALLROOT/data/siprd $!\n";}
if(!-d "$INSTALLROOT/data/static")
   {system("cp -rf $SRCROOT/data/cdl $INSTALLROOT/data/");
    system("cp -rf $SRCROOT/data/static $INSTALLROOT/data/");}
if(!-d "$INSTALLROOT/graphics") {mkdir "$INSTALLROOT/graphics",0777 
                      or warn "Can't mkdir $INSTALLROOT/graphics $!\n";}

# Make sure we have a data root as specified...if not, create it.
if (! -d "$DATAROOT") {
  mkdir "$DATAROOT", 0777 ||  die "Could not mkdir $DATAROOT!\n";
}

# Make the EXT_DATAROOT as necessary
if (! -d "$EXTDATAROOT") {
  my $result = system ("cp -r $SRCROOT/extdata $EXTDATAROOT");
  if ($result) { die "Could not make $EXTDATAROOT!\n";}
}else{
  if (! -d "$EXTDATAROOT/static"){
    my $result = system ("cp -r $SRCROOT/extdata/static $EXTDATAROOT/.");
    if ($result) { die "Problem updating $EXTDATAROOT/static\n" }
  }
  if (! -d "$EXTDATAROOT/log" ) {
    mkdir "$EXTDATAROOT/log", 0777 ||
      die "Could not mkdir $EXTDATAROOT/log\n";
  }
  if (! -d "$EXTDATAROOT/extprd" ) {
    mkdir "$EXTDATAROOT/extprd", 0777 ||
      die "Could not mkdir $EXTDATAROOT/extprd\n";
  }
  if (! -d "$EXTDATAROOT/work" ) {
    mkdir "$EXTDATAROOT/work", 0777 ||
        die "Could not mkdir $EXTDATAROOT/work\n";
  }
}

require "$SRCROOT/etc/wrfsi_utils.pm";

# Touch gui/src/makefile.inc for gui/src/Makefile.
system("touch $SRCROOT/gui/src/makefile.inc");

chdir "$SRCROOT";
print "Make and install wrfsi binaries in $INSTALLROOT\n";
system("make realclean > make_install.log");
system("make all  >> make_install.log");
system("make ggminstall >> make_install.log");
system("make gprepinstall >> make_install.log");
system("make hinterpinstall >> make_install.log");
system("make vinterpinstall >> make_install.log");
system("make utilinstall >> make_install.log"); 
system("make spostinstall >> make_install.log"); 

print "Done. Checking file $SRCROOT/make_install.log\n";
open(MIO,"$SRCROOT/make_install.log");
my @mio=<MIO>;
close(MIO);
foreach(@mio){next if !/error/i && !/ignored/i;
              print "$_\n";}

#copy source root scripts to installroot
#---------------------------------------
if($SRCROOT ne $INSTALLROOT){
   opendir(PLS,"$SRCROOT/etc");
   my @pscripts=readdir PLS;
   foreach(@pscripts){
      next if(/^\./ || /\.in$/);
      system("cp $SRCROOT/etc/$_ $INSTALLROOT/etc/");
      chmod 0775, "$INSTALLROOT/etc/$_";
   }
}

#modify scripts (as indicated) in etc
#------------------------------------


my (@perl_scripts) = qw(wrfsi.pl grib_prep.pl localize_domain.pl laps_driver.pl
                       window_domain_rt.pl  rt_wrf.pl run_wrf.pl
                       setup_mpiconf.pl wrfprep.pl setup-mpi.sh
                       ftp_eta_tiles.csh process_eta218_tiles.csh
                       grib_prep218.pl generate_images.pl sync_wrfnl.pl);
my ($perl_script, $fname_len, $filename);
chdir "$SRCROOT/etc";
my $type="wrfsi";
foreach $perl_script (@perl_scripts) {

   &wrfsi_utils::make_script($SRCROOT, $INSTALLROOT,
                             $perl_script, $type);
   # Print "Done making $perl_script\n";
   chmod 0775, "$INSTALLROOT/etc/$perl_script";
}

# Copy NCL script to SRCROOT/graphics/ncl.
chmod 0775, "$SRCROOT/etc/generate_images.pl";
system("cp -f $SRCROOT/etc/generate_images.pl $SRCROOT/graphics/ncl");

# Make all scripts and files available to the INSTALLROOT. 
if($SRCROOT ne $INSTALLROOT){
   foreach $perl_script (@perl_scripts) {
      system("cp $SRCROOT/etc/$perl_script $INSTALLROOT/etc/.");
      chmod 0775, "$INSTALLROOT/etc/$perl_script";
   }
   system("cp -rf $SRCROOT/data/static $INSTALLROOT/data/");
   system("cp -rf $SRCROOT/data/cdl    $INSTALLROOT/data/");

   # Make NCL script and files available to the INSTALLROOT. 
   system("cp -rf $SRCROOT/graphics/ncl $INSTALLROOT/graphics");
}


#  Set up the templates directory using the TEMPLATE variable

my $deftemp = "$TEMPLATES/default";
if (! -d "$TEMPLATES"){
  mkdir "$TEMPLATES", 0777;
  system ("cp -r $SRCROOT/data/static $deftemp");
}else{
  # Open the permissions up for writing, since
  # a previous run may have closed permissions
  system("chmod -R a+w $deftemp");
  if (! -d "$deftemp"){
    system ("cp -r $SRCROOT/data/static $deftemp"); 
  }else{
    system ("cp $SRCROOT/data/static/* $deftemp/.");
  }
}

# Create default dataroot.txt file for gui.
open(DT, ">$deftemp/dataroot.txt");
print DT "$DATAROOT\n";
close(DT);

# Copy default domain.gif image to deftemp for gui.
system("cp -f $SRCROOT/gui/data/static/domain.gif $deftemp"); 

# Edit the template directory/wrfsi.nl for some entries we know.
open(WNL, "$deftemp/wrfsi.nl");
my @nllines = <WNL>;
close(WNL);
open(WNL, ">$deftemp/wrfsi.nl");
foreach $fline (@nllines){

   if ($fline =~ /TOPO_30S/i) {
     $fline = " TOPO_30S = \'$GEOGDATAROOT/topo_30s\',\n";
   }
   if ($fline =~ /LANDUSE_30S/i) {
     $fline = " LANDUSE_30S = \'$GEOGDATAROOT/landuse_30s\',\n";
   }
   if ($fline =~ /SOILTYPE_TOP_30S/i) {
     $fline = " SOILTYPE_TOP_30S = \'$GEOGDATAROOT/soiltype_top_30s\',\n";
   }
   if ($fline =~ /SOILTYPE_BOT_30S/i) {
     $fline = " SOILTYPE_BOT_30S = \'$GEOGDATAROOT/soiltype_bot_30s\',\n";
   }
   if ($fline =~ /GREENFRAC/i) {
     $fline = " GREENFRAC = \'$GEOGDATAROOT/greenfrac\',\n";
   }
   if ($fline =~ /SOILTEMP_1DEG/i) {
     $fline = " SOILTEMP_1DEG = \'$GEOGDATAROOT/soiltemp_1deg\',\n";
   }
   if ($fline =~ /ALBEDO_NCEP/i) {
     $fline = " ALBEDO_NCEP = \'$GEOGDATAROOT/albedo_ncep\',\n";
   }
   if ($fline =~ /MAXSNOWALB/i) {
     $fline = " MAXSNOWALB = \'$GEOGDATAROOT/maxsnowalb\',\n";
   }
   if ($fline =~ /ISLOPE/i) {
     $fline = " ISLOPE = \'$GEOGDATAROOT/islope\',\n";
   }
   if ($fline =~ /LBCPATH/i) {
     $fline = " LBCPATH = \'$EXTDATAROOT\/extprd',\n";
   }
   if ($fline =~ /ANALPATH/i) {
     $fline = " ANALPATH = \'$EXTDATAROOT/extprd\',\n";
   }
   if ($fline =~ /CONSTANTS_PATH/i) {
     $fline = " CONSTANTS_PATH = \'$EXTDATAROOT/extprd\',\n";
   }
   if ($fline =~ /LSMPATH/i) {
     $fline = " LSMPATH = \'$EXTDATAROOT/extprd\',\n";
   }
   print WNL $fline;
}
close(WNL);

# Make the default temp directory read-only for safety.
system("chmod -R a-w $deftemp");

# Make SRCROOT/data/static/wrfsi.nl identical to the templates/default/wrfsi.nl.
system("cp -f $deftemp/wrfsi.nl $SRCROOT/data/static/wrfsi.nl"); 

print "Check installation in $INSTALLROOT/etc\n";
system("ls -rtal $INSTALLROOT/etc/*.pl");
print "Check installation in $INSTALLROOT/bin\n";
system("ls -rtal $INSTALLROOT/bin");


# Check to see if GEOG data is present.  
my ($topo_30s, $landuse_30s, $soiltype_top_30s,
    $soiltype_bot_30s, $greenfrac, $soiltemp_1deg, $albedo_ncep,
    $islope, $maxsnowalb);

print "\n";
print "Checking to see if GEOG data is present...\n";
print "  30 second topography  ..........  ";
if (-e "$GEOGDATAROOT/topo_30s/UHEADER") {
  print "found\n";
  $topo_30s = 1;
}else{
  print "not found\n";
}

print "\n";
print "  30 second USGS land use ........  ";
if (-e "$GEOGDATAROOT/landuse_30s/VHEADER") {
  print "found\n";
  $landuse_30s = 1;
}else{
  print "not found\n";
}

print "\n"; 
print "  30 second soil category (top)...  ";
if (-e "$GEOGDATAROOT/soiltype_top_30s/OHEADER") {
  print "found\n";
  $soiltype_top_30s = 1;
}else{
  print "not found\n";
}

print "\n";
print "  30 second soil category (bot)...  ";
if (-e "$GEOGDATAROOT/soiltype_bot_30s/OHEADER") {
  print "found\n";
   $soiltype_bot_30s = 1;
}else{
  print "not found\n";
}

print "\n";
print "  Greenness fraction .............  ";
if (-e "$GEOGDATAROOT/greenfrac/GHEADER") {
  print "found\n";
  $greenfrac = 1;
}else{
  print "not found\n";
}

print "\n";
print "  1 deg deep layer soil temp .....  ";
if (-e "$GEOGDATAROOT/soiltemp_1deg/THEADER") {
  print "found\n";
  $soiltemp_1deg = 1;
}else{
  print "not found\n";
}

print "\n";
print "  NCEP monthly albedo ............  ";
if (-e "$GEOGDATAROOT/albedo_ncep/AHEADER") {
  print "found\n";
  $albedo_ncep = 1;
}else{
  print "not found\n";
}

print "\n";
print "  NCEP Slope Category ............  ";
if (-e "$GEOGDATAROOT/islope/IHEADER") {
  print "found\n";
  $islope = 1;
}else{
  print "not found\n";
}

print "\n";
print "  NCEP Max Snow Albedo ...........  ";
if (-e "$GEOGDATAROOT/maxsnowalb/MHEADER") {
  print "found\n";
  $maxsnowalb = 1;
}else{
  print "not found\n";
}
# Create the config_paths file for use by the GUI and document all paths.
# ---------------------------------------------
open (CFG, ">$INSTALLROOT/config_paths");
print CFG "\$DATAROOT='$DATAROOT';\n";
print CFG "\$EXT_DATAROOT='$EXTDATAROOT';\n";
print CFG "\$GEOG_DATAROOT='$GEOGDATAROOT';\n";
print CFG "\$TEMPLATES='$TEMPLATES';\n";
print CFG "\$INSTALLROOT='$INSTALLROOT';\n";
print CFG "\$SOURCE_ROOT='$SRCROOT';\n";
print CFG "\$UI_TEMPDIR='$UI_TEMPDIR';\n";
print CFG "\$PATH_TO_PERL='$PATH_TO_PERL';\n\n";
print CFG "# This file records the values for the default paths as\n";
print CFG "# defined when the install script (install_wrfsi.pl) ran.\n";
print CFG "# This file was designed for the WRF SI GUI. Definitions:\n#\n";
print CFG "# DATAROOT:  Directory under which MOAD_DATAROOTs will be created.\n#\n";
print CFG "# EXT_DATAROOT:  Directory containing configuration files, \n";
print CFG "# work space, and output from the grib_prep program. This\n";
print CFG "# EXT_DATAROOT can support multiple MOAD_DATAROOTs.\n#\n";
print CFG "# GEOG_DATAROOT:  Top directory containing subdirectories of \n";
print CFG "# tile land use, topography, etc. data for use when localizing.\n#\n";
print CFG "# TEMPLATES:  Directory containing template files.\n#\n";
print CFG "# INSTALLROOT:  Path to compile/installed code and scripts.\n#\n";
print CFG "# SOURCE_ROOT:  Path to original source code.\n#\n";
print CFG "# PATH_TO_PERL: Path to the system location for perl.;\n\n";
close (CFG);


if($install_wrfsi_gui =~ /^y/i) {

  # Run $SRCROOT/gui/install_gui.pl, which will:
  # o modify gui scripts (as indicated) in gui and gui/guiTk
  #   via calls to &wrfsi_utils::make_script(...), etc
  # o check for installation of Perl/Tk by running the gui.
  # o install Perl/Tk, if necessary.
  #------------------------------------
  chdir "$SRCROOT";
  system("$PATH_TO_PERL gui/install_gui.pl --source_root=$SRCROOT --installroot=$INSTALLROOT --ui_tempdir=$UI_TEMPDIR 2> gui/gui_install.log");

}


# Installation complete
# ---------------------------------------------
print "\n\n";
print "**********************************************************************\n"; 
print "Installation complete.  Next steps:\n";
print "**********************************************************************\n\n";
if ( (! $topo_30s) or (! $landuse_30s) or 
     (! $soiltype_top_30s) or (! $soiltype_bot_30s) or (! $greenfrac) or
     (! $soiltemp_1deg) or (! $albedo_ncep) ) {
  print "-- Populate the global geographical (topography, etc.) data sets:\n";
  print "    a.  cd into $GEOGDATAROOT\n";
  print "    b.  FTP anonomously to aftp.fsl.noaa.gov\n";
  print "    c.  cd into divisions/frd-laps/WRFSI/Geog_Data\n";
  print "    d.  Obtain the following files:\n";
  if (! $topo_30s){
    print "        topo_30s/topo_30s.??.tar.gz\n";
  }
  if (! $landuse_30s) {
    print "        landuse_30s/landuse_30s.??.tar.gz\n";
  }
  if (! $soiltype_top_30s){
    print "        soiltype_top_30s/soiltype_top_30s.??.tar.gz\n";
  }
  if (! $soiltype_bot_30s){
    print "        soiltype_bot_30s/soiltype_top_30s.??.tar.gz\n";
  }
  if (! $greenfrac) {
    print "        greenfrac.tar.gz\n";
  }
  if (! $soiltemp_1deg) {
    print "        soiltemp_1deg.tar.gz\n";
  }
  if (! $albedo_ncep) {
    print "        albedo_ncep.tar.gz\n";
  }
  if (! $islope) {
    print "        islope.tar.gz\n";
  }
  if (! $maxsnowalb) {
    print "        maxsnowalb.tar.gz\n";
  }
  print "\n";
  print "    e.  Untar (tar xvfz filename) on each of the downloaded files.\n\n";
}

print "-- Set up grib_prep to ingest your GRIB data sets:\n";
print "    a.  cd $EXTDATAROOT/static\n";
print "    b.  Edit grib_prep.nl and/or the Vtables\n\n";
if (-e "$INSTALLROOT/wrf_tools") {
  print "-- Use the WRFSI GUI to set up your domains.\n";
  print "   >>> use this script '$INSTALLROOT/wrf_tools' to start the WRFSI GUI <<<\n\n";
}
print "-- See $SRCROOT/HOW_TO_RUN.txt for more info.\n";



exit;
