!dis   
!dis    Open Source License/Disclaimer, Forecast Systems Laboratory
!dis    NOAA/OAR/FSL, 325 Broadway Boulder, CO 80305
!dis    
!dis    This software is distributed under the Open Source Definition,
!dis    which may be found at http://www.opensource.org/osd.html.
!dis    
!dis    In particular, redistribution and use in source and binary forms,
!dis    with or without modification, are permitted provided that the
!dis    following conditions are met:
!dis    
!dis    - Redistributions of source code must retain this notice, this
!dis    list of conditions and the following disclaimer.
!dis    
!dis    - Redistributions in binary form must provide access to this
!dis    notice, this list of conditions and the following disclaimer, and
!dis    the underlying source code.
!dis    
!dis    - All modifications to this software must be clearly documented,
!dis    and are solely the responsibility of the agent making the
!dis    modifications.
!dis    
!dis    - If significant modifications or enhancements are made to this
!dis    software, the FSL Software Policy Manager
!dis    (softwaremgr@fsl.noaa.gov) should be notified.
!dis    
!dis    THIS SOFTWARE AND ITS DOCUMENTATION ARE IN THE PUBLIC DOMAIN
!dis    AND ARE FURNISHED "AS IS."  THE AUTHORS, THE UNITED STATES
!dis    GOVERNMENT, ITS INSTRUMENTALITIES, OFFICERS, EMPLOYEES, AND
!dis    AGENTS MAKE NO WARRANTY, EXPRESS OR IMPLIED, AS TO THE USEFULNESS
!dis    OF THE SOFTWARE AND DOCUMENTATION FOR ANY PURPOSE.  THEY ASSUME
!dis    NO RESPONSIBILITY (1) FOR THE USE OF THE SOFTWARE AND
!dis    DOCUMENTATION; OR (2) TO PROVIDE TECHNICAL SUPPORT TO USERS.
!dis   
!dis 


MODULE domain_info

! This module sets up all of the information pertaining to a single WRF domain
! (projection info, dimensions, lat/lon arrays, etc.) for items needed during
! horizontal interpolation from a source grid to the WRF grid and for populating
! the WRF domain metadata record.


  USE map_utils
  USE wrfsi_static
  USE wrf_metadata
  USE hinterp_setup

  IMPLICIT NONE
  
  
  ! Variables needed during interpolation 
  TYPE(proj_info)   :: proj_wrf
  TYPE(proj_info)   :: proj_moad
  REAL, ALLOCATABLE :: latitude( : , : , : )  ! (nx,ny,num_staggers)
  REAL, ALLOCATABLE :: longitude( : , : , : )
  REAL, ALLOCATABLE :: landmask_wrf ( : , : ) ! Land(1)/Water(0) Mask on T grid
!mp
  REAL, ALLOCATABLE :: cosalp( : , : )
  REAL, ALLOCATABLE :: sinalp( : , : )
!mp
	
  INTEGER           :: nx
  INTEGER           :: ny
CONTAINS
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE init_wrf_domain(moad_dataroot,domain_num)

    ! This routine will read the WRF static data and populate the global and
    ! domain metadata fields (global_meta and dom_meta) related to the 
    ! projection, dimenstions, etc.  Things related to nesting are somewhat
    ! hardcoded for now until nesting is supported.  

    IMPLICIT NONE
   
    CHARACTER(LEN=*), INTENT(IN)       :: moad_dataroot
    INTEGER, INTENT(IN)                :: domain_num
    CHARACTER(LEN=32)                  :: projection
!mp
    CHARACTER(LEN=3)                   :: varnam
!mp
    REAL                               :: lat1
    REAL                               :: lon1
    REAL                               :: dx_meters
    REAL                               :: dy_meters
    REAL                               :: stdlon
    REAL                               :: truelat1
    REAL                               :: truelat2
    CHARACTER (LEN=4)                  :: horiz_stagger_type
    CHARACTER (LEN=8)                  :: known_loc
    INTEGER                            :: parent_id
    INTEGER                            :: origin_parent_x,end_parent_x
    INTEGER                            :: origin_parent_y,end_parent_y
    INTEGER                            :: origin_parent_z
    INTEGER                            :: ratio_to_parent,nest_nx,nest_ny
    REAL, ALLOCATABLE                  :: lat_temp( : , : )
    REAL, ALLOCATABLE                  :: lon_temp( : , : )
    INTEGER                            :: s,nx_s,ny_s
    REAL                               :: latc, lonc ! Center lat/lon

    ! Get the dimensions.  
    CALL get_wrfsi_static_dims(moad_dataroot, domain_num,nx, ny)
    
    ! Get the projection
    CALL get_wrfsi_static_proj(moad_dataroot,domain_num,projection, lat1, lon1, &
                               dx_meters, dy_meters, stdlon, truelat1, &
                               truelat2)


    ! Safety check to ensure longitudes range from -180->180 E

    IF ( stdlon .GT. 180 ) stdlon = stdlon - 360
    IF (stdlon .LT. -180.) stdlon = stdlon + 360
   
    IF ( lon1 .GT. 180 ) lon1= lon1 - 360
    IF (stdlon .LT. -180.) lon1 = lon1 + 360

    ! Initialize the proj_wrf structure for this domain
    IF (projection(1:8) .EQ. 'MERCATOR') THEN
      CALL map_set(PROJ_MERC,lat1,lon1,dx_meters,0.,truelat1, 0.,nx,ny,proj_wrf)
    ELSE IF (projection(1:17) .EQ. 'LAMBERT CONFORMAL')THEN
      CALL map_set(PROJ_LC, lat1, lon1, dx_meters, stdlon, truelat1,truelat2, &
                   nx,ny,proj_wrf)
    ELSE IF (projection(1:19) .EQ. 'POLAR STEREOGRAPHIC')THEN
      CALL map_set(PROJ_PS, lat1, lon1, dx_meters, stdlon, truelat1, 0., &
                  nx,ny,proj_wrf)
!mp
    ELSE IF (projection(1:14) .EQ. 'ROTATED LATLON')THEN
      CALL map_set(PROJ_ROTLAT, lat1, lon1, dx_meters, stdlon, truelat1, 0., &
                  nx,ny,proj_wrf)
!mp
    ELSE
      PRINT '(2A)', 'Unrecognized map projection: ', projection
      STOP 'INIT_WRF_DOMAIN'
    ENDIF
    IF (domain_num .EQ. 1) THEN
      proj_moad = proj_wrf
    ENDIF

    ! Some metadata info for where this domain sits w.r.t. its parent.
    CALL get_wrfsi_static_nestinfo(moad_dataroot,domain_num,parent_id, &
       origin_parent_x, origin_parent_y, end_parent_x,end_parent_y, &
       ratio_to_parent,nest_nx,nest_ny) 

 
    ! Set up for staggering, which is somewhat hardcoded for an
    ! Arakawa-C stagger for now.  In the case of the Arakawa-C stagger,
    ! hinterp interpolates all of the met variables to the non-staggered
    ! grid ("n"), except for the LSM "masked" variables such as snow cover,
    ! ice, soil stuff, etc.  So, for hinterp, all we need are the lat/lon
    ! arrays for the non-staggered grid and the lat/lon arrays for the
    ! mass grid. 

!mp	generalize

        IF (projection(1:14) .EQ. 'ROTATED LATLON')THEN
    horiz_stagger_type = 'AR-E'
    global_meta%horiz_stagger_type = horiz_stagger_type   
	ELSE
    horiz_stagger_type = 'AR-C'
    global_meta%horiz_stagger_type = horiz_stagger_type   
        ENDIF

!mp

    IF (ALLOCATED(latitude)) DEALLOCATE(latitude)
    IF (ALLOCATED(longitude)) DEALLOCATE(longitude)
    IF (ALLOCATED(cosalp)) DEALLOCATE(cosalp)
    IF (ALLOCATED(sinalp)) DEALLOCATE(sinalp)
 
    IF (horiz_stagger_type .EQ. 'AR-C') THEN
      global_meta%num_stagger_xy = 4 ! (T,U, and V grids plus non-staggered grid)

      global_meta%stagger_dir_x(t_ind) = 0.5  ! T grid
      global_meta%stagger_dir_y(t_ind) = 0.5  ! T grid
 
      global_meta%stagger_dir_x(u_ind) = 0.0  ! U grid
      global_meta%stagger_dir_y(u_ind) = 0.5  ! U grid

      global_meta%stagger_dir_x(v_ind) = 0.5  ! V grid
      global_meta%stagger_dir_y(v_ind) = 0.0  ! V grid

      global_meta%stagger_dir_x(n_ind) = 0.0   ! Non-staggered
      global_meta%stagger_dir_y(n_ind) = 0.0

      ALLOCATE( latitude (nx,ny,global_meta%num_stagger_xy) )
      ALLOCATE( longitude(nx,ny,global_meta%num_stagger_xy) )
      ALLOCATE( lat_temp(nx,ny) )
      ALLOCATE( lon_temp(nx,ny) )

      ! Get non-staggered lat/lons
      CALL get_wrfsi_static_latlon(moad_dataroot,domain_num, 'N', lat_temp, lon_temp)
      latitude(:,:,n_ind) = lat_temp(:,:)
      longitude(:,:,n_ind) = lon_temp(:,:)
      
      ! Get latlons for T grid
      CALL get_wrfsi_static_latlon(moad_dataroot,domain_num, 'T', lat_temp, lon_temp)
      latitude(:,:,t_ind) = lat_temp(:,:)
      longitude(:,:,t_ind) = lon_temp(:,:)      

      ! Get latlons for U grid
      CALL get_wrfsi_static_latlon(moad_dataroot,domain_num, 'U', lat_temp, lon_temp)
      latitude(:,:,u_ind) = lat_temp(:,:)
      longitude(:,:,u_ind) = lon_temp(:,:) 

      ! Get latlons for V grid
      CALL get_wrfsi_static_latlon(moad_dataroot,domain_num, 'V', lat_temp, lon_temp)
      latitude(:,:,v_ind) = lat_temp(:,:)
      longitude(:,:,v_ind) = lon_temp(:,:)

      DEALLOCATE(lat_temp)
      DEALLOCATE(lon_temp)

!mp
     ELSEIF (horiz_stagger_type .EQ. 'AR-E') THEN

      global_meta%num_stagger_xy = 2 ! (H and wind grids)

      global_meta%stagger_dir_x(1) = 0.0  ! H grid
      global_meta%stagger_dir_y(1) = 0.0  ! H grid
 
      global_meta%stagger_dir_x(2) = 0.0  ! wind grid
      global_meta%stagger_dir_y(2) = 0.0  ! wind grid

      ALLOCATE( latitude (nx,ny,global_meta%num_stagger_xy) )
      ALLOCATE( longitude(nx,ny,global_meta%num_stagger_xy) )
      ALLOCATE( lat_temp(nx,ny) )
      ALLOCATE( lon_temp(nx,ny) )

      ! MASS Point lat/lons
      CALL get_wrfsi_static_latlon(moad_dataroot,domain_num, 'H', lat_temp, lon_temp)
      latitude(:,:,1) = lat_temp(:,:)
      longitude(:,:,1) = lon_temp(:,:)
      
      ! WIND point lat/lons
      CALL get_wrfsi_static_latlon(moad_dataroot,domain_num, 'W', lat_temp, lon_temp)
      latitude(:,:,2) = lat_temp(:,:)
      longitude(:,:,2) = lon_temp(:,:)      

	ALLOCATE(COSALP(nx,ny),SINALP(nx,ny))
	varnam='cpr'
	CALL get_wrfsi_static_2d(moad_dataroot,domain_num, varnam, COSALP)
	varnam='spr'
	CALL get_wrfsi_static_2d(moad_dataroot,domain_num, varnam, SINALP)

!mp

	write(6,*) 'attempting to deal with E-grid'
	write(6,*) 'lat,lon: ', latitude(1,1,1), longitude(1,1,1)
	write(6,*) 'rotation angles: ', COSALP(1,1), SINALP(1,1)

	ELSE

      PRINT '(2A)', 'Horizontal stagger type not supported: ', horiz_stagger_type
      STOP 'INIT_WRF_DOMAIN'
    ENDIF   

   
    ! If this is the MOAD (domain_num = 1), then populate as much of 
    ! global metadata as possible that has not already been filled

    ! We need to compute the center lat/lon of the domain and store these
    ! values into the global metadata. This is a change requested by NCAR
    ! (we used to store the SW corner) and implemented by B. Shaw.
    ! in Mar 2002. NOTE: ONLY THE MOAD center is stored in the global metadata.
    ! 
     
    IF (domain_num .EQ. 1) THEN
      known_loc = 'CENTER  '

      CALL ij_to_latlon(proj_moad, FLOAT(nx+1)/2., FLOAT(ny+1)/2., &
                        latc,lonc)

	if (proj_wrf%code .eq. 203) then ! rotlat
	latc=truelat1
	lonc=stdlon
	endif 

      PRINT '("MOAD CENTER LAT/LON =",2F10.4)', latc,lonc

	write(6,*) 'in here latc,lonc: ',latc,lonc 

      global_meta%map_projection = projection
      global_meta%moad_known_lat = latc
      global_meta%moad_known_lon = lonc
      global_meta%moad_known_loc = known_loc
      global_meta%moad_stand_lats(1) = truelat1
      global_meta%moad_stand_lats(2) = truelat2
      global_meta%moad_stand_lons(1) = stdlon
      global_meta%moad_delta_x = dx_meters
      global_meta%moad_delta_y = dy_meters
      global_meta%num_stagger_z = 1
      global_meta%stagger_dir_z(:) = 0.
      global_meta%num_domains = num_domains_to_proc
    ENDIF

    ! Populate as many of the domain metadata fields as possible
    PRINT *, 'Populating domain metadata for domain: ', domain_num
    dom_meta%id = domain_num
    dom_meta%parent_id = parent_id
    dom_meta%dyn_init_src = 'SI      '
    dom_meta%static_init_src = 'SI      '
    dom_meta%origin_parent_x = origin_parent_x
    dom_meta%origin_parent_y = origin_parent_y
    dom_meta%origin_parent_z = origin_parent_z
    dom_meta%ratio_to_parent = ratio_to_parent
    dom_meta%delta_x = dx_meters
    dom_meta%delta_y = dy_meters
    dom_meta%top_level = ptop_in_Pa
    dom_meta%xdim = nx
    dom_meta%ydim = ny
  
    ! Fill dom_meta%corner_lats/corner_lons, clockwise starting with SW
      
    DO s = 1, global_meta%num_stagger_xy
            SELECT CASE (s)
        CASE(t_ind)
          nx_s = nx - 1
          ny_s = ny - 1
        CASE(u_ind)
          nx_s = nx
          ny_s = ny - 1
        CASE(v_ind)
          nx_s = nx - 1
          ny_s = ny
        CASE(n_ind)
          nx_s = nx
          ny_s = ny
      END SELECT
      dom_meta%corner_lats(1,s) = latitude(1,1,s)
      dom_meta%corner_lons(1,s) = longitude(1,1,s)
      dom_meta%corner_lats(2,s) = latitude(1,ny_s,s)
      dom_meta%corner_lons(2,s) = longitude(1,ny_s,s)
      dom_meta%corner_lats(3,s) = latitude(nx_s,ny_s,s)
      dom_meta%corner_lons(3,s) = longitude(nx_s,ny_s,s)
      dom_meta%corner_lats(4,s) = latitude(nx_s,1,s)
      dom_meta%corner_lons(4,s) = longitude(nx_s,1,s)
    ENDDO

    ! Now, if we desire to input any data needed by the land
    ! surface model component of WRF, we need to get the land
    ! mask field for this domain for use in the masked
    ! interpolation routines.  Note that the land mask 
    ! is valid for the T grid!

    IF (ALLOCATED(landmask_wrf)) DEALLOCATE(landmask_wrf)
    ALLOCATE(landmask_wrf(nx,ny))
    CALL get_wrfsi_static_landmask(moad_dataroot,domain_num,landmask_wrf)
    RETURN
  END SUBROUTINE init_wrf_domain                    

END MODULE domain_info
