!dis   
!dis    Open Source License/Disclaimer, Forecast Systems Laboratory
!dis    NOAA/OAR/FSL, 325 Broadway Boulder, CO 80305
!dis    
!dis    This software is distributed under the Open Source Definition,
!dis    which may be found at http://www.opensource.org/osd.html.
!dis    
!dis    In particular, redistribution and use in source and binary forms,
!dis    with or without modification, are permitted provided that the
!dis    following conditions are met:
!dis    
!dis    - Redistributions of source code must retain this notice, this
!dis    list of conditions and the following disclaimer.
!dis    
!dis    - Redistributions in binary form must provide access to this
!dis    notice, this list of conditions and the following disclaimer, and
!dis    the underlying source code.
!dis    
!dis    - All modifications to this software must be clearly documented,
!dis    and are solely the responsibility of the agent making the
!dis    modifications.
!dis    
!dis    - If significant modifications or enhancements are made to this
!dis    software, the FSL Software Policy Manager
!dis    (softwaremgr@fsl.noaa.gov) should be notified.
!dis    
!dis    THIS SOFTWARE AND ITS DOCUMENTATION ARE IN THE PUBLIC DOMAIN
!dis    AND ARE FURNISHED "AS IS."  THE AUTHORS, THE UNITED STATES
!dis    GOVERNMENT, ITS INSTRUMENTALITIES, OFFICERS, EMPLOYEES, AND
!dis    AGENTS MAKE NO WARRANTY, EXPRESS OR IMPLIED, AS TO THE USEFULNESS
!dis    OF THE SOFTWARE AND DOCUMENTATION FOR ANY PURPOSE.  THEY ASSUME
!dis    NO RESPONSIBILITY (1) FOR THE USE OF THE SOFTWARE AND
!dis    DOCUMENTATION; OR (2) TO PROVIDE TECHNICAL SUPPORT TO USERS.
!dis   
!dis 

MODULE hinterp_setup

   USE wrf_metadata
   USE wrfsi_maxdims
   USE date_pack
   USE wrfsi_nl

   IMPLICIT NONE

   CHARACTER(LEN=255)              :: moad_dataroot
   ! Setup for processing times
   INTEGER                         :: num_output_times
   CHARACTER(LEN=19), ALLOCATABLE  :: time_list(:)
 
   ! Setup for domain processing
   INTEGER                         :: num_domains_to_proc
   INTEGER,ALLOCATABLE             :: domain_ids(:)
   
   ! Configuration for interpolation of pressure-level files
   REAL                            :: ptop_in_Pa
   INTEGER                         :: num_new_levels
   REAL, ALLOCATABLE               :: new_levels (:)

   ! Miscellaneous configuration items
   INTEGER :: interp_method
   INTEGER :: interp_method_lsm
   LOGICAL :: verbose

   ! Input file processing from grib_prep
   CHARACTER (LEN=200) , ALLOCATABLE :: init_prefix (:),lbc_prefix(:)
   CHARACTER (LEN=200)               :: lsm_prefix
   CHARACTER (LEN=200) , ALLOCATABLE :: constants_names(:)
   CHARACTER (LEN=200)               :: output_prefix
   INTEGER :: num_files_per_single_time_init, num_files_per_single_time_lbc
   INTEGER :: num_files_constants
   LOGICAL :: separate_lsm
 
 
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
CONTAINS
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE init_hinterp(moad_dataroot)

   IMPLICIT NONE
   
   CHARACTER(LEN=*), INTENT(IN) :: moad_dataroot
   CHARACTER(LEN=255)           :: nl_file
   INTEGER                      :: nl_unit
   LOGICAL                      :: nl_exists
   INTEGER                      :: status
   LOGICAL                      :: already_opened
 
   CHARACTER(LEN=80)      :: simulation_name
   CHARACTER(LEN=80)	  :: user_desc
   INTEGER                :: start_year       , &
                             start_month      , &
                             start_day        , &
                             start_hour       , &
                             start_minute = 0 , &
                             start_second = 0
   INTEGER                :: end_year         , &
                             end_month        , &
                             end_day          , &
                             end_hour         , &
                             end_minute   = 0 , &
                             end_second   = 0
   INTEGER                :: interval
   
   INTEGER                :: i , j
   
   NAMELIST / project_id / simulation_name, user_desc
   NAMELIST / filetimespec / start_year   , start_month  , start_day    , &
                        start_hour   , start_minute , start_second , &
                        end_year     , end_month    , end_day      , &
                        end_hour     , end_minute   , end_second   , &
                        interval

   !  This data is the local stuff for the time loop.  We
   !  compute the incremental times from the starting and
   !  ending times defined in the NAMELIST. 

   CHARACTER (LEN=19) :: current_time, prev_time, start_time, end_time
   CHARACTER (LEN=19),ALLOCATABLE :: output_times_temp(:)
   CHARACTER(LEN=200) :: path_to_data

   num_new_levels= 0 
   ! Build the namelist file name
   nl_file = TRIM(moad_dataroot) // '/static/wrfsi.nl'
   INQUIRE(FILE=nl_file, EXIST=nl_exists)
   IF (.NOT. nl_exists) THEN
     PRINT '(2a)', 'WRFSI Namelist not found: ', nl_file
     STOP 'INIT_HINERP'
   ENDIF
  
   !  Find a spare UNIT number for the NAMELIST file.

   namelist_unit : DO nl_unit = 10, 99
      INQUIRE (UNIT=nl_unit, OPENED=already_opened)
      IF ( .NOT. already_opened) THEN
         EXIT namelist_unit
      END IF
   END DO namelist_unit   
   
    !  Play it safe and REWIND the file.

   PRINT '(A,A/)','Trying to open file: ',nl_file

   !  OPEN the NAMELIST file with the given name and UNIT.

   OPEN ( UNIT=nl_unit , &
          FILE=nl_file , &
          STATUS = 'OLD' , &
          FORM = 'FORMATTED' , &
          IOSTAT = status )
   
   IF ( status .NE. 0 ) THEN
      PRINT '(A)' , 'Error opening NAMELIST file.'
      STOP 'INIT_HINTERP'
   END IF

   REWIND(UNIT=nl_unit)

   ! Set default values where appropriate
   
   !  Read each record from the NAMELIST file.
   READ ( UNIT = nl_unit, NML = project_id )
   REWIND (nl_unit)
   READ ( UNIT = nl_unit , NML = filetimespec )

   PRINT '(A)','Successfully read the project and filetimespec namelists'
   CLOSE (UNIT=nl_unit)

   CALL read_interp_nl(moad_dataroot)  

   ! The number of domains that will be processed will
   ! be one more than the number of active sub-nests to
   ! account for MOAD, which we will always run.
   num_domains_to_proc = 1 + num_active_subnests
   interp_method = hinterp_method
   interp_method_lsm = lsm_hinterp_method
   verbose = verbose_log
   ptop_in_pa = ptop_pa 

   ! Domain processing.
   ALLOCATE(domain_ids(num_domains_to_proc))
   ! Set the first domain ID to be MOAD...always
   domain_ids(1) = 1
   IF (num_domains_to_proc .GT. 1) THEN
     domain_ids(2:num_domains_to_proc) = active_subnests(1:num_active_subnests)
   ENDIF 
   IF ( verbose ) THEN
      WRITE ( UNIT = 6 , NML = filetimespec )
   END IF

   IF ( verbose ) THEN
      WRITE ( UNIT = 6 , FMT='(A)' ) ' &HINTERP_CONTROL'
      WRITE ( UNIT = 6 , FMT='(A,I4)' ) ' INTERP_METHOD = ',interp_method
      WRITE ( UNIT = 6 , FMT='(A)' ) ' /'
   END IF

   num_files_constants = 1
   DO WHILE ( constants_full_name(num_files_constants)(1:20) .NE. '                    ' )
      num_files_constants = num_files_constants + 1
   END DO
   num_files_constants = num_files_constants - 1
    
   num_files_per_single_time_init = 1
   DO WHILE ( init_root(num_files_per_single_time_init)(1:20) .NE. '                    ' )
      num_files_per_single_time_init = num_files_per_single_time_init + 1
   END DO
   num_files_per_single_time_init = num_files_per_single_time_init - 1

   num_files_per_single_time_lbc = 1
   DO WHILE ( lbc_root(num_files_per_single_time_lbc)(1:20) .NE. '                    ' )
      num_files_per_single_time_lbc = num_files_per_single_time_lbc + 1
   END DO
   num_files_per_single_time_lbc = num_files_per_single_time_lbc - 1

   IF ( num_files_per_single_time_init .GT. 0 ) THEN
      IF ( verbose ) THEN
         WRITE ( UNIT = 6 , FMT='(A)' ) ' &HINTERP_CONTROL'
         WRITE ( UNIT = 6 , ADVANCE='NO' , FMT='(A)' ) ' ROOT = '
         WRITE ( UNIT = 6 , FMT='(A,1X)' ) &
           init_root(1:num_files_per_single_time_init)
         IF ( num_files_constants .GT. 0 ) THEN
            WRITE ( UNIT = 6 , ADVANCE='NO' , FMT='(A)' ) ' CONSTANTS_FULL_NAME = '
            WRITE ( UNIT = 6 , FMT='(A,1X)' ) constants_full_name(1:num_files_constants)
         END IF
         WRITE ( UNIT = 6 , FMT='(A)' ) ' /'
      END IF
   ELSE
      PRINT '(A)','You must supply at least a single root name for the first guess file.'
      STOP 'INIT_HINTERP'
   END IF
   IF ( ( num_files_constants .EQ. 0 ) .AND. ( verbose ) ) THEN
      PRINT '(A)','No constants file found (such as for SST or SNOW), continuing.'
   END IF
   IF (lsm_root(1:5) .NE. '     ') THEN

     ! Check to see if LSM field is the same as the initial condition field.
     ! If it is, we do not need a separate lsm root
     separate_lsm = .true.
     DO i = 1, num_files_per_single_time_init 
       IF ( TRIM(lsm_root).EQ.TRIM(init_root(i)) ) THEN
         print *, 'LSM and INIT roots are the same'
         separate_lsm = .false.
       ENDIF
     ENDDO
   ELSE
     separate_lsm = .false.
   ENDIF
  
   ! New version now assumes input file are in current working directory
   ! when executable is run.  
   path_to_data = './'
   ALLOCATE(init_prefix(num_files_per_single_time_init))
   DO i = 1, num_files_per_single_time_init
     init_prefix(i) = TRIM(path_to_data) // TRIM(init_root(i))
     IF (verbose) PRINT '(2a)', 'Using IC from: ', TRIM(init_prefix(i))
   ENDDO

   ALLOCATE(lbc_prefix(num_files_per_single_time_lbc))
   DO i = 1, num_files_per_single_time_lbc
     lbc_prefix(i) = TRIM(path_to_data) // TRIM(lbc_root(i))
     IF (verbose) PRINT '(2a)', 'Using LBC from: ', TRIM(lbc_prefix(i))
   ENDDO

   IF (num_files_constants .GT. 0) THEN
     ALLOCATE (constants_names(num_files_constants))
     DO i = 1, num_files_constants
       constants_names(i) = TRIM(path_to_data) // TRIM(constants_full_name(i))
       IF (verbose) PRINT '(2a)', 'Using input from: ', TRIM(constants_names(i))
     ENDDO
   ENDIF

   lsm_prefix = TRIM(path_to_data) // TRIM(lsm_root)
   output_prefix = TRIM(moad_dataroot) // '/siprd/hinterp'
   IF (verbose) PRINT '(2a)', 'Writing output to: ', TRIM(output_prefix)
        
   ! Set up times

   !  Did the user use a 4-digit year, it is required ya know.

   IF ((start_year .LT. 1900) .OR. (end_year .LT. 1900)) THEN
      PRINT '(A)','You must specify 4-digit years in filetimespec!'
      STOP 'INIT_HINTERP'
   END IF

   ! Build starting and ending time strings
   WRITE(start_time, '(I4.4,"-",I2.2,"-",I2.2,"_",I2.2,":",I2.2,":",I2.2)') &
     start_year, start_month, start_day, start_hour, start_minute, start_second
   WRITE(end_time, '(I4.4,"-",I2.2,"-",I2.2,"_",I2.2,":",I2.2,":",I2.2)') &
     end_year, end_month, end_day, end_hour, end_minute, end_second   
 
   !  Is the ending time before the starting time?
   IF (start_time .GT. end_time) THEN
     PRINT '(4A)','The ending date ',end_time, ' is prior to the starting date ', start_time 
     STOP 'INIT_HINTERP'
   END IF
  
   ! Build list of times
   IF (start_time .EQ. end_time) THEN
     num_output_times = 1
     ALLOCATE(time_list(1))
     time_list(1) = start_time
   ELSE
     IF (interval .GT. 0) THEN
       ALLOCATE(output_times_temp(500))
       num_output_times = 1
       prev_time = start_time
       current_time = prev_time !C.Arthur
       output_times_temp(1) = start_time
       DO WHILE(current_time .LT. end_time)
         CALL geth_newdate(current_time, prev_time, interval)
         prev_time = current_time
         IF (current_time .LE. end_time) THEN
           num_output_times = num_output_times + 1
           output_times_temp(num_output_times) = current_time
         ENDIF
         IF (verbose) PRINT '(2A)', 'The current time is: ',current_time
       ENDDO
       ALLOCATE(time_list(num_output_times))
       time_list(:) = output_times_temp(1:num_output_times)
       DEALLOCATE(output_times_temp)
     ELSE
       PRINT '(A)', 'You must set INTERVAL > 0 seconds!'
       STOP 'INIT_HINTERP'
     ENDIF
   ENDIF

   IF (verbose) THEN
     PRINT '(A,I4)', 'Number of output times: ', num_output_times
     PRINT '(A)',  'List of times:'
     DO i = 1, num_output_times
       PRINT '(A)', time_list(i)
     ENDDO
   ENDIF

   !  The input pressure arrays need to be in bottom-up order.

   ! Fill a few items in the global metadata record

   global_meta%simulation_name = simulation_name
   global_meta%user_desc = user_desc
   global_meta%si_version = 2
END SUBROUTINE init_hinterp
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
END MODULE hinterp_setup
