!dis
!dis    Open Source License/Disclaimer, Forecast Systems Laboratory
!dis    NOAA/OAR/FSL, 325 Broadway Boulder, CO 80305
!dis
!dis    This software is distributed under the Open Source Definition,
!dis    which may be found at http://www.opensource.org/osd.html.
!dis
!dis    In particular, redistribution and use in source and binary forms,
!dis    with or without modification, are permitted provided that the
!dis    following conditions are met:
!dis
!dis    - Redistributions of source code must retain this notice, this
!dis    list of conditions and the following disclaimer.
!dis
!dis    - Redistributions in binary form must provide access to this
!dis    notice, this list of conditions and the following disclaimer, and
!dis    the underlying source code.
!dis
!dis    - All modifications to this software must be clearly documented,
!dis    and are solely the responsibility of the agent making the
!dis    modifications.
!dis
!dis    - If significant modifications or enhancements are made to this
!dis    software, the FSL Software Policy Manager
!dis    (softwaremgr@fsl.noaa.gov) should be notified.
!dis
!dis    THIS SOFTWARE AND ITS DOCUMENTATION ARE IN THE PUBLIC DOMAIN
!dis    AND ARE FURNISHED "AS IS."  THE AUTHORS, THE UNITED STATES
!dis    GOVERNMENT, ITS INSTRUMENTALITIES, OFFICERS, EMPLOYEES, AND
!dis    AGENTS MAKE NO WARRANTY, EXPRESS OR IMPLIED, AS TO THE USEFULNESS
!dis    OF THE SOFTWARE AND DOCUMENTATION FOR ANY PURPOSE.  THEY ASSUME
!dis    NO RESPONSIBILITY (1) FOR THE USE OF THE SOFTWARE AND
!dis    DOCUMENTATION; OR (2) TO PROVIDE TECHNICAL SUPPORT TO USERS.
!dis
!dis                                   

MODULE diagnostic_vars

  USE physical_constants

CONTAINS
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  FUNCTION compute_theta(t_k, p_pa) RESULT (theta)
    
    ! Computes potential temperature using a temperature in deg K
    ! and pressure in Pascals.
    IMPLICIT NONE
    REAL 	:: t_k
    REAL	:: p_pa
    REAL 	:: theta

    theta = t_k*( Pref/p_pa )**RovCp
  END FUNCTION compute_theta
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  FUNCTION compute_density(t_k, p_pa) RESULT(rho)

    ! Computes density (kg/m{-3}) given temperature (K) and pressure (Pa)
    IMPLICIT NONE
    REAL 	:: t_k
    REAL	:: p_pa
    REAL	:: rho

    rho = p_pa / (Rd * t_k)
  END FUNCTION compute_density
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  FUNCTION compute_vapor_mixing_ratio(t_k, p_pa, rh_pct, rh_liq_flag) RESULT(wvmr)

    ! Computes the water vapor mixing ratio (dimensionless) given
    ! temperature (K), pressure (Pa), relative humidity (%), and a flag
    ! that tells the function whether or not the RH being provided is
    ! with respect to liquid everywhere or with respect to liquid only
    ! at temperature > Tref_frz.  This formulation comes from a paper by
    ! Buck (1991)

    IMPLICIT NONE
    REAL	:: t_k
    REAL	:: p_pa
    REAL	:: rh_pct
    LOGICAL     :: rh_liq_flag
    REAL        :: wvmr

    REAL	:: e,es

    !  Compute saturation vapor pressure using an approx.
    !  developed by Buck (1981)

    IF ((rh_liq_flag).OR.((.NOT.rh_liq_flag).AND.(t_k.GE.Tref_frz)))THEN

      ! Compute saturation vapor pressure wrt liquid
    
      es = 100.0 * 6.1121 * EXP( (17.502*t_k-4780.6713)/(t_k-32.18) )

    ELSE 
 
      ! Compute es wrt ice
      ! Factor of 100 converts mb to Pa
      es = 100.0 * 6.1115 * EXP ( (22.452*t_k-6132.7638)/(t_k-0.60) )

    ENDIF

    ! From McKee (1993 CSU AT650 class notes) the values above
    ! need to be multiplied by 1.004 at pressures greater than 800 mb

    IF (p_pa.GE.80000.) es = es * 1.004

    ! Compute e from RH

    e = rh_pct*.01 * es

    ! Compute wvmr from e and p_pa

    wvmr = 0.622 * e / (p_pa - e )
  END FUNCTION compute_vapor_mixing_ratio
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  FUNCTION compute_lower_pressure(ptop_pa,thick_m,tv_mean_k) RESULT(pbot_pa)
   
    ! Function to compute pressure at bottom of a layer specified by
    ! a top pressure (Pa), a thickness (m), and a mean virtual temp (K).
    ! Uses the hypsometric equation.

    IMPLICIT NONE
    REAL	:: ptop_pa
    REAL	:: thick_m
    REAL	:: tv_mean_k
    REAL        :: pbot_pa

    pbot_pa = ptop_pa * EXP( (thick_m*g) / (Rd*tv_mean_k) )
  END FUNCTION compute_lower_pressure
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
 FUNCTION compute_virtual_temp(t_k, wvmr) RESULT(tv_k)

    ! Computes virtual temperature from temperature and water vapor mixing
    ! ratio

    IMPLICIT NONE
    REAL	:: t_k
    REAL	:: wvmr
    REAL        :: tv_k

    tv_k = t_k * (1.0 + 0.61*wvmr)
  END FUNCTION compute_virtual_temp
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
END MODULE diagnostic_vars


