MODULE grid_utils
  
  USE wrfsi_io , ONLY : DryRun

! This module contains utilities related to grid manipulation
! (e.g., creating data for the B and C grids on an Arakawa-C
!  type stagger when only the A grid is available)

CONTAINS
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE arakawa_c_n2t(datain, nx, ny, nz, dataout)
    
    ! Staggers a 3D array of data from the non-staggered points
    ! to the mass grid of an Arakawa C stagger.

    IMPLICIT NONE
    INTEGER, INTENT(IN)                :: nx
    INTEGER, INTENT(IN)                :: ny
    INTEGER, INTENT(IN)                :: nz
    REAL, INTENT(IN)                   :: datain(nx,ny,nz)
    REAL, INTENT(OUT)                  :: dataout(nx,ny,nz)
   
    INTEGER                            :: i,j,k
    IF(.not.DryRun)PRINT *, 'Staggering to T grid (Arakawa C)'
    DO k = 1, nz
      DO j = 1, ny-1
        DO i = 1, nx-1
          dataout(i,j,k) = 0.25 * ( datain(i,j,k)    + &
                                    datain(i+1,j,k)  + &
                                    datain(i+1,j+1,k)+ &
                                    datain(i,j+1,k) )
        ENDDO
        ! Fill unused rightmost column
        dataout(nx,j,k) = dataout(nx-1,j,k)
      ENDDO
      ! Fill unused uppermost row
      dataout(:,ny,k) = dataout(:,ny-1,k)
    ENDDO
    RETURN
  END SUBROUTINE arakawa_c_n2t
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE arakawa_c_n2u(datain, nx, ny, nz, dataout)

    ! Staggers a 3D array of data from the non-staggered points
    ! to the U grid of an Arakawa C stagger.

    IMPLICIT NONE
    INTEGER, INTENT(IN)                :: nx
    INTEGER, INTENT(IN)                :: ny
    INTEGER, INTENT(IN)                :: nz
    REAL, INTENT(IN)                   :: datain(nx,ny,nz)
    REAL, INTENT(OUT)                  :: dataout(nx,ny,nz)

    INTEGER                            :: i,j,k
    IF(.not.DryRun)PRINT *, 'Staggering to U grid (Arakawa C)'
    DO k = 1, nz
      DO j = 1, ny-1
        DO i = 1, nx
          dataout(i,j,k) = 0.50 * ( datain(i,j,k)    + &
                                    datain(i,j+1,k) )
        ENDDO
      ENDDO
      ! Fill unused uppermost row
      dataout(:,ny,k) = dataout(:,ny-1,k)
    ENDDO
    RETURN
  END SUBROUTINE arakawa_c_n2u                    
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE arakawa_c_n2v(datain, nx, ny, nz, dataout)

    ! Staggers a 3D array of data from the non-staggered points
    ! to the V grid of an Arakawa C stagger.

    IMPLICIT NONE
    INTEGER, INTENT(IN)                :: nx
    INTEGER, INTENT(IN)                :: ny
    INTEGER, INTENT(IN)                :: nz
    REAL, INTENT(IN)                   :: datain(nx,ny,nz)
    REAL, INTENT(OUT)                  :: dataout(nx,ny,nz)

    INTEGER                            :: i,j,k
    IF(.not.DryRun)PRINT *, 'Staggering to V Grid (Arakawa C)'
    DO k = 1, nz
      DO j = 1, ny
        DO i = 1, nx-1
          dataout(i,j,k) = 0.50 * ( datain(i,j,k)    + &
                                    datain(i+1,j,k) )
        ENDDO
        ! Fill unused right column
        dataout(nx,j,k) = dataout(nx-1,j,k)
      ENDDO
    ENDDO
    RETURN
  END SUBROUTINE arakawa_c_n2v                              
END MODULE grid_utils
