MODULE wrfsi_io

  USE wrf_metadata
  USE date_pack
  USE map_utils
  IMPLICIT NONE
 
  INTEGER, PARAMETER    :: metafile_in = 10
  INTEGER, PARAMETER    :: metafile_out = 11 
  INTEGER, PARAMETER    :: domfile_in = 20     
  INTEGER, PARAMETER    :: domfile_out = 21
  CHARACTER(LEN=200)    :: metafile_name_in, metafile_name_out
  CHARACTER(LEN=200)    :: domfile_name_in
  CHARACTER(LEN=255)    :: domfile_name_out
  CHARACTER(LEN=255)    :: WrfFileName
  LOGICAL               :: metafile_in_opened, metafile_out_opened
  LOGICAL               :: domfile_in_opened  = .false.
  LOGICAL               :: domfile_out_opened = .false.
  LOGICAL               :: output_file_opened = .false.
  INTEGER               :: open_status
  INTEGER		:: status
  LOGICAL               :: eof
  logical               :: DryRun
  integer               :: WrfDataHandle

 ! These arrays will be used to return the data.  They have
 ! the same shape as the maximum number of dimensions
 ! possible.  When the variable metadata is read in,
 ! the type (real or int), shape (ndim) and value 
 ! for each of the dimensions are determined and the
 ! appropriate array (real_array or int_array) is allocated
 ! beginning with the left-most dimension in order.

  REAL, ALLOCATABLE     :: real_array(:,:,:,:,:)
  INTEGER, ALLOCATABLE  :: int_array(:,:,:,:,:)
  REAL			:: real_scalar
  INTEGER		:: int_scalar
  INTEGER		:: dimsizes(5)
  TYPE (wrfvar_metadata) :: var_info
CONTAINS

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE get_global_metadata(prefix)

    IMPLICIT NONE

    CHARACTER(LEN=*),INTENT(IN)  :: prefix
    CHARACTER (LEN=200)  :: filename_temp

    filename_temp = TRIM(prefix) // '.global.metadata'
    IF ( filename_temp .EQ. metafile_name_in ) THEN
      IF (metafile_in_opened) THEN
        REWIND (metafile_in)
      ELSE
        OPEN ( UNIT=metafile_in , &
            FILE=TRIM(metafile_name_in) , &
            STATUS = 'OLD' , &
            FORM = 'UNFORMATTED' , &
            IOSTAT = open_status )
        IF (open_status.NE.0) THEN
          PRINT  '(A,I5)' , 'Error opening global metadata file:',open_status
          STOP 'metadata_file_open'
        ELSE
          metafile_in_opened = .true.
        ENDIF
      ENDIF
    ELSE
      metafile_name_in = filename_temp
      IF (metafile_in_opened) THEN
        CLOSE(metafile_in)
        metafile_in_opened = .false.
      ENDIF
      OPEN ( UNIT=metafile_in , &
            FILE=TRIM(metafile_name_in) , &
            STATUS = 'OLD' , &
            FORM = 'UNFORMATTED' , &
            IOSTAT = open_status )
      IF (open_status.NE.0) THEN
        PRINT  '(A,I5)' , 'Error opening global metadata file:',open_status
        STOP 'metadata_file_open'
      ELSE
          metafile_in_opened = .true.
      ENDIF
    ENDIF

    READ(metafile_in) global_meta%simulation_name, global_meta%user_desc, &
                   global_meta%si_version, global_meta%anal_version, &
                   global_meta%wrf_version, global_meta%post_version, &
                   global_meta%map_projection, global_meta%moad_known_lat, &
                   global_meta%moad_known_lon, global_meta%moad_known_loc, &
                   global_meta%moad_stand_lats, global_meta%moad_stand_lons, &
                   global_meta%moad_delta_x, global_meta%moad_delta_y, &
                   global_meta%horiz_stagger_type, global_meta%num_stagger_xy, &
                   global_meta%stagger_dir_x, global_meta%stagger_dir_y, &
                   global_meta%num_stagger_z, global_meta%stagger_dir_z, &
                   global_meta%vertical_coord, global_meta%num_domains, &
                   global_meta%init_date, global_meta%init_time, &
                   global_meta%end_date, global_meta%end_time, &
                   global_meta%lu_source, global_meta%lu_water, &
                   global_meta%lu_ice, global_meta%st_water
  
    CLOSE(metafile_in)
    metafile_in_opened = .false.
    RETURN
  END SUBROUTINE get_global_metadata
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE write_global_metadata(prefix)
!   write the metadata in binary

    IMPLICIT NONE
    CHARACTER (LEN=*), INTENT(IN)  :: prefix

!   Write the metadata to the binary file
    metafile_name_out = TRIM(prefix) // '.global.metadata'
    OPEN ( UNIT=metafile_out , &
       FILE=TRIM(metafile_name_out) , &
       STATUS = 'UNKNOWN' , &
       FORM = 'UNFORMATTED' , &
       IOSTAT = open_status )
    IF (open_status.NE.0) THEN
      PRINT  '(A,I5)' , 'Error opening global metadata file:',open_status
      STOP 'metadata_file_open'
    ELSE
      metafile_out_opened = .true.
    ENDIF

    WRITE(metafile_out) global_meta%simulation_name, global_meta%user_desc, &
                   global_meta%si_version, global_meta%anal_version, &
                   global_meta%wrf_version, global_meta%post_version, &
                   global_meta%map_projection, global_meta%moad_known_lat, &
                   global_meta%moad_known_lon, global_meta%moad_known_loc, &
                   global_meta%moad_stand_lats, global_meta%moad_stand_lons, &
                   global_meta%moad_delta_x, global_meta%moad_delta_y, &
                   global_meta%horiz_stagger_type, global_meta%num_stagger_xy, &
                   global_meta%stagger_dir_x, global_meta%stagger_dir_y, &
                   global_meta%num_stagger_z, global_meta%stagger_dir_z, &
                   global_meta%vertical_coord, global_meta%num_domains, &
                   global_meta%init_date, global_meta%init_time, &
                   global_meta%end_date, global_meta%end_time, &
                   global_meta%lu_source, global_meta%lu_water, &
                   global_meta%lu_ice, global_meta%st_water

    CLOSE(metafile_out)
    metafile_out_opened = .false.

    RETURN
  END SUBROUTINE write_global_metadata 
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! 
  SUBROUTINE wrf_write_global_metadata ( output_vcoord_char ) ! write the metadata using the wrf I/O API

    IMPLICIT NONE
    CHARACTER (LEN=4) , INTENT(IN) :: output_vcoord_char
    integer DH
    CHARACTER (LEN=80) :: dum_str
    INTEGER :: dum_int, YYYY, MM, DD, JJJ, status  !mp-BLS

    DH = WrfDataHandle
    call ext_ncd_put_dom_ti_char    (DH,'simulation_name'   ,global_meta%simulation_name                  ,Status)
    call ext_ncd_put_dom_ti_char    (DH,'user_desc'         ,global_meta%user_desc                        ,Status)
    call ext_ncd_put_dom_ti_integer (DH,'si_version'        ,global_meta%si_version     ,1                ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'anal_version'      ,global_meta%anal_version   ,1                ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'wrf_version'       ,global_meta%wrf_version    ,1                ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'post_version'      ,global_meta%post_version   ,1                ,Status)
    call ext_ncd_put_dom_ti_char    (DH,'map_projection'    ,global_meta%map_projection                   ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'moad_known_lat'    ,global_meta%moad_known_lat ,1                ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'moad_known_lon'    ,global_meta%moad_known_lon ,1                ,Status)
!   call ext_ncd_put_dom_ti_char    (DH,'moad_known_loc'    ,global_meta%moad_known_loc                   ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'moad_stand_lats'   ,global_meta%moad_stand_lats,max_standard_lats,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'moad_stand_lons'   ,global_meta%moad_stand_lons,max_standard_lons,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'moad_delta_x'      ,global_meta%moad_delta_x   ,1                ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'moad_delta_y'      ,global_meta%moad_delta_y   ,1                ,Status)
!   call ext_ncd_put_dom_ti_char    (DH,'horiz_stagger_type',global_meta%horiz_stagger_type               ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'num_stagger_xy'    ,global_meta%num_stagger_xy ,1                ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'stagger_dir_x'     ,global_meta%stagger_dir_x  ,max_staggers_xy  ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'stagger_dir_y'     ,global_meta%stagger_dir_y  ,max_staggers_xy  ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'num_stagger_z'     ,global_meta%num_stagger_z  ,1                ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'stagger_dir_z'     ,global_meta%stagger_dir_z  ,max_staggers_z   ,Status)
!   call ext_ncd_put_dom_ti_char    (DH,'vertical_coord'    ,global_meta%vertical_coord                   ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'num_domains'       ,global_meta%num_domains    ,1                ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'init_date'         ,global_meta%init_date      ,1                ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'init_time'         ,global_meta%init_time      ,1                ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'end_date'          ,global_meta%end_date       ,1                ,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'end_time'          ,global_meta%end_time       ,1                ,Status)
!   call ext_ncd_put_dom_ti_char    (DH,'lu_source'         ,global_meta%lu_source                        ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'lu_water'          ,global_meta%lu_water       ,1                ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'lu_ice'            ,global_meta%lu_ice         ,1                ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'st_water'          ,global_meta%st_water       ,1                ,Status)

!=================================================================================
!=================================================================================

    dum_str = '                                                                                '
    WRITE ( dum_str , FMT = '(A,I2.2,A)' ) "OUTPUT FROM WRF SI V" , global_meta%si_version , " PREPROCESSOR"
    call ext_ncd_put_dom_ti_char    (DH,'TITLE'             ,TRIM(dum_str)                                ,Status)

    dum_str = '                                                                                '
        !write(6,*) 'pieces going into dum_str: ', global_meta%init_date, &
         !     global_meta%init_time

        YYYY=INT(global_meta%init_date/1000)
        JJJ= global_meta%init_date - YYYY*1000
        !write(6,*) 'YYYY: ', YYYY
        !write(6,*) 'JJJ: ', JJJ

        call CV_JUL_MMDD_LP(JJJ,YYYY,MM,DD,STATUS)

        !write(6,*) 'return MM, DD: ', MM,DD

    !write(6,*) global_meta%init_date/1000,INT(global_meta%init_time/3600),&
  ! (INT(global_meta%init_time)-3600*INT(global_meta%init_time/3600))/60

!    WRITE ( dum_str , FMT = '(I4,"-",2x,"-",2x,"_",I2.2,":",i2.2,":00.0000")' ) global_meta%init_date/1000 , &
!        INT(global_meta%init_time/3600) , (INT(global_meta%init_time)-3600*INT(global_meta%init_time/3600))/60
  WRITE (dum_str,FMT = '(I4,"-",I2.2,"-",I2.2,"_",I2.2,":",i2.2,":00.0000")' ) &
     YYYY,MM,DD,INT(global_meta%init_time/3600),  &
     (INT(global_meta%init_time)-3600*INT(global_meta%init_time/3600))/60

        !write(6,*) 'dum_str: ', dum_str
        !write(6,*) 'trim(dumstr): ', trim(dum_str)


    call ext_ncd_put_dom_ti_char    (DH,'START_DATE'    ,TRIM(dum_str)                                ,Status)

    call ext_ncd_put_dom_ti_real    (DH,'MOAD_CEN_LAT'  ,global_meta%moad_known_lat ,1                ,Status)
    call ext_ncd_put_dom_ti_real    (DH,'STAND_LON'     ,global_meta%moad_stand_lons(1) ,1            ,Status)
    call ext_ncd_put_dom_ti_real    (DH,'TRUELAT1'      ,global_meta%moad_stand_lats(1) ,1            ,Status)
    call ext_ncd_put_dom_ti_real    (DH,'TRUELAT2'      ,global_meta%moad_stand_lats(2) ,1            ,Status)

   ! write(6,*) ' global_meta%map_projection: ',  global_meta%map_projection
    IF      ( global_meta%map_projection(1:7) .EQ. 'LAMBERT'  ) THEN
       dum_int = 1
    ELSE IF ( global_meta%map_projection(1:5) .EQ. 'POLAR'    ) THEN
       dum_int = 2
    ELSE IF ( global_meta%map_projection(1:8) .EQ. 'MERCATOR' ) THEN
       dum_int = 3
    ELSE IF ( global_meta%map_projection(1:8) .EQ. 'ROTATED ' ) THEN
       dum_int = 203
    END IF
    call ext_ncd_put_dom_ti_integer (DH,'MAP_PROJ'          ,dum_int                     ,1               ,Status)

    !write(6,*) 'output_vcoord_char(1:4): ', output_vcoord_char(1:4)

    IF      ( output_vcoord_char(1:4) .EQ. 'ZETA' ) THEN
      global_meta%output_vcoord = 1
    ELSE IF ( output_vcoord_char(1:4) .EQ. 'ETAP' ) THEN
      global_meta%output_vcoord = 2
    ELSE IF ( output_vcoord_char(1:4) .EQ. 'NMMH' ) THEN
!!      S-L is 3
      global_meta%output_vcoord = 4

    ELSE
      PRINT '(2A)' , 'improper coord specified ',output_vcoord_char(1:4)
      STOP 'IMPROPER_OUTPUT_COORD'
    END IF
    call ext_ncd_put_dom_ti_integer (DH,'DYN_OPT'           ,global_meta%output_vcoord  ,1                ,Status)

    call ext_ncd_put_dom_ti_integer (DH,'ISWATER'           ,global_meta%lu_water       ,1                ,Status)
    call ext_ncd_put_dom_ti_integer (DH,'ISICE'             ,global_meta%lu_ice         ,1                ,Status)

    call ext_ncd_put_dom_ti_char    (DH,'MMINLU'            ,global_meta%lu_source                        ,Status)

!=================================================================================
!=================================================================================

    call ext_ncd_put_dom_ti_integer (DH,'WEST-EAST_GRID_DIMENSION'   , dom_meta%xdim                       ,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'SOUTH-NORTH_GRID_DIMENSION' , dom_meta%ydim                       ,1,Status)
!    call ext_ncd_put_dom_ti_integer (DH,'BOTTOM-TOP_GRID_DIMENSION'  , dom_meta%zdim
    call ext_ncd_put_dom_ti_real    (DH,'DX'                         , dom_meta%delta_x                    ,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'DY'                         , dom_meta%delta_y                    ,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'P_TOP'                       ,dom_meta%top_level                  ,1,Status)

!=================================================================================
!=================================================================================

    RETURN
  END SUBROUTINE wrf_write_global_metadata
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE get_domain_metadata(domain_id,date_str,prefix)

    IMPLICIT NONE
    INTEGER, INTENT(IN) :: domain_id
    CHARACTER(LEN=200)  :: filename_temp
    CHARACTER(LEN=19),INTENT(IN)   :: date_str
    CHARACTER(LEN=*),INTENT(IN)    :: prefix
    CHARACTER(LEN=3)    :: domain_str
    status = 0
    WRITE(domain_str, '(A1,I2.2)') 'd',domain_id
    filename_temp =  TRIM(prefix) // '.' //domain_str// &
         '.' // date_str
    IF ( filename_temp .EQ. domfile_name_in ) THEN
      IF (domfile_in_opened) THEN
        REWIND (domfile_in)
      ELSE
        OPEN ( UNIT=domfile_in , &
            FILE=TRIM(domfile_name_in) , &
            STATUS = 'OLD' , &
            FORM = 'UNFORMATTED' , &
            IOSTAT = open_status )
        IF (open_status.NE.0) THEN
          PRINT  '(A,I5)' , 'Error opening domain file:',open_status
          PRINT '(2A)', 'Filename = ', TRIM(domfile_name_in)
          status = 1                               
        ELSE
          domfile_in_opened = .true.
        ENDIF
      ENDIF
    ELSE
      domfile_name_in = filename_temp
      IF (metafile_in_opened) THEN
        CLOSE(domfile_in)
        domfile_in_opened = .false.
      ENDIF
      OPEN ( UNIT=domfile_in , &
            FILE=TRIM(domfile_name_in) , &
            STATUS = 'OLD' , &
            FORM = 'UNFORMATTED' , &
            IOSTAT = open_status )
      IF (open_status.NE.0) THEN
        PRINT  '(A,I5)' , 'Error opening domain file:',open_status
        PRINT '(2A)', 'Filename = ', TRIM(domfile_name_in)  
        status = 1                    
      ELSE
          domfile_in_opened = .true.
      ENDIF
    ENDIF
    IF (status .EQ. 0) THEN
      READ (domfile_in) dom_meta%id,dom_meta%parent_id,dom_meta%dyn_init_src,&
        dom_meta%static_init_src, dom_meta%vt_date, dom_meta%vt_time, &
        dom_meta%origin_parent_x, dom_meta%origin_parent_y, &
        dom_meta%ratio_to_parent, dom_meta%delta_x, dom_meta%delta_y, &
        dom_meta%top_level, dom_meta%origin_parent_z, &
        dom_meta%corner_lats, dom_meta%corner_lons, dom_meta%xdim, &
        dom_meta%ydim, dom_meta%zdim
    ENDIF
    RETURN
    END SUBROUTINE get_domain_metadata

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE get_variable(prefix,varname,domain_id,date_str,iostatus) 

    IMPLICIT NONE
    CHARACTER(LEN=*)    :: prefix
    CHARACTER (LEN=8)	:: varname
    INTEGER		:: domain_id
    CHARACTER (LEN=19)	:: date_str
    LOGICAL             :: found_var
    INTEGER             :: iostatus
    found_var = .false.
    iostatus = 0
    ! Call get_domain_metadeta to handle the file opening and advancment
    ! past the metadata record
    CALL get_domain_metadata(domain_id,date_str,prefix)
    IF (status .NE.0.) THEN
      STOP 'problem_reading_domain_metadata'
    ENDIF

    ! At this point, the file should be open and the pointer ready
    ! to read the first variable metadata record
    read_loop: DO WHILE(.NOT.found_var)
      CALL read_next_variable
      IF (status .EQ. 0) THEN
        IF (var_info%name .EQ. varname) THEN
          found_var = .true.
          EXIT read_loop
        ENDIF
      ELSE
        EXIT read_loop
      ENDIF
    ENDDO read_loop
99  IF (.not.found_var) THEN
       iostatus = 1
    ENDIF 
    RETURN
  END SUBROUTINE get_variable
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE read_next_variable
    ! This routine reads in the next variable metdata record into
    ! var_info and variable into real_array.  It can only be called
    ! when the domain data file has already been opened and the
    ! domain metadata record has been read.  

    
    IMPLICIT NONE
    status = 0 
    eof = .false.   
    IF (.NOT.domfile_in_opened) THEN 
      PRINT '(A)','Read_next_variable called, but domfile_in is not open.'
      PRINT '(A)','This should not happen, so I am quitting.'
    ENDIF
    READ (domfile_in, END=999) var_info%name, var_info%units, &
      var_info%description, var_info%domain_id, var_info%ndim, &
      var_info%dim_val, var_info%dim_desc, var_info%start_index, &
      var_info%stop_index, var_info%h_stagger_index, var_info%v_stagger_index,&
      var_info%array_order, var_info%field_type, var_info%field_source_prog, &
      var_info%source_desc, var_info%field_time_type, var_info%vt_date_start, &
      var_info%vt_time_start, var_info%vt_date_stop, var_info%vt_time_stop

    dimsizes = var_info%stop_index - var_info%start_index + 1
    IF (TRIM(var_info%field_type).EQ.'REAL') THEN
      IF (ALLOCATED(real_array)) DEALLOCATE(real_array)
      numdims_real: SELECT CASE (var_info%ndim)
        CASE(0)
          READ(domfile_in) real_scalar
        CASE(1)
          ALLOCATE(real_array(dimsizes(1),1,1,1,1))
          READ(domfile_in) real_array
        CASE(2)
          ALLOCATE(real_array(dimsizes(1),dimsizes(2),1,1,1))
          READ(domfile_in) real_array
        CASE(3)
          ALLOCATE(real_array(dimsizes(1),dimsizes(2),dimsizes(3),1,1))
          READ(domfile_in) real_array
        CASE(4)
          ALLOCATE(real_array(dimsizes(1),dimsizes(2),dimsizes(3),&
                  dimsizes(4),1))
          READ(domfile_in) real_array
        CASE(5)
          ALLOCATE(real_array(dimsizes(1),dimsizes(2),dimsizes(3),&
                  dimsizes(4),dimsizes(5)))
          READ(domfile_in) real_array
      END SELECT numdims_real
    ELSE IF  (TRIM(var_info%field_type).EQ.'INT') THEN
      IF (ALLOCATED(int_array)) DEALLOCATE(int_array)
      numdims_int: SELECT CASE (var_info%ndim)
        CASE(0)
          READ(domfile_in) int_scalar
        CASE(1)
          ALLOCATE(int_array(dimsizes(1),1,1,1,1))
          READ(domfile_in) int_array
        CASE(2)
          ALLOCATE(int_array(dimsizes(1),dimsizes(2),1,1,1))
          READ(domfile_in) int_array
        CASE(3)
          ALLOCATE(int_array(dimsizes(1),dimsizes(2),dimsizes(3),1,1))
          READ(domfile_in) int_array
        CASE(4)
          ALLOCATE(int_array(dimsizes(1),dimsizes(2),dimsizes(3),&
                  dimsizes(4),1))
          READ(domfile_in) int_array
        CASE(5)
          ALLOCATE(int_array(dimsizes(1),dimsizes(2),dimsizes(3),&
                  dimsizes(4),dimsizes(5)))
          READ(domfile_in) int_array
      END SELECT numdims_int
    ENDIF
    RETURN
 999 eof = .true.
    status = 2
  END SUBROUTINE read_next_variable
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
SUBROUTINE init_new_output_file(dom_new,prefix,date)

  IMPLICIT NONE
  TYPE(wrf_domain_metadata), INTENT(IN) :: dom_new
  CHARACTER (LEN=*)        , INTENT(IN) :: prefix
  CHARACTER (LEN=19)       , INTENT(IN) :: date

  CHARACTER(LEN=2)                      :: dom_str
  INTEGER                               :: Status

  if(DryRun) then
    WRITE (dom_str, '(I2.2)') dom_new%id
    domfile_name_out = TRIM(prefix) // '.d' //dom_str // '.' // date
    OPEN ( UNIT=domfile_out , FILE=TRIM(domfile_name_out) , &
     STATUS = 'UNKNOWN' , FORM = 'UNFORMATTED' , IOSTAT = open_status )
    REWIND(domfile_out)
    IF (open_status.NE.0) THEN
      PRINT  '(A,I5)' , 'Error opening domain file:',open_status
      status = 1
      stop 'could not open output file'
    ELSE
      domfile_out_opened = .true.
      output_file_opened = .true.
      WRITE (domfile_out) dom_new%id,dom_new%parent_id,dom_new%dyn_init_src,&
        dom_new%static_init_src, dom_new%vt_date, dom_new%vt_time, &
        dom_new%origin_parent_x, dom_new%origin_parent_y, &
        dom_new%ratio_to_parent, dom_new%delta_x, dom_new%delta_y, &
        dom_new%top_level, dom_new%origin_parent_z, &
        dom_new%corner_lats, dom_new%corner_lons, dom_new%xdim, &
        dom_new%ydim, dom_new%zdim
    ENDIF
  endif !DryRun
  RETURN
END SUBROUTINE init_new_output_file
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
SUBROUTINE init_new_static_file(dom_new,prefix)

  IMPLICIT NONE
  TYPE(wrf_domain_metadata), INTENT(IN) :: dom_new
  CHARACTER (LEN=*)        , INTENT(IN) :: prefix

  CHARACTER(LEN=2)                      :: dom_str
  INTEGER                               :: Status

  if(DryRun) then
    WRITE (dom_str, '(I2.2)') dom_new%id
    domfile_name_out = TRIM(prefix) // '.d' //dom_str
    OPEN ( UNIT=domfile_out , FILE=TRIM(domfile_name_out) , &
     STATUS = 'UNKNOWN' , FORM = 'UNFORMATTED' , IOSTAT = open_status )
    REWIND(domfile_out)
    IF (open_status.NE.0) THEN
      PRINT  '(A,I5)' , 'Error opening domain file:',open_status
      status = 1
      stop 'could not open output file'
    ELSE
      domfile_out_opened = .true.
      output_file_opened = .true.
      WRITE (domfile_out) dom_new%id,dom_new%parent_id,dom_new%dyn_init_src,&
        dom_new%static_init_src, dom_new%vt_date, dom_new%vt_time, &
        dom_new%origin_parent_x, dom_new%origin_parent_y, &
        dom_new%ratio_to_parent, dom_new%delta_x, dom_new%delta_y, &
        dom_new%top_level, dom_new%origin_parent_z, &
        dom_new%corner_lats, dom_new%corner_lons, dom_new%xdim, &
        dom_new%ydim, dom_new%zdim
    ENDIF
  endif !DryRun
  RETURN
END SUBROUTINE init_new_static_file

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
SUBROUTINE init_new_wrf_output_file(dom_new,wrf_prefix,date)

  IMPLICIT NONE
  TYPE(wrf_domain_metadata), INTENT(IN) :: dom_new
  CHARACTER (LEN=*)        , INTENT(IN) :: wrf_prefix
  CHARACTER (LEN=19)       , INTENT(IN) :: date

  CHARACTER(LEN=2)                      :: dom_str
  INTEGER                               :: Comm, IOComm, DH, Status, N
  CHARACTER(LEN=10)                     :: SysDepInfo
  CHARACTER(LEN=255)                    :: WrfCDLName
  CHARACTER(LEN=550)                    :: SysCmd
  LOGICAL                               :: havecdl, ncmade
  TYPE(proj_info)                       :: proj_dom
  REAL                                  :: ric, rjc, clat, clon
  if(DryRun) then
    ! Open the wrf file for output
    !
    ! JPE: Look for a cdl file and use it if it exists, this avoids the need for a DryRun
    !
    WRITE (dom_str, '(I2.2)') dom_new%id
    WrfFileName = TRIM(wrf_prefix) // '.d' //dom_str // '.' // date
    ! Make sure this file does not already exist due to a previous run
    Inquire(FILE=WrfFileName,EXIST=ncmade)
    IF (ncmade) THEN
      write (SysCmd,*) 'rm -f ' // TRIM(WrfFileName)
      CALL system(SysCmd)
    ENDIF
    WrfCDLName = TRIM(wrf_prefix) // '.d' //dom_str // '.cdl'
    Inquire(FILE=WrfCDLName ,EXIST=havecdl)
    if(havecdl) then
       write(SysCmd,*) 'ncgen -b -o '//Trim(WrfFileName)//' '//Trim(WrfCDLName)
       call system(SysCmd)
       Inquire(FILE=WrfCDLName ,EXIST=ncmade)
       if(ncmade) then
          DryRun = .false.
          PRINT *, Trim(WrfFileName), ' created from cdl ',Trim(WrfCDLName)
       endif
    endif

    Comm        = 1      
    IOComm      = 1   !Not used
    SysDepInfo  = ' ' !Not used
    call ext_ncd_open_for_write_begin(WrfFileName,Comm,IOComm,SysDepInfo,WrfDataHandle,Status)
    if(Status /= 0) then
      write(6,*) 'ext_ncd_open_for_write_begin ERROR in ',__FILE__,', line', __LINE__
    endif
    output_file_opened = .true.
 endif
 if(output_file_opened) then
    N = 4*max_staggers_xy
    DH = WrfDataHandle
    call ext_ncd_put_dom_ti_integer (DH,'grid_id'        ,dom_new%id             ,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'parent_id'      ,dom_new%parent_id      ,1,Status)
!   call ext_ncd_put_dom_ti_char    (DH,'dyn_init_src'   ,dom_new%dyn_init_src     ,Status)
!   call ext_ncd_put_dom_ti_char    (DH,'static_init_src',dom_new%static_init_src  ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'vt_date'        ,dom_new%vt_date        ,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'vt_time'        ,dom_new%vt_time        ,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'i_parent_start' ,dom_new%origin_parent_x,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'j_parent_start' ,dom_new%origin_parent_y,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'i_parent_end'   ,dom_new%end_parent_x,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'j_parent_end'   ,dom_new%end_parent_y,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'parent_grid_ratio',dom_new%ratio_to_parent,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'delta_x'        ,dom_new%delta_x        ,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'delta_y'        ,dom_new%delta_y        ,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'top_level'      ,dom_new%top_level      ,1,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'origin_parent_z',dom_new%origin_parent_z,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'corner_lats'    ,dom_new%corner_lats    ,N,Status)
    call ext_ncd_put_dom_ti_real    (DH,'corner_lons'    ,dom_new%corner_lons    ,N,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'xdim'           ,dom_new%xdim           ,1,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'ydim'           ,dom_new%ydim           ,1,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'zdim'           ,dom_new%zdim           ,1,Status)
!   call wrf_write_global_metadata

!=================================================================================
!=================================================================================

    call ext_ncd_put_dom_ti_integer (DH,'WEST-EAST_GRID_DIMENSION'   , dom_new%xdim      ,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'SOUTH-NORTH_GRID_DIMENSION' , dom_new%ydim      ,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'BOTTOM-TOP_GRID_DIMENSION'  , dom_new%zdim      ,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'DX'                         , dom_new%delta_x   ,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'DY'                         , dom_new%delta_y   ,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'P_TOP'                      , dom_new%top_level ,1,Status)

! Compute center lat/lon of this grid...
    IF (dom_new%id .EQ. 1) THEN
      clat = global_meta%moad_known_lat
      clon = global_meta%moad_known_lon
    ELSE 
      SELECT CASE (global_meta%map_projection)
        CASE ('MERCATOR                        ')
          CALL map_set(PROJ_MERC, dom_new%corner_lats(1,n_ind), dom_new%corner_lons(1,n_ind), &
            dom_new%delta_x, 0.,global_meta%moad_stand_lats(1),0.,dom_new%xdim,dom_new%ydim,proj_dom)
        CASE ('LAMBERT CONFORMAL               ')
          CALL map_set(PROJ_LC, dom_new%corner_lats(1,n_ind), dom_new%corner_lons(1,n_ind), &
            dom_new%delta_x,global_meta%moad_stand_lons(1),global_meta%moad_stand_lats(1),&
            global_meta%moad_stand_lats(2), &
            dom_new%xdim,dom_new%ydim,proj_dom)
        CASE ('POLAR STEREOGRAPHIC             ')
          CALL map_set(PROJ_PS, dom_new%corner_lats(1,n_ind), dom_new%corner_lons(1,n_ind), &
            dom_new%delta_x, global_meta%moad_stand_lons(1),global_meta%moad_stand_lats(1),0., &
            dom_new%xdim,dom_new%ydim,proj_dom)
        CASE ('ROTATED LATLON                  ')
          CALL map_set(PROJ_PS, dom_new%corner_lats(1,n_ind), dom_new%corner_lons(1,n_ind), &
            dom_new%delta_x, global_meta%moad_stand_lons(1),global_meta%moad_stand_lats(1),0., &
            dom_new%xdim,dom_new%ydim,proj_dom)
      END SELECT
      ric = 0.5*FLOAT(dom_new%xdim + 1)
      rjc = 0.5*FLOAT(dom_new%ydim + 1)
      CALL ij_to_latlon(proj_dom, ric, rjc, clat, clon)
    ENDIF
    call ext_ncd_put_dom_ti_real    (DH,'CEN_LAT'                    , clat                              ,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'CEN_LON'                    , clon                              ,1,Status)

!=================================================================================
!=================================================================================

  endif !output_file_opened
  RETURN
END SUBROUTINE init_new_wrf_output_file
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
SUBROUTINE init_new_wrf_static_file(dom_new,wrf_prefix)

  IMPLICIT NONE
  TYPE(wrf_domain_metadata), INTENT(IN) :: dom_new
  CHARACTER (LEN=*)        , INTENT(IN) :: wrf_prefix
  CHARACTER(LEN=2)                      :: dom_str
  INTEGER                               :: Comm, IOComm, DH, Status, N
  CHARACTER(LEN=10)                     :: SysDepInfo
  CHARACTER(LEN=255)                    :: WrfCDLName
  CHARACTER(LEN=550)                    :: SysCmd
  LOGICAL                               :: havecdl, ncmade
  TYPE(proj_info)                       :: proj_dom
  REAL                                  :: ric, rjc, clat, clon

  if(DryRun) then
    ! Open the wrf file for output
    !
    ! JPE: Look for a cdl file and use it if it exists, this avoids the need for a DryRun
    !
    WRITE (dom_str, '(I2.2)') dom_new%id
    WrfFileName = TRIM(wrf_prefix) // '_d' //dom_str 
    ! Make sure this file does not already exist due to a previous run
    Inquire(FILE=WrfFileName,EXIST=ncmade)
    IF (ncmade) THEN
      write (SysCmd,*) 'rm -f ' // TRIM(WrfFileName)
      CALL system(SysCmd)
    ENDIF
    WrfCDLName = TRIM(wrf_prefix) // '_d' //dom_str // '.cdl'
    Inquire(FILE=WrfCDLName ,EXIST=havecdl)
    if(havecdl) then
       write(SysCmd,*) 'ncgen -b -o '//Trim(WrfFileName)//' '//Trim(WrfCDLName)
       call system(SysCmd)
       Inquire(FILE=WrfCDLName ,EXIST=ncmade)
       if(ncmade) then
          DryRun = .false.
          PRINT *, Trim(WrfFileName), ' created from cdl ',Trim(WrfCDLName)
       endif
    endif

    Comm        = 1
    IOComm      = 1   !Not used
    SysDepInfo  = ' ' !Not used
    PRINT *, 'Calling ext_ncd_open_for_write_begin: ',WrfFileNAme
    call ext_ncd_open_for_write_begin(WrfFileName,Comm,IOComm,SysDepInfo,WrfDataHandle,Status)
    if(Status /= 0) then
      write(6,*) 'ext_ncd_open_for_write_begin ERROR in ',__FILE__,', line', __LINE__
    endif
    output_file_opened = .true.
 endif
 if(output_file_opened) then
    N = 4*max_staggers_xy
    DH = WrfDataHandle
    call ext_ncd_put_dom_ti_integer (DH,'grid_id'        ,dom_new%id             ,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'parent_id'      ,dom_new%parent_id      ,1,Status)
!   call ext_ncd_put_dom_ti_char    (DH,'dyn_init_src'   ,dom_new%dyn_init_src     ,Status)
!   call ext_ncd_put_dom_ti_char    (DH,'static_init_src',dom_new%static_init_src  ,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'vt_date'        ,dom_new%vt_date        ,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'vt_time'        ,dom_new%vt_time        ,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'i_parent_start' ,dom_new%origin_parent_x,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'j_parent_start' ,dom_new%origin_parent_y,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'i_parent_end'   ,dom_new%end_parent_x,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'j_parent_end'   ,dom_new%end_parent_y,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'parent_grid_ratio',dom_new%ratio_to_parent,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'delta_x'        ,dom_new%delta_x        ,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'delta_y'        ,dom_new%delta_y        ,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'top_level'      ,dom_new%top_level      ,1,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'origin_parent_z',dom_new%origin_parent_z,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'corner_lats'    ,dom_new%corner_lats    ,N,Status)
    call ext_ncd_put_dom_ti_real    (DH,'corner_lons'    ,dom_new%corner_lons    ,N,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'xdim'           ,dom_new%xdim           ,1,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'ydim'           ,dom_new%ydim           ,1,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'zdim'           ,dom_new%zdim           ,1,Status)
!   call wrf_write_global_metadata

!=================================================================================
!=================================================================================

    call ext_ncd_put_dom_ti_integer (DH,'WEST-EAST_GRID_DIMENSION'   , dom_new%xdim                       ,1,Status)
    call ext_ncd_put_dom_ti_integer (DH,'SOUTH-NORTH_GRID_DIMENSION' , dom_new%ydim                       ,1,Status)
!   call ext_ncd_put_dom_ti_integer (DH,'BOTTOM-TOP_GRID_DIMENSION'  , dom_new%zdim                       ,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'DX'                         , dom_new%delta_x                    ,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'DY'                         , dom_new%delta_y                    ,1,Status)
!   call ext_ncd_put_dom_ti_real    (DH,'P_TOP'                       ,dom_new%top_level                  ,1,Status)
! Compute center lat/lon of this grid...
    SELECT CASE (global_meta%map_projection)
      CASE ('MERCATOR                        ')
        CALL map_set(PROJ_MERC, dom_new%corner_lats(1,n_ind), dom_new%corner_lons(1,n_ind), &
          dom_new%delta_x, 0.,global_meta%moad_stand_lats(1),0.,dom_new%xdim,dom_new%ydim,proj_dom)
      CASE ('LAMBERT CONFORMAL               ')
        CALL map_set(PROJ_LC, dom_new%corner_lats(1,n_ind), dom_new%corner_lons(1,n_ind), &
          dom_new%delta_x,global_meta%moad_stand_lons(1),global_meta%moad_stand_lats(1),&
          global_meta%moad_stand_lats(2), &
          dom_new%xdim,dom_new%ydim,proj_dom)
      CASE ('POLAR STEREOGRAPHIC             ')
        CALL map_set(PROJ_PS, dom_new%corner_lats(1,n_ind), dom_new%corner_lons(1,n_ind), &
          dom_new%delta_x, global_meta%moad_stand_lons(1),global_meta%moad_stand_lats(1),0., &
          dom_new%xdim,dom_new%ydim,proj_dom)
      CASE ('ROTATED LATLON                  ')
        CALL map_set(PROJ_PS, dom_new%corner_lats(1,n_ind), dom_new%corner_lons(1,n_ind), &
          dom_new%delta_x, global_meta%moad_stand_lons(1),global_meta%moad_stand_lats(1),0., &
          dom_new%xdim,dom_new%ydim,proj_dom)
    END SELECT
    ric = 0.5*FLOAT(dom_new%xdim + 1)
    rjc = 0.5*FLOAT(dom_new%ydim + 1)
    CALL ij_to_latlon(proj_dom, ric, rjc, clat, clon)
    call ext_ncd_put_dom_ti_real    (DH,'CEN_LAT'                    , clat                              ,1,Status)
    call ext_ncd_put_dom_ti_real    (DH,'CEN_LON'                    , clon                              ,1,Status)

  CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_STATIC' , 1 , 1 ,Status)

!=================================================================================
!=================================================================================

  endif !output_file_opened
  RETURN
END SUBROUTINE init_new_wrf_static_file

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE output_variable_metadata(var_meta_out)

    IMPLICIT NONE
    TYPE(wrfvar_metadata) :: var_meta_out

    if(.not.DryRun) then
      PRINT '(3A,I1,A,5I5)', 'Writing variable: ', var_meta_out%name, &
      '  NDIM = ', var_meta_out%ndim, '  Dim Values: ', &
         var_meta_out%dim_val
    endif
    WRITE(domfile_out) var_meta_out%name, var_meta_out%units, &
      var_meta_out%description, var_meta_out%domain_id, var_meta_out%ndim, &
      var_meta_out%dim_val, var_meta_out%dim_desc, var_meta_out%start_index, &
      var_meta_out%stop_index, var_meta_out%h_stagger_index, var_meta_out%v_stagger_index,&
      var_meta_out%array_order, var_meta_out%field_type, var_meta_out%field_source_prog, &
      var_meta_out%source_desc, var_meta_out%field_time_type, var_meta_out%vt_date_start, &
      var_meta_out%vt_time_start, var_meta_out%vt_date_stop, var_meta_out%vt_time_stop

  END SUBROUTINE output_variable_metadata

 SUBROUTINE output_variable_wrf_metadata(var_meta_out,Name,DateStr)

 IMPLICIT NONE
 TYPE(wrfvar_metadata)               :: var_meta_out
 CHARACTER*(*)         , INTENT (IN) :: Name
 CHARACTER(LEN=19)     , INTENT (IN) :: DateStr
 integer                             :: i,DH
 character(len=9)                    :: str
! from module_vinterp -- MUST AGREE
 INTEGER, PARAMETER            :: zstag_full_index = 2
 INTEGER, PARAMETER            :: zstag_half_index = 1

 DH = WrfDataHandle

if(DryRun) then
!  call ext_ncd_put_var_ti_char   (DH,'name'             ,Name,var_meta_out%name                       ,Status)
   call ext_ncd_put_var_ti_char   (DH,'units'            ,Name,var_meta_out%units                      ,Status)
   call ext_ncd_put_var_ti_char   (DH,'description'      ,Name,var_meta_out%description                ,Status)
!  call ext_ncd_put_var_ti_integer(DH,'domain_id'        ,Name,var_meta_out%domain_id      ,1          ,Status)
!  call ext_ncd_put_var_ti_integer(DH,'ndim'             ,Name,var_meta_out%ndim           ,1          ,Status)
!  call ext_ncd_put_var_ti_integer(DH,'dim_val'          ,Name,var_meta_out%dim_val        ,var_maxdims,Status)
!  do i=1,var_maxdims
!    write(str,"('dim_desc',I1)") I
!    call ext_ncd_put_var_ti_char (DH,str                ,Name,var_meta_out%dim_desc(i)                ,Status)
!  enddo
!  call ext_ncd_put_var_ti_integer(DH,'start_index'      ,Name,var_meta_out%start_index    ,var_maxdims,Status)
!  call ext_ncd_put_var_ti_integer(DH,'stop_index'       ,Name,var_meta_out%stop_index     ,var_maxdims,Status)
!  call ext_ncd_put_var_ti_integer(DH,'h_stagger_index'  ,Name,var_meta_out%h_stagger_index,1          ,Status)
!  call ext_ncd_put_var_ti_integer(DH,'v_stagger_index'  ,Name,var_meta_out%v_stagger_index,1          ,Status)
!  call ext_ncd_put_var_ti_char   (DH,'array_order'      ,Name,var_meta_out%array_order                ,Status)
!  call ext_ncd_put_var_ti_char   (DH,'field_type'       ,Name,var_meta_out%field_type                 ,Status)
!  call ext_ncd_put_var_ti_char   (DH,'field_source_prog',Name,var_meta_out%field_source_prog          ,Status)
!  call ext_ncd_put_var_ti_char   (DH,'source_desc'      ,Name,var_meta_out%source_desc                ,Status)
!  call ext_ncd_put_var_ti_char   (DH,'field_time_type'  ,Name,var_meta_out%field_time_type            ,Status)
!  call ext_ncd_put_var_ti_integer(DH,'vt_date_start'    ,Name,var_meta_out%vt_date_start  ,1          ,Status)
!  call ext_ncd_put_var_ti_real   (DH,'vt_time_start'    ,Name,var_meta_out%vt_time_start  ,1          ,Status)
!  call ext_ncd_put_var_ti_integer(DH,'vt_date_stop'     ,Name,var_meta_out%vt_date_stop   ,1          ,Status)
!  call ext_ncd_put_var_ti_real   (DH,'vt_time_stop'     ,Name,var_meta_out%vt_time_stop   ,1          ,Status) 

  IF      ( var_meta_out%v_stagger_index .EQ. zstag_full_index ) THEN
    call ext_ncd_put_var_ti_char   (DH,'stagger'     ,Name,'Z'               ,Status)
  ELSE
    IF      ( var_meta_out%h_stagger_index .EQ. u_ind ) THEN
     call ext_ncd_put_var_ti_char   (DH,'stagger'     ,Name,'X'               ,Status)
    ELSE IF ( var_meta_out%h_stagger_index .EQ. v_ind ) THEN
     call ext_ncd_put_var_ti_char   (DH,'stagger'     ,Name,'Y'               ,Status)
    ELSE IF ( var_meta_out%h_stagger_index .EQ. t_ind ) THEN
     call ext_ncd_put_var_ti_char   (DH,'stagger'     ,Name,' '               ,Status)
    ELSE
     call ext_ncd_put_var_ti_char   (DH,'stagger'     ,Name,' '               ,Status)
    ENDIF
  ENDIF
endif ! dryrun
  !  Global metadata that is dependent on the variables that are output.

  !  3d Moisture/hydrometeors
if (.not. DryRun) then
  IF      ( Name(1:8) == 'QLIQUID ' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_QR      ' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'QCLOUD  ' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_QC      ' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'QSNOW   ' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_QS      ' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'QICE    ' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_QI      ' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'QGRAUPEL' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_QG      ' , 1 , 1 ,Status)

  !  2d LSM-ish fields, various levels possible levels of soil temperature,
  !  total soil moisture, and liquid soil moisture.

  ELSE IF ( Name(1:8) == 'ST000010' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_ST000010' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'ST010040' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_ST010040' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'ST040100' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_ST040100' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'ST100200' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_ST100200' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'ST010200' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_ST010200' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILT000' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILT000' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILT005' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILT005' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILT020' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILT020' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILT040' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILT040' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILT160' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILT160' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILT300' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILT300' , 1 , 1 ,Status)

  ELSE IF ( Name(1:8) == 'SM000010' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SM000010' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SM010040' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SM010040' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SM040100' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SM040100' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SM100200' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SM100200' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SM010200' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SM010200' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILM000' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILM000' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILM005' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILM005' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILM020' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILM020' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILM040' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILM040' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILM160' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILM160' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILM300' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILM300' , 1 , 1 ,Status)

  ELSE IF ( Name(1:8) == 'SW000010' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SW000010' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SW010040' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SW010040' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SW040100' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SW040100' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SW100200' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SW100200' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SW010200' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SW010200' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILW000' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILW000' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILW005' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILW005' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILW020' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILW020' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILW040' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILW040' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILW160' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILW160' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'SOILW300' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILW300' , 1 , 1 ,Status)

  !  2d Another helpful LSM field is the elevation of the input model's
  !  topography,  This is used to adjust the soil temps depending on 
  !  the forecast model's orography.

  ELSE IF ( ( Name(1:8) == 'TOPOSOIL' ).OR.( Name(1:8) == 'SOILHGT ')) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_TOPOSOIL' , 1 , 1 ,Status)
 
  !  2d Yet more helpful LSM fields - the physical height of the snow.

  ELSE IF ( Name(1:8) == 'SNOWH   ' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SNOWH' , 1 , 1 ,Status)
 
  !  2d Is there a real SST field in the house?

  ELSE IF ( Name(1:8) == 'SST     ' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SST     ' , 1 , 1 ,Status)

  ! Added SOILCAT and VEGCAT from background data
  ELSE IF ( Name(1:8) == 'SOILCAT ' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_SOILCAT ' , 1 , 1 ,Status)
  ELSE IF ( Name(1:8) == 'VEGCAT  ' ) THEN
    CALL ext_ncd_put_dom_ti_integer (DH , 'FLAG_VEGCAT  ' , 1 , 1 ,Status)
  END IF
endif ! not dryrun
 END SUBROUTINE output_variable_wrf_metadata

END MODULE wrfsi_io

