!
! CRTM_Model_Profiles
!
! Module to define a set of climatological model profiles
!
!
! CREATION HISTORY:
!       Written by:     Paul van Delst, CIMSS/SSEC 13-Nov-2001
!                       paul.vandelst@noaa.gov
!       Modified by:    Yong Chen, NESDIS/CIRA 22-Aug-2007
!                       Yong.Chen@noaa.gov
!                       Paul van Delst, 22-Oct-2007
!                       paul.vandelst@noaa.gov
!


MODULE CRTM_Model_Profiles

  ! -----------------
  ! Environment setup
  ! -----------------
  ! Module use
  USE Type_Kinds            , ONLY: fp
  USE CRTM_Atmosphere_Define, ONLY: N_MODELS => N_VALID_CLIMATOLOGY_MODELS, &
                                    TROPICAL              , &
                                    MIDLATITUDE_SUMMER    , &
                                    MIDLATITUDE_WINTER    , &
                                    SUBARCTIC_SUMMER      , &
                                    SUBARCTIC_WINTER      , &
                                    US_STANDARD_ATMOSPHERE

  ! Disable implicit typing
  IMPLICIT NONE


  ! ------------
  ! Visibilities
  ! ------------
  PRIVATE
  ! Parameters
  PUBLIC :: MODEL_LEVEL_PRESSURE
  ! Procedures
  PUBLIC :: CRTM_Get_Model_Profile


  ! -----------------
  ! Module parameters
  ! -----------------
  CHARACTER(*), PARAMETER :: MODULE_RCS_ID = &
  '$Id: CRTM_Model_Profiles.f90 60152 2015-08-13 19:19:13Z paul.vandelst@noaa.gov $'

  ! Model dimension info
  INTEGER, PARAMETER :: N_MODEL_LEVELS    = 101
  INTEGER, PARAMETER :: N_MODEL_LAYERS    = N_MODEL_LEVELS-1
  INTEGER, PARAMETER :: N_MODEL_ABSORBERS = 6

  ! The absorber indices
  INTEGER, PARAMETER :: H2O = 1
  INTEGER, PARAMETER ::  O3 = 2
  INTEGER, PARAMETER :: CO2 = 3
  INTEGER, PARAMETER :: N2O = 4
  INTEGER, PARAMETER ::  CO = 5
  INTEGER, PARAMETER :: CH4 = 6

  ! The absorber ids
  INTEGER, PARAMETER :: H2O_ID = 1
  INTEGER, PARAMETER :: CO2_ID = 2
  INTEGER, PARAMETER ::  O3_ID = 3
  INTEGER, PARAMETER :: N2O_ID = 4
  INTEGER, PARAMETER ::  CO_ID = 5
  INTEGER, PARAMETER :: CH4_ID = 6
  
  ! The absorber id->index map
  ! This maps the absorber ID to the local absorber array INDEX.
  INTEGER, PARAMETER :: ABSORBER_ID_MAP(N_MODEL_ABSORBERS) = &
    (/ H2O, &
       CO2, &
        O3, &
       N2O, &
        CO, &
       CH4 /)
  REAL(fp), PARAMETER :: MODEL_LEVEL_PRESSURE(0:N_MODEL_LAYERS) = &
  (/  0.00500000_fp,    0.01606500_fp,    0.03838300_fp,    0.07687900_fp,    0.13695400_fp, &
      0.22441200_fp,    0.34540400_fp,    0.50637400_fp,    0.71402300_fp,    0.97527400_fp, &
      1.29724000_fp,    1.68720000_fp,    2.15257300_fp,    2.70089700_fp,    3.33981200_fp, &
      4.07703800_fp,    4.92036200_fp,    5.87762300_fp,    6.95669500_fp,    8.16548000_fp, &
      9.51188900_fp,   11.00383500_fp,   12.64922300_fp,   14.45593600_fp,   16.43183300_fp, &
     18.58473200_fp,   20.92240800_fp,   23.45258300_fp,   26.18291800_fp,   29.12100900_fp, &
     32.27437800_fp,   35.65046700_fp,   39.25663300_fp,   43.10014400_fp,   47.18817100_fp, &
     51.52778600_fp,   56.12595300_fp,   60.98953000_fp,   66.12525900_fp,   71.53976800_fp, &
     77.23956000_fp,   83.23101600_fp,   89.52039000_fp,   96.11380300_fp,  103.01724400_fp, &
    110.23656500_fp,  117.77748100_fp,  125.64556200_fp,  133.84624000_fp,  142.38479600_fp, &
    151.26636600_fp,  160.49593900_fp,  170.07834800_fp,  180.01827900_fp,  190.32026000_fp, &
    200.98866500_fp,  212.02771200_fp,  223.44146100_fp,  235.23381400_fp,  247.40851400_fp, &
    259.96914200_fp,  272.91912000_fp,  286.26170600_fp,  300.00000000_fp,  314.13693600_fp, &
    328.67528600_fp,  343.61765900_fp,  358.96650300_fp,  374.72409800_fp,  390.89256600_fp, &
    407.47386100_fp,  424.46977600_fp,  441.88194100_fp,  459.71182100_fp,  477.96072200_fp, &
    496.62978500_fp,  515.71998900_fp,  535.23215300_fp,  555.16693500_fp,  575.52483200_fp, &
    596.30618200_fp,  617.51116300_fp,  639.13979700_fp,  661.19194600_fp,  683.66731600_fp, &
    706.56546000_fp,  729.88577200_fp,  753.62749400_fp,  777.78971600_fp,  802.37137600_fp, &
    827.37125900_fp,  852.78800300_fp,  878.62009600_fp,  904.86588000_fp,  931.52354900_fp, &
    958.59115400_fp,  986.06660100_fp, 1013.94765500_fp, 1042.23194000_fp, 1070.91694000_fp, &
   1100.00000000_fp /)
  
  
  ! ----------------------
  ! Module data statements
  ! ----------------------
  REAL(fp), SAVE :: Level_Temperature(0:N_MODEL_LAYERS,N_MODELS)
  DATA Level_Temperature(:,TROPICAL) / &
    178.174_fp,  192.328_fp,  210.111_fp,  225.459_fp,  239.120_fp, &
    251.516_fp,  258.971_fp,  264.536_fp,  268.260_fp,  269.940_fp, &
    267.898_fp,  263.813_fp,  259.762_fp,  256.009_fp,  252.506_fp, &
    249.223_fp,  246.228_fp,  243.425_fp,  240.822_fp,  238.354_fp, &
    236.044_fp,  233.852_fp,  231.779_fp,  229.858_fp,  228.013_fp, &
    226.216_fp,  224.456_fp,  222.760_fp,  221.135_fp,  219.623_fp, &
    218.157_fp,  216.716_fp,  215.232_fp,  213.323_fp,  211.116_fp, &
    208.960_fp,  206.863_fp,  204.840_fp,  202.874_fp,  201.054_fp, &
    199.289_fp,  197.557_fp,  195.862_fp,  195.130_fp,  196.031_fp, &
    196.910_fp,  199.292_fp,  201.792_fp,  204.249_fp,  206.692_fp, &
    209.083_fp,  211.535_fp,  214.055_fp,  216.524_fp,  218.876_fp, &
    221.164_fp,  223.408_fp,  225.700_fp,  227.958_fp,  230.178_fp, &
    232.509_fp,  234.797_fp,  237.043_fp,  239.252_fp,  241.422_fp, &
    243.553_fp,  245.698_fp,  247.807_fp,  249.880_fp,  251.983_fp, &
    254.067_fp,  256.118_fp,  258.148_fp,  260.155_fp,  262.131_fp, &
    264.092_fp,  266.071_fp,  268.020_fp,  269.939_fp,  271.870_fp, &
    273.782_fp,  275.665_fp,  277.531_fp,  279.397_fp,  281.235_fp, &
    283.047_fp,  284.395_fp,  285.475_fp,  286.540_fp,  287.590_fp, &
    289.118_fp,  290.683_fp,  292.227_fp,  293.750_fp,  295.281_fp, &
    296.790_fp,  298.280_fp,  299.749_fp,  301.199_fp,  302.630_fp, &
    304.043_fp /
  DATA Level_Temperature(:,MIDLATITUDE_SUMMER) / &
    166.103_fp,  181.157_fp,  202.859_fp,  222.255_fp,  239.663_fp, &   
    252.229_fp,  261.666_fp,  268.978_fp,  272.709_fp,  275.659_fp, &   
    275.105_fp,  270.621_fp,  265.928_fp,  261.515_fp,  257.445_fp, &   
    253.718_fp,  250.248_fp,  247.060_fp,  244.068_fp,  241.271_fp, &   
    238.659_fp,  236.454_fp,  234.345_fp,  232.403_fp,  230.575_fp, &   
    228.818_fp,  227.618_fp,  226.594_fp,  225.605_fp,  224.701_fp, &   
    223.884_fp,  223.178_fp,  222.456_fp,  221.710_fp,  221.003_fp, &   
    220.320_fp,  219.654_fp,  218.993_fp,  218.317_fp,  217.695_fp, &   
    217.154_fp,  216.627_fp,  216.116_fp,  215.700_fp,  215.700_fp, &   
    215.700_fp,  215.700_fp,  215.700_fp,  215.700_fp,  215.700_fp, &   
    215.700_fp,  215.730_fp,  215.767_fp,  216.038_fp,  218.372_fp, &   
    220.660_fp,  222.920_fp,  225.181_fp,  227.399_fp,  229.604_fp, &   
    231.820_fp,  233.995_fp,  236.134_fp,  238.241_fp,  240.310_fp, &   
    242.374_fp,  244.466_fp,  246.522_fp,  248.550_fp,  250.576_fp, &   
    252.568_fp,  254.527_fp,  256.478_fp,  258.399_fp,  260.290_fp, &   
    262.111_fp,  263.867_fp,  265.596_fp,  267.301_fp,  269.024_fp, &   
    270.722_fp,  272.394_fp,  274.060_fp,  275.718_fp,  277.352_fp, &   
    278.963_fp,  280.560_fp,  282.137_fp,  283.691_fp,  285.218_fp, &   
    286.393_fp,  287.551_fp,  288.694_fp,  289.823_fp,  290.949_fp, &   
    292.059_fp,  293.155_fp,  294.236_fp,  295.303_fp,  296.356_fp, &   
    297.395_fp /
  DATA Level_Temperature(:,MIDLATITUDE_WINTER) / &
    200.964_fp,  216.091_fp,  227.925_fp,  237.836_fp,  246.210_fp, &  
    253.447_fp,  259.898_fp,  263.297_fp,  265.617_fp,  264.331_fp, &  
    258.371_fp,  252.296_fp,  246.806_fp,  241.773_fp,  237.138_fp, &  
    232.901_fp,  228.974_fp,  225.394_fp,  222.050_fp,  219.798_fp, &  
    218.606_fp,  217.468_fp,  216.770_fp,  216.126_fp,  215.508_fp, &  
    215.407_fp,  215.317_fp,  215.230_fp,  215.200_fp,  215.200_fp, &  
    215.200_fp,  215.200_fp,  215.200_fp,  215.200_fp,  215.200_fp, &  
    215.200_fp,  215.200_fp,  215.200_fp,  215.363_fp,  215.611_fp, &  
    215.854_fp,  216.092_fp,  216.324_fp,  216.551_fp,  216.773_fp, &  
    216.988_fp,  217.199_fp,  217.406_fp,  217.607_fp,  217.805_fp, &  
    217.998_fp,  218.188_fp,  218.374_fp,  218.557_fp,  218.736_fp, &  
    218.911_fp,  219.083_fp,  219.251_fp,  219.417_fp,  219.580_fp, &  
    220.181_fp,  222.085_fp,  223.955_fp,  225.794_fp,  227.652_fp, &  
    229.477_fp,  231.270_fp,  233.065_fp,  234.839_fp,  236.584_fp, &  
    238.315_fp,  240.046_fp,  241.750_fp,  243.426_fp,  245.108_fp, &  
    246.771_fp,  248.408_fp,  250.028_fp,  251.653_fp,  253.253_fp, &  
    254.830_fp,  256.399_fp,  257.966_fp,  259.509_fp,  261.030_fp, &  
    262.193_fp,  263.071_fp,  263.936_fp,  264.789_fp,  265.636_fp, &  
    266.477_fp,  267.306_fp,  268.124_fp,  268.933_fp,  269.738_fp, &  
    270.532_fp,  271.316_fp,  272.089_fp,  272.852_fp,  273.605_fp, &  
    274.348_fp /
  DATA Level_Temperature(:,SUBARCTIC_SUMMER) / &
    162.600_fp,  176.793_fp,  198.862_fp,  219.134_fp,  237.464_fp, &   
    254.166_fp,  265.996_fp,  272.936_fp,  275.498_fp,  277.137_fp, &   
    276.306_fp,  274.243_fp,  271.376_fp,  267.499_fp,  262.519_fp, &   
    257.948_fp,  253.686_fp,  249.780_fp,  246.155_fp,  242.879_fp, &   
    239.836_fp,  237.823_fp,  235.897_fp,  234.239_fp,  232.785_fp, &   
    231.387_fp,  230.336_fp,  229.438_fp,  228.572_fp,  227.634_fp, &   
    226.602_fp,  225.672_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  226.105_fp,  228.344_fp,  230.542_fp,  232.719_fp, &   
    234.915_fp,  237.074_fp,  239.195_fp,  241.311_fp,  243.391_fp, &   
    245.437_fp,  247.492_fp,  249.535_fp,  251.545_fp,  253.541_fp, &   
    255.569_fp,  257.566_fp,  259.532_fp,  261.175_fp,  262.673_fp, &   
    264.149_fp,  265.604_fp,  267.058_fp,  268.491_fp,  269.903_fp, &   
    271.305_fp,  272.712_fp,  274.099_fp,  275.466_fp,  276.825_fp, &   
    278.180_fp,  279.516_fp,  280.835_fp,  282.152_fp,  283.486_fp, &   
    284.801_fp,  286.099_fp,  287.379_fp,  288.643_fp,  289.889_fp, &   
    291.120_fp /
  DATA Level_Temperature(:,SUBARCTIC_WINTER) / &
    214.590_fp,  231.481_fp,  244.770_fp,  248.281_fp,  250.442_fp, &   
    255.518_fp,  259.144_fp,  259.262_fp,  255.110_fp,  249.389_fp, &   
    244.240_fp,  239.569_fp,  235.403_fp,  231.542_fp,  227.948_fp, &   
    224.653_fp,  221.850_fp,  220.096_fp,  218.457_fp,  217.428_fp, &   
    216.448_fp,  215.538_fp,  214.690_fp,  213.877_fp,  213.104_fp, &   
    212.364_fp,  211.653_fp,  211.345_fp,  211.756_fp,  212.155_fp, &   
    212.541_fp,  212.915_fp,  213.277_fp,  213.629_fp,  213.971_fp, &   
    214.303_fp,  214.627_fp,  214.942_fp,  215.249_fp,  215.547_fp, &   
    215.839_fp,  216.123_fp,  216.401_fp,  216.673_fp,  216.938_fp, &   
    217.198_fp,  217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp, &   
    217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp, &   
    217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp, &   
    217.200_fp,  217.200_fp,  217.457_fp,  218.476_fp,  219.477_fp, &   
    220.460_fp,  222.270_fp,  224.190_fp,  226.077_fp,  227.963_fp, &   
    229.873_fp,  231.752_fp,  233.601_fp,  235.457_fp,  237.298_fp, &   
    239.109_fp,  240.893_fp,  242.699_fp,  244.477_fp,  246.229_fp, &   
    247.892_fp,  249.174_fp,  250.437_fp,  251.682_fp,  252.835_fp, &   
    253.620_fp,  254.394_fp,  255.157_fp,  255.909_fp,  256.660_fp, &   
    257.400_fp,  258.129_fp,  258.849_fp,  258.826_fp,  258.408_fp, &   
    257.995_fp,  257.588_fp,  257.187_fp,  256.790_fp,  256.399_fp, &   
    256.013_fp /
  DATA Level_Temperature(:,US_STANDARD_ATMOSPHERE) / &
    190.195_fp,  203.667_fp,  215.175_fp,  226.809_fp,  237.789_fp, &   
    247.507_fp,  256.483_fp,  263.555_fp,  268.955_fp,  270.636_fp, &   
    267.045_fp,  261.561_fp,  256.386_fp,  251.670_fp,  247.321_fp, &   
    243.261_fp,  239.551_fp,  236.057_fp,  232.759_fp,  229.832_fp, &   
    228.503_fp,  227.233_fp,  226.133_fp,  225.245_fp,  224.392_fp, &   
    223.595_fp,  222.847_fp,  222.126_fp,  221.425_fp,  220.733_fp, &   
    220.065_fp,  219.420_fp,  218.797_fp,  218.196_fp,  217.614_fp, &   
    217.106_fp,  216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp, &   
    216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp, &   
    216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp, &   
    216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp, &   
    216.723_fp,  216.757_fp,  216.790_fp,  218.296_fp,  220.415_fp, &   
    222.495_fp,  224.553_fp,  226.585_fp,  228.580_fp,  230.577_fp, &   
    232.588_fp,  234.564_fp,  236.514_fp,  238.474_fp,  240.401_fp, &   
    242.296_fp,  244.201_fp,  246.087_fp,  247.943_fp,  249.783_fp, &   
    251.627_fp,  253.442_fp,  255.229_fp,  257.021_fp,  258.798_fp, &   
    260.549_fp,  262.275_fp,  264.015_fp,  265.730_fp,  267.420_fp, &   
    269.095_fp,  270.776_fp,  272.433_fp,  274.067_fp,  275.689_fp, &   
    277.314_fp,  278.917_fp,  280.497_fp,  282.066_fp,  283.643_fp, &   
    285.200_fp,  286.736_fp,  288.251_fp,  289.746_fp,  291.221_fp, &   
    292.678_fp /

  REAL(fp), SAVE :: Level_Absorber(0:N_MODEL_LAYERS,N_MODEL_ABSORBERS,N_MODELS)
  DATA Level_Absorber(:,H2O,TROPICAL) / &
 8.77980825E-04_fp, 1.63665427E-03_fp, 2.41525886E-03_fp, 3.01358863E-03_fp, 3.42673415E-03_fp, &
 3.69727056E-03_fp, 3.73182898E-03_fp, 3.73182898E-03_fp, 3.73182898E-03_fp, 3.70485491E-03_fp, & 
 3.62553040E-03_fp, 3.52250711E-03_fp, 3.42918702E-03_fp, 3.30368092E-03_fp, 3.18356224E-03_fp, & 
 3.07218031E-03_fp, 2.96914223E-03_fp, 2.87229788E-03_fp, 2.78234404E-03_fp, 2.69709168E-03_fp, & 
 2.61724153E-03_fp, 2.54151913E-03_fp, 2.46345214E-03_fp, 2.37323353E-03_fp, 2.28665909E-03_fp, & 
 2.20863685E-03_fp, 2.14020216E-03_fp, 2.07426306E-03_fp, 2.01766533E-03_fp, 1.99628659E-03_fp, & 
 1.90185294E-03_fp, 1.79636495E-03_fp, 1.75789394E-03_fp, 1.71098213E-03_fp, 1.65816603E-03_fp, & 
 1.63469692E-03_fp, 1.61839270E-03_fp, 1.61712589E-03_fp, 1.61712589E-03_fp, 1.65651334E-03_fp, & 
 1.69871321E-03_fp, 1.73942166E-03_fp, 1.77895400E-03_fp, 1.81305422E-03_fp, 1.83851684E-03_fp, & 
 1.86338111E-03_fp, 2.07865778E-03_fp, 2.31078836E-03_fp, 2.60268897E-03_fp, 3.11383907E-03_fp, & 
 3.61397564E-03_fp, 4.29053420E-03_fp, 5.15159631E-03_fp, 5.99495710E-03_fp, 9.54264560E-03_fp, & 
 1.36728122E-02_fp, 1.77218010E-02_fp, 2.69138794E-02_fp, 3.64198574E-02_fp, 4.62694696E-02_fp, & 
 7.10905559E-02_fp, 9.54559132E-02_fp, 1.19808735E-01_fp, 1.65311675E-01_fp, 2.10009241E-01_fp, & 
 2.53925406E-01_fp, 3.23805339E-01_fp, 3.93088062E-01_fp, 4.61199718E-01_fp, 5.57061296E-01_fp, & 
 6.58709582E-01_fp, 7.58695269E-01_fp, 8.89551039E-01_fp, 1.04316485E+00_fp, 1.19433850E+00_fp, & 
 1.36357009E+00_fp, 1.59234548E+00_fp, 1.81758872E+00_fp, 2.03938326E+00_fp, 2.24071288E+00_fp, & 
 2.43503683E+00_fp, 2.62646230E+00_fp, 2.96715923E+00_fp, 3.68750198E+00_fp, 4.39735364E+00_fp, & 
 5.09695512E+00_fp, 6.07752900E+00_fp, 7.20936332E+00_fp, 8.32520255E+00_fp, 9.42539639E+00_fp, & 
 1.01510542E+01_fp, 1.08244053E+01_fp, 1.14885020E+01_fp, 1.21559103E+01_fp, 1.31774892E+01_fp, & 
 1.41852861E+01_fp, 1.51795783E+01_fp, 1.61606176E+01_fp, 1.71286637E+01_fp, 1.80839545E+01_fp, & 
 1.90267237E+01_fp /
  DATA Level_Absorber(:,O3,TROPICAL) / &
 4.76282998E-01_fp, 2.63575307E-01_fp, 2.38323078E-01_fp, 4.34252158E-01_fp, 7.32103525E-01_fp, &
 1.05832797E+00_fp, 1.49900320E+00_fp, 1.96708261E+00_fp, 2.51463612E+00_fp, 3.08189624E+00_fp, &   
 3.82225579E+00_fp, 4.75582031E+00_fp, 5.80609455E+00_fp, 6.90465318E+00_fp, 7.85315879E+00_fp, &   
 8.62916701E+00_fp, 9.17872204E+00_fp, 9.64583838E+00_fp, 9.76328654E+00_fp, 9.83182051E+00_fp, &   
 9.68129051E+00_fp, 9.45808993E+00_fp, 9.15268309E+00_fp, 8.60873569E+00_fp, 8.08675973E+00_fp, &   
 7.46417712E+00_fp, 6.70969487E+00_fp, 5.98272612E+00_fp, 5.26763918E+00_fp, 4.51144304E+00_fp, &   
 3.87334450E+00_fp, 3.28178900E+00_fp, 2.66325567E+00_fp, 2.20360475E+00_fp, 1.86393581E+00_fp, &   
 1.62600430E+00_fp, 1.41629416E+00_fp, 1.19079017E+00_fp, 9.69572281E-01_fp, 7.60019680E-01_fp, &   
 5.56473977E-01_fp, 4.22289982E-01_fp, 3.16357103E-01_fp, 2.34147587E-01_fp, 1.90916468E-01_fp, &   
 1.48701241E-01_fp, 1.37969522E-01_fp, 1.30953035E-01_fp, 1.23887238E-01_fp, 1.16261319E-01_fp, &   
 1.08799712E-01_fp, 1.02767939E-01_fp, 9.82121846E-02_fp, 9.37500852E-02_fp, 8.87007788E-02_fp, &   
 8.35895445E-02_fp, 7.85787706E-02_fp, 7.42657223E-02_fp, 7.00914630E-02_fp, 6.60152554E-02_fp, &   
 6.25765013E-02_fp, 5.92008847E-02_fp, 5.59111488E-02_fp, 5.39198530E-02_fp, 5.19638019E-02_fp, &   
 5.00419464E-02_fp, 4.83436243E-02_fp, 4.66785655E-02_fp, 4.50416508E-02_fp, 4.40871060E-02_fp, &   
 4.33155387E-02_fp, 4.25565913E-02_fp, 4.18230593E-02_fp, 4.11113222E-02_fp, 4.04108911E-02_fp, &   
 3.97271370E-02_fp, 3.90712599E-02_fp, 3.84255092E-02_fp, 3.77896455E-02_fp, 3.71872689E-02_fp, &   
 3.65994968E-02_fp, 3.60204917E-02_fp, 3.55648316E-02_fp, 3.54061032E-02_fp, 3.52496865E-02_fp, &   
 3.50955285E-02_fp, 3.47584482E-02_fp, 3.43210595E-02_fp, 3.38898521E-02_fp, 3.34646906E-02_fp, &   
 3.29662458E-02_fp, 3.24653767E-02_fp, 3.19713915E-02_fp, 3.14763682E-02_fp, 3.07596938E-02_fp, &   
 3.00526880E-02_fp, 2.93551562E-02_fp, 2.86669218E-02_fp, 2.79878026E-02_fp, 2.73176317E-02_fp, &   
 2.66562451E-02_fp /
  DATA Level_Absorber(:,CO2,TROPICAL) / &
 3.74360000e+02_fp, 3.74361536e+02_fp, 3.74363355e+02_fp, 3.74366645e+02_fp, 3.74372000e+02_fp,&
 3.74380012e+02_fp, 3.74391000e+02_fp, 3.74404976e+02_fp, 3.74427783e+02_fp, 3.74467018e+02_fp,&
 3.74521289e+02_fp, 3.74580048e+02_fp, 3.74628801e+02_fp, 3.74664910e+02_fp, 3.74699114e+02_fp,&
 3.74739482e+02_fp, 3.74792946e+02_fp, 3.74861608e+02_fp, 3.74938271e+02_fp, 3.75015741e+02_fp,&
 3.75084934e+02_fp, 3.75140614e+02_fp, 3.75186705e+02_fp, 3.75224301e+02_fp, 3.75257801e+02_fp,&
 3.75293898e+02_fp, 3.75341078e+02_fp, 3.75402006e+02_fp, 3.75483301e+02_fp, 3.75585596e+02_fp,&
 3.75708307e+02_fp, 3.75851657e+02_fp, 3.76012795e+02_fp, 3.76192873e+02_fp, 3.76383982e+02_fp,&
 3.76585651e+02_fp, 3.76809813e+02_fp, 3.77081169e+02_fp, 3.77391271e+02_fp, 3.77732795e+02_fp,&
 3.78108482e+02_fp, 3.78481072e+02_fp, 3.78820139e+02_fp, 3.79146343e+02_fp, 3.79424024e+02_fp,&
 3.79651892e+02_fp, 3.79872699e+02_fp, 3.80073036e+02_fp, 3.80265614e+02_fp, 3.80453361e+02_fp,&
 3.80639313e+02_fp, 3.80813470e+02_fp, 3.80970199e+02_fp, 3.81124934e+02_fp, 3.81280663e+02_fp,&
 3.81437259e+02_fp, 3.81594518e+02_fp, 3.81753006e+02_fp, 3.81910795e+02_fp, 3.82065175e+02_fp,&
 3.82213512e+02_fp, 3.82349223e+02_fp, 3.82473964e+02_fp, 3.82577331e+02_fp, 3.82656946e+02_fp,&
 3.82723325e+02_fp, 3.82777620e+02_fp, 3.82823139e+02_fp, 3.82861807e+02_fp, 3.82896325e+02_fp,&
 3.82929211e+02_fp, 3.82963229e+02_fp, 3.83001367e+02_fp, 3.83044151e+02_fp, 3.83086855e+02_fp,&
 3.83128253e+02_fp, 3.83164506e+02_fp, 3.83194127e+02_fp, 3.83218072e+02_fp, 3.83237331e+02_fp,&
 3.83252313e+02_fp, 3.83264940e+02_fp, 3.83277627e+02_fp, 3.83291488e+02_fp, 3.83308524e+02_fp,&
 3.83328380e+02_fp, 3.83347090e+02_fp, 3.83365096e+02_fp, 3.83394620e+02_fp, 3.83444994e+02_fp,&
 3.83510928e+02_fp, 3.83581313e+02_fp, 3.83656398e+02_fp, 3.83737054e+02_fp, 3.83815572e+02_fp,&
 3.83884500e+02_fp, 3.83941349e+02_fp, 3.83985090e+02_fp, 3.84014928e+02_fp, 3.84031542e+02_fp,&
 3.84037247e+02_fp /
  DATA Level_Absorber(:,N2O,TROPICAL) / &
 5.90294582e-04_fp, 7.86351255e-04_fp, 1.01522316e-03_fp, 1.28618783e-03_fp, 1.60853310e-03_fp,&
 2.01337053e-03_fp, 2.59797101e-03_fp, 3.29258542e-03_fp, 4.25213249e-03_fp, 6.75783097e-03_fp,&
 1.16944249e-02_fp, 1.80296550e-02_fp, 2.67314773e-02_fp, 3.85736817e-02_fp, 5.10524380e-02_fp,&
 6.40658680e-02_fp, 7.77945300e-02_fp, 9.11964559e-02_fp, 1.02770322e-01_fp, 1.13621418e-01_fp,&
 1.24199207e-01_fp, 1.34385570e-01_fp, 1.43289317e-01_fp, 1.49526613e-01_fp, 1.55511913e-01_fp,&
 1.61150838e-01_fp, 1.66432154e-01_fp, 1.71520884e-01_fp, 1.77031958e-01_fp, 1.85212594e-01_fp,&
 1.91861329e-01_fp, 1.97692921e-01_fp, 2.02888699e-01_fp, 2.09780841e-01_fp, 2.17876138e-01_fp,&
 2.26838285e-01_fp, 2.35803338e-01_fp, 2.44031605e-01_fp, 2.51995340e-01_fp, 2.58779329e-01_fp,&
 2.65292787e-01_fp, 2.70581419e-01_fp, 2.75327194e-01_fp, 2.79711107e-01_fp, 2.83559355e-01_fp,&
 2.87317105e-01_fp, 2.89991708e-01_fp, 2.92492258e-01_fp, 2.94857299e-01_fp, 2.96893337e-01_fp,&
 2.98885506e-01_fp, 3.00803155e-01_fp, 3.02646505e-01_fp, 3.04451986e-01_fp, 3.06135770e-01_fp,&
 3.07765541e-01_fp, 3.09363285e-01_fp, 3.10954176e-01_fp, 3.12516926e-01_fp, 3.14043961e-01_fp,&
 3.15361366e-01_fp, 3.16654575e-01_fp, 3.17910448e-01_fp, 3.18445922e-01_fp, 3.18971920e-01_fp,&
 3.19488720e-01_fp, 3.19656558e-01_fp, 3.19813935e-01_fp, 3.19968653e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp /
  DATA Level_Absorber(:,CO,TROPICAL) / &
 3.52812285e+00_fp, 1.11741668e+00_fp, 3.86747360e-01_fp, 1.54743622e-01_fp, 8.78259542e-02_fp,&
 7.16614777e-02_fp, 6.40422809e-02_fp, 5.83586828e-02_fp, 5.34678407e-02_fp, 4.82750435e-02_fp,&
 4.30645954e-02_fp, 3.89983832e-02_fp, 3.54201339e-02_fp, 3.25130324e-02_fp, 3.00019827e-02_fp,&
 2.78529759e-02_fp, 2.63238431e-02_fp, 2.50210223e-02_fp, 2.39375827e-02_fp, 2.29278597e-02_fp,&
 2.18287310e-02_fp, 2.07289285e-02_fp, 1.96818700e-02_fp, 1.86918806e-02_fp, 1.77418882e-02_fp,&
 1.69387390e-02_fp, 1.63068673e-02_fp, 1.56980371e-02_fp, 1.50643976e-02_fp, 1.42325850e-02_fp,&
 1.35591265e-02_fp, 1.29813464e-02_fp, 1.25174376e-02_fp, 1.23200000e-02_fp, 1.23200000e-02_fp,&
 1.27506375e-02_fp, 1.32696669e-02_fp, 1.43235123e-02_fp, 1.53951754e-02_fp, 1.72504724e-02_fp,&
 1.91366614e-02_fp, 2.12856984e-02_fp, 2.35018057e-02_fp, 2.57606828e-02_fp, 2.81351339e-02_fp,&
 3.04537459e-02_fp, 3.36726406e-02_fp, 3.69270879e-02_fp, 4.03112947e-02_fp, 4.43241409e-02_fp,&
 4.82505240e-02_fp, 5.27364400e-02_fp, 5.78112971e-02_fp, 6.27818956e-02_fp, 6.78325727e-02_fp,&
 7.28259138e-02_fp, 7.77211271e-02_fp, 8.18562263e-02_fp, 8.58499210e-02_fp, 8.97524961e-02_fp,&
 9.31237001e-02_fp, 9.64329889e-02_fp, 9.96838639e-02_fp, 1.02956948e-01_fp, 1.06172111e-01_fp,&
 1.09331051e-01_fp, 1.12249350e-01_fp, 1.15113623e-01_fp, 1.17929487e-01_fp, 1.20057230e-01_fp,&
 1.21986162e-01_fp, 1.23883520e-01_fp, 1.25413019e-01_fp, 1.26660078e-01_fp, 1.27887330e-01_fp,&
 1.28910042e-01_fp, 1.29353203e-01_fp, 1.29789520e-01_fp, 1.30219160e-01_fp, 1.30510903e-01_fp,&
 1.30767695e-01_fp, 1.31020658e-01_fp, 1.31493200e-01_fp, 1.32523538e-01_fp, 1.33538878e-01_fp,&
 1.34539553e-01_fp, 1.35768992e-01_fp, 1.37118952e-01_fp, 1.38449842e-01_fp, 1.39762065e-01_fp,&
 1.41105284e-01_fp, 1.42435717e-01_fp, 1.43747865e-01_fp, 1.45042048e-01_fp, 1.46317267e-01_fp,&
 1.47575290e-01_fp, 1.48816449e-01_fp, 1.50041068e-01_fp, 1.51249462e-01_fp, 1.52441934e-01_fp,&
 1.53618781e-01_fp /
  DATA Level_Absorber(:,CH4,TROPICAL) / &
 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp,&
 1.50000000e-01_fp, 1.58607339e-01_fp, 1.72598336e-01_fp, 1.97189468e-01_fp, 2.39181242e-01_fp,&
 3.07727014e-01_fp, 3.81062249e-01_fp, 4.54802692e-01_fp, 5.25697446e-01_fp, 5.90423804e-01_fp,&
 6.48924544e-01_fp, 6.97229645e-01_fp, 7.40933834e-01_fp, 7.81440295e-01_fp, 8.19818909e-01_fp,&
 8.55643574e-01_fp, 8.89571062e-01_fp, 9.20809063e-01_fp, 9.47426361e-01_fp, 9.72968281e-01_fp,&
 9.96515296e-01_fp, 1.01789205e+00_fp, 1.03848929e+00_fp, 1.06258095e+00_fp, 1.10589020e+00_fp,&
 1.15260620e+00_fp, 1.20057459e+00_fp, 1.25067674e+00_fp, 1.29916852e+00_fp, 1.34615521e+00_fp,&
 1.38501413e+00_fp, 1.42118891e+00_fp, 1.45003518e+00_fp, 1.47756414e+00_fp, 1.49730920e+00_fp,&
 1.51585446e+00_fp, 1.53094691e+00_fp, 1.54450627e+00_fp, 1.55735341e+00_fp, 1.56922567e+00_fp,&
 1.58081873e+00_fp, 1.58986706e+00_fp, 1.59845103e+00_fp, 1.60674605e+00_fp, 1.61452001e+00_fp,&
 1.62212648e+00_fp, 1.62950203e+00_fp, 1.63664971e+00_fp, 1.64365056e+00_fp, 1.64983151e+00_fp,&
 1.65572643e+00_fp, 1.66150550e+00_fp, 1.66620095e+00_fp, 1.67071556e+00_fp, 1.67511272e+00_fp,&
 1.67849068e+00_fp, 1.68180660e+00_fp, 1.68505224e+00_fp, 1.68772961e+00_fp, 1.69035960e+00_fp,&
 1.69294360e+00_fp, 1.69425246e+00_fp, 1.69551148e+00_fp, 1.69674922e+00_fp, 1.69750233e+00_fp,&
 1.69812457e+00_fp, 1.69873662e+00_fp, 1.69917391e+00_fp, 1.69947807e+00_fp, 1.69977740e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp /

  DATA Level_Absorber(:,H2O,MIDLATITUDE_SUMMER) / &
 8.80599931E-04_fp, 1.47572005E-03_fp, 1.97813347E-03_fp, 2.38351250E-03_fp, 2.72802818E-03_fp, &   
 3.00292035E-03_fp, 3.19132561E-03_fp, 3.32180485E-03_fp, 3.37724401E-03_fp, 3.42084323E-03_fp, &   
 3.42028608E-03_fp, 3.39397328E-03_fp, 3.32997533E-03_fp, 3.25260801E-03_fp, 3.17150369E-03_fp, &   
 3.13410940E-03_fp, 3.10449317E-03_fp, 3.08824187E-03_fp, 3.06740482E-03_fp, 3.03934609E-03_fp, &   
 3.01055985E-03_fp, 2.97174541E-03_fp, 2.93462097E-03_fp, 2.88485311E-03_fp, 2.83070490E-03_fp, &   
 2.77866814E-03_fp, 2.72916030E-03_fp, 2.68161015E-03_fp, 2.63574105E-03_fp, 2.57094018E-03_fp, &   
 2.48649721E-03_fp, 2.42662857E-03_fp, 2.34999817E-03_fp, 2.25339352E-03_fp, 2.19271705E-03_fp, &   
 2.13957036E-03_fp, 2.08783721E-03_fp, 2.04260672E-03_fp, 2.01023719E-03_fp, 1.98452690E-03_fp, &   
 1.96920466E-03_fp, 1.96410491E-03_fp, 1.97853791E-03_fp, 1.99496639E-03_fp, 2.02268300E-03_fp, &   
 2.04974830E-03_fp, 2.07583594E-03_fp, 2.10129201E-03_fp, 2.29281897E-03_fp, 2.67060180E-03_fp, &   
 3.04024471E-03_fp, 3.67850181E-03_fp, 4.36794406E-03_fp, 5.46397037E-03_fp, 1.02532782E-02_fp, &   
 1.49470824E-02_fp, 2.22350362E-02_fp, 3.65409017E-02_fp, 5.05732548E-02_fp, 7.11067223E-02_fp, &   
 1.03253429E-01_fp, 1.34809905E-01_fp, 1.67179038E-01_fp, 2.01107829E-01_fp, 2.34436101E-01_fp, &   
 2.71873055E-01_fp, 3.18610664E-01_fp, 3.64549797E-01_fp, 4.14550309E-01_fp, 4.86968165E-01_fp, &   
 5.58187485E-01_fp, 6.28241915E-01_fp, 7.17769497E-01_fp, 8.07854932E-01_fp, 8.96509358E-01_fp, &   
 1.00673091E+00_fp, 1.13686247E+00_fp, 1.26498487E+00_fp, 1.40046198E+00_fp, 1.68417177E+00_fp, &   
 1.96361553E+00_fp, 2.23889119E+00_fp, 2.56503717E+00_fp, 2.93826004E+00_fp, 3.30604729E+00_fp, &   
 3.66852373E+00_fp, 4.24303573E+00_fp, 4.84696946E+00_fp, 5.44236840E+00_fp, 6.03073164E+00_fp, &   
 6.69658337E+00_fp, 7.35322328E+00_fp, 8.00083839E+00_fp, 8.65543046E+00_fp, 9.43033313E+00_fp, &   
 1.01947817E+01_fp, 1.09489865E+01_fp, 1.16931386E+01_fp, 1.24274348E+01_fp, 1.31520557E+01_fp, &   
 1.38671785E+01_fp /
  DATA Level_Absorber(:,O3,MIDLATITUDE_SUMMER) / &
 5.28761628E-01_fp, 1.96792365E-01_fp, 2.54518897E-01_fp, 4.75536727E-01_fp, 7.92056224E-01_fp, &   
 1.15672290E+00_fp, 1.48711958E+00_fp, 1.78681001E+00_fp, 2.33267719E+00_fp, 2.85792951E+00_fp, &   
 3.51791554E+00_fp, 4.36402526E+00_fp, 5.39690129E+00_fp, 6.48151796E+00_fp, 7.56018675E+00_fp, &   
 8.25159185E+00_fp, 8.73449876E+00_fp, 8.83901351E+00_fp, 8.75395997E+00_fp, 8.39305957E+00_fp, &   
 8.02923474E+00_fp, 7.57159438E+00_fp, 7.13388001E+00_fp, 6.75296026E+00_fp, 6.40472425E+00_fp, &   
 6.07006743E+00_fp, 5.70200930E+00_fp, 5.33504602E+00_fp, 4.98105597E+00_fp, 4.53413503E+00_fp, &   
 3.99106853E+00_fp, 3.60604361E+00_fp, 3.25661050E+00_fp, 2.94597039E+00_fp, 2.65143375E+00_fp, &   
 2.37328364E+00_fp, 2.15148077E+00_fp, 1.92042065E+00_fp, 1.66020348E+00_fp, 1.40703933E+00_fp, &   
 1.16068975E+00_fp, 9.52782106E-01_fp, 8.13550699E-01_fp, 6.92511553E-01_fp, 6.47949041E-01_fp, &   
 6.04433707E-01_fp, 5.62490237E-01_fp, 5.21562211E-01_fp, 4.89261010E-01_fp, 4.66483670E-01_fp, &   
 4.44197104E-01_fp, 3.97334483E-01_fp, 3.45605470E-01_fp, 2.97181021E-01_fp, 2.69526368E-01_fp, &   
 2.42423177E-01_fp, 2.18830070E-01_fp, 2.03628367E-01_fp, 1.88717304E-01_fp, 1.73248858E-01_fp, &   
 1.56581717E-01_fp, 1.40220595E-01_fp, 1.27885420E-01_fp, 1.21531632E-01_fp, 1.15290303E-01_fp, &   
 1.09042505E-01_fp, 1.02657654E-01_fp, 9.63818824E-02_fp, 9.05268627E-02_fp, 8.62821781E-02_fp, &   
 8.21077443E-02_fp, 7.80015891E-02_fp, 7.39311148E-02_fp, 6.99229303E-02_fp, 6.59784159E-02_fp, &   
 6.27189244E-02_fp, 6.00970366E-02_fp, 5.75156292E-02_fp, 5.50040365E-02_fp, 5.30191697E-02_fp, &   
 5.10641484E-02_fp, 4.91382877E-02_fp, 4.73518022E-02_fp, 4.56961666E-02_fp, 4.40646437E-02_fp, &   
 4.24566797E-02_fp, 4.10229808E-02_fp, 3.96358393E-02_fp, 3.82683009E-02_fp, 3.69259339E-02_fp, &   
 3.59937718E-02_fp, 3.50745060E-02_fp, 3.41678744E-02_fp, 3.32825328E-02_fp, 3.24819665E-02_fp, &   
 3.16922005E-02_fp, 3.09130175E-02_fp, 3.01442203E-02_fp, 2.93856052E-02_fp, 2.86369860E-02_fp, &   
 2.78981793E-02_fp /
  DATA Level_Absorber(:,CO2,MIDLATITUDE_SUMMER) / &
 3.74360000e+02_fp, 3.74361536e+02_fp, 3.74363355e+02_fp, 3.74366645e+02_fp, 3.74372000e+02_fp,&
 3.74380012e+02_fp, 3.74391000e+02_fp, 3.74404976e+02_fp, 3.74427783e+02_fp, 3.74467018e+02_fp,&
 3.74521289e+02_fp, 3.74580048e+02_fp, 3.74628801e+02_fp, 3.74664910e+02_fp, 3.74699114e+02_fp,&
 3.74739482e+02_fp, 3.74792946e+02_fp, 3.74861608e+02_fp, 3.74938271e+02_fp, 3.75015741e+02_fp,&
 3.75084934e+02_fp, 3.75140614e+02_fp, 3.75186705e+02_fp, 3.75224301e+02_fp, 3.75257801e+02_fp,&
 3.75293898e+02_fp, 3.75341078e+02_fp, 3.75402006e+02_fp, 3.75483301e+02_fp, 3.75585596e+02_fp,&
 3.75708307e+02_fp, 3.75851657e+02_fp, 3.76012795e+02_fp, 3.76192873e+02_fp, 3.76383982e+02_fp,&
 3.76585651e+02_fp, 3.76809813e+02_fp, 3.77081169e+02_fp, 3.77391271e+02_fp, 3.77732795e+02_fp,&
 3.78108482e+02_fp, 3.78481072e+02_fp, 3.78820139e+02_fp, 3.79146343e+02_fp, 3.79424024e+02_fp,&
 3.79651892e+02_fp, 3.79872699e+02_fp, 3.80073036e+02_fp, 3.80265614e+02_fp, 3.80453361e+02_fp,&
 3.80639313e+02_fp, 3.80813470e+02_fp, 3.80970199e+02_fp, 3.81124934e+02_fp, 3.81280663e+02_fp,&
 3.81437259e+02_fp, 3.81594518e+02_fp, 3.81753006e+02_fp, 3.81910795e+02_fp, 3.82065175e+02_fp,&
 3.82213512e+02_fp, 3.82349223e+02_fp, 3.82473964e+02_fp, 3.82577331e+02_fp, 3.82656946e+02_fp,&
 3.82723325e+02_fp, 3.82777620e+02_fp, 3.82823139e+02_fp, 3.82861807e+02_fp, 3.82896325e+02_fp,&
 3.82929211e+02_fp, 3.82963229e+02_fp, 3.83001367e+02_fp, 3.83044151e+02_fp, 3.83086855e+02_fp,&
 3.83128253e+02_fp, 3.83164506e+02_fp, 3.83194127e+02_fp, 3.83218072e+02_fp, 3.83237331e+02_fp,&
 3.83252313e+02_fp, 3.83264940e+02_fp, 3.83277627e+02_fp, 3.83291488e+02_fp, 3.83308524e+02_fp,&
 3.83328380e+02_fp, 3.83347090e+02_fp, 3.83365096e+02_fp, 3.83394620e+02_fp, 3.83444994e+02_fp,&
 3.83510928e+02_fp, 3.83581313e+02_fp, 3.83656398e+02_fp, 3.83737054e+02_fp, 3.83815572e+02_fp,&
 3.83884500e+02_fp, 3.83941349e+02_fp, 3.83985090e+02_fp, 3.84014928e+02_fp, 3.84031542e+02_fp,&
 3.84037247e+02_fp /
  DATA Level_Absorber(:,N2O,MIDLATITUDE_SUMMER) / &
 4.44272444e-04_fp, 5.51336175e-04_fp, 6.72162275e-04_fp, 8.22959935e-04_fp, 1.01506641e-03_fp,&
 1.28899910e-03_fp, 1.63390485e-03_fp, 2.01325058e-03_fp, 2.66128474e-03_fp, 3.38910082e-03_fp,&
 5.36322688e-03_fp, 7.34627544e-03_fp, 1.01729639e-02_fp, 1.36170547e-02_fp, 1.74294441e-02_fp,&
 2.21371446e-02_fp, 2.72182590e-02_fp, 3.33641284e-02_fp, 3.92555822e-02_fp, 4.49532363e-02_fp,&
 5.06993205e-02_fp, 5.79385725e-02_fp, 6.48623503e-02_fp, 7.02113679e-02_fp, 7.47662764e-02_fp,&
 7.91435800e-02_fp, 8.24240699e-02_fp, 8.53352826e-02_fp, 8.81436407e-02_fp, 9.17866925e-02_fp,&
 9.63283383e-02_fp, 1.00999959e-01_fp, 1.06683310e-01_fp, 1.13579530e-01_fp, 1.23399170e-01_fp,&
 1.34196806e-01_fp, 1.49944713e-01_fp, 1.64456225e-01_fp, 1.76738307e-01_fp, 1.90453894e-01_fp,&
 2.06762225e-01_fp, 2.21287632e-01_fp, 2.32750973e-01_fp, 2.43522809e-01_fp, 2.51989738e-01_fp,&
 2.60257554e-01_fp, 2.65338594e-01_fp, 2.69963449e-01_fp, 2.73760284e-01_fp, 2.76645402e-01_fp,&
 2.79468355e-01_fp, 2.81828533e-01_fp, 2.84045468e-01_fp, 2.86278232e-01_fp, 2.89007777e-01_fp,&
 2.91682900e-01_fp, 2.94105738e-01_fp, 2.95949414e-01_fp, 2.97757858e-01_fp, 3.00224080e-01_fp,&
 3.03871092e-01_fp, 3.07451123e-01_fp, 3.10472940e-01_fp, 3.12678656e-01_fp, 3.14845340e-01_fp,&
 3.16631854e-01_fp, 3.17661672e-01_fp, 3.18673891e-01_fp, 3.19526914e-01_fp, 3.19682739e-01_fp,&
 3.19835987e-01_fp, 3.19986726e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp /
  DATA Level_Absorber(:,CO,MIDLATITUDE_SUMMER) / &
 1.37767494e+00_fp, 5.57965300e-01_fp, 2.39585360e-01_fp, 1.07586828e-01_fp, 6.51303472e-02_fp,&
 5.16888743e-02_fp, 4.39230705e-02_fp, 3.94284528e-02_fp, 3.77490113e-02_fp, 3.63193831e-02_fp,&
 3.48375732e-02_fp, 3.33147544e-02_fp, 3.19873796e-02_fp, 3.07597204e-02_fp, 2.95981823e-02_fp,&
 2.81191347e-02_fp, 2.67650414e-02_fp, 2.55682692e-02_fp, 2.44665687e-02_fp, 2.34695921e-02_fp,&
 2.24856690e-02_fp, 2.13581764e-02_fp, 2.02798179e-02_fp, 1.92755631e-02_fp, 1.83248827e-02_fp,&
 1.74112712e-02_fp, 1.67208634e-02_fp, 1.61062061e-02_fp, 1.55132649e-02_fp, 1.48078768e-02_fp,&
 1.39861601e-02_fp, 1.33893734e-02_fp, 1.28549115e-02_fp, 1.23889507e-02_fp, 1.23200000e-02_fp,&
 1.23861211e-02_fp, 1.29350798e-02_fp, 1.36569733e-02_fp, 1.47915046e-02_fp, 1.62652790e-02_fp,&
 1.83198332e-02_fp, 2.04831706e-02_fp, 2.29104287e-02_fp, 2.53243310e-02_fp, 2.79089728e-02_fp,&
 3.04328322e-02_fp, 3.39608441e-02_fp, 3.75297588e-02_fp, 4.13501941e-02_fp, 4.54652835e-02_fp,&
 4.94917069e-02_fp, 5.43611506e-02_fp, 5.93455611e-02_fp, 6.42671772e-02_fp, 6.94389467e-02_fp,&
 7.45076007e-02_fp, 7.92373561e-02_fp, 8.32377855e-02_fp, 8.71617682e-02_fp, 9.08749833e-02_fp,&
 9.42765887e-02_fp, 9.76157205e-02_fp, 1.00894232e-01_fp, 1.04113919e-01_fp, 1.07276631e-01_fp,&
 1.10343709e-01_fp, 1.13272254e-01_fp, 1.16150752e-01_fp, 1.18833734e-01_fp, 1.20765964e-01_fp,&
 1.22666234e-01_fp, 1.24535399e-01_fp, 1.25821421e-01_fp, 1.27033334e-01_fp, 1.28225997e-01_fp,&
 1.29027858e-01_fp, 1.29466791e-01_fp, 1.29898945e-01_fp, 1.30315105e-01_fp, 1.30573625e-01_fp,&
 1.30828258e-01_fp, 1.31079093e-01_fp, 1.31730107e-01_fp, 1.32752782e-01_fp, 1.33760572e-01_fp,&
 1.34753805e-01_fp, 1.36033546e-01_fp, 1.37347121e-01_fp, 1.38642142e-01_fp, 1.39920093e-01_fp,&
 1.41251754e-01_fp, 1.42564990e-01_fp, 1.43860178e-01_fp, 1.45136666e-01_fp, 1.46387549e-01_fp,&
 1.47621563e-01_fp, 1.48839036e-01_fp, 1.50040284e-01_fp, 1.51225617e-01_fp, 1.52395332e-01_fp,&
 1.53549720e-01_fp /
  DATA Level_Absorber(:,CH4,MIDLATITUDE_SUMMER) / &
 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp,&
 1.50000000e-01_fp, 1.52769504e-01_fp, 1.57204193e-01_fp, 1.77111000e-01_fp, 1.98037434e-01_fp,&
 2.39280928e-01_fp, 2.80650840e-01_fp, 3.20355156e-01_fp, 3.57473145e-01_fp, 3.92076264e-01_fp,&
 4.24362792e-01_fp, 4.54872141e-01_fp, 4.83876882e-01_fp, 5.11460808e-01_fp, 5.37806255e-01_fp,&
 5.63850970e-01_fp, 5.93931310e-01_fp, 6.22700800e-01_fp, 6.49559097e-01_fp, 6.75015046e-01_fp,&
 6.99478417e-01_fp, 7.25509106e-01_fp, 7.51257436e-01_fp, 7.76096066e-01_fp, 8.19108788e-01_fp,&
 8.81383992e-01_fp, 9.41062665e-01_fp, 9.99670977e-01_fp, 1.05745012e+00_fp, 1.10974729e+00_fp,&
 1.15867523e+00_fp, 1.19749049e+00_fp, 1.23307224e+00_fp, 1.26273659e+00_fp, 1.28880857e+00_fp,&
 1.30950192e+00_fp, 1.32819400e+00_fp, 1.34350940e+00_fp, 1.35854608e+00_fp, 1.37369743e+00_fp,&
 1.38849246e+00_fp, 1.40200310e+00_fp, 1.41510003e+00_fp, 1.42719056e+00_fp, 1.43819956e+00_fp,&
 1.44897136e+00_fp, 1.45953315e+00_fp, 1.46987885e+00_fp, 1.48006168e+00_fp, 1.49047705e+00_fp,&
 1.50068475e+00_fp, 1.51124436e+00_fp, 1.52307171e+00_fp, 1.53467305e+00_fp, 1.54657860e+00_fp,&
 1.55918976e+00_fp, 1.57156930e+00_fp, 1.58369042e+00_fp, 1.59554203e+00_fp, 1.60718392e+00_fp,&
 1.61645186e+00_fp, 1.62095731e+00_fp, 1.62538577e+00_fp, 1.63007656e+00_fp, 1.63630956e+00_fp,&
 1.64243946e+00_fp, 1.64846903e+00_fp, 1.65529090e+00_fp, 1.66208944e+00_fp, 1.66877998e+00_fp,&
 1.67427858e+00_fp, 1.67866791e+00_fp, 1.68298945e+00_fp, 1.68716783e+00_fp, 1.69004028e+00_fp,&
 1.69286953e+00_fp, 1.69565659e+00_fp, 1.69742982e+00_fp, 1.69825901e+00_fp, 1.69907614e+00_fp,&
 1.69988146e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp /
    
  DATA Level_Absorber(:,H2O,MIDLATITUDE_WINTER) / &
 8.74332246E-04_fp, 1.49718079E-03_fp, 1.95195793E-03_fp, 2.35711591E-03_fp, 2.65468785E-03_fp, &  
 2.85767582E-03_fp, 3.00096567E-03_fp, 3.04944903E-03_fp, 3.08308098E-03_fp, 3.10985748E-03_fp, &  
 3.10985748E-03_fp, 3.10985748E-03_fp, 3.09351587E-03_fp, 3.07299566E-03_fp, 3.05427505E-03_fp, &  
 3.03702602E-03_fp, 3.02095600E-03_fp, 3.00616912E-03_fp, 2.99230417E-03_fp, 2.97922469E-03_fp, &  
 2.96686654E-03_fp, 2.95506941E-03_fp, 2.94405213E-03_fp, 2.93351392E-03_fp, 2.92340138E-03_fp, &  
 2.91367493E-03_fp, 2.90431481E-03_fp, 2.89529602E-03_fp, 2.87835857E-03_fp, 2.85745031E-03_fp, &  
 2.83684182E-03_fp, 2.82482238E-03_fp, 2.81705932E-03_fp, 2.80603976E-03_fp, 2.79887173E-03_fp, &  
 2.79887173E-03_fp, 2.79887173E-03_fp, 2.79887173E-03_fp, 2.79887173E-03_fp, 2.79887173E-03_fp, &  
 2.79887173E-03_fp, 2.79887173E-03_fp, 2.81434083E-03_fp, 2.84255984E-03_fp, 2.87009063E-03_fp, &  
 2.89695072E-03_fp, 2.92319027E-03_fp, 2.94883936E-03_fp, 2.97391660E-03_fp, 3.01152488E-03_fp, &  
 3.05970643E-03_fp, 3.10686597E-03_fp, 3.32681239E-03_fp, 3.55402722E-03_fp, 3.91087891E-03_fp, &  
 4.78254265E-03_fp, 5.63707400E-03_fp, 7.47526348E-03_fp, 1.15169670E-02_fp, 1.54824914E-02_fp, &  
 1.97084441E-02_fp, 2.48533895E-02_fp, 2.99050194E-02_fp, 3.51272761E-02_fp, 4.51472902E-02_fp, &  
 5.49921348E-02_fp, 6.46667487E-02_fp, 8.45843276E-02_fp, 1.07466432E-01_fp, 1.29966066E-01_fp, &  
 1.62099986E-01_fp, 2.12027416E-01_fp, 2.61145512E-01_fp, 3.09475916E-01_fp, 3.63203514E-01_fp, &  
 4.17297509E-01_fp, 4.70548251E-01_fp, 5.28053350E-01_fp, 6.04854643E-01_fp, 6.80491245E-01_fp, &  
 7.54990531E-01_fp, 8.54663471E-01_fp, 9.85885149E-01_fp, 1.11518275E+00_fp, 1.24259726E+00_fp, &  
 1.35998736E+00_fp, 1.46918467E+00_fp, 1.57682899E+00_fp, 1.68295208E+00_fp, 1.78567781E+00_fp, &  
 1.88517492E+00_fp, 1.98329553E+00_fp, 2.08006757E+00_fp, 2.18395930E+00_fp, 2.30730386E+00_fp, &  
 2.42898440E+00_fp, 2.54903439E+00_fp, 2.66748424E+00_fp, 2.78436530E+00_fp, 2.89970630E+00_fp, &  
 3.01353544E+00_fp /
  DATA Level_Absorber(:,O3,MIDLATITUDE_WINTER) / &
 5.13824109E-01_fp, 2.41632954E-01_fp, 3.01139649E-01_fp, 4.80917555E-01_fp, 7.91364237E-01_fp, &  
 1.18908931E+00_fp, 1.64984950E+00_fp, 2.25519636E+00_fp, 2.88203086E+00_fp, 3.80482427E+00_fp, &  
 4.62171912E+00_fp, 5.64746315E+00_fp, 6.42547774E+00_fp, 6.95559661E+00_fp, 7.13618951E+00_fp, &  
 7.16580449E+00_fp, 7.11413005E+00_fp, 6.99974493E+00_fp, 6.86599321E+00_fp, 6.65957726E+00_fp, &  
 6.38140684E+00_fp, 6.11586435E+00_fp, 5.93419689E+00_fp, 5.76476456E+00_fp, 5.60217603E+00_fp, &  
 5.44579517E+00_fp, 5.29530398E+00_fp, 5.15030072E+00_fp, 4.92238564E+00_fp, 4.65345678E+00_fp, &  
 4.38838356E+00_fp, 4.13446410E+00_fp, 3.88989110E+00_fp, 3.65366241E+00_fp, 3.38741275E+00_fp, &  
 3.05569077E+00_fp, 2.73064175E+00_fp, 2.41212341E+00_fp, 2.13743042E+00_fp, 1.88945447E+00_fp, &  
 1.67691975E+00_fp, 1.48641733E+00_fp, 1.32538682E+00_fp, 1.18927599E+00_fp, 1.07098983E+00_fp, &  
 9.84619020E-01_fp, 9.00243622E-01_fp, 8.58883440E-01_fp, 8.18564495E-01_fp, 7.79803354E-01_fp, &  
 7.42464818E-01_fp, 7.05918287E-01_fp, 6.40673216E-01_fp, 5.74770590E-01_fp, 5.11627432E-01_fp, &  
 4.55289027E-01_fp, 4.00057942E-01_fp, 3.49484701E-01_fp, 3.07909396E-01_fp, 2.67117714E-01_fp, &  
 2.30569056E-01_fp, 2.05080137E-01_fp, 1.80053518E-01_fp, 1.55870755E-01_fp, 1.39525396E-01_fp, &  
 1.23465785E-01_fp, 1.07683866E-01_fp, 9.82162019E-02_fp, 9.08257439E-02_fp, 8.35588159E-02_fp, &  
 7.68041071E-02_fp, 7.08774194E-02_fp, 6.50468045E-02_fp, 5.93096935E-02_fp, 5.57481739E-02_fp, &  
 5.26523318E-02_fp, 4.96047496E-02_fp, 4.65702815E-02_fp, 4.34473838E-02_fp, 4.03718448E-02_fp, &  
 3.73425513E-02_fp, 3.52424995E-02_fp, 3.42842253E-02_fp, 3.33400020E-02_fp, 3.24095305E-02_fp, &  
 3.15057165E-02_fp, 3.06253763E-02_fp, 2.97575562E-02_fp, 2.89020001E-02_fp, 2.84289367E-02_fp, &  
 2.83112407E-02_fp, 2.81951729E-02_fp, 2.80807004E-02_fp, 2.79853632E-02_fp, 2.79347499E-02_fp, &  
 2.78848194E-02_fp, 2.78355580E-02_fp, 2.77869531E-02_fp, 2.77389920E-02_fp, 2.76916629E-02_fp, &  
 2.76449541E-02_fp /
  DATA Level_Absorber(:,CO2,MIDLATITUDE_WINTER) / &
 3.74360000e+02_fp, 3.74361536e+02_fp, 3.74363355e+02_fp, 3.74366645e+02_fp, 3.74372000e+02_fp,&
 3.74380012e+02_fp, 3.74391000e+02_fp, 3.74404976e+02_fp, 3.74427783e+02_fp, 3.74467018e+02_fp,&
 3.74521289e+02_fp, 3.74580048e+02_fp, 3.74628801e+02_fp, 3.74664910e+02_fp, 3.74699114e+02_fp,&
 3.74739482e+02_fp, 3.74792946e+02_fp, 3.74861608e+02_fp, 3.74938271e+02_fp, 3.75015741e+02_fp,&
 3.75084934e+02_fp, 3.75140614e+02_fp, 3.75186705e+02_fp, 3.75224301e+02_fp, 3.75257801e+02_fp,&
 3.75293898e+02_fp, 3.75341078e+02_fp, 3.75402006e+02_fp, 3.75483301e+02_fp, 3.75585596e+02_fp,&
 3.75708307e+02_fp, 3.75851657e+02_fp, 3.76012795e+02_fp, 3.76192873e+02_fp, 3.76383982e+02_fp,&
 3.76585651e+02_fp, 3.76809813e+02_fp, 3.77081169e+02_fp, 3.77391271e+02_fp, 3.77732795e+02_fp,&
 3.78108482e+02_fp, 3.78481072e+02_fp, 3.78820139e+02_fp, 3.79146343e+02_fp, 3.79424024e+02_fp,&
 3.79651892e+02_fp, 3.79872699e+02_fp, 3.80073036e+02_fp, 3.80265614e+02_fp, 3.80453361e+02_fp,&
 3.80639313e+02_fp, 3.80813470e+02_fp, 3.80970199e+02_fp, 3.81124934e+02_fp, 3.81280663e+02_fp,&
 3.81437259e+02_fp, 3.81594518e+02_fp, 3.81753006e+02_fp, 3.81910795e+02_fp, 3.82065175e+02_fp,&
 3.82213512e+02_fp, 3.82349223e+02_fp, 3.82473964e+02_fp, 3.82577331e+02_fp, 3.82656946e+02_fp,&
 3.82723325e+02_fp, 3.82777620e+02_fp, 3.82823139e+02_fp, 3.82861807e+02_fp, 3.82896325e+02_fp,&
 3.82929211e+02_fp, 3.82963229e+02_fp, 3.83001367e+02_fp, 3.83044151e+02_fp, 3.83086855e+02_fp,&
 3.83128253e+02_fp, 3.83164506e+02_fp, 3.83194127e+02_fp, 3.83218072e+02_fp, 3.83237331e+02_fp,&
 3.83252313e+02_fp, 3.83264940e+02_fp, 3.83277627e+02_fp, 3.83291488e+02_fp, 3.83308524e+02_fp,&
 3.83328380e+02_fp, 3.83347090e+02_fp, 3.83365096e+02_fp, 3.83394620e+02_fp, 3.83444994e+02_fp,&
 3.83510928e+02_fp, 3.83581313e+02_fp, 3.83656398e+02_fp, 3.83737054e+02_fp, 3.83815572e+02_fp,&
 3.83884500e+02_fp, 3.83941349e+02_fp, 3.83985090e+02_fp, 3.84014928e+02_fp, 3.84031542e+02_fp,&
 3.84037247e+02_fp /
  DATA Level_Absorber(:,N2O,MIDLATITUDE_WINTER) / &
 5.32276359e-04_fp, 7.23016133e-04_fp, 9.47945748e-04_fp, 1.22656559e-03_fp, 1.58250604e-03_fp,&
 2.03057156e-03_fp, 2.57026674e-03_fp, 3.44052011e-03_fp, 4.70738858e-03_fp, 8.65747482e-03_fp,&
 1.32391894e-02_fp, 1.93061481e-02_fp, 2.56077835e-02_fp, 3.17531520e-02_fp, 3.76105133e-02_fp,&
 4.36059374e-02_fp, 4.95376252e-02_fp, 5.45353578e-02_fp, 5.90382101e-02_fp, 6.40643550e-02_fp,&
 6.96198615e-02_fp, 7.49232807e-02_fp, 7.91165296e-02_fp, 8.30778781e-02_fp, 8.68791824e-02_fp,&
 8.99154484e-02_fp, 9.28289249e-02_fp, 9.56361598e-02_fp, 1.00246690e-01_fp, 1.05684336e-01_fp,&
 1.10720681e-01_fp, 1.16786637e-01_fp, 1.23333665e-01_fp, 1.31129223e-01_fp, 1.41059910e-01_fp,&
 1.55379299e-01_fp, 1.68366138e-01_fp, 1.80151508e-01_fp, 1.95061984e-01_fp, 2.11478013e-01_fp,&
 2.25000121e-01_fp, 2.36763755e-01_fp, 2.46825487e-01_fp, 2.55445858e-01_fp, 2.62739108e-01_fp,&
 2.67619001e-01_fp, 2.72386226e-01_fp, 2.75524844e-01_fp, 2.78589110e-01_fp, 2.81257053e-01_fp,&
 2.83581018e-01_fp, 2.85855719e-01_fp, 2.88651003e-01_fp, 2.91427399e-01_fp, 2.93981427e-01_fp,&
 2.95838354e-01_fp, 2.97658790e-01_fp, 3.00002012e-01_fp, 3.03549523e-01_fp, 3.07030153e-01_fp,&
 3.10136589e-01_fp, 3.12263305e-01_fp, 3.14351453e-01_fp, 3.16350257e-01_fp, 3.17340887e-01_fp,&
 3.18314194e-01_fp, 3.19270678e-01_fp, 3.19613722e-01_fp, 3.19761590e-01_fp, 3.19906986e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp /
  DATA Level_Absorber(:,CO,MIDLATITUDE_WINTER) / &
 4.34896419e+00_fp, 2.34472097e+00_fp, 1.29914967e+00_fp, 7.14807742e-01_fp, 4.16057775e-01_fp,&
 2.53527748e-01_fp, 1.58357108e-01_fp, 1.02771632e-01_fp, 5.90536475e-02_fp, 3.71617519e-02_fp,&
 3.04969215e-02_fp, 2.70650268e-02_fp, 2.46909672e-02_fp, 2.29690134e-02_fp, 2.18613521e-02_fp,&
 2.09570291e-02_fp, 2.01819828e-02_fp, 1.94787351e-02_fp, 1.88233694e-02_fp, 1.82191633e-02_fp,&
 1.76628179e-02_fp, 1.71317172e-02_fp, 1.67285960e-02_fp, 1.63490656e-02_fp, 1.59848688e-02_fp,&
 1.56715859e-02_fp, 1.53706069e-02_fp, 1.50806033e-02_fp, 1.45448405e-02_fp, 1.38917852e-02_fp,&
 1.32754956e-02_fp, 1.27596248e-02_fp, 1.23200000e-02_fp, 1.23200000e-02_fp, 1.25057649e-02_fp,&
 1.30531084e-02_fp, 1.39253234e-02_fp, 1.50826255e-02_fp, 1.68458149e-02_fp, 1.89139370e-02_fp,&
 2.12692565e-02_fp, 2.37600987e-02_fp, 2.63325170e-02_fp, 2.89639987e-02_fp, 3.19548689e-02_fp,&
 3.57205917e-02_fp, 3.93993709e-02_fp, 4.38670145e-02_fp, 4.82376259e-02_fp, 5.30762733e-02_fp,&
 5.83013232e-02_fp, 6.34156076e-02_fp, 6.87629524e-02_fp, 7.40234922e-02_fp, 7.89676257e-02_fp,&
 8.29968067e-02_fp, 8.69468074e-02_fp, 9.06678577e-02_fp, 9.39766577e-02_fp, 9.72230773e-02_fp,&
 1.00403260e-01_fp, 1.03507630e-01_fp, 1.06555702e-01_fp, 1.09542919e-01_fp, 1.12360024e-01_fp,&
 1.15127865e-01_fp, 1.17847864e-01_fp, 1.19910147e-01_fp, 1.21743714e-01_fp, 1.23546628e-01_fp,&
 1.25120364e-01_fp, 1.26303387e-01_fp, 1.27467242e-01_fp, 1.28612429e-01_fp, 1.29152081e-01_fp,&
 1.29567816e-01_fp, 1.29977071e-01_fp, 1.30349154e-01_fp, 1.30592920e-01_fp, 1.30832988e-01_fp,&
 1.31069445e-01_fp, 1.31630992e-01_fp, 1.32597104e-01_fp, 1.33549041e-01_fp, 1.34487123e-01_fp,&
 1.35604109e-01_fp, 1.36858158e-01_fp, 1.38094364e-01_fp, 1.39313107e-01_fp, 1.40535555e-01_fp,&
 1.41760555e-01_fp, 1.42968607e-01_fp, 1.44160056e-01_fp, 1.45332653e-01_fp, 1.46482954e-01_fp,&
 1.47617742e-01_fp, 1.48737320e-01_fp, 1.49841978e-01_fp, 1.50932000e-01_fp, 1.52007660e-01_fp,&
 1.53069225e-01_fp /
  DATA Level_Absorber(:,CH4,MIDLATITUDE_WINTER) / &
 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp,&
 1.51999548e-01_fp, 1.56869968e-01_fp, 1.76961071e-01_fp, 2.00547370e-01_fp, 2.44302970e-01_fp,&
 2.88471902e-01_fp, 3.29417957e-01_fp, 3.67663628e-01_fp, 4.03292534e-01_fp, 4.36401976e-01_fp,&
 4.67558697e-01_fp, 4.96958787e-01_fp, 5.28259891e-01_fp, 5.59289448e-01_fp, 5.88481357e-01_fp,&
 6.15980717e-01_fp, 6.42232262e-01_fp, 6.66747463e-01_fp, 6.90197020e-01_fp, 7.12699181e-01_fp,&
 7.37703969e-01_fp, 7.61812385e-01_fp, 7.85041673e-01_fp, 8.31687107e-01_fp, 8.90821477e-01_fp,&
 9.52450437e-01_fp, 1.01148653e+00_fp, 1.06822443e+00_fp, 1.12019482e+00_fp, 1.16713489e+00_fp,&
 1.20583594e+00_fp, 1.24008873e+00_fp, 1.27034847e+00_fp, 1.29465569e+00_fp, 1.31548570e+00_fp,&
 1.33315401e+00_fp, 1.34887060e+00_fp, 1.36445613e+00_fp, 1.37988206e+00_fp, 1.39464172e+00_fp,&
 1.40846089e+00_fp, 1.42196099e+00_fp, 1.43392375e+00_fp, 1.44561634e+00_fp, 1.45686625e+00_fp,&
 1.46771142e+00_fp, 1.47832669e+00_fp, 1.48911567e+00_fp, 1.49970981e+00_fp, 1.51044689e+00_fp,&
 1.52235925e+00_fp, 1.53403752e+00_fp, 1.54581070e+00_fp, 1.55807779e+00_fp, 1.57011361e+00_fp,&
 1.58188316e+00_fp, 1.59331029e+00_fp, 1.60453019e+00_fp, 1.61521988e+00_fp, 1.61955388e+00_fp,&
 1.62381210e+00_fp, 1.62799671e+00_fp, 1.63354886e+00_fp, 1.63946359e+00_fp, 1.64527944e+00_fp,&
 1.65135814e+00_fp, 1.65799461e+00_fp, 1.66452356e+00_fp, 1.67094777e+00_fp, 1.67552081e+00_fp,&
 1.67967816e+00_fp, 1.68377071e+00_fp, 1.68754615e+00_fp, 1.69025466e+00_fp, 1.69292208e+00_fp,&
 1.69554939e+00_fp, 1.69734945e+00_fp, 1.69813279e+00_fp, 1.69890463e+00_fp, 1.69966523e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp /

  DATA Level_Absorber(:,H2O,SUBARCTIC_SUMMER) / &
 8.69383728E-04_fp, 1.36116616E-03_fp, 1.76469831E-03_fp, 2.10026204E-03_fp, 2.44573806E-03_fp, &  
 2.68348563E-03_fp, 2.86237283E-03_fp, 2.99606196E-03_fp, 3.04567272E-03_fp, 3.07754036E-03_fp, &  
 3.10655579E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &  
 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &  
 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10332744E-03_fp, 3.09229744E-03_fp, &  
 3.08169754E-03_fp, 3.07164176E-03_fp, 3.06200913E-03_fp, 3.05271704E-03_fp, 3.03799773E-03_fp, &  
 3.01659788E-03_fp, 3.00831971E-03_fp, 2.99845479E-03_fp, 2.98689086E-03_fp, 2.95285888E-03_fp, &  
 2.91662195E-03_fp, 2.88155558E-03_fp, 2.84738628E-03_fp, 2.81403006E-03_fp, 2.76435656E-03_fp, &  
 2.70130163E-03_fp, 2.63110641E-03_fp, 2.55605408E-03_fp, 2.51179113E-03_fp, 2.49757023E-03_fp, &  
 2.48788599E-03_fp, 2.48788599E-03_fp, 2.48788599E-03_fp, 2.48788599E-03_fp, 2.48788599E-03_fp, &  
 2.55305414E-03_fp, 2.66197293E-03_fp, 2.77071777E-03_fp, 3.13346408E-03_fp, 3.48887207E-03_fp, &  
 4.22682497E-03_fp, 5.83106127E-03_fp, 7.40422819E-03_fp, 1.09552220E-02_fp, 1.69722497E-02_fp, &  
 2.28773144E-02_fp, 3.34187290E-02_fp, 5.08394269E-02_fp, 6.79481093E-02_fp, 9.32770644E-02_fp, &  
 1.45896166E-01_fp, 1.97605412E-01_fp, 2.48431246E-01_fp, 3.24174426E-01_fp, 3.98706765E-01_fp, &  
 4.72005572E-01_fp, 5.61681354E-01_fp, 6.58406539E-01_fp, 7.53580570E-01_fp, 8.61983503E-01_fp, &  
 1.02204873E+00_fp, 1.17961877E+00_fp, 1.33475601E+00_fp, 1.52343275E+00_fp, 1.72391468E+00_fp, &  
 1.92138205E+00_fp, 2.11943114E+00_fp, 2.36062938E+00_fp, 2.59829098E+00_fp, 2.83249127E+00_fp, &  
 3.08782910E+00_fp, 3.40062556E+00_fp, 3.70897345E+00_fp, 4.01296376E+00_fp, 4.31618141E+00_fp, &  
 4.62075034E+00_fp, 4.92110566E+00_fp, 5.21733292E+00_fp, 5.57739905E+00_fp, 6.06579203E+00_fp, &  
 6.54759618E+00_fp, 7.02294405E+00_fp, 7.49195601E+00_fp, 7.95475623E+00_fp, 8.41145844E+00_fp, &  
 8.86217434E+00_fp /
  DATA Level_Absorber(:,O3,SUBARCTIC_SUMMER) / &
 6.02447863E-01_fp, 1.85384866E-01_fp, 2.45755187E-01_fp, 4.43875328E-01_fp, 7.61277121E-01_fp, &  
 1.05158664E+00_fp, 1.34585211E+00_fp, 1.65291517E+00_fp, 2.07443467E+00_fp, 2.48432665E+00_fp, &  
 3.12568202E+00_fp, 3.95253307E+00_fp, 4.85091650E+00_fp, 5.83269455E+00_fp, 6.90940812E+00_fp, &  
 7.44284770E+00_fp, 7.78765316E+00_fp, 7.73487051E+00_fp, 7.58385625E+00_fp, 7.21986676E+00_fp, &  
 6.85994658E+00_fp, 6.36681745E+00_fp, 5.89515953E+00_fp, 5.61600841E+00_fp, 5.47413698E+00_fp, &  
 5.33779770E+00_fp, 5.11691393E+00_fp, 4.86911782E+00_fp, 4.63008203E+00_fp, 4.40678739E+00_fp, &  
 4.20034843E+00_fp, 3.86871267E+00_fp, 3.57850376E+00_fp, 3.33060533E+00_fp, 2.98547464E+00_fp, &  
 2.63590625E+00_fp, 2.29762991E+00_fp, 2.01200498E+00_fp, 1.79748565E+00_fp, 1.58901363E+00_fp, &  
 1.38625538E+00_fp, 1.21632217E+00_fp, 1.07152004E+00_fp, 9.65303396E-01_fp, 8.96710707E-01_fp, &  
 8.30054162E-01_fp, 7.65667586E-01_fp, 7.02740825E-01_fp, 6.58995663E-01_fp, 6.17019138E-01_fp, &  
 5.76716277E-01_fp, 5.37801027E-01_fp, 4.99725104E-01_fp, 4.65860708E-01_fp, 4.32681386E-01_fp, &  
 3.98552843E-01_fp, 3.61453664E-01_fp, 3.25072988E-01_fp, 2.87804478E-01_fp, 2.49240970E-01_fp, &  
 2.11395041E-01_fp, 1.82920946E-01_fp, 1.67893374E-01_fp, 1.53134955E-01_fp, 1.38232663E-01_fp, &  
 1.22272790E-01_fp, 1.06588886E-01_fp, 9.11729285E-02_fp, 8.69304911E-02_fp, 8.27823305E-02_fp, &  
 7.87028231E-02_fp, 7.50455252E-02_fp, 7.16194928E-02_fp, 6.82484027E-02_fp, 6.48620845E-02_fp, &  
 6.12800313E-02_fp, 5.77538175E-02_fp, 5.42820463E-02_fp, 5.15875999E-02_fp, 4.92297424E-02_fp, &  
 4.69073391E-02_fp, 4.46667180E-02_fp, 4.30751373E-02_fp, 4.15068937E-02_fp, 3.99614901E-02_fp, &  
 3.84894396E-02_fp, 3.71657147E-02_fp, 3.58608157E-02_fp, 3.45743575E-02_fp, 3.33635638E-02_fp, &  
 3.22617141E-02_fp, 3.11751081E-02_fp, 3.01034363E-02_fp, 2.89659113E-02_fp, 2.76858761E-02_fp, &  
 2.64231096E-02_fp, 2.51772645E-02_fp, 2.39480253E-02_fp, 2.27350665E-02_fp, 2.15380901E-02_fp, &  
 2.03568032E-02_fp /
  DATA Level_Absorber(:,CO2,SUBARCTIC_SUMMER) / &
 3.74360000e+02_fp, 3.74361536e+02_fp, 3.74363355e+02_fp, 3.74366645e+02_fp, 3.74372000e+02_fp,&
 3.74380012e+02_fp, 3.74391000e+02_fp, 3.74404976e+02_fp, 3.74427783e+02_fp, 3.74467018e+02_fp,&
 3.74521289e+02_fp, 3.74580048e+02_fp, 3.74628801e+02_fp, 3.74664910e+02_fp, 3.74699114e+02_fp,&
 3.74739482e+02_fp, 3.74792946e+02_fp, 3.74861608e+02_fp, 3.74938271e+02_fp, 3.75015741e+02_fp,&
 3.75084934e+02_fp, 3.75140614e+02_fp, 3.75186705e+02_fp, 3.75224301e+02_fp, 3.75257801e+02_fp,&
 3.75293898e+02_fp, 3.75341078e+02_fp, 3.75402006e+02_fp, 3.75483301e+02_fp, 3.75585596e+02_fp,&
 3.75708307e+02_fp, 3.75851657e+02_fp, 3.76012795e+02_fp, 3.76192873e+02_fp, 3.76383982e+02_fp,&
 3.76585651e+02_fp, 3.76809813e+02_fp, 3.77081169e+02_fp, 3.77391271e+02_fp, 3.77732795e+02_fp,&
 3.78108482e+02_fp, 3.78481072e+02_fp, 3.78820139e+02_fp, 3.79146343e+02_fp, 3.79424024e+02_fp,&
 3.79651892e+02_fp, 3.79872699e+02_fp, 3.80073036e+02_fp, 3.80265614e+02_fp, 3.80453361e+02_fp,&
 3.80639313e+02_fp, 3.80813470e+02_fp, 3.80970199e+02_fp, 3.81124934e+02_fp, 3.81280663e+02_fp,&
 3.81437259e+02_fp, 3.81594518e+02_fp, 3.81753006e+02_fp, 3.81910795e+02_fp, 3.82065175e+02_fp,&
 3.82213512e+02_fp, 3.82349223e+02_fp, 3.82473964e+02_fp, 3.82577331e+02_fp, 3.82656946e+02_fp,&
 3.82723325e+02_fp, 3.82777620e+02_fp, 3.82823139e+02_fp, 3.82861807e+02_fp, 3.82896325e+02_fp,&
 3.82929211e+02_fp, 3.82963229e+02_fp, 3.83001367e+02_fp, 3.83044151e+02_fp, 3.83086855e+02_fp,&
 3.83128253e+02_fp, 3.83164506e+02_fp, 3.83194127e+02_fp, 3.83218072e+02_fp, 3.83237331e+02_fp,&
 3.83252313e+02_fp, 3.83264940e+02_fp, 3.83277627e+02_fp, 3.83291488e+02_fp, 3.83308524e+02_fp,&
 3.83328380e+02_fp, 3.83347090e+02_fp, 3.83365096e+02_fp, 3.83394620e+02_fp, 3.83444994e+02_fp,&
 3.83510928e+02_fp, 3.83581313e+02_fp, 3.83656398e+02_fp, 3.83737054e+02_fp, 3.83815572e+02_fp,&
 3.83884500e+02_fp, 3.83941349e+02_fp, 3.83985090e+02_fp, 3.84014928e+02_fp, 3.84031542e+02_fp,&
 3.84037247e+02_fp /
  DATA Level_Absorber(:,N2O,SUBARCTIC_SUMMER) / &
 4.04678710e-04_fp, 4.88629908e-04_fp, 5.81597226e-04_fp, 6.98665057e-04_fp, 8.66076326e-04_fp,&
 1.09254970e-03_fp, 1.37473925e-03_fp, 1.71306712e-03_fp, 2.24713918e-03_fp, 2.77477543e-03_fp,&
 4.37100282e-03_fp, 5.85837731e-03_fp, 7.75858832e-03_fp, 1.00121396e-02_fp, 1.25283411e-02_fp,&
 1.64767185e-02_fp, 2.08269136e-02_fp, 2.65436487e-02_fp, 3.19488939e-02_fp, 3.70537919e-02_fp,&
 4.21364042e-02_fp, 4.94801353e-02_fp, 5.65038392e-02_fp, 6.23204035e-02_fp, 6.72362289e-02_fp,&
 7.19603764e-02_fp, 7.65838738e-02_fp, 8.10689382e-02_fp, 8.53955431e-02_fp, 9.12237456e-02_fp,&
 9.89169668e-02_fp, 1.06397144e-01_fp, 1.13723457e-01_fp, 1.20912521e-01_fp, 1.28719501e-01_fp,&
 1.37318014e-01_fp, 1.49552266e-01_fp, 1.62199775e-01_fp, 1.75607050e-01_fp, 1.87748687e-01_fp,&
 1.98799005e-01_fp, 2.07516551e-01_fp, 2.14467026e-01_fp, 2.20282502e-01_fp, 2.24992563e-01_fp,&
 2.29170681e-01_fp, 2.32304171e-01_fp, 2.35366599e-01_fp, 2.38698351e-01_fp, 2.41972507e-01_fp,&
 2.44720297e-01_fp, 2.47094148e-01_fp, 2.49416788e-01_fp, 2.51486287e-01_fp, 2.53513908e-01_fp,&
 2.55456000e-01_fp, 2.57257968e-01_fp, 2.59025021e-01_fp, 2.60756013e-01_fp, 2.62451477e-01_fp,&
 2.64115398e-01_fp, 2.65772579e-01_fp, 2.67435203e-01_fp, 2.69068048e-01_fp, 2.70714811e-01_fp,&
 2.72472087e-01_fp, 2.74198987e-01_fp, 2.75896376e-01_fp, 2.77827320e-01_fp, 2.79726544e-01_fp,&
 2.81594355e-01_fp, 2.83870321e-01_fp, 2.86321664e-01_fp, 2.88733685e-01_fp, 2.91342676e-01_fp,&
 2.94762429e-01_fp, 2.98128883e-01_fp, 3.01443350e-01_fp, 3.03495087e-01_fp, 3.05020759e-01_fp,&
 3.06523484e-01_fp, 3.07940251e-01_fp, 3.08505789e-01_fp, 3.09063031e-01_fp, 3.09612161e-01_fp,&
 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp,&
 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp,&
 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp, 3.10000000e-01_fp,&
 3.10000000e-01_fp /
  DATA Level_Absorber(:,CO,SUBARCTIC_SUMMER) / &
 1.71842575e+00_fp, 6.59646252e-01_fp, 2.60919500e-01_fp, 1.11713491e-01_fp, 6.89879139e-02_fp,&
 5.25152363e-02_fp, 4.31507231e-02_fp, 3.73850267e-02_fp, 3.44871859e-02_fp, 3.22844327e-02_fp,&
 3.02244094e-02_fp, 2.85601870e-02_fp, 2.72113285e-02_fp, 2.60616500e-02_fp, 2.50454787e-02_fp,&
 2.41130105e-02_fp, 2.32324874e-02_fp, 2.23879097e-02_fp, 2.15599695e-02_fp, 2.07137029e-02_fp,&
 1.99069220e-02_fp, 1.91302207e-02_fp, 1.83873668e-02_fp, 1.77461225e-02_fp, 1.71821859e-02_fp,&
 1.66402382e-02_fp, 1.61718864e-02_fp, 1.57413450e-02_fp, 1.53260149e-02_fp, 1.47582182e-02_fp,&
 1.40012823e-02_fp, 1.33838115e-02_fp, 1.28421903e-02_fp, 1.23773801e-02_fp, 1.23200000e-02_fp,&
 1.24257527e-02_fp, 1.29839052e-02_fp, 1.37895804e-02_fp, 1.49586948e-02_fp, 1.66470195e-02_fp,&
 1.87607730e-02_fp, 2.11187891e-02_fp, 2.36431630e-02_fp, 2.62316032e-02_fp, 2.88838703e-02_fp,&
 3.18494987e-02_fp, 3.55925170e-02_fp, 3.92506499e-02_fp, 4.38548884e-02_fp, 4.84051253e-02_fp,&
 5.33909524e-02_fp, 5.86406656e-02_fp, 6.37839543e-02_fp, 6.92022781e-02_fp, 7.45109596e-02_fp,&
 7.93937256e-02_fp, 8.34569858e-02_fp, 8.74415173e-02_fp, 9.11194131e-02_fp, 9.44372049e-02_fp,&
 9.76932685e-02_fp, 1.00884966e-01_fp, 1.04011979e-01_fp, 1.07082982e-01_fp, 1.10074068e-01_fp,&
 1.12929641e-01_fp, 1.15735854e-01_fp, 1.18494110e-01_fp, 1.20396727e-01_fp, 1.22265806e-01_fp,&
 1.24103969e-01_fp, 1.25515276e-01_fp, 1.26711764e-01_fp, 1.27889061e-01_fp, 1.28894408e-01_fp,&
 1.29329122e-01_fp, 1.29757061e-01_fp, 1.30178392e-01_fp, 1.30479196e-01_fp, 1.30728851e-01_fp,&
 1.30974752e-01_fp, 1.31270918e-01_fp, 1.32267343e-01_fp, 1.33249150e-01_fp, 1.34216665e-01_fp,&
 1.35274569e-01_fp, 1.36577451e-01_fp, 1.37861795e-01_fp, 1.39127997e-01_fp, 1.40395403e-01_fp,&
 1.41675457e-01_fp, 1.42937800e-01_fp, 1.44182795e-01_fp, 1.45411068e-01_fp, 1.46623220e-01_fp,&
 1.47819026e-01_fp, 1.48998803e-01_fp, 1.50162858e-01_fp, 1.51311490e-01_fp, 1.52444988e-01_fp,&
 1.53563633e-01_fp /
  DATA Level_Absorber(:,CH4,SUBARCTIC_SUMMER) / &
 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp,&
 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.57489541e-01_fp, 1.65685832e-01_fp,&
 1.96393702e-01_fp, 2.33487726e-01_fp, 2.74168722e-01_fp, 3.18562823e-01_fp, 3.66746837e-01_fp,&
 4.14976317e-01_fp, 4.59660892e-01_fp, 4.98405892e-01_fp, 5.33407516e-01_fp, 5.62890349e-01_fp,&
 5.90648953e-01_fp, 6.13415753e-01_fp, 6.35190414e-01_fp, 6.54993271e-01_fp, 6.73223677e-01_fp,&
 6.90743243e-01_fp, 7.07523538e-01_fp, 7.23661097e-01_fp, 7.39228508e-01_fp, 7.64354093e-01_fp,&
 8.01237515e-01_fp, 8.38470079e-01_fp, 8.75503278e-01_fp, 9.12440196e-01_fp, 9.55424196e-01_fp,&
 9.99027270e-01_fp, 1.04508895e+00_fp, 1.09069924e+00_fp, 1.13628397e+00_fp, 1.17725478e+00_fp,&
 1.21425814e+00_fp, 1.24673563e+00_fp, 1.27569594e+00_fp, 1.30202817e+00_fp, 1.32580711e+00_fp,&
 1.34824959e+00_fp, 1.36842412e+00_fp, 1.38814112e+00_fp, 1.40745203e+00_fp, 1.42634139e+00_fp,&
 1.44261492e+00_fp, 1.45701369e+00_fp, 1.47110683e+00_fp, 1.48427637e+00_fp, 1.49717942e+00_fp,&
 1.50992471e+00_fp, 1.52264448e+00_fp, 1.53511779e+00_fp, 1.54748480e+00_fp, 1.55978523e+00_fp,&
 1.57185681e+00_fp, 1.58365632e+00_fp, 1.59516679e+00_fp, 1.60647110e+00_fp, 1.61603703e+00_fp,&
 1.62043022e+00_fp, 1.62474747e+00_fp, 1.62899094e+00_fp, 1.63511848e+00_fp, 1.64114776e+00_fp,&
 1.64707732e+00_fp, 1.65357350e+00_fp, 1.66028551e+00_fp, 1.66688985e+00_fp, 1.67294408e+00_fp,&
 1.67729122e+00_fp, 1.68157061e+00_fp, 1.68578392e+00_fp, 1.68899107e+00_fp, 1.69176502e+00_fp,&
 1.69449724e+00_fp, 1.69705750e+00_fp, 1.69786541e+00_fp, 1.69866147e+00_fp, 1.69944594e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp /

  DATA Level_Absorber(:,H2O,SUBARCTIC_WINTER) / &
 8.84696872E-04_fp, 1.54955175E-03_fp, 2.03052785E-03_fp, 2.47065038E-03_fp, 2.74191225E-03_fp, &   
 2.92146556E-03_fp, 3.03039715E-03_fp, 3.06709313E-03_fp, 3.10012079E-03_fp, 3.10985748E-03_fp, &   
 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &   
 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &   
 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &   
 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.09146619E-03_fp, &   
 3.07147389E-03_fp, 3.05208862E-03_fp, 3.03329059E-03_fp, 3.01505782E-03_fp, 2.99733188E-03_fp, &   
 2.98010536E-03_fp, 2.96333121E-03_fp, 2.94701069E-03_fp, 2.93111777E-03_fp, 2.91562533E-03_fp, &   
 2.90051362E-03_fp, 2.88576944E-03_fp, 2.87137086E-03_fp, 2.85730694E-03_fp, 2.84353388E-03_fp, &   
 2.83008447E-03_fp, 2.81700923E-03_fp, 2.80423099E-03_fp, 2.79170619E-03_fp, 2.77943253E-03_fp, &   
 2.77862415E-03_fp, 3.14321091E-03_fp, 3.50019707E-03_fp, 4.03488780E-03_fp, 4.91460361E-03_fp, &   
 5.77677700E-03_fp, 7.22889321E-03_fp, 9.30815524E-03_fp, 1.13476641E-02_fp, 1.33261166E-02_fp, &   
 1.52411067E-02_fp, 1.71209366E-02_fp, 1.87015151E-02_fp, 1.94619961E-02_fp, 2.02090171E-02_fp, &   
 2.09429787E-02_fp, 3.85915946E-02_fp, 5.87588759E-02_fp, 7.85852755E-02_fp, 9.68791925E-02_fp, &   
 1.12588983E-01_fp, 1.28041819E-01_fp, 1.43244161E-01_fp, 1.71432368E-01_fp, 2.04090211E-01_fp, &   
 2.36234320E-01_fp, 2.67877346E-01_fp, 3.27052478E-01_fp, 3.85437353E-01_fp, 4.42936821E-01_fp, &   
 5.00201056E-01_fp, 5.60201978E-01_fp, 6.19315101E-01_fp, 6.77561459E-01_fp, 7.32076449E-01_fp, &   
 7.71902320E-01_fp, 8.11157232E-01_fp, 8.49853863E-01_fp, 8.87881652E-01_fp, 9.15307184E-01_fp, &   
 9.42350623E-01_fp, 9.69019926E-01_fp, 9.95322689E-01_fp, 9.85632908E-01_fp, 9.56886655E-01_fp, &   
 9.28528213E-01_fp, 9.00549779E-01_fp, 8.72944270E-01_fp, 8.45704377E-01_fp, 8.18823407E-01_fp, &   
 7.92294783E-01_fp /
  DATA Level_Absorber(:,O3,SUBARCTIC_WINTER) / &
 6.64488655E-01_fp, 2.70387123E-01_fp, 4.89988085E-01_fp, 6.44061875E-01_fp, 8.95052643E-01_fp, &   
 1.31605255E+00_fp, 1.82244415E+00_fp, 2.41243879E+00_fp, 2.87476351E+00_fp, 3.67618430E+00_fp, &   
 4.54517916E+00_fp, 5.26145086E+00_fp, 5.80774687E+00_fp, 6.07829864E+00_fp, 6.24554631E+00_fp, &   
 6.21897368E+00_fp, 6.16447409E+00_fp, 6.02603072E+00_fp, 5.89135176E+00_fp, 5.68564390E+00_fp, &   
 5.48967750E+00_fp, 5.30381111E+00_fp, 5.12709769E+00_fp, 4.95779587E+00_fp, 4.85867873E+00_fp, &   
 4.79704146E+00_fp, 4.73772631E+00_fp, 4.67583708E+00_fp, 4.60726127E+00_fp, 4.54086129E+00_fp, &   
 4.42972331E+00_fp, 4.24271853E+00_fp, 4.10758583E+00_fp, 3.98549190E+00_fp, 3.81449433E+00_fp, &   
 3.59662883E+00_fp, 3.27299705E+00_fp, 2.94629363E+00_fp, 2.61411140E+00_fp, 2.31486882E+00_fp, &   
 2.04760764E+00_fp, 1.81770663E+00_fp, 1.63250734E+00_fp, 1.46370947E+00_fp, 1.33084427E+00_fp, &   
 1.20110129E+00_fp, 1.07496781E+00_fp, 9.51699345E-01_fp, 8.42396537E-01_fp, 7.43729115E-01_fp, &   
 6.47186109E-01_fp, 5.52641076E-01_fp, 4.60067038E-01_fp, 3.93909310E-01_fp, 3.76229324E-01_fp, &   
 3.58901894E-01_fp, 3.41887263E-01_fp, 3.25172172E-01_fp, 3.08776655E-01_fp, 2.86854038E-01_fp, &   
 2.58462510E-01_fp, 2.30592267E-01_fp, 2.01994858E-01_fp, 1.70228779E-01_fp, 1.39024942E-01_fp, &   
 1.08366609E-01_fp, 9.61530320E-02_fp, 8.71327802E-02_fp, 7.82649952E-02_fp, 6.97853309E-02_fp, &   
 6.19016706E-02_fp, 5.41469585E-02_fp, 4.65179519E-02_fp, 4.31724147E-02_fp, 4.14266931E-02_fp, &   
 3.97084331E-02_fp, 3.80169583E-02_fp, 3.65604959E-02_fp, 3.51271963E-02_fp, 3.37156327E-02_fp, &   
 3.23437308E-02_fp, 3.10974815E-02_fp, 2.98696723E-02_fp, 2.86598662E-02_fp, 2.74879278E-02_fp, &   
 2.64305466E-02_fp, 2.53883245E-02_fp, 2.43609248E-02_fp, 2.33525872E-02_fp, 2.27333876E-02_fp, &   
 2.21228147E-02_fp, 2.15206889E-02_fp, 2.09268386E-02_fp, 2.03303186E-02_fp, 1.97360877E-02_fp, &   
 1.91498734E-02_fp, 1.85715146E-02_fp, 1.80008647E-02_fp, 1.74377726E-02_fp, 1.68821001E-02_fp, &   
 1.63337111E-02_fp /
  DATA Level_Absorber(:,CO2,SUBARCTIC_WINTER) / &
 3.74360000e+02_fp, 3.74361536e+02_fp, 3.74363355e+02_fp, 3.74366645e+02_fp, 3.74372000e+02_fp,&
 3.74380012e+02_fp, 3.74391000e+02_fp, 3.74404976e+02_fp, 3.74427783e+02_fp, 3.74467018e+02_fp,&
 3.74521289e+02_fp, 3.74580048e+02_fp, 3.74628801e+02_fp, 3.74664910e+02_fp, 3.74699114e+02_fp,&
 3.74739482e+02_fp, 3.74792946e+02_fp, 3.74861608e+02_fp, 3.74938271e+02_fp, 3.75015741e+02_fp,&
 3.75084934e+02_fp, 3.75140614e+02_fp, 3.75186705e+02_fp, 3.75224301e+02_fp, 3.75257801e+02_fp,&
 3.75293898e+02_fp, 3.75341078e+02_fp, 3.75402006e+02_fp, 3.75483301e+02_fp, 3.75585596e+02_fp,&
 3.75708307e+02_fp, 3.75851657e+02_fp, 3.76012795e+02_fp, 3.76192873e+02_fp, 3.76383982e+02_fp,&
 3.76585651e+02_fp, 3.76809813e+02_fp, 3.77081169e+02_fp, 3.77391271e+02_fp, 3.77732795e+02_fp,&
 3.78108482e+02_fp, 3.78481072e+02_fp, 3.78820139e+02_fp, 3.79146343e+02_fp, 3.79424024e+02_fp,&
 3.79651892e+02_fp, 3.79872699e+02_fp, 3.80073036e+02_fp, 3.80265614e+02_fp, 3.80453361e+02_fp,&
 3.80639313e+02_fp, 3.80813470e+02_fp, 3.80970199e+02_fp, 3.81124934e+02_fp, 3.81280663e+02_fp,&
 3.81437259e+02_fp, 3.81594518e+02_fp, 3.81753006e+02_fp, 3.81910795e+02_fp, 3.82065175e+02_fp,&
 3.82213512e+02_fp, 3.82349223e+02_fp, 3.82473964e+02_fp, 3.82577331e+02_fp, 3.82656946e+02_fp,&
 3.82723325e+02_fp, 3.82777620e+02_fp, 3.82823139e+02_fp, 3.82861807e+02_fp, 3.82896325e+02_fp,&
 3.82929211e+02_fp, 3.82963229e+02_fp, 3.83001367e+02_fp, 3.83044151e+02_fp, 3.83086855e+02_fp,&
 3.83128253e+02_fp, 3.83164506e+02_fp, 3.83194127e+02_fp, 3.83218072e+02_fp, 3.83237331e+02_fp,&
 3.83252313e+02_fp, 3.83264940e+02_fp, 3.83277627e+02_fp, 3.83291488e+02_fp, 3.83308524e+02_fp,&
 3.83328380e+02_fp, 3.83347090e+02_fp, 3.83365096e+02_fp, 3.83394620e+02_fp, 3.83444994e+02_fp,&
 3.83510928e+02_fp, 3.83581313e+02_fp, 3.83656398e+02_fp, 3.83737054e+02_fp, 3.83815572e+02_fp,&
 3.83884500e+02_fp, 3.83941349e+02_fp, 3.83985090e+02_fp, 3.84014928e+02_fp, 3.84031542e+02_fp,&
 3.84037247e+02_fp /
  DATA Level_Absorber(:,N2O,SUBARCTIC_WINTER) / &
 5.17198983e-04_fp, 7.11536301e-04_fp, 9.48007739e-04_fp, 1.24801137e-03_fp, 1.64041627e-03_fp,&
 2.16645303e-03_fp, 2.83368434e-03_fp, 3.69729454e-03_fp, 6.09623043e-03_fp, 9.27422811e-03_fp,&
 1.31757691e-02_fp, 1.80052348e-02_fp, 2.37410463e-02_fp, 2.96412008e-02_fp, 3.54866165e-02_fp,&
 4.20289857e-02_fp, 4.79313556e-02_fp, 5.27310397e-02_fp, 5.73774069e-02_fp, 6.39764445e-02_fp,&
 7.02630984e-02_fp, 7.56102331e-02_fp, 8.01481893e-02_fp, 8.44958826e-02_fp, 8.79882612e-02_fp,&
 9.09838274e-02_fp, 9.38665113e-02_fp, 9.77779709e-02_fp, 1.03675568e-01_fp, 1.08971973e-01_fp,&
 1.14729917e-01_fp, 1.21462068e-01_fp, 1.29099406e-01_fp, 1.37452693e-01_fp, 1.52215216e-01_fp,&
 1.65924691e-01_fp, 1.77898993e-01_fp, 1.92127311e-01_fp, 2.09042820e-01_fp, 2.23468543e-01_fp,&
 2.35470992e-01_fp, 2.46008954e-01_fp, 2.54805887e-01_fp, 2.62466946e-01_fp, 2.67471565e-01_fp,&
 2.72358493e-01_fp, 2.75567476e-01_fp, 2.78690269e-01_fp, 2.81382493e-01_fp, 2.83750501e-01_fp,&
 2.86085530e-01_fp, 2.88959724e-01_fp, 2.91773947e-01_fp, 2.94245610e-01_fp, 2.96119685e-01_fp,&
 2.97956394e-01_fp, 3.00636131e-01_fp, 3.04213144e-01_fp, 3.07721801e-01_fp, 3.10578647e-01_fp,&
 3.12692245e-01_fp, 3.14767024e-01_fp, 3.16541666e-01_fp, 3.17500641e-01_fp, 3.18442645e-01_fp,&
 3.19368177e-01_fp, 3.19624635e-01_fp, 3.19767904e-01_fp, 3.19908752e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp /
  DATA Level_Absorber(:,CO,SUBARCTIC_WINTER) / &
 4.35656598e+00_fp, 2.43514308e+00_fp, 1.55469614e+00_fp, 1.08495496e+00_fp, 7.79402033e-01_fp,&
 5.84873371e-01_fp, 4.46359217e-01_fp, 3.46017623e-01_fp, 2.46910564e-01_fp, 1.75979932e-01_fp,&
 1.29950471e-01_fp, 9.61525780e-02_fp, 6.92625474e-02_fp, 5.43125173e-02_fp, 4.31666585e-02_fp,&
 3.64648489e-02_fp, 3.08725149e-02_fp, 2.77250297e-02_fp, 2.47823468e-02_fp, 2.29351100e-02_fp,&
 2.11753172e-02_fp, 1.97639849e-02_fp, 1.86507012e-02_fp, 1.75840942e-02_fp, 1.68047114e-02_fp,&
 1.61852579e-02_fp, 1.55891474e-02_fp, 1.49176343e-02_fp, 1.40878561e-02_fp, 1.34500082e-02_fp,&
 1.28943113e-02_fp, 1.24268008e-02_fp, 1.23200000e-02_fp, 1.23678829e-02_fp, 1.29321646e-02_fp,&
 1.36855778e-02_fp, 1.48614326e-02_fp, 1.64760993e-02_fp, 1.86071469e-02_fp, 2.09449587e-02_fp,&
 2.34863679e-02_fp, 2.60832597e-02_fp, 2.87686390e-02_fp, 3.17448465e-02_fp, 3.56068187e-02_fp,&
 3.93779700e-02_fp, 4.39278210e-02_fp, 4.83819098e-02_fp, 5.33583043e-02_fp, 5.86823769e-02_fp,&
 6.39020575e-02_fp, 6.93478990e-02_fp, 7.46801107e-02_fp, 7.95408520e-02_fp, 8.36072411e-02_fp,&
 8.75925540e-02_fp, 9.12593075e-02_fp, 9.45956240e-02_fp, 9.78681847e-02_fp, 1.01048533e-01_fp,&
 1.04133754e-01_fp, 1.07162312e-01_fp, 1.10087238e-01_fp, 1.12814322e-01_fp, 1.15493148e-01_fp,&
 1.18125130e-01_fp, 1.20045469e-01_fp, 1.21822008e-01_fp, 1.23568521e-01_fp, 1.25099574e-01_fp,&
 1.26251507e-01_fp, 1.27384585e-01_fp, 1.28499304e-01_fp, 1.29099439e-01_fp, 1.29505421e-01_fp,&
 1.29905015e-01_fp, 1.30298382e-01_fp, 1.30538057e-01_fp, 1.30773454e-01_fp, 1.31005280e-01_fp,&
 1.31341808e-01_fp, 1.32290598e-01_fp, 1.33225354e-01_fp, 1.34146395e-01_fp, 1.35111222e-01_fp,&
 1.36337883e-01_fp, 1.37546960e-01_fp, 1.38738834e-01_fp, 1.39914322e-01_fp, 1.41110500e-01_fp,&
 1.42290016e-01_fp, 1.43453214e-01_fp, 1.44600424e-01_fp, 1.45721631e-01_fp, 1.46822057e-01_fp,&
 1.47907643e-01_fp, 1.48978677e-01_fp, 1.50035439e-01_fp, 1.51078199e-01_fp, 1.52107220e-01_fp,&
 1.53122757e-01_fp /
  DATA Level_Absorber(:,CH4,SUBARCTIC_WINTER) / &
 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp,&
 1.55069153e-01_fp, 1.67676017e-01_fp, 1.90682027e-01_fp, 2.30495504e-01_fp, 2.79166399e-01_fp,&
 3.31746928e-01_fp, 3.86884549e-01_fp, 4.41306475e-01_fp, 4.87840601e-01_fp, 5.29545878e-01_fp,&
 5.63825555e-01_fp, 5.94823817e-01_fp, 6.20252912e-01_fp, 6.44276841e-01_fp, 6.65135395e-01_fp,&
 6.85006552e-01_fp, 7.03873196e-01_fp, 7.21827104e-01_fp, 7.39028258e-01_fp, 7.55478495e-01_fp,&
 7.71257609e-01_fp, 7.86442117e-01_fp, 8.08616151e-01_fp, 8.44138889e-01_fp, 8.79243617e-01_fp,&
 9.13830688e-01_fp, 9.48239458e-01_fp, 9.82854212e-01_fp, 1.01738566e+00_fp, 1.05728437e+00_fp,&
 1.09726582e+00_fp, 1.13879831e+00_fp, 1.17731675e+00_fp, 1.21257869e+00_fp, 1.24474140e+00_fp,&
 1.27389715e+00_fp, 1.30069819e+00_fp, 1.32477401e+00_fp, 1.34768553e+00_fp, 1.36850120e+00_fp,&
 1.38882736e+00_fp, 1.40775479e+00_fp, 1.42624501e+00_fp, 1.44252537e+00_fp, 1.45712809e+00_fp,&
 1.47139389e+00_fp, 1.48463031e+00_fp, 1.49759055e+00_fp, 1.51038528e+00_fp, 1.52311484e+00_fp,&
 1.53559060e+00_fp, 1.54800344e+00_fp, 1.56037255e+00_fp, 1.57250529e+00_fp, 1.58425840e+00_fp,&
 1.59561505e+00_fp, 1.60676311e+00_fp, 1.61605729e+00_fp, 1.62025280e+00_fp, 1.62437407e+00_fp,&
 1.62842328e+00_fp, 1.63398538e+00_fp, 1.63971615e+00_fp, 1.64535007e+00_fp, 1.65124152e+00_fp,&
 1.65770358e+00_fp, 1.66405987e+00_fp, 1.67031317e+00_fp, 1.67499439e+00_fp, 1.67905421e+00_fp,&
 1.68305015e+00_fp, 1.68698382e+00_fp, 1.68964508e+00_fp, 1.69226060e+00_fp, 1.69483644e+00_fp,&
 1.69711498e+00_fp, 1.69788427e+00_fp, 1.69864218e+00_fp, 1.69938897e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp /

  DATA Level_Absorber(:,H2O,US_STANDARD_ATMOSPHERE) / &
 8.86997040E-04_fp, 1.52427987E-03_fp, 2.01101378E-03_fp, 2.40599752E-03_fp, 2.72437587E-03_fp, &   
 2.96236227E-03_fp, 3.10396312E-03_fp, 3.19368814E-03_fp, 3.23610094E-03_fp, 3.25981037E-03_fp, &   
 3.25671227E-03_fp, 3.23196162E-03_fp, 3.19285497E-03_fp, 3.13971111E-03_fp, 3.10625749E-03_fp, &   
 3.08100578E-03_fp, 3.06248483E-03_fp, 3.04448043E-03_fp, 3.02081059E-03_fp, 2.99803553E-03_fp, &   
 2.97440450E-03_fp, 2.95184624E-03_fp, 2.92511340E-03_fp, 2.89196774E-03_fp, 2.86016092E-03_fp, &   
 2.82977471E-03_fp, 2.80069281E-03_fp, 2.77267144E-03_fp, 2.73864263E-03_fp, 2.68478764E-03_fp, &   
 2.64118787E-03_fp, 2.59714199E-03_fp, 2.54483821E-03_fp, 2.50568261E-03_fp, 2.47311126E-03_fp, &   
 2.44672175E-03_fp, 2.42271064E-03_fp, 2.40621873E-03_fp, 2.39238370E-03_fp, 2.38458004E-03_fp, &   
 2.38110201E-03_fp, 2.38850647E-03_fp, 2.39914434E-03_fp, 2.42737549E-03_fp, 2.45492999E-03_fp, &   
 2.71901083E-03_fp, 2.99417069E-03_fp, 3.24509817E-03_fp, 3.47714919E-03_fp, 3.78040842E-03_fp, &   
 4.95999293E-03_fp, 6.11455674E-03_fp, 7.57663188E-03_fp, 9.42311177E-03_fp, 1.12322375E-02_fp, &   
 1.42466434E-02_fp, 1.78602949E-02_fp, 2.14039606E-02_fp, 2.73154771E-02_fp, 3.41753036E-02_fp, &   
 4.09074844E-02_fp, 5.42724979E-02_fp, 7.17133280E-02_fp, 8.88417820E-02_fp, 1.15945509E-01_fp, &   
 1.56046471E-01_fp, 1.95454032E-01_fp, 2.34255117E-01_fp, 2.72750760E-01_fp, 3.10602956E-01_fp, &   
 3.47828689E-01_fp, 4.06535547E-01_fp, 4.70306893E-01_fp, 5.33055556E-01_fp, 6.01901014E-01_fp, &   
 6.85096335E-01_fp, 7.66994753E-01_fp, 8.47628708E-01_fp, 9.65099457E-01_fp, 1.09450413E+00_fp, &   
 1.22196301E+00_fp, 1.34952968E+00_fp, 1.52006224E+00_fp, 1.68809431E+00_fp, 1.85367915E+00_fp, &   
 2.03383153E+00_fp, 2.26692831E+00_fp, 2.49671002E+00_fp, 2.72324444E+00_fp, 2.94770972E+00_fp, &   
 3.17163654E+00_fp, 3.39246541E+00_fp, 3.61025923E+00_fp, 3.83453885E+00_fp, 4.08727861E+00_fp, &   
 4.33660869E+00_fp, 4.58259770E+00_fp, 4.82530792E+00_fp, 5.06480362E+00_fp, 5.30114363E+00_fp, &   
 5.53438577E+00_fp /
  DATA Level_Absorber(:,O3,US_STANDARD_ATMOSPHERE) / &
 4.73303997E-01_fp, 2.74146902E-01_fp, 2.80243651E-01_fp, 5.10480276E-01_fp, 8.31073636E-01_fp, &   
 1.12571713E+00_fp, 1.58104620E+00_fp, 2.16173433E+00_fp, 2.87091990E+00_fp, 3.74371439E+00_fp, &   
 4.73885897E+00_fp, 5.61330664E+00_fp, 6.34570524E+00_fp, 7.09761422E+00_fp, 7.50525354E+00_fp, &   
 7.77591671E+00_fp, 7.81936233E+00_fp, 7.80536994E+00_fp, 7.56992926E+00_fp, 7.33375237E+00_fp, &   
 7.02220367E+00_fp, 6.72479815E+00_fp, 6.44285088E+00_fp, 6.17639446E+00_fp, 5.92070090E+00_fp, &   
 5.68739719E+00_fp, 5.47387081E+00_fp, 5.26813104E+00_fp, 5.03222902E+00_fp, 4.69211312E+00_fp, &   
 4.38133099E+00_fp, 4.07406867E+00_fp, 3.74952993E+00_fp, 3.39674005E+00_fp, 3.03656514E+00_fp, &   
 2.78144823E+00_fp, 2.52642430E+00_fp, 2.23528370E+00_fp, 1.96713487E+00_fp, 1.74480839E+00_fp, &   
 1.53398068E+00_fp, 1.34350343E+00_fp, 1.16379657E+00_fp, 1.01995634E+00_fp, 8.79563704E-01_fp, &   
 7.81925294E-01_fp, 6.89400605E-01_fp, 6.15902146E-01_fp, 5.56537854E-01_fp, 4.99365854E-01_fp, &   
 4.53753714E-01_fp, 4.09109075E-01_fp, 3.72418996E-01_fp, 3.45262678E-01_fp, 3.18655731E-01_fp, &   
 2.88188006E-01_fp, 2.55989671E-01_fp, 2.24414924E-01_fp, 1.95655516E-01_fp, 1.68400469E-01_fp, &   
 1.41652577E-01_fp, 1.23541576E-01_fp, 1.10965251E-01_fp, 9.86141761E-02_fp, 8.73600481E-02_fp, &   
 7.74538294E-02_fp, 6.77189026E-02_fp, 5.91969666E-02_fp, 5.63118453E-02_fp, 5.34749481E-02_fp, &   
 5.06850024E-02_fp, 4.80182173E-02_fp, 4.54157757E-02_fp, 4.28550687E-02_fp, 4.08112252E-02_fp, &   
 3.98355325E-02_fp, 3.88750495E-02_fp, 3.79293958E-02_fp, 3.69055940E-02_fp, 3.58639488E-02_fp, &   
 3.48379664E-02_fp, 3.38620749E-02_fp, 3.36773243E-02_fp, 3.34952826E-02_fp, 3.33158922E-02_fp, &   
 3.31308212E-02_fp, 3.29213219E-02_fp, 3.27148022E-02_fp, 3.25112010E-02_fp, 3.21398618E-02_fp, &   
 3.13748034E-02_fp, 3.06203294E-02_fp, 2.98762248E-02_fp, 2.91576054E-02_fp, 2.84997705E-02_fp, &   
 2.78508104E-02_fp, 2.72105464E-02_fp, 2.65788165E-02_fp, 2.59554534E-02_fp, 2.53403040E-02_fp, &   
 2.47332178E-02_fp /
  DATA Level_Absorber(:,CO2,US_STANDARD_ATMOSPHERE) / &
 3.74360000e+02_fp, 3.74361536e+02_fp, 3.74363355e+02_fp, 3.74366645e+02_fp, 3.74372000e+02_fp,&
 3.74380012e+02_fp, 3.74391000e+02_fp, 3.74404976e+02_fp, 3.74427783e+02_fp, 3.74467018e+02_fp,&
 3.74521289e+02_fp, 3.74580048e+02_fp, 3.74628801e+02_fp, 3.74664910e+02_fp, 3.74699114e+02_fp,&
 3.74739482e+02_fp, 3.74792946e+02_fp, 3.74861608e+02_fp, 3.74938271e+02_fp, 3.75015741e+02_fp,&
 3.75084934e+02_fp, 3.75140614e+02_fp, 3.75186705e+02_fp, 3.75224301e+02_fp, 3.75257801e+02_fp,&
 3.75293898e+02_fp, 3.75341078e+02_fp, 3.75402006e+02_fp, 3.75483301e+02_fp, 3.75585596e+02_fp,&
 3.75708307e+02_fp, 3.75851657e+02_fp, 3.76012795e+02_fp, 3.76192873e+02_fp, 3.76383982e+02_fp,&
 3.76585651e+02_fp, 3.76809813e+02_fp, 3.77081169e+02_fp, 3.77391271e+02_fp, 3.77732795e+02_fp,&
 3.78108482e+02_fp, 3.78481072e+02_fp, 3.78820139e+02_fp, 3.79146343e+02_fp, 3.79424024e+02_fp,&
 3.79651892e+02_fp, 3.79872699e+02_fp, 3.80073036e+02_fp, 3.80265614e+02_fp, 3.80453361e+02_fp,&
 3.80639313e+02_fp, 3.80813470e+02_fp, 3.80970199e+02_fp, 3.81124934e+02_fp, 3.81280663e+02_fp,&
 3.81437259e+02_fp, 3.81594518e+02_fp, 3.81753006e+02_fp, 3.81910795e+02_fp, 3.82065175e+02_fp,&
 3.82213512e+02_fp, 3.82349223e+02_fp, 3.82473964e+02_fp, 3.82577331e+02_fp, 3.82656946e+02_fp,&
 3.82723325e+02_fp, 3.82777620e+02_fp, 3.82823139e+02_fp, 3.82861807e+02_fp, 3.82896325e+02_fp,&
 3.82929211e+02_fp, 3.82963229e+02_fp, 3.83001367e+02_fp, 3.83044151e+02_fp, 3.83086855e+02_fp,&
 3.83128253e+02_fp, 3.83164506e+02_fp, 3.83194127e+02_fp, 3.83218072e+02_fp, 3.83237331e+02_fp,&
 3.83252313e+02_fp, 3.83264940e+02_fp, 3.83277627e+02_fp, 3.83291488e+02_fp, 3.83308524e+02_fp,&
 3.83328380e+02_fp, 3.83347090e+02_fp, 3.83365096e+02_fp, 3.83394620e+02_fp, 3.83444994e+02_fp,&
 3.83510928e+02_fp, 3.83581313e+02_fp, 3.83656398e+02_fp, 3.83737054e+02_fp, 3.83815572e+02_fp,&
 3.83884500e+02_fp, 3.83941349e+02_fp, 3.83985090e+02_fp, 3.84014928e+02_fp, 3.84031542e+02_fp,&
 3.84037247e+02_fp /
  DATA Level_Absorber(:,N2O,US_STANDARD_ATMOSPHERE) / &
 5.89486325e-04_fp, 7.99946346e-04_fp, 1.04611880e-03_fp, 1.33724706e-03_fp, 1.68957918e-03_fp,&
 2.09942616e-03_fp, 2.70755658e-03_fp, 3.48736380e-03_fp, 4.44336017e-03_fp, 7.72942738e-03_fp,&
 1.30073625e-02_fp, 2.03461663e-02_fp, 2.98432661e-02_fp, 4.18880873e-02_fp, 5.40792692e-02_fp,&
 6.58805180e-02_fp, 8.04411489e-02_fp, 9.43692730e-02_fp, 1.06420058e-01_fp, 1.17701210e-01_fp,&
 1.27237718e-01_fp, 1.36341497e-01_fp, 1.44126168e-01_fp, 1.50236934e-01_fp, 1.56100816e-01_fp,&
 1.61635299e-01_fp, 1.66872091e-01_fp, 1.71917921e-01_fp, 1.77678822e-01_fp, 1.85921925e-01_fp,&
 1.92424046e-01_fp, 1.98214388e-01_fp, 2.03446973e-01_fp, 2.10881540e-01_fp, 2.19202070e-01_fp,&
 2.28789801e-01_fp, 2.38051329e-01_fp, 2.46642502e-01_fp, 2.54743418e-01_fp, 2.61970292e-01_fp,&
 2.68584503e-01_fp, 2.73917915e-01_fp, 2.78988261e-01_fp, 2.83254910e-01_fp, 2.87419312e-01_fp,&
 2.90390197e-01_fp, 2.93213136e-01_fp, 2.95690079e-01_fp, 2.97903682e-01_fp, 3.00050400e-01_fp,&
 3.01938057e-01_fp, 3.03785697e-01_fp, 3.05562317e-01_fp, 3.07261854e-01_fp, 3.08926999e-01_fp,&
 3.10513780e-01_fp, 3.12045420e-01_fp, 3.13547383e-01_fp, 3.14897771e-01_fp, 3.16169239e-01_fp,&
 3.17417050e-01_fp, 3.18213314e-01_fp, 3.18721192e-01_fp, 3.19219973e-01_fp, 3.19567457e-01_fp,&
 3.19722145e-01_fp, 3.19874159e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp, 3.20000000e-01_fp,&
 3.20000000e-01_fp /
  DATA Level_Absorber(:,CO,US_STANDARD_ATMOSPHERE) / &
 3.00647777e+00_fp, 1.05484796e+00_fp, 4.37113095e-01_fp, 2.42958176e-01_fp, 1.60383696e-01_fp,&
 1.05793717e-01_fp, 7.91855724e-02_fp, 6.07096526e-02_fp, 4.95672747e-02_fp, 4.03060655e-02_fp,&
 3.45252104e-02_fp, 3.08152747e-02_fp, 2.78069818e-02_fp, 2.55716490e-02_fp, 2.38328681e-02_fp,&
 2.23333516e-02_fp, 2.10958744e-02_fp, 1.99815939e-02_fp, 1.91748256e-02_fp, 1.84330003e-02_fp,&
 1.79010834e-02_fp, 1.73933030e-02_fp, 1.69355053e-02_fp, 1.65375950e-02_fp, 1.61557609e-02_fp,&
 1.58112322e-02_fp, 1.54995184e-02_fp, 1.51991714e-02_fp, 1.48088029e-02_fp, 1.41299591e-02_fp,&
 1.35022431e-02_fp, 1.29347868e-02_fp, 1.24675917e-02_fp, 1.23200000e-02_fp, 1.23200000e-02_fp,&
 1.28636200e-02_fp, 1.35187590e-02_fp, 1.46748552e-02_fp, 1.60817398e-02_fp, 1.81745222e-02_fp,&
 2.03532098e-02_fp, 2.28437230e-02_fp, 2.53146716e-02_fp, 2.79472850e-02_fp, 3.05168094e-02_fp,&
 3.41912717e-02_fp, 3.78653057e-02_fp, 4.19526285e-02_fp, 4.63154386e-02_fp, 5.06640593e-02_fp,&
 5.58608944e-02_fp, 6.09475620e-02_fp, 6.60756108e-02_fp, 7.12827006e-02_fp, 7.63844225e-02_fp,&
 8.07307713e-02_fp, 8.46449614e-02_fp, 8.84833111e-02_fp, 9.19373730e-02_fp, 9.51910260e-02_fp,&
 9.83841442e-02_fp, 1.01535134e-01_fp, 1.04639536e-01_fp, 1.07688333e-01_fp, 1.10627726e-01_fp,&
 1.13443039e-01_fp, 1.16209689e-01_fp, 1.18799981e-01_fp, 1.20669129e-01_fp, 1.22507030e-01_fp,&
 1.24314531e-01_fp, 1.25646968e-01_fp, 1.26836493e-01_fp, 1.28006938e-01_fp, 1.28934641e-01_fp,&
 1.29360087e-01_fp, 1.29778903e-01_fp, 1.30191251e-01_fp, 1.30482931e-01_fp, 1.30728989e-01_fp,&
 1.30971346e-01_fp, 1.31242494e-01_fp, 1.32233190e-01_fp, 1.33209351e-01_fp, 1.34171304e-01_fp,&
 1.35203575e-01_fp, 1.36496783e-01_fp, 1.37771591e-01_fp, 1.39028391e-01_fp, 1.40283563e-01_fp,&
 1.41558660e-01_fp, 1.42816117e-01_fp, 1.44056291e-01_fp, 1.45281170e-01_fp, 1.46494885e-01_fp,&
 1.47692233e-01_fp, 1.48873531e-01_fp, 1.50039087e-01_fp, 1.51189201e-01_fp, 1.52324160e-01_fp,&
 1.53444248e-01_fp /
  DATA Level_Absorber(:,CH4,US_STANDARD_ATMOSPHERE) / &
 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp, 1.50000000e-01_fp,&
 1.50552291e-01_fp, 1.60308394e-01_fp, 1.77517906e-01_fp, 2.02072607e-01_fp, 2.53316226e-01_fp,&
 3.24972887e-01_fp, 4.00692685e-01_fp, 4.74959957e-01_fp, 5.44959372e-01_fp, 6.04030660e-01_fp,&
 6.57082145e-01_fp, 7.05860369e-01_fp, 7.51727113e-01_fp, 7.94348838e-01_fp, 8.34000842e-01_fp,&
 8.65763874e-01_fp, 8.96085623e-01_fp, 9.24380276e-01_fp, 9.50457613e-01_fp, 9.75481387e-01_fp,&
 9.98476211e-01_fp, 1.01967275e+00_fp, 1.04009635e+00_fp, 1.06600553e+00_fp, 1.10964549e+00_fp,&
 1.15707124e+00_fp, 1.20560302e+00_fp, 1.25606009e+00_fp, 1.30555719e+00_fp, 1.35385117e+00_fp,&
 1.39288867e+00_fp, 1.42936262e+00_fp, 1.45906050e+00_fp, 1.48581806e+00_fp, 1.50639458e+00_fp,&
 1.52524144e+00_fp, 1.54047976e+00_fp, 1.55512336e+00_fp, 1.56828642e+00_fp, 1.58113405e+00_fp,&
 1.59123501e+00_fp, 1.60092569e+00_fp, 1.60992576e+00_fp, 1.61837769e+00_fp, 1.62658318e+00_fp,&
 1.63390267e+00_fp, 1.64106699e+00_fp, 1.64775732e+00_fp, 1.65390458e+00_fp, 1.65992744e+00_fp,&
 1.66492870e+00_fp, 1.66935343e+00_fp, 1.67369244e+00_fp, 1.67730198e+00_fp, 1.68056215e+00_fp,&
 1.68376167e+00_fp, 1.68656657e+00_fp, 1.68910596e+00_fp, 1.69159986e+00_fp, 1.69353966e+00_fp,&
 1.69477716e+00_fp, 1.69599327e+00_fp, 1.69709677e+00_fp, 1.69769972e+00_fp, 1.69829259e+00_fp,&
 1.69887566e+00_fp, 1.69923097e+00_fp, 1.69952110e+00_fp, 1.69980657e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp, 1.70000000e+00_fp,&
 1.70000000e+00_fp /


CONTAINS


!--------------------------------------------------------------------------------
!:sdoc+:
!
! NAME:
!       CRTM_Get_Model_Profile
!
! PURPOSE:
!       Subroutine to retrieve climatological model profile levels for
!       supplementing a user input profile for use in the CRTM. 
!
!       Currently the only absorbers available are water vapour and ozone.
!
! CALLING SEQUENCE:
!       CALL CRTM_Get_Model_Profile( Absorber_Id, &
!                                    Pressure   , &
!                                    Temperature, &
!                                    Absorber   , &
!                                    Model=Model  )
!                                    
!
! INPUTS:
!       Absorber_Id:     Integer array containing the absorber id's.
!                        UNITS:      N/A
!                        TYPE:       INTEGER
!                        DIMENSION:  Rank-1 (J)
!                        ATTRIBUTES: INTENT(IN)
!
! OUTPUTS:
!       Pressure:        Model profile pressure levels.
!                        UNITS:      hectoPascals
!                        TYPE:       REAL(fp)
!                        DIMENSION:  Rank-1 (0:)
!                        ATTRIBUTES: INTENT(OUT)
!
!       Temperature:     Model profile level temperatures.
!                        UNITS:      Kelvin
!                        TYPE:       REAL(fp)
!                        DIMENSION:  Rank-1 (0:)
!                        ATTRIBUTES: INTENT(OUT)
!
!       Absorber:        Model profile level absorber amounts.
!                        Note that the number and order of the absorbers will
!                        change in the future.
!                        UNITS:      g/kg (H2O) and ppmv (everything else)
!                        TYPE:       REAL(fp)
!                        DIMENSION:  Rank-2 (0:,J)
!                        ATTRIBUTES: INTENT(OUT)
!
! OPTIONAL INPUTS:
!       Model:           Integer flag defining the climatological profile
!                        to retrieve. Valid values are:
!                          1: Tropical        
!                          2: Midlatitude summer   
!                          3: Midlatitude winter   
!                          4: Subarctic summer    
!                          5: Subarctic winter    
!                          6: U.S. Standard Atmosphere (DEFAULT)
!                        If no model is specified, or if an invalid model
!                        is specified, the U.S. Standard Atmosphere model
!                        is used.
!                        UNITS:      N/A
!                        TYPE:       INTEGER
!                        DIMENSION:  Scalar
!                        ATTRIBUTES: INTENT(IN), OPTIONAL
!
! COMMENTS:
!       - The model profiles are specified at the AIRS 101 pressure levels. If
!         the output arguments are larger than this, only the 101 defined 
!         levels are filled with profile data.
!       - Currently the only absorbers available are water vapour and ozone.
!
!:sdoc-:
!--------------------------------------------------------------------------------

  SUBROUTINE CRTM_Get_Model_Profile( absorber_id, p, t, a, Model )
    ! Arguments
    INTEGER ,           INTENT(IN)  :: absorber_id(:)
    REAL(fp),           INTENT(OUT) :: p(0:), t(0:), a(0:,:) 
    INTEGER , OPTIONAL, INTENT(IN)  :: Model            
    ! Local Variables
    INTEGER :: i, j, k
    
    ! Use US Std Atm if passed model is invalid
    i = US_STANDARD_ATMOSPHERE
    IF ( PRESENT(Model) ) THEN
      i = Model
      IF ( i < 1 .OR. i > N_MODELS ) i = US_STANDARD_ATMOSPHERE
    END IF

    ! Get the requireds number of levels    
    k = MIN(SIZE(p)-1,N_MODEL_LAYERS)
    p = MODEL_LEVEL_PRESSURE(0:k)
    t = Level_Temperature(0:k,i)
    DO j = 1, SIZE(absorber_id)
      a(0:k,j) = Level_Absorber(0:k,ABSORBER_ID_MAP(absorber_id(j)),i)
    END DO
    
  END SUBROUTINE CRTM_Get_Model_Profile

END MODULE CRTM_Model_Profiles
