   subroutine endrun
   call abort
   return
   end
module TridiagonalMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: TridiagonalMod
!
! !DESCRIPTION:
! Tridiagonal matrix solution
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: Tridiagonal
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: Tridiagonal
!
! !INTERFACE:
  subroutine Tridiagonal (lbc, ubc, lbj, ubj, jtop, numf, filter, &
                          a, b, c, r, u)
!
! !DESCRIPTION:
! Tridiagonal matrix solution
!
! !USES:
    use shr_kind_mod, only: r8 => shr_kind_r8
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)    :: lbc, ubc               ! lbinning and ubing column indices
    integer , intent(in)    :: lbj, ubj               ! lbinning and ubing level indices
    integer , intent(in)    :: jtop(lbc:ubc)          ! top level for each column
    integer , intent(in)    :: numf                   ! filter dimension
    integer , intent(in)    :: filter(1:numf)         ! filter
    real(r8), intent(in)    :: a(lbc:ubc, lbj:ubj)    ! "a" left off diagonal of tridiagonal matrix
    real(r8), intent(in)    :: b(lbc:ubc, lbj:ubj)    ! "b" diagonal column for tridiagonal matrix
    real(r8), intent(in)    :: c(lbc:ubc, lbj:ubj)    ! "c" right off diagonal tridiagonal matrix
    real(r8), intent(in)    :: r(lbc:ubc, lbj:ubj)    ! "r" forcing term of tridiagonal matrix
    real(r8), intent(inout) :: u(lbc:ubc, lbj:ubj)    ! solution
!
! !CALLED FROM:
! subroutine BiogeophysicsLake in module BiogeophysicsLakeMod
! subroutine SoilTemperature in module SoilTemperatureMod
! subroutine SoilWater in module HydrologyMod
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
!  1 July 2003: Mariana Vertenstein; modified for vectorization
!
!
! !OTHER LOCAL VARIABLES:
!EOP
!
    integer  :: j,ci,fc                   !indices
    real(r8) :: gam(lbc:ubc,lbj:ubj)      !temporary
    real(r8) :: bet(lbc:ubc)              !temporary
!-----------------------------------------------------------------------

    ! Solve the matrix

!dir$ concurrent
!cdir nodep
    do fc = 1,numf
       ci = filter(fc)
       bet(ci) = b(ci,jtop(ci))
    end do

    do j = lbj, ubj
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1,numf
          ci = filter(fc)
          if (j >= jtop(ci)) then
             if (j == jtop(ci)) then
                u(ci,j) = r(ci,j) / bet(ci)
              else
                gam(ci,j) = c(ci,j-1) / bet(ci)
                bet(ci) = b(ci,j) - a(ci,j) * gam(ci,j)
                u(ci,j) = (r(ci,j) - a(ci,j)*u(ci,j-1)) / bet(ci)

             end if
          end if
       end do
    end do

!Cray X1 unroll directive used here as work-around for compiler issue 2003/10/20
!dir$ unroll 0
    do j = ubj-1,lbj,-1
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1,numf
          ci = filter(fc)
          if (j >= jtop(ci)) then
             u(ci,j) = u(ci,j) - gam(ci,j+1) * u(ci,j+1)

          end if
       end do
    end do

  end subroutine Tridiagonal

end module TridiagonalMod
module globals

!----------------------------------------------------------------------- 
!BOP
!
! !MODULE: globals
!
! !DESCRIPTION: 
! Module of global time-related control variables
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
!
! !PUBLIC TYPES:
  implicit none
!  save
  integer  :: nstep      !time step number
  real(r8):: dtime      !land model time step (sec)
!ylu add   dt may be same as dtime, dtime=get_step_size(), dt=real(get_step_size(),r8)
  real(r8):: dt         !radiation time step (sec)
  integer :: iyear0
  integer :: day_per_year  ! Get the number of days per year for currrent year

!end add
  logical :: is_perpetual  = .false.    ! true when using perpetual calendar

  integer :: year
  integer :: month      !current month (1 -> 12)
  integer :: day        !current day (1 -> 31)
  integer :: secs       ! seconds into current date
  real(r8):: calday     !calendar day

  integer :: yrp1
  integer :: monp1      !current month (1 -> 12)
  integer :: dayp1      !current day (1 -> 31)
  integer :: secp1
  real(r8):: caldayp1   !calendar day for next time step

  integer :: nbdate
!                              
!EOP
!----------------------------------------------------------------------- 
 contains
   subroutine globals_mod
   end subroutine globals_mod
end module globals
module nanMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: nanMod
!
! !DESCRIPTION:
! Set parameters for the floating point flags "inf" Infinity
! and "nan" not-a-number. As well as "bigint" the point
! at which integers start to overflow. These values are used
! to initialize arrays with as a way to detect if arrays
! are being used before being set.
! Note that bigint is the largest possible 32-bit integer.
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
!
! !PUBLIC TYPES:
  implicit none
  save
#ifdef __PGI
! quiet nan for portland group compilers
  real(r8), parameter :: inf = O'0777600000000000000000'
  real(r8), parameter :: nan = O'0777700000000000000000'
  integer,  parameter :: bigint = O'17777777777'
#elif __GNUC__
  real(r8), parameter :: inf = 1.e19
  real(r8), parameter :: nan = 1.e21
  integer,  parameter :: bigint = O'17777777777'
#else
! signaling nan otherwise
  real(r8), parameter :: inf = O'0777600000000000000000'
  real(r8), parameter :: nan = O'0777610000000000000000'
  integer,  parameter :: bigint = O'17777777777'
#endif
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein based on cam module created by
! CCM core group
!
!EOP
!-----------------------------------------------------------------------
 contains
   subroutine nanMod_mod
   end subroutine nanMod_mod
end module nanMod
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: mkrank
!
! !INTERFACE:
subroutine mkrank (n, a, miss, iv, num)
!
! !DESCRIPTION:
! Return indices of largest [num] values in array [a]
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
!
! !ARGUMENTS:
  implicit none
  integer , intent(in) :: n        !array length
  real(r8), intent(in) :: a(0:n)   !array to be ranked
  integer , intent(in) :: miss     !missing data value
  integer , intent(in) :: num      !number of largest values requested
  integer , intent(out):: iv(num)  !index to [num] largest values in array [a]
!
! !CALLED FROM:
! subroutine mkpft
! subroutine mksoicol
! subroutine mksoitex
!
! !REVISION HISTORY:
! Author: Gordon Bonan
!
!EOP
!
! !LOCAL VARIABLES:
  real(r8) a_max       !maximum value in array
  integer i            !array index
  real(r8) delmax      !tolerance for finding if larger value
  integer m            !do loop index
  integer k            !do loop index
  logical exclude      !true if data value has already been chosen
!-----------------------------------------------------------------------

  delmax = 1.e-06

  ! Find index of largest non-zero number

  iv(1) = miss
  a_max = -9999.

  do i = 0, n
     if (a(i)>0. .and. (a(i)-a_max)>delmax) then
        a_max = a(i)
        iv(1)  = i
     end if
  end do

  ! iv(1) = miss indicates no values > 0. this is an error

  if (iv(1) == miss) then
     write (6,*) 'MKRANK error: iv(1) = missing'
     call endrun
  end if

  ! Find indices of the next [num]-1 largest non-zero number.
  ! iv(m) = miss if there are no more values > 0

  do m = 2, num
     iv(m) = miss
     a_max = -9999.
     do i = 0, n

        ! exclude if data value has already been chosen

        exclude = .false.
        do k = 1, m-1
           if (i == iv(k)) exclude = .true.
        end do

        ! if not already chosen, see if it is the largest of
        ! the remaining values

        if (.not. exclude) then
           if (a(i)>0. .and. (a(i)-a_max)>delmax) then
              a_max = a(i)
              iv(m)  = i
           end if
        end if
     end do
  end do

  return
end subroutine mkrank
module clm_varpar

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: clm_varpar
!
! !DESCRIPTION:
! Module containing CLM parameters
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
!
! !PUBLIC TYPES:
  implicit none
  save
!
! Define land surface 2-d grid. This sets the model resolution according
! to cpp directives LSMLON and LSMLAT in preproc.h.
!
!  integer, parameter :: lsmlon = LSMLON     ! maximum number of longitude points on lsm grid
!  integer, parameter :: lsmlat = LSMLAT     ! number of latitude points on lsm grid
! tcx make it dynamic, read from surface dataset

  integer, parameter :: lsmlon   = 1           ! maximum number of longitude points on lsm grid
  integer, parameter :: lsmlat   = 1        ! number of latitude points on lsm grid

! Define number of levels

  integer, parameter :: nlevsoi     =  10   ! number of soil layers
  integer, parameter :: nlevlak     =  10   ! number of lake layers
  integer, parameter :: nlevsno     =   5   ! maximum number of snow layers
!ylu add
  integer, parameter :: nlevgrnd    =  10 ! 10 ! 10 ! 10 ! 10 ! 10 ! 10 ! 10 ! 10 ! 10 ! 15     ! number of ground layers (includes lower layers that are hydrologically inactive)
  integer, parameter :: nlevurb     = nlevgrnd! number of urban layers (must equal nlevgrnd right now)
!ylu end
! Define miscellaneous parameters

  integer, parameter :: numwat      =   5   ! number of water types (soil, ice, 2 lakes, wetland)
 ! integer, parameter :: npftpar     =  32   ! number of pft parameters (in LPJ - DGVM only)
  integer, parameter :: numrad      =   2   ! number of solar radiation bands: vis, nir
!ylu add 
  integer, parameter :: numsolar    =   2   ! number of solar type bands: direct, diffuse
!ylu end
  integer, parameter :: ndst        =   4   ! number of dust size classes (BGC only)
  integer, parameter :: dst_src_nbr =   3   ! number of size distns in src soil (BGC only)
  integer, parameter :: sz_nbr      = 200   ! number of sub-grid bins in large bin of dust size distribution (BGC only)
  integer, parameter :: nvoc        =   5   ! number of voc categories (BGC only)
!Not found in CLM3.5, putting back in from CLM3
!ylu remove
  integer, parameter :: numcol      =   8   !number of soil color types

! Define parameters for RTM river routing model

  integer, parameter :: rtmlon = 720  !number of rtm longitudes
  integer, parameter :: rtmlat = 360  !number of rtm latitudes

! Define indices used in surface file read
! maxpatch_pft  = max number of vegetated pfts in naturally vegetated landunit
! maxpatch_crop = max number of crop pfts in crop landunit
!ylu add 10/15/10
#if (defined CROP)
  integer, parameter :: numpft         = 20  ! number of plant types
  integer, parameter :: numcft         =  6     ! actual # of crops
  integer, parameter :: numveg         = 16     ! number of veg types (without specific crop)
#else
  integer, parameter :: numpft         = 16     ! actual # of pfts (without bare)
  integer, parameter :: numcft         =  2     ! actual # of crops
  integer, parameter :: numveg         = numpft ! number of veg types (without specific crop)
#endif
  integer, parameter :: maxpatch_urb   = 1  !  5  ! the current coupling not include urban. 
#ifdef CROP
  integer, parameter :: maxpatch_cft   = 4    !YL changed from 2 to 4
#else
  integer, parameter :: maxpatch_cft   = 2
#endif

  integer, parameter :: maxpatch_pft   = 4
  integer, parameter :: npatch_urban   = maxpatch_pft + 1
  integer, parameter :: npatch_lake    = npatch_urban + maxpatch_urb
  integer, parameter :: npatch_wet     = npatch_lake  + 1
  integer, parameter :: npatch_glacier = npatch_wet   + 1
  integer, parameter :: npatch_crop    = npatch_glacier + maxpatch_cft
  integer, parameter :: maxpatch       = npatch_crop

!ylu add
#if (defined CROP)
  integer, parameter :: max_pft_per_gcell = numpft+1 + 3 + maxpatch_urb
#else
  integer, parameter :: max_pft_per_gcell = numpft+1 + 3 + maxpatch_urb + numcft
#endif
 integer, parameter :: max_pft_per_lu    = max(numpft+1, numcft, maxpatch_urb)
  integer, parameter :: max_pft_per_col   = max(numpft+1, numcft, maxpatch_urb)


!Are these constants used?  I don't see max_col_per_lunit referenced anywhere.
!ylu remove
!  integer, parameter :: max_pft_per_gcell = numpft+1 + 4 + maxpatch_cft
!  integer, parameter :: max_pft_per_lu    = max(numpft+1, maxpatch_cft)
!  integer, parameter :: max_pft_per_col   = numpft+1

!  integer, parameter :: max_pft_per_col     = maxpatch_pft
!#if (defined NOCOMPETE)
!  integer, parameter :: max_col_per_lunit   = maxpatch_pft
!#else
!  integer, parameter :: max_col_per_lunit   = 1
!#endif
!Shouldn't this be 1?
!  integer, parameter :: max_lunit_per_gcell = 5            !(soil,urban,lake,wetland,glacier)

contains
	subroutine clm_varpar_mod
	end subroutine clm_varpar_mod
!------------------------------------------------------------------------------
end module clm_varpar
module clm_varcon
!----------------------------------------------------------------------- 
!BOP
!
! !MODULE: clm_varcon
!
! !DESCRIPTION: 
! Module containing various model constants 
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
  use clm_varpar, only : numcol,numrad,nlevlak,&
                         maxpatch_pft,numpft,nlevgrnd
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!----------------------------------------------------------------------- 
  
  integer, private :: i  ! loop index
  !------------------------------------------------------------------
  ! Initialize physical constants
  !------------------------------------------------------------------
  real(r8), parameter :: cday   = 86400.0_r8   !sec in calendar day ~ sec

  integer,  parameter :: idx_Mie_snw_mx = 1471           ! number of effective radius indices used in Mie lookup table [idx]
  integer,  parameter :: idx_T_max      = 11             ! maxiumum temperature index used in aging lookup table [idx]
  integer,  parameter :: idx_Tgrd_max   = 31             ! maxiumum temperature gradient index used in aging lookup table [idx]
  integer,  parameter :: idx_rhos_max   = 8              ! maxiumum snow density index used in aging lookup table [idx]
  integer,  parameter :: numrad_snw  =   5               ! number of spectral bands used in snow model [nbr]


  real(r8), parameter :: pie    = 3.141592653589793_r8 ! pi
  real(r8), parameter :: rpi    = 3.141592653589793_r8 ! pi
  real(r8), parameter :: grav   = 9.80616_r8   !gravity constant [m/s2]
  real(r8), parameter :: sb     = 5.67e-8_r8   !stefan-boltzmann constant  [W/m2/K4]
  real(r8), parameter :: vkc    = 0.4_r8       !von Karman constant [-]
  real(r8), parameter :: rgas   = 8314.468_r8  !Universal gas constant ~ J/K/kmole
  real(r8), parameter :: rwat   = 461.5046_r8  !gas constant for water vapor [J/(kg K)]
  real(r8), parameter :: rair   = 287.0423_r8  !gas constant for dry air [J/kg/K]
  real(r8), parameter :: roverg = 47062.73_r8  !Rw/g constant = (8.3144/0.018)/(9.80616)*1000. mm/K
  real(r8), parameter :: cpliq  = 4.188e3_r8   !Specific heat of water [J/kg-K]
  real(r8), parameter :: cpice  = 2.11727e3_r8 !Specific heat of ice [J/kg-K]
  real(r8), parameter :: cpair  = 1.00464e3_r8 !specific heat of dry air [J/kg/K]
  real(r8), parameter :: hvap   = 2.501e6_r8   !Latent heat of evap for water [J/kg]
  real(r8), parameter :: hfus   = 3.337e5_r8   !Latent heat of fusion for ice [J/kg]
  real(r8), parameter :: hsub   = 2.501e6_r8+3.337e5_r8 !Latent heat of sublimation    [J/kg]
  real(r8), parameter :: denh2o = 1.000e3_r8   !density of liquid water [kg/m3]
  real(r8), parameter :: denice = 0.917e3_r8   !density of ice [kg/m3]
  real(r8), parameter :: tkair  = 0.023_r8        !thermal conductivity of air   [W/m/k]
  real(r8), parameter :: tkice  = 2.290_r8        !thermal conductivity of ice   [W/m/k]
  real(r8), parameter :: tkwat  = 0.6_r8          !thermal conductivity of water [W/m/k]
  real(r8), parameter :: tfrz   = 273.16_r8    !freezing temperature [K]
  real(r8), parameter :: tcrit  = 2.5_r8          !critical temperature to determine rain or snow
  real(r8), parameter :: po2    = 0.209_r8        !constant atmospheric partial pressure  O2 (mol/mol)
  real(r8), parameter :: pco2   = 355.e-06     !constant atmospheric partial pressure CO2 (mol/mol)
  real(r8), parameter :: pstd   = 101325.0_r8  !standard pressure ~ pascals

  real(r8), parameter :: bdsno = 250.            !bulk density snow (kg/m**3)

  real(r8), parameter :: re = 6.37122e6_r8*0.001 !radius of earth (km)
!CLM4 --ylu
  real(r8), public, parameter ::  secspday= 86400.0_r8  ! Seconds per day
  real(r8), public, parameter ::  spval = 1.e36_r8  ! special value for real data
  integer , public, parameter :: ispval = -9999     ! special value for int data
  real(r8) :: alpha_aero = 1.0_r8   !constant for aerodynamic parameter weighting
  real(r8) :: tlsai_crit = 2.0_r8   !critical value of elai+esai for which aerodynamic parameters are maximum
  real(r8) :: watmin = 0.01_r8      !minimum soil moisture (mm)


!!
  ! These are tunable constants from clm2_3

  real(r8), parameter :: zlnd = 0.01      !Roughness length for soil [m]
  real(r8), parameter :: zsno = 0.0024    !Roughness length for snow [m]
  real(r8), parameter :: csoilc = 0.004   !Drag coefficient for soil under canopy [-]
  real(r8), parameter :: capr   = 0.34    !Tuning factor to turn first layer T into surface T  
  real(r8), parameter :: cnfac  = 0.5     !Crank Nicholson factor between 0 and 1
  real(r8), parameter :: ssi    = 0.033   !Irreducible water saturation of snow
  real(r8), parameter :: wimp   = 0.05    !Water impremeable if porosity less than wimp
  real(r8), parameter :: pondmx = 10.0    !Ponding depth (mm)
!new from CLM4 ylu add
  real(r8) :: pondmx_urban = 1.0_r8  !Ponding depth for urban roof and impervious road (mm)
  real(r8) :: o2_molar_const = 0.209_r8   !constant atmospheric O2 molar ratio (mol/mol)
  real(r8), parameter :: maxwattabfract = 1.0 !Max water table fraction for landunit that is not wet or ice
!!!!!
!new from CLM4 add by ylu 
#if (defined C13)
  ! 4/14/05: PET
  ! Adding isotope code
  real(r8), parameter :: preind_atm_del13c = -6.0   ! preindustrial value for atmospheric del13C
  real(r8), parameter :: preind_atm_ratio = SHR_CONST_PDB + (preind_atm_del13c * SHR_CONST_PDB)/1000.0  ! 13C/12C
  real(r8) :: c13ratio = preind_atm_ratio/(1.0+preind_atm_ratio) ! 13C/(12+13)C preind atmosphere
#endif
  real(r8), parameter :: ht_efficiency_factor = 0.75_r8 !efficiency factor for urban heating (-)
  real(r8), parameter :: ac_efficiency_factor = 0.25_r8 !efficiency factor for urban air conditioning (-)
  real(r8) :: ht_wasteheat_factor = 1.0_r8/ht_efficiency_factor  !wasteheat factor for urban heating (-)
  real(r8) :: ac_wasteheat_factor = 1.0_r8/ac_efficiency_factor  !wasteheat factor for urban air conditioning (-)
  real(r8) :: wasteheat_limit = 100._r8  !limit on wasteheat (W/m2)


  !------------------------------------------------------------------
  ! Initialize water type constants
  !------------------------------------------------------------------

  ! "water" types 
  !   1     soil
  !   2     land ice (glacier)
  !   3     deep lake
  !   4     shallow lake
  !   5     wetland: swamp, marsh, etc

  integer,parameter :: istsoil = 1  !soil         "water" type
  integer,parameter :: istice  = 2  !land ice     "water" type
  integer,parameter :: istdlak = 3  !deep lake    "water" type
  integer,parameter :: istslak = 4  !shallow lake "water" type
  integer,parameter :: istwet  = 5  !wetland      "water" type
  integer,parameter :: isturb  = 6  !urban        landunit type
!new from CLM4 add by ylu 
#ifdef CROP
  integer,parameter :: istcrop = 7  !crop         landunit type
#endif
  integer,parameter :: icol_roof        = 61
  integer,parameter :: icol_sunwall     = 62
  integer,parameter :: icol_shadewall   = 63
  integer,parameter :: icol_road_imperv = 64
  integer,parameter :: icol_road_perv   = 65

!Yaqiong Lu mv fndepdyn and fpftdyn from clm_varctl to clm_varcon
  logical,            public :: set_caerdep_from_file = .true.  ! if reading in carbon aerosol deposition from file
  logical,            public :: set_dustdep_from_file = .true.  ! if reading in dust aerosol deposition from file
! Landunit logic
!
  logical, public :: create_crop_landunit = .false.     ! true => separate crop landunit is not created by default
  logical, public :: allocate_all_vegpfts = .false.     ! true => allocate memory for all possible vegetated pfts on
                                                        ! vegetated landunit if at least one pft has nonzero weight

  character(len=256), public :: faerdep      = ' '               ! aerosol depos
  character(len=256), public :: fndepdyn   = ' '        ! dynamic nitrogen deposition data file name
  character(len=256), public :: fpftdyn    = ' '        ! dynamic landuse dataset

  ! snow and aerosol Mie parameters:
  ! (arrays declared here, but are set in iniTimeConst)
  ! (idx_Mie_snw_mx is number of snow radii with defined parameters (i.e. from 30um to 1500um))

  ! direct-beam weighted ice optical properties
  real(r8) :: ss_alb_snw_drc(idx_Mie_snw_mx,numrad_snw)
  real(r8) :: asm_prm_snw_drc(idx_Mie_snw_mx,numrad_snw)
  real(r8) :: ext_cff_mss_snw_drc(idx_Mie_snw_mx,numrad_snw)

  ! diffuse radiation weighted ice optical properties
  real(r8) :: ss_alb_snw_dfs(idx_Mie_snw_mx,numrad_snw)
  real(r8) :: asm_prm_snw_dfs(idx_Mie_snw_mx,numrad_snw)
  real(r8) :: ext_cff_mss_snw_dfs(idx_Mie_snw_mx,numrad_snw)


  ! hydrophiliic BC
  real(r8) :: ss_alb_bc1(1,numrad_snw)
  real(r8) :: asm_prm_bc1(1,numrad_snw)
  real(r8) :: ext_cff_mss_bc1(1,numrad_snw)

  ! hydrophobic BC
  real(r8) :: ss_alb_bc2(1,numrad_snw)
  real(r8) :: asm_prm_bc2(1,numrad_snw)
  real(r8) :: ext_cff_mss_bc2(1,numrad_snw)

  ! hydrophobic OC
  real(r8) :: ss_alb_oc1(1,numrad_snw)
  real(r8) :: asm_prm_oc1(1,numrad_snw)
  real(r8) :: ext_cff_mss_oc1(1,numrad_snw)

  ! hydrophilic OC
  real(r8) :: ss_alb_oc2(1,numrad_snw)
  real(r8) :: asm_prm_oc2(1,numrad_snw)
  real(r8) :: ext_cff_mss_oc2(1,numrad_snw)

  ! dust species 1:
  real(r8) :: ss_alb_dst1(1,numrad_snw)
  real(r8) :: asm_prm_dst1(1,numrad_snw)
  real(r8) :: ext_cff_mss_dst1(1,numrad_snw)

  ! dust species 2:
  real(r8) :: ss_alb_dst2(1,numrad_snw)
  real(r8) :: asm_prm_dst2(1,numrad_snw)
  real(r8) :: ext_cff_mss_dst2(1,numrad_snw)

  ! dust species 3:
  real(r8) :: ss_alb_dst3(1,numrad_snw)
  real(r8) :: asm_prm_dst3(1,numrad_snw)
  real(r8) :: ext_cff_mss_dst3(1,numrad_snw)

  ! dust species 4:
  real(r8) :: ss_alb_dst4(1,numrad_snw)
  real(r8) :: asm_prm_dst4(1,numrad_snw)
  real(r8) :: ext_cff_mss_dst4(1,numrad_snw)

 data(ss_alb_bc1(1,i),i=1,5) / 0.515945305512804, 0.434313626536424, 0.346103765992635,& 
    0.275522926330555, 0.138576096442815/

 data(asm_prm_bc1(1,i),i=1,5) / 0.521517715996158, 0.34457189840306, 0.244048159248401,& 
    0.188518513380877, 0.103316928297739/

 data(ext_cff_mss_bc1(1,i),i=1,5) /25368.6111954733, 12520.3846877849, 7738.643174918, &
    5744.35461327268, 3526.76546641382/

 data(ss_alb_bc2(1,i),i=1,5) /0.287685315976181, 0.186577277125224, 0.123152237089201, &
    0.0883462885905543, 0.0403421562269378/

 data(asm_prm_bc2(1,i),i=1,5) /0.350231881885906, 0.211924244128064, 0.146188682542913, &
    0.112009439045293, 0.060565694843084/

 data(ext_cff_mss_bc2(1,i),i=1,5) / 11398.4540724821, 5922.76076637376, 4039.88947595266,& 
    3261.62137894056, 2223.60028513459/

 data(ss_alb_oc1(1,i),i=1,5) / 0.996738033108225, 0.993951726870337, 0.98995641641622, &
    0.986792757460599, 0.950852907010411/

 data(asm_prm_oc1(1,i),i=1,5) / 0.771317243327679, 0.745701825432596, 0.721705644101165,& 
    0.702407207901621, 0.643447858916726/

 data(ext_cff_mss_oc1(1,i),i=1,5) / 37773.5353898986, 22112.4459872647, 14719.3405499929,& 
    10940.4200945733, 5441.11949854352/

 data(ss_alb_oc2(1,i),i=1,5) / 0.963132440682188, 0.920560323320592, 0.860191636407288, &
    0.813824138511211, 0.744011091642019/

 data(asm_prm_oc2(1,i),i=1,5) / 0.618810265705101, 0.57310868510342, 0.537906606684992, &
    0.511257182926184, 0.440320412154112/

 data(ext_cff_mss_oc2(1,i),i=1,5) /3288.85206279517, 1485.50576885264, 871.90125135612, &
    606.005758817735, 247.996083891168/

 data(ss_alb_dst1(1,i),i=1,5) /0.97891105715305, 0.994175916042451, 0.993357580762207, &
    0.992545751316266, 0.953291550046772/

 data(asm_prm_dst1(1,i),i=1,5) /0.690908112844937, 0.717759065247993, 0.671511248292627,& 
    0.614225462567888, 0.436682950958558/

 data(ext_cff_mss_dst1(1,i),i=1,5) /2686.90326329624, 2419.98140297723, 1627.51690973548,& 
    1138.23252303209, 466.104227277046/

 data(ss_alb_dst2(1,i),i=1,5) / 0.943752248802793, 0.984191668599419, 0.989309063917025, &
    0.991793946836264, 0.982999590668913/

 data(asm_prm_dst2(1,i),i=1,5) /0.699478684452806, 0.651992387581091, 0.695738438913831, &
    0.724417176862696, 0.701481090364134/

 data(ext_cff_mss_dst2(1,i),i=1,5) /841.089434044834, 987.406197502421, 1183.52284776972, &
    1267.30625580205, 993.497508579304/

 data(ss_alb_dst3(1,i),i=1,5) /0.904044530646049, 0.964651629694555, 0.968275809551522, &
    0.972598419874107, 0.977612418329876/

 data(asm_prm_dst3(1,i),i=1,5) /0.785636278417498, 0.749796744517699, 0.683301177698451, &
    0.629720518882672, 0.665531587501598/

 data(ext_cff_mss_dst3(1,i),i=1,5) /387.85423560755, 419.109723948302, 399.559447343404, &
    397.191283865122, 503.14317519429/
 
 data(ss_alb_dst4(1,i),i=1,5) /0.849818195355416, 0.940460325044343, 0.948316305534169, &
    0.952841175117807, 0.955379528193802/
 
 data(asm_prm_dst4(1,i),i=1,5) /0.849818195355416, 0.940460325044343, 0.948316305534169, &
    0.952841175117807, 0.955379528193802/

 data(ext_cff_mss_dst4(1,i),i=1,5) /196.638063554016, 202.877379461792, 208.304425287341, &
    204.723737634461, 228.755667038372/


  ! best-fit parameters for snow aging defined over:
  !  11 temperatures from 225 to 273 K
  !  31 temperature gradients from 0 to 300 K/m
  !   8 snow densities from 0 to 350 kg/m3
  ! (arrays declared here, but are set in iniTimeConst)
  real(r8) :: snowage_tau(idx_T_max,idx_Tgrd_max,idx_rhos_max)
  real(r8) :: snowage_kappa(idx_T_max,idx_Tgrd_max,idx_rhos_max)
  real(r8) :: snowage_drdt0(idx_T_max,idx_Tgrd_max,idx_rhos_max)


  real, dimension(idx_Mie_snw_mx*numrad_snw) ::  &
                         xx_ext_cff_mss_snw_dfs  &
                        ,xx_ss_alb_snw_drc       &
                        ,xx_asm_prm_snw_drc      &
                        ,xx_ext_cff_mss_snw_drc  &
                        ,xx_ss_alb_snw_dfs       &
                        ,xx_asm_prm_snw_dfs      

  real, dimension(idx_rhos_max*idx_Tgrd_max*idx_T_max) ::   &
                         xx_snowage_tau                     &
                        ,xx_snowage_kappa                   &
                        ,xx_snowage_drdt0         


  real(r8) :: ndep ! Sum of NOy and NHx deposition (unit: g(N)/m2/year)
  data ndep/0.1600056/

  real(r8),dimension(1:12) :: bcphidry,bcphodry,bcphiwet,ocphidry,ocphodry,ocphiwet,dstx01wd,dstx01dd,dstx02wd,&
                          dstx02dd,dstx03wd,dstx03dd,dstx04wd,dstx04dd
  !hydrophilic BC wet deposition (unit: kg/m2/s)
  data(bcphiwet(i),i=1,12)/2.825279e-13,2.804302e-13,2.806464e-13,2.776603e-13,2.867702e-13,2.840975e-13,&
                           3.122134e-13,3.540193e-13,3.618796e-13,3.123423e-13,2.668725e-13,2.721869e-13/
  !hydrophilic BC dry deposition (unit: kg/m2/s)
  data(bcphidry(i),i=1,12)/4.379167e-14,4.140940e-14,3.956216e-14,3.461795e-14,3.561638e-14,3.812630e-14,&
                           4.509564e-14,5.387520e-14,4.985846e-14,4.057210e-14,3.778306e-14,4.178772e-14/
  !hydrophobic BC dry deposition (unit: kg/m2/s)
  data(bcphodry(i),i=1,12)/4.192595e-14,3.831034e-14,3.536048e-14,3.209042e-14,3.280311e-14,3.226350e-14,&
                           3.723765e-14,4.297412e-14,4.106369e-14,3.602615e-14,3.536953e-14,4.030912e-14/
  !hydrophilic OC wet deposition (unit: kg/m2/s)
  data(ocphiwet(i),i=1,12)/1.162276e-12,1.151254e-12,1.188579e-12,1.186147e-12,1.340542e-12,1.292835e-12,&
                           1.628738e-12,2.033289e-12,1.964814e-12,1.479005e-12,1.043205e-12,1.068595e-12/
  !hydrophilic OC dry deposition (unit: kg/m2/s)
  data(ocphidry(i),i=1,12)/2.152982e-13,1.993085e-13,1.982182e-13,1.799778e-13,2.096774e-13,2.264119e-13,&
                           3.075992e-13,3.972984e-13,3.344011e-13,2.181304e-13,1.666979e-13,1.974062e-13/
  !hydrophobic OC dry deposition (unit: kg/m2/s)
  data(ocphodry(i),i=1,12)/1.041400e-13,9.450685e-14,9.076748e-14,8.334433e-14,9.459879e-14,9.190213e-14,&
                           1.252610e-13,1.566317e-13,1.342872e-13,9.783121e-14,8.087127e-14,9.675401e-14/
  !DSTX01 wet deposition flux at bottom (unit: kg/m2/s)
  data(dstx01wd(i),i=1,12)/3.954503e-12,4.835873e-12,5.138886e-12,4.327863e-12,4.352995e-12,5.446991e-12,&
                           5.994205e-12,5.140828e-12,3.412828e-12,2.943823e-12,3.267167e-12,3.414306e-12/                           
  !DSTX01 dry deposition flux at bottom (unit: kg/m2/s)
  data(dstx01dd(i),i=1,12)/1.926454e-13,2.188806e-13,2.054299e-13,1.452168e-13,1.216905e-13,1.291714e-13,&
                           1.238305e-13,1.022406e-13,8.948773e-14,1.024716e-13,1.347662e-13,1.688275e-13/
  !DSTX02 wet deposition flux at bottom (unit: kg/m2/s)
  data(dstx02wd(i),i=1,12)/9.846976e-12,1.203580e-11,1.324912e-11,1.146517e-11,1.176165e-11,1.479383e-11,&
                           1.656127e-11,1.427957e-11,9.381504e-12,7.933820e-12,8.429268e-12,8.695841e-12/
  !DSTX02 dry deposition flux at bottom (unit: kg/m2/s)
  data(dstx02dd(i),i=1,12)/2.207384e-12,2.523390e-12,2.099760e-12,1.318037e-12,1.071989e-12,1.305896e-12,&
                           1.065086e-12,8.545297e-13,7.591564e-13,9.132561e-13,1.344110e-12,1.683045e-12/
  !DSTX03 wet deposition flux at bottom (unit: kg/m2/s)
  data(dstx03wd(i),i=1,12)/5.689729e-12,7.006299e-12,8.480560e-12,8.957637e-12,1.042770e-11,1.315425e-11,&
                           1.529579e-11,1.397714e-11,9.306412e-12,7.171395e-12,6.230214e-12,5.392280e-12/
  !DSTX03 dry deposition flux at bottom (unit: kg/m2/s)
  data(dstx03dd(i),i=1,12)/1.344186e-11,1.552927e-11,1.442798e-11,9.362479e-12,8.622053e-12,1.158499e-11,&
                           1.128677e-11,8.671572e-12,6.141916e-12,6.720502e-12,8.372052e-12,1.090343e-11/
  !DSTX04 wet deposition flux at bottom (unit: kg/m2/s)
  data(dstx04wd(i),i=1,12)/5.657587e-12,7.503811e-12,1.001585e-11,1.095202e-11,1.382148e-11,1.919693e-11,&
                           2.390845e-11,2.121497e-11,1.201019e-11,7.470685e-12,5.650550e-12,4.622456e-12/
  !DSTX04 dry deposition flux at bottom (unit: kg/m2/s)
  data(dstx04dd(i),i=1,12)/7.075009e-11,8.168510e-11,8.081875e-11,6.024911e-11,6.014012e-11,7.693025e-11,&
                           7.988822e-11,6.632887e-11,4.771782e-11,4.599348e-11,4.981839e-11,5.885732e-11/ 


  real(r8) :: organic(1:nlevgrnd)!organic matter density at soil levels 
                                 !(unit: kg/m3 (assumed carbon content 0.58 gC per gOM)
  data(organic(i),i=1,nlevgrnd)/0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0/
! The following data are global average for each soil layer
!  data(organic(i),i=1,nlevgrnd)/15.36,15.12,13.22,10.80,8.31,6.09,4.37,3.12,0.00,0.00/

 real(r8) :: fmax ! maximum fractional saturated area!
                  ! this parameter is for soil hydrology, and it is used in SoilHydrologyMod.F
                  ! this is an ajustable parameter for a specific region -- Jiming Jin
 data fmax/0.366/

 real(r8) :: efisop(1:6)! emission factors for isoprene (ug/m2/h1)
 data(efisop(i), i=1,6)/3025.2,& ! broadleaf trees
                        554.6 ,& ! fineleaf evergreen    
                        131.0 ,& ! fineleaf deciduous
                        2629.5,& ! shrubs
                        164.3 ,& ! grass
                        14.1/    ! crops
  !------------------------------------------------------------------
  ! Initialize miscellaneous radiation constants
  !------------------------------------------------------------------


  integer,parameter :: num_landcover_types  = 24  !24 (USGS)


  ! saturated soil albedos for 8 color classes: 1=vis, 2=nir

  real(r8) :: albsat(numcol,numrad) !wet soil albedo by color class and waveband
  data(albsat(i,1),i=1,8)/0.12,0.11,0.10,0.09,0.08,0.07,0.06,0.05/
  data(albsat(i,2),i=1,8)/0.24,0.22,0.20,0.18,0.16,0.14,0.12,0.10/

  ! dry soil albedos for 8 color classes: 1=vis, 2=nir 

  real(r8) :: albdry(numcol,numrad) !dry soil albedo by color class and waveband
  data(albdry(i,1),i=1,8)/0.24,0.22,0.20,0.18,0.16,0.14,0.12,0.10/
  data(albdry(i,2),i=1,8)/0.48,0.44,0.40,0.36,0.32,0.28,0.24,0.20/

  ! albedo land ice: 1=vis, 2=nir

  real(r8) :: albice(numrad)        !albedo land ice by waveband
  data (albice(i),i=1,numrad) /0.80, 0.55/

  ! albedo frozen lakes: 1=vis, 2=nir 

  real(r8) :: alblak(numrad)        !albedo frozen lakes by waveband
  data (alblak(i),i=1,numrad) /0.60, 0.40/

  ! omega,betad,betai for snow 

  real(r8),parameter :: betads  = 0.5       !two-stream parameter betad for snow
  real(r8),parameter :: betais  = 0.5       !two-stream parameter betai for snow
  real(r8) :: omegas(numrad)      !two-stream parameter omega for snow by band
  data (omegas(i),i=1,numrad) /0.8, 0.4/

  !------------------------------------------------------------------
  ! Soil and Lake depths are constants for now
  ! The values for the following arrays are set in routine iniTimeConst
  !------------------------------------------------------------------

  real(r8) :: zlak(1:nlevlak)     !lake z  (layers) 
  real(r8) :: dzlak(1:nlevlak)    !lake dz (thickness)
  real(r8) :: zsoi(1:nlevgrnd)     !soil z  (layers)
  real(r8) :: dzsoi(1:nlevgrnd)    !soil dz (thickness)
  real(r8) :: zisoi(0:nlevgrnd)    !soil zi (interfaces)  

    real(r8) :: sand(19)                           ! percent sand
    real(r8) :: clay(19)                           ! percent clay
    integer  :: soic(19)
    integer  :: plant(24,maxpatch_pft)
    real(r8) :: cover(24,maxpatch_pft)

    data(sand(i), i=1,19)/92.,80.,66.,20.,5.,43.,60.,&
      10.,32.,51., 6.,22.,39.7,0.,100.,54.,17.,100.,92./

    data(clay(i), i=1,19)/ 3., 5.,10.,15.,5.,18.,27.,&
      33.,33.,41.,47.,58.,14.7,0., 0., 8.5,54.,  0., 3./

    data(soic(i), i=1,19)/1,2,2,3,3,4,5,5,6,7,7,8,8,0,&
                          1,1,4,7,1/

!  soil type from MM5
! (1)  sand
! (2)  loamy-sand
! (3)  sandy-loam
! (4)  silt-loam
! (5)  silt
! (6)  loam
! (7)  sandy-clay-loam
! (8)  silty-clay-loam
! (9)  clay-loam
! (10) sandy-clay
! (11) silty-clay
! (12) clay
! (13) organic-material,
! (14) water
! (15) bedrock
! (16) other(land-ice)
! (17) playa
! (18) lava
! (19) white-sand
!----------------------------------------------------------------------------
  data (plant(i,1),i=1,24) / 0,  15,  15,  15,  15,  15, &
                            14,   9,   9,  14,   7,   3, &
                             4,   1,   1,   0,   0,   4, &
                            11,  11,   2,  11,  11,   0/
  data (cover(i,1),i=1,24) /100.,  85.,  85.,  85.,  50.,  40., &
                             60.,  80.,  50.,  70.,  75.,  50., &
                             95.,  75.,  37., 100., 100.,  80., &
                             10.,  30.,  13.,  20.,  10., 100./

  data (plant(i,2),i=1,24) / 0,   0,   0,   0,  14,   3, &
                            13,   0,  14,   6,   0,   0, &
                             0,   0,   7,   0,   0,   0, &
                             0,  12,   3,  12,  12,   0/
  data (cover(i,2),i=1,24) /  0.,  15.,  15.,  15.,  35.,  30., &
                             20.,  20.,  30.,  30.,  25.,  50., &
                              5.,  25.,  37.,   0.,   0.,  20., &
                             90.,  30.,  13.,  20.,  10.,   0./

  data (plant(i,3),i=1,24) / 0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0, &
                             0,   0,  10,   0,   0,   0/

  data (cover(i,3),i=1,24) /  0.,   0.,   0.,   0.,  15.,  30., &
                             20.,   0.,  20.,   0.,   0.,   0., &
                              0.,   0.,  26.,   0.,   0.,   0., &
                              0.,  40.,  24.,  60.,  80.,   0./

  data (plant(i,4),i=1,24) / 0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0/

  data (cover(i,4),i=1,24) / 0.,   0.,   0.,   0.,   0.,   0., &
                             0.,   0.,   0.,   0.,   0.,   0., &
                             0.,   0.,   0.,   0.,   0.,   0., &
                             0.,   0.,  50.,   0.,   0.,   0./

!-----------------------------------------------------------------------

!USGS vegetation 24 categories
!
!Urban and Built-Up Land            1
!Dryland Cropland and Pasture       2
!Irrigated Cropland and Pasture     3
!Mixed Dryland/Irrg. C.P.           4
!Cropland/Grassland Mosaic          5
!Cropland/Woodland Mosaic           6
!Grassland                          7
!Shrubland                          8
!Mixed Shrubland/Grassland          9
!Savanna                           10
!Deciduous Broadleaf Forest        11
!Deciduous Needleleaf Forest       12
!Evergreen Broadleaf Forest        13
!Evergreen Needleleaf Forest       14
!Mixed Forest                      15
!Water Bodies                      16
!Herbaceous Wetland                17
!Wooded Wetland                    18
!Barren or Sparsely Vegetated      19
!Herbaceous Tundra                 20
!Wooded Tundra                     21
!Mixed Tundra                      22
!Bare Ground Tundra                23
!Snow or Ice                       24
!-----------------------------------------------------------------------
    real(r8):: lai(numpft,12),sai(numpft,12)

    real(r8):: hvt(16),hvb(16)  

    data (hvt(i),i=1,16) /17.0,17.0,14.0,35.0,35.0,18.0,20.0,20.0,&
      0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5/
    data (hvb(i),i=1,16) /8.50, 8.50,7.00,1.00,1.00,10.00,11.50,11.50,&
      0.10,0.10,0.10,0.01,0.10,0.01,0.01,0.01/


    data (lai(1,i),i=1,12) &
        /4.1,4.2,4.6,4.8,4.9,5.0,4.8,4.7,4.6,4.2,4.0,4.0/
    data (lai(2,i),i=1,12) &
        /4.1,4.2,4.6,4.8,4.9,5.0,4.8,4.7,4.6,4.2,4.0,4.0/
    data (lai(3,i),i=1,12) &
        /0.0,0.0,0.0,0.6,1.2,2.0,2.6,1.7,1.0,0.5,0.2,0.0/
    data (lai(4,i),i=1,12) &
        /4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5/
    data (lai(5,i),i=1,12) &
        /4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5/
    data (lai(6,i),i=1,12) &
        /0.8,0.7,0.4,0.5,0.5,0.7,1.7,3.0,2.5,1.6,1.0,1.0/
    data (lai(7,i),i=1,12) &
        /0.0,0.0,0.3,1.2,3.0,4.7,4.5,3.4,1.2,0.3,0.0,0.0/
    data (lai(8,i),i=1,12) &
        /0.0,0.0,0.3,1.2,3.0,4.7,4.5,3.4,1.2,0.3,0.0,0.0/
    data (lai(9,i),i=1,12) &
        /1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0/
    data (lai(10,i),i=1,12) &
        /0.9,0.8,0.2,0.2,0.0,0.0,0.0,0.2,0.4,0.5,0.6,0.8/
    data (lai(11,i),i=1,12) &
        /0.0,0.0,0.0,0.0,0.0,0.2,1.4,1.2,0.0,0.0,0.0,0.0/
    data (lai(12,i),i=1,12) &
        /0.4,0.5,0.6,0.7,1.2,3.0,3.5,1.5,0.7,0.6,0.5,0.4/
    data (lai(13,i),i=1,12) &
        /0.0,0.0,0.0,0.0,0.0,0.2,1.4,1.2,0.0,0.0,0.0,0.0/
    data (lai(14,i),i=1,12) &
        /0.4,0.5,0.6,0.7,1.2,3.0,3.5,1.5,0.7,0.6,0.5,0.4/
    data (lai(15,i),i=1,12) &
        /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
    data (lai(16,i),i=1,12) &
        /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
!-----------------------------------------------------------------------
    data (sai(1,i),i=1,12) &
       /0.4,0.5,0.4,0.3,0.4,0.5,0.5,0.6,0.6,0.7,0.6,0.5/
    data (sai(2,i),i=1,12) &
       /0.4,0.5,0.4,0.3,0.4,0.5,0.5,0.6,0.6,0.7,0.6,0.5/
    data (sai(3,i),i=1,12) &
       /0.3,0.3,0.3,0.4,0.4,0.4,1.7,1.2,1.0,0.8,0.6,0.5/
    data (sai(4,i),i=1,12) &
       /0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5/
    data (sai(5,i),i=1,12) &
       /0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5/
    data (sai(6,i),i=1,12) &
       /0.4,0.3,0.5,0.3,0.3,0.3,0.3,0.7,0.7,1.1,0.9,0.2/
    data (sai(7,i),i=1,12) &
       /0.4,0.4,0.4,0.4,0.5,0.4,0.9,1.4,2.6,1.4,0.6,0.4/
    data (sai(8,i),i=1,12) &
       /0.4,0.4,0.4,0.4,0.5,0.4,0.9,1.4,2.6,1.4,0.6,0.4/
    data (sai(9,i),i=1,12) &
       /0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3/
    data (sai(10,i),i=1,12) &
       /0.1,0.2,0.6,0.1,0.6,0.0,0.1,0.1,0.1,0.1,0.1,0.1/
    data (sai(11,i),i=1,12) &
       /0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.5,1.4,0.1,0.1,0.1/
    data (sai(12,i),i=1,12) &
       /0.3,0.3,0.3,0.3,0.3,0.4,0.8,2.3,1.1,0.4,0.4,0.4/
    data (sai(13,i),i=1,12) &
       /0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.5,1.4,0.1,0.1,0.1/
    data (sai(14,i),i=1,12) &
       /0.3,0.3,0.3,0.3,0.3,0.4,0.8,2.3,1.1,0.4,0.4,0.4/
    data (sai(15,i),i=1,12) &
       /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
    data (sai(16,i),i=1,12) &
       /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
!----------------------------------------------------------------------------
  character(len=40) pftname(0:numpft)
  real(r8) dleaf(0:numpft)       !characteristic leaf dimension (m)
  real(r8) c3psn(0:numpft)       !photosynthetic pathway: 0. = c4, 1. = c3
  real(r8) vcmx25(0:numpft)      !max rate of carboxylation at 25C (umol CO2/m**2/s)
  real(r8) mp(0:numpft)          !slope of conductance-to-photosynthesis relationship
  real(r8) qe25(0:numpft)        !quantum efficiency at 25C (umol CO2 / umol photon)
  real(r8) xl(0:numpft)          !leaf/stem orientation index
  real(r8) rhol(0:numpft,numrad) !leaf reflectance: 1=vis, 2=nir
  real(r8) rhos(0:numpft,numrad) !stem reflectance: 1=vis, 2=nir
  real(r8) taul(0:numpft,numrad) !leaf transmittance: 1=vis, 2=nir
  real(r8) taus(0:numpft,numrad) !stem transmittance: 1=vis, 2=nir
  real(r8) z0mr(0:numpft)        !ratio of momentum roughness length to canopy top height (-)
  real(r8) displar(0:numpft)     !ratio of displacement height to canopy top height (-)
  real(r8) roota_par(0:numpft)   !CLM rooting distribution parameter [1/m]
  real(r8) rootb_par(0:numpft)   !CLM rooting distribution parameter [1/m]

data (pftname(i),i=1,16)/'needleleaf_evergreen_temperate_tree',&
                           'needleleaf_evergreen_boreal_tree'   ,&
                           'needleleaf_deciduous_boreal_tree'   ,&
                           'broadleaf_evergreen_tropical_tree'  ,&
                           'broadleaf_evergreen_temperate_tree' ,&
                           'broadleaf_deciduous_tropical_tree'  ,&
                           'broadleaf_deciduous_temperate_tree' ,&
                           'broadleaf_deciduous_boreal_tree'    ,&
                           'broadleaf_evergreen_shrub'          ,&
                           'broadleaf_deciduous_temperate_shrub',&
                           'broadleaf_deciduous_boreal_shrub'   ,&
                           'c3_arctic_grass'                    ,&
                           'c3_non-arctic_grass'                ,&
                           'c4_grass'                           ,&
                           'corn'                               ,&
                           'wheat'/

  data (z0mr(i),i=1,16)/ 0.055, 0.055, 0.055, 0.075, 0.075,  &
       0.055,0.055, 0.055, 0.120, 0.120, 0.120, 0.120, 0.120,&
       0.120, 0.120, 0.120/

  data (displar(i),i=1,16)/ 0.67, 0.67, 0.67, 0.67, 0.67, &
         0.67, 0.67, 0.67, 0.68, 0.68, 0.68, 0.68, 0.68,  &
         0.68, 0.68, 0.68/

  data (dleaf(i),i=1,16)/ 0.04, 0.04, 0.04, 0.04, 0.04,&
         0.04, 0.04, 0.04, 0.04, 0.04, 0.04, 0.04, 0.04,&
         0.04, 0.04, 0.04/

  data (c3psn(i),i=1,16)/1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,&
        1.0,1.0,1.0,1.0,1.0,0.0,1.0,1.0/

  data (vcmx25(i),i=1,16)/51.0,43.0,43.0,75.0,69.0,40.0,&
       51.0,51.0,17.0,17.0,33.0,43.0,43.0,24.0,50.0,50.0/

  data (mp(i),i=1,16)/6.0,6.0,6.0,9.0,9.0,9.0,9.0,9.0,&
        9.0,9.0,9.0,9.0,9.0,5.0,9.0,9.0/

  data (qe25(i),i=1,16)/ 0.06, 0.06, 0.06, 0.06, 0.06,&
        0.06, 0.06, 0.06, 0.06, 0.06, 0.06, 0.06, 0.06,&
        0.04, 0.06, 0.06/

  data (rhol(i,1),i=1,16)/ 0.07, 0.07, 0.07, 0.10, 0.10,&
        0.10, 0.10, 0.10, 0.07, 0.10, 0.10, 0.11, 0.11,&
        0.11, 0.11, 0.11/

  data (rhol(i,2),i=1,16)/ 0.35, 0.35, 0.35, 0.45, 0.45,&
        0.45, 0.45, 0.45, 0.35, 0.45, 0.45, 0.58, 0.58, &
        0.58, 0.58, 0.58/

  data (rhos(i,1),i=1,16) /0.16, 0.16, 0.16, 0.16, 0.16,&
         0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.36, 0.36,&
         0.36, 0.36, 0.36/

  data (rhos(i,2),i=1,16)/ 0.39, 0.39, 0.39, 0.39, 0.39,&
        0.39, 0.39, 0.39, 0.39, 0.39, 0.39, 0.58, 0.58, &
        0.58, 0.58, 0.58/

  data (taul(i,1),i=1,16)/ 0.05, 0.05, 0.05, 0.05, 0.05,&
        0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.07, 0.07,&
        0.07, 0.07, 0.07/

  data (taul(i,2),i=1,16)/ 0.10, 0.10, 0.10, 0.25, 0.25,&
        0.25, 0.25, 0.25, 0.10, 0.25, 0.25, 0.25, 0.25, &
        0.25, 0.25, 0.25/

  data (taus(i,1),i=1,16)/0.001, 0.001, 0.001, 0.001,&
       0.001,0.001, 0.001, 0.001, 0.001, 0.001, 0.001,&
       0.220, 0.220, 0.220, 0.220, 0.220/

  data (taus(i,2),i=1,16)/ 0.001, 0.001, 0.001, 0.001,&
       0.001, 0.001, 0.001, 0.001, 0.001, 0.001, &
       0.001, 0.380, 0.380, 0.380, 0.380, 0.380/

  data (xl(i),i=1,16)/0.01,0.01,0.01,0.10,0.10, 0.01,&
       0.25, 0.25, 0.01, 0.25, 0.25, -0.30, -0.30,&
       -0.30, -0.30, -0.30/

  data (roota_par(i),i=1,16)/ 7.0, 7.0, 7.0, 7.0,&
      7.0, 6.0, 6.0, 6.0, 7.0, 7.0, 7.0, 11.0, &
      11.0, 11.0,  6.0,  6.0/

  data (rootb_par(i),i=1,16)/ 2.0, 2.0, 2.0, &
     1.0, 1.0, 2.0, 2.0, 2.0, 1.5, 1.5, 1.5, &
     2.0, 2.0, 2.0, 3.0, 3.0/

contains

 subroutine var_par
 end subroutine var_par

end module clm_varcon
module clm_varsur

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: clm_varsur
!
! !DESCRIPTION:
! Module containing 2-d surface boundary data information
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
  use clm_varpar, only : maxpatch,lsmlon, lsmlat, nlevsoi
!
! !PUBLIC TYPES:
  implicit none
  save
!
! land model grid
!
!ylu add
 
!   real(r8),allocatable :: pctspec(:)         ! percent of spec lunits wrt gcell
!ylu end 

  integer :: numlon(lsmlat)             !longitude points for each latitude strip
  real(r8):: latixy(1)      !latitude of grid cell (degrees)
  real(r8):: longxy(1)      !longitude of grid cell (degrees)
  real(r8):: area(1)        !grid cell area (km**2)
  real(r8):: landarea                   !total land area for all gridcells (km^2)
  real(r8):: lats(lsmlat+1)             !grid cell latitude, southern edge (degrees)
  real(r8):: lonw(lsmlon+1,lsmlat)      !grid cell longitude, western edge (degrees)
  real(r8):: lsmedge(4)                 !North,East,South,West edges of grid (deg)
  logical :: pole_points                !true => grid has pole points
  logical :: fullgrid  = .true.         !true => no grid reduction towards poles
  logical :: offline_rdgrid             !true => read offline grid rather than creating it
!
! fractional land and mask
!
!  integer  landmask(smlon,lsmlat)      !land mask: 1 = land. 0 = ocean
!  real(r8) landfrac(lsmlon,lsmlat)      !fractional land
!
! surface boundary data
!
  real(r8), allocatable :: gti(:)
  integer , allocatable :: soic2d(:)   !soil color
  real(r8) , allocatable :: efisop2d(:,:) 
  real(r8), allocatable :: sand3d(:,:) !soil texture: percent sand
  real(r8), allocatable :: clay3d(:,:) !soil texture: percent clay
  real(r8), allocatable :: organic3d(:,:) !organic matter: kg/m3
  real(r8), allocatable :: pctgla(:)   !percent of grid cell that is glacier
  real(r8), allocatable :: pctlak(:)   !percent of grid cell that is lake
  real(r8), allocatable :: pctwet(:)   !percent of grid cell that is wetland
  real(r8), allocatable :: pcturb(:)   !percent of grid cell that is urbanized
  integer , allocatable :: vegxy(:,:) ! vegetation type
  real(r8), allocatable,target :: wtxy(:,:)  ! subgrid weights

!
! !PUBLIC MEMBER FUNCTIONS:
  public :: varsur_alloc    !allocates 2d surface data needed for initialization
  public :: varsur_dealloc  !deallocates 2d surface data needed for initialization
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: varsur_alloc
!
! !INTERFACE:
  subroutine varsur_alloc

!
! !DESCRIPTION:
! Allocate dynamic memory for module variables
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!
! LOCAL VARIABLES:
    integer :: ier,begg,endg                   !error code
    
  begg=1
  endg=1

!-----------------------------------------------------------------------
    allocate (vegxy(1,maxpatch), &
              wtxy(1,maxpatch),  &
              stat=ier)
    if (ier /= 0) then
       write(6,*)'initialize allocation error' 
       call endrun()
    endif



    allocate (soic2d(begg:endg), &
              gti(begg:endg), &
              efisop2d(1:6,begg:endg),&
              sand3d(begg:endg,nlevsoi), &
              clay3d(begg:endg,nlevsoi), &
              organic3d(begg:endg,nlevsoi), &
              pctgla(begg:endg), &
              pctlak(begg:endg), &
              pctwet(begg:endg), &
              pcturb(begg:endg), stat=ier)
    if (ier /= 0) then
       write(6,*)'varsur_alloc(): allocation error'
       call endrun()
    endif

  end subroutine varsur_alloc

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: varsur_dealloc
!
! !INTERFACE:
  subroutine varsur_dealloc
!
! !DESCRIPTION:
! Deallocate dynamic memory for module variables
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!-----------------------------------------------------------------------

    deallocate (soic2d, &
                gti,    &
               efisop2d,&
                sand3d, &
                clay3d, &
               organic3d,&
                pctgla, &
                pctlak, &
                pctwet, &
                pcturb, &
                wtxy,   &
                vegxy)

  end subroutine varsur_dealloc

end module clm_varsur
!#include <misc.h>
!#include <preproc.h>

module clmtype

!----------------------------------------------------------------------- 
!BOP
!
! !MODULE: clmtype
!
! !DESCRIPTION: 
! Define derived type hierarchy. Includes declaration of
! the clm derived type and 1d mapping arrays. 
!
! -------------------------------------------------------- 
! gridcell types can have values of 
! -------------------------------------------------------- 
!   1 => default
! -------------------------------------------------------- 
! landunits types can have values of (see clm_varcon.F90)
! -------------------------------------------------------- 
!   1  => (istsoil) soil (vegetated or bare soil landunit)
!   2  => (istice)  land ice
!   3  => (istdlak) deep lake
!   4  => (istslak) shall lake (not currently implemented)
!   5  => (istwet)  wetland
!   6  => (isturb)  urban 
!   7  => (istcrop) crop (only for CROP configuration)
! -------------------------------------------------------- 
! column types can have values of
! -------------------------------------------------------- 
!   1  => (istsoil)          soil (vegetated or bare soil)
!   2  => (istice)           land ice
!   3  => (istdlak)          deep lake
!   4  => (istslak)          shallow lake 
!   5  => (istwet)           wetland
!   61 => (icol_roof)        urban roof
!   62 => (icol_sunwall)     urban sunwall
!   63 => (icol_shadewall)   urban shadewall
!   64 => (icol_road_imperv) urban impervious road
!   65 => (icol_road_perv)   urban pervious road
! -------------------------------------------------------- 
! pft types can have values of
! -------------------------------------------------------- 
!   0  => not vegetated
!   1  => needleleaf evergreen temperate tree
!   2  => needleleaf evergreen boreal tree
!   3  => needleleaf deciduous boreal tree
!   4  => broadleaf evergreen tropical tree
!   5  => broadleaf evergreen temperate tree
!   6  => broadleaf deciduous tropical tree
!   7  => broadleaf deciduous temperate tree
!   8  => broadleaf deciduous boreal tree
!   9  => broadleaf evergreen shrub
!   10 => broadleaf deciduous temperate shrub
!   11 => broadleaf deciduous boreal shrub
!   12 => c3 arctic grass
!   13 => c3 non-arctic grass
!   14 => c4 grass
!   15 => corn
!   16 => wheat
! -------------------------------------------------------- 
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
!
! !PUBLIC TYPES:
  implicit none

!                              
! !REVISION HISTORY:
! Created by Peter Thornton and Mariana Vertenstein
!
!*******************************************************************************
!----------------------------------------------------
! Begin definition of conservation check structures
!----------------------------------------------------
! energy balance structure
!----------------------------------------------------
type, public :: energy_balance_type
   real(r8), pointer :: errsoi(:)        !soil/lake energy conservation error (W/m**2)
   real(r8), pointer :: errseb(:)        !surface energy conservation error (W/m**2)
   real(r8), pointer :: errsol(:)        !solar radiation conservation error (W/m**2)
   real(r8), pointer :: errlon(:)        !longwave radiation conservation error (W/m**2)
end type energy_balance_type

!----------------------------------------------------
! water balance structure
!----------------------------------------------------
type, public :: water_balance_type
   real(r8), pointer :: begwb(:)         !water mass begining of the time step
   real(r8), pointer :: endwb(:)         !water mass end of the time step
   real(r8), pointer :: errh2o(:)        !water conservation error (mm H2O)
end type water_balance_type

!----------------------------------------------------
! carbon balance structure
!----------------------------------------------------
type, public :: carbon_balance_type
   real(r8), pointer :: begcb(:)         !carbon mass, beginning of time step (gC/m**2)
   real(r8), pointer :: endcb(:)         !carbon mass, end of time step (gC/m**2)
   real(r8), pointer :: errcb(:)         !carbon balance error for the timestep (gC/m**2)
end type carbon_balance_type

!----------------------------------------------------
! nitrogen balance structure
!----------------------------------------------------
type, public :: nitrogen_balance_type
   real(r8), pointer :: begnb(:)         !nitrogen mass, beginning of time step (gN/m**2)
   real(r8), pointer :: endnb(:)         !nitrogen mass, end of time step (gN/m**2)
   real(r8), pointer :: errnb(:)         !nitrogen balance error for the timestep (gN/m**2)
end type nitrogen_balance_type

!----------------------------------------------------
! End definition of conservation check structures
!----------------------------------------------------
!*******************************************************************************

!*******************************************************************************
!----------------------------------------------------
! Begin definition of structures defined at the pft_type level
!----------------------------------------------------
! pft physical state variables structure
!----------------------------------------------------
type, public :: pft_pstate_type
   integer , pointer :: frac_veg_nosno(:)       !fraction of vegetation not covered by snow (0 OR 1) [-] 
   integer , pointer :: frac_veg_nosno_alb(:)   !fraction of vegetation not covered by snow (0 OR 1) [-] 
   real(r8), pointer :: emv(:)                  !vegetation emissivity
   real(r8), pointer :: z0mv(:)                 !roughness length over vegetation, momentum [m]
   real(r8), pointer :: z0hv(:)                 !roughness length over vegetation, sensible heat [m]
   real(r8), pointer :: z0qv(:)                 !roughness length over vegetation, latent heat [m]
   real(r8), pointer :: rootfr(:,:)             !fraction of roots in each soil layer  (nlevgrnd)
   real(r8), pointer :: rootr(:,:)              !effective fraction of roots in each soil layer  (nlevgrnd)
   real(r8), pointer :: rresis(:,:)             !root resistance by layer (0-1)  (nlevgrnd)
   real(r8), pointer :: dewmx(:)                !Maximum allowed dew [mm]
   real(r8), pointer :: rssun(:)                !sunlit stomatal resistance (s/m)
   real(r8), pointer :: rssha(:)                !shaded stomatal resistance (s/m)
   real(r8), pointer :: laisun(:)               !sunlit projected leaf area index
   real(r8), pointer :: laisha(:)               !shaded projected leaf area index
   real(r8), pointer :: btran(:)                !transpiration wetness factor (0 to 1)
   real(r8), pointer :: fsun(:)                 !sunlit fraction of canopy
   real(r8), pointer :: tlai(:)                 !one-sided leaf area index, no burying by snow
   real(r8), pointer :: tsai(:)                 !one-sided stem area index, no burying by snow
   real(r8), pointer :: elai(:)                 !one-sided leaf area index with burying by snow
   real(r8), pointer :: esai(:)                 !one-sided stem area index with burying by snow
   real(r8), pointer :: fwet(:)                 !fraction of canopy that is wet (0 to 1)
   real(r8), pointer :: fdry(:)                 !fraction of foliage that is green and dry [-] (new)
   real(r8), pointer :: dt_veg(:)               !change in t_veg, last iteration (Kelvin)
   real(r8), pointer :: htop(:)                 !canopy top (m)
   real(r8), pointer :: hbot(:)                 !canopy bottom (m)
   real(r8), pointer :: z0m(:)                  !momentum roughness length (m)
   real(r8), pointer :: displa(:)               !displacement height (m)
   real(r8), pointer :: albd(:,:)               !surface albedo (direct)                       (numrad)
   real(r8), pointer :: albi(:,:)               !surface albedo (indirect)                      (numrad)
   real(r8), pointer :: fabd(:,:)               !flux absorbed by veg per unit direct flux     (numrad)
   real(r8), pointer :: fabi(:,:)               !flux absorbed by veg per unit diffuse flux    (numrad)
   real(r8), pointer :: ftdd(:,:)               !down direct flux below veg per unit dir flx   (numrad)
   real(r8), pointer :: ftid(:,:)               !down diffuse flux below veg per unit dir flx  (numrad)
   real(r8), pointer :: ftii(:,:)               !down diffuse flux below veg per unit dif flx  (numrad)
   real(r8), pointer :: u10(:)                  !10-m wind (m/s) (for dust model)
   real(r8), pointer :: ram1(:)                 !aerodynamical resistance (s/m)
   real(r8), pointer :: fv(:)                   !friction velocity (m/s) (for dust model)
   real(r8), pointer :: forc_hgt_u_pft(:)       !wind forcing height (10m+z0m+d) (m)
   real(r8), pointer :: forc_hgt_t_pft(:)       !temperature forcing height (10m+z0m+d) (m)
   real(r8), pointer :: forc_hgt_q_pft(:)       !specific humidity forcing height (10m+z0m+d) (m)
#if (defined CROP)
   real(r8), pointer :: hdidx(:)
   real(r8), pointer :: cumvd(:)
   real(r8), pointer :: htmx(:)      ! max hgt attained by a crop during yr
   real(r8), pointer :: vf(:)        ! vernalization factor for wheat
   real(r8), pointer :: gddmaturity(:)
   real(r8), pointer :: gdd0(:)
   real(r8), pointer :: gdd8(:)
   real(r8), pointer :: gdd10(:)
   real(r8), pointer :: gdd020(:)
   real(r8), pointer :: gdd820(:)
   real(r8), pointer :: gdd1020(:)
   real(r8), pointer :: gddplant(:)  ! accum gdd past planting date for crop
   real(r8), pointer :: gddtsoi(:)
   real(r8), pointer :: huileaf(:)
   real(r8), pointer :: huigrain(:)
   real(r8), pointer :: a10tmin(:)
   real(r8), pointer :: a5tmin(:)
   real(r8), pointer :: aleafi(:)
   real(r8), pointer :: astemi(:)
   real(r8), pointer :: aleaf(:)
   real(r8), pointer :: astem(:)
   integer , pointer :: croplive(:)
   integer , pointer :: cropplant(:) ! this and next could be 2-D to
   integer , pointer :: harvdate(:)  ! facilitate crop rotation
   integer , pointer :: idop(:)
   integer , pointer :: peaklai(:)   ! 1: max allowed lai; 0: not at max
#endif
   real(r8), pointer :: vds(:) 		        !deposition velocity term (m/s) (for dry dep SO4, NH4NO3)
   ! new variables for CN code
   real(r8), pointer :: slasun(:)     !specific leaf area for sunlit canopy, projected area basis (m^2/gC)
   real(r8), pointer :: slasha(:)     !specific leaf area for shaded canopy, projected area basis (m^2/gC)
   real(r8), pointer :: lncsun(:)     !leaf N concentration per unit projected LAI (gN leaf/m^2)
   real(r8), pointer :: lncsha(:)     !leaf N concentration per unit projected LAI (gN leaf/m^2)
   real(r8), pointer :: vcmxsun(:)    !sunlit leaf Vcmax (umolCO2/m^2/s)
   real(r8), pointer :: vcmxsha(:)    !shaded leaf Vcmax (umolCO2/m^2/s)
   real(r8), pointer :: gdir(:)       !leaf projection in solar direction (0 to 1)
   real(r8), pointer :: omega(:,:)    !fraction of intercepted radiation that is scattered (0 to 1)
   real(r8), pointer :: eff_kid(:,:)  !effective extinction coefficient for indirect from direct
   real(r8), pointer :: eff_kii(:,:)  !effective extinction coefficient for indirect from indirect
   real(r8), pointer :: sun_faid(:,:) !fraction sun canopy absorbed indirect from direct
   real(r8), pointer :: sun_faii(:,:) !fraction sun canopy absorbed indirect from indirect
   real(r8), pointer :: sha_faid(:,:) !fraction shade canopy absorbed indirect from direct
   real(r8), pointer :: sha_faii(:,:) !fraction shade canopy absorbed indirect from indirect
   ! 4/14/05: PET
   ! Adding isotope code
   real(r8), pointer :: cisun(:)       !sunlit intracellular CO2 (Pa)
   real(r8), pointer :: cisha(:)       !shaded intracellular CO2 (Pa)
#if (defined C13)
   real(r8), pointer :: alphapsnsun(:) !sunlit 13c fractionation ([])
   real(r8), pointer :: alphapsnsha(:) !shaded 13c fractionation ([])
#endif
   ! heald: added outside of CASA definition
   real(r8), pointer :: sandfrac(:)    ! sand fraction
   real(r8), pointer :: clayfrac(:)    ! clay fraction
   ! for dry deposition of chemical tracers
   real(r8), pointer :: mlaidiff(:)    ! difference between lai month one and month two
   real(r8), pointer :: rb1(:)         ! aerodynamical resistance (s/m)
   real(r8), pointer :: annlai(:,:)    ! 12 months of monthly lai from input data set  

   
#if (defined CASA)
   real(r8), pointer :: Closs(:,:)  ! C lost to atm
   real(r8), pointer :: Ctrans(:,:) ! C transfers out of pool types
   real(r8), pointer :: Resp_C(:,:) ! C respired
   real(r8), pointer :: Tpool_C(:,:)! Total C pool size
   real(r8), pointer :: eff(:,:)
   real(r8), pointer :: frac_donor(:,:)
   real(r8), pointer :: livefr(:,:) !live fraction
   real(r8), pointer :: pet(:)      !potential evaporation (mm h2o/s)
   real(r8), pointer :: co2flux(:)  ! net CO2 flux (g C/m2/sec) [+= atm]
   real(r8), pointer :: fnpp(:)     ! NPP  (g C/m2/sec)
   real(r8), pointer :: soilt(:)    !soil temp for top 30cm
   real(r8), pointer :: smoist(:)   !soil moisture for top 30cm
   real(r8), pointer :: sz(:)       !thickness of soil layers contributing to output
   real(r8), pointer :: watopt(:)   !optimal soil water content for et for top 30cm (mm3/mm3)
   real(r8), pointer :: watdry(:)   !soil water when et stops for top 30cm (mm3/mm3)
   real(r8), pointer :: soiltc(:)   !soil temp for entire column
   real(r8), pointer :: smoistc(:)  !soil moisture for entire column
   real(r8), pointer :: szc(:)      !thickness of soil layers contributing to output
   real(r8), pointer :: watoptc(:)  !optimal soil water content for et for entire column (mm3/mm3)
   real(r8), pointer :: watdryc(:)  !soil water when et stops for entire column (mm3/mm3)
   real(r8), pointer :: Wlim(:)     !Water limitation min value
   real(r8), pointer :: litterscalar(:)
   real(r8), pointer :: rootlitscalar(:)
   real(r8), pointer :: stressCD(:) ! cold and drought stress function (sec-1)
                                    ! add to "annK(m,LEAF)" and "annK(m,FROOT)"
                                    ! in casa_litterfall.F
   real(r8), pointer :: excessC(:)  ! excess Carbon (gC/m2/timestep)
   real(r8), pointer :: bgtemp(:)   ! temperature dependence
   real(r8), pointer :: bgmoist(:)  ! moisture dependence
   real(r8), pointer :: plai(:)     ! prognostic LAI (m2 leaf/m2 ground)
   real(r8), pointer :: Cflux(:)    ! Carbon flux
   real(r8), pointer :: XSCpool(:)
   real(r8), pointer :: tday(:)      ! daily accumulated temperature (deg C)
   real(r8), pointer :: tdayavg(:)   ! daily averaged temperature (deg C)
   real(r8), pointer :: tcount(:)    ! counter for daily avg temp
   real(r8), pointer :: degday(:)    ! accumulated degree days (deg C)
   real(r8), pointer :: ndegday(:)   ! counter for number of degree days
   real(r8), pointer :: stressT(:)   ! temperature stress function for leaf
                                     ! loss apply to Litterfall of deciduous veg   
   real(r8), pointer :: stressW(:)   ! water stress function for leaf loss
   real(r8), pointer :: iseabeg(:)   ! index for start of growing season
   real(r8), pointer :: nstepbeg(:)  ! nstep at start of growing season
   real(r8), pointer :: lgrow(:)     ! growing season index (0 or 1) to be
                                     ! passed daily to CASA to get NPP
#if (defined CLAMP)
   ! Summary variables added for the C-LAMP Experiments
   real(r8), pointer :: casa_agnpp(:)        ! above-ground net primary production [gC/m2/s]
   real(r8), pointer :: casa_ar(:)           ! autotrophic respiration [gC/m2/s]
   real(r8), pointer :: casa_bgnpp(:)        ! below-ground net primary production [gC/m2/s]
   real(r8), pointer :: casa_cwdc(:)         ! coarse woody debris C [gC/m2]
   real(r8), pointer :: casa_cwdc_hr(:)      ! cwd heterotrophic respiration [gC/m2/s]
   real(r8), pointer :: casa_cwdc_loss(:)    ! cwd C loss [gC/m2/s]
   real(r8), pointer :: casa_frootc(:)       ! fine root C [gC/m2]
   real(r8), pointer :: casa_frootc_alloc(:) ! fine root C allocation [gC/m2/s]
   real(r8), pointer :: casa_frootc_loss(:)  ! fine root C loss [gC/m2/s]
   real(r8), pointer :: casa_gpp(:)          ! gross primary production [gC/m2/s]
   real(r8), pointer :: casa_hr(:)           ! total heterotrophic respiration [gC/m2/s]
   real(r8), pointer :: casa_leafc(:)        ! leaf C [gC/m2]
   real(r8), pointer :: casa_leafc_alloc(:)  ! leaf C allocation [gC/m2/s]
   real(r8), pointer :: casa_leafc_loss(:)   ! leaf C loss [gC/m2/s]
   real(r8), pointer :: casa_litterc(:)      ! total litter C (excluding cwd C) [gC/m2]
   real(r8), pointer :: casa_litterc_hr(:)   ! litter heterotrophic respiration [gC/m2/s]
   real(r8), pointer :: casa_litterc_loss(:) ! litter C loss [gC/m2/s]
   real(r8), pointer :: casa_nee(:)          ! net ecosystem exchange [gC/m2/s]
   real(r8), pointer :: casa_nep(:)          ! net ecosystem production [gC/m2/s]
   real(r8), pointer :: casa_npp(:)          ! net primary production [gC/m2/s]
   real(r8), pointer :: casa_soilc(:)        ! total soil organic matter C (excluding cwd and litter C) [gC/m2]
   real(r8), pointer :: casa_soilc_hr(:)     ! soil heterotrophic respiration [gC/m2/s]
   real(r8), pointer :: casa_soilc_loss(:)   ! total soil organic matter C loss [gC/m2/s]
   real(r8), pointer :: casa_woodc(:)        ! wood C [gC/m2]
   real(r8), pointer :: casa_woodc_alloc(:)  ! wood C allocation [gC/m2/s]
   real(r8), pointer :: casa_woodc_loss(:)   ! wood C loss [gC/m2/s]
#endif
#endif
end type pft_pstate_type

!----------------------------------------------------
! pft ecophysiological constants structure
!----------------------------------------------------
type, public :: pft_epc_type
   integer , pointer :: noveg(:)                !value for not vegetated
   integer , pointer :: tree(:)                 !tree or not?
   real(r8), pointer :: smpso(:)                !soil water potential at full stomatal opening (mm)
   real(r8), pointer :: smpsc(:)                !soil water potential at full stomatal closure (mm)
   real(r8), pointer :: fnitr(:)                !foliage nitrogen limitation factor (-)
   real(r8), pointer :: foln(:)                 !foliage nitrogen (%)
   real(r8), pointer :: dleaf(:)                !characteristic leaf dimension (m)
   real(r8), pointer :: c3psn(:)                !photosynthetic pathway: 0. = c4, 1. = c3
   real(r8), pointer :: vcmx25(:)               !max rate of carboxylation at 25C (umol CO2/m**2/s)
   real(r8), pointer :: mp(:)                   !slope of conductance-to-photosynthesis relationship
   real(r8), pointer :: qe25(:)                 !quantum efficiency at 25C (umol CO2 / umol photon)
   real(r8), pointer :: xl(:)                   !leaf/stem orientation index
   real(r8), pointer :: rhol(:,:)               !leaf reflectance: 1=vis, 2=nir   (numrad)
   real(r8), pointer :: rhos(:,:)               !stem reflectance: 1=vis, 2=nir   (numrad)
   real(r8), pointer :: taul(:,:)               !leaf transmittance: 1=vis, 2=nir (numrad)
   real(r8), pointer :: taus(:,:)               !stem transmittance: 1=vis, 2=nir (numrad)
   real(r8), pointer :: z0mr(:)                 !ratio of momentum roughness length to canopy top height (-)
   real(r8), pointer :: displar(:)              !ratio of displacement height to canopy top height (-)
   real(r8), pointer :: roota_par(:)            !CLM rooting distribution parameter [1/m]
   real(r8), pointer :: rootb_par(:)            !CLM rooting distribution parameter [1/m]
   real(r8), pointer :: sla(:)                  !specific leaf area [m2 leaf g-1 carbon]
   ! new variables for CN code
   real(r8), pointer :: dwood(:)           !wood density (gC/m3)
   real(r8), pointer :: slatop(:)    !specific leaf area at top of canopy, projected area basis [m^2/gC]
   real(r8), pointer :: dsladlai(:)  !dSLA/dLAI, projected area basis [m^2/gC]
   real(r8), pointer :: leafcn(:)    !leaf C:N (gC/gN)
   real(r8), pointer :: flnr(:)      !fraction of leaf N in the Rubisco enzyme (gN Rubisco / gN leaf)
   real(r8), pointer :: woody(:)     !binary flag for woody lifeform (1=woody, 0=not woody)
   real(r8), pointer :: lflitcn(:)      !leaf litter C:N (gC/gN)
   real(r8), pointer :: frootcn(:)      !fine root C:N (gC/gN)
   real(r8), pointer :: livewdcn(:)     !live wood (phloem and ray parenchyma) C:N (gC/gN)
   real(r8), pointer :: deadwdcn(:)     !dead wood (xylem and heartwood) C:N (gC/gN)
#ifdef CROP
   real(r8), pointer :: graincn(:)      !grain C:N (gC/gN)
#endif
   real(r8), pointer :: froot_leaf(:)   !allocation parameter: new fine root C per new leaf C (gC/gC)
   real(r8), pointer :: stem_leaf(:)    !allocation parameter: new stem c per new leaf C (gC/gC)
   real(r8), pointer :: croot_stem(:)   !allocation parameter: new coarse root C per new stem C (gC/gC)
   real(r8), pointer :: flivewd(:)      !allocation parameter: fraction of new wood that is live (phloem and ray parenchyma) (no units)
   real(r8), pointer :: fcur(:)         !allocation parameter: fraction of allocation that goes to currently displayed growth, remainder to storage
   real(r8), pointer :: lf_flab(:)      !leaf litter labile fraction
   real(r8), pointer :: lf_fcel(:)      !leaf litter cellulose fraction
   real(r8), pointer :: lf_flig(:)      !leaf litter lignin fraction
   real(r8), pointer :: fr_flab(:)      !fine root litter labile fraction
   real(r8), pointer :: fr_fcel(:)      !fine root litter cellulose fraction
   real(r8), pointer :: fr_flig(:)      !fine root litter lignin fraction
   real(r8), pointer :: dw_fcel(:)      !dead wood cellulose fraction
   real(r8), pointer :: dw_flig(:)      !dead wood lignin fraction
   real(r8), pointer :: leaf_long(:)    !leaf longevity (yrs)
   real(r8), pointer :: evergreen(:)    !binary flag for evergreen leaf habit (0 or 1)
   real(r8), pointer :: stress_decid(:) !binary flag for stress-deciduous leaf habit (0 or 1)
   real(r8), pointer :: season_decid(:) !binary flag for seasonal-deciduous leaf habit (0 or 1)
   ! new variables for fire code
   real(r8), pointer :: resist(:)       !resistance to fire (no units)
end type pft_epc_type

#if (defined CNDV) || (defined CROP)
!----------------------------------------------------
! pft DGVM-specific ecophysiological constants structure
!----------------------------------------------------
type, public :: pft_dgvepc_type
   real(r8), pointer :: crownarea_max(:)   !tree maximum crown area [m2]
   real(r8), pointer :: tcmin(:)           !minimum coldest monthly mean temperature [units?]
   real(r8), pointer :: tcmax(:)           !maximum coldest monthly mean temperature [units?]
   real(r8), pointer :: gddmin(:)          !minimum growing degree days (at or above 5 C)
   real(r8), pointer :: twmax(:)           !upper limit of temperature of the warmest month [units?]
   real(r8), pointer :: reinickerp(:)      !parameter in allometric equation
   real(r8), pointer :: allom1(:)          !parameter in allometric
   real(r8), pointer :: allom2(:)          !parameter in allometric
   real(r8), pointer :: allom3(:)          !parameter in allometric
end type pft_dgvepc_type
#endif

!----------------------------------------------------
! pft ecophysiological variables structure
!----------------------------------------------------
type, public :: pft_epv_type
   real(r8), pointer :: dormant_flag(:)         !dormancy flag
   real(r8), pointer :: days_active(:)          !number of days since last dormancy
   real(r8), pointer :: onset_flag(:)           !onset flag
   real(r8), pointer :: onset_counter(:)        !onset days counter
   real(r8), pointer :: onset_gddflag(:)        !onset flag for growing degree day sum
   real(r8), pointer :: onset_fdd(:)            !onset freezing degree days counter
   real(r8), pointer :: onset_gdd(:)            !onset growing degree days
   real(r8), pointer :: onset_swi(:)            !onset soil water index
   real(r8), pointer :: offset_flag(:)          !offset flag
   real(r8), pointer :: offset_counter(:)       !offset days counter
   real(r8), pointer :: offset_fdd(:)           !offset freezing degree days counter
   real(r8), pointer :: offset_swi(:)           !offset soil water index
   real(r8), pointer :: lgsf(:)                 !long growing season factor [0-1]
   real(r8), pointer :: bglfr(:)                !background litterfall rate (1/s)
   real(r8), pointer :: bgtr(:)                 !background transfer growth rate (1/s)
   real(r8), pointer :: dayl(:)                 !daylength (seconds)
   real(r8), pointer :: prev_dayl(:)            !daylength from previous timestep (seconds)
   real(r8), pointer :: annavg_t2m(:)           !annual average 2m air temperature (K)
   real(r8), pointer :: tempavg_t2m(:)          !temporary average 2m air temperature (K)
   real(r8), pointer :: gpp(:)                  !GPP flux before downregulation (gC/m2/s)
   real(r8), pointer :: availc(:)               !C flux available for allocation (gC/m2/s)
   real(r8), pointer :: xsmrpool_recover(:)     !C flux assigned to recovery of negative cpool (gC/m2/s)
#if (defined C13)
   real(r8), pointer :: xsmrpool_c13ratio(:)    !C13/C(12+13) ratio for xsmrpool (proportion)
#endif
   real(r8), pointer :: alloc_pnow(:)           !fraction of current allocation to display as new growth (DIM)
   real(r8), pointer :: c_allometry(:)          !C allocation index (DIM)
   real(r8), pointer :: n_allometry(:)          !N allocation index (DIM)
   real(r8), pointer :: plant_ndemand(:)        !N flux required to support initial GPP (gN/m2/s)
   real(r8), pointer :: tempsum_potential_gpp(:)!temporary annual sum of potential GPP
   real(r8), pointer :: annsum_potential_gpp(:) !annual sum of potential GPP
   real(r8), pointer :: tempmax_retransn(:)     !temporary annual max of retranslocated N pool (gN/m2)
   real(r8), pointer :: annmax_retransn(:)      !annual max of retranslocated N pool (gN/m2)
   real(r8), pointer :: avail_retransn(:)       !N flux available from retranslocation pool (gN/m2/s)
   real(r8), pointer :: plant_nalloc(:)         !total allocated N flux (gN/m2/s)
   real(r8), pointer :: plant_calloc(:)         !total allocated C flux (gC/m2/s)
   real(r8), pointer :: excess_cflux(:)         !C flux not allocated due to downregulation (gC/m2/s)
   real(r8), pointer :: downreg(:)              !fractional reduction in GPP due to N limitation (DIM)
   real(r8), pointer :: prev_leafc_to_litter(:) !previous timestep leaf C litterfall flux (gC/m2/s)
   real(r8), pointer :: prev_frootc_to_litter(:)!previous timestep froot C litterfall flux (gC/m2/s)
   real(r8), pointer :: tempsum_npp(:)          !temporary annual sum of NPP (gC/m2/yr)
   real(r8), pointer :: annsum_npp(:)           !annual sum of NPP (gC/m2/yr)
#if (defined CNDV)
   real(r8), pointer :: tempsum_litfall(:)      !temporary annual sum of litfall (gC/m2/yr)
   real(r8), pointer :: annsum_litfall(:)       !annual sum of litfall (gC/m2/yr)
#endif
#if (defined C13)
   real(r8), pointer :: rc13_canair(:)          !C13O2/C12O2 in canopy air
   real(r8), pointer :: rc13_psnsun(:)          !C13O2/C12O2 in sunlit canopy psn flux
   real(r8), pointer :: rc13_psnsha(:)          !C13O2/C12O2 in shaded canopy psn flux
#endif
end type pft_epv_type                        

!----------------------------------------------------
! pft energy state variables structure
!----------------------------------------------------
type, public :: pft_estate_type
   real(r8), pointer :: t_ref2m(:)            !2 m height surface air temperature (Kelvin)
   real(r8), pointer :: t_ref2m_min(:)        !daily minimum of average 2 m height surface air temperature (K)
   real(r8), pointer :: t_ref2m_max(:)        !daily maximum of average 2 m height surface air temperature (K)
   real(r8), pointer :: t_ref2m_min_inst(:)   !instantaneous daily min of average 2 m height surface air temp (K)
   real(r8), pointer :: t_ref2m_max_inst(:)   !instantaneous daily max of average 2 m height surface air temp (K)
   real(r8), pointer :: q_ref2m(:)            !2 m height surface specific humidity (kg/kg)
   real(r8), pointer :: t_ref2m_u(:)          !Urban 2 m height surface air temperature (Kelvin)
   real(r8), pointer :: t_ref2m_r(:)          !Rural 2 m height surface air temperature (Kelvin)
   real(r8), pointer :: t_ref2m_min_u(:)      !Urban daily minimum of average 2 m height surface air temperature (K)
   real(r8), pointer :: t_ref2m_min_r(:)      !Rural daily minimum of average 2 m height surface air temperature (K)
   real(r8), pointer :: t_ref2m_max_u(:)      !Urban daily maximum of average 2 m height surface air temperature (K)
   real(r8), pointer :: t_ref2m_max_r(:)      !Rural daily maximum of average 2 m height surface air temperature (K)
   real(r8), pointer :: t_ref2m_min_inst_u(:) !Urban instantaneous daily min of average 2 m height surface air temp (K)
   real(r8), pointer :: t_ref2m_min_inst_r(:) !Rural instantaneous daily min of average 2 m height surface air temp (K)
   real(r8), pointer :: t_ref2m_max_inst_u(:) !Urban instantaneous daily max of average 2 m height surface air temp (K)
   real(r8), pointer :: t_ref2m_max_inst_r(:) !Rural instantaneous daily max of average 2 m height surface air temp (K)
   real(r8), pointer :: rh_ref2m(:)           !2 m height surface relative humidity (%)
   real(r8), pointer :: rh_ref2m_u(:)         !Urban 2 m height surface relative humidity (%)
   real(r8), pointer :: rh_ref2m_r(:)         !Rural 2 m height surface relative humidity (%)
   real(r8), pointer :: t_veg(:)              !vegetation temperature (Kelvin)
   real(r8), pointer :: thm(:)                !intermediate variable (forc_t+0.0098*forc_hgt_t_pft)
end type pft_estate_type

!----------------------------------------------------
! pft water state variables structure
!----------------------------------------------------
type, public :: pft_wstate_type
   real(r8), pointer :: h2ocan(:)         !canopy water (mm H2O)
end type pft_wstate_type

!----------------------------------------------------
! pft carbon state variables structure
!----------------------------------------------------
type, public :: pft_cstate_type
   real(r8), pointer :: leafcmax(:)           ! (gC/m2) ann max leaf C
#if (defined CROP)
   real(r8), pointer :: grainc(:)             ! (gC/m2) grain C
   real(r8), pointer :: grainc_storage(:)     ! (gC/m2) grain C storage
   real(r8), pointer :: grainc_xfer(:)        ! (gC/m2) grain C transfer
#endif
   real(r8), pointer :: leafc(:)              ! (gC/m2) leaf C
   real(r8), pointer :: leafc_storage(:)      ! (gC/m2) leaf C storage
   real(r8), pointer :: leafc_xfer(:)         ! (gC/m2) leaf C transfer
   real(r8), pointer :: frootc(:)             ! (gC/m2) fine root C
   real(r8), pointer :: frootc_storage(:)     ! (gC/m2) fine root C storage
   real(r8), pointer :: frootc_xfer(:)        ! (gC/m2) fine root C transfer
   real(r8), pointer :: livestemc(:)          ! (gC/m2) live stem C
   real(r8), pointer :: livestemc_storage(:)  ! (gC/m2) live stem C storage
   real(r8), pointer :: livestemc_xfer(:)     ! (gC/m2) live stem C transfer
   real(r8), pointer :: deadstemc(:)          ! (gC/m2) dead stem C
   real(r8), pointer :: deadstemc_storage(:)  ! (gC/m2) dead stem C storage
   real(r8), pointer :: deadstemc_xfer(:)     ! (gC/m2) dead stem C transfer
   real(r8), pointer :: livecrootc(:)         ! (gC/m2) live coarse root C
   real(r8), pointer :: livecrootc_storage(:) ! (gC/m2) live coarse root C storage
   real(r8), pointer :: livecrootc_xfer(:)    ! (gC/m2) live coarse root C transfer
   real(r8), pointer :: deadcrootc(:)         ! (gC/m2) dead coarse root C
   real(r8), pointer :: deadcrootc_storage(:) ! (gC/m2) dead coarse root C storage
   real(r8), pointer :: deadcrootc_xfer(:)    ! (gC/m2) dead coarse root C transfer
   real(r8), pointer :: gresp_storage(:)      ! (gC/m2) growth respiration storage
   real(r8), pointer :: gresp_xfer(:)         ! (gC/m2) growth respiration transfer
   real(r8), pointer :: cpool(:)              ! (gC/m2) temporary photosynthate C pool
   real(r8), pointer :: xsmrpool(:)           ! (gC/m2) abstract C pool to meet excess MR demand
   real(r8), pointer :: pft_ctrunc(:)         ! (gC/m2) pft-level sink for C truncation
   ! summary (diagnostic) state variables, not involved in mass balance
   real(r8), pointer :: dispvegc(:)           ! (gC/m2) displayed veg carbon, excluding storage and cpool
   real(r8), pointer :: storvegc(:)           ! (gC/m2) stored vegetation carbon, excluding cpool
   real(r8), pointer :: totvegc(:)            ! (gC/m2) total vegetation carbon, excluding cpool
   real(r8), pointer :: totpftc(:)            ! (gC/m2) total pft-level carbon, including cpool
#if (defined CLAMP) && (defined CN)
   ! CLAMP summary (diagnostic) variable
   real(r8), pointer :: woodc(:)              ! (gC/m2) wood C
#endif
end type pft_cstate_type

!----------------------------------------------------
! pft nitrogen state variables structure
!----------------------------------------------------
type, public :: pft_nstate_type
#if (defined CROP)
   real(r8), pointer :: grainn(:)             ! (gN/m2) grain N 
   real(r8), pointer :: grainn_storage(:)     ! (gN/m2) grain N storage
   real(r8), pointer :: grainn_xfer(:)        ! (gN/m2) grain N transfer
#endif
   real(r8), pointer :: leafn(:)              ! (gN/m2) leaf N 
   real(r8), pointer :: leafn_storage(:)      ! (gN/m2) leaf N storage
   real(r8), pointer :: leafn_xfer(:)         ! (gN/m2) leaf N transfer
   real(r8), pointer :: frootn(:)             ! (gN/m2) fine root N
   real(r8), pointer :: frootn_storage(:)     ! (gN/m2) fine root N storage
   real(r8), pointer :: frootn_xfer(:)        ! (gN/m2) fine root N transfer
   real(r8), pointer :: livestemn(:)          ! (gN/m2) live stem N
   real(r8), pointer :: livestemn_storage(:)  ! (gN/m2) live stem N storage
   real(r8), pointer :: livestemn_xfer(:)     ! (gN/m2) live stem N transfer
   real(r8), pointer :: deadstemn(:)          ! (gN/m2) dead stem N
   real(r8), pointer :: deadstemn_storage(:)  ! (gN/m2) dead stem N storage
   real(r8), pointer :: deadstemn_xfer(:)     ! (gN/m2) dead stem N transfer
   real(r8), pointer :: livecrootn(:)         ! (gN/m2) live coarse root N
   real(r8), pointer :: livecrootn_storage(:) ! (gN/m2) live coarse root N storage
   real(r8), pointer :: livecrootn_xfer(:)    ! (gN/m2) live coarse root N transfer
   real(r8), pointer :: deadcrootn(:)         ! (gN/m2) dead coarse root N
   real(r8), pointer :: deadcrootn_storage(:) ! (gN/m2) dead coarse root N storage
   real(r8), pointer :: deadcrootn_xfer(:)    ! (gN/m2) dead coarse root N transfer
   real(r8), pointer :: retransn(:)           ! (gN/m2) plant pool of retranslocated N
   real(r8), pointer :: npool(:)              ! (gN/m2) temporary plant N pool
   real(r8), pointer :: pft_ntrunc(:)         ! (gN/m2) pft-level sink for N truncation
   ! summary (diagnostic) state variables, not involved in mass balance
   real(r8), pointer :: dispvegn(:)           ! (gN/m2) displayed veg nitrogen, excluding storage
   real(r8), pointer :: storvegn(:)           ! (gN/m2) stored vegetation nitrogen
   real(r8), pointer :: totvegn(:)            ! (gN/m2) total vegetation nitrogen
   real(r8), pointer :: totpftn(:)            ! (gN/m2) total pft-level nitrogen
end type pft_nstate_type

!----------------------------------------------------
! pft VOC state variables structure
!----------------------------------------------------
type, public :: pft_vstate_type
   real(r8), pointer :: t_veg24(:)             ! 24hr average vegetation temperature (K)
   real(r8), pointer :: t_veg240(:)            ! 240hr average vegetation temperature (Kelvin)
   real(r8), pointer :: fsd24(:)               ! 24hr average of direct beam radiation 
   real(r8), pointer :: fsd240(:)              ! 240hr average of direct beam radiation 
   real(r8), pointer :: fsi24(:)               ! 24hr average of diffuse beam radiation 
   real(r8), pointer :: fsi240(:)              ! 240hr average of diffuse beam radiation 
   real(r8), pointer :: fsun24(:)              ! 24hr average of sunlit fraction of canopy 
   real(r8), pointer :: fsun240(:)             ! 240hr average of sunlit fraction of canopy
   real(r8), pointer :: elai_p(:)              ! leaf area index average over timestep 
end type pft_vstate_type

#if (defined CNDV) || (defined CROP)
!----------------------------------------------------
! pft DGVM state variables structure
!----------------------------------------------------
type, public :: pft_dgvstate_type
   real(r8), pointer :: agddtw(:)              !accumulated growing degree days above twmax
   real(r8), pointer :: agdd(:)                !accumulated growing degree days above 5
   real(r8), pointer :: t10(:)                 !10-day running mean of the 2 m temperature (K)
   real(r8), pointer :: t_mo(:)                !30-day average temperature (Kelvin)
   real(r8), pointer :: t_mo_min(:)            !annual min of t_mo (Kelvin)
   real(r8), pointer :: prec365(:)             !365-day running mean of tot. precipitation
   logical , pointer :: present(:)             !whether PFT present in patch
   logical , pointer :: pftmayexist(:)         !if .false. then exclude seasonal decid pfts from tropics
   real(r8), pointer :: nind(:)                !number of individuals (#/m**2)
   real(r8), pointer :: lm_ind(:)              !individual leaf mass
   real(r8), pointer :: lai_ind(:)             !LAI per individual
   real(r8), pointer :: fpcinc(:)              !foliar projective cover increment (fraction) 
   real(r8), pointer :: fpcgrid(:)             !foliar projective cover on gridcell (fraction)
   real(r8), pointer :: fpcgridold(:)          !last yr's fpcgrid
   real(r8), pointer :: crownarea(:)           !area that each individual tree takes up (m^2)
   real(r8), pointer :: greffic(:)
   real(r8), pointer :: heatstress(:)
end type pft_dgvstate_type
#endif

!----------------------------------------------------
! pft energy flux variables structure
!----------------------------------------------------
type, public :: pft_eflux_type
   real(r8), pointer :: sabg(:)              !solar radiation absorbed by ground (W/m**2)
   real(r8), pointer :: sabv(:)              !solar radiation absorbed by vegetation (W/m**2)
   real(r8), pointer :: fsa(:)               !solar radiation absorbed (total) (W/m**2)
   real(r8), pointer :: fsa_u(:)             !urban solar radiation absorbed (total) (W/m**2)
   real(r8), pointer :: fsa_r(:)             !rural solar radiation absorbed (total) (W/m**2)
   real(r8), pointer :: fsr(:)               !solar radiation reflected (W/m**2)
   real(r8), pointer :: parsun(:)            !average absorbed PAR for sunlit leaves (W/m**2)
   real(r8), pointer :: parsha(:)            !average absorbed PAR for shaded leaves (W/m**2)
   real(r8), pointer :: dlrad(:)             !downward longwave radiation below the canopy [W/m2]
   real(r8), pointer :: ulrad(:)             !upward longwave radiation above the canopy [W/m2]
   real(r8), pointer :: eflx_lh_tot(:)       !total latent heat flux (W/m**2)  [+ to atm]
   real(r8), pointer :: eflx_lh_tot_u(:)     !urban total latent heat flux (W/m**2)  [+ to atm]
   real(r8), pointer :: eflx_lh_tot_r(:)     !rural total latent heat flux (W/m**2)  [+ to atm]
   real(r8), pointer :: eflx_lh_grnd(:)      !ground evaporation heat flux (W/m**2) [+ to atm]
   real(r8), pointer :: eflx_soil_grnd(:)    !soil heat flux (W/m**2) [+ = into soil]
   real(r8), pointer :: eflx_soil_grnd_u(:)  !urban soil heat flux (W/m**2) [+ = into soil]
   real(r8), pointer :: eflx_soil_grnd_r(:)  !rural soil heat flux (W/m**2) [+ = into soil]
   real(r8), pointer :: eflx_sh_tot(:)       !total sensible heat flux (W/m**2) [+ to atm]
   real(r8), pointer :: eflx_sh_tot_u(:)     !urban total sensible heat flux (W/m**2) [+ to atm]
   real(r8), pointer :: eflx_sh_tot_r(:)     !rural total sensible heat flux (W/m**2) [+ to atm]
   real(r8), pointer :: eflx_sh_grnd(:)      !sensible heat flux from ground (W/m**2) [+ to atm]
   real(r8), pointer :: eflx_sh_veg(:)       !sensible heat flux from leaves (W/m**2) [+ to atm]
   real(r8), pointer :: eflx_lh_vege(:)      !veg evaporation heat flux (W/m**2) [+ to atm]
   real(r8), pointer :: eflx_lh_vegt(:)      !veg transpiration heat flux (W/m**2) [+ to atm]
   real(r8), pointer :: eflx_wasteheat_pft(:) !sensible heat flux from domestic heating/cooling sources of waste heat (W/m**2)
   real(r8), pointer :: eflx_heat_from_ac_pft(:) !sensible heat flux put back into canyon due to removal by AC (W/m**2)
   real(r8), pointer :: eflx_traffic_pft(:)      !traffic sensible heat flux (W/m**2)
   real(r8), pointer :: eflx_anthro(:)           !total anthropogenic heat flux (W/m**2)
   real(r8), pointer :: cgrnd(:)             !deriv. of soil energy flux wrt to soil temp [w/m2/k]
   real(r8), pointer :: cgrndl(:)            !deriv. of soil latent heat flux wrt soil temp [w/m**2/k]
   real(r8), pointer :: cgrnds(:)            !deriv. of soil sensible heat flux wrt soil temp [w/m2/k]
   real(r8), pointer :: eflx_gnet(:)         !net heat flux into ground (W/m**2)
   real(r8), pointer :: dgnetdT(:)           !derivative of net ground heat flux wrt soil temp (W/m**2 K)
   real(r8), pointer :: eflx_lwrad_out(:)    !emitted infrared (longwave) radiation (W/m**2)
   real(r8), pointer :: eflx_lwrad_net(:)    !net infrared (longwave) rad (W/m**2) [+ = to atm]
   real(r8), pointer :: eflx_lwrad_net_u(:)  !urban net infrared (longwave) rad (W/m**2) [+ = to atm]
   real(r8), pointer :: eflx_lwrad_net_r(:)  !rural net infrared (longwave) rad (W/m**2) [+ = to atm]
   real(r8), pointer :: netrad(:)            !net radiation (W/m**2) [+ = to sfc]
   real(r8), pointer :: fsds_vis_d(:)        !incident direct beam vis solar radiation (W/m**2)
   real(r8), pointer :: fsds_nir_d(:)        !incident direct beam nir solar radiation (W/m**2)
   real(r8), pointer :: fsds_vis_i(:)        !incident diffuse vis solar radiation (W/m**2)
   real(r8), pointer :: fsds_nir_i(:)        !incident diffuse nir solar radiation (W/m**2)
   real(r8), pointer :: fsr_vis_d(:)         !reflected direct beam vis solar radiation (W/m**2)
   real(r8), pointer :: fsr_nir_d(:)         !reflected direct beam nir solar radiation (W/m**2)
   real(r8), pointer :: fsr_vis_i(:)         !reflected diffuse vis solar radiation (W/m**2)
   real(r8), pointer :: fsr_nir_i(:)         !reflected diffuse nir solar radiation (W/m**2)
   real(r8), pointer :: fsds_vis_d_ln(:)     !incident direct beam vis solar radiation at local noon (W/m**2)
   real(r8), pointer :: fsds_nir_d_ln(:)     !incident direct beam nir solar radiation at local noon (W/m**2)
   real(r8), pointer :: fsr_vis_d_ln(:)      !reflected direct beam vis solar radiation at local noon (W/m**2)
   real(r8), pointer :: fsr_nir_d_ln(:)      !reflected direct beam nir solar radiation at local noon (W/m**2)
   real(r8), pointer :: sun_add(:,:)      !sun canopy absorbed direct from direct (W/m**2)
   real(r8), pointer :: tot_aid(:,:)      !total canopy absorbed indirect from direct (W/m**2)
   real(r8), pointer :: sun_aid(:,:)      !sun canopy absorbed indirect from direct (W/m**2)
   real(r8), pointer :: sun_aii(:,:)      !sun canopy absorbed indirect from indirect (W/m**2)
   real(r8), pointer :: sha_aid(:,:)      !shade canopy absorbed indirect from direct (W/m**2)
   real(r8), pointer :: sha_aii(:,:)      !shade canopy absorbed indirect from indirect (W/m**2)
   real(r8), pointer :: sun_atot(:,:)     !sun canopy total absorbed (W/m**2)
   real(r8), pointer :: sha_atot(:,:)     !shade canopy total absorbed (W/m**2)
   real(r8), pointer :: sun_alf(:,:)      !sun canopy total absorbed by leaves (W/m**2)
   real(r8), pointer :: sha_alf(:,:)      !shade canopy total absored by leaves (W/m**2)
   real(r8), pointer :: sun_aperlai(:,:)  !sun canopy total absorbed per unit LAI (W/m**2)
   real(r8), pointer :: sha_aperlai(:,:)  !shade canopy total absorbed per unit LAI (W/m**2)
   real(r8), pointer :: sabg_lyr(:,:)     ! absorbed radiation in each snow layer and top soil layer (pft,lyr) [W/m2]
   real(r8), pointer :: sfc_frc_aer(:)    ! surface forcing of snow with all aerosols (pft) [W/m2]
   real(r8), pointer :: sfc_frc_bc(:)     ! surface forcing of snow with BC (pft) [W/m2]
   real(r8), pointer :: sfc_frc_oc(:)     ! surface forcing of snow with OC (pft) [W/m2]
   real(r8), pointer :: sfc_frc_dst(:)    ! surface forcing of snow with dust (pft) [W/m2]
   real(r8), pointer :: sfc_frc_aer_sno(:)! surface forcing of snow with all aerosols, averaged only when snow is present (pft) [W/m2]
   real(r8), pointer :: sfc_frc_bc_sno(:) ! surface forcing of snow with BC, averaged only when snow is present (pft) [W/m2]
   real(r8), pointer :: sfc_frc_oc_sno(:) ! surface forcing of snow with OC, averaged only when snow is present (pft) [W/m2]
   real(r8), pointer :: sfc_frc_dst_sno(:)! surface forcing of snow with dust, averaged only when snow is present (pft) [W/m2]
   real(r8), pointer :: fsr_sno_vd(:)     ! reflected direct beam vis solar radiation from snow (W/m**2)
   real(r8), pointer :: fsr_sno_nd(:)     ! reflected direct beam NIR solar radiation from snow (W/m**2)
   real(r8), pointer :: fsr_sno_vi(:)     ! reflected diffuse vis solar radiation from snow (W/m**2)
   real(r8), pointer :: fsr_sno_ni(:)     ! reflected diffuse NIR solar radiation from snow (W/m**2)
   real(r8), pointer :: fsds_sno_vd(:)    ! incident visible, direct radiation on snow (for history files)  [W/m2]
   real(r8), pointer :: fsds_sno_nd(:)    ! incident near-IR, direct radiation on snow (for history files)  [W/m2]
   real(r8), pointer :: fsds_sno_vi(:)    ! incident visible, diffuse radiation on snow (for history files) [W/m2]
   real(r8), pointer :: fsds_sno_ni(:)    ! incident near-IR, diffuse radiation on snow (for history files) [W/m2]
end type pft_eflux_type

!----------------------------------------------------
! pft momentum flux variables structure
!----------------------------------------------------
type, public :: pft_mflux_type
   real(r8),pointer ::  taux(:)           !wind (shear) stress: e-w (kg/m/s**2)
   real(r8),pointer ::  tauy(:)           !wind (shear) stress: n-s (kg/m/s**2)
end type pft_mflux_type

!----------------------------------------------------
! pft water flux variables structure
!----------------------------------------------------
type, public :: pft_wflux_type
   real(r8), pointer :: qflx_prec_intr(:) !interception of precipitation [mm/s]
   real(r8), pointer :: qflx_prec_grnd(:) !water onto ground including canopy runoff [kg/(m2 s)]
   real(r8), pointer :: qflx_rain_grnd(:) !rain on ground after interception (mm H2O/s) [+]
   real(r8), pointer :: qflx_snow_grnd(:) !snow on ground after interception (mm H2O/s) [+]
   real(r8), pointer :: qflx_snwcp_ice(:) !excess snowfall due to snow capping (mm H2O /s) [+]
   real(r8), pointer :: qflx_snwcp_liq(:) !excess rainfall due to snow capping (mm H2O /s) [+]
   real(r8), pointer :: qflx_evap_veg(:)  !vegetation evaporation (mm H2O/s) (+ = to atm)
   real(r8), pointer :: qflx_tran_veg(:)  !vegetation transpiration (mm H2O/s) (+ = to atm)
   real(r8), pointer :: qflx_evap_can(:)  !evaporation from leaves and stems 
   real(r8), pointer :: qflx_evap_soi(:)  !soil evaporation (mm H2O/s) (+ = to atm)
   real(r8), pointer :: qflx_evap_tot(:)  !qflx_evap_soi + qflx_evap_veg + qflx_tran_veg
   real(r8), pointer :: qflx_evap_grnd(:) !ground surface evaporation rate (mm H2O/s) [+]
   real(r8), pointer :: qflx_dew_grnd(:)  !ground surface dew formation (mm H2O /s) [+]
   real(r8), pointer :: qflx_sub_snow(:)  !sublimation rate from snow pack (mm H2O /s) [+]
   real(r8), pointer :: qflx_dew_snow(:)  !surface dew added to snow pack (mm H2O /s) [+]
end type pft_wflux_type

!----------------------------------------------------
! pft carbon flux variables structure
!----------------------------------------------------
type, public :: pft_cflux_type
   real(r8), pointer :: psnsun(:)         !sunlit leaf photosynthesis (umol CO2 /m**2/ s)
   real(r8), pointer :: psnsha(:)         !shaded leaf photosynthesis (umol CO2 /m**2/ s)
   real(r8), pointer :: fpsn(:)           !photosynthesis (umol CO2 /m**2 /s)
   real(r8), pointer :: fco2(:)           !net CO2 flux (umol CO2 /m**2 /s) [+ = to atm]
   ! new variables for CN code
   ! gap mortality fluxes
   real(r8), pointer :: m_leafc_to_litter(:)                 ! leaf C mortality (gC/m2/s)
   real(r8), pointer :: m_leafc_storage_to_litter(:)         ! leaf C storage mortality (gC/m2/s)
   real(r8), pointer :: m_leafc_xfer_to_litter(:)            ! leaf C transfer mortality (gC/m2/s)
   real(r8), pointer :: m_frootc_to_litter(:)                ! fine root C mortality (gC/m2/s)
   real(r8), pointer :: m_frootc_storage_to_litter(:)        ! fine root C storage mortality (gC/m2/s)
   real(r8), pointer :: m_frootc_xfer_to_litter(:)           ! fine root C transfer mortality (gC/m2/s)
   real(r8), pointer :: m_livestemc_to_litter(:)             ! live stem C mortality (gC/m2/s)
   real(r8), pointer :: m_livestemc_storage_to_litter(:)     ! live stem C storage mortality (gC/m2/s)
   real(r8), pointer :: m_livestemc_xfer_to_litter(:)        ! live stem C transfer mortality (gC/m2/s)
   real(r8), pointer :: m_deadstemc_to_litter(:)             ! dead stem C mortality (gC/m2/s)
   real(r8), pointer :: m_deadstemc_storage_to_litter(:)     ! dead stem C storage mortality (gC/m2/s)
   real(r8), pointer :: m_deadstemc_xfer_to_litter(:)        ! dead stem C transfer mortality (gC/m2/s)
   real(r8), pointer :: m_livecrootc_to_litter(:)            ! live coarse root C mortality (gC/m2/s)
   real(r8), pointer :: m_livecrootc_storage_to_litter(:)    ! live coarse root C storage mortality (gC/m2/s)
   real(r8), pointer :: m_livecrootc_xfer_to_litter(:)       ! live coarse root C transfer mortality (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_to_litter(:)            ! dead coarse root C mortality (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_storage_to_litter(:)    ! dead coarse root C storage mortality (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_xfer_to_litter(:)       ! dead coarse root C transfer mortality (gC/m2/s)
   real(r8), pointer :: m_gresp_storage_to_litter(:)         ! growth respiration storage mortality (gC/m2/s)
   real(r8), pointer :: m_gresp_xfer_to_litter(:)            ! growth respiration transfer mortality (gC/m2/s)
   ! harvest mortality fluxes
   real(r8), pointer :: hrv_leafc_to_litter(:)               ! leaf C harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_leafc_storage_to_litter(:)       ! leaf C storage harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_leafc_xfer_to_litter(:)          ! leaf C transfer harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_frootc_to_litter(:)              ! fine root C harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_frootc_storage_to_litter(:)      ! fine root C storage harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_frootc_xfer_to_litter(:)         ! fine root C transfer harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_livestemc_to_litter(:)           ! live stem C harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_livestemc_storage_to_litter(:)   ! live stem C storage harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_livestemc_xfer_to_litter(:)      ! live stem C transfer harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_deadstemc_to_prod10c(:)          ! dead stem C harvest to 10-year product pool (gC/m2/s)
   real(r8), pointer :: hrv_deadstemc_to_prod100c(:)         ! dead stem C harvest to 100-year product pool (gC/m2/s)
   real(r8), pointer :: hrv_deadstemc_storage_to_litter(:)   ! dead stem C storage harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_deadstemc_xfer_to_litter(:)      ! dead stem C transfer harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_livecrootc_to_litter(:)          ! live coarse root C harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_livecrootc_storage_to_litter(:)  ! live coarse root C storage harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_livecrootc_xfer_to_litter(:)     ! live coarse root C transfer harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_deadcrootc_to_litter(:)          ! dead coarse root C harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_deadcrootc_storage_to_litter(:)  ! dead coarse root C storage harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_deadcrootc_xfer_to_litter(:)     ! dead coarse root C transfer harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_gresp_storage_to_litter(:)       ! growth respiration storage harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_gresp_xfer_to_litter(:)          ! growth respiration transfer harvest mortality (gC/m2/s)
   real(r8), pointer :: hrv_xsmrpool_to_atm(:)               ! excess MR pool harvest mortality (gC/m2/s)
   ! PFT-level fire fluxes
   real(r8), pointer :: m_leafc_to_fire(:)                   ! leaf C fire loss (gC/m2/s)
   real(r8), pointer :: m_leafc_storage_to_fire(:)           ! leaf C storage fire loss (gC/m2/s)
   real(r8), pointer :: m_leafc_xfer_to_fire(:)              ! leaf C transfer fire loss (gC/m2/s)
   real(r8), pointer :: m_frootc_to_fire(:)                  ! fine root C fire loss (gC/m2/s)
   real(r8), pointer :: m_frootc_storage_to_fire(:)          ! fine root C storage fire loss (gC/m2/s)
   real(r8), pointer :: m_frootc_xfer_to_fire(:)             ! fine root C transfer fire loss (gC/m2/s)
   real(r8), pointer :: m_livestemc_to_fire(:)               ! live stem C fire loss (gC/m2/s)
   real(r8), pointer :: m_livestemc_storage_to_fire(:)       ! live stem C storage fire loss (gC/m2/s)
   real(r8), pointer :: m_livestemc_xfer_to_fire(:)          ! live stem C transfer fire loss (gC/m2/s)
   real(r8), pointer :: m_deadstemc_to_fire(:)               ! dead stem C fire loss (gC/m2/s)
   real(r8), pointer :: m_deadstemc_to_litter_fire(:)        ! dead stem C fire mortality to litter (gC/m2/s)
   real(r8), pointer :: m_deadstemc_storage_to_fire(:)       ! dead stem C storage fire loss (gC/m2/s)
   real(r8), pointer :: m_deadstemc_xfer_to_fire(:)          ! dead stem C transfer fire loss (gC/m2/s)
   real(r8), pointer :: m_livecrootc_to_fire(:)              ! live coarse root C fire loss (gC/m2/s)
   real(r8), pointer :: m_livecrootc_storage_to_fire(:)      ! live coarse root C storage fire loss (gC/m2/s)
   real(r8), pointer :: m_livecrootc_xfer_to_fire(:)         ! live coarse root C transfer fire loss (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_to_fire(:)              ! dead coarse root C fire loss (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_to_litter_fire(:)       ! dead coarse root C fire mortality to litter (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_storage_to_fire(:)      ! dead coarse root C storage fire loss (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_xfer_to_fire(:)         ! dead coarse root C transfer fire loss (gC/m2/s)
   real(r8), pointer :: m_gresp_storage_to_fire(:)           ! growth respiration storage fire loss (gC/m2/s)
   real(r8), pointer :: m_gresp_xfer_to_fire(:)              ! growth respiration transfer fire loss (gC/m2/s)
   ! phenology fluxes from transfer pools                     
#if (defined CROP)
   real(r8), pointer :: grainc_xfer_to_grainc(:)             ! grain C growth from storage (gC/m2/s)
#endif
   real(r8), pointer :: leafc_xfer_to_leafc(:)               ! leaf C growth from storage (gC/m2/s)
   real(r8), pointer :: frootc_xfer_to_frootc(:)             ! fine root C growth from storage (gC/m2/s)
   real(r8), pointer :: livestemc_xfer_to_livestemc(:)       ! live stem C growth from storage (gC/m2/s)
   real(r8), pointer :: deadstemc_xfer_to_deadstemc(:)       ! dead stem C growth from storage (gC/m2/s)
   real(r8), pointer :: livecrootc_xfer_to_livecrootc(:)     ! live coarse root C growth from storage (gC/m2/s)
   real(r8), pointer :: deadcrootc_xfer_to_deadcrootc(:)     ! dead coarse root C growth from storage (gC/m2/s)
   ! leaf and fine root litterfall                           
   real(r8), pointer :: leafc_to_litter(:)                   ! leaf C litterfall (gC/m2/s)
   real(r8), pointer :: frootc_to_litter(:)                  ! fine root C litterfall (gC/m2/s)
#if (defined CROP)
   real(r8), pointer :: livestemc_to_litter(:)               ! live stem C litterfall (gC/m2/s)
   real(r8), pointer :: grainc_to_food(:)                    ! grain C to food (gC/m2/s)
#endif
   ! maintenance respiration fluxes                          
   real(r8), pointer :: leaf_mr(:)                           ! leaf maintenance respiration (gC/m2/s)
   real(r8), pointer :: froot_mr(:)                          ! fine root maintenance respiration (gC/m2/s)
   real(r8), pointer :: livestem_mr(:)                       ! live stem maintenance respiration (gC/m2/s)
   real(r8), pointer :: livecroot_mr(:)                      ! live coarse root maintenance respiration (gC/m2/s)
   real(r8), pointer :: leaf_curmr(:)                        ! leaf maintenance respiration from current GPP (gC/m2/s)
   real(r8), pointer :: froot_curmr(:)                       ! fine root maintenance respiration from current GPP (gC/m2/s)
   real(r8), pointer :: livestem_curmr(:)                    ! live stem maintenance respiration from current GPP (gC/m2/s)
   real(r8), pointer :: livecroot_curmr(:)                   ! live coarse root maintenance respiration from current GPP (gC/m2/s)
   real(r8), pointer :: leaf_xsmr(:)                         ! leaf maintenance respiration from storage (gC/m2/s)
   real(r8), pointer :: froot_xsmr(:)                        ! fine root maintenance respiration from storage (gC/m2/s)
   real(r8), pointer :: livestem_xsmr(:)                     ! live stem maintenance respiration from storage (gC/m2/s)
   real(r8), pointer :: livecroot_xsmr(:)                    ! live coarse root maintenance respiration from storage (gC/m2/s)
   ! photosynthesis fluxes                                   
   real(r8), pointer :: psnsun_to_cpool(:)                   ! C fixation from sunlit canopy (gC/m2/s)
   real(r8), pointer :: psnshade_to_cpool(:)                 ! C fixation from shaded canopy (gC/m2/s)
   ! allocation fluxes, from current GPP                     
   real(r8), pointer :: cpool_to_xsmrpool(:)                 ! allocation to maintenance respiration storage pool (gC/m2/s)
#if (defined CROP)
   real(r8), pointer :: cpool_to_grainc(:)                   ! allocation to grain C (gC/m2/s)
   real(r8), pointer :: cpool_to_grainc_storage(:)           ! allocation to grain C storage (gC/m2/s)
#endif
   real(r8), pointer :: cpool_to_leafc(:)                    ! allocation to leaf C (gC/m2/s)
   real(r8), pointer :: cpool_to_leafc_storage(:)            ! allocation to leaf C storage (gC/m2/s)
   real(r8), pointer :: cpool_to_frootc(:)                   ! allocation to fine root C (gC/m2/s)
   real(r8), pointer :: cpool_to_frootc_storage(:)           ! allocation to fine root C storage (gC/m2/s)
   real(r8), pointer :: cpool_to_livestemc(:)                ! allocation to live stem C (gC/m2/s)
   real(r8), pointer :: cpool_to_livestemc_storage(:)        ! allocation to live stem C storage (gC/m2/s)
   real(r8), pointer :: cpool_to_deadstemc(:)                ! allocation to dead stem C (gC/m2/s)
   real(r8), pointer :: cpool_to_deadstemc_storage(:)        ! allocation to dead stem C storage (gC/m2/s)
   real(r8), pointer :: cpool_to_livecrootc(:)               ! allocation to live coarse root C (gC/m2/s)
   real(r8), pointer :: cpool_to_livecrootc_storage(:)       ! allocation to live coarse root C storage (gC/m2/s)
   real(r8), pointer :: cpool_to_deadcrootc(:)               ! allocation to dead coarse root C (gC/m2/s)
   real(r8), pointer :: cpool_to_deadcrootc_storage(:)       ! allocation to dead coarse root C storage (gC/m2/s)
   real(r8), pointer :: cpool_to_gresp_storage(:)            ! allocation to growth respiration storage (gC/m2/s)
   ! growth respiration fluxes                               
#if (defined CROP)
   real(r8), pointer :: xsmrpool_to_atm(:)
   real(r8), pointer :: cpool_grain_gr(:)                    ! grain growth respiration (gC/m2/s)
   real(r8), pointer :: cpool_grain_storage_gr(:)            ! grain growth respiration to storage (gC/m2/s)
   real(r8), pointer :: transfer_grain_gr(:)                 ! grain growth respiration from storage (gC/m2/s)
#endif
   real(r8), pointer :: cpool_leaf_gr(:)                     ! leaf growth respiration (gC/m2/s)
   real(r8), pointer :: cpool_leaf_storage_gr(:)             ! leaf growth respiration to storage (gC/m2/s)
   real(r8), pointer :: transfer_leaf_gr(:)                  ! leaf growth respiration from storage (gC/m2/s)
   real(r8), pointer :: cpool_froot_gr(:)                    ! fine root growth respiration (gC/m2/s)
   real(r8), pointer :: cpool_froot_storage_gr(:)            ! fine root  growth respiration to storage (gC/m2/s)
   real(r8), pointer :: transfer_froot_gr(:)                 ! fine root  growth respiration from storage (gC/m2/s)
   real(r8), pointer :: cpool_livestem_gr(:)                 ! live stem growth respiration (gC/m2/s)
   real(r8), pointer :: cpool_livestem_storage_gr(:)         ! live stem growth respiration to storage (gC/m2/s)
   real(r8), pointer :: transfer_livestem_gr(:)              ! live stem growth respiration from storage (gC/m2/s)
   real(r8), pointer :: cpool_deadstem_gr(:)                 ! dead stem growth respiration (gC/m2/s)
   real(r8), pointer :: cpool_deadstem_storage_gr(:)         ! dead stem growth respiration to storage (gC/m2/s)
   real(r8), pointer :: transfer_deadstem_gr(:)              ! dead stem growth respiration from storage (gC/m2/s)
   real(r8), pointer :: cpool_livecroot_gr(:)                ! live coarse root growth respiration (gC/m2/s)
   real(r8), pointer :: cpool_livecroot_storage_gr(:)        ! live coarse root growth respiration to storage (gC/m2/s)
   real(r8), pointer :: transfer_livecroot_gr(:)             ! live coarse root growth respiration from storage (gC/m2/s)
   real(r8), pointer :: cpool_deadcroot_gr(:)                ! dead coarse root growth respiration (gC/m2/s)
   real(r8), pointer :: cpool_deadcroot_storage_gr(:)        ! dead coarse root growth respiration to storage (gC/m2/s)
   real(r8), pointer :: transfer_deadcroot_gr(:)             ! dead coarse root growth respiration from storage (gC/m2/s)
   ! annual turnover of storage to transfer pools            
#if (defined CROP)
   real(r8), pointer :: grainc_storage_to_xfer(:)            ! grain C shift storage to transfer (gC/m2/s)
#endif
   real(r8), pointer :: leafc_storage_to_xfer(:)             ! leaf C shift storage to transfer (gC/m2/s)
   real(r8), pointer :: frootc_storage_to_xfer(:)            ! fine root C shift storage to transfer (gC/m2/s)
   real(r8), pointer :: livestemc_storage_to_xfer(:)         ! live stem C shift storage to transfer (gC/m2/s)
   real(r8), pointer :: deadstemc_storage_to_xfer(:)         ! dead stem C shift storage to transfer (gC/m2/s)
   real(r8), pointer :: livecrootc_storage_to_xfer(:)        ! live coarse root C shift storage to transfer (gC/m2/s)
   real(r8), pointer :: deadcrootc_storage_to_xfer(:)        ! dead coarse root C shift storage to transfer (gC/m2/s)
   real(r8), pointer :: gresp_storage_to_xfer(:)             ! growth respiration shift storage to transfer (gC/m2/s)
   ! turnover of livewood to deadwood
   real(r8), pointer :: livestemc_to_deadstemc(:)            ! live stem C turnover (gC/m2/s)
   real(r8), pointer :: livecrootc_to_deadcrootc(:)          ! live coarse root C turnover (gC/m2/s)
   ! summary (diagnostic) flux variables, not involved in mass balance
   real(r8), pointer :: gpp(:)            ! (gC/m2/s) gross primary production 
   real(r8), pointer :: mr(:)             ! (gC/m2/s) maintenance respiration
   real(r8), pointer :: current_gr(:)     ! (gC/m2/s) growth resp for new growth displayed in this timestep
   real(r8), pointer :: transfer_gr(:)    ! (gC/m2/s) growth resp for transfer growth displayed in this timestep
   real(r8), pointer :: storage_gr(:)     ! (gC/m2/s) growth resp for growth sent to storage for later display
   real(r8), pointer :: gr(:)             ! (gC/m2/s) total growth respiration
   real(r8), pointer :: ar(:)             ! (gC/m2/s) autotrophic respiration (MR + GR)
   real(r8), pointer :: rr(:)             ! (gC/m2/s) root respiration (fine root MR + total root GR)
   real(r8), pointer :: npp(:)            ! (gC/m2/s) net primary production
   real(r8), pointer :: agnpp(:)          ! (gC/m2/s) aboveground NPP
   real(r8), pointer :: bgnpp(:)          ! (gC/m2/s) belowground NPP
   real(r8), pointer :: litfall(:)        ! (gC/m2/s) litterfall (leaves and fine roots)
   real(r8), pointer :: vegfire(:)        ! (gC/m2/s) pft-level fire loss (obsolete, mark for removal)
   real(r8), pointer :: wood_harvestc(:)  ! (gC/m2/s) pft-level wood harvest (to product pools)
   real(r8), pointer :: pft_cinputs(:)    ! (gC/m2/s) pft-level carbon inputs (for balance checking)
   real(r8), pointer :: pft_coutputs(:)   ! (gC/m2/s) pft-level carbon outputs (for balance checking)
#if (defined CLAMP) && (defined CN)
   ! CLAMP summary (diagnostic) variables, not involved in mass balance
   real(r8), pointer :: frootc_alloc(:)   ! (gC/m2/s) pft-level fine root C alloc
   real(r8), pointer :: frootc_loss(:)    ! (gC/m2/s) pft-level fine root C loss
   real(r8), pointer :: leafc_alloc(:)    ! (gC/m2/s) pft-level leaf C alloc
   real(r8), pointer :: leafc_loss(:)     ! (gC/m2/s) pft-level leaf C loss
   real(r8), pointer :: woodc_alloc(:)    ! (gC/m2/s) pft-level wood C alloc
   real(r8), pointer :: woodc_loss(:)     ! (gC/m2/s) pft-level wood C loss
#endif
   ! new variables for fire code
   real(r8), pointer :: pft_fire_closs(:) ! (gC/m2/s) total pft-level fire C loss 
end type pft_cflux_type

!----------------------------------------------------
! pft nitrogen flux variables structure
!----------------------------------------------------
type, public :: pft_nflux_type
   ! new variables for CN code
   ! gap mortality fluxes
   real(r8), pointer :: m_leafn_to_litter(:)                ! leaf N mortality (gN/m2/s)
   real(r8), pointer :: m_frootn_to_litter(:)               ! fine root N mortality (gN/m2/s)
   real(r8), pointer :: m_leafn_storage_to_litter(:)        ! leaf N storage mortality (gN/m2/s)
   real(r8), pointer :: m_frootn_storage_to_litter(:)       ! fine root N storage mortality (gN/m2/s)
   real(r8), pointer :: m_livestemn_storage_to_litter(:)    ! live stem N storage mortality (gN/m2/s)
   real(r8), pointer :: m_deadstemn_storage_to_litter(:)    ! dead stem N storage mortality (gN/m2/s)
   real(r8), pointer :: m_livecrootn_storage_to_litter(:)   ! live coarse root N storage mortality (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_storage_to_litter(:)   ! dead coarse root N storage mortality (gN/m2/s)
   real(r8), pointer :: m_leafn_xfer_to_litter(:)           ! leaf N transfer mortality (gN/m2/s)
   real(r8), pointer :: m_frootn_xfer_to_litter(:)          ! fine root N transfer mortality (gN/m2/s)
   real(r8), pointer :: m_livestemn_xfer_to_litter(:)       ! live stem N transfer mortality (gN/m2/s)
   real(r8), pointer :: m_deadstemn_xfer_to_litter(:)       ! dead stem N transfer mortality (gN/m2/s)
   real(r8), pointer :: m_livecrootn_xfer_to_litter(:)      ! live coarse root N transfer mortality (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_xfer_to_litter(:)      ! dead coarse root N transfer mortality (gN/m2/s)
   real(r8), pointer :: m_livestemn_to_litter(:)            ! live stem N mortality (gN/m2/s)
   real(r8), pointer :: m_deadstemn_to_litter(:)            ! dead stem N mortality (gN/m2/s)
   real(r8), pointer :: m_livecrootn_to_litter(:)           ! live coarse root N mortality (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_to_litter(:)           ! dead coarse root N mortality (gN/m2/s)
   real(r8), pointer :: m_retransn_to_litter(:)             ! retranslocated N pool mortality (gN/m2/s)
   ! harvest mortality fluxes
   real(r8), pointer :: hrv_leafn_to_litter(:)                ! leaf N harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_frootn_to_litter(:)               ! fine root N harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_leafn_storage_to_litter(:)        ! leaf N storage harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_frootn_storage_to_litter(:)       ! fine root N storage harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_livestemn_storage_to_litter(:)    ! live stem N storage harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_deadstemn_storage_to_litter(:)    ! dead stem N storage harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_livecrootn_storage_to_litter(:)   ! live coarse root N storage harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_deadcrootn_storage_to_litter(:)   ! dead coarse root N storage harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_leafn_xfer_to_litter(:)           ! leaf N transfer harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_frootn_xfer_to_litter(:)          ! fine root N transfer harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_livestemn_xfer_to_litter(:)       ! live stem N transfer harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_deadstemn_xfer_to_litter(:)       ! dead stem N transfer harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_livecrootn_xfer_to_litter(:)      ! live coarse root N transfer harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_deadcrootn_xfer_to_litter(:)      ! dead coarse root N transfer harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_livestemn_to_litter(:)            ! live stem N harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_deadstemn_to_prod10n(:)           ! dead stem N harvest to 10-year product pool (gN/m2/s)
   real(r8), pointer :: hrv_deadstemn_to_prod100n(:)          ! dead stem N harvest to 100-year product pool (gN/m2/s)
   real(r8), pointer :: hrv_livecrootn_to_litter(:)           ! live coarse root N harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_deadcrootn_to_litter(:)           ! dead coarse root N harvest mortality (gN/m2/s)
   real(r8), pointer :: hrv_retransn_to_litter(:)             ! retranslocated N pool harvest mortality (gN/m2/s)
   ! fire mortality fluxes
   real(r8), pointer :: m_leafn_to_fire(:)                  ! leaf N fire loss (gN/m2/s)
   real(r8), pointer :: m_leafn_storage_to_fire(:)          ! leaf N storage fire loss (gN/m2/s)
   real(r8), pointer :: m_leafn_xfer_to_fire(:)             ! leaf N transfer fire loss (gN/m2/s)
   real(r8), pointer :: m_frootn_to_fire(:)                 ! fine root N fire loss (gN/m2/s)
   real(r8), pointer :: m_frootn_storage_to_fire(:)         ! fine root N storage fire loss (gN/m2/s)
   real(r8), pointer :: m_frootn_xfer_to_fire(:)            ! fine root N transfer fire loss (gN/m2/s)
   real(r8), pointer :: m_livestemn_to_fire(:)              ! live stem N fire loss (gN/m2/s)
   real(r8), pointer :: m_livestemn_storage_to_fire(:)      ! live stem N storage fire loss (gN/m2/s)
   real(r8), pointer :: m_livestemn_xfer_to_fire(:)         ! live stem N transfer fire loss (gN/m2/s)
   real(r8), pointer :: m_deadstemn_to_fire(:)              ! dead stem N fire loss (gN/m2/s)
   real(r8), pointer :: m_deadstemn_to_litter_fire(:)       ! dead stem N fire mortality to litter (gN/m2/s)
   real(r8), pointer :: m_deadstemn_storage_to_fire(:)      ! dead stem N storage fire loss (gN/m2/s)
   real(r8), pointer :: m_deadstemn_xfer_to_fire(:)         ! dead stem N transfer fire loss (gN/m2/s)
   real(r8), pointer :: m_livecrootn_to_fire(:)             ! live coarse root N fire loss (gN/m2/s)
   real(r8), pointer :: m_livecrootn_storage_to_fire(:)     ! live coarse root N storage fire loss (gN/m2/s)
   real(r8), pointer :: m_livecrootn_xfer_to_fire(:)        ! live coarse root N transfer fire loss (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_to_fire(:)             ! dead coarse root N fire loss (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_to_litter_fire(:)      ! dead coarse root N fire mortality to litter (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_storage_to_fire(:)     ! dead coarse root N storage fire loss (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_xfer_to_fire(:)        ! dead coarse root N transfer fire loss (gN/m2/s)
   real(r8), pointer :: m_retransn_to_fire(:)               ! retranslocated N pool fire loss (gN/m2/s)
   ! phenology fluxes from transfer pool                     
#if (defined CROP)
   real(r8), pointer :: grainn_xfer_to_grainn(:)            ! grain N growth from storage (gN/m2/s)
#endif
   real(r8), pointer :: leafn_xfer_to_leafn(:)              ! leaf N growth from storage (gN/m2/s)
   real(r8), pointer :: frootn_xfer_to_frootn(:)            ! fine root N growth from storage (gN/m2/s)
   real(r8), pointer :: livestemn_xfer_to_livestemn(:)      ! live stem N growth from storage (gN/m2/s)
   real(r8), pointer :: deadstemn_xfer_to_deadstemn(:)      ! dead stem N growth from storage (gN/m2/s)
   real(r8), pointer :: livecrootn_xfer_to_livecrootn(:)    ! live coarse root N growth from storage (gN/m2/s)
   real(r8), pointer :: deadcrootn_xfer_to_deadcrootn(:)    ! dead coarse root N growth from storage (gN/m2/s)
   ! litterfall fluxes
#if (defined CROP)
   real(r8), pointer :: livestemn_to_litter(:)              ! livestem N to litter (gN/m2/s)
   real(r8), pointer :: grainn_to_food(:)                   ! grain N to food (gN/m2/s)
#endif
   real(r8), pointer :: leafn_to_litter(:)                  ! leaf N litterfall (gN/m2/s)
   real(r8), pointer :: leafn_to_retransn(:)                ! leaf N to retranslocated N pool (gN/m2/s)
   real(r8), pointer :: frootn_to_litter(:)                 ! fine root N litterfall (gN/m2/s)
   ! allocation fluxes
   real(r8), pointer :: retransn_to_npool(:)                ! deployment of retranslocated N (gN/m2/s)       
   real(r8), pointer :: sminn_to_npool(:)                   ! deployment of soil mineral N uptake (gN/m2/s)
#if (defined CROP)
   real(r8), pointer :: npool_to_grainn(:)                  ! allocation to grain N (gN/m2/s)
   real(r8), pointer :: npool_to_grainn_storage(:)          ! allocation to grain N storage (gN/m2/s)
#endif
   real(r8), pointer :: npool_to_leafn(:)                   ! allocation to leaf N (gN/m2/s)
   real(r8), pointer :: npool_to_leafn_storage(:)           ! allocation to leaf N storage (gN/m2/s)
   real(r8), pointer :: npool_to_frootn(:)                  ! allocation to fine root N (gN/m2/s)
   real(r8), pointer :: npool_to_frootn_storage(:)          ! allocation to fine root N storage (gN/m2/s)
   real(r8), pointer :: npool_to_livestemn(:)               ! allocation to live stem N (gN/m2/s)
   real(r8), pointer :: npool_to_livestemn_storage(:)       ! allocation to live stem N storage (gN/m2/s)
   real(r8), pointer :: npool_to_deadstemn(:)               ! allocation to dead stem N (gN/m2/s)
   real(r8), pointer :: npool_to_deadstemn_storage(:)       ! allocation to dead stem N storage (gN/m2/s)
   real(r8), pointer :: npool_to_livecrootn(:)              ! allocation to live coarse root N (gN/m2/s)
   real(r8), pointer :: npool_to_livecrootn_storage(:)      ! allocation to live coarse root N storage (gN/m2/s)
   real(r8), pointer :: npool_to_deadcrootn(:)              ! allocation to dead coarse root N (gN/m2/s)
   real(r8), pointer :: npool_to_deadcrootn_storage(:)      ! allocation to dead coarse root N storage (gN/m2/s)
   ! annual turnover of storage to transfer pools           
#if (defined CROP)
   real(r8), pointer :: grainn_storage_to_xfer(:)           ! grain N shift storage to transfer (gN/m2/s)
#endif
   real(r8), pointer :: leafn_storage_to_xfer(:)            ! leaf N shift storage to transfer (gN/m2/s)
   real(r8), pointer :: frootn_storage_to_xfer(:)           ! fine root N shift storage to transfer (gN/m2/s)
   real(r8), pointer :: livestemn_storage_to_xfer(:)        ! live stem N shift storage to transfer (gN/m2/s)
   real(r8), pointer :: deadstemn_storage_to_xfer(:)        ! dead stem N shift storage to transfer (gN/m2/s)
   real(r8), pointer :: livecrootn_storage_to_xfer(:)       ! live coarse root N shift storage to transfer (gN/m2/s)
   real(r8), pointer :: deadcrootn_storage_to_xfer(:)       ! dead coarse root N shift storage to transfer (gN/m2/s)
   ! turnover of livewood to deadwood, with retranslocation 
   real(r8), pointer :: livestemn_to_deadstemn(:)           ! live stem N turnover (gN/m2/s)
   real(r8), pointer :: livestemn_to_retransn(:)            ! live stem N to retranslocated N pool (gN/m2/s)
   real(r8), pointer :: livecrootn_to_deadcrootn(:)         ! live coarse root N turnover (gN/m2/s)
   real(r8), pointer :: livecrootn_to_retransn(:)           ! live coarse root N to retranslocated N pool (gN/m2/s)
   ! summary (diagnostic) flux variables, not involved in mass balance
   real(r8), pointer :: ndeploy(:)                          ! total N deployed to growth and storage (gN/m2/s)
   real(r8), pointer :: pft_ninputs(:)                      ! total N inputs to pft-level (gN/m2/s)
   real(r8), pointer :: pft_noutputs(:)                     ! total N outputs from pft-level (gN/m2/s)
   real(r8), pointer :: wood_harvestn(:)                    ! total N losses to wood product pools (gN/m2/s)
   ! new variables for fire code 
   real(r8), pointer :: pft_fire_nloss(:)                   ! total pft-level fire N loss (gN/m2/s) 
end type pft_nflux_type

!----------------------------------------------------
! pft VOC flux variables structure
!----------------------------------------------------
type, public :: pft_vflux_type
   real(r8), pointer :: vocflx_tot(:)     !total VOC flux into atmosphere [ug C m-2 h-1]
   real(r8), pointer :: vocflx(:,:)       !(nvoc) VOC flux [ug C m-2 h-1]
   real(r8), pointer :: vocflx_1(:)       !vocflx(1) (for history output) [ug C m-2 h-1]
   real(r8), pointer :: vocflx_2(:)       !vocflx(2) (for history output) [ug C m-2 h-1]
   real(r8), pointer :: vocflx_3(:)       !vocflx(3) (for history output) [ug C m-2 h-1]
   real(r8), pointer :: vocflx_4(:)       !vocflx(4) (for history output) [ug C m-2 h-1]
   real(r8), pointer :: vocflx_5(:)       !vocflx(5) (for history output) [ug C m-2 h-1]
   real(r8), pointer :: Eopt_out(:)       !Eopt coefficient
   real(r8), pointer :: topt_out(:)       !topt coefficient
   real(r8), pointer :: alpha_out(:)      !alpha coefficient
   real(r8), pointer :: cp_out(:)         !cp coefficient
   real(r8), pointer :: paru_out(:)
   real(r8), pointer :: par24u_out(:)
   real(r8), pointer :: par240u_out(:)
   real(r8), pointer :: para_out(:)
   real(r8), pointer :: par24a_out(:)
   real(r8), pointer :: par240a_out(:)
   real(r8), pointer :: gamma_out(:)
   real(r8), pointer :: gammaL_out(:)
   real(r8), pointer :: gammaT_out(:)
   real(r8), pointer :: gammaP_out(:)
   real(r8), pointer :: gammaA_out(:)
   real(r8), pointer :: gammaS_out(:)
end type pft_vflux_type

!----------------------------------------------------
! pft dry dep velocity variables structure
!----------------------------------------------------
type, public :: pft_depvd_type
   real(r8), pointer :: drydepvel(:,:)
end type pft_depvd_type

!----------------------------------------------------
! pft dust flux variables structure
!----------------------------------------------------
type, public :: pft_dflux_type
   real(r8), pointer :: flx_mss_vrt_dst(:,:)    !(ndst)  !surface dust emission (kg/m**2/s) [ + = to atm]
   real(r8), pointer :: flx_mss_vrt_dst_tot(:)  !total dust flux into atmosphere
   real(r8), pointer :: vlc_trb(:,:)            !(ndst) turbulent deposition velocity (m/s)
   real(r8), pointer :: vlc_trb_1(:)            !turbulent deposition velocity 1(m/s)
   real(r8), pointer :: vlc_trb_2(:)            !turbulent deposition velocity 2(m/s)
   real(r8), pointer :: vlc_trb_3(:)            !turbulent deposition velocity 3(m/s)
   real(r8), pointer :: vlc_trb_4(:)            !turbulent deposition velocity 4(m/s)
end type pft_dflux_type

!----------------------------------------------------
! End definition of structures defined at the pft_type level
!----------------------------------------------------
!*******************************************************************************


!*******************************************************************************
!----------------------------------------------------
! Begin definition of structures defined at the column_type level
!----------------------------------------------------
! column physical state variables structure
!----------------------------------------------------
type, public :: column_pstate_type
   type(pft_pstate_type) :: pps_a            !pft-level pstate variables averaged to the column
   integer , pointer :: snl(:)                !number of snow layers
   integer , pointer :: isoicol(:)            !soil color class
   real(r8), pointer :: bsw(:,:)              !Clapp and Hornberger "b" (nlevgrnd)  
   real(r8), pointer :: watsat(:,:)           !volumetric soil water at saturation (porosity) (nlevgrnd) 
   real(r8), pointer :: watdry(:,:)           !btran parameter for btran=0
   real(r8), pointer :: watopt(:,:)           !btran parameter for btran = 1
   real(r8), pointer :: hksat(:,:)            !hydraulic conductivity at saturation (mm H2O /s) (nlevgrnd) 
   real(r8), pointer :: sucsat(:,:)           !minimum soil suction (mm) (nlevgrnd) 
   real(r8), pointer :: hkdepth(:)            !decay factor (m)
   real(r8), pointer :: wtfact(:)             !maximum saturated fraction for a gridcell
   real(r8), pointer :: fracice(:,:)          !fractional impermeability (-)
   real(r8), pointer :: csol(:,:)             !heat capacity, soil solids (J/m**3/Kelvin) (nlevgrnd) 
   real(r8), pointer :: tkmg(:,:)             !thermal conductivity, soil minerals  [W/m-K] (new) (nlevgrnd) 
   real(r8), pointer :: tkdry(:,:)            !thermal conductivity, dry soil (W/m/Kelvin) (nlevgrnd) 
   real(r8), pointer :: tksatu(:,:)           !thermal conductivity, saturated soil [W/m-K] (new) (nlevgrnd) 
   real(r8), pointer :: smpmin(:)             !restriction for min of soil potential (mm) (new)
   real(r8), pointer :: gwc_thr(:)            !threshold soil moisture based on clay content
   real(r8), pointer :: mss_frc_cly_vld(:)    ![frc] Mass fraction clay limited to 0.20
   real(r8), pointer :: mbl_bsn_fct(:)        !basin factor
   logical , pointer :: do_capsnow(:)         !true => do snow capping
   real(r8), pointer :: snowdp(:)             !snow height (m)
   real(r8), pointer :: frac_sno(:)           !fraction of ground covered by snow (0 to 1)
   real(r8), pointer :: zi(:,:)               !interface level below a "z" level (m) (-nlevsno+0:nlevgrnd) 
   real(r8), pointer :: dz(:,:)               !layer thickness (m)  (-nlevsno+1:nlevgrnd) 
   real(r8), pointer :: z(:,:)                !layer depth (m) (-nlevsno+1:nlevgrnd) 
   real(r8), pointer :: frac_iceold(:,:)      !fraction of ice relative to the tot water (new) (-nlevsno+1:nlevgrnd) 
   integer , pointer :: imelt(:,:)            !flag for melting (=1), freezing (=2), Not=0 (new) (-nlevsno+1:nlevgrnd) 
   real(r8), pointer :: eff_porosity(:,:)     !effective porosity = porosity - vol_ice (nlevgrnd) 
   real(r8), pointer :: emg(:)                !ground emissivity
   real(r8), pointer :: z0mg(:)               !roughness length over ground, momentum [m]
   real(r8), pointer :: z0hg(:)               !roughness length over ground, sensible heat [m]
   real(r8), pointer :: z0qg(:)               !roughness length over ground, latent heat [m]
   real(r8), pointer :: htvp(:)               !latent heat of vapor of water (or sublimation) [j/kg]
   real(r8), pointer :: beta(:)               !coefficient of convective velocity [-]
   real(r8), pointer :: zii(:)                !convective boundary height [m]
   real(r8), pointer :: albgrd(:,:)           !ground albedo (direct) (numrad)
   real(r8), pointer :: albgri(:,:)           !ground albedo (diffuse) (numrad)
   real(r8), pointer :: rootr_column(:,:)     !effective fraction of roots in each soil layer (nlevgrnd)  
   real(r8), pointer :: rootfr_road_perv(:,:) !fraction of roots in each soil layer for urban pervious road
   real(r8), pointer :: rootr_road_perv(:,:)  !effective fraction of roots in each soil layer of urban pervious road
   real(r8), pointer :: wf(:)                 !soil water as frac. of whc for top 0.5 m
!  real(r8), pointer :: xirrig(:)             !irrigation rate
   real(r8), pointer :: max_dayl(:)           !maximum daylength for this column (s)
   ! new variables for CN code
   real(r8), pointer :: bsw2(:,:)        !Clapp and Hornberger "b" for CN code
   real(r8), pointer :: psisat(:,:)        !soil water potential at saturation for CN code (MPa)
   real(r8), pointer :: vwcsat(:,:)        !volumetric water content at saturation for CN code (m3/m3)
   real(r8), pointer :: decl(:)              ! solar declination angle (radians)
   real(r8), pointer :: coszen(:)            !cosine of solar zenith angle
   real(r8), pointer :: soilpsi(:,:)         !soil water potential in each soil layer (MPa)
   real(r8), pointer :: fpi(:)           !fraction of potential immobilization (no units)
   real(r8), pointer :: fpg(:)           !fraction of potential gpp (no units)
   real(r8), pointer :: annsum_counter(:) !seconds since last annual accumulator turnover
   real(r8), pointer :: cannsum_npp(:)    !annual sum of NPP, averaged from pft-level (gC/m2/yr)
   real(r8), pointer :: cannavg_t2m(:)    !annual average of 2m air temperature, averaged from pft-level (K)
   real(r8), pointer :: watfc(:,:)        !volumetric soil water at field capacity (nlevsoi)
   ! new variables for fire code
   real(r8), pointer :: me(:)                 !moisture of extinction (proportion) 
   real(r8), pointer :: fire_prob(:)          !daily fire probability (0-1) 
   real(r8), pointer :: mean_fire_prob(:)     !e-folding mean of daily fire probability (0-1) 
   real(r8), pointer :: fireseasonl(:)        !annual fire season length (days, <= 365) 
   real(r8), pointer :: farea_burned(:)       !timestep fractional area burned (proportion) 
   real(r8), pointer :: ann_farea_burned(:)   !annual total fractional area burned (proportion)
   real(r8), pointer :: albsnd_hst(:,:)       ! snow albedo, direct, for history files (col,bnd) [frc]
   real(r8), pointer :: albsni_hst(:,:)       ! snow albedo, diffuse, for history files (col,bnd) [frc]
   real(r8), pointer :: albsod(:,:)           ! soil albedo: direct (col,bnd) [frc]
   real(r8), pointer :: albsoi(:,:)           ! soil albedo: diffuse (col,bnd) [frc]
   real(r8), pointer :: flx_absdv(:,:)        ! absorbed flux per unit incident direct flux: VIS (col,lyr) [frc]
   real(r8), pointer :: flx_absdn(:,:)        ! absorbed flux per unit incident direct flux: NIR (col,lyr) [frc]
   real(r8), pointer :: flx_absiv(:,:)        ! absorbed flux per unit incident diffuse flux: VIS (col,lyr) [frc]
   real(r8), pointer :: flx_absin(:,:)        ! absorbed flux per unit incident diffuse flux: NIR (col,lyr) [frc]
   real(r8), pointer :: snw_rds(:,:)          ! snow grain radius (col,lyr) [m^-6, microns]
   real(r8), pointer :: snw_rds_top(:)        ! snow grain radius, top layer (col) [m^-6, microns]
   real(r8), pointer :: sno_liq_top(:)        ! snow liquid water fraction (mass), top layer (col) [fraction]
   real(r8), pointer :: mss_bcpho(:,:)        ! mass of hydrophobic BC in snow (col,lyr) [kg]
   real(r8), pointer :: mss_bcphi(:,:)        ! mass of hydrophillic BC in snow (col,lyr) [kg]
   real(r8), pointer :: mss_bctot(:,:)        ! total mass of BC in snow (pho+phi) (col,lyr) [kg]
   real(r8), pointer :: mss_bc_col(:)         ! column-integrated mass of total BC (col) [kg]
   real(r8), pointer :: mss_bc_top(:)         ! top-layer mass of total BC (col) [kg]
   real(r8), pointer :: mss_ocpho(:,:)        ! mass of hydrophobic OC in snow (col,lyr) [kg]
   real(r8), pointer :: mss_ocphi(:,:)        ! mass of hydrophillic OC in snow (col,lyr) [kg]
   real(r8), pointer :: mss_octot(:,:)        ! total mass of OC in snow (pho+phi) (col,lyr) [kg]
   real(r8), pointer :: mss_oc_col(:)         ! column-integrated mass of total OC (col) [kg]
   real(r8), pointer :: mss_oc_top(:)         ! top-layer mass of total OC (col) [kg]
   real(r8), pointer :: mss_dst1(:,:)         ! mass of dust species 1 in snow (col,lyr) [kg]
   real(r8), pointer :: mss_dst2(:,:)         ! mass of dust species 2 in snow (col,lyr) [kg]
   real(r8), pointer :: mss_dst3(:,:)         ! mass of dust species 3 in snow (col,lyr) [kg]
   real(r8), pointer :: mss_dst4(:,:)         ! mass of dust species 4 in snow (col,lyr) [kg]
   real(r8), pointer :: mss_dsttot(:,:)       ! total mass of dust in snow (col,lyr) [kg]
   real(r8), pointer :: mss_dst_col(:)        ! column-integrated mass of dust in snow (col) [kg]
   real(r8), pointer :: mss_dst_top(:)        ! top-layer mass of dust in snow (col) [kg]
   real(r8), pointer :: h2osno_top(:)         ! top-layer mass of snow (col) [kg]
   real(r8), pointer :: mss_cnc_bcphi(:,:)    ! mass concentration of hydrophilic BC in snow (col,lyr) [kg/kg]
   real(r8), pointer :: mss_cnc_bcpho(:,:)    ! mass concentration of hydrophilic BC in snow (col,lyr) [kg/kg]
   real(r8), pointer :: mss_cnc_ocphi(:,:)    ! mass concentration of hydrophilic OC in snow (col,lyr) [kg/kg]
   real(r8), pointer :: mss_cnc_ocpho(:,:)    ! mass concentration of hydrophilic OC in snow (col,lyr) [kg/kg]
   real(r8), pointer :: mss_cnc_dst1(:,:)     ! mass concentration of dust species 1 in snow (col,lyr) [kg/kg]
   real(r8), pointer :: mss_cnc_dst2(:,:)     ! mass concentration of dust species 2 in snow (col,lyr) [kg/kg]
   real(r8), pointer :: mss_cnc_dst3(:,:)     ! mass concentration of dust species 3 in snow (col,lyr) [kg/kg]
   real(r8), pointer :: mss_cnc_dst4(:,:)     ! mass concentration of dust species 4 in snow (col,lyr) [kg/kg]
   real(r8), pointer :: albgrd_pur(:,:)       ! pure snow ground direct albedo (numrad)
   real(r8), pointer :: albgri_pur(:,:)       ! pure snow ground diffuse albedo (numrad)
   real(r8), pointer :: albgrd_bc(:,:)        ! ground direct albedo without BC  (numrad)
   real(r8), pointer :: albgri_bc(:,:)        ! ground diffuse albedo without BC (numrad)
   real(r8), pointer :: albgrd_oc(:,:)        ! ground direct albedo without OC  (numrad)
   real(r8), pointer :: albgri_oc(:,:)        ! ground diffuse albedo without OC (numrad)
   real(r8), pointer :: albgrd_dst(:,:)       ! ground direct albedo without dust  (numrad)
   real(r8), pointer :: albgri_dst(:,:)       ! ground diffuse albedo without dust (numrad)
   real(r8), pointer :: dTdz_top(:)           ! temperature gradient in top layer  [K m-1]
   real(r8), pointer :: snot_top(:)           ! temperature of top snow layer [K]
end type column_pstate_type

!----------------------------------------------------
! column energy state variables structure
!----------------------------------------------------
type, public :: column_estate_type
   type(pft_estate_type):: pes_a              !pft-level energy state variables averaged to the column
   real(r8), pointer :: t_grnd(:)             !ground temperature (Kelvin)
   real(r8), pointer :: t_grnd_u(:)           !Urban ground temperature (Kelvin)
   real(r8), pointer :: t_grnd_r(:)           !Rural ground temperature (Kelvin)
   real(r8), pointer :: dt_grnd(:)            !change in t_grnd, last iteration (Kelvin)
   real(r8), pointer :: t_soisno(:,:)         !soil temperature (Kelvin)  (-nlevsno+1:nlevgrnd) 
   real(r8), pointer :: t_soi_10cm(:)         !soil temperature in top 10cm of soil (Kelvin)
   real(r8), pointer :: t_lake(:,:)           !lake temperature (Kelvin)  (1:nlevlak)          
   real(r8), pointer :: tssbef(:,:)           !soil/snow temperature before update (-nlevsno+1:nlevgrnd) 
   real(r8), pointer :: thv(:)                !virtual potential temperature (kelvin)
   real(r8), pointer :: hc_soi(:)             !soil heat content (MJ/m2)
   real(r8), pointer :: hc_soisno(:)          !soil plus snow heat content (MJ/m2)
end type column_estate_type

!----------------------------------------------------
! column water state variables structure
!----------------------------------------------------
type, public :: column_wstate_type
   type(pft_wstate_type):: pws_a             !pft-level water state variables averaged to the column
   real(r8), pointer :: h2osno(:)             !snow water (mm H2O)
   real(r8), pointer :: h2osoi_liq(:,:)       !liquid water (kg/m2) (new) (-nlevsno+1:nlevgrnd)    
   real(r8), pointer :: h2osoi_ice(:,:)       !ice lens (kg/m2) (new) (-nlevsno+1:nlevgrnd)    
   real(r8), pointer :: h2osoi_liqice_10cm(:) !liquid water + ice lens in top 10cm of soil (kg/m2)
   real(r8), pointer :: h2osoi_vol(:,:)       !volumetric soil water (0<=h2osoi_vol<=watsat) [m3/m3]  (nlevgrnd)  
   real(r8), pointer :: h2osno_old(:)         !snow mass for previous time step (kg/m2) (new)
   real(r8), pointer :: qg(:)                 !ground specific humidity [kg/kg]
   real(r8), pointer :: dqgdT(:)              !d(qg)/dT
   real(r8), pointer :: snowice(:)            !average snow ice lens
   real(r8), pointer :: snowliq(:)            !average snow liquid water
   real(r8) ,pointer :: soilalpha(:)          !factor that reduces ground saturated specific humidity (-)
   real(r8), pointer :: soilbeta(:)           !factor that reduces ground evaporation L&P1992(-)
   real(r8) ,pointer :: soilalpha_u(:)        !urban factor that reduces ground saturated specific humidity (-)
   real(r8), pointer :: zwt(:)                !water table depth
   real(r8), pointer :: fcov(:)               !fractional impermeable area
   real(r8), pointer :: wa(:)                 !water in the unconfined aquifer (mm)
   real(r8), pointer :: wt(:)                 !total water storage (unsaturated soil water + groundwater) (mm)
   real(r8), pointer :: qcharge(:)            !aquifer recharge rate (mm/s)
   real(r8), pointer :: smp_l(:,:)            !soil matric potential (mm)
   real(r8), pointer :: hk_l(:,:)             !hydraulic conductivity (mm/s)
   real(r8), pointer :: fsat(:)               !fractional area with water table at surface
end type column_wstate_type

!----------------------------------------------------
! column carbon state variables structure
!----------------------------------------------------
type, public :: column_cstate_type
   type(pft_cstate_type):: pcs_a              !pft-level carbon state variables averaged to the column
   ! NOTE: the soilc variable is used by the original CLM C-cycle code,
   ! and is not used by the CN code
   real(r8), pointer :: soilc(:)              !soil carbon (kg C /m**2)
   ! BGC variables
   real(r8), pointer :: cwdc(:)               ! (gC/m2) coarse woody debris C
   real(r8), pointer :: litr1c(:)             ! (gC/m2) litter labile C
   real(r8), pointer :: litr2c(:)             ! (gC/m2) litter cellulose C
   real(r8), pointer :: litr3c(:)             ! (gC/m2) litter lignin C
   real(r8), pointer :: soil1c(:)             ! (gC/m2) soil organic matter C (fast pool)
   real(r8), pointer :: soil2c(:)             ! (gC/m2) soil organic matter C (medium pool)
   real(r8), pointer :: soil3c(:)             ! (gC/m2) soil organic matter C (slow pool)
   real(r8), pointer :: soil4c(:)             ! (gC/m2) soil organic matter C (slowest pool)
   real(r8), pointer :: col_ctrunc(:)         ! (gC/m2) column-level sink for C truncation
   ! pools for dynamic landcover
   real(r8), pointer :: seedc(:)              ! (gC/m2) column-level pool for seeding new PFTs
   real(r8), pointer :: prod10c(:)            ! (gC/m2) wood product C pool, 10-year lifespan
   real(r8), pointer :: prod100c(:)           ! (gC/m2) wood product C pool, 100-year lifespan
   real(r8), pointer :: totprodc(:)           ! (gC/m2) total wood product C
   ! summary (diagnostic) state variables, not involved in mass balance
   real(r8), pointer :: totlitc(:)            ! (gC/m2) total litter carbon
   real(r8), pointer :: totsomc(:)            ! (gC/m2) total soil organic matter carbon
   real(r8), pointer :: totecosysc(:)         ! (gC/m2) total ecosystem carbon, incl veg but excl cpool
   real(r8), pointer :: totcolc(:)            ! (gC/m2) total column carbon, incl veg and cpool
   
end type column_cstate_type

!----------------------------------------------------
! column nitrogen state variables structure
!----------------------------------------------------
type, public :: column_nstate_type
   type(pft_nstate_type):: pns_a              !pft-level nitrogen state variables averaged to the column
   ! BGC variables
   real(r8), pointer :: cwdn(:)               ! (gN/m2) coarse woody debris N
   real(r8), pointer :: litr1n(:)             ! (gN/m2) litter labile N
   real(r8), pointer :: litr2n(:)             ! (gN/m2) litter cellulose N
   real(r8), pointer :: litr3n(:)             ! (gN/m2) litter lignin N
   real(r8), pointer :: soil1n(:)             ! (gN/m2) soil organic matter N (fast pool)
   real(r8), pointer :: soil2n(:)             ! (gN/m2) soil organic matter N (medium pool)
   real(r8), pointer :: soil3n(:)             ! (gN/m2) soil orgainc matter N (slow pool)
   real(r8), pointer :: soil4n(:)             ! (gN/m2) soil orgainc matter N (slowest pool)
   real(r8), pointer :: sminn(:)              ! (gN/m2) soil mineral N
   real(r8), pointer :: col_ntrunc(:)         ! (gN/m2) column-level sink for N truncation
   ! wood product pools, for dynamic landcover
   real(r8), pointer :: seedn(:)              ! (gN/m2) column-level pool for seeding new PFTs
   real(r8), pointer :: prod10n(:)            ! (gN/m2) wood product N pool, 10-year lifespan
   real(r8), pointer :: prod100n(:)           ! (gN/m2) wood product N pool, 100-year lifespan
   real(r8), pointer :: totprodn(:)           ! (gN/m2) total wood product N
   ! summary (diagnostic) state variables, not involved in mass balance
   real(r8), pointer :: totlitn(:)            ! (gN/m2) total litter nitrogen
   real(r8), pointer :: totsomn(:)            ! (gN/m2) total soil organic matter nitrogen
   real(r8), pointer :: totecosysn(:)         ! (gN/m2) total ecosystem nitrogen, incl veg 
   real(r8), pointer :: totcoln(:)            ! (gN/m2) total column nitrogen, incl veg
end type column_nstate_type

!----------------------------------------------------
! column VOC state variables structure
!----------------------------------------------------
type, public :: column_vstate_type
   type(pft_vstate_type):: pvs_a              !pft-level VOC state variables averaged to the column
end type column_vstate_type

#if (defined CNDV)
!----------------------------------------------------
! column DGVM state variables structure
!----------------------------------------------------
type, public :: column_dgvstate_type
   type(pft_dgvstate_type):: pdgvs_a
end type column_dgvstate_type
#endif

!----------------------------------------------------
! column dust state variables structure
!----------------------------------------------------
type, public :: column_dstate_type
   real(r8), pointer :: dummy_entry(:)
end type column_dstate_type

!----------------------------------------------------
! column energy flux variables structure
!----------------------------------------------------
type, public :: column_eflux_type
   type(pft_eflux_type):: pef_a	              ! pft-level energy flux variables averaged to the column
   real(r8), pointer :: eflx_snomelt(:)       ! snow melt heat flux (W/m**2)
   real(r8), pointer :: eflx_snomelt_u(:)     ! urban snow melt heat flux (W/m**2)
   real(r8), pointer :: eflx_snomelt_r(:)     ! rural snow melt heat flux (W/m**2)
   real(r8), pointer :: eflx_impsoil(:)	      ! implicit evaporation for soil temperature equation
   real(r8), pointer :: eflx_fgr12(:)         ! ground heat flux between soil layers 1 and 2 (W/m2)
   ! Urban variable
   real(r8), pointer :: eflx_building_heat(:) ! heat flux from urban building interior to urban walls, roof (W/m**2)
   real(r8), pointer :: eflx_urban_ac(:)      ! urban air conditioning flux (W/m**2)
   real(r8), pointer :: eflx_urban_heat(:)    ! urban heating flux (W/m**2)
end type column_eflux_type

!----------------------------------------------------
! column momentum flux variables structure
!----------------------------------------------------
type, public :: column_mflux_type
   type(pft_mflux_type)::  pmf_a        ! pft-level momentum flux variables averaged to the column
end type column_mflux_type

!----------------------------------------------------
! column water flux variables structure
!----------------------------------------------------
type, public :: column_wflux_type
   type(pft_wflux_type):: pwf_a	        ! pft-level water flux variables averaged to the column
   real(r8), pointer :: qflx_infl(:)	! infiltration (mm H2O /s)
   real(r8), pointer :: qflx_surf(:)	! surface runoff (mm H2O /s)
   real(r8), pointer :: qflx_drain(:) 	! sub-surface runoff (mm H2O /s)
   real(r8), pointer :: qflx_top_soil(:)! net water input into soil from top (mm/s)
   real(r8), pointer :: qflx_snomelt(:) ! snow melt (mm H2O /s)
   real(r8), pointer :: qflx_qrgwl(:) 	! qflx_surf at glaciers, wetlands, lakes
   real(r8), pointer :: qflx_runoff(:) 	! total runoff (qflx_drain+qflx_surf+qflx_qrgwl) (mm H2O /s)
   real(r8), pointer :: qflx_runoff_u(:)! Urban total runoff (qflx_drain+qflx_surf) (mm H2O /s)
   real(r8), pointer :: qflx_runoff_r(:)! Rural total runoff (qflx_drain+qflx_surf+qflx_qrgwl) (mm H2O /s)
   real(r8), pointer :: qmelt(:) 	! snow melt [mm/s]
   real(r8), pointer :: h2ocan_loss(:)  ! mass balance correction term for dynamic weights
   real(r8), pointer :: qflx_rsub_sat(:)    ! soil saturation excess [mm/s]
   real(r8), pointer :: flx_bc_dep_dry(:)   ! dry (BCPHO+BCPHI) BC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_bc_dep_wet(:)   ! wet (BCPHI) BC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_bc_dep_pho(:)   ! hydrophobic BC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_bc_dep_phi(:)   ! hydrophillic BC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_bc_dep(:)       ! total (dry+wet) BC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_oc_dep_dry(:)   ! dry (OCPHO+OCPHI) OC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_oc_dep_wet(:)   ! wet (OCPHI) OC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_oc_dep_pho(:)   ! hydrophobic OC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_oc_dep_phi(:)   ! hydrophillic OC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_oc_dep(:)       ! total (dry+wet) OC deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep_dry1(:) ! dust species 1 dry deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep_wet1(:) ! dust species 1 wet deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep_dry2(:) ! dust species 2 dry deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep_wet2(:) ! dust species 2 wet deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep_dry3(:) ! dust species 3 dry deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep_wet3(:) ! dust species 3 wet deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep_dry4(:) ! dust species 4 dry deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep_wet4(:) ! dust species 4 wet deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: flx_dst_dep(:)      ! total (dry+wet) dust deposition on ground (positive definite) (col) [kg/s]
   real(r8), pointer :: qflx_snofrz_lyr(:,:)! snow freezing rate (positive definite) (col,lyr) [kg m-2 s-1]
end type column_wflux_type

!----------------------------------------------------
! column carbon flux variables structure
!----------------------------------------------------
type, public :: column_cflux_type
   type(pft_cflux_type):: pcf_a                           !pft-level carbon flux variables averaged to the column
   ! new variables for CN code
   ! column-level gap mortality fluxes
   real(r8), pointer :: m_leafc_to_litr1c(:)              ! leaf C mortality to litter 1 C (gC/m2/s) 
   real(r8), pointer :: m_leafc_to_litr2c(:)              ! leaf C mortality to litter 2 C (gC/m2/s)
   real(r8), pointer :: m_leafc_to_litr3c(:)              ! leaf C mortality to litter 3 C (gC/m2/s)
   real(r8), pointer :: m_frootc_to_litr1c(:)             ! fine root C mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_frootc_to_litr2c(:)             ! fine root C mortality to litter 2 C (gC/m2/s)
   real(r8), pointer :: m_frootc_to_litr3c(:)             ! fine root C mortality to litter 3 C (gC/m2/s)
   real(r8), pointer :: m_livestemc_to_cwdc(:)            ! live stem C mortality to coarse woody debris C (gC/m2/s)
   real(r8), pointer :: m_deadstemc_to_cwdc(:)            ! dead stem C mortality to coarse woody debris C (gC/m2/s)
   real(r8), pointer :: m_livecrootc_to_cwdc(:)           ! live coarse root C mortality to coarse woody debris C (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_to_cwdc(:)           ! dead coarse root C mortality to coarse woody debris C (gC/m2/s)
   real(r8), pointer :: m_leafc_storage_to_litr1c(:)      ! leaf C storage mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_frootc_storage_to_litr1c(:)     ! fine root C storage mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_livestemc_storage_to_litr1c(:)  ! live stem C storage mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_deadstemc_storage_to_litr1c(:)  ! dead stem C storage mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_livecrootc_storage_to_litr1c(:) ! live coarse root C storage mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_storage_to_litr1c(:) ! dead coarse root C storage mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_gresp_storage_to_litr1c(:)      ! growth respiration storage mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_leafc_xfer_to_litr1c(:)         ! leaf C transfer mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_frootc_xfer_to_litr1c(:)        ! fine root C transfer mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_livestemc_xfer_to_litr1c(:)     ! live stem C transfer mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_deadstemc_xfer_to_litr1c(:)     ! dead stem C transfer mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_livecrootc_xfer_to_litr1c(:)    ! live coarse root C transfer mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_xfer_to_litr1c(:)    ! dead coarse root C transfer mortality to litter 1 C (gC/m2/s)
   real(r8), pointer :: m_gresp_xfer_to_litr1c(:)         ! growth respiration transfer mortality to litter 1 C (gC/m2/s)
   ! column-level harvest mortality fluxes
   real(r8), pointer :: hrv_leafc_to_litr1c(:)               ! leaf C harvest mortality to litter 1 C (gC/m2/s)                         
   real(r8), pointer :: hrv_leafc_to_litr2c(:)               ! leaf C harvest mortality to litter 2 C (gC/m2/s)                        
   real(r8), pointer :: hrv_leafc_to_litr3c(:)               ! leaf C harvest mortality to litter 3 C (gC/m2/s)                        
   real(r8), pointer :: hrv_frootc_to_litr1c(:)              ! fine root C harvest mortality to litter 1 C (gC/m2/s)                   
   real(r8), pointer :: hrv_frootc_to_litr2c(:)              ! fine root C harvest mortality to litter 2 C (gC/m2/s)                   
   real(r8), pointer :: hrv_frootc_to_litr3c(:)              ! fine root C harvest mortality to litter 3 C (gC/m2/s)                   
   real(r8), pointer :: hrv_livestemc_to_cwdc(:)             ! live stem C harvest mortality to coarse woody debris C (gC/m2/s)        
   real(r8), pointer :: hrv_deadstemc_to_prod10c(:)          ! dead stem C harvest mortality to 10-year product pool (gC/m2/s)        
   real(r8), pointer :: hrv_deadstemc_to_prod100c(:)         ! dead stem C harvest mortality to 100-year product pool (gC/m2/s)        
   real(r8), pointer :: hrv_livecrootc_to_cwdc(:)            ! live coarse root C harvest mortality to coarse woody debris C (gC/m2/s) 
   real(r8), pointer :: hrv_deadcrootc_to_cwdc(:)            ! dead coarse root C harvest mortality to coarse woody debris C (gC/m2/s) 
   real(r8), pointer :: hrv_leafc_storage_to_litr1c(:)       ! leaf C storage harvest mortality to litter 1 C (gC/m2/s)                
   real(r8), pointer :: hrv_frootc_storage_to_litr1c(:)      ! fine root C storage harvest mortality to litter 1 C (gC/m2/s)           
   real(r8), pointer :: hrv_livestemc_storage_to_litr1c(:)   ! live stem C storage harvest mortality to litter 1 C (gC/m2/s)           
   real(r8), pointer :: hrv_deadstemc_storage_to_litr1c(:)   ! dead stem C storage harvest mortality to litter 1 C (gC/m2/s)           
   real(r8), pointer :: hrv_livecrootc_storage_to_litr1c(:)  ! live coarse root C storage harvest mortality to litter 1 C (gC/m2/s)    
   real(r8), pointer :: hrv_deadcrootc_storage_to_litr1c(:)  ! dead coarse root C storage harvest mortality to litter 1 C (gC/m2/s)    
   real(r8), pointer :: hrv_gresp_storage_to_litr1c(:)       ! growth respiration storage harvest mortality to litter 1 C (gC/m2/s)    
   real(r8), pointer :: hrv_leafc_xfer_to_litr1c(:)          ! leaf C transfer harvest mortality to litter 1 C (gC/m2/s)               
   real(r8), pointer :: hrv_frootc_xfer_to_litr1c(:)         ! fine root C transfer harvest mortality to litter 1 C (gC/m2/s)          
   real(r8), pointer :: hrv_livestemc_xfer_to_litr1c(:)      ! live stem C transfer harvest mortality to litter 1 C (gC/m2/s)          
   real(r8), pointer :: hrv_deadstemc_xfer_to_litr1c(:)      ! dead stem C transfer harvest mortality to litter 1 C (gC/m2/s)          
   real(r8), pointer :: hrv_livecrootc_xfer_to_litr1c(:)     ! live coarse root C transfer harvest mortality to litter 1 C (gC/m2/s)   
   real(r8), pointer :: hrv_deadcrootc_xfer_to_litr1c(:)     ! dead coarse root C transfer harvest mortality to litter 1 C (gC/m2/s)   
   real(r8), pointer :: hrv_gresp_xfer_to_litr1c(:)          ! growth respiration transfer harvest mortality to litter 1 C (gC/m2/s)   
   ! column-level fire fluxes
   real(r8), pointer :: m_deadstemc_to_cwdc_fire(:)       ! dead stem C to coarse woody debris C by fire (gC/m2/s)
   real(r8), pointer :: m_deadcrootc_to_cwdc_fire(:)      ! dead coarse root C to to woody debris C by fire (gC/m2/s)
   real(r8), pointer :: m_litr1c_to_fire(:)               ! litter 1 C fire loss (gC/m2/s)
   real(r8), pointer :: m_litr2c_to_fire(:)               ! litter 2 C fire loss (gC/m2/s)
   real(r8), pointer :: m_litr3c_to_fire(:)               ! litter 3 C fire loss (gC/m2/s)
   real(r8), pointer :: m_cwdc_to_fire(:)                 ! coarse woody debris C fire loss (gC/m2/s)
   ! litterfall fluxes
#if (defined CROP)
   real(r8), pointer :: grainc_to_litr1c(:)               ! grain C litterfall to litter 1 C (gC/m2/s)
   real(r8), pointer :: grainc_to_litr2c(:)               ! grain C litterfall to litter 2 C (gC/m2/s)
   real(r8), pointer :: grainc_to_litr3c(:)               ! grain C litterfall to litter 3 C (gC/m2/s)
   real(r8), pointer :: livestemc_to_litr1c(:)            ! livestem C litterfall to litter 1 C (gC/m2/s)
   real(r8), pointer :: livestemc_to_litr2c(:)            ! livestem C litterfall to litter 2 C (gC/m2/s)
   real(r8), pointer :: livestemc_to_litr3c(:)            ! livestem C litterfall to litter 3 C (gC/m2/s)
#endif
   real(r8), pointer :: leafc_to_litr1c(:)                ! leaf C litterfall to litter 1 C (gC/m2/s)
   real(r8), pointer :: leafc_to_litr2c(:)                ! leaf C litterfall to litter 2 C (gC/m2/s)
   real(r8), pointer :: leafc_to_litr3c(:)                ! leaf C litterfall to litter 3 C (gC/m2/s)
   real(r8), pointer :: frootc_to_litr1c(:)               ! fine root C litterfall to litter 1 C (gC/m2/s)
   real(r8), pointer :: frootc_to_litr2c(:)               ! fine root C litterfall to litter 2 C (gC/m2/s)
   real(r8), pointer :: frootc_to_litr3c(:)               ! fine root C litterfall to litter 3 C (gC/m2/s)
   ! decomposition fluxes
   real(r8), pointer :: cwdc_to_litr2c(:)     ! decomp. of coarse woody debris C to litter 2 C (gC/m2/s)
   real(r8), pointer :: cwdc_to_litr3c(:)     ! decomp. of coarse woody debris C to litter 3 C (gC/m2/s)
   real(r8), pointer :: litr1_hr(:)           ! het. resp. from litter 1 C (gC/m2/s)
   real(r8), pointer :: litr1c_to_soil1c(:)   ! decomp. of litter 1 C to SOM 1 C (gC/m2/s)
   real(r8), pointer :: litr2_hr(:)           ! het. resp. from litter 2 C (gC/m2/s)
   real(r8), pointer :: litr2c_to_soil2c(:)   ! decomp. of litter 2 C to SOM 2 C (gC/m2/s)
   real(r8), pointer :: litr3_hr(:)           ! het. resp. from litter 3 C (gC/m2/s)
   real(r8), pointer :: litr3c_to_soil3c(:)   ! decomp. of litter 3 C to SOM 3 C (gC/m2/s)
   real(r8), pointer :: soil1_hr(:)           ! het. resp. from SOM 1 C (gC/m2/s)
   real(r8), pointer :: soil1c_to_soil2c(:)   ! decomp. of SOM 1 C to SOM 2 C (gC/m2/s)
   real(r8), pointer :: soil2_hr(:)           ! het. resp. from SOM 2 C (gC/m2/s)
   real(r8), pointer :: soil2c_to_soil3c(:)   ! decomp. of SOM 2 C to SOM 3 C (gC/m2/s)
   real(r8), pointer :: soil3_hr(:)           ! het. resp. from SOM 3 C (gC/m2/s)
   real(r8), pointer :: soil3c_to_soil4c(:)   ! decomp. of SOM 3 C to SOM 4 C (gC/m2/s)
   real(r8), pointer :: soil4_hr(:)           ! het. resp. from SOM 4 C (gC/m2/s)
   ! dynamic landcover fluxes
#ifdef CN
   real(r8), pointer :: dwt_seedc_to_leaf(:)      ! (gC/m2/s) seed source to PFT-level
   real(r8), pointer :: dwt_seedc_to_deadstem(:)  ! (gC/m2/s) seed source to PFT-level
   real(r8), pointer :: dwt_conv_cflux(:)         ! (gC/m2/s) conversion C flux (immediate loss to atm)
   real(r8), pointer :: dwt_prod10c_gain(:)       ! (gC/m2/s) addition to 10-yr wood product pool
   real(r8), pointer :: dwt_prod100c_gain(:)      ! (gC/m2/s) addition to 100-yr wood product pool
   real(r8), pointer :: dwt_frootc_to_litr1c(:)   ! (gC/m2/s) fine root to litter due to landcover change
   real(r8), pointer :: dwt_frootc_to_litr2c(:)   ! (gC/m2/s) fine root to litter due to landcover change
   real(r8), pointer :: dwt_frootc_to_litr3c(:)   ! (gC/m2/s) fine root to litter due to landcover change
   real(r8), pointer :: dwt_livecrootc_to_cwdc(:) ! (gC/m2/s) live coarse root to CWD due to landcover change
   real(r8), pointer :: dwt_deadcrootc_to_cwdc(:) ! (gC/m2/s) dead coarse root to CWD due to landcover change
   real(r8), pointer :: dwt_closs(:)              ! (gC/m2/s) total carbon loss from product pools and conversion
   real(r8), pointer :: landuseflux(:)            ! (gC/m2/s) dwt_closs+product_closs
   real(r8), pointer :: landuptake(:)             ! (gC/m2/s) nee-landuseflux
   ! wood product pool loss fluxes
   real(r8), pointer :: prod10c_loss(:)           ! (gC/m2/s) decomposition loss from 10-yr wood product pool
   real(r8), pointer :: prod100c_loss(:)          ! (gC/m2/s) decomposition loss from 100-yr wood product pool
   real(r8), pointer :: product_closs(:)          ! (gC/m2/s) total wood product carbon loss
#endif
   ! summary (diagnostic) flux variables, not involved in mass balance
   real(r8), pointer :: lithr(:)         ! (gC/m2/s) litter heterotrophic respiration 
   real(r8), pointer :: somhr(:)         ! (gC/m2/s) soil organic matter heterotrophic respiration
   real(r8), pointer :: hr(:)            ! (gC/m2/s) total heterotrophic respiration
   real(r8), pointer :: sr(:)            ! (gC/m2/s) total soil respiration (HR + root resp)
   real(r8), pointer :: er(:)            ! (gC/m2/s) total ecosystem respiration, autotrophic + heterotrophic
   real(r8), pointer :: litfire(:)       ! (gC/m2/s) litter fire losses
   real(r8), pointer :: somfire(:)       ! (gC/m2/s) soil organic matter fire losses
   real(r8), pointer :: totfire(:)       ! (gC/m2/s) total ecosystem fire losses
   real(r8), pointer :: nep(:)           ! (gC/m2/s) net ecosystem production, excludes fire, landuse, and harvest flux, positive for sink
   real(r8), pointer :: nbp(:)           ! (gC/m2/s) net biome production, includes fire, landuse, and harvest flux, positive for sink
   real(r8), pointer :: nee(:)           ! (gC/m2/s) net ecosystem exchange of carbon, includes fire, landuse, harvest, and hrv_xsmrpool flux, positive for source
   real(r8), pointer :: col_cinputs(:)   ! (gC/m2/s) total column-level carbon inputs (for balance check)
   real(r8), pointer :: col_coutputs(:)  ! (gC/m2/s) total column-level carbon outputs (for balance check) 

#if (defined CLAMP) && (defined CN)
   ! CLAMP summary (diagnostic) flux variables, not involved in mass balance
   real(r8), pointer :: cwdc_hr(:)       ! (gC/m2/s) col-level coarse woody debris C heterotrophic respiration
   real(r8), pointer :: cwdc_loss(:)     ! (gC/m2/s) col-level coarse woody debris C loss
   real(r8), pointer :: litterc_loss(:)  ! (gC/m2/s) col-level litter C loss
#endif

   ! new variables for fire
   real(r8), pointer :: col_fire_closs(:) ! (gC/m2/s) total column-level fire C loss
end type column_cflux_type

!----------------------------------------------------
! column nitrogen flux variables structure
!----------------------------------------------------
type, public :: column_nflux_type
   type(pft_nflux_type):: pnf_a        !pft-level nitrogen flux variables averaged to the column
   ! new variables for CN code
   ! deposition fluxes
   real(r8), pointer :: ndep_to_sminn(:)                   ! atmospheric N deposition to soil mineral N (gN/m2/s)
   real(r8), pointer :: nfix_to_sminn(:)                   ! symbiotic/asymbiotic N fixation to soil mineral N (gN/m2/s) 
   ! column-level gap mortality fluxes
   real(r8), pointer :: m_leafn_to_litr1n(:)               ! leaf N mortality to litter 1 N (gC/m2/s)
   real(r8), pointer :: m_leafn_to_litr2n(:)               ! leaf N mortality to litter 2 N (gC/m2/s)
   real(r8), pointer :: m_leafn_to_litr3n(:)               ! leaf N mortality to litter 3 N (gC/m2/s)
   real(r8), pointer :: m_frootn_to_litr1n(:)              ! fine root N mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_frootn_to_litr2n(:)              ! fine root N mortality to litter 2 N (gN/m2/s)
   real(r8), pointer :: m_frootn_to_litr3n(:)              ! fine root N mortality to litter 3 N (gN/m2/s)
   real(r8), pointer :: m_livestemn_to_cwdn(:)             ! live stem N mortality to coarse woody debris N (gN/m2/s)
   real(r8), pointer :: m_deadstemn_to_cwdn(:)             ! dead stem N mortality to coarse woody debris N (gN/m2/s)
   real(r8), pointer :: m_livecrootn_to_cwdn(:)            ! live coarse root N mortality to coarse woody debris N (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_to_cwdn(:)            ! dead coarse root N mortality to coarse woody debris N (gN/m2/s)
   real(r8), pointer :: m_retransn_to_litr1n(:)            ! retranslocated N pool mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_leafn_storage_to_litr1n(:)       ! leaf N storage mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_frootn_storage_to_litr1n(:)      ! fine root N storage mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_livestemn_storage_to_litr1n(:)   ! live stem N storage mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_deadstemn_storage_to_litr1n(:)   ! dead stem N storage mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_livecrootn_storage_to_litr1n(:)  ! live coarse root N storage mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_storage_to_litr1n(:)  ! dead coarse root N storage mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_leafn_xfer_to_litr1n(:)          ! leaf N transfer mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_frootn_xfer_to_litr1n(:)         ! fine root N transfer mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_livestemn_xfer_to_litr1n(:)      ! live stem N transfer mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_deadstemn_xfer_to_litr1n(:)      ! dead stem N transfer mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_livecrootn_xfer_to_litr1n(:)     ! live coarse root N transfer mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_xfer_to_litr1n(:)     ! dead coarse root N transfer mortality to litter 1 N (gN/m2/s)
   ! column-level harvest fluxes
   real(r8), pointer :: hrv_leafn_to_litr1n(:)               ! leaf N harvest mortality to litter 1 N (gC/m2/s)
   real(r8), pointer :: hrv_leafn_to_litr2n(:)               ! leaf N harvest mortality to litter 2 N (gC/m2/s)
   real(r8), pointer :: hrv_leafn_to_litr3n(:)               ! leaf N harvest mortality to litter 3 N (gC/m2/s)
   real(r8), pointer :: hrv_frootn_to_litr1n(:)              ! fine root N harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_frootn_to_litr2n(:)              ! fine root N harvest mortality to litter 2 N (gN/m2/s)
   real(r8), pointer :: hrv_frootn_to_litr3n(:)              ! fine root N harvest mortality to litter 3 N (gN/m2/s)
   real(r8), pointer :: hrv_livestemn_to_cwdn(:)             ! live stem N harvest mortality to coarse woody debris N (gN/m2/s)
   real(r8), pointer :: hrv_deadstemn_to_prod10n(:)          ! dead stem N harvest mortality to 10-year product pool (gN/m2/s)
   real(r8), pointer :: hrv_deadstemn_to_prod100n(:)         ! dead stem N harvest mortality to 100-year product pool (gN/m2/s)
   real(r8), pointer :: hrv_livecrootn_to_cwdn(:)            ! live coarse root N harvest mortality to coarse woody debris N (gN/m2/s)
   real(r8), pointer :: hrv_deadcrootn_to_cwdn(:)            ! dead coarse root N harvest mortality to coarse woody debris N (gN/m2/s)
   real(r8), pointer :: hrv_retransn_to_litr1n(:)            ! retranslocated N pool harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_leafn_storage_to_litr1n(:)       ! leaf N storage harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_frootn_storage_to_litr1n(:)      ! fine root N storage harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_livestemn_storage_to_litr1n(:)   ! live stem N storage harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_deadstemn_storage_to_litr1n(:)   ! dead stem N storage harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_livecrootn_storage_to_litr1n(:)  ! live coarse root N storage harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_deadcrootn_storage_to_litr1n(:)  ! dead coarse root N storage harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_leafn_xfer_to_litr1n(:)          ! leaf N transfer harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_frootn_xfer_to_litr1n(:)         ! fine root N transfer harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_livestemn_xfer_to_litr1n(:)      ! live stem N transfer harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_deadstemn_xfer_to_litr1n(:)      ! dead stem N transfer harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_livecrootn_xfer_to_litr1n(:)     ! live coarse root N transfer harvest mortality to litter 1 N (gN/m2/s)
   real(r8), pointer :: hrv_deadcrootn_xfer_to_litr1n(:)     ! dead coarse root N transfer harvest mortality to litter 1 N (gN/m2/s)
   ! column-level fire fluxes
   real(r8), pointer :: m_deadstemn_to_cwdn_fire(:)        ! dead stem N to coarse woody debris N by fire (gN/m2/s)
   real(r8), pointer :: m_deadcrootn_to_cwdn_fire(:)       ! dead coarse root N to to woody debris N by fire (gN/m2/s)
   real(r8), pointer :: m_litr1n_to_fire(:)                ! litter 1 N fire loss (gN/m2/s)
   real(r8), pointer :: m_litr2n_to_fire(:)                ! litter 2 N fire loss (gN/m2/s)
   real(r8), pointer :: m_litr3n_to_fire(:)                ! litter 3 N fire loss (gN/m2/s)
   real(r8), pointer :: m_cwdn_to_fire(:)                  ! coarse woody debris N fire loss (gN/m2/s)
   ! litterfall fluxes
#if (defined CROP)
   real(r8), pointer :: livestemn_to_litr1n(:)   ! livestem N litterfall to litter 1 N (gN/m2/s)
   real(r8), pointer :: livestemn_to_litr2n(:)   ! livestem N litterfall to litter 2 N (gN/m2/s)
   real(r8), pointer :: livestemn_to_litr3n(:)   ! livestem N litterfall to litter 3 N (gN/m2/s)
   real(r8), pointer :: grainn_to_litr1n(:)      ! grain N litterfall to litter 1 N (gN/m2/s)
   real(r8), pointer :: grainn_to_litr2n(:)      ! grain N litterfall to litter 2 N (gN/m2/s)
   real(r8), pointer :: grainn_to_litr3n(:)      ! grain N litterfall to litter 3 N (gN/m2/s)
#endif
   real(r8), pointer :: leafn_to_litr1n(:)       ! leaf N litterfall to litter 1 N (gN/m2/s)
   real(r8), pointer :: leafn_to_litr2n(:)       ! leaf N litterfall to litter 2 N (gN/m2/s)
   real(r8), pointer :: leafn_to_litr3n(:)       ! leaf N litterfall to litter 3 N (gN/m2/s)
   real(r8), pointer :: frootn_to_litr1n(:)      ! fine root N litterfall to litter 1 N (gN/m2/s)
   real(r8), pointer :: frootn_to_litr2n(:)      ! fine root N litterfall to litter 2 N (gN/m2/s)
   real(r8), pointer :: frootn_to_litr3n(:)      ! fine root N litterfall to litter 3 N (gN/m2/s)
   ! decomposition fluxes
   real(r8), pointer :: cwdn_to_litr2n(:)        ! decomp. of coarse woody debris N to litter 2 N (gN/m2/s)
   real(r8), pointer :: cwdn_to_litr3n(:)        ! decomp. of coarse woody debris N to litter 3 N (gN/m2/s)
   real(r8), pointer :: litr1n_to_soil1n(:)      ! decomp. of litter 1 N to SOM 1 N (gN/m2/s)
   real(r8), pointer :: sminn_to_soil1n_l1(:)    ! mineral N flux for decomp. of litter 1 to SOM 1 (gN/m2/s)
   real(r8), pointer :: litr2n_to_soil2n(:)      ! decomp. of litter 2 N to SOM 2 N (gN/m2/s)
   real(r8), pointer :: sminn_to_soil2n_l2(:)    ! mineral N flux for decomp. of litter 2 to SOM 2 (gN/m2/s)
   real(r8), pointer :: litr3n_to_soil3n(:)      ! decomp. of litter 3 N to SOM 3 N (gN/m2/s)
   real(r8), pointer :: sminn_to_soil3n_l3(:)    ! mineral N flux for decomp. of litter 3 to SOM 3 (gN/m2/s)
   real(r8), pointer :: soil1n_to_soil2n(:)      ! decomp. of SOM 1 N to SOM 2 N (gN/m2/s)
   real(r8), pointer :: sminn_to_soil2n_s1(:)    ! mineral N flux for decomp. of SOM 1 to SOM 2 (gN/m2/s)
   real(r8), pointer :: soil2n_to_soil3n(:)      ! decomp. of SOM 2 N to SOM 3 N (gN/m2/s)
   real(r8), pointer :: sminn_to_soil3n_s2(:)    ! mineral N flux for decomp. of SOM 2 to SOM 3 (gN/m2/s)
   real(r8), pointer :: soil3n_to_soil4n(:)      ! decomp. of SOM 3 N to SOM 4 N (gN/m2/s)
   real(r8), pointer :: sminn_to_soil4n_s3(:)    ! mineral N flux for decomp. of SOM 3 to SOM 4 (gN/m2/s)
   real(r8), pointer :: soil4n_to_sminn(:)       ! N mineralization for decomp. of SOM 4 (gN/m2/s)
   ! denitrification fluxes
   real(r8), pointer :: sminn_to_denit_l1s1(:)   ! denitrification for decomp. of litter 1 to SOM 1 (gN/m2/s) 
   real(r8), pointer :: sminn_to_denit_l2s2(:)   ! denitrification for decomp. of litter 2 to SOM 2 (gN/m2/s)
   real(r8), pointer :: sminn_to_denit_l3s3(:)   ! denitrification for decomp. of litter 3 to SOM 3 (gN/m2/s)
   real(r8), pointer :: sminn_to_denit_s1s2(:)   ! denitrification for decomp. of SOM 1 to SOM 2 (gN/m2/s)
   real(r8), pointer :: sminn_to_denit_s2s3(:)   ! denitrification for decomp. of SOM 2 to SOM 3 (gN/m2/s)
   real(r8), pointer :: sminn_to_denit_s3s4(:)   ! denitrification for decomp. of SOM 3 to SOM 4 (gN/m2/s)
   real(r8), pointer :: sminn_to_denit_s4(:)     ! denitrification for decomp. of SOM 4 (gN/m2/s)
   real(r8), pointer :: sminn_to_denit_excess(:) ! denitrification from excess mineral N pool (gN/m2/s)
   ! leaching fluxes
   real(r8), pointer :: sminn_leached(:)         ! soil mineral N pool loss to leaching (gN/m2/s)
   ! dynamic landcover fluxes
   real(r8), pointer :: dwt_seedn_to_leaf(:)      ! (gN/m2/s) seed source to PFT-level
   real(r8), pointer :: dwt_seedn_to_deadstem(:)  ! (gN/m2/s) seed source to PFT-level
   real(r8), pointer :: dwt_conv_nflux(:)         ! (gN/m2/s) conversion N flux (immediate loss to atm)
   real(r8), pointer :: dwt_prod10n_gain(:)       ! (gN/m2/s) addition to 10-yr wood product pool
   real(r8), pointer :: dwt_prod100n_gain(:)      ! (gN/m2/s) addition to 100-yr wood product pool
   real(r8), pointer :: dwt_frootn_to_litr1n(:)   ! (gN/m2/s) fine root to litter due to landcover change
   real(r8), pointer :: dwt_frootn_to_litr2n(:)   ! (gN/m2/s) fine root to litter due to landcover change
   real(r8), pointer :: dwt_frootn_to_litr3n(:)   ! (gN/m2/s) fine root to litter due to landcover change
   real(r8), pointer :: dwt_livecrootn_to_cwdn(:) ! (gN/m2/s) live coarse root to CWD due to landcover change
   real(r8), pointer :: dwt_deadcrootn_to_cwdn(:) ! (gN/m2/s) dead coarse root to CWD due to landcover change
   real(r8), pointer :: dwt_nloss(:)              ! (gN/m2/s) total nitrogen loss from product pools and conversion
   ! wood product pool loss fluxes
   real(r8), pointer :: prod10n_loss(:)           ! (gN/m2/s) decomposition loss from 10-yr wood product pool
   real(r8), pointer :: prod100n_loss(:)          ! (gN/m2/s) decomposition loss from 100-yr wood product pool
   real(r8), pointer :: product_nloss(:)          ! (gN/m2/s) total wood product nitrogen loss
   ! summary (diagnostic) flux variables, not involved in mass balance
   real(r8), pointer :: potential_immob(:)       ! potential N immobilization (gN/m2/s)
   real(r8), pointer :: actual_immob(:)          ! actual N immobilization (gN/m2/s)
   real(r8), pointer :: sminn_to_plant(:)        ! plant uptake of soil mineral N (gN/m2/s)
   real(r8), pointer :: supplement_to_sminn(:)   ! supplemental N supply (gN/m2/s)
   real(r8), pointer :: gross_nmin(:)            ! gross rate of N mineralization (gN/m2/s)
   real(r8), pointer :: net_nmin(:)              ! net rate of N mineralization (gN/m2/s)
   real(r8), pointer :: denit(:)                 ! total rate of denitrification (gN/m2/s)
   real(r8), pointer :: col_ninputs(:)           ! column-level N inputs (gN/m2/s)
   real(r8), pointer :: col_noutputs(:)          ! column-level N outputs (gN/m2/s)
   ! new variables for fire
   real(r8), pointer :: col_fire_nloss(:)        ! total column-level fire N loss (gN/m2/s)
end type column_nflux_type

!----------------------------------------------------
! column VOC flux variables structure
!----------------------------------------------------
type, public :: column_vflux_type
   type(pft_vflux_type):: pvf_a         !pft-level VOC flux variables averaged to the column
end type column_vflux_type

!----------------------------------------------------
! column dust flux variables structure
!----------------------------------------------------
type, public :: column_dflux_type
   type(pft_dflux_type):: pdf_a         !pft-level dust flux variables averaged to the column
end type column_dflux_type

!----------------------------------------------------
! End definition of structures defined at the column_type level
!----------------------------------------------------
!*******************************************************************************


!*******************************************************************************
!----------------------------------------------------
! Begin definition of structures defined at the landunit_type level
!----------------------------------------------------
! landunit physical state variables structure
! note - landunit type can be vegetated (includes bare soil), deep lake, 
! shallow lake, wetland, glacier or urban
!----------------------------------------------------
type, public :: landunit_pstate_type
   type(column_pstate_type):: cps_a            !column-level physical state variables averaged to landunit
   ! Urban variables
   real(r8), pointer :: t_building(:)         ! internal building temperature (K)
   real(r8), pointer :: t_building_max(:)     ! maximum internal building temperature (K)
   real(r8), pointer :: t_building_min(:)     ! minimum internal building temperature (K)
   real(r8), pointer :: tk_wall(:,:)          ! thermal conductivity of urban wall (W/m/K)
   real(r8), pointer :: tk_roof(:,:)          ! thermal conductivity of urban roof (W/m/K)
   real(r8), pointer :: tk_improad(:,:)       ! thermal conductivity of urban impervious road (W/m/K)
   real(r8), pointer :: cv_wall(:,:)          ! heat capacity of urban wall (J/m^3/K)
   real(r8), pointer :: cv_roof(:,:)          ! heat capacity of urban roof (J/m^3/K)
   real(r8), pointer :: cv_improad(:,:)       ! heat capacity of urban impervious road (J/m^3/K)
   real(r8), pointer :: thick_wall(:)         ! total thickness of urban wall (m)
   real(r8), pointer :: thick_roof(:)         ! total thickness of urban roof (m)
   integer, pointer :: nlev_improad(:)        ! number of impervious road layers (-)
   real(r8), pointer :: vf_sr(:)              ! view factor of sky for road
   real(r8), pointer :: vf_wr(:)              ! view factor of one wall for road
   real(r8), pointer :: vf_sw(:)              ! view factor of sky for one wall
   real(r8), pointer :: vf_rw(:)              ! view factor of road for one wall
   real(r8), pointer :: vf_ww(:)              ! view factor of opposing wall for one wall
   real(r8), pointer :: taf(:)                ! urban canopy air temperature (K)
   real(r8), pointer :: qaf(:)                ! urban canopy air specific humidity (kg/kg)
   real(r8), pointer :: sabs_roof_dir(:,:)       ! direct solar absorbed by roof per unit ground area per unit incident flux
   real(r8), pointer :: sabs_roof_dif(:,:)       ! diffuse solar absorbed by roof per unit ground area per unit incident flux
   real(r8), pointer :: sabs_sunwall_dir(:,:)    ! direct  solar absorbed by sunwall per unit wall area per unit incident flux
   real(r8), pointer :: sabs_sunwall_dif(:,:)    ! diffuse solar absorbed by sunwall per unit wall area per unit incident flux
   real(r8), pointer :: sabs_shadewall_dir(:,:)  ! direct  solar absorbed by shadewall per unit wall area per unit incident flux
   real(r8), pointer :: sabs_shadewall_dif(:,:)  ! diffuse solar absorbed by shadewall per unit wall area per unit incident flux
   real(r8), pointer :: sabs_improad_dir(:,:)    ! direct  solar absorbed by impervious road per unit ground area per unit incident flux
   real(r8), pointer :: sabs_improad_dif(:,:)    ! diffuse solar absorbed by impervious road per unit ground area per unit incident flux
   real(r8), pointer :: sabs_perroad_dir(:,:)    ! direct  solar absorbed by pervious road per unit ground area per unit incident flux
   real(r8), pointer :: sabs_perroad_dif(:,:)    ! diffuse solar absorbed by pervious road per unit ground area per unit incident flux
end type landunit_pstate_type

!----------------------------------------------------
! landunit energy state variables structure
!----------------------------------------------------
type, public :: landunit_estate_type
   type(column_estate_type):: ces_a            !column-level energy state variables averaged to landunit
end type landunit_estate_type

!----------------------------------------------------
! landunit water state variables structure
!----------------------------------------------------
type, public :: landunit_wstate_type
   type(column_wstate_type):: cws_a            !column-level water state variables averaged to landunit
end type landunit_wstate_type

!----------------------------------------------------
! landunit carbon state variables structure
!----------------------------------------------------
type, public :: landunit_cstate_type
   type(column_cstate_type):: ccs_a            !column-level carbon state variables averaged to landunit
end type landunit_cstate_type

!----------------------------------------------------
! landunit nitrogen state variables structure
!----------------------------------------------------
type, public :: landunit_nstate_type
   type(column_nstate_type):: cns_a            !column-level nitrogen state variables averaged to landunit
end type landunit_nstate_type

!----------------------------------------------------
! landunit VOC state variables structure
!----------------------------------------------------
type, public :: landunit_vstate_type
   real(r8):: dummy_entry
end type landunit_vstate_type

!----------------------------------------------------
! landunit DGVM state variables structure
!----------------------------------------------------
type, public :: landunit_dgvstate_type
   real(r8):: dummy_entry
end type landunit_dgvstate_type

!----------------------------------------------------
! landunit dust state variables structure
!----------------------------------------------------
type, public :: landunit_dstate_type
   type(column_dstate_type):: cds_a            !column-level dust state variables averaged to landunit
end type landunit_dstate_type

!----------------------------------------------------
! landunit energy flux variables structure
!----------------------------------------------------
type, public :: landunit_eflux_type
   type(column_eflux_type)::	cef_a		! column-level energy flux variables averaged to landunit
   ! Urban variables
   real(r8), pointer :: eflx_traffic_factor(:)  ! multiplicative traffic factor for sensible heat flux from urban traffic (-)
   real(r8), pointer :: eflx_traffic(:)         ! traffic sensible heat flux (W/m**2)
   real(r8), pointer :: eflx_wasteheat(:)       ! sensible heat flux from domestic heating/cooling sources of waste heat (W/m**2)
   real(r8), pointer :: eflx_heat_from_ac(:)    ! sensible heat flux to be put back into canyon due to removal by AC (W/m**2)
end type landunit_eflux_type

!----------------------------------------------------
! landunit momentum flux variables structure
!----------------------------------------------------
type, public :: landunit_mflux_type
   type(pft_mflux_type):: pmf_a                !pft-level momentum flux variables averaged to landunit
end type landunit_mflux_type

!----------------------------------------------------
! landunit water flux variables structure
!----------------------------------------------------
type, public :: landunit_wflux_type
   type(column_wflux_type):: cwf_a             !column-level water flux variables averaged to landunit
end type landunit_wflux_type

!----------------------------------------------------
! landunit carbon flux variables structure
!----------------------------------------------------
type, public :: landunit_cflux_type
   type(column_cflux_type):: ccf_a             !column-level carbon flux variables averaged to landunit
end type landunit_cflux_type

!----------------------------------------------------
! landunit nitrogen flux variables structure
!----------------------------------------------------
type, public :: landunit_nflux_type
   type(column_nflux_type):: cnf_a             !column-level nitrogen flux variables averaged to landunit
end type landunit_nflux_type

!----------------------------------------------------
! landunit VOC flux variables structure
!----------------------------------------------------
type, public :: landunit_vflux_type
   type(pft_vflux_type):: pvf_a                !pft-level VOC flux variables averaged to landunit
end type landunit_vflux_type

!----------------------------------------------------
! landunit dust flux variables structure
!----------------------------------------------------
type, public :: landunit_dflux_type
   type(pft_dflux_type):: pdf_a                !pft-level dust flux variables averaged to landunit
end type landunit_dflux_type

!----------------------------------------------------
! End definition of structures defined at the landunit_type level
!----------------------------------------------------
!*******************************************************************************


!*******************************************************************************
!----------------------------------------------------
! Begin definition of structures defined at the gridcell_type level
!----------------------------------------------------
! gridcell physical state variables structure
!----------------------------------------------------
type, public :: gridcell_pstate_type
   type(column_pstate_type):: cps_a   !column-level physical state variables averaged to gridcell
end type gridcell_pstate_type

!----------------------------------------------------
! gridcell energy state variables structure
!----------------------------------------------------
type, public :: gridcell_estate_type
   type(column_estate_type):: ces_a            !column-level energy state variables averaged to gridcell
   real(r8), pointer :: gc_heat1(:)            ! initial gridcell total heat content
   real(r8), pointer :: gc_heat2(:)            ! post land cover change total heat content
end type gridcell_estate_type

!----------------------------------------------------
! gridcell water state variables structure
!----------------------------------------------------
type, public :: gridcell_wstate_type
   type(column_wstate_type):: cws_a            !column-level water state variables averaged to gridcell
   real(r8), pointer :: gc_liq1(:)             ! initial gridcell total h2o liq content
   real(r8), pointer :: gc_liq2(:)             ! post land cover change total liq content
   real(r8), pointer :: gc_ice1(:)             ! initial gridcell total h2o liq content
   real(r8), pointer :: gc_ice2(:)             ! post land cover change total ice content
end type gridcell_wstate_type

!----------------------------------------------------
! gridcell carbon state variables structure
!----------------------------------------------------
type, public :: gridcell_cstate_type
   type(column_cstate_type):: ccs_a            !column-level carbon state variables averaged to gridcell
end type gridcell_cstate_type

!----------------------------------------------------
! gridcell nitrogen state variables structure
!----------------------------------------------------
type, public :: gridcell_nstate_type
   type(column_nstate_type):: cns_a            !column-level nitrogen state variables averaged to gridcell
end type gridcell_nstate_type

!----------------------------------------------------
! gridcell VOC state variables structure
!----------------------------------------------------
type, public :: gridcell_vstate_type
   type(column_vstate_type):: cvs_a            !column-level VOC state variables averaged to gridcell
end type gridcell_vstate_type

!----------------------------------------------------
! gridcell VOC emission factor variables structure (heald)
!----------------------------------------------------
type, public :: gridcell_efstate_type
   real(r8), pointer      :: efisop(:,:)    ! isoprene emission factors
end type gridcell_efstate_type

!----------------------------------------------------
! gridcell dust state variables structure
!----------------------------------------------------
type, public :: gridcell_dstate_type
   type(column_dstate_type):: cds_a            !column-level dust state variables averaged to gridcell
end type gridcell_dstate_type

#if (defined CNDV)
!----------------------------------------------------
! gridcell DGVM state variables structure
!----------------------------------------------------
type, public :: gridcell_dgvstate_type
   real(r8), pointer :: agdd20(:)      !20-yr running mean of agdd
   real(r8), pointer :: tmomin20(:)    !20-yr running mean of tmomin
   real(r8), pointer :: t10min(:)      !ann minimum of 10-day running mean (K)
end type gridcell_dgvstate_type
#endif

!----------------------------------------------------
! gridcell energy flux variables structure
!----------------------------------------------------
type, public :: gridcell_eflux_type
   type(column_eflux_type):: cef_a             !column-level energy flux variables averaged to gridcell
   real(r8), pointer :: eflx_sh_totg(:)   ! total grid-level sensible heat flux
   real(r8), pointer :: eflx_dynbal(:)    ! dynamic land cover change conversion energy flux
end type gridcell_eflux_type

!----------------------------------------------------
! gridcell momentum flux variables structure
!-- -------------------------------------------------
type, public :: gridcell_mflux_type
   type(pft_mflux_type):: pmf_a                !pft-level momentum flux variables averaged to gridcell
end type gridcell_mflux_type

!----------------------------------------------------
! gridcell water flux variables structure
!----------------------------------------------------
type, public :: gridcell_wflux_type
   type(column_wflux_type):: cwf_a             !column-level water flux variables averaged to gridcell
   real(r8), pointer :: qflx_runoffg(:)      ! total grid-level liq runoff
   real(r8), pointer :: qflx_snwcp_iceg(:)   ! total grid-level ice runoff
   real(r8), pointer :: qflx_liq_dynbal(:)   ! liq dynamic land cover change conversion runoff flux
   real(r8), pointer :: qflx_ice_dynbal(:)   ! ice dynamic land cover change conversion runoff flux
end type gridcell_wflux_type

!----------------------------------------------------
! gridcell carbon flux variables structure
!----------------------------------------------------
type, public :: gridcell_cflux_type
   type(column_cflux_type):: ccf_a             !column-level carbon flux variables averaged to gridcell
end type gridcell_cflux_type

!----------------------------------------------------
! gridcell nitrogen flux variables structure
!----------------------------------------------------
type, public :: gridcell_nflux_type
   type(column_nflux_type):: cnf_a             !column-level nitrogen flux variables averaged to gridcell
end type gridcell_nflux_type

!----------------------------------------------------
! gridcell VOC flux variables structure
!----------------------------------------------------
type, public :: gridcell_vflux_type
   type(pft_vflux_type):: pvf_a                !pft-level VOC flux variables averaged to gridcell
end type gridcell_vflux_type

!----------------------------------------------------
! gridcell dust flux variables structure
!----------------------------------------------------
type, public :: gridcell_dflux_type
   type(pft_dflux_type):: pdf_a                !pft-level dust flux variables averaged to gridcell
end type gridcell_dflux_type

!----------------------------------------------------
! End definition of structures defined at the gridcell_type level
!----------------------------------------------------
!*******************************************************************************


!*******************************************************************************
!----------------------------------------------------
! Begin definition of structures defined at the CLM level
!----------------------------------------------------
! CLM physical state variables structure
!----------------------------------------------------
type, public :: model_pstate_type
   type(column_pstate_type) :: cps_a           !column-level physical state variables globally averaged
end type model_pstate_type

!----------------------------------------------------
! CLM energy state variables structure
!----------------------------------------------------
type, public :: model_estate_type
   type(column_estate_type):: ces_a            !column-level energy state variables globally averaged
end type model_estate_type

!----------------------------------------------------
! CLM water state variables structure
!----------------------------------------------------
type, public :: model_wstate_type
   type(column_wstate_type):: cws_a            !column-level water state variables globally averaged
end type model_wstate_type

!----------------------------------------------------
! CLM carbon state variables structure
!----------------------------------------------------
type, public :: model_cstate_type
   type(column_cstate_type):: ccs_a            !column-level carbon state variables globally averaged
end type model_cstate_type

!----------------------------------------------------
! CLM nitrogen state variables structure
!----------------------------------------------------
type, public :: model_nstate_type
   type(column_nstate_type):: cns_a            !column-level nitrogen state variables globally averaged
end type model_nstate_type

!----------------------------------------------------
! CLM VOC state variables structure
!----------------------------------------------------
type, public :: model_vstate_type
   type(column_vstate_type):: cvs_a            !column-level VOC state variables globally averaged
end type model_vstate_type

!----------------------------------------------------
! CLM dust state variables structure
!----------------------------------------------------
type, public :: model_dstate_type
   type(column_dstate_type):: cds_a            !column-level dust state variables globally averaged
end type model_dstate_type

!----------------------------------------------------
! CLM energy flux variables structure
!----------------------------------------------------
type, public :: model_eflux_type
   type(column_eflux_type):: cef_a             !column-level energy flux variables globally averaged
end type model_eflux_type

!----------------------------------------------------
! CLM momentum flux variables structure
!----------------------------------------------------
type, public :: model_mflux_type
   type(pft_mflux_type):: pmf_a                !pft-level momentum flux variables globally averaged
end type model_mflux_type

!----------------------------------------------------
! CLM water flux variables structure
!----------------------------------------------------
type, public :: model_wflux_type
   type(column_wflux_type):: cwf_a             !column-level water flux variables globally averaged
end type model_wflux_type

!----------------------------------------------------
! CLM carbon flux variables structure
!----------------------------------------------------
type, public :: model_cflux_type
   type(column_cflux_type):: ccf_a             !column-level carbon flux variables globally averaged
end type model_cflux_type

!----------------------------------------------------
! CLM nitrogen flux variables structure
!----------------------------------------------------
type, public :: model_nflux_type
   type(column_nflux_type):: cnf_a             !column-level nitrogen flux variables globally averaged
end type model_nflux_type

!----------------------------------------------------
! CLM VOC flux variables structure
!----------------------------------------------------
type, public :: model_vflux_type
   type(pft_vflux_type):: pvf_a                !pft-level VOC flux variables globally averaged
end type model_vflux_type

!----------------------------------------------------
! CLM dust flux variables structure
!----------------------------------------------------
type, public :: model_dflux_type
   type(pft_dflux_type):: pdf_a                !pft-level dust flux variables globally averaged
end type model_dflux_type

!----------------------------------------------------
! End definition of structures defined at the model_type level
!----------------------------------------------------

!*******************************************************************************
!----------------------------------------------------
! Begin definition of spatial scaling hierarchy
!----------------------------------------------------

!----------------------------------------------------
! define the pft structure
!----------------------------------------------------

type, public :: pft_type

   ! g/l/c/p hierarchy, local g/l/c/p cells only
   integer, pointer :: column(:)        !index into column level quantities
   real(r8), pointer :: wtcol(:)        !weight (relative to column) 
   integer, pointer :: landunit(:)      !index into landunit level quantities
   real(r8), pointer :: wtlunit(:)      !weight (relative to landunit) 
   integer, pointer :: gridcell(:)      !index into gridcell level quantities
   real(r8), pointer :: wtgcell(:)      !weight (relative to gridcell) 

   ! topological mapping functionality
   integer , pointer :: itype(:)        !pft vegetation 
   integer , pointer :: mxy(:)          !m index for laixy(i,j,m),etc.
   real(r8), pointer :: area(:)         !total land area for this pft (km^2)

   ! conservation check structures for the pft level
   type(energy_balance_type)   :: pebal !energy balance structure
   type(water_balance_type)    :: pwbal !water balance structure
   type(carbon_balance_type)   :: pcbal !carbon balance structure
   type(nitrogen_balance_type) :: pnbal !nitrogen balance structure
   
#if (defined CNDV) || (defined CROP)
   ! DGVM state variables
   type(pft_dgvstate_type) :: pdgvs     !pft DGVM state variables
#endif
   
   ! CN ecophysiological variables
   type(pft_epv_type)    :: pepv        !pft ecophysiological variables
   
   ! state variables defined at the pft level
   type(pft_pstate_type) :: pps         !physical state variables
   type(pft_estate_type) :: pes         !pft energy state
   type(pft_wstate_type) :: pws         !pft water state
   type(pft_cstate_type) :: pcs         !pft carbon state
   type(pft_nstate_type) :: pns         !pft nitrogen state
   type(pft_vstate_type) :: pvs         !pft VOC state

   ! flux variables defined at the pft level
   type(pft_eflux_type)  :: pef         !pft energy flux
   type(pft_mflux_type)  :: pmf         !pft momentum flux
   type(pft_wflux_type)  :: pwf         !pft water flux
   type(pft_cflux_type)  :: pcf         !pft carbon flux
   type(pft_nflux_type)  :: pnf         !pft nitrogen flux
   type(pft_vflux_type)  :: pvf         !pft VOC flux
   type(pft_dflux_type)  :: pdf         !pft dust flux
   type(pft_depvd_type)  :: pdd         !dry dep velocity
   
#if (defined C13)
   ! 4/14/05: PET
   ! Adding isotope code
   type(pft_cstate_type) :: pc13s       !pft carbon-13 state
   type(pft_cflux_type)  :: pc13f       !pft carbon-13 flux
#endif
   
end type pft_type

!----------------------------------------------------
! define the column structure
!----------------------------------------------------

type, public :: column_type

   type(pft_type)   :: p       !plant functional type (pft) data structure 

   ! g/l/c/p hierarchy, local g/l/c/p cells only
   integer , pointer :: landunit(:)     !index into landunit level quantities
   real(r8), pointer :: wtlunit(:)      !weight (relative to landunit)
   integer , pointer :: gridcell(:)     !index into gridcell level quantities
   real(r8), pointer :: wtgcell(:)      !weight (relative to gridcell)
   integer , pointer :: pfti(:)         !beginning pft index for each column
   integer , pointer :: pftf(:)         !ending pft index for each column
   integer , pointer :: npfts(:)        !number of pfts for each column
   
   ! topological mapping functionality
   integer , pointer :: itype(:)        !column type
   real(r8), pointer :: area(:)         !total land area for this column (km^2)

   ! conservation check structures for the column level
   type(energy_balance_type)   :: cebal !energy balance structure
   type(water_balance_type)    :: cwbal !water balance structure
   type(carbon_balance_type)   :: ccbal !carbon balance structure
   type(nitrogen_balance_type) :: cnbal !nitrogen balance structure
   
   ! state variables defined at the column level
   type(column_pstate_type) :: cps      !column physical state variables
   type(column_estate_type) :: ces      !column energy state
   type(column_wstate_type) :: cws      !column water state
   type(column_cstate_type) :: ccs      !column carbon state
   type(column_nstate_type) :: cns      !column nitrogen state
   type(column_dstate_type) :: cds      !column dust state
   
   ! flux variables defined at the column level
   type(column_eflux_type) :: cef       !column energy flux
   type(column_mflux_type) :: cmf       !column momentum flux
   type(column_wflux_type) :: cwf       !column water flux
   type(column_cflux_type) :: ccf       !column carbon flux
   type(column_nflux_type) :: cnf       !column nitrogen flux
   type(column_vflux_type) :: cvf       !column VOC flux
   type(column_dflux_type) :: cdf       !column dust flux

#if (defined CNDV)
   ! dgvm variables defined at the column level
   type (column_dgvstate_type) :: cdgvs !column DGVM structure
#endif
   
#if (defined C13)
   ! 4/14/05: PET
   ! Adding isotope code
   type(column_cstate_type) :: cc13s    !column carbon-13 state
   type(column_cflux_type)  :: cc13f    !column carbon-13 flux
#endif
   
end type column_type

!----------------------------------------------------
! define the geomorphological land unit structure
!----------------------------------------------------

type, public :: landunit_type
   type(column_type) :: c                 !column data structure (soil/snow/canopy columns)

   ! g/l/c/p hierarchy, local g/l/c/p cells only
   integer , pointer :: gridcell(:)       !index into gridcell level quantities
   real(r8), pointer :: wtgcell(:)        !weight (relative to gridcell)
   integer , pointer :: coli(:)           !beginning column index per landunit
   integer , pointer :: colf(:)           !ending column index for each landunit
   integer , pointer :: ncolumns(:)       !number of columns for each landunit
   integer , pointer :: pfti(:)           !beginning pft index for each landunit
   integer , pointer :: pftf(:)           !ending pft index for each landunit
   integer , pointer :: npfts(:)          !number of pfts for each landunit

   real(r8), pointer :: area(:)          !total land area for this landunit (km^2)

   ! Urban canyon related properties
   real(r8), pointer :: canyon_hwr(:)     ! urban landunit canyon height to width ratio (-)   
   real(r8), pointer :: wtroad_perv(:)    ! urban landunit weight of pervious road column to total road (-)
   real(r8), pointer :: wtlunit_roof(:)   ! weight of roof with respect to urban landunit (-)

   ! Urban related info MV - this should be moved to land physical state - MV
   real(r8), pointer :: ht_roof(:)        ! height of urban roof (m)
   real(r8), pointer :: wind_hgt_canyon(:)! height above road at which wind in canyon is to be computed (m)
   real(r8), pointer :: z_0_town(:)       ! urban landunit momentum roughness length (m)
   real(r8), pointer :: z_d_town(:)       ! urban landunit displacement height (m)
   
   ! topological mapping functionality
   integer , pointer :: itype(:)        !landunit type
   logical , pointer :: ifspecial(:)    !BOOL: true=>landunit is not vegetated
   logical , pointer :: lakpoi(:)       !BOOL: true=>lake point
   logical , pointer :: urbpoi(:)       !BOOL: true=>urban point

   ! conservation check structures for the landunit level
   type(energy_balance_type)   :: lebal !energy balance structure
   type(water_balance_type)    :: lwbal !water balance structure
   type(carbon_balance_type)   :: lcbal !carbon balance structure
   type(nitrogen_balance_type) :: lnbal !nitrogen balance structure
   
   ! state variables defined at the land unit level
   type(landunit_pstate_type) :: lps    !land unit physical state variables
   type(landunit_estate_type) :: les    !average of energy states all columns
   type(landunit_wstate_type) :: lws    !average of water states all columns
   type(landunit_cstate_type) :: lcs    !average of carbon states all columns
   type(landunit_nstate_type) :: lns    !average of nitrogen states all columns
   type(landunit_vstate_type) :: lvs    !average of VOC states all columns
   type(landunit_dstate_type) :: lds    !average of dust states all columns
   
   ! flux variables defined at the landunit level
   type(landunit_eflux_type) :: lef     !average of energy fluxes all columns
   type(landunit_mflux_type) :: lmf     !average of momentum fluxes all columns
   type(landunit_wflux_type) :: lwf     !average of water fluxes all columns
   type(landunit_cflux_type) :: lcf     !average of carbon fluxes all columns
   type(landunit_nflux_type) :: lnf     !average of nitrogen fluxes all columns
   type(landunit_vflux_type) :: lvf     !average of VOC fluxes all columns
   type(landunit_dflux_type) :: ldf     !average of dust fluxes all columns
end type landunit_type

!----------------------------------------------------
! define the gridcell structure
!----------------------------------------------------

type, public :: gridcell_type

   type(landunit_type) :: l             !geomorphological landunits

   ! g/l/c/p hierarchy, local g/l/c/p cells only
   integer, pointer :: luni(:)          !beginning landunit index 
   integer, pointer :: lunf(:)          !ending landunit index 
   integer, pointer :: nlandunits(:)    !number of landunit for each gridcell
   integer, pointer :: coli(:)          !beginning column index
   integer, pointer :: colf(:)          !ending column index
   integer, pointer :: ncolumns(:)      !number of columns for each gridcell
   integer, pointer :: pfti(:)          !beginning pft index
   integer, pointer :: pftf(:)          !ending pft index
   integer, pointer :: npfts(:)         !number of pfts for each gridcell

   ! topological mapping functionality, local 1d gdc arrays
   integer , pointer :: gindex(:)       !global index
   real(r8), pointer :: area(:)         !total land area, gridcell (km^2)
   real(r8), pointer :: lat(:)          !latitude (radians)
   real(r8), pointer :: lon(:)          !longitude (radians)
   real(r8), pointer :: latdeg(:)       !latitude (degrees)
   real(r8), pointer :: londeg(:)       !longitude (degrees)
   integer , pointer :: gindex_a(:)     !"atm" global index
   real(r8), pointer :: lat_a(:) 	!"atm" latitude (radians) for albedo
   real(r8), pointer :: lon_a(:)        !"atm" longitude (radians) for albedo
   real(r8), pointer :: latdeg_a(:)     !"atm" latitude (degrees) for albedo
   real(r8), pointer :: londeg_a(:)     !"atm" longitude (degrees) for albedo

   ! conservation check structures for the gridcell level
   type(energy_balance_type)   :: gebal !energy balance structure
   type(water_balance_type)    :: gwbal !water balance structure
   type(carbon_balance_type)   :: gcbal !carbon balance structure
   type(nitrogen_balance_type) :: gnbal !nitrogen balance structure

#if (defined CNDV)
   ! dgvm variables defined at the gridcell level
   type(gridcell_dgvstate_type):: gdgvs !gridcell DGVM structure
#endif
   
   ! state variables defined at the gridcell level
   type(gridcell_pstate_type) :: gps    !gridcell physical state variables
   type(gridcell_estate_type) :: ges    !average of energy states all landunits
   type(gridcell_wstate_type) :: gws    !average of water states all landunits
   type(gridcell_cstate_type) :: gcs    !average of carbon states all landunits
   type(gridcell_nstate_type) :: gns    !average of nitrogen states all landus
   type(gridcell_vstate_type) :: gvs    !average of VOC states all landunits
   type(gridcell_efstate_type):: gve	!gridcell VOC emission factors
   type(gridcell_dstate_type) :: gds    !average of dust states all landunits
   
   ! flux variables defined at the gridcell level
   type(gridcell_eflux_type) :: gef     !average of energy fluxes all landunits
   type(gridcell_wflux_type) :: gwf     !average of water fluxes all landunits
   type(gridcell_cflux_type) :: gcf     !average of carbon fluxes all landunits
   type(gridcell_nflux_type) :: gnf     !average of nitrogen fluxes all landus
   type(gridcell_vflux_type) :: gvf     !average of VOC fluxes all landunits
   type(gridcell_dflux_type) :: gdf     !average of dust fluxes all landunits

end type gridcell_type

!----------------------------------------------------
! define the top-level (model) structure 
!----------------------------------------------------

type, public :: model_type
   ! lower level in hierarch
   type(gridcell_type) :: g    !gridicell data structure
   integer  :: ngridcells      !number of gridcells for this process
   real(r8) :: area            !total land area for all gridcells (km^2)

   ! conservation check structures for the clm (global) level
   type(energy_balance_type)   :: mebal !energy balance structure
   type(water_balance_type)    :: mwbal !water balance structure
   type(carbon_balance_type)   :: mcbal !carbon balnace structure
   type(nitrogen_balance_type) :: mnbal !nitrogen balance structure
   
   ! globally average state variables 
   type(model_pstate_type) ::  mps      !clm physical state variables
   type(model_estate_type) ::  mes      !average of energy states all gridcells
   type(model_wstate_type) ::  mws      !average of water states all gridcells
   type(model_cstate_type) ::  mcs      !average of carbon states all gridcells
   type(model_nstate_type) ::  mns      !average of nitrogen states all gcells
   type(model_vstate_type) ::  mvs      !average of VOC states all gridcells
   type(model_dstate_type) ::  mds      !average of dust states all gridcells
   
   ! globally averaged flux variables 
   type(model_eflux_type) ::   mef      !average of energy fluxes all gridcells
   type(model_wflux_type) ::   mwf      !average of water fluxes all gridcells
   type(model_cflux_type) ::   mcf      !average of carbon fluxes all gridcells
   type(model_nflux_type) ::   mnf      !average of nitrogen fluxes all gcells
   type(model_vflux_type) ::   mvf      !average of VOC fluxes all gridcells
   type(model_dflux_type) ::   mdf      !average of dust fluxes all gridcells
end type model_type

type atm2lnd_type
  real(r8), pointer :: forc_t(:)       !atmospheric temperature (Kelvin)
  real(r8), pointer :: forc_u(:)       !atm wind speed, east direction (m/s)
  real(r8), pointer :: forc_v(:)       !atm wind speed, north direction (m/s)
  real(r8), pointer :: forc_wind(:)    !atmospheric wind speed   
  real(r8), pointer :: forc_q(:)       !atmospheric specific humidity (kg/kg)
  real(r8), pointer :: forc_hgt(:)     !atmospheric reference height (m)
  real(r8), pointer :: forc_hgt_u(:)   !obs height of wind [m] (new)
  real(r8), pointer :: forc_hgt_t(:)   !obs height of temperature [m] (new)
  real(r8), pointer :: forc_hgt_q(:)   !obs height of humidity [m] (new)
  real(r8), pointer :: forc_pbot(:)    !atmospheric pressure (Pa)
  real(r8), pointer :: forc_th(:)      !atm potential temperature (Kelvin)
  real(r8), pointer :: forc_vp(:)      !atmospheric vapor pressure (Pa)
  real(r8), pointer :: forc_rho(:)     !density (kg/m**3)
  real(r8), pointer :: forc_rh(:)      !atmospheric relative humidity (%)
  real(r8), pointer :: forc_psrf(:)    !surface pressure (Pa)
  real(r8), pointer :: forc_pco2(:)    !CO2 partial pressure (Pa)
  real(r8), pointer :: forc_lwrad(:)   !downwrd IR longwave radiation (W/m**2)
  real(r8), pointer :: forc_solad(:,:) !direct beam radiation (numrad)
                                       !(vis=forc_sols , nir=forc_soll )
  real(r8), pointer :: forc_solai(:,:) !diffuse radiation (numrad)
                                       !(vis=forc_solsd, nir=forc_solld)
  real(r8), pointer :: forc_solar(:)   !incident solar radiation
  real(r8), pointer :: forc_rain(:)    !rain rate [mm/s]
  real(r8), pointer :: forc_snow(:)    !snow rate [mm/s]
  real(r8), pointer :: forc_ndep(:)    !nitrogen deposition rate (gN/m2/s)
  real(r8), pointer :: rainf(:)        !ALMA rain+snow [mm/s]
#if (defined C13)
  ! 4/14/05: PET
  ! Adding isotope code
  real(r8), pointer :: forc_pc13o2(:)  !C13O2 partial pressure (Pa)
#endif
  real(r8), pointer :: forc_po2(:)     !O2 partial pressure (Pa)
  real(r8), pointer :: forc_aer(:,:)   ! aerosol deposition array
end type atm2lnd_type

type(atm2lnd_type), public, target, save :: clm_a2l

!----------------------------------------------------
! End definition of spatial scaling hierarchy
!----------------------------------------------------
!*******************************************************************************

!*******************************************************************************
!----------------------------------------------------
! Declare single instance of clmtype
!----------------------------------------------------
type(model_type)    , public, target     , save :: clm3

!----------------------------------------------------
! Declare single instance of array of ecophysiological constant types
!----------------------------------------------------
type(pft_epc_type), public, target, save :: pftcon

#if (defined CNDV) || (defined CROP)
!----------------------------------------------------
! Declare single instance of array of dgvm ecophysiological constant types
!----------------------------------------------------
type(pft_dgvepc_type), public, target, save :: dgv_pftcon
#endif

character(len=8), parameter, public :: gratm  = 'atmgrid'   ! name of atmgrid
character(len=8), parameter, public :: grlnd  = 'lndgrid'   ! name of lndgrid
character(len=8), parameter, public :: nameg  = 'gridcell'  ! name of gridcells
character(len=8), parameter, public :: namel  = 'landunit'  ! name of landunits
character(len=8), parameter, public :: namec  = 'column'    ! name of columns
character(len=8), parameter, public :: namep  = 'pft'       ! name of pfts
character(len=8), parameter, public :: allrof = 'allrof'    ! name of rtm, runoff

!
!EOP
!----------------------------------------------------------------------- 
 contains
   subroutine clmtype_mod
   end subroutine clmtype_mod


end module clmtype  
MODULE module_sf_clm

!October 15, 2012
!Jiming Jin: initial coupling WRF with CLM 
!Yaqiong Lu and Jiming Jin:  CLM version 4.0 update with WRF
!
!------------------------------------------------------
  use shr_kind_mod, only: r8 => shr_kind_r8
  use clm_varpar, only: numpft, clm_varpar_mod,nlevgrnd, nlevsoi,nlevlak,nlevsno,maxpatch_pft
  use clm_varcon, only: hvap, hsub,tfrz, vkc, sb  ,&
                        snowage_drdt0,ndep,organic,fmax,efisop
  
!
CONTAINS
!
    subroutine clmdrv(zgcmxy     ,forc_qxy   ,ps   ,forc_txy    ,tsxy  &  
                   ,shxy       ,qfx        ,lhxy        ,soiflx      ,qgh       &
                   ,gsw, swdown,ra_sw_physics &   
                   ,history_interval ,flwdsxy    ,smstav      ,smstot      ,qsfxy     &
                   ,qdnxy      ,ivgtyp     ,isltyp      ,vegfra      ,albxy     &
                   ,znt        ,z0         ,tmn         ,xland       ,xice      &
                   ,emiss      ,snowc      ,qsfc        ,prec        ,maxpatch  &
                   ,num_soil_layers        ,dt          ,dzs         ,nstep     &
                   ,smois      ,tslb       ,snow        ,canwat      ,chs       &
                   ,chs2                                                        &
                   ,sh2o       ,snowh      ,forc_uxy    ,forc_vxy    ,shdmin    &
                   ,shdmax     ,acsnom     ,acsnow      ,dx          ,xlat      &
                   ,xlong,ht                                                    &   
                   ,ids,ide, jds,jde, kds,kde                    &
                   ,ims,ime, jms,jme, kms,kme                    &
                   ,its,ite, jts,jte, kts,kte                    &
                   ,inest, sf_urban_physics,                               &
!Optional urban
                CMR_SFCDIF,CHR_SFCDIF,CMC_SFCDIF,CHC_SFCDIF,   &
                tr_urb2d,tb_urb2d,tg_urb2d,tc_urb2d,qc_urb2d,  & !H urban
                uc_urb2d,                                       & !H urban
                xxxr_urb2d,xxxb_urb2d,xxxg_urb2d,xxxc_urb2d,    & !H urban
                trl_urb3d,tbl_urb3d,tgl_urb3d,                  & !H urban
                sh_urb2d,lh_urb2d,g_urb2d,rn_urb2d,ts_urb2d,    & !H urban
                psim_urb2d,psih_urb2d,u10_urb2d,v10_urb2d,      & !O urban
                GZ1OZ0_urb2d, AKMS_URB2D,                       & !O urban
                th2_urb2d,q2_urb2d,ust_urb2d,                   & !O urban
               declin_urb,cosz_urb2d,omg_urb2d,                & !I urban
                xlat_urb2d,                                     & !I urban
                num_roof_layers, num_wall_layers,               & !I urban
                num_road_layers, DZR, DZB, DZG,                 & !I urban
                FRC_URB2D, UTYPE_URB2D                          & ! urban
! subgrids
               ,numc,nump,sabv,sabg,lwup,snl, &
                snowdp,wtc,wtp,h2osno,t_grnd,t_veg,         &
                h2ocan,h2ocan_col,t2m_max,t2m_min,t2clm ,    &
                t_ref2m,h2osoi_liq_s1,                 &
                h2osoi_liq_s2,h2osoi_liq_s3,h2osoi_liq_s4,          &
                h2osoi_liq_s5,h2osoi_liq1,h2osoi_liq2,              &
                h2osoi_liq3,h2osoi_liq4,h2osoi_liq5,h2osoi_liq6,    &
                h2osoi_liq7,h2osoi_liq8,h2osoi_liq9,h2osoi_liq10,   &
                h2osoi_ice_s1,h2osoi_ice_s2,                        &
                h2osoi_ice_s3,h2osoi_ice_s4,h2osoi_ice_s5,          &
                h2osoi_ice1,h2osoi_ice2,h2osoi_ice3,h2osoi_ice4,    &
                h2osoi_ice5,h2osoi_ice6,h2osoi_ice7,                &
                h2osoi_ice8,h2osoi_ice9,h2osoi_ice10,               &
                t_soisno_s1,t_soisno_s2,t_soisno_s3,t_soisno_s4,    &
                t_soisno_s5,t_soisno1,t_soisno2,t_soisno3,          &
                t_soisno4,t_soisno5,t_soisno6,t_soisno7,            &
                t_soisno8,t_soisno9,t_soisno10,                     &
                dzsnow1,dzsnow2,dzsnow3,dzsnow4,dzsnow5,            &
                snowrds1,snowrds2,snowrds3,snowrds4,snowrds5,       &
                t_lake1,t_lake2,t_lake3,t_lake4,t_lake5,            &
                t_lake6,t_lake7,t_lake8,t_lake9,t_lake10,           &
                h2osoi_vol1,h2osoi_vol2,h2osoi_vol3,                &
                h2osoi_vol4,h2osoi_vol5,h2osoi_vol6,                &
                h2osoi_vol7,h2osoi_vol8,                            &
                h2osoi_vol9,h2osoi_vol10,                           &
                q_ref2m,                                   &
                ALBEDOsubgrid,LHsubgrid,HFXsubgrid,LWUPsubgrid,     &
                Q2subgrid,SABVsubgrid,SABGsubgrid,NRAsubgrid,SWUPsubgrid,&
                LHsoi,LHveg,LHtran,&
                alswvisdir, alswvisdif, alswnirdir, alswnirdif,      & ! clm
                swvisdir, swvisdif, swnirdir, swnirdif               & ! clm
#ifdef CN
!CROP&CN restart and outputs
                ,dyntlai,dyntsai,dyntop,dynbot &
                ,htmx,croplive,gdd1020,gdd820,gdd020,grainc,grainc_storage  &
                ,grainc_xfer,grainn,grainn_storage,grainn_xfer,days_active  &
                ,onset_flag,onset_counter,onset_gddflag,onset_fdd,onset_gdd &
                ,onset_swi,offset_flag,offset_counter,offset_fdd,offset_swi &
                ,dayl,annavg_t2m,tempavg_t2m,tempsum_potential_gpp          &
                ,annsum_potential_gpp,tempmax_retransn,annmax_retransn      &
                ,prev_leafc_to_litter,prev_frootc_to_litter,tempsum_npp     &
                ,annsum_npp,leafc,leafc_storage,leafc_xfer,frootc           &
                ,frootc_storage,frootc_xfer,livestemc,livestemc_storage     &
                ,livestemc_xfer,deadstemc,deadstemc_storage,deadstemc_xfer  &
                ,livecrootc,livecrootc_storage,livecrootc_xfer,deadcrootc   &
                ,deadcrootc_storage,deadcrootc_xfer,cpool,pft_ctrunc        &
                ,leafn,leafn_storage,leafn_xfer,frootn,frootn_storage       &
                ,frootn_xfer,livestemn,livestemn_storage,livestemn_xfer     &  
                ,deadstemn,deadstemn_storage,deadstemn_xfer,livecrootn      &
                ,livecrootn_storage,livecrootn_xfer,deadcrootn              &
                ,deadcrootn_storage,deadcrootn_xfer,npool,pft_ntrunc        &
                ,gresp_storage,gresp_xfer,xsmrpool,annsum_counter           &
                ,cannsum_npp,cannavg_t2m,wf,me,mean_fire_prob,cwdc,litr1c   &
                ,litr2c,litr3c,soil1c,soil2c,soil3c,soil4c,seedc,col_ctrunc &
                ,prod10c,prod100c,cwdn,litr1n,litr2n,litr3n,soil1n,soil2n   &
                ,soil3n,soil4n,seedn,col_ntrunc,prod10n,prod100n,sminn      &
                ,totlitc,dwt_seedc_to_leaf,dwt_seedc_to_deadstem,dwt_conv_cflux &
                ,dwt_prod10c_gain,dwt_prod100c_gain,prod100c_loss,dwt_frootc_to_litr1c &
                ,dwt_frootc_to_litr2c,dwt_frootc_to_litr3c,dwt_livecrootc_to_cwdc &
                ,dwt_deadcrootc_to_cwdc,dwt_seedn_to_leaf,dwt_seedn_to_deadstem &
                ,dwt_conv_nflux,dwt_prod10n_gain,dwt_prod100n_gain,prod100n_loss &
                ,dwt_frootn_to_litr1n,dwt_frootn_to_litr2n, dwt_frootn_to_litr3n &
                , dwt_livecrootn_to_cwdn,dwt_deadcrootn_to_cwdn,retransn &
#endif
                 )


  USE module_date_time
  USE module_sf_urban, only: urban
  USE module_configure

  implicit none

  integer,  intent(in)       ::     ids,ide, jds,jde, kds,kde,  &
                                    ims,ime, jms,jme, kms,kme,  &
                                    its,ite, jts,jte, kts,kte

  integer,intent(in)  :: num_soil_layers,maxpatch,nstep,sf_urban_physics,&
                         ra_sw_physics,history_interval
  real,dimension(ims:ime,1:num_soil_layers,jms:jme ),intent(inout) ::&
                                                         smois, & ! total soil moisture
                                                         sh2o,  & ! new soil liquid
                                                         tslb     ! TSLB     STEMP

  real,intent(in) :: dt,dx
  real(r8) :: dtt
  real, dimension(1:num_soil_layers), intent(in)::dzs

  real,dimension(ims:ime,jms:jme ),intent(inout) ::&
                    smstav     ,smstot                &
                   ,znt        ,snowc      ,qsfc       ,snow        ,snowh &
                   ,canwat     ,acsnom     ,acsnow,   emiss, z0

  real,dimension(ims:ime,jms:jme ),intent(in) ::&
                 vegfra, tmn,shdmin,shdmax

  real,dimension(ims:ime,jms:jme ),intent(in) ::&
                   qgh,chs,chs2

  real(r8) :: efisop_buf(6)
  logical :: found = .false.
  integer,   dimension(ims:ime,jms:jme ),intent(inout) :: numc,nump
  real,  dimension(ims:ime,jms:jme ),intent(inout) :: soiflx,sabv,sabg,lwup,t2m_max,t2m_min
  integer,  dimension(ims:ime,1:maxpatch,jms:jme ) :: snl,snl1
  real,  dimension(ims:ime,1:maxpatch,jms:jme ),intent(inout) ::  &
                snowdp,wtc,wtp,h2osno,t_grnd,t_veg,         &
                h2ocan,h2ocan_col,     &
                t_ref2m,h2osoi_liq_s1,              &
                h2osoi_liq_s2,h2osoi_liq_s3,h2osoi_liq_s4,          &
                h2osoi_liq_s5,h2osoi_liq1,h2osoi_liq2,              &
                h2osoi_liq3,h2osoi_liq4,h2osoi_liq5,h2osoi_liq6,    &
                h2osoi_liq7,h2osoi_liq8,h2osoi_liq9,h2osoi_liq10,   &
                h2osoi_ice_s1,h2osoi_ice_s2,                        &
                h2osoi_ice_s3,h2osoi_ice_s4,h2osoi_ice_s5,          &
                h2osoi_ice1,h2osoi_ice2,h2osoi_ice3,h2osoi_ice4,    &
                h2osoi_ice5,h2osoi_ice6,h2osoi_ice7,                &
                h2osoi_ice8,h2osoi_ice9,h2osoi_ice10,               &
                t_soisno_s1,t_soisno_s2,t_soisno_s3,t_soisno_s4,    &
                t_soisno_s5,t_soisno1,t_soisno2,t_soisno3,          &
                t_soisno4,t_soisno5,t_soisno6,t_soisno7,            &
                t_soisno8,t_soisno9,t_soisno10,                     &
                dzsnow1,dzsnow2,dzsnow3,dzsnow4,dzsnow5,            &
                snowrds1,snowrds2,snowrds3,snowrds4,snowrds5,       &
                t_lake1,t_lake2,t_lake3,t_lake4,t_lake5,            &
                t_lake6,t_lake7,t_lake8,t_lake9,t_lake10,           &
                h2osoi_vol1,h2osoi_vol2,h2osoi_vol3,                &
                h2osoi_vol4,h2osoi_vol5,h2osoi_vol6,                &
                h2osoi_vol7,h2osoi_vol8,                            &
                h2osoi_vol9,h2osoi_vol10,                           &
                q_ref2m,                                            &
                ALBEDOsubgrid,LHsubgrid,HFXsubgrid,LWUPsubgrid,     &
                Q2subgrid,SABVsubgrid,SABGsubgrid,NRAsubgrid,       &
                SWUPsubgrid,LHsoi,LHveg,LHtran

 real(r8) :: gti_buf

#ifdef CN
 real, dimension(ims:ime,1:maxpatch,jms:jme),intent(in) :: dyntlai,dyntsai,dyntop,dynbot

!ADD_NEW_VAR
  integer,   dimension(ims:ime,1:maxpatch,jms:jme ),intent(inout) :: croplive
  real,dimension(ims:ime,1:maxpatch,jms:jme),intent(inout) :: &
                 htmx,gdd1020,gdd820,gdd020,grainc,grainc_storage  &
                ,grainc_xfer,grainn,grainn_storage,grainn_xfer,days_active  &
                ,onset_flag,onset_counter,onset_gddflag,onset_fdd,onset_gdd &
                ,onset_swi,offset_flag,offset_counter,offset_fdd,offset_swi &
                ,dayl,annavg_t2m,tempavg_t2m,tempsum_potential_gpp          &
                ,annsum_potential_gpp,tempmax_retransn,annmax_retransn      &
                ,prev_leafc_to_litter,prev_frootc_to_litter,tempsum_npp     &
                ,annsum_npp,leafc,leafc_storage,leafc_xfer,frootc           &
                ,frootc_storage,frootc_xfer,livestemc,livestemc_storage     &
                ,livestemc_xfer,deadstemc,deadstemc_storage,deadstemc_xfer  &
                ,livecrootc,livecrootc_storage,livecrootc_xfer,deadcrootc   &
                ,deadcrootc_storage,deadcrootc_xfer,cpool,pft_ctrunc        &
                ,leafn,leafn_storage,leafn_xfer,frootn,frootn_storage       &
                ,frootn_xfer,livestemn,livestemn_storage,livestemn_xfer     &
                ,deadstemn,deadstemn_storage,deadstemn_xfer,livecrootn      &
                ,livecrootn_storage,livecrootn_xfer,deadcrootn              &
                ,deadcrootn_storage,deadcrootn_xfer,npool,pft_ntrunc        &
                ,gresp_storage,gresp_xfer,xsmrpool,annsum_counter           &
                ,cannsum_npp,cannavg_t2m,wf,me,mean_fire_prob,cwdc,litr1c   &
                ,litr2c,litr3c,soil1c,soil2c,soil3c,soil4c,seedc,col_ctrunc &
                ,prod10c,prod100c,cwdn,litr1n,litr2n,litr3n,soil1n,soil2n   &
                ,soil3n,soil4n,seedn,col_ntrunc,prod10n,prod100n,sminn      &
                ,totlitc,dwt_seedc_to_leaf,dwt_seedc_to_deadstem,dwt_conv_cflux &
                ,dwt_prod10c_gain,dwt_prod100c_gain,prod100c_loss,dwt_frootc_to_litr1c &
                ,dwt_frootc_to_litr2c,dwt_frootc_to_litr3c,dwt_livecrootc_to_cwdc &
                ,dwt_deadcrootc_to_cwdc,dwt_seedn_to_leaf,dwt_seedn_to_deadstem & 
                ,dwt_conv_nflux,dwt_prod10n_gain,dwt_prod100n_gain,prod100n_loss &
                ,dwt_frootn_to_litr1n,dwt_frootn_to_litr2n, dwt_frootn_to_litr3n &
                , dwt_livecrootn_to_cwdn,dwt_deadcrootn_to_cwdn,retransn 

#endif
!!!

  integer  :: i,j,m,inest,k
  real, dimension(ims:ime, kms:kme,jms:jme),intent(in) ::&
            forc_txy,forc_uxy,forc_vxy,forc_qxy,zgcmxy,ps
  
  real :: flwdsxy(ims:ime,jms:jme)           !downward longwave rad onto surface (W/m**2)
  real :: gsw(ims:ime,jms:jme)               !downward solar rad onto surface (W/m**2)

  real :: swdown(ims:ime,jms:jme)
  real, dimension(ims:ime,jms:jme),intent(in) :: swvisdir, swvisdif, swnirdir,swnirdif
  real, dimension(ims:ime,jms:jme),intent(out):: alswvisdir,alswvisdif,alswnirdir,alswnirdif

  real :: xlat (ims:ime,jms:jme)
  real :: xlong(ims:ime,jms:jme)
  real :: ht(ims:ime,jms:jme)
  real :: xland (ims:ime,jms:jme)
  real :: xice (ims:ime,jms:jme)
  real :: prec (ims:ime,jms:jme)              !total precipitation rate (mm; accumlated precipitation within DT)
  integer  :: ivgtyp(ims:ime,jms:jme)
  integer  :: isltyp (ims:ime,jms:jme)

  real :: albxy(ims:ime,jms:jme)           
  real :: tsxy(ims:ime,jms:jme)           
  real :: t2clm(ims:ime,jms:jme)
  real :: shxy(ims:ime,jms:jme)           
  real :: lhxy(ims:ime,jms:jme)           
  real :: qfx(ims:ime,jms:jme)   ! kg/(sm^2) =>mm/s       
  real :: qsfxy(ims:ime,jms:jme) 
  real :: qdnxy(ims:ime,jms:jme) 

  real(r8) :: alswvisdir_buf,alswvisdif_buf,alswnirdir_buf,alswnirdif_buf
  real(r8) :: swvisdir_buf,swvisdif_buf,swnirdir_buf,swnirdif_buf

  real(r8) :: albxy_buf
  real(r8) :: tsxy_buf,trefxy_buf
  real(r8) :: shxy_buf
  real(r8) :: lhxy_buf
  real(r8) :: qsfxy_buf
  real(r8) :: qdnxy_buf
  real(r8) :: soiflx_buf
  real(r8) :: sabv_buf
  real(r8) :: sabg_buf
  real(r8) :: lwup_buf
  real(r8) :: znt_buf
  real(r8) :: rhoxy_buf

  real(r8) :: swd_buf
  real(r8) :: forc_sols_buf
  real(r8) :: forc_soll_buf
  real(r8) :: forc_solsd_buf
  real(r8) :: forc_solld_buf
  real(r8) :: area_buf
  real(r8) :: forc_pbot_buf
  real(r8) :: forc_txy_buf
  real(r8) :: forc_uxy_buf
  real(r8) :: forc_vxy_buf
  real(r8) :: forc_qxy_buf
  real(r8) :: zgcmxy_buf
  real(r8) :: prec_buf
  real(r8) :: flwdsxy_buf
  real(r8) :: forc_psrfxy_buf

!ADD_NEW_VAR
  real(r8) :: forc_ndepxy_buf
!!!

  real(r8) :: xlat_buf 
  real(r8) :: xlon_buf

  real(r8),dimension(maxpatch,-nlevsno+1:nlevgrnd) :: dzclm
  real(r8),dimension(maxpatch,-nlevsno+1:nlevgrnd) :: zclm
  real(r8),dimension(maxpatch,-nlevsno:nlevgrnd)   :: ziclm

  real(r8),dimension(maxpatch,-nlevsno+1:nlevgrnd)   :: &
                          h2osoi_liq_buf, &
                          h2osoi_ice_buf, &
                          t_soisno_buf

  real(r8),dimension(maxpatch,-nlevsno+1:0) ::snw_rds_buf

      
  real(r8),dimension(maxpatch,1:num_soil_layers)   :: &
                         t_lake_buf, h2osoi_vol_buf
  integer  :: lndmsk
!------------------------------------------------------------------------
 real(r8),dimension(maxpatch) :: organic_buf

  real(r8), dimension(maxpatch)  ::                 &
                snowdp_buf,wtc_buf,wtp_buf,h2osno_buf,t_grnd_buf,t_veg_buf,         & 
                h2ocan_buf,h2ocan_col_buf,     &
                t_ref2m_buf, q_ref2m_buf,                          &
                albedosubgrid_buf, lhsubgrid_buf, hfxsubgrid_buf, lwupsubgrid_buf, &
                q2subgrid_buf,sabgsubgrid_buf,sabvsubgrid_buf,nrasubgrid_buf,swupsubgrid_buf,&
                lhsoi_buf,lhveg_buf,lhtran_buf,tlai_buf,tsai_buf,htop_buf,hbot_buf
#ifdef CN
!CROP&CN buf variables
  integer,dimension(maxpatch) :: croplive_buf
   real(r8), dimension(maxpatch)  ::  &
                 htmx_buf,gdd1020_buf,gdd820_buf,gdd020_buf,grainc_buf,grainc_storage_buf  &
                ,grainc_xfer_buf,grainn_buf,grainn_storage_buf,grainn_xfer_buf,days_active_buf  &
                ,onset_flag_buf,onset_counter_buf,onset_gddflag_buf,onset_fdd_buf,onset_gdd_buf &
                ,onset_swi_buf,offset_flag_buf,offset_counter_buf,offset_fdd_buf,offset_swi_buf &
                ,dayl_buf,annavg_t2m_buf,tempavg_t2m_buf,tempsum_potential_gpp_buf          &
                ,annsum_potential_gpp_buf,tempmax_retransn_buf,annmax_retransn_buf      &
                ,prev_leafc_to_litter_buf,prev_frootc_to_litter_buf,tempsum_npp_buf     &
                ,annsum_npp_buf,leafc_buf,leafc_storage_buf,leafc_xfer_buf,frootc_buf           &
                ,frootc_storage_buf,frootc_xfer_buf,livestemc_buf,livestemc_storage_buf     &
                ,livestemc_xfer_buf,deadstemc_buf,deadstemc_storage_buf,deadstemc_xfer_buf  &
                ,livecrootc_buf,livecrootc_storage_buf,livecrootc_xfer_buf,deadcrootc_buf   &
                ,deadcrootc_storage_buf,deadcrootc_xfer_buf,cpool_buf,pft_ctrunc_buf        &
                ,leafn_buf,leafn_storage_buf,leafn_xfer_buf,frootn_buf,frootn_storage_buf       &
                ,frootn_xfer_buf,livestemn_buf,livestemn_storage_buf,livestemn_xfer_buf     &
                ,deadstemn_buf,deadstemn_storage_buf,deadstemn_xfer_buf,livecrootn_buf      &
                ,livecrootn_storage_buf,livecrootn_xfer_buf,deadcrootn_buf              &
                ,deadcrootn_storage_buf,deadcrootn_xfer_buf,npool_buf,pft_ntrunc_buf        &
                ,gresp_storage_buf,gresp_xfer_buf,xsmrpool_buf,annsum_counter_buf           &
                ,cannsum_npp_buf,cannavg_t2m_buf,wf_buf,me_buf,mean_fire_prob_buf,cwdc_buf,litr1c_buf   &
                ,litr2c_buf,litr3c_buf,soil1c_buf,soil2c_buf,soil3c_buf,soil4c_buf,seedc_buf,col_ctrunc_buf &
                ,prod10c_buf,prod100c_buf,cwdn_buf,litr1n_buf,litr2n_buf,litr3n_buf,soil1n_buf,soil2n_buf   &
                ,soil3n_buf,soil4n_buf,seedn_buf,col_ntrunc_buf,prod10n_buf,prod100n_buf,sminn_buf&
                ,totlitc_buf,dwt_seedc_to_leaf_buf,dwt_seedc_to_deadstem_buf,dwt_conv_cflux_buf &
                ,dwt_prod10c_gain_buf,dwt_prod100c_gain_buf,prod100c_loss_buf,dwt_frootc_to_litr1c_buf &
                ,dwt_frootc_to_litr2c_buf,dwt_frootc_to_litr3c_buf,dwt_livecrootc_to_cwdc_buf &
                ,dwt_deadcrootc_to_cwdc_buf,dwt_seedn_to_leaf_buf,dwt_seedn_to_deadstem_buf &
                ,dwt_conv_nflux_buf,dwt_prod10n_gain_buf,dwt_prod100n_gain_buf,prod100n_loss_buf &
                ,dwt_frootn_to_litr1n_buf,dwt_frootn_to_litr2n_buf, dwt_frootn_to_litr3n_buf &
                , dwt_livecrootn_to_cwdn_buf,dwt_deadcrootn_to_cwdn_buf,retransn_buf  
#endif
! ----------------------------------------------------------------------
! DECLARATIONS START - urban
! ----------------------------------------------------------------------

! input variables surface_driver --> lsm 
     INTEGER, INTENT(IN) :: num_roof_layers
     INTEGER, INTENT(IN) :: num_wall_layers
     INTEGER, INTENT(IN) :: num_road_layers
     REAL, OPTIONAL, DIMENSION(1:num_roof_layers), INTENT(IN) :: DZR
     REAL, OPTIONAL, DIMENSION(1:num_wall_layers), INTENT(IN) :: DZB
     REAL, OPTIONAL, DIMENSION(1:num_road_layers), INTENT(IN) :: DZG
     REAL, OPTIONAL, INTENT(IN) :: DECLIN_URB
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(IN) :: COSZ_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(IN) :: OMG_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(IN) :: XLAT_URB2D

! input variables lsm --> urban
     INTEGER :: UTYPE_URB ! urban type [urban=1, suburban=2, rural=3]
     REAL :: TA_URB       ! potential temp at 1st atmospheric level [K]
     REAL :: QA_URB       ! mixing ratio at 1st atmospheric level  [kg/kg]
     REAL :: UA_URB       ! wind speed at 1st atmospheric level    [m/s]
     REAL :: U1_URB       ! u at 1st atmospheric level             [m/s]
     REAL :: V1_URB       ! v at 1st atmospheric level             [m/s]
     REAL :: SSG_URB      ! downward total short wave radiation    [W/m/m]
     REAL :: LLG_URB      ! downward long wave radiation           [W/m/m]
     REAL :: RAIN_URB     ! precipitation                          [mm/h]
     REAL :: RHOO_URB     ! air density                            [kg/m^3]
     REAL :: ZA_URB       ! first atmospheric level                [m]
     REAL :: DELT_URB     ! time step                              [s]
     REAL :: SSGD_URB     ! downward direct short wave radiation   [W/m/m]
     REAL :: SSGQ_URB     ! downward diffuse short wave radiation  [W/m/m]
     REAL :: XLAT_URB     ! latitude                               [deg]
     REAL :: COSZ_URB     ! cosz
     REAL :: OMG_URB      ! hour angle
     REAL :: ZNT_URB      ! roughness length                       [m]
     REAL :: TR_URB
     REAL :: TB_URB
     REAL :: TG_URB
     REAL :: TC_URB
     REAL :: QC_URB
     REAL :: UC_URB
     REAL :: XXXR_URB
     REAL :: XXXB_URB
     REAL :: XXXG_URB
     REAL :: XXXC_URB
     REAL, DIMENSION(1:num_roof_layers) :: TRL_URB  ! roof layer temp [K]
     REAL, DIMENSION(1:num_wall_layers) :: TBL_URB  ! wall layer temp [K]
     REAL, DIMENSION(1:num_road_layers) :: TGL_URB  ! road layer temp [K]
     LOGICAL  :: LSOLAR_URB

! state variable surface_driver <--> lsm <--> urban
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: TR_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: TB_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: TG_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: TC_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: QC_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: UC_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: XXXR_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: XXXB_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: XXXG_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: XXXC_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: SH_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: LH_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: G_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: RN_URB2D
!
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: TS_URB2D

     REAL, OPTIONAL, DIMENSION( ims:ime, 1:num_roof_layers, jms:jme ), INTENT(INOUT) :: TRL_URB3D
     REAL, OPTIONAL, DIMENSION( ims:ime, 1:num_wall_layers, jms:jme ), INTENT(INOUT) :: TBL_URB3D
     REAL, OPTIONAL, DIMENSION( ims:ime, 1:num_road_layers, jms:jme ), INTENT(INOUT) :: TGL_URB3D

   REAL :: CMR_URB, CHR_URB, CMC_URB, CHC_URB

   REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: CMR_SFCDIF
   REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: CHR_SFCDIF
   REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: CMC_SFCDIF
   REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(INOUT) :: CHC_SFCDIF

! output variable lsm --> surface_driver
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: PSIM_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: PSIH_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: GZ1OZ0_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: U10_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: V10_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: TH2_URB2D
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: Q2_URB2D
!
     REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: AKMS_URB2D
!
     REAL, DIMENSION( ims:ime, jms:jme ), INTENT(OUT) :: UST_URB2D
     REAL, DIMENSION( ims:ime, jms:jme ), INTENT(IN) :: FRC_URB2D
     INTEGER, DIMENSION( ims:ime, jms:jme ), INTENT(IN) :: UTYPE_URB2D


! output variables urban --> lsm
     REAL :: TS_URB     ! surface radiative temperature    [K]
     REAL :: QS_URB     ! surface humidity                 [-]
     REAL :: SH_URB     ! sensible heat flux               [W/m/m]
     REAL :: LH_URB     ! latent heat flux                 [W/m/m]
     REAL :: LH_KINEMATIC_URB ! latent heat flux, kinetic  [kg/m/m/s]
     REAL :: SW_URB     ! upward short wave radiation flux [W/m/m]
     REAL :: ALB_URB    ! time-varying albedo            [fraction]
     REAL :: LW_URB     ! upward long wave radiation flux  [W/m/m]
     REAL :: G_URB      ! heat flux into the ground        [W/m/m]
     REAL :: RN_URB     ! net radiation                    [W/m/m]
     REAL :: PSIM_URB   ! shear f for momentum             [-]
     REAL :: PSIH_URB   ! shear f for heat                 [-]
     REAL :: GZ1OZ0_URB   ! shear f for heat                 [-]
     REAL :: U10_URB    ! wind u component at 10 m         [m/s]
     REAL :: V10_URB    ! wind v component at 10 m         [m/s]
     REAL :: TH2_URB    ! potential temperature at 2 m     [K]
     REAL :: Q2_URB     ! humidity at 2 m                  [-]
     REAL :: CHS_URB
     REAL :: CHS2_URB
     REAL :: UST_URB
! ----------------------------------------------------------------------
! DECLARATIONS END - urban
! ----------------------------------------------------------------------
 CHARACTER(len=24) ::  nextstep_date, cdate,simulation_start_date
 INTEGER simulation_start_year   , &
         simulation_start_month  , &
         simulation_start_day    , &
         simulation_start_hour   , &
         simulation_start_minute , &
         simulation_start_second

 integer  :: myr,mon,mday,mhr,mint,msc,mtsec,myr1,mon1,mday1,mhr1,mint1,msc1,mtsec1
 integer  :: myrs,mons,mdays,mhrs,mints,mscs,mtsecs
 integer  :: julyr,julday, julyr1,julday1
 integer  :: mbdate
 integer  :: msec,msec1
 integer  :: ns
 real(r8) :: calday,calday1
 real     :: gmt,gmt1
 integer(selected_int_kind(12))  :: idts
 integer  :: idt
 real(r8) :: dsqmin, dsq
 character*256 :: msg
 real :: mh_urb,stdh_urb,lp_urb,hgt_urb,frc_urb,lb_urb,check
 real, dimension(4) :: lf_urb
 
! ----------------------------------------------------------------------
   call clm_varpar_mod

   call CLMDebug('Now in clmdrv')

!  if((nlevsoi /= num_soil_layers) .or. (nlevlak/= num_soil_layers)) then
!    print*,'nlevsoi and nlevlak must be equal to num_soil_layers in CLM; Stop in module_sf_clm.F'
!    call endrun() 
!  end if

  dtt = dt
  
   write(msg,*) 'dt=',dt,'jts=',jts,'jte=',jte,'its=',its,'ite=',ite
   call CLMDebug(msg)

  CALL nl_get_simulation_start_year   ( 1, simulation_start_year   )
  CALL nl_get_simulation_start_month  ( 1, simulation_start_month  )
  CALL nl_get_simulation_start_day    ( 1, simulation_start_day    )
  CALL nl_get_simulation_start_hour   ( 1, simulation_start_hour   )
  CALL nl_get_simulation_start_minute ( 1, simulation_start_minute )
  CALL nl_get_simulation_start_second ( 1, simulation_start_second )
  WRITE ( simulation_start_date(1:19) , FMT = '(I4.4,"-",I2.2,"-",I2.2,"_",I2.2,":",I2.2,":",I2.2)' ) &
           simulation_start_year,simulation_start_month,simulation_start_day,&
           simulation_start_hour,simulation_start_minute,simulation_start_second
  simulation_start_date(1:24) = simulation_start_date(1:19)//'.0000'

  CALL split_date_char (simulation_start_date, myrs, mons, mdays, mhrs, mints, mscs, mtsecs) 

  idt  = nint(dtt)*nstep
  idts = nint(dtt)*nstep
  if(idt/=idts) then
     print*,'The integer idt and idts is too large; Stop in module_sf_clm.F', idt,idts
     call endrun()
  end if
  CALL geth_newdate (cdate(1:19), simulation_start_date(1:19), idt) ! dt in seconds
  cdate(1:24) = cdate(1:19)//'.0000'
  CALL split_date_char (cdate, myr, mon, mday, mhr, mint, msc, mtsec )
  CALL geth_newdate (nextstep_date(1:19), cdate(1:19), nint(dtt)) ! dtt in seconds
  nextstep_date(1:24) = nextstep_date(1:19)//'.0000'
  CALL split_date_char (nextstep_date, myr1, mon1, mday1, mhr1, mint1, msc1, mtsec1) 
  CALL get_julgmt(cdate,julyr,julday, gmt)   !module_date_time.F ../share/
  CALL get_julgmt(nextstep_date,julyr1,julday1, gmt1)

  msec  = mhr*3600  + mint*60
  msec1 = mhr1*3600 + mint1*60
  calday = julday  + gmt/24.0
  calday1= julday1 + gmt1/24.0
  mbdate = myrs*10000 + mons*100 + mdays


!write(6,*) 'at nstep=',nstep,'snowage_drdt0(1,31,8)=',snowage_drdt0(1,31,8)


  do j=jts,jte

   do i=its,ite
      if(xland(i,j) == 1.0) then
        lndmsk = 1
      else
        lndmsk = 0
      end if

  
    if(lndmsk == 1) then
      qsfxy_buf       = qsfxy(i,j)
      qdnxy_buf       = qdnxy(i,j)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      xlon_buf       = xlong(i,j) + 360.0
      xlat_buf       = xlat(i,j)
      albxy_buf      = albxy(i,j) 
      if(gsw(i,j)>0.0.and.albxy_buf<0.99.and.albxy_buf>0.0)then
         swd_buf     = gsw(i,j)/(1.-albxy_buf)
         swdown(i,j) = gsw(i,j)/(1.-albxy_buf)
      else
         swd_buf     = 0.0
         swdown(i,j) = 0.0
      end if

    if(ra_sw_physics .ne. 3) then   !if not CAM scheme, the use 0.35/0.15 for coupling --ylu
      forc_sols_buf  = swd_buf*0.35
      forc_soll_buf  = swd_buf*0.35
      forc_solsd_buf = swd_buf*0.15
      forc_solld_buf = swd_buf*0.15
    else                            !if use cam radiation, then we can directly use the seperate swd --ylu
      forc_sols_buf  = swvisdir(i,j)
      forc_soll_buf  = swnirdir(i,j)
      forc_solsd_buf = swvisdif(i,j)
      forc_solld_buf = swnirdif(i,j)
    end if
   

      area_buf       = dx*dx/1.e6 !(km^2)
      forc_pbot_buf  = ps(i,1,j) !Pa
      forc_txy_buf   = forc_txy(i,1,j)
      forc_uxy_buf   = forc_uxy(i,1,j)
      forc_vxy_buf   = forc_vxy(i,1,j)
      forc_qxy_buf   = forc_qxy(i,1,j)
      zgcmxy_buf     = zgcmxy(i,1,j)
      prec_buf       = prec(i,j)/dtt ! mm/s
      flwdsxy_buf    = flwdsxy(i,j)
      forc_psrfxy_buf= ps(i,1,j) ! Pa
!ADD_NEW_VAR
      forc_ndepxy_buf=ndep/(86400._r8 * 365._r8)
!!!

      efisop_buf(1:6) = efisop(1:6)
      gti_buf      = fmax


      soiflx(i,j) = 0.0
      sabv(i,j)   = 0.0
      sabg(i,j)   = 0.0
      lwup(i,j)   = 0.0
      soiflx_buf     = 0.0
      sabv_buf       = 0.0
      sabg_buf       = 0.0
      lwup_buf       = 0.0

     swvisdir_buf = swvisdir(i,j)
     swvisdif_buf = swvisdif(i,j)
     swnirdir_buf = swnirdir(i,j)
     swnirdif_buf = swnirdif(i,j)

     do m=1,maxpatch
      do k =1,nlevgrnd 
        zclm(m,k) =  0.025*(exp(0.5*(k-0.5))-1.)
      end do

       dzclm(m,1) = 0.5*(zclm(m,1)+zclm(m,2))
       do k = 2,nlevgrnd-1
         dzclm(m,k)= 0.5*(zclm(m,k+1)-zclm(m,k-1))
       enddo
       dzclm(m,nlevgrnd) = zclm(m,nlevgrnd)-zclm(m,nlevgrnd-1)

      ziclm(m,0) = 0.0
      do k =1,nlevgrnd-1 
        ziclm(m,k) =  0.5*(zclm(m,k) + zclm(m,k+1)) 
      end do
        ziclm(m,nlevgrnd) =  zclm(m,nlevgrnd) + 0.5*dzclm(m,nlevgrnd)

         dzclm(m,-4) = dzsnow5(i,m,j)
         dzclm(m,-3) = dzsnow4(i,m,j)
         dzclm(m,-2) = dzsnow3(i,m,j)
         dzclm(m,-1) = dzsnow2(i,m,j)
         dzclm(m,0)  = dzsnow1(i,m,j)


      do k=0,-nlevsno+1, -1
        zclm(m,k)     = ziclm(m,k) - 0.5*dzclm(m,k)
        ziclm(m,k-1)  = ziclm(m,k) - dzclm(m,k)
      end do

      snl1(i,m,j) = snl(i,m,j)
      snowdp_buf(m)     = snowdp(i,m,j)
!      snowage_buf(m)    = snowage(i,m,j)
      snw_rds_buf(m,-4) = snowrds5(i,m,j)
      snw_rds_buf(m,-3) = snowrds4(i,m,j)
      snw_rds_buf(m,-2) = snowrds3(i,m,j)
      snw_rds_buf(m,-1) = snowrds2(i,m,j)
      snw_rds_buf(m,0) = snowrds1(i,m,j)

      h2osoi_liq_buf(m,-4) = h2osoi_liq_s5(i,m,j)
      h2osoi_liq_buf(m,-3) = h2osoi_liq_s4(i,m,j)
      h2osoi_liq_buf(m,-2) = h2osoi_liq_s3(i,m,j)
      h2osoi_liq_buf(m,-1) = h2osoi_liq_s2(i,m,j)
      h2osoi_liq_buf(m,0)  = h2osoi_liq_s1(i,m,j)
      h2osoi_liq_buf(m,1)  = h2osoi_liq1(i,m,j)
      h2osoi_liq_buf(m,2)  = h2osoi_liq2(i,m,j)
      h2osoi_liq_buf(m,3)  = h2osoi_liq3(i,m,j)
      h2osoi_liq_buf(m,4)  = h2osoi_liq4(i,m,j)
      h2osoi_liq_buf(m,5)  = h2osoi_liq5(i,m,j)
      h2osoi_liq_buf(m,6)  = h2osoi_liq6(i,m,j)
      h2osoi_liq_buf(m,7)  = h2osoi_liq7(i,m,j)
      h2osoi_liq_buf(m,8)  = h2osoi_liq8(i,m,j)
      h2osoi_liq_buf(m,9)  = h2osoi_liq9(i,m,j)
      h2osoi_liq_buf(m,10) = h2osoi_liq10(i,m,j)

      h2osoi_ice_buf(m,-4) = h2osoi_ice_s5(i,m,j)
      h2osoi_ice_buf(m,-3) = h2osoi_ice_s4(i,m,j)
      h2osoi_ice_buf(m,-2) = h2osoi_ice_s3(i,m,j)
      h2osoi_ice_buf(m,-1) = h2osoi_ice_s2(i,m,j)
      h2osoi_ice_buf(m,0)  = h2osoi_ice_s1(i,m,j)
      h2osoi_ice_buf(m,1)  = h2osoi_ice1(i,m,j)
      h2osoi_ice_buf(m,2)  = h2osoi_ice2(i,m,j)
      h2osoi_ice_buf(m,3)  = h2osoi_ice3(i,m,j)
      h2osoi_ice_buf(m,4)  = h2osoi_ice4(i,m,j)
      h2osoi_ice_buf(m,5)  = h2osoi_ice5(i,m,j)
      h2osoi_ice_buf(m,6)  = h2osoi_ice6(i,m,j)
      h2osoi_ice_buf(m,7)  = h2osoi_ice7(i,m,j)
      h2osoi_ice_buf(m,8)  = h2osoi_ice8(i,m,j)
      h2osoi_ice_buf(m,9)  = h2osoi_ice9(i,m,j)
      h2osoi_ice_buf(m,10) = h2osoi_ice10(i,m,j)

      t_soisno_buf(m,-4) = t_soisno_s5(i,m,j)
      t_soisno_buf(m,-3) = t_soisno_s4(i,m,j)
      t_soisno_buf(m,-2) = t_soisno_s3(i,m,j)
      t_soisno_buf(m,-1) = t_soisno_s2(i,m,j)
      t_soisno_buf(m,0)  = t_soisno_s1(i,m,j)
      t_soisno_buf(m,1)  = t_soisno1(i,m,j)
      t_soisno_buf(m,2)  = t_soisno2(i,m,j)
      t_soisno_buf(m,3)  = t_soisno3(i,m,j)
      t_soisno_buf(m,4)  = t_soisno4(i,m,j)
      t_soisno_buf(m,5)  = t_soisno5(i,m,j)
      t_soisno_buf(m,6)  = t_soisno6(i,m,j)
      t_soisno_buf(m,7)  = t_soisno7(i,m,j)
      t_soisno_buf(m,8)  = t_soisno8(i,m,j)
      t_soisno_buf(m,9)  = t_soisno9(i,m,j)
      t_soisno_buf(m,10) = t_soisno10(i,m,j)


      t_lake_buf(m,1)  = t_lake1(i,m,j)
      t_lake_buf(m,2)  = t_lake2(i,m,j)
      t_lake_buf(m,3)  = t_lake3(i,m,j)
      t_lake_buf(m,4)  = t_lake4(i,m,j)
      t_lake_buf(m,5)  = t_lake5(i,m,j)
      t_lake_buf(m,6)  = t_lake6(i,m,j)
      t_lake_buf(m,7)  = t_lake7(i,m,j)
      t_lake_buf(m,8)  = t_lake8(i,m,j)
      t_lake_buf(m,9)  = t_lake9(i,m,j)
      t_lake_buf(m,10) = t_lake10(i,m,j)

      h2osoi_vol_buf(m,1)  = h2osoi_vol1(i,m,j)
      h2osoi_vol_buf(m,2)  = h2osoi_vol2(i,m,j)
      h2osoi_vol_buf(m,3)  = h2osoi_vol3(i,m,j)
      h2osoi_vol_buf(m,4)  = h2osoi_vol4(i,m,j)
      h2osoi_vol_buf(m,5)  = h2osoi_vol5(i,m,j)
      h2osoi_vol_buf(m,6)  = h2osoi_vol6(i,m,j)
      h2osoi_vol_buf(m,7)  = h2osoi_vol7(i,m,j)
      h2osoi_vol_buf(m,8)  = h2osoi_vol8(i,m,j)
      h2osoi_vol_buf(m,9)  = h2osoi_vol9(i,m,j)
      h2osoi_vol_buf(m,10) = h2osoi_vol10(i,m,j)

      t_grnd_buf(m)     = t_grnd(i,m,j)
      t_veg_buf(m)      = t_veg(i,m,j)
      h2ocan_buf(m)     = h2ocan(i,m,j)
      h2ocan_col_buf(m) = h2ocan_col(i,m,j)
      h2osno_buf(m)     = h2osno(i,m,j)
      albedosubgrid_buf(m) = albedosubgrid(i,m,j)
      lhsubgrid_buf(m)  = lhsubgrid(i,m,j)
      hfxsubgrid_buf(m) = hfxsubgrid(i,m,j)
      lwupsubgrid_buf(m)= lwupsubgrid(i,m,j)
      q2subgrid_buf(m)  = q2subgrid(i,m,j)
!ylu 01/14/09
      sabvsubgrid_buf(m) = sabvsubgrid(i,m,j)
      sabgsubgrid_buf(m) = sabgsubgrid(i,m,j)
      nrasubgrid_buf(m) = nrasubgrid(i,m,j)
      swupsubgrid_buf(m) = swupsubgrid(i,m,j)      
!ylu 04/07/09 add three component of LH to output file
     lhsoi_buf(m) = lhsoi(i,m,j)
     lhveg_buf(m) = lhveg(i,m,j)
     lhtran_buf(m) = lhtran(i,m,j) 
!!!
#ifdef CN
!ADD_NEW_VAR 02/14/2011
     tlai_buf(m) = dyntlai(i,m,j)
     tsai_buf(m) = dyntsai(i,m,j)
     htop_buf(m) = dyntop(i,m,j)
     hbot_buf(m) = dynbot(i,m,j)
#endif

     organic_buf(m) = organic(m)

      t_ref2m_buf(m)    = t_ref2m(i,m,j)
      q_ref2m_buf(m)    = q_ref2m(i,m,j)
#ifdef CN
!CROP CN VARS
!ylu 05/31/11
      htmx_buf(m)                   = htmx(i,m,j)
      croplive_buf(m)               = croplive(i,m,j)
      gdd1020_buf(m)                = gdd1020(i,m,j)
      gdd820_buf(m)                 = gdd820(i,m,j)
      gdd020_buf(m)                 = gdd020(i,m,j)
      grainc_buf(m)                 = grainc(i,m,j)
      grainc_storage_buf(m)         = grainc_storage(i,m,j)
      grainc_xfer_buf(m)            = grainc_xfer(i,m,j)
      grainn_buf(m)                 = grainn(i,m,j)
      grainn_storage_buf(m)         = grainn_storage(i,m,j)
      grainn_xfer_buf(m)            = grainn_xfer(i,m,j)
      days_active_buf(m)            = days_active(i,m,j)
      onset_flag_buf(m)             = onset_flag(i,m,j)
      onset_counter_buf(m)          = onset_counter(i,m,j)
      onset_gddflag_buf(m)          = onset_gddflag(i,m,j)
      onset_fdd_buf(m)              = onset_fdd(i,m,j) 
      onset_gdd_buf(m)              = onset_gdd(i,m,j)
      onset_swi_buf(m)              = onset_swi(i,m,j)
      offset_flag_buf(m)            = offset_flag(i,m,j)
      offset_counter_buf(m)         = offset_counter(i,m,j)
      offset_fdd_buf(m)             = offset_fdd(i,m,j)
      offset_swi_buf(m)             = offset_swi(i,m,j)
      dayl_buf(m)                   = dayl(i,m,j)
      annavg_t2m_buf(m)             = annavg_t2m(i,m,j)
      tempavg_t2m_buf(m)            = tempavg_t2m(i,m,j)
      tempsum_potential_gpp_buf(m)  = tempsum_potential_gpp(i,m,j)
      annsum_potential_gpp_buf(m)   = annsum_potential_gpp(i,m,j)
      tempmax_retransn_buf(m)       = tempmax_retransn(i,m,j)
      annmax_retransn_buf(m)        = annmax_retransn(i,m,j)
      prev_leafc_to_litter_buf(m)   = prev_leafc_to_litter(i,m,j)
      prev_frootc_to_litter_buf(m)  = prev_frootc_to_litter(i,m,j)
      tempsum_npp_buf(m)            = tempsum_npp(i,m,j)
      annsum_npp_buf(m)             = annsum_npp(i,m,j)
      leafc_buf(m)                  = leafc(i,m,j)
      leafc_storage_buf(m)          = leafc_storage(i,m,j)
      leafc_xfer_buf(m)             = leafc_xfer(i,m,j)
      frootc_buf(m)                 = frootc(i,m,j)
      frootc_storage_buf(m)         = frootc_storage(i,m,j)
      frootc_xfer_buf(m)            = frootc_xfer(i,m,j)
      livestemc_buf(m)              = livestemc(i,m,j)
      livestemc_storage_buf(m)      = livestemc_storage(i,m,j)
      livestemc_xfer_buf(m)         = livestemc_xfer(i,m,j)
      deadstemc_buf(m)              = deadstemc(i,m,j)
      deadstemc_storage_buf(m)      = deadstemc_storage(i,m,j)
      deadstemc_xfer_buf(m)         = deadstemc_xfer(i,m,j)
      livecrootc_buf(m)             = livecrootc(i,m,j)
      livecrootc_storage_buf(m)     = livecrootc_storage(i,m,j)
      livecrootc_xfer_buf(m)        = livecrootc_xfer(i,m,j)
      deadcrootc_buf(m)             = deadcrootc(i,m,j)
      deadcrootc_storage_buf(m)     = deadcrootc_storage(i,m,j)
      deadcrootc_xfer_buf(m)        = deadcrootc_xfer(i,m,j)
      cpool_buf(m)                  = cpool(i,m,j)
      pft_ctrunc_buf(m)             = pft_ctrunc(i,m,j)
      leafn_buf(m)                  = leafn(i,m,j)
      leafn_storage_buf(m)          = leafn_storage(i,m,j)
      leafn_xfer_buf(m)             = leafn_xfer(i,m,j)
      frootn_buf(m)                 = frootn(i,m,j)
      frootn_storage_buf(m)         = frootn_storage(i,m,j)
      frootn_xfer_buf(m)            = frootn_xfer(i,m,j)
      livestemn_buf(m)              = livestemn(i,m,j)
      livestemn_storage_buf(m)      = livestemn_storage(i,m,j)
      livestemn_xfer_buf(m)         = livestemn_xfer(i,m,j)
      deadstemn_buf(m)              = deadstemn(i,m,j)
      deadstemn_storage_buf(m)      = deadstemn_storage(i,m,j)
      deadstemn_xfer_buf(m)         = deadstemn_xfer(i,m,j)
      livecrootn_buf(m)             = livecrootn(i,m,j)
      livecrootn_storage_buf(m)     = livecrootn_storage(i,m,j)
      livecrootn_xfer_buf(m)        = livecrootn_xfer(i,m,j)
      deadcrootn_buf(m)             = deadcrootn(i,m,j)
      deadcrootn_storage_buf(m)     = deadcrootn_storage(i,m,j)
      deadcrootn_xfer_buf(m)        = deadcrootn_xfer(i,m,j)
      npool_buf(m)                  = npool(i,m,j)
      pft_ntrunc_buf(m)             = pft_ntrunc(i,m,j)
      gresp_storage_buf(m)          = gresp_storage(i,m,j)
      gresp_xfer_buf(m)             = gresp_xfer(i,m,j)
      xsmrpool_buf(m)               = xsmrpool(i,m,j)
      annsum_counter_buf(m)         = annsum_counter(i,m,j)
      cannsum_npp_buf(m)            = cannsum_npp(i,m,j)
      cannavg_t2m_buf(m)            = cannavg_t2m(i,m,j)
      wf_buf(m)                     = wf(i,m,j)
      me_buf(m)                     = me(i,m,j)
      mean_fire_prob_buf(m)         = mean_fire_prob(i,m,j)
      cwdc_buf(m)                   = cwdc(i,m,j)
      litr1c_buf(m)                 = litr1c(i,m,j)
      litr2c_buf(m)                 = litr2c(i,m,j)
      litr3c_buf(m)                 = litr3c(i,m,j)
      soil1c_buf(m)                 = soil1c(i,m,j)
      soil2c_buf(m)                 = soil2c(i,m,j)
      soil3c_buf(m)                 = soil3c(i,m,j)
      soil4c_buf(m)                 = soil4c(i,m,j)
      seedc_buf(m)                  = seedc(i,m,j)
      col_ctrunc_buf(m)             = col_ctrunc(i,m,j)
      prod10c_buf(m)                = prod10c(i,m,j)
      prod100c_buf(m)               = prod100c(i,m,j)
      cwdn_buf(m)                   = cwdn(i,m,j)
      litr1n_buf(m)                 = litr1n(i,m,j)
      litr2n_buf(m)                 = litr2n(i,m,j)
      litr3n_buf(m)                 = litr3n(i,m,j)
      soil1n_buf(m)                 = soil1n(i,m,j)
      soil2n_buf(m)                 = soil2n(i,m,j)
      soil3n_buf(m)                 = soil3n(i,m,j)
      soil4n_buf(m)                 = soil4n(i,m,j)
      seedn_buf(m)                  = seedn(i,m,j)
      col_ntrunc_buf(m)             = col_ntrunc(i,m,j)
      prod10n_buf(m)                = prod10n(i,m,j)
      prod100n_buf(m)               = prod100n(i,m,j)
      sminn_buf(m)                  = sminn(i,m,j)

      totlitc_buf(m)                = totlitc(i,m,j)
      dwt_seedc_to_leaf_buf(m)      = dwt_seedc_to_leaf(i,m,j)
      dwt_seedc_to_deadstem_buf(m)  = dwt_seedc_to_deadstem(i,m,j)
      dwt_conv_cflux_buf(m)         = dwt_conv_cflux(i,m,j) 
      dwt_prod10c_gain_buf(m)       = dwt_prod10c_gain(i,m,j)
      dwt_prod100c_gain_buf(m)      = dwt_prod100c_gain(i,m,j)
      prod100c_loss_buf(m)          = prod100c_loss(i,m,j)
      dwt_frootc_to_litr1c_buf(m)   = dwt_frootc_to_litr1c(i,m,j)
      dwt_frootc_to_litr2c_buf(m)   = dwt_frootc_to_litr2c(i,m,j)
      dwt_frootc_to_litr3c_buf(m)   = dwt_frootc_to_litr3c(i,m,j)
      dwt_livecrootc_to_cwdc_buf(m) = dwt_livecrootc_to_cwdc(i,m,j)
      dwt_deadcrootc_to_cwdc_buf(m) = dwt_deadcrootc_to_cwdc(i,m,j)
      dwt_seedn_to_leaf_buf(m)      = dwt_seedn_to_leaf(i,m,j)
      dwt_seedn_to_deadstem_buf(m)  = dwt_seedn_to_deadstem(i,m,j)
      dwt_conv_nflux_buf(m)         = dwt_conv_nflux(i,m,j)
      dwt_prod10n_gain_buf(m)       = dwt_prod10n_gain(i,m,j)
      dwt_prod100n_gain_buf(m)      = dwt_prod100n_gain(i,m,j)
      prod100n_loss_buf(m)          = prod100n_loss(i,m,j)
      dwt_frootn_to_litr1n_buf(m)   = dwt_frootn_to_litr1n(i,m,j)
      dwt_frootn_to_litr2n_buf(m)   = dwt_frootn_to_litr2n(i,m,j)
      dwt_frootn_to_litr3n_buf(m)   = dwt_frootn_to_litr3n(i,m,j)
      dwt_livecrootn_to_cwdn_buf(m) = dwt_livecrootn_to_cwdn(i,m,j)
      dwt_deadcrootn_to_cwdn_buf(m) = dwt_deadcrootn_to_cwdn(i,m,j)
      retransn_buf(m)               = retransn(i,m,j)
#endif
    end do !!!!!!!!!!!!!!!!! m=1, maxpatch
       
   !  if(lndmsk  == 1) then

          call clm(forc_txy_buf        ,forc_uxy_buf           ,forc_vxy_buf    &
                  ,forc_qxy_buf        ,zgcmxy_buf             ,prec_buf        &
                  ,flwdsxy_buf         ,forc_sols_buf          ,forc_soll_buf   &
                  ,forc_solsd_buf      ,forc_solld_buf         ,forc_pbot_buf   &
                  ,forc_psrfxy_buf     ,ivgtyp(i,j)            ,isltyp(i,j)     &
                  ,lndmsk              ,xlat_buf               ,xlon_buf        &
                  ,area_buf            ,dtt                    ,myr             &
                  ,mon                 ,mday                   ,msec            &
                  ,calday              ,myr1                   ,mon1            &
                  ,mday1               ,msec1                  ,calday1         &
                  ,mbdate              ,qsfxy_buf              ,qdnxy_buf       &
                  ,snl1(i,:,j)          ,snowdp_buf            ,snw_rds_buf     & 
                  ,dzclm               ,zclm                   ,ziclm           &
                  ,h2osno_buf          ,h2osoi_liq_buf         ,h2osoi_ice_buf  &
                  ,t_grnd_buf          ,t_soisno_buf           ,t_lake_buf      &
                  ,t_veg_buf           ,h2ocan_buf             ,h2ocan_col_buf  &
                  ,h2osoi_vol_buf      ,wtc_buf                ,wtp_buf         &
                  ,numc(i,j)           ,nump(i,j)                &
                  ,t_ref2m_buf         ,albxy_buf        ,tsxy_buf,  trefxy_buf        &
                  ,shxy_buf            ,lhxy_buf               ,nstep           &
                  ,inest               ,i                      ,j               &
                  ,soiflx_buf          ,sabv_buf               ,sabg_buf        &
                  ,lwup_buf            ,znt_buf                ,q_ref2m_buf     &
                  ,rhoxy_buf                                                    &
                  ,albedosubgrid_buf   ,lhsubgrid_buf          ,hfxsubgrid_buf  &
                  ,lwupsubgrid_buf     ,q2subgrid_buf          ,sabvsubgrid_buf &
                  ,sabgsubgrid_buf     ,nrasubgrid_buf         ,swupsubgrid_buf &
                  ,lhsoi_buf          ,lhveg_buf               ,lhtran_buf      &
                  ,organic_buf        ,efisop_buf              ,gti_buf         &
                  ,alswnirdir_buf  ,alswnirdif_buf,alswvisdir_buf,alswvisdif_buf&
#ifdef CN
!CROP and CN restart and outputs
                ,forc_ndepxy_buf    ,organic_buf  ,tlai_buf ,tsai_buf,htop_buf,hbot_buf    &  !ADD_NEW_VAR
                ,htmx_buf,croplive_buf,gdd1020_buf,gdd820_buf,gdd020_buf,grainc_buf,grainc_storage_buf  &
                ,grainc_xfer_buf,grainn_buf,grainn_storage_buf,grainn_xfer_buf,days_active_buf  &
                ,onset_flag_buf,onset_counter_buf,onset_gddflag_buf,onset_fdd_buf,onset_gdd_buf &
                ,onset_swi_buf,offset_flag_buf,offset_counter_buf,offset_fdd_buf,offset_swi_buf &
                ,dayl_buf,annavg_t2m_buf,tempavg_t2m_buf,tempsum_potential_gpp_buf          &
                ,annsum_potential_gpp_buf,tempmax_retransn_buf,annmax_retransn_buf      &
                ,prev_leafc_to_litter_buf,prev_frootc_to_litter_buf,tempsum_npp_buf     &
                ,annsum_npp_buf,leafc_buf,leafc_storage_buf,leafc_xfer_buf,frootc_buf           &
                ,frootc_storage_buf,frootc_xfer_buf,livestemc_buf,livestemc_storage_buf     &
                ,livestemc_xfer_buf,deadstemc_buf,deadstemc_storage_buf,deadstemc_xfer_buf  &
                ,livecrootc_buf,livecrootc_storage_buf,livecrootc_xfer_buf,deadcrootc_buf   &
                ,deadcrootc_storage_buf,deadcrootc_xfer_buf,cpool_buf,pft_ctrunc_buf        &
                ,leafn_buf,leafn_storage_buf,leafn_xfer_buf,frootn_buf,frootn_storage_buf       &
                ,frootn_xfer_buf,livestemn_buf,livestemn_storage_buf,livestemn_xfer_buf     &
                ,deadstemn_buf,deadstemn_storage_buf,deadstemn_xfer_buf,livecrootn_buf      &
                ,livecrootn_storage_buf,livecrootn_xfer_buf,deadcrootn_buf              &
                ,deadcrootn_storage_buf,deadcrootn_xfer_buf,npool_buf,pft_ntrunc_buf        &
                ,gresp_storage_buf,gresp_xfer_buf,xsmrpool_buf,annsum_counter_buf           &
                ,cannsum_npp_buf,cannavg_t2m_buf,wf_buf,me_buf,mean_fire_prob_buf,cwdc_buf,litr1c_buf   &
                ,litr2c_buf,litr3c_buf,soil1c_buf,soil2c_buf,soil3c_buf,soil4c_buf,seedc_buf,col_ctrunc_buf &
                ,prod10c_buf,prod100c_buf,cwdn_buf,litr1n_buf,litr2n_buf,litr3n_buf,soil1n_buf,soil2n_buf   &
                ,soil3n_buf,soil4n_buf,seedn_buf,col_ntrunc_buf,prod10n_buf,prod100n_buf,sminn_buf &
               ,totlitc_buf,dwt_seedc_to_leaf_buf,dwt_seedc_to_deadstem_buf,dwt_conv_cflux_buf &
                ,dwt_prod10c_gain_buf,dwt_prod100c_gain_buf,prod100c_loss_buf,dwt_frootc_to_litr1c_buf &
                ,dwt_frootc_to_litr2c_buf,dwt_frootc_to_litr3c_buf,dwt_livecrootc_to_cwdc_buf &
                ,dwt_deadcrootc_to_cwdc_buf,dwt_seedn_to_leaf_buf,dwt_seedn_to_deadstem_buf &
                ,dwt_conv_nflux_buf,dwt_prod10n_gain_buf,dwt_prod100n_gain_buf,prod100n_loss_buf &
                ,dwt_frootn_to_litr1n_buf,dwt_frootn_to_litr2n_buf, dwt_frootn_to_litr3n_buf &
                , dwt_livecrootn_to_cwdn_buf,dwt_deadcrootn_to_cwdn_buf,retransn_buf &                                                 
#endif
                  )
                 
                 if(albxy_buf ==  1) albxy_buf = 0.991

                  albxy(i,j)          = albxy_buf
                  call CLMDebug('get albxy')
                  snowh(i,j)          = sum(snowdp_buf(1:numc(i,j))*wtc_buf(1:numc(i,j)))
                  call CLMDebug('get snowh')
                  snow(i,j)           = sum(h2osno_buf(1:numc(i,j))*wtc_buf(1:numc(i,j)))
                  call CLMDebug('get snow')
                  canwat(i,j)         = sum(h2ocan_buf(1:nump(i,j))*wtp_buf(1:nump(i,j)))
                 call CLMDebug('get canwat') 
              if (ivgtyp(i,j) /= 16 .and. ivgtyp(i,j) /= 24) then
                  do k=1,nlevgrnd
                     smois(i,k,j)     = sum(h2osoi_vol_buf(1:numc(i,j),k)*wtc_buf(1:numc(i,j)))
                     tslb (i,k,j)     = sum(t_soisno_buf(1:numc(i,j),k)*wtc_buf(1:numc(i,j)))
                  end do !over levels
              end if
                  call CLMDebug('get tslb')
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                  tsxy(i,j)           = tsxy_buf
                  qsfxy(i,j)          = qsfxy_buf
                  qdnxy(i,j)          = qdnxy_buf
                  soiflx(i,j)         = soiflx_buf
                  sabv(i,j)           = sabv_buf
                  sabg(i,j)           = sabg_buf
                  lwup(i,j)           = lwup_buf
                  znt(i,j)            = znt_buf
                  z0(i,j)             = znt(i,j)
                  alswvisdir(i,j)     = alswvisdir_buf
                  alswvisdif(i,j)     = alswvisdif_buf
                  alswnirdir(i,j)     = alswnirdir_buf
                  alswnirdif(i,j)     = alswnirdif_buf

                  t2clm(i,j)       = trefxy_buf
!Accumulate T2 MAX/MIN between history interval --Yaqiong Lu
                  if(mod(dt*(nstep-1),60.*history_interval)==0) then 
                    t2m_max(i,j) = 0.0
                    t2m_min(i,j) = 999.0
                  else
                    t2m_max(i,j) = max(t2m_max(i,j),t2clm(i,j)) !the t2m_max/min will save the max/min along each history interval chunk.  
                    t2m_min(i,j) = min(t2m_min(i,j),t2clm(i,j))
                  end if
             call CLMDebug('module clm mark1')
                  emiss(i,j) = lwup(i,j)/(sb * tsxy(i,j)**4)
                  shxy(i,j)           = shxy_buf
                  lhxy(i,j)           = lhxy_buf
                  if(tsxy(i,j)>=tfrz) then
                    qfx(i,j)          = lhxy_buf/hvap ! heat flux (W/m^2)=>mass flux(kg/(sm^2))
                  else
                    qfx(i,j)          = lhxy_buf/hsub ! heat flux (W/m^2)=>mass flux(kg/(sm^2))     
                  end if
                  
                  qsfc(i,j)  = forc_qxy(i,1,j) +qfx(i,j)/(rhoxy_buf*chs(i,j))

              do m=1,maxpatch
                  snowdp(i,m,j)       = snowdp_buf(m)
      !            snowage(i,m,j)      = snowage_buf(m)
 
                  snl(i,m,j)         = snl1(i,m,j)

                  dzsnow5(i,m,j)      = dzclm(m,-4)
                  dzsnow4(i,m,j)      = dzclm(m,-3)
                  dzsnow3(i,m,j)      = dzclm(m,-2)
                  dzsnow2(i,m,j)      = dzclm(m,-1)
                  dzsnow1(i,m,j)      = dzclm(m,0)

                  snowrds5(i,m,j)     = snw_rds_buf(m,-4)
                  snowrds4(i,m,j)     = snw_rds_buf(m,-3)
                  snowrds3(i,m,j)     = snw_rds_buf(m,-2)
                  snowrds2(i,m,j)     = snw_rds_buf(m,-1)
                  snowrds1(i,m,j)     = snw_rds_buf(m,0)

                  h2osno(i,m,j)       = h2osno_buf(m)
                  t_grnd(i,m,j)       = t_grnd_buf(m)
                  t_veg(i,m,j)        = t_veg_buf(m)
                  h2ocan(i,m,j)       = h2ocan_buf(m)
                  h2ocan_col(i,m,j)   = h2ocan_col_buf(m)
                  wtc(i,m,j)          = wtc_buf(m)
                  wtp(i,m,j)          = wtp_buf(m)
            call CLMDebug('module clm mark2')
                  
                  h2osoi_liq_s5(i,m,j) = h2osoi_liq_buf(m,-4)
                  h2osoi_liq_s4(i,m,j) = h2osoi_liq_buf(m,-3)
                  h2osoi_liq_s3(i,m,j) = h2osoi_liq_buf(m,-2)
                  h2osoi_liq_s2(i,m,j) = h2osoi_liq_buf(m,-1)
                  h2osoi_liq_s1(i,m,j) = h2osoi_liq_buf(m,0)
                  h2osoi_liq1(i,m,j)   = h2osoi_liq_buf(m,1)
                  h2osoi_liq2(i,m,j)   = h2osoi_liq_buf(m,2)
                  h2osoi_liq3(i,m,j)   = h2osoi_liq_buf(m,3)
                  h2osoi_liq4(i,m,j)   = h2osoi_liq_buf(m,4)
                  h2osoi_liq5(i,m,j)   = h2osoi_liq_buf(m,5)
                  h2osoi_liq6(i,m,j)   = h2osoi_liq_buf(m,6)
                  h2osoi_liq7(i,m,j)   = h2osoi_liq_buf(m,7)
                  h2osoi_liq8(i,m,j)   = h2osoi_liq_buf(m,8)
                  h2osoi_liq9(i,m,j)   = h2osoi_liq_buf(m,9)
                  h2osoi_liq10(i,m,j)  = h2osoi_liq_buf(m,10)

                  h2osoi_ice_s5(i,m,j) = h2osoi_ice_buf(m,-4)
                  h2osoi_ice_s4(i,m,j) = h2osoi_ice_buf(m,-3)
                  h2osoi_ice_s3(i,m,j) = h2osoi_ice_buf(m,-2)
                  h2osoi_ice_s2(i,m,j) = h2osoi_ice_buf(m,-1)
                  h2osoi_ice_s1(i,m,j) = h2osoi_ice_buf(m,0)
                  h2osoi_ice1(i,m,j)   = h2osoi_ice_buf(m,1)
                  h2osoi_ice2(i,m,j)   = h2osoi_ice_buf(m,2)
                  h2osoi_ice3(i,m,j)   = h2osoi_ice_buf(m,3)
                  h2osoi_ice4(i,m,j)   = h2osoi_ice_buf(m,4)
                  h2osoi_ice5(i,m,j)   = h2osoi_ice_buf(m,5)
                  h2osoi_ice6(i,m,j)   = h2osoi_ice_buf(m,6)
                  h2osoi_ice7(i,m,j)   = h2osoi_ice_buf(m,7)
                  h2osoi_ice8(i,m,j)   = h2osoi_ice_buf(m,8)
                  h2osoi_ice9(i,m,j)   = h2osoi_ice_buf(m,9)
                  h2osoi_ice10(i,m,j)  = h2osoi_ice_buf(m,10)

            call CLMDebug('module clm mark3')

                  t_soisno_s5(i,m,j) = t_soisno_buf(m,-4)
                  t_soisno_s4(i,m,j) = t_soisno_buf(m,-3)
                  t_soisno_s3(i,m,j) = t_soisno_buf(m,-2)
                  t_soisno_s2(i,m,j) = t_soisno_buf(m,-1)
                  t_soisno_s1(i,m,j) = t_soisno_buf(m,0)
                  t_soisno1(i,m,j)   = t_soisno_buf(m,1)
                  t_soisno2(i,m,j)   = t_soisno_buf(m,2)
                  t_soisno3(i,m,j)   = t_soisno_buf(m,3)
                  t_soisno4(i,m,j)   = t_soisno_buf(m,4)
                  t_soisno5(i,m,j)   = t_soisno_buf(m,5)
                  t_soisno6(i,m,j)   = t_soisno_buf(m,6)
                  t_soisno7(i,m,j)   = t_soisno_buf(m,7)
                  t_soisno8(i,m,j)   = t_soisno_buf(m,8)
                  t_soisno9(i,m,j)   = t_soisno_buf(m,9)
                  t_soisno10(i,m,j)  = t_soisno_buf(m,10)


                  t_lake1(i,m,j)   = t_lake_buf(m,1)
                  t_lake2(i,m,j)   = t_lake_buf(m,2)
                  t_lake3(i,m,j)   = t_lake_buf(m,3)
                  t_lake4(i,m,j)   = t_lake_buf(m,4)
                  t_lake5(i,m,j)   = t_lake_buf(m,5)
                  t_lake6(i,m,j)   = t_lake_buf(m,6)
                  t_lake7(i,m,j)   = t_lake_buf(m,7)
                  t_lake8(i,m,j)   = t_lake_buf(m,8)
                  t_lake9(i,m,j)   = t_lake_buf(m,9)
                  t_lake10(i,m,j)  = t_lake_buf(m,10)

                  h2osoi_vol1(i,m,j)   = h2osoi_vol_buf(m,1)
                  h2osoi_vol2(i,m,j)   = h2osoi_vol_buf(m,2)
                  h2osoi_vol3(i,m,j)   = h2osoi_vol_buf(m,3)
                  h2osoi_vol4(i,m,j)   = h2osoi_vol_buf(m,4)
                  h2osoi_vol5(i,m,j)   = h2osoi_vol_buf(m,5)
                  h2osoi_vol6(i,m,j)   = h2osoi_vol_buf(m,6)
                  h2osoi_vol7(i,m,j)   = h2osoi_vol_buf(m,7)
                  h2osoi_vol8(i,m,j)   = h2osoi_vol_buf(m,8)
                  h2osoi_vol9(i,m,j)   = h2osoi_vol_buf(m,9)
                  h2osoi_vol10(i,m,j)  = h2osoi_vol_buf(m,10)

            call CLMDebug('module clm mark4')

                  t_ref2m(i,m,j)      = t_ref2m_buf(m)    
                  q_ref2m(i,m,j)      = q_ref2m_buf(m)
!!!!New patch-level variables
                  albedosubgrid(i,m,j)= albedosubgrid_buf(m)
                  lhsubgrid(i,m,j)    = lhsubgrid_buf(m)
                  hfxsubgrid(i,m,j)   = hfxsubgrid_buf(m)
                  lwupsubgrid(i,m,j)  = lwupsubgrid_buf(m)
                  q2subgrid(i,m,j)    = q2subgrid_buf(m)
!!ylu 01/14/09
                  sabvsubgrid(i,m,j)  = sabvsubgrid_buf(m)
                  sabgsubgrid(i,m,j)  = sabgsubgrid_buf(m)
                  nrasubgrid(i,m,j)   = nrasubgrid_buf(m)
                  swupsubgrid(i,m,j)  = swupsubgrid_buf(m)
!ylu 04/07/09
                  lhsoi(i,m,j)       = lhsoi_buf(m)
                  lhveg(i,m,j)       = lhveg_buf(m)
                  lhtran(i,m,j)      = lhtran_buf(m)
#ifdef CN                  
                  dyntlai(i,m,j)     = tlai_buf(m)
                  dyntsai(i,m,j)     = tsai_buf(m)
                  dyntop(i,m,j)      = htop_buf(m)
                  dynbot(i,m,j)      = hbot_buf(m)

            call CLMDebug('module clm mark5')
!CROP CN VARS
!ylu 05/31/11
                    htmx(i,m,j)                        = htmx_buf(m)
                    croplive(i,m,j)                    = croplive_buf(m)
                    gdd1020(i,m,j)                     = gdd1020_buf(m)
                    gdd820(i,m,j)                      = gdd820_buf(m)
                    gdd020(i,m,j)                      = gdd020_buf(m)
                    grainc(i,m,j)                      = grainc_buf(m)
                    grainc_storage(i,m,j)              = grainc_storage_buf(m) 
                    grainc_xfer(i,m,j)                 = grainc_xfer_buf(m)
                    grainn(i,m,j)                      = grainn_buf(m)
                    grainn_storage(i,m,j)              = grainn_storage_buf(m)
                    grainn_xfer(i,m,j)                 = grainn_xfer_buf(m)
                    days_active(i,m,j)                 = days_active_buf(m)
                    onset_flag(i,m,j)                  = onset_flag_buf(m)
                    onset_counter(i,m,j)               = onset_counter_buf(m)
                    onset_gddflag(i,m,j)               = onset_gddflag_buf(m)
                    onset_fdd(i,m,j)                   = onset_fdd_buf(m)
                    onset_gdd(i,m,j)                   = onset_gdd_buf(m)
                    onset_swi(i,m,j)                   = onset_swi_buf(m)
                    offset_flag(i,m,j)                 = offset_flag_buf(m) 
                    offset_counter(i,m,j)              = offset_counter_buf(m)
                    offset_fdd(i,m,j)                  = offset_fdd_buf(m)
                    offset_swi(i,m,j)                  = offset_swi_buf(m)
                    dayl(i,m,j)                        = dayl_buf(m)
                    annavg_t2m(i,m,j)                  = annavg_t2m_buf(m)
                    tempavg_t2m(i,m,j)                 = tempavg_t2m_buf(m)
                    tempsum_potential_gpp(i,m,j)       = tempsum_potential_gpp_buf(m)
                    annsum_potential_gpp(i,m,j)        = annsum_potential_gpp_buf(m)
                    tempmax_retransn(i,m,j)            = tempmax_retransn_buf(m)
                    annmax_retransn(i,m,j)             = annmax_retransn_buf(m) 
                    prev_leafc_to_litter(i,m,j)        = prev_leafc_to_litter_buf(m)
                    prev_frootc_to_litter(i,m,j)       = prev_frootc_to_litter_buf(m)
                    tempsum_npp(i,m,j)                 = tempsum_npp_buf(m)
                    annsum_npp(i,m,j)                  = annsum_npp_buf(m)
                    leafc(i,m,j)                       = annsum_npp_buf(m)
                    leafc_storage(i,m,j)               = leafc_storage_buf(m)
                    leafc_xfer(i,m,j)                  = leafc_xfer_buf(m)
                    frootc(i,m,j)                      = frootc_buf(m)
                    frootc_storage(i,m,j)              = frootc_storage_buf(m)
                    frootc_xfer(i,m,j)                 = frootc_xfer_buf(m)
                    livestemc(i,m,j)                   = livestemc_buf(m)
                    livestemc_storage(i,m,j)           = livestemc_storage_buf(m)
                    livestemc_xfer(i,m,j)              = livestemc_xfer_buf(m)
                    deadstemc(i,m,j)                   = deadstemc_buf(m)
                    deadstemc_storage(i,m,j)           = deadstemc_storage_buf(m)
                    deadstemc_xfer(i,m,j)              = deadstemc_xfer_buf(m)
                    livecrootc(i,m,j)                  = livecrootc_buf(m)
                    livecrootc_storage(i,m,j)          = livecrootc_storage_buf(m)
                    livecrootc_xfer(i,m,j)             = livecrootc_xfer_buf(m)
                    deadcrootc(i,m,j)                  = deadcrootc_buf(m)
                    deadcrootc_storage(i,m,j)          = deadcrootc_storage_buf(m)
                    deadcrootc_xfer(i,m,j)             = deadcrootc_xfer_buf(m)
                    cpool(i,m,j)                       = cpool_buf(m)
                    pft_ctrunc(i,m,j)                  = pft_ctrunc_buf(m)
                    leafn(i,m,j)                       = leafn_buf(m)
                    leafn_storage(i,m,j)               = leafn_storage_buf(m)
                    leafn_xfer(i,m,j)                  = leafn_xfer_buf(m)
                    frootn(i,m,j)                      = frootn_buf(m)
                    frootn_storage(i,m,j)              = frootn_storage_buf(m)
                    frootn_xfer(i,m,j)                 = frootn_xfer_buf(m)
                    livestemn(i,m,j)                   = livestemn_buf(m)
                    livestemn_storage(i,m,j)           = livestemn_storage_buf(m)
                    livestemn_xfer(i,m,j)              = livestemn_xfer_buf(m)
                    deadstemn(i,m,j)                   = deadstemn_buf(m)
                    deadstemn_storage(i,m,j)           = deadstemn_storage_buf(m)
                    deadstemn_xfer(i,m,j)              = deadstemn_xfer_buf(m)
                    livecrootn(i,m,j)                  = livecrootn_buf(m)
                    livecrootn_storage(i,m,j)          = livecrootn_storage_buf(m)
                    livecrootn_xfer(i,m,j)             = livecrootn_xfer_buf(m)
                    deadcrootn(i,m,j)                  = deadcrootn_buf(m)
                    deadcrootn_storage(i,m,j)          = deadcrootn_storage_buf(m)
                    deadcrootn_xfer(i,m,j)             = deadcrootn_xfer_buf(m)
                    npool(i,m,j)                       = npool_buf(m)
                    pft_ntrunc(i,m,j)                  = pft_ntrunc_buf(m)
                    gresp_storage(i,m,j)               = gresp_storage_buf(m)
                    gresp_xfer(i,m,j)                  = gresp_xfer_buf(m) 
                    xsmrpool(i,m,j)                    = xsmrpool_buf(m)
                    annsum_counter(i,m,j)              = annsum_counter_buf(m)
                    cannsum_npp(i,m,j)                 = cannsum_npp_buf(m)
                    cannavg_t2m(i,m,j)                 = cannavg_t2m_buf(m)
                    wf(i,m,j)                          = wf_buf(m)
                    me(i,m,j)                          = me_buf(m)
                    mean_fire_prob(i,m,j)              = mean_fire_prob_buf(m)
                    cwdc(i,m,j)                        = cwdc_buf(m)
                    litr1c(i,m,j)                      = litr1c_buf(m)
                    litr2c(i,m,j)                      = litr2c_buf(m)
                    litr3c(i,m,j)                      = litr3c_buf(m) 
                    soil1c(i,m,j)                      = soil1c_buf(m)
                    soil2c(i,m,j)                      = soil2c_buf(m)
                    soil3c(i,m,j)                      = soil3c_buf(m)
                    soil4c(i,m,j)                      = soil4c_buf(m)
                    seedc(i,m,j)                       = seedc_buf(m)
                    col_ctrunc(i,m,j)                  = col_ctrunc_buf(m)
                    prod10c(i,m,j)                     = prod10c_buf(m)
                    prod100c(i,m,j)                    = prod100c_buf(m)
                    cwdn(i,m,j)                        = cwdn_buf(m)
                    litr1n(i,m,j)                      = litr1n_buf(m) 
                    litr2n(i,m,j)                      = litr2n_buf(m)
                    litr3n(i,m,j)                      = litr3n_buf(m) 
                    soil1n(i,m,j)                      = soil1n_buf(m)
                    soil2n(i,m,j)                      = soil2n_buf(m)
                    soil3n(i,m,j)                      = soil3n_buf(m)
                    soil4n(i,m,j)                      = soil4n_buf(m)
                    seedn(i,m,j)                       = seedn_buf(m)
                    col_ntrunc(i,m,j)                  = col_ntrunc_buf(m)
                    prod10n(i,m,j)                     = prod10n_buf(m)
                    prod100n(i,m,j)                    = prod100n_buf(m)
                    sminn(i,m,j)                       = sminn_buf(m) 
                    totlitc(i,m,j)                     = totlitc_buf(m)
                    dwt_seedc_to_leaf(i,m,j)           = dwt_seedc_to_leaf_buf(m)
                    dwt_seedc_to_deadstem(i,m,j)       = dwt_seedc_to_deadstem_buf(m)
                    dwt_conv_cflux(i,m,j)              = dwt_conv_cflux_buf(m)
                    dwt_prod10c_gain(i,m,j)            = dwt_prod10c_gain_buf(m)
                    dwt_prod100c_gain(i,m,j)           = dwt_prod100c_gain_buf(m)
                    prod100c_loss(i,m,j)               = prod100c_loss_buf(m)
                    dwt_frootc_to_litr1c(i,m,j)        = dwt_frootc_to_litr1c_buf(m)
                    dwt_frootc_to_litr2c(i,m,j)        = dwt_frootc_to_litr2c_buf(m)
                    dwt_frootc_to_litr3c(i,m,j)        = dwt_frootc_to_litr3c_buf(m)
                    dwt_livecrootc_to_cwdc(i,m,j)      = dwt_livecrootc_to_cwdc_buf(m)
                    dwt_deadcrootc_to_cwdc(i,m,j)      = dwt_deadcrootc_to_cwdc_buf(m)
                    dwt_seedn_to_leaf(i,m,j)           = dwt_seedn_to_leaf_buf(m)
                    dwt_seedn_to_deadstem(i,m,j)       = dwt_seedn_to_deadstem_buf(m)
                    dwt_conv_nflux(i,m,j)              = dwt_conv_nflux_buf(m)
                    dwt_prod10n_gain(i,m,j)            = dwt_prod10n_gain_buf(m)
                    dwt_prod100n_gain(i,m,j)           = dwt_prod100n_gain_buf(m)
                    prod100n_loss(i,m,j)               = prod100n_loss_buf(m)
                    dwt_frootn_to_litr1n(i,m,j)        = dwt_frootn_to_litr1n_buf(m)
                    dwt_frootn_to_litr2n(i,m,j)        = dwt_frootn_to_litr2n_buf(m)
                    dwt_frootn_to_litr3n(i,m,j)        = dwt_frootn_to_litr3n_buf(m)
                    dwt_livecrootn_to_cwdn(i,m,j)      =  dwt_livecrootn_to_cwdn_buf(m)
                    dwt_deadcrootn_to_cwdn(i,m,j)      = dwt_deadcrootn_to_cwdn_buf(m)
                    retransn(i,m,j)                    = retransn_buf(m)
#endif


              end do !!!!!!!!!!!!! m = 1, maxpatch
      end if
        call CLMDebug('good before call urban')
        IF (sf_urban_physics == 1 ) THEN                                              ! Beginning of UCM CALL if block
!--------------------------------------
! URBAN CANOPY MODEL START - urban
!--------------------------------------
! Input variables lsm --> urban


          IF( IVGTYP(I,J) == 1 .or. IVGTYP(I,J) == 31 .or. &
              IVGTYP(I,J) == 32 .or. IVGTYP(I,J) == 33 ) THEN

! Call urban
      forc_sols_buf  = swd_buf*0.35
      forc_soll_buf  = swd_buf*0.35
      forc_solsd_buf = swd_buf*0.15
      forc_solld_buf = swd_buf*0.15
      area_buf       = dx*dx/1.e6 !(km^2)
      forc_pbot_buf  = ps(i,1,j) ! Pa
      forc_txy_buf   = forc_txy(i,1,j)
      forc_uxy_buf   = forc_uxy(i,1,j)
      forc_vxy_buf   = forc_vxy(i,1,j)
      forc_qxy_buf   = forc_qxy(i,1,j)
      zgcmxy_buf     = zgcmxy(i,1,j)
      prec_buf       = prec(i,j)/dtt ! mm/s
      flwdsxy_buf    = flwdsxy(i,j) 
      forc_psrfxy_buf= ps(i,1,j) !  Pa
!
            UTYPE_URB = UTYPE_URB2D(I,J) !urban type (low, high or industrial)

            TA_URB    = forc_txy(i,1,j)  ! [K]
            QA_URB    = forc_qxy(i,1,j)  ! [kg/kg]
            UA_URB    = SQRT(forc_uxy(i,1,j)**2.+forc_vxy(i,1,j)**2.)
            U1_URB    = forc_uxy(i,1,j)
            V1_URB    = forc_vxy(i,1,j)
            IF(UA_URB < 1.) UA_URB=1.    ! [m/s]
            SSG_URB   = swd_buf            ! [W/m/m]
            SSGD_URB  = 0.8*swd_buf        ! [W/m/m]
            SSGQ_URB  = SSG_URB-SSGD_URB ! [W/m/m]
            LLG_URB   = flwdsxy(i,j)    ! [W/m/m]
            RAIN_URB  = prec(i,j)      ! [mm]
            RHOO_URB  = ps(i,1,j)/(287.04 * forc_txy(i,1,j) * (1.0+ 0.61 * forc_qxy(i,1,j))) ![kg/m/m/m]
            ZA_URB    = zgcmxy_buf             ! [m]
            DELT_URB  = DT               ! [sec]
            XLAT_URB  = XLAT_URB2D(I,J)  ! [deg]
            COSZ_URB  = COSZ_URB2D(I,J)  !
            OMG_URB   = OMG_URB2D(I,J)   !
            ZNT_URB   = ZNT(I,J)

            LSOLAR_URB = .FALSE.

            TR_URB = TR_URB2D(I,J)
            TB_URB = TB_URB2D(I,J)
            TG_URB = TG_URB2D(I,J)
            TC_URB = TC_URB2D(I,J)
            QC_URB = QC_URB2D(I,J)
            UC_URB = UC_URB2D(I,J)

            DO K = 1,num_roof_layers
              TRL_URB(K) = TRL_URB3D(I,K,J)
            END DO
            DO K = 1,num_wall_layers
              TBL_URB(K) = TBL_URB3D(I,K,J)
            END DO
            DO K = 1,num_road_layers
              TGL_URB(K) = TGL_URB3D(I,K,J)
            END DO

            XXXR_URB = XXXR_URB2D(I,J)
            XXXB_URB = XXXB_URB2D(I,J)
            XXXG_URB = XXXG_URB2D(I,J)
            XXXC_URB = XXXC_URB2D(I,J)
!
            CHS_URB  = CHS(I,J)
            CHS2_URB = CHS2(I,J)
! Jin
            IF (PRESENT(CMR_SFCDIF)) THEN
               CMR_URB = CMR_SFCDIF(I,J)
               CHR_URB = CHR_SFCDIF(I,J)
               CMC_URB = CMC_SFCDIF(I,J)
               CHC_URB = CHC_SFCDIF(I,J)
            ENDIF
! initialize NUDAPT variables to zero
            lp_urb = 0.
            lb_urb = 0.
            hgt_urb = 0.
            mh_urb = 0.
            stdh_urb = 0.
            do k = 1,4
              lf_urb(k) = 0.
            enddo
            frc_urb = FRC_URB2D(I,J)
            check = 0.
!

! Call urban
            CALL urban(LSOLAR_URB,                                      & ! I
                       num_roof_layers,num_wall_layers,num_road_layers, & ! C
                       DZR,DZB,DZG,                                     & ! C
                       UTYPE_URB,TA_URB,QA_URB,UA_URB,U1_URB,V1_URB,SSG_URB, & ! I
                       SSGD_URB,SSGQ_URB,LLG_URB,RAIN_URB,RHOO_URB,     & ! I
                       ZA_URB,DECLIN_URB,COSZ_URB,OMG_URB,              & ! I
                       XLAT_URB,DELT_URB,ZNT_URB,                       & ! I
                       CHS_URB, CHS2_URB,                               & ! I
                       TR_URB, TB_URB, TG_URB, TC_URB, QC_URB,UC_URB,   & ! H
                       TRL_URB,TBL_URB,TGL_URB,                         & ! H
                       XXXR_URB, XXXB_URB, XXXG_URB, XXXC_URB,          & ! H
                       TS_URB,QS_URB,SH_URB,LH_URB,LH_KINEMATIC_URB,    & ! O
                       SW_URB,ALB_URB,LW_URB,G_URB,RN_URB,PSIM_URB,PSIH_URB, & ! O
                       GZ1OZ0_URB,                                      & !O
                       CMR_URB, CHR_URB, CMC_URB, CHC_URB,              &
                       U10_URB, V10_URB, TH2_URB, Q2_URB,               & ! O
                       UST_URB,mh_urb, stdh_urb, lf_urb, lp_urb,        & ! 0
                       hgt_urb,frc_urb,lb_urb, check)

            TS_URB2D(I,J) = TS_URB

            albxy(i,j) = FRC_URB2D(I,J)*ALB_URB+(1-FRC_URB2D(I,J))*albxy_buf   ![-]
            shxy(i,j)  = FRC_URB2D(I,J)*SH_URB+(1-FRC_URB2D(I,J))*shxy_buf     ![W/m/m]
            qfx(i,j)   = FRC_URB2D(I,J)*LH_KINEMATIC_URB &
                     + (1-FRC_URB2D(I,J))*qfx(i,j)                ![kg/m/m/s]
            lhxy(i,j) = FRC_URB2D(I,J)*LH_URB+(1-FRC_URB2D(I,J))*lhxy_buf     ![W/m/m]
            soiflx(i,j) = FRC_URB2D(I,J)*G_URB+(1-FRC_URB2D(I,J))*soiflx_buf  ![W/m/m]
            tsxy(i,j)  = FRC_URB2D(I,J)*TS_URB+(1-FRC_URB2D(I,J))*tsxy_buf          ![K]
            qsfc(i,j)  = FRC_URB2D(I,J)*QS_URB+(1-FRC_URB2D(I,J))*qsfc(i,j)         ![-]

! Renew Urban State Varialbes

            TR_URB2D(I,J) = TR_URB
            TB_URB2D(I,J) = TB_URB
            TG_URB2D(I,J) = TG_URB
            TC_URB2D(I,J) = TC_URB
            QC_URB2D(I,J) = QC_URB
            UC_URB2D(I,J) = UC_URB

            DO K = 1,num_roof_layers
              TRL_URB3D(I,K,J) = TRL_URB(K)
            END DO
            DO K = 1,num_wall_layers
              TBL_URB3D(I,K,J) = TBL_URB(K)
            END DO
            DO K = 1,num_road_layers
              TGL_URB3D(I,K,J) = TGL_URB(K)
            END DO
            XXXR_URB2D(I,J) = XXXR_URB
            XXXB_URB2D(I,J) = XXXB_URB
            XXXG_URB2D(I,J) = XXXG_URB
            XXXC_URB2D(I,J) = XXXC_URB

            SH_URB2D(I,J)    = SH_URB
            LH_URB2D(I,J)    = LH_URB
            G_URB2D(I,J)     = G_URB
            RN_URB2D(I,J)    = RN_URB
            PSIM_URB2D(I,J)  = PSIM_URB
            PSIH_URB2D(I,J)  = PSIH_URB
            GZ1OZ0_URB2D(I,J)= GZ1OZ0_URB
            U10_URB2D(I,J)   = U10_URB
            V10_URB2D(I,J)   = V10_URB
            TH2_URB2D(I,J)   = TH2_URB
            Q2_URB2D(I,J)    = Q2_URB
            UST_URB2D(I,J)   = UST_URB
            AKMS_URB2D(I,J)  = vkc * UST_URB2D(I,J)/(GZ1OZ0_URB2D(I,J)-PSIM_URB2D(I,J))
            IF (PRESENT(CMR_SFCDIF)) THEN
               CMR_SFCDIF(I,J) = CMR_URB
               CHR_SFCDIF(I,J) = CHR_URB
               CMC_SFCDIF(I,J) = CMC_URB
               CHC_SFCDIF(I,J) = CHC_URB
            ENDIF
          END IF

         ENDIF                                   ! end of urban CALL if block


    do m=1,maxpatch
      if(snl(i,m,j)<-10 .or. snl(i,m,j) >10) found=.true. 
         
    end do
  
   if(found) then

        write(6,*) 'in module_sf_clm, right after clm(), found snl ERROR! at i=',i,'j=',j
        found=.false.
   end if



  end do ! of i loop
  end do ! of j loop

   do i=its,ite
    do j=jts,jte
     do m=1,maxpatch
      if(snl(i,m,j)<-10 .or. snl(i,m,j) >10) found=.true.
     end do 
      if(found) then
        write(6,*) 'in module_sf_clm, finish all clm loop, found snl ERROR! at i=',i,'j=',j
        write(6,*) 'snl(',i,':',j,')=',snl(i,:,j)
        found=.false.
       end if

    end do
  end do


        call CLMDebug('clmdrv() success finished')   !DEL 
end subroutine clmdrv
!------------------------------------------------------------------------
  subroutine clminit(VEGFRA,SNOW,SNOWC,SNOWH,CANWAT,SMSTAV,        &
                     SMSTOT, SFCRUNOFF,UDRUNOFF,ACSNOW,            &
                     ACSNOM,IVGTYP,ISLTYP,TSLB,SMOIS,SH2O,ZS,DZS,  &
                     FNDSOILW, FNDSNOWH,                           &
                     num_soil_layers, restart,                     &
                     allowed_to_read ,                             &
                     ids,ide, jds,jde, kds,kde,                    &
                     ims,ime, jms,jme, kms,kme,                    &
                     its,ite, jts,jte, kts,kte,                    &
                     maxpatch                                      &
                    ,numc,nump,snl,                                      &
                     snowdp,wtc,wtp,h2osno,t_grnd,t_veg,         &
                     h2ocan,h2ocan_col,t2m_max,t2m_min,     &
                     t_ref2m,h2osoi_liq_s1,              &
                     h2osoi_liq_s2,h2osoi_liq_s3,h2osoi_liq_s4,          &
                     h2osoi_liq_s5,h2osoi_liq1,h2osoi_liq2,              &
                     h2osoi_liq3,h2osoi_liq4,h2osoi_liq5,h2osoi_liq6,    &
                     h2osoi_liq7,h2osoi_liq8,h2osoi_liq9,h2osoi_liq10,   &
                     h2osoi_ice_s1,h2osoi_ice_s2,                        &
                     h2osoi_ice_s3,h2osoi_ice_s4,h2osoi_ice_s5,          &
                     h2osoi_ice1,h2osoi_ice2,h2osoi_ice3,h2osoi_ice4,    &
                     h2osoi_ice5,h2osoi_ice6,h2osoi_ice7,                &
                     h2osoi_ice8,h2osoi_ice9,h2osoi_ice10,               &
                     t_soisno_s1,t_soisno_s2,t_soisno_s3,t_soisno_s4,    &
                     t_soisno_s5,t_soisno1,t_soisno2,t_soisno3,          &
                     t_soisno4,t_soisno5,t_soisno6,t_soisno7,            &
                     t_soisno8,t_soisno9,t_soisno10,                     &
                     dzsnow1,dzsnow2,dzsnow3,dzsnow4,dzsnow5,            &
                     snowrds1,snowrds2,snowrds3,snowrds4,snowrds5,       &
                     t_lake1,t_lake2,t_lake3,t_lake4,t_lake5,            &
                     t_lake6,t_lake7,t_lake8,t_lake9,t_lake10,           &
                     h2osoi_vol1,h2osoi_vol2,h2osoi_vol3,                &
                     h2osoi_vol4,h2osoi_vol5,h2osoi_vol6,                &
                     h2osoi_vol7,h2osoi_vol8,                            &
                     h2osoi_vol9,h2osoi_vol10,                           &
                     ht,xland,xice                                        &
                    ,albedosubgrid,lhsubgrid,hfxsubgrid,lwupsubgrid,q2subgrid &
                    ,sabvsubgrid,sabgsubgrid,nrasubgrid,swupsubgrid,      &
                    lhsoi,lhveg,lhtran &
!#ifdef CN 
!                    ,dyntlai,dyntsai,dyntop,dynbot &  !ADD_NEW_VAR
!#endif
                                                                         )

  USE module_wrf_error
   use clm_varcon, only :snowage_tau,snowage_kappa,snowage_drdt0            &
                        ,ss_alb_snw_drc,asm_prm_snw_drc                     &
                        ,ext_cff_mss_snw_drc,ss_alb_snw_dfs,asm_prm_snw_dfs &
                        ,ext_cff_mss_snw_dfs      &
                        ,xx_ss_alb_snw_drc        &
                        ,xx_asm_prm_snw_drc       &
                        ,xx_ext_cff_mss_snw_drc   &
                        ,xx_ss_alb_snw_dfs        &
                        ,xx_asm_prm_snw_dfs       &
                        ,xx_ext_cff_mss_snw_dfs   &
                        ,xx_snowage_tau           &
                        ,xx_snowage_kappa         &
                        ,xx_snowage_drdt0         &
                        ,idx_Mie_snw_mx           &
                        ,idx_T_max                &
                        ,idx_Tgrd_max             &
                        ,idx_rhos_max             &
                        ,numrad_snw 

!New in CLM4_crop
!#if (defined CROP)
!    USE  CropIniMod      , only : initialcrop
!#endif

  implicit none

  INTEGER,  INTENT(IN   )   ::     ids,ide, jds,jde, kds,kde,  &
                                   ims,ime, jms,jme, kms,kme,  &
                                   its,ite, jts,jte, kts,kte

  logical, external :: wrf_dm_on_monitor
  integer :: ix

  INTEGER, INTENT(IN)       ::     num_soil_layers,maxpatch
    
   LOGICAL , INTENT(IN) :: restart , allowed_to_read

   REAL,    DIMENSION( num_soil_layers), INTENT(INOUT) :: zs, dzs

   REAL,    DIMENSION( ims:ime, num_soil_layers, jms:jme )    , &
            INTENT(INOUT)    ::                          SMOIS, &  !Total soil moisture
                                                         SH2O,  &  !liquid soil moisture
                                                         TSLB      !STEMP
     
   REAL,    DIMENSION( ims:ime, jms:jme )                     , &
            INTENT(INOUT)    ::                           SNOW, & 
                                                         SNOWH, &
                                                         SNOWC, &
                                                        CANWAT, &
                                                        SMSTAV, &
                                                        SMSTOT, &
                                                     SFCRUNOFF, &
                                                      UDRUNOFF, &
                                                        ACSNOW, &
                                                        VEGFRA, &
                                                        ACSNOM

   REAL,  DIMENSION( ims:ime, jms:jme ), INTENT(IN) :: HT
   INTEGER, DIMENSION( ims:ime, jms:jme )                     , &
            INTENT(IN)       ::                         IVGTYP, &
                                                        ISLTYP
   REAL, DIMENSION( ims:ime, jms:jme )                     , &
            INTENT(IN)       ::                         XLAND,xice
   LOGICAL, DIMENSION( ims:ime, jms:jme ) :: lake

   LOGICAL, INTENT(IN)       ::                      FNDSOILW , &
                                                     FNDSNOWH

  integer,   dimension(ims:ime,jms:jme ),intent(inout) :: numc,nump
  integer,   dimension(ims:ime,1:maxpatch,jms:jme ),intent(inout) :: snl
  real,  dimension(ims:ime,jms:jme ),intent(inout) :: t2m_max,t2m_min
  real,  dimension(ims:ime,1:maxpatch,jms:jme ),intent(inout) ::  &
                snowdp,wtc,wtp,h2osno,t_grnd,t_veg,         &
                h2ocan,h2ocan_col,   &
                t_ref2m,h2osoi_liq_s1,              &
                h2osoi_liq_s2,h2osoi_liq_s3,h2osoi_liq_s4,          &
                h2osoi_liq_s5,h2osoi_liq1,h2osoi_liq2,              &
                h2osoi_liq3,h2osoi_liq4,h2osoi_liq5,h2osoi_liq6,    &
                h2osoi_liq7,h2osoi_liq8,h2osoi_liq9,h2osoi_liq10,   &
                h2osoi_ice_s1,h2osoi_ice_s2,                        &
                h2osoi_ice_s3,h2osoi_ice_s4,h2osoi_ice_s5,          &
                h2osoi_ice1,h2osoi_ice2,h2osoi_ice3,h2osoi_ice4,    &
                h2osoi_ice5,h2osoi_ice6,h2osoi_ice7,                &
                h2osoi_ice8,h2osoi_ice9,h2osoi_ice10,               &
                t_soisno_s1,t_soisno_s2,t_soisno_s3,t_soisno_s4,    &
                t_soisno_s5,t_soisno1,t_soisno2,t_soisno3,          &
                t_soisno4,t_soisno5,t_soisno6,t_soisno7,            &
                t_soisno8,t_soisno9,t_soisno10,                     &
                dzsnow1,dzsnow2,dzsnow3,dzsnow4,dzsnow5,            &
                snowrds1,snowrds2,snowrds3,snowrds4,snowrds5,       &
                t_lake1,t_lake2,t_lake3,t_lake4,t_lake5,            &
                t_lake6,t_lake7,t_lake8,t_lake9,t_lake10,           &
                h2osoi_vol1,h2osoi_vol2,h2osoi_vol3,                &
                h2osoi_vol4,h2osoi_vol5,h2osoi_vol6,                &
                h2osoi_vol7,h2osoi_vol8,                            &
                h2osoi_vol9,h2osoi_vol10,                           &
                ALBEDOsubgrid,LHsubgrid,HFXsubgrid,LWUPsubgrid,     &
                Q2subgrid,SABVsubgrid,SABGsubgrid,NRAsubgrid,SWUPsubgrid,&
                LHsoi,LHveg,LHtran
#ifdef CN
     real,  dimension(ims:ime,1:maxpatch,jms:jme ),intent(inout) :: dyntlai,dyntsai,dyntop,dynbot    !ADD_NEW_VAR                                       
#endif


   INTEGER                   :: L
   REAL                      :: BX, SMCMAX, PSISAT, FREE
   INTEGER                   :: errflag
   INTEGER                   :: itf,jtf,j,i,k,m
   LOGICAL                   :: opened

   integer                   :: lu_unit

   call CLMDebug('Now in clminit.')

IF ( wrf_dm_on_monitor() ) THEN
     DO i=10,99
        INQUIRE ( i , OPENED = opened )
          IF ( .NOT. opened ) THEN
                lu_unit=i
                GOTO 2011
            ENDIF
         ENDDO
         lu_unit = -1
 2011   CONTINUE

        if(lu_unit<0) then
          write(6,*) 'Can not assign unit to read CLM input data in clminit'
          call endrun()
        end if

  open(lu_unit,file='CLM_ALB_ICE_DRC_DATA')
  read(lu_unit,*) ((ss_alb_snw_drc(i,j),j=1,numrad_snw),i=1,idx_Mie_snw_mx)
  close(lu_unit)

  open(lu_unit,file='CLM_ASM_ICE_DRC_DATA')
  read(lu_unit,*) ((asm_prm_snw_drc(i,j),j=1,numrad_snw),i=1,idx_Mie_snw_mx)
  close(lu_unit)

  open(lu_unit,file='CLM_EXT_ICE_DRC_DATA')
  read(lu_unit,*) ((ext_cff_mss_snw_drc(i,j),j=1,numrad_snw),i=1,idx_Mie_snw_mx)
  close(lu_unit)

  open(lu_unit,file='CLM_ALB_ICE_DFS_DATA')
  read(lu_unit,*) ((ss_alb_snw_dfs(i,j),j=1,numrad_snw),i=1,idx_Mie_snw_mx)
  close(lu_unit)

  open(lu_unit,file='CLM_ASM_ICE_DFS_DATA')
  read(lu_unit,*) ((asm_prm_snw_dfs(i,j),j=1,numrad_snw),i=1,idx_Mie_snw_mx)
  close(lu_unit)

  open(lu_unit,file='CLM_EXT_ICE_DFS_DATA')
  read(lu_unit,*) ((ext_cff_mss_snw_dfs(i,j),j=1,numrad_snw),i=1,idx_Mie_snw_mx)
  close(lu_unit)

  open(lu_unit,file='CLM_TAU_DATA')
  read(lu_unit,*) &
  (((snowage_tau(i,j,k),i=1,idx_T_max),j=1,idx_Tgrd_max),k=1,idx_rhos_max)
  close(lu_unit)

  open(lu_unit,file='CLM_KAPPA_DATA')
  read(lu_unit,*) &
  (((snowage_kappa(i,j,k),i=1,idx_T_max),j=1,idx_Tgrd_max),k=1,idx_rhos_max)
  close(lu_unit)

  open(lu_unit,file='CLM_DRDSDT0_DATA')
  read(lu_unit,*)&
  (((snowage_drdt0(i,j,k),i=1,idx_T_max),j=1,idx_Tgrd_max),k=1,idx_rhos_max)
  close(lu_unit)
END IF

  ix = 0
  do i=1, idx_Mie_snw_mx
  do j=1, numrad_snw
    ix = ix + 1
    xx_ss_alb_snw_drc(ix)      = ss_alb_snw_drc(i,j)
    xx_asm_prm_snw_drc(ix)     = asm_prm_snw_drc(i,j)
    xx_ext_cff_mss_snw_drc(ix) = ext_cff_mss_snw_drc(i,j)
    xx_ss_alb_snw_dfs(ix)      = ss_alb_snw_dfs(i,j)
    xx_asm_prm_snw_dfs(ix)     = asm_prm_snw_dfs(i,j)
    xx_ext_cff_mss_snw_dfs(ix) = ext_cff_mss_snw_dfs(i,j)
  end do
  end do


 ix = 0
 do i=1,idx_T_max
 do j=1,idx_Tgrd_max
 do k=1,idx_rhos_max
    ix = ix + 1
    xx_snowage_tau(ix)   = snowage_tau(i,j,k)
    xx_snowage_kappa(ix) = snowage_kappa(i,j,k)
    xx_snowage_drdt0(ix) = snowage_drdt0(i,j,k)
 end do
 end do
 end do

  CALL wrf_dm_bcast_real(xx_ss_alb_snw_drc,      numrad_snw*idx_Mie_snw_mx )
  CALL wrf_dm_bcast_real(xx_asm_prm_snw_drc,     numrad_snw*idx_Mie_snw_mx )
  CALL wrf_dm_bcast_real(xx_ext_cff_mss_snw_drc, numrad_snw*idx_Mie_snw_mx )

  CALL wrf_dm_bcast_real(xx_ss_alb_snw_dfs,      numrad_snw*idx_Mie_snw_mx )
  CALL wrf_dm_bcast_real(xx_asm_prm_snw_dfs,     numrad_snw*idx_Mie_snw_mx )
  CALL wrf_dm_bcast_real(xx_ext_cff_mss_snw_dfs, numrad_snw*idx_Mie_snw_mx )

  CALL wrf_dm_bcast_real(xx_snowage_tau,  idx_T_max*idx_Tgrd_max*idx_rhos_max)
  CALL wrf_dm_bcast_real(xx_snowage_kappa,idx_T_max*idx_Tgrd_max*idx_rhos_max)
  CALL wrf_dm_bcast_real(xx_snowage_drdt0,idx_T_max*idx_Tgrd_max*idx_rhos_max)

 IF(restart) return

   itf=min0(ite,ide-1)
   jtf=min0(jte,jde-1)


   errflag = 0
   DO j = jts,jtf
     DO i = its,itf
       IF ( ISLTYP( i,j ) .LT. 1 ) THEN
         errflag = 1
         WRITE(wrf_err_message,*)"CLM: clminit: out of range ISLTYP ",i,j,ISLTYP( i,j )
         CALL wrf_message(wrf_err_message)
       ENDIF
     ENDDO
   ENDDO
   IF ( errflag .EQ. 1 ) THEN
      CALL wrf_error_fatal( "CLM: clminit: out of range value "// &
                            "of ISLTYP. Is this field in the input?" )
   ENDIF
!------------------------------------------------------------------------------
          DO j = jts,jtf
          DO i = its,itf


          if((xland(i,j)-1.5).ge.0.)then

             If(xice(i,j).eq.1)print*,' SEA-ICE AT WATER POINT, i=',i,'j=',j
  
              smstav(i,j)=1.0
              smstot(i,j)=1.0
              smois(i,:,j)=1.0
              tslb(i,:,j)=273.16
            else if(xice(i,j).eq.1.)then

              smstav(i,j)=1.0
              smstot(i,j)=1.0
              smois(i,:,j)=1.0
            endif

            snowh(i,j)=snow(i,j)*0.005               ! SNOW in kg/m^2 and SNOWH in m

            snowdp(i,:,j) = snowh(i,j)

          ENDDO
          ENDDO

       do i=its,itf
       do j=jts,jtf
           snl(i,:,j) = 0                  !-999.0 
           h2osoi_liq_s5(i,:,j) = -999.0 
           h2osoi_liq_s4(i,:,j) = -999.0 
           h2osoi_liq_s3(i,:,j) = -999.0 
           h2osoi_liq_s2(i,:,j) = -999.0 
           h2osoi_liq_s1(i,:,j) = -999.0 
           h2osoi_liq1(i,:,j)   = -999.0 
           h2osoi_liq2(i,:,j)   = -999.0 
           h2osoi_liq3(i,:,j)   = -999.0 
           h2osoi_liq4(i,:,j)   = -999.0 
           h2osoi_liq5(i,:,j)   = -999.0 
           h2osoi_liq6(i,:,j)   = -999.0 
           h2osoi_liq7(i,:,j)   = -999.0 
           h2osoi_liq8(i,:,j)   = -999.0 
           h2osoi_liq9(i,:,j)   = -999.0 
           h2osoi_liq10(i,:,j)  = -999.0 

           h2osoi_ice_s5(i,:,j) = -999.0 
           h2osoi_ice_s4(i,:,j) = -999.0 
           h2osoi_ice_s3(i,:,j) = -999.0 
           h2osoi_ice_s2(i,:,j) = -999.0 
           h2osoi_ice_s1(i,:,j) = -999.0 
           h2osoi_ice1(i,:,j)   = -999.0 
           h2osoi_ice2(i,:,j)   = -999.0 
           h2osoi_ice3(i,:,j)   = -999.0 
           h2osoi_ice4(i,:,j)   = -999.0 
           h2osoi_ice5(i,:,j)   = -999.0 
           h2osoi_ice6(i,:,j)   = -999.0 
           h2osoi_ice7(i,:,j)   = -999.0 
           h2osoi_ice8(i,:,j)   = -999.0 
           h2osoi_ice9(i,:,j)   = -999.0 
           h2osoi_ice10(i,:,j)  = -999.0 

        !   snowage(i,:,j)= 0.0
!Could this be the snow bug?
!           if(ivgtyp(i,j).eq.24) then
!             h2osno(i,:,j) = 1000.0 ! mm
!           else
             h2osno(i,:,j) = snow(i,j) ! mm
!           end if
       end do
       end do

     !  write(6,*) '-------in clminit--------'
     !  write(6,*) 'snl=',snl
     !  call CLMDebug('clminit mark1')
!------------------------------------------------------------------------------
       do i=its,itf
       do j=jts,jtf
         numc(i,j) = 0
         nump(i,j) = 0
         wtc(i,:,j) = 0.0
         wtp(i,:,j) = 0.0
#ifdef CN
         dyntlai(i,:,j) = 0.0
         dyntsai(i,:,j) = 0.0
         dyntop(i,:,j) = 0.0
         dynbot(i,:,j) = 0.0
#endif
       end do
       end do
!------------------------------------------------------------------------------
       do i=its,itf
         do j=jts,jtf
!            if(ivgtyp(i,j)==16.and.ht(i,j)>=1.e-5) then
!!!!!!Lakes Disabled.  See comments above.
            if(0 == 1) then
!!!!!!
                lake(i,j)  = .true.
            else
                lake(i,j)  = .false.
            end if
        end do
       end do
!------------------------------------------------------------------------------
! for snow
      do m=1,maxpatch
      do i=its,itf
      do j=jts,jtf
               dzsnow1(i,m,j) = 0.0
               dzsnow2(i,m,j) = 0.0
               dzsnow3(i,m,j) = 0.0
               dzsnow4(i,m,j) = 0.0
               dzsnow5(i,m,j) = 0.0
               if(snowdp(i,m,j).lt.0.01) then
                     snl(i,m,j) = 0
                     dzsnow1(i,m,j) = 0.0
                     dzsnow2(i,m,j) = 0.0
                     dzsnow3(i,m,j) = 0.0
                     dzsnow4(i,m,j) = 0.0
                     dzsnow5(i,m,j) = 0.0
               else
                if(snowdp(i,m,j).ge.0.01.and.snowdp(i,m,j).le.0.03) then
                   snl(i,m,j) = -1
                   dzsnow1(i,m,j) = snowdp(i,m,j)
                else if(snowdp(i,m,j).gt.0.03.and.snowdp(i,m,j).le.0.04) then
                   snl(i,m,j) = -2
                   dzsnow2(i,m,j) = snowdp(i,m,j)/2.
                   dzsnow1(i,m,j) = snowdp(i,m,j)/2.
                else if(snowdp(i,m,j).gt.0.04.and.snowdp(i,m,j).le.0.07) then
                   snl(i,m,j) = -2
                   dzsnow2(i,m,j) = 0.02
                   dzsnow1(i,m,j) = snowdp(i,m,j)- dzsnow2(i,m,j)
                else if(snowdp(i,m,j).gt.0.07.and.snowdp(i,m,j).le.0.12) then
                   snl(i,m,j) = -3
                   dzsnow3(i,m,j) = 0.02
                   dzsnow2(i,m,j) = (snowdp(i,m,j) - 0.02)/2.0
                   dzsnow1(i,m,j) = (snowdp(i,m,j) - 0.02)/2.0
                else if(snowdp(i,m,j).gt.0.12.and.snowdp(i,m,j).le.0.18) then
                   snl(i,m,j) = -3
                   dzsnow3(i,m,j) = 0.02
                   dzsnow2(i,m,j) = 0.05
                   dzsnow1(i,m,j)= snowdp(i,m,j)-dzsnow3(i,m,j)-dzsnow2(i,m,j)
                else if(snowdp(i,m,j).gt.0.18.and.snowdp(i,m,j).le.0.29) then
                   snl(i,m,j) = -4
                   dzsnow4(i,m,j) = 0.02
                   dzsnow3(i,m,j) = 0.05
                   dzsnow2(i,m,j) = (snowdp(i,m,j)-dzsnow4(i,m,j)-dzsnow3(i,m,j))/2.0
                   dzsnow1(i,m,j) = dzsnow2(i,m,j)
                else if(snowdp(i,m,j).gt.0.29.and.snowdp(i,m,j).le.0.41) then
                   snl(i,m,j) = -4
                   dzsnow4(i,m,j) = 0.02
                   dzsnow3(i,m,j) = 0.05
                   dzsnow2(i,m,j) = 0.11
                   dzsnow1(i,m,j) = snowdp(i,m,j)-dzsnow4(i,m,j)-dzsnow3(i,m,j)-dzsnow2(i,m,j)
                else if(snowdp(i,m,j).gt.0.41.and.snowdp(i,m,j).le.0.64) then
                   snl(i,m,j) = -5
                   dzsnow5(i,m,j) = 0.02
                   dzsnow4(i,m,j) = 0.05
                   dzsnow3(i,m,j) = 0.11
                   dzsnow2(i,m,j) = (snowdp(i,m,j)-dzsnow5(i,m,j)-dzsnow4(i,m,j)-dzsnow3(i,m,j))/2.0
                   dzsnow1(i,m,j) = (snowdp(i,m,j)-dzsnow5(i,m,j)-dzsnow4(i,m,j)-dzsnow3(i,m,j))/2.0
                else if(snowdp(i,m,j).gt.0.64) then
                   snl(i,m,j) = -5
                   dzsnow5(i,m,j) = 0.02
                   dzsnow4(i,m,j)= 0.05
                   dzsnow3(i,m,j) = 0.11
                   dzsnow2(i,m,j) = 0.23
                   dzsnow1(i,m,j) = snowdp(i,m,j)-dzsnow5(i,m,j)-dzsnow4(i,m,j)-dzsnow3(i,m,j)-dzsnow2(i,m,j)
                end if
            end if ! start from snowdp(i,m,j).lt.0.01
       end do
       end do
       end do
 
      !write(6,*) 'after assign snl=',snl

!------------------------------------------------------------------------------
!snow radius
    do m=1,maxpatch
      do i=its,itf
        do j=jts,jtf
          if(snl(i,m,j) == -5) then
          snowrds1(i,m,j) = 54.526    !snw_rds_min = 54.526
          snowrds2(i,m,j) = 54.526
          snowrds3(i,m,j) = 54.526
          snowrds4(i,m,j) = 54.526
          snowrds5(i,m,j) = 54.526
          else if(snl(i,m,j) == -4) then
          snowrds1(i,m,j) = 54.526
          snowrds2(i,m,j) = 54.526
          snowrds3(i,m,j) = 54.526
          snowrds4(i,m,j) = 54.526
          snowrds5(i,m,j) = 0.0
           else if(snl(i,m,j) == -3) then
          snowrds1(i,m,j) = 54.526    
          snowrds2(i,m,j) = 54.526
          snowrds3(i,m,j) = 54.526
          snowrds4(i,m,j) = 0.0
          snowrds5(i,m,j) = 0.0
          else if(snl(i,m,j) == -2) then
          snowrds1(i,m,j) = 54.526
          snowrds2(i,m,j) = 54.526
          snowrds3(i,m,j) = 0.0
          snowrds4(i,m,j) = 0.0
          snowrds5(i,m,j) = 0.0
          else if(snl(i,m,j) == -1) then
          snowrds1(i,m,j) = 54.526
          snowrds2(i,m,j) = 0.0
          snowrds3(i,m,j) = 0.0
          snowrds4(i,m,j) = 0.0
          snowrds5(i,m,j) = 0.0
          else if(snl(i,m,j) == 0) then
          snowrds1(i,m,j) = 0.0
          snowrds2(i,m,j) = 0.0
          snowrds3(i,m,j) = 0.0
          snowrds4(i,m,j) = 0.0
          snowrds5(i,m,j) = 0.0
         end if
        end do
       end do
    end do


!------------------------------------------------------------------------------
       do i=its,itf
          do j=jts,jtf
              h2ocan(i,:,j) = 0.0
              h2ocan_col(i,:,j) = 0.0
              sfcrunoff(i,j) = 0.0 
              udrunoff(i,j) = 0.0
          end do
       end do
!------------------------------------------------------------------------------
! initialize temperature and moisture
      do i=its,itf
      do j=jts,jtf
           t_soisno_s5(i,:,j) = -999.0 
           t_soisno_s4(i,:,j) = -999.0 
           t_soisno_s3(i,:,j) = -999.0 
           t_soisno_s2(i,:,j) = -999.0 
           t_soisno_s1(i,:,j) = -999.0 
           t_soisno1(i,:,j)   = -999.0 
           t_soisno2(i,:,j)   = -999.0 
           t_soisno3(i,:,j)   = -999.0 
           t_soisno4(i,:,j)   = -999.0 
           t_soisno5(i,:,j)   = -999.0 
           t_soisno6(i,:,j)   = -999.0 
           t_soisno7(i,:,j)   = -999.0 
           t_soisno8(i,:,j)   = -999.0 
           t_soisno9(i,:,j)   = -999.0 
           t_soisno10(i,:,j)  = -999.0 

           t_lake1(i,:,j)     = -999.0 
           t_lake2(i,:,j)     = -999.0 
           t_lake3(i,:,j)     = -999.0 
           t_lake4(i,:,j)     = -999.0 
           t_lake5(i,:,j)     = -999.0 
           t_lake6(i,:,j)     = -999.0 
           t_lake7(i,:,j)     = -999.0 
           t_lake8(i,:,j)     = -999.0 
           t_lake9(i,:,j)     = -999.0 
           t_lake10(i,:,j)    = -999.0 
      end do
      end do

      do i=its,itf
      do j=jts,jtf
           do k=1,num_soil_layers
             if(ivgtyp(i,j).eq.24.and.tslb(i,k,j) .gt.tfrz) tslb(i,k,j)=tfrz
           end do
           t_soisno_s5(i,:,j) = tslb(i,1,j)
           t_soisno_s4(i,:,j) = tslb(i,1,j)
           t_soisno_s3(i,:,j) = tslb(i,1,j)
           t_soisno_s2(i,:,j) = tslb(i,1,j)
           t_soisno_s1(i,:,j) = tslb(i,1,j)
           t_soisno1(i,:,j) = tslb(i,1,j)
           t_soisno2(i,:,j) = tslb(i,2,j)
           t_soisno3(i,:,j) = tslb(i,3,j)
           t_soisno4(i,:,j) = tslb(i,4,j)
           t_soisno5(i,:,j) = tslb(i,5,j)
           t_soisno6(i,:,j) = tslb(i,6,j)
           t_soisno7(i,:,j) = tslb(i,7,j)
           t_soisno8(i,:,j) = tslb(i,8,j)
           t_soisno9(i,:,j) = tslb(i,9,j)
           t_soisno10(i,:,j)= tslb(i,10,j)

           t_grnd(i,:,j) = tslb(i,1,j)
           t_veg(i,:,j)  = tslb(i,1,j)
      end do 
      end do 

      do i=its,itf
      do j=jts,jtf
        if(lake(i,j)) then
              t_lake1(i,:,j)     = 277.0 
              t_lake2(i,:,j)     = 277.0 
              t_lake3(i,:,j)     = 277.0 
              t_lake4(i,:,j)     = 277.0 
              t_lake5(i,:,j)     = 277.0 
              t_lake6(i,:,j)     = 277.0 
              t_lake7(i,:,j)     = 277.0 
              t_lake8(i,:,j)     = 277.0 
              t_lake9(i,:,j)     = 277.0 
              t_lake10(i,:,j)    = 277.0 
              t_grnd(i,:,j)      = 277.0
        end if
      end do
      end do

        call CLMDebug('clminit mark2')
! for moisture

      do i=its,itf
      do j=jts,jtf
         h2osoi_vol1(i,:,j)   = smois(i,1,j)
         h2osoi_vol2(i,:,j)   = smois(i,2,j)
         h2osoi_vol3(i,:,j)   = smois(i,3,j)
         h2osoi_vol4(i,:,j)   = smois(i,4,j)
         h2osoi_vol5(i,:,j)   = smois(i,5,j)
         h2osoi_vol6(i,:,j)   = smois(i,6,j)
         h2osoi_vol7(i,:,j)   = smois(i,7,j)
         h2osoi_vol8(i,:,j)   = smois(i,8,j)
         h2osoi_vol9(i,:,j)   = smois(i,9,j)
         h2osoi_vol10(i,:,j)  = smois(i,10,j)

         h2osoi_liq_s5(i,:,j) = 0.0
         h2osoi_liq_s4(i,:,j) = 0.0
         h2osoi_liq_s3(i,:,j) = 0.0
         h2osoi_liq_s2(i,:,j) = 0.0
         h2osoi_liq_s1(i,:,j) = 0.0
         h2osoi_ice_s5(i,:,j) = 1.0 
         h2osoi_ice_s4(i,:,j) = 1.0 
         h2osoi_ice_s3(i,:,j) = 1.0 
         h2osoi_ice_s2(i,:,j) = 1.0 
         h2osoi_ice_s1(i,:,j) = 1.0 

         do m = 1, maxpatch
          if(t_soisno1(i,m,j) <tfrz.and.t_soisno1(i,m,j)/=-999.0) then
             h2osoi_ice1(i,m,j)   = dzs(1)*0.917e3*h2osoi_vol1(i,m,j)
             h2osoi_liq1(i,m,j)   = 0.0
          else if (t_soisno1(i,m,j) >= tfrz) then
             h2osoi_ice1(i,m,j)   = 0.0
             h2osoi_liq1(i,m,j)   = dzs(1)*1000.0*h2osoi_vol1(i,m,j)
          end if

          if(t_soisno2(i,m,j) <tfrz.and.t_soisno2(i,m,j)/=-999.0) then
             h2osoi_ice2(i,m,j)   = dzs(2)*0.917e3*h2osoi_vol2(i,m,j)
             h2osoi_liq2(i,m,j)   = 0.0
          else if (t_soisno2(i,m,j) >= tfrz) then
             h2osoi_ice2(i,m,j)   = 0.0
             h2osoi_liq2(i,m,j)   = dzs(2)*1000.0*h2osoi_vol2(i,m,j)
          end if

          if(t_soisno3(i,m,j) <tfrz.and.t_soisno3(i,m,j)/=-999.0) then
             h2osoi_ice3(i,m,j)   = dzs(3)*0.917e3*h2osoi_vol3(i,m,j)
             h2osoi_liq3(i,m,j)   = 0.0
          else if (t_soisno3(i,m,j) >= tfrz) then
             h2osoi_ice3(i,m,j)   = 0.0
             h2osoi_liq3(i,m,j)   = dzs(3)*1000.0*h2osoi_vol3(i,m,j)
          end if

          if(t_soisno4(i,m,j) <tfrz.and.t_soisno4(i,m,j)/=-999.0) then
             h2osoi_ice4(i,m,j)   = dzs(4)*0.917e4*h2osoi_vol4(i,m,j)
             h2osoi_liq4(i,m,j)   = 0.0
          else if (t_soisno4(i,m,j) >= tfrz) then
             h2osoi_ice4(i,m,j)   = 0.0
             h2osoi_liq4(i,m,j)   = dzs(4)*1000.0*h2osoi_vol4(i,m,j)
          end if

          if(t_soisno5(i,m,j) <tfrz.and.t_soisno5(i,m,j)/=-999.0) then
             h2osoi_ice5(i,m,j)   = dzs(5)*0.917e4*h2osoi_vol5(i,m,j)
             h2osoi_liq5(i,m,j)   = 0.0
          else if (t_soisno5(i,m,j) >= tfrz) then
             h2osoi_ice5(i,m,j)   = 0.0
             h2osoi_liq5(i,m,j)   = dzs(5)*1000.0*h2osoi_vol5(i,m,j)
          end if

          if(t_soisno6(i,m,j) <tfrz.and.t_soisno6(i,m,j)/=-999.0) then
             h2osoi_ice6(i,m,j)   = dzs(6)*0.917e4*h2osoi_vol6(i,m,j)
             h2osoi_liq6(i,m,j)   = 0.0
          else if (t_soisno6(i,m,j) >= tfrz) then
             h2osoi_ice6(i,m,j)   = 0.0
             h2osoi_liq6(i,m,j)   = dzs(6)*1000.0*h2osoi_vol6(i,m,j)
          end if

          if(t_soisno7(i,m,j) <tfrz.and.t_soisno7(i,m,j)/=-999.0) then
             h2osoi_ice7(i,m,j)   = dzs(7)*0.917e4*h2osoi_vol7(i,m,j)
             h2osoi_liq7(i,m,j)   = 0.0
          else if (t_soisno7(i,m,j) >= tfrz) then
             h2osoi_ice7(i,m,j)   = 0.0
             h2osoi_liq7(i,m,j)   = dzs(7)*1000.0*h2osoi_vol7(i,m,j)
          end if

          if(t_soisno8(i,m,j) <tfrz.and.t_soisno8(i,m,j)/=-999.0) then
             h2osoi_ice8(i,m,j)   = dzs(8)*0.917e4*h2osoi_vol8(i,m,j)
             h2osoi_liq8(i,m,j)   = 0.0
          else if (t_soisno8(i,m,j) >= tfrz) then
             h2osoi_ice8(i,m,j)   = 0.0
             h2osoi_liq8(i,m,j)   = dzs(8)*1000.0*h2osoi_vol8(i,m,j)
          end if

          if(t_soisno9(i,m,j) <tfrz.and.t_soisno9(i,m,j)/=-999.0) then
             h2osoi_ice9(i,m,j)   = dzs(9)*0.917e4*h2osoi_vol9(i,m,j)
             h2osoi_liq9(i,m,j)   = 0.0
          else if (t_soisno9(i,m,j) >= tfrz) then
             h2osoi_ice9(i,m,j)   = 0.0
             h2osoi_liq9(i,m,j)   = dzs(9)*1000.0*h2osoi_vol9(i,m,j)
          end if

          if(t_soisno10(i,m,j) <tfrz.and.t_soisno10(i,m,j)/=-999.0) then
             h2osoi_ice10(i,m,j)   = dzs(10)*0.917e4*h2osoi_vol10(i,m,j)
             h2osoi_liq10(i,m,j)   = 0.0
          else if (t_soisno10(i,m,j) >= tfrz) then
             h2osoi_ice10(i,m,j)   = 0.0
             h2osoi_liq10(i,m,j)   = dzs(10)*1000.0*h2osoi_vol10(i,m,j)
          end if
        end do

      end do
      end do
!------------------------------------------------------------------------------
 
      call CLMDebug('clminit mark 4') 
      do i=its,itf
      do j=jts,jtf
        t2m_max(i,j)             = tslb(i,1,j)
        t2m_min(i,j)             = tslb(i,1,j)
        t_ref2m(i,:,j)           = tslb(i,1,j)
        albedosubgrid(i,:,j) = 0.0
        lhsubgrid(i,:,j)     = 0.0
        hfxsubgrid(i,:,j)    = 0.0
        lwupsubgrid(i,:,j)   = 0.0
        q2subgrid(i,:,j)     = 0.0
        sabvsubgrid(i,:,j)   = 0.0
        sabgsubgrid(i,:,j)   = 0.0
        nrasubgrid(i,:,j)    = 0.0
        swupsubgrid(i,:,j)   = 0.0
!!
        lhsoi(i,:,j)  = 0.0
        lhveg(i,:,j)  = 0.0
        lhtran(i,:,j) = 0.0 
      end do
      end do

      do i=its,itf
      do j=jts,jtf
       do k=1, num_soil_layers
          if(tslb(i,k,j) >= tfrz )  then
            sh2o(i,k,j) = smois(i,k,j)
          else
            sh2o(i,k,j) = 0.0
          end if
       end do
      end do
      end do

   call CLMDebug('clminit done')
!------------------------------------------------------------------------------
 END SUBROUTINE clminit
!------------------------------------------------------------------------------
END MODULE module_sf_clm
module decompMod
!------------------------------------------------------------------------------
!BOP
!
! !MODULE: decompMod
!
! !USES:
  use shr_kind_mod, only : r8 => shr_kind_r8
  use clm_varpar  , only : lsmlon, lsmlat, maxpatch, maxpatch_pft, &
                           npatch_crop, npatch_urban, npatch_glacier
  use clm_varsur  , only : numlon

  implicit none

  integer, public :: ncells
  integer, public :: nlunits
  integer, public :: ncols
  integer, public :: npfts

  public initDecomp              ! initializes land surface decomposition
                                 ! into clumps and processors
  public get_gcell_info          ! updates gridcell, landunits, columns and
                                 ! pfts counters
  public get_gcell_xyind         ! returns ixy and jxy for each grid cell

  public get_proc_bounds         ! beg and end gridcell, landunit, column,
                                 ! pft indices for this processor

  save 

  private

  type gcell_decomp
     integer :: gsn     ! corresponding cell index in south->north gridcell array
     integer :: li      ! beginning landunit index
     integer :: lf      ! ending landunit index
     integer :: ci      ! beginning column index
     integer :: cf      ! ending column index
     integer :: pi      ! beginning pft index
     integer :: pf      ! ending pft index
  end type gcell_decomp
  type(gcell_decomp), allocatable :: gcelldc(:)

contains
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: initDecomp
!
! !INTERFACE:
  subroutine initDecomp
!
! !DESCRIPTION:
! This subroutine initializes the land surface decomposition into a clump
! data structure.
!
! !USES:
    use clmtype
!
! !ARGUMENTS:
    implicit none
                                                           ! weights
!
! !LOCAL VARIABLES:
    integer :: ppc                    ! min number of pfts per clump
    integer :: lpc                    ! min number of landunits per clump
    integer :: ppclump                ! min pfts per clump
    integer :: i,j,cid,pid            ! indices
    integer :: gi,li,ci,pi            ! indices
    integer :: gf,lf,cf,pf            ! indices
    integer :: g,l,c,p,n,m            ! indices
    integer :: gdc,gsn                ! indices
    integer :: nzero                  ! first clump with zero gridcells
!    integer :: ncells                 ! total gridcells
!    integer :: nlunits                ! total landunits
!    integer :: ncols                  ! total columns
!    integer :: npfts                  ! total pfts
    integer :: nveg                   ! number of pfts in vegetated landunit
    integer :: numg                   ! total number of gridcells across all
                                      ! processors
    integer :: numl                   ! total number of landunits across all
                                      ! processors
    integer :: numc                   ! total number of columns across all
                                      ! processors
    integer :: nump                   ! total number of pfts across all
                                      ! processors
    logical :: error = .false.        ! temporary for finding full clump
    integer :: ilunits, icols, ipfts  ! temporaries
    integer :: ng                     ! temporaries
    integer :: nl                     ! temporaries
    integer :: nc                     ! temporaries
    integer :: np                     ! temporaries
    integer :: ier                    ! error code
    character*256 :: msg
    integer :: begg,endg

    begg=1
    endg=1
!

! !CALLED FROM:
! subroutine initialize
!
! !REVISION HISTORY:
! 2002.09.11  Forrest Hoffman  Creation.
!
!EOP
!------------------------------------------------------------------------------

    ! Find total global number of grid cells, landunits, columns and pfts

    ncells = 0
    nlunits = 0
    ncols = 0
    npfts = 0
    
    msg= ''
    write(msg,*) 'lsmlat=',lsmlat,'numlon=',numlon
    call CLMDebug(msg)
  
       do g = begg,endg
             call get_gcell_info (g,  nlunits=ilunits, ncols=icols, npfts=ipfts)
             ncells  = ncells  + 1
             nlunits = nlunits + ilunits
             ncols   = ncols   + icols
             npfts   = npfts   + ipfts
       end do

  end subroutine initDecomp
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: get_gcell_info
!
! !INTERFACE:
   subroutine get_gcell_info (g, nlunits, ncols, npfts, &
                              nveg, wtveg, ncrop, wtcrop)
!

   use clm_varsur , only: wtxy
! !DESCRIPTION:
! Obtain gridcell properties.
!
! !ARGUMENTS:
     implicit none
     integer ,intent(in) :: g                                                       ! weights
     integer , optional, intent(out) :: nlunits  ! number of landunits
     integer , optional, intent(out) :: ncols    ! number of columns
     integer , optional, intent(out) :: npfts    ! number of pfts
     integer , optional, intent(out) :: nveg     ! number of vegetated pfts
                                                 ! in naturally vegetated
                                                 ! landunit
     real(r8), optional, intent(out) :: wtveg    ! weight (relative to
                                                 ! gridcell) of naturally
                                                 ! vegetated landunit
     integer , optional, intent(out) :: ncrop    ! number of crop pfts in
                                                 ! crop landunit
     real(r8), optional, intent(out) :: wtcrop   ! weight (relative to
                                                 ! gridcell) of crop landunit
!
! !CALLED FROM:
! subroutines initDecomp
!
! !REVISION HISTORY:
! 2002.09.11  Mariana Vertenstein  Creation.
!
!EOP
!
! !LOCAL VARIABLES:
     integer  :: m       ! loop index
     integer  :: nvegl   ! number of vegetated pfts in naturally vegetated landunit
     real(r8) :: wtvegl  ! weight (relative to gridcell) of vegetated landunit
     integer  :: nvegc   ! number of crop pfts in crop landunit
     real(r8) :: wtvegc  ! weight (relative to gridcell) of crop landunit
     integer  :: ilunits ! number of landunits in gridcell
     integer  :: icols   ! number of columns in gridcell
     integer  :: ipfts   ! number of pfts in gridcell
!------------------------------------------------------------------------------

     ! Initialize pfts, columns and landunits counters for gridcell

     ipfts   = 0
     icols   = 0
     ilunits = 0

     ! Set total number of pfts in gridcell
     do m = 1,maxpatch
        if (wtxy(g,m) > 0.0) ipfts = ipfts + 1
     end do

     ! Set naturally vegetated landunit

     nvegl = 0
     wtvegl = 0.0
     do m = 1, maxpatch_pft
        if (wtxy(g,m) > 0.0) then
           nvegl = nvegl + 1
           wtvegl = wtvegl + wtxy(g,m)
        end if
     end do
     if (nvegl > 0) ilunits = ilunits + 1
#if (defined NOCOMPETE)
     if (nvegl > 0) icols = icols + nvegl ! each pft on vegetated landunit has its own column
#else
     if (nvegl > 0) icols = icols + 1     ! the vegetated landunit has one column
#endif

     ! Set special landunits

     do m = npatch_urban, npatch_glacier
        if (wtxy(g,m) > 0.0) ilunits = ilunits + 1
        if (wtxy(g,m) > 0.0) icols = icols + 1
     end do

     ! Set crop landunit if appropriate

     nvegc = 0
     wtvegc = 0.0

        do m = npatch_glacier+1, npatch_crop
           if (wtxy(g,m) > 0.0) then
              nvegc = nvegc + 1
              wtvegc = wtvegc + wtxy(g,m)
           end if
        end do
        if (nvegc > 0) ilunits = ilunits + 1
        if (nvegc > 0) icols = icols + nvegc

     if (present(nlunits)) nlunits = ilunits
     if (present(ncols))   ncols   = icols
     if (present(npfts))   npfts   = ipfts
     if (present(nveg))    nveg    = nvegl
     if (present(wtveg))   wtveg   = wtvegl
     if (present(ncrop))   ncrop   = nvegc
     if (present(wtcrop))  wtcrop  = wtvegc

 end subroutine get_gcell_info

!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: get_proc_bounds
!
! !INTERFACE:
   subroutine get_proc_bounds (begg, endg, begl, endl, begc, endc, &
                               begp, endp)
!
! !ARGUMENTS:
     implicit none
     integer,optional, intent(out) :: begp, endp  ! proc beginning and ending
                                         ! pft indices
     integer,optional, intent(out) :: begc, endc  ! proc beginning and ending
                                         ! column indices
     integer,optional, intent(out) :: begl, endl  ! proc beginning and ending
                                         ! landunit indices
     integer,optional, intent(out) :: begg, endg  ! proc beginning and ending
                                         ! gridcell indices
! !DESCRIPTION:
! Retrieve gridcell, landunit, column, and pft bounds for process.
!
! !REVISION HISTORY:
! 2003.09.12  Mariana Vertenstein  Creation.
!
!EOP
!------------------------------------------------------------------------------

     if(present(begp)) begp = 1
     if(present(endp)) endp = npfts
     if(present(begc)) begc = 1
     if(present(endc)) endc = ncols
     if(present(begl)) begl = 1
     if(present(endl)) endl = nlunits
     if(present(begg)) begg = 1
     if(present(endg)) endg = 1

   end subroutine get_proc_bounds

!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: get_gcell_xyind
!
! !INTERFACE:
   subroutine get_gcell_xyind(lbg, ubg)
!
! !DESCRIPTION:
! Retrieve x,y indices of a gridcell.
!
! !ARGUMENTS:
     implicit none
     integer, intent(in) :: lbg
     integer, intent(in) :: ubg
!
! !REVISION HISTORY:
! 2003.09.12  Mariana Vertenstein  Creation.
!
!EOP
!
! !LOCAL VARIABLES:
     integer :: g     ! indices
     integer :: i, j
     integer :: ier                    ! error code
!------------------------------------------------------------------------------

!dir$ concurrent
!cdir nodep
    allocate(gcelldc(ncells), stat=ier)
    g = 0

       do j=1,lsmlat
          numlon(j) = lsmlon
       end do


    do j = 1, lsmlat
       do i = 1, numlon(j)
          g = g + 1
       end do
    end do
    do g = lbg,ubg
    end do
    deallocate(gcelldc)

   end subroutine get_gcell_xyind

end module decompMod
!==============================================================================
subroutine CLMDebug( str )

  IMPLICIT NONE
  CHARACTER*(*), str

#if (defined DEBUGCLM)
  print*, TRIM(str)
  call flush(6)
#endif

end subroutine CLMDebug

module clmtypeInitMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: clmtypeInitMod
!
! !DESCRIPTION:
! Allocate clmtype components and initialize them to signaling NaN.
!
! !USES:
  use shr_kind_mod, only : r8 => shr_kind_r8
  use nanMod      , only : nan, bigint
  use clmtype
  use clm_varpar  , only : maxpatch_pft, nlevsno, nlevgrnd, numrad, nlevlak, &
                           numpft, ndst, nvoc, nlevurb, nlevsoi
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: initClmtype
!
! !REVISION HISTORY:
! Created by Peter Thornton and Mariana Vertenstein
! Modified by Colette L. Heald (05/06) for VOC emission factors
! 3/17/08 David Lawrence, changed nlevsoi to nlevgrnd where appropriate
!
! !PRIVATE MEMBER FUNCTIONS:
  private :: init_pft_type
  private :: init_column_type
  private :: init_landunit_type
  private :: init_gridcell_type
  private :: init_energy_balance_type
  private :: init_water_balance_type
  private :: init_pft_ecophys_constants
#if (defined CNDV) || (defined CROP)
  private :: init_pft_DGVMecophys_constants
#endif
  private :: init_pft_pstate_type
  private :: init_pft_epv_type
#if (defined CNDV) || (defined CROP)
  private :: init_pft_pdgvstate_type
#endif
  private :: init_pft_vstate_type
  private :: init_pft_estate_type
  private :: init_pft_wstate_type
  private :: init_pft_cstate_type
  private :: init_pft_nstate_type
  private :: init_pft_eflux_type
  private :: init_pft_mflux_type
  private :: init_pft_wflux_type
  private :: init_pft_cflux_type
  private :: init_pft_nflux_type
  private :: init_pft_vflux_type
  private :: init_pft_dflux_type
  private :: init_pft_depvd_type
  private :: init_column_pstate_type
  private :: init_column_estate_type
  private :: init_column_wstate_type
  private :: init_column_cstate_type
  private :: init_column_nstate_type
  private :: init_column_eflux_type
  private :: init_column_wflux_type
  private :: init_column_cflux_type
  private :: init_column_nflux_type
  private :: init_landunit_pstate_type
  private :: init_landunit_eflux_type
  private :: init_gridcell_pstate_type
  private :: init_gridcell_efstate_type
  private :: init_gridcell_wflux_type
!ylu add these two subroutine was called but not declared in clm4
  private :: init_gridcell_wstate_type
  private :: init_gridcell_estate_type
  private :: init_atm2lnd_type



!have to deallocate all the allocated vars at every time step
  private :: dealloc_pft_type
  private :: dealloc_column_type
  private :: dealloc_landunit_type
  private :: dealloc_gridcell_type
  private :: dealloc_energy_balance_type
  private :: dealloc_water_balance_type
  private :: dealloc_pft_ecophys_constants
#if (defined CNDV) || (defined CROP)
  private :: dealloc_pft_DGVMecophys_constants
#endif
  private :: dealloc_pft_pstate_type
  private :: dealloc_pft_epv_type
#if (defined CNDV) || (defined CROP)
  private :: dealloc_pft_pdgvstate_type
#endif
  private :: dealloc_pft_vstate_type
  private :: dealloc_pft_estate_type
  private :: dealloc_pft_wstate_type
  private :: dealloc_pft_cstate_type
  private :: dealloc_pft_nstate_type
  private :: dealloc_pft_eflux_type
  private :: dealloc_pft_mflux_type
  private :: dealloc_pft_wflux_type
  private :: dealloc_pft_cflux_type
  private :: dealloc_pft_nflux_type
  private :: dealloc_pft_vflux_type
  private :: dealloc_pft_dflux_type
  private :: dealloc_pft_depvd_type
  private :: dealloc_column_pstate_type
  private :: dealloc_column_estate_type
  private :: dealloc_column_wstate_type
  private :: dealloc_column_cstate_type
  private :: dealloc_column_nstate_type
  private :: dealloc_column_eflux_type
  private :: dealloc_column_wflux_type
  private :: dealloc_column_cflux_type
  private :: dealloc_column_nflux_type
  private :: dealloc_landunit_pstate_type
  private :: dealloc_landunit_eflux_type
  private :: dealloc_gridcell_pstate_type
  private :: dealloc_gridcell_efstate_type
  private :: dealloc_gridcell_wflux_type
!ylu add these two subroutine was called but not declared in clm4
  private :: dealloc_gridcell_wstate_type
  private :: dealloc_gridcell_estate_type
  private :: dealloc_atm2lnd_type


!EOP
!----------------------------------------------------

contains

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: initClmtype
!
! !INTERFACE:
  subroutine initClmtype()
!
! !DESCRIPTION:
! Initialize clmtype components to signaling nan
! The following clmtype components should NOT be initialized here
! since they are set in routine clm_map which is called before this
! routine is invoked
!    *%area, *%wt, *%wtlnd, *%wtxy, *%ixy, *%jxy, *%mxy, %snindex
!    *%ifspecial, *%ityplun, *%itype
!    *%pfti, *%pftf, *%pftn
!    *%coli, *%colf, *%coln
!    *%luni, *%lunf, *%lunn
!
! !USES:
    use decompMod , only : get_proc_bounds
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!
! LOCAL VARAIBLES:
    integer :: begp, endp   ! per-proc beginning and ending pft indices
    integer :: begc, endc   ! per-proc beginning and ending column indices
    integer :: begl, endl   ! per-proc beginning and ending landunit indices
    integer :: begg, endg   ! per-proc gridcell ending gridcell indices
!------------------------------------------------------------------------

    ! Determine necessary indices

    call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

    call init_pft_type     (begp, endp, clm3%g%l%c%p)
    call init_column_type  (begc, endc, clm3%g%l%c)
    call init_landunit_type(begl, endl, clm3%g%l)
    call init_gridcell_type(begg, endg, clm3%g)

    ! pft ecophysiological constants

    call init_pft_ecophys_constants()

#if (defined CNDV)
    ! pft DGVM-specific ecophysiological constants

    call init_pft_DGVMecophys_constants()
#endif

    ! energy balance structures (all levels)

    call init_energy_balance_type(begp, endp, clm3%g%l%c%p%pebal)
    call init_energy_balance_type(begc, endc, clm3%g%l%c%cebal)
    call init_energy_balance_type(begl, endl, clm3%g%l%lebal)
    call init_energy_balance_type(begg, endg, clm3%g%gebal)
    call init_energy_balance_type(1,       1, clm3%mebal)

    ! water balance structures (all levels)

    call init_water_balance_type(begp, endp, clm3%g%l%c%p%pwbal)
    call init_water_balance_type(begc, endc, clm3%g%l%c%cwbal)
    call init_water_balance_type(begl, endl, clm3%g%l%lwbal)
    call init_water_balance_type(begg, endg, clm3%g%gwbal)
    call init_water_balance_type(1,       1, clm3%mwbal)

    ! carbon balance structures (pft and column levels)

    call init_carbon_balance_type(begp, endp, clm3%g%l%c%p%pcbal)
    call init_carbon_balance_type(begc, endc, clm3%g%l%c%ccbal)

    ! nitrogen balance structures (pft and column levels)

    call init_nitrogen_balance_type(begp, endp, clm3%g%l%c%p%pnbal)
    call init_nitrogen_balance_type(begc, endc, clm3%g%l%c%cnbal)

    ! pft physical state variables at pft level and averaged to the column

    call init_pft_pstate_type(begp, endp, clm3%g%l%c%p%pps)
    call init_pft_pstate_type(begc, endc, clm3%g%l%c%cps%pps_a)

    ! pft ecophysiological variables (only at the pft level for now)
    call init_pft_epv_type(begp, endp, clm3%g%l%c%p%pepv)

#if (defined CNDV) || (defined CROP)
    ! pft DGVM state variables at pft level and averaged to column

    call init_pft_pdgvstate_type(begp, endp, clm3%g%l%c%p%pdgvs)
#endif
#if (defined CNDV)
    call init_pft_pdgvstate_type(begc, endc, clm3%g%l%c%cdgvs%pdgvs_a)
#endif
    call init_pft_vstate_type(begp, endp, clm3%g%l%c%p%pvs)

    ! pft energy state variables at the pft level and averaged to the column

    call init_pft_estate_type(begp, endp, clm3%g%l%c%p%pes)
    call init_pft_estate_type(begc, endc, clm3%g%l%c%ces%pes_a)

    ! pft water state variables at the pft level and averaged to the column

    call init_pft_wstate_type(begp, endp, clm3%g%l%c%p%pws)
    call init_pft_wstate_type(begc, endc, clm3%g%l%c%cws%pws_a)

    ! pft carbon state variables at the pft level and averaged to the column

    call init_pft_cstate_type(begp, endp, clm3%g%l%c%p%pcs)
    call init_pft_cstate_type(begc, endc, clm3%g%l%c%ccs%pcs_a)
#if (defined C13)
    ! 4/14/05: PET
    ! Adding isotope code
    call init_pft_cstate_type(begp, endp, clm3%g%l%c%p%pc13s)
    call init_pft_cstate_type(begc, endc, clm3%g%l%c%cc13s%pcs_a)
#endif

    ! pft nitrogen state variables at the pft level and averaged to the column

    call init_pft_nstate_type(begp, endp, clm3%g%l%c%p%pns)
    call init_pft_nstate_type(begc, endc, clm3%g%l%c%cns%pns_a)

    ! pft energy flux variables at pft level and averaged to column

    call init_pft_eflux_type(begp, endp, clm3%g%l%c%p%pef)
    call init_pft_eflux_type(begc, endc, clm3%g%l%c%cef%pef_a)

    ! pft momentum flux variables at pft level and averaged to the column

    call init_pft_mflux_type(begp, endp, clm3%g%l%c%p%pmf)
    call init_pft_mflux_type(begc, endc, clm3%g%l%c%cmf%pmf_a)

    ! pft water flux variables

    call init_pft_wflux_type(begp, endp, clm3%g%l%c%p%pwf)
    call init_pft_wflux_type(begc, endc, clm3%g%l%c%cwf%pwf_a)

    ! pft carbon flux variables at pft level and averaged to column

    call init_pft_cflux_type(begp, endp, clm3%g%l%c%p%pcf)
    call init_pft_cflux_type(begc, endc, clm3%g%l%c%ccf%pcf_a)
#if (defined C13)
    ! 4/14/05: PET
    ! Adding isotope code
    call init_pft_cflux_type(begp, endp, clm3%g%l%c%p%pc13f)
    call init_pft_cflux_type(begc, endc, clm3%g%l%c%cc13f%pcf_a)
#endif

    ! pft nitrogen flux variables at pft level and averaged to column

    call init_pft_nflux_type(begp, endp, clm3%g%l%c%p%pnf)
    call init_pft_nflux_type(begc, endc, clm3%g%l%c%cnf%pnf_a)

    ! pft VOC flux variables at pft level and averaged to column

    call init_pft_vflux_type(begp, endp, clm3%g%l%c%p%pvf)
    call init_pft_vflux_type(begc, endc, clm3%g%l%c%cvf%pvf_a)

    ! gridcell VOC emission factors (heald, 05/06)

    call init_gridcell_efstate_type(begg, endg, clm3%g%gve)

    ! pft dust flux variables at pft level and averaged to column

    call init_pft_dflux_type(begp, endp, clm3%g%l%c%p%pdf)
    call init_pft_dflux_type(begc, endc, clm3%g%l%c%cdf%pdf_a)

    ! pft dry dep velocity variables at pft level and averaged to column

    call init_pft_depvd_type(begp, endp, clm3%g%l%c%p%pdd)

    ! column physical state variables at column level and averaged to
    ! the landunit and gridcell and model

    call init_column_pstate_type(begc, endc, clm3%g%l%c%cps)
    call init_column_pstate_type(begl, endl, clm3%g%l%lps%cps_a)
    call init_column_pstate_type(begg, endg, clm3%g%gps%cps_a)
    call init_column_pstate_type(1,       1, clm3%mps%cps_a)

    ! column energy state variables at column level and averaged to
    ! the landunit and gridcell and model

    call init_column_estate_type(begc, endc, clm3%g%l%c%ces)
    call init_column_estate_type(begl, endl, clm3%g%l%les%ces_a)
    call init_column_estate_type(begg, endg, clm3%g%ges%ces_a)
    call init_column_estate_type(1,       1, clm3%mes%ces_a)

    ! column water state variables at column level and averaged to
    ! the landunit and gridcell and model

    call init_column_wstate_type(begc, endc, clm3%g%l%c%cws)
    call init_column_wstate_type(begl, endl, clm3%g%l%lws%cws_a)
    call init_column_wstate_type(begg, endg, clm3%g%gws%cws_a)
    call init_column_wstate_type(1,       1, clm3%mws%cws_a)

    ! column carbon state variables at column level and averaged to
    ! the landunit and gridcell and model

    call init_column_cstate_type(begc, endc, clm3%g%l%c%ccs)
    call init_column_cstate_type(begl, endl, clm3%g%l%lcs%ccs_a)
    call init_column_cstate_type(begg, endg, clm3%g%gcs%ccs_a)
    call init_column_cstate_type(1,       1, clm3%mcs%ccs_a)
#if (defined C13)
    ! 4/14/05: PET
    ! Adding isotope code
    call init_column_cstate_type(begc, endc, clm3%g%l%c%cc13s)
#endif

    ! column nitrogen state variables at column level and averaged to
    ! the landunit and gridcell and model

    call init_column_nstate_type(begc, endc, clm3%g%l%c%cns)
    call init_column_nstate_type(begl, endl, clm3%g%l%lns%cns_a)
    call init_column_nstate_type(begg, endg, clm3%g%gns%cns_a)
    call init_column_nstate_type(1,       1, clm3%mns%cns_a)

    ! column energy flux variables at column level and averaged to
    ! the landunit and gridcell and model

    call init_column_eflux_type(begc, endc, clm3%g%l%c%cef)
    call init_column_eflux_type(begl, endl, clm3%g%l%lef%cef_a)
    call init_column_eflux_type(begg, endg, clm3%g%gef%cef_a)
    call init_column_eflux_type(1,       1, clm3%mef%cef_a)

    ! column water flux variables at column level and averaged to
    ! landunit, gridcell and model level

    call init_column_wflux_type(begc, endc, clm3%g%l%c%cwf)
    call init_column_wflux_type(begl, endl, clm3%g%l%lwf%cwf_a)
    call init_column_wflux_type(begg, endg, clm3%g%gwf%cwf_a)
    call init_column_wflux_type(1,       1, clm3%mwf%cwf_a)

    ! column carbon flux variables at column level

    call init_column_cflux_type(begc, endc, clm3%g%l%c%ccf)
#if (defined C13)
    ! 4/14/05: PET
    ! Adding isotope code
    call init_column_cflux_type(begc, endc, clm3%g%l%c%cc13f)
#endif

    ! column nitrogen flux variables at column level

    call init_column_nflux_type(begc, endc, clm3%g%l%c%cnf)

    ! land unit physical state variables

    call init_landunit_pstate_type(begl, endl, clm3%g%l%lps)

    ! land unit energy flux variables 

    call init_landunit_eflux_type(begl, endl, clm3%g%l%lef)

#if (defined CNDV)
    ! gridcell DGVM variables

    call init_gridcell_dgvstate_type(begg, endg, clm3%g%gdgvs)
#endif

    ! gridcell physical state variables

    call init_gridcell_pstate_type(begg, endg, clm3%g%gps)

    ! gridcell: water flux variables

    call init_gridcell_wflux_type(begg, endg, clm3%g%gwf)

    ! gridcell: energy flux variables

    call init_gridcell_eflux_type(begg, endg, clm3%g%gef)

    ! gridcell: water state variables

    call init_gridcell_wstate_type(begg, endg, clm3%g%gws)

    ! gridcell: energy state variables

    call init_gridcell_estate_type(begg, endg, clm3%g%ges)

    call init_atm2lnd_type  (begg    , endg    , clm_a2l)

  end subroutine initClmtype

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_type
!
! !INTERFACE:
  subroutine init_pft_type (beg, end, p)
!
! !DESCRIPTION:
! Initialize components of pft_type structure
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type(pft_type), intent(inout):: p
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(p%gridcell(beg:end),p%wtgcell(beg:end))
    allocate(p%landunit(beg:end),p%wtlunit(beg:end))
    allocate(p%column  (beg:end),p%wtcol  (beg:end))

    allocate(p%itype(beg:end))
    allocate(p%mxy(beg:end))
    allocate(p%area(beg:end))

  end subroutine init_pft_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_type
!
! !INTERFACE:
  subroutine init_column_type (beg, end, c)
!
! !DESCRIPTION:
! Initialize components of column_type structure
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type(column_type), intent(inout):: c
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

   allocate(c%gridcell(beg:end),c%wtgcell(beg:end))
   allocate(c%landunit(beg:end),c%wtlunit(beg:end))

   allocate(c%pfti(beg:end),c%pftf(beg:end),c%npfts(beg:end))

   allocate(c%itype(beg:end))

   allocate(c%area(beg:end))

  end subroutine init_column_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_landunit_type
!
! !INTERFACE:
  subroutine init_landunit_type (beg, end,l)
!
! !DESCRIPTION:
! Initialize components of landunit_type structure
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type(landunit_type), intent(inout):: l
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

   allocate(l%gridcell(beg:end),l%wtgcell(beg:end))

   allocate(l%coli(beg:end),l%colf(beg:end),l%ncolumns(beg:end))
   allocate(l%pfti(beg:end),l%pftf(beg:end),l%npfts   (beg:end))

   allocate(l%itype(beg:end))
   allocate(l%ifspecial(beg:end))
   allocate(l%lakpoi(beg:end))
   allocate(l%urbpoi(beg:end))

   ! MV - these should be moved to landunit physical state -MV
   allocate(l%canyon_hwr(beg:end))
   allocate(l%wtroad_perv(beg:end))
   allocate(l%ht_roof(beg:end))
   allocate(l%wtlunit_roof(beg:end))
   allocate(l%wind_hgt_canyon(beg:end))
   allocate(l%z_0_town(beg:end))
   allocate(l%z_d_town(beg:end))
   allocate(l%area(beg:end))


   l%canyon_hwr(beg:end)  = nan
   l%wtroad_perv(beg:end) = nan
   l%ht_roof(beg:end) = nan
   l%wtlunit_roof(beg:end) = nan
   l%wind_hgt_canyon(beg:end) = nan
   l%z_0_town(beg:end) = nan
   l%z_d_town(beg:end) = nan

  end subroutine init_landunit_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_gridcell_type
!
! !INTERFACE:
  subroutine init_gridcell_type (beg, end,g)
!
! !DESCRIPTION:
! Initialize components of gridcell_type structure
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type(gridcell_type), intent(inout):: g
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

   allocate(g%luni(beg:end),g%lunf(beg:end),g%nlandunits(beg:end))
   allocate(g%coli(beg:end),g%colf(beg:end),g%ncolumns  (beg:end))
   allocate(g%pfti(beg:end),g%pftf(beg:end),g%npfts     (beg:end))

   allocate(g%gindex(beg:end))
   allocate(g%area(beg:end))
   allocate(g%lat(beg:end))
   allocate(g%lon(beg:end))
   allocate(g%latdeg(beg:end))
   allocate(g%londeg(beg:end))
   allocate(g%gindex_a(beg:end))
   allocate(g%lat_a(beg:end))
   allocate(g%lon_a(beg:end))
   allocate(g%latdeg_a(beg:end))
   allocate(g%londeg_a(beg:end))

  end subroutine init_gridcell_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_energy_balance_type
!
! !INTERFACE:
  subroutine init_energy_balance_type(beg, end, ebal)
!
! !DESCRIPTION:
! Initialize energy balance variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type(energy_balance_type), intent(inout):: ebal
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(ebal%errsoi(beg:end))
    allocate(ebal%errseb(beg:end))
    allocate(ebal%errsol(beg:end))
    allocate(ebal%errlon(beg:end))

    ebal%errsoi(beg:end) = nan
    ebal%errseb(beg:end) = nan
    ebal%errsol(beg:end) = nan
    ebal%errlon(beg:end) = nan

  end subroutine init_energy_balance_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_water_balance_type
!
! !INTERFACE:
  subroutine init_water_balance_type(beg, end, wbal)
!
! !DESCRIPTION:
! Initialize water balance variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type(water_balance_type), intent(inout):: wbal
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(wbal%begwb(beg:end))
    allocate(wbal%endwb(beg:end))
    allocate(wbal%errh2o(beg:end))

    wbal%begwb(beg:end) = nan
    wbal%endwb(beg:end) = nan
    wbal%errh2o(beg:end) = nan

  end subroutine init_water_balance_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_carbon_balance_type
!
! !INTERFACE:
  subroutine init_carbon_balance_type(beg, end, cbal)
!
! !DESCRIPTION:
! Initialize carbon balance variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type(carbon_balance_type), intent(inout):: cbal
!
! !REVISION HISTORY:
! Created by Peter Thornton, 12/11/2003
!
!EOP
!------------------------------------------------------------------------

    allocate(cbal%begcb(beg:end))
    allocate(cbal%endcb(beg:end))
    allocate(cbal%errcb(beg:end))

    cbal%begcb(beg:end) = nan
    cbal%endcb(beg:end) = nan
    cbal%errcb(beg:end) = nan

  end subroutine init_carbon_balance_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_nitrogen_balance_type
!
! !INTERFACE:
  subroutine init_nitrogen_balance_type(beg, end, nbal)
!
! !DESCRIPTION:
! Initialize nitrogen balance variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type(nitrogen_balance_type), intent(inout):: nbal
!
! !REVISION HISTORY:
! Created by Peter Thornton, 12/11/2003
!
!EOP
!------------------------------------------------------------------------

    allocate(nbal%begnb(beg:end))
    allocate(nbal%endnb(beg:end))
    allocate(nbal%errnb(beg:end))

    nbal%begnb(beg:end) = nan
    nbal%endnb(beg:end) = nan
    nbal%errnb(beg:end) = nan

  end subroutine init_nitrogen_balance_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_ecophys_constants
!
! !INTERFACE:
  subroutine init_pft_ecophys_constants()
!
! !DESCRIPTION:
! Initialize pft physical state
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pftcon%noveg(0:numpft))
    allocate(pftcon%tree(0:numpft))
    allocate(pftcon%smpso(0:numpft)) 
    allocate(pftcon%smpsc(0:numpft)) 
    allocate(pftcon%fnitr(0:numpft))
    allocate(pftcon%foln(0:numpft))
    allocate(pftcon%dleaf(0:numpft))
    allocate(pftcon%c3psn(0:numpft))
    allocate(pftcon%vcmx25(0:numpft))
    allocate(pftcon%mp(0:numpft))
    allocate(pftcon%qe25(0:numpft))
    allocate(pftcon%xl(0:numpft))
    allocate(pftcon%rhol(0:numpft,numrad))
    allocate(pftcon%rhos(0:numpft,numrad))
    allocate(pftcon%taul(0:numpft,numrad))
    allocate(pftcon%taus(0:numpft,numrad))
    allocate(pftcon%z0mr(0:numpft))
    allocate(pftcon%displar(0:numpft))
    allocate(pftcon%roota_par(0:numpft))
    allocate(pftcon%rootb_par(0:numpft))
    allocate(pftcon%sla(0:numpft))
    allocate(pftcon%slatop(0:numpft))
    allocate(pftcon%dsladlai(0:numpft))
    allocate(pftcon%leafcn(0:numpft))
    allocate(pftcon%flnr(0:numpft))
    allocate(pftcon%woody(0:numpft))
    allocate(pftcon%lflitcn(0:numpft))
    allocate(pftcon%frootcn(0:numpft))
    allocate(pftcon%livewdcn(0:numpft))
    allocate(pftcon%deadwdcn(0:numpft))
#ifdef CROP
    allocate(pftcon%graincn(0:numpft))
#endif
    allocate(pftcon%froot_leaf(0:numpft))
    allocate(pftcon%stem_leaf(0:numpft))
    allocate(pftcon%croot_stem(0:numpft))
    allocate(pftcon%flivewd(0:numpft))
    allocate(pftcon%fcur(0:numpft))
    allocate(pftcon%lf_flab(0:numpft))
    allocate(pftcon%lf_fcel(0:numpft))
    allocate(pftcon%lf_flig(0:numpft))
    allocate(pftcon%fr_flab(0:numpft))
    allocate(pftcon%fr_fcel(0:numpft))
    allocate(pftcon%fr_flig(0:numpft))
    allocate(pftcon%dw_fcel(0:numpft))
    allocate(pftcon%dw_flig(0:numpft))
    allocate(pftcon%leaf_long(0:numpft))
    allocate(pftcon%evergreen(0:numpft))
    allocate(pftcon%stress_decid(0:numpft))
    allocate(pftcon%season_decid(0:numpft))
    allocate(pftcon%resist(0:numpft))
    allocate(pftcon%dwood(0:numpft))

    pftcon%noveg(:) = bigint
    pftcon%tree(:) = bigint
    pftcon%smpso(:) = nan
    pftcon%smpsc(:) = nan
    pftcon%fnitr(:) = nan
    pftcon%foln(:) = nan
    pftcon%dleaf(:) = nan
    pftcon%c3psn(:) = nan
    pftcon%vcmx25(:) = nan
    pftcon%mp(:) = nan
    pftcon%qe25(:) = nan
    pftcon%xl(:) = nan
    pftcon%rhol(:,:numrad) = nan
    pftcon%rhos(:,:numrad) = nan
    pftcon%taul(:,:numrad) = nan
    pftcon%taus(:,:numrad) = nan
    pftcon%z0mr(:) = nan
    pftcon%displar(:) = nan
    pftcon%roota_par(:) = nan
    pftcon%rootb_par(:) = nan
    pftcon%sla(:) = nan
    pftcon%slatop(:) = nan
    pftcon%dsladlai(:) = nan
    pftcon%leafcn(:) = nan
    pftcon%flnr(:) = nan
    pftcon%woody(:) = nan
    pftcon%lflitcn(:) = nan
    pftcon%frootcn(:) = nan
    pftcon%livewdcn(:) = nan
    pftcon%deadwdcn(:) = nan
#ifdef CROP
    pftcon%graincn(:) = nan
#endif
    pftcon%froot_leaf(:) = nan
    pftcon%stem_leaf(:) = nan
    pftcon%croot_stem(:) = nan
    pftcon%flivewd(:) = nan
    pftcon%fcur(:) = nan
    pftcon%lf_flab(:) = nan
    pftcon%lf_fcel(:) = nan
    pftcon%lf_flig(:) = nan
    pftcon%fr_flab(:) = nan
    pftcon%fr_fcel(:) = nan
    pftcon%fr_flig(:) = nan
    pftcon%dw_fcel(:) = nan
    pftcon%dw_flig(:) = nan
    pftcon%leaf_long(:) = nan
    pftcon%evergreen(:) = nan
    pftcon%stress_decid(:) = nan
    pftcon%season_decid(:) = nan
    pftcon%resist(:) = nan
    pftcon%dwood(:) = nan

  end subroutine init_pft_ecophys_constants

#if (defined CNDV) || (defined CROP)
!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_DGVMecophys_constants
!
! !INTERFACE:
  subroutine init_pft_DGVMecophys_constants()
!
! !DESCRIPTION:
! Initialize pft physical state
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(dgv_pftcon%crownarea_max(0:numpft))
    allocate(dgv_pftcon%tcmin(0:numpft))
    allocate(dgv_pftcon%tcmax(0:numpft))
    allocate(dgv_pftcon%gddmin(0:numpft))
    allocate(dgv_pftcon%twmax(0:numpft))
    allocate(dgv_pftcon%reinickerp(0:numpft))
    allocate(dgv_pftcon%allom1(0:numpft))
    allocate(dgv_pftcon%allom2(0:numpft))
    allocate(dgv_pftcon%allom3(0:numpft))

    dgv_pftcon%crownarea_max(:) = nan
    dgv_pftcon%tcmin(:) = nan
    dgv_pftcon%tcmax(:) = nan
    dgv_pftcon%gddmin(:) = nan
    dgv_pftcon%twmax(:) = nan
    dgv_pftcon%reinickerp(:) = nan
    dgv_pftcon%allom1(:) = nan
    dgv_pftcon%allom2(:) = nan
    dgv_pftcon%allom3(:) = nan

  end subroutine init_pft_DGVMecophys_constants
#endif

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_pstate_type
!
! !INTERFACE:
  subroutine init_pft_pstate_type(beg, end, pps)
!
! !DESCRIPTION:
! Initialize pft physical state
!
! !USES:
    use clm_varcon, only : spval
#if (defined CASA)
    use CASAMod   , only : npools, nresp_pools, nlive, npool_types
#endif
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_pstate_type), intent(inout):: pps
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pps%frac_veg_nosno(beg:end))
    allocate(pps%frac_veg_nosno_alb(beg:end))
    allocate(pps%emv(beg:end))
    allocate(pps%z0mv(beg:end))
    allocate(pps%z0hv(beg:end))
    allocate(pps%z0qv(beg:end))
    allocate(pps%rootfr(beg:end,1:nlevgrnd))
    allocate(pps%rootr(beg:end,1:nlevgrnd))
    allocate(pps%rresis(beg:end,1:nlevgrnd))
    allocate(pps%dewmx(beg:end))
    allocate(pps%rssun(beg:end))
    allocate(pps%rssha(beg:end))
    allocate(pps%laisun(beg:end))
    allocate(pps%laisha(beg:end))
    allocate(pps%btran(beg:end))
    allocate(pps%fsun(beg:end))
    allocate(pps%tlai(beg:end))
    allocate(pps%tsai(beg:end))
    allocate(pps%elai(beg:end))
    allocate(pps%esai(beg:end))
    allocate(pps%fwet(beg:end))
    allocate(pps%fdry(beg:end))
    allocate(pps%dt_veg(beg:end))
    allocate(pps%htop(beg:end))
    allocate(pps%hbot(beg:end))
    allocate(pps%z0m(beg:end))
    allocate(pps%displa(beg:end))
    allocate(pps%albd(beg:end,1:numrad))
    allocate(pps%albi(beg:end,1:numrad))
    allocate(pps%fabd(beg:end,1:numrad))
    allocate(pps%fabi(beg:end,1:numrad))
    allocate(pps%ftdd(beg:end,1:numrad))
    allocate(pps%ftid(beg:end,1:numrad))
    allocate(pps%ftii(beg:end,1:numrad))
    allocate(pps%u10(beg:end))
    allocate(pps%fv(beg:end))
    allocate(pps%ram1(beg:end))
#if (defined CROP)
    allocate(pps%hdidx(beg:end))
    allocate(pps%cumvd(beg:end))
    allocate(pps%htmx(beg:end))
    allocate(pps%vf(beg:end))
    allocate(pps%gddmaturity(beg:end))
    allocate(pps%gdd0(beg:end))
    allocate(pps%gdd8(beg:end))
    allocate(pps%gdd10(beg:end))
    allocate(pps%gdd020(beg:end))
    allocate(pps%gdd820(beg:end))
    allocate(pps%gdd1020(beg:end))
    allocate(pps%gddplant(beg:end))
    allocate(pps%gddtsoi(beg:end))
    allocate(pps%huileaf(beg:end))
    allocate(pps%huigrain(beg:end))
    allocate(pps%a10tmin(beg:end))
    allocate(pps%a5tmin(beg:end))
    allocate(pps%aleafi(beg:end))
    allocate(pps%astemi(beg:end))
    allocate(pps%aleaf(beg:end))
    allocate(pps%astem(beg:end))
    allocate(pps%croplive(beg:end))
    allocate(pps%cropplant(beg:end)) !,numpft)) ! make 2-D if using
    allocate(pps%harvdate(beg:end))  !,numpft)) ! crop rotation
    allocate(pps%idop(beg:end))
    allocate(pps%peaklai(beg:end))
#endif
    allocate(pps%vds(beg:end))
    allocate(pps%slasun(beg:end))
    allocate(pps%slasha(beg:end))
    allocate(pps%lncsun(beg:end))
    allocate(pps%lncsha(beg:end))
    allocate(pps%vcmxsun(beg:end))
    allocate(pps%vcmxsha(beg:end))
    allocate(pps%gdir(beg:end))
    allocate(pps%omega(beg:end,1:numrad))
    allocate(pps%eff_kid(beg:end,1:numrad))
    allocate(pps%eff_kii(beg:end,1:numrad))
    allocate(pps%sun_faid(beg:end,1:numrad))
    allocate(pps%sun_faii(beg:end,1:numrad))
    allocate(pps%sha_faid(beg:end,1:numrad))
    allocate(pps%sha_faii(beg:end,1:numrad))
    allocate(pps%forc_hgt_u_pft(beg:end))
    allocate(pps%forc_hgt_t_pft(beg:end))
    allocate(pps%forc_hgt_q_pft(beg:end))
    ! 4/14/05: PET
    ! Adding isotope code
    allocate(pps%cisun(beg:end))
    allocate(pps%cisha(beg:end))
#if (defined C13)
    allocate(pps%alphapsnsun(beg:end))
    allocate(pps%alphapsnsha(beg:end))
#endif
    ! heald: added from CASA definition
    allocate(pps%sandfrac(beg:end))
    allocate(pps%clayfrac(beg:end))
    pps%sandfrac(beg:end) = nan
    pps%clayfrac(beg:end) = nan
    allocate(pps%mlaidiff(beg:end))
    allocate(pps%rb1(beg:end))
    allocate(pps%annlai(12,beg:end))
    pps%mlaidiff(beg:end) = nan
    pps%rb1(beg:end) = nan
    pps%annlai(:,:) = nan

    
#if (defined CASA)
    allocate(pps%Closs(beg:end,npools))  ! C lost to atm
    allocate(pps%Ctrans(beg:end,npool_types))  ! C transfers out of pool types
    allocate(pps%Resp_C(beg:end,npools))
    allocate(pps%Tpool_C(beg:end,npools))! Total C pool size
    allocate(pps%eff(beg:end,nresp_pools))
    allocate(pps%frac_donor(beg:end,nresp_pools))
    allocate(pps%livefr(beg:end,nlive))  !live fraction
    allocate(pps%pet(beg:end))           !potential evaporation (mm h2o/s)
    allocate(pps%co2flux(beg:end))       ! net CO2 flux (g C/m2/sec) [+= atm]
    allocate(pps%fnpp(beg:end))          ! NPP  (g C/m2/sec)
    allocate(pps%soilt(beg:end))         !soil temp for top 30cm
    allocate(pps%smoist(beg:end))        !soil moisture for top 30cm
    allocate(pps%sz(beg:end))
    allocate(pps%watopt(beg:end))
    allocate(pps%watdry(beg:end))
    allocate(pps%soiltc(beg:end))         !soil temp for entire column
    allocate(pps%smoistc(beg:end))        !soil moisture for entire column
    allocate(pps%szc(beg:end))
    allocate(pps%watoptc(beg:end))
    allocate(pps%watdryc(beg:end))
    allocate(pps%Wlim(beg:end))
    allocate(pps%litterscalar(beg:end))
    allocate(pps%rootlitscalar(beg:end))
    allocate(pps%stressCD(beg:end))
    allocate(pps%excessC(beg:end))       ! excess Carbon (gC/m2/timestep)
    allocate(pps%bgtemp(beg:end))
    allocate(pps%bgmoist(beg:end))
    allocate(pps%plai(beg:end))          ! prognostic LAI (m2 leaf/m2 ground)
    allocate(pps%Cflux(beg:end))
    allocate(pps%XSCpool(beg:end))
    allocate(pps%tday(beg:end))     ! daily accumulated temperature (deg C)
    allocate(pps%tdayavg(beg:end))  ! daily averaged temperature (deg C)
    allocate(pps%tcount(beg:end))   ! counter for daily avg temp
    allocate(pps%degday(beg:end))   ! accumulated degree days (deg C)
    allocate(pps%ndegday(beg:end))  ! counter for number of degree days
    allocate(pps%stressT(beg:end))
    allocate(pps%stressW(beg:end))  ! water stress function for leaf loss
    allocate(pps%iseabeg(beg:end))  ! index for start of growing season
    allocate(pps%nstepbeg(beg:end)) ! nstep at start of growing season
    allocate(pps%lgrow(beg:end))    ! growing season index (0 or 1) to be
                                    ! passed daily to CASA to get NPP
#if (defined CLAMP)
    ! Summary variables added for the C-LAMP Experiments
    allocate(pps%casa_agnpp(beg:end))
    allocate(pps%casa_ar(beg:end))
    allocate(pps%casa_bgnpp(beg:end))
    allocate(pps%casa_cwdc(beg:end))
    allocate(pps%casa_cwdc_hr(beg:end))
    allocate(pps%casa_cwdc_loss(beg:end))
    allocate(pps%casa_frootc(beg:end))
    allocate(pps%casa_frootc_alloc(beg:end))
    allocate(pps%casa_frootc_loss(beg:end))
    allocate(pps%casa_gpp(beg:end))
    allocate(pps%casa_hr(beg:end))
    allocate(pps%casa_leafc(beg:end))
    allocate(pps%casa_leafc_alloc(beg:end))
    allocate(pps%casa_leafc_loss(beg:end))
    allocate(pps%casa_litterc(beg:end))
    allocate(pps%casa_litterc_hr(beg:end))
    allocate(pps%casa_litterc_loss(beg:end))
    allocate(pps%casa_nee(beg:end))
    allocate(pps%casa_nep(beg:end))
    allocate(pps%casa_npp(beg:end))
    allocate(pps%casa_soilc(beg:end))
    allocate(pps%casa_soilc_hr(beg:end))
    allocate(pps%casa_soilc_loss(beg:end))
    allocate(pps%casa_woodc(beg:end))
    allocate(pps%casa_woodc_alloc(beg:end))
    allocate(pps%casa_woodc_loss(beg:end))
#endif
#endif

    pps%frac_veg_nosno(beg:end) = bigint
    pps%frac_veg_nosno_alb(beg:end) = 0
    pps%emv(beg:end) = nan
    pps%z0mv(beg:end) = nan
    pps%z0hv(beg:end) = nan
    pps%z0qv(beg:end) = nan
    pps%rootfr(beg:end,:nlevgrnd) = spval
    pps%rootr (beg:end,:nlevgrnd) = spval
    pps%rresis(beg:end,:nlevgrnd) = spval
    pps%dewmx(beg:end) = nan
    pps%rssun(beg:end) = nan
    pps%rssha(beg:end) = nan
    pps%laisun(beg:end) = nan
    pps%laisha(beg:end) = nan
    pps%btran(beg:end) = nan
    pps%fsun(beg:end) = spval
    pps%tlai(beg:end) = 0._r8
    pps%tsai(beg:end) = 0._r8
    pps%elai(beg:end) = 0._r8
    pps%esai(beg:end) = 0._r8
    pps%fwet(beg:end) = nan
    pps%fdry(beg:end) = nan
    pps%dt_veg(beg:end) = nan
    pps%htop(beg:end) = 0._r8
    pps%hbot(beg:end) = 0._r8
    pps%z0m(beg:end) = nan
    pps%displa(beg:end) = nan
    pps%albd(beg:end,:numrad) = nan
    pps%albi(beg:end,:numrad) = nan
    pps%fabd(beg:end,:numrad) = nan
    pps%fabi(beg:end,:numrad) = nan
    pps%ftdd(beg:end,:numrad) = nan
    pps%ftid(beg:end,:numrad) = nan
    pps%ftii(beg:end,:numrad) = nan
    pps%u10(beg:end) = nan
    pps%fv(beg:end) = nan
    pps%ram1(beg:end) = nan
#if (defined CROP)
    pps%hdidx(beg:end) = nan
    pps%cumvd(beg:end) = nan
    pps%htmx(beg:end) = nan
    pps%vf(beg:end) = nan
    pps%gddmaturity(beg:end) = nan
    pps%gdd0(beg:end) = nan
    pps%gdd8(beg:end) = nan
    pps%gdd10(beg:end) = nan
    pps%gdd020(beg:end) = nan
    pps%gdd820(beg:end) = nan
    pps%gdd1020(beg:end) = nan
    pps%gddplant(beg:end) = nan
    pps%gddtsoi(beg:end) = nan
    pps%huileaf(beg:end) = nan
    pps%huigrain(beg:end) = nan
    pps%a10tmin(beg:end) = nan
    pps%a5tmin(beg:end) = nan
    pps%aleafi(beg:end) = nan
    pps%astemi(beg:end) = nan
    pps%aleaf(beg:end) = nan
    pps%astem(beg:end) = nan
    pps%croplive(beg:end) = bigint
    pps%cropplant(beg:end) = bigint
    pps%harvdate(beg:end) = bigint
    pps%idop(beg:end) = bigint
    pps%peaklai(beg:end) = bigint
#endif
    pps%vds(beg:end) = nan
    pps%slasun(beg:end) = nan
    pps%slasha(beg:end) = nan
    pps%lncsun(beg:end) = nan
    pps%lncsha(beg:end) = nan
    pps%vcmxsun(beg:end) = nan
    pps%vcmxsha(beg:end) = nan
    pps%gdir(beg:end) = nan
    pps%omega(beg:end,1:numrad) = nan
    pps%eff_kid(beg:end,1:numrad) = nan
    pps%eff_kii(beg:end,1:numrad) = nan
    pps%sun_faid(beg:end,1:numrad) = nan
    pps%sun_faii(beg:end,1:numrad) = nan
    pps%sha_faid(beg:end,1:numrad) = nan
    pps%sha_faii(beg:end,1:numrad) = nan
    pps%forc_hgt_u_pft(beg:end) = nan
    pps%forc_hgt_t_pft(beg:end) = nan
    pps%forc_hgt_q_pft(beg:end) = nan
    ! 4/14/05: PET
    ! Adding isotope code
    pps%cisun(beg:end) = nan
    pps%cisha(beg:end) = nan
#if (defined C13)
    pps%alphapsnsun(beg:end) = nan
    pps%alphapsnsha(beg:end) = nan
#endif

#if (defined CASA)
    pps%Closs(beg:end,:npools) = spval   !init w/ spval the variables that
    pps%Ctrans(beg:end,:npool_types) = spval   !init w/ spval the variables that
    pps%Resp_C(beg:end,:npools) = nan    !go to history, because CASA
    pps%Tpool_C(beg:end,:npools) = spval !routines do not get called on
    pps%livefr(beg:end,:nlive) = spval   !first timestep of nsrest=0 and
    pps%pet(beg:end) = spval             !history would get nans
    pps%co2flux(beg:end) = nan           !in the first timestep
    pps%fnpp(beg:end) = nan
    pps%excessC(beg:end) = spval
    pps%bgtemp(beg:end) = spval
    pps%bgmoist(beg:end) = spval
    pps%plai(beg:end) = spval
    pps%Cflux(beg:end) = nan
    pps%XSCpool(beg:end) = spval
    pps%tdayavg(beg:end) = spval
    pps%degday(beg:end) = spval
    pps%stressT(beg:end) = spval
    pps%stressW(beg:end) = spval
    pps%stressCD(beg:end) = spval
    pps%iseabeg(beg:end) = spval
    pps%nstepbeg(beg:end) = spval
    pps%lgrow(beg:end) = spval
    pps%eff(beg:end,:nresp_pools) = nan
    pps%frac_donor(beg:end,:nresp_pools) = nan
    pps%soilt(beg:end) = spval                  ! on history file
    pps%smoist(beg:end) = spval                 ! on history file
    pps%sz(beg:end) = nan
    pps%watopt(beg:end) = nan
    pps%watdry(beg:end) = nan
    pps%soiltc(beg:end) = nan
    pps%smoistc(beg:end) = nan
    pps%szc(beg:end) = nan
    pps%watoptc(beg:end) = spval                ! on history file
    pps%watdryc(beg:end) = spval                ! on history file
    pps%Wlim(beg:end) = spval                   ! on history file
    pps%litterscalar(beg:end) = nan
    pps%rootlitscalar(beg:end) = nan
    pps%tday(beg:end) = nan
    pps%tcount(beg:end) = nan
    pps%ndegday(beg:end) = nan
#if (defined CLAMP)
    ! Summary variables added for the C-LAMP Experiments
    pps%casa_agnpp(beg:end) = nan
    pps%casa_ar(beg:end) = nan
    pps%casa_bgnpp(beg:end) = nan
    pps%casa_cwdc(beg:end) = nan
    pps%casa_cwdc_hr(beg:end) = nan
    pps%casa_cwdc_loss(beg:end) = nan
    pps%casa_frootc(beg:end) = nan
    pps%casa_frootc_alloc(beg:end) = nan
    pps%casa_frootc_loss(beg:end) = nan
    pps%casa_gpp(beg:end) = nan
    pps%casa_hr(beg:end) = nan
    pps%casa_leafc(beg:end) = nan
    pps%casa_leafc_alloc(beg:end) = nan
    pps%casa_leafc_loss(beg:end) = nan
    pps%casa_litterc(beg:end) = nan
    pps%casa_litterc_loss(beg:end) = nan
    pps%casa_nee(beg:end) = nan
    pps%casa_nep(beg:end) = nan
    pps%casa_npp(beg:end) = nan
    pps%casa_soilc(beg:end) = nan
    pps%casa_soilc_hr(beg:end) = nan
    pps%casa_soilc_loss(beg:end) = nan
    pps%casa_woodc(beg:end) = nan
    pps%casa_woodc_alloc(beg:end) = nan
    pps%casa_woodc_loss(beg:end) = nan
#endif
#endif

  end subroutine init_pft_pstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_epv_type
!
! !INTERFACE:
  subroutine init_pft_epv_type(beg, end, pepv)
!
! !DESCRIPTION:
! Initialize pft ecophysiological variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_epv_type), intent(inout):: pepv
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

    allocate(pepv%dormant_flag(beg:end))
    allocate(pepv%days_active(beg:end))
    allocate(pepv%onset_flag(beg:end))
    allocate(pepv%onset_counter(beg:end))
    allocate(pepv%onset_gddflag(beg:end))
    allocate(pepv%onset_fdd(beg:end))
    allocate(pepv%onset_gdd(beg:end))
    allocate(pepv%onset_swi(beg:end))
    allocate(pepv%offset_flag(beg:end))
    allocate(pepv%offset_counter(beg:end))
    allocate(pepv%offset_fdd(beg:end))
    allocate(pepv%offset_swi(beg:end))
    allocate(pepv%lgsf(beg:end))
    allocate(pepv%bglfr(beg:end))
    allocate(pepv%bgtr(beg:end))
    allocate(pepv%dayl(beg:end))
    allocate(pepv%prev_dayl(beg:end))
    allocate(pepv%annavg_t2m(beg:end))
    allocate(pepv%tempavg_t2m(beg:end))
    allocate(pepv%gpp(beg:end))
    allocate(pepv%availc(beg:end))
    allocate(pepv%xsmrpool_recover(beg:end))
#if (defined C13)
    allocate(pepv%xsmrpool_c13ratio(beg:end))
#endif
    allocate(pepv%alloc_pnow(beg:end))
    allocate(pepv%c_allometry(beg:end))
    allocate(pepv%n_allometry(beg:end))
    allocate(pepv%plant_ndemand(beg:end))
    allocate(pepv%tempsum_potential_gpp(beg:end))
    allocate(pepv%annsum_potential_gpp(beg:end))
    allocate(pepv%tempmax_retransn(beg:end))
    allocate(pepv%annmax_retransn(beg:end))
    allocate(pepv%avail_retransn(beg:end))
    allocate(pepv%plant_nalloc(beg:end))
    allocate(pepv%plant_calloc(beg:end))
    allocate(pepv%excess_cflux(beg:end))
    allocate(pepv%downreg(beg:end))
    allocate(pepv%prev_leafc_to_litter(beg:end))
    allocate(pepv%prev_frootc_to_litter(beg:end))
    allocate(pepv%tempsum_npp(beg:end))
    allocate(pepv%annsum_npp(beg:end))
#if (defined CNDV)
    allocate(pepv%tempsum_litfall(beg:end))
    allocate(pepv%annsum_litfall(beg:end))
#endif
#if (defined C13)
    ! 4/21/05, PET
    ! Adding isotope code
    allocate(pepv%rc13_canair(beg:end))
    allocate(pepv%rc13_psnsun(beg:end))
    allocate(pepv%rc13_psnsha(beg:end))
#endif

    pepv%dormant_flag(beg:end) = nan
    pepv%days_active(beg:end) = nan
    pepv%onset_flag(beg:end) = nan
    pepv%onset_counter(beg:end) = nan
    pepv%onset_gddflag(beg:end) = nan
    pepv%onset_fdd(beg:end) = nan
    pepv%onset_gdd(beg:end) = nan
    pepv%onset_swi(beg:end) = nan
    pepv%offset_flag(beg:end) = nan
    pepv%offset_counter(beg:end) = nan
    pepv%offset_fdd(beg:end) = nan
    pepv%offset_swi(beg:end) = nan
    pepv%lgsf(beg:end) = nan
    pepv%bglfr(beg:end) = nan
    pepv%bgtr(beg:end) = nan
    pepv%dayl(beg:end) = nan
    pepv%prev_dayl(beg:end) = nan
    pepv%annavg_t2m(beg:end) = nan
    pepv%tempavg_t2m(beg:end) = nan
    pepv%gpp(beg:end) = nan
    pepv%availc(beg:end) = nan
    pepv%xsmrpool_recover(beg:end) = nan
#if (defined C13)
    pepv%xsmrpool_c13ratio(beg:end) = nan
#endif
    pepv%alloc_pnow(beg:end) = nan
    pepv%c_allometry(beg:end) = nan
    pepv%n_allometry(beg:end) = nan
    pepv%plant_ndemand(beg:end) = nan
    pepv%tempsum_potential_gpp(beg:end) = nan
    pepv%annsum_potential_gpp(beg:end) = nan
    pepv%tempmax_retransn(beg:end) = nan
    pepv%annmax_retransn(beg:end) = nan
    pepv%avail_retransn(beg:end) = nan
    pepv%plant_nalloc(beg:end) = nan
    pepv%plant_calloc(beg:end) = nan
    pepv%excess_cflux(beg:end) = nan
    pepv%downreg(beg:end) = nan
    pepv%prev_leafc_to_litter(beg:end) = nan
    pepv%prev_frootc_to_litter(beg:end) = nan
    pepv%tempsum_npp(beg:end) = nan
    pepv%annsum_npp(beg:end) = nan
#if (defined CNDV)
    pepv%tempsum_litfall(beg:end) = nan
    pepv%annsum_litfall(beg:end) = nan
#endif
#if (defined C13)
    ! 4/21/05, PET
    ! Adding isotope code
    pepv%rc13_canair(beg:end) = nan
    pepv%rc13_psnsun(beg:end) = nan
    pepv%rc13_psnsha(beg:end) = nan
#endif
    
  end subroutine init_pft_epv_type

#if (defined CNDV) || (defined CROP)
!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_pdgvstate_type
!
! !INTERFACE:
  subroutine init_pft_pdgvstate_type(beg, end, pdgvs)
!
! !DESCRIPTION:
! Initialize pft DGVM state variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_dgvstate_type), intent(inout):: pdgvs
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pdgvs%agddtw(beg:end))
    allocate(pdgvs%agdd(beg:end))
    allocate(pdgvs%t10(beg:end))
    allocate(pdgvs%t_mo(beg:end))
    allocate(pdgvs%t_mo_min(beg:end))
    allocate(pdgvs%prec365(beg:end))
    allocate(pdgvs%present(beg:end))
    allocate(pdgvs%pftmayexist(beg:end))
    allocate(pdgvs%nind(beg:end))
    allocate(pdgvs%lm_ind(beg:end))
    allocate(pdgvs%lai_ind(beg:end))
    allocate(pdgvs%fpcinc(beg:end))
    allocate(pdgvs%fpcgrid(beg:end))
    allocate(pdgvs%fpcgridold(beg:end))
    allocate(pdgvs%crownarea(beg:end))
    allocate(pdgvs%greffic(beg:end))
    allocate(pdgvs%heatstress(beg:end))

    pdgvs%agddtw(beg:end)           = nan
    pdgvs%agdd(beg:end)             = nan
    pdgvs%t10(beg:end)              = nan
    pdgvs%t_mo(beg:end)             = nan
    pdgvs%t_mo_min(beg:end)         = nan
    pdgvs%prec365(beg:end)          = nan
    pdgvs%present(beg:end)          = .false.
    pdgvs%pftmayexist(beg:end)      = .true.
    pdgvs%nind(beg:end)             = nan
    pdgvs%lm_ind(beg:end)           = nan
    pdgvs%lai_ind(beg:end)          = nan
    pdgvs%fpcinc(beg:end)           = nan
    pdgvs%fpcgrid(beg:end)          = nan
    pdgvs%fpcgridold(beg:end)       = nan
    pdgvs%crownarea(beg:end)        = nan
    pdgvs%greffic(beg:end)          = nan
    pdgvs%heatstress(beg:end)       = nan

  end subroutine init_pft_pdgvstate_type
#endif

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_vstate_type
!
! !INTERFACE:
  subroutine init_pft_vstate_type(beg, end, pvs)
!
! !DESCRIPTION:
! Initialize pft VOC variables
!
! !USES:
    use clm_varcon, only : spval
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_vstate_type), intent(inout):: pvs
!
! !REVISION HISTORY:
! Created by Erik Kluzek
!
!EOP
!------------------------------------------------------------------------

    allocate(pvs%t_veg24 (beg:end))
    allocate(pvs%t_veg240(beg:end))
    allocate(pvs%fsd24   (beg:end))
    allocate(pvs%fsd240  (beg:end))
    allocate(pvs%fsi24   (beg:end))
    allocate(pvs%fsi240  (beg:end))
    allocate(pvs%fsun24  (beg:end))
    allocate(pvs%fsun240 (beg:end))
    allocate(pvs%elai_p  (beg:end))

    pvs%t_veg24 (beg:end)   = spval
    pvs%t_veg240(beg:end)   = spval
    pvs%fsd24   (beg:end)   = spval
    pvs%fsd240  (beg:end)   = spval
    pvs%fsi24   (beg:end)   = spval
    pvs%fsi240  (beg:end)   = spval
    pvs%fsun24  (beg:end)   = spval
    pvs%fsun240 (beg:end)   = spval
    pvs%elai_p  (beg:end)   = spval
  end subroutine init_pft_vstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_estate_type
!
! !INTERFACE:
  subroutine init_pft_estate_type(beg, end, pes)
!
! !DESCRIPTION:
! Initialize pft energy state
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_estate_type), intent(inout):: pes
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pes%t_ref2m(beg:end))
    allocate(pes%t_ref2m_min(beg:end))
    allocate(pes%t_ref2m_max(beg:end))
    allocate(pes%t_ref2m_min_inst(beg:end))
    allocate(pes%t_ref2m_max_inst(beg:end))
    allocate(pes%q_ref2m(beg:end))
    allocate(pes%t_ref2m_u(beg:end))
    allocate(pes%t_ref2m_r(beg:end))
    allocate(pes%t_ref2m_min_u(beg:end))
    allocate(pes%t_ref2m_min_r(beg:end))
    allocate(pes%t_ref2m_max_u(beg:end))
    allocate(pes%t_ref2m_max_r(beg:end))
    allocate(pes%t_ref2m_min_inst_u(beg:end))
    allocate(pes%t_ref2m_min_inst_r(beg:end))
    allocate(pes%t_ref2m_max_inst_u(beg:end))
    allocate(pes%t_ref2m_max_inst_r(beg:end))
    allocate(pes%rh_ref2m(beg:end))
    allocate(pes%rh_ref2m_u(beg:end))
    allocate(pes%rh_ref2m_r(beg:end))
    allocate(pes%t_veg(beg:end))
    allocate(pes%thm(beg:end))

    pes%t_ref2m(beg:end) = nan
    pes%t_ref2m_min(beg:end) = nan
    pes%t_ref2m_max(beg:end) = nan
    pes%t_ref2m_min_inst(beg:end) = nan
    pes%t_ref2m_max_inst(beg:end) = nan
    pes%q_ref2m(beg:end) = nan
    pes%t_ref2m_u(beg:end) = nan
    pes%t_ref2m_r(beg:end) = nan
    pes%t_ref2m_min_u(beg:end) = nan
    pes%t_ref2m_min_r(beg:end) = nan
    pes%t_ref2m_max_u(beg:end) = nan
    pes%t_ref2m_max_r(beg:end) = nan
    pes%t_ref2m_min_inst_u(beg:end) = nan
    pes%t_ref2m_min_inst_r(beg:end) = nan
    pes%t_ref2m_max_inst_u(beg:end) = nan
    pes%t_ref2m_max_inst_r(beg:end) = nan
    pes%rh_ref2m(beg:end) = nan
    pes%rh_ref2m_u(beg:end) = nan
    pes%rh_ref2m_r(beg:end) = nan
    pes%t_veg(beg:end) = nan
    pes%thm(beg:end) = nan

  end subroutine init_pft_estate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_wstate_type
!
! !INTERFACE:
  subroutine init_pft_wstate_type(beg, end, pws)
!
! !DESCRIPTION:
! Initialize pft water state
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_wstate_type), intent(inout):: pws !pft water state
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pws%h2ocan(beg:end))
    pws%h2ocan(beg:end) = nan

  end subroutine init_pft_wstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_cstate_type
!
! !INTERFACE:
  subroutine init_pft_cstate_type(beg, end, pcs)
!
! !DESCRIPTION:
! Initialize pft carbon state
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_cstate_type), intent(inout):: pcs !pft carbon state
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

    allocate(pcs%leafc(beg:end))
    allocate(pcs%leafc_storage(beg:end))
    allocate(pcs%leafc_xfer(beg:end))
    allocate(pcs%frootc(beg:end))
    allocate(pcs%frootc_storage(beg:end))
    allocate(pcs%frootc_xfer(beg:end))
    allocate(pcs%livestemc(beg:end))
    allocate(pcs%livestemc_storage(beg:end))
    allocate(pcs%livestemc_xfer(beg:end))
    allocate(pcs%deadstemc(beg:end))
    allocate(pcs%deadstemc_storage(beg:end))
    allocate(pcs%deadstemc_xfer(beg:end))
    allocate(pcs%livecrootc(beg:end))
    allocate(pcs%livecrootc_storage(beg:end))
    allocate(pcs%livecrootc_xfer(beg:end))
    allocate(pcs%deadcrootc(beg:end))
    allocate(pcs%deadcrootc_storage(beg:end))
    allocate(pcs%deadcrootc_xfer(beg:end))
    allocate(pcs%gresp_storage(beg:end))
    allocate(pcs%gresp_xfer(beg:end))
    allocate(pcs%cpool(beg:end))
    allocate(pcs%xsmrpool(beg:end))
    allocate(pcs%pft_ctrunc(beg:end))
    allocate(pcs%dispvegc(beg:end))
    allocate(pcs%storvegc(beg:end))
    allocate(pcs%totvegc(beg:end))
    allocate(pcs%totpftc(beg:end))
    allocate(pcs%leafcmax(beg:end))
#if (defined CROP)
    allocate(pcs%grainc(beg:end))
    allocate(pcs%grainc_storage(beg:end))
    allocate(pcs%grainc_xfer(beg:end))
#endif
#if (defined CLAMP) && (defined CN)
    !CLAMP
    allocate(pcs%woodc(beg:end))
#endif

    pcs%leafc(beg:end) = nan
    pcs%leafc_storage(beg:end) = nan
    pcs%leafc_xfer(beg:end) = nan
    pcs%frootc(beg:end) = nan
    pcs%frootc_storage(beg:end) = nan
    pcs%frootc_xfer(beg:end) = nan
    pcs%livestemc(beg:end) = nan
    pcs%livestemc_storage(beg:end) = nan
    pcs%livestemc_xfer(beg:end) = nan
    pcs%deadstemc(beg:end) = nan
    pcs%deadstemc_storage(beg:end) = nan
    pcs%deadstemc_xfer(beg:end) = nan
    pcs%livecrootc(beg:end) = nan
    pcs%livecrootc_storage(beg:end) = nan
    pcs%livecrootc_xfer(beg:end) = nan
    pcs%deadcrootc(beg:end) = nan
    pcs%deadcrootc_storage(beg:end) = nan
    pcs%deadcrootc_xfer(beg:end) = nan
    pcs%gresp_storage(beg:end) = nan
    pcs%gresp_xfer(beg:end) = nan
    pcs%cpool(beg:end) = nan
    pcs%xsmrpool(beg:end) = nan
    pcs%pft_ctrunc(beg:end) = nan
    pcs%dispvegc(beg:end) = nan
    pcs%storvegc(beg:end) = nan
    pcs%totvegc(beg:end) = nan
    pcs%totpftc(beg:end) = nan
    pcs%leafcmax(beg:end) = nan
#if (defined CROP)
    pcs%grainc(beg:end) = nan
    pcs%grainc_storage(beg:end) = nan
    pcs%grainc_xfer(beg:end) = nan
#endif
#if (defined CLAMP) && (defined CN)
    !CLAMP
    pcs%woodc(beg:end) = nan
#endif

  end subroutine init_pft_cstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_nstate_type
!
! !INTERFACE:
  subroutine init_pft_nstate_type(beg, end, pns)
!
! !DESCRIPTION:
! Initialize pft nitrogen state
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_nstate_type), intent(inout):: pns !pft nitrogen state
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

#if (defined CROP)
    allocate(pns%grainn(beg:end))
    allocate(pns%grainn_storage(beg:end))
    allocate(pns%grainn_xfer(beg:end))
#endif
    allocate(pns%leafn(beg:end))
    allocate(pns%leafn_storage(beg:end))
    allocate(pns%leafn_xfer(beg:end))
    allocate(pns%frootn(beg:end))
    allocate(pns%frootn_storage(beg:end))
    allocate(pns%frootn_xfer(beg:end))
    allocate(pns%livestemn(beg:end))
    allocate(pns%livestemn_storage(beg:end))
    allocate(pns%livestemn_xfer(beg:end))
    allocate(pns%deadstemn(beg:end))
    allocate(pns%deadstemn_storage(beg:end))
    allocate(pns%deadstemn_xfer(beg:end))
    allocate(pns%livecrootn(beg:end))
    allocate(pns%livecrootn_storage(beg:end))
    allocate(pns%livecrootn_xfer(beg:end))
    allocate(pns%deadcrootn(beg:end))
    allocate(pns%deadcrootn_storage(beg:end))
    allocate(pns%deadcrootn_xfer(beg:end))
    allocate(pns%retransn(beg:end))
    allocate(pns%npool(beg:end))
    allocate(pns%pft_ntrunc(beg:end))
    allocate(pns%dispvegn(beg:end))
    allocate(pns%storvegn(beg:end))
    allocate(pns%totvegn(beg:end))
    allocate(pns%totpftn(beg:end))

#if (defined CROP)
    pns%grainn(beg:end) = nan
    pns%grainn_storage(beg:end) = nan
    pns%grainn_xfer(beg:end) = nan
#endif
    pns%leafn(beg:end) = nan
    pns%leafn_storage(beg:end) = nan
    pns%leafn_xfer(beg:end) = nan
    pns%frootn(beg:end) = nan
    pns%frootn_storage(beg:end) = nan
    pns%frootn_xfer(beg:end) = nan
    pns%livestemn(beg:end) = nan
    pns%livestemn_storage(beg:end) = nan
    pns%livestemn_xfer(beg:end) = nan
    pns%deadstemn(beg:end) = nan
    pns%deadstemn_storage(beg:end) = nan
    pns%deadstemn_xfer(beg:end) = nan
    pns%livecrootn(beg:end) = nan
    pns%livecrootn_storage(beg:end) = nan
    pns%livecrootn_xfer(beg:end) = nan
    pns%deadcrootn(beg:end) = nan
    pns%deadcrootn_storage(beg:end) = nan
    pns%deadcrootn_xfer(beg:end) = nan
    pns%retransn(beg:end) = nan
    pns%npool(beg:end) = nan
    pns%pft_ntrunc(beg:end) = nan
    pns%dispvegn(beg:end) = nan
    pns%storvegn(beg:end) = nan
    pns%totvegn(beg:end) = nan
    pns%totpftn(beg:end) = nan

  end subroutine init_pft_nstate_type
!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_eflux_type
!
! !INTERFACE:
  subroutine init_pft_eflux_type(beg, end, pef)
!
! !DESCRIPTION:
! Initialize pft energy flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_eflux_type), intent(inout):: pef
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pef%sabg(beg:end))
    allocate(pef%sabv(beg:end))
    allocate(pef%fsa(beg:end))
    allocate(pef%fsa_u(beg:end))
    allocate(pef%fsa_r(beg:end))
    allocate(pef%fsr(beg:end))
    allocate(pef%parsun(beg:end))
    allocate(pef%parsha(beg:end))
    allocate(pef%dlrad(beg:end))
    allocate(pef%ulrad(beg:end))
    allocate(pef%eflx_lh_tot(beg:end))
    allocate(pef%eflx_lh_tot_u(beg:end))
    allocate(pef%eflx_lh_tot_r(beg:end))
    allocate(pef%eflx_lh_grnd(beg:end))
    allocate(pef%eflx_soil_grnd(beg:end))
    allocate(pef%eflx_soil_grnd_u(beg:end))
    allocate(pef%eflx_soil_grnd_r(beg:end))
    allocate(pef%eflx_sh_tot(beg:end))
    allocate(pef%eflx_sh_tot_u(beg:end))
    allocate(pef%eflx_sh_tot_r(beg:end))
    allocate(pef%eflx_sh_grnd(beg:end))
    allocate(pef%eflx_sh_veg(beg:end))
    allocate(pef%eflx_lh_vege(beg:end))
    allocate(pef%eflx_lh_vegt(beg:end))
    allocate(pef%eflx_wasteheat_pft(beg:end))
    allocate(pef%eflx_heat_from_ac_pft(beg:end))
    allocate(pef%eflx_traffic_pft(beg:end))
    allocate(pef%eflx_anthro(beg:end))
    allocate(pef%cgrnd(beg:end))
    allocate(pef%cgrndl(beg:end))
    allocate(pef%cgrnds(beg:end))
    allocate(pef%eflx_gnet(beg:end))
    allocate(pef%dgnetdT(beg:end))
    allocate(pef%eflx_lwrad_out(beg:end))
    allocate(pef%eflx_lwrad_net(beg:end))
    allocate(pef%eflx_lwrad_net_u(beg:end))
    allocate(pef%eflx_lwrad_net_r(beg:end))
    allocate(pef%netrad(beg:end))
    allocate(pef%fsds_vis_d(beg:end))
    allocate(pef%fsds_nir_d(beg:end))
    allocate(pef%fsds_vis_i(beg:end))
    allocate(pef%fsds_nir_i(beg:end))
    allocate(pef%fsr_vis_d(beg:end))
    allocate(pef%fsr_nir_d(beg:end))
    allocate(pef%fsr_vis_i(beg:end))
    allocate(pef%fsr_nir_i(beg:end))
    allocate(pef%fsds_vis_d_ln(beg:end))
    allocate(pef%fsds_nir_d_ln(beg:end))
    allocate(pef%fsr_vis_d_ln(beg:end))
    allocate(pef%fsr_nir_d_ln(beg:end))
    allocate(pef%sun_add(beg:end,1:numrad))
    allocate(pef%tot_aid(beg:end,1:numrad))
    allocate(pef%sun_aid(beg:end,1:numrad))
    allocate(pef%sun_aii(beg:end,1:numrad))
    allocate(pef%sha_aid(beg:end,1:numrad))
    allocate(pef%sha_aii(beg:end,1:numrad))
    allocate(pef%sun_atot(beg:end,1:numrad))
    allocate(pef%sha_atot(beg:end,1:numrad))
    allocate(pef%sun_alf(beg:end,1:numrad))
    allocate(pef%sha_alf(beg:end,1:numrad))
    allocate(pef%sun_aperlai(beg:end,1:numrad))
    allocate(pef%sha_aperlai(beg:end,1:numrad))
    allocate(pef%sabg_lyr(beg:end,-nlevsno+1:1))
    allocate(pef%sfc_frc_aer(beg:end))
    allocate(pef%sfc_frc_bc(beg:end))
    allocate(pef%sfc_frc_oc(beg:end))
    allocate(pef%sfc_frc_dst(beg:end))
    allocate(pef%sfc_frc_aer_sno(beg:end))
    allocate(pef%sfc_frc_bc_sno(beg:end))
    allocate(pef%sfc_frc_oc_sno(beg:end))
    allocate(pef%sfc_frc_dst_sno(beg:end))
    allocate(pef%fsr_sno_vd(beg:end))
    allocate(pef%fsr_sno_nd(beg:end))
    allocate(pef%fsr_sno_vi(beg:end))
    allocate(pef%fsr_sno_ni(beg:end))
    allocate(pef%fsds_sno_vd(beg:end))
    allocate(pef%fsds_sno_nd(beg:end))
    allocate(pef%fsds_sno_vi(beg:end))
    allocate(pef%fsds_sno_ni(beg:end))

    pef%sabg(beg:end) = nan
    pef%sabv(beg:end) = nan
    pef%fsa(beg:end) = nan
    pef%fsa_u(beg:end) = nan
    pef%fsa_r(beg:end) = nan
    pef%fsr(beg:end) = nan
    pef%parsun(beg:end) = nan
    pef%parsha(beg:end) = nan
    pef%dlrad(beg:end) = nan
    pef%ulrad(beg:end) = nan
    pef%eflx_lh_tot(beg:end) = nan
    pef%eflx_lh_tot_u(beg:end) = nan
    pef%eflx_lh_tot_r(beg:end) = nan
    pef%eflx_lh_grnd(beg:end) = nan
    pef%eflx_soil_grnd(beg:end) = nan
    pef%eflx_soil_grnd_u(beg:end) = nan
    pef%eflx_soil_grnd_r(beg:end) = nan
    pef%eflx_sh_tot(beg:end) = nan
    pef%eflx_sh_tot_u(beg:end) = nan
    pef%eflx_sh_tot_r(beg:end) = nan
    pef%eflx_sh_grnd(beg:end) = nan
    pef%eflx_sh_veg(beg:end) = nan
    pef%eflx_lh_vege(beg:end) = nan
    pef%eflx_lh_vegt(beg:end) = nan
    pef%eflx_wasteheat_pft(beg:end) = nan
    pef%eflx_heat_from_ac_pft(beg:end) = nan
    pef%eflx_traffic_pft(beg:end) = nan
    pef%eflx_anthro(beg:end) = nan
    pef%cgrnd(beg:end) = nan
    pef%cgrndl(beg:end) = nan
    pef%cgrnds(beg:end) = nan
    pef%eflx_gnet(beg:end) = nan
    pef%dgnetdT(beg:end) = nan
    pef%eflx_lwrad_out(beg:end) = nan
    pef%eflx_lwrad_net(beg:end) = nan
    pef%eflx_lwrad_net_u(beg:end) = nan
    pef%eflx_lwrad_net_r(beg:end) = nan
    pef%netrad(beg:end) = nan
    pef%fsds_vis_d(beg:end) = nan
    pef%fsds_nir_d(beg:end) = nan
    pef%fsds_vis_i(beg:end) = nan
    pef%fsds_nir_i(beg:end) = nan
    pef%fsr_vis_d(beg:end) = nan
    pef%fsr_nir_d(beg:end) = nan
    pef%fsr_vis_i(beg:end) = nan
    pef%fsr_nir_i(beg:end) = nan
    pef%fsds_vis_d_ln(beg:end) = nan
    pef%fsds_nir_d_ln(beg:end) = nan
    pef%fsr_vis_d_ln(beg:end) = nan
    pef%fsr_nir_d_ln(beg:end) = nan
    pef%sun_add(beg:end,1:numrad) = nan
    pef%tot_aid(beg:end,1:numrad) = nan
    pef%sun_aid(beg:end,1:numrad) = nan
    pef%sun_aii(beg:end,1:numrad) = nan
    pef%sha_aid(beg:end,1:numrad) = nan
    pef%sha_aii(beg:end,1:numrad) = nan
    pef%sun_atot(beg:end,1:numrad) = nan
    pef%sha_atot(beg:end,1:numrad) = nan
    pef%sun_alf(beg:end,1:numrad) = nan
    pef%sha_alf(beg:end,1:numrad) = nan
    pef%sun_aperlai(beg:end,1:numrad) = nan
    pef%sha_aperlai(beg:end,1:numrad) = nan
    pef%sabg_lyr(beg:end,-nlevsno+1:1) = nan
    pef%sfc_frc_aer(beg:end) = nan
    pef%sfc_frc_bc(beg:end) = nan
    pef%sfc_frc_oc(beg:end) = nan
    pef%sfc_frc_dst(beg:end) = nan
    pef%sfc_frc_aer_sno(beg:end) = nan
    pef%sfc_frc_bc_sno(beg:end) = nan
    pef%sfc_frc_oc_sno(beg:end) = nan
    pef%sfc_frc_dst_sno(beg:end) = nan
    pef%fsr_sno_vd(beg:end) = nan
    pef%fsr_sno_nd(beg:end) = nan
    pef%fsr_sno_vi(beg:end) = nan
    pef%fsr_sno_ni(beg:end) = nan
    pef%fsds_sno_vd(beg:end) = nan
    pef%fsds_sno_nd(beg:end) = nan
    pef%fsds_sno_vi(beg:end) = nan
    pef%fsds_sno_ni(beg:end) = nan
  end subroutine init_pft_eflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_mflux_type
!
! !INTERFACE:
  subroutine init_pft_mflux_type(beg, end, pmf)
!
! !DESCRIPTION:
! Initialize pft momentum flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_mflux_type), intent(inout) :: pmf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pmf%taux(beg:end))
    allocate(pmf%tauy(beg:end))

    pmf%taux(beg:end) = nan
    pmf%tauy(beg:end) = nan

  end subroutine init_pft_mflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_wflux_type
!
! !INTERFACE:
  subroutine init_pft_wflux_type(beg, end, pwf)
!
! !DESCRIPTION:
! Initialize pft water flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_wflux_type), intent(inout) :: pwf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pwf%qflx_prec_intr(beg:end))
    allocate(pwf%qflx_prec_grnd(beg:end))
    allocate(pwf%qflx_rain_grnd(beg:end))
    allocate(pwf%qflx_snow_grnd(beg:end))
    allocate(pwf%qflx_snwcp_liq(beg:end))
    allocate(pwf%qflx_snwcp_ice(beg:end))
    allocate(pwf%qflx_evap_veg(beg:end))
    allocate(pwf%qflx_tran_veg(beg:end))
    allocate(pwf%qflx_evap_can(beg:end))
    allocate(pwf%qflx_evap_soi(beg:end))
    allocate(pwf%qflx_evap_tot(beg:end))
    allocate(pwf%qflx_evap_grnd(beg:end))
    allocate(pwf%qflx_dew_grnd(beg:end))
    allocate(pwf%qflx_sub_snow(beg:end))
    allocate(pwf%qflx_dew_snow(beg:end))

    pwf%qflx_prec_intr(beg:end) = nan
    pwf%qflx_prec_grnd(beg:end) = nan
    pwf%qflx_rain_grnd(beg:end) = nan
    pwf%qflx_snow_grnd(beg:end) = nan
    pwf%qflx_snwcp_liq(beg:end) = nan
    pwf%qflx_snwcp_ice(beg:end) = nan
    pwf%qflx_evap_veg(beg:end) = nan
    pwf%qflx_tran_veg(beg:end) = nan
    pwf%qflx_evap_can(beg:end) = nan
    pwf%qflx_evap_soi(beg:end) = nan
    pwf%qflx_evap_tot(beg:end) = nan
    pwf%qflx_evap_grnd(beg:end) = nan
    pwf%qflx_dew_grnd(beg:end) = nan
    pwf%qflx_sub_snow(beg:end) = nan
    pwf%qflx_dew_snow(beg:end) = nan

  end subroutine init_pft_wflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_cflux_type
!
! !INTERFACE:
  subroutine init_pft_cflux_type(beg, end, pcf)
!
! !DESCRIPTION:
! Initialize pft carbon flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_cflux_type), intent(inout) :: pcf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pcf%psnsun(beg:end))
    allocate(pcf%psnsha(beg:end))
    allocate(pcf%fpsn(beg:end))
    allocate(pcf%fco2(beg:end))

    allocate(pcf%m_leafc_to_litter(beg:end))
    allocate(pcf%m_frootc_to_litter(beg:end))
    allocate(pcf%m_leafc_storage_to_litter(beg:end))
    allocate(pcf%m_frootc_storage_to_litter(beg:end))
    allocate(pcf%m_livestemc_storage_to_litter(beg:end))
    allocate(pcf%m_deadstemc_storage_to_litter(beg:end))
    allocate(pcf%m_livecrootc_storage_to_litter(beg:end))
    allocate(pcf%m_deadcrootc_storage_to_litter(beg:end))
    allocate(pcf%m_leafc_xfer_to_litter(beg:end))
    allocate(pcf%m_frootc_xfer_to_litter(beg:end))
    allocate(pcf%m_livestemc_xfer_to_litter(beg:end))
    allocate(pcf%m_deadstemc_xfer_to_litter(beg:end))
    allocate(pcf%m_livecrootc_xfer_to_litter(beg:end))
    allocate(pcf%m_deadcrootc_xfer_to_litter(beg:end))
    allocate(pcf%m_livestemc_to_litter(beg:end))
    allocate(pcf%m_deadstemc_to_litter(beg:end))
    allocate(pcf%m_livecrootc_to_litter(beg:end))
    allocate(pcf%m_deadcrootc_to_litter(beg:end))
    allocate(pcf%m_gresp_storage_to_litter(beg:end))
    allocate(pcf%m_gresp_xfer_to_litter(beg:end))
    allocate(pcf%hrv_leafc_to_litter(beg:end))             
    allocate(pcf%hrv_leafc_storage_to_litter(beg:end))     
    allocate(pcf%hrv_leafc_xfer_to_litter(beg:end))        
    allocate(pcf%hrv_frootc_to_litter(beg:end))            
    allocate(pcf%hrv_frootc_storage_to_litter(beg:end))    
    allocate(pcf%hrv_frootc_xfer_to_litter(beg:end))       
    allocate(pcf%hrv_livestemc_to_litter(beg:end))         
    allocate(pcf%hrv_livestemc_storage_to_litter(beg:end)) 
    allocate(pcf%hrv_livestemc_xfer_to_litter(beg:end))    
    allocate(pcf%hrv_deadstemc_to_prod10c(beg:end))        
    allocate(pcf%hrv_deadstemc_to_prod100c(beg:end))       
    allocate(pcf%hrv_deadstemc_storage_to_litter(beg:end)) 
    allocate(pcf%hrv_deadstemc_xfer_to_litter(beg:end))    
    allocate(pcf%hrv_livecrootc_to_litter(beg:end))        
    allocate(pcf%hrv_livecrootc_storage_to_litter(beg:end))
    allocate(pcf%hrv_livecrootc_xfer_to_litter(beg:end))   
    allocate(pcf%hrv_deadcrootc_to_litter(beg:end))        
    allocate(pcf%hrv_deadcrootc_storage_to_litter(beg:end))
    allocate(pcf%hrv_deadcrootc_xfer_to_litter(beg:end))   
    allocate(pcf%hrv_gresp_storage_to_litter(beg:end))     
    allocate(pcf%hrv_gresp_xfer_to_litter(beg:end))        
    allocate(pcf%hrv_xsmrpool_to_atm(beg:end))                 
    allocate(pcf%m_leafc_to_fire(beg:end))
    allocate(pcf%m_frootc_to_fire(beg:end))
    allocate(pcf%m_leafc_storage_to_fire(beg:end))
    allocate(pcf%m_frootc_storage_to_fire(beg:end))
    allocate(pcf%m_livestemc_storage_to_fire(beg:end))
    allocate(pcf%m_deadstemc_storage_to_fire(beg:end))
    allocate(pcf%m_livecrootc_storage_to_fire(beg:end))
    allocate(pcf%m_deadcrootc_storage_to_fire(beg:end))
    allocate(pcf%m_leafc_xfer_to_fire(beg:end))
    allocate(pcf%m_frootc_xfer_to_fire(beg:end))
    allocate(pcf%m_livestemc_xfer_to_fire(beg:end))
    allocate(pcf%m_deadstemc_xfer_to_fire(beg:end))
    allocate(pcf%m_livecrootc_xfer_to_fire(beg:end))
    allocate(pcf%m_deadcrootc_xfer_to_fire(beg:end))
    allocate(pcf%m_livestemc_to_fire(beg:end))
    allocate(pcf%m_deadstemc_to_fire(beg:end))
    allocate(pcf%m_deadstemc_to_litter_fire(beg:end))
    allocate(pcf%m_livecrootc_to_fire(beg:end))
    allocate(pcf%m_deadcrootc_to_fire(beg:end))
    allocate(pcf%m_deadcrootc_to_litter_fire(beg:end))
    allocate(pcf%m_gresp_storage_to_fire(beg:end))
    allocate(pcf%m_gresp_xfer_to_fire(beg:end))
    allocate(pcf%leafc_xfer_to_leafc(beg:end))
    allocate(pcf%frootc_xfer_to_frootc(beg:end))
    allocate(pcf%livestemc_xfer_to_livestemc(beg:end))
    allocate(pcf%deadstemc_xfer_to_deadstemc(beg:end))
    allocate(pcf%livecrootc_xfer_to_livecrootc(beg:end))
    allocate(pcf%deadcrootc_xfer_to_deadcrootc(beg:end))
    allocate(pcf%leafc_to_litter(beg:end))
    allocate(pcf%frootc_to_litter(beg:end))
    allocate(pcf%leaf_mr(beg:end))
    allocate(pcf%froot_mr(beg:end))
    allocate(pcf%livestem_mr(beg:end))
    allocate(pcf%livecroot_mr(beg:end))
    allocate(pcf%leaf_curmr(beg:end))
    allocate(pcf%froot_curmr(beg:end))
    allocate(pcf%livestem_curmr(beg:end))
    allocate(pcf%livecroot_curmr(beg:end))
    allocate(pcf%leaf_xsmr(beg:end))
    allocate(pcf%froot_xsmr(beg:end))
    allocate(pcf%livestem_xsmr(beg:end))
    allocate(pcf%livecroot_xsmr(beg:end))
    allocate(pcf%psnsun_to_cpool(beg:end))
    allocate(pcf%psnshade_to_cpool(beg:end))
    allocate(pcf%cpool_to_xsmrpool(beg:end))
    allocate(pcf%cpool_to_leafc(beg:end))
    allocate(pcf%cpool_to_leafc_storage(beg:end))
    allocate(pcf%cpool_to_frootc(beg:end))
    allocate(pcf%cpool_to_frootc_storage(beg:end))
    allocate(pcf%cpool_to_livestemc(beg:end))
    allocate(pcf%cpool_to_livestemc_storage(beg:end))
    allocate(pcf%cpool_to_deadstemc(beg:end))
    allocate(pcf%cpool_to_deadstemc_storage(beg:end))
    allocate(pcf%cpool_to_livecrootc(beg:end))
    allocate(pcf%cpool_to_livecrootc_storage(beg:end))
    allocate(pcf%cpool_to_deadcrootc(beg:end))
    allocate(pcf%cpool_to_deadcrootc_storage(beg:end))
    allocate(pcf%cpool_to_gresp_storage(beg:end))
    allocate(pcf%cpool_leaf_gr(beg:end))
    allocate(pcf%cpool_leaf_storage_gr(beg:end))
    allocate(pcf%transfer_leaf_gr(beg:end))
    allocate(pcf%cpool_froot_gr(beg:end))
    allocate(pcf%cpool_froot_storage_gr(beg:end))
    allocate(pcf%transfer_froot_gr(beg:end))
    allocate(pcf%cpool_livestem_gr(beg:end))
    allocate(pcf%cpool_livestem_storage_gr(beg:end))
    allocate(pcf%transfer_livestem_gr(beg:end))
    allocate(pcf%cpool_deadstem_gr(beg:end))
    allocate(pcf%cpool_deadstem_storage_gr(beg:end))
    allocate(pcf%transfer_deadstem_gr(beg:end))
    allocate(pcf%cpool_livecroot_gr(beg:end))
    allocate(pcf%cpool_livecroot_storage_gr(beg:end))
    allocate(pcf%transfer_livecroot_gr(beg:end))
    allocate(pcf%cpool_deadcroot_gr(beg:end))
    allocate(pcf%cpool_deadcroot_storage_gr(beg:end))
    allocate(pcf%transfer_deadcroot_gr(beg:end))
    allocate(pcf%leafc_storage_to_xfer(beg:end))
    allocate(pcf%frootc_storage_to_xfer(beg:end))
    allocate(pcf%livestemc_storage_to_xfer(beg:end))
    allocate(pcf%deadstemc_storage_to_xfer(beg:end))
    allocate(pcf%livecrootc_storage_to_xfer(beg:end))
    allocate(pcf%deadcrootc_storage_to_xfer(beg:end))
    allocate(pcf%gresp_storage_to_xfer(beg:end))
    allocate(pcf%livestemc_to_deadstemc(beg:end))
    allocate(pcf%livecrootc_to_deadcrootc(beg:end))
    allocate(pcf%gpp(beg:end))
    allocate(pcf%mr(beg:end))
    allocate(pcf%current_gr(beg:end))
    allocate(pcf%transfer_gr(beg:end))
    allocate(pcf%storage_gr(beg:end))
    allocate(pcf%gr(beg:end))
    allocate(pcf%ar(beg:end))
    allocate(pcf%rr(beg:end))
    allocate(pcf%npp(beg:end))
    allocate(pcf%agnpp(beg:end))
    allocate(pcf%bgnpp(beg:end))
    allocate(pcf%litfall(beg:end))
    allocate(pcf%vegfire(beg:end))
    allocate(pcf%wood_harvestc(beg:end))
    allocate(pcf%pft_cinputs(beg:end))
    allocate(pcf%pft_coutputs(beg:end))
    allocate(pcf%pft_fire_closs(beg:end))
#if (defined CROP)
    allocate(pcf%xsmrpool_to_atm(beg:end))
    allocate(pcf%grainc_xfer_to_grainc(beg:end))
    allocate(pcf%livestemc_to_litter(beg:end))
    allocate(pcf%grainc_to_food(beg:end))
    allocate(pcf%cpool_to_grainc(beg:end))
    allocate(pcf%cpool_to_grainc_storage(beg:end))
    allocate(pcf%cpool_grain_gr(beg:end))
    allocate(pcf%cpool_grain_storage_gr(beg:end))
    allocate(pcf%transfer_grain_gr(beg:end))
    allocate(pcf%grainc_storage_to_xfer(beg:end))
#endif
#if (defined CLAMP) && (defined CN)
    !CLAMP
    allocate(pcf%frootc_alloc(beg:end))
    allocate(pcf%frootc_loss(beg:end))
    allocate(pcf%leafc_alloc(beg:end))
    allocate(pcf%leafc_loss(beg:end))
    allocate(pcf%woodc_alloc(beg:end))
    allocate(pcf%woodc_loss(beg:end))
#endif

    pcf%psnsun(beg:end) = nan
    pcf%psnsha(beg:end) = nan
    pcf%fpsn(beg:end) = nan
    pcf%fco2(beg:end) = 0._r8

    pcf%m_leafc_to_litter(beg:end) = nan
    pcf%m_frootc_to_litter(beg:end) = nan
    pcf%m_leafc_storage_to_litter(beg:end) = nan
    pcf%m_frootc_storage_to_litter(beg:end) = nan
    pcf%m_livestemc_storage_to_litter(beg:end) = nan
    pcf%m_deadstemc_storage_to_litter(beg:end) = nan
    pcf%m_livecrootc_storage_to_litter(beg:end) = nan
    pcf%m_deadcrootc_storage_to_litter(beg:end) = nan
    pcf%m_leafc_xfer_to_litter(beg:end) = nan
    pcf%m_frootc_xfer_to_litter(beg:end) = nan
    pcf%m_livestemc_xfer_to_litter(beg:end) = nan
    pcf%m_deadstemc_xfer_to_litter(beg:end) = nan
    pcf%m_livecrootc_xfer_to_litter(beg:end) = nan
    pcf%m_deadcrootc_xfer_to_litter(beg:end) = nan
    pcf%m_livestemc_to_litter(beg:end) = nan
    pcf%m_deadstemc_to_litter(beg:end) = nan
    pcf%m_livecrootc_to_litter(beg:end) = nan
    pcf%m_deadcrootc_to_litter(beg:end) = nan
    pcf%m_gresp_storage_to_litter(beg:end) = nan
    pcf%m_gresp_xfer_to_litter(beg:end) = nan
    pcf%hrv_leafc_to_litter(beg:end) = nan             
    pcf%hrv_leafc_storage_to_litter(beg:end) = nan     
    pcf%hrv_leafc_xfer_to_litter(beg:end) = nan        
    pcf%hrv_frootc_to_litter(beg:end) = nan            
    pcf%hrv_frootc_storage_to_litter(beg:end) = nan    
    pcf%hrv_frootc_xfer_to_litter(beg:end) = nan       
    pcf%hrv_livestemc_to_litter(beg:end) = nan         
    pcf%hrv_livestemc_storage_to_litter(beg:end) = nan 
    pcf%hrv_livestemc_xfer_to_litter(beg:end) = nan    
    pcf%hrv_deadstemc_to_prod10c(beg:end) = nan        
    pcf%hrv_deadstemc_to_prod100c(beg:end) = nan       
    pcf%hrv_deadstemc_storage_to_litter(beg:end) = nan 
    pcf%hrv_deadstemc_xfer_to_litter(beg:end) = nan    
    pcf%hrv_livecrootc_to_litter(beg:end) = nan        
    pcf%hrv_livecrootc_storage_to_litter(beg:end) = nan
    pcf%hrv_livecrootc_xfer_to_litter(beg:end) = nan   
    pcf%hrv_deadcrootc_to_litter(beg:end) = nan        
    pcf%hrv_deadcrootc_storage_to_litter(beg:end) = nan
    pcf%hrv_deadcrootc_xfer_to_litter(beg:end) = nan   
    pcf%hrv_gresp_storage_to_litter(beg:end) = nan     
    pcf%hrv_gresp_xfer_to_litter(beg:end) = nan        
    pcf%hrv_xsmrpool_to_atm(beg:end) = nan                 
    pcf%m_leafc_to_fire(beg:end) = nan
    pcf%m_frootc_to_fire(beg:end) = nan
    pcf%m_leafc_storage_to_fire(beg:end) = nan
    pcf%m_frootc_storage_to_fire(beg:end) = nan
    pcf%m_livestemc_storage_to_fire(beg:end) = nan
    pcf%m_deadstemc_storage_to_fire(beg:end) = nan
    pcf%m_livecrootc_storage_to_fire(beg:end) = nan
    pcf%m_deadcrootc_storage_to_fire(beg:end) = nan
    pcf%m_leafc_xfer_to_fire(beg:end) = nan
    pcf%m_frootc_xfer_to_fire(beg:end) = nan
    pcf%m_livestemc_xfer_to_fire(beg:end) = nan
    pcf%m_deadstemc_xfer_to_fire(beg:end) = nan
    pcf%m_livecrootc_xfer_to_fire(beg:end) = nan
    pcf%m_deadcrootc_xfer_to_fire(beg:end) = nan
    pcf%m_livestemc_to_fire(beg:end) = nan
    pcf%m_deadstemc_to_fire(beg:end) = nan
    pcf%m_deadstemc_to_litter_fire(beg:end) = nan
    pcf%m_livecrootc_to_fire(beg:end) = nan
    pcf%m_deadcrootc_to_fire(beg:end) = nan
    pcf%m_deadcrootc_to_litter_fire(beg:end) = nan
    pcf%m_gresp_storage_to_fire(beg:end) = nan
    pcf%m_gresp_xfer_to_fire(beg:end) = nan
    pcf%leafc_xfer_to_leafc(beg:end) = nan
    pcf%frootc_xfer_to_frootc(beg:end) = nan
    pcf%livestemc_xfer_to_livestemc(beg:end) = nan
    pcf%deadstemc_xfer_to_deadstemc(beg:end) = nan
    pcf%livecrootc_xfer_to_livecrootc(beg:end) = nan
    pcf%deadcrootc_xfer_to_deadcrootc(beg:end) = nan
    pcf%leafc_to_litter(beg:end) = nan
    pcf%frootc_to_litter(beg:end) = nan
    pcf%leaf_mr(beg:end) = nan
    pcf%froot_mr(beg:end) = nan
    pcf%livestem_mr(beg:end) = nan
    pcf%livecroot_mr(beg:end) = nan
    pcf%leaf_curmr(beg:end) = nan
    pcf%froot_curmr(beg:end) = nan
    pcf%livestem_curmr(beg:end) = nan
    pcf%livecroot_curmr(beg:end) = nan
    pcf%leaf_xsmr(beg:end) = nan
    pcf%froot_xsmr(beg:end) = nan
    pcf%livestem_xsmr(beg:end) = nan
    pcf%livecroot_xsmr(beg:end) = nan
    pcf%psnsun_to_cpool(beg:end) = nan
    pcf%psnshade_to_cpool(beg:end) = nan
    pcf%cpool_to_xsmrpool(beg:end) = nan
    pcf%cpool_to_leafc(beg:end) = nan
    pcf%cpool_to_leafc_storage(beg:end) = nan
    pcf%cpool_to_frootc(beg:end) = nan
    pcf%cpool_to_frootc_storage(beg:end) = nan
    pcf%cpool_to_livestemc(beg:end) = nan
    pcf%cpool_to_livestemc_storage(beg:end) = nan
    pcf%cpool_to_deadstemc(beg:end) = nan
    pcf%cpool_to_deadstemc_storage(beg:end) = nan
    pcf%cpool_to_livecrootc(beg:end) = nan
    pcf%cpool_to_livecrootc_storage(beg:end) = nan
    pcf%cpool_to_deadcrootc(beg:end) = nan
    pcf%cpool_to_deadcrootc_storage(beg:end) = nan
    pcf%cpool_to_gresp_storage(beg:end) = nan
    pcf%cpool_leaf_gr(beg:end) = nan
    pcf%cpool_leaf_storage_gr(beg:end) = nan
    pcf%transfer_leaf_gr(beg:end) = nan
    pcf%cpool_froot_gr(beg:end) = nan
    pcf%cpool_froot_storage_gr(beg:end) = nan
    pcf%transfer_froot_gr(beg:end) = nan
    pcf%cpool_livestem_gr(beg:end) = nan
    pcf%cpool_livestem_storage_gr(beg:end) = nan
    pcf%transfer_livestem_gr(beg:end) = nan
    pcf%cpool_deadstem_gr(beg:end) = nan
    pcf%cpool_deadstem_storage_gr(beg:end) = nan
    pcf%transfer_deadstem_gr(beg:end) = nan
    pcf%cpool_livecroot_gr(beg:end) = nan
    pcf%cpool_livecroot_storage_gr(beg:end) = nan
    pcf%transfer_livecroot_gr(beg:end) = nan
    pcf%cpool_deadcroot_gr(beg:end) = nan
    pcf%cpool_deadcroot_storage_gr(beg:end) = nan
    pcf%transfer_deadcroot_gr(beg:end) = nan
    pcf%leafc_storage_to_xfer(beg:end) = nan
    pcf%frootc_storage_to_xfer(beg:end) = nan
    pcf%livestemc_storage_to_xfer(beg:end) = nan
    pcf%deadstemc_storage_to_xfer(beg:end) = nan
    pcf%livecrootc_storage_to_xfer(beg:end) = nan
    pcf%deadcrootc_storage_to_xfer(beg:end) = nan
    pcf%gresp_storage_to_xfer(beg:end) = nan
    pcf%livestemc_to_deadstemc(beg:end) = nan
    pcf%livecrootc_to_deadcrootc(beg:end) = nan
    pcf%gpp(beg:end) = nan
    pcf%mr(beg:end) = nan
    pcf%current_gr(beg:end) = nan
    pcf%transfer_gr(beg:end) = nan
    pcf%storage_gr(beg:end) = nan
    pcf%gr(beg:end) = nan
    pcf%ar(beg:end) = nan
    pcf%rr(beg:end) = nan
    pcf%npp(beg:end) = nan
    pcf%agnpp(beg:end) = nan
    pcf%bgnpp(beg:end) = nan
    pcf%litfall(beg:end) = nan
    pcf%vegfire(beg:end) = nan
    pcf%wood_harvestc(beg:end) = nan
    pcf%pft_cinputs(beg:end) = nan
    pcf%pft_coutputs(beg:end) = nan
    pcf%pft_fire_closs(beg:end) = nan
#if (defined CROP)
    pcf%xsmrpool_to_atm(beg:end) = nan
    pcf%grainc_xfer_to_grainc(beg:end) = nan
    pcf%livestemc_to_litter(beg:end) = nan
    pcf%grainc_to_food(beg:end) = nan
    pcf%cpool_to_grainc(beg:end) = nan
    pcf%cpool_to_grainc_storage(beg:end) = nan
    pcf%cpool_grain_gr(beg:end) = nan
    pcf%cpool_grain_storage_gr(beg:end) = nan
    pcf%transfer_grain_gr(beg:end) = nan
    pcf%grainc_storage_to_xfer(beg:end) = nan
#endif
#if (defined CLAMP) && (defined CN)
    !CLAMP
    pcf%frootc_alloc(beg:end) = nan
    pcf%frootc_loss(beg:end) = nan
    pcf%leafc_alloc(beg:end) = nan
    pcf%leafc_loss(beg:end) = nan
    pcf%woodc_alloc(beg:end) = nan
    pcf%woodc_loss(beg:end) = nan
#endif

  end subroutine init_pft_cflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_nflux_type
!
! !INTERFACE:
  subroutine init_pft_nflux_type(beg, end, pnf)
!
! !DESCRIPTION:
! Initialize pft nitrogen flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_nflux_type), intent(inout) :: pnf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pnf%m_leafn_to_litter(beg:end))
    allocate(pnf%m_frootn_to_litter(beg:end))
    allocate(pnf%m_leafn_storage_to_litter(beg:end))
    allocate(pnf%m_frootn_storage_to_litter(beg:end))
    allocate(pnf%m_livestemn_storage_to_litter(beg:end))
    allocate(pnf%m_deadstemn_storage_to_litter(beg:end))
    allocate(pnf%m_livecrootn_storage_to_litter(beg:end))
    allocate(pnf%m_deadcrootn_storage_to_litter(beg:end))
    allocate(pnf%m_leafn_xfer_to_litter(beg:end))
    allocate(pnf%m_frootn_xfer_to_litter(beg:end))
    allocate(pnf%m_livestemn_xfer_to_litter(beg:end))
    allocate(pnf%m_deadstemn_xfer_to_litter(beg:end))
    allocate(pnf%m_livecrootn_xfer_to_litter(beg:end))
    allocate(pnf%m_deadcrootn_xfer_to_litter(beg:end))
    allocate(pnf%m_livestemn_to_litter(beg:end))
    allocate(pnf%m_deadstemn_to_litter(beg:end))
    allocate(pnf%m_livecrootn_to_litter(beg:end))
    allocate(pnf%m_deadcrootn_to_litter(beg:end))
    allocate(pnf%m_retransn_to_litter(beg:end))
    allocate(pnf%hrv_leafn_to_litter(beg:end))             
    allocate(pnf%hrv_frootn_to_litter(beg:end))            
    allocate(pnf%hrv_leafn_storage_to_litter(beg:end))     
    allocate(pnf%hrv_frootn_storage_to_litter(beg:end))    
    allocate(pnf%hrv_livestemn_storage_to_litter(beg:end)) 
    allocate(pnf%hrv_deadstemn_storage_to_litter(beg:end)) 
    allocate(pnf%hrv_livecrootn_storage_to_litter(beg:end))
    allocate(pnf%hrv_deadcrootn_storage_to_litter(beg:end))
    allocate(pnf%hrv_leafn_xfer_to_litter(beg:end))        
    allocate(pnf%hrv_frootn_xfer_to_litter(beg:end))       
    allocate(pnf%hrv_livestemn_xfer_to_litter(beg:end))    
    allocate(pnf%hrv_deadstemn_xfer_to_litter(beg:end))    
    allocate(pnf%hrv_livecrootn_xfer_to_litter(beg:end))   
    allocate(pnf%hrv_deadcrootn_xfer_to_litter(beg:end))   
    allocate(pnf%hrv_livestemn_to_litter(beg:end))         
    allocate(pnf%hrv_deadstemn_to_prod10n(beg:end))        
    allocate(pnf%hrv_deadstemn_to_prod100n(beg:end))       
    allocate(pnf%hrv_livecrootn_to_litter(beg:end))        
    allocate(pnf%hrv_deadcrootn_to_litter(beg:end))        
    allocate(pnf%hrv_retransn_to_litter(beg:end))              
    allocate(pnf%m_leafn_to_fire(beg:end))
    allocate(pnf%m_frootn_to_fire(beg:end))
    allocate(pnf%m_leafn_storage_to_fire(beg:end))
    allocate(pnf%m_frootn_storage_to_fire(beg:end))
    allocate(pnf%m_livestemn_storage_to_fire(beg:end))
    allocate(pnf%m_deadstemn_storage_to_fire(beg:end))
    allocate(pnf%m_livecrootn_storage_to_fire(beg:end))
    allocate(pnf%m_deadcrootn_storage_to_fire(beg:end))
    allocate(pnf%m_leafn_xfer_to_fire(beg:end))
    allocate(pnf%m_frootn_xfer_to_fire(beg:end))
    allocate(pnf%m_livestemn_xfer_to_fire(beg:end))
    allocate(pnf%m_deadstemn_xfer_to_fire(beg:end))
    allocate(pnf%m_livecrootn_xfer_to_fire(beg:end))
    allocate(pnf%m_deadcrootn_xfer_to_fire(beg:end))
    allocate(pnf%m_livestemn_to_fire(beg:end))
    allocate(pnf%m_deadstemn_to_fire(beg:end))
    allocate(pnf%m_deadstemn_to_litter_fire(beg:end))
    allocate(pnf%m_livecrootn_to_fire(beg:end))
    allocate(pnf%m_deadcrootn_to_fire(beg:end))
    allocate(pnf%m_deadcrootn_to_litter_fire(beg:end))
    allocate(pnf%m_retransn_to_fire(beg:end))
    allocate(pnf%leafn_xfer_to_leafn(beg:end))
    allocate(pnf%frootn_xfer_to_frootn(beg:end))
    allocate(pnf%livestemn_xfer_to_livestemn(beg:end))
    allocate(pnf%deadstemn_xfer_to_deadstemn(beg:end))
    allocate(pnf%livecrootn_xfer_to_livecrootn(beg:end))
    allocate(pnf%deadcrootn_xfer_to_deadcrootn(beg:end))
    allocate(pnf%leafn_to_litter(beg:end))
    allocate(pnf%leafn_to_retransn(beg:end))
    allocate(pnf%frootn_to_litter(beg:end))
    allocate(pnf%retransn_to_npool(beg:end))
    allocate(pnf%sminn_to_npool(beg:end))
    allocate(pnf%npool_to_leafn(beg:end))
    allocate(pnf%npool_to_leafn_storage(beg:end))
    allocate(pnf%npool_to_frootn(beg:end))
    allocate(pnf%npool_to_frootn_storage(beg:end))
    allocate(pnf%npool_to_livestemn(beg:end))
    allocate(pnf%npool_to_livestemn_storage(beg:end))
    allocate(pnf%npool_to_deadstemn(beg:end))
    allocate(pnf%npool_to_deadstemn_storage(beg:end))
    allocate(pnf%npool_to_livecrootn(beg:end))
    allocate(pnf%npool_to_livecrootn_storage(beg:end))
    allocate(pnf%npool_to_deadcrootn(beg:end))
    allocate(pnf%npool_to_deadcrootn_storage(beg:end))
    allocate(pnf%leafn_storage_to_xfer(beg:end))
    allocate(pnf%frootn_storage_to_xfer(beg:end))
    allocate(pnf%livestemn_storage_to_xfer(beg:end))
    allocate(pnf%deadstemn_storage_to_xfer(beg:end))
    allocate(pnf%livecrootn_storage_to_xfer(beg:end))
    allocate(pnf%deadcrootn_storage_to_xfer(beg:end))
    allocate(pnf%livestemn_to_deadstemn(beg:end))
    allocate(pnf%livestemn_to_retransn(beg:end))
    allocate(pnf%livecrootn_to_deadcrootn(beg:end))
    allocate(pnf%livecrootn_to_retransn(beg:end))
    allocate(pnf%ndeploy(beg:end))
    allocate(pnf%pft_ninputs(beg:end))
    allocate(pnf%pft_noutputs(beg:end))
    allocate(pnf%wood_harvestn(beg:end))
    allocate(pnf%pft_fire_nloss(beg:end))
#if (defined CROP)
    allocate(pnf%grainn_xfer_to_grainn(beg:end))
    allocate(pnf%livestemn_to_litter(beg:end))
    allocate(pnf%grainn_to_food(beg:end))
    allocate(pnf%npool_to_grainn(beg:end))
    allocate(pnf%npool_to_grainn_storage(beg:end))
    allocate(pnf%grainn_storage_to_xfer(beg:end))
#endif

    pnf%m_leafn_to_litter(beg:end) = nan
    pnf%m_frootn_to_litter(beg:end) = nan
    pnf%m_leafn_storage_to_litter(beg:end) = nan
    pnf%m_frootn_storage_to_litter(beg:end) = nan
    pnf%m_livestemn_storage_to_litter(beg:end) = nan
    pnf%m_deadstemn_storage_to_litter(beg:end) = nan
    pnf%m_livecrootn_storage_to_litter(beg:end) = nan
    pnf%m_deadcrootn_storage_to_litter(beg:end) = nan
    pnf%m_leafn_xfer_to_litter(beg:end) = nan
    pnf%m_frootn_xfer_to_litter(beg:end) = nan
    pnf%m_livestemn_xfer_to_litter(beg:end) = nan
    pnf%m_deadstemn_xfer_to_litter(beg:end) = nan
    pnf%m_livecrootn_xfer_to_litter(beg:end) = nan
    pnf%m_deadcrootn_xfer_to_litter(beg:end) = nan
    pnf%m_livestemn_to_litter(beg:end) = nan
    pnf%m_deadstemn_to_litter(beg:end) = nan
    pnf%m_livecrootn_to_litter(beg:end) = nan
    pnf%m_deadcrootn_to_litter(beg:end) = nan
    pnf%m_retransn_to_litter(beg:end) = nan
    pnf%hrv_leafn_to_litter(beg:end) = nan             
    pnf%hrv_frootn_to_litter(beg:end) = nan            
    pnf%hrv_leafn_storage_to_litter(beg:end) = nan     
    pnf%hrv_frootn_storage_to_litter(beg:end) = nan    
    pnf%hrv_livestemn_storage_to_litter(beg:end) = nan 
    pnf%hrv_deadstemn_storage_to_litter(beg:end) = nan 
    pnf%hrv_livecrootn_storage_to_litter(beg:end) = nan
    pnf%hrv_deadcrootn_storage_to_litter(beg:end) = nan
    pnf%hrv_leafn_xfer_to_litter(beg:end) = nan        
    pnf%hrv_frootn_xfer_to_litter(beg:end) = nan       
    pnf%hrv_livestemn_xfer_to_litter(beg:end) = nan    
    pnf%hrv_deadstemn_xfer_to_litter(beg:end) = nan    
    pnf%hrv_livecrootn_xfer_to_litter(beg:end) = nan   
    pnf%hrv_deadcrootn_xfer_to_litter(beg:end) = nan   
    pnf%hrv_livestemn_to_litter(beg:end) = nan         
    pnf%hrv_deadstemn_to_prod10n(beg:end) = nan        
    pnf%hrv_deadstemn_to_prod100n(beg:end) = nan       
    pnf%hrv_livecrootn_to_litter(beg:end) = nan        
    pnf%hrv_deadcrootn_to_litter(beg:end) = nan        
    pnf%hrv_retransn_to_litter(beg:end) = nan           
    pnf%m_leafn_to_fire(beg:end) = nan
    pnf%m_frootn_to_fire(beg:end) = nan
    pnf%m_leafn_storage_to_fire(beg:end) = nan
    pnf%m_frootn_storage_to_fire(beg:end) = nan
    pnf%m_livestemn_storage_to_fire(beg:end) = nan
    pnf%m_deadstemn_storage_to_fire(beg:end) = nan
    pnf%m_livecrootn_storage_to_fire(beg:end) = nan
    pnf%m_deadcrootn_storage_to_fire(beg:end) = nan
    pnf%m_leafn_xfer_to_fire(beg:end) = nan
    pnf%m_frootn_xfer_to_fire(beg:end) = nan
    pnf%m_livestemn_xfer_to_fire(beg:end) = nan
    pnf%m_deadstemn_xfer_to_fire(beg:end) = nan
    pnf%m_livecrootn_xfer_to_fire(beg:end) = nan
    pnf%m_deadcrootn_xfer_to_fire(beg:end) = nan
    pnf%m_livestemn_to_fire(beg:end) = nan
    pnf%m_deadstemn_to_fire(beg:end) = nan
    pnf%m_deadstemn_to_litter_fire(beg:end) = nan
    pnf%m_livecrootn_to_fire(beg:end) = nan
    pnf%m_deadcrootn_to_fire(beg:end) = nan
    pnf%m_deadcrootn_to_litter_fire(beg:end) = nan
    pnf%m_retransn_to_fire(beg:end) = nan
    pnf%leafn_xfer_to_leafn(beg:end) = nan
    pnf%frootn_xfer_to_frootn(beg:end) = nan
    pnf%livestemn_xfer_to_livestemn(beg:end) = nan
    pnf%deadstemn_xfer_to_deadstemn(beg:end) = nan
    pnf%livecrootn_xfer_to_livecrootn(beg:end) = nan
    pnf%deadcrootn_xfer_to_deadcrootn(beg:end) = nan
    pnf%leafn_to_litter(beg:end) = nan
    pnf%leafn_to_retransn(beg:end) = nan
    pnf%frootn_to_litter(beg:end) = nan
    pnf%retransn_to_npool(beg:end) = nan
    pnf%sminn_to_npool(beg:end) = nan
    pnf%npool_to_leafn(beg:end) = nan
    pnf%npool_to_leafn_storage(beg:end) = nan
    pnf%npool_to_frootn(beg:end) = nan
    pnf%npool_to_frootn_storage(beg:end) = nan
    pnf%npool_to_livestemn(beg:end) = nan
    pnf%npool_to_livestemn_storage(beg:end) = nan
    pnf%npool_to_deadstemn(beg:end) = nan
    pnf%npool_to_deadstemn_storage(beg:end) = nan
    pnf%npool_to_livecrootn(beg:end) = nan
    pnf%npool_to_livecrootn_storage(beg:end) = nan
    pnf%npool_to_deadcrootn(beg:end) = nan
    pnf%npool_to_deadcrootn_storage(beg:end) = nan
    pnf%leafn_storage_to_xfer(beg:end) = nan
    pnf%frootn_storage_to_xfer(beg:end) = nan
    pnf%livestemn_storage_to_xfer(beg:end) = nan
    pnf%deadstemn_storage_to_xfer(beg:end) = nan
    pnf%livecrootn_storage_to_xfer(beg:end) = nan
    pnf%deadcrootn_storage_to_xfer(beg:end) = nan
    pnf%livestemn_to_deadstemn(beg:end) = nan
    pnf%livestemn_to_retransn(beg:end) = nan
    pnf%livecrootn_to_deadcrootn(beg:end) = nan
    pnf%livecrootn_to_retransn(beg:end) = nan
    pnf%ndeploy(beg:end) = nan
    pnf%pft_ninputs(beg:end) = nan
    pnf%pft_noutputs(beg:end) = nan
    pnf%wood_harvestn(beg:end) = nan
    pnf%pft_fire_nloss(beg:end) = nan
#if (defined CROP)
    pnf%grainn_xfer_to_grainn(beg:end) = nan
    pnf%livestemn_to_litter(beg:end) = nan
    pnf%grainn_to_food(beg:end) = nan
    pnf%npool_to_grainn(beg:end) = nan
    pnf%npool_to_grainn_storage(beg:end) = nan
    pnf%grainn_storage_to_xfer(beg:end) = nan
#endif

  end subroutine init_pft_nflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_vflux_type
!
! !INTERFACE:
  subroutine init_pft_vflux_type(beg, end, pvf)
!
! !DESCRIPTION:
! Initialize pft VOC flux variables
!
    use clm_varcon, only : spval
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_vflux_type), intent(inout) :: pvf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
! (heald, 08/06)
!
!EOP
!------------------------------------------------------------------------

    allocate(pvf%vocflx_tot(beg:end))
    allocate(pvf%vocflx(beg:end,1:nvoc))
    allocate(pvf%vocflx_1(beg:end))
    allocate(pvf%vocflx_2(beg:end))
    allocate(pvf%vocflx_3(beg:end))
    allocate(pvf%vocflx_4(beg:end))
    allocate(pvf%vocflx_5(beg:end))
    allocate(pvf%Eopt_out(beg:end))
    allocate(pvf%topt_out(beg:end))
    allocate(pvf%alpha_out(beg:end))
    allocate(pvf%cp_out(beg:end))
    allocate(pvf%para_out(beg:end))
    allocate(pvf%par24a_out(beg:end))
    allocate(pvf%par240a_out(beg:end))
    allocate(pvf%paru_out(beg:end))
    allocate(pvf%par24u_out(beg:end))
    allocate(pvf%par240u_out(beg:end))
    allocate(pvf%gamma_out(beg:end))
    allocate(pvf%gammaL_out(beg:end))
    allocate(pvf%gammaT_out(beg:end))
    allocate(pvf%gammaP_out(beg:end))
    allocate(pvf%gammaA_out(beg:end))
    allocate(pvf%gammaS_out(beg:end))

    pvf%vocflx_tot(beg:end) = spval
    pvf%vocflx(beg:end,1:nvoc) = spval
    pvf%vocflx_1(beg:end) = spval
    pvf%vocflx_2(beg:end) = spval
    pvf%vocflx_3(beg:end) = spval
    pvf%vocflx_4(beg:end) = spval
    pvf%vocflx_5(beg:end) = spval
    pvf%Eopt_out(beg:end) = nan
    pvf%topt_out(beg:end) = nan
    pvf%alpha_out(beg:end) = nan
    pvf%cp_out(beg:end) = nan
    pvf%para_out(beg:end) = nan
    pvf%par24a_out(beg:end) = nan
    pvf%par240a_out(beg:end) = nan
    pvf%paru_out(beg:end) = nan
    pvf%par24u_out(beg:end) = nan
    pvf%par240u_out(beg:end) = nan
    pvf%gamma_out(beg:end) = nan
    pvf%gammaL_out(beg:end) = nan
    pvf%gammaT_out(beg:end) = nan
    pvf%gammaP_out(beg:end) = nan
    pvf%gammaA_out(beg:end) = nan
    pvf%gammaS_out(beg:end) = nan

  end subroutine init_pft_vflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_dflux_type
!
! !INTERFACE:
  subroutine init_pft_dflux_type(beg, end, pdf)
!
! !DESCRIPTION:
! Initialize pft dust flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_dflux_type), intent(inout):: pdf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(pdf%flx_mss_vrt_dst(beg:end,1:ndst))
    allocate(pdf%flx_mss_vrt_dst_tot(beg:end))
    allocate(pdf%vlc_trb(beg:end,1:ndst))
    allocate(pdf%vlc_trb_1(beg:end))
    allocate(pdf%vlc_trb_2(beg:end))
    allocate(pdf%vlc_trb_3(beg:end))
    allocate(pdf%vlc_trb_4(beg:end))

    pdf%flx_mss_vrt_dst(beg:end,1:ndst) = nan
    pdf%flx_mss_vrt_dst_tot(beg:end) = nan
    pdf%vlc_trb(beg:end,1:ndst) = nan
    pdf%vlc_trb_1(beg:end) = nan
    pdf%vlc_trb_2(beg:end) = nan
    pdf%vlc_trb_3(beg:end) = nan
    pdf%vlc_trb_4(beg:end) = nan

  end subroutine init_pft_dflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_pft_depvd_type
!
! !INTERFACE:
  subroutine init_pft_depvd_type(beg, end, pdd)

!    use seq_drydep_mod, only:  n_drydep
!
! !DESCRIPTION:
! Initialize pft dep velocity variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (pft_depvd_type), intent(inout):: pdd
    integer :: i
!
! !REVISION HISTORY:
! Created by James Sulzman 541-929-6183
!
!EOP
!------------------------------------------------------------------------

!    if ( n_drydep > 0 )then
!       allocate(pdd%drydepvel(beg:end,n_drydep))
!       pdd%drydepvel = nan
!    end if

  end subroutine init_pft_depvd_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_pstate_type
!
! !INTERFACE:
  subroutine init_column_pstate_type(beg, end, cps)
!
! !DESCRIPTION:
! Initialize column physical state variables
!
! !USES:
    use clm_varcon, only : spval
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_pstate_type), intent(inout):: cps
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(cps%snl(beg:end))      !* cannot be averaged up
    allocate(cps%isoicol(beg:end))  !* cannot be averaged up
    allocate(cps%bsw(beg:end,nlevgrnd))
    allocate(cps%watsat(beg:end,nlevgrnd))
    allocate(cps%watfc(beg:end,nlevgrnd))
    allocate(cps%watdry(beg:end,nlevgrnd)) 
    allocate(cps%watopt(beg:end,nlevgrnd)) 
    allocate(cps%hksat(beg:end,nlevgrnd))
    allocate(cps%sucsat(beg:end,nlevgrnd))
    allocate(cps%csol(beg:end,nlevgrnd))
    allocate(cps%tkmg(beg:end,nlevgrnd))
    allocate(cps%tkdry(beg:end,nlevgrnd))
    allocate(cps%tksatu(beg:end,nlevgrnd))
    allocate(cps%smpmin(beg:end))
    allocate(cps%hkdepth(beg:end))
    allocate(cps%wtfact(beg:end))
    allocate(cps%fracice(beg:end,nlevgrnd))
    allocate(cps%gwc_thr(beg:end))
    allocate(cps%mss_frc_cly_vld(beg:end))
    allocate(cps%mbl_bsn_fct(beg:end))
    allocate(cps%do_capsnow(beg:end))
    allocate(cps%snowdp(beg:end))
    allocate(cps%frac_sno (beg:end))
    allocate(cps%zi(beg:end,-nlevsno+0:nlevgrnd))
    allocate(cps%dz(beg:end,-nlevsno+1:nlevgrnd))
    allocate(cps%z (beg:end,-nlevsno+1:nlevgrnd))
    allocate(cps%frac_iceold(beg:end,-nlevsno+1:nlevgrnd))
    allocate(cps%imelt(beg:end,-nlevsno+1:nlevgrnd))
    allocate(cps%eff_porosity(beg:end,nlevgrnd))
    allocate(cps%emg(beg:end))
    allocate(cps%z0mg(beg:end))
    allocate(cps%z0hg(beg:end))
    allocate(cps%z0qg(beg:end))
    allocate(cps%htvp(beg:end))
    allocate(cps%beta(beg:end))
    allocate(cps%zii(beg:end))
    allocate(cps%albgrd(beg:end,numrad))
    allocate(cps%albgri(beg:end,numrad))
    allocate(cps%rootr_column(beg:end,nlevgrnd))
    allocate(cps%rootfr_road_perv(beg:end,nlevgrnd))
    allocate(cps%rootr_road_perv(beg:end,nlevgrnd))
    allocate(cps%wf(beg:end))
!   allocate(cps%xirrig(beg:end))
    allocate(cps%max_dayl(beg:end))
    allocate(cps%bsw2(beg:end,nlevgrnd))
    allocate(cps%psisat(beg:end,nlevgrnd))
    allocate(cps%vwcsat(beg:end,nlevgrnd))
    allocate(cps%soilpsi(beg:end,nlevgrnd))
    allocate(cps%decl(beg:end))
    allocate(cps%coszen(beg:end))
    allocate(cps%fpi(beg:end))
    allocate(cps%fpg(beg:end))
    allocate(cps%annsum_counter(beg:end))
    allocate(cps%cannsum_npp(beg:end))
    allocate(cps%cannavg_t2m(beg:end))
    allocate(cps%me(beg:end))
    allocate(cps%fire_prob(beg:end))
    allocate(cps%mean_fire_prob(beg:end))
    allocate(cps%fireseasonl(beg:end))
    allocate(cps%farea_burned(beg:end))
    allocate(cps%ann_farea_burned(beg:end))
    allocate(cps%albsnd_hst(beg:end,numrad))
    allocate(cps%albsni_hst(beg:end,numrad))
    allocate(cps%albsod(beg:end,numrad))
    allocate(cps%albsoi(beg:end,numrad))
    allocate(cps%flx_absdv(beg:end,-nlevsno+1:1))
    allocate(cps%flx_absdn(beg:end,-nlevsno+1:1))
    allocate(cps%flx_absiv(beg:end,-nlevsno+1:1))
    allocate(cps%flx_absin(beg:end,-nlevsno+1:1))
    allocate(cps%snw_rds(beg:end,-nlevsno+1:0))
    allocate(cps%snw_rds_top(beg:end))
    allocate(cps%sno_liq_top(beg:end))
    allocate(cps%mss_bcpho(beg:end,-nlevsno+1:0))
    allocate(cps%mss_bcphi(beg:end,-nlevsno+1:0))
    allocate(cps%mss_bctot(beg:end,-nlevsno+1:0))
    allocate(cps%mss_bc_col(beg:end))
    allocate(cps%mss_bc_top(beg:end))
    allocate(cps%mss_ocpho(beg:end,-nlevsno+1:0))
    allocate(cps%mss_ocphi(beg:end,-nlevsno+1:0))
    allocate(cps%mss_octot(beg:end,-nlevsno+1:0))
    allocate(cps%mss_oc_col(beg:end))
    allocate(cps%mss_oc_top(beg:end))
    allocate(cps%mss_dst1(beg:end,-nlevsno+1:0))
    allocate(cps%mss_dst2(beg:end,-nlevsno+1:0))
    allocate(cps%mss_dst3(beg:end,-nlevsno+1:0))
    allocate(cps%mss_dst4(beg:end,-nlevsno+1:0))
    allocate(cps%mss_dsttot(beg:end,-nlevsno+1:0))
    allocate(cps%mss_dst_col(beg:end))
    allocate(cps%mss_dst_top(beg:end))
    allocate(cps%h2osno_top(beg:end))
    allocate(cps%mss_cnc_bcphi(beg:end,-nlevsno+1:0))
    allocate(cps%mss_cnc_bcpho(beg:end,-nlevsno+1:0))
    allocate(cps%mss_cnc_ocphi(beg:end,-nlevsno+1:0))
    allocate(cps%mss_cnc_ocpho(beg:end,-nlevsno+1:0))
    allocate(cps%mss_cnc_dst1(beg:end,-nlevsno+1:0))
    allocate(cps%mss_cnc_dst2(beg:end,-nlevsno+1:0))
    allocate(cps%mss_cnc_dst3(beg:end,-nlevsno+1:0))
    allocate(cps%mss_cnc_dst4(beg:end,-nlevsno+1:0))
    allocate(cps%albgrd_pur(beg:end,numrad))
    allocate(cps%albgri_pur(beg:end,numrad))
    allocate(cps%albgrd_bc(beg:end,numrad))
    allocate(cps%albgri_bc(beg:end,numrad))
    allocate(cps%albgrd_oc(beg:end,numrad))
    allocate(cps%albgri_oc(beg:end,numrad))
    allocate(cps%albgrd_dst(beg:end,numrad))
    allocate(cps%albgri_dst(beg:end,numrad))
    allocate(cps%dTdz_top(beg:end))
    allocate(cps%snot_top(beg:end))

    cps%isoicol(beg:end) = bigint
    cps%bsw(beg:end,1:nlevgrnd) = nan
    cps%watsat(beg:end,1:nlevgrnd) = nan
    cps%watfc(beg:end,1:nlevgrnd) = nan
    cps%watdry(beg:end,1:nlevgrnd) = nan
    cps%watopt(beg:end,1:nlevgrnd) = nan
    cps%hksat(beg:end,1:nlevgrnd) = nan
    cps%sucsat(beg:end,1:nlevgrnd) = nan
    cps%csol(beg:end,1:nlevgrnd) = nan
    cps%tkmg(beg:end,1:nlevgrnd) = nan
    cps%tkdry(beg:end,1:nlevgrnd) = nan
    cps%tksatu(beg:end,1:nlevgrnd) = nan
    cps%smpmin(beg:end) = nan
    cps%hkdepth(beg:end) = nan
    cps%wtfact(beg:end) = nan
    cps%fracice(beg:end,1:nlevgrnd) = nan
    cps%gwc_thr(beg:end) = nan
    cps%mss_frc_cly_vld(beg:end) = nan
    cps%mbl_bsn_fct(beg:end) = nan
    cps%do_capsnow (beg:end)= .false.
    cps%snowdp(beg:end) = nan
    cps%frac_sno(beg:end) = nan
    cps%zi(beg:end,-nlevsno+0:nlevgrnd) = nan
    cps%dz(beg:end,-nlevsno+1:nlevgrnd) = nan
    cps%z (beg:end,-nlevsno+1:nlevgrnd) = nan
    cps%frac_iceold(beg:end,-nlevsno+1:nlevgrnd) = spval
    cps%imelt(beg:end,-nlevsno+1:nlevgrnd) = bigint
    cps%eff_porosity(beg:end,1:nlevgrnd) = spval
    cps%emg(beg:end) = nan
    cps%z0mg(beg:end) = nan
    cps%z0hg(beg:end) = nan
    cps%z0qg(beg:end) = nan
    cps%htvp(beg:end) = nan
    cps%beta(beg:end) = nan
    cps%zii(beg:end) = nan
    cps%albgrd(beg:end,:numrad) = nan
    cps%albgri(beg:end,:numrad) = nan
    cps%rootr_column(beg:end,1:nlevgrnd) = spval
    cps%rootfr_road_perv(beg:end,1:nlevurb) = nan
    cps%rootr_road_perv(beg:end,1:nlevurb) = nan
    cps%wf(beg:end) = nan
!   cps%xirrig(beg:end) = 0._r8
    cps%bsw2(beg:end,1:nlevgrnd) = nan
    cps%psisat(beg:end,1:nlevgrnd) = nan
    cps%vwcsat(beg:end,1:nlevgrnd) = nan
    cps%soilpsi(beg:end,1:nlevgrnd) = spval
    cps%decl(beg:end) = nan
    cps%coszen(beg:end) = nan
    cps%fpi(beg:end) = nan
    cps%fpg(beg:end) = nan
    cps%annsum_counter(beg:end) = nan
    cps%cannsum_npp(beg:end) = nan
    cps%cannavg_t2m(beg:end) = nan
    cps%me(beg:end) = nan
    cps%fire_prob(beg:end) = nan
    cps%mean_fire_prob(beg:end) = nan
    cps%fireseasonl(beg:end) = nan
    cps%farea_burned(beg:end) = nan
    cps%ann_farea_burned(beg:end) = nan
    cps%albsnd_hst(beg:end,:numrad) = spval
    cps%albsni_hst(beg:end,:numrad) = spval
    cps%albsod(beg:end,:numrad) = nan
    cps%albsoi(beg:end,:numrad) = nan
    cps%flx_absdv(beg:end,-nlevsno+1:1) = spval
    cps%flx_absdn(beg:end,-nlevsno+1:1) = spval
    cps%flx_absiv(beg:end,-nlevsno+1:1) = spval
    cps%flx_absin(beg:end,-nlevsno+1:1) = spval
    cps%snw_rds(beg:end,-nlevsno+1:0) = nan
    cps%snw_rds_top(beg:end) = nan
    cps%sno_liq_top(beg:end) = nan
    cps%mss_bcpho(beg:end,-nlevsno+1:0) = nan
    cps%mss_bcphi(beg:end,-nlevsno+1:0) = nan
    cps%mss_bctot(beg:end,-nlevsno+1:0) = nan
    cps%mss_bc_col(beg:end) = nan
    cps%mss_bc_top(beg:end) = nan
    cps%mss_ocpho(beg:end,-nlevsno+1:0) = nan
    cps%mss_ocphi(beg:end,-nlevsno+1:0) = nan
    cps%mss_octot(beg:end,-nlevsno+1:0) = nan
    cps%mss_oc_col(beg:end) = nan
    cps%mss_oc_top(beg:end) = nan
    cps%mss_dst1(beg:end,-nlevsno+1:0) = nan
    cps%mss_dst2(beg:end,-nlevsno+1:0) = nan
    cps%mss_dst3(beg:end,-nlevsno+1:0) = nan
    cps%mss_dst4(beg:end,-nlevsno+1:0) = nan
    cps%mss_dsttot(beg:end,-nlevsno+1:0) = nan
    cps%mss_dst_col(beg:end) = nan
    cps%mss_dst_top(beg:end) = nan
    cps%h2osno_top(beg:end) = nan
    cps%mss_cnc_bcphi(beg:end,-nlevsno+1:0) = nan
    cps%mss_cnc_bcpho(beg:end,-nlevsno+1:0) = nan
    cps%mss_cnc_ocphi(beg:end,-nlevsno+1:0) = nan
    cps%mss_cnc_ocpho(beg:end,-nlevsno+1:0) = nan
    cps%mss_cnc_dst1(beg:end,-nlevsno+1:0) = nan
    cps%mss_cnc_dst2(beg:end,-nlevsno+1:0) = nan
    cps%mss_cnc_dst3(beg:end,-nlevsno+1:0) = nan
    cps%mss_cnc_dst4(beg:end,-nlevsno+1:0) = nan
    cps%albgrd_pur(beg:end,:numrad) = nan
    cps%albgri_pur(beg:end,:numrad) = nan
    cps%albgrd_bc(beg:end,:numrad) = nan
    cps%albgri_bc(beg:end,:numrad) = nan
    cps%albgrd_oc(beg:end,:numrad) = nan
    cps%albgri_oc(beg:end,:numrad) = nan 
    cps%albgrd_dst(beg:end,:numrad) = nan
    cps%albgri_dst(beg:end,:numrad) = nan
    cps%dTdz_top(beg:end) = nan
    cps%snot_top(beg:end) = nan
  end subroutine init_column_pstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_estate_type
!
! !INTERFACE:
  subroutine init_column_estate_type(beg, end, ces)
!
! !DESCRIPTION:
! Initialize column energy state variables
!
! !USES:
    use clm_varcon, only : spval
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_estate_type), intent(inout):: ces
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------
    allocate(ces%t_grnd(beg:end))
    allocate(ces%t_grnd_u(beg:end))
    allocate(ces%t_grnd_r(beg:end))
    allocate(ces%dt_grnd(beg:end))
    allocate(ces%t_soisno(beg:end,-nlevsno+1:nlevgrnd))
    allocate(ces%t_soi_10cm(beg:end))
    allocate(ces%t_lake(beg:end,1:nlevlak))
    allocate(ces%tssbef(beg:end,-nlevsno+1:nlevgrnd))
    allocate(ces%thv(beg:end))
    allocate(ces%hc_soi(beg:end))
    allocate(ces%hc_soisno(beg:end))

    ces%t_grnd(beg:end)    = nan
    ces%t_grnd_u(beg:end)  = nan
    ces%t_grnd_r(beg:end)  = nan
    ces%dt_grnd(beg:end)   = nan
    ces%t_soisno(beg:end,-nlevsno+1:nlevgrnd) = spval
    ces%t_soi_10cm(beg:end) = spval
    ces%t_lake(beg:end,1:nlevlak)            = nan
    ces%tssbef(beg:end,-nlevsno+1:nlevgrnd)   = nan
    ces%thv(beg:end)       = nan
    ces%hc_soi(beg:end)    = nan
    ces%hc_soisno(beg:end) = nan

  end subroutine init_column_estate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_wstate_type
!
! !INTERFACE:
  subroutine init_column_wstate_type(beg, end, cws)
!
! !DESCRIPTION:
! Initialize column water state variables
!
! !USES:
    use clm_varcon, only : spval
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_wstate_type), intent(inout):: cws !column water state
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(cws%h2osno(beg:end))
    allocate(cws%h2osoi_liq(beg:end,-nlevsno+1:nlevgrnd))
    allocate(cws%h2osoi_ice(beg:end,-nlevsno+1:nlevgrnd))
    allocate(cws%h2osoi_liqice_10cm(beg:end))
    allocate(cws%h2osoi_vol(beg:end,1:nlevgrnd))
    allocate(cws%h2osno_old(beg:end))
    allocate(cws%qg(beg:end))
    allocate(cws%dqgdT(beg:end))
    allocate(cws%snowice(beg:end))
    allocate(cws%snowliq(beg:end))
    allocate(cws%soilalpha(beg:end))
    allocate(cws%soilbeta(beg:end))
    allocate(cws%soilalpha_u(beg:end))
    allocate(cws%zwt(beg:end))
    allocate(cws%fcov(beg:end))
    allocate(cws%fsat(beg:end))
    allocate(cws%wa(beg:end))
    allocate(cws%wt(beg:end))
    allocate(cws%qcharge(beg:end))
    allocate(cws%smp_l(beg:end,1:nlevgrnd))
    allocate(cws%hk_l(beg:end,1:nlevgrnd))

    cws%h2osno(beg:end) = nan
    cws%h2osoi_liq(beg:end,-nlevsno+1:nlevgrnd)= spval
    cws%h2osoi_ice(beg:end,-nlevsno+1:nlevgrnd) = spval
    cws%h2osoi_liqice_10cm(beg:end) = spval
    cws%h2osoi_vol(beg:end,1:nlevgrnd) = spval
    cws%h2osno_old(beg:end) = nan
    cws%qg(beg:end) = nan
    cws%dqgdT(beg:end) = nan
    cws%snowice(beg:end) = nan
    cws%snowliq(beg:end) = nan
    cws%soilalpha(beg:end) = nan
    cws%soilbeta(beg:end) = nan
    cws%soilalpha_u(beg:end) = nan
    cws%zwt(beg:end) = nan
    cws%fcov(beg:end) = nan
    cws%fsat(beg:end) = nan
    cws%wa(beg:end) = nan
    cws%wt(beg:end) = nan
    cws%qcharge(beg:end) = nan
    cws%smp_l(beg:end,1:nlevgrnd) = spval
    cws%hk_l(beg:end,1:nlevgrnd) = spval

  end subroutine init_column_wstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_cstate_type
!
! !INTERFACE:
  subroutine init_column_cstate_type(beg, end, ccs)
!
! !DESCRIPTION:
! Initialize column carbon state variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_cstate_type), intent(inout):: ccs
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(ccs%soilc(beg:end))
    allocate(ccs%cwdc(beg:end))
    allocate(ccs%litr1c(beg:end))
    allocate(ccs%litr2c(beg:end))
    allocate(ccs%litr3c(beg:end))
    allocate(ccs%soil1c(beg:end))
    allocate(ccs%soil2c(beg:end))
    allocate(ccs%soil3c(beg:end))
    allocate(ccs%soil4c(beg:end))
    allocate(ccs%seedc(beg:end))
    allocate(ccs%col_ctrunc(beg:end))
    allocate(ccs%prod10c(beg:end))
    allocate(ccs%prod100c(beg:end))
    allocate(ccs%totprodc(beg:end))
    allocate(ccs%totlitc(beg:end))
    allocate(ccs%totsomc(beg:end))
    allocate(ccs%totecosysc(beg:end))
    allocate(ccs%totcolc(beg:end))

    ccs%soilc(beg:end) = nan
    ccs%cwdc(beg:end) = nan
    ccs%litr1c(beg:end) = nan
    ccs%litr2c(beg:end) = nan
    ccs%litr3c(beg:end) = nan
    ccs%soil1c(beg:end) = nan
    ccs%soil2c(beg:end) = nan
    ccs%soil3c(beg:end) = nan
    ccs%soil4c(beg:end) = nan
    ccs%seedc(beg:end) = nan
    ccs%col_ctrunc(beg:end) = nan
    ccs%prod10c(beg:end) = nan
    ccs%prod100c(beg:end) = nan
    ccs%totprodc(beg:end) = nan
    ccs%totlitc(beg:end) = nan
    ccs%totsomc(beg:end) = nan
    ccs%totecosysc(beg:end) = nan
    ccs%totcolc(beg:end) = nan

  end subroutine init_column_cstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_nstate_type
!
! !INTERFACE:
  subroutine init_column_nstate_type(beg, end, cns)
!
! !DESCRIPTION:
! Initialize column nitrogen state variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_nstate_type), intent(inout):: cns
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

    allocate(cns%cwdn(beg:end))
    allocate(cns%litr1n(beg:end))
    allocate(cns%litr2n(beg:end))
    allocate(cns%litr3n(beg:end))
    allocate(cns%soil1n(beg:end))
    allocate(cns%soil2n(beg:end))
    allocate(cns%soil3n(beg:end))
    allocate(cns%soil4n(beg:end))
    allocate(cns%sminn(beg:end))
    allocate(cns%col_ntrunc(beg:end))
    allocate(cns%seedn(beg:end))
    allocate(cns%prod10n(beg:end))
    allocate(cns%prod100n(beg:end))
    allocate(cns%totprodn(beg:end))
    allocate(cns%totlitn(beg:end))
    allocate(cns%totsomn(beg:end))
    allocate(cns%totecosysn(beg:end))
    allocate(cns%totcoln(beg:end))

    cns%cwdn(beg:end) = nan
    cns%litr1n(beg:end) = nan
    cns%litr2n(beg:end) = nan
    cns%litr3n(beg:end) = nan
    cns%soil1n(beg:end) = nan
    cns%soil2n(beg:end) = nan
    cns%soil3n(beg:end) = nan
    cns%soil4n(beg:end) = nan
    cns%sminn(beg:end) = nan
    cns%col_ntrunc(beg:end) = nan
    cns%seedn(beg:end) = nan
    cns%prod10n(beg:end) = nan
    cns%prod100n(beg:end) = nan
    cns%totprodn(beg:end) = nan
    cns%totlitn(beg:end) = nan
    cns%totsomn(beg:end) = nan
    cns%totecosysn(beg:end) = nan
    cns%totcoln(beg:end) = nan

  end subroutine init_column_nstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_eflux_type
!
! !INTERFACE:
  subroutine init_column_eflux_type(beg, end, cef)
!
! !DESCRIPTION:
! Initialize column energy flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_eflux_type), intent(inout):: cef
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(cef%eflx_snomelt(beg:end))
    allocate(cef%eflx_snomelt_u(beg:end))
    allocate(cef%eflx_snomelt_r(beg:end))
    allocate(cef%eflx_impsoil(beg:end))
    allocate(cef%eflx_fgr12(beg:end))
    allocate(cef%eflx_building_heat(beg:end))
    allocate(cef%eflx_urban_ac(beg:end))
    allocate(cef%eflx_urban_heat(beg:end))

    cef%eflx_snomelt(beg:end)       = nan
    cef%eflx_snomelt_u(beg:end)       = nan
    cef%eflx_snomelt_r(beg:end)       = nan
    cef%eflx_impsoil(beg:end)       = nan
    cef%eflx_fgr12(beg:end)         = nan
    cef%eflx_building_heat(beg:end) = nan
    cef%eflx_urban_ac(beg:end) = nan
    cef%eflx_urban_heat(beg:end) = nan

  end subroutine init_column_eflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_wflux_type
!
! !INTERFACE:
  subroutine init_column_wflux_type(beg, end, cwf)
!
! !DESCRIPTION:
! Initialize column water flux variables
!
! !USES:
    use clm_varcon, only : spval
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_wflux_type), intent(inout):: cwf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(cwf%qflx_infl(beg:end))
    allocate(cwf%qflx_surf(beg:end))
    allocate(cwf%qflx_drain(beg:end))
    allocate(cwf%qflx_top_soil(beg:end))
    allocate(cwf%qflx_snomelt(beg:end))
    allocate(cwf%qflx_qrgwl(beg:end))
    allocate(cwf%qflx_runoff(beg:end))
    allocate(cwf%qflx_runoff_u(beg:end))
    allocate(cwf%qflx_runoff_r(beg:end))
    allocate(cwf%qmelt(beg:end))
    allocate(cwf%h2ocan_loss(beg:end))
    allocate(cwf%qflx_rsub_sat(beg:end))
    allocate(cwf%flx_bc_dep_dry(beg:end))
    allocate(cwf%flx_bc_dep_wet(beg:end))
    allocate(cwf%flx_bc_dep_pho(beg:end))
    allocate(cwf%flx_bc_dep_phi(beg:end))
    allocate(cwf%flx_bc_dep(beg:end))
    allocate(cwf%flx_oc_dep_dry(beg:end))
    allocate(cwf%flx_oc_dep_wet(beg:end))
    allocate(cwf%flx_oc_dep_pho(beg:end))
    allocate(cwf%flx_oc_dep_phi(beg:end))
    allocate(cwf%flx_oc_dep(beg:end))
    allocate(cwf%flx_dst_dep_dry1(beg:end))
    allocate(cwf%flx_dst_dep_wet1(beg:end))
    allocate(cwf%flx_dst_dep_dry2(beg:end))
    allocate(cwf%flx_dst_dep_wet2(beg:end))
    allocate(cwf%flx_dst_dep_dry3(beg:end))
    allocate(cwf%flx_dst_dep_wet3(beg:end))
    allocate(cwf%flx_dst_dep_dry4(beg:end))
    allocate(cwf%flx_dst_dep_wet4(beg:end))
    allocate(cwf%flx_dst_dep(beg:end))
    allocate(cwf%qflx_snofrz_lyr(beg:end,-nlevsno+1:0))

    cwf%qflx_infl(beg:end) = nan
    cwf%qflx_surf(beg:end) = nan
    cwf%qflx_drain(beg:end) = nan
    cwf%qflx_top_soil(beg:end) = nan
    cwf%qflx_snomelt(beg:end) = nan
    cwf%qflx_qrgwl(beg:end) = nan
    cwf%qflx_runoff(beg:end) = nan
    cwf%qflx_runoff_u(beg:end) = nan
    cwf%qflx_runoff_r(beg:end) = nan
    cwf%qmelt(beg:end) = nan
    cwf%h2ocan_loss(beg:end) = nan
    cwf%qflx_rsub_sat(beg:end) = nan
    cwf%flx_bc_dep_dry(beg:end) = nan
    cwf%flx_bc_dep_wet(beg:end) = nan
    cwf%flx_bc_dep_pho(beg:end) = nan
    cwf%flx_bc_dep_phi(beg:end) = nan
    cwf%flx_bc_dep(beg:end) = nan
    cwf%flx_oc_dep_dry(beg:end) = nan
    cwf%flx_oc_dep_wet(beg:end) = nan
    cwf%flx_oc_dep_pho(beg:end) = nan
    cwf%flx_oc_dep_phi(beg:end) = nan
    cwf%flx_oc_dep(beg:end) = nan
    cwf%flx_dst_dep_dry1(beg:end) = nan
    cwf%flx_dst_dep_wet1(beg:end) = nan
    cwf%flx_dst_dep_dry2(beg:end) = nan
    cwf%flx_dst_dep_wet2(beg:end) = nan
    cwf%flx_dst_dep_dry3(beg:end) = nan
    cwf%flx_dst_dep_wet3(beg:end) = nan
    cwf%flx_dst_dep_dry4(beg:end) = nan
    cwf%flx_dst_dep_wet4(beg:end) = nan
    cwf%flx_dst_dep(beg:end) = nan
    cwf%qflx_snofrz_lyr(beg:end,-nlevsno+1:0) = spval

  end subroutine init_column_wflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_cflux_type
!
! !INTERFACE:
  subroutine init_column_cflux_type(beg, end, ccf)
!
! !DESCRIPTION:
! Initialize column carbon flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_cflux_type), intent(inout):: ccf
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

    allocate(ccf%m_leafc_to_litr1c(beg:end))
    allocate(ccf%m_leafc_to_litr2c(beg:end))
    allocate(ccf%m_leafc_to_litr3c(beg:end))
    allocate(ccf%m_frootc_to_litr1c(beg:end))
    allocate(ccf%m_frootc_to_litr2c(beg:end))
    allocate(ccf%m_frootc_to_litr3c(beg:end))
    allocate(ccf%m_leafc_storage_to_litr1c(beg:end))
    allocate(ccf%m_frootc_storage_to_litr1c(beg:end))
    allocate(ccf%m_livestemc_storage_to_litr1c(beg:end))
    allocate(ccf%m_deadstemc_storage_to_litr1c(beg:end))
    allocate(ccf%m_livecrootc_storage_to_litr1c(beg:end))
    allocate(ccf%m_deadcrootc_storage_to_litr1c(beg:end))
    allocate(ccf%m_leafc_xfer_to_litr1c(beg:end))
    allocate(ccf%m_frootc_xfer_to_litr1c(beg:end))
    allocate(ccf%m_livestemc_xfer_to_litr1c(beg:end))
    allocate(ccf%m_deadstemc_xfer_to_litr1c(beg:end))
    allocate(ccf%m_livecrootc_xfer_to_litr1c(beg:end))
    allocate(ccf%m_deadcrootc_xfer_to_litr1c(beg:end))
    allocate(ccf%m_livestemc_to_cwdc(beg:end))
    allocate(ccf%m_deadstemc_to_cwdc(beg:end))
    allocate(ccf%m_livecrootc_to_cwdc(beg:end))
    allocate(ccf%m_deadcrootc_to_cwdc(beg:end))
    allocate(ccf%m_gresp_storage_to_litr1c(beg:end))
    allocate(ccf%m_gresp_xfer_to_litr1c(beg:end))
    allocate(ccf%m_deadstemc_to_cwdc_fire(beg:end))
    allocate(ccf%m_deadcrootc_to_cwdc_fire(beg:end))
    allocate(ccf%hrv_leafc_to_litr1c(beg:end))             
    allocate(ccf%hrv_leafc_to_litr2c(beg:end))             
    allocate(ccf%hrv_leafc_to_litr3c(beg:end))             
    allocate(ccf%hrv_frootc_to_litr1c(beg:end))            
    allocate(ccf%hrv_frootc_to_litr2c(beg:end))            
    allocate(ccf%hrv_frootc_to_litr3c(beg:end))            
    allocate(ccf%hrv_livestemc_to_cwdc(beg:end))           
    allocate(ccf%hrv_deadstemc_to_prod10c(beg:end))        
    allocate(ccf%hrv_deadstemc_to_prod100c(beg:end))       
    allocate(ccf%hrv_livecrootc_to_cwdc(beg:end))          
    allocate(ccf%hrv_deadcrootc_to_cwdc(beg:end))          
    allocate(ccf%hrv_leafc_storage_to_litr1c(beg:end))     
    allocate(ccf%hrv_frootc_storage_to_litr1c(beg:end))    
    allocate(ccf%hrv_livestemc_storage_to_litr1c(beg:end)) 
    allocate(ccf%hrv_deadstemc_storage_to_litr1c(beg:end)) 
    allocate(ccf%hrv_livecrootc_storage_to_litr1c(beg:end))
    allocate(ccf%hrv_deadcrootc_storage_to_litr1c(beg:end))
    allocate(ccf%hrv_gresp_storage_to_litr1c(beg:end))     
    allocate(ccf%hrv_leafc_xfer_to_litr1c(beg:end))        
    allocate(ccf%hrv_frootc_xfer_to_litr1c(beg:end))       
    allocate(ccf%hrv_livestemc_xfer_to_litr1c(beg:end))    
    allocate(ccf%hrv_deadstemc_xfer_to_litr1c(beg:end))    
    allocate(ccf%hrv_livecrootc_xfer_to_litr1c(beg:end))   
    allocate(ccf%hrv_deadcrootc_xfer_to_litr1c(beg:end))   
    allocate(ccf%hrv_gresp_xfer_to_litr1c(beg:end))        
    allocate(ccf%m_litr1c_to_fire(beg:end))
    allocate(ccf%m_litr2c_to_fire(beg:end))
    allocate(ccf%m_litr3c_to_fire(beg:end))
    allocate(ccf%m_cwdc_to_fire(beg:end))
#if (defined CROP)
    allocate(ccf%grainc_to_litr1c(beg:end))
    allocate(ccf%grainc_to_litr2c(beg:end))
    allocate(ccf%grainc_to_litr3c(beg:end))
    allocate(ccf%livestemc_to_litr1c(beg:end))
    allocate(ccf%livestemc_to_litr2c(beg:end))
    allocate(ccf%livestemc_to_litr3c(beg:end))
#endif
    allocate(ccf%leafc_to_litr1c(beg:end))
    allocate(ccf%leafc_to_litr2c(beg:end))
    allocate(ccf%leafc_to_litr3c(beg:end))
    allocate(ccf%frootc_to_litr1c(beg:end))
    allocate(ccf%frootc_to_litr2c(beg:end))
    allocate(ccf%frootc_to_litr3c(beg:end))
    allocate(ccf%cwdc_to_litr2c(beg:end))
    allocate(ccf%cwdc_to_litr3c(beg:end))
    allocate(ccf%litr1_hr(beg:end))
    allocate(ccf%litr1c_to_soil1c(beg:end))
    allocate(ccf%litr2_hr(beg:end))
    allocate(ccf%litr2c_to_soil2c(beg:end))
    allocate(ccf%litr3_hr(beg:end))
    allocate(ccf%litr3c_to_soil3c(beg:end))
    allocate(ccf%soil1_hr(beg:end))
    allocate(ccf%soil1c_to_soil2c(beg:end))
    allocate(ccf%soil2_hr(beg:end))
    allocate(ccf%soil2c_to_soil3c(beg:end))
    allocate(ccf%soil3_hr(beg:end))
    allocate(ccf%soil3c_to_soil4c(beg:end))
    allocate(ccf%soil4_hr(beg:end))
#ifdef CN
    allocate(ccf%dwt_seedc_to_leaf(beg:end))
    allocate(ccf%dwt_seedc_to_deadstem(beg:end))
    allocate(ccf%dwt_conv_cflux(beg:end))
    allocate(ccf%dwt_prod10c_gain(beg:end))
    allocate(ccf%dwt_prod100c_gain(beg:end))
    allocate(ccf%dwt_frootc_to_litr1c(beg:end))
    allocate(ccf%dwt_frootc_to_litr2c(beg:end))
    allocate(ccf%dwt_frootc_to_litr3c(beg:end))
    allocate(ccf%dwt_livecrootc_to_cwdc(beg:end))
    allocate(ccf%dwt_deadcrootc_to_cwdc(beg:end))
    allocate(ccf%dwt_closs(beg:end))
    allocate(ccf%landuseflux(beg:end))
    allocate(ccf%landuptake(beg:end))
    allocate(ccf%prod10c_loss(beg:end))
    allocate(ccf%prod100c_loss(beg:end))
    allocate(ccf%product_closs(beg:end))
#endif
    allocate(ccf%lithr(beg:end))
    allocate(ccf%somhr(beg:end))
    allocate(ccf%hr(beg:end))
    allocate(ccf%sr(beg:end))
    allocate(ccf%er(beg:end))
    allocate(ccf%litfire(beg:end))
    allocate(ccf%somfire(beg:end))
    allocate(ccf%totfire(beg:end))
    allocate(ccf%nep(beg:end))
    allocate(ccf%nbp(beg:end))
    allocate(ccf%nee(beg:end))
    allocate(ccf%col_cinputs(beg:end))
    allocate(ccf%col_coutputs(beg:end))
    allocate(ccf%col_fire_closs(beg:end))

#if (defined CLAMP) && (defined CN)
    !CLAMP
    allocate(ccf%cwdc_hr(beg:end))
    allocate(ccf%cwdc_loss(beg:end))
    allocate(ccf%litterc_loss(beg:end))
#endif

    ccf%m_leafc_to_litr1c(beg:end)                = nan
    ccf%m_leafc_to_litr2c(beg:end)                = nan
    ccf%m_leafc_to_litr3c(beg:end)                = nan
    ccf%m_frootc_to_litr1c(beg:end)               = nan
    ccf%m_frootc_to_litr2c(beg:end)               = nan
    ccf%m_frootc_to_litr3c(beg:end)               = nan
    ccf%m_leafc_storage_to_litr1c(beg:end)        = nan
    ccf%m_frootc_storage_to_litr1c(beg:end)       = nan
    ccf%m_livestemc_storage_to_litr1c(beg:end)    = nan
    ccf%m_deadstemc_storage_to_litr1c(beg:end)    = nan
    ccf%m_livecrootc_storage_to_litr1c(beg:end)   = nan
    ccf%m_deadcrootc_storage_to_litr1c(beg:end)   = nan
    ccf%m_leafc_xfer_to_litr1c(beg:end)           = nan
    ccf%m_frootc_xfer_to_litr1c(beg:end)          = nan
    ccf%m_livestemc_xfer_to_litr1c(beg:end)       = nan
    ccf%m_deadstemc_xfer_to_litr1c(beg:end)       = nan
    ccf%m_livecrootc_xfer_to_litr1c(beg:end)      = nan
    ccf%m_deadcrootc_xfer_to_litr1c(beg:end)      = nan
    ccf%m_livestemc_to_cwdc(beg:end)              = nan
    ccf%m_deadstemc_to_cwdc(beg:end)              = nan
    ccf%m_livecrootc_to_cwdc(beg:end)             = nan
    ccf%m_deadcrootc_to_cwdc(beg:end)             = nan
    ccf%m_gresp_storage_to_litr1c(beg:end)        = nan
    ccf%m_gresp_xfer_to_litr1c(beg:end)           = nan
    ccf%m_deadstemc_to_cwdc_fire(beg:end)         = nan
    ccf%m_deadcrootc_to_cwdc_fire(beg:end)        = nan
    ccf%hrv_leafc_to_litr1c(beg:end)              = nan             
    ccf%hrv_leafc_to_litr2c(beg:end)              = nan             
    ccf%hrv_leafc_to_litr3c(beg:end)              = nan             
    ccf%hrv_frootc_to_litr1c(beg:end)             = nan            
    ccf%hrv_frootc_to_litr2c(beg:end)             = nan            
    ccf%hrv_frootc_to_litr3c(beg:end)             = nan            
    ccf%hrv_livestemc_to_cwdc(beg:end)            = nan           
    ccf%hrv_deadstemc_to_prod10c(beg:end)         = nan        
    ccf%hrv_deadstemc_to_prod100c(beg:end)        = nan       
    ccf%hrv_livecrootc_to_cwdc(beg:end)           = nan          
    ccf%hrv_deadcrootc_to_cwdc(beg:end)           = nan          
    ccf%hrv_leafc_storage_to_litr1c(beg:end)      = nan     
    ccf%hrv_frootc_storage_to_litr1c(beg:end)     = nan    
    ccf%hrv_livestemc_storage_to_litr1c(beg:end)  = nan 
    ccf%hrv_deadstemc_storage_to_litr1c(beg:end)  = nan 
    ccf%hrv_livecrootc_storage_to_litr1c(beg:end) = nan
    ccf%hrv_deadcrootc_storage_to_litr1c(beg:end) = nan
#if (defined CROP)
    ccf%grainc_to_litr1c(beg:end) = nan
    ccf%grainc_to_litr2c(beg:end) = nan
    ccf%grainc_to_litr3c(beg:end) = nan
    ccf%livestemc_to_litr1c(beg:end) = nan
    ccf%livestemc_to_litr2c(beg:end) = nan
    ccf%livestemc_to_litr3c(beg:end) = nan
#endif
    ccf%hrv_gresp_storage_to_litr1c(beg:end)      = nan     
    ccf%hrv_leafc_xfer_to_litr1c(beg:end)         = nan        
    ccf%hrv_frootc_xfer_to_litr1c(beg:end)        = nan       
    ccf%hrv_livestemc_xfer_to_litr1c(beg:end)     = nan    
    ccf%hrv_deadstemc_xfer_to_litr1c(beg:end)     = nan    
    ccf%hrv_livecrootc_xfer_to_litr1c(beg:end)    = nan   
    ccf%hrv_deadcrootc_xfer_to_litr1c(beg:end)    = nan   
    ccf%hrv_gresp_xfer_to_litr1c(beg:end)         = nan        
    ccf%m_litr1c_to_fire(beg:end)                 = nan
    ccf%m_litr2c_to_fire(beg:end)                 = nan
    ccf%m_litr3c_to_fire(beg:end)                 = nan
    ccf%m_cwdc_to_fire(beg:end)                   = nan
    ccf%leafc_to_litr1c(beg:end)                  = nan
    ccf%leafc_to_litr2c(beg:end)                  = nan
    ccf%leafc_to_litr3c(beg:end)                  = nan
    ccf%frootc_to_litr1c(beg:end)                 = nan
    ccf%frootc_to_litr2c(beg:end)                 = nan
    ccf%frootc_to_litr3c(beg:end)                 = nan
    ccf%cwdc_to_litr2c(beg:end)                   = nan
    ccf%cwdc_to_litr3c(beg:end)                   = nan
    ccf%litr1_hr(beg:end)                         = nan
    ccf%litr1c_to_soil1c(beg:end)                 = nan
    ccf%litr2_hr(beg:end)                         = nan
    ccf%litr2c_to_soil2c(beg:end)                 = nan
    ccf%litr3_hr(beg:end)                         = nan
    ccf%litr3c_to_soil3c(beg:end)                 = nan
    ccf%soil1_hr(beg:end)                         = nan
    ccf%soil1c_to_soil2c(beg:end)                 = nan
    ccf%soil2_hr(beg:end)                         = nan
    ccf%soil2c_to_soil3c(beg:end)                 = nan
    ccf%soil3_hr(beg:end)                         = nan
    ccf%soil3c_to_soil4c(beg:end)                 = nan
    ccf%soil4_hr(beg:end)                         = nan
#if (defined CN)
    ccf%dwt_seedc_to_leaf(beg:end)                = nan
    ccf%dwt_seedc_to_deadstem(beg:end)            = nan
    ccf%dwt_conv_cflux(beg:end)                   = nan
    ccf%dwt_prod10c_gain(beg:end)                 = nan
    ccf%dwt_prod100c_gain(beg:end)                = nan
    ccf%dwt_frootc_to_litr1c(beg:end)             = nan
    ccf%dwt_frootc_to_litr2c(beg:end)             = nan
    ccf%dwt_frootc_to_litr3c(beg:end)             = nan
    ccf%dwt_livecrootc_to_cwdc(beg:end)           = nan
    ccf%dwt_deadcrootc_to_cwdc(beg:end)           = nan
    ccf%dwt_closs(beg:end)                        = nan
    ccf%landuseflux(beg:end)                      = nan
    ccf%landuptake(beg:end)                       = nan
    ccf%prod10c_loss(beg:end)                     = nan
    ccf%prod100c_loss(beg:end)                    = nan
    ccf%product_closs(beg:end)                    = nan
#endif
    ccf%lithr(beg:end)                            = nan
    ccf%somhr(beg:end)                            = nan
    ccf%hr(beg:end)                               = nan
    ccf%sr(beg:end)                               = nan
    ccf%er(beg:end)                               = nan
    ccf%litfire(beg:end)                          = nan
    ccf%somfire(beg:end)                          = nan
    ccf%totfire(beg:end)                          = nan
    ccf%nep(beg:end)                              = nan
    ccf%nbp(beg:end)                              = nan
    ccf%nee(beg:end)                              = nan
    ccf%col_cinputs(beg:end)                      = nan
    ccf%col_coutputs(beg:end)                     = nan
    ccf%col_fire_closs(beg:end)                   = nan

#if (defined CLAMP) && (defined CN)
    !CLAMP
    ccf%cwdc_hr(beg:end)                          = nan
    ccf%cwdc_loss(beg:end)                        = nan
    ccf%litterc_loss(beg:end)                     = nan
#endif

  end subroutine init_column_cflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_column_nflux_type
!
! !INTERFACE:
  subroutine init_column_nflux_type(beg, end, cnf)
!
! !DESCRIPTION:
! Initialize column nitrogen flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (column_nflux_type), intent(inout):: cnf
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

    allocate(cnf%ndep_to_sminn(beg:end))
    allocate(cnf%nfix_to_sminn(beg:end))
    allocate(cnf%m_leafn_to_litr1n(beg:end))
    allocate(cnf%m_leafn_to_litr2n(beg:end))
    allocate(cnf%m_leafn_to_litr3n(beg:end))
    allocate(cnf%m_frootn_to_litr1n(beg:end))
    allocate(cnf%m_frootn_to_litr2n(beg:end))
    allocate(cnf%m_frootn_to_litr3n(beg:end))
    allocate(cnf%m_leafn_storage_to_litr1n(beg:end))
    allocate(cnf%m_frootn_storage_to_litr1n(beg:end))
    allocate(cnf%m_livestemn_storage_to_litr1n(beg:end))
    allocate(cnf%m_deadstemn_storage_to_litr1n(beg:end))
    allocate(cnf%m_livecrootn_storage_to_litr1n(beg:end))
    allocate(cnf%m_deadcrootn_storage_to_litr1n(beg:end))
    allocate(cnf%m_leafn_xfer_to_litr1n(beg:end))
    allocate(cnf%m_frootn_xfer_to_litr1n(beg:end))
    allocate(cnf%m_livestemn_xfer_to_litr1n(beg:end))
    allocate(cnf%m_deadstemn_xfer_to_litr1n(beg:end))
    allocate(cnf%m_livecrootn_xfer_to_litr1n(beg:end))
    allocate(cnf%m_deadcrootn_xfer_to_litr1n(beg:end))
    allocate(cnf%m_livestemn_to_cwdn(beg:end))
    allocate(cnf%m_deadstemn_to_cwdn(beg:end))
    allocate(cnf%m_livecrootn_to_cwdn(beg:end))
    allocate(cnf%m_deadcrootn_to_cwdn(beg:end))
    allocate(cnf%m_retransn_to_litr1n(beg:end))
    allocate(cnf%hrv_leafn_to_litr1n(beg:end))             
    allocate(cnf%hrv_leafn_to_litr2n(beg:end))             
    allocate(cnf%hrv_leafn_to_litr3n(beg:end))             
    allocate(cnf%hrv_frootn_to_litr1n(beg:end))            
    allocate(cnf%hrv_frootn_to_litr2n(beg:end))            
    allocate(cnf%hrv_frootn_to_litr3n(beg:end))            
    allocate(cnf%hrv_livestemn_to_cwdn(beg:end))           
    allocate(cnf%hrv_deadstemn_to_prod10n(beg:end))        
    allocate(cnf%hrv_deadstemn_to_prod100n(beg:end))       
    allocate(cnf%hrv_livecrootn_to_cwdn(beg:end))          
    allocate(cnf%hrv_deadcrootn_to_cwdn(beg:end))          
    allocate(cnf%hrv_retransn_to_litr1n(beg:end))          
    allocate(cnf%hrv_leafn_storage_to_litr1n(beg:end))     
    allocate(cnf%hrv_frootn_storage_to_litr1n(beg:end))    
    allocate(cnf%hrv_livestemn_storage_to_litr1n(beg:end)) 
    allocate(cnf%hrv_deadstemn_storage_to_litr1n(beg:end)) 
    allocate(cnf%hrv_livecrootn_storage_to_litr1n(beg:end))
    allocate(cnf%hrv_deadcrootn_storage_to_litr1n(beg:end))
    allocate(cnf%hrv_leafn_xfer_to_litr1n(beg:end))        
    allocate(cnf%hrv_frootn_xfer_to_litr1n(beg:end))       
    allocate(cnf%hrv_livestemn_xfer_to_litr1n(beg:end))    
    allocate(cnf%hrv_deadstemn_xfer_to_litr1n(beg:end))    
    allocate(cnf%hrv_livecrootn_xfer_to_litr1n(beg:end))   
    allocate(cnf%hrv_deadcrootn_xfer_to_litr1n(beg:end))   
    allocate(cnf%m_deadstemn_to_cwdn_fire(beg:end))
    allocate(cnf%m_deadcrootn_to_cwdn_fire(beg:end))
    allocate(cnf%m_litr1n_to_fire(beg:end))
    allocate(cnf%m_litr2n_to_fire(beg:end))
    allocate(cnf%m_litr3n_to_fire(beg:end))
    allocate(cnf%m_cwdn_to_fire(beg:end))
#if (defined CROP)
    allocate(cnf%grainn_to_litr1n(beg:end))
    allocate(cnf%grainn_to_litr2n(beg:end))
    allocate(cnf%grainn_to_litr3n(beg:end))
    allocate(cnf%livestemn_to_litr1n(beg:end))
    allocate(cnf%livestemn_to_litr2n(beg:end))
    allocate(cnf%livestemn_to_litr3n(beg:end))
#endif
    allocate(cnf%leafn_to_litr1n(beg:end))
    allocate(cnf%leafn_to_litr2n(beg:end))
    allocate(cnf%leafn_to_litr3n(beg:end))
    allocate(cnf%frootn_to_litr1n(beg:end))
    allocate(cnf%frootn_to_litr2n(beg:end))
    allocate(cnf%frootn_to_litr3n(beg:end))
    allocate(cnf%cwdn_to_litr2n(beg:end))
    allocate(cnf%cwdn_to_litr3n(beg:end))
    allocate(cnf%litr1n_to_soil1n(beg:end))
    allocate(cnf%sminn_to_soil1n_l1(beg:end))
    allocate(cnf%litr2n_to_soil2n(beg:end))
    allocate(cnf%sminn_to_soil2n_l2(beg:end))
    allocate(cnf%litr3n_to_soil3n(beg:end))
    allocate(cnf%sminn_to_soil3n_l3(beg:end))
    allocate(cnf%soil1n_to_soil2n(beg:end))
    allocate(cnf%sminn_to_soil2n_s1(beg:end))
    allocate(cnf%soil2n_to_soil3n(beg:end))
    allocate(cnf%sminn_to_soil3n_s2(beg:end))
    allocate(cnf%soil3n_to_soil4n(beg:end))
    allocate(cnf%sminn_to_soil4n_s3(beg:end))
    allocate(cnf%soil4n_to_sminn(beg:end))
    allocate(cnf%sminn_to_denit_l1s1(beg:end))
    allocate(cnf%sminn_to_denit_l2s2(beg:end))
    allocate(cnf%sminn_to_denit_l3s3(beg:end))
    allocate(cnf%sminn_to_denit_s1s2(beg:end))
    allocate(cnf%sminn_to_denit_s2s3(beg:end))
    allocate(cnf%sminn_to_denit_s3s4(beg:end))
    allocate(cnf%sminn_to_denit_s4(beg:end))
    allocate(cnf%sminn_to_denit_excess(beg:end))
    allocate(cnf%sminn_leached(beg:end))
    allocate(cnf%dwt_seedn_to_leaf(beg:end))
    allocate(cnf%dwt_seedn_to_deadstem(beg:end))
    allocate(cnf%dwt_conv_nflux(beg:end))
    allocate(cnf%dwt_prod10n_gain(beg:end))
    allocate(cnf%dwt_prod100n_gain(beg:end))
    allocate(cnf%dwt_frootn_to_litr1n(beg:end))
    allocate(cnf%dwt_frootn_to_litr2n(beg:end))
    allocate(cnf%dwt_frootn_to_litr3n(beg:end))
    allocate(cnf%dwt_livecrootn_to_cwdn(beg:end))
    allocate(cnf%dwt_deadcrootn_to_cwdn(beg:end))
    allocate(cnf%dwt_nloss(beg:end))
    allocate(cnf%prod10n_loss(beg:end))
    allocate(cnf%prod100n_loss(beg:end))
    allocate(cnf%product_nloss(beg:end))
    allocate(cnf%potential_immob(beg:end))
    allocate(cnf%actual_immob(beg:end))
    allocate(cnf%sminn_to_plant(beg:end))
    allocate(cnf%supplement_to_sminn(beg:end))
    allocate(cnf%gross_nmin(beg:end))
    allocate(cnf%net_nmin(beg:end))
    allocate(cnf%denit(beg:end))
    allocate(cnf%col_ninputs(beg:end))
    allocate(cnf%col_noutputs(beg:end))
    allocate(cnf%col_fire_nloss(beg:end))

    cnf%ndep_to_sminn(beg:end) = nan
    cnf%nfix_to_sminn(beg:end) = nan
    cnf%m_leafn_to_litr1n(beg:end) = nan
    cnf%m_leafn_to_litr2n(beg:end) = nan
    cnf%m_leafn_to_litr3n(beg:end) = nan
    cnf%m_frootn_to_litr1n(beg:end) = nan
    cnf%m_frootn_to_litr2n(beg:end) = nan
    cnf%m_frootn_to_litr3n(beg:end) = nan
    cnf%m_leafn_storage_to_litr1n(beg:end) = nan
    cnf%m_frootn_storage_to_litr1n(beg:end) = nan
    cnf%m_livestemn_storage_to_litr1n(beg:end) = nan
    cnf%m_deadstemn_storage_to_litr1n(beg:end) = nan
    cnf%m_livecrootn_storage_to_litr1n(beg:end) = nan
    cnf%m_deadcrootn_storage_to_litr1n(beg:end) = nan
    cnf%m_leafn_xfer_to_litr1n(beg:end) = nan
    cnf%m_frootn_xfer_to_litr1n(beg:end) = nan
    cnf%m_livestemn_xfer_to_litr1n(beg:end) = nan
    cnf%m_deadstemn_xfer_to_litr1n(beg:end) = nan
    cnf%m_livecrootn_xfer_to_litr1n(beg:end) = nan
    cnf%m_deadcrootn_xfer_to_litr1n(beg:end) = nan
    cnf%m_livestemn_to_cwdn(beg:end) = nan
    cnf%m_deadstemn_to_cwdn(beg:end) = nan
    cnf%m_livecrootn_to_cwdn(beg:end) = nan
    cnf%m_deadcrootn_to_cwdn(beg:end) = nan
    cnf%m_retransn_to_litr1n(beg:end) = nan
    cnf%hrv_leafn_to_litr1n(beg:end) = nan             
    cnf%hrv_leafn_to_litr2n(beg:end) = nan             
    cnf%hrv_leafn_to_litr3n(beg:end) = nan             
    cnf%hrv_frootn_to_litr1n(beg:end) = nan            
    cnf%hrv_frootn_to_litr2n(beg:end) = nan            
    cnf%hrv_frootn_to_litr3n(beg:end) = nan            
    cnf%hrv_livestemn_to_cwdn(beg:end) = nan           
    cnf%hrv_deadstemn_to_prod10n(beg:end) = nan        
    cnf%hrv_deadstemn_to_prod100n(beg:end) = nan       
    cnf%hrv_livecrootn_to_cwdn(beg:end) = nan          
    cnf%hrv_deadcrootn_to_cwdn(beg:end) = nan          
    cnf%hrv_retransn_to_litr1n(beg:end) = nan          
    cnf%hrv_leafn_storage_to_litr1n(beg:end) = nan     
    cnf%hrv_frootn_storage_to_litr1n(beg:end) = nan    
    cnf%hrv_livestemn_storage_to_litr1n(beg:end) = nan 
    cnf%hrv_deadstemn_storage_to_litr1n(beg:end) = nan 
    cnf%hrv_livecrootn_storage_to_litr1n(beg:end) = nan
    cnf%hrv_deadcrootn_storage_to_litr1n(beg:end) = nan
    cnf%hrv_leafn_xfer_to_litr1n(beg:end) = nan        
    cnf%hrv_frootn_xfer_to_litr1n(beg:end) = nan       
    cnf%hrv_livestemn_xfer_to_litr1n(beg:end) = nan    
    cnf%hrv_deadstemn_xfer_to_litr1n(beg:end) = nan    
    cnf%hrv_livecrootn_xfer_to_litr1n(beg:end) = nan   
    cnf%hrv_deadcrootn_xfer_to_litr1n(beg:end) = nan   
    cnf%m_deadstemn_to_cwdn_fire(beg:end) = nan
    cnf%m_deadcrootn_to_cwdn_fire(beg:end) = nan
    cnf%m_litr1n_to_fire(beg:end) = nan
    cnf%m_litr2n_to_fire(beg:end) = nan
    cnf%m_litr3n_to_fire(beg:end) = nan
    cnf%m_cwdn_to_fire(beg:end) = nan
#if (defined CROP)
    cnf%grainn_to_litr1n(beg:end) = nan
    cnf%grainn_to_litr2n(beg:end) = nan
    cnf%grainn_to_litr3n(beg:end) = nan
    cnf%livestemn_to_litr1n(beg:end) = nan
    cnf%livestemn_to_litr2n(beg:end) = nan
    cnf%livestemn_to_litr3n(beg:end) = nan
#endif
    cnf%leafn_to_litr1n(beg:end) = nan
    cnf%leafn_to_litr2n(beg:end) = nan
    cnf%leafn_to_litr3n(beg:end) = nan
    cnf%frootn_to_litr1n(beg:end) = nan
    cnf%frootn_to_litr2n(beg:end) = nan
    cnf%frootn_to_litr3n(beg:end) = nan
    cnf%cwdn_to_litr2n(beg:end) = nan
    cnf%cwdn_to_litr3n(beg:end) = nan
    cnf%litr1n_to_soil1n(beg:end) = nan
    cnf%sminn_to_soil1n_l1(beg:end) = nan
    cnf%litr2n_to_soil2n(beg:end) = nan
    cnf%sminn_to_soil2n_l2(beg:end) = nan
    cnf%litr3n_to_soil3n(beg:end) = nan
    cnf%sminn_to_soil3n_l3(beg:end) = nan
    cnf%soil1n_to_soil2n(beg:end) = nan
    cnf%sminn_to_soil2n_s1(beg:end) = nan
    cnf%soil2n_to_soil3n(beg:end) = nan
    cnf%sminn_to_soil3n_s2(beg:end) = nan
    cnf%soil3n_to_soil4n(beg:end) = nan
    cnf%sminn_to_soil4n_s3(beg:end) = nan
    cnf%soil4n_to_sminn(beg:end) = nan
    cnf%sminn_to_denit_l1s1(beg:end) = nan
    cnf%sminn_to_denit_l2s2(beg:end) = nan
    cnf%sminn_to_denit_l3s3(beg:end) = nan
    cnf%sminn_to_denit_s1s2(beg:end) = nan
    cnf%sminn_to_denit_s2s3(beg:end) = nan
    cnf%sminn_to_denit_s3s4(beg:end) = nan
    cnf%sminn_to_denit_s4(beg:end) = nan
    cnf%sminn_to_denit_excess(beg:end) = nan
    cnf%sminn_leached(beg:end) = nan
    cnf%dwt_seedn_to_leaf(beg:end) = nan
    cnf%dwt_seedn_to_deadstem(beg:end) = nan
    cnf%dwt_conv_nflux(beg:end) = nan
    cnf%dwt_prod10n_gain(beg:end) = nan
    cnf%dwt_prod100n_gain(beg:end) = nan
    cnf%dwt_frootn_to_litr1n(beg:end) = nan
    cnf%dwt_frootn_to_litr2n(beg:end) = nan
    cnf%dwt_frootn_to_litr3n(beg:end) = nan
    cnf%dwt_livecrootn_to_cwdn(beg:end) = nan
    cnf%dwt_deadcrootn_to_cwdn(beg:end) = nan
    cnf%dwt_nloss(beg:end) = nan
    cnf%prod10n_loss(beg:end) = nan
    cnf%prod100n_loss(beg:end) = nan
    cnf%product_nloss(beg:end) = nan
    cnf%potential_immob(beg:end) = nan
    cnf%actual_immob(beg:end) = nan
    cnf%sminn_to_plant(beg:end) = nan
    cnf%supplement_to_sminn(beg:end) = nan
    cnf%gross_nmin(beg:end) = nan
    cnf%net_nmin(beg:end) = nan
    cnf%denit(beg:end) = nan
    cnf%col_ninputs(beg:end) = nan
    cnf%col_noutputs(beg:end) = nan
    cnf%col_fire_nloss(beg:end) = nan

  end subroutine init_column_nflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_landunit_pstate_type
!
! !INTERFACE:
  subroutine init_landunit_pstate_type(beg, end, lps)
!
! !DESCRIPTION:
! Initialize landunit physical state variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (landunit_pstate_type), intent(inout):: lps
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(lps%t_building(beg:end))
    allocate(lps%t_building_max(beg:end))
    allocate(lps%t_building_min(beg:end))
    allocate(lps%tk_wall(beg:end,nlevurb))
    allocate(lps%tk_roof(beg:end,nlevurb))
    allocate(lps%tk_improad(beg:end,nlevgrnd))
    allocate(lps%cv_wall(beg:end,nlevurb))
    allocate(lps%cv_roof(beg:end,nlevurb))
    allocate(lps%cv_improad(beg:end,nlevgrnd))
    allocate(lps%thick_wall(beg:end))
    allocate(lps%thick_roof(beg:end))
    allocate(lps%nlev_improad(beg:end))
    allocate(lps%vf_sr(beg:end))
    allocate(lps%vf_wr(beg:end))
    allocate(lps%vf_sw(beg:end))
    allocate(lps%vf_rw(beg:end))
    allocate(lps%vf_ww(beg:end))
    allocate(lps%taf(beg:end))
    allocate(lps%qaf(beg:end))
    allocate(lps%sabs_roof_dir(beg:end,1:numrad))
    allocate(lps%sabs_roof_dif(beg:end,1:numrad))
    allocate(lps%sabs_sunwall_dir(beg:end,1:numrad))
    allocate(lps%sabs_sunwall_dif(beg:end,1:numrad))
    allocate(lps%sabs_shadewall_dir(beg:end,1:numrad))
    allocate(lps%sabs_shadewall_dif(beg:end,1:numrad))
    allocate(lps%sabs_improad_dir(beg:end,1:numrad))
    allocate(lps%sabs_improad_dif(beg:end,1:numrad))
    allocate(lps%sabs_perroad_dir(beg:end,1:numrad))
    allocate(lps%sabs_perroad_dif(beg:end,1:numrad))

    lps%t_building(beg:end) = nan
    lps%t_building_max(beg:end) = nan
    lps%t_building_min(beg:end) = nan
    lps%tk_wall(beg:end,1:nlevurb) = nan
    lps%tk_roof(beg:end,1:nlevurb) = nan
    lps%tk_improad(beg:end,1:nlevgrnd) = nan
    lps%cv_wall(beg:end,1:nlevurb) = nan
    lps%cv_roof(beg:end,1:nlevurb) = nan
    lps%cv_improad(beg:end,1:nlevgrnd) = nan
    lps%cv_improad(beg:end,1:5) = nan
    lps%thick_wall(beg:end) = nan
    lps%thick_roof(beg:end) = nan
    lps%nlev_improad(beg:end) = bigint
    lps%vf_sr(beg:end) = nan
    lps%vf_wr(beg:end) = nan
    lps%vf_sw(beg:end) = nan
    lps%vf_rw(beg:end) = nan
    lps%vf_ww(beg:end) = nan
    lps%taf(beg:end) = nan
    lps%qaf(beg:end) = nan
    lps%sabs_roof_dir(beg:end,1:numrad) = nan
    lps%sabs_roof_dif(beg:end,1:numrad) = nan
    lps%sabs_sunwall_dir(beg:end,1:numrad) = nan
    lps%sabs_sunwall_dif(beg:end,1:numrad) = nan
    lps%sabs_shadewall_dir(beg:end,1:numrad) = nan
    lps%sabs_shadewall_dif(beg:end,1:numrad) = nan
    lps%sabs_improad_dir(beg:end,1:numrad) = nan
    lps%sabs_improad_dif(beg:end,1:numrad) = nan
    lps%sabs_perroad_dir(beg:end,1:numrad) = nan
    lps%sabs_perroad_dif(beg:end,1:numrad) = nan

  end subroutine init_landunit_pstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_landunit_eflux_type
!
! !INTERFACE:
  subroutine init_landunit_eflux_type(beg, end, lef)
!
! !DESCRIPTION: 
! Initialize landunit energy flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end 
    type (landunit_eflux_type), intent(inout):: lef 
!
! !REVISION HISTORY:
! Created by Keith Oleson
!
!EOP
!------------------------------------------------------------------------

    allocate(lef%eflx_traffic(beg:end))
    allocate(lef%eflx_traffic_factor(beg:end))
    allocate(lef%eflx_wasteheat(beg:end))
    allocate(lef%eflx_heat_from_ac(beg:end))

    lef%eflx_traffic(beg:end) = nan
    lef%eflx_traffic_factor(beg:end) = nan
    lef%eflx_wasteheat(beg:end) = nan
    lef%eflx_heat_from_ac(beg:end) = nan

  end subroutine init_landunit_eflux_type

#if (defined CNDV)
!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_gridcell_dgvstate_type
!
! !INTERFACE:
  subroutine init_gridcell_dgvstate_type(beg, end, gps)
!
! !DESCRIPTION:
! Initialize gridcell DGVM variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (gridcell_dgvstate_type), intent(inout):: gps
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(gps%agdd20(beg:end))
    allocate(gps%tmomin20(beg:end))
    allocate(gps%t10min(beg:end))
    gps%agdd20(beg:end) = nan
    gps%tmomin20(beg:end) = nan
    gps%t10min(beg:end) = nan

  end subroutine init_gridcell_dgvstate_type
#endif

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_gridcell_pstate_type
!
! !INTERFACE:
  subroutine init_gridcell_pstate_type(beg, end, gps)
!
! !DESCRIPTION:
! Initialize gridcell physical state variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (gridcell_pstate_type), intent(inout):: gps
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------
    
    
    !allocate(gps%bcphiwet2t(beg:end,1:2))
    !allocate(gps%bcphidry2t(beg:end,1:2))
    !allocate(gps%bcphodry2t(beg:end,1:2))
    !allocate(gps%ocphiwet2t(beg:end,1:2))
    !allocate(gps%ocphidry2t(beg:end,1:2))
    !allocate(gps%ocphodry2t(beg:end,1:2))
    !allocate(gps%dstx01wd2t(beg:end,1:2))
    !allocate(gps%dstx01dd2t(beg:end,1:2))
    !allocate(gps%dstx02wd2t(beg:end,1:2))
    !allocate(gps%dstx02dd2t(beg:end,1:2))
    !allocate(gps%dstx03wd2t(beg:end,1:2))
    !allocate(gps%dstx03dd2t(beg:end,1:2))
    !allocate(gps%dstx04wd2t(beg:end,1:2))
    !allocate(gps%dstx04dd2t(beg:end,1:2))
    
    !gps%bcphiwet2t(beg:end,1:2) = nan
    !gps%bcphidry2t(beg:end,1:2) = nan
    !gps%bcphodry2t(beg:end,1:2) = nan
    !gps%ocphiwet2t(beg:end,1:2) = nan
    !gps%ocphidry2t(beg:end,1:2) = nan
    !gps%ocphodry2t(beg:end,1:2) = nan
    !gps%dstx01wd2t(beg:end,1:2) = nan
    !gps%dstx01dd2t(beg:end,1:2) = nan
    !gps%dstx02wd2t(beg:end,1:2) = nan
    !gps%dstx02dd2t(beg:end,1:2) = nan
    !gps%dstx03wd2t(beg:end,1:2) = nan
    !gps%dstx03dd2t(beg:end,1:2) = nan
    !gps%dstx04wd2t(beg:end,1:2) = nan
    !gps%dstx04dd2t(beg:end,1:2) = nan

  end subroutine init_gridcell_pstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_gridcell_efstate_type
!
! !INTERFACE:
  subroutine init_gridcell_efstate_type(beg, end, gve)
!
! !DESCRIPTION:
! Initialize gridcell isoprene emission factor variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (gridcell_efstate_type), intent(inout) :: gve
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein (heald)
!
!EOP
!------------------------------------------------------------------------

    allocate(gve%efisop(6,beg:end))
    gve%efisop(:,beg:end) = nan

  end subroutine init_gridcell_efstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_gridcell_wflux_type
!
! !INTERFACE:
  subroutine init_gridcell_wflux_type(beg, end, gwf)
!
! !DESCRIPTION:
! Initialize gridcell water flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (gridcell_wflux_type), intent(inout):: gwf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    allocate(gwf%qflx_runoffg(beg:end))
    allocate(gwf%qflx_snwcp_iceg(beg:end))
    allocate(gwf%qflx_liq_dynbal(beg:end))
    allocate(gwf%qflx_ice_dynbal(beg:end))

    gwf%qflx_runoffg(beg:end) = nan
    gwf%qflx_snwcp_iceg(beg:end) = nan
    gwf%qflx_liq_dynbal(beg:end) = nan
    gwf%qflx_ice_dynbal(beg:end) = nan

  end subroutine init_gridcell_wflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_gridcell_eflux_type
!
! !INTERFACE:
  subroutine init_gridcell_eflux_type(beg, end, gef)
!
! !DESCRIPTION:
! Initialize gridcell energy flux variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (gridcell_eflux_type), intent(inout):: gef
!
! !REVISION HISTORY:
! Created by David Lawrence
!
!EOP
!------------------------------------------------------------------------
    allocate(gef%eflx_sh_totg(beg:end))
    allocate(gef%eflx_dynbal(beg:end))

    gef%eflx_sh_totg(beg:end) = nan
    gef%eflx_dynbal(beg:end) = nan

  end subroutine init_gridcell_eflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_gridcell_wstate_type
!
! !INTERFACE:
  subroutine init_gridcell_wstate_type(beg, end, gws)
!
! !DESCRIPTION:
! Initialize gridcell water state variables
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (gridcell_wstate_type), intent(inout):: gws
!
! !REVISION HISTORY:
! Created by David Lawrence
!
!EOP
!------------------------------------------------------------------------
    allocate(gws%gc_liq1(beg:end))
    allocate(gws%gc_liq2(beg:end))
    allocate(gws%gc_ice1(beg:end))     
    allocate(gws%gc_ice2(beg:end))    

    gws%gc_liq1(beg:end) = nan
    gws%gc_liq2(beg:end) = nan
    gws%gc_ice1(beg:end) = nan     
    gws%gc_ice2(beg:end) = nan    

  end subroutine init_gridcell_wstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_gridcell_estate_type
!
! !INTERFACE:
  subroutine init_gridcell_estate_type(beg, end, ges)
!
! !DESCRIPTION:
! Initialize gridcell energy state variables     
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: beg, end
    type (gridcell_estate_type), intent(inout):: ges
!
! !REVISION HISTORY:
! Created by David Lawrence
!
!EOP
!------------------------------------------------------------------------
    allocate(ges%gc_heat1(beg:end))     
    allocate(ges%gc_heat2(beg:end))    

    ges%gc_heat1(beg:end) = nan     
    ges%gc_heat2(beg:end) = nan    

  end subroutine init_gridcell_estate_type


! !INTERFACE:
  subroutine init_atm2lnd_type(beg, end, a2l)
!
! !DESCRIPTION:
! Initialize atmospheric variables required by the land
! 
! !ARGUMENTS:
  implicit none
  integer, intent(in) :: beg, end
  type (atm2lnd_type), intent(inout):: a2l
! 
! !REVISION HISTORY:
! Created by Mariana Vertenstein
! Modified by T Craig, 11/01/05 for finemesh project
!
!
! !LOCAL VARIABLES:
!EOP
  real(r8) :: ival   ! initial value
  integer  :: beg_atm, end_atm
!------------------------------------------------------------------------

  allocate(a2l%forc_t(beg:end))
  allocate(a2l%forc_u(beg:end))
  allocate(a2l%forc_v(beg:end))
  allocate(a2l%forc_wind(beg:end))
  allocate(a2l%forc_q(beg:end))
  allocate(a2l%forc_rh(beg:end))
  allocate(a2l%forc_hgt(beg:end))
  allocate(a2l%forc_hgt_u(beg:end))
  allocate(a2l%forc_hgt_t(beg:end))
  allocate(a2l%forc_hgt_q(beg:end))
  allocate(a2l%forc_pbot(beg:end))
  allocate(a2l%forc_th(beg:end))
  allocate(a2l%forc_vp(beg:end))
  allocate(a2l%forc_rho(beg:end))
  allocate(a2l%forc_psrf(beg:end))
  allocate(a2l%forc_pco2(beg:end))
  allocate(a2l%forc_lwrad(beg:end))
  allocate(a2l%forc_solad(beg:end,numrad))
  allocate(a2l%forc_solai(beg:end,numrad))
  allocate(a2l%forc_solar(beg:end))
  allocate(a2l%forc_rain(beg:end))
  allocate(a2l%forc_snow(beg:end))
  allocate(a2l%forc_ndep(beg:end))
  allocate(a2l%rainf(beg:end))
#if (defined C13)
  ! 4/14/05: PET
  ! Adding isotope code
  allocate(a2l%forc_pc13o2(beg:end))
#endif
  allocate(a2l%forc_po2(beg:end))
  allocate(a2l%forc_aer(beg:end,14))

! ival = nan      ! causes core dump in map_maparray, tcx fix
  ival = 0.0_r8

  a2l%forc_t(beg:end) = ival
  a2l%forc_u(beg:end) = ival
  a2l%forc_v(beg:end) = ival
  a2l%forc_wind(beg:end) = ival
  a2l%forc_q(beg:end) = ival
  a2l%forc_rh(beg:end) = ival
  a2l%forc_hgt(beg:end) = ival
  a2l%forc_hgt_u(beg:end) = ival
  a2l%forc_hgt_t(beg:end) = ival
  a2l%forc_hgt_q(beg:end) = ival
  a2l%forc_pbot(beg:end) = ival
  a2l%forc_th(beg:end) = ival
  a2l%forc_vp(beg:end) = ival
  a2l%forc_rho(beg:end) = ival
  a2l%forc_psrf(beg:end) = ival
  a2l%forc_pco2(beg:end) = ival
  a2l%forc_lwrad(beg:end) = ival
  a2l%forc_solad(beg:end,1:numrad) = ival
  a2l%forc_solai(beg:end,1:numrad) = ival
  a2l%forc_solar(beg:end) = ival
  a2l%forc_rain(beg:end) = ival
  a2l%forc_snow(beg:end) = ival
  a2l%forc_ndep(beg:end) = ival
  a2l%rainf(beg:end) = nan
#if (defined C13)
  ! 4/14/05: PET
  ! Adding isotope code
  a2l%forc_pc13o2(beg:end) = ival
#endif
  a2l%forc_po2(beg:end) = ival
  a2l%forc_aer(beg:end,:) = ival

end subroutine init_atm2lnd_type





  subroutine clmtype_dealloc()
!
! !ARGUMENTS:
    implicit none

    call dealloc_pft_type     (    clm3%g%l%c%p)
    call dealloc_column_type  (    clm3%g%l%c)
    call dealloc_landunit_type(    clm3%g%l)
    call dealloc_gridcell_type(    clm3%g)
   ! pft ecophysiological constants
  
    call dealloc_pft_ecophys_constants()
  
#if (defined CNDV)
    ! pft DGVM-specific ecophysiological constants
  
    call dealloc_pft_DGVMecophys_constants()
#endif
  
    ! energy balance structures (all levels)

    call dealloc_energy_balance_type(    clm3%g%l%c%p%pebal)
    call dealloc_energy_balance_type(    clm3%g%l%c%cebal)
    call dealloc_energy_balance_type(    clm3%g%l%lebal)
    call dealloc_energy_balance_type(    clm3%g%gebal)
    call dealloc_energy_balance_type(          clm3%mebal)

    ! water balance structures (all levels)

    call dealloc_water_balance_type(    clm3%g%l%c%p%pwbal)
    call dealloc_water_balance_type(    clm3%g%l%c%cwbal)
    call dealloc_water_balance_type(    clm3%g%l%lwbal)
    call dealloc_water_balance_type(    clm3%g%gwbal)
    call dealloc_water_balance_type(          clm3%mwbal)

    ! carbon balance structures (pft and column levels)

    call dealloc_carbon_balance_type(    clm3%g%l%c%p%pcbal)
    call dealloc_carbon_balance_type(    clm3%g%l%c%ccbal)

    ! nitrogen balance structures (pft and column levels)

    call dealloc_nitrogen_balance_type(    clm3%g%l%c%p%pnbal)
    call dealloc_nitrogen_balance_type(    clm3%g%l%c%cnbal)
    
    ! pft physical state variables at pft level and averaged to the column

    call dealloc_pft_pstate_type(    clm3%g%l%c%p%pps)
    call dealloc_pft_pstate_type(    clm3%g%l%c%cps%pps_a)

    ! pft ecophysiological variables (only at the pft level for now)
    call dealloc_pft_epv_type(    clm3%g%l%c%p%pepv)

#if (defined CNDV) || (defined CROP)
    ! pft DGVM state variables at pft level and averaged to column
    
    call dealloc_pft_pdgvstate_type(    clm3%g%l%c%p%pdgvs)
#endif
#if (defined CNDV)
    call dealloc_pft_pdgvstate_type(    clm3%g%l%c%cdgvs%pdgvs_a)
#endif
    call dealloc_pft_vstate_type(    clm3%g%l%c%p%pvs)
    
    ! pft energy state variables at the pft level and averaged to the column

    call dealloc_pft_estate_type(    clm3%g%l%c%p%pes)
    call dealloc_pft_estate_type(    clm3%g%l%c%ces%pes_a)

    ! pft water state variables at the pft level and averaged to the column

    call dealloc_pft_wstate_type(    clm3%g%l%c%p%pws)
    call dealloc_pft_wstate_type(    clm3%g%l%c%cws%pws_a)
    
    ! pft carbon state variables at the pft level and averaged to the column

    call dealloc_pft_cstate_type(    clm3%g%l%c%p%pcs)
    call dealloc_pft_cstate_type(    clm3%g%l%c%ccs%pcs_a)
#if (defined C13)
    ! 4/14/05: PET
    ! Adding isotope code
    call dealloc_pft_cstate_type(    clm3%g%l%c%p%pc13s)
    call dealloc_pft_cstate_type(    clm3%g%l%c%cc13s%pcs_a)
#endif

    ! pft nitrogen state variables at the pft level and averaged to the column

    call dealloc_pft_nstate_type(    clm3%g%l%c%p%pns)
    call dealloc_pft_nstate_type(    clm3%g%l%c%cns%pns_a)

    ! pft energy flux variables at pft level and averaged to column

    call dealloc_pft_eflux_type(    clm3%g%l%c%p%pef)
    call dealloc_pft_eflux_type(    clm3%g%l%c%cef%pef_a)

    ! pft momentum flux variables at pft level and averaged to the column

    call dealloc_pft_mflux_type(    clm3%g%l%c%p%pmf)
    call dealloc_pft_mflux_type(    clm3%g%l%c%cmf%pmf_a)

    ! pft water flux variables

    call dealloc_pft_wflux_type(    clm3%g%l%c%p%pwf)
    call dealloc_pft_wflux_type(    clm3%g%l%c%cwf%pwf_a)

    ! pft carbon flux variables at pft level and averaged to column

    call dealloc_pft_cflux_type(    clm3%g%l%c%p%pcf)
    call dealloc_pft_cflux_type(    clm3%g%l%c%ccf%pcf_a)
#if (defined C13)
    ! 4/14/05: PET
    ! Adding isotope code
    call dealloc_pft_cflux_type(    clm3%g%l%c%p%pc13f)
    call dealloc_pft_cflux_type(    clm3%g%l%c%cc13f%pcf_a)
#endif

    ! pft nitrogen flux variables at pft level and averaged to column

    call dealloc_pft_nflux_type(    clm3%g%l%c%p%pnf)
    call dealloc_pft_nflux_type(    clm3%g%l%c%cnf%pnf_a)

    ! pft VOC flux variables at pft level and averaged to column

    call dealloc_pft_vflux_type(    clm3%g%l%c%p%pvf)
    call dealloc_pft_vflux_type(    clm3%g%l%c%cvf%pvf_a)

    ! gridcell VOC emission factors (heald, 05/06)

    call dealloc_gridcell_efstate_type(    clm3%g%gve)

    ! pft dust flux variables at pft level and averaged to column

    call dealloc_pft_dflux_type(    clm3%g%l%c%p%pdf)
    call dealloc_pft_dflux_type(    clm3%g%l%c%cdf%pdf_a)

    ! pft dry dep velocity variables at pft level and averaged to column

    call dealloc_pft_depvd_type(    clm3%g%l%c%p%pdd)

    ! column physical state variables at column level and averaged to
    ! the landunit and gridcell and model

    call dealloc_column_pstate_type(    clm3%g%l%c%cps)
    call dealloc_column_pstate_type(    clm3%g%l%lps%cps_a)
    call dealloc_column_pstate_type(    clm3%g%gps%cps_a)
    call dealloc_column_pstate_type(          clm3%mps%cps_a)
    
    ! column energy state variables at column level and averaged to
    ! the landunit and gridcell and model

    call dealloc_column_estate_type(    clm3%g%l%c%ces)
    call dealloc_column_estate_type(    clm3%g%l%les%ces_a)
    call dealloc_column_estate_type(    clm3%g%ges%ces_a)
    call dealloc_column_estate_type(          clm3%mes%ces_a)
    
    ! column water state variables at column level and averaged to
    ! the landunit and gridcell and model

    call dealloc_column_wstate_type(    clm3%g%l%c%cws)
    call dealloc_column_wstate_type(    clm3%g%l%lws%cws_a)
    call dealloc_column_wstate_type(    clm3%g%gws%cws_a)
    call dealloc_column_wstate_type(          clm3%mws%cws_a)

    ! column carbon state variables at column level and averaged to
    ! the landunit and gridcell and model

    call dealloc_column_cstate_type(    clm3%g%l%c%ccs)
    call dealloc_column_cstate_type(    clm3%g%l%lcs%ccs_a)
    call dealloc_column_cstate_type(    clm3%g%gcs%ccs_a)
    call dealloc_column_cstate_type(          clm3%mcs%ccs_a)
#if (defined C13)
    ! 4/14/05: PET
    ! Adding isotope code
    call dealloc_column_cstate_type(    clm3%g%l%c%cc13s)
#endif

    ! column nitrogen state variables at column level and averaged to
    ! the landunit and gridcell and model
    
    call dealloc_column_nstate_type(    clm3%g%l%c%cns)
    call dealloc_column_nstate_type(    clm3%g%l%lns%cns_a)
    call dealloc_column_nstate_type(    clm3%g%gns%cns_a)
    call dealloc_column_nstate_type(          clm3%mns%cns_a)

    ! column energy flux variables at column level and averaged to
    ! the landunit and gridcell and model

    call dealloc_column_eflux_type(    clm3%g%l%c%cef)
    call dealloc_column_eflux_type(    clm3%g%l%lef%cef_a)
    call dealloc_column_eflux_type(    clm3%g%gef%cef_a)
    call dealloc_column_eflux_type(          clm3%mef%cef_a)

    ! column water flux variables at column level and averaged to
    ! landunit, gridcell and model level

    call dealloc_column_wflux_type(    clm3%g%l%c%cwf)
    call dealloc_column_wflux_type(    clm3%g%l%lwf%cwf_a)
    call dealloc_column_wflux_type(    clm3%g%gwf%cwf_a)
    call dealloc_column_wflux_type(          clm3%mwf%cwf_a)
    
    ! column carbon flux variables at column level

    call dealloc_column_cflux_type(    clm3%g%l%c%ccf)
#if (defined C13)
    ! 4/14/05: PET
    ! Adding isotope code
    call dealloc_column_cflux_type(    clm3%g%l%c%cc13f)
#endif

    ! column nitrogen flux variables at column level

    call dealloc_column_nflux_type(    clm3%g%l%c%cnf)

    ! land unit physical state variables

    call dealloc_landunit_pstate_type(    clm3%g%l%lps)
   call CLMDebug('mark1')
    ! land unit energy flux variables 

    call dealloc_landunit_eflux_type(    clm3%g%l%lef)

#if (defined CNDV)
    ! gridcell DGVM variables

    call dealloc_gridcell_dgvstate_type(    clm3%g%gdgvs)
#endif

    ! gridcell physical state variables

    call dealloc_gridcell_pstate_type(    clm3%g%gps)

    ! gridcell: water flux variables

    call dealloc_gridcell_wflux_type(    clm3%g%gwf)

    ! gridcell: energy flux variables

    call dealloc_gridcell_eflux_type(    clm3%g%gef)

       call CLMDebug('mark2')

    ! gridcell: water state variables

    call dealloc_gridcell_wstate_type(    clm3%g%gws)

    ! gridcell: energy state variables

    call dealloc_gridcell_estate_type(    clm3%g%ges)

     call CLMDebug('mark3')

    call dealloc_atm2lnd_type  ( clm_a2l)

     call CLMDebug('done clmtype_dealloc')

end subroutine clmtype_dealloc


!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_type
!
! !INTERFACE:
  subroutine dealloc_pft_type (    p)
!
    implicit none
    type(pft_type), intent(inout):: p

    deallocate(p%gridcell ,p%wtgcell )
    deallocate(p%landunit ,p%wtlunit )
    deallocate(p%column   ,p%wtcol   )

    deallocate(p%itype )
    deallocate(p%mxy )
    deallocate(p%area)

  end subroutine dealloc_pft_type

! !IROUTINE: dealloc_column_type
!
! !INTERFACE:
  subroutine dealloc_column_type (    c)
!
  implicit none
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------
    type(column_type), intent(inout):: c

   deallocate(c%gridcell ,c%wtgcell )
   deallocate(c%landunit ,c%wtlunit )    

   deallocate(c%pfti ,c%pftf ,c%npfts )

   deallocate(c%itype )

   deallocate(c%area)

  end subroutine dealloc_column_type

!------------------------------------------------------------------------
!BOP
! 
! !IROUTINE: dealloc_landunit_type
!
! !INTERFACE:
  subroutine dealloc_landunit_type (   l)
!
! !DESCRIPTION:
! Initialize components of landunit_type structure
! 
! !ARGUMENTS:
    implicit none
    type(landunit_type), intent(inout):: l
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!   
!EOP
!------------------------------------------------------------------------

   deallocate(l%gridcell ,l%wtgcell )

   deallocate(l%coli ,l%colf ,l%ncolumns )
   deallocate(l%pfti ,l%pftf ,l%npfts    )

   deallocate(l%itype )
   deallocate(l%ifspecial )
   deallocate(l%lakpoi )
   deallocate(l%urbpoi )

   ! MV - these should be moved to landunit physical state -MV
   deallocate(l%canyon_hwr )
   deallocate(l%wtroad_perv )
   deallocate(l%ht_roof )
   deallocate(l%wtlunit_roof )
   deallocate(l%wind_hgt_canyon )
   deallocate(l%z_0_town )
   deallocate(l%z_d_town )

   deallocate(l%area)


  end subroutine dealloc_landunit_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_gridcell_type
!
! !INTERFACE:
  subroutine dealloc_gridcell_type (   g)
!
! !DESCRIPTION:
! Initialize components of gridcell_type structure
!
! !ARGUMENTS:
    implicit none
    type(gridcell_type), intent(inout):: g
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

   deallocate(g%luni ,g%lunf ,g%nlandunits )
   deallocate(g%coli ,g%colf ,g%ncolumns   )
   deallocate(g%pfti ,g%pftf ,g%npfts      )

   deallocate(g%gindex )
   deallocate(g%area )
   deallocate(g%lat )
   deallocate(g%lon )
   deallocate(g%latdeg )
   deallocate(g%londeg )
   deallocate(g%gindex_a )
   deallocate(g%lat_a )
   deallocate(g%lon_a )
   deallocate(g%latdeg_a )
   deallocate(g%londeg_a )

  end subroutine dealloc_gridcell_type

!------------------------------------------------------------------------
!BOP
!
                                                                               ! !IROUTINE: dealloc_energy_balance_type
!  
! !INTERFACE:
  subroutine dealloc_energy_balance_type(    ebal)
!
! !DESCRIPTION:
! Initialize energy balance variables
!  
! !ARGUMENTS:
    implicit none
    type(energy_balance_type), intent(inout):: ebal
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!  
!EOP
!------------------------------------------------------------------------
   
    deallocate(ebal%errsoi )
    deallocate(ebal%errseb )
    deallocate(ebal%errsol )
    deallocate(ebal%errlon )


  end subroutine dealloc_energy_balance_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_water_balance_type
!   
! !INTERFACE:
  subroutine dealloc_water_balance_type(    wbal)
!
! !DESCRIPTION:
! Initialize water balance variables
!
! !ARGUMENTS:
    implicit none
    type(water_balance_type), intent(inout):: wbal
!  
! !REVISION HISTORY: 
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------
   
    deallocate(wbal%begwb )
    deallocate(wbal%endwb )
    deallocate(wbal%errh2o )
   

  end subroutine dealloc_water_balance_type

!------------------------------------------------------------------------
!BOP
! !IROUTINE: dealloc_carbon_balance_type
!
! !INTERFACE:
  subroutine dealloc_carbon_balance_type(    cbal)
!
! !DESCRIPTION:
! Initialize carbon balance variables
!
! !ARGUMENTS:
    implicit none
    type(carbon_balance_type), intent(inout):: cbal
!
! !REVISION HISTORY:
! Created by Peter Thornton, 12/11/2003
!
!EOP
!------------------------------------------------------------------------

    deallocate(cbal%begcb )
    deallocate(cbal%endcb )
    deallocate(cbal%errcb )


  end subroutine dealloc_carbon_balance_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_nitrogen_balance_type
!
! !INTERFACE:
  subroutine dealloc_nitrogen_balance_type(    nbal)
!
! !DESCRIPTION:
! Initialize nitrogen balance variables
!
! !ARGUMENTS:
    implicit none
    type(nitrogen_balance_type), intent(inout):: nbal
!
! !REVISION HISTORY:
! Created by Peter Thornton, 12/11/2003
!
!EOP
!------------------------------------------------------------------------

    deallocate(nbal%begnb )
    deallocate(nbal%endnb )
    deallocate(nbal%errnb )


  end subroutine dealloc_nitrogen_balance_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_ecophys_constants
! !INTERFACE:
  subroutine dealloc_pft_ecophys_constants()
!
! !DESCRIPTION:
! Initialize pft physical state
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!   
!EOP
!------------------------------------------------------------------------

    deallocate(pftcon%noveg )
    deallocate(pftcon%tree )
    deallocate(pftcon%smpso ) 
    deallocate(pftcon%smpsc )
    deallocate(pftcon%fnitr )
    deallocate(pftcon%foln )
    deallocate(pftcon%dleaf )
    deallocate(pftcon%c3psn )
    deallocate(pftcon%vcmx25 )
    deallocate(pftcon%mp )
    deallocate(pftcon%qe25 )
    deallocate(pftcon%xl )
    deallocate(pftcon%rhol)
    deallocate(pftcon%rhos)
    deallocate(pftcon%taul)
    deallocate(pftcon%taus)
    deallocate(pftcon%z0mr )
    deallocate(pftcon%displar )
    deallocate(pftcon%roota_par )
    deallocate(pftcon%rootb_par )
    deallocate(pftcon%sla )
    deallocate(pftcon%slatop )
    deallocate(pftcon%dsladlai )
    deallocate(pftcon%leafcn )
    deallocate(pftcon%flnr )
    deallocate(pftcon%woody )
    deallocate(pftcon%lflitcn )
    deallocate(pftcon%frootcn )
    deallocate(pftcon%livewdcn ) 
    deallocate(pftcon%deadwdcn )
#ifdef CROP
    deallocate(pftcon%graincn )
#endif
    deallocate(pftcon%froot_leaf )
    deallocate(pftcon%stem_leaf )
    deallocate(pftcon%croot_stem )
    deallocate(pftcon%flivewd )
    deallocate(pftcon%fcur )
    deallocate(pftcon%lf_flab )
    deallocate(pftcon%lf_fcel )
    deallocate(pftcon%lf_flig )
    deallocate(pftcon%fr_flab )
    deallocate(pftcon%fr_fcel )
    deallocate(pftcon%fr_flig )
    deallocate(pftcon%dw_fcel )
    deallocate(pftcon%dw_flig )
    deallocate(pftcon%leaf_long )
    deallocate(pftcon%evergreen )
    deallocate(pftcon%stress_decid )
    deallocate(pftcon%season_decid )
    deallocate(pftcon%resist )
    deallocate(pftcon%dwood )


  end subroutine dealloc_pft_ecophys_constants

#if (defined CNDV) || (defined CROP)
!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_DGVMecophys_constants
!
! !INTERFACE:
  subroutine dealloc_pft_DGVMecophys_constants()
! !DESCRIPTION:
! Initialize pft physical state
!
! !ARGUMENTS:
    implicit none
!   
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!   
!EOP
!------------------------------------------------------------------------
    
    deallocate(dgv_pftcon%crownarea_max )
    deallocate(dgv_pftcon%tcmin )
    deallocate(dgv_pftcon%tcmax )
    deallocate(dgv_pftcon%gddmin )
    deallocate(dgv_pftcon%twmax )
    deallocate(dgv_pftcon%reinickerp )
    deallocate(dgv_pftcon%allom1 )
    deallocate(dgv_pftcon%allom2 )
    deallocate(dgv_pftcon%allom3 )
    
    
  end subroutine dealloc_pft_DGVMecophys_constants
#endif

!------------------------------------------------------------------------
!BOP
!   
! !IROUTINE: dealloc_pft_pstate_type
!   
! !INTERFACE:
  subroutine dealloc_pft_pstate_type(    pps)
!   
! !DESCRIPTION:
! Initialize pft physical state
!   
! !USES:
    use clm_varcon, only : spval
#if (defined CASA)
    use CASAMod   , only : npools, nresp_pools, nlive, npool_types
#endif
! !ARGUMENTS:
    implicit none
    type (pft_pstate_type), intent(inout):: pps
! 
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(pps%frac_veg_nosno )
    deallocate(pps%frac_veg_nosno_alb )
    deallocate(pps%emv )
    deallocate(pps%z0mv )
    deallocate(pps%z0hv )
    deallocate(pps%z0qv )
    deallocate(pps%rootfr )
    deallocate(pps%rootr )
    deallocate(pps%rresis )
    deallocate(pps%dewmx )
    deallocate(pps%rssun )
    deallocate(pps%rssha )
    deallocate(pps%laisun )
    deallocate(pps%laisha )
    deallocate(pps%btran )
    deallocate(pps%fsun )
    deallocate(pps%tlai )
    deallocate(pps%tsai )
    deallocate(pps%elai )
    deallocate(pps%esai )
    deallocate(pps%fwet )
    deallocate(pps%fdry )
    deallocate(pps%dt_veg )
    deallocate(pps%htop )
    deallocate(pps%hbot )
    deallocate(pps%z0m )
    deallocate(pps%displa )
    deallocate(pps%albd )
    deallocate(pps%albi )
    deallocate(pps%fabd )
    deallocate(pps%fabi )
    deallocate(pps%ftdd )
    deallocate(pps%ftid )
    deallocate(pps%ftii )
    deallocate(pps%u10 )
    deallocate(pps%fv )
    deallocate(pps%ram1 )
#if (defined CROP)
    deallocate(pps%hdidx )
    deallocate(pps%cumvd )
    deallocate(pps%htmx )
    deallocate(pps%vf )
    deallocate(pps%gddmaturity )
    deallocate(pps%gdd0 )
    deallocate(pps%gdd8 )
    deallocate(pps%gdd10 )
    deallocate(pps%gdd020 )
    deallocate(pps%gdd820 )
    deallocate(pps%gdd1020 )
    deallocate(pps%gddplant )
    deallocate(pps%gddtsoi )
    deallocate(pps%huileaf )
    deallocate(pps%huigrain )
    deallocate(pps%a10tmin )
    deallocate(pps%a5tmin )
    deallocate(pps%aleafi )
    deallocate(pps%astemi )
    deallocate(pps%aleaf )
    deallocate(pps%astem )
    deallocate(pps%croplive )
    deallocate(pps%cropplant ) !,numpft)) ! make 2-D if using
    deallocate(pps%harvdate )  !,numpft)) ! crop rotation
    deallocate(pps%idop )
    deallocate(pps%peaklai )
#endif
    deallocate(pps%vds )
    deallocate(pps%slasun )
    deallocate(pps%slasha )
    deallocate(pps%lncsun )
    deallocate(pps%lncsha )
    deallocate(pps%vcmxsun )
    deallocate(pps%vcmxsha )
    deallocate(pps%gdir ) 
    deallocate(pps%omega )
    deallocate(pps%eff_kid )
    deallocate(pps%eff_kii )
    deallocate(pps%sun_faid )
    deallocate(pps%sun_faii )
    deallocate(pps%sha_faid )
    deallocate(pps%sha_faii )
    deallocate(pps%forc_hgt_u_pft )
    deallocate(pps%forc_hgt_t_pft )
    deallocate(pps%forc_hgt_q_pft )
    ! 4/14/05: PET
    ! Adding isotope code
    deallocate(pps%cisun )
    deallocate(pps%cisha )
#if (defined C13)
    deallocate(pps%alphapsnsun )
    deallocate(pps%alphapsnsha )
#endif
    ! heald: added from CASA defdeallocion
    deallocate(pps%sandfrac )
    deallocate(pps%clayfrac )
    deallocate(pps%mlaidiff )
    deallocate(pps%rb1 )
    deallocate(pps%annlai)
    
    
#if (defined CASA)
    deallocate(pps%Closs)  ! C lost to atm
    deallocate(pps%Ctrans)  ! C transfers out of pool types
    deallocate(pps%Resp_C)
    deallocate(pps%Tpool_C)! Total C pool size
    deallocate(pps%eff)
    deallocate(pps%frac_donor)
    deallocate(pps%livefr)  !live fraction
    deallocate(pps%pet )           !potential evaporation (mm h2o/s)
    deallocate(pps%co2flux )       ! net CO2 flux (g C/m2/sec) [+= atm]
    deallocate(pps%fnpp )          ! NPP  (g C/m2/sec)
    deallocate(pps%soilt )         !soil temp for top 30cm
    deallocate(pps%smoist )        !soil moisture for top 30cm
    deallocate(pps%sz )
    deallocate(pps%watopt ) 
    deallocate(pps%watdry )
    deallocate(pps%soiltc )         !soil temp for entire column
    deallocate(pps%smoistc )        !soil moisture for entire column
    deallocate(pps%szc )
    deallocate(pps%watoptc )
    deallocate(pps%watdryc )
    deallocate(pps%Wlim )
    deallocate(pps%litterscalar )
    deallocate(pps%rootlitscalar )
    deallocate(pps%stressCD )
    deallocate(pps%excessC )       ! excess Carbon (gC/m2/timestep)
    deallocate(pps%bgtemp )
    deallocate(pps%bgmoist )
    deallocate(pps%plai )          ! prognostic LAI (m2 leaf/m2 ground)
    deallocate(pps%Cflux )
    deallocate(pps%XSCpool )
    deallocate(pps%tday )     ! daily accumulated temperature (deg C)
    deallocate(pps%tdayavg )  ! daily averaged temperature (deg C)
    deallocate(pps%tcount )   ! counter for daily avg temp
    deallocate(pps%degday )   ! accumulated degree days (deg C)
    deallocate(pps%ndegday )  ! counter for number of degree days
    deallocate(pps%stressT )
    deallocate(pps%stressW )  ! water stress function for leaf loss
    deallocate(pps%iseabeg )  ! index for start of growing season
    deallocate(pps%nstepbeg ) ! nstep at start of growing season
    deallocate(pps%lgrow )    ! growing season index (0 or 1) to be
                                    ! passed daily to CASA to get NPP
#if (defined CLAMP)
    ! Summary variables added for the C-LAMP Experiments
    deallocate(pps%casa_agnpp )
    deallocate(pps%casa_ar )
    deallocate(pps%casa_bgnpp )
    deallocate(pps%casa_cwdc )
    deallocate(pps%casa_cwdc_hr )
    deallocate(pps%casa_cwdc_loss )
    deallocate(pps%casa_frootc )
    deallocate(pps%casa_frootc_alloc )
    deallocate(pps%casa_frootc_loss )
    deallocate(pps%casa_gpp )
    deallocate(pps%casa_hr )
    deallocate(pps%casa_leafc )
    deallocate(pps%casa_leafc_alloc )
    deallocate(pps%casa_leafc_loss )
    deallocate(pps%casa_litterc )
    deallocate(pps%casa_litterc_hr )
    deallocate(pps%casa_litterc_loss )
    deallocate(pps%casa_nee )
    deallocate(pps%casa_nep )
    deallocate(pps%casa_npp )
    deallocate(pps%casa_soilc )
    deallocate(pps%casa_soilc_hr )
    deallocate(pps%casa_soilc_loss )
    deallocate(pps%casa_woodc )
    deallocate(pps%casa_woodc_alloc )
    deallocate(pps%casa_woodc_loss )
#endif
#endif
  end subroutine dealloc_pft_pstate_type
    
!------------------------------------------------------------------------
!BOP
!   
! !IROUTINE: dealloc_pft_epv_type
!   
! !INTERFACE: 
  subroutine dealloc_pft_epv_type(    pepv)
!   
! !DESCRIPTION:  
! Initialize pft ecophysiological variables
!   
! !ARGUMENTS:
    implicit none 
    type (pft_epv_type), intent(inout):: pepv
!   
! !REVISION HISTORY:
! Created by Peter Thornton
!   
!EOP
!------------------------------------------------------------------------
    
    deallocate(pepv%dormant_flag )
    deallocate(pepv%days_active )
    deallocate(pepv%onset_flag )
    deallocate(pepv%onset_counter )
    deallocate(pepv%onset_gddflag )
    deallocate(pepv%onset_fdd )
    deallocate(pepv%onset_gdd )
    deallocate(pepv%onset_swi )
    deallocate(pepv%offset_flag )
    deallocate(pepv%offset_counter )
    deallocate(pepv%offset_fdd )
    deallocate(pepv%offset_swi )
    deallocate(pepv%lgsf )
    deallocate(pepv%bglfr )
    deallocate(pepv%bgtr )
    deallocate(pepv%dayl )
    deallocate(pepv%prev_dayl )
    deallocate(pepv%annavg_t2m )
    deallocate(pepv%tempavg_t2m )
    deallocate(pepv%gpp ) 
    deallocate(pepv%availc )
    deallocate(pepv%xsmrpool_recover )
#if (defined C13)
    deallocate(pepv%xsmrpool_c13ratio )
#endif
    deallocate(pepv%alloc_pnow )
    deallocate(pepv%c_allometry )
    deallocate(pepv%n_allometry )
    deallocate(pepv%plant_ndemand )
    deallocate(pepv%tempsum_potential_gpp )
    deallocate(pepv%annsum_potential_gpp )
    deallocate(pepv%tempmax_retransn )
    deallocate(pepv%annmax_retransn )
    deallocate(pepv%avail_retransn )
    deallocate(pepv%plant_nalloc )
    deallocate(pepv%plant_calloc )
    deallocate(pepv%excess_cflux )
    deallocate(pepv%downreg )
    deallocate(pepv%prev_leafc_to_litter )
    deallocate(pepv%prev_frootc_to_litter )
    deallocate(pepv%tempsum_npp )
    deallocate(pepv%annsum_npp )
#if (defined CNDV)
    deallocate(pepv%tempsum_litfall )
    deallocate(pepv%annsum_litfall )
#endif
#if (defined C13)
    ! 4/21/05, PET
    ! Adding isotope code
    deallocate(pepv%rc13_canair )
    deallocate(pepv%rc13_psnsun )
    deallocate(pepv%rc13_psnsha )
#endif

    
  end subroutine dealloc_pft_epv_type
    
#if (defined CNDV) || (defined CROP)
!------------------------------------------------------------------------
!BOP
!   
! !IROUTINE: dealloc_pft_pdgvstate_type
!   
! !INTERFACE:
  subroutine dealloc_pft_pdgvstate_type(    pdgvs)
!
! !DESCRIPTION:
! Initialize pft DGVM state variables
!   
! !ARGUMENTS:
    implicit none
    type (pft_dgvstate_type), intent(inout):: pdgvs
!   
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!   
!EOP
!------------------------------------------------------------------------
    
    deallocate(pdgvs%agddtw )
    deallocate(pdgvs%agdd )
    deallocate(pdgvs%t10 )
    deallocate(pdgvs%t_mo )
    deallocate(pdgvs%t_mo_min )
    deallocate(pdgvs%prec365 )
    deallocate(pdgvs%present )
    deallocate(pdgvs%pftmayexist )
    deallocate(pdgvs%nind )
    deallocate(pdgvs%lm_ind )
    deallocate(pdgvs%lai_ind )
    deallocate(pdgvs%fpcinc )
    deallocate(pdgvs%fpcgrid )
    deallocate(pdgvs%fpcgridold )
    deallocate(pdgvs%crownarea )
    deallocate(pdgvs%greffic )
    deallocate(pdgvs%heatstress ) 
  end subroutine dealloc_pft_pdgvstate_type
#endif
!
! !IROUTINE: dealloc_pft_vstate_type
!
! !INTERFACE:
  subroutine dealloc_pft_vstate_type(    pvs)
!
! !DESCRIPTION:
! Initialize pft VOC variables
!
! !USES:
! !ARGUMENTS:
    implicit none
    type (pft_vstate_type), intent(inout):: pvs
!
! !REVISION HISTORY:
! Created by Erik Kluzek
!
!EOP
!------------------------------------------------------------------------

    deallocate(pvs%t_veg24  )
    deallocate(pvs%t_veg240 )
    deallocate(pvs%fsd24    )
    deallocate(pvs%fsd240   )
    deallocate(pvs%fsi24    )
    deallocate(pvs%fsi240   )
    deallocate(pvs%fsun24   )
    deallocate(pvs%fsun240  )
    deallocate(pvs%elai_p   )

  end subroutine dealloc_pft_vstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_estate_type
!
! !INTERFACE:
  subroutine dealloc_pft_estate_type(    pes)
!
! !DESCRIPTION:
! Initialize pft energy state
!
! !ARGUMENTS:
    implicit none
    type (pft_estate_type), intent(inout):: pes
!
! !REVISION HISTORY:
    deallocate(pes%t_ref2m )
    deallocate(pes%t_ref2m_min )
    deallocate(pes%t_ref2m_max )
    deallocate(pes%t_ref2m_min_inst )
    deallocate(pes%t_ref2m_max_inst )
    deallocate(pes%q_ref2m )
    deallocate(pes%t_ref2m_u )
    deallocate(pes%t_ref2m_r )
    deallocate(pes%t_ref2m_min_u )
    deallocate(pes%t_ref2m_min_r )
    deallocate(pes%t_ref2m_max_u )
    deallocate(pes%t_ref2m_max_r )
    deallocate(pes%t_ref2m_min_inst_u )
    deallocate(pes%t_ref2m_min_inst_r )
    deallocate(pes%t_ref2m_max_inst_u )
    deallocate(pes%t_ref2m_max_inst_r )
    deallocate(pes%rh_ref2m )
    deallocate(pes%rh_ref2m_u )
    deallocate(pes%rh_ref2m_r )
    deallocate(pes%t_veg )
    deallocate(pes%thm )


  end subroutine dealloc_pft_estate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_wstate_type
! 
! !INTERFACE:
  subroutine dealloc_pft_wstate_type(    pws)
!
! !DESCRIPTION:
! Initialize pft water state
!   
! !ARGUMENTS:
    implicit none
    type (pft_wstate_type), intent(inout):: pws !pft water state
                                                                                  !
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(pws%h2ocan )

  end subroutine dealloc_pft_wstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_cstate_type
!
! !INTERFACE:
  subroutine dealloc_pft_cstate_type(    pcs)
!
! !DESCRIPTION:
! Initialize pft carbon state
!
! !ARGUMENTS:
    implicit none
    type (pft_cstate_type), intent(inout):: pcs !pft carbon state
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

    deallocate(pcs%leafc )
    deallocate(pcs%leafc_storage )
    deallocate(pcs%leafc_xfer )
    deallocate(pcs%frootc )
    deallocate(pcs%frootc_storage )
    deallocate(pcs%frootc_xfer )
    deallocate(pcs%livestemc )
    deallocate(pcs%livestemc_storage )
    deallocate(pcs%livestemc_xfer )
    deallocate(pcs%deadstemc )
    deallocate(pcs%deadstemc_storage )
    deallocate(pcs%deadstemc_xfer )
    deallocate(pcs%livecrootc )
    deallocate(pcs%livecrootc_storage )
    deallocate(pcs%livecrootc_xfer )
    deallocate(pcs%deadcrootc )
    deallocate(pcs%deadcrootc_storage )
    deallocate(pcs%deadcrootc_xfer )
    deallocate(pcs%gresp_storage )
    deallocate(pcs%gresp_xfer )
    deallocate(pcs%cpool )
    deallocate(pcs%xsmrpool )
    deallocate(pcs%pft_ctrunc )
    deallocate(pcs%dispvegc )
    deallocate(pcs%storvegc )
    deallocate(pcs%totvegc )
    deallocate(pcs%totpftc )
    deallocate(pcs%leafcmax )
#if (defined CROP)
    deallocate(pcs%grainc )
    deallocate(pcs%grainc_storage )
    deallocate(pcs%grainc_xfer )
#endif
#if (defined CLAMP) && (defined CN)
    !CLAMP
    deallocate(pcs%woodc )
#endif

    
  end subroutine dealloc_pft_cstate_type
    
!------------------------------------------------------------------------
!BOP
!   
! !IROUTINE: dealloc_pft_nstate_type
!   
! !INTERFACE:
  subroutine dealloc_pft_nstate_type(    pns)
!   
! !DESCRIPTION:
! Initialize pft nitrogen state
!   
! !ARGUMENTS:
    implicit none
    type (pft_nstate_type), intent(inout):: pns !pft nitrogen state
!   
! !REVISION HISTORY:
! Created by Peter Thornton

#if (defined CROP)
    deallocate(pns%grainn )
    deallocate(pns%grainn_storage )
    deallocate(pns%grainn_xfer )
#endif
    deallocate(pns%leafn )
    deallocate(pns%leafn_storage )
    deallocate(pns%leafn_xfer )
    deallocate(pns%frootn )
    deallocate(pns%frootn_storage )
    deallocate(pns%frootn_xfer )
    deallocate(pns%livestemn )
    deallocate(pns%livestemn_storage )
    deallocate(pns%livestemn_xfer )
    deallocate(pns%deadstemn )
    deallocate(pns%deadstemn_storage )
    deallocate(pns%deadstemn_xfer )
    deallocate(pns%livecrootn )
    deallocate(pns%livecrootn_storage )
    deallocate(pns%livecrootn_xfer )
    deallocate(pns%deadcrootn )
    deallocate(pns%deadcrootn_storage )
    deallocate(pns%deadcrootn_xfer )
    deallocate(pns%retransn )
    deallocate(pns%npool )
    deallocate(pns%pft_ntrunc )
    deallocate(pns%dispvegn )
    deallocate(pns%storvegn )
    deallocate(pns%totvegn )
    deallocate(pns%totpftn )
 end subroutine dealloc_pft_nstate_type
! !IROUTINE: dealloc_pft_eflux_type
!   
! !INTERFACE:
  subroutine dealloc_pft_eflux_type(    pef)
!
! !DESCRIPTION:
! Initialize pft energy flux variables
!   
! !ARGUMENTS:
    implicit none
    type (pft_eflux_type), intent(inout):: pef
!   
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!   
!EOP
!------------------------------------------------------------------------
    
    deallocate(pef%sabg )
    deallocate(pef%sabv )
    deallocate(pef%fsa )
    deallocate(pef%fsa_u )
    deallocate(pef%fsa_r )
    deallocate(pef%fsr ) 
    deallocate(pef%parsun )
    deallocate(pef%parsha ) 
    deallocate(pef%dlrad )
    deallocate(pef%ulrad ) 
    deallocate(pef%eflx_lh_tot )
    deallocate(pef%eflx_lh_tot_u )
    deallocate(pef%eflx_lh_tot_r )
    deallocate(pef%eflx_lh_grnd )
    deallocate(pef%eflx_soil_grnd )
    deallocate(pef%eflx_soil_grnd_u )
    deallocate(pef%eflx_soil_grnd_r )
    deallocate(pef%eflx_sh_tot )
    deallocate(pef%eflx_sh_tot_u )
    deallocate(pef%eflx_sh_tot_r )
    deallocate(pef%eflx_sh_grnd )
    deallocate(pef%eflx_sh_veg )
    deallocate(pef%eflx_lh_vege )
    deallocate(pef%eflx_lh_vegt )
    deallocate(pef%eflx_wasteheat_pft )
    deallocate(pef%eflx_heat_from_ac_pft )
    deallocate(pef%eflx_traffic_pft )
    deallocate(pef%eflx_anthro )
    deallocate(pef%cgrnd )
    deallocate(pef%cgrndl )
    deallocate(pef%cgrnds ) 
    deallocate(pef%eflx_gnet )
    deallocate(pef%dgnetdT )
    deallocate(pef%eflx_lwrad_out )
    deallocate(pef%eflx_lwrad_net )
    deallocate(pef%eflx_lwrad_net_u )
    deallocate(pef%eflx_lwrad_net_r )
    deallocate(pef%netrad )
    deallocate(pef%fsds_vis_d )
    deallocate(pef%fsds_nir_d )
    deallocate(pef%fsds_vis_i )
    deallocate(pef%fsds_nir_i )
    deallocate(pef%fsr_vis_d )
    deallocate(pef%fsr_nir_d )
    deallocate(pef%fsr_vis_i )
                                                                         
    deallocate(pef%fsr_nir_i )
    deallocate(pef%fsds_vis_d_ln )
    deallocate(pef%fsds_nir_d_ln )
    deallocate(pef%fsr_vis_d_ln )
    deallocate(pef%fsr_nir_d_ln )
    deallocate(pef%sun_add )
    deallocate(pef%tot_aid )
    deallocate(pef%sun_aid )
    deallocate(pef%sun_aii )
    deallocate(pef%sha_aid )
    deallocate(pef%sha_aii )
    deallocate(pef%sun_atot )
    deallocate(pef%sha_atot )
    deallocate(pef%sun_alf )
    deallocate(pef%sha_alf )
    deallocate(pef%sun_aperlai )
    deallocate(pef%sha_aperlai )
    deallocate(pef%sabg_lyr)
    deallocate(pef%sfc_frc_aer )
    deallocate(pef%sfc_frc_bc )
    deallocate(pef%sfc_frc_oc )
    deallocate(pef%sfc_frc_dst )
    deallocate(pef%sfc_frc_aer_sno )
    deallocate(pef%sfc_frc_bc_sno )
    deallocate(pef%sfc_frc_oc_sno )
    deallocate(pef%sfc_frc_dst_sno )
    deallocate(pef%fsr_sno_vd )
    deallocate(pef%fsr_sno_nd )
    deallocate(pef%fsr_sno_vi )
    deallocate(pef%fsr_sno_ni )
    deallocate(pef%fsds_sno_vd )
    deallocate(pef%fsds_sno_nd )
    deallocate(pef%fsds_sno_vi )
    deallocate(pef%fsds_sno_ni )
  end subroutine dealloc_pft_eflux_type
    
!------------------------------------------------------------------------
!BOP
!   
! !IROUTINE: dealloc_pft_mflux_type
!   
! !INTERFACE:
  subroutine dealloc_pft_mflux_type(    pmf)
!   
! !DESCRIPTION:
! Initialize pft momentum flux variables
!   
! !ARGUMENTS:
    implicit none
    type (pft_mflux_type), intent(inout) :: pmf
   deallocate(pmf%taux )
    deallocate(pmf%tauy )


  end subroutine dealloc_pft_mflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_wflux_type
!
! !INTERFACE:
  subroutine dealloc_pft_wflux_type(    pwf)
!
! !DESCRIPTION:
! Initialize pft water flux variables
!
! !ARGUMENTS:
    implicit none
    type (pft_wflux_type), intent(inout) :: pwf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(pwf%qflx_prec_intr )
    deallocate(pwf%qflx_prec_grnd )
    deallocate(pwf%qflx_rain_grnd )
    deallocate(pwf%qflx_snow_grnd )
    deallocate(pwf%qflx_snwcp_liq )
    deallocate(pwf%qflx_snwcp_ice )
    deallocate(pwf%qflx_evap_veg )
    deallocate(pwf%qflx_tran_veg )
    deallocate(pwf%qflx_evap_can )
    deallocate(pwf%qflx_evap_soi )
    deallocate(pwf%qflx_evap_tot )
    deallocate(pwf%qflx_evap_grnd )
    deallocate(pwf%qflx_dew_grnd )
    deallocate(pwf%qflx_sub_snow )
    deallocate(pwf%qflx_dew_snow )

  end subroutine dealloc_pft_wflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_cflux_type
!
! !INTERFACE: 
  subroutine dealloc_pft_cflux_type(    pcf)
!
! !DESCRIPTION:
! Initialize pft carbon flux variables
!
! !ARGUMENTS:
    implicit none
    type (pft_cflux_type), intent(inout) :: pcf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(pcf%psnsun )
    deallocate(pcf%psnsha )  
    deallocate(pcf%fpsn )
    deallocate(pcf%fco2 )

    deallocate(pcf%m_leafc_to_litter )
    deallocate(pcf%m_frootc_to_litter )
    deallocate(pcf%m_leafc_storage_to_litter )
    deallocate(pcf%m_frootc_storage_to_litter )
    deallocate(pcf%m_livestemc_storage_to_litter )
    deallocate(pcf%m_deadstemc_storage_to_litter )
    deallocate(pcf%m_livecrootc_storage_to_litter )
    deallocate(pcf%m_deadcrootc_storage_to_litter )
    deallocate(pcf%m_leafc_xfer_to_litter )
    deallocate(pcf%m_frootc_xfer_to_litter )
    deallocate(pcf%m_livestemc_xfer_to_litter )
    deallocate(pcf%m_deadstemc_xfer_to_litter )
    deallocate(pcf%m_livecrootc_xfer_to_litter )
    deallocate(pcf%m_deadcrootc_xfer_to_litter )
    deallocate(pcf%m_livestemc_to_litter )
    deallocate(pcf%m_deadstemc_to_litter )
    deallocate(pcf%m_livecrootc_to_litter )
    deallocate(pcf%m_deadcrootc_to_litter )
    deallocate(pcf%m_gresp_storage_to_litter )
    deallocate(pcf%m_gresp_xfer_to_litter )
    deallocate(pcf%hrv_leafc_to_litter )
    deallocate(pcf%hrv_leafc_storage_to_litter )
    deallocate(pcf%hrv_leafc_xfer_to_litter )
    deallocate(pcf%hrv_frootc_to_litter )
    deallocate(pcf%hrv_frootc_storage_to_litter )
    deallocate(pcf%hrv_frootc_xfer_to_litter )
    deallocate(pcf%hrv_livestemc_to_litter )
    deallocate(pcf%hrv_livestemc_storage_to_litter )
    deallocate(pcf%hrv_livestemc_xfer_to_litter )
    deallocate(pcf%hrv_deadstemc_to_prod10c )
    deallocate(pcf%hrv_deadstemc_to_prod100c )
    deallocate(pcf%hrv_deadstemc_storage_to_litter )
    deallocate(pcf%hrv_deadstemc_xfer_to_litter )
    deallocate(pcf%hrv_livecrootc_to_litter )
    deallocate(pcf%hrv_livecrootc_storage_to_litter )
    deallocate(pcf%hrv_livecrootc_xfer_to_litter )
    deallocate(pcf%hrv_deadcrootc_to_litter )
    deallocate(pcf%hrv_deadcrootc_storage_to_litter )
    deallocate(pcf%hrv_deadcrootc_xfer_to_litter )
    deallocate(pcf%hrv_gresp_storage_to_litter )
    deallocate(pcf%hrv_gresp_xfer_to_litter )
    deallocate(pcf%hrv_xsmrpool_to_atm )
    deallocate(pcf%m_leafc_to_fire )
    deallocate(pcf%m_frootc_to_fire )
    deallocate(pcf%m_leafc_storage_to_fire )
    deallocate(pcf%m_frootc_storage_to_fire )
    deallocate(pcf%m_livestemc_storage_to_fire )
    deallocate(pcf%m_deadstemc_storage_to_fire )
    deallocate(pcf%m_livecrootc_storage_to_fire )
    deallocate(pcf%m_deadcrootc_storage_to_fire )
    deallocate(pcf%m_leafc_xfer_to_fire )
    deallocate(pcf%m_frootc_xfer_to_fire )
    deallocate(pcf%m_livestemc_xfer_to_fire )
    deallocate(pcf%m_deadstemc_xfer_to_fire )
    deallocate(pcf%m_livecrootc_xfer_to_fire )
    deallocate(pcf%m_deadcrootc_xfer_to_fire )
    deallocate(pcf%m_livestemc_to_fire )
    deallocate(pcf%m_deadstemc_to_fire )
    deallocate(pcf%m_deadstemc_to_litter_fire )
    deallocate(pcf%m_livecrootc_to_fire )
    deallocate(pcf%m_deadcrootc_to_fire )
    deallocate(pcf%m_deadcrootc_to_litter_fire )
    deallocate(pcf%m_gresp_storage_to_fire )
    deallocate(pcf%m_gresp_xfer_to_fire )
    deallocate(pcf%leafc_xfer_to_leafc )
    deallocate(pcf%frootc_xfer_to_frootc )
    deallocate(pcf%livestemc_xfer_to_livestemc )
    deallocate(pcf%deadstemc_xfer_to_deadstemc )
    deallocate(pcf%livecrootc_xfer_to_livecrootc )
    deallocate(pcf%deadcrootc_xfer_to_deadcrootc )
    deallocate(pcf%leafc_to_litter )
    deallocate(pcf%frootc_to_litter )
    deallocate(pcf%leaf_mr )
    deallocate(pcf%froot_mr )
    deallocate(pcf%livestem_mr )
    deallocate(pcf%livecroot_mr )
    deallocate(pcf%leaf_curmr )
    deallocate(pcf%froot_curmr )
    deallocate(pcf%livestem_curmr )
    deallocate(pcf%livecroot_curmr )
    deallocate(pcf%leaf_xsmr )
    deallocate(pcf%froot_xsmr )
    deallocate(pcf%livestem_xsmr )
    deallocate(pcf%livecroot_xsmr )
    deallocate(pcf%psnsun_to_cpool )
    deallocate(pcf%psnshade_to_cpool )
    deallocate(pcf%cpool_to_xsmrpool )
    deallocate(pcf%cpool_to_leafc )
    deallocate(pcf%cpool_to_leafc_storage )
    deallocate(pcf%cpool_to_frootc )
    deallocate(pcf%cpool_to_frootc_storage )
    deallocate(pcf%cpool_to_livestemc )
    deallocate(pcf%cpool_to_livestemc_storage )
    deallocate(pcf%cpool_to_deadstemc )
    deallocate(pcf%cpool_to_deadstemc_storage )
    deallocate(pcf%cpool_to_livecrootc )
    deallocate(pcf%cpool_to_livecrootc_storage )
    deallocate(pcf%cpool_to_deadcrootc )
    deallocate(pcf%cpool_to_deadcrootc_storage )
    deallocate(pcf%cpool_to_gresp_storage )
    deallocate(pcf%cpool_leaf_gr )
    deallocate(pcf%cpool_leaf_storage_gr )
    deallocate(pcf%transfer_leaf_gr )
    deallocate(pcf%cpool_froot_gr )
    deallocate(pcf%cpool_froot_storage_gr )
    deallocate(pcf%transfer_froot_gr )
    deallocate(pcf%cpool_livestem_gr )
    deallocate(pcf%cpool_livestem_storage_gr )
    deallocate(pcf%transfer_livestem_gr )
    deallocate(pcf%cpool_deadstem_gr )
    deallocate(pcf%cpool_deadstem_storage_gr )
    deallocate(pcf%transfer_deadstem_gr )
    deallocate(pcf%cpool_livecroot_gr )
    deallocate(pcf%cpool_livecroot_storage_gr )
    deallocate(pcf%transfer_livecroot_gr )
    deallocate(pcf%cpool_deadcroot_gr )
    deallocate(pcf%cpool_deadcroot_storage_gr )
    deallocate(pcf%transfer_deadcroot_gr )
    deallocate(pcf%leafc_storage_to_xfer )
    deallocate(pcf%frootc_storage_to_xfer ) 
    deallocate(pcf%livestemc_storage_to_xfer )
    deallocate(pcf%deadstemc_storage_to_xfer )
    deallocate(pcf%livecrootc_storage_to_xfer )
    deallocate(pcf%deadcrootc_storage_to_xfer )
    deallocate(pcf%gresp_storage_to_xfer )
    deallocate(pcf%livestemc_to_deadstemc )
    deallocate(pcf%livecrootc_to_deadcrootc )
    deallocate(pcf%gpp )
    deallocate(pcf%mr )
    deallocate(pcf%current_gr )
    deallocate(pcf%transfer_gr )
    deallocate(pcf%storage_gr )
    deallocate(pcf%gr )
    deallocate(pcf%ar )
    deallocate(pcf%rr )
    deallocate(pcf%npp )
    deallocate(pcf%agnpp )
    deallocate(pcf%bgnpp )
    deallocate(pcf%litfall )
    deallocate(pcf%vegfire )
    deallocate(pcf%wood_harvestc )
    deallocate(pcf%pft_cinputs )
    deallocate(pcf%pft_coutputs )
    deallocate(pcf%pft_fire_closs )
#if (defined CROP)
    deallocate(pcf%xsmrpool_to_atm )
    deallocate(pcf%grainc_xfer_to_grainc )
    deallocate(pcf%livestemc_to_litter )
    deallocate(pcf%grainc_to_food )
    deallocate(pcf%cpool_to_grainc )
    deallocate(pcf%cpool_to_grainc_storage )
    deallocate(pcf%cpool_grain_gr )
    deallocate(pcf%cpool_grain_storage_gr )
    deallocate(pcf%transfer_grain_gr )
    deallocate(pcf%grainc_storage_to_xfer )
#endif
#if (defined CLAMP) && (defined CN)
    !CLAMP
    deallocate(pcf%frootc_alloc )
    deallocate(pcf%frootc_loss )
    deallocate(pcf%leafc_alloc )
    deallocate(pcf%leafc_loss )
    deallocate(pcf%woodc_alloc )
    deallocate(pcf%woodc_loss )
#endif
  end subroutine dealloc_pft_cflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_nflux_type
!
! !INTERFACE:
  subroutine dealloc_pft_nflux_type(    pnf)
!
! !DESCRIPTION:
! Initialize pft nitrogen flux variables
!
! !ARGUMENTS:
    implicit none
    type (pft_nflux_type), intent(inout) :: pnf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(pnf%m_leafn_to_litter )
    deallocate(pnf%m_frootn_to_litter )
    deallocate(pnf%m_leafn_storage_to_litter )
    deallocate(pnf%m_frootn_storage_to_litter )
    deallocate(pnf%m_livestemn_storage_to_litter )
    deallocate(pnf%m_deadstemn_storage_to_litter )
    deallocate(pnf%m_livecrootn_storage_to_litter )
    deallocate(pnf%m_deadcrootn_storage_to_litter )
    deallocate(pnf%m_leafn_xfer_to_litter )
    deallocate(pnf%m_frootn_xfer_to_litter )
    deallocate(pnf%m_livestemn_xfer_to_litter )
    deallocate(pnf%m_deadstemn_xfer_to_litter )
    deallocate(pnf%m_livecrootn_xfer_to_litter )
    deallocate(pnf%m_deadcrootn_xfer_to_litter )
    deallocate(pnf%m_livestemn_to_litter )
    deallocate(pnf%m_deadstemn_to_litter )
    deallocate(pnf%m_livecrootn_to_litter )
    deallocate(pnf%m_deadcrootn_to_litter )
    deallocate(pnf%m_retransn_to_litter )
    deallocate(pnf%hrv_leafn_to_litter )
    deallocate(pnf%hrv_frootn_to_litter )
    deallocate(pnf%hrv_leafn_storage_to_litter )
    deallocate(pnf%hrv_frootn_storage_to_litter )
    deallocate(pnf%hrv_livestemn_storage_to_litter )
    deallocate(pnf%hrv_deadstemn_storage_to_litter )
    deallocate(pnf%hrv_livecrootn_storage_to_litter )
    deallocate(pnf%hrv_deadcrootn_storage_to_litter )
    deallocate(pnf%hrv_leafn_xfer_to_litter )
    deallocate(pnf%hrv_frootn_xfer_to_litter )
    deallocate(pnf%hrv_livestemn_xfer_to_litter )
    deallocate(pnf%hrv_deadstemn_xfer_to_litter )
    deallocate(pnf%hrv_livecrootn_xfer_to_litter )
    deallocate(pnf%hrv_deadcrootn_xfer_to_litter )
    deallocate(pnf%hrv_livestemn_to_litter )
    deallocate(pnf%hrv_deadstemn_to_prod10n )
    deallocate(pnf%hrv_deadstemn_to_prod100n )
    deallocate(pnf%hrv_livecrootn_to_litter )
    deallocate(pnf%hrv_deadcrootn_to_litter )
    deallocate(pnf%hrv_retransn_to_litter )
    deallocate(pnf%m_leafn_to_fire )
    deallocate(pnf%m_frootn_to_fire )
    deallocate(pnf%m_leafn_storage_to_fire )
    deallocate(pnf%m_frootn_storage_to_fire )
    deallocate(pnf%m_livestemn_storage_to_fire )
    deallocate(pnf%m_deadstemn_storage_to_fire )
    deallocate(pnf%m_livecrootn_storage_to_fire )
    deallocate(pnf%m_deadcrootn_storage_to_fire )
    deallocate(pnf%m_leafn_xfer_to_fire )
    deallocate(pnf%m_frootn_xfer_to_fire )
    deallocate(pnf%m_livestemn_xfer_to_fire )
    deallocate(pnf%m_deadstemn_xfer_to_fire )
    deallocate(pnf%m_livecrootn_xfer_to_fire )
    deallocate(pnf%m_deadcrootn_xfer_to_fire )
    deallocate(pnf%m_livestemn_to_fire )
    deallocate(pnf%m_deadstemn_to_fire )
    deallocate(pnf%m_deadstemn_to_litter_fire )
    deallocate(pnf%m_livecrootn_to_fire )
    deallocate(pnf%m_deadcrootn_to_fire )
    deallocate(pnf%m_deadcrootn_to_litter_fire )
    deallocate(pnf%m_retransn_to_fire )
    deallocate(pnf%leafn_xfer_to_leafn )
    deallocate(pnf%frootn_xfer_to_frootn )
    deallocate(pnf%livestemn_xfer_to_livestemn )
    deallocate(pnf%deadstemn_xfer_to_deadstemn )
    deallocate(pnf%livecrootn_xfer_to_livecrootn )
    deallocate(pnf%deadcrootn_xfer_to_deadcrootn )
    deallocate(pnf%leafn_to_litter ) 
    deallocate(pnf%leafn_to_retransn )
    deallocate(pnf%frootn_to_litter )
    deallocate(pnf%retransn_to_npool )
    deallocate(pnf%sminn_to_npool )
    deallocate(pnf%npool_to_leafn )
    deallocate(pnf%npool_to_leafn_storage )
    deallocate(pnf%npool_to_frootn )
    deallocate(pnf%npool_to_frootn_storage )
    deallocate(pnf%npool_to_livestemn )
    deallocate(pnf%npool_to_livestemn_storage )
    deallocate(pnf%npool_to_deadstemn )
    deallocate(pnf%npool_to_deadstemn_storage )
    deallocate(pnf%npool_to_livecrootn )
    deallocate(pnf%npool_to_livecrootn_storage )
    deallocate(pnf%npool_to_deadcrootn )
    deallocate(pnf%npool_to_deadcrootn_storage )
    deallocate(pnf%leafn_storage_to_xfer )
    deallocate(pnf%frootn_storage_to_xfer )
    deallocate(pnf%livestemn_storage_to_xfer )
    deallocate(pnf%deadstemn_storage_to_xfer )
    deallocate(pnf%livecrootn_storage_to_xfer )
    deallocate(pnf%deadcrootn_storage_to_xfer )
    deallocate(pnf%livestemn_to_deadstemn )
    deallocate(pnf%livestemn_to_retransn )
    deallocate(pnf%livecrootn_to_deadcrootn )
    deallocate(pnf%livecrootn_to_retransn )
    deallocate(pnf%ndeploy )
    deallocate(pnf%pft_ninputs )
    deallocate(pnf%pft_noutputs )
    deallocate(pnf%wood_harvestn )
    deallocate(pnf%pft_fire_nloss )
#if (defined CROP)
    deallocate(pnf%grainn_xfer_to_grainn )
    deallocate(pnf%livestemn_to_litter )
    deallocate(pnf%grainn_to_food )
    deallocate(pnf%npool_to_grainn )
    deallocate(pnf%npool_to_grainn_storage )
    deallocate(pnf%grainn_storage_to_xfer )
#endif
  end subroutine dealloc_pft_nflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_vflux_type
!
! !INTERFACE:
  subroutine dealloc_pft_vflux_type(    pvf)
!
! !DESCRIPTION:
! Initialize pft VOC flux variables
!
! !ARGUMENTS:
    implicit none
    type (pft_vflux_type), intent(inout) :: pvf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
! (heald, 08/06)
!
!EOP
!------------------------------------------------------------------------

    deallocate(pvf%vocflx_tot )
    deallocate(pvf%vocflx)
    deallocate(pvf%vocflx_1 )
    deallocate(pvf%vocflx_2 )
    deallocate(pvf%vocflx_3 )
    deallocate(pvf%vocflx_4 )
    deallocate(pvf%vocflx_5 )
    deallocate(pvf%Eopt_out )
    deallocate(pvf%topt_out )
    deallocate(pvf%alpha_out )
    deallocate(pvf%cp_out )
    deallocate(pvf%para_out )
    deallocate(pvf%par24a_out )
    deallocate(pvf%par240a_out )
    deallocate(pvf%paru_out )
    deallocate(pvf%par24u_out )
    deallocate(pvf%par240u_out )
    deallocate(pvf%gamma_out )
    deallocate(pvf%gammaL_out )
    deallocate(pvf%gammaT_out )
    deallocate(pvf%gammaP_out )
    deallocate(pvf%gammaA_out )
    deallocate(pvf%gammaS_out )
  end subroutine dealloc_pft_vflux_type 

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_pft_dflux_type
!
! !INTERFACE:
  subroutine dealloc_pft_dflux_type(    pdf)
!   
! !DESCRIPTION:
! Initialize pft dust flux variables
!
! !ARGUMENTS:
    implicit none
    type (pft_dflux_type), intent(inout):: pdf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!   
!EOP
!------------------------------------------------------------------------
    
    deallocate(pdf%flx_mss_vrt_dst)
    deallocate(pdf%flx_mss_vrt_dst_tot )
    deallocate(pdf%vlc_trb)
    deallocate(pdf%vlc_trb_1 )
    deallocate(pdf%vlc_trb_2 )
    deallocate(pdf%vlc_trb_3 )
    deallocate(pdf%vlc_trb_4 )
    
    
  end subroutine dealloc_pft_dflux_type

  subroutine dealloc_pft_depvd_type(    pdd)

!
! !DESCRIPTION:
! Initialize pft dep velocity variables
!
! !ARGUMENTS:
    implicit none
    type (pft_depvd_type), intent(inout):: pdd
!
! !REVISION HISTORY:
! Created by James Sulzman 541-929-6183
!
!EOP
!------------------------------------------------------------------------

!   if (allocated(pdd%drydepvel))  deallocate(pdd%drydepvel)

  end subroutine dealloc_pft_depvd_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_column_pstate_type
!
! !INTERFACE:
  subroutine dealloc_column_pstate_type(    cps)
!
! !DESCRIPTION:
! Initialize column physical state variables
!
! !USES:
! !ARGUMENTS:
    implicit none
    type (column_pstate_type), intent(inout):: cps
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(cps%snl )      !* cannot be averaged up
    deallocate(cps%isoicol )  !* cannot be averaged up
    deallocate(cps%bsw)
    deallocate(cps%watsat)
    deallocate(cps%watfc)
    deallocate(cps%watdry)
    deallocate(cps%watopt)
    deallocate(cps%hksat)
    deallocate(cps%sucsat)
    deallocate(cps%csol)
    deallocate(cps%tkmg)
    deallocate(cps%tkdry)
    deallocate(cps%tksatu)
    deallocate(cps%smpmin )
    deallocate(cps%hkdepth )
    deallocate(cps%wtfact )
    deallocate(cps%fracice)
    deallocate(cps%gwc_thr )
    deallocate(cps%mss_frc_cly_vld )
    deallocate(cps%mbl_bsn_fct )
    deallocate(cps%do_capsnow )
    deallocate(cps%snowdp )
    deallocate(cps%frac_sno  )
    deallocate(cps%zi)
    deallocate(cps%dz)
    deallocate(cps%z )
    deallocate(cps%frac_iceold)
    deallocate(cps%imelt)
    deallocate(cps%eff_porosity)
    deallocate(cps%emg )
    deallocate(cps%z0mg )
    deallocate(cps%z0hg )
    deallocate(cps%z0qg )
    deallocate(cps%htvp )
    deallocate(cps%beta )
    deallocate(cps%zii )
    deallocate(cps%albgrd)
    deallocate(cps%albgri)
    deallocate(cps%rootr_column)
    deallocate(cps%rootfr_road_perv)
    deallocate(cps%rootr_road_perv)
    deallocate(cps%wf )
!   deallocate(cps%xirrig )
    deallocate(cps%max_dayl )
    deallocate(cps%bsw2)
    deallocate(cps%psisat)
    deallocate(cps%vwcsat)
    deallocate(cps%soilpsi)
    deallocate(cps%decl )
    deallocate(cps%coszen )
    deallocate(cps%fpi )
    deallocate(cps%fpg )
    deallocate(cps%annsum_counter )
    deallocate(cps%cannsum_npp )
    deallocate(cps%cannavg_t2m )
    deallocate(cps%me )
    deallocate(cps%fire_prob )
    deallocate(cps%mean_fire_prob )
    deallocate(cps%fireseasonl )
    deallocate(cps%farea_burned )
    deallocate(cps%ann_farea_burned )
    deallocate(cps%albsnd_hst)
    deallocate(cps%albsni_hst)
    deallocate(cps%albsod)
    deallocate(cps%albsoi)
    deallocate(cps%flx_absdv)
    deallocate(cps%flx_absdn)
    deallocate(cps%flx_absiv )
    deallocate(cps%flx_absin )
    deallocate(cps%snw_rds )
    deallocate(cps%snw_rds_top )
    deallocate(cps%sno_liq_top )
    deallocate(cps%mss_bcpho )
    deallocate(cps%mss_bcphi )
    deallocate(cps%mss_bctot )
    deallocate(cps%mss_bc_col )
    deallocate(cps%mss_bc_top )
    deallocate(cps%mss_ocpho )
    deallocate(cps%mss_ocphi )
    deallocate(cps%mss_octot )
    deallocate(cps%mss_oc_col )
    deallocate(cps%mss_oc_top )
    deallocate(cps%mss_dst1 )
    deallocate(cps%mss_dst2 )
    deallocate(cps%mss_dst3 )
    deallocate(cps%mss_dst4 )
    deallocate(cps%mss_dsttot )
    deallocate(cps%mss_dst_col )
    deallocate(cps%mss_dst_top )
    deallocate(cps%h2osno_top )
    deallocate(cps%mss_cnc_bcphi )
    deallocate(cps%mss_cnc_bcpho )
    deallocate(cps%mss_cnc_ocphi )
    deallocate(cps%mss_cnc_ocpho )
    deallocate(cps%mss_cnc_dst1 )
    deallocate(cps%mss_cnc_dst2 )
    deallocate(cps%mss_cnc_dst3 )
    deallocate(cps%mss_cnc_dst4 )
    deallocate(cps%albgrd_pur )
    deallocate(cps%albgri_pur )
    deallocate(cps%albgrd_bc )
    deallocate(cps%albgri_bc )
    deallocate(cps%albgrd_oc )
    deallocate(cps%albgri_oc )
    deallocate(cps%albgrd_dst )
    deallocate(cps%albgri_dst )
    deallocate(cps%dTdz_top )
    deallocate(cps%snot_top )

  end subroutine dealloc_column_pstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_column_estate_type
!
! !INTERFACE:
  subroutine dealloc_column_estate_type(    ces)
!
! !DESCRIPTION:
! Initialize column energy state variables
!
! !USES:
! !ARGUMENTS:
    implicit none
    type (column_estate_type), intent(inout):: ces
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------
    deallocate(ces%t_grnd )
    deallocate(ces%t_grnd_u )
    deallocate(ces%t_grnd_r )
    deallocate(ces%dt_grnd )
    deallocate(ces%t_soisno)
    deallocate(ces%t_soi_10cm )
    deallocate(ces%t_lake)
    deallocate(ces%tssbef)
    deallocate(ces%thv )
    deallocate(ces%hc_soi )
    deallocate(ces%hc_soisno )


  end subroutine dealloc_column_estate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_column_wstate_type
!
! !INTERFACE:
  subroutine dealloc_column_wstate_type(    cws)
!
! !USES:
! !ARGUMENTS:
    implicit none 
    type (column_wstate_type), intent(inout):: cws !column water state
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------
  
    deallocate(cws%h2osno )
    deallocate(cws%h2osoi_liq)
    deallocate(cws%h2osoi_ice)
    deallocate(cws%h2osoi_liqice_10cm )
    deallocate(cws%h2osoi_vol )
    deallocate(cws%h2osno_old )
    deallocate(cws%qg )
    deallocate(cws%dqgdT )
    deallocate(cws%snowice ) 
    deallocate(cws%snowliq )
    deallocate(cws%soilalpha )
    deallocate(cws%soilbeta )
    deallocate(cws%soilalpha_u )
    deallocate(cws%zwt )
    deallocate(cws%fcov )
    deallocate(cws%fsat )
    deallocate(cws%wa )
    deallocate(cws%wt )
    deallocate(cws%qcharge )
    deallocate(cws%smp_l ) 
    deallocate(cws%hk_l )
    

  end subroutine dealloc_column_wstate_type

!------------------------------------------------------------------------
!BOP
! 
! !IROUTINE: dealloc_column_cstate_type
!
! !INTERFACE:
  subroutine dealloc_column_cstate_type(    ccs)
! !DESCRIPTION:
! Initialize column carbon state variables
!
! !ARGUMENTS:
    implicit none
    type (column_cstate_type), intent(inout):: ccs
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(ccs%soilc )
    deallocate(ccs%cwdc )
    deallocate(ccs%litr1c )
    deallocate(ccs%litr2c )
    deallocate(ccs%litr3c )
    deallocate(ccs%soil1c )
    deallocate(ccs%soil2c )
    deallocate(ccs%soil3c )
    deallocate(ccs%soil4c )
    deallocate(ccs%seedc )
    deallocate(ccs%col_ctrunc )
    deallocate(ccs%prod10c )
    deallocate(ccs%prod100c )
    deallocate(ccs%totprodc )
    deallocate(ccs%totlitc )
    deallocate(ccs%totsomc )
    deallocate(ccs%totecosysc )
    deallocate(ccs%totcolc )


  end subroutine dealloc_column_cstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_column_nstate_type
!
! !INTERFACE:
  subroutine dealloc_column_nstate_type(    cns)
!
! !DESCRIPTION:
! Initialize column nitrogen state variables
!
! !ARGUMENTS:
    implicit none
    type (column_nstate_type), intent(inout):: cns
!
! !REVISION HISTORY:
! Created by Peter Thornton  
!   
!EOP
!------------------------------------------------------------------------

    deallocate(cns%cwdn )
    deallocate(cns%litr1n )
    deallocate(cns%litr2n )
    deallocate(cns%litr3n )
    deallocate(cns%soil1n )
    deallocate(cns%soil2n )
    deallocate(cns%soil3n )
    deallocate(cns%soil4n )
    deallocate(cns%sminn )
    deallocate(cns%col_ntrunc )
    deallocate(cns%seedn )
    deallocate(cns%prod10n )
    deallocate(cns%prod100n )
    deallocate(cns%totprodn )
    deallocate(cns%totlitn )
    deallocate(cns%totsomn )
    deallocate(cns%totecosysn )
    deallocate(cns%totcoln )
    
    
  end subroutine dealloc_column_nstate_type
    
!------------------------------------------------------------------------
!BOP
! 
! !IROUTINE: dealloc_column_eflux_type
!
! !INTERFACE:
  subroutine dealloc_column_eflux_type(    cef)
!
! !DESCRIPTION:
! Initialize column energy flux variables
! 
! !ARGUMENTS:
    implicit none
    type (column_eflux_type), intent(inout):: cef
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(cef%eflx_snomelt )
    deallocate(cef%eflx_snomelt_u )
    deallocate(cef%eflx_snomelt_r )
    deallocate(cef%eflx_impsoil )
    deallocate(cef%eflx_fgr12 )
    deallocate(cef%eflx_building_heat )
    deallocate(cef%eflx_urban_ac )
    deallocate(cef%eflx_urban_heat )


  end subroutine dealloc_column_eflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_column_wflux_type
!
! !INTERFACE:
  subroutine dealloc_column_wflux_type(    cwf)
!
! !DESCRIPTION:
! Initialize column water flux variables
!
! !USES:
! !ARGUMENTS:
    implicit none
    type (column_wflux_type), intent(inout):: cwf
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(cwf%qflx_infl )
    deallocate(cwf%qflx_surf )
    deallocate(cwf%qflx_drain )
    deallocate(cwf%qflx_top_soil )
    deallocate(cwf%qflx_snomelt )
    deallocate(cwf%qflx_qrgwl )
    deallocate(cwf%qflx_runoff )
    deallocate(cwf%qflx_runoff_u )
    deallocate(cwf%qflx_runoff_r )
    deallocate(cwf%qmelt )
    deallocate(cwf%h2ocan_loss )
    deallocate(cwf%qflx_rsub_sat )
    deallocate(cwf%flx_bc_dep_dry )
    deallocate(cwf%flx_bc_dep_wet )
    deallocate(cwf%flx_bc_dep_pho )
    deallocate(cwf%flx_bc_dep_phi )
    deallocate(cwf%flx_bc_dep )
    deallocate(cwf%flx_oc_dep_dry )
    deallocate(cwf%flx_oc_dep_wet )
    deallocate(cwf%flx_oc_dep_pho )
    deallocate(cwf%flx_oc_dep_phi )
    deallocate(cwf%flx_oc_dep )
    deallocate(cwf%flx_dst_dep_dry1 )
    deallocate(cwf%flx_dst_dep_wet1 )
    deallocate(cwf%flx_dst_dep_dry2 )
    deallocate(cwf%flx_dst_dep_wet2 )
    deallocate(cwf%flx_dst_dep_dry3 )
    deallocate(cwf%flx_dst_dep_wet3 ) 
    deallocate(cwf%flx_dst_dep_dry4 )
    deallocate(cwf%flx_dst_dep_wet4 )
    deallocate(cwf%flx_dst_dep )
    deallocate(cwf%qflx_snofrz_lyr)
    
  end subroutine dealloc_column_wflux_type
    
!------------------------------------------------------------------------
!BOP
!   
! !IROUTINE: dealloc_column_cflux_type
!   
! !INTERFACE:
  subroutine dealloc_column_cflux_type(    ccf)
!   
! !DESCRIPTION:
! Initialize column carbon flux variables
!   
! !ARGUMENTS:
    implicit none
    type (column_cflux_type), intent(inout):: ccf
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

    deallocate(ccf%m_leafc_to_litr1c )
    deallocate(ccf%m_leafc_to_litr2c )
    deallocate(ccf%m_leafc_to_litr3c )
    deallocate(ccf%m_frootc_to_litr1c )
    deallocate(ccf%m_frootc_to_litr2c )
    deallocate(ccf%m_frootc_to_litr3c )
    deallocate(ccf%m_leafc_storage_to_litr1c )
    deallocate(ccf%m_frootc_storage_to_litr1c )
    deallocate(ccf%m_livestemc_storage_to_litr1c )
    deallocate(ccf%m_deadstemc_storage_to_litr1c )
    deallocate(ccf%m_livecrootc_storage_to_litr1c )
    deallocate(ccf%m_deadcrootc_storage_to_litr1c )
    deallocate(ccf%m_leafc_xfer_to_litr1c )
    deallocate(ccf%m_frootc_xfer_to_litr1c )
    deallocate(ccf%m_livestemc_xfer_to_litr1c )
    deallocate(ccf%m_deadstemc_xfer_to_litr1c )
    deallocate(ccf%m_livecrootc_xfer_to_litr1c )
    deallocate(ccf%m_deadcrootc_xfer_to_litr1c )
    deallocate(ccf%m_livestemc_to_cwdc )
    deallocate(ccf%m_deadstemc_to_cwdc )
    deallocate(ccf%m_livecrootc_to_cwdc )
    deallocate(ccf%m_deadcrootc_to_cwdc )
    deallocate(ccf%m_gresp_storage_to_litr1c )
    deallocate(ccf%m_gresp_xfer_to_litr1c )
    deallocate(ccf%m_deadstemc_to_cwdc_fire )
    deallocate(ccf%m_deadcrootc_to_cwdc_fire )
    deallocate(ccf%hrv_leafc_to_litr1c )
    deallocate(ccf%hrv_leafc_to_litr2c )
    deallocate(ccf%hrv_leafc_to_litr3c )
    deallocate(ccf%hrv_frootc_to_litr1c )
    deallocate(ccf%hrv_frootc_to_litr2c )
    deallocate(ccf%hrv_frootc_to_litr3c )
    deallocate(ccf%hrv_livestemc_to_cwdc )
    deallocate(ccf%hrv_deadstemc_to_prod10c )
    deallocate(ccf%hrv_deadstemc_to_prod100c )
    deallocate(ccf%hrv_livecrootc_to_cwdc )
    deallocate(ccf%hrv_deadcrootc_to_cwdc )
    deallocate(ccf%hrv_leafc_storage_to_litr1c )
    deallocate(ccf%hrv_frootc_storage_to_litr1c )
    deallocate(ccf%hrv_livestemc_storage_to_litr1c )
    deallocate(ccf%hrv_deadstemc_storage_to_litr1c )
    deallocate(ccf%hrv_livecrootc_storage_to_litr1c )
    deallocate(ccf%hrv_deadcrootc_storage_to_litr1c )
    deallocate(ccf%hrv_gresp_storage_to_litr1c )
    deallocate(ccf%hrv_leafc_xfer_to_litr1c )
    deallocate(ccf%hrv_frootc_xfer_to_litr1c )
    deallocate(ccf%hrv_livestemc_xfer_to_litr1c )
    deallocate(ccf%hrv_deadstemc_xfer_to_litr1c )
    deallocate(ccf%hrv_livecrootc_xfer_to_litr1c )
    deallocate(ccf%hrv_deadcrootc_xfer_to_litr1c )
    deallocate(ccf%hrv_gresp_xfer_to_litr1c )
    deallocate(ccf%m_litr1c_to_fire )
    deallocate(ccf%m_litr2c_to_fire )
    deallocate(ccf%m_litr3c_to_fire )
    deallocate(ccf%m_cwdc_to_fire )
#if (defined CROP)
    deallocate(ccf%grainc_to_litr1c )
    deallocate(ccf%grainc_to_litr2c )
    deallocate(ccf%grainc_to_litr3c )
    deallocate(ccf%livestemc_to_litr1c )
    deallocate(ccf%livestemc_to_litr2c )
    deallocate(ccf%livestemc_to_litr3c )
#endif
    deallocate(ccf%leafc_to_litr1c )
    deallocate(ccf%leafc_to_litr2c )
    deallocate(ccf%leafc_to_litr3c )
    deallocate(ccf%frootc_to_litr1c )
    deallocate(ccf%frootc_to_litr2c )
    deallocate(ccf%frootc_to_litr3c )
    deallocate(ccf%cwdc_to_litr2c )
    deallocate(ccf%cwdc_to_litr3c )
    deallocate(ccf%litr1_hr )
    deallocate(ccf%litr1c_to_soil1c )
    deallocate(ccf%litr2_hr )
    deallocate(ccf%litr2c_to_soil2c )
    deallocate(ccf%litr3_hr )
    deallocate(ccf%litr3c_to_soil3c )
    deallocate(ccf%soil1_hr )
    deallocate(ccf%soil1c_to_soil2c )
    deallocate(ccf%soil2_hr )
    deallocate(ccf%soil2c_to_soil3c )
    deallocate(ccf%soil3_hr )
    deallocate(ccf%soil3c_to_soil4c )
    deallocate(ccf%soil4_hr )
#ifdef CN
    deallocate(ccf%dwt_seedc_to_leaf ) 
    deallocate(ccf%dwt_seedc_to_deadstem )
    deallocate(ccf%dwt_conv_cflux )
    deallocate(ccf%dwt_prod10c_gain )
    deallocate(ccf%dwt_prod100c_gain )
    deallocate(ccf%dwt_frootc_to_litr1c )
    deallocate(ccf%dwt_frootc_to_litr2c )
    deallocate(ccf%dwt_frootc_to_litr3c )
    deallocate(ccf%dwt_livecrootc_to_cwdc )
    deallocate(ccf%dwt_deadcrootc_to_cwdc )
    deallocate(ccf%dwt_closs )
    deallocate(ccf%landuseflux )
    deallocate(ccf%landuptake )
    deallocate(ccf%prod10c_loss )
    deallocate(ccf%prod100c_loss )
    deallocate(ccf%product_closs )
#endif
    deallocate(ccf%lithr )
    deallocate(ccf%somhr )
    deallocate(ccf%hr )
    deallocate(ccf%sr )
    deallocate(ccf%er )
    deallocate(ccf%litfire )
    deallocate(ccf%somfire )
    deallocate(ccf%totfire )
    deallocate(ccf%nep )
    deallocate(ccf%nbp )
    deallocate(ccf%nee )
    deallocate(ccf%col_cinputs )
    deallocate(ccf%col_coutputs )
    deallocate(ccf%col_fire_closs )
    
#if (defined CLAMP) && (defined CN) 
    !CLAMP
    deallocate(ccf%cwdc_hr )
    deallocate(ccf%cwdc_loss )
    deallocate(ccf%litterc_loss )
#endif
    
 end subroutine dealloc_column_cflux_type




  subroutine dealloc_column_nflux_type(    cnf)
!
! !DESCRIPTION:
! Initialize column nitrogen flux variables
!
! !ARGUMENTS:
    implicit none
    type (column_nflux_type), intent(inout):: cnf
!
! !REVISION HISTORY:
! Created by Peter Thornton
!
!EOP
!------------------------------------------------------------------------

    deallocate(cnf%ndep_to_sminn )
    deallocate(cnf%nfix_to_sminn )
    deallocate(cnf%m_leafn_to_litr1n )
    deallocate(cnf%m_leafn_to_litr2n )
    deallocate(cnf%m_leafn_to_litr3n )
    deallocate(cnf%m_frootn_to_litr1n )
    deallocate(cnf%m_frootn_to_litr2n )
    deallocate(cnf%m_frootn_to_litr3n )
    deallocate(cnf%m_leafn_storage_to_litr1n )
    deallocate(cnf%m_frootn_storage_to_litr1n )
    deallocate(cnf%m_livestemn_storage_to_litr1n )
    deallocate(cnf%m_deadstemn_storage_to_litr1n )
    deallocate(cnf%m_livecrootn_storage_to_litr1n )
    deallocate(cnf%m_deadcrootn_storage_to_litr1n )
    deallocate(cnf%m_leafn_xfer_to_litr1n )
    deallocate(cnf%m_frootn_xfer_to_litr1n )
    deallocate(cnf%m_livestemn_xfer_to_litr1n )
    deallocate(cnf%m_deadstemn_xfer_to_litr1n )
    deallocate(cnf%m_livecrootn_xfer_to_litr1n )
    deallocate(cnf%m_deadcrootn_xfer_to_litr1n )
    deallocate(cnf%m_livestemn_to_cwdn )
    deallocate(cnf%m_deadstemn_to_cwdn )
    deallocate(cnf%m_livecrootn_to_cwdn )
    deallocate(cnf%m_deadcrootn_to_cwdn )
    deallocate(cnf%m_retransn_to_litr1n )
    deallocate(cnf%hrv_leafn_to_litr1n )
    deallocate(cnf%hrv_leafn_to_litr2n )
    deallocate(cnf%hrv_leafn_to_litr3n )
    deallocate(cnf%hrv_frootn_to_litr1n )
    deallocate(cnf%hrv_frootn_to_litr2n )
    deallocate(cnf%hrv_frootn_to_litr3n )
    deallocate(cnf%hrv_livestemn_to_cwdn )
    deallocate(cnf%hrv_deadstemn_to_prod10n )
    deallocate(cnf%hrv_deadstemn_to_prod100n )
    deallocate(cnf%hrv_livecrootn_to_cwdn )
    deallocate(cnf%hrv_deadcrootn_to_cwdn )
    deallocate(cnf%hrv_retransn_to_litr1n )
    deallocate(cnf%hrv_leafn_storage_to_litr1n )
    deallocate(cnf%hrv_frootn_storage_to_litr1n )
    deallocate(cnf%hrv_livestemn_storage_to_litr1n )
    deallocate(cnf%hrv_deadstemn_storage_to_litr1n )
    deallocate(cnf%hrv_livecrootn_storage_to_litr1n )
    deallocate(cnf%hrv_deadcrootn_storage_to_litr1n )
    deallocate(cnf%hrv_leafn_xfer_to_litr1n )
    deallocate(cnf%hrv_frootn_xfer_to_litr1n )
    deallocate(cnf%hrv_livestemn_xfer_to_litr1n )
    deallocate(cnf%hrv_deadstemn_xfer_to_litr1n )
    deallocate(cnf%hrv_livecrootn_xfer_to_litr1n )
    deallocate(cnf%hrv_deadcrootn_xfer_to_litr1n )
    deallocate(cnf%m_deadstemn_to_cwdn_fire )
    deallocate(cnf%m_deadcrootn_to_cwdn_fire )
    deallocate(cnf%m_litr1n_to_fire )
    deallocate(cnf%m_litr2n_to_fire )
    deallocate(cnf%m_litr3n_to_fire )
    deallocate(cnf%m_cwdn_to_fire )
#if (defined CROP)
    deallocate(cnf%grainn_to_litr1n )
    deallocate(cnf%grainn_to_litr2n )
    deallocate(cnf%grainn_to_litr3n )
    deallocate(cnf%livestemn_to_litr1n )
    deallocate(cnf%livestemn_to_litr2n )
    deallocate(cnf%livestemn_to_litr3n )
#endif
    deallocate(cnf%leafn_to_litr1n )
    deallocate(cnf%leafn_to_litr2n )
    deallocate(cnf%leafn_to_litr3n )
    deallocate(cnf%frootn_to_litr1n )
    deallocate(cnf%frootn_to_litr2n )
    deallocate(cnf%frootn_to_litr3n )
    deallocate(cnf%cwdn_to_litr2n )
    deallocate(cnf%cwdn_to_litr3n )
    deallocate(cnf%litr1n_to_soil1n ) 
    deallocate(cnf%sminn_to_soil1n_l1 )
    deallocate(cnf%litr2n_to_soil2n ) 
    deallocate(cnf%sminn_to_soil2n_l2 )
    deallocate(cnf%litr3n_to_soil3n )
    deallocate(cnf%sminn_to_soil3n_l3 )
    deallocate(cnf%soil1n_to_soil2n )
    deallocate(cnf%sminn_to_soil2n_s1 )
    deallocate(cnf%soil2n_to_soil3n )
    deallocate(cnf%sminn_to_soil3n_s2 )
    deallocate(cnf%soil3n_to_soil4n )
    deallocate(cnf%sminn_to_soil4n_s3 )
    deallocate(cnf%soil4n_to_sminn )
    deallocate(cnf%sminn_to_denit_l1s1 )
    deallocate(cnf%sminn_to_denit_l2s2 )
    deallocate(cnf%sminn_to_denit_l3s3 )
    deallocate(cnf%sminn_to_denit_s1s2 )
    deallocate(cnf%sminn_to_denit_s2s3 )
    deallocate(cnf%sminn_to_denit_s3s4 )
    deallocate(cnf%sminn_to_denit_s4 )
    deallocate(cnf%sminn_to_denit_excess )
    deallocate(cnf%sminn_leached )
    deallocate(cnf%dwt_seedn_to_leaf ) 
    deallocate(cnf%dwt_seedn_to_deadstem )
    deallocate(cnf%dwt_conv_nflux )
    deallocate(cnf%dwt_prod10n_gain )
    deallocate(cnf%dwt_prod100n_gain )
    deallocate(cnf%dwt_frootn_to_litr1n )
    deallocate(cnf%dwt_frootn_to_litr2n )
    deallocate(cnf%dwt_frootn_to_litr3n ) 
    deallocate(cnf%dwt_livecrootn_to_cwdn )
    deallocate(cnf%dwt_deadcrootn_to_cwdn )
    deallocate(cnf%dwt_nloss )
    deallocate(cnf%prod10n_loss )
    deallocate(cnf%prod100n_loss )
    deallocate(cnf%product_nloss )
    deallocate(cnf%potential_immob )
    deallocate(cnf%actual_immob )
    deallocate(cnf%sminn_to_plant )
    deallocate(cnf%supplement_to_sminn )
    deallocate(cnf%gross_nmin )
    deallocate(cnf%net_nmin )
    deallocate(cnf%denit )
    deallocate(cnf%col_ninputs )
    deallocate(cnf%col_noutputs )
   deallocate(cnf%col_fire_nloss )

  end subroutine dealloc_column_nflux_type
    
!------------------------------------------------------------------------
!BOP
!   
! !IROUTINE: dealloc_landunit_pstate_type
!   
! !INTERFACE:
  subroutine dealloc_landunit_pstate_type(    lps)
!   
! !ARGUMENTS:
    implicit none
    type (landunit_pstate_type), intent(inout):: lps
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------

    deallocate(lps%t_building )
    deallocate(lps%t_building_max )
    deallocate(lps%t_building_min )
    deallocate(lps%tk_wall)
    deallocate(lps%tk_roof)
    deallocate(lps%tk_improad)
    deallocate(lps%cv_wall)
    deallocate(lps%cv_roof)
    deallocate(lps%cv_improad)
    deallocate(lps%thick_wall )
    deallocate(lps%thick_roof )
    deallocate(lps%nlev_improad )
    deallocate(lps%vf_sr )
    deallocate(lps%vf_wr )
    deallocate(lps%vf_sw )
    deallocate(lps%vf_rw )
    deallocate(lps%vf_ww )
    deallocate(lps%taf )
    deallocate(lps%qaf )
    deallocate(lps%sabs_roof_dir )
    deallocate(lps%sabs_roof_dif )
    deallocate(lps%sabs_sunwall_dir )
    deallocate(lps%sabs_sunwall_dif )
    deallocate(lps%sabs_shadewall_dir )
    deallocate(lps%sabs_shadewall_dif )
    deallocate(lps%sabs_improad_dir )
    deallocate(lps%sabs_improad_dif )
    deallocate(lps%sabs_perroad_dir )
    deallocate(lps%sabs_perroad_dif )
  end subroutine dealloc_landunit_pstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_landunit_eflux_type
!   
! !INTERFACE:
  subroutine dealloc_landunit_eflux_type(    lef)
!   
! !DESCRIPTION: 
! Initialize landunit energy flux variables
!   
! !ARGUMENTS:
    implicit none
    type (landunit_eflux_type), intent(inout):: lef
!   
! !REVISION HISTORY:
! Created by Keith Oleson
!   
!EOP
!------------------------------------------------------------------------
    
    deallocate(lef%eflx_traffic )
    deallocate(lef%eflx_traffic_factor )
    deallocate(lef%eflx_wasteheat ) 
    deallocate(lef%eflx_heat_from_ac )
    
    
  end subroutine dealloc_landunit_eflux_type

#if (defined CNDV)
!------------------------------------------------------------------------
!BOP
!   
! !IROUTINE: dealloc_gridcell_dgvstate_type
!   
! !INTERFACE:
  subroutine dealloc_gridcell_dgvstate_type(    gps)
!   
! !DESCRIPTION:
! Initialize gridcell DGVM variables
!   
! !ARGUMENTS:  
    implicit none
    type (gridcell_dgvstate_type), intent(inout):: gps
!   
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!   
!EOP
!------------------------------------------------------------------------
    
    deallocate(gps%agdd20 )
    deallocate(gps%tmomin20 )
    deallocate(gps%t10min )

  end subroutine dealloc_gridcell_dgvstate_type
#endif

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_gridcell_pstate_type
!
! !INTERFACE:
  subroutine dealloc_gridcell_pstate_type(    gps)
!
! !DESCRIPTION:
! Initialize gridcell physical state variables
!
! !ARGUMENTS:
    implicit none
    type (gridcell_pstate_type), intent(inout):: gps
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------



  end subroutine dealloc_gridcell_pstate_type



 subroutine dealloc_gridcell_efstate_type(    gve)
!
! !DESCRIPTION:
! Initialize gridcell isoprene emission factor variables
!
! !ARGUMENTS:
    implicit none
    type (gridcell_efstate_type), intent(inout) :: gve
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein (heald)
!
!EOP
!------------------------------------------------------------------------

    deallocate(gve%efisop)
    
  end subroutine dealloc_gridcell_efstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_gridcell_wflux_type
!
! !INTERFACE:
  subroutine dealloc_gridcell_wflux_type(    gwf)
!
! !DESCRIPTION:
! Initialize gridcell water flux variables
!   
! !ARGUMENTS:
    implicit none
    type (gridcell_wflux_type), intent(inout):: gwf
!   
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!   
!EOP
!------------------------------------------------------------------------

    deallocate(gwf%qflx_runoffg )
    deallocate(gwf%qflx_snwcp_iceg )
    deallocate(gwf%qflx_liq_dynbal )
    deallocate(gwf%qflx_ice_dynbal )
    
    
  end subroutine dealloc_gridcell_wflux_type
    
!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_gridcell_eflux_type
!
 !INTERFACE:
  subroutine dealloc_gridcell_eflux_type(    gef)
!
! !DESCRIPTION:
! Initialize gridcell energy flux variables
!
! !ARGUMENTS:
    implicit none
    type (gridcell_eflux_type), intent(inout):: gef
!
! !REVISION HISTORY:
! Created by David Lawrence
!
!EOP
!------------------------------------------------------------------------
    deallocate(gef%eflx_sh_totg )
    deallocate(gef%eflx_dynbal )


  end subroutine dealloc_gridcell_eflux_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_gridcell_wstate_type
!
! !INTERFACE:
  subroutine dealloc_gridcell_wstate_type(    gws)
!
! !DESCRIPTION:
! Initialize gridcell water state variables
!
! !ARGUMENTS:
    implicit none
    type (gridcell_wstate_type), intent(inout):: gws
!
! !REVISION HISTORY:
! Created by David Lawrence
!
!EOP
!------------------------------------------------------------------------
    deallocate(gws%gc_liq1 )
    deallocate(gws%gc_liq2 )
    deallocate(gws%gc_ice1 )
    deallocate(gws%gc_ice2 )


  end subroutine dealloc_gridcell_wstate_type

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: dealloc_gridcell_estate_type
!
! !INTERFACE:
  subroutine dealloc_gridcell_estate_type(    ges)
! !DESCRIPTION:
! Initialize gridcell energy state variables     
! 
! !ARGUMENTS:
    implicit none
    type (gridcell_estate_type), intent(inout):: ges
!
! !REVISION HISTORY:
! Created by David Lawrence  
!   
!EOP
!------------------------------------------------------------------------
    deallocate(ges%gc_heat1 )
    deallocate(ges%gc_heat2 )

    
  end subroutine dealloc_gridcell_estate_type

   
   subroutine dealloc_atm2lnd_type(a2l)
    implicit none
  type (atm2lnd_type), intent(inout):: a2l
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
! Modified by T Craig, 11/01/05 for finemesh project
!
!
! !LOCAL VARIABLES:
!EOP
!------------------------------------------------------------------------

  deallocate(a2l%forc_t)
  deallocate(a2l%forc_u)
  deallocate(a2l%forc_v)
  deallocate(a2l%forc_wind)
  deallocate(a2l%forc_q)
  deallocate(a2l%forc_rh)
  deallocate(a2l%forc_hgt)
  deallocate(a2l%forc_hgt_u)
  deallocate(a2l%forc_hgt_t)
  deallocate(a2l%forc_hgt_q)
  deallocate(a2l%forc_pbot)
  deallocate(a2l%forc_th)
  deallocate(a2l%forc_vp)
  deallocate(a2l%forc_rho)
  deallocate(a2l%forc_psrf)
  deallocate(a2l%forc_pco2)
  deallocate(a2l%forc_lwrad)
  deallocate(a2l%forc_solad)
  deallocate(a2l%forc_solai)
  deallocate(a2l%forc_solar)
  deallocate(a2l%forc_rain)
  deallocate(a2l%forc_snow)
  deallocate(a2l%forc_ndep)
  deallocate(a2l%rainf)
#if (defined C13)
  ! 4/14/05: PET
  ! Adding isotope code
  deallocate(a2l%forc_pc13o2)
#endif
  deallocate(a2l%forc_po2)
  deallocate(a2l%forc_aer)

 end  subroutine dealloc_atm2lnd_type                                               




end module clmtypeInitMod
module pftvarcon

!----------------------------------------------------------------------- 
!BOP
!
! !MODULE: pftvarcon
! 
! !DESCRIPTION: 
! Module containing vegetation constants and method to 
! eads and initialize vegetation (PFT) constants.
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
  use clm_varpar,only : numpft, numrad

!
! !PUBLIC TYPES:
  implicit none
  save
!
! Vegetation type constants
!ylu add
! Set specific vegetation type values



  integer,parameter :: noveg                  = 0                 !value for not vegetated 
  integer,parameter :: ndllf_evr_tmp_tree     = 1  !value for Needleleaf evergreen temperate tree
  integer,parameter :: ndllf_evr_brl_tree     = 2  !value for Needleleaf evergreen boreal tree
  integer,parameter :: ndllf_dcd_brl_tree     = 3  !value for Needleleaf deciduous boreal tree
  integer,parameter :: nbrdlf_evr_trp_tree    = 4  !value for Broadleaf evergreen tropical tree
  integer,parameter :: nbrdlf_evr_tmp_tree    = 5  !value for Broadleaf evergreen temperate tree
  integer,parameter :: nbrdlf_dcd_trp_tree    = 6  !value for Broadleaf deciduous tropical tree
  integer,parameter :: nbrdlf_dcd_tmp_tree    = 7  !value for Broadleaf deciduous temperate tree
  integer,parameter :: nbrdlf_dcd_brl_tree    = 8  !value for Broadleaf deciduous boreal tree
  integer :: ntree                  !value for last type of tree
  integer,parameter :: nbrdlf_evr_shrub       = 9  !value for Broadleaf evergreen shrub
  integer,parameter :: nbrdlf_dcd_tmp_shrub   = 10 !value for Broadleaf deciduous temperate shrub
  integer,parameter :: nbrdlf_dcd_brl_shrub   = 11 !value for Broadleaf deciduous boreal shrub
  integer,parameter :: nc3_arctic_grass       = 12 !value for C3 arctic grass
  integer,parameter :: nc3_nonarctic_grass    = 13 !value for C3 non-arctic grass
  integer,parameter :: nc4_grass              = 14 !value for C4 grass
  integer,parameter :: nc3crop                = 15 !value for generic C3 crop
  integer,parameter :: nc4crop                = 16 !value for generic C4 crop
#if (defined CROP)
  integer :: npcropmin              !value for first crop
  integer,parameter :: ncorn                  = 17 !value for corn
  integer,parameter :: nswheat                = 18 !value for spring wheat
  integer,parameter :: nwwheat                = 19 !value for winter wheat
  integer,parameter :: nsoybean               = 20 !value for soybean
  integer :: npcropmax              !value for last prognostic crop in list
  real(r8):: mxtmp(0:numpft)        !parameter used in accFlds
  real(r8):: baset(0:numpft)        !parameter used in accFlds
  real(r8):: declfact(0:numpft)     !parameter used in CNAllocation
  real(r8):: bfact(0:numpft)        !parameter used in CNAllocation
  real(r8):: aleaff(0:numpft)       !parameter used in CNAllocation
  real(r8):: arootf(0:numpft)       !parameter used in CNAllocation
  real(r8):: astemf(0:numpft)       !parameter used in CNAllocation
  real(r8):: arooti(0:numpft)       !parameter used in CNAllocation
  real(r8):: fleafi(0:numpft)       !parameter used in CNAllocation
  real(r8):: allconsl(0:numpft)     !parameter used in CNAllocation
  real(r8):: allconss(0:numpft)     !parameter used in CNAllocation
  real(r8):: ztopmx(0:numpft)       !parameter used in CNVegStructUpdate
  real(r8):: laimx(0:numpft)        !parameter used in CNVegStructUpdate
  real(r8):: gddmin(0:numpft)       !parameter used in CNPhenology
  real(r8):: hybgdd(0:numpft)       !parameter used in CNPhenology
  real(r8):: lfemerg(0:numpft)      !parameter used in CNPhenology
  real(r8):: grnfill(0:numpft)      !parameter used in CNPhenology
  integer :: mxmat(0:numpft)        !parameter used in CNPhenology
#endif

  real(r8):: crop(0:numpft)        ! crop pft: 0. = not crop, 1. = crop pft

!-----------------------------------------------------------------------
  character(len=40) pftname(0:numpft)
  real(r8):: dleaf(0:numpft)       !characteristic leaf dimension (m)
  real(r8):: c3psn(0:numpft)       !photosynthetic pathway: 0. = c4, 1. = c3
  real(r8):: vcmx25(0:numpft)      !max rate of carboxylation at 25C (umol CO2/m**2/s)
  real(r8):: mp(0:numpft)          !slope of conductance-to-photosynthesis relationship
  real(r8):: qe25(0:numpft)        !quantum efficiency at 25C (umol CO2 / umol photon)
  real(r8):: xl(0:numpft)          !leaf/stem orientation index
  real(r8):: rhol(0:numpft,numrad) !leaf reflectance: 1=vis, 2=nir
  real(r8):: rhos(0:numpft,numrad) !stem reflectance: 1=vis, 2=nir
  real(r8):: taul(0:numpft,numrad) !leaf transmittance: 1=vis, 2=nir
  real(r8):: taus(0:numpft,numrad) !stem transmittance: 1=vis, 2=nir
  real(r8):: z0mr(0:numpft)        !ratio of momentum roughness length to canopy top height (-)
  real(r8):: displar(0:numpft)     !ratio of displacement height to canopy top height (-)
  real(r8):: roota_par(0:numpft)   !CLM rooting distribution parameter [1/m]
  real(r8):: rootb_par(0:numpft)   !CLM rooting distribution parameter [1/m]
  real(r8):: slatop(0:numpft)      !SLA at top of canopy [m^2/gC]
  real(r8):: dsladlai(0:numpft)    !dSLA/dLAI [m^2/gC]
  real(r8):: leafcn(0:numpft)      !leaf C:N [gC/gN]
  real(r8):: flnr(0:numpft)        !fraction of leaf N in Rubisco [no units]
  real(r8):: smpso(0:numpft)       !soil water potential at full stomatal opening (mm)
  real(r8):: smpsc(0:numpft)       !soil water potential at full stomatal closure (mm)
  real(r8):: fnitr(0:numpft)       !foliage nitrogen limitation factor (-)
  real(r8):: woody(0:numpft)       !woody lifeform flag (0 or 1)
  real(r8):: lflitcn(0:numpft)      !leaf litter C:N (gC/gN)
  real(r8):: frootcn(0:numpft)      !fine root C:N (gC/gN)
  real(r8):: livewdcn(0:numpft)     !live wood (phloem and ray parenchyma) C:N (gC/gN)
  real(r8):: deadwdcn(0:numpft)     !dead wood (xylem and heartwood) C:N (gC/gN)
  real(r8):: froot_leaf(0:numpft)   !allocation parameter: new fine root C per new leaf C (gC/gC) 
  real(r8):: stem_leaf(0:numpft)    !allocation parameter: new stem c per new leaf C (gC/gC)
  real(r8):: croot_stem(0:numpft)   !allocation parameter: new coarse root C per new stem C (gC/gC)
  real(r8):: flivewd(0:numpft)      !allocation parameter: fraction of new wood that is live (phloem and ray parenchyma) (no units)
  real(r8):: fcur(0:numpft)         !allocation parameter: fraction of allocation that goes to currently displayed growth, remainder to storage
  real(r8):: lf_flab(0:numpft)      !leaf litter labile fraction
  real(r8):: lf_fcel(0:numpft)      !leaf litter cellulose fraction
  real(r8):: lf_flig(0:numpft)      !leaf litter lignin fraction
  real(r8):: fr_flab(0:numpft)      !fine root litter labile fraction
  real(r8):: fr_fcel(0:numpft)      !fine root litter cellulose fraction
  real(r8):: fr_flig(0:numpft)      !fine root litter lignin fraction
  real(r8):: dw_fcel(0:numpft)      !dead wood cellulose fraction
  real(r8):: dw_flig(0:numpft)      !dead wood lignin fraction
  real(r8):: leaf_long(0:numpft)    !leaf longevity (yrs)
  real(r8):: evergreen(0:numpft)    !binary flag for evergreen leaf habit (0 or 1)
  real(r8):: stress_decid(0:numpft) !binary flag for stress-deciduous leaf habit (0 or 1)
  real(r8):: season_decid(0:numpft) !binary flag for seasonal-deciduous leaf habit (0 or 1)
  ! new pft parameters for CN-fire code
  real(r8):: resist(0:numpft)       !resistance to fire (no units)
  real(r8):: pftpar20(0:numpft)       !tree maximum crown area (m2)
  real(r8):: pftpar28(0:numpft)       !min coldest monthly mean temperature
  real(r8):: pftpar29(0:numpft)       !max coldest monthly mean temperature
  real(r8):: pftpar30(0:numpft)       !min growing degree days (>= 5 deg C)
  real(r8):: pftpar31(0:numpft)       !upper limit of temperature of the warmest month (twmax)
  ! for crop
  real(r8):: graincn(0:numpft)      !grain C:N (gC/gN)
 

  real(r8), parameter :: reinickerp = 1.6_r8 !parameter in allometric equation
  real(r8), parameter :: dwood  = 2.5e5_r8   !cn wood density (gC/m3); lpj:2.0e5
  real(r8), parameter :: allom1 = 100.0_r8   !parameters in
  real(r8), parameter :: allom2 =  40.0_r8   !...allometric
  real(r8), parameter :: allom3 =   0.5_r8   !...equations
  real(r8), parameter :: allom1s = 250.0_r8  !modified for shrubs by
  real(r8), parameter :: allom2s =   8.0_r8  !X.D.Z



! Created by Sam Levis (put into module form by Mariana Vertenstein)

    character(len=40) expected_pftnames(0:numpft) 

    integer, private :: i  ! loop index





   data (expected_pftnames(i),i=1,numpft) / &
                 'needleleaf_evergreen_temperate_tree'  &
               , 'needleleaf_evergreen_boreal_tree   '  &
               , 'needleleaf_deciduous_boreal_tree   '  &
               , 'broadleaf_evergreen_tropical_tree  '  &
               , 'broadleaf_evergreen_temperate_tree '  &
               , 'broadleaf_deciduous_tropical_tree  '  &
               , 'broadleaf_deciduous_temperate_tree '  &
               , 'broadleaf_deciduous_boreal_tree    '  &
               , 'broadleaf_evergreen_shrub          '  &
               , 'broadleaf_deciduous_temperate_shrub'  &
               , 'broadleaf_deciduous_boreal_shrub   '  &
               , 'c3_arctic_grass                    '  &
               , 'c3_non-arctic_grass                '  &
               , 'c4_grass                           '  &
               , 'c3_crop                            '  &
               , 'c4_crop                            '  &
#if (defined CROP)
               , 'corn                               '  &
               , 'spring_wheat                       '  &
               , 'winter_wheat                       '  &
               , 'soybean                            '  &
#endif
    /

!ylu 10/18/10 add new physiology data for CLM4 and CROP

data (pftname(i),i=1,numpft)/'needleleaf_evergreen_temperate_tree'&
               , 'needleleaf_evergreen_boreal_tree   '  &
               , 'needleleaf_deciduous_boreal_tree   '  &
               , 'broadleaf_evergreen_tropical_tree  '  &
               , 'broadleaf_evergreen_temperate_tree '  &
               , 'broadleaf_deciduous_tropical_tree  '  &
               , 'broadleaf_deciduous_temperate_tree '  &
               , 'broadleaf_deciduous_boreal_tree    '  &
               , 'broadleaf_evergreen_shrub          '  &
               , 'broadleaf_deciduous_temperate_shrub'  &
               , 'broadleaf_deciduous_boreal_shrub   '  &
               , 'c3_arctic_grass                    '  &
               , 'c3_non-arctic_grass                '  &
               , 'c4_grass                           '  &
               , 'c3_crop                            '  &
               , 'c4_crop                            '  &
#if (defined CROP)
               , 'corn                               '  &
               , 'spring_wheat                       '  &
               , 'winter_wheat                       '  &
               , 'soybean                            '  &
#endif
    /

  data (z0mr(i),i=1,numpft)/ 0.055,0.055,0.055,0.075,0.075,&
         0.055,0.055,0.055,0.120,0.120,0.120,0.120,0.120,&
         0.120,0.120,0.120&
#if (defined CROP)
         ,0.120,0.120,0.120,0.120/
#else      
         /
#endif

  data (displar(i),i=1,numpft)/0.67,0.67,0.67,0.67,0.67,0.67,&
         0.67,0.67,0.68,0.68,0.68,0.68,0.68,0.68,0.68,0.68&
#if (defined CROP)
         ,0.68,0.68,0.68,0.68/
#else      
         /
#endif

  data (dleaf(i),i=1,numpft)/ 0.04,0.04,0.04,0.04,0.04,0.04,&
         0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04&
#if (defined CROP)
         ,0.04,0.04,0.04,0.04/
#else      
         /
#endif

  data (c3psn(i),i=1,numpft)/1,1,1,1,1,1,1,1,1,1,1,1,1,0,1,1&
#if (defined CROP)
       ,0,1,1,1 /
#else      
         /
#endif 

  data (vcmx25(i),i=1,numpft)/51,43,51,75,69,40,51,51,17,17,&
         33,43,43,24,50,50&
#if (defined CROP)
        ,50,50,50,50 /
#else      
         /
#endif 

  data (mp(i),i=1,numpft)/6,6,6,9,9,9,9,9,9,9,9,9,9,5,9,9&
#if (defined CROP)
      ,4,9,9,9/
#else      
         /
#endif

  data (qe25(i),i=1,numpft)/ 0.06,0.06,0.06,0.06,0.06,0.06,&
       0.06,0.06,0.06,0.06,0.06,0.06,0.06,0.04,0.06,0.06&
#if (defined CROP)
      ,0.04,0.06,0.06,0.06/
#else      
         /
#endif

  data (rhol(i,1),i=1,numpft)/0.07,0.07,0.07,0.10,0.10,0.10,&
       0.10,0.10,0.07,0.10,0.10,0.11,0.11,0.11,0.11,0.11&
#if (defined CROP)
      ,0.11,0.11,0.11,0.11/
#else      
         /
#endif


  data (rhol(i,2),i=1,numpft)/ 0.35,0.35,0.35,0.45,0.45,0.45,&
        0.45,0.45,0.35,0.45,0.45,0.35,0.35,0.35,0.35,0.35&
#if (defined CROP)
        ,0.35,0.35,0.35,0.35/
#else      
         /
#endif

  data (rhos(i,1),i=1,numpft) /0.16,0.16,0.16,0.16,0.16,0.16,&
       0.16,0.16,0.16,0.16,0.16,0.31,0.31,0.31,0.31,0.31&
#if (defined CROP)
       ,0.31,0.31,0.31,0.31/  
#else      
         /
#endif


  data (rhos(i,2),i=1,numpft)/0.39,0.39,0.39,0.39,0.39,0.39,&
       0.39,0.39,0.39,0.39,0.39,0.53,0.53,0.53,0.53,0.53&
#if (defined CROP)
      ,0.53,0.53,0.53,0.53/
#else      
         /
#endif

  data (taul(i,1),i=1,numpft)/0.05,0.05,0.05,0.05,0.05,0.05,&
       0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05&
#if (defined CROP)
      ,0.05,0.05,0.05,0.05/
#else      
         /
#endif


  data (taul(i,2),i=1,numpft)/0.10,0.10,0.10,0.25,0.25,0.25,&
      0.25,0.25,0.10,0.25,0.25,0.34,0.34,0.34,0.34,0.34&
#if (defined CROP)
      ,0.34,0.34,0.34,0.34/
#else      
         /
#endif


  data (taus(i,1),i=1,numpft)/0.001,0.001,0.001,0.001,0.001,&
      0.001,0.001,0.001,0.001,0.001,0.001,0.120,0.120,0.120,0.120,0.120&
#if (defined CROP) 
     ,0.120,0.120,0.120,0.120/
#else      
         /
#endif


  data (taus(i,2),i=1,numpft)/ 0.001,0.001,0.001,0.001,0.001,0.001,&
     0.001,0.001,0.001,0.001,0.001,0.250,0.250,0.250,0.250,0.250&
#if (defined CROP)
,0.250,0.250,0.250,0.250/
#else      
         /
#endif

  data (xl(i),i=1,numpft)/0.01, 0.01, 0.01, 0.10, 0.10, 0.01,&
     0.25, 0.25, 0.01, 0.25, 0.25,-0.30,-0.30,-0.30,-0.30,-0.30&
#if (defined CROP)
    ,-0.50, 0.65, 0.65,-0.50/
#else      
         /
#endif

  data (roota_par(i),i=1,numpft)/ 7, 7, 7, 7, 7, 6, 6, 6, 7,&
       7, 7,11,11,11, 6, 6&
#if (defined CROP)
      , 6, 6, 6, 6/
#else      
         /
#endif

  data (rootb_par(i),i=1,numpft)/ 2.0,2.0,2.0,1.0,1.0,2.0,2.0,&
       2.0,1.5,1.5,1.5,2.0,2.0,2.0,3.0,3.0&
#if (defined CROP)
      ,3.0,3.0,3.0,3.0/
#else      
         /
#endif 

  data (slatop(i),i=1,numpft)/0.010,0.008,0.024,0.012,0.012,0.030,&
       0.030,0.030,0.012,0.030,0.030,0.030,0.030,0.030,0.030,0.030&
#if (defined CROP)
      ,0.050,0.070,0.070,0.070/
#else      
         /
#endif 

  data (dsladlai(i),i=1,numpft)/0.00125,0.00100,0.00300,0.00150,0.00150,&
       0.00400,0.00400,0.00400,0.00000,0.00000,0.00000,0.00000,0.00000,&
       0.00000,0.00000,0.00000&
#if (defined CROP)
       ,0.00000,0.00000,0.00000,0.00000/
#else      
         /
#endif 

  data (leafcn(i),i=1,numpft)/35,40,25,30,30,25,25,25,30,25,25,&
       25,25,25,25,25&
#if (defined CROP)
      ,25,25,25,25/
#else      
         /
#endif

 data (flnr(i),i=1,numpft)/0.05,0.04,0.08,0.06,0.06,0.09,0.09,0.09,&
      0.06,0.09,0.09,0.09,0.09,0.09,0.10,0.10&
#if (defined CROP)
     ,0.10,0.20,0.20,0.10/
#else      
         /
#endif

 data (smpso(i),i=1,numpft)/-66000,-66000,-66000,-66000,-66000,-35000,&
     -35000,-35000,-83000,-83000,-83000,-74000,-74000,-74000,-74000,-74000&
#if (defined CROP)
      ,-74000,-74000,-74000,-74000/
#else      
         /
#endif

 data (smpsc(i),i=1,numpft)/-255000,-255000,-255000,-255000,-255000,-224000,&
     -224000,-224000,-428000,-428000,-428000,-275000,-275000,-275000,-275000,-275000&
#if (defined CROP)
     ,-275000,-275000,-275000,-275000/
#else      
         /
#endif

 data(fnitr(i),i=1,numpft)/0.72,0.78,0.79,0.83,0.71,0.66,0.64,0.70,0.62,&
     0.60,0.76,0.68,0.61,0.64,0.61,0.61&
#if (defined CROP)
    ,0.61,0.61,0.61,0.61/
#else       
         /
#endif


 data(woody(i),i=1,numpft)/1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0&
#if (defined CROP)
     ,0,0,0,0/
#else      
         /
#endif


 data(lflitcn(i),i=1,numpft)/70,80,50,60,60,50,50,50,60,50,50,50,50,50,50,50&
#if (defined CROP)
  ,25,25,25,25/ 
#else      
         /
#endif


 data(frootcn(i),i=1,numpft)/42,42,42,42,42,42,42,42,42,42,42,42,42,42,42,42&
#if (defined CROP)
   ,42,42,42,42/
#else      
         /
#endif

 data(livewdcn(i),i=1,numpft)/50,50,50,50,50,50,50,50,50,50,50, 50, 50, 50, 50, 50&
#if (defined CROP)
   ,50,50,50,50/
#else      
         /
#endif

 data(deadwdcn(i),i=1,numpft)/500,500,500,500,500,500,500,500,500,500,500,&
       500,  500,  500,  500,  500&
#if (defined CROP)
     ,500,500,500,500/
#else      
         /
#endif


 data(froot_leaf(i),i=1,numpft)/1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2&
#if (defined CROP)
      ,2,2,2,2/
#else      
         /
#endif

 data(stem_leaf(i),i=1,numpft)/-1.0,-1.0,-1.0,-1.0,-1.0,-1.0,-1.0,-1.0,&
       0.2, 0.2, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0&
#if (defined CROP)
     , 0.0, 0.0, 0.0, 0.0/
#else      
         /
#endif


 data(croot_stem(i),i=1,numpft)/0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,&
       0.3,0.0,0.0,0.0,0.0,0.0&
#if (defined CROP)
       ,0.0,0.0,0.0,0.0/
#else      
         /
#endif

 data(flivewd(i),i=1,numpft)/0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.5,0.5,0.1,&
       0.0,0.0,0.0,0.0,0.0&
#if (defined CROP)
      ,1.0,1.0,1.0,1.0/
#else      
         /
#endif


 data(fcur(i),i=1,numpft)/1,1,0,1,1,0,0,0,1,0,0,0,0,0,0,0&
#if (defined CROP)   
   ,1,1,1,1/
#else      
         /
#endif

 data(lf_flab(i),i=1,numpft)/0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,&
      0.25,0.25,0.25,0.25,0.25,0.25,0.25&
#if (defined CROP) 
     ,0.25,0.25,0.25,0.25/
#else      
         /
#endif

 data(lf_fcel(i),i=1,numpft)/0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,&
      0.5,0.5,0.5,0.5,0.5&
#if (defined CROP)     
     ,0.5,0.5,0.5,0.5/
#else      
         /
#endif

 data(lf_flig(i),i=1,numpft)/0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,&
      0.25,0.25,0.25,0.25,0.25,0.25,0.25&
#if (defined CROP) 
     ,0.25,0.25,0.25,0.25/
#else      
         /
#endif

 data(fr_flab(i),i=1,numpft)/0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,&
      0.25,0.25,0.25,0.25,0.25,0.25,0.25&
#if (defined CROP)
      ,0.25,0.25,0.25,0.25/
#else      
         /
#endif

 data(fr_fcel(i),i=1,numpft)/0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,&
      0.5,0.5,0.5,0.5,0.5&
#if (defined CROP)  
     ,0.5,0.5,0.5,0.5/
#else      
         /
#endif

 data(fr_flig(i),i=1,numpft)/0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,&
      0.25,0.25,0.25,0.25,0.25,0.25,0.25&
#if (defined CROP)
     ,0.25,0.25,0.25,0.25/
#else       
         /
#endif

 data(dw_fcel(i),i=1,numpft)/0.75,0.75,0.75,0.75,0.75,0.75,0.75,0.75,0.75,&
      0.75,0.75,0.75,0.75,0.75,0.75,0.75&
#if (defined CROP)
     ,0.75,0.75,0.75,0.75/
#else      
         /
#endif


 data(dw_flig(i),i=1,numpft)/0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,0.25,&
      0.25,0.25,0.25,0.25,0.25,0.25,0.25&
#if (defined CROP)     
     ,0.25,0.25,0.25,0.25/
#else      
         /
#endif


 data(leaf_long(i),i=1,numpft)/3.0,6.0,1.0,1.5,1.5,1.0,1.0,1.0,1.5,1.0,1.0,&
       1.0,1.0,1.0,1.0,1.0&
#if (defined CROP)   
      ,1.0,1.0,1.0,1.0/
#else      
         /
#endif

 
 data(evergreen(i),i=1,numpft)/1,1,0,1,1,0,0,0,1,0,0,0,0,0,0,0&
#if (defined CROP) 
    ,0,0,0,0/
#else      
         /
#endif

 
 data(stress_decid(i),i=1,numpft)/0,0,0,0,0,1,0,0,0,1,1,1,1,1,1,1&
#if (defined CROP)
    ,0,0,0,0/
#else      
         /
#endif

 
 data(season_decid(i),i=1,numpft)/0,0,1,0,0,0,1,1,0,0,0,0,0,0,0,0&
#if (defined CROP)    
    ,0,0,0,0/
#else      
         /
#endif

 data(resist(i),i=1,numpft)/0.12,0.12,0.12,0.12,0.12,0.12,0.12,0.12,&
     0.12,0.12,0.12,0.12,0.12,0.12,1.00,1.00&
#if (defined CROP)
    ,1.00,1.00,1.00,1.00/
#else      
         /
#endif


 data(pftpar20(i),i=1,numpft)/15,15,15,15,15,15,15,15, 5, 5, 5, 0, 0, 0, 0, 0&
#if (defined CROP)
    , 0, 0, 0, 0/
#else      
         /
#endif


 data(pftpar28(i),i=1,numpft)/ -2.0,  -32.5, 9999.9,   15.5,    3.0,   15.5,&
       -17.0,-1000.0, 9999.9,  -17.0,-1000.0,-1000.0,  -17.0,15.5, 9999.9, 9999.9&
#if (defined CROP)
   , 9999.9, 9999.9, 9999.9, 9999.9/
#else      
         /
#endif


 data(pftpar29(i),i=1,numpft)/ 22.0,  -2.0,  -2.0,1000.0,  18.8,1000.0,  15.5,&
      -2.0,1000.0,1000.0,  -2.0, -17.0,  15.5,1000.0,1000.0,1000.0&
#if (defined CROP)
     ,1000.0,1000.0,1000.0,1000.0/
#else      
         /
#endif

 data(pftpar30(i),i=1,numpft)/900, 600, 350,   0,1200,   0,1200, 350,   0,1200,&
       350,   0,   0,   0,   0,   0&
#if (defined CROP)
    ,   0,   0,   0,   0/
#else      
         /
#endif


 data(pftpar31(i),i=1,numpft)/1000,  23,  23,1000,1000,1000,1000,  23,1000,1000,&
       23,1000,1000,1000,1000,1000&
#if (defined CROP)
      ,1000,1000,1000,1000/
#else      
         /
#endif


 data(graincn(i),i=1,numpft)/0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0&
#if (defined CROP)    
     ,50,50,50,50/
#else      
         /
#endif



  public :: pftconrd ! Read and initialize vegetation (PFT) constants 


!
!EOP
!----------------------------------------------------------------------- 

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: pftconrd
!
! !INTERFACE:
  subroutine pftconrd

    use nanMod    , only : inf



!
! !DESCRIPTION: 
! Read and initialize vegetation (PFT) constants 
!
! !USES:
!
! !ARGUMENTS:
    implicit none
!
! !CALLED FROM:
! routine initialize in module initializeMod
!
! !REVISION HISTORY:
! Created by Gordon Bonan
!
!EOP
!
! !LOCAL VARIABLES:
    integer :: i,n              !loop indices
    integer :: ier              !error code
!-----------------------------------------------------------------------


    ! Set value for last type of tree

    ntree = nbrdlf_dcd_brl_tree  !value for last type of tree

    ! Set value for non-vegetated

!ylu moved to top    noveg = 0  !value


!ylu add

#if (defined CROP)
    npcropmin            = ncorn                ! first prognostic crop
    npcropmax            = nsoybean             ! last prognostic crop in list
#endif


    ! Assign unit number to file. Get local file. 
    ! Open file and read PFT's.
    ! Close and release file.

    crop(:)  = 0
    crop(nc3crop:numpft) = 1
  
!  crop(15) = 1
!    crop(16) = 1


#if (defined CROP)
       mxtmp(:)                      =  0._r8
       mxtmp(ncorn)                  = 30._r8
       mxtmp(nswheat)                = 26._r8
       mxtmp(nwwheat)                = 26._r8
       mxtmp(nsoybean)               = 30._r8

       baset(:)                      =  0._r8
       baset(ncorn)                  =  8._r8
       baset(nswheat)                =  0._r8
       baset(nwwheat)                =  0._r8
       baset(nsoybean)               = 10._r8

       declfact(:)                   = 0.00_r8
     declfact(npcropmin:npcropmax) = 1.05_r8

       bfact(:)                      = 0.00_r8
       bfact(npcropmin:npcropmax)    = 0.10_r8

       aleaff(:)                     = 0._r8

       arootf(:)                     = 0.00_r8
       arootf(ncorn)                 = 0.05_r8
       arootf(nsoybean)              = 0.20_r8

       astemf(:)                     = 0.00_r8
       astemf(nswheat)               = 0.05_r8
       astemf(nwwheat)               = 0.05_r8
       astemf(nsoybean)              = 0.30_r8

       arooti(:)                     = 0.0_r8
       arooti(ncorn)                 = 0.4_r8
       arooti(nswheat)               = 0.3_r8
       arooti(nwwheat)               = 0.3_r8
       arooti(nsoybean)              = 0.5_r8

       fleafi(:)                     = 0.000_r8
       fleafi(ncorn)                 = 0.800_r8
       fleafi(nswheat)               = 0.750_r8
       fleafi(nwwheat)               = 0.425_r8
       fleafi(nsoybean)              = 0.850_r8

       allconsl(:)                   = 0._r8
       allconsl(ncorn)               = 5._r8
       allconsl(nswheat)             = 3._r8
       allconsl(nwwheat)             = 3._r8
       allconsl(nsoybean)            = 2._r8

       allconss(:)                   = 0._r8
       allconss(ncorn)               = 2._r8
       allconss(nswheat)             = 1._r8
       allconss(nwwheat)             = 1._r8
       allconss(nsoybean)            = 5._r8

       ztopmx(:)                     = 0.00_r8
       ztopmx(ncorn)                 = 2.50_r8
       ztopmx(nswheat)               = 1.20_r8
       ztopmx(nwwheat)               = 1.20_r8
       ztopmx(nsoybean)              = 0.75_r8

       laimx(:)                      = 0._r8
       laimx(ncorn)                  = 5._r8
       laimx(nswheat)                = 7._r8
       laimx(nwwheat)                = 7._r8
       laimx(nsoybean)               = 6._r8

       gddmin(:)                     =  0._r8
       gddmin(ncorn)                 = 50._r8
       gddmin(nswheat)               = 50._r8
       gddmin(nwwheat)               = 50._r8
       gddmin(nsoybean)              = 50._r8

       hybgdd(:)                     =    0._r8
       hybgdd(ncorn)                 = 1700._r8
       hybgdd(nswheat)               = 1700._r8
       hybgdd(nwwheat)               = 1700._r8
       hybgdd(nsoybean)              = 1900._r8

       lfemerg(:)                    = 0.00_r8
       lfemerg(ncorn)                = 0.03_r8
       lfemerg(nswheat)              = 0.05_r8
       lfemerg(nwwheat)              = 0.05_r8
       lfemerg(nsoybean)             = 0.03_r8
       grnfill(:)                    = 0.00_r8
       grnfill(ncorn)                = 0.65_r8
       grnfill(nswheat)              = 0.60_r8
       grnfill(nwwheat)              = 0.40_r8
       grnfill(nsoybean)             = 0.70_r8

       mxmat(:)                      = 0
       mxmat(ncorn)                  = 165
       mxmat(nswheat)                = 150
       mxmat(nwwheat)                = 265
       mxmat(nsoybean)               = 150
#endif

    pftname(noveg)      = 'not_vegetated'
       z0mr(noveg)         = 0._r8
       displar(noveg)      = 0._r8
       dleaf(noveg)        = 0._r8
       c3psn(noveg)        = 1._r8
       vcmx25(noveg)       = 0._r8
       mp(noveg)           = 9._r8
       qe25(noveg)         = 0._r8
       rhol(noveg,1)       = 0._r8
       rhol(noveg,2)       = 0._r8
       rhos(noveg,1)       = 0._r8
       rhos(noveg,2)       = 0._r8
       taul(noveg,1)       = 0._r8
       taul(noveg,2)       = 0._r8
       taus(noveg,1)       = 0._r8
       taus(noveg,2)       = 0._r8
       xl(noveg)           = 0._r8
       roota_par(noveg)    = 0._r8
       rootb_par(noveg)    = 0._r8
       crop(noveg)         = 0._r8
       smpso(noveg)        = 0._r8
       smpsc(noveg)        = 0._r8
       fnitr(noveg)        = 0._r8
       slatop(noveg)       = 0._r8
       dsladlai(noveg)     = 0._r8
       leafcn(noveg)       = 1._r8
       flnr(noveg)         = 0._r8
       ! begin variables used only for CN code
       woody(noveg)        = 0._r8
       lflitcn(noveg)      = 1._r8
       frootcn(noveg)      = 1._r8
       livewdcn(noveg)     = 1._r8
       deadwdcn(noveg)     = 1._r8
#if (defined CROP)
       ! begin variables used only for CROP
       graincn(noveg)      = 1._r8
       mxtmp(noveg)        = 0._r8
       baset(noveg)        = 0._r8
       declfact(noveg)     = 0._r8
       bfact(noveg)        = 0._r8
       aleaff(noveg)       = 0._r8
       arootf(noveg)       = 0._r8
       astemf(noveg)       = 0._r8
       arooti(noveg)       = 0._r8
       fleafi(noveg)       = 0._r8
       allconsl(noveg)     = 0._r8
       allconss(noveg)     = 0._r8
       ztopmx(noveg)       = 0._r8
       laimx(noveg)        = 0._r8
       gddmin(noveg)       = 0._r8
       hybgdd(noveg)       = 0._r8
       lfemerg(noveg)      = 0._r8
       grnfill(noveg)      = 0._r8
       mxmat(noveg)        = 0
       ! end variables used only for CROP
#endif
       froot_leaf(noveg)   = 0._r8
       stem_leaf(noveg)    = 0._r8
       croot_stem(noveg)   = 0._r8
       flivewd(noveg)      = 0._r8
       fcur(noveg)         = 0._r8
       lf_flab(noveg)      = 0._r8
       lf_fcel(noveg)      = 0._r8
       lf_flig(noveg)      = 0._r8
       fr_flab(noveg)      = 0._r8
       fr_fcel(noveg)      = 0._r8
       fr_flig(noveg)      = 0._r8
       dw_fcel(noveg)      = 0._r8
       dw_flig(noveg)      = 0._r8
       leaf_long(noveg)    = 0._r8
       evergreen(noveg)    = 0._r8
       stress_decid(noveg) = 0._r8
       season_decid(noveg) = 0._r8
       resist(noveg)       = 1._r8
       pftpar20(noveg) = inf
       pftpar28(noveg) = 9999.9_r8
       pftpar29(noveg) = 1000.0_r8
       pftpar30(noveg) =    0.0_r8
       pftpar31(noveg) = 1000.0_r8


  end subroutine pftconrd

end module pftvarcon

module pftdynMod

!---------------------------------------------------------------------------
!BOP
!
! !MODULE: pftdynMod
!
! !USES:
  use clmtype
  use decompMod   , only : get_proc_bounds
  use clm_varpar  , only : max_pft_per_col
  use shr_kind_mod, only : r8 => shr_kind_r8
!
! !DESCRIPTION:
! Determine pft weights at current time using dynamic landuse datasets.
! ASSUMES that only have one dynamic landuse dataset.
!
! !PUBLIC TYPES:
! implicit none
  private
  save
  public :: pftdyn_init            !not used 02/23/11 ylu
  public :: pftdyn_interp          !not used 02/23/11 ylu
  public :: pftdyn_wbal_init       !not used 02/23/11 ylu
  public :: pftdyn_wbal            !not used 02/23/11 ylu
#ifdef CN
  public :: pftdyn_cnbal           !Call in driver.F
#ifdef CNDV
  public :: pftwt_init             !not used 02/23/11 ylu
  public :: pftwt_interp           !not used 02/23/11 ylu 
#endif
  public :: CNHarvest              !Call in CNEcosystemDynMod.F
  public :: CNHarvestPftToColumn   !not used 02/23/11 ylu
#endif
!
! !REVISION HISTORY:
! Created by Peter Thornton
! slevis modified to handle CNDV and CROP
! 19 May 2009: PET - modified to handle harvest fluxes
!
!EOP
!
! ! PRIVATE TYPES
  real(r8), parameter :: days_per_year = 365._r8
  integer , pointer   :: yearspft(:)
  real(r8), pointer   :: wtpft1(:,:)   
  real(r8), pointer   :: wtpft2(:,:)
  real(r8), pointer   :: harvest(:)   
  real(r8), pointer   :: wtcol_old(:)
  integer :: nt1
  integer :: nt2
  integer :: ntimes
  logical :: do_harvest
  integer :: ncid
!---------------------------------------------------------------------------

contains
  
  
#ifdef CN
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: pftdyn_cnbal
!
! !INTERFACE:
  subroutine pftdyn_cnbal()
!
! !DESCRIPTION:
! modify pft-level state and flux variables to maintain carbon and nitrogen balance with
! dynamic pft-weights.
!
! !USES:
    use shr_kind_mod, only : r8 => shr_kind_r8
    use shr_const_mod,only : SHR_CONST_PDB
    use decompMod   , only : get_proc_bounds
    use clm_varcon  , only : istsoil
#ifdef CROP
    use clm_varcon  , only : istcrop
#endif
    use clm_varpar  , only : numveg, numpft
#if (defined C13)
    use clm_varcon  , only : c13ratio
#endif
!    use clm_time_manager, only : get_step_size
    use globals     , only: dt
!
! !ARGUMENTS:
    implicit none
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: begp, endp    ! proc beginning and ending pft indices
    integer  :: begc, endc    ! proc beginning and ending column indices
    integer  :: begl, endl    ! proc beginning and ending landunit indices
    integer  :: begg, endg    ! proc beginning and ending gridcell indices
    integer  :: pi,p,c,l,g    ! indices
    integer  :: ier           ! error code
    real(r8) :: dwt           ! change in pft weight (relative to column)
!    real(r8) :: dt            ! land model time step (sec)
    real(r8) :: init_h2ocan   ! initial canopy water mass
    real(r8) :: new_h2ocan    ! canopy water mass after weight shift
    real(r8), allocatable :: dwt_leafc_seed(:)       ! pft-level mass gain due to seeding of new area
    real(r8), allocatable :: dwt_leafn_seed(:)       ! pft-level mass gain due to seeding of new area
#if (defined C13)
    real(r8), allocatable :: dwt_leafc13_seed(:)     ! pft-level mass gain due to seeding of new area
#endif
    real(r8), allocatable :: dwt_deadstemc_seed(:)       ! pft-level mass gain due to seeding of new area
    real(r8), allocatable :: dwt_deadstemn_seed(:)       ! pft-level mass gain due to seeding of new area
#if (defined C13)
    real(r8), allocatable :: dwt_deadstemc13_seed(:)     ! pft-level mass gain due to seeding of new area
#endif
    real(r8), allocatable :: dwt_frootc_to_litter(:)       ! pft-level mass loss due to weight shift
    real(r8), allocatable :: dwt_livecrootc_to_litter(:)   ! pft-level mass loss due to weight shift
    real(r8), allocatable :: dwt_deadcrootc_to_litter(:)   ! pft-level mass loss due to weight shift
#if (defined C13)
    real(r8), allocatable, target :: dwt_frootc13_to_litter(:)     ! pft-level mass loss due to weight shift
    real(r8), allocatable, target :: dwt_livecrootc13_to_litter(:) ! pft-level mass loss due to weight shift
    real(r8), allocatable, target :: dwt_deadcrootc13_to_litter(:) ! pft-level mass loss due to weight shift
#endif
    real(r8), allocatable, target :: dwt_frootn_to_litter(:)       ! pft-level mass loss due to weight shift
    real(r8), allocatable, target :: dwt_livecrootn_to_litter(:)   ! pft-level mass loss due to weight shift
    real(r8), allocatable, target :: dwt_deadcrootn_to_litter(:)   ! pft-level mass loss due to weight shift
    real(r8), allocatable :: conv_cflux(:)         ! pft-level mass loss due to weight shift
    real(r8), allocatable :: prod10_cflux(:)       ! pft-level mass loss due to weight shift
    real(r8), allocatable :: prod100_cflux(:)      ! pft-level mass loss due to weight shift
#if (defined C13)
    real(r8), allocatable, target :: conv_c13flux(:)       ! pft-level mass loss due to weight shift
    real(r8), allocatable, target :: prod10_c13flux(:)     ! pft-level mass loss due to weight shift
    real(r8), allocatable, target :: prod100_c13flux(:)    ! pft-level mass loss due to weight shift
#endif
    real(r8), allocatable, target :: conv_nflux(:)         ! pft-level mass loss due to weight shift
    real(r8), allocatable, target :: prod10_nflux(:)       ! pft-level mass loss due to weight shift
    real(r8), allocatable, target :: prod100_nflux(:)      ! pft-level mass loss due to weight shift
#if (defined C13)
    real(r8) :: c3_del13c     ! typical del13C for C3 photosynthesis (permil, relative to PDB)
    real(r8) :: c4_del13c     ! typical del13C for C4 photosynthesis (permil, relative to PDB)
    real(r8) :: c3_r1         ! isotope ratio (13c/12c) for C3 photosynthesis
    real(r8) :: c4_r1         ! isotope ratio (13c/12c) for C4 photosynthesis
    real(r8) :: c3_r2         ! isotope ratio (13c/[12c+13c]) for C3 photosynthesis
    real(r8) :: c4_r2         ! isotope ratio (13c/[12c+13c]) for C4 photosynthesis
#endif
    real(r8) :: t1,t2,wt_new,wt_old
    real(r8) :: init_state, change_state, new_state
	real(r8) :: tot_leaf, pleaf, pstor, pxfer
	real(r8) :: leafc_seed, leafn_seed
	real(r8) :: deadstemc_seed, deadstemn_seed
#if (defined C13)
        real(r8) :: leafc13_seed, deadstemc13_seed
#endif
    real(r8), pointer :: dwt_ptr0, dwt_ptr1, dwt_ptr2, dwt_ptr3, ptr
    real(r8) :: pconv(0:numpft)    ! proportion of deadstem to conversion flux
    real(r8) :: pprod10(0:numpft)  ! proportion of deadstem to 10-yr product pool
    real(r8) :: pprod100(0:numpft) ! proportion of deadstem to 100-yr product pool
    type(landunit_type), pointer :: lptr         ! pointer to landunit derived subtype
    type(column_type),   pointer :: cptr         ! pointer to column derived subtype
    type(pft_type)   ,   pointer :: pptr         ! pointer to pft derived subtype
    character(len=32) :: subname='pftdyn_cbal' ! subroutine name
!-----------------------------------------------------------------------
    
    ! (dangerous hardwiring) (should put this into pftphysiology file)
    ! set deadstem proportions
    ! veg type:      0       1       2       3       4       5       6       7       8       9      10      11      12     &
    !                13      14      15      16
    pconv(0:numveg)   = &
                 (/0.0_r8, 0.6_r8, 0.6_r8, 0.6_r8, 0.6_r8, 0.6_r8, 0.6_r8, 0.6_r8, 0.6_r8, 0.8_r8, 0.8_r8, 0.8_r8, 1.0_r8, &
                   1.0_r8, 1.0_r8, 1.0_r8, 1.0_r8/)
    pprod10(0:numveg) = &
                 (/0.0_r8, 0.3_r8, 0.3_r8, 0.3_r8, 0.4_r8, 0.3_r8, 0.4_r8, 0.3_r8, 0.3_r8, 0.2_r8, 0.2_r8, 0.2_r8, 0.0_r8, &
                   0.0_r8, 0.0_r8, 0.0_r8, 0.0_r8/)
    pprod100(0:numveg) = &
                 (/0.0_r8, 0.1_r8, 0.1_r8, 0.1_r8, 0.0_r8, 0.1_r8, 0.0_r8, 0.1_r8, 0.1_r8, 0.0_r8, 0.0_r8, 0.0_r8, 0.0_r8, &
                   0.0_r8, 0.0_r8, 0.0_r8, 0.0_r8/)
#ifdef CROP
    !                17 - 20 (dangerous hardwiring)
    pconv(numveg+1:numpft)   =    0.0_r8
    pprod10(numveg+1:numpft) =    0.0_r8
    pprod100(numveg+1:numpft) =   0.0_r8
#endif
    
    ! Set pointers into derived type

    lptr => clm3%g%l
    cptr => clm3%g%l%c
    pptr => clm3%g%l%c%p

    ! Get relevant sizes

    call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

    ! Allocate pft-level mass loss arrays
    allocate(dwt_leafc_seed(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_leafc_seed'; call endrun()
    end if
    allocate(dwt_leafn_seed(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_leafn_seed'; call endrun()
    end if
#if (defined C13)
    allocate(dwt_leafc13_seed(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_leafc13_seed'; call endrun()
    end if
#endif
    allocate(dwt_deadstemc_seed(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_deadstemc_seed'; call endrun()
    end if
    allocate(dwt_deadstemn_seed(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_deadstemn_seed'; call endrun()
    end if
#if (defined C13)
    allocate(dwt_deadstemc13_seed(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_deadstemc13_seed'; call endrun()
    end if
#endif
    allocate(dwt_frootc_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_frootc_to_litter'; call endrun()
    end if
    allocate(dwt_livecrootc_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_livecrootc_to_litter'; call endrun()
    end if
    allocate(dwt_deadcrootc_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_deadcrootc_to_litter'; call endrun()
    end if
#if (defined C13)
    allocate(dwt_frootc13_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_frootc13_to_litter'; call endrun()
    end if
    allocate(dwt_livecrootc13_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_livecrootc13_to_litter'; call endrun()
    end if
    allocate(dwt_deadcrootc13_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_deadcrootc13_to_litter'; call endrun()
    end if
#endif
    allocate(dwt_frootn_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_frootn_to_litter'; call endrun()
    end if
    allocate(dwt_livecrootn_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_livecrootn_to_litter'; call endrun()
    end if
    allocate(dwt_deadcrootn_to_litter(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for dwt_deadcrootn_to_litter'; call endrun()
    end if
    allocate(conv_cflux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for conv_cflux'; call endrun()
    end if
    allocate(prod10_cflux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for prod10_cflux'; call endrun()
    end if
    allocate(prod100_cflux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for prod100_cflux'; call endrun()
    end if
#if (defined C13)
    allocate(conv_c13flux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for conv_c13flux'; call endrun()
    end if
    allocate(prod10_c13flux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for prod10_c13flux'; call endrun()
    end if
    allocate(prod100_c13flux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for prod100_c13flux'; call endrun()
    end if
#endif
    allocate(conv_nflux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for conv_nflux'; call endrun()
    end if
    allocate(prod10_nflux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for prod10_nflux'; call endrun()
    end if
    allocate(prod100_nflux(begp:endp), stat=ier)
    if (ier /= 0) then
          write(6,*)subname,' allocation error for prod100_nflux'; call endrun()
    end if

    ! Get time step
!    dt = real( get_step_size(), r8 )

	do p = begp,endp
		! initialize all the pft-level local flux arrays
		dwt_leafc_seed(p) = 0._r8
		dwt_leafn_seed(p) = 0._r8
#if (defined C13)
		dwt_leafc13_seed(p) = 0._r8
#endif
		dwt_deadstemc_seed(p) = 0._r8
		dwt_deadstemn_seed(p) = 0._r8
#if (defined C13)
		dwt_deadstemc13_seed(p) = 0._r8
#endif
		dwt_frootc_to_litter(p) = 0._r8
		dwt_livecrootc_to_litter(p) = 0._r8
		dwt_deadcrootc_to_litter(p) = 0._r8
#if (defined C13)
		dwt_frootc13_to_litter(p) = 0._r8
		dwt_livecrootc13_to_litter(p) = 0._r8
		dwt_deadcrootc13_to_litter(p) = 0._r8
#endif
		dwt_frootn_to_litter(p) = 0._r8
		dwt_livecrootn_to_litter(p) = 0._r8
		dwt_deadcrootn_to_litter(p) = 0._r8
		conv_cflux(p) = 0._r8
		prod10_cflux(p) = 0._r8
		prod100_cflux(p) = 0._r8
#if (defined C13)
		conv_c13flux(p) = 0._r8
		prod10_c13flux(p) = 0._r8
		prod100_c13flux(p) = 0._r8
#endif
		conv_nflux(p) = 0._r8
		prod10_nflux(p) = 0._r8
		prod100_nflux(p) = 0._r8
       
		l = pptr%landunit(p)
		c = pptr%column(p)
#if (defined CNDV) || (! defined CROP)
		if (lptr%itype(l) == istsoil) then ! CNDV incompatible with dynLU
#else
		if (lptr%itype(l) == istsoil .or. lptr%itype(l) == istcrop) then
#endif

			! calculate the change in weight for the timestep
			dwt = pptr%wtcol(p)-wtcol_old(p)

			! PFTs for which weight increases on this timestep
			if (dwt > 0._r8) then

				! first identify PFTs that are initiating on this timestep
				! and set all the necessary state and flux variables
				if (wtcol_old(p) == 0._r8) then

					! set initial conditions for PFT that is being initiated
					! in this time step.  Based on the settings in cnIniTimeVar.

					! pft-level carbon state variables
					pptr%pcs%leafc(p)              = 0._r8
					pptr%pcs%leafc_storage(p)      = 0._r8
					pptr%pcs%leafc_xfer(p)         = 0._r8
					pptr%pcs%frootc(p)             = 0._r8
					pptr%pcs%frootc_storage(p)     = 0._r8
					pptr%pcs%frootc_xfer(p)        = 0._r8
					pptr%pcs%livestemc(p)          = 0._r8
					pptr%pcs%livestemc_storage(p)  = 0._r8
					pptr%pcs%livestemc_xfer(p)     = 0._r8
					pptr%pcs%deadstemc(p)          = 0._r8
					pptr%pcs%deadstemc_storage(p)  = 0._r8
					pptr%pcs%deadstemc_xfer(p)     = 0._r8
					pptr%pcs%livecrootc(p)         = 0._r8
					pptr%pcs%livecrootc_storage(p) = 0._r8
					pptr%pcs%livecrootc_xfer(p)    = 0._r8
					pptr%pcs%deadcrootc(p)         = 0._r8
					pptr%pcs%deadcrootc_storage(p) = 0._r8
					pptr%pcs%deadcrootc_xfer(p)    = 0._r8
					pptr%pcs%gresp_storage(p)      = 0._r8
					pptr%pcs%gresp_xfer(p)         = 0._r8
					pptr%pcs%cpool(p)              = 0._r8
					pptr%pcs%xsmrpool(p)           = 0._r8
					pptr%pcs%pft_ctrunc(p)         = 0._r8
					pptr%pcs%dispvegc(p)           = 0._r8
					pptr%pcs%storvegc(p)           = 0._r8
					pptr%pcs%totvegc(p)            = 0._r8
					pptr%pcs%totpftc(p)            = 0._r8

#if (defined C13)
					! pft-level carbon-13 state variables
					pptr%pc13s%leafc(p)              = 0._r8
					pptr%pc13s%leafc_storage(p)      = 0._r8
					pptr%pc13s%leafc_xfer(p)         = 0._r8
					pptr%pc13s%frootc(p)             = 0._r8
					pptr%pc13s%frootc_storage(p)     = 0._r8
					pptr%pc13s%frootc_xfer(p)        = 0._r8
					pptr%pc13s%livestemc(p)          = 0._r8
					pptr%pc13s%livestemc_storage(p)  = 0._r8
					pptr%pc13s%livestemc_xfer(p)     = 0._r8
					pptr%pc13s%deadstemc(p)          = 0._r8
					pptr%pc13s%deadstemc_storage(p)  = 0._r8
					pptr%pc13s%deadstemc_xfer(p)     = 0._r8
					pptr%pc13s%livecrootc(p)         = 0._r8
					pptr%pc13s%livecrootc_storage(p) = 0._r8
					pptr%pc13s%livecrootc_xfer(p)    = 0._r8
					pptr%pc13s%deadcrootc(p)         = 0._r8
					pptr%pc13s%deadcrootc_storage(p) = 0._r8
					pptr%pc13s%deadcrootc_xfer(p)    = 0._r8
					pptr%pc13s%gresp_storage(p)      = 0._r8
					pptr%pc13s%gresp_xfer(p)         = 0._r8
					pptr%pc13s%cpool(p)              = 0._r8
					pptr%pc13s%xsmrpool(p)           = 0._r8
					pptr%pc13s%pft_ctrunc(p)         = 0._r8
					pptr%pc13s%dispvegc(p)           = 0._r8
					pptr%pc13s%storvegc(p)           = 0._r8
					pptr%pc13s%totvegc(p)            = 0._r8
					pptr%pc13s%totpftc(p)            = 0._r8
#endif

					! pft-level nitrogen state variables
					pptr%pns%leafn(p)	           = 0._r8
					pptr%pns%leafn_storage(p)      = 0._r8
					pptr%pns%leafn_xfer(p)         = 0._r8
					pptr%pns%frootn(p)	           = 0._r8
					pptr%pns%frootn_storage(p)     = 0._r8
					pptr%pns%frootn_xfer(p)        = 0._r8
					pptr%pns%livestemn(p)	       = 0._r8
					pptr%pns%livestemn_storage(p)  = 0._r8
					pptr%pns%livestemn_xfer(p)     = 0._r8
					pptr%pns%deadstemn(p)	       = 0._r8
					pptr%pns%deadstemn_storage(p)  = 0._r8
					pptr%pns%deadstemn_xfer(p)     = 0._r8
					pptr%pns%livecrootn(p)         = 0._r8
					pptr%pns%livecrootn_storage(p) = 0._r8
					pptr%pns%livecrootn_xfer(p)    = 0._r8
					pptr%pns%deadcrootn(p)         = 0._r8
					pptr%pns%deadcrootn_storage(p) = 0._r8
					pptr%pns%deadcrootn_xfer(p)    = 0._r8
					pptr%pns%retransn(p)	       = 0._r8
					pptr%pns%npool(p)	           = 0._r8
					pptr%pns%pft_ntrunc(p)         = 0._r8
					pptr%pns%dispvegn(p)           = 0._r8
					pptr%pns%storvegn(p)           = 0._r8
					pptr%pns%totvegn(p)            = 0._r8
					pptr%pns%totpftn (p)           = 0._r8

					! initialize same flux and epv variables that are set
					! in CNiniTimeVar
					pptr%pcf%psnsun(p) = 0._r8
					pptr%pcf%psnsha(p) = 0._r8
#if (defined C13)
					pptr%pc13f%psnsun(p) = 0._r8
					pptr%pc13f%psnsha(p) = 0._r8
#endif
					pptr%pps%laisun(p) = 0._r8
					pptr%pps%laisha(p) = 0._r8
					pptr%pps%lncsun(p) = 0._r8
					pptr%pps%lncsha(p) = 0._r8
					pptr%pps%vcmxsun(p) = 0._r8
					pptr%pps%vcmxsha(p) = 0._r8
#if (defined C13)
					pptr%pps%alphapsnsun(p) = 0._r8
					pptr%pps%alphapsnsha(p) = 0._r8
#endif

					pptr%pepv%dormant_flag(p) = 1._r8
					pptr%pepv%days_active(p) = 0._r8
					pptr%pepv%onset_flag(p) = 0._r8
					pptr%pepv%onset_counter(p) = 0._r8
					pptr%pepv%onset_gddflag(p) = 0._r8
					pptr%pepv%onset_fdd(p) = 0._r8
					pptr%pepv%onset_gdd(p) = 0._r8
					pptr%pepv%onset_swi(p) = 0.0_r8
					pptr%pepv%offset_flag(p) = 0._r8
					pptr%pepv%offset_counter(p) = 0._r8
					pptr%pepv%offset_fdd(p) = 0._r8
					pptr%pepv%offset_swi(p) = 0._r8
					pptr%pepv%lgsf(p) = 0._r8
					pptr%pepv%bglfr(p) = 0._r8
					pptr%pepv%bgtr(p) = 0._r8
					! difference from CNiniTimeVar: using column-level
					! information to initialize annavg_t2m.
					pptr%pepv%annavg_t2m(p) = cptr%cps%cannavg_t2m(c)
					pptr%pepv%tempavg_t2m(p) = 0._r8
					pptr%pepv%gpp(p) = 0._r8
					pptr%pepv%availc(p) = 0._r8
					pptr%pepv%xsmrpool_recover(p) = 0._r8
#if (defined C13)
					pptr%pepv%xsmrpool_c13ratio(p) = c13ratio
#endif
					pptr%pepv%alloc_pnow(p) = 1._r8
					pptr%pepv%c_allometry(p) = 0._r8
					pptr%pepv%n_allometry(p) = 0._r8
					pptr%pepv%plant_ndemand(p) = 0._r8
					pptr%pepv%tempsum_potential_gpp(p) = 0._r8
					pptr%pepv%annsum_potential_gpp(p) = 0._r8
					pptr%pepv%tempmax_retransn(p) = 0._r8
					pptr%pepv%annmax_retransn(p) = 0._r8
					pptr%pepv%avail_retransn(p) = 0._r8
					pptr%pepv%plant_nalloc(p) = 0._r8
					pptr%pepv%plant_calloc(p) = 0._r8
					pptr%pepv%excess_cflux(p) = 0._r8
					pptr%pepv%downreg(p) = 0._r8
					pptr%pepv%prev_leafc_to_litter(p) = 0._r8
					pptr%pepv%prev_frootc_to_litter(p) = 0._r8
					pptr%pepv%tempsum_npp(p) = 0._r8
					pptr%pepv%annsum_npp(p) = 0._r8
#if (defined C13)
					pptr%pepv%rc13_canair(p) = 0._r8
					pptr%pepv%rc13_psnsun(p) = 0._r8
					pptr%pepv%rc13_psnsha(p) = 0._r8
#endif

				end if  ! end initialization of new pft

				! (still in dwt > 0 block)

				! set the seed sources for leaf and deadstem
				! leaf source is split later between leaf, leaf_storage, leaf_xfer
				leafc_seed   = 0._r8
				leafn_seed   = 0._r8
#if (defined C13)
				leafc13_seed = 0._r8
#endif
				deadstemc_seed   = 0._r8
				deadstemn_seed   = 0._r8
#if (defined C13)
				deadstemc13_seed = 0._r8
#endif
				if (pptr%itype(p) /= 0) then
					leafc_seed = 1._r8
					leafn_seed  = leafc_seed / pftcon%leafcn(pptr%itype(p))
					if (pftcon%woody(pptr%itype(p)) == 1._r8) then
						deadstemc_seed = 0.1_r8
						deadstemn_seed = deadstemc_seed / pftcon%deadwdcn(pptr%itype(p))
					end if

#if (defined C13)
					! 13c state is initialized assuming del13c = -28 permil for C3, and -13 permil for C4.
					! That translates to ratios of (13c/(12c+13c)) of 0.01080455 for C3, and 0.01096945 for C4
					! based on the following formulae: 
					! r1 (13/12) = PDB + (del13c * PDB)/1000.0
					! r2 (13/(13+12)) = r1/(1+r1)
					! PDB = 0.0112372_R8  (ratio of 13C/12C in Pee Dee Belemnite, C isotope standard)
					c3_del13c = -28._r8
					c4_del13c = -13._r8
					c3_r1 = SHR_CONST_PDB + ((c3_del13c*SHR_CONST_PDB)/1000._r8)
					c3_r2 = c3_r1/(1._r8 + c3_r1)
					c4_r1 = SHR_CONST_PDB + ((c4_del13c*SHR_CONST_PDB)/1000._r8)
					c4_r2 = c4_r1/(1._r8 + c4_r1)

					if (pftcon%c3psn(pptr%itype(p)) == 1._r8) then
						leafc13_seed     = leafc_seed     * c3_r2
						deadstemc13_seed = deadstemc_seed * c3_r2
					else
						leafc13_seed     = leafc_seed     * c4_r2
						deadstemc13_seed = deadstemc_seed * c4_r2
					end if 
#endif
				end if

				! When PFT area expands (dwt > 0), the pft-level mass density 
				! is modified to conserve the original pft mass distributed
				! over the new (larger) area, plus a term to account for the 
				! introduction of new seed source for leaf and deadstem
				t1 = wtcol_old(p)/pptr%wtcol(p)
				t2 = dwt/pptr%wtcol(p)

				tot_leaf = pptr%pcs%leafc(p) + pptr%pcs%leafc_storage(p) + pptr%pcs%leafc_xfer(p)
				pleaf = 0._r8
				pstor = 0._r8
				pxfer = 0._r8
				if (tot_leaf /= 0._r8) then
					! when adding seed source to non-zero leaf state, use current proportions
					pleaf = pptr%pcs%leafc(p)/tot_leaf
					pstor = pptr%pcs%leafc_storage(p)/tot_leaf
					pxfer = pptr%pcs%leafc_xfer(p)/tot_leaf
				else
					! when initiating from zero leaf state, use evergreen flag to set proportions
					if (pftcon%evergreen(pptr%itype(p)) == 1._r8) then
						pleaf = 1._r8
					else
						pstor = 1._r8
					end if
				end if 
				pptr%pcs%leafc(p)         = pptr%pcs%leafc(p)*t1         + leafc_seed*pleaf*t2
				pptr%pcs%leafc_storage(p) = pptr%pcs%leafc_storage(p)*t1 + leafc_seed*pstor*t2
				pptr%pcs%leafc_xfer(p)    = pptr%pcs%leafc_xfer(p)*t1    + leafc_seed*pxfer*t2
				pptr%pcs%frootc(p)  		   = pptr%pcs%frootc(p) 			* t1
				pptr%pcs%frootc_storage(p)     = pptr%pcs%frootc_storage(p) 	* t1
				pptr%pcs%frootc_xfer(p) 	   = pptr%pcs%frootc_xfer(p)		* t1
				pptr%pcs%livestemc(p)		   = pptr%pcs%livestemc(p)  		* t1
				pptr%pcs%livestemc_storage(p)  = pptr%pcs%livestemc_storage(p)  * t1
				pptr%pcs%livestemc_xfer(p)     = pptr%pcs%livestemc_xfer(p) 	* t1
				pptr%pcs%deadstemc(p)     = pptr%pcs%deadstemc(p)*t1     + deadstemc_seed*t2
				pptr%pcs%deadstemc_storage(p)  = pptr%pcs%deadstemc_storage(p)  * t1
				pptr%pcs%deadstemc_xfer(p)     = pptr%pcs%deadstemc_xfer(p) 	* t1
				pptr%pcs%livecrootc(p)  	   = pptr%pcs%livecrootc(p) 		* t1
				pptr%pcs%livecrootc_storage(p) = pptr%pcs%livecrootc_storage(p) * t1
				pptr%pcs%livecrootc_xfer(p)    = pptr%pcs%livecrootc_xfer(p)	* t1
				pptr%pcs%deadcrootc(p)  	   = pptr%pcs%deadcrootc(p) 		* t1
				pptr%pcs%deadcrootc_storage(p) = pptr%pcs%deadcrootc_storage(p) * t1
				pptr%pcs%deadcrootc_xfer(p)    = pptr%pcs%deadcrootc_xfer(p)	* t1
				pptr%pcs%gresp_storage(p)	   = pptr%pcs%gresp_storage(p)  	* t1
				pptr%pcs%gresp_xfer(p)  	   = pptr%pcs%gresp_xfer(p) 		* t1
				pptr%pcs%cpool(p)			   = pptr%pcs%cpool(p)  			* t1
				pptr%pcs%xsmrpool(p)		   = pptr%pcs%xsmrpool(p)			* t1
				pptr%pcs%pft_ctrunc(p)  	   = pptr%pcs%pft_ctrunc(p) 		* t1
				pptr%pcs%dispvegc(p)		   = pptr%pcs%dispvegc(p)			* t1
				pptr%pcs%storvegc(p)		   = pptr%pcs%storvegc(p)			* t1
				pptr%pcs%totvegc(p) 		   = pptr%pcs%totvegc(p)			* t1
				pptr%pcs%totpftc(p) 		   = pptr%pcs%totpftc(p)			* t1

#if (defined C13)
				! pft-level carbon-13 state variables 
				tot_leaf = pptr%pc13s%leafc(p) + pptr%pc13s%leafc_storage(p) + pptr%pc13s%leafc_xfer(p)
				pleaf = 0._r8
				pstor = 0._r8
				pxfer = 0._r8
				if (tot_leaf /= 0._r8) then
					pleaf = pptr%pc13s%leafc(p)/tot_leaf
					pstor = pptr%pc13s%leafc_storage(p)/tot_leaf
					pxfer = pptr%pc13s%leafc_xfer(p)/tot_leaf
				else
					! when initiating from zero leaf state, use evergreen flag to set proportions
					if (pftcon%evergreen(pptr%itype(p)) == 1._r8) then
						pleaf = 1._r8
					else
						pstor = 1._r8
					end if
				end if 
				pptr%pc13s%leafc(p)         = pptr%pc13s%leafc(p)*t1         + leafc13_seed*pleaf*t2
				pptr%pc13s%leafc_storage(p) = pptr%pc13s%leafc_storage(p)*t1 + leafc13_seed*pstor*t2
				pptr%pc13s%leafc_xfer(p)    = pptr%pc13s%leafc_xfer(p)*t1    + leafc13_seed*pxfer*t2
				pptr%pc13s%frootc(p)			 = pptr%pc13s%frootc(p) 		* t1
				pptr%pc13s%frootc_storage(p)	         = pptr%pc13s%frootc_storage(p) 	* t1
				pptr%pc13s%frootc_xfer(p)		 = pptr%pc13s%frootc_xfer(p)		* t1
				pptr%pc13s%livestemc(p) 		 = pptr%pc13s%livestemc(p)  		* t1
				pptr%pc13s%livestemc_storage(p)          = pptr%pc13s%livestemc_storage(p)      * t1
				pptr%pc13s%livestemc_xfer(p)	         = pptr%pc13s%livestemc_xfer(p) 	* t1
				pptr%pc13s%deadstemc(p)                  = pptr%pc13s%deadstemc(p)*t1     + deadstemc13_seed*t2
				pptr%pc13s%deadstemc_storage(p)          = pptr%pc13s%deadstemc_storage(p)      * t1
				pptr%pc13s%deadstemc_xfer(p)	         = pptr%pc13s%deadstemc_xfer(p) 	* t1
				pptr%pc13s%livecrootc(p)		 = pptr%pc13s%livecrootc(p) 		* t1
				pptr%pc13s%livecrootc_storage(p)         = pptr%pc13s%livecrootc_storage(p)     * t1
				pptr%pc13s%livecrootc_xfer(p)	         = pptr%pc13s%livecrootc_xfer(p)	* t1
				pptr%pc13s%deadcrootc(p)		 = pptr%pc13s%deadcrootc(p) 		* t1
				pptr%pc13s%deadcrootc_storage(p)         = pptr%pc13s%deadcrootc_storage(p)     * t1
				pptr%pc13s%deadcrootc_xfer(p)	         = pptr%pc13s%deadcrootc_xfer(p)	* t1
				pptr%pc13s%gresp_storage(p) 	         = pptr%pc13s%gresp_storage(p)  	* t1
				pptr%pc13s%gresp_xfer(p)		 = pptr%pc13s%gresp_xfer(p) 		* t1
				pptr%pc13s%cpool(p) 			 = pptr%pc13s%cpool(p)  		* t1
				pptr%pc13s%xsmrpool(p)  		 = pptr%pc13s%xsmrpool(p)		* t1
				pptr%pc13s%pft_ctrunc(p)		 = pptr%pc13s%pft_ctrunc(p) 		* t1
				pptr%pc13s%dispvegc(p)  		 = pptr%pc13s%dispvegc(p)		* t1
				pptr%pc13s%storvegc(p)  		 = pptr%pc13s%storvegc(p)		* t1
				pptr%pc13s%totvegc(p)			 = pptr%pc13s%totvegc(p)		* t1
				pptr%pc13s%totpftc(p)			 = pptr%pc13s%totpftc(p)		* t1
#endif

				tot_leaf = pptr%pns%leafn(p) + pptr%pns%leafn_storage(p) + pptr%pns%leafn_xfer(p)
				pleaf = 0._r8
				pstor = 0._r8
				pxfer = 0._r8
				if (tot_leaf /= 0._r8) then
					pleaf = pptr%pns%leafn(p)/tot_leaf
					pstor = pptr%pns%leafn_storage(p)/tot_leaf
					pxfer = pptr%pns%leafn_xfer(p)/tot_leaf
				else
					! when initiating from zero leaf state, use evergreen flag to set proportions
					if (pftcon%evergreen(pptr%itype(p)) == 1._r8) then
						pleaf = 1._r8
					else
						pstor = 1._r8
					end if
				end if 
				! pft-level nitrogen state variables
				pptr%pns%leafn(p)         = pptr%pns%leafn(p)*t1         + leafn_seed*pleaf*t2
				pptr%pns%leafn_storage(p) = pptr%pns%leafn_storage(p)*t1 + leafn_seed*pstor*t2
				pptr%pns%leafn_xfer(p)    = pptr%pns%leafn_xfer(p)*t1    + leafn_seed*pxfer*t2
				pptr%pns%frootn(p)  		   = pptr%pns%frootn(p) 		* t1
				pptr%pns%frootn_storage(p)         = pptr%pns%frootn_storage(p) 	* t1
				pptr%pns%frootn_xfer(p) 	   = pptr%pns%frootn_xfer(p)		* t1
				pptr%pns%livestemn(p)		   = pptr%pns%livestemn(p)  		* t1
				pptr%pns%livestemn_storage(p)      = pptr%pns%livestemn_storage(p)      * t1
				pptr%pns%livestemn_xfer(p)         = pptr%pns%livestemn_xfer(p) 	* t1
				pptr%pns%deadstemn(p)              = pptr%pns%deadstemn(p)*t1     + deadstemn_seed*t2
				pptr%pns%deadstemn_storage(p)      = pptr%pns%deadstemn_storage(p)      * t1
				pptr%pns%deadstemn_xfer(p)         = pptr%pns%deadstemn_xfer(p) 	* t1
				pptr%pns%livecrootn(p)  	   = pptr%pns%livecrootn(p) 		* t1
				pptr%pns%livecrootn_storage(p)     = pptr%pns%livecrootn_storage(p)     * t1
				pptr%pns%livecrootn_xfer(p)        = pptr%pns%livecrootn_xfer(p)	* t1
				pptr%pns%deadcrootn(p)  	   = pptr%pns%deadcrootn(p) 		* t1
				pptr%pns%deadcrootn_storage(p)     = pptr%pns%deadcrootn_storage(p)     * t1
				pptr%pns%deadcrootn_xfer(p)        = pptr%pns%deadcrootn_xfer(p)        * t1
				pptr%pns%retransn(p)		   = pptr%pns%retransn(p)		* t1
				pptr%pns%npool(p)		   = pptr%pns%npool(p)  		* t1
				pptr%pns%pft_ntrunc(p)  	   = pptr%pns%pft_ntrunc(p)        	* t1
				pptr%pns%dispvegn(p)		   = pptr%pns%dispvegn(p)		* t1
				pptr%pns%storvegn(p)		   = pptr%pns%storvegn(p)		* t1
				pptr%pns%totvegn(p) 		   = pptr%pns%totvegn(p)		* t1
				pptr%pns%totpftn(p) 		   = pptr%pns%totpftn(p)		* t1

				! update temporary seed source arrays
				! These are calculated in terms of the required contributions from
				! column-level seed source
				dwt_leafc_seed(p)   = leafc_seed   * dwt
#if (defined C13)
				dwt_leafc13_seed(p) = leafc13_seed * dwt
#endif
				dwt_leafn_seed(p)   = leafn_seed   * dwt
				dwt_deadstemc_seed(p)   = deadstemc_seed   * dwt
#if (defined C13)
				dwt_deadstemc13_seed(p) = deadstemc13_seed * dwt
#endif
				dwt_deadstemn_seed(p)   = deadstemn_seed   * dwt

			else if (dwt < 0._r8) then

				! if the pft lost weight on the timestep, then the carbon and nitrogen state
				! variables are directed to litter, CWD, and wood product pools.

				! N.B. : the conv_cflux, prod10_cflux, and prod100_cflux fluxes are accumulated
				! as negative values, but the fluxes for pft-to-litter are accumulated as 
				! positive values

				! set local weight variables for this pft
				wt_new = pptr%wtcol(p)
				wt_old = wtcol_old(p)

				!---------------
				! C state update
				!---------------

				! leafc 
				ptr => pptr%pcs%leafc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! leafc_storage 
				ptr => pptr%pcs%leafc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! leafc_xfer 
				ptr => pptr%pcs%leafc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! frootc 
				ptr => pptr%pcs%frootc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_frootc_to_litter(p) = dwt_frootc_to_litter(p) - change_state
				else
					ptr = 0._r8
					dwt_frootc_to_litter(p) = dwt_frootc_to_litter(p) + init_state
				end if

				! frootc_storage 
				ptr => pptr%pcs%frootc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! frootc_xfer 
				ptr => pptr%pcs%frootc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! livestemc 
				ptr => pptr%pcs%livestemc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! livestemc_storage 
				ptr => pptr%pcs%livestemc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! livestemc_xfer 
				ptr => pptr%pcs%livestemc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! deadstemc 
				ptr => pptr%pcs%deadstemc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state*pconv(pptr%itype(p))
					prod10_cflux(p) = prod10_cflux(p) + change_state*pprod10(pptr%itype(p))
					prod100_cflux(p) = prod100_cflux(p) + change_state*pprod100(pptr%itype(p))
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state*pconv(pptr%itype(p))
					prod10_cflux(p) = prod10_cflux(p) - init_state*pprod10(pptr%itype(p))
					prod100_cflux(p) = prod100_cflux(p) - init_state*pprod100(pptr%itype(p))
				end if

				! deadstemc_storage 
				ptr => pptr%pcs%deadstemc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! deadstemc_xfer 
				ptr => pptr%pcs%deadstemc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! livecrootc 
				ptr => pptr%pcs%livecrootc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_livecrootc_to_litter(p) = dwt_livecrootc_to_litter(p) - change_state
				else
					ptr = 0._r8
					dwt_livecrootc_to_litter(p) = dwt_livecrootc_to_litter(p) + init_state
				end if

				! livecrootc_storage 
				ptr => pptr%pcs%livecrootc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! livecrootc_xfer 
				ptr => pptr%pcs%livecrootc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! deadcrootc 
				ptr => pptr%pcs%deadcrootc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_deadcrootc_to_litter(p) = dwt_deadcrootc_to_litter(p) - change_state
				else
					ptr = 0._r8
					dwt_deadcrootc_to_litter(p) = dwt_deadcrootc_to_litter(p) + init_state
				end if

				! deadcrootc_storage 
				ptr => pptr%pcs%deadcrootc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! deadcrootc_xfer 
				ptr => pptr%pcs%deadcrootc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! gresp_storage 
				ptr => pptr%pcs%gresp_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! gresp_xfer 
				ptr => pptr%pcs%gresp_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! cpool 
				ptr => pptr%pcs%cpool(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! xsmrpool 
				ptr => pptr%pcs%xsmrpool(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

				! pft_ctrunc 
				ptr => pptr%pcs%pft_ctrunc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					conv_cflux(p) = conv_cflux(p) + change_state
				else
					ptr = 0._r8
					conv_cflux(p) = conv_cflux(p) - init_state
				end if

#if (defined C13)
				!-----------------
				! C13 state update
				!-----------------

				! set pointers to the conversion and product pool fluxes for this pft
				! dwt_ptr0 is reserved for local assignment to dwt_xxx_to_litter fluxes
				dwt_ptr1 => conv_c13flux(p)
				dwt_ptr2 => prod10_c13flux(p)
				dwt_ptr3 => prod100_c13flux(p)

				! leafc 
				ptr => pptr%pc13s%leafc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! leafc_storage 
				ptr => pptr%pc13s%leafc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! leafc_xfer 
				ptr => pptr%pc13s%leafc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! frootc 
				ptr => pptr%pc13s%frootc(p)
				dwt_ptr0 => dwt_frootc13_to_litter(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr0 = dwt_ptr0 - change_state
				else
					ptr = 0._r8
					dwt_ptr0 = dwt_ptr0 + init_state
				end if

				! frootc_storage 
				ptr => pptr%pc13s%frootc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! frootc_xfer 
				ptr => pptr%pc13s%frootc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livestemc 
				ptr => pptr%pc13s%livestemc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livestemc_storage 
				ptr => pptr%pc13s%livestemc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livestemc_xfer 
				ptr => pptr%pc13s%livestemc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! deadstemc 
				ptr => pptr%pc13s%deadstemc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state*pconv(pptr%itype(p))
					dwt_ptr2 = dwt_ptr2 + change_state*pprod10(pptr%itype(p))
					dwt_ptr3 = dwt_ptr3 + change_state*pprod100(pptr%itype(p))
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state*pconv(pptr%itype(p))
					dwt_ptr2 = dwt_ptr2 - init_state*pprod10(pptr%itype(p))
					dwt_ptr3 = dwt_ptr3 - init_state*pprod100(pptr%itype(p))
				end if

				! deadstemc_storage 
				ptr => pptr%pc13s%deadstemc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! deadstemc_xfer 
				ptr => pptr%pc13s%deadstemc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livecrootc 
				ptr => pptr%pc13s%livecrootc(p)
				dwt_ptr0 => dwt_livecrootc13_to_litter(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr0 = dwt_ptr0 - change_state
				else
					ptr = 0._r8
					dwt_ptr0 = dwt_ptr0 + init_state
				end if

				! livecrootc_storage 
				ptr => pptr%pc13s%livecrootc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livecrootc_xfer 
				ptr => pptr%pc13s%livecrootc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! deadcrootc 
				ptr => pptr%pc13s%deadcrootc(p)
				dwt_ptr0 => dwt_deadcrootc13_to_litter(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr0 = dwt_ptr0 - change_state
				else
					ptr = 0._r8
					dwt_ptr0 = dwt_ptr0 + init_state
				end if

				! deadcrootc_storage 
				ptr => pptr%pc13s%deadcrootc_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! deadcrootc_xfer 
				ptr => pptr%pc13s%deadcrootc_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! gresp_storage 
				ptr => pptr%pc13s%gresp_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! gresp_xfer 
				ptr => pptr%pc13s%gresp_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! cpool 
				ptr => pptr%pc13s%cpool(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! pft_ctrunc 
				ptr => pptr%pc13s%pft_ctrunc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if
#endif

				!---------------
				! N state update
				!---------------

				! set pointers to the conversion and product pool fluxes for this pft
				! dwt_ptr0 is reserved for local assignment to dwt_xxx_to_litter fluxes
				dwt_ptr1 => conv_nflux(p)
				dwt_ptr2 => prod10_nflux(p)
				dwt_ptr3 => prod100_nflux(p)

				! leafn 
				ptr => pptr%pns%leafn(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! leafn_storage  
				ptr => pptr%pns%leafn_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! leafn_xfer  
				ptr => pptr%pns%leafn_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! frootn 
				ptr => pptr%pns%frootn(p)
				dwt_ptr0 => dwt_frootn_to_litter(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr0 = dwt_ptr0 - change_state
				else
					ptr = 0._r8
					dwt_ptr0 = dwt_ptr0 + init_state
				end if

				! frootn_storage 
				ptr => pptr%pns%frootn_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! frootn_xfer  
				ptr => pptr%pns%frootn_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livestemn  
				ptr => pptr%pns%livestemn(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livestemn_storage 
				ptr => pptr%pns%livestemn_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livestemn_xfer 
				ptr => pptr%pns%livestemn_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! deadstemn 
				ptr => pptr%pns%deadstemn(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state*pconv(pptr%itype(p))
					dwt_ptr2 = dwt_ptr2 + change_state*pprod10(pptr%itype(p))
					dwt_ptr3 = dwt_ptr3 + change_state*pprod100(pptr%itype(p))
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state*pconv(pptr%itype(p))
					dwt_ptr2 = dwt_ptr2 - init_state*pprod10(pptr%itype(p))
					dwt_ptr3 = dwt_ptr3 - init_state*pprod100(pptr%itype(p))
				end if

				! deadstemn_storage 
				ptr => pptr%pns%deadstemn_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! deadstemn_xfer 
				ptr => pptr%pns%deadstemn_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livecrootn 
				ptr => pptr%pns%livecrootn(p)
				dwt_ptr0 => dwt_livecrootn_to_litter(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr0 = dwt_ptr0 - change_state
				else
					ptr = 0._r8
					dwt_ptr0 = dwt_ptr0 + init_state
				end if

				! livecrootn_storage  
				ptr => pptr%pns%livecrootn_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! livecrootn_xfer  
				ptr => pptr%pns%livecrootn_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! deadcrootn 
				ptr => pptr%pns%deadcrootn(p)
				dwt_ptr0 => dwt_deadcrootn_to_litter(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr0 = dwt_ptr0 - change_state
				else
					ptr = 0._r8
					dwt_ptr0 = dwt_ptr0 + init_state
				end if

				! deadcrootn_storage  
				ptr => pptr%pns%deadcrootn_storage(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! deadcrootn_xfer  
				ptr => pptr%pns%deadcrootn_xfer(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! retransn  
				ptr => pptr%pns%retransn(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if

				! npool  
				ptr => pptr%pns%npool(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if
				
				! pft_ntrunc  
				ptr => pptr%pns%pft_ntrunc(p)
				init_state = ptr*wt_old
				change_state = ptr*dwt
				new_state = init_state+change_state
				if (wt_new /= 0._r8) then
					ptr = new_state/wt_new
					dwt_ptr1 = dwt_ptr1 + change_state
				else
					ptr = 0._r8
					dwt_ptr1 = dwt_ptr1 - init_state
				end if
				
			end if       ! weight decreasing
		end if           ! is soil
	end do               ! pft loop
    
	! calculate column-level seeding fluxes
	do pi = 1,max_pft_per_col
		do c = begc, endc
			if ( pi <=  cptr%npfts(c) ) then
				p = cptr%pfti(c) + pi - 1
				
				! C fluxes
				cptr%ccf%dwt_seedc_to_leaf(c) = cptr%ccf%dwt_seedc_to_leaf(c) + dwt_leafc_seed(p)/dt
				cptr%ccf%dwt_seedc_to_deadstem(c) = cptr%ccf%dwt_seedc_to_deadstem(c) &
                                                                    + dwt_deadstemc_seed(p)/dt
				
#if (defined C13)
				! C13 fluxes
                                cptr%cc13f%dwt_seedc_to_leaf(c) = cptr%cc13f%dwt_seedc_to_leaf(c) + dwt_leafc13_seed(p)/dt
                                cptr%cc13f%dwt_seedc_to_deadstem(c) = cptr%cc13f%dwt_seedc_to_deadstem(c) &
                                                                      + dwt_deadstemc13_seed(p)/dt
#endif
				
				! N fluxes
				cptr%cnf%dwt_seedn_to_leaf(c) = cptr%cnf%dwt_seedn_to_leaf(c) + dwt_leafn_seed(p)/dt
				cptr%cnf%dwt_seedn_to_deadstem(c) = cptr%cnf%dwt_seedn_to_deadstem(c) &
                                                                    + dwt_deadstemn_seed(p)/dt
			end if
		end do
	end do


	! calculate pft-to-column for fluxes into litter and CWD pools
	do pi = 1,max_pft_per_col
		do c = begc, endc
			if ( pi <=  cptr%npfts(c) ) then
				p = cptr%pfti(c) + pi - 1

				! fine root litter carbon fluxes
				cptr%ccf%dwt_frootc_to_litr1c(c) = cptr%ccf%dwt_frootc_to_litr1c(c) + &
                                                            (dwt_frootc_to_litter(p)*pftcon%fr_flab(pptr%itype(p)))/dt
				cptr%ccf%dwt_frootc_to_litr2c(c) = cptr%ccf%dwt_frootc_to_litr2c(c) + &
                                                            (dwt_frootc_to_litter(p)*pftcon%fr_fcel(pptr%itype(p)))/dt
				cptr%ccf%dwt_frootc_to_litr3c(c) = cptr%ccf%dwt_frootc_to_litr3c(c) + &
                                                            (dwt_frootc_to_litter(p)*pftcon%fr_flig(pptr%itype(p)))/dt

#if (defined C13)
				! fine root litter C13 fluxes
				cptr%cc13f%dwt_frootc_to_litr1c(c) = cptr%cc13f%dwt_frootc_to_litr1c(c) + &
                                                            (dwt_frootc13_to_litter(p)*pftcon%fr_flab(pptr%itype(p)))/dt
				cptr%cc13f%dwt_frootc_to_litr2c(c) = cptr%cc13f%dwt_frootc_to_litr2c(c) + &
                                                            (dwt_frootc13_to_litter(p)*pftcon%fr_fcel(pptr%itype(p)))/dt
				cptr%cc13f%dwt_frootc_to_litr3c(c) = cptr%cc13f%dwt_frootc_to_litr3c(c) + &
                                                            (dwt_frootc13_to_litter(p)*pftcon%fr_flig(pptr%itype(p)))/dt
#endif

				! fine root litter nitrogen fluxes
				cptr%cnf%dwt_frootn_to_litr1n(c) = cptr%cnf%dwt_frootn_to_litr1n(c) + &
                                                            (dwt_frootn_to_litter(p)*pftcon%fr_flab(pptr%itype(p)))/dt
				cptr%cnf%dwt_frootn_to_litr2n(c) = cptr%cnf%dwt_frootn_to_litr2n(c) + &
                                                            (dwt_frootn_to_litter(p)*pftcon%fr_fcel(pptr%itype(p)))/dt
				cptr%cnf%dwt_frootn_to_litr3n(c) = cptr%cnf%dwt_frootn_to_litr3n(c) + &
                                                            (dwt_frootn_to_litter(p)*pftcon%fr_flig(pptr%itype(p)))/dt

				! livecroot fluxes to cwd
				cptr%ccf%dwt_livecrootc_to_cwdc(c) = cptr%ccf%dwt_livecrootc_to_cwdc(c) + &
                                                            (dwt_livecrootc_to_litter(p))/dt
#if (defined C13)
				cptr%cc13f%dwt_livecrootc_to_cwdc(c) = cptr%cc13f%dwt_livecrootc_to_cwdc(c) + &
                                                            (dwt_livecrootc13_to_litter(p))/dt
#endif
				cptr%cnf%dwt_livecrootn_to_cwdn(c) = cptr%cnf%dwt_livecrootn_to_cwdn(c) + &
                                                            (dwt_livecrootn_to_litter(p))/dt

				! deadcroot fluxes to cwd
				cptr%ccf%dwt_deadcrootc_to_cwdc(c) = cptr%ccf%dwt_deadcrootc_to_cwdc(c) + &
                                                            (dwt_deadcrootc_to_litter(p))/dt
#if (defined C13)
				cptr%cc13f%dwt_deadcrootc_to_cwdc(c) = cptr%cc13f%dwt_deadcrootc_to_cwdc(c) + &
                                                            (dwt_deadcrootc13_to_litter(p))/dt
#endif
				cptr%cnf%dwt_deadcrootn_to_cwdn(c) = cptr%cnf%dwt_deadcrootn_to_cwdn(c) + &
                                                            (dwt_deadcrootn_to_litter(p))/dt
			end if
		end do
	end do

	! calculate pft-to-column for fluxes into product pools and conversion flux
	do pi = 1,max_pft_per_col
		do c = begc,endc
			if (pi <= cptr%npfts(c)) then
				p = cptr%pfti(c) + pi - 1

				! column-level fluxes are accumulated as positive fluxes.
				! column-level C flux updates
				cptr%ccf%dwt_conv_cflux(c) = cptr%ccf%dwt_conv_cflux(c) - conv_cflux(p)/dt
				cptr%ccf%dwt_prod10c_gain(c) = cptr%ccf%dwt_prod10c_gain(c) - prod10_cflux(p)/dt
				cptr%ccf%dwt_prod100c_gain(c) = cptr%ccf%dwt_prod100c_gain(c) - prod100_cflux(p)/dt

#if (defined C13)
				! column-level C13 flux updates
				cptr%cc13f%dwt_conv_cflux(c) = cptr%cc13f%dwt_conv_cflux(c) - conv_c13flux(p)/dt
				cptr%cc13f%dwt_prod10c_gain(c) = cptr%cc13f%dwt_prod10c_gain(c) - prod10_c13flux(p)/dt
				cptr%cc13f%dwt_prod100c_gain(c) = cptr%cc13f%dwt_prod100c_gain(c) - prod100_c13flux(p)/dt
#endif

				! column-level N flux updates
				cptr%cnf%dwt_conv_nflux(c) = cptr%cnf%dwt_conv_nflux(c) - conv_nflux(p)/dt
				cptr%cnf%dwt_prod10n_gain(c) = cptr%cnf%dwt_prod10n_gain(c) - prod10_nflux(p)/dt
				cptr%cnf%dwt_prod100n_gain(c) = cptr%cnf%dwt_prod100n_gain(c) - prod100_nflux(p)/dt

			end if
		end do
	end do

	! Deallocate pft-level flux arrays
        deallocate(dwt_leafc_seed)
        deallocate(dwt_leafn_seed)
#if (defined C13)
        deallocate(dwt_leafc13_seed)
#endif
        deallocate(dwt_deadstemc_seed)
        deallocate(dwt_deadstemn_seed)
#if (defined C13)
        deallocate(dwt_deadstemc13_seed)
#endif
	deallocate(dwt_frootc_to_litter)
	deallocate(dwt_livecrootc_to_litter)
	deallocate(dwt_deadcrootc_to_litter)
#if (defined C13)
	deallocate(dwt_frootc13_to_litter)
	deallocate(dwt_livecrootc13_to_litter)
	deallocate(dwt_deadcrootc13_to_litter)
#endif
	deallocate(dwt_frootn_to_litter)
	deallocate(dwt_livecrootn_to_litter)
	deallocate(dwt_deadcrootn_to_litter)
	deallocate(conv_cflux)
	deallocate(prod10_cflux)
	deallocate(prod100_cflux)
#if (defined C13)
	deallocate(conv_c13flux)
	deallocate(prod10_c13flux)
	deallocate(prod100_c13flux)
#endif
	deallocate(conv_nflux)
	deallocate(prod10_nflux)
	deallocate(prod100_nflux)
    
end subroutine pftdyn_cnbal
#endif


!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: CNHarvest
!
! !INTERFACE:
subroutine CNHarvest (num_soilc, filter_soilc, num_soilp, filter_soilp)
!
! !DESCRIPTION:
! Harvest mortality routine for coupled carbon-nitrogen code (CN)
!
! !USES:
   use clmtype
   use pftvarcon, only : noveg, nbrdlf_evr_shrub
!
! !ARGUMENTS:
   implicit none
   integer, intent(in) :: num_soilc       ! number of soil columns in filter
   integer, intent(in) :: filter_soilc(:) ! column filter for soil points
   integer, intent(in) :: num_soilp       ! number of soil pfts in filter
   integer, intent(in) :: filter_soilp(:) ! pft filter for soil points
!
! !CALLED FROM:
! subroutine CNEcosystemDyn
!
! !REVISION HISTORY:
! 3/29/04: Created by Peter Thornton
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arrays
   integer , pointer :: pgridcell(:)   ! pft-level index into gridcell-level quantities
   integer , pointer :: ivt(:)         ! pft vegetation type

   real(r8), pointer :: leafc(:)              ! (gC/m2) leaf C
   real(r8), pointer :: frootc(:)             ! (gC/m2) fine root C
   real(r8), pointer :: livestemc(:)          ! (gC/m2) live stem C
   real(r8), pointer :: deadstemc(:)          ! (gC/m2) dead stem C
   real(r8), pointer :: livecrootc(:)         ! (gC/m2) live coarse root C
   real(r8), pointer :: deadcrootc(:)         ! (gC/m2) dead coarse root C
   real(r8), pointer :: xsmrpool(:)           ! (gC/m2) abstract C pool to meet excess MR demand
   real(r8), pointer :: leafc_storage(:)      ! (gC/m2) leaf C storage
   real(r8), pointer :: frootc_storage(:)     ! (gC/m2) fine root C storage
   real(r8), pointer :: livestemc_storage(:)  ! (gC/m2) live stem C storage
   real(r8), pointer :: deadstemc_storage(:)  ! (gC/m2) dead stem C storage
   real(r8), pointer :: livecrootc_storage(:) ! (gC/m2) live coarse root C storage
   real(r8), pointer :: deadcrootc_storage(:) ! (gC/m2) dead coarse root C storage
   real(r8), pointer :: gresp_storage(:)      ! (gC/m2) growth respiration storage
   real(r8), pointer :: leafc_xfer(:)         ! (gC/m2) leaf C transfer
   real(r8), pointer :: frootc_xfer(:)        ! (gC/m2) fine root C transfer
   real(r8), pointer :: livestemc_xfer(:)     ! (gC/m2) live stem C transfer
   real(r8), pointer :: deadstemc_xfer(:)     ! (gC/m2) dead stem C transfer
   real(r8), pointer :: livecrootc_xfer(:)    ! (gC/m2) live coarse root C transfer
   real(r8), pointer :: deadcrootc_xfer(:)    ! (gC/m2) dead coarse root C transfer
   real(r8), pointer :: gresp_xfer(:)         ! (gC/m2) growth respiration transfer
   real(r8), pointer :: leafn(:)              ! (gN/m2) leaf N
   real(r8), pointer :: frootn(:)             ! (gN/m2) fine root N
   real(r8), pointer :: livestemn(:)          ! (gN/m2) live stem N
   real(r8), pointer :: deadstemn(:)          ! (gN/m2) dead stem N
   real(r8), pointer :: livecrootn(:)         ! (gN/m2) live coarse root N
   real(r8), pointer :: deadcrootn(:)         ! (gN/m2) dead coarse root N
   real(r8), pointer :: retransn(:)           ! (gN/m2) plant pool of retranslocated N
   real(r8), pointer :: leafn_storage(:)      ! (gN/m2) leaf N storage
   real(r8), pointer :: frootn_storage(:)     ! (gN/m2) fine root N storage
   real(r8), pointer :: livestemn_storage(:)  ! (gN/m2) live stem N storage
   real(r8), pointer :: deadstemn_storage(:)  ! (gN/m2) dead stem N storage
   real(r8), pointer :: livecrootn_storage(:) ! (gN/m2) live coarse root N storage
   real(r8), pointer :: deadcrootn_storage(:) ! (gN/m2) dead coarse root N storage
   real(r8), pointer :: leafn_xfer(:)         ! (gN/m2) leaf N transfer
   real(r8), pointer :: frootn_xfer(:)        ! (gN/m2) fine root N transfer
   real(r8), pointer :: livestemn_xfer(:)     ! (gN/m2) live stem N transfer
   real(r8), pointer :: deadstemn_xfer(:)     ! (gN/m2) dead stem N transfer
   real(r8), pointer :: livecrootn_xfer(:)    ! (gN/m2) live coarse root N transfer
   real(r8), pointer :: deadcrootn_xfer(:)    ! (gN/m2) dead coarse root N transfer
!
! local pointers to implicit in/out arrays
!
! local pointers to implicit out arrays
   real(r8), pointer :: hrv_leafc_to_litter(:)
   real(r8), pointer :: hrv_frootc_to_litter(:)
   real(r8), pointer :: hrv_livestemc_to_litter(:)
   real(r8), pointer :: hrv_deadstemc_to_prod10c(:)
   real(r8), pointer :: hrv_deadstemc_to_prod100c(:)
   real(r8), pointer :: hrv_livecrootc_to_litter(:)
   real(r8), pointer :: hrv_deadcrootc_to_litter(:)
   real(r8), pointer :: hrv_xsmrpool_to_atm(:)
   real(r8), pointer :: hrv_leafc_storage_to_litter(:)
   real(r8), pointer :: hrv_frootc_storage_to_litter(:)
   real(r8), pointer :: hrv_livestemc_storage_to_litter(:)
   real(r8), pointer :: hrv_deadstemc_storage_to_litter(:)
   real(r8), pointer :: hrv_livecrootc_storage_to_litter(:)
   real(r8), pointer :: hrv_deadcrootc_storage_to_litter(:)
   real(r8), pointer :: hrv_gresp_storage_to_litter(:)
   real(r8), pointer :: hrv_leafc_xfer_to_litter(:)
   real(r8), pointer :: hrv_frootc_xfer_to_litter(:)
   real(r8), pointer :: hrv_livestemc_xfer_to_litter(:)
   real(r8), pointer :: hrv_deadstemc_xfer_to_litter(:)
   real(r8), pointer :: hrv_livecrootc_xfer_to_litter(:)
   real(r8), pointer :: hrv_deadcrootc_xfer_to_litter(:)
   real(r8), pointer :: hrv_gresp_xfer_to_litter(:)
   real(r8), pointer :: hrv_leafn_to_litter(:)
   real(r8), pointer :: hrv_frootn_to_litter(:)
   real(r8), pointer :: hrv_livestemn_to_litter(:)
   real(r8), pointer :: hrv_deadstemn_to_prod10n(:)
   real(r8), pointer :: hrv_deadstemn_to_prod100n(:)
   real(r8), pointer :: hrv_livecrootn_to_litter(:)
   real(r8), pointer :: hrv_deadcrootn_to_litter(:)
   real(r8), pointer :: hrv_retransn_to_litter(:)
   real(r8), pointer :: hrv_leafn_storage_to_litter(:)
   real(r8), pointer :: hrv_frootn_storage_to_litter(:)
   real(r8), pointer :: hrv_livestemn_storage_to_litter(:)
   real(r8), pointer :: hrv_deadstemn_storage_to_litter(:)
   real(r8), pointer :: hrv_livecrootn_storage_to_litter(:)
   real(r8), pointer :: hrv_deadcrootn_storage_to_litter(:)
   real(r8), pointer :: hrv_leafn_xfer_to_litter(:)
   real(r8), pointer :: hrv_frootn_xfer_to_litter(:)
   real(r8), pointer :: hrv_livestemn_xfer_to_litter(:)
   real(r8), pointer :: hrv_deadstemn_xfer_to_litter(:)
   real(r8), pointer :: hrv_livecrootn_xfer_to_litter(:)
   real(r8), pointer :: hrv_deadcrootn_xfer_to_litter(:)
!
! !OTHER LOCAL VARIABLES:
   integer :: p                         ! pft index
   integer :: g                         ! gridcell index
   integer :: fp                        ! pft filter index
   real(r8):: am                        ! rate for fractional harvest mortality (1/yr)
   real(r8):: m                         ! rate for fractional harvest mortality (1/s)
   real(r8) :: pprod10(1:8)   ! proportion of deadstem to 10-yr product pool  (for tree pfts - 1 through 8)
!EOP
!-----------------------------------------------------------------------

   ! assign local pointers to pft-level arrays
   pgridcell                      => clm3%g%l%c%p%gridcell
   
   ivt                            => clm3%g%l%c%p%itype
   leafc                          => clm3%g%l%c%p%pcs%leafc
   frootc                         => clm3%g%l%c%p%pcs%frootc
   livestemc                      => clm3%g%l%c%p%pcs%livestemc
   deadstemc                      => clm3%g%l%c%p%pcs%deadstemc
   livecrootc                     => clm3%g%l%c%p%pcs%livecrootc
   deadcrootc                     => clm3%g%l%c%p%pcs%deadcrootc
   xsmrpool                       => clm3%g%l%c%p%pcs%xsmrpool
   leafc_storage                  => clm3%g%l%c%p%pcs%leafc_storage
   frootc_storage                 => clm3%g%l%c%p%pcs%frootc_storage
   livestemc_storage              => clm3%g%l%c%p%pcs%livestemc_storage
   deadstemc_storage              => clm3%g%l%c%p%pcs%deadstemc_storage
   livecrootc_storage             => clm3%g%l%c%p%pcs%livecrootc_storage
   deadcrootc_storage             => clm3%g%l%c%p%pcs%deadcrootc_storage
   gresp_storage                  => clm3%g%l%c%p%pcs%gresp_storage
   leafc_xfer                     => clm3%g%l%c%p%pcs%leafc_xfer
   frootc_xfer                    => clm3%g%l%c%p%pcs%frootc_xfer
   livestemc_xfer                 => clm3%g%l%c%p%pcs%livestemc_xfer
   deadstemc_xfer                 => clm3%g%l%c%p%pcs%deadstemc_xfer
   livecrootc_xfer                => clm3%g%l%c%p%pcs%livecrootc_xfer
   deadcrootc_xfer                => clm3%g%l%c%p%pcs%deadcrootc_xfer
   gresp_xfer                     => clm3%g%l%c%p%pcs%gresp_xfer
   leafn                          => clm3%g%l%c%p%pns%leafn
   frootn                         => clm3%g%l%c%p%pns%frootn
   livestemn                      => clm3%g%l%c%p%pns%livestemn
   deadstemn                      => clm3%g%l%c%p%pns%deadstemn
   livecrootn                     => clm3%g%l%c%p%pns%livecrootn
   deadcrootn                     => clm3%g%l%c%p%pns%deadcrootn
   retransn                       => clm3%g%l%c%p%pns%retransn
   leafn_storage                  => clm3%g%l%c%p%pns%leafn_storage
   frootn_storage                 => clm3%g%l%c%p%pns%frootn_storage
   livestemn_storage              => clm3%g%l%c%p%pns%livestemn_storage
   deadstemn_storage              => clm3%g%l%c%p%pns%deadstemn_storage
   livecrootn_storage             => clm3%g%l%c%p%pns%livecrootn_storage
   deadcrootn_storage             => clm3%g%l%c%p%pns%deadcrootn_storage
   leafn_xfer                     => clm3%g%l%c%p%pns%leafn_xfer
   frootn_xfer                    => clm3%g%l%c%p%pns%frootn_xfer
   livestemn_xfer                 => clm3%g%l%c%p%pns%livestemn_xfer
   deadstemn_xfer                 => clm3%g%l%c%p%pns%deadstemn_xfer
   livecrootn_xfer                => clm3%g%l%c%p%pns%livecrootn_xfer
   deadcrootn_xfer                => clm3%g%l%c%p%pns%deadcrootn_xfer
   hrv_leafc_to_litter              => clm3%g%l%c%p%pcf%hrv_leafc_to_litter
   hrv_frootc_to_litter             => clm3%g%l%c%p%pcf%hrv_frootc_to_litter
   hrv_livestemc_to_litter          => clm3%g%l%c%p%pcf%hrv_livestemc_to_litter
   hrv_deadstemc_to_prod10c         => clm3%g%l%c%p%pcf%hrv_deadstemc_to_prod10c
   hrv_deadstemc_to_prod100c        => clm3%g%l%c%p%pcf%hrv_deadstemc_to_prod100c
   hrv_livecrootc_to_litter         => clm3%g%l%c%p%pcf%hrv_livecrootc_to_litter
   hrv_deadcrootc_to_litter         => clm3%g%l%c%p%pcf%hrv_deadcrootc_to_litter
   hrv_xsmrpool_to_atm              => clm3%g%l%c%p%pcf%hrv_xsmrpool_to_atm
   hrv_leafc_storage_to_litter      => clm3%g%l%c%p%pcf%hrv_leafc_storage_to_litter
   hrv_frootc_storage_to_litter     => clm3%g%l%c%p%pcf%hrv_frootc_storage_to_litter
   hrv_livestemc_storage_to_litter  => clm3%g%l%c%p%pcf%hrv_livestemc_storage_to_litter
   hrv_deadstemc_storage_to_litter  => clm3%g%l%c%p%pcf%hrv_deadstemc_storage_to_litter
   hrv_livecrootc_storage_to_litter => clm3%g%l%c%p%pcf%hrv_livecrootc_storage_to_litter
   hrv_deadcrootc_storage_to_litter => clm3%g%l%c%p%pcf%hrv_deadcrootc_storage_to_litter
   hrv_gresp_storage_to_litter      => clm3%g%l%c%p%pcf%hrv_gresp_storage_to_litter
   hrv_leafc_xfer_to_litter         => clm3%g%l%c%p%pcf%hrv_leafc_xfer_to_litter
   hrv_frootc_xfer_to_litter        => clm3%g%l%c%p%pcf%hrv_frootc_xfer_to_litter
   hrv_livestemc_xfer_to_litter     => clm3%g%l%c%p%pcf%hrv_livestemc_xfer_to_litter
   hrv_deadstemc_xfer_to_litter     => clm3%g%l%c%p%pcf%hrv_deadstemc_xfer_to_litter
   hrv_livecrootc_xfer_to_litter    => clm3%g%l%c%p%pcf%hrv_livecrootc_xfer_to_litter
   hrv_deadcrootc_xfer_to_litter    => clm3%g%l%c%p%pcf%hrv_deadcrootc_xfer_to_litter
   hrv_gresp_xfer_to_litter         => clm3%g%l%c%p%pcf%hrv_gresp_xfer_to_litter
   hrv_leafn_to_litter              => clm3%g%l%c%p%pnf%hrv_leafn_to_litter
   hrv_frootn_to_litter             => clm3%g%l%c%p%pnf%hrv_frootn_to_litter
   hrv_livestemn_to_litter          => clm3%g%l%c%p%pnf%hrv_livestemn_to_litter
   hrv_deadstemn_to_prod10n         => clm3%g%l%c%p%pnf%hrv_deadstemn_to_prod10n
   hrv_deadstemn_to_prod100n        => clm3%g%l%c%p%pnf%hrv_deadstemn_to_prod100n
   hrv_livecrootn_to_litter         => clm3%g%l%c%p%pnf%hrv_livecrootn_to_litter
   hrv_deadcrootn_to_litter         => clm3%g%l%c%p%pnf%hrv_deadcrootn_to_litter
   hrv_retransn_to_litter           => clm3%g%l%c%p%pnf%hrv_retransn_to_litter
   hrv_leafn_storage_to_litter      => clm3%g%l%c%p%pnf%hrv_leafn_storage_to_litter
   hrv_frootn_storage_to_litter     => clm3%g%l%c%p%pnf%hrv_frootn_storage_to_litter
   hrv_livestemn_storage_to_litter  => clm3%g%l%c%p%pnf%hrv_livestemn_storage_to_litter
   hrv_deadstemn_storage_to_litter  => clm3%g%l%c%p%pnf%hrv_deadstemn_storage_to_litter
   hrv_livecrootn_storage_to_litter => clm3%g%l%c%p%pnf%hrv_livecrootn_storage_to_litter
   hrv_deadcrootn_storage_to_litter => clm3%g%l%c%p%pnf%hrv_deadcrootn_storage_to_litter
   hrv_leafn_xfer_to_litter         => clm3%g%l%c%p%pnf%hrv_leafn_xfer_to_litter
   hrv_frootn_xfer_to_litter        => clm3%g%l%c%p%pnf%hrv_frootn_xfer_to_litter
   hrv_livestemn_xfer_to_litter     => clm3%g%l%c%p%pnf%hrv_livestemn_xfer_to_litter
   hrv_deadstemn_xfer_to_litter     => clm3%g%l%c%p%pnf%hrv_deadstemn_xfer_to_litter
   hrv_livecrootn_xfer_to_litter    => clm3%g%l%c%p%pnf%hrv_livecrootn_xfer_to_litter
   hrv_deadcrootn_xfer_to_litter    => clm3%g%l%c%p%pnf%hrv_deadcrootn_xfer_to_litter

   ! set deadstem proportions to 10-year product pool. 
   ! remainder (1-pprod10) is assumed to go to 100-year product pool
   ! veg type:       1        2        3       4        5       6        7        8      
   pprod10 =    (/0.75_r8, 0.75_r8, 0.75_r8, 1.0_r8, 0.75_r8, 1.0_r8, 0.75_r8, 0.75_r8/)

   ! pft loop
   do fp = 1,num_soilp
      p = filter_soilp(fp)
      g = pgridcell(p)
      
      ! If this is a tree pft, then
      ! get the annual harvest "mortality" rate (am) from harvest array
      ! and convert to rate per second
      if (ivt(p) > noveg .and. ivt(p) < nbrdlf_evr_shrub) then

         if (do_harvest) then
            am = harvest(g)
            m  = am/(365._r8 * 86400._r8)
         else
            m = 0._r8
         end if   

         ! pft-level harvest carbon fluxes
         ! displayed pools
         hrv_leafc_to_litter(p)               = leafc(p)               * m
         hrv_frootc_to_litter(p)              = frootc(p)              * m
         hrv_livestemc_to_litter(p)           = livestemc(p)           * m
         hrv_deadstemc_to_prod10c(p)          = deadstemc(p)           * m * pprod10(ivt(p))
         hrv_deadstemc_to_prod100c(p)         = deadstemc(p)           * m * (1.0_r8 - pprod10(ivt(p)))
         hrv_livecrootc_to_litter(p)          = livecrootc(p)          * m
         hrv_deadcrootc_to_litter(p)          = deadcrootc(p)          * m
         hrv_xsmrpool_to_atm(p)               = xsmrpool(p)            * m

         ! storage pools
         hrv_leafc_storage_to_litter(p)       = leafc_storage(p)       * m
         hrv_frootc_storage_to_litter(p)      = frootc_storage(p)      * m
         hrv_livestemc_storage_to_litter(p)   = livestemc_storage(p)   * m
         hrv_deadstemc_storage_to_litter(p)   = deadstemc_storage(p)   * m
         hrv_livecrootc_storage_to_litter(p)  = livecrootc_storage(p)  * m
         hrv_deadcrootc_storage_to_litter(p)  = deadcrootc_storage(p)  * m
         hrv_gresp_storage_to_litter(p)       = gresp_storage(p)       * m

         ! transfer pools
         hrv_leafc_xfer_to_litter(p)          = leafc_xfer(p)          * m
         hrv_frootc_xfer_to_litter(p)         = frootc_xfer(p)         * m
         hrv_livestemc_xfer_to_litter(p)      = livestemc_xfer(p)      * m
         hrv_deadstemc_xfer_to_litter(p)      = deadstemc_xfer(p)      * m
         hrv_livecrootc_xfer_to_litter(p)     = livecrootc_xfer(p)     * m
         hrv_deadcrootc_xfer_to_litter(p)     = deadcrootc_xfer(p)     * m
         hrv_gresp_xfer_to_litter(p)          = gresp_xfer(p)          * m

         ! pft-level harvest mortality nitrogen fluxes
         ! displayed pools
         hrv_leafn_to_litter(p)               = leafn(p)               * m
         hrv_frootn_to_litter(p)              = frootn(p)              * m
         hrv_livestemn_to_litter(p)           = livestemn(p)           * m
         hrv_deadstemn_to_prod10n(p)          = deadstemn(p)           * m * pprod10(ivt(p))
         hrv_deadstemn_to_prod100n(p)         = deadstemn(p)           * m * (1.0_r8 - pprod10(ivt(p)))
         hrv_livecrootn_to_litter(p)          = livecrootn(p)          * m
         hrv_deadcrootn_to_litter(p)          = deadcrootn(p)          * m
         hrv_retransn_to_litter(p)            = retransn(p)            * m

         ! storage pools
         hrv_leafn_storage_to_litter(p)       = leafn_storage(p)       * m
         hrv_frootn_storage_to_litter(p)      = frootn_storage(p)      * m
         hrv_livestemn_storage_to_litter(p)   = livestemn_storage(p)   * m
         hrv_deadstemn_storage_to_litter(p)   = deadstemn_storage(p)   * m
         hrv_livecrootn_storage_to_litter(p)  = livecrootn_storage(p)  * m
         hrv_deadcrootn_storage_to_litter(p)  = deadcrootn_storage(p)  * m

         ! transfer pools
         hrv_leafn_xfer_to_litter(p)          = leafn_xfer(p)          * m
         hrv_frootn_xfer_to_litter(p)         = frootn_xfer(p)         * m
         hrv_livestemn_xfer_to_litter(p)      = livestemn_xfer(p)      * m
         hrv_deadstemn_xfer_to_litter(p)      = deadstemn_xfer(p)      * m
         hrv_livecrootn_xfer_to_litter(p)     = livecrootn_xfer(p)     * m
         hrv_deadcrootn_xfer_to_litter(p)     = deadcrootn_xfer(p)     * m
         
      end if  ! end tree block

   end do ! end of pft loop

   ! gather all pft-level litterfall fluxes from harvest to the column
   ! for litter C and N inputs

   call CNHarvestPftToColumn(num_soilc, filter_soilc)

end subroutine CNHarvest
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: CNHarvestPftToColumn
!
! !INTERFACE:
subroutine CNHarvestPftToColumn (num_soilc, filter_soilc)
!
! !DESCRIPTION:
! called at the end of CNHarvest to gather all pft-level harvest litterfall fluxes
! to the column level and assign them to the three litter pools
!
! !USES:
  use clmtype
  use clm_varpar, only : max_pft_per_col, maxpatch_pft
!
! !ARGUMENTS:
  implicit none
  integer, intent(in) :: num_soilc       ! number of soil columns in filter
  integer, intent(in) :: filter_soilc(:) ! soil column filter
!
! !CALLED FROM:
! subroutine CNphenology
!
! !REVISION HISTORY:
! 9/8/03: Created by Peter Thornton
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in scalars
   integer , pointer :: ivt(:)      ! pft vegetation type
   real(r8), pointer :: wtcol(:)    ! pft weight relative to column (0-1)
   real(r8), pointer :: pwtgcell(:) ! weight of pft relative to corresponding gridcell
   real(r8), pointer :: lf_flab(:)  ! leaf litter labile fraction
   real(r8), pointer :: lf_fcel(:)  ! leaf litter cellulose fraction
   real(r8), pointer :: lf_flig(:)  ! leaf litter lignin fraction
   real(r8), pointer :: fr_flab(:)  ! fine root litter labile fraction
   real(r8), pointer :: fr_fcel(:)  ! fine root litter cellulose fraction
   real(r8), pointer :: fr_flig(:)  ! fine root litter lignin fraction
   integer , pointer :: npfts(:)    ! number of pfts for each column
   integer , pointer :: pfti(:)     ! beginning pft index for each column
   real(r8), pointer :: hrv_leafc_to_litter(:)
   real(r8), pointer :: hrv_frootc_to_litter(:)
   real(r8), pointer :: hrv_livestemc_to_litter(:)
   real(r8), pointer :: phrv_deadstemc_to_prod10c(:)
   real(r8), pointer :: phrv_deadstemc_to_prod100c(:)
   real(r8), pointer :: hrv_livecrootc_to_litter(:)
   real(r8), pointer :: hrv_deadcrootc_to_litter(:)
   real(r8), pointer :: hrv_leafc_storage_to_litter(:)
   real(r8), pointer :: hrv_frootc_storage_to_litter(:)
   real(r8), pointer :: hrv_livestemc_storage_to_litter(:)
   real(r8), pointer :: hrv_deadstemc_storage_to_litter(:)
   real(r8), pointer :: hrv_livecrootc_storage_to_litter(:)
   real(r8), pointer :: hrv_deadcrootc_storage_to_litter(:)
   real(r8), pointer :: hrv_gresp_storage_to_litter(:)
   real(r8), pointer :: hrv_leafc_xfer_to_litter(:)
   real(r8), pointer :: hrv_frootc_xfer_to_litter(:)
   real(r8), pointer :: hrv_livestemc_xfer_to_litter(:)
   real(r8), pointer :: hrv_deadstemc_xfer_to_litter(:)
   real(r8), pointer :: hrv_livecrootc_xfer_to_litter(:)
   real(r8), pointer :: hrv_deadcrootc_xfer_to_litter(:)
   real(r8), pointer :: hrv_gresp_xfer_to_litter(:)
   real(r8), pointer :: hrv_leafn_to_litter(:)
   real(r8), pointer :: hrv_frootn_to_litter(:)
   real(r8), pointer :: hrv_livestemn_to_litter(:)
   real(r8), pointer :: phrv_deadstemn_to_prod10n(:)
   real(r8), pointer :: phrv_deadstemn_to_prod100n(:)
   real(r8), pointer :: hrv_livecrootn_to_litter(:)
   real(r8), pointer :: hrv_deadcrootn_to_litter(:)
   real(r8), pointer :: hrv_retransn_to_litter(:)
   real(r8), pointer :: hrv_leafn_storage_to_litter(:)
   real(r8), pointer :: hrv_frootn_storage_to_litter(:)
   real(r8), pointer :: hrv_livestemn_storage_to_litter(:)
   real(r8), pointer :: hrv_deadstemn_storage_to_litter(:)
   real(r8), pointer :: hrv_livecrootn_storage_to_litter(:)
   real(r8), pointer :: hrv_deadcrootn_storage_to_litter(:)
   real(r8), pointer :: hrv_leafn_xfer_to_litter(:)
   real(r8), pointer :: hrv_frootn_xfer_to_litter(:)
   real(r8), pointer :: hrv_livestemn_xfer_to_litter(:)
   real(r8), pointer :: hrv_deadstemn_xfer_to_litter(:)
   real(r8), pointer :: hrv_livecrootn_xfer_to_litter(:)
   real(r8), pointer :: hrv_deadcrootn_xfer_to_litter(:)
!
! local pointers to implicit in/out arrays
   real(r8), pointer :: hrv_leafc_to_litr1c(:)
   real(r8), pointer :: hrv_leafc_to_litr2c(:)
   real(r8), pointer :: hrv_leafc_to_litr3c(:)
   real(r8), pointer :: hrv_frootc_to_litr1c(:)
   real(r8), pointer :: hrv_frootc_to_litr2c(:)
   real(r8), pointer :: hrv_frootc_to_litr3c(:)
   real(r8), pointer :: hrv_livestemc_to_cwdc(:)
   real(r8), pointer :: chrv_deadstemc_to_prod10c(:)
   real(r8), pointer :: chrv_deadstemc_to_prod100c(:)
   real(r8), pointer :: hrv_livecrootc_to_cwdc(:)
   real(r8), pointer :: hrv_deadcrootc_to_cwdc(:)
   real(r8), pointer :: hrv_leafc_storage_to_litr1c(:)
   real(r8), pointer :: hrv_frootc_storage_to_litr1c(:)
   real(r8), pointer :: hrv_livestemc_storage_to_litr1c(:)
   real(r8), pointer :: hrv_deadstemc_storage_to_litr1c(:)
   real(r8), pointer :: hrv_livecrootc_storage_to_litr1c(:)
   real(r8), pointer :: hrv_deadcrootc_storage_to_litr1c(:)
   real(r8), pointer :: hrv_gresp_storage_to_litr1c(:)
   real(r8), pointer :: hrv_leafc_xfer_to_litr1c(:)
   real(r8), pointer :: hrv_frootc_xfer_to_litr1c(:)
   real(r8), pointer :: hrv_livestemc_xfer_to_litr1c(:)
   real(r8), pointer :: hrv_deadstemc_xfer_to_litr1c(:)
   real(r8), pointer :: hrv_livecrootc_xfer_to_litr1c(:)
   real(r8), pointer :: hrv_deadcrootc_xfer_to_litr1c(:)
   real(r8), pointer :: hrv_gresp_xfer_to_litr1c(:)
   real(r8), pointer :: hrv_leafn_to_litr1n(:)
   real(r8), pointer :: hrv_leafn_to_litr2n(:)
   real(r8), pointer :: hrv_leafn_to_litr3n(:)
   real(r8), pointer :: hrv_frootn_to_litr1n(:)
   real(r8), pointer :: hrv_frootn_to_litr2n(:)
   real(r8), pointer :: hrv_frootn_to_litr3n(:)
   real(r8), pointer :: hrv_livestemn_to_cwdn(:)
   real(r8), pointer :: chrv_deadstemn_to_prod10n(:)
   real(r8), pointer :: chrv_deadstemn_to_prod100n(:)
   real(r8), pointer :: hrv_livecrootn_to_cwdn(:)
   real(r8), pointer :: hrv_deadcrootn_to_cwdn(:)
   real(r8), pointer :: hrv_retransn_to_litr1n(:)
   real(r8), pointer :: hrv_leafn_storage_to_litr1n(:)
   real(r8), pointer :: hrv_frootn_storage_to_litr1n(:)
   real(r8), pointer :: hrv_livestemn_storage_to_litr1n(:)
   real(r8), pointer :: hrv_deadstemn_storage_to_litr1n(:)
   real(r8), pointer :: hrv_livecrootn_storage_to_litr1n(:)
   real(r8), pointer :: hrv_deadcrootn_storage_to_litr1n(:)
   real(r8), pointer :: hrv_leafn_xfer_to_litr1n(:)
   real(r8), pointer :: hrv_frootn_xfer_to_litr1n(:)
   real(r8), pointer :: hrv_livestemn_xfer_to_litr1n(:)
   real(r8), pointer :: hrv_deadstemn_xfer_to_litr1n(:)
   real(r8), pointer :: hrv_livecrootn_xfer_to_litr1n(:)
   real(r8), pointer :: hrv_deadcrootn_xfer_to_litr1n(:)
!
! local pointers to implicit out arrays
!
!
! !OTHER LOCAL VARIABLES:
   integer :: fc,c,pi,p               ! indices
!EOP
!-----------------------------------------------------------------------

   ! assign local pointers
   lf_flab                        => pftcon%lf_flab
   lf_fcel                        => pftcon%lf_fcel
   lf_flig                        => pftcon%lf_flig
   fr_flab                        => pftcon%fr_flab
   fr_fcel                        => pftcon%fr_fcel
   fr_flig                        => pftcon%fr_flig

   ! assign local pointers to column-level arrays
   npfts                          => clm3%g%l%c%npfts
   pfti                           => clm3%g%l%c%pfti
   hrv_leafc_to_litr1c              => clm3%g%l%c%ccf%hrv_leafc_to_litr1c
   hrv_leafc_to_litr2c              => clm3%g%l%c%ccf%hrv_leafc_to_litr2c
   hrv_leafc_to_litr3c              => clm3%g%l%c%ccf%hrv_leafc_to_litr3c
   hrv_frootc_to_litr1c             => clm3%g%l%c%ccf%hrv_frootc_to_litr1c
   hrv_frootc_to_litr2c             => clm3%g%l%c%ccf%hrv_frootc_to_litr2c
   hrv_frootc_to_litr3c             => clm3%g%l%c%ccf%hrv_frootc_to_litr3c
   hrv_livestemc_to_cwdc            => clm3%g%l%c%ccf%hrv_livestemc_to_cwdc
   chrv_deadstemc_to_prod10c        => clm3%g%l%c%ccf%hrv_deadstemc_to_prod10c
   chrv_deadstemc_to_prod100c       => clm3%g%l%c%ccf%hrv_deadstemc_to_prod100c
   hrv_livecrootc_to_cwdc           => clm3%g%l%c%ccf%hrv_livecrootc_to_cwdc
   hrv_deadcrootc_to_cwdc           => clm3%g%l%c%ccf%hrv_deadcrootc_to_cwdc
   hrv_leafc_storage_to_litr1c      => clm3%g%l%c%ccf%hrv_leafc_storage_to_litr1c
   hrv_frootc_storage_to_litr1c     => clm3%g%l%c%ccf%hrv_frootc_storage_to_litr1c
   hrv_livestemc_storage_to_litr1c  => clm3%g%l%c%ccf%hrv_livestemc_storage_to_litr1c
   hrv_deadstemc_storage_to_litr1c  => clm3%g%l%c%ccf%hrv_deadstemc_storage_to_litr1c
   hrv_livecrootc_storage_to_litr1c => clm3%g%l%c%ccf%hrv_livecrootc_storage_to_litr1c
   hrv_deadcrootc_storage_to_litr1c => clm3%g%l%c%ccf%hrv_deadcrootc_storage_to_litr1c
   hrv_gresp_storage_to_litr1c      => clm3%g%l%c%ccf%hrv_gresp_storage_to_litr1c
   hrv_leafc_xfer_to_litr1c         => clm3%g%l%c%ccf%hrv_leafc_xfer_to_litr1c
   hrv_frootc_xfer_to_litr1c        => clm3%g%l%c%ccf%hrv_frootc_xfer_to_litr1c
   hrv_livestemc_xfer_to_litr1c     => clm3%g%l%c%ccf%hrv_livestemc_xfer_to_litr1c
   hrv_deadstemc_xfer_to_litr1c     => clm3%g%l%c%ccf%hrv_deadstemc_xfer_to_litr1c
   hrv_livecrootc_xfer_to_litr1c    => clm3%g%l%c%ccf%hrv_livecrootc_xfer_to_litr1c
   hrv_deadcrootc_xfer_to_litr1c    => clm3%g%l%c%ccf%hrv_deadcrootc_xfer_to_litr1c
   hrv_gresp_xfer_to_litr1c         => clm3%g%l%c%ccf%hrv_gresp_xfer_to_litr1c
   hrv_leafn_to_litr1n              => clm3%g%l%c%cnf%hrv_leafn_to_litr1n
   hrv_leafn_to_litr2n              => clm3%g%l%c%cnf%hrv_leafn_to_litr2n
   hrv_leafn_to_litr3n              => clm3%g%l%c%cnf%hrv_leafn_to_litr3n
   hrv_frootn_to_litr1n             => clm3%g%l%c%cnf%hrv_frootn_to_litr1n
   hrv_frootn_to_litr2n             => clm3%g%l%c%cnf%hrv_frootn_to_litr2n
   hrv_frootn_to_litr3n             => clm3%g%l%c%cnf%hrv_frootn_to_litr3n
   hrv_livestemn_to_cwdn            => clm3%g%l%c%cnf%hrv_livestemn_to_cwdn
   chrv_deadstemn_to_prod10n        => clm3%g%l%c%cnf%hrv_deadstemn_to_prod10n
   chrv_deadstemn_to_prod100n       => clm3%g%l%c%cnf%hrv_deadstemn_to_prod100n
   hrv_livecrootn_to_cwdn           => clm3%g%l%c%cnf%hrv_livecrootn_to_cwdn
   hrv_deadcrootn_to_cwdn           => clm3%g%l%c%cnf%hrv_deadcrootn_to_cwdn
   hrv_retransn_to_litr1n           => clm3%g%l%c%cnf%hrv_retransn_to_litr1n
   hrv_leafn_storage_to_litr1n      => clm3%g%l%c%cnf%hrv_leafn_storage_to_litr1n
   hrv_frootn_storage_to_litr1n     => clm3%g%l%c%cnf%hrv_frootn_storage_to_litr1n
   hrv_livestemn_storage_to_litr1n  => clm3%g%l%c%cnf%hrv_livestemn_storage_to_litr1n
   hrv_deadstemn_storage_to_litr1n  => clm3%g%l%c%cnf%hrv_deadstemn_storage_to_litr1n
   hrv_livecrootn_storage_to_litr1n => clm3%g%l%c%cnf%hrv_livecrootn_storage_to_litr1n
   hrv_deadcrootn_storage_to_litr1n => clm3%g%l%c%cnf%hrv_deadcrootn_storage_to_litr1n
   hrv_leafn_xfer_to_litr1n         => clm3%g%l%c%cnf%hrv_leafn_xfer_to_litr1n
   hrv_frootn_xfer_to_litr1n        => clm3%g%l%c%cnf%hrv_frootn_xfer_to_litr1n
   hrv_livestemn_xfer_to_litr1n     => clm3%g%l%c%cnf%hrv_livestemn_xfer_to_litr1n
   hrv_deadstemn_xfer_to_litr1n     => clm3%g%l%c%cnf%hrv_deadstemn_xfer_to_litr1n
   hrv_livecrootn_xfer_to_litr1n    => clm3%g%l%c%cnf%hrv_livecrootn_xfer_to_litr1n
   hrv_deadcrootn_xfer_to_litr1n    => clm3%g%l%c%cnf%hrv_deadcrootn_xfer_to_litr1n

   ! assign local pointers to pft-level arrays
   ivt                            => clm3%g%l%c%p%itype
   wtcol                          => clm3%g%l%c%p%wtcol
   pwtgcell                       => clm3%g%l%c%p%wtgcell  
   hrv_leafc_to_litter              => clm3%g%l%c%p%pcf%hrv_leafc_to_litter
   hrv_frootc_to_litter             => clm3%g%l%c%p%pcf%hrv_frootc_to_litter
   hrv_livestemc_to_litter          => clm3%g%l%c%p%pcf%hrv_livestemc_to_litter
   phrv_deadstemc_to_prod10c        => clm3%g%l%c%p%pcf%hrv_deadstemc_to_prod10c
   phrv_deadstemc_to_prod100c       => clm3%g%l%c%p%pcf%hrv_deadstemc_to_prod100c
   hrv_livecrootc_to_litter         => clm3%g%l%c%p%pcf%hrv_livecrootc_to_litter
   hrv_deadcrootc_to_litter         => clm3%g%l%c%p%pcf%hrv_deadcrootc_to_litter
   hrv_leafc_storage_to_litter      => clm3%g%l%c%p%pcf%hrv_leafc_storage_to_litter
   hrv_frootc_storage_to_litter     => clm3%g%l%c%p%pcf%hrv_frootc_storage_to_litter
   hrv_livestemc_storage_to_litter  => clm3%g%l%c%p%pcf%hrv_livestemc_storage_to_litter
   hrv_deadstemc_storage_to_litter  => clm3%g%l%c%p%pcf%hrv_deadstemc_storage_to_litter
   hrv_livecrootc_storage_to_litter => clm3%g%l%c%p%pcf%hrv_livecrootc_storage_to_litter
   hrv_deadcrootc_storage_to_litter => clm3%g%l%c%p%pcf%hrv_deadcrootc_storage_to_litter
   hrv_gresp_storage_to_litter      => clm3%g%l%c%p%pcf%hrv_gresp_storage_to_litter
   hrv_leafc_xfer_to_litter         => clm3%g%l%c%p%pcf%hrv_leafc_xfer_to_litter
   hrv_frootc_xfer_to_litter        => clm3%g%l%c%p%pcf%hrv_frootc_xfer_to_litter
   hrv_livestemc_xfer_to_litter     => clm3%g%l%c%p%pcf%hrv_livestemc_xfer_to_litter
   hrv_deadstemc_xfer_to_litter     => clm3%g%l%c%p%pcf%hrv_deadstemc_xfer_to_litter
   hrv_livecrootc_xfer_to_litter    => clm3%g%l%c%p%pcf%hrv_livecrootc_xfer_to_litter
   hrv_deadcrootc_xfer_to_litter    => clm3%g%l%c%p%pcf%hrv_deadcrootc_xfer_to_litter
   hrv_gresp_xfer_to_litter         => clm3%g%l%c%p%pcf%hrv_gresp_xfer_to_litter
   hrv_leafn_to_litter              => clm3%g%l%c%p%pnf%hrv_leafn_to_litter
   hrv_frootn_to_litter             => clm3%g%l%c%p%pnf%hrv_frootn_to_litter
   hrv_livestemn_to_litter          => clm3%g%l%c%p%pnf%hrv_livestemn_to_litter
   phrv_deadstemn_to_prod10n        => clm3%g%l%c%p%pnf%hrv_deadstemn_to_prod10n
   phrv_deadstemn_to_prod100n       => clm3%g%l%c%p%pnf%hrv_deadstemn_to_prod100n
   hrv_livecrootn_to_litter         => clm3%g%l%c%p%pnf%hrv_livecrootn_to_litter
   hrv_deadcrootn_to_litter         => clm3%g%l%c%p%pnf%hrv_deadcrootn_to_litter
   hrv_retransn_to_litter           => clm3%g%l%c%p%pnf%hrv_retransn_to_litter
   hrv_leafn_storage_to_litter      => clm3%g%l%c%p%pnf%hrv_leafn_storage_to_litter
   hrv_frootn_storage_to_litter     => clm3%g%l%c%p%pnf%hrv_frootn_storage_to_litter
   hrv_livestemn_storage_to_litter  => clm3%g%l%c%p%pnf%hrv_livestemn_storage_to_litter
   hrv_deadstemn_storage_to_litter  => clm3%g%l%c%p%pnf%hrv_deadstemn_storage_to_litter
   hrv_livecrootn_storage_to_litter => clm3%g%l%c%p%pnf%hrv_livecrootn_storage_to_litter
   hrv_deadcrootn_storage_to_litter => clm3%g%l%c%p%pnf%hrv_deadcrootn_storage_to_litter
   hrv_leafn_xfer_to_litter         => clm3%g%l%c%p%pnf%hrv_leafn_xfer_to_litter
   hrv_frootn_xfer_to_litter        => clm3%g%l%c%p%pnf%hrv_frootn_xfer_to_litter
   hrv_livestemn_xfer_to_litter     => clm3%g%l%c%p%pnf%hrv_livestemn_xfer_to_litter
   hrv_deadstemn_xfer_to_litter     => clm3%g%l%c%p%pnf%hrv_deadstemn_xfer_to_litter
   hrv_livecrootn_xfer_to_litter    => clm3%g%l%c%p%pnf%hrv_livecrootn_xfer_to_litter
   hrv_deadcrootn_xfer_to_litter    => clm3%g%l%c%p%pnf%hrv_deadcrootn_xfer_to_litter

   do pi = 1,maxpatch_pft
      do fc = 1,num_soilc
         c = filter_soilc(fc)

         if (pi <=  npfts(c)) then
            p = pfti(c) + pi - 1

            if (pwtgcell(p)>0._r8) then

               ! leaf harvest mortality carbon fluxes
               hrv_leafc_to_litr1c(c) = hrv_leafc_to_litr1c(c) + &
                  hrv_leafc_to_litter(p) * lf_flab(ivt(p)) * wtcol(p)
               hrv_leafc_to_litr2c(c) = hrv_leafc_to_litr2c(c) + &
                  hrv_leafc_to_litter(p) * lf_fcel(ivt(p)) * wtcol(p)
               hrv_leafc_to_litr3c(c) = hrv_leafc_to_litr3c(c) + &
                  hrv_leafc_to_litter(p) * lf_flig(ivt(p)) * wtcol(p)

               ! fine root harvest mortality carbon fluxes
               hrv_frootc_to_litr1c(c) = hrv_frootc_to_litr1c(c) + &
                  hrv_frootc_to_litter(p) * fr_flab(ivt(p)) * wtcol(p)
               hrv_frootc_to_litr2c(c) = hrv_frootc_to_litr2c(c) + &
                  hrv_frootc_to_litter(p) * fr_fcel(ivt(p)) * wtcol(p)
               hrv_frootc_to_litr3c(c) = hrv_frootc_to_litr3c(c) + &
                  hrv_frootc_to_litter(p) * fr_flig(ivt(p)) * wtcol(p)

               ! wood harvest mortality carbon fluxes
               hrv_livestemc_to_cwdc(c)  = hrv_livestemc_to_cwdc(c)  + &
                  hrv_livestemc_to_litter(p)  * wtcol(p)
               chrv_deadstemc_to_prod10c(c)  = chrv_deadstemc_to_prod10c(c)  + &
                  phrv_deadstemc_to_prod10c(p)  * wtcol(p)
               chrv_deadstemc_to_prod100c(c)  = chrv_deadstemc_to_prod100c(c)  + &
                  phrv_deadstemc_to_prod100c(p)  * wtcol(p)
               hrv_livecrootc_to_cwdc(c) = hrv_livecrootc_to_cwdc(c) + &
                  hrv_livecrootc_to_litter(p) * wtcol(p)
               hrv_deadcrootc_to_cwdc(c) = hrv_deadcrootc_to_cwdc(c) + &
                  hrv_deadcrootc_to_litter(p) * wtcol(p)

               ! storage harvest mortality carbon fluxes
               hrv_leafc_storage_to_litr1c(c)      = hrv_leafc_storage_to_litr1c(c)      + &
                  hrv_leafc_storage_to_litter(p)      * wtcol(p)
               hrv_frootc_storage_to_litr1c(c)     = hrv_frootc_storage_to_litr1c(c)     + &
                  hrv_frootc_storage_to_litter(p)     * wtcol(p)
               hrv_livestemc_storage_to_litr1c(c)  = hrv_livestemc_storage_to_litr1c(c)  + &
                  hrv_livestemc_storage_to_litter(p)  * wtcol(p)
               hrv_deadstemc_storage_to_litr1c(c)  = hrv_deadstemc_storage_to_litr1c(c)  + &
                  hrv_deadstemc_storage_to_litter(p)  * wtcol(p)
               hrv_livecrootc_storage_to_litr1c(c) = hrv_livecrootc_storage_to_litr1c(c) + &
                  hrv_livecrootc_storage_to_litter(p) * wtcol(p)
               hrv_deadcrootc_storage_to_litr1c(c) = hrv_deadcrootc_storage_to_litr1c(c) + &
                  hrv_deadcrootc_storage_to_litter(p) * wtcol(p)
               hrv_gresp_storage_to_litr1c(c)      = hrv_gresp_storage_to_litr1c(c)      + &
                  hrv_gresp_storage_to_litter(p)      * wtcol(p)

               ! transfer harvest mortality carbon fluxes
               hrv_leafc_xfer_to_litr1c(c)      = hrv_leafc_xfer_to_litr1c(c)      + &
                  hrv_leafc_xfer_to_litter(p)      * wtcol(p)
               hrv_frootc_xfer_to_litr1c(c)     = hrv_frootc_xfer_to_litr1c(c)     + &
                  hrv_frootc_xfer_to_litter(p)     * wtcol(p)
               hrv_livestemc_xfer_to_litr1c(c)  = hrv_livestemc_xfer_to_litr1c(c)  + &
                  hrv_livestemc_xfer_to_litter(p)  * wtcol(p)
               hrv_deadstemc_xfer_to_litr1c(c)  = hrv_deadstemc_xfer_to_litr1c(c)  + &
                  hrv_deadstemc_xfer_to_litter(p)  * wtcol(p)
               hrv_livecrootc_xfer_to_litr1c(c) = hrv_livecrootc_xfer_to_litr1c(c) + &
                  hrv_livecrootc_xfer_to_litter(p) * wtcol(p)
               hrv_deadcrootc_xfer_to_litr1c(c) = hrv_deadcrootc_xfer_to_litr1c(c) + &
                  hrv_deadcrootc_xfer_to_litter(p) * wtcol(p)
               hrv_gresp_xfer_to_litr1c(c)      = hrv_gresp_xfer_to_litr1c(c)      + &
                  hrv_gresp_xfer_to_litter(p)      * wtcol(p)

               ! leaf harvest mortality nitrogen fluxes
               hrv_leafn_to_litr1n(c) = hrv_leafn_to_litr1n(c) + &
                  hrv_leafn_to_litter(p) * lf_flab(ivt(p)) * wtcol(p)
               hrv_leafn_to_litr2n(c) = hrv_leafn_to_litr2n(c) + &
                  hrv_leafn_to_litter(p) * lf_fcel(ivt(p)) * wtcol(p)
               hrv_leafn_to_litr3n(c) = hrv_leafn_to_litr3n(c) + &
                  hrv_leafn_to_litter(p) * lf_flig(ivt(p)) * wtcol(p)

               ! fine root litter nitrogen fluxes
               hrv_frootn_to_litr1n(c) = hrv_frootn_to_litr1n(c) + &
                  hrv_frootn_to_litter(p) * fr_flab(ivt(p)) * wtcol(p)
               hrv_frootn_to_litr2n(c) = hrv_frootn_to_litr2n(c) + &
                  hrv_frootn_to_litter(p) * fr_fcel(ivt(p)) * wtcol(p)
               hrv_frootn_to_litr3n(c) = hrv_frootn_to_litr3n(c) + &
                  hrv_frootn_to_litter(p) * fr_flig(ivt(p)) * wtcol(p)

               ! wood harvest mortality nitrogen fluxes
               hrv_livestemn_to_cwdn(c)  = hrv_livestemn_to_cwdn(c)  + &
                  hrv_livestemn_to_litter(p)  * wtcol(p)
               chrv_deadstemn_to_prod10n(c)  = chrv_deadstemn_to_prod10n(c)  + &
                  phrv_deadstemn_to_prod10n(p)  * wtcol(p)
               chrv_deadstemn_to_prod100n(c)  = chrv_deadstemn_to_prod100n(c)  + &
                  phrv_deadstemn_to_prod100n(p)  * wtcol(p)
               hrv_livecrootn_to_cwdn(c) = hrv_livecrootn_to_cwdn(c) + &
                  hrv_livecrootn_to_litter(p) * wtcol(p)
               hrv_deadcrootn_to_cwdn(c) = hrv_deadcrootn_to_cwdn(c) + &
                  hrv_deadcrootn_to_litter(p) * wtcol(p)

               ! retranslocated N pool harvest mortality fluxes
               hrv_retransn_to_litr1n(c) = hrv_retransn_to_litr1n(c) + &
                  hrv_retransn_to_litter(p) * wtcol(p)

               ! storage harvest mortality nitrogen fluxes
               hrv_leafn_storage_to_litr1n(c)      = hrv_leafn_storage_to_litr1n(c)      + &
                  hrv_leafn_storage_to_litter(p)      * wtcol(p)
               hrv_frootn_storage_to_litr1n(c)     = hrv_frootn_storage_to_litr1n(c)     + &
                  hrv_frootn_storage_to_litter(p)     * wtcol(p)
               hrv_livestemn_storage_to_litr1n(c)  = hrv_livestemn_storage_to_litr1n(c)  + &
                  hrv_livestemn_storage_to_litter(p)  * wtcol(p)
               hrv_deadstemn_storage_to_litr1n(c)  = hrv_deadstemn_storage_to_litr1n(c)  + &
                  hrv_deadstemn_storage_to_litter(p)  * wtcol(p)
               hrv_livecrootn_storage_to_litr1n(c) = hrv_livecrootn_storage_to_litr1n(c) + &
                  hrv_livecrootn_storage_to_litter(p) * wtcol(p)
               hrv_deadcrootn_storage_to_litr1n(c) = hrv_deadcrootn_storage_to_litr1n(c) + &
                  hrv_deadcrootn_storage_to_litter(p) * wtcol(p)

               ! transfer harvest mortality nitrogen fluxes
               hrv_leafn_xfer_to_litr1n(c)      = hrv_leafn_xfer_to_litr1n(c)      + &
                  hrv_leafn_xfer_to_litter(p)      * wtcol(p)
               hrv_frootn_xfer_to_litr1n(c)     = hrv_frootn_xfer_to_litr1n(c)     + &
                  hrv_frootn_xfer_to_litter(p)     * wtcol(p)
               hrv_livestemn_xfer_to_litr1n(c)  = hrv_livestemn_xfer_to_litr1n(c)  + &
                  hrv_livestemn_xfer_to_litter(p)  * wtcol(p)
               hrv_deadstemn_xfer_to_litr1n(c)  = hrv_deadstemn_xfer_to_litr1n(c)  + &
                  hrv_deadstemn_xfer_to_litter(p)  * wtcol(p)
               hrv_livecrootn_xfer_to_litr1n(c) = hrv_livecrootn_xfer_to_litr1n(c) + &
                  hrv_livecrootn_xfer_to_litter(p) * wtcol(p)
               hrv_deadcrootn_xfer_to_litr1n(c) = hrv_deadcrootn_xfer_to_litr1n(c) + &
                  hrv_deadcrootn_xfer_to_litter(p) * wtcol(p)

            end if
         end if

      end do

   end do

end subroutine CNHarvestPftToColumn
!-----------------------------------------------------------------------

! !ROUTINE: pftdyn_wbal_init
!
! !INTERFACE:
  subroutine pftdyn_wbal_init()
!
! !DESCRIPTION:
! initialize the column-level mass-balance correction term.
! Called in every timestep.
!
! !USES:
!
! !ARGUMENTS:
    implicit none
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: begp, endp    ! proc beginning and ending pft indices
    integer  :: begc, endc    ! proc beginning and ending column indices
    integer  :: begl, endl    ! proc beginning and ending landunit indices
    integer  :: begg, endg    ! proc beginning and ending gridcell indices
    integer  :: c             ! indices
    type(column_type),   pointer :: cptr         ! pointer to column derived subtype
!-----------------------------------------------------------------------

    ! Set pointers into derived type

    cptr => clm3%g%l%c

    ! Get relevant sizes

    call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

    ! set column-level canopy water mass balance correction flux
    ! term to 0 at the beginning of every timestep

    do c = begc,endc
       cptr%cwf%h2ocan_loss(c) = 0._r8
    end do

  end subroutine pftdyn_wbal_init



end module pftdynMod

module filterMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: filterMod
!
! !DESCRIPTION:
! Module of filters used for processing columns and pfts of particular
! types, including lake, non-lake, urban, soil, snow, non-snow, and
! naturally-vegetated patches.
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
!
! !PUBLIC TYPES:
  implicit none
  save


  type clumpfilter
#if (defined CNDV)
     integer, pointer :: natvegp(:)      ! CNDV nat-vegetated (present) filter (pfts)
     integer :: num_natvegp              ! number of pfts in nat-vegetated filter
#endif
#if (defined CROP)
     integer, pointer :: pcropp(:)       ! prognostic crop filter (pfts)
     integer :: num_pcropp               ! number of pfts in prognostic crop filter
     integer, pointer :: soilnopcropp(:) ! soil w/o prog. crops (pfts)
     integer :: num_soilnopcropp         ! number of pfts in soil w/o prog crops
#endif

     integer, pointer :: lakep(:)        ! lake filter (pfts)
     integer :: num_lakep                ! number of pfts in lake filter
     integer, pointer :: nolakep(:)      ! non-lake filter (pfts)
     integer :: num_nolakep              ! number of pfts in non-lake filter
     integer, pointer :: lakec(:)        ! lake filter (columns)
     integer :: num_lakec                ! number of columns in lake filter
     integer, pointer :: nolakec(:)      ! non-lake filter (columns)
     integer :: num_nolakec              ! number of columns in non-lake filter

     integer, pointer :: soilc(:)        ! soil filter (columns)
     integer :: num_soilc                ! number of columns in soil filter 
     integer, pointer :: soilp(:)        ! soil filter (pfts)
     integer :: num_soilp                ! number of pfts in soil filter 

     integer, pointer :: snowc(:)        ! snow filter (columns) 
     integer :: num_snowc                ! number of columns in snow filter 
     integer, pointer :: nosnowc(:)      ! non-snow filter (columns) 
     integer :: num_nosnowc              ! number of columns in non-snow filter 

     integer, pointer :: hydrologyc(:)   ! hydrology filter (columns)
     integer :: num_hydrologyc           ! number of columns in hydrology filter 

     integer, pointer :: urbanl(:)       ! urban filter (landunits)
     integer :: num_urbanl               ! number of landunits in urban filter 
     integer, pointer :: nourbanl(:)     ! non-urban filter (landunits)
     integer :: num_nourbanl             ! number of landunits in non-urban filter 

     integer, pointer :: urbanc(:)       ! urban filter (columns)
     integer :: num_urbanc               ! number of columns in urban filter
     integer, pointer :: nourbanc(:)     ! non-urban filter (columns)
     integer :: num_nourbanc             ! number of columns in non-urban filter

     integer, pointer :: urbanp(:)       ! urban filter (pfts)
     integer :: num_urbanp               ! number of pfts in urban filter
     integer, pointer :: nourbanp(:)     ! non-urban filter (pfts)
     integer :: num_nourbanp             ! number of pfts in non-urban filter

     integer, pointer :: nolakeurbanp(:) ! non-lake, non-urban filter (pfts)
     integer :: num_nolakeurbanp         ! number of pfts in non-lake, non-urban filter

  end type clumpfilter
  public clumpfilter

  type(clumpfilter), public :: filter
!
  public allocFilters   ! allocate memory for filters
  public setFilters     ! set filters
  public filters_dealloc
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
! 11/13/03, Peter Thornton: Added soilp and num_soilp
! Jan/08, S. Levis: Added crop-related filters
!
!EOP
!-----------------------------------------------------------------------

contains

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: allocFilters
!
! !INTERFACE:
  subroutine allocFilters()
!
! !DESCRIPTION:
! Allocate CLM filters.
!
! !USES:
    use clmtype
    use decompMod , only : get_proc_bounds
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
! 2004.04.27 DGVM naturally-vegetated filter added by Forrest Hoffman
!
!EOP
!
! LOCAL VARAIBLES:
    integer :: begp, endp  ! per-clump beginning and ending pft indices
    integer :: begc, endc  ! per-clump beginning and ending column indices
    integer :: begl, endl  ! per-clump beginning and ending landunit indices
    integer :: begg, endg  ! per-clump beginning and ending gridcell indices
!------------------------------------------------------------------------

    ! Determine clump variables for this processor

       call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

       allocate(filter%lakep(endp-begp+1))
       allocate(filter%nolakep(endp-begp+1))
       allocate(filter%nolakeurbanp(endp-begp+1))

       allocate(filter%lakec(endc-begc+1))
       allocate(filter%nolakec(endc-begc+1))

       allocate(filter%soilc(endc-begc+1))
       allocate(filter%soilp(endp-begp+1))

       allocate(filter%snowc(endc-begc+1))
       allocate(filter%nosnowc(endc-begc+1))

#if (defined CNDV)
       allocate(filter%natvegp(endp-begp+1))
#endif

       allocate(filter%hydrologyc(endc-begc+1))

       allocate(filter%urbanp(endp-begp+1))
       allocate(filter%nourbanp(endp-begp+1))

       allocate(filter%urbanc(endc-begc+1))
       allocate(filter%nourbanc(endc-begc+1))

       allocate(filter%urbanl(endl-begl+1))
       allocate(filter%nourbanl(endl-begl+1))

#if (defined CROP)
       allocate(filter%pcropp(endp-begp+1))
       allocate(filter%soilnopcropp(endp-begp+1))
#endif

  end subroutine allocFilters

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: setFilters
!
! !INTERFACE:
  subroutine setFilters()
!
! !DESCRIPTION:
! Set CLM filters.
!
! !USES:
    use clmtype
    use decompMod , only : get_proc_bounds
#if (defined CROP)
    use pftvarcon , only : npcropmin
#endif
    use clm_varcon, only : istsoil, isturb, icol_road_perv
#ifdef CROP
    use clm_varcon, only : istcrop
#endif
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
! 2004.04.27 DGVM naturally-vegetated filter added by Forrest Hoffman
! 2005.09.12 Urban related filters added by Mariana Vertenstein
!
!EOP
!
! LOCAL VARAIBLES:
    integer , pointer :: ctype(:) ! column type
    integer :: c,l,p       ! column, landunit, pft indices
    integer :: fl          ! lake filter index
    integer :: fnl,fnlu    ! non-lake filter index
    integer :: fs          ! soil filter index
    integer :: f, fn       ! general indices
    integer :: begp, endp  ! per-clump beginning and ending pft indices
    integer :: begc, endc  ! per-clump beginning and ending column indices
    integer :: begl, endl  ! per-clump beginning and ending landunit indices
    integer :: begg, endg  ! per-clump beginning and ending gridcell indices
!------------------------------------------------------------------------

    ctype => clm3%g%l%c%itype

    ! Loop over clumps on this processor


       ! Determine clump boundaries

       call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

       ! Create lake and non-lake filters at column-level 

       fl = 0
       fnl = 0
       do c = begc,endc
          l = clm3%g%l%c%landunit(c)
          if (clm3%g%l%lakpoi(l)) then
             fl = fl + 1
             filter%lakec(fl) = c
          else
             fnl = fnl + 1
             filter%nolakec(fnl) = c
          end if
       end do
       filter%num_lakec = fl
       filter%num_nolakec = fnl

       ! Create lake and non-lake filters at pft-level 
       ! Filter will only be active if weight of pft wrt gcell is nonzero

       fl = 0
       fnl = 0
       fnlu = 0
       do p = begp,endp
          if (clm3%g%l%c%p%wtgcell(p) > 0._r8) then
             l = clm3%g%l%c%p%landunit(p)
             if (clm3%g%l%lakpoi(l) ) then
                fl = fl + 1
                filter%lakep(fl) = p
             else
                fnl = fnl + 1
                filter%nolakep(fnl) = p
                if (clm3%g%l%itype(l) /= isturb) then
                   fnlu = fnlu + 1
                   filter%nolakeurbanp(fnlu) = p
                end if
             end if
          end if
       end do
       filter%num_lakep = fl
       filter%num_nolakep = fnl
       filter%num_nolakeurbanp = fnlu

       ! Create soil filter at column-level

       fs = 0
       do c = begc,endc
          l = clm3%g%l%c%landunit(c)
#ifndef CROP
          if (clm3%g%l%itype(l) == istsoil) then
#else
          if (clm3%g%l%itype(l) == istsoil .or. clm3%g%l%itype(l) == istcrop) then
#endif
             fs = fs + 1
             filter%soilc(fs) = c
          end if
       end do
       filter%num_soilc = fs

       ! Create soil filter at pft-level
       ! Filter will only be active if weight of pft wrt gcell is nonzero

       fs = 0
       do p = begp,endp
          if (clm3%g%l%c%p%wtgcell(p) > 0._r8) then
             l = clm3%g%l%c%p%landunit(p)
#ifndef CROP
             if (clm3%g%l%itype(l) == istsoil) then
#else
             if (clm3%g%l%itype(l) == istsoil .or. clm3%g%l%itype(l) == istcrop) then
#endif
                fs = fs + 1
                filter%soilp(fs) = p
             end if
          end if
       end do
       filter%num_soilp = fs

#if (defined CROP)
       ! Create prognostic crop and soil w/o prog. crop filters at pft-level
       ! according to where the CROP model should be used

       fl = 0
       fnl = 0
       do p = begp,endp
          if (clm3%g%l%c%p%wtgcell(p) > 0._r8) then
             if (clm3%g%l%c%p%itype(p) >= npcropmin) then !skips 2 generic crop types
                fl = fl + 1
                filter%pcropp(fl) = p
             else
                l = clm3%g%l%c%p%landunit(p)
                if (clm3%g%l%itype(l) == istsoil .or. clm3%g%l%itype(l) == istcrop) then
                   fnl = fnl + 1
                   filter%soilnopcropp(fnl) = p
                end if
             end if
          end if
       end do
       filter%num_pcropp = fl
#endif

       ! Create column-level hydrology filter (soil and Urban pervious road cols) 

       f = 0
       do c = begc,endc
          l = clm3%g%l%c%landunit(c)
#ifndef CROP
          if (clm3%g%l%itype(l) == istsoil .or. ctype(c) == icol_road_perv ) then
#else
          if (clm3%g%l%itype(l) == istsoil .or. clm3%g%l%itype(l) == istcrop .or. ctype(c) == icol_road_perv ) then
#endif
             f = f + 1
             filter%hydrologyc(f) = c
          end if
       end do
       filter%num_hydrologyc = f

       ! Create landunit-level urban and non-urban filters

       f = 0
       fn = 0
       do l = begl,endl
          if (clm3%g%l%itype(l) == isturb) then
             f = f + 1
             filter%urbanl(f) = l
          else
             fn = fn + 1
             filter%nourbanl(fn) = l
          end if
       end do
       filter%num_urbanl = f
       filter%num_nourbanl = fn

       ! Create column-level urban and non-urban filters

       f = 0
       fn = 0
       do c = begc,endc
          l = clm3%g%l%c%landunit(c)
          if (clm3%g%l%itype(l) == isturb) then
             f = f + 1
             filter%urbanc(f) = c
          else
             fn = fn + 1
             filter%nourbanc(fn) = c
          end if
       end do
       filter%num_urbanc = f
       filter%num_nourbanc = fn

       ! Create pft-level urban and non-urban filters

       f = 0
       fn = 0
       do p = begp,endp
          l = clm3%g%l%c%p%landunit(p)
          if (clm3%g%l%itype(l) == isturb .and. clm3%g%l%c%p%wtgcell(p) > 0._r8) then
             f = f + 1
             filter%urbanp(f) = p
          else
             fn = fn + 1
             filter%nourbanp(fn) = p 
          end if
       end do
       filter%num_urbanp = f
       filter%num_nourbanp = fn

       ! Note: snow filters are reconstructed each time step in Hydrology2
       ! Note: CNDV "pft present" filter is reconstructed each time CNDV is run


  end subroutine setFilters

  subroutine filters_dealloc
!
    implicit none
     
       deallocate(filter%lakep)
   call CLMDebug('mark1')
       deallocate(filter%nolakep)
       deallocate(filter%nolakeurbanp)

       deallocate(filter%lakec)
       deallocate(filter%nolakec)

       deallocate(filter%soilc)
       deallocate(filter%soilp)

   
       deallocate(filter%snowc)
       deallocate(filter%nosnowc)

#if (defined CNDV)
       deallocate(filter%natvegp)
#endif

       deallocate(filter%hydrologyc)

       deallocate(filter%urbanp)
       deallocate(filter%nourbanp)

       deallocate(filter%urbanc)
       deallocate(filter%nourbanc)

       deallocate(filter%urbanl)
       deallocate(filter%nourbanl)

#if (defined CROP)
       deallocate(filter%pcropp)
       deallocate(filter%soilnopcropp)
#endif

  call CLMDebug('done  filters_dealloc')
  end subroutine filters_dealloc


end module filterMod

!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: iniTimeConst
!
! !INTERFACE:
subroutine iniTimeConst
!
! !DESCRIPTION:
! Initialize time invariant clm variables
! 1) removed references to shallow lake - since it is not used
! 2) ***Make c%z, c%zi and c%dz allocatable depending on if you
!    have lake or soil
! 3) rootfr only initialized for soil points
!
! !USES:
  use shr_kind_mod, only : r8 => shr_kind_r8
  use nanMod      , only : nan
  use clmtype
  use decompMod   , only : get_proc_bounds
  use clm_varpar  , only : nlevsoi, nlevgrnd, nlevlak, lsmlon, lsmlat, numpft, numrad, nlevurb
  use clm_varcon  , only : istice, istdlak, istwet, isturb, &
                           icol_roof, icol_sunwall, icol_shadewall, icol_road_perv, icol_road_imperv, &
                           zlak, dzlak, zsoi, dzsoi, zisoi, spval
  use pftvarcon   , only : noveg, ntree, roota_par, rootb_par,  &
                           smpso, smpsc, fnitr, nbrdlf_dcd_brl_shrub, &
                           z0mr, displar, dleaf, rhol, rhos, taul, taus, xl, &
                           qe25, vcmx25, mp, c3psn, slatop, dsladlai, leafcn, flnr, woody, &
                           lflitcn, frootcn, livewdcn, deadwdcn, froot_leaf, stem_leaf, croot_stem, &
                           flivewd, fcur, lf_flab, lf_fcel, lf_flig, fr_flab, fr_fcel, fr_flig, &
                           dw_fcel, dw_flig, leaf_long, evergreen, stress_decid, season_decid, &
                           resist, pftpar20, pftpar28, pftpar29, pftpar30, pftpar31, &
                           allom1s, allom2s, &
                           allom1 , allom2 , allom3  , reinickerp, dwood
#if (defined CROP)
  use pftvarcon   , only : graincn
#endif
  use globals     , only : nstep
  use clm_varsur  , only : gti,soic2d,efisop2d,sand3d,clay3d,organic3d
!
! !ARGUMENTS:
  implicit none
!
! !CALLED FROM:
! subroutine initialize in module initializeMod.
!
! !REVISION HISTORY:
! Created by Gordon Bonan.
! Updated to clm2.1 data structrues by Mariana Vertenstein
! 4/26/05, Peter Thornton: Eliminated exponential decrease in saturated hydraulic
!   conductivity (hksat) with depth. 
! Updated: Colette L. Heald (05/06) reading in VOC emission factors
! 27 February 2008: Keith Oleson; Qing Liu (2004) saturated hydraulic conductivity 
! and matric potential
! 29 February 2008: David Lawrence; modified soil thermal and hydraulic properties to
! account for organic matter
! 18 March 2008: David Lawrence; nlevgrnd changes
! 03/28/08 Mark Flanner, read in netcdf files for SNICAR parameters
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
  integer , pointer :: ivt(:)             !  vegetation type index
  integer , pointer :: pcolumn(:)         ! column index of corresponding pft
  integer , pointer :: pgridcell(:)       ! gridcell index of corresponding pft
  integer , pointer :: clandunit(:)       ! landunit index of column
  integer , pointer :: cgridcell(:)       ! gridcell index of column
  integer , pointer :: ctype(:)           ! column type index
  integer , pointer :: ltype(:)           ! landunit type index
  real(r8), pointer :: thick_wall(:)      ! total thickness of urban wall
  real(r8), pointer :: thick_roof(:)      ! total thickness of urban roof
  real(r8), pointer :: lat(:)             ! gridcell latitude (radians)
!
! local pointers to implicit out arguments
!
  real(r8), pointer :: z(:,:)             ! layer depth (m)
  real(r8), pointer :: zi(:,:)            ! interface level below a "z" level (m)
  real(r8), pointer :: dz(:,:)            ! layer thickness depth (m)
  real(r8), pointer :: rootfr(:,:)        ! fraction of roots in each soil layer
  real(r8), pointer :: rootfr_road_perv(:,:) ! fraction of roots in each soil layer for urban pervious road
  real(r8), pointer :: rresis(:,:)        !root resistance by layer (0-1)  (nlevgrnd)	
  real(r8), pointer :: dewmx(:)           ! maximum allowed dew [mm]
  real(r8), pointer :: bsw(:,:)           ! Clapp and Hornberger "b" (nlevgrnd)  
  real(r8), pointer :: bsw2(:,:)          ! Clapp and Hornberger "b" for CN code
  real(r8), pointer :: psisat(:,:)        ! soil water potential at saturation for CN code (MPa)
  real(r8), pointer :: vwcsat(:,:)        ! volumetric water content at saturation for CN code (m3/m3)
  real(r8), pointer :: watsat(:,:)        ! volumetric soil water at saturation (porosity) (nlevgrnd) 
  real(r8), pointer :: watfc(:,:)         ! volumetric soil water at field capacity (nlevsoi)
  real(r8), pointer :: watdry(:,:)        ! btran parameter for btran=0
  real(r8), pointer :: watopt(:,:)        ! btran parameter for btran = 1
  real(r8), pointer :: hksat(:,:)         ! hydraulic conductivity at saturation (mm H2O /s) (nlevgrnd) 
  real(r8), pointer :: sucsat(:,:)        ! minimum soil suction (mm) (nlevgrnd) 
  real(r8), pointer :: csol(:,:)          ! heat capacity, soil solids (J/m**3/Kelvin) (nlevgrnd) 
  real(r8), pointer :: tkmg(:,:)          ! thermal conductivity, soil minerals  [W/m-K] (new) (nlevgrnd) 
  real(r8), pointer :: tkdry(:,:)         ! thermal conductivity, dry soil (W/m/Kelvin) (nlevgrnd) 
  real(r8), pointer :: tksatu(:,:)        ! thermal conductivity, saturated soil [W/m-K] (new) (nlevgrnd) 
  real(r8), pointer :: wtfact(:)          ! maximum saturated fraction for a gridcell
  real(r8), pointer :: smpmin(:)          ! restriction for min of soil potential (mm) (new)
  real(r8), pointer :: hkdepth(:)         ! decay factor (m)
  integer , pointer :: isoicol(:)         ! soil color class
  real(r8), pointer :: gwc_thr(:)         ! threshold soil moisture based on clay content
  real(r8), pointer :: mss_frc_cly_vld(:) ! [frc] Mass fraction clay limited to 0.20
  real(r8), pointer :: forc_ndep(:)       ! nitrogen deposition rate (gN/m2/s)
  real(r8), pointer :: efisop(:,:)        ! emission factors for isoprene (ug isoprene m-2 h-1)
  real(r8), pointer :: max_dayl(:)        ! maximum daylength (s)
  real(r8), pointer :: sandfrac(:)
  real(r8), pointer :: clayfrac(:)
!
!
! !OTHER LOCAL VARIABLES:
!EOP
  integer  :: ncid             ! netCDF file id 
  integer  :: n,j,ib,lev,bottom! indices
  integer  :: g,l,c,p          ! indices
  integer  :: m                ! vegetation type index
  real(r8) :: bd               ! bulk density of dry soil material [kg/m^3]
  real(r8) :: tkm              ! mineral conductivity
  real(r8) :: xksat            ! maximum hydraulic conductivity of soil [mm/s]
  real(r8) :: scalez = 0.025_r8   ! Soil layer thickness discretization (m)
  real(r8) :: clay,sand        ! temporaries
  real(r8) :: slope,intercept        ! temporary, for rooting distribution
  real(r8) :: temp, max_decl   ! temporary, for calculation of max_dayl
  integer  :: begp, endp       ! per-proc beginning and ending pft indices
  integer  :: begc, endc       ! per-proc beginning and ending column indices
  integer  :: begl, endl       ! per-proc beginning and ending landunit indices
  integer  :: begg, endg       ! per-proc gridcell ending gridcell indices


  real(r8) :: om_frac                ! organic matter fraction
  real(r8) :: om_watsat    = 0.9_r8  ! porosity of organic soil
  real(r8) :: om_hksat     = 0.1_r8  ! saturated hydraulic conductivity of organic soil [mm/s]
  real(r8) :: om_tkm       = 0.25_r8 ! thermal conductivity of organic soil (Farouki, 1986) [W/m/K]
  real(r8) :: om_sucsat    = 10.3_r8 ! saturated suction for organic matter (Letts, 2000)
  real(r8) :: om_csol      = 2.5_r8  ! heat capacity of peat soil *10^6 (J/K m3) (Farouki, 1986)
  real(r8) :: om_tkd       = 0.05_r8 ! thermal conductivity of dry organic soil (Farouki, 1981)
  real(r8) :: om_b         = 2.7_r8  ! Clapp Hornberger paramater for oragnic soil (Letts, 2000)
  real(r8) :: organic_max  = 130._r8 ! organic matter (kg/m3) where soil is assumed to act like peat 
  real(r8) :: csol_bedrock = 2.0e6_r8 ! vol. heat capacity of granite/sandstone  J/(m3 K)(Shabbir, 2000)
  real(r8) :: pc           = 0.5_r8   ! percolation threshold
  real(r8) :: pcbeta       = 0.139_r8 ! percolation exponent
  real(r8) :: perc_frac               ! "percolating" fraction of organic soil
  real(r8) :: perc_norm               ! normalize to 1 when 100% organic soil
  real(r8) :: uncon_hksat             ! series conductivity of mineral/organic soil
  real(r8) :: uncon_frac              ! fraction of "unconnected" soil
  integer  :: start(3),count(3)      ! netcdf start/count arrays
  integer  :: varid                  ! netCDF id's
  integer  :: ret

  integer  :: ier                                ! error status
  character(len=256) :: locfn                    ! local filename
  character(len= 32) :: subname = 'iniTimeConst' ! subroutine name
  integer :: mxsoil_color                        ! maximum number of soil color classes
  real(r8), allocatable :: zurb_wall(:,:)        ! wall (layer node depth)
  real(r8), allocatable :: zurb_roof(:,:)        ! roof (layer node depth)
  real(r8), allocatable :: dzurb_wall(:,:)       ! wall (layer thickness)
  real(r8), allocatable :: dzurb_roof(:,:)       ! roof (layer thickness)
  real(r8), allocatable :: ziurb_wall(:,:)       ! wall (layer interface)
  real(r8), allocatable :: ziurb_roof(:,:)       ! roof (layer interface)
!------------------------------------------------------------------------

  integer :: closelatidx,closelonidx
  real(r8):: closelat,closelon
  integer :: iostat

!------------------------------------------------------------------------

 !  write(6,*) 'Attempting to initialize time invariant variables'

  call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)


  efisop          => clm3%g%gve%efisop

  ! Assign local pointers to derived subtypes components (gridcell-level)
  lat             => clm3%g%lat
     
  ! Assign local pointers to derived subtypes components (landunit-level)

  ltype               => clm3%g%l%itype
  thick_wall          => clm3%g%l%lps%thick_wall
  thick_roof          => clm3%g%l%lps%thick_roof

  ! Assign local pointers to derived subtypes components (column-level)

  ctype           => clm3%g%l%c%itype
  clandunit       => clm3%g%l%c%landunit
  cgridcell       => clm3%g%l%c%gridcell
  z               => clm3%g%l%c%cps%z
  dz              => clm3%g%l%c%cps%dz
  zi              => clm3%g%l%c%cps%zi
  bsw             => clm3%g%l%c%cps%bsw
  bsw2            => clm3%g%l%c%cps%bsw2
  psisat          => clm3%g%l%c%cps%psisat
  vwcsat          => clm3%g%l%c%cps%vwcsat
  watsat          => clm3%g%l%c%cps%watsat
  watfc           => clm3%g%l%c%cps%watfc
  watdry          => clm3%g%l%c%cps%watdry  
  watopt          => clm3%g%l%c%cps%watopt  
  rootfr_road_perv => clm3%g%l%c%cps%rootfr_road_perv
  hksat           => clm3%g%l%c%cps%hksat
  sucsat          => clm3%g%l%c%cps%sucsat
  tkmg            => clm3%g%l%c%cps%tkmg
  tksatu          => clm3%g%l%c%cps%tksatu
  tkdry           => clm3%g%l%c%cps%tkdry
  csol            => clm3%g%l%c%cps%csol
  smpmin          => clm3%g%l%c%cps%smpmin
  hkdepth         => clm3%g%l%c%cps%hkdepth
  wtfact          => clm3%g%l%c%cps%wtfact
  isoicol         => clm3%g%l%c%cps%isoicol
  gwc_thr         => clm3%g%l%c%cps%gwc_thr
  mss_frc_cly_vld => clm3%g%l%c%cps%mss_frc_cly_vld
  max_dayl        => clm3%g%l%c%cps%max_dayl
  forc_ndep       => clm_a2l%forc_ndep


  ! Assign local pointers to derived subtypes components (pft-level)

  ivt             => clm3%g%l%c%p%itype
  pgridcell       => clm3%g%l%c%p%gridcell
  pcolumn         => clm3%g%l%c%p%column
  dewmx           => clm3%g%l%c%p%pps%dewmx
  rootfr          => clm3%g%l%c%p%pps%rootfr
  rresis          => clm3%g%l%c%p%pps%rresis
  sandfrac        => clm3%g%l%c%p%pps%sandfrac
  clayfrac        => clm3%g%l%c%p%pps%clayfrac

  allocate(zurb_wall(begl:endl,nlevurb), zurb_roof(begl:endl,nlevurb), &
           dzurb_wall(begl:endl,nlevurb), dzurb_roof(begl:endl,nlevurb), &
           ziurb_wall(begl:endl,0:nlevurb), ziurb_roof(begl:endl,0:nlevurb),  stat=ier)
  if (ier /= 0) then
     call endrun( 'iniTimeConst: allocation error for zurb_wall,zurb_roof,dzurb_wall,dzurb_roof,ziurb_wall,ziurb_roof' )
  end if

  ! --------------------------------------------------------------------
  ! Read soil color, sand and clay from surface dataset 
  ! --------------------------------------------------------------------

  call CLMDebug('TimeConst mark1')
  
  do p = begp,endp
     g = pgridcell(p)
     sandfrac(p) = sand3d(g,1)/100.0_r8
     clayfrac(p) = clay3d(g,1)/100.0_r8
  end do


   do m = 0,numpft
      if (m <= ntree) then
         pftcon%tree(m) = 1
      else
         pftcon%tree(m) = 0
      end if
      pftcon%z0mr(m) = z0mr(m)
      pftcon%displar(m) = displar(m)
      pftcon%dleaf(m) = dleaf(m)
      pftcon%xl(m) = xl(m)
      do ib = 1,numrad
         pftcon%rhol(m,ib) = rhol(m,ib)
         pftcon%rhos(m,ib) = rhos(m,ib)
         pftcon%taul(m,ib) = taul(m,ib)
         pftcon%taus(m,ib) = taus(m,ib)
      end do
      pftcon%qe25(m) = qe25(m)
      pftcon%vcmx25(m) = vcmx25(m)
      pftcon%mp(m) = mp(m)
      pftcon%c3psn(m) = c3psn(m)
      pftcon%slatop(m) = slatop(m)
      pftcon%dsladlai(m) = dsladlai(m)
      pftcon%leafcn(m) = leafcn(m)
      pftcon%flnr(m) = flnr(m)
      pftcon%smpso(m) = smpso(m)
      pftcon%smpsc(m) = smpsc(m)
      pftcon%fnitr(m) = fnitr(m)
      pftcon%woody(m) = woody(m)
      pftcon%lflitcn(m) = lflitcn(m)
      pftcon%frootcn(m) = frootcn(m)
      pftcon%livewdcn(m) = livewdcn(m)
      pftcon%deadwdcn(m) = deadwdcn(m)
#if (defined CROP)
      pftcon%graincn(m) = graincn(m)
#endif
      pftcon%froot_leaf(m) = froot_leaf(m)
      pftcon%stem_leaf(m) = stem_leaf(m)
      pftcon%croot_stem(m) = croot_stem(m)
      pftcon%flivewd(m) = flivewd(m)
      pftcon%fcur(m) = fcur(m)
      pftcon%lf_flab(m) = lf_flab(m)
      pftcon%lf_fcel(m) = lf_fcel(m)
      pftcon%lf_flig(m) = lf_flig(m)
      pftcon%fr_flab(m) = fr_flab(m)
      pftcon%fr_fcel(m) = fr_fcel(m)
      pftcon%fr_flig(m) = fr_flig(m)
      pftcon%dw_fcel(m) = dw_fcel(m)
      pftcon%dw_flig(m) = dw_flig(m)
      pftcon%leaf_long(m) = leaf_long(m)
      pftcon%evergreen(m) = evergreen(m)
      pftcon%stress_decid(m) = stress_decid(m)
      pftcon%season_decid(m) = season_decid(m)
      pftcon%resist(m) = resist(m)
      pftcon%dwood(m) = dwood
   end do

#ifdef CNDV
   do m = 0,numpft
      dgv_pftcon%crownarea_max(m) = pftpar20(m)
      dgv_pftcon%tcmin(m) = pftpar28(m)
      dgv_pftcon%tcmax(m) = pftpar29(m)
      dgv_pftcon%gddmin(m) = pftpar30(m)
      dgv_pftcon%twmax(m) = pftpar31(m)
      dgv_pftcon%reinickerp(m) = reinickerp
      dgv_pftcon%allom1(m) = allom1
      dgv_pftcon%allom2(m) = allom2
      dgv_pftcon%allom3(m) = allom3
      ! modification for shrubs by X.D.Z
      if (m > ntree .and. m <= nbrdlf_dcd_brl_shrub ) then 
         dgv_pftcon%allom1(m) = allom1s
         dgv_pftcon%allom2(m) = allom2s
      end if
   end do
#endif

   ! --------------------------------------------------------------------
   ! Define layer structure for soil, lakes, urban walls and roof 
   ! Vertical profile of snow is not initialized here 
   ! --------------------------------------------------------------------

   ! Lake layers (assumed same for all lake patches)

   dzlak(1) = 0.1_r8
   dzlak(2) = 1._r8
   dzlak(3) = 2._r8
   dzlak(4) = 3._r8
   dzlak(5) = 4._r8
   dzlak(6) = 5._r8
   dzlak(7) = 7._r8
   dzlak(8) = 7._r8
   dzlak(9) = 10.45_r8
   dzlak(10)= 10.45_r8

   zlak(1) =  0.05_r8
   zlak(2) =  0.6_r8
   zlak(3) =  2.1_r8
   zlak(4) =  4.6_r8
   zlak(5) =  8.1_r8
   zlak(6) = 12.6_r8
   zlak(7) = 18.6_r8
   zlak(8) = 25.6_r8
   zlak(9) = 34.325_r8
   zlak(10)= 44.775_r8

   ! Soil layers and interfaces (assumed same for all non-lake patches)
   ! "0" refers to soil surface and "nlevsoi" refers to the bottom of model soil

   do j = 1, nlevgrnd
      zsoi(j) = scalez*(exp(0.5_r8*(j-0.5_r8))-1._r8)    !node depths
   enddo

   dzsoi(1) = 0.5_r8*(zsoi(1)+zsoi(2))             !thickness b/n two interfaces
   do j = 2,nlevgrnd-1
      dzsoi(j)= 0.5_r8*(zsoi(j+1)-zsoi(j-1))
   enddo
   dzsoi(nlevgrnd) = zsoi(nlevgrnd)-zsoi(nlevgrnd-1)

   zisoi(0) = 0._r8
   do j = 1, nlevgrnd-1
      zisoi(j) = 0.5_r8*(zsoi(j)+zsoi(j+1))         !interface depths
   enddo
   zisoi(nlevgrnd) = zsoi(nlevgrnd) + 0.5_r8*dzsoi(nlevgrnd)

   ! Column level initialization for urban wall and roof layers and interfaces
   do l = begl, endl

   ! "0" refers to urban wall/roof surface and "nlevsoi" refers to urban wall/roof bottom
    if (ltype(l)==isturb) then
   
      do j = 1, nlevurb
        zurb_wall(l,j) = (j-0.5)*(thick_wall(l)/float(nlevurb))  !node depths
      end do
      do j = 1, nlevurb
        zurb_roof(l,j) = (j-0.5)*(thick_roof(l)/float(nlevurb))  !node depths
      end do

      dzurb_wall(l,1) = 0.5*(zurb_wall(l,1)+zurb_wall(l,2))    !thickness b/n two interfaces
      do j = 2,nlevurb-1
        dzurb_wall(l,j)= 0.5*(zurb_wall(l,j+1)-zurb_wall(l,j-1)) 
      enddo
      dzurb_wall(l,nlevurb) = zurb_wall(l,nlevurb)-zurb_wall(l,nlevurb-1)

      dzurb_roof(l,1) = 0.5*(zurb_roof(l,1)+zurb_roof(l,2))    !thickness b/n two interfaces
      do j = 2,nlevurb-1
        dzurb_roof(l,j)= 0.5*(zurb_roof(l,j+1)-zurb_roof(l,j-1)) 
      enddo
      dzurb_roof(l,nlevurb) = zurb_roof(l,nlevurb)-zurb_roof(l,nlevurb-1)

      ziurb_wall(l,0) = 0.
      do j = 1, nlevurb-1
        ziurb_wall(l,j) = 0.5*(zurb_wall(l,j)+zurb_wall(l,j+1))          !interface depths
      enddo
      ziurb_wall(l,nlevurb) = zurb_wall(l,nlevurb) + 0.5*dzurb_wall(l,nlevurb)

      ziurb_roof(l,0) = 0.
      do j = 1, nlevurb-1
        ziurb_roof(l,j) = 0.5*(zurb_roof(l,j)+zurb_roof(l,j+1))          !interface depths
      enddo
      ziurb_roof(l,nlevurb) = zurb_roof(l,nlevurb) + 0.5*dzurb_roof(l,nlevurb)
    end if
   end do

   ! --------------------------------------------------------------------
   ! Initialize nitrogen deposition values 
   ! for now these are constants by gridcell, eventually they
   ! will be variables from the atmosphere, and at some point in between
   ! they will be specified time varying fields.
   ! --------------------------------------------------------------------

   ! Grid level initialization
   do g = begg, endg

      ! nitrogen deposition (forcing flux from atmosphere)
      ! convert rate from 1/yr -> 1/s

!ndep moved to module_sf_clm and clm      
!      forc_ndep(g) = ndep(g)/(86400._r8 * 365._r8)
      
      ! VOC emission factors
      ! Set gridcell and landunit indices
      efisop(:,g)=efisop2d(:,g)

   end do
   
 !  write(6,*) 'efisop=',efisop

   call CLMDebug('mark2')

   ! --------------------------------------------------------------------
   ! Initialize soil and lake levels
   ! Initialize soil color, thermal and hydraulic properties
   ! --------------------------------------------------------------------

   ! Column level initialization
   do c = begc, endc

      ! Set gridcell and landunit indices
      g = cgridcell(c)
      l = clandunit(c)
      
      ! initialize maximum daylength, based on latitude and maximum declination
      ! maximum declination hardwired for present-day orbital parameters, 
      ! +/- 23.4667 degrees = +/- 0.409571 radians, use negative value for S. Hem
      call CLMDebug('mark21')
      max_decl = 0.409571
      if (lat(g) .lt. 0._r8) max_decl = -max_decl
      temp = -(sin(lat(g))*sin(max_decl))/(cos(lat(g)) * cos(max_decl))
      temp = min(1._r8,max(-1._r8,temp))
      max_dayl(c) = 2.0_r8 * 13750.9871_r8 * acos(temp)
      ! Initialize restriction for min of soil potential (mm)
      smpmin(c) = -1.e8_r8

      ! Decay factor (m)
      hkdepth(c) = 1._r8/2.5_r8

  call CLMDebug('mark22')
      ! Maximum saturated fraction
      wtfact(c) = gti(g)
   call CLMDebug('mark23')

      ! Soil color
      isoicol(c) = soic2d(g)

      ! Soil hydraulic and thermal properties
        ! Note that urban roof, sunwall and shadewall thermal properties used to 
        ! derive thermal conductivity and heat capacity are set to special 
        ! value because thermal conductivity and heat capacity for urban 
        ! roof, sunwall and shadewall are prescribed in SoilThermProp.F90 in 
        ! SoilTemperatureMod.F90
      if (ltype(l)==istdlak .or. ltype(l)==istwet .or. ltype(l)==istice) then
         do lev = 1,nlevgrnd
            bsw(c,lev)    = spval
            bsw2(c,lev)   = spval
            psisat(c,lev) = spval
            vwcsat(c,lev) = spval
            watsat(c,lev) = spval
            watfc(c,lev)  = spval
            hksat(c,lev)  = spval
            sucsat(c,lev) = spval
            tkmg(c,lev)   = spval
            tksatu(c,lev) = spval
            tkdry(c,lev)  = spval
            if (ltype(l)==istwet .and. lev > nlevsoi) then
               csol(c,lev) = csol_bedrock
            else
               csol(c,lev)= spval
            endif
            watdry(c,lev) = spval 
            watopt(c,lev) = spval 
         end do
      else if (ltype(l)==isturb .and. (ctype(c) /= icol_road_perv) .and. (ctype(c) /= icol_road_imperv) )then
         ! Urban Roof, sunwall, shadewall properties set to special value
         do lev = 1,nlevurb
            watsat(c,lev) = spval
            watfc(c,lev)  = spval
            bsw(c,lev)    = spval
            bsw2(c,lev)   = spval
            psisat(c,lev) = spval
            vwcsat(c,lev) = spval
            hksat(c,lev)  = spval
            sucsat(c,lev) = spval
            tkmg(c,lev)   = spval
            tksatu(c,lev) = spval
            tkdry(c,lev)  = spval
            csol(c,lev)   = spval
            watdry(c,lev) = spval 
            watopt(c,lev) = spval 
         end do
      else  ! soil columns of both urban and non-urban types
         do lev = 1,nlevgrnd
            ! duplicate clay and sand values from 10th soil layer
            if (lev .le. nlevsoi) then
               clay    = clay3d(g,lev)
               sand    = sand3d(g,lev)
               om_frac = (organic3d(g,lev)/organic_max)**2._r8
            else
               clay    = clay3d(g,nlevsoi)
               sand    = sand3d(g,nlevsoi)
               om_frac = 0._r8
            endif
            ! No organic matter for urban
            if (ltype(l)==isturb) then
              om_frac = 0._r8
            end if
            ! Note that the following properties are overwritten for urban impervious road 
            ! layers that are not soil in SoilThermProp.F90 within SoilTemperatureMod.F90
            watsat(c,lev) = 0.489_r8 - 0.00126_r8*sand
            bsw(c,lev)    = 2.91 + 0.159*clay
            sucsat(c,lev) = 10._r8 * ( 10._r8**(1.88_r8-0.0131_r8*sand) )
            bd            = (1._r8-watsat(c,lev))*2.7e3_r8 
            watsat(c,lev) = (1._r8 - om_frac)*watsat(c,lev) + om_watsat*om_frac
            tkm           = (1._r8-om_frac)*(8.80_r8*sand+2.92_r8*clay)/(sand+clay)+om_tkm*om_frac ! W/(m K)
            bsw(c,lev)    = (1._r8-om_frac)*(2.91_r8 + 0.159_r8*clay) + om_frac*om_b   
            bsw2(c,lev)   = -(3.10_r8 + 0.157_r8*clay - 0.003_r8*sand)
            psisat(c,lev) = -(exp((1.54_r8 - 0.0095_r8*sand + 0.0063_r8*(100.0_r8-sand-clay))*log(10.0_r8))*9.8e-5_r8)
            vwcsat(c,lev) = (50.5_r8 - 0.142_r8*sand - 0.037_r8*clay)/100.0_r8
            sucsat(c,lev) = (1._r8-om_frac)*sucsat(c,lev) + om_sucsat*om_frac  
            xksat         = 0.0070556 *( 10.**(-0.884+0.0153*sand) ) ! mm/s

            ! perc_frac is zero unless perf_frac greater than percolation threshold
            if (om_frac > pc) then
               perc_norm=(1._r8 - pc)**(-pcbeta)
               perc_frac=perc_norm*(om_frac - pc)**pcbeta
            else
               perc_frac=0._r8
            endif
            ! uncon_frac is fraction of mineral soil plus fraction of "nonpercolating" organic soil
            uncon_frac=(1._r8-om_frac)+(1._r8-perc_frac)*om_frac
            ! uncon_hksat is series addition of mineral/organic conductivites
            if (om_frac .lt. 1._r8) then
              uncon_hksat=uncon_frac/((1._r8-om_frac)/xksat &
                   +((1._r8-perc_frac)*om_frac)/om_hksat)
            else
              uncon_hksat = 0._r8
            end if
            hksat(c,lev)  = uncon_frac*uncon_hksat + (perc_frac*om_frac)*om_hksat

            tkmg(c,lev)   = tkm ** (1._r8- watsat(c,lev))           
            tksatu(c,lev) = tkmg(c,lev)*0.57_r8**watsat(c,lev)
            tkdry(c,lev)  = ((0.135_r8*bd + 64.7_r8) / (2.7e3_r8 - 0.947_r8*bd))*(1._r8-om_frac) + &
                            om_tkd*om_frac  
            csol(c,lev)   = ((1._r8-om_frac)*(2.128_r8*sand+2.385_r8*clay) / (sand+clay) +   &
                           om_csol*om_frac)*1.e6_r8  ! J/(m3 K)  
            if (lev .gt. nlevsoi) then
               csol(c,lev) = csol_bedrock
            endif
            watdry(c,lev) = watsat(c,lev) * (316230._r8/sucsat(c,lev)) ** (-1._r8/bsw(c,lev)) 
            watopt(c,lev) = watsat(c,lev) * (158490._r8/sucsat(c,lev)) ** (-1._r8/bsw(c,lev)) 
            !! added by K.Sakaguchi for beta from Lee and Pielke, 1992
            ! water content at field capacity, defined as hk = 0.1 mm/day
            ! used eqn (7.70) in CLM3 technote with k = 0.1 (mm/day) / 86400 (day/sec)
            watfc(c,lev) = watsat(c,lev) * (0.1_r8 / (hksat(c,lev)*86400._r8))**(1._r8/(2._r8*bsw(c,lev)+3._r8))
         end do
         !
         ! Urban pervious and impervious road
         !
         ! Impervious road layers -- same as above except set watdry and watopt as missing
         if (ctype(c) == icol_road_imperv) then
            do lev = 1,nlevgrnd
               watdry(c,lev) = spval 
               watopt(c,lev) = spval 
            end do
         ! pervious road layers -- same as above except also set rootfr_road_perv
         ! Currently, pervious road has same properties as soil
         else if (ctype(c) == icol_road_perv) then 
            do lev = 1, nlevgrnd
               rootfr_road_perv(c,lev) = 0._r8
            enddo
            do lev = 1,nlevsoi
               rootfr_road_perv(c,lev) = 0.1_r8  ! uniform profile
            end do
         end if
      endif
      ! Define lake or non-lake levels, layers and interfaces
      if (ltype(l) == istdlak) then
         z(c,1:nlevlak)  = zlak(1:nlevlak)
         dz(c,1:nlevlak) = dzlak(1:nlevlak)
      else if (ltype(l) == isturb) then
         if (ctype(c)==icol_sunwall .or. ctype(c)==icol_shadewall) then
            z(c,1:nlevurb)  = zurb_wall(l,1:nlevurb)
            zi(c,0:nlevurb) = ziurb_wall(l,0:nlevurb)
            dz(c,1:nlevurb) = dzurb_wall(l,1:nlevurb)
         else if (ctype(c)==icol_roof) then
            z(c,1:nlevurb)  = zurb_roof(l,1:nlevurb)
            zi(c,0:nlevurb) = ziurb_roof(l,0:nlevurb)
            dz(c,1:nlevurb) = dzurb_roof(l,1:nlevurb)
         else
            z(c,1:nlevurb)  = zsoi(1:nlevurb)
            zi(c,0:nlevurb) = zisoi(0:nlevurb)
            dz(c,1:nlevurb) = dzsoi(1:nlevurb)
         end if
      else
         z(c,1:nlevgrnd)  = zsoi(1:nlevgrnd)
         zi(c,0:nlevgrnd) = zisoi(0:nlevgrnd)
         dz(c,1:nlevgrnd) = dzsoi(1:nlevgrnd)
      end if

      ! Initialize terms needed for dust model
      clay = clay3d(g,1)
      gwc_thr(c) = 0.17_r8 + 0.14_r8*clay*0.01_r8
      mss_frc_cly_vld(c) = min(clay*0.01_r8, 0.20_r8)

   end do


    call CLMDebug('mark3')
   ! pft level initialization
   do p = begp, endp

      ! Initialize maximum allowed dew

      dewmx(p)  = 0.1_r8

      ! Initialize root fraction (computing from surface, d is depth in meter):
      ! Y = 1 -1/2 (exp(-ad)+exp(-bd) under the constraint that
      ! Y(d =0.1m) = 1-beta^(10 cm) and Y(d=d_obs)=0.99 with
      ! beta & d_obs given in Zeng et al. (1998).

      c = pcolumn(p)
      if (ivt(p) /= noveg) then
         do lev = 1, nlevgrnd
            rootfr(p,lev) = 0._r8
         enddo
         do lev = 1, nlevsoi-1
            rootfr(p,lev) = .5_r8*( exp(-roota_par(ivt(p)) * zi(c,lev-1))  &
                               + exp(-rootb_par(ivt(p)) * zi(c,lev-1))  &
                               - exp(-roota_par(ivt(p)) * zi(c,lev  ))  &
                               - exp(-rootb_par(ivt(p)) * zi(c,lev  )) )
         end do
         rootfr(p,nlevsoi) = .5_r8*( exp(-roota_par(ivt(p)) * zi(c,nlevsoi-1))  &
                                + exp(-rootb_par(ivt(p)) * zi(c,nlevsoi-1)) )
         rootfr(p,nlevsoi+1:nlevgrnd) =  0.0_r8

      else
         rootfr(p,1:nlevsoi) = 0._r8
      endif
      
      ! initialize rresis, for use in ecosystemdyn
      do lev = 1,nlevgrnd
         rresis(p,lev) = 0._r8
      end do

   end do ! end pft level initialization
   
#if (defined CN)
   ! initialize the CN variables for special landunits, including lake points
   if(nstep==1)  call CNiniSpecial()
#endif





   call CLMDebug('Successfully initialized time invariant variables')

  deallocate(zurb_wall)
  deallocate(zurb_roof)
  deallocate(dzurb_wall)
  deallocate(dzurb_roof)
  deallocate(ziurb_wall)
  deallocate(ziurb_roof)


end subroutine iniTimeConst

module QSatMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: QSatMod
!
! !DESCRIPTION:
! Computes saturation mixing ratio and the change in saturation
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: QSat
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: QSat
!
! !INTERFACE:
  subroutine QSat (T, p, es, esdT, qs, qsdT)
!
! !DESCRIPTION:
! Computes saturation mixing ratio and the change in saturation
! mixing ratio with respect to temperature.
! Reference:  Polynomial approximations from:
!             Piotr J. Flatau, et al.,1992:  Polynomial fits to saturation
!             vapor pressure.  Journal of Applied Meteorology, 31, 1507-1513.
!
! !USES:
    use shr_kind_mod , only: r8 => shr_kind_r8
    use shr_const_mod, only: SHR_CONST_TKFRZ
!
! !ARGUMENTS:
    implicit none
    real(r8), intent(in)  :: T        ! temperature (K)
    real(r8), intent(in)  :: p        ! surface atmospheric pressure (pa)
    real(r8), intent(out) :: es       ! vapor pressure (pa)
    real(r8), intent(out) :: esdT     ! d(es)/d(T)
    real(r8), intent(out) :: qs       ! humidity (kg/kg)
    real(r8), intent(out) :: qsdT     ! d(qs)/d(T)
!
! !CALLED FROM:
! subroutine Biogeophysics1 in module Biogeophysics1Mod
! subroutine BiogeophysicsLake in module BiogeophysicsLakeMod
! subroutine CanopyFluxesMod CanopyFluxesMod
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
!
!
! !LOCAL VARIABLES:
!EOP
!
    real(r8) :: T_limit
    real(r8) :: td,vp,vp1,vp2
!
! For water vapor (temperature range 0C-100C)
!
    real(r8), parameter :: a0 =  6.11213476_r8
    real(r8), parameter :: a1 =  0.444007856_r8
    real(r8), parameter :: a2 =  0.143064234e-01_r8
    real(r8), parameter :: a3 =  0.264461437e-03_r8
    real(r8), parameter :: a4 =  0.305903558e-05_r8
    real(r8), parameter :: a5 =  0.196237241e-07_r8
    real(r8), parameter :: a6 =  0.892344772e-10_r8
    real(r8), parameter :: a7 = -0.373208410e-12_r8
    real(r8), parameter :: a8 =  0.209339997e-15_r8
!
! For derivative:water vapor
!
    real(r8), parameter :: b0 =  0.444017302_r8
    real(r8), parameter :: b1 =  0.286064092e-01_r8
    real(r8), parameter :: b2 =  0.794683137e-03_r8
    real(r8), parameter :: b3 =  0.121211669e-04_r8
    real(r8), parameter :: b4 =  0.103354611e-06_r8
    real(r8), parameter :: b5 =  0.404125005e-09_r8
    real(r8), parameter :: b6 = -0.788037859e-12_r8
    real(r8), parameter :: b7 = -0.114596802e-13_r8
    real(r8), parameter :: b8 =  0.381294516e-16_r8
!
! For ice (temperature range -75C-0C)
!
    real(r8), parameter :: c0 =  6.11123516_r8
    real(r8), parameter :: c1 =  0.503109514_r8
    real(r8), parameter :: c2 =  0.188369801e-01_r8
    real(r8), parameter :: c3 =  0.420547422e-03_r8
    real(r8), parameter :: c4 =  0.614396778e-05_r8
    real(r8), parameter :: c5 =  0.602780717e-07_r8
    real(r8), parameter :: c6 =  0.387940929e-09_r8
    real(r8), parameter :: c7 =  0.149436277e-11_r8
    real(r8), parameter :: c8 =  0.262655803e-14_r8
!
! For derivative:ice
!
    real(r8), parameter :: d0 =  0.503277922_r8
    real(r8), parameter :: d1 =  0.377289173e-01_r8
    real(r8), parameter :: d2 =  0.126801703e-02_r8
    real(r8), parameter :: d3 =  0.249468427e-04_r8
    real(r8), parameter :: d4 =  0.313703411e-06_r8
    real(r8), parameter :: d5 =  0.257180651e-08_r8
    real(r8), parameter :: d6 =  0.133268878e-10_r8
    real(r8), parameter :: d7 =  0.394116744e-13_r8
    real(r8), parameter :: d8 =  0.498070196e-16_r8
!-----------------------------------------------------------------------

    T_limit = T - SHR_CONST_TKFRZ
    if (T_limit > 100.0_r8) T_limit=100.0_r8
    if (T_limit < -75.0_r8) T_limit=-75.0_r8

    td       = T_limit
    if (td >= 0.0_r8) then
       es   = a0 + td*(a1 + td*(a2 + td*(a3 + td*(a4 &
            + td*(a5 + td*(a6 + td*(a7 + td*a8)))))))
       esdT = b0 + td*(b1 + td*(b2 + td*(b3 + td*(b4 &
            + td*(b5 + td*(b6 + td*(b7 + td*b8)))))))
    else
       es   = c0 + td*(c1 + td*(c2 + td*(c3 + td*(c4 &
            + td*(c5 + td*(c6 + td*(c7 + td*c8)))))))
       esdT = d0 + td*(d1 + td*(d2 + td*(d3 + td*(d4 &
            + td*(d5 + td*(d6 + td*(d7 + td*d8)))))))
    endif

    es    = es    * 100._r8            ! pa
    esdT  = esdT  * 100._r8            ! pa/K

    vp    = 1.0_r8   / (p - 0.378_r8*es)
    vp1   = 0.622_r8 * vp
    vp2   = vp1   * vp

    qs    = es    * vp1             ! kg/kg
    qsdT  = esdT  * vp2 * p         ! 1 / K

  end subroutine QSat

end module QSatMod
module initGridcellsMod

!Edited to 3.5 from Jiming Jin's 3.0 version by Zack Subin, 7/17/08.
!Latdeg, londeg, & area for l, c, & p, and itype for g, was eliminated.
!To prevent redoing equations, areas are put back in clmtype, but 
!latdeg & londeg are redundant and left out.

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: initGridcellsMod
!
! !DESCRIPTION:
! Initializes sub-grid mapping for each land grid cell
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
  use clmtype
  use clm_varpar, only : lsmlon, lsmlat,npatch_urban,npatch_glacier,npatch_crop, maxpatch, maxpatch_pft
  use clm_varsur, only : wtxy,vegxy,numlon, area, latixy, longxy

! !PUBLIC TYPES:
  implicit none
  private
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public initGridcells      ! Initialize sub-grid gridcell mapping
!
! !PIVATE MEMBER FUNCTIONS:
  private landunit_veg_compete
  private landunit_veg_noncompete
  private landunit_special
  private landunit_crop_noncompete
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!
! !LOCAL MODULE VARIABLES:
  type(gridcell_type), pointer :: gptr  ! pointer to gridcell derived subtype
  type(landunit_type), pointer :: lptr  ! pointer to landunit derived subtype
  type(column_type)  , pointer :: cptr  ! pointer to column derived subtype
  type(pft_type)     , pointer :: pptr  ! pointer to pft derived subtype
!-----------------------------------------------------------------------

contains

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: initGridcells
!
! !INTERFACE:
  subroutine initGridcells
!
! !DESCRIPTION:
! Initialize sub-grid mapping and allocates space for derived type
! hierarchy.  For each land gridcell determine landunit, column and
! pft properties.  Note that ngcells, nlunits, ncols and npfts are
! per-processor totals here and are currently not used for anything other
! than placeholders.  Determine if there are any vegetated landunits and
! if so---the weight of the vegetated landunit relative to the gridcell
! The first landunit contains all the vegetated patches (if any) For now,
! the vegetated patches will all be gathered on a single landunit, with
! each vegetated type having its own column on that landunit.  The special
! patches (urban, lake, wetland, glacier) each get their own landunit
! having a single column and one non-vegetated pfts
!
! !USES:
    use decompMod , only : get_proc_bounds, get_gcell_xyind, &
                           get_gcell_info 
    use clm_varcon, only : pie
!
! !ARGUMENTS:
    implicit none
                                                          ! weights
!
! !REVISION HISTORY:
! Created by Peter Thornton and Mariana Vertenstein
!
!EOP
!
! !LOCAL VARIABLES:
    integer :: g,i,j,m,n,gi,li,ci,pi ! indices
    integer :: ngcells     ! temporary dummy
    integer :: nlunits     ! temporary dummy
    integer :: ncols       ! temporary dummy
    integer :: npfts       ! temporary dummy
    integer :: nveg        ! number of pfts in naturally vegetated landunit
    real(r8):: wtveg       ! weight (relative to gridcell) of naturally vegetated landunit
    integer :: ncrop       ! number of crop pfts in crop landunit
    real(r8):: wtcrop      ! weight (relative to gridcell) of crop landunit
    integer :: begp, endp  ! per-proc beginning and ending pft indices
    integer :: begc, endc  ! per-proc beginning and ending column indices
    integer :: begl, endl  ! per-proc beginning and ending landunit indices
    integer :: begg, endg  ! per-proc gridcell ending gridcell indices
    integer :: ier         ! error status
    integer :: ilunits, icols, ipfts  ! temporaries
!New variables -- comment for now, probably not needed
!    integer :: nlake          ! number of pfts (columns) in lake landunit
!    real(r8):: wtlake         ! weight (gridcell) of lake landunit
!    integer :: nwetland       ! number of pfts (columns) in wetland landunit
!    real(r8):: wtwetland      ! weight (gridcell) of wetland landunit
!    integer :: nglacier       ! number of pfts (columns) in glacier landunit
!    real(r8):: wtglacier      ! weight (gridcell) of glacier landunit
!!!!!!!!!!!!!
!------------------------------------------------------------------------

    ! Set pointers into derived types for this module

    gptr => clm3%g
    lptr => clm3%g%l
    cptr => clm3%g%l%c
    pptr => clm3%g%l%c%p

    ! Determine necessary indices

    call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)
    call get_gcell_xyind(begg, endg)


    ! Determine number of land gridcells on this processor

    clm3%ngridcells = endg - begg + 1

    ! Determine gridcell properties.
    ! Set area, weight, and type information for this gridcell.
    ! For now there is only one type of gridcell, value = 1
    ! Still need to resolve the calculation of area for the gridcell

    ngcells = begg-1
    nlunits = begl-1
    ncols   = begc-1
    npfts   = begp-1

    do gi = begg, endg

       ! Get 2d grid indices


!
       gptr%area(gi)   = area(gi)

  call CLMDebug('mark0')
!       gptr%itype(gi)  = 1
      !gptr%wtglob(g)  = gptr%area(g)/clm3%area
       gptr%lat(gi)    = latixy(gi) * pie/180.
       gptr%lon(gi)    = longxy(gi) * pie/180.
       gptr%latdeg(gi) = latixy(gi)
       gptr%londeg(gi) = longxy(gi)
!      gptr%landfrac(gi) = landfrac(i,j)
!

       gptr%luni(gi) = nlunits + 1
       gptr%coli(gi) = ncols   + 1
       gptr%pfti(gi) = npfts   + 1

       call get_gcell_info(gi, nlunits=ilunits, ncols=icols, npfts=ipfts)

       ngcells = ngcells + 1
       nlunits = nlunits + ilunits
       ncols   = ncols   + icols
       npfts   = npfts   + ipfts

       gptr%lunf(gi) = nlunits
       gptr%colf(gi) = ncols
       gptr%pftf(gi) = npfts

       gptr%nlandunits(gi) = gptr%lunf(gi) - gptr%luni(gi) + 1
       gptr%ncolumns(gi)   = gptr%colf(gi) - gptr%coli(gi) + 1
       gptr%npfts(gi)      = gptr%pftf(gi) - gptr%pfti(gi) + 1

    end do

     call CLMDebug('mark1')

    ! For each land gridcell determine landunit, column and pft properties.

    ngcells = 0
    nlunits = 0
    ncols   = 0
    npfts   = 0

    li = begl - 1
    ci = begc - 1
    pi = begp - 1

    do gi = begg,endg

       ! Determine 2d lat and lon indices


       ! Obtain gridcell properties

       call get_gcell_info(gi, nveg=nveg, wtveg=wtveg, ncrop=ncrop, wtcrop=wtcrop)

       ! Determine naturally vegetated landunit

#if (defined NOCOMPETE)
       if (nveg > 0) call landunit_veg_noncompete(nveg, wtveg,  i, j, gi, li, ci, pi)
#else
       if (nveg > 0) call landunit_veg_compete(nveg, wtveg, i, j, gi, li, ci, pi)
#endif

       ! Determine crop landunit.

       if (ncrop > 0) call landunit_crop_noncompete(ncrop, wtcrop,  i, j, gi, li, ci, pi)

       ! Determine special landunits (urban, lake, wetland, glacier).

       do m = npatch_urban, npatch_glacier
          if (wtxy(gi,m) > 0.) call landunit_special(i, j, m, gi, li, ci, pi)
       end do

    end do


  end subroutine initGridcells

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: landunit_veg_compete
!
! !INTERFACE:
  subroutine landunit_veg_compete (nveg, wtveg,  i, j, &
                                   gi, li, ci, pi)
!
! !DESCRIPTION:
! Initialize vegetated landunit with competition
!
! !USES:
    use clm_varcon, only : istsoil
!
! !ARGUMENTS:
    implicit none
    integer , intent(in) :: nveg   ! number of vegetated patches in gridcell
    real(r8), intent(in) :: wtveg  ! weight relative to gridcell of veg
                                   ! landunit
    integer , intent(in) :: i      ! 2d longitude index
    integer , intent(in) :: j      ! 2d latitude index
    integer , intent(in) :: gi     ! gridcell index
    integer , intent(inout) :: li  ! landunit index
    integer , intent(inout) :: ci  ! column index
    integer , intent(inout) :: pi  ! pft index
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!
! !LOCAL VARIABLES:
    integer  :: m                          ! indices
!------------------------------------------------------------------------

    ! Set landunit properties
    ! Increment landunits and set indices into lower levels in hierarchy and higher levels
    ! in hierarchy and topological mapping functionality

    li = li + 1
    lptr%ncolumns(li) = 1
    lptr%coli(li) = ci + 1
    lptr%colf(li) = ci + 1
    lptr%npfts(li) = nveg
    lptr%pfti(li) = pi + 1
    lptr%pftf(li) = pi + nveg
!
    lptr%area(li) = gptr%area(gi) * wtveg
!
    lptr%gridcell(li) = gi
    lptr%wtgcell(li) = wtveg
!
!    lptr%latdeg(li) = latixy(i,j)
!    lptr%londeg(li) = longxy(i,j)
!
    lptr%ifspecial(li) = .false.
    lptr%lakpoi(li) = .false.
    lptr%urbpoi(li) = .false.
    lptr%itype(li) = istsoil

    ! Set column properties for this landunit
    ! Increment column  - set only one column on compete landunit -  and set indices into
    ! lower levels in hierarchy, higher levels in hierarchy and topological mapping
    ! functionality (currently all columns have type 1)

    ci = ci + 1
    cptr%npfts(ci) = nveg
    cptr%pfti(ci) = pi + 1
    cptr%pftf(ci) = pi + nveg
 


!
    cptr%area(ci) = lptr%area(li)
!
    cptr%landunit(ci) = li
    cptr%gridcell(ci) = gi
    cptr%wtlunit(ci) = 1.0
    cptr%wtgcell(ci) = wtveg
!
!    cptr%latdeg(ci) = latixy(i,j)
!    cptr%londeg(ci) = longxy(i,j)
!
    cptr%itype(ci) = 1

    ! Set pft properties for this landunit
    ! Topological mapping functionality

!dir$ concurrent
!cdir nodep
    do m = 1,maxpatch_pft
       if (wtxy(gi,m) > 0.) then
          pi = pi+1
          pptr%column(pi) = ci
          pptr%landunit(pi) = li
          pptr%gridcell(pi) = gi
          pptr%wtcol(pi) = wtxy(gi,m) / wtveg
          pptr%wtlunit(pi) = wtxy(gi,m) / wtveg
          pptr%wtgcell(pi) = wtxy(gi,m)
!
          pptr%area(pi) = cptr%area(ci) * pptr%wtcol(pi)
!
          pptr%mxy(pi) = m
!
!          pptr%latdeg(pi) = latixy(i,j)
!          pptr%londeg(pi) = longxy(i,j)
!
          pptr%itype(pi) = vegxy(gi,m)
       end if ! non-zero weight for this pft
    end do ! loop through maxpatch_pft

  end subroutine landunit_veg_compete

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: landunit_veg_noncompete
!
! !INTERFACE:
  subroutine landunit_veg_noncompete (nveg, wtveg, i, j, &
                                      gi, li, ci, pi)
!
! !DESCRIPTION:
! Initialize vegetated landunit without competition
!
! !USES:
    use clm_varcon, only : istsoil
!
! !ARGUMENTS:
    implicit none
    integer , intent(in) :: nveg       ! number of vegetated patches in gridcell
    real(r8), intent(in) :: wtveg      ! weight relative to gridcell of veg landunit
    integer , intent(in) :: i          ! 2d longitude index
    integer , intent(in) :: j          ! 2d latitude index
    integer , intent(in) :: gi         ! gridcell index
    integer , intent(inout) :: li      ! landunit index
    integer , intent(inout) :: ci      ! column index
    integer , intent(inout) :: pi      ! pft index
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!
! !LOCAL VARIABLES:
    integer  :: m                          ! indices
    real(r8) :: wtlunit                    ! weight relative to landunit
!------------------------------------------------------------------------

    ! Set landunit properties
    ! Increment landunits and set indices into lower levels in hierarchy and higher levels
    ! in hierarchy and topological mapping functionality

    li = li + 1
    lptr%ncolumns(li) = nveg
    lptr%coli(li) = ci + 1
    lptr%colf(li) = ci + nveg
    lptr%npfts(li) = nveg
    lptr%pfti(li) = pi + 1
    lptr%pftf(li) = pi + nveg
!
    lptr%area(li) = gptr%area(gi) * wtveg
!
    lptr%gridcell(li) = gi
    lptr%wtgcell(li) = wtveg
!
!    lptr%latdeg(li) = latixy(i,j)
!    lptr%londeg(li) = longxy(i,j)
!
    lptr%ifspecial(li) = .false.
    lptr%lakpoi(li) = .false.
    lptr%itype(li) = istsoil

    ! Set column properties for this landunit
    ! Increment column  - each column has its own pft -  and set indices into
    ! lower levels in hierarchy, higher levels in hierarchy and topological mapping
    ! functionality (currently all columns have type 1)
    ! Set column and pft properties
    ! Loop through regular (vegetated) patches, assign one column for each
    ! vegetated patch with non-zero weight. The weights for each column on
    ! the vegetated landunit must add to one when summed over the landunit,
    ! so the wtxy(i,j,m) values are taken relative to the total wtveg.

!dir$ concurrent
!cdir nodep
    do m = 1, maxpatch_pft
       if (wtxy(gi,m) > 0.) then

          ! Determine weight relative to landunit of pft/column

          wtlunit = wtxy(gi,m) / wtveg

          ! Increment number of columns on landunit

          ci = ci + 1
          cptr%npfts(ci) = 1
          cptr%pfti(ci) = ci
          cptr%pftf(ci) = ci
!
          cptr%area(ci) = lptr%area(li) * wtlunit
!
          cptr%landunit(ci) = li
          cptr%gridcell(ci) = gi
          cptr%wtlunit(ci) = wtlunit
!
          cptr%wtgcell(ci) = cptr%area(ci) / gptr%area(gi)
!
!
!          cptr%latdeg(ci) = latixy(i,j)
!          cptr%londeg(ci) = longxy(i,j)
!
          cptr%itype(ci) = 1

          ! Increment number of pfts on this landunit
          ! Set area, weight (relative to column) and type information for this pft
          ! For now, a single pft per column, so weight = 1
          ! pft type comes from the m dimension of wtxy()
          ! Set grid index, weight (relative to grid cell)
          ! and m index (needed for laixy, etc. reference)

          pi = pi + 1
          pptr%column(pi) = ci
          pptr%landunit(pi) = li
          pptr%gridcell(pi) = gi
          pptr%wtcol(pi) = 1.0
          pptr%wtlunit(pi) = cptr%wtlunit(ci)
!
          pptr%area(pi) = cptr%area(ci)
          pptr%wtgcell(pi) = pptr%area(pi) / gptr%area(gi)
!
          pptr%mxy(pi) = m
!
!          pptr%latdeg(pi) = latixy(i,j)
!          pptr%londeg(pi) = longxy(i,j)
!
          pptr%itype(pi) = vegxy(gi,m)

       end if   ! end if non-zero weight
    end do   ! end loop through the possible vegetated patch indices

  end subroutine landunit_veg_noncompete

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: landunit_special
!
! !INTERFACE:
  subroutine landunit_special ( i, j, m, gi, li, ci, pi)
!
! !DESCRIPTION:
! Initialize special landunits (urban, lake, wetland, glacier)
!
! !USES:
    use pftvarcon, only : noveg
    use clm_varcon, only : istice, istwet, istdlak, isturb
    use clm_varpar, only : npatch_lake, npatch_wet 
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: i            !2-dim longitude index
    integer, intent(in) :: j            !2-dim latitude index
    integer, intent(in) :: m            !2-dim PFT patch index
    integer, intent(in) :: gi           !gridcell index
    integer, intent(inout) :: li        !landunit index
    integer, intent(inout) :: ci        !column index
    integer, intent(inout) :: pi        !pft index
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!
! !LOCAL VARIABLES:
    integer  :: c             !column loop index
    integer  :: ncols         !number of columns
    integer  :: npfts         !number of pfts
    integer  :: ier           !error status
    real(r8) :: weight        !temporary weight
    integer  :: itype         !landunit type
!------------------------------------------------------------------------

    ! Define landunit type

    if (m == npatch_lake) then         !deep lake (from pctlak)
       itype = istdlak
    else if (m == npatch_wet) then     !wetland (from pctwet)
       itype = istwet
    else if (m == npatch_glacier) then !glacier (from pctgla)
       itype = istice
    else if (m == npatch_urban) then   !urban (from pcturb)
       itype = isturb
    else                               !error
       write(6,*)'special landunit are currently only:', &
            ' deep lake, wetland, glacier or urban)'
       call endrun()
    endif

    ! Determine landunit index and landunit properties

    li = li + 1
    lptr%ncolumns(li) = 1
    lptr%coli(li) = ci + 1
    lptr%colf(li) = ci + 1
    lptr%npfts(li) = 1
    lptr%pfti(li) = pi + 1
    lptr%pftf(li) = pi + 1
!
    lptr%area(li) = gptr%area(gi) * wtxy(gi,m)
    lptr%gridcell(li) = gi
    lptr%wtgcell(li) = lptr%area(li) / gptr%area(gi)
!
!
!    lptr%latdeg(li) = latixy(i,j)
!    lptr%londeg(li) = longxy(i,j)
!
    lptr%ifspecial(li) = .true.
    if (itype == istdlak) then
       lptr%lakpoi(li) = .true.
    else
       lptr%lakpoi(li) = .false.
    end if
    lptr%itype(li) = itype

    ! For the special landunits there currently is only one column
    ! Later, the age classes will be implemented on different columns within
    ! the same landunit, so the column type will correspond to an age class

    ncols = 1

    ! Loop through columns for this landunit and set the column properties
    ! We know that there is only one column for the special landunit - but
    ! the loop is included for future consistency.

    do c = 1,ncols

       ! Determine column index and column properties
       ! For now all columns have the same type, value = 1

       weight = 1.0/ncols

       ci = ci + c
       cptr%npfts(ci) = 1
       cptr%pfti(ci) = pi + 1
       cptr%pftf(ci) = pi + 1


!
       cptr%area(ci) = lptr%area(li) * weight
!
       cptr%landunit(ci) = li
       cptr%gridcell(ci) = gi
       cptr%wtlunit(ci) = weight
!
       cptr%wtgcell(ci) = cptr%area(ci) / gptr%area(gi)
!
!
!       cptr%latdeg(ci) = latixy(i,j)
!       cptr%londeg(ci) = longxy(i,j)
!
       cptr%itype(ci) = 1

       ! Determine pft index and pft properties
       ! Each column has one non-vegetated pft
       ! Set area, weight (relative to column), and type information
       ! for this non-vegetated pft
       ! Set grid index, weight (relative to grid cell) and
       ! m index (needed for laixy, etc. reference)

       npfts = 1
       weight = 1.0/npfts

       pi = pi + 1
       pptr%column(pi) = ci
       pptr%landunit(pi) = li
       pptr%gridcell(pi) = gi
!
       pptr%area(pi) = lptr%area(li) * weight
!
       pptr%wtcol(pi) = weight
       pptr%wtlunit(pi) = cptr%wtlunit(ci)
!
       pptr%wtgcell(pi) = pptr%area(pi) / gptr%area(gi)
!
       pptr%mxy(pi) = m
!
!       pptr%latdeg(pi) = latixy(i,j)
!       pptr%londeg(pi) = longxy(i,j)
!
       pptr%itype(pi) = noveg

    end do   ! end loop through ncolumns

  end subroutine landunit_special

!------------------------------------------------------------------------
!BOP
!
! !IROUTINE: landunit_crop_noncompete
!
! !INTERFACE:
  subroutine landunit_crop_noncompete (ncrop, wtcrop,  i, j, &
                                       gi, li, ci, pi)
!
! !DESCRIPTION:
! Initialize crop landunit without competition
!
! !USES:
    use clm_varcon, only : istsoil
#ifdef CN
    use clm_varcon, only : istsoil,istcrop
#endif

    use clm_varpar, only : npatch_crop
!
! !ARGUMENTS:
    implicit none
    integer , intent(in) :: ncrop       ! number of vegetated patches in gridcell
    real(r8), intent(in) :: wtcrop      ! weight relative to gridcell of veg landunit
    integer , intent(in) :: i          ! 2d longitude index
    integer , intent(in) :: j          ! 2d latitude index
    integer , intent(in) :: gi         ! gridcell index
    integer , intent(inout) :: li      ! landunit index
    integer , intent(inout) :: ci      ! column index
    integer , intent(inout) :: pi      ! pft index
!
! !REVISION HISTORY:
! Created by Sam Levis
!
!EOP
!
! !LOCAL VARIABLES:
    integer  :: m                          ! indices
    real(r8) :: wtlunit                    ! weight relative to landunit
!------------------------------------------------------------------------

    ! Set landunit properties
    ! Increment landunits and set indices into lower levels in hierarchy and higher levels
    ! in hierarchy and topological mapping functionality

    li = li + 1
    lptr%ncolumns(li) = ncrop
    lptr%coli(li) = ci + 1
    lptr%colf(li) = ci + ncrop
    lptr%npfts(li) = ncrop
    lptr%pfti(li) = pi + 1
    lptr%pftf(li) = pi + ncrop
!
    lptr%area(li) = gptr%area(gi) * wtcrop
    lptr%gridcell(li) = gi
    lptr%wtgcell(li) = wtcrop
!
!
!    lptr%latdeg(li) = latixy(i,j)
!    lptr%londeg(li) = longxy(i,j)
!
    lptr%ifspecial(li) = .false.
    lptr%lakpoi(li) = .false.
    lptr%urbpoi(li)    = .false.
#ifdef CROP
    lptr%itype(li) = istcrop
#else
    lptr%itype(li) = istsoil
#endif
    ! Set column properties for this landunit
    ! Increment column  - each column has its own pft -  and set indices into
    ! lower levels in hierarchy, higher levels in hierarchy and topological mapping
    ! functionality (currently all columns have type 1)
    ! Set column and pft properties
    ! Loop through regular (vegetated) patches, assign one column for each
    ! vegetated patch with non-zero weight. The weights for each column on
    ! the vegetated landunit must add to one when summed over the landunit,
    ! so the wtxy(i,j,m) values are taken relative to the total wtcrop.

!dir$ concurrent
!cdir nodep
    do m = npatch_glacier+1, npatch_crop
       if (wtxy(gi,m) > 0.) then

          ! Determine weight of crop pft/column relative to crop landunit

          wtlunit = wtxy(gi,m) / wtcrop

          ! Increment number of columns on landunit

          ci = ci + 1
          cptr%npfts(ci) = 1
!
          cptr%area(ci) = lptr%area(li) * wtlunit
          cptr%landunit(ci) = li
          cptr%gridcell(ci) = gi
          cptr%wtlunit(ci) = wtlunit
          cptr%wtgcell(ci) = cptr%area(ci) / gptr%area(gi)
!
!
!          cptr%latdeg(ci) = latixy(i,j)
!          cptr%londeg(ci) = longxy(i,j)
!
          cptr%itype(ci) = 1

          ! Increment number of pfts on this landunit
          ! Set area, weight (relative to column) and type information for this pft
          ! For now, a single pft per column, so weight relative to column is 1
          ! pft type comes from the m dimension of wtxy()
          ! Set grid index, weight relative to grid cell and m index (needed for laixy, etc.)

          pi = pi + 1
          pptr%column(pi) = ci
          pptr%landunit(pi) = li
          pptr%gridcell(pi) = gi
!
          pptr%wtcol(pi) = 1.0
          pptr%wtlunit(pi) = cptr%wtlunit(ci)
          pptr%area(pi) = cptr%area(ci)
          pptr%wtgcell(pi) = pptr%area(pi) / gptr%area(gi)
!
          pptr%mxy(pi) = m
!
!          pptr%latdeg(pi) = latixy(i,j)
!          pptr%londeg(pi) = longxy(i,j)
!
          pptr%itype(pi) = vegxy(gi,m)

          ! Set pft indices for column

          cptr%pfti(ci) = pi
          cptr%pftf(ci) = pi
 


 
       end if   ! end if non-zero weight
    end do   ! end loop through the possible vegetated patch indices

  end subroutine landunit_crop_noncompete

end module initGridcellsMod

module FracWetMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: FracWetMod
!
! !DESCRIPTION:
! Determine fraction of vegetated surfaces which are wet and
! fraction of elai which is dry.
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: FracWet
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: FracWet
!
! !INTERFACE:
  subroutine FracWet(numf, filter)
!
! !DESCRIPTION:
! Determine fraction of vegetated surfaces which are wet and
! fraction of elai which is dry. The variable ``fwet'' is the
! fraction of all vegetation surfaces which are wet including
! stem area which contribute to evaporation. The variable ``fdry''
! is the fraction of elai which is dry because only leaves
! can transpire.  Adjusted for stem area which does not transpire.
!
! !USES:
    use shr_kind_mod, only: r8 => shr_kind_r8
    use clmtype
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: numf                  ! number of filter non-lake points
    integer, intent(in) :: filter(numf)          ! pft filter for non-lake points
!
! !CALLED FROM:
! subroutine Hydrology1 in module Hydrology1Mod
!
! !REVISION HISTORY:
! Created by Keith Oleson and M. Vertenstein
! 03/08/29 Mariana Vertenstein : Migrated to vectorized code
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
    integer , pointer :: frac_veg_nosno(:) ! fraction of veg not covered by snow (0/1 now) [-]
    real(r8), pointer :: dewmx(:)          ! Maximum allowed dew [mm]
    real(r8), pointer :: elai(:)           ! one-sided leaf area index with burying by snow
    real(r8), pointer :: esai(:)           ! one-sided stem area index with burying by snow
    real(r8), pointer :: h2ocan(:)         ! total canopy water (mm H2O)
!
! local pointers to implicit out arguments
!
    real(r8), pointer :: fwet(:)           ! fraction of canopy that is wet (0 to 1)
    real(r8), pointer :: fdry(:)           ! fraction of foliage that is green and dry [-] (new)
!
!
! !OTHER LOCAL VARIABLES:
!EOP
!
    integer  :: fp,p             ! indices
    real(r8) :: vegt             ! frac_veg_nosno*lsai
    real(r8) :: dewmxi           ! inverse of maximum allowed dew [1/mm]
!-----------------------------------------------------------------------

    ! Assign local pointers to derived subtypes components (pft-level)

    frac_veg_nosno => clm3%g%l%c%p%pps%frac_veg_nosno
    dewmx => clm3%g%l%c%p%pps%dewmx
    elai => clm3%g%l%c%p%pps%elai
    esai => clm3%g%l%c%p%pps%esai
    h2ocan => clm3%g%l%c%p%pws%h2ocan
    fwet => clm3%g%l%c%p%pps%fwet
    fdry => clm3%g%l%c%p%pps%fdry

    ! Compute fraction of canopy that is wet and dry

!dir$ concurrent
!cdir nodep
    do fp = 1,numf
       p = filter(fp)
       if (frac_veg_nosno(p) == 1) then
          if (h2ocan(p) > 0._r8) then
             vegt    = frac_veg_nosno(p)*(elai(p) + esai(p))
             dewmxi  = 1.0_r8/dewmx(p)
             fwet(p) = ((dewmxi/vegt)*h2ocan(p))**0.666666666666_r8
             fwet(p) = min (fwet(p),1.0_r8)   ! Check for maximum limit of fwet
          else
             fwet(p) = 0._r8
          end if
          fdry(p) = (1._r8-fwet(p))*elai(p)/(elai(p)+esai(p))
#if (defined PERGRO)
          fwet(p) = 0._r8
          fdry(p) = elai(p)/(elai(p)+esai(p))
#endif
       else
          fwet(p) = 0._r8
          fdry(p) = 0._r8
       end if
    end do

  end subroutine FracWet

end module FracWetMod

module FrictionVelocityMod

!------------------------------------------------------------------------------
!BOP
!
! !MODULE: FrictionVelocityMod
!
! !DESCRIPTION:
! Calculation of the friction velocity, relation for potential
! temperature and humidity profiles of surface boundary layer.
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: FrictionVelocity       ! Calculate friction velocity
  public :: MoninObukIni           ! Initialization of the Monin-Obukhov length
!
! !PRIVATE MEMBER FUNCTIONS:
  private :: StabilityFunc1        ! Stability function for rib < 0.
  private :: StabilityFunc2        ! Stability function for rib < 0.
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!------------------------------------------------------------------------------

contains

!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FrictionVelocity
!
! !INTERFACE:
  subroutine FrictionVelocity(lbn, ubn, fn, filtern, &
                              displa, z0m, z0h, z0q, &
                              obu, iter, ur, um, ustar, &
                              temp1, temp2, temp12m, temp22m, fm, landunit_index)
!
! !DESCRIPTION:
! Calculation of the friction velocity, relation for potential
! temperature and humidity profiles of surface boundary layer.
! The scheme is based on the work of Zeng et al. (1998):
! Intercomparison of bulk aerodynamic algorithms for the computation
! of sea surface fluxes using TOGA CORE and TAO data. J. Climate,
! Vol. 11, 2628-2644.
!
! !USES:
   use clmtype
   use clm_varcon, only : vkc
!
! !ARGUMENTS:
   implicit none
   integer , intent(in)  :: lbn, ubn         ! pft/landunit array bounds
   integer , intent(in)  :: fn               ! number of filtered pft/landunit elements
   integer , intent(in)  :: filtern(fn)      ! pft/landunit filter
   real(r8), intent(in)  :: displa(lbn:ubn)  ! displacement height (m)
   real(r8), intent(in)  :: z0m(lbn:ubn)     ! roughness length over vegetation, momentum [m]
   real(r8), intent(in)  :: z0h(lbn:ubn)     ! roughness length over vegetation, sensible heat [m]
   real(r8), intent(in)  :: z0q(lbn:ubn)     ! roughness length over vegetation, latent heat [m]
   real(r8), intent(in)  :: obu(lbn:ubn)     ! monin-obukhov length (m)
   integer,  intent(in)  :: iter             ! iteration number
   real(r8), intent(in)  :: ur(lbn:ubn)      ! wind speed at reference height [m/s]
   real(r8), intent(in)  :: um(lbn:ubn)      ! wind speed including the stablity effect [m/s]
   logical,  optional, intent(in)  :: landunit_index  ! optional argument that defines landunit or pft level
   real(r8), intent(out) :: ustar(lbn:ubn)   ! friction velocity [m/s]
   real(r8), intent(out) :: temp1(lbn:ubn)   ! relation for potential temperature profile
   real(r8), intent(out) :: temp12m(lbn:ubn) ! relation for potential temperature profile applied at 2-m
   real(r8), intent(out) :: temp2(lbn:ubn)   ! relation for specific humidity profile
   real(r8), intent(out) :: temp22m(lbn:ubn) ! relation for specific humidity profile applied at 2-m
   real(r8), intent(inout) :: fm(lbn:ubn)    ! diagnose 10m wind (DUST only)
!
! !CALLED FROM:
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
! 12/19/01, Peter Thornton
! Added arguments to eliminate passing clm derived type into this function.
! Created by Mariana Vertenstein
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
   integer , pointer :: ngridcell(:)      !pft/landunit gridcell index
   real(r8), pointer :: forc_hgt_u_pft(:) !observational height of wind at pft level [m]
   real(r8), pointer :: forc_hgt_t_pft(:) !observational height of temperature at pft level [m]
   real(r8), pointer :: forc_hgt_q_pft(:) !observational height of specific humidity at pft level [m]
   integer , pointer :: pfti(:)           !beginning pfti index for landunit
   integer , pointer :: pftf(:)           !final pft index for landunit
!
! local pointers to implicit out arguments
!
   real(r8), pointer :: u10(:)         ! 10-m wind (m/s) (for dust model)
   real(r8), pointer :: fv(:)          ! friction velocity (m/s) (for dust model)
   real(r8), pointer :: vds(:)         ! dry deposition velocity term (m/s) (for SO4 NH4NO3)
!
!
! !OTHER LOCAL VARIABLES:
!EOP
!
   real(r8), parameter :: zetam = 1.574_r8 ! transition point of flux-gradient relation (wind profile)
   real(r8), parameter :: zetat = 0.465_r8 ! transition point of flux-gradient relation (temp. profile)
   integer :: f                            ! pft/landunit filter index
   integer :: n                            ! pft/landunit index
   integer :: g                            ! gridcell index
   integer :: pp                           ! pfti,pftf index
   real(r8):: zldis(lbn:ubn)               ! reference height "minus" zero displacement heght [m]
   real(r8):: zeta(lbn:ubn)                ! dimensionless height used in Monin-Obukhov theory
#if (defined DUST)
   real(r8) :: tmp1,tmp2,tmp3,tmp4         ! Used to diagnose the 10 meter wind
   real(r8) :: fmnew                       ! Used to diagnose the 10 meter wind
   real(r8) :: fm10                        ! Used to diagnose the 10 meter wind
   real(r8) :: zeta10                      ! Used to diagnose the 10 meter wind
#endif
   real(r8) :: vds_tmp                     ! Temporary for dry deposition velocity
!------------------------------------------------------------------------------

   ! Assign local pointers to derived type members (gridcell-level)

   if (present(landunit_index)) then
     ngridcell  => clm3%g%l%gridcell
   else
     ngridcell  => clm3%g%l%c%p%gridcell
   end if

   vds        => clm3%g%l%c%p%pps%vds
   u10        => clm3%g%l%c%p%pps%u10
   fv         => clm3%g%l%c%p%pps%fv

   ! Assign local pointers to derived type members (pft or landunit-level)

   pfti             => clm3%g%l%pfti
   pftf             => clm3%g%l%pftf

   ! Assign local pointers to derived type members (pft-level)

   forc_hgt_u_pft => clm3%g%l%c%p%pps%forc_hgt_u_pft
   forc_hgt_t_pft => clm3%g%l%c%p%pps%forc_hgt_t_pft
   forc_hgt_q_pft => clm3%g%l%c%p%pps%forc_hgt_q_pft

   ! Adjustment factors for unstable (moz < 0) or stable (moz > 0) conditions.

#if (!defined PERGRO)

!dir$ concurrent
!cdir nodep
   do f = 1, fn
      n = filtern(f)
      g = ngridcell(n)

      ! Wind profile

      if (present(landunit_index)) then
        zldis(n) = forc_hgt_u_pft(pfti(n))-displa(n)
      else
        zldis(n) = forc_hgt_u_pft(n)-displa(n)
      end if
      zeta(n) = zldis(n)/obu(n)
      if (zeta(n) < -zetam) then
         ustar(n) = vkc*um(n)/(log(-zetam*obu(n)/z0m(n))&
              - StabilityFunc1(-zetam) &
              + StabilityFunc1(z0m(n)/obu(n)) &
              + 1.14_r8*((-zeta(n))**0.333_r8-(zetam)**0.333_r8))
      else if (zeta(n) < 0._r8) then
         ustar(n) = vkc*um(n)/(log(zldis(n)/z0m(n))&
              - StabilityFunc1(zeta(n))&
              + StabilityFunc1(z0m(n)/obu(n)))
      else if (zeta(n) <=  1._r8) then
         ustar(n) = vkc*um(n)/(log(zldis(n)/z0m(n)) + 5._r8*zeta(n) -5._r8*z0m(n)/obu(n))
      else
         ustar(n) = vkc*um(n)/(log(obu(n)/z0m(n))+5._r8-5._r8*z0m(n)/obu(n) &
              +(5._r8*log(zeta(n))+zeta(n)-1._r8))
      end if
      
      if (zeta(n) < 0._r8) then
         vds_tmp = 2.e-3_r8*ustar(n) * ( 1._r8 + (300._r8/(-obu(n)))**0.666_r8)
      else
         vds_tmp = 2.e-3_r8*ustar(n)
      endif

      if (present(landunit_index)) then
         do pp = pfti(n),pftf(n)
            vds(pp) = vds_tmp
         end do
      else
         vds(n) = vds_tmp
      end if

      ! Temperature profile

      if (present(landunit_index)) then
        zldis(n) = forc_hgt_t_pft(pfti(n))-displa(n)
      else
        zldis(n) = forc_hgt_t_pft(n)-displa(n)
      end if
      zeta(n) = zldis(n)/obu(n)
      if (zeta(n) < -zetat) then
         temp1(n) = vkc/(log(-zetat*obu(n)/z0h(n))&
              - StabilityFunc2(-zetat) &
              + StabilityFunc2(z0h(n)/obu(n)) &
              + 0.8_r8*((zetat)**(-0.333_r8)-(-zeta(n))**(-0.333_r8)))
      else if (zeta(n) < 0._r8) then
         temp1(n) = vkc/(log(zldis(n)/z0h(n)) &
              - StabilityFunc2(zeta(n)) &
              + StabilityFunc2(z0h(n)/obu(n)))
      else if (zeta(n) <=  1._r8) then
         temp1(n) = vkc/(log(zldis(n)/z0h(n)) + 5._r8*zeta(n) - 5._r8*z0h(n)/obu(n))
      else
         temp1(n) = vkc/(log(obu(n)/z0h(n)) + 5._r8 - 5._r8*z0h(n)/obu(n) &
              + (5._r8*log(zeta(n))+zeta(n)-1._r8))
      end if

      ! Humidity profile

      if (present(landunit_index)) then
       if (forc_hgt_q_pft(pfti(n)) == forc_hgt_t_pft(pfti(n)) .and. z0q(n) == z0h(n)) then
         temp2(n) = temp1(n)
       else
         zldis(n) = forc_hgt_q_pft(pfti(n))-displa(n)
         zeta(n) = zldis(n)/obu(n)
         if (zeta(n) < -zetat) then
            temp2(n) = vkc/(log(-zetat*obu(n)/z0q(n)) &
                 - StabilityFunc2(-zetat) &
                 + StabilityFunc2(z0q(n)/obu(n)) &
                 + 0.8_r8*((zetat)**(-0.333_r8)-(-zeta(n))**(-0.333_r8)))
         else if (zeta(n) < 0._r8) then
            temp2(n) = vkc/(log(zldis(n)/z0q(n)) &
                 - StabilityFunc2(zeta(n)) &
                 + StabilityFunc2(z0q(n)/obu(n)))
         else if (zeta(n) <=  1._r8) then
            temp2(n) = vkc/(log(zldis(n)/z0q(n)) + 5._r8*zeta(n)-5._r8*z0q(n)/obu(n))
         else
            temp2(n) = vkc/(log(obu(n)/z0q(n)) + 5._r8 - 5._r8*z0q(n)/obu(n) &
                 + (5._r8*log(zeta(n))+zeta(n)-1._r8))
         end if
       end if
      else
       if (forc_hgt_q_pft(n) == forc_hgt_t_pft(n) .and. z0q(n) == z0h(n)) then
         temp2(n) = temp1(n)
       else
         zldis(n) = forc_hgt_q_pft(n)-displa(n)
         zeta(n) = zldis(n)/obu(n)
         if (zeta(n) < -zetat) then
            temp2(n) = vkc/(log(-zetat*obu(n)/z0q(n)) &
                 - StabilityFunc2(-zetat) &
                 + StabilityFunc2(z0q(n)/obu(n)) &
                 + 0.8_r8*((zetat)**(-0.333_r8)-(-zeta(n))**(-0.333_r8)))
         else if (zeta(n) < 0._r8) then
            temp2(n) = vkc/(log(zldis(n)/z0q(n)) &
                 - StabilityFunc2(zeta(n)) &
                 + StabilityFunc2(z0q(n)/obu(n)))
         else if (zeta(n) <=  1._r8) then
            temp2(n) = vkc/(log(zldis(n)/z0q(n)) + 5._r8*zeta(n)-5._r8*z0q(n)/obu(n))
         else
            temp2(n) = vkc/(log(obu(n)/z0q(n)) + 5._r8 - 5._r8*z0q(n)/obu(n) &
                 + (5._r8*log(zeta(n))+zeta(n)-1._r8))
         end if
       endif
      endif

      ! Temperature profile applied at 2-m

      zldis(n) = 2.0_r8 + z0h(n)
      zeta(n) = zldis(n)/obu(n)
      if (zeta(n) < -zetat) then
         temp12m(n) = vkc/(log(-zetat*obu(n)/z0h(n))&
              - StabilityFunc2(-zetat) &
              + StabilityFunc2(z0h(n)/obu(n)) &
              + 0.8_r8*((zetat)**(-0.333_r8)-(-zeta(n))**(-0.333_r8)))
      else if (zeta(n) < 0._r8) then
         temp12m(n) = vkc/(log(zldis(n)/z0h(n)) &
              - StabilityFunc2(zeta(n))  &
              + StabilityFunc2(z0h(n)/obu(n)))
      else if (zeta(n) <=  1._r8) then
         temp12m(n) = vkc/(log(zldis(n)/z0h(n)) + 5._r8*zeta(n) - 5._r8*z0h(n)/obu(n))
      else
         temp12m(n) = vkc/(log(obu(n)/z0h(n)) + 5._r8 - 5._r8*z0h(n)/obu(n) &
              + (5._r8*log(zeta(n))+zeta(n)-1._r8))
      end if

      ! Humidity profile applied at 2-m

      if (z0q(n) == z0h(n)) then
         temp22m(n) = temp12m(n)
      else
         zldis(n) = 2.0_r8 + z0q(n)
         zeta(n) = zldis(n)/obu(n)
         if (zeta(n) < -zetat) then
            temp22m(n) = vkc/(log(-zetat*obu(n)/z0q(n)) - &
                 StabilityFunc2(-zetat) + StabilityFunc2(z0q(n)/obu(n)) &
                 + 0.8_r8*((zetat)**(-0.333_r8)-(-zeta(n))**(-0.333_r8)))
         else if (zeta(n) < 0._r8) then
            temp22m(n) = vkc/(log(zldis(n)/z0q(n)) - &
                 StabilityFunc2(zeta(n))+StabilityFunc2(z0q(n)/obu(n)))
         else if (zeta(n) <=  1._r8) then
            temp22m(n) = vkc/(log(zldis(n)/z0q(n)) + 5._r8*zeta(n)-5._r8*z0q(n)/obu(n))
         else
            temp22m(n) = vkc/(log(obu(n)/z0q(n)) + 5._r8 - 5._r8*z0q(n)/obu(n) &
                 + (5._r8*log(zeta(n))+zeta(n)-1._r8))
         end if
      end if

#if (defined DUST)
      ! diagnose 10-m wind for dust model (dstmbl.F)
      ! Notes from C. Zender's dst.F:
      ! According to Bon96 p. 62, the displacement height d (here displa) is
      ! 0.0 <= d <= 0.34 m in dust source regions (i.e., regions w/o trees).
      ! Therefore d <= 0.034*z1 and may safely be neglected.
      ! Code from LSM routine SurfaceTemperature was used to obtain u10

      if (present(landunit_index)) then
        zldis(n) = forc_hgt_u_pft(pfti(n))-displa(n)
      else
        zldis(n) = forc_hgt_u_pft(n)-displa(n)
      end if
      zeta(n) = zldis(n)/obu(n)
      if (min(zeta(n), 1._r8) < 0._r8) then
         tmp1 = (1._r8 - 16._r8*min(zeta(n),1._r8))**0.25_r8
         tmp2 = log((1._r8+tmp1*tmp1)/2._r8)
         tmp3 = log((1._r8+tmp1)/2._r8)
         fmnew = 2._r8*tmp3 + tmp2 - 2._r8*atan(tmp1) + 1.5707963_r8
      else
         fmnew = -5._r8*min(zeta(n),1._r8)
      endif
      if (iter == 1) then
         fm(n) = fmnew
      else
         fm(n) = 0.5_r8 * (fm(n)+fmnew)
      end if
      zeta10 = min(10._r8/obu(n), 1._r8)
      if (zeta(n) == 0._r8) zeta10 = 0._r8
      if (zeta10 < 0._r8) then
         tmp1 = (1.0_r8 - 16.0_r8 * zeta10)**0.25_r8
         tmp2 = log((1.0_r8 + tmp1*tmp1)/2.0_r8)
         tmp3 = log((1.0_r8 + tmp1)/2.0_r8)
         fm10 = 2.0_r8*tmp3 + tmp2 - 2.0_r8*atan(tmp1) + 1.5707963_r8
      else                ! not stable
         fm10 = -5.0_r8 * zeta10
      end if
      if (present(landunit_index)) then
        tmp4 = log( max( 1.0_8, forc_hgt_u_pft(pfti(n)) / 10._r8) )
      else 
        tmp4 = log( max( 1.0_8, forc_hgt_u_pft(n) / 10._r8) )
      end if
      if (present(landunit_index)) then
        do pp = pfti(n),pftf(n)
          u10(pp) = ur(n) - ustar(n)/vkc * (tmp4 - fm(n) + fm10)
          fv(pp)  = ustar(n)
        end do 
      else
        u10(n) = ur(n) - ustar(n)/vkc * (tmp4 - fm(n) + fm10)
        fv(n)  = ustar(n)
      end if
#endif

   end do
#endif


#if (defined PERGRO)

   !===============================================================================
   ! The following only applies when PERGRO is defined
   !===============================================================================

!dir$ concurrent
!cdir nodep
   do f = 1, fn
      n = filtern(f)
      g = ngridcell(n)

      if (present(landunit_index)) then
        zldis(n) = forc_hgt_u_pft(pfti(n))-displa(n)
      else
        zldis(n) = forc_hgt_u_pft(n)-displa(n)
      end if
      zeta(n) = zldis(n)/obu(n)
      if (zeta(n) < -zetam) then           ! zeta < -1
         ustar(n) = vkc * um(n) / log(-zetam*obu(n)/z0m(n))
      else if (zeta(n) < 0._r8) then         ! -1 <= zeta < 0
         ustar(n) = vkc * um(n) / log(zldis(n)/z0m(n))
      else if (zeta(n) <= 1._r8) then        !  0 <= ztea <= 1
         ustar(n)=vkc * um(n)/log(zldis(n)/z0m(n))
      else                             !  1 < zeta, phi=5+zeta
         ustar(n)=vkc * um(n)/log(obu(n)/z0m(n))
      endif

      if (present(landunit_index)) then
        zldis(n) = forc_hgt_t_pft(pfti(n))-displa(n)
      else
        zldis(n) = forc_hgt_t_pft(n)-displa(n)
      end if
      zeta(n) = zldis(n)/obu(n)
      if (zeta(n) < -zetat) then
         temp1(n)=vkc/log(-zetat*obu(n)/z0h(n))
      else if (zeta(n) < 0._r8) then
         temp1(n)=vkc/log(zldis(n)/z0h(n))
      else if (zeta(n) <= 1._r8) then
         temp1(n)=vkc/log(zldis(n)/z0h(n))
      else
         temp1(n)=vkc/log(obu(n)/z0h(n))
      end if

      if (present(landunit_index)) then
        zldis(n) = forc_hgt_q_pft(pfti(n))-displa(n)
      else
        zldis(n) = forc_hgt_q_pft(n)-displa(n)
      end if
      zeta(n) = zldis(n)/obu(n)
      if (zeta(n) < -zetat) then
         temp2(n)=vkc/log(-zetat*obu(n)/z0q(n))
      else if (zeta(n) < 0._r8) then
         temp2(n)=vkc/log(zldis(n)/z0q(n))
      else if (zeta(n) <= 1._r8) then
         temp2(n)=vkc/log(zldis(n)/z0q(n))
      else
         temp2(n)=vkc/log(obu(n)/z0q(n))
      end if

      zldis(n) = 2.0_r8 + z0h(n)
      zeta(n) = zldis(n)/obu(n)
      if (zeta(n) < -zetat) then
         temp12m(n)=vkc/log(-zetat*obu(n)/z0h(n))
      else if (zeta(n) < 0._r8) then
         temp12m(n)=vkc/log(zldis(n)/z0h(n))
      else if (zeta(n) <= 1._r8) then
         temp12m(n)=vkc/log(zldis(n)/z0h(n))
      else
         temp12m(n)=vkc/log(obu(n)/z0h(n))
      end if

      zldis(n) = 2.0_r8 + z0q(n)
      zeta(n) = zldis(n)/obu(n)
      if (zeta(n) < -zetat) then
         temp22m(n)=vkc/log(-zetat*obu(n)/z0q(n))
      else if (zeta(n) < 0._r8) then
         temp22m(n)=vkc/log(zldis(n)/z0q(n))
      else if (zeta(n) <= 1._r8) then
         temp22m(n)=vkc/log(zldis(n)/z0q(n))
      else
         temp22m(n)=vkc/log(obu(n)/z0q(n))
      end if
#if (defined DUST)
      ! diagnose 10-m wind for dust model (dstmbl.F)
      ! Notes from C. Zender's dst.F:
      ! According to Bon96 p. 62, the displacement height d (here displa) is
      ! 0.0 <= d <= 0.34 m in dust source regions (i.e., regions w/o trees).
      ! Therefore d <= 0.034*z1 and may safely be neglected.
      ! Code from LSM routine SurfaceTemperature was used to obtain u10

      if (present(landunit_index)) then
        zldis(n) = forc_hgt_u_pft(pfti(n))-displa(n)
      else
        zldis(n) = forc_hgt_u_pft(n)-displa(n)
      end if 
      zeta(n) = zldis(n)/obu(n)
      if (min(zeta(n), 1._r8) < 0._r8) then
         tmp1 = (1._r8 - 16._r8*min(zeta(n),1._r8))**0.25_r8
         tmp2 = log((1._r8+tmp1*tmp1)/2._r8)
         tmp3 = log((1._r8+tmp1)/2._r8)
         fmnew = 2._r8*tmp3 + tmp2 - 2._r8*atan(tmp1) + 1.5707963_r8
      else
         fmnew = -5._r8*min(zeta(n),1._r8)
      endif
      if (iter == 1) then
         fm(n) = fmnew
      else
         fm(n) = 0.5_r8 * (fm(n)+fmnew)
      end if
      zeta10 = min(10._r8/obu(n), 1._r8)
      if (zeta(n) == 0._r8) zeta10 = 0._r8
      if (zeta10 < 0._r8) then
         tmp1 = (1.0_r8 - 16.0 * zeta10)**0.25_r8
         tmp2 = log((1.0_r8 + tmp1*tmp1)/2.0_r8)
         tmp3 = log((1.0_r8 + tmp1)/2.0_r8)
         fm10 = 2.0_r8*tmp3 + tmp2 - 2.0_r8*atan(tmp1) + 1.5707963_r8
      else                ! not stable
         fm10 = -5.0_r8 * zeta10
      end if
      if (present(landunit_index)) then
        tmp4 = log( max( 1.0_r8, forc_hgt_u_pft(pfti(n)) / 10._r8 ) )
      else
        tmp4 = log( max( 1.0_r8, forc_hgt_u_pft(n) / 10._r8 ) )
      end if
      if (present(landunit_index)) then
        do pp = pfti(n),pftf(n)
          u10(pp) = ur(n) - ustar(n)/vkc * (tmp4 - fm(n) + fm10)
          fv(pp)  = ustar(n)
        end do 
      else
        u10(n) = ur(n) - ustar(n)/vkc * (tmp4 - fm(n) + fm10)
        fv(n)  = ustar(n)
      end if
#endif
   end do

#endif

   end subroutine FrictionVelocity

!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: StabilityFunc
!
! !INTERFACE:
   real(r8) function StabilityFunc1(zeta)
!
! !DESCRIPTION:
! Stability function for rib < 0.
!
! !USES:
      use shr_const_mod, only: SHR_CONST_PI
!
! !ARGUMENTS:
      implicit none
      real(r8), intent(in) :: zeta  ! dimensionless height used in Monin-Obukhov theory
!
! !CALLED FROM:
! subroutine FrictionVelocity in this module
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
!
!
! !LOCAL VARIABLES:
!EOP
      real(r8) :: chik, chik2
!------------------------------------------------------------------------------

      chik2 = sqrt(1._r8-16._r8*zeta)
      chik = sqrt(chik2)
      StabilityFunc1 = 2._r8*log((1._r8+chik)*0.5_r8) &
           + log((1._r8+chik2)*0.5_r8)-2._r8*atan(chik)+SHR_CONST_PI*0.5_r8

    end function StabilityFunc1

!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: StabilityFunc2
!
! !INTERFACE:
   real(r8) function StabilityFunc2(zeta)
!
! !DESCRIPTION:
! Stability function for rib < 0.
!
! !USES:
     use shr_const_mod, only: SHR_CONST_PI
!
! !ARGUMENTS:
     implicit none
     real(r8), intent(in) :: zeta  ! dimensionless height used in Monin-Obukhov theory
!
! !CALLED FROM:
! subroutine FrictionVelocity in this module
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
!
!
! !LOCAL VARIABLES:
!EOP
     real(r8) :: chik2
!------------------------------------------------------------------------------

     chik2 = sqrt(1._r8-16._r8*zeta)
     StabilityFunc2 = 2._r8*log((1._r8+chik2)*0.5_r8)

   end function StabilityFunc2

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: MoninObukIni
!
! !INTERFACE:
  subroutine MoninObukIni (ur, thv, dthv, zldis, z0m, um, obu)
!
! !DESCRIPTION:
! Initialization of the Monin-Obukhov length.
! The scheme is based on the work of Zeng et al. (1998):
! Intercomparison of bulk aerodynamic algorithms for the computation
! of sea surface fluxes using TOGA CORE and TAO data. J. Climate,
! Vol. 11, 2628-2644.
!
! !USES:
    use clm_varcon, only : grav
!
! !ARGUMENTS:
    implicit none
    real(r8), intent(in)  :: ur    ! wind speed at reference height [m/s]
    real(r8), intent(in)  :: thv   ! virtual potential temperature (kelvin)
    real(r8), intent(in)  :: dthv  ! diff of vir. poten. temp. between ref. height and surface
    real(r8), intent(in)  :: zldis ! reference height "minus" zero displacement heght [m]
    real(r8), intent(in)  :: z0m   ! roughness length, momentum [m]
    real(r8), intent(out) :: um    ! wind speed including the stability effect [m/s]
    real(r8), intent(out) :: obu   ! monin-obukhov length (m)
!
! !CALLED FROM:
! subroutine BareGroundFluxes in module BareGroundFluxesMod.F90
! subroutine BiogeophysicsLake in module BiogeophysicsLakeMod.F90
! subroutine CanopyFluxes in module CanopyFluxesMod.F90
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
!
!
! !LOCAL VARIABLES:
!EOP
!
    real(r8) :: wc    ! convective velocity [m/s]
    real(r8) :: rib   ! bulk Richardson number
    real(r8) :: zeta  ! dimensionless height used in Monin-Obukhov theory
    real(r8) :: ustar ! friction velocity [m/s]
!-----------------------------------------------------------------------

    ! Initial values of u* and convective velocity

    ustar=0.06_r8
    wc=0.5_r8
    if (dthv >= 0._r8) then
       um=max(ur,0.1_r8)
    else
       um=sqrt(ur*ur+wc*wc)
    endif

    rib=grav*zldis*dthv/(thv*um*um)
#if (defined PERGRO)
    rib = 0._r8
#endif

    if (rib >= 0._r8) then      ! neutral or stable
       zeta = rib*log(zldis/z0m)/(1._r8-5._r8*min(rib,0.19_r8))
       zeta = min(2._r8,max(zeta,0.01_r8 ))
    else                     ! unstable
       zeta=rib*log(zldis/z0m)
       zeta = max(-100._r8,min(zeta,-0.01_r8 ))
    endif

    obu=zldis/zeta

  end subroutine MoninObukIni

end module FrictionVelocityMod

module VOCEmissionMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: VOCEmissionMod
!
! !DESCRIPTION:
! Volatile organic compound emission
!
! !USES:
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: VOCEmission
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: VOCEmission
!
! !INTERFACE:
  subroutine VOCEmission (lbp, ubp, num_soilp, filter_soilp )
!
! ! NEW DESCRIPTION
! Volatile organic compound emission
! This code simulates volatile organic compound emissions following:
! 1. Isoprene: Guenther et al., 2006 description of MEGAN emissions
!     following equations 2-9, 16-17, 20
! 2. Monoterpenes/OVOCs/ORVOCs/CO: algorithm presented in Guenther, A., 
!    1999: Modeling Biogenic Volatile Organic Compound Emissions to the 
!    Atmosphere. In Reactive Hydrocarbons in the Atmosphere, Ch. 3
!    With updates from MEGAN online user's guide 
!    ( http://acd.ucar.edu/~guenther/MEGAN/MEGANusersguide.pdf)
! This model relies on the assumption that 90% of isoprene and monoterpene
! emissions originate from canopy foliage:
!    E= epsilon * gamma * rho
! VOC flux (E) [ugC m-2 h-1] is calculated from baseline emission
! factors (epsilon) [ugC m-2 h-1] which are mapped for each PFT (isoprene)
! or constant for each PFT (others).  Note that for constant EFs the units
! of [ugC g-1 h-1] must be multiplied by the source density factor.
! The emission activity factor (gamma) [unitless] for isoprene includes 
! dependence on PPFT, temperature, LAI, leaf age and soil moisture.  
! The canopy environment constant was calculated offline for CLM+CAM at 
! standard conditions.
! The emission activity factor for the other emissions depends on temperature.
! We assume that the escape efficiency (rho) here is unity following
! Guenther et al., 2006.
! Subroutine written to operate at the patch level.
! IN FINAL IMPLEMENTATION, REMEMBER:
! 1. may wish to call this routine only as freq. as rad. calculations
! 2. may wish to place epsilon values directly in pft-physiology file
! Output: vocflx(nvoc) !VOC flux [ug C m-2 h-1]
!
!
! !USES:
    use shr_kind_mod , only : r8 => shr_kind_r8
    use clmtype
    use clm_varpar   , only : nvoc, numpft
    use shr_const_mod, only : SHR_CONST_RGAS
    use clm_varcon   , only : denice
    use clm_varpar   , only : nlevsoi
    use pftvarcon    , only : ndllf_evr_tmp_tree,  ndllf_evr_brl_tree,    &
                              ndllf_dcd_brl_tree,  nbrdlf_evr_trp_tree,   &
                              nbrdlf_evr_tmp_tree, nbrdlf_dcd_brl_shrub,  &
                              nbrdlf_dcd_trp_tree, nbrdlf_dcd_tmp_tree,   &
                              nbrdlf_dcd_brl_tree, nbrdlf_evr_shrub,      &
                              nc3_arctic_grass,    nc4_grass,    noveg
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: lbp, ubp                    ! pft bounds
    integer, intent(in) :: num_soilp                   ! number of columns in soil pft filter
    integer, intent(in) :: filter_soilp(num_soilp)     ! pft filter for soil
!
! !CALLED FROM:
!
! !REVISION HISTORY:
! Author: Sam Levis
! 2/1/02, Peter Thornton: migration to new data structure
! 4/15/06, Colette L. Heald: modify for updated MEGAN model (Guenther et al., 2006)
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
    integer , pointer :: pgridcell(:)     ! gridcell index of corresponding pft
    integer , pointer :: pcolumn(:)       ! column index of corresponding pft
    integer , pointer :: ivt(:)           ! pft vegetation type for current
    real(r8), pointer :: t_veg(:)         ! pft vegetation temperature (Kelvin)
    real(r8), pointer :: fsun(:)          ! sunlit fraction of canopy
    real(r8), pointer :: elai(:)          ! one-sided leaf area index with burying by snow
    real(r8), pointer :: clayfrac(:)      ! fraction of soil that is clay
    real(r8), pointer :: sandfrac(:)      ! fraction of soil that is sand
    real(r8), pointer :: forc_solad(:,:)  ! direct beam radiation (visible only)
    real(r8), pointer :: forc_solai(:,:)  ! diffuse radiation     (visible only)
    real(r8), pointer :: sla(:)           ! specific leaf area [m2 leaf g-1 C]
    real(r8), pointer :: h2osoi_vol(:,:)  ! volumetric soil water (m3/m3)
    real(r8), pointer :: h2osoi_ice(:,:)  ! ice soil content (kg/m3)
    real(r8), pointer :: dz(:,:)          ! depth of layer (m)
    real(r8), pointer :: coszen(:)        ! cosine of solar zenith angle
    real(r8), pointer :: efisop(:,:)      ! emission factors for isoprene for each pft [ug C m-2 h-1]
    real(r8), pointer :: elai_p(:)        ! one-sided leaf area index from previous timestep
    real(r8), pointer :: t_veg24(:)       ! avg pft vegetation temperature for last 24 hrs
    real(r8), pointer :: t_veg240(:)      ! avg pft vegetation temperature for last 240 hrs
    real(r8), pointer :: fsun24(:)        ! sunlit fraction of canopy last 24 hrs
    real(r8), pointer :: fsun240(:)       ! sunlit fraction of canopy last 240 hrs
    real(r8), pointer :: forc_solad24(:)  ! direct beam radiation last 24hrs  (visible only)
    real(r8), pointer :: forc_solai24(:)  ! diffuse radiation  last 24hrs     (visible only)
    real(r8), pointer :: forc_solad240(:) ! direct beam radiation last 240hrs (visible only)
    real(r8), pointer :: forc_solai240(:) ! diffuse radiation  last 240hrs    (visible only)
    real(r8), pointer :: bsw(:,:)         ! Clapp and Hornberger "b" (nlevgrnd)
    real(r8), pointer :: watsat(:,:)      ! volumetric soil water at saturation (porosity) (nlevgrnd)
    real(r8), pointer :: sucsat(:,:)      ! minimum soil suction (mm) (nlevgrnd)

    real(r8), parameter :: smpmax = 2.57e5_r8 ! maximum soil matrix potential
!
! local pointers to original implicit out arrays
!
    real(r8), pointer :: vocflx(:,:)      ! VOC flux [ug C m-2 h-1]
    real(r8), pointer :: vocflx_tot(:)    ! VOC flux [ug C m-2 h-1]
    real(r8), pointer :: vocflx_1(:)      ! VOC flux(1) [ug C m-2 h-1]
    real(r8), pointer :: vocflx_2(:)      ! VOC flux(2) [ug C m-2 h-1]
    real(r8), pointer :: vocflx_3(:)      ! VOC flux(3) [ug C m-2 h-1]
    real(r8), pointer :: vocflx_4(:)      ! VOC flux(4) [ug C m-2 h-1]
    real(r8), pointer :: vocflx_5(:)      ! VOC flux(5) [ug C m-2 h-1]
    real(r8), pointer :: Eopt_out(:)     
    real(r8), pointer :: topt_out(:)
    real(r8), pointer :: alpha_out(:)
    real(r8), pointer :: cp_out(:)
    real(r8), pointer :: paru_out(:)
    real(r8), pointer :: par24u_out(:)
    real(r8), pointer :: par240u_out(:)
    real(r8), pointer :: para_out(:)
    real(r8), pointer :: par24a_out(:)
    real(r8), pointer :: par240a_out(:)
    real(r8), pointer :: gamma_out(:)
    real(r8), pointer :: gammaT_out(:)
    real(r8), pointer :: gammaP_out(:)
    real(r8), pointer :: gammaL_out(:)
    real(r8), pointer :: gammaA_out(:)
    real(r8), pointer :: gammaS_out(:)
!
!
! !OTHER LOCAL VARIABLES:
!EOP
!
    integer  :: fp,p,g,c,n,j            ! indices
    integer  :: ct_bad
    real(r8) :: epsilon(lbp:ubp)        ! emission factor [ugC m-2 h-1]
    real(r8) :: par                     ! temporary
    real(r8) :: par24                   ! temporary
    real(r8) :: par240                  ! temporary
    real(r8) :: density                 ! source density factor [g dry wgt foliar mass/m2 ground]
    real(r8) :: gamma(lbp:ubp)          ! activity factor (accounting for light, T, age, LAI conditions)
    real(r8) :: gamma_p                 ! activity factor for PPFD
    real(r8) :: gamma_l                 ! activity factor for PPFD & LAI
    real(r8) :: gamma_t                 ! activity factor for temperature
    real(r8) :: gamma_a                 ! activity factor for leaf age
    real(r8) :: gamma_sm                ! activity factor for soil moisture
    real(r8) :: x                       ! temporary 
    real(r8) :: Eopt                    ! temporary 
    real(r8) :: topt                    ! temporary 
    real(r8) :: cp                      ! temporary
    real(r8) :: alpha                   ! temporary
    real(r8) :: elai_prev               ! lai for previous timestep
    real(r8) :: fnew, fgro, fmat, fsen  ! fractions of leaves at different phenological stages
    real(r8) :: nl                      ! temporary number of soil levels
    real(r8) :: theta_ice               ! water content in ice in m3/m3
    real(r8) :: wilt                    ! wilting point in m3/m3
    real(r8) :: theta1                  ! temporary
!
! Constants
!
    real(r8), parameter :: R   = SHR_CONST_RGAS*0.001_r8 ! univ. gas constant [J K-1 mol-1]
    real(r8), parameter :: scale_mw =0.882_r8            ! conversion factor for isoprene -> carbon
    real(r8), parameter :: alpha_fix = 0.001_r8          ! empirical coefficient
    real(r8), parameter :: cp_fix = 1.21_r8              ! empirical coefficient
    real(r8), parameter :: ct1 = 95.0_r8                 ! empirical coefficient (70 in User's Guide)
    real(r8), parameter :: ct2 = 230.0_r8                ! empirical coefficient  (200 in User's Guide)
    real(r8), parameter :: ct3 = 0.00831_r8              ! empirical coefficient (0.0083 in User's Guide)
    real(r8), parameter :: topt_fix = 317._r8            ! std temperature [K]
    real(r8), parameter :: Eopt_fix = 2.26_r8            ! empirical coefficient
    real(r8), parameter :: tstd = 303.15_r8              ! std temperature [K]
    real(r8), parameter :: bet = 0.09_r8                 ! beta empirical coefficient [K-1]
    real(r8), parameter :: clai1 = 0.49_r8               ! empirical coefficient
    real(r8), parameter :: clai2 = 0.2_r8                ! empirical coefficient
    real(r8), parameter :: clai3 = 5.0_r8                ! empirical coefficient
    real(r8), parameter :: Anew = 0.01_r8                ! relative emission factor for new plants
    real(r8), parameter :: Agro = 0.5_r8                 ! relative emission factor for new plants
    real(r8), parameter :: Amat = 1.0_r8                 ! relative emission factor for new plants
    real(r8), parameter :: Asen = 0.33_r8                ! relative emission factor for new plants
    real(r8), parameter :: cce = 0.40_r8                 ! factor to set emissions to unity @ std
    real(r8), parameter :: cce1 = 0.47_r8                ! same as Cce but for non-accumulated vars
    real(r8), parameter :: ca1 = 0.004_r8                ! empirical coefficent for alpha
    real(r8), parameter :: ca2 = 0.0005_r8               ! empirical coefficent for alpha
    real(r8), parameter :: ca3 = 0.0468_r8               ! empirical coefficent for cp
    real(r8), parameter :: par0_sun = 200._r8            ! std conditions for past 24 hrs [umol/m2/s]
    real(r8), parameter :: par0_shade = 50._r8           ! std conditions for past 24 hrs [umol/m2/s]
    real(r8), parameter :: co1 = 313._r8                 ! empirical coefficient
    real(r8), parameter :: co2 = 0.6_r8                  ! empirical coefficient
    real(r8), parameter :: co3 = 2.034_r8                ! empirical coefficient
    real(r8), parameter :: co4 = 0.05_r8                 ! empirical coefficient
    real(r8), parameter :: tstd0 = 297_r8                ! std temperature [K]
    real(r8), parameter :: deltheta1=0.06_r8             ! empirical coefficient
!
! These are the values from version of genesis-ibis / 1000.
! CN calculates its own sla [m2 leaf g-1 C]
! Divide by 2 in the equation to get dry weight foliar mass from grams carbon
!
    real(r8) :: hardwire_sla(0:numpft)
    real(r8) :: slarea(lbp:ubp)           ! Specific leaf areas [m2 leaf g-1 C]
    real(r8) :: hardwire_droot(0:numpft)  ! Root depth [m]
!-----------------------------------------------------------------------

    ! Assign local pointers to derived type members (gridcell-level)
    forc_solad => clm_a2l%forc_solad
    forc_solai => clm_a2l%forc_solai
    efisop     => clm3%g%gve%efisop

    ! Assign local pointers to derived subtypes components (column-level)
    h2osoi_vol       => clm3%g%l%c%cws%h2osoi_vol
    h2osoi_ice       => clm3%g%l%c%cws%h2osoi_ice
    dz               => clm3%g%l%c%cps%dz
    bsw              => clm3%g%l%c%cps%bsw
    watsat           => clm3%g%l%c%cps%watsat
    sucsat           => clm3%g%l%c%cps%sucsat

    ! Assign local pointers to derived subtypes components (pft-level)
    pgridcell        => clm3%g%l%c%p%gridcell
    pcolumn          => clm3%g%l%c%p%column
    ivt              => clm3%g%l%c%p%itype
    t_veg            => clm3%g%l%c%p%pes%t_veg
    fsun             => clm3%g%l%c%p%pps%fsun
    elai             => clm3%g%l%c%p%pps%elai
    clayfrac         => clm3%g%l%c%p%pps%clayfrac
    sandfrac         => clm3%g%l%c%p%pps%sandfrac
    vocflx           => clm3%g%l%c%p%pvf%vocflx
    vocflx_tot       => clm3%g%l%c%p%pvf%vocflx_tot
    vocflx_1         => clm3%g%l%c%p%pvf%vocflx_1
    vocflx_2         => clm3%g%l%c%p%pvf%vocflx_2
    vocflx_3         => clm3%g%l%c%p%pvf%vocflx_3
    vocflx_4         => clm3%g%l%c%p%pvf%vocflx_4
    vocflx_5         => clm3%g%l%c%p%pvf%vocflx_5
    Eopt_out         => clm3%g%l%c%p%pvf%Eopt_out
    topt_out         => clm3%g%l%c%p%pvf%topt_out
    alpha_out        => clm3%g%l%c%p%pvf%alpha_out
    cp_out           => clm3%g%l%c%p%pvf%cp_out
    paru_out         => clm3%g%l%c%p%pvf%paru_out
    par24u_out       => clm3%g%l%c%p%pvf%par24u_out
    par240u_out      => clm3%g%l%c%p%pvf%par240u_out
    para_out         => clm3%g%l%c%p%pvf%para_out
    par24a_out       => clm3%g%l%c%p%pvf%par24a_out
    par240a_out      => clm3%g%l%c%p%pvf%par240a_out
    gammaL_out       => clm3%g%l%c%p%pvf%gammaL_out
    gammaT_out       => clm3%g%l%c%p%pvf%gammaT_out
    gammaP_out       => clm3%g%l%c%p%pvf%gammaP_out
    gammaA_out       => clm3%g%l%c%p%pvf%gammaA_out
    gammaS_out       => clm3%g%l%c%p%pvf%gammaS_out
    gamma_out        => clm3%g%l%c%p%pvf%gamma_out
    sla              => clm3%g%l%c%p%pps%slasha

    t_veg24          => clm3%g%l%c%p%pvs%t_veg24
    t_veg240         => clm3%g%l%c%p%pvs%t_veg240
    forc_solad24     => clm3%g%l%c%p%pvs%fsd24
    forc_solad240    => clm3%g%l%c%p%pvs%fsd240
    forc_solai24     => clm3%g%l%c%p%pvs%fsi24
    forc_solai240    => clm3%g%l%c%p%pvs%fsi240
    fsun24           => clm3%g%l%c%p%pvs%fsun24
    fsun240          => clm3%g%l%c%p%pvs%fsun240
    elai_p           => clm3%g%l%c%p%pvs%elai_p

    hardwire_sla(noveg)                                    = 0._r8     ! bare-soil

    hardwire_sla(ndllf_evr_tmp_tree)                       = 0.0125_r8 !needleleaf
    hardwire_sla(ndllf_evr_brl_tree)                       = 0.0125_r8 !Gordon Bonan suggests NET = 0.0076
    hardwire_sla(ndllf_dcd_brl_tree)                       = 0.0125_r8 !Gordon Bonan suggests NDT = 0.0200

    hardwire_sla(nbrdlf_evr_trp_tree)                      = 0.0250_r8 !broadleaf
    hardwire_sla(nbrdlf_evr_tmp_tree)                      = 0.0250_r8 !Gordon Bonan suggests BET = 0.0178
    hardwire_sla(nbrdlf_dcd_trp_tree)                      = 0.0250_r8 !Gordon Bonan suggests BDT = 0.0274
    hardwire_sla(nbrdlf_dcd_tmp_tree:nbrdlf_dcd_brl_shrub) = 0.0250_r8 

    hardwire_sla(nc3_arctic_grass:numpft)                  = 0.0200_r8 !grass/crop

! root depth (m) (defined based on Zeng et al., 2001, cf Guenther 2006)

    hardwire_droot(noveg)                                     = 0._r8   ! bare-soil
    hardwire_droot(ndllf_evr_tmp_tree:ndllf_evr_brl_tree)     = 1.8_r8  ! evergreen tree
    hardwire_droot(ndllf_dcd_brl_tree)                        = 2.0_r8  ! needleleaf deciduous boreal tree
    hardwire_droot(nbrdlf_evr_trp_tree:nbrdlf_evr_tmp_tree)   = 3.0_r8  ! broadleaf evergreen tree
    hardwire_droot(nbrdlf_dcd_trp_tree:nbrdlf_dcd_brl_tree)   = 2.0_r8  ! broadleaf deciduous tree
    hardwire_droot(nbrdlf_evr_shrub:nbrdlf_dcd_brl_shrub)     = 2.5_r8  ! shrub
    hardwire_droot(nc3_arctic_grass:numpft)                   = 1.5_r8  ! grass/crop

! initialize variables which get passed to the atmosphere
    vocflx(lbp:ubp, :)=0._r8

    ! Determine specific leaf array
    do fp = 1,num_soilp
       p = filter_soilp(fp)

       slarea(p) = hardwire_sla(ivt(p))

    end do


    ! Begin loop through voc species
    !_______________________________________________________________________________

    do n = 1, nvoc
       select case (n)

       case(1)	

          do fp = 1,num_soilp
             p = filter_soilp(fp)
             g = pgridcell(p)


             ! epsilon: use gridded values for 6 PFTs specified by MEGAN following
             ! -------  Guenther et al. (2006).  Map the numpft CLM PFTs to these 6.
             !          Units: [ug C m-2 h-1] (convert input files from units of 
             !                 [ug isop m-2 h-1])
    	     epsilon(p) = 0._r8

             ! isoprenes:
             if (     ivt(p) == ndllf_evr_tmp_tree  &
             .or.     ivt(p) == ndllf_evr_brl_tree) then     !fineleaf evergreen
                	epsilon(p) = efisop(2,g)*scale_mw
             else if (ivt(p) == ndllf_dcd_brl_tree) then     !fineleaf deciduous
                	epsilon(p) = efisop(3,g)*scale_mw
             else if (ivt(p) >= nbrdlf_evr_trp_tree &
             .and.    ivt(p) <= nbrdlf_dcd_brl_tree) then    !broadleaf trees
                	epsilon(p) = efisop(1,g)*scale_mw
             else if (ivt(p) >= nbrdlf_evr_shrub &
             .and.    ivt(p) <= nbrdlf_dcd_brl_shrub) then   !shrubs
                	epsilon(p) = efisop(4,g)*scale_mw
             else if (ivt(p) >= nc3_arctic_grass &
             .and.    ivt(p) <= nc4_grass) then              !grass
                	epsilon(p) = efisop(5,g)*scale_mw
             else if (ivt(p) >  nc4_grass) then              !crops
                	epsilon(p) =efisop(6,g)*scale_mw
             end if

          end do

       case(2)

          do fp = 1,num_soilp
             p = filter_soilp(fp)
             g = pgridcell(p)

             ! epsilon: use values from table 3 in Guenther (1997) which originate in
             ! -------  Guenther et al. (1995). In the comments below, I mention the pft
             !          category as described in table 3. Some values were taken directly
             !          from Guenther et al. (1995). Units: [ugC g-1 h-1]
             !          Values were updated on 1/2002 (Guenther, personal communication)

             ! monoterpenes:
             epsilon(p) = 0._r8
             ! monoterpenes:
             if (     ivt(p) >= ndllf_evr_tmp_tree &
             .and.    ivt(p) <= ndllf_evr_brl_tree) then     !needleleaf evergreen
                epsilon(p) = 2.0_r8
             else if (ivt(p) == ndllf_dcd_brl_tree) then     !needleleaf deciduous
                epsilon(p) = 1.6_r8
             else if (ivt(p) >= nbrdlf_evr_trp_tree  &
             .and.    ivt(p) <= nbrdlf_dcd_brl_tree) then    !broadleaf everg trop
                epsilon(p) = 0.4_r8
             else if (ivt(p) >= nbrdlf_evr_shrub &
             .and.    ivt(p) <= nbrdlf_dcd_brl_shrub) then   !other woody veg
                epsilon(p) = 0.8_r8
             else if (ivt(p) >= nc3_arctic_grass &
             .and.    ivt(p) <= numpft) then                 !grass & crop
                epsilon(p) = 0.1_r8
             end if
          end do

       case (3)
          do fp = 1,num_soilp
             p = filter_soilp(fp)
             g = pgridcell(p)

             ! other VOCs (OVOCs)
             epsilon(p) = 1.0_r8                 !Guenther (personal communication)
          end do

       case (4)
          do fp = 1,num_soilp
             p = filter_soilp(fp)
             g = pgridcell(p)

             ! other reactive VOCs (ORVOCs)
             epsilon(p) = 1.0_r8                 !Guenther (personal communication)
          end do

       case (5)
          do fp = 1,num_soilp
             p = filter_soilp(fp)
             g = pgridcell(p)

             ! CO
             epsilon(p) = 0.3_r8                 !Guenther (personal communication)
          end do


       case default

          write(6,*)'only nvocs up to index 5 are currently supported'
          call endrun()

       end select
       
       
       ct_bad=0

       select case (n)

       case (1)

          do fp = 1,num_soilp
             p = filter_soilp(fp)
             g = pgridcell(p)
             c = pcolumn(p)


             ! gamma: Activity factor. Units [dimensionless]
             ! =====  For isoprene include activity factors for LAI,PPFD, T, leaf age, and soil moisture

             ! Activity factor for LAI (Guenther et al., 2006)
             !------------------------
             ! Guenther et al., 2006 eq 3
             if ( (fsun240(p) > 0.0_r8) .and. (fsun240(p) < 1.e30_r8) ) then 
                 gamma_l = cce * elai(p)
             else
                 gamma_l = cce1 * elai(p)
             end if
	     gammaL_out(p)=gamma_l

             ! Activity factor for PPFD (Guenther et al., 2006)
             !-------------------------
	     ! With distinction between sunlit and shaded leafs, weight scalings by
             ! fsun and fshade 
             ! Scale total incident par by fraction of sunlit leaves (added on 1/2002)
             ! multiply w/m2 by 4.6 to get umol/m2/s for par (added 8/14/02)

             ! fvitt -- forc_solad240, forc_solai240 can be zero when CLM finidat is specified
             !          which will cause par240 to be zero and produce NaNs via log(par240)
             ! dml   -- fsun240 can be equal to or greater than one before 10 day averages are
             !           set on startup or if a new pft comes online during land cover change.
             !           Avoid this problem by only doing calculations with fsun240 when fsun240 is
             !           between 0 and 1
             if ( (fsun240(p) > 0._r8) .and. (fsun240(p) < 1._r8) .and.  (forc_solad240(p) > 0._r8) &
             .and. (forc_solai240(p) > 0._r8)) then
                ! With alpha and cp calculated based on eq 6 and 7:
                ! Note indexing for accumulated variables is all at pft level
                ! SUN:
                par = (forc_solad(g,1) + fsun(p) * forc_solai(g,1)) * 4.6_r8
                par24 = (forc_solad24(p) + fsun24(p) * forc_solai24(p)) * 4.6_r8
                par240 = (forc_solad240(p) + fsun240(p) * forc_solai240(p)) * 4.6_r8
                alpha = ca1 - ca2 * log(par240)
                cp = ca3 * exp(ca2 * (par24-par0_sun))*par240**(0.6_r8)
                gamma_p = fsun(p) * ( cp * alpha*par * (1._r8 + alpha*alpha*par*par)**(-0.5_r8) )
	        paru_out(p)=par
		par24u_out(p)=par24
                par240u_out(p)=par240
                ! SHADE:
                par = ((1._r8 - fsun(p)) * forc_solai(g,1)) * 4.6_r8
                par24 = ((1._r8 - fsun24(p)) * forc_solai24(p)) * 4.6_r8
                par240 = ((1._r8 - fsun240(p)) * forc_solai240(p)) * 4.6_r8
                alpha = ca1 - ca2 * log(par240)
                cp = ca3 * exp(ca2 * (par24-par0_shade))*par240**(0.6_r8)
                par = ((1._r8 - fsun(p)) * forc_solai(g,1)) * 4.6_r8
                gamma_p = gamma_p + (1-fsun(p)) * (cp*alpha*par*(1._r8 + alpha*alpha*par*par)**(-0.5_r8))
                para_out(p)=par
		par24a_out(p)=par24
 		par240a_out(p)=par240
             else
                ! With fixed alpha and cp (from MEGAN User's Guide):
                ! SUN: direct + diffuse  
                par = (forc_solad(g,1) + fsun(p) * forc_solai(g,1)) * 4.6_r8
                alpha = alpha_fix
                cp = cp_fix
                gamma_p = fsun(p) * ( cp * alpha*par * (1._r8 + alpha*alpha*par*par)**(-0.5_r8) )
		paru_out(p)=par
	        par24u_out(p)=-999
	        par240u_out(p)=-999
                ! SHADE: diffuse 
                par = ((1._r8 - fsun(p)) * forc_solai(g,1)) * 4.6_r8
                gamma_p = gamma_p + (1-fsun(p)) * (cp*alpha*par*(1._r8 + alpha*alpha*par*par)**(-0.5_r8))
		para_out(p)=par
                par24a_out(p)=-999
                par240a_out(p)=-999
             end if 
             alpha_out(p)=alpha
             cp_out(p)=cp
             gammaP_out(p)=gamma_p


             ! Activity factor for temperature (Guenther et al., 2006)
             !--------------------------------
             if ( (t_veg240(p) > 0.0_r8) .and. (t_veg240(p) < 1.e30_r8) ) then 
                ! topt and Eopt from eq 8 and 9:
                topt = co1 + (co2 * (t_veg240(p)-tstd0))
                Eopt = co3 * exp (co4 * (t_veg24(p)-tstd0)) * exp(co4 * (t_veg240(p) -tstd0))
	     else
                topt = topt_fix
                Eopt = Eopt_fix
             endif 
             x = ( (1._r8/topt) - (1._r8/(t_veg(p))) ) / ct3
             gamma_t = Eopt * ( ct2 * exp(ct1 * x)/(ct2 - ct1 * (1._r8 - exp(ct2 * x))) )
             topt_out(p)=topt
             Eopt_out(p)=Eopt
             gammaT_out(p)=gamma_t


             ! Activity factor for leaf age (Guenther et al., 2006)
             !-----------------------------
             ! If not CNDV elai is constant therefore gamma_a=1.0
             ! gamma_a set to unity for evergreens (PFTs 1, 2, 4, 5)
             ! Note that we assume here that the time step is shorter than the number of 
             !days after budbreak required to induce isoprene emissions (ti=12 days) and 
             ! the number of days after budbreak to reach peak emission (tm=28 days)
	     if ( (ivt(p) == ndllf_dcd_brl_tree) .or. (ivt(p) >= nbrdlf_dcd_trp_tree) ) then  ! non-evergreen

                if ( (elai_p(p) > 0.0_r8) .and. (elai_p(p) < 1.e30_r8) )then 
                   elai_prev = 2._r8*elai_p(p)-elai(p)  ! have accumulated average lai over last timestep
                   if (elai_prev == elai(p)) then
                      fnew = 0.0_r8
                      fgro = 0.0_r8
                      fmat = 1.0_r8
                      fsen = 0.0_r8
                   else if (elai_prev > elai(p)) then
                      fnew = 0.0_r8
                      fgro = 0.0_r8
                      fmat = 1.0_r8 - (elai_prev - elai(p))/elai_prev
                      fsen = (elai_prev - elai(p))/elai_prev
                   else if (elai_prev < elai(p)) then
                      fnew = 1 - (elai_prev / elai(p))
                      fgro = 0.0_r8
                      fmat = (elai_prev / elai(p))
                      fsen = 0.0_r8
                   end if             
                
                   gamma_a = fnew * Anew + fgro * Agro + fmat * Amat + fsen * Asen
	        else
                   gamma_a = 1.0_r8
                end if

             else
                gamma_a = 1.0_r8
             end if
             gammaA_out(p)=gamma_a


             ! Activity factor for soil moisture (Guenther et al., 2006) 
             !----------------------------------
             ! Calculate the mean scaling factor throughout the root depth.
             ! wilting point potential is in units of matric potential (mm) 
             ! (1 J/Kg = 0.001 MPa, approx = 0.1 m)
             ! convert to volumetric soil water using equation 7.118 of the CLM4 Technical Note
             if ((clayfrac(p) > 0) .and. (sandfrac(p) > 0)) then 
               gamma_sm = 0._r8
	       nl=0._r8

               do j = 1,nlevsoi
	         if  (sum(dz(c,1:j)) < hardwire_droot(ivt(p)))  then
                   theta_ice = h2osoi_ice(c,j)/(dz(c,j)*denice)
                   wilt = ((smpmax/sucsat(c,j))**(-1._r8/bsw(c,j))) * (watsat(c,j) - theta_ice)
                   theta1 = wilt + deltheta1
                   if (h2osoi_vol(c,j) >= theta1) then 
             	      gamma_sm = gamma_sm + 1._r8
                   else if ( (h2osoi_vol(c,j) > wilt) .and. (h2osoi_vol(c,j) < theta1) ) then
		      gamma_sm = gamma_sm + ( h2osoi_vol(c,j) - wilt ) / deltheta1
                   else
		      gamma_sm = gamma_sm + 0._r8
                   end if
		   nl=nl+1._r8
                 end if
 	       end do 

	       if (nl > 0) then
	         gamma_sm = gamma_sm/nl
	       endif
             else
	       gamma_sm = 1.0_r8
             end if
             gammaS_out(p)=gamma_sm


             ! Calculate total scaling factor
             !--------------------------------
	     gamma(p) = gamma_l * gamma_p * gamma_t * gamma_a * gamma_sm
             if ( (gamma(p) >=0.0_r8) .and. (gamma(p)< 100._r8) ) then
                gamma_out(p)=gamma(p)
	     else
                gamma_out(p)=gamma(p)
                write(6,*) 'clh GAMMA: ',gamma(p),gamma_l,gamma_p,gamma_t,gamma_a,gamma_sm
             end if

          end do

       case (2,3,4,5)

          do fp = 1,num_soilp
             p = filter_soilp(fp)
             g = pgridcell(p)

             ! gamma: Activity factor. Units [dimensionless]
             ! -----  For monoterpenes, OVOCs, ORVOCs, CO include simple activity factors 
             !        for LAI and T only (Guenther et al., 1995)
             gamma_t = exp(bet * (t_veg(p) - tstd))
	     gamma(p)=gamma_t

          end do

       end select

       do fp = 1,num_soilp
          p = filter_soilp(fp)
          g = pgridcell(p)

          ! density: Source density factor [g dry weight foliar mass m-2 ground]
          ! -------  Other than isoprene, need to convert EF units from 
          ! [ug g-1 h-1] to [ug m-2 h-1]
          if (ivt(p) > noveg) then
             density = elai(p) / (slarea(p) * 0.5_r8)
          else
             density = 0._r8
          end if

          ! calculate the voc flux
          ! ----------------------
	  select case (n)

          case(1)
              vocflx(p,n) = epsilon(p) * gamma(p) 

          case(2,3,4,5)
              vocflx(p,n) = epsilon(p) * gamma(p) * density

          end select


       end do   ! end pft loop

    end do   ! end voc species loop
    !_______________________________________________________________________________

    ! Calculate total voc flux and individual components for history output

    do fp = 1,num_soilp
       p = filter_soilp(fp)
       vocflx_tot(p) = 0._r8
    end do
    do n = 1, nvoc
       do fp = 1,num_soilp
          p = filter_soilp(fp)
          vocflx_tot(p) = vocflx_tot(p) + vocflx(p,n)
       end do
    end do
    do fp = 1,num_soilp
       p = filter_soilp(fp)
       g = pgridcell(p)
       vocflx_1(p) = vocflx(p,1)
       vocflx_2(p) = vocflx(p,2)
       vocflx_3(p) = vocflx(p,3)
       vocflx_4(p) = vocflx(p,4)
       vocflx_5(p) = vocflx(p,5)
    end do

  end subroutine VOCEmission

end module VOCEmissionMod
module dynlandMod

!---------------------------------------------------------------------------
!BOP
!
! !MODULE: dynlandMod
!
! !USES:
   use clmtype
   use decompMod   , only : get_proc_bounds
   use shr_kind_mod, only : r8 => shr_kind_r8
!
! !DESCRIPTION:
! Compute heat and water content to track conservation wrt dynamic land use
!
! !PUBLIC TYPES:
   implicit none
   private
   save
   public :: dynland_hwcontent
!
! !REVISION HISTORY:
!    2009-feb-20 B. Kauffman, created by
!
!EOP
!
! ! PRIVATE TYPES

!===============================================================================

contains
  
!===============================================================================
!BOP
!
! !ROUTINE: dynland_hwcontent
!
! !INTERFACE:

   subroutine dynland_hwcontent(begg,endg,gcell_liq,gcell_ice,gcell_heat)
 
! !DESCRIPTION:
!    Compute grid-level heat and water content
!
! !REVISION HISTORY:
!    2009-feb-20 B. Kauffman, created by
!
! !USES:

   use clm_varcon, only : istsoil,istice,istwet, istdlak,istslak,isturb
#ifdef CROP
   use clm_varcon, only : istcrop
#endif
   use clm_varcon, only : icol_road_perv,icol_road_imperv,icol_roof
   use clm_varcon, only : icol_sunwall,icol_shadewall
   use clm_varcon, only : cpice,  cpliq
   use clm_varpar, only : nlevsno, nlevgrnd

   implicit none

! !ARGUMENTS:

   integer , intent(in)  :: begg, endg              ! proc beg & end gridcell indices
   real(r8), intent(out) :: gcell_liq(begg:endg)
   real(r8), intent(out) :: gcell_ice  (begg:endg)
   real(r8), intent(out) :: gcell_heat (begg:endg)
 
! !LOCAL VARIABLES:
!EOP

   integer  :: li,lf         ! loop initial/final indicies
   integer  :: ci,cf         ! loop initial/final indicies
   integer  :: pi,pf         ! loop initial/final indicies

   integer  :: g,l,c,p,k     ! loop indicies (grid,lunit,column,pft,vertical level)

   real(r8) :: wtgcell       ! weight relative to grid cell
   real(r8) :: wtcol         ! weight relative to column
   real(r8) :: liq           ! sum of liquid water at column level
   real(r8) :: ice           ! sum of frozen water at column level
   real(r8) :: heat          ! sum of heat content at column level
   real(r8) :: cv            ! heat capacity [J/(m^2 K)]

   integer ,pointer :: ltype(:)          ! landunit type index
   integer ,pointer :: ctype(:)          ! column   type index
   integer ,pointer :: ptype(:)          ! pft      type index

   integer,  pointer :: nlev_improad(:)  ! number of impervious road layers
   real(r8), pointer :: cv_wall(:,:)     ! thermal conductivity of urban wall
   real(r8), pointer :: cv_roof(:,:)     ! thermal conductivity of urban roof
   real(r8), pointer :: cv_improad(:,:)  ! thermal conductivity of urban impervious road

   integer , pointer :: snl(:)           ! number of snow layers
   real(r8), pointer :: t_soisno(:,:)    ! soil temperature (Kelvin)
   real(r8), pointer :: h2osno(:)        ! snow water (mm H2O)
   real(r8), pointer :: h2osoi_liq(:,:)  ! liquid water (kg/m2)
   real(r8), pointer :: h2osoi_ice(:,:)  ! frozen water (kg/m2)
   real(r8), pointer :: watsat(:,:)      ! volumetric soil water at saturation (porosity)
   real(r8), pointer :: csol(:,:)        ! heat capacity, soil solids (J/m**3/Kelvin)
   real(r8), pointer :: dz(:,:)          ! layer depth (m)
   real(r8), pointer :: wa(:,:)          ! h2o in underground aquifer

   type(gridcell_type), pointer :: gptr  ! pointer to gridcell derived subtype
   type(landunit_type), pointer :: lptr  ! pointer to landunit derived subtype
   type(column_type)  , pointer :: cptr  ! pointer to column derived subtype
   type(pft_type)     , pointer :: pptr  ! pointer to pft derived subtype

!-------------------------------------------------------------------------------
! Note: this routine does not compute heat or water content of lakes.
!
!-------------------------------------------------------------------------------

   ! Set pointers into derived type

   gptr => clm3%g
   lptr => clm3%g%l
   cptr => clm3%g%l%c
   pptr => clm3%g%l%c%p

   ltype => clm3%g%l%itype
   ctype => clm3%g%l%c%itype
   ptype => clm3%g%l%c%p%itype

   nlev_improad => clm3%g%l%lps%nlev_improad
   cv_wall      => clm3%g%l%lps%cv_wall
   cv_roof      => clm3%g%l%lps%cv_roof
   cv_improad   => clm3%g%l%lps%cv_improad

   snl          => clm3%g%l%c%cps%snl
   watsat       => clm3%g%l%c%cps%watsat
   csol         => clm3%g%l%c%cps%csol
   dz           => clm3%g%l%c%cps%dz
   t_soisno     => clm3%g%l%c%ces%t_soisno
   h2osoi_liq   => clm3%g%l%c%cws%h2osoi_liq
   h2osoi_ice   => clm3%g%l%c%cws%h2osoi_ice
   h2osno       => clm3%g%l%c%cws%h2osno

   ! Get relevant sizes

   do g = begg,endg ! loop over grid cells

      gcell_liq  (g) = 0.0_r8   ! sum for one grid cell
      gcell_ice  (g) = 0.0_r8   ! sum for one grid cell
      gcell_heat (g) = 0.0_r8   ! sum for one grid cell

      li = gptr%luni(g)
      lf = gptr%lunf(g)
      do l = li,lf   ! loop over land units  

         ci = lptr%coli(l)
         cf = lptr%colf(l)
         do c = ci,cf   ! loop over columns

            liq   = 0.0_r8 ! sum for one column
            ice   = 0.0_r8
            heat  = 0.0_r8

            !--- water & ice, above ground only ---
#ifndef CROP
            if ( (ltype(l) == istsoil                                  )  &
#else
            if ( (ltype(l) == istsoil .or. ltype(l) == istcrop         )  &
#endif
            .or. (ltype(l) == istwet                                   )  &
            .or. (ltype(l) == istice                                   )  &
            .or. (ltype(l) == isturb .and. ctype(c) == icol_roof       )  &
            .or. (ltype(l) == isturb .and. ctype(c) == icol_road_imperv)  &
            .or. (ltype(l) == isturb .and. ctype(c) == icol_road_perv  )) then

               if ( snl(c) < 0 ) then
                  do k = snl(c)+1,0 ! loop over snow layers
                     liq   = liq   + clm3%g%l%c%cws%h2osoi_liq(c,k)
                     ice   = ice   + clm3%g%l%c%cws%h2osoi_ice(c,k)
                  end do
               else                 ! no snow layers exist
                  ice = ice + cptr%cws%h2osno(c)
               end if
            end if

            !--- water & ice, below ground only ---
#ifndef CROP
            if ( (ltype(l) == istsoil                                  )  &
#else
            if ( (ltype(l) == istsoil .or. ltype(l) == istcrop         )  &
#endif
            .or. (ltype(l) == istwet                                   )  &
            .or. (ltype(l) == istice                                   )  &
            .or. (ltype(l) == isturb .and. ctype(c) == icol_road_perv  )) then
               do k = 1,nlevgrnd
                  liq   = liq   + cptr%cws%h2osoi_liq(c,k)
                  ice   = ice   + cptr%cws%h2osoi_ice(c,k)
               end do
            end if

            !--- water in aquifer ---
#ifndef CROP
            if ( (ltype(l) == istsoil                                  )  &
#else
            if ( (ltype(l) == istsoil .or. ltype(l) == istcrop         )  &
#endif
            .or. (ltype(l) == istwet                                   )  &
            .or. (ltype(l) == istice                                   )  &
            .or. (ltype(l) == isturb .and. ctype(c) == icol_road_perv  )) then
               liq = liq + cptr%cws%wa(c)
            end if

            !--- water in canopy (at pft level) ---
#ifndef CROP
            if (ltype(l) == istsoil                         ) then
#else
            if (ltype(l) == istsoil .or. ltype(l) == istcrop) then   ! note: soil specified at LU level
#endif
               pi = cptr%pfti(c)
               pf = cptr%pftf(c)
               do p = pi,pf ! loop over pfts
                  wtcol = pptr%wtcol(p)
                  liq = liq + pptr%pws%h2ocan(p) * wtcol
               end do
            end if

            if ( (ltype(l) /= istslak) .and. ltype(l) /= istdlak) then

               !--- heat content, below ground only ---
               do k = 1,nlevgrnd
                  if (ctype(c)==icol_sunwall .OR. ctype(c)==icol_shadewall) then
                      cv = cv_wall(l,k) * dz(c,k)
                   else if (ctype(c) == icol_roof) then
                      cv = cv_roof(l,k) * dz(c,k)
                   else if (ctype(c) == icol_road_imperv .and. k >= 1 .and. k <= nlev_improad(l)) then
                      cv = cv_improad(l,k) * dz(c,k)
                   else if (ltype(l) /= istwet .AND. ltype(l) /= istice) then
                      cv = csol(c,k)*(1-watsat(c,k))*dz(c,k) + (h2osoi_ice(c,k)*cpice + h2osoi_liq(c,k)*cpliq)
                   else
                      cv = (h2osoi_ice(c,k)*cpice + h2osoi_liq(c,k)*cpliq)
                   endif
                   heat = heat + cv*t_soisno(c,k) / 1.e6_r8 
                end do

               !--- heat content, above ground only ---
               if ( snl(c) < 0 ) then
                  do k = snl(c)+1,0 ! loop over snow layers
                     cv = cpliq*h2osoi_liq(c,k) + cpice*h2osoi_ice(c,k)
                     heat = heat + cv*t_soisno(c,k) / 1.e6_r8
                  end do
               else if ( h2osno(c) > 0.0_r8) then
                  k = 1
                  cv = cpice*h2osno(c)
                  heat = heat + cv*t_soisno(c,k) / 1.e6_r8
               end if

            end if

            !--- scale x/m^2 column-level values into x/m^2 gridcell-level values ---
            wtgcell = cptr%wtgcell(c)
            gcell_liq  (g) = gcell_liq  (g) + liq   * wtgcell
            gcell_ice  (g) = gcell_ice  (g) + ice   * wtgcell
            gcell_heat (g) = gcell_heat (g) + heat  * wtgcell

         end do ! column loop      
      end do ! landunit loop
   end do ! grid cell loop

   end subroutine dynland_hwcontent

!===============================================================================

end module dynlandMod

module subgridAveMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: subgridAveMod
!
! !DESCRIPTION:
! Utilities to perfrom subgrid averaging
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
  use clmtype , only : clm3
  use clm_varcon, only : spval, isturb,  icol_roof, icol_sunwall, icol_shadewall, &
                         icol_road_perv, icol_road_imperv

! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: p2c   ! Perfrom an average from pfts to columns
  public :: p2l   ! Perfrom an average from pfts to landunits
  public :: p2g   ! Perfrom an average from pfts to gridcells
  public :: c2l   ! Perfrom an average from columns to landunits
  public :: c2g   ! Perfrom an average from columns to gridcells
  public :: l2g   ! Perfrom an average from landunits to gridcells

  interface p2c
     module procedure p2c_1d
     module procedure p2c_2d
     module procedure p2c_1d_filter
     module procedure p2c_2d_filter
  end interface
  interface p2l
     module procedure p2l_1d
     module procedure p2l_2d
  end interface
  interface p2g
     module procedure p2g_1d
     module procedure p2g_2d
  end interface
  interface c2l
     module procedure c2l_1d
     module procedure c2l_2d
  end interface
  interface c2g
     module procedure c2g_1d
     module procedure c2g_2d
  end interface
  interface l2g
     module procedure l2g_1d
     module procedure l2g_2d
  end interface
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: p2c_1d
!
! !INTERFACE:
  subroutine p2c_1d (lbp, ubp, lbc, ubc, parr, carr, p2c_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from pfts to columns.
! Averaging is only done for points that are not equal to "spval".
!
! !USES:
    use clm_varpar, only : max_pft_per_col
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbp, ubp              ! beginning and ending pft
    integer , intent(in)  :: lbc, ubc              ! beginning and ending column
    real(r8), intent(in)  :: parr(lbp:ubp)         ! pft array
    real(r8), intent(out) :: carr(lbc:ubc)         ! column array
    character(len=*), intent(in) :: p2c_scale_type ! scale type
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: pi,p,c,index           ! indices
    real(r8) :: scale_p2c(lbp:ubp)     ! scale factor for column->landunit mapping
    logical  :: found                  ! temporary for error check
    real(r8) :: sumwt(lbc:ubc)         ! sum of weights
    real(r8), pointer :: wtcol(:)      ! weight of pft relative to column
    integer , pointer :: pcolumn(:)    ! column index of corresponding pft
    integer , pointer :: npfts(:)      ! number of pfts in column
    integer , pointer :: pfti(:)       ! initial pft index in column
!------------------------------------------------------------------------

    wtcol    => clm3%g%l%c%p%wtcol
    pcolumn  => clm3%g%l%c%p%column
    npfts    => clm3%g%l%c%npfts
    pfti     => clm3%g%l%c%pfti

    if (p2c_scale_type == 'unity') then
       do p = lbp,ubp
          scale_p2c(p) = 1.0_r8
       end do
    else
       write(6,*)'p2c_1d error: scale type ',p2c_scale_type,' not supported'
       call endrun()
    end if

    carr(lbc:ubc) = spval
    sumwt(lbc:ubc) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
    do pi = 1,max_pft_per_col
!dir$ concurrent
!cdir nodep
       do c = lbc,ubc
          if (pi <= npfts(c)) then
             p = pfti(c) + pi - 1
             if (wtcol(p) /= 0._r8) then
                if (parr(p) /= spval) then
                   carr(c) = 0._r8
                end if
             end if
          end if
       end do
    end do
!dir$ nointerchange
    do pi = 1,max_pft_per_col
!dir$ concurrent
!cdir nodep
       do c = lbc,ubc
          if (pi <= npfts(c)) then
             p = pfti(c) + pi - 1
             if (wtcol(p) /= 0._r8) then
                if (parr(p) /= spval) then
                   carr(c) = carr(c) + parr(p) * scale_p2c(p) * wtcol(p)
                   sumwt(c) = sumwt(c) + wtcol(p)
                end if
             end if
          end if
       end do
    end do
#else
    do p = lbp,ubp
       if (wtcol(p) /= 0._r8) then
          if (parr(p) /= spval) then
             c = pcolumn(p)
             if (sumwt(c) == 0._r8) carr(c) = 0._r8
             carr(c) = carr(c) + parr(p) * scale_p2c(p) * wtcol(p)
             sumwt(c) = sumwt(c) + wtcol(p)
          end if
       end if
    end do
#endif
    found = .false.
    do c = lbc,ubc
       if (sumwt(c) > 1.0_r8 + 1.e-6_r8) then
          found = .true.
          index = c
       else if (sumwt(c) /= 0._r8) then
          carr(c) = carr(c)/sumwt(c)
       end if
    end do
    if (found) then
       write(6,*)'p2c error: sumwt is greater than 1.0 at c= ',index
       call endrun()
    end if

  end subroutine p2c_1d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: p2c_2d
!
! !INTERFACE:
  subroutine p2c_2d (lbp, ubp, lbc, ubc, num2d, parr, carr, p2c_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from landunits to gridcells.
! Averaging is only done for points that are not equal to "spval".
!
! !USES:
    use clm_varpar, only : max_pft_per_col
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbp, ubp              ! beginning and ending pft
    integer , intent(in)  :: lbc, ubc              ! beginning and ending column
    integer , intent(in)  :: num2d                 ! size of second dimension
    real(r8), intent(in)  :: parr(lbp:ubp,num2d)   ! pft array
    real(r8), intent(out) :: carr(lbc:ubc,num2d)   ! column array
    character(len=*), intent(in) :: p2c_scale_type ! scale type
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: j,pi,p,c,index         ! indices
    real(r8) :: scale_p2c(lbp:ubp)     ! scale factor for column->landunit mapping
    logical  :: found                  ! temporary for error check
    real(r8) :: sumwt(lbc:ubc)         ! sum of weights
    real(r8), pointer :: wtcol(:)      ! weight of pft relative to column
    integer , pointer :: pcolumn(:)    ! column index of corresponding pft
    integer , pointer :: npfts(:)      ! number of pfts in column
    integer , pointer :: pfti(:)       ! initial pft index in column
!------------------------------------------------------------------------

    wtcol    => clm3%g%l%c%p%wtcol
    pcolumn  => clm3%g%l%c%p%column
    npfts    => clm3%g%l%c%npfts
    pfti     => clm3%g%l%c%pfti

    if (p2c_scale_type == 'unity') then
       do p = lbp,ubp
          scale_p2c(p) = 1.0_r8
       end do
    else
       write(6,*)'p2c_2d error: scale type ',p2c_scale_type,' not supported'
       call endrun()
    end if

    carr(:,:) = spval
    do j = 1,num2d
       sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
       do pi = 1,max_pft_per_col
!dir$ concurrent
!cdir nodep
          do c = lbc,ubc
             if (pi <= npfts(c)) then
                p = pfti(c) + pi - 1
                if (wtcol(p) /= 0._r8) then
                   if (parr(p,j) /= spval) then
                      carr(c,j) = 0._r8
                   end if
                end if
             end if
          end do
       end do
!dir$ nointerchange
       do pi = 1,max_pft_per_col
!dir$ concurrent
!cdir nodep
          do c = lbc,ubc
             if (pi <= npfts(c)) then
                p = pfti(c) + pi - 1
                if (wtcol(p) /= 0._r8) then
                   if (parr(p,j) /= spval) then
                      carr(c,j) = carr(c,j) + parr(p,j) * scale_p2c(p) * wtcol(p)
                      sumwt(c) = sumwt(c) + wtcol(p)
                   end if
                end if
             end if
          end do
       end do
#else
       do p = lbp,ubp
          if (wtcol(p) /= 0._r8) then
             if (parr(p,j) /= spval) then
                c = pcolumn(p)
                if (sumwt(c) == 0._r8) carr(c,j) = 0._r8
                carr(c,j) = carr(c,j) + parr(p,j) * scale_p2c(p) * wtcol(p)
                sumwt(c) = sumwt(c) + wtcol(p)
             end if
          end if
       end do
#endif
       found = .false.
       do c = lbc,ubc
          if (sumwt(c) > 1.0_r8 + 1.e-6_r8) then
             found = .true.
             index = c
          else if (sumwt(c) /= 0._r8) then
             carr(c,j) = carr(c,j)/sumwt(c)
          end if
       end do
       if (found) then
          write(6,*)'p2c_2d error: sumwt is greater than 1.0 at c= ',index,' lev= ',j
          call endrun()
       end if
    end do 
  end subroutine p2c_2d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: p2c_1d_filter
!
! !INTERFACE:
  subroutine p2c_1d_filter (numfc, filterc, pftarr, colarr)
!
! !DESCRIPTION:
! perform pft to column averaging for single level pft arrays
!
! !USES:
    use clm_varpar, only : max_pft_per_col
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: numfc
    integer , intent(in)  :: filterc(numfc)
    real(r8), pointer     :: pftarr(:)
    real(r8), pointer     :: colarr(:)
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer :: fc,c,pi,p           ! indices
    integer , pointer :: npfts(:)
    integer , pointer :: pfti(:)
    integer , pointer :: pftf(:)
    real(r8), pointer :: wtcol(:)
    real(r8), pointer :: wtgcell(:)
!-----------------------------------------------------------------------

    npfts   => clm3%g%l%c%npfts
    pfti    => clm3%g%l%c%pfti
    pftf    => clm3%g%l%c%pftf
    wtcol   => clm3%g%l%c%p%wtcol
    wtgcell => clm3%g%l%c%p%wtgcell

#if (defined CPP_VECTOR)
!dir$ concurrent
!cdir nodep
    do fc = 1,numfc
       c = filterc(fc)
       colarr(c) = 0._r8
    end do
!dir$ nointerchange
    do pi = 1,max_pft_per_col
!dir$ concurrent
!cdir nodep
       do fc = 1,numfc
          c = filterc(fc)
          if ( pi <=  npfts(c) ) then
             p = pfti(c) + pi - 1
             if (wtgcell(p) > 0._r8) colarr(c) = colarr(c) + pftarr(p) * wtcol(p)
          end if
       end do
    end do
#else
    do fc = 1,numfc
       c = filterc(fc)
       colarr(c) = 0._r8
       do p = pfti(c), pftf(c)
          if (wtgcell(p) > 0._r8) colarr(c) = colarr(c) + pftarr(p) * wtcol(p)
       end do
    end do
#endif

  end subroutine p2c_1d_filter

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: p2c_2d_filter
!
! !INTERFACE:
  subroutine p2c_2d_filter (lev, numfc, filterc, pftarr, colarr)
!
! !DESCRIPTION:
! perform pft to column averaging for multi level pft arrays
!
! !USES:
    use clm_varpar, only : max_pft_per_col

! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lev
    integer , intent(in)  :: numfc
    integer , intent(in)  :: filterc(numfc)
    real(r8), pointer     :: pftarr(:,:)
    real(r8), pointer     :: colarr(:,:)
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer :: fc,c,pi,p,j    ! indices
    integer , pointer :: npfts(:)
    integer , pointer :: pfti(:)
    integer , pointer :: pftf(:)
    real(r8), pointer :: wtcol(:)
!-----------------------------------------------------------------------

    npfts => clm3%g%l%c%npfts
    pfti  => clm3%g%l%c%pfti
    pftf  => clm3%g%l%c%pftf
    wtcol => clm3%g%l%c%p%wtcol

#if (defined CPP_VECTOR)
    do j = 1,lev
!dir$ concurrent
!cdir nodep
       do fc = 1,numfc
          c = filterc(fc)
          colarr(c,j) = 0._r8
       end do
!dir$ nointerchange
       do pi = 1,max_pft_per_col
!dir$ concurrent
!cdir nodep
          do fc = 1,numfc
             c = filterc(fc)
             if ( pi <=  npfts(c) ) then
                p = pfti(c) + pi - 1
                colarr(c,j) = colarr(c,j) + pftarr(p,j) * wtcol(p)
             end if
          end do
       end do
    end do
#else
    do j = 1,lev
       do fc = 1,numfc
          c = filterc(fc)
          colarr(c,j) = 0._r8
          do p = pfti(c), pftf(c)
             colarr(c,j) = colarr(c,j) + pftarr(p,j) * wtcol(p)
          end do
       end do
    end do
#endif

  end subroutine p2c_2d_filter

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: p2l_1d
!
! !INTERFACE:
  subroutine p2l_1d (lbp, ubp, lbc, ubc, lbl, ubl, parr, larr, &
       p2c_scale_type, c2l_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from pfts to landunits
! Averaging is only done for points that are not equal to "spval".
!
! !USES:
    use clm_varpar, only : max_pft_per_lu
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbp, ubp              ! beginning and ending pft indices
    integer , intent(in)  :: lbc, ubc              ! beginning and ending column indices
    integer , intent(in)  :: lbl, ubl              ! beginning and ending landunit indices
    real(r8), intent(in)  :: parr(lbp:ubp)         ! input column array
    real(r8), intent(out) :: larr(lbl:ubl)         ! output landunit array
    character(len=*), intent(in) :: p2c_scale_type ! scale factor type for averaging
    character(len=*), intent(in) :: c2l_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: pi,p,c,l,index         ! indices
    logical  :: found                  ! temporary for error check
    real(r8) :: sumwt(lbl:ubl)         ! sum of weights
    real(r8) :: scale_p2c(lbc:ubc)     ! scale factor for pft->column mapping
    real(r8) :: scale_c2l(lbc:ubc)     ! scale factor for column->landunit mapping
    real(r8), pointer :: wtlunit(:)    ! weight of pft relative to landunit
    integer , pointer :: pcolumn(:)    ! column of corresponding pft
    integer , pointer :: plandunit(:)  ! landunit of corresponding pft
    integer , pointer :: npfts(:)      ! number of pfts in landunit
    integer , pointer :: pfti(:)       ! initial pft index in landunit
    integer , pointer :: clandunit(:)  ! landunit of corresponding column
    integer , pointer :: ctype(:)      ! column type
    integer , pointer :: ltype(:)      ! landunit type
    real(r8), pointer :: canyon_hwr(:) ! urban canyon height to width ratio
!------------------------------------------------------------------------

    canyon_hwr => clm3%g%l%canyon_hwr
    ltype      => clm3%g%l%itype
    ctype      => clm3%g%l%c%itype
    clandunit  => clm3%g%l%c%landunit
    wtlunit    => clm3%g%l%c%p%wtlunit
    pcolumn    => clm3%g%l%c%p%column
    plandunit  => clm3%g%l%c%p%landunit
    npfts      => clm3%g%l%npfts
    pfti       => clm3%g%l%pfti

    if (c2l_scale_type == 'unity') then
       do c = lbc,ubc
          scale_c2l(c) = 1.0_r8
       end do
    else if (c2l_scale_type == 'urbanf') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0_r8
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbans') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0 / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbanh') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = spval
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else
       write(6,*)'p2l_1d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

    if (p2c_scale_type == 'unity') then
       do p = lbp,ubp
          scale_p2c(p) = 1.0_r8
       end do
    else
       write(6,*)'p2l_1d error: scale type ',p2c_scale_type,' not supported'
       call endrun()
    end if

    larr(:) = spval
    sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
    do pi = 1,max_pft_per_lu
!dir$ concurrent
!cdir nodep
       do l = lbl,ubl
          if (pi <= npfts(l)) then
             p = pfti(l) + pi - 1
             if (wtlunit(p) /= 0._r8) then
                if (parr(p) /= spval) then
                   larr(l) = 0._r8
                end if
             end if
          end if
       end do
    end do
!dir$ nointerchange
    do pi = 1,max_pft_per_lu
!dir$ concurrent
!cdir nodep
       do l = lbl,ubl
          if (pi <= npfts(l)) then
             p = pfti(l) + pi - 1
             if (wtlunit(p) /= 0._r8) then
                c = pcolumn(p)
                if (parr(p) /= spval .and. scale_c2l(c) /= spval) then
                   larr(l) = larr(l) + parr(p) * scale_p2c(p) * scale_c2l(c) * wtlunit(p)
                   sumwt(l) = sumwt(l) + wtlunit(p)
                end if
             end if
          end if
       end do
    end do
#else
    do p = lbp,ubp
       if (wtlunit(p) /= 0._r8) then
          c = pcolumn(p)
          if (parr(p) /= spval .and. scale_c2l(c) /= spval) then
             l = plandunit(p)
             if (sumwt(l) == 0._r8) larr(l) = 0._r8
             larr(l) = larr(l) + parr(p) * scale_p2c(p) * scale_c2l(c) * wtlunit(p)
             sumwt(l) = sumwt(l) + wtlunit(p)
          end if
       end if
    end do
#endif
    found = .false.
    do l = lbl,ubl
       if (sumwt(l) > 1.0_r8 + 1.e-6_r8) then
          found = .true.
          index = l
       else if (sumwt(l) /= 0._r8) then
          larr(l) = larr(l)/sumwt(l)
       end if
    end do
    if (found) then
       write(6,*)'p2l_1d error: sumwt is greater than 1.0 at l= ',index
       call endrun()
    end if

  end subroutine p2l_1d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: p2l_2d
!
! !INTERFACE:
  subroutine p2l_2d(lbp, ubp, lbc, ubc, lbl, ubl, num2d, parr, larr, &
       p2c_scale_type, c2l_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from pfts to landunits
! Averaging is only done for points that are not equal to "spval".
!
! !USES:
    use clm_varpar, only : max_pft_per_lu
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbp, ubp              ! beginning and ending pft indices
    integer , intent(in)  :: lbc, ubc              ! beginning and ending column indices
    integer , intent(in)  :: lbl, ubl              ! beginning and ending landunit indices
    integer , intent(in)  :: num2d                 ! size of second dimension
    real(r8), intent(in)  :: parr(lbp:ubp,num2d)   ! input pft array
    real(r8), intent(out) :: larr(lbl:ubl,num2d)   ! output gridcell array
    character(len=*), intent(in) :: p2c_scale_type ! scale factor type for averaging
    character(len=*), intent(in) :: c2l_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: j,pi,p,c,l,index       ! indices
    logical  :: found                  ! temporary for error check
    real(r8) :: sumwt(lbl:ubl)         ! sum of weights
    real(r8) :: scale_p2c(lbc:ubc)     ! scale factor for pft->column mapping
    real(r8) :: scale_c2l(lbc:ubc)     ! scale factor for column->landunit mapping
    real(r8), pointer :: wtlunit(:)    ! weight of pft relative to landunit
    integer , pointer :: pcolumn(:)    ! column of corresponding pft
    integer , pointer :: plandunit(:)  ! landunit of corresponding pft
    integer , pointer :: npfts(:)      ! number of pfts in landunit
    integer , pointer :: pfti(:)       ! initial pft index in landunit
    integer , pointer :: clandunit(:)  ! landunit of corresponding column
    integer , pointer :: ctype(:)      ! column type
    integer , pointer :: ltype(:)      ! landunit type
    real(r8), pointer :: canyon_hwr(:) ! urban canyon height to width ratio
!------------------------------------------------------------------------

    canyon_hwr => clm3%g%l%canyon_hwr
    ltype      => clm3%g%l%itype
    clandunit  => clm3%g%l%c%landunit
    ctype      => clm3%g%l%c%itype
    wtlunit   => clm3%g%l%c%p%wtlunit
    pcolumn   => clm3%g%l%c%p%column
    plandunit => clm3%g%l%c%p%landunit
    npfts     => clm3%g%l%npfts
    pfti      => clm3%g%l%pfti

    if (c2l_scale_type == 'unity') then
       do c = lbc,ubc
          scale_c2l(c) = 1.0_r8
       end do
    else if (c2l_scale_type == 'urbanf') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0_r8
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbans') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0 / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbanh') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = spval
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else
       write(6,*)'p2l_2d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

    if (p2c_scale_type == 'unity') then
       do p = lbp,ubp
          scale_p2c(p) = 1.0_r8
       end do
    else
       write(6,*)'p2l_2d error: scale type ',p2c_scale_type,' not supported'
       call endrun()
    end if

    larr(:,:) = spval
    do j = 1,num2d
       sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
       do pi = 1,max_pft_per_lu
!dir$ concurrent
!cdir nodep
          do l = lbl,ubl
             if (pi <= npfts(l)) then
                p = pfti(l) + pi - 1
                if (wtlunit(p) /= 0._r8) then
                   if (parr(p,j) /= spval) then
                      larr(l,j) = 0._r8
                   end if
                end if
             end if
          end do
       end do
!dir$ nointerchange
       do pi = 1,max_pft_per_lu
!dir$ concurrent
!cdir nodep
          do l = lbl,ubl
             if (pi <= npfts(l)) then
                p = pfti(l) + pi - 1
                if (wtlunit(p) /= 0._r8) then
                   c = pcolumn(p)
                   if (parr(p,j) /= spval .and. scale_c2l(c) /= spval) then
                      larr(l,j) = larr(l,j) + parr(p,j) * scale_p2c(p) * scale_c2l(c) * wtlunit(p)
                      sumwt(l) = sumwt(l) + wtlunit(p)
                   end if
                end if
             end if
          end do
       end do
#else
       do p = lbp,ubp
          if (wtlunit(p) /= 0._r8) then
             c = pcolumn(p)
             if (parr(p,j) /= spval .and. scale_c2l(c) /= spval) then
                l = plandunit(p)
                if (sumwt(l) == 0._r8) larr(l,j) = 0._r8
                larr(l,j) = larr(l,j) + parr(p,j) * scale_p2c(p) * scale_c2l(c) * wtlunit(p)
                sumwt(l) = sumwt(l) + wtlunit(p)
             end if
          end if
       end do
#endif
       found = .false.
       do l = lbl,ubl
          if (sumwt(l) > 1.0_r8 + 1.e-6_r8) then
             found = .true.
             index = l
          else if (sumwt(l) /= 0._r8) then
             larr(l,j) = larr(l,j)/sumwt(l)
          end if
       end do
       if (found) then
          write(6,*)'p2l_2d error: sumwt is greater than 1.0 at l= ',index,' j= ',j
          call endrun()
       end if
    end do

  end subroutine p2l_2d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: p2g_1d
!
! !INTERFACE:
  subroutine p2g_1d(lbp, ubp, lbc, ubc, lbl, ubl, lbg, ubg, parr, garr, &
       p2c_scale_type, c2l_scale_type, l2g_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from pfts to gridcells.
! Averaging is only done for points that are not equal to "spval".
!
! !USES:
    use clm_varpar, only : max_pft_per_gcell
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbp, ubp            ! beginning and ending pft indices
    integer , intent(in)  :: lbc, ubc            ! beginning and ending column indices
    integer , intent(in)  :: lbl, ubl            ! beginning and ending landunit indices
    integer , intent(in)  :: lbg, ubg            ! beginning and ending gridcell indices
    real(r8), intent(in)  :: parr(lbp:ubp)       ! input pft array
    real(r8), intent(out) :: garr(lbg:ubg)       ! output gridcell array
    character(len=*), intent(in) :: p2c_scale_type ! scale factor type for averaging
    character(len=*), intent(in) :: c2l_scale_type ! scale factor type for averaging
    character(len=*), intent(in) :: l2g_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!  !LOCAL VARIABLES:
!EOP
    integer  :: pi,p,c,l,g,index       ! indices
    logical  :: found                  ! temporary for error check
    real(r8) :: scale_p2c(lbp:ubp)     ! scale factor
    real(r8) :: scale_c2l(lbc:ubc)     ! scale factor
    real(r8) :: scale_l2g(lbl:ubl)     ! scale factor
    real(r8) :: sumwt(lbg:ubg)         ! sum of weights
    real(r8), pointer :: wtgcell(:)    ! weight of pfts relative to gridcells
    integer , pointer :: pcolumn(:)    ! column of corresponding pft
    integer , pointer :: plandunit(:)  ! landunit of corresponding pft
    integer , pointer :: pgridcell(:)  ! gridcell of corresponding pft
    integer , pointer :: npfts(:)      ! number of pfts in gridcell
    integer , pointer :: pfti(:)       ! initial pft index in gridcell
    integer , pointer :: ctype(:)      ! column type
    integer , pointer :: clandunit(:)  ! landunit of corresponding column
    integer , pointer :: ltype(:)      ! landunit type
    real(r8), pointer :: canyon_hwr(:) ! urban canyon height to width ratio
!------------------------------------------------------------------------

    canyon_hwr => clm3%g%l%canyon_hwr
    ltype      => clm3%g%l%itype
    clandunit  => clm3%g%l%c%landunit
    ctype      => clm3%g%l%c%itype
    wtgcell   => clm3%g%l%c%p%wtgcell
    pcolumn   => clm3%g%l%c%p%column
    pgridcell => clm3%g%l%c%p%gridcell
    plandunit => clm3%g%l%c%p%landunit
    npfts     => clm3%g%npfts
    pfti      => clm3%g%pfti

    if (l2g_scale_type == 'unity') then
       do l = lbl,ubl
          scale_l2g(l) = 1.0_r8
       end do
    else
       write(6,*)'p2g_1d error: scale type ',l2g_scale_type,' not supported'
       call endrun()
    end if

    if (c2l_scale_type == 'unity') then
       do c = lbc,ubc
          scale_c2l(c) = 1.0_r8
       end do
    else if (c2l_scale_type == 'urbanf') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0_r8
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbans') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0 / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbanh') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = spval
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else
       write(6,*)'p2g_1d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

    if (p2c_scale_type == 'unity') then
       do p = lbp,ubp
          scale_p2c(p) = 1.0_r8
       end do
    else
       write(6,*)'p2g_1d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

    garr(:) = spval
    sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
    do pi = 1,max_pft_per_gcell
!dir$ concurrent
!cdir nodep
       do g = lbg,ubg
          if (pi <= npfts(g)) then
             p = pfti(g) + pi - 1
             if (wtgcell(p) /= 0._r8) then
                if (parr(p) /= spval) then
                   garr(g) = 0._r8
                end if
             end if
          end if
       end do
    end do
!dir$ nointerchange
    do pi = 1,max_pft_per_gcell
!dir$ concurrent
!cdir nodep
       do g = lbg,ubg
          if (pi <= npfts(g)) then
             p = pfti(g) + pi - 1
             if (wtgcell(p) /= 0._r8) then
                c = pcolumn(p)
                if (parr(p) /= spval .and. scale_c2l(c) /= spval) then
                   l = plandunit(p)
                   garr(g) = garr(g) + parr(p) * scale_p2c(p) * scale_c2l(c) * scale_l2g(l) * wtgcell(p)
                   sumwt(g) = sumwt(g) + wtgcell(p)
                end if
             end if
          end if
       end do
    end do
#else
    do p = lbp,ubp
       if (wtgcell(p) /= 0._r8) then
          c = pcolumn(p)
          if (parr(p) /= spval .and. scale_c2l(c) /= spval) then
             l = plandunit(p)
             g = pgridcell(p)
             if (sumwt(g) == 0._r8) garr(g) = 0._r8
             garr(g) = garr(g) + parr(p) * scale_p2c(p) * scale_c2l(c) * scale_l2g(l) * wtgcell(p)
             sumwt(g) = sumwt(g) + wtgcell(p)
          end if
       end if
    end do
#endif
    found = .false.
    do g = lbg, ubg
       if (sumwt(g) > 1.0_r8 + 1.e-6_r8) then
          found = .true.
          index = g
       else if (sumwt(g) /= 0._r8) then
          garr(g) = garr(g)/sumwt(g)
       end if
    end do
    if (found) then
       write(6,*)'p2g_1d error: sumwt is greater than 1.0 at g= ',index
       call endrun()
    end if

  end subroutine p2g_1d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: p2g_2d
!
! !INTERFACE:
  subroutine p2g_2d(lbp, ubp, lbc, ubc, lbl, ubl, lbg, ubg, num2d, &
       parr, garr, p2c_scale_type, c2l_scale_type, l2g_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from pfts to gridcells.
! Averaging is only done for points that are not equal to "spval".
!
! !USES:
    use clm_varpar, only : max_pft_per_gcell
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbp, ubp              ! beginning and ending pft indices
    integer , intent(in)  :: lbc, ubc              ! beginning and ending column indices
    integer , intent(in)  :: lbl, ubl              ! beginning and ending landunit indices
    integer , intent(in)  :: lbg, ubg              ! beginning and ending gridcell indices
    integer , intent(in)  :: num2d                 ! size of second dimension
    real(r8), intent(in)  :: parr(lbp:ubp,num2d)   ! input pft array
    real(r8), intent(out) :: garr(lbg:ubg,num2d)   ! output gridcell array
    character(len=*), intent(in) :: p2c_scale_type ! scale factor type for averaging
    character(len=*), intent(in) :: c2l_scale_type ! scale factor type for averaging
    character(len=*), intent(in) :: l2g_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: j,pi,p,c,l,g,index     ! indices
    logical  :: found                  ! temporary for error check
    real(r8) :: scale_p2c(lbp:ubp)     ! scale factor
    real(r8) :: scale_c2l(lbc:ubc)     ! scale factor
    real(r8) :: scale_l2g(lbl:ubl)     ! scale factor
    real(r8) :: sumwt(lbg:ubg)         ! sum of weights
    real(r8), pointer :: wtgcell(:)    ! weight of pfts relative to gridcells
    integer , pointer :: pcolumn(:)    ! column of corresponding pft
    integer , pointer :: plandunit(:)  ! landunit of corresponding pft
    integer , pointer :: pgridcell(:)  ! gridcell of corresponding pft
    integer , pointer :: npfts(:)      ! number of pfts in gridcell
    integer , pointer :: pfti(:)       ! initial pft index in gridcell
    integer , pointer :: clandunit(:)  ! landunit of corresponding column
    integer , pointer :: ctype(:)      ! column type
    integer , pointer :: ltype(:)      ! landunit type
    real(r8), pointer :: canyon_hwr(:) ! urban canyon height to width ratio
!------------------------------------------------------------------------

    canyon_hwr   => clm3%g%l%canyon_hwr
    ltype        => clm3%g%l%itype
    clandunit    => clm3%g%l%c%landunit
    ctype        => clm3%g%l%c%itype
    wtgcell      => clm3%g%l%c%p%wtgcell
    pcolumn      => clm3%g%l%c%p%column
    pgridcell    => clm3%g%l%c%p%gridcell
    plandunit    => clm3%g%l%c%p%landunit
    npfts        => clm3%g%npfts
    pfti         => clm3%g%pfti

    if (l2g_scale_type == 'unity') then
       do l = lbl,ubl
          scale_l2g(l) = 1.0_r8
       end do
    else
       write(6,*)'p2g_2d error: scale type ',l2g_scale_type,' not supported'
       call endrun()
    end if

    if (c2l_scale_type == 'unity') then
       do c = lbc,ubc
          scale_c2l(c) = 1.0_r8
       end do
    else if (c2l_scale_type == 'urbanf') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0_r8
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbans') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0 / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbanh') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = spval 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = spval 
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else
       write(6,*)'p2g_2d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

    if (p2c_scale_type == 'unity') then
       do p = lbp,ubp
          scale_p2c(p) = 1.0_r8
       end do
    else
       write(6,*)'p2g_2d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

    garr(:,:) = spval
    do j = 1,num2d
       sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
       do pi = 1,max_pft_per_gcell
!dir$ concurrent
!cdir nodep
          do g = lbg,ubg
             if (pi <= npfts(g)) then
                p = pfti(g) + pi - 1
                if (wtgcell(p) /= 0._r8) then
                   if (parr(p,j) /= spval) then
                      garr(g,j) = 0._r8
                   end if
                end if
             end if
          end do
       end do
!dir$ nointerchange
       do pi = 1,max_pft_per_gcell
!dir$ concurrent
!cdir nodep
          do g = lbg,ubg
             if (pi <= npfts(g)) then
                p = pfti(g) + pi - 1
                if (wtgcell(p) /= 0._r8) then
                   c = pcolumn(p)
                   if (parr(p,j) /= spval .and. scale_c2l(c) /= spval) then
                      l = plandunit(p)
                      garr(g,j) = garr(g,j) + parr(p,j) * scale_p2c(p) * scale_c2l(c) * scale_l2g(l) * wtgcell(p)
                      sumwt(g) = sumwt(g) + wtgcell(p)
                   end if
                end if
             end if
          end do
       end do
#else
       do p = lbp,ubp
          if (wtgcell(p) /= 0._r8) then
             c = pcolumn(p)
             if (parr(p,j) /= spval .and. scale_c2l(c) /= spval) then
                l = plandunit(p)
                g = pgridcell(p)
                if (sumwt(g) == 0._r8) garr(g,j) = 0._r8
                garr(g,j) = garr(g,j) + parr(p,j) * scale_p2c(p) * scale_c2l(c) * scale_l2g(l) * wtgcell(p)
                sumwt(g) = sumwt(g) + wtgcell(p)
             end if
          end if
       end do
#endif
       found = .false.
       do g = lbg, ubg
          if (sumwt(g) > 1.0_r8 + 1.e-6_r8) then
             found = .true.
             index = g
          else if (sumwt(g) /= 0._r8) then
             garr(g,j) = garr(g,j)/sumwt(g)
          end if
       end do
       if (found) then
          write(6,*)'p2g_2d error: sumwt gt 1.0 at g/sumwt = ',index,sumwt(index)
          call endrun()
       end if
    end do

  end subroutine p2g_2d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: c2l_1d
!
! !INTERFACE:
  subroutine c2l_1d (lbc, ubc, lbl, ubl, carr, larr, c2l_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from columns to landunits
! Averaging is only done for points that are not equal to "spval".
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbc, ubc      ! beginning and ending column indices
    integer , intent(in)  :: lbl, ubl      ! beginning and ending landunit indices
    real(r8), intent(in)  :: carr(lbc:ubc) ! input column array
    real(r8), intent(out) :: larr(lbl:ubl) ! output landunit array
    character(len=*), intent(in) :: c2l_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: ci,c,l,index           ! indices
    integer  :: max_col_per_lu         ! max columns per landunit; on the fly
    logical  :: found                  ! temporary for error check
    real(r8) :: scale_c2l(lbc:ubc)     ! scale factor for column->landunit mapping
    real(r8) :: sumwt(lbl:ubl)         ! sum of weights
    real(r8), pointer :: wtlunit(:)    ! weight of landunits relative to gridcells
    integer , pointer :: clandunit(:)  ! gridcell of corresponding column
    integer , pointer :: ncolumns(:)   ! number of columns in landunit
    integer , pointer :: coli(:)       ! initial column index in landunit
    integer , pointer :: ctype(:)      ! column type
    integer , pointer :: ltype(:)      ! landunit type
    real(r8), pointer :: canyon_hwr(:) ! urban canyon height to width ratio
!------------------------------------------------------------------------

    ctype      => clm3%g%l%c%itype
    ltype      => clm3%g%l%itype
    canyon_hwr => clm3%g%l%canyon_hwr
    wtlunit    => clm3%g%l%c%wtlunit
    clandunit  => clm3%g%l%c%landunit
    ncolumns   => clm3%g%l%ncolumns
    coli       => clm3%g%l%coli

    if (c2l_scale_type == 'unity') then
       do c = lbc,ubc
          scale_c2l(c) = 1.0_r8
       end do
    else if (c2l_scale_type == 'urbanf') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0_r8
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbans') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0 / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbanh') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = spval
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else
       write(6,*)'c2l_1d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

    larr(:) = spval
    sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
    max_col_per_lu = 0
    do l = lbl,ubl
       max_col_per_lu = max(ncolumns(l), max_col_per_lu)
    end do
!dir$ nointerchange
    do ci = 1,max_col_per_lu
!dir$ concurrent
!cdir nodep
       do l = lbl,ubl
          if (ci <= ncolumns(l)) then
             c = coli(l) + ci - 1
             if (wtlunit(c) /= 0._r8) then
                if (carr(c) /= spval) then
                   larr(l) = 0._r8
                end if
             end if
          end if
       end do
    end do
!dir$ nointerchange
    do ci = 1,max_col_per_lu
!dir$ concurrent
!cdir nodep
       do l = lbl,ubl
          if (ci <= ncolumns(l)) then
             c = coli(l) + ci - 1
             if (wtlunit(c) /= 0._r8) then
                if (carr(c) /= spval .and. scale_c2l(c) /= spval) then
                   larr(l) = larr(l) + carr(c) * scale_c2l(c) * wtlunit(c)
                   sumwt(l) = sumwt(l) + wtlunit(c)
                end if
             end if
          end if
      end do
    end do
#else
    do c = lbc,ubc
       if (wtlunit(c) /= 0._r8) then
          if (carr(c) /= spval .and. scale_c2l(c) /= spval) then
             l = clandunit(c)
             if (sumwt(l) == 0._r8) larr(l) = 0._r8
             larr(l) = larr(l) + carr(c) * scale_c2l(c) * wtlunit(c)
             sumwt(l) = sumwt(l) + wtlunit(c)
          end if
       end if
    end do
#endif
    found = .false.
    do l = lbl,ubl
       if (sumwt(l) > 1.0_r8 + 1.e-6_r8) then
          found = .true.
          index = l
       else if (sumwt(l) /= 0._r8) then
          larr(l) = larr(l)/sumwt(l)
       end if
    end do
    if (found) then
       write(6,*)'c2l_1d error: sumwt is greater than 1.0 at l= ',index
       call endrun()
    end if

  end subroutine c2l_1d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: c2l_2d
!
! !INTERFACE:
  subroutine c2l_2d (lbc, ubc, lbl, ubl, num2d, carr, larr, c2l_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from columns to landunits
! Averaging is only done for points that are not equal to "spval".
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbc, ubc            ! beginning and ending column indices
    integer , intent(in)  :: lbl, ubl            ! beginning and ending landunit indices
    integer , intent(in)  :: num2d               ! size of second dimension
    real(r8), intent(in)  :: carr(lbc:ubc,num2d) ! input column array
    real(r8), intent(out) :: larr(lbl:ubl,num2d) ! output landunit array
    character(len=*), intent(in) :: c2l_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: j,l,ci,c,index         ! indices
    integer  :: max_col_per_lu         ! max columns per landunit; on the fly
    logical  :: found                  ! temporary for error check
    real(r8) :: scale_c2l(lbc:ubc)        ! scale factor for column->landunit mapping
    real(r8) :: sumwt(lbl:ubl)         ! sum of weights
    real(r8), pointer :: wtlunit(:)    ! weight of column relative to landunit
    integer , pointer :: clandunit(:)  ! landunit of corresponding column
    integer , pointer :: ncolumns(:)   ! number of columns in landunit
    integer , pointer :: coli(:)       ! initial column index in landunit
    integer , pointer :: ctype(:)      ! column type
    integer , pointer :: ltype(:)      ! landunit type
    real(r8), pointer :: canyon_hwr(:) ! urban canyon height to width ratio
!------------------------------------------------------------------------

    ctype      => clm3%g%l%c%itype
    ltype      => clm3%g%l%itype
    canyon_hwr => clm3%g%l%canyon_hwr
    wtlunit    => clm3%g%l%c%wtlunit
    clandunit  => clm3%g%l%c%landunit
    ncolumns   => clm3%g%l%ncolumns
    coli       => clm3%g%l%coli

    if (c2l_scale_type == 'unity') then
       do c = lbc,ubc
          scale_c2l(c) = 1.0_r8
       end do
    else if (c2l_scale_type == 'urbanf') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0_r8
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbans') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0 / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbanh') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = spval
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else
       write(6,*)'c2l_2d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

#if (defined CPP_VECTOR)
    max_col_per_lu = 0
    do l = lbl,ubl
       max_col_per_lu = max(ncolumns(l), max_col_per_lu)
    end do
#endif

    larr(:,:) = spval
    do j = 1,num2d
       sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
       do ci = 1,max_col_per_lu
!dir$ concurrent
!cdir nodep
          do l = lbl,ubl
             if (ci <= ncolumns(l)) then
                c = coli(l) + ci - 1
                if (wtlunit(c) /= 0._r8) then
                   if (carr(c,j) /= spval) then
                      larr(l,j) = 0._r8
                   end if
                end if
             end if
          end do
       end do
!dir$ nointerchange
       do ci = 1,max_col_per_lu
!dir$ concurrent
!cdir nodep
          do l = lbl,ubl
             if (ci <= ncolumns(l)) then
                c = coli(l) + ci - 1
                if (wtlunit(c) /= 0._r8) then
                   if (carr(c,j) /= spval .and. scale_c2l(c) /= spval) then
                      larr(l,j) = larr(l,j) + carr(c,j) * scale_c2l(c) * wtlunit(c)
                      sumwt(l) = sumwt(l) + wtlunit(c)
                   end if
                end if
             end if
          end do
       end do
#else
       do c = lbc,ubc
          if (wtlunit(c) /= 0._r8) then
             if (carr(c,j) /= spval .and. scale_c2l(c) /= spval) then
                l = clandunit(c)
                if (sumwt(l) == 0._r8) larr(l,j) = 0._r8
                larr(l,j) = larr(l,j) + carr(c,j) * scale_c2l(c) * wtlunit(c)
                sumwt(l) = sumwt(l) + wtlunit(c)
             end if
          end if
       end do
#endif
       found = .false.
       do l = lbl,ubl
          if (sumwt(l) > 1.0_r8 + 1.e-6_r8) then
             found = .true.
             index = l
          else if (sumwt(l) /= 0._r8) then
             larr(l,j) = larr(l,j)/sumwt(l)
          end if
       end do
       if (found) then
          write(6,*)'c2l_2d error: sumwt is greater than 1.0 at l= ',index,' lev= ',j
          call endrun()
       end if
    end do

  end subroutine c2l_2d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: c2g_1d
!
! !INTERFACE:
  subroutine c2g_1d(lbc, ubc, lbl, ubl, lbg, ubg, carr, garr, &
       c2l_scale_type, l2g_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from columns to gridcells.
! Averaging is only done for points that are not equal to "spval".
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbc, ubc              ! beginning and ending column indices
    integer , intent(in)  :: lbl, ubl              ! beginning and ending landunit indices
    integer , intent(in)  :: lbg, ubg              ! beginning and ending landunit indices
    real(r8), intent(in)  :: carr(lbc:ubc)         ! input column array
    real(r8), intent(out) :: garr(lbg:ubg)         ! output gridcell array
    character(len=*), intent(in) :: c2l_scale_type ! scale factor type for averaging
    character(len=*), intent(in) :: l2g_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: ci,c,l,g,index         ! indices
    integer  :: max_col_per_gcell      ! max columns per gridcell; on the fly
    logical  :: found                  ! temporary for error check
    real(r8) :: scale_c2l(lbc:ubc)     ! scale factor
    real(r8) :: scale_l2g(lbl:ubl)     ! scale factor
    real(r8) :: sumwt(lbg:ubg)         ! sum of weights
    real(r8), pointer :: wtgcell(:)    ! weight of columns relative to gridcells
    integer , pointer :: clandunit(:)  ! landunit of corresponding column
    integer , pointer :: cgridcell(:)  ! gridcell of corresponding column
    integer , pointer :: ncolumns(:)   ! number of columns in gridcell
    integer , pointer :: coli(:)       ! initial column index in gridcell
    integer , pointer :: ctype(:)      ! column type
    integer , pointer :: ltype(:)      ! landunit type
    real(r8), pointer :: canyon_hwr(:) ! urban canyon height to width ratio
!------------------------------------------------------------------------

    ctype      => clm3%g%l%c%itype
    ltype      => clm3%g%l%itype
    canyon_hwr => clm3%g%l%canyon_hwr
    wtgcell    => clm3%g%l%c%wtgcell
    clandunit  => clm3%g%l%c%landunit
    cgridcell  => clm3%g%l%c%gridcell
    ncolumns   => clm3%g%ncolumns
    coli       => clm3%g%coli

    if (l2g_scale_type == 'unity') then
       do l = lbl,ubl
          scale_l2g(l) = 1.0_r8
       end do
    else
       write(6,*)'c2l_1d error: scale type ',l2g_scale_type,' not supported'
       call endrun()
    end if

    if (c2l_scale_type == 'unity') then
       do c = lbc,ubc
          scale_c2l(c) = 1.0_r8
       end do
    else if (c2l_scale_type == 'urbanf') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0_r8
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbans') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0 / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbanh') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = spval
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else
       write(6,*)'c2l_1d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

    garr(:) = spval
    sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
    max_col_per_gcell = 0
    do g = lbg,ubg
       max_col_per_gcell = max(ncolumns(g), max_col_per_gcell)
    end do
!dir$ nointerchange
    do ci = 1,max_col_per_gcell
!dir$ concurrent
!cdir nodep
       do g = lbg,ubg
          if (ci <= ncolumns(g)) then
             c = coli(g) + ci - 1
             if (wtgcell(c) /= 0._r8) then
                if (carr(c) /= spval) then
                   garr(g) = 0._r8
                end if
             end if
          end if
       end do
    end do
!dir$ nointerchange
    do ci = 1,max_col_per_gcell
!dir$ concurrent
!cdir nodep
       do g = lbg,ubg
          if (ci <= ncolumns(g)) then
             c = coli(g) + ci - 1
             if (wtgcell(c) /= 0._r8) then
                if (carr(c) /= spval .and. scale_c2l(c) /= spval) then
                   l = clandunit(c)
                   garr(g) = garr(g) + carr(c) * scale_c2l(c) * scale_l2g(l) * wtgcell(c)
                   sumwt(g) = sumwt(g) + wtgcell(c)
                end if
             end if
          end if
       end do
    end do
#else
    do c = lbc,ubc
       if ( wtgcell(c) /= 0._r8) then
          if (carr(c) /= spval .and. scale_c2l(c) /= spval) then
             l = clandunit(c)
             g = cgridcell(c)
             if (sumwt(g) == 0._r8) garr(g) = 0._r8
             garr(g) = garr(g) + carr(c) * scale_c2l(c) * scale_l2g(l) * wtgcell(c)
             sumwt(g) = sumwt(g) + wtgcell(c)
          end if
       end if
    end do
#endif
    found = .false.
    do g = lbg, ubg
       if (sumwt(g) > 1.0_r8 + 1.e-6_r8) then
          found = .true.
          index = g
       else if (sumwt(g) /= 0._r8) then
          garr(g) = garr(g)/sumwt(g)
       end if
    end do
    if (found) then
       write(6,*)'c2g_1d error: sumwt is greater than 1.0 at g= ',index
       call endrun()
    end if

  end subroutine c2g_1d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: c2g_2d
!
! !INTERFACE:
  subroutine c2g_2d(lbc, ubc, lbl, ubl, lbg, ubg, num2d, carr, garr, &
       c2l_scale_type, l2g_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from columns to gridcells.
! Averaging is only done for points that are not equal to "spval".
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbc, ubc              ! beginning and ending column indices
    integer , intent(in)  :: lbl, ubl              ! beginning and ending landunit indices
    integer , intent(in)  :: lbg, ubg              ! beginning and ending gridcell indices
    integer , intent(in)  :: num2d                 ! size of second dimension
    real(r8), intent(in)  :: carr(lbc:ubc,num2d)   ! input column array
    real(r8), intent(out) :: garr(lbg:ubg,num2d)   ! output gridcell array
    character(len=*), intent(in) :: c2l_scale_type ! scale factor type for averaging
    character(len=*), intent(in) :: l2g_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: j,ci,c,g,l,index       ! indices
    integer  :: max_col_per_gcell      ! max columns per gridcell; on the fly
    logical  :: found                  ! temporary for error check
    real(r8) :: scale_c2l(lbc:ubc)     ! scale factor
    real(r8) :: scale_l2g(lbl:ubl)     ! scale factor
    real(r8) :: sumwt(lbg:ubg)         ! sum of weights
    real(r8), pointer :: wtgcell(:)    ! weight of columns relative to gridcells
    integer , pointer :: clandunit(:)  ! landunit of corresponding column
    integer , pointer :: cgridcell(:)  ! gridcell of corresponding column
    integer , pointer :: ncolumns(:)   ! number of columns in gridcell
    integer , pointer :: coli(:)       ! initial column index in gridcell
    integer , pointer :: ctype(:)      ! column type
    integer , pointer :: ltype(:)      ! landunit type
    real(r8), pointer :: canyon_hwr(:) ! urban canyon height to width ratio
!------------------------------------------------------------------------

    ctype      => clm3%g%l%c%itype
    ltype      => clm3%g%l%itype
    canyon_hwr => clm3%g%l%canyon_hwr
    wtgcell    => clm3%g%l%c%wtgcell
    clandunit  => clm3%g%l%c%landunit
    cgridcell  => clm3%g%l%c%gridcell
    ncolumns   => clm3%g%ncolumns
    coli       => clm3%g%coli

    if (l2g_scale_type == 'unity') then
       do l = lbl,ubl
          scale_l2g(l) = 1.0_r8
       end do
    else
       write(6,*)'c2g_2d error: scale type ',l2g_scale_type,' not supported'
       call endrun()
    end if
    if (c2l_scale_type == 'unity') then
       do c = lbc,ubc
          scale_c2l(c) = 1.0_r8
       end do
    else if (c2l_scale_type == 'urbanf') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = 3.0 * canyon_hwr(l) 
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0_r8
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbans') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = (3.0 * canyon_hwr(l)) / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = 3.0 / (2.*canyon_hwr(l) + 1.)
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = 1.0_r8
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else if (c2l_scale_type == 'urbanh') then
       do c = lbc,ubc
          l = clandunit(c) 
          if (ltype(l) == isturb) then
             if (ctype(c) == icol_sunwall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_shadewall) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_road_perv .or. ctype(c) == icol_road_imperv) then
                scale_c2l(c) = spval
             else if (ctype(c) == icol_roof) then
                scale_c2l(c) = spval
             end if
          else
             scale_c2l(c) = 1.0_r8
          end if
       end do
    else
       write(6,*)'c2g_2d error: scale type ',c2l_scale_type,' not supported'
       call endrun()
    end if

#if (defined CPP_VECTOR)
    max_col_per_gcell = 0
    do g = lbg,ubg
       max_col_per_gcell = max(ncolumns(g), max_col_per_gcell)
    end do
#endif

    garr(:,:) = spval
    do j = 1,num2d
       sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
       do ci = 1,max_col_per_gcell
!dir$ concurrent
!cdir nodep
          do g = lbg,ubg
             if (ci <= ncolumns(g)) then
                c = coli(g) + ci - 1
                if (wtgcell(c) /= 0._r8) then
                   if (carr(c,j) /= spval) then
                      garr(g,j) = 0._r8
                   end if
                end if
             end if
          end do
       end do
!dir$ nointerchange
       do ci = 1,max_col_per_gcell
!dir$ concurrent
!cdir nodep
          do g = lbg,ubg
             if (ci <= ncolumns(g)) then
                c = coli(g) + ci - 1
                if (wtgcell(c) /= 0._r8) then
                   if (carr(c,j) /= spval .and. scale_c2l(c) /= spval) then
                      l = clandunit(c)
                      garr(g,j) = garr(g,j) + carr(c,j) * scale_c2l(c) * scale_l2g(l) * wtgcell(c)
                      sumwt(g) = sumwt(g) + wtgcell(c)
                   end if
                end if
             end if
          end do
       end do
#else
       do c = lbc,ubc
          if (wtgcell(c) /= 0._r8) then
             if (carr(c,j) /= spval .and. scale_c2l(c) /= spval) then
                l = clandunit(c)
                g = cgridcell(c)
                if (sumwt(g) == 0._r8) garr(g,j) = 0._r8
                garr(g,j) = garr(g,j) + carr(c,j) * scale_c2l(c) * scale_l2g(l) * wtgcell(c)
                sumwt(g) = sumwt(g) + wtgcell(c)
             end if
          end if
       end do
#endif
       found = .false.
       do g = lbg, ubg
          if (sumwt(g) > 1.0_r8 + 1.e-6_r8) then
             found = .true.
             index = g
          else if (sumwt(g) /= 0._r8) then
             garr(g,j) = garr(g,j)/sumwt(g)
          end if
       end do
       if (found) then
          write(6,*)'c2g_2d error: sumwt is greater than 1.0 at g= ',index
          call endrun()
       end if
    end do

  end subroutine c2g_2d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: l2g_1d
!
! !INTERFACE:
  subroutine l2g_1d(lbl, ubl, lbg, ubg, larr, garr, l2g_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from landunits to gridcells.
! Averaging is only done for points that are not equal to "spval".
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbl, ubl       ! beginning and ending sub landunit indices
    integer , intent(in)  :: lbg, ubg       ! beginning and ending gridcell indices
    real(r8), intent(in)  :: larr(lbl:ubl)  ! input landunit array
    real(r8), intent(out) :: garr(lbg:ubg)  ! output gridcell array
    character(len=*), intent(in) :: l2g_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: li,l,g,index           ! indices
    integer  :: max_lu_per_gcell       ! max landunits per gridcell; on the fly
    logical  :: found                  ! temporary for error check
    real(r8) :: scale_l2g(lbl:ubl)     ! scale factor
    real(r8) :: sumwt(lbg:ubg)         ! sum of weights
    real(r8), pointer :: wtgcell(:)    ! weight of landunits relative to gridcells
    integer , pointer :: lgridcell(:)  ! gridcell of corresponding landunit
    integer , pointer :: nlandunits(:) ! number of landunits in gridcell
    integer , pointer :: luni(:)       ! initial landunit index in gridcell
!------------------------------------------------------------------------

    wtgcell    => clm3%g%l%wtgcell
    lgridcell  => clm3%g%l%gridcell
    nlandunits => clm3%g%nlandunits
    luni       => clm3%g%luni

    if (l2g_scale_type == 'unity') then
       do l = lbl,ubl
          scale_l2g(l) = 1.0_r8
       end do
    else
       write(6,*)'l2g_1d error: scale type ',l2g_scale_type,' not supported'
       call endrun()
    end if

    garr(:) = spval
    sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
    max_lu_per_gcell = 0
    do g = lbg,ubg
       max_lu_per_gcell = max(nlandunits(g), max_lu_per_gcell)
    end do
!dir$ nointerchange
    do li = 1,max_lu_per_gcell
!dir$ concurrent
!cdir nodep
       do g = lbg,ubg
          if (li <= nlandunits(g)) then
             l = luni(g) + li - 1
             if (wtgcell(l) /= 0._r8) then
                if (larr(l) /= spval) then
                   garr(g) = 0._r8
                end if
             end if
          end if
       end do
    end do
!dir$ nointerchange
    do li = 1,max_lu_per_gcell
!dir$ concurrent
!cdir nodep
       do g = lbg,ubg
          if (li <= nlandunits(g)) then
             l = luni(g) + li - 1
             if (wtgcell(l) /= 0._r8) then
                if (larr(l) /= spval) then
                   garr(g) = garr(g) + larr(l) * scale_l2g(l) * wtgcell(l)
                   sumwt(g) = sumwt(g) + wtgcell(l)
                end if
             end if
          end if
       end do
    end do
#else
    do l = lbl,ubl
       if (wtgcell(l) /= 0._r8) then
          if (larr(l) /= spval) then
             g = lgridcell(l)
             if (sumwt(g) == 0._r8) garr(g) = 0._r8
             garr(g) = garr(g) + larr(l) * scale_l2g(l) * wtgcell(l)
             sumwt(g) = sumwt(g) + wtgcell(l)
          end if
       end if
    end do
#endif
    found = .false.
    do g = lbg, ubg
       if (sumwt(g) > 1.0_r8 + 1.e-6_r8) then
          found = .true.
          index = g
       else if (sumwt(g) /= 0._r8) then
          garr(g) = garr(g)/sumwt(g)
       end if
    end do
    if (found) then
       write(6,*)'l2g_1d error: sumwt is greater than 1.0 at g= ',index
       call endrun()
    end if

  end subroutine l2g_1d

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: l2g_2d
!
! !INTERFACE:
  subroutine l2g_2d(lbl, ubl, lbg, ubg, num2d, larr, garr, l2g_scale_type)
!
! !DESCRIPTION:
! Perfrom subgrid-average from landunits to gridcells.
! Averaging is only done for points that are not equal to "spval".
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: lbl, ubl             ! beginning and ending column indices
    integer , intent(in)  :: lbg, ubg             ! beginning and ending gridcell indices
    integer , intent(in)  :: num2d                ! size of second dimension
    real(r8), intent(in)  :: larr(lbl:ubl,num2d)  ! input landunit array
    real(r8), intent(out) :: garr(lbg:ubg,num2d)  ! output gridcell array
    character(len=*), intent(in) :: l2g_scale_type ! scale factor type for averaging
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein 12/03
!
!
! !LOCAL VARIABLES:
!EOP
    integer  :: j,g,li,l,index         ! indices
    integer  :: max_lu_per_gcell       ! max landunits per gridcell; on the fly
    logical  :: found                  ! temporary for error check
    real(r8) :: scale_l2g(lbl:ubl)     ! scale factor
    real(r8) :: sumwt(lbg:ubg)         ! sum of weights
    real(r8), pointer :: wtgcell(:)    ! weight of landunits relative to gridcells
    integer , pointer :: lgridcell(:)  ! gridcell of corresponding landunit
    integer , pointer :: nlandunits(:) ! number of landunits in gridcell
    integer , pointer :: luni(:)       ! initial landunit index in gridcell
!------------------------------------------------------------------------

    wtgcell   => clm3%g%l%wtgcell
    lgridcell => clm3%g%l%gridcell
    nlandunits => clm3%g%nlandunits
    luni       => clm3%g%luni

    if (l2g_scale_type == 'unity') then
       do l = lbl,ubl
          scale_l2g(l) = 1.0_r8
       end do
    else
       write(6,*)'l2g_2d error: scale type ',l2g_scale_type,' not supported'
       call endrun()
    end if

#if (defined CPP_VECTOR)
    max_lu_per_gcell = 0
    do g = lbg,ubg
       max_lu_per_gcell = max(nlandunits(g), max_lu_per_gcell)
    end do
#endif

    garr(:,:) = spval
    do j = 1,num2d
       sumwt(:) = 0._r8
#if (defined CPP_VECTOR)
!dir$ nointerchange
       do li = 1,max_lu_per_gcell
!dir$ concurrent
!cdir nodep
          do g = lbg,ubg
             if (li <= nlandunits(g)) then
                l = luni(g) + li - 1
                if (wtgcell(l) /= 0._r8) then
                   if (larr(l,j) /= spval) then
                      garr(g,j) = 0._r8
                   end if
                end if
             end if
          end do
       end do
!dir$ nointerchange
       do li = 1,max_lu_per_gcell
!dir$ concurrent
!cdir nodep
          do g = lbg,ubg
             if (li <= nlandunits(g)) then
                l = luni(g) + li - 1
                if (wtgcell(l) /= 0._r8) then
                   if (larr(l,j) /= spval) then
                      garr(g,j) = garr(g,j) + larr(l,j) * scale_l2g(l) * wtgcell(l)
                      sumwt(g) = sumwt(g) + wtgcell(l)
                   end if
                end if
             end if
          end do
       end do
#else
       do l = lbl,ubl
          if (wtgcell(l) /= 0._r8) then
             if (larr(l,j) /= spval) then
                g = lgridcell(l)
                if (sumwt(g) == 0._r8) garr(g,j) = 0._r8
                garr(g,j) = garr(g,j) + larr(l,j) * scale_l2g(l) * wtgcell(l)
                sumwt(g) = sumwt(g) + wtgcell(l)
             end if
          end if
       end do
#endif
       found = .false.
       do g = lbg,ubg
          if (sumwt(g) > 1.0_r8 + 1.e-6_r8) then
             found = .true.
             index= g
          else if (sumwt(g) /= 0._r8) then
             garr(g,j) = garr(g,j)/sumwt(g)
          end if
       end do
       if (found) then
          write(6,*)'l2g_2d error: sumwt is greater than 1.0 at g= ',index,' lev= ',j
          call endrun()
       end if
    end do

  end subroutine l2g_2d

end module subgridAveMod
module pft2colMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: pft2colMod
!
! !DESCRIPTION:
! Contains calls to methods to perfom averages over from pfts to columns
! for model variables.
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
  use subgridAveMod
  use clmtype
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: p2c  ! obtain column properties from average over column pfts
!
! !REVISION HISTORY:
! 03/09/08: Created by Mariana Vertenstein
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: pft2col
!
! !INTERFACE:
  subroutine pft2col (lbc, ubc, num_nolakec, filter_nolakec)
!
! !DESCRIPTION:
! Averages over all pfts for variables defined over both soil and lake
! to provide the column-level averages of state and flux variables
! defined at the pft level.
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: lbc, ubc                    ! column bounds
    integer, intent(in) :: num_nolakec                 ! number of column non-lake points in column filter
    integer, intent(in) :: filter_nolakec(ubc-lbc+1)   ! column filter for non-lake points
!
! !REVISION HISTORY:
! 03/09/08: Created by Mariana Vertenstein
!
!EOP
!
! !OTHER LOCAL VARIABLES:
    integer :: c,fc                      ! indices
    integer :: num_allc                  ! number of total column points
    integer :: filter_allc(ubc-lbc+1)    ! filter for all column points
    real(r8), pointer :: ptrp(:)         ! pointer to input pft array
    real(r8), pointer :: ptrc(:)         ! pointer to output column array
! -----------------------------------------------------------------

    ! Set up a filter for all column points

    num_allc = ubc-lbc+1
    fc = 0
    do c = lbc,ubc
       fc = fc + 1
       filter_allc(fc) = c
    end do

    ! Averaging for pft water state variables

    ptrp => clm3%g%l%c%p%pws%h2ocan
    ptrc => clm3%g%l%c%cws%pws_a%h2ocan
    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)

    ! Averaging for pft water flux variables

    ptrp => clm3%g%l%c%p%pwf%qflx_evap_tot
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_evap_tot
    call p2c (num_allc, filter_allc, ptrp, ptrc)

    ptrp => clm3%g%l%c%p%pwf%qflx_rain_grnd
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_rain_grnd
    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)

    ptrp => clm3%g%l%c%p%pwf%qflx_snow_grnd
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_snow_grnd
    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)

!    ptrp => clm3%g%l%c%p%pwf%qflx_snowcap
!    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_snowcap
!    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)
!CLM4 
    ptrp => clm3%g%l%c%p%pwf%qflx_snwcp_liq
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_snwcp_liq
    call p2c (num_allc, filter_allc, ptrp, ptrc)

    ptrp => clm3%g%l%c%p%pwf%qflx_snwcp_ice
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_snwcp_ice
    call p2c (num_allc, filter_allc, ptrp, ptrc)
!!!

    ptrp => clm3%g%l%c%p%pwf%qflx_tran_veg
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_tran_veg
    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)

    ptrp => clm3%g%l%c%p%pwf%qflx_evap_grnd
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_evap_grnd
    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)

    ptrp => clm3%g%l%c%p%pwf%qflx_dew_grnd
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_dew_grnd
    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)

    ptrp => clm3%g%l%c%p%pwf%qflx_sub_snow
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_sub_snow
    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)

    ptrp => clm3%g%l%c%p%pwf%qflx_dew_snow
    ptrc => clm3%g%l%c%cwf%pwf_a%qflx_dew_snow
    call p2c (num_nolakec, filter_nolakec, ptrp, ptrc)

  end subroutine pft2col

end module pft2colMod
!===============================================================================
! SVN $Id: shr_orb_mod.F90 6752 2007-10-04 21:02:15Z jwolfe $
! SVN $URL: https://svn-ccsm-models.cgd.ucar.edu/csm_share/trunk_tags/share3_090706b/shr/shr_orb_mod.F90 $
!===============================================================================

MODULE shr_orb_mod

   use shr_kind_mod
   use shr_const_mod

   IMPLICIT none

   !----------------------------------------------------------------------------
   ! PUBLIC: Interfaces and global data
   !----------------------------------------------------------------------------
   public :: shr_orb_cosz
   public :: shr_orb_params
   public :: shr_orb_decl
!   public :: shr_orb_print

   real   (SHR_KIND_R8),public,parameter :: SHR_ORB_UNDEF_REAL = 1.e36_SHR_KIND_R8 ! undefined real 
   integer,public,parameter :: SHR_ORB_UNDEF_INT  = 2000000000        ! undefined int

   !----------------------------------------------------------------------------
   ! PRIVATE: by default everything else is private to this module
   !----------------------------------------------------------------------------
   private

   real   (SHR_KIND_R8),parameter :: pi                 = SHR_CONST_PI
   real   (SHR_KIND_R8),parameter :: SHR_ORB_ECCEN_MIN  =   0.0_SHR_KIND_R8 ! min value for eccen
   real   (SHR_KIND_R8),parameter :: SHR_ORB_ECCEN_MAX  =   0.1_SHR_KIND_R8 ! max value for eccen
   real   (SHR_KIND_R8),parameter :: SHR_ORB_OBLIQ_MIN  = -90.0_SHR_KIND_R8 ! min value for obliq
   real   (SHR_KIND_R8),parameter :: SHR_ORB_OBLIQ_MAX  = +90.0_SHR_KIND_R8 ! max value for obliq
   real   (SHR_KIND_R8),parameter :: SHR_ORB_MVELP_MIN  =   0.0_SHR_KIND_R8 ! min value for mvelp
   real   (SHR_KIND_R8),parameter :: SHR_ORB_MVELP_MAX  = 360.0_SHR_KIND_R8 ! max value for mvelp

!===============================================================================
CONTAINS
!===============================================================================

real(SHR_KIND_R8) FUNCTION shr_orb_cosz(jday,lat,lon,declin)

   !----------------------------------------------------------------------------
   !
   ! FUNCTION to return the cosine of the solar zenith angle.
   ! Assumes 365.0 days/year.
   !
   !--------------- Code History -----------------------------------------------
   !
   ! Original Author: Brian Kauffman
   ! Date:            Jan/98
   ! History:         adapted from statement FUNCTION in share/orb_cosz.h
   !
   !----------------------------------------------------------------------------

   real   (SHR_KIND_R8),intent(in) :: jday   ! Julian cal day (1.xx to 365.xx)
   real   (SHR_KIND_R8),intent(in) :: lat    ! Centered latitude (radians)
   real   (SHR_KIND_R8),intent(in) :: lon    ! Centered longitude (radians)
   real   (SHR_KIND_R8),intent(in) :: declin ! Solar declination (radians)

   !----------------------------------------------------------------------------

   shr_orb_cosz = sin(lat)*sin(declin) - &
   &              cos(lat)*cos(declin)*cos(jday*2.0_SHR_KIND_R8*pi + lon)

END FUNCTION shr_orb_cosz

!===============================================================================

SUBROUTINE shr_orb_params( iyear_AD , eccen  , obliq , mvelp     ,     &
           &               obliqr   , lambm0 , mvelpp)

!-------------------------------------------------------------------------------
!
! Calculate earths orbital parameters using Dave Threshers formula which 
! came from Berger, Andre.  1978  "A Simple Algorithm to Compute Long-Term 
! Variations of Daily Insolation".  Contribution 18, Institute of Astronomy 
! and Geophysics, Universite Catholique de Louvain, Louvain-la-Neuve, Belgium
!
!------------------------------Code history-------------------------------------
!
! Original Author: Erik Kluzek
! Date:            Oct/97
!
!-------------------------------------------------------------------------------

   !----------------------------- Arguments ------------------------------------
   real   (SHR_KIND_R8),intent(inout) :: eccen     ! orbital eccentricity
   real   (SHR_KIND_R8),intent(inout) :: obliq     ! obliquity in degrees
   real   (SHR_KIND_R8),intent(inout) :: mvelp     ! moving vernal equinox long
   integer,intent(in)    :: iyear_AD  ! Year to calculate orbit for
   real   (SHR_KIND_R8),intent(out)   :: obliqr    ! Earths obliquity in rad
   real   (SHR_KIND_R8),intent(out)   :: lambm0    ! Mean long of perihelion at
                                                   ! vernal equinox (radians)
   real   (SHR_KIND_R8),intent(out)   :: mvelpp    ! moving vernal equinox long
                                                   ! of perihelion plus pi (rad)

   !------------------------------ Parameters ----------------------------------
   integer,parameter :: poblen =47 ! # of elements in series wrt obliquity
   integer,parameter :: pecclen=19 ! # of elements in series wrt eccentricity
   integer,parameter :: pmvelen=78 ! # of elements in series wrt vernal equinox
   real   (SHR_KIND_R8),parameter :: psecdeg = 1.0_SHR_KIND_R8/3600.0_SHR_KIND_R8 ! arc sec to deg conversion

   real   (SHR_KIND_R8) :: degrad = pi/180._SHR_KIND_R8   ! degree to radian conversion factor
   real   (SHR_KIND_R8) :: yb4_1950AD         ! number of years before 1950 AD
 
   ! Cosine series data for computation of obliquity: amplitude (arc seconds),
   ! rate (arc seconds/year), phase (degrees).
 
   real   (SHR_KIND_R8), parameter :: obamp(poblen) =  & ! amplitudes for obliquity cos series
   &      (/   -2462.2214466_SHR_KIND_R8, -857.3232075_SHR_KIND_R8, -629.3231835_SHR_KIND_R8,   &
   &            -414.2804924_SHR_KIND_R8, -311.7632587_SHR_KIND_R8,  308.9408604_SHR_KIND_R8,   &
   &            -162.5533601_SHR_KIND_R8, -116.1077911_SHR_KIND_R8,  101.1189923_SHR_KIND_R8,   &
   &             -67.6856209_SHR_KIND_R8,   24.9079067_SHR_KIND_R8,   22.5811241_SHR_KIND_R8,   &
   &             -21.1648355_SHR_KIND_R8,  -15.6549876_SHR_KIND_R8,   15.3936813_SHR_KIND_R8,   &
   &              14.6660938_SHR_KIND_R8,  -11.7273029_SHR_KIND_R8,   10.2742696_SHR_KIND_R8,   &
   &               6.4914588_SHR_KIND_R8,    5.8539148_SHR_KIND_R8,   -5.4872205_SHR_KIND_R8,   &
   &              -5.4290191_SHR_KIND_R8,    5.1609570_SHR_KIND_R8,    5.0786314_SHR_KIND_R8,   &
   &              -4.0735782_SHR_KIND_R8,    3.7227167_SHR_KIND_R8,    3.3971932_SHR_KIND_R8,   &
   &              -2.8347004_SHR_KIND_R8,   -2.6550721_SHR_KIND_R8,   -2.5717867_SHR_KIND_R8,   &
   &              -2.4712188_SHR_KIND_R8,    2.4625410_SHR_KIND_R8,    2.2464112_SHR_KIND_R8,   &
   &              -2.0755511_SHR_KIND_R8,   -1.9713669_SHR_KIND_R8,   -1.8813061_SHR_KIND_R8,   &
   &              -1.8468785_SHR_KIND_R8,    1.8186742_SHR_KIND_R8,    1.7601888_SHR_KIND_R8,   &
   &              -1.5428851_SHR_KIND_R8,    1.4738838_SHR_KIND_R8,   -1.4593669_SHR_KIND_R8,   &
   &               1.4192259_SHR_KIND_R8,   -1.1818980_SHR_KIND_R8,    1.1756474_SHR_KIND_R8,   &
   &              -1.1316126_SHR_KIND_R8,    1.0896928_SHR_KIND_R8/)
 
   real   (SHR_KIND_R8), parameter :: obrate(poblen) = & ! rates for obliquity cosine series
   &        (/  31.609974_SHR_KIND_R8, 32.620504_SHR_KIND_R8, 24.172203_SHR_KIND_R8,   &
   &            31.983787_SHR_KIND_R8, 44.828336_SHR_KIND_R8, 30.973257_SHR_KIND_R8,   &
   &            43.668246_SHR_KIND_R8, 32.246691_SHR_KIND_R8, 30.599444_SHR_KIND_R8,   &
   &            42.681324_SHR_KIND_R8, 43.836462_SHR_KIND_R8, 47.439436_SHR_KIND_R8,   &
   &            63.219948_SHR_KIND_R8, 64.230478_SHR_KIND_R8,  1.010530_SHR_KIND_R8,   &
   &             7.437771_SHR_KIND_R8, 55.782177_SHR_KIND_R8,  0.373813_SHR_KIND_R8,   &
   &            13.218362_SHR_KIND_R8, 62.583231_SHR_KIND_R8, 63.593761_SHR_KIND_R8,   &
   &            76.438310_SHR_KIND_R8, 45.815258_SHR_KIND_R8,  8.448301_SHR_KIND_R8,   &
   &            56.792707_SHR_KIND_R8, 49.747842_SHR_KIND_R8, 12.058272_SHR_KIND_R8,   &
   &            75.278220_SHR_KIND_R8, 65.241008_SHR_KIND_R8, 64.604291_SHR_KIND_R8,   &
   &             1.647247_SHR_KIND_R8,  7.811584_SHR_KIND_R8, 12.207832_SHR_KIND_R8,   &
   &            63.856665_SHR_KIND_R8, 56.155990_SHR_KIND_R8, 77.448840_SHR_KIND_R8,   &
   &             6.801054_SHR_KIND_R8, 62.209418_SHR_KIND_R8, 20.656133_SHR_KIND_R8,   &
   &            48.344406_SHR_KIND_R8, 55.145460_SHR_KIND_R8, 69.000539_SHR_KIND_R8,   &
   &            11.071350_SHR_KIND_R8, 74.291298_SHR_KIND_R8, 11.047742_SHR_KIND_R8,   &
   &             0.636717_SHR_KIND_R8, 12.844549_SHR_KIND_R8/)
 
   real   (SHR_KIND_R8), parameter :: obphas(poblen) = & ! phases for obliquity cosine series
   &      (/    251.9025_SHR_KIND_R8, 280.8325_SHR_KIND_R8, 128.3057_SHR_KIND_R8,   &
   &            292.7252_SHR_KIND_R8,  15.3747_SHR_KIND_R8, 263.7951_SHR_KIND_R8,   &
   &            308.4258_SHR_KIND_R8, 240.0099_SHR_KIND_R8, 222.9725_SHR_KIND_R8,   &
   &            268.7809_SHR_KIND_R8, 316.7998_SHR_KIND_R8, 319.6024_SHR_KIND_R8,   &
   &            143.8050_SHR_KIND_R8, 172.7351_SHR_KIND_R8,  28.9300_SHR_KIND_R8,   &
   &            123.5968_SHR_KIND_R8,  20.2082_SHR_KIND_R8,  40.8226_SHR_KIND_R8,   &
   &            123.4722_SHR_KIND_R8, 155.6977_SHR_KIND_R8, 184.6277_SHR_KIND_R8,   &
   &            267.2772_SHR_KIND_R8,  55.0196_SHR_KIND_R8, 152.5268_SHR_KIND_R8,   &
   &             49.1382_SHR_KIND_R8, 204.6609_SHR_KIND_R8,  56.5233_SHR_KIND_R8,   &
   &            200.3284_SHR_KIND_R8, 201.6651_SHR_KIND_R8, 213.5577_SHR_KIND_R8,   &
   &             17.0374_SHR_KIND_R8, 164.4194_SHR_KIND_R8,  94.5422_SHR_KIND_R8,   &
   &            131.9124_SHR_KIND_R8,  61.0309_SHR_KIND_R8, 296.2073_SHR_KIND_R8,   &
   &            135.4894_SHR_KIND_R8, 114.8750_SHR_KIND_R8, 247.0691_SHR_KIND_R8,   &
   &            256.6114_SHR_KIND_R8,  32.1008_SHR_KIND_R8, 143.6804_SHR_KIND_R8,   &
   &             16.8784_SHR_KIND_R8, 160.6835_SHR_KIND_R8,  27.5932_SHR_KIND_R8,   &
   &            348.1074_SHR_KIND_R8,  82.6496_SHR_KIND_R8/)
 
   ! Cosine/sine series data for computation of eccentricity and fixed vernal 
   ! equinox longitude of perihelion (fvelp): amplitude, 
   ! rate (arc seconds/year), phase (degrees).
 
   real   (SHR_KIND_R8), parameter :: ecamp (pecclen) = & ! ampl for eccen/fvelp cos/sin series
   &      (/   0.01860798_SHR_KIND_R8,  0.01627522_SHR_KIND_R8, -0.01300660_SHR_KIND_R8,   &
   &           0.00988829_SHR_KIND_R8, -0.00336700_SHR_KIND_R8,  0.00333077_SHR_KIND_R8,   &
   &          -0.00235400_SHR_KIND_R8,  0.00140015_SHR_KIND_R8,  0.00100700_SHR_KIND_R8,   &
   &           0.00085700_SHR_KIND_R8,  0.00064990_SHR_KIND_R8,  0.00059900_SHR_KIND_R8,   &
   &           0.00037800_SHR_KIND_R8, -0.00033700_SHR_KIND_R8,  0.00027600_SHR_KIND_R8,   &
   &           0.00018200_SHR_KIND_R8, -0.00017400_SHR_KIND_R8, -0.00012400_SHR_KIND_R8,   &
   &           0.00001250_SHR_KIND_R8/)
 
   real   (SHR_KIND_R8), parameter :: ecrate(pecclen) = & ! rates for eccen/fvelp cos/sin series
   &      (/    4.2072050_SHR_KIND_R8,  7.3460910_SHR_KIND_R8, 17.8572630_SHR_KIND_R8,  &
   &           17.2205460_SHR_KIND_R8, 16.8467330_SHR_KIND_R8,  5.1990790_SHR_KIND_R8,  &
   &           18.2310760_SHR_KIND_R8, 26.2167580_SHR_KIND_R8,  6.3591690_SHR_KIND_R8,  &
   &           16.2100160_SHR_KIND_R8,  3.0651810_SHR_KIND_R8, 16.5838290_SHR_KIND_R8,  &
   &           18.4939800_SHR_KIND_R8,  6.1909530_SHR_KIND_R8, 18.8677930_SHR_KIND_R8,  &
   &           17.4255670_SHR_KIND_R8,  6.1860010_SHR_KIND_R8, 18.4174410_SHR_KIND_R8,  &
   &            0.6678630_SHR_KIND_R8/)
 
   real   (SHR_KIND_R8), parameter :: ecphas(pecclen) = & ! phases for eccen/fvelp cos/sin series
   &      (/    28.620089_SHR_KIND_R8, 193.788772_SHR_KIND_R8, 308.307024_SHR_KIND_R8,  &
   &           320.199637_SHR_KIND_R8, 279.376984_SHR_KIND_R8,  87.195000_SHR_KIND_R8,  &
   &           349.129677_SHR_KIND_R8, 128.443387_SHR_KIND_R8, 154.143880_SHR_KIND_R8,  &
   &           291.269597_SHR_KIND_R8, 114.860583_SHR_KIND_R8, 332.092251_SHR_KIND_R8,  &
   &           296.414411_SHR_KIND_R8, 145.769910_SHR_KIND_R8, 337.237063_SHR_KIND_R8,  &
   &           152.092288_SHR_KIND_R8, 126.839891_SHR_KIND_R8, 210.667199_SHR_KIND_R8,  &
   &            72.108838_SHR_KIND_R8/)
 
   ! Sine series data for computation of moving vernal equinox longitude of 
   ! perihelion: amplitude (arc seconds), rate (arc sec/year), phase (degrees).      
 
   real   (SHR_KIND_R8), parameter :: mvamp (pmvelen) = & ! amplitudes for mvelp sine series 
   &      (/   7391.0225890_SHR_KIND_R8, 2555.1526947_SHR_KIND_R8, 2022.7629188_SHR_KIND_R8,  &
   &          -1973.6517951_SHR_KIND_R8, 1240.2321818_SHR_KIND_R8,  953.8679112_SHR_KIND_R8,  &
   &           -931.7537108_SHR_KIND_R8,  872.3795383_SHR_KIND_R8,  606.3544732_SHR_KIND_R8,  &
   &           -496.0274038_SHR_KIND_R8,  456.9608039_SHR_KIND_R8,  346.9462320_SHR_KIND_R8,  &
   &           -305.8412902_SHR_KIND_R8,  249.6173246_SHR_KIND_R8, -199.1027200_SHR_KIND_R8,  &
   &            191.0560889_SHR_KIND_R8, -175.2936572_SHR_KIND_R8,  165.9068833_SHR_KIND_R8,  &
   &            161.1285917_SHR_KIND_R8,  139.7878093_SHR_KIND_R8, -133.5228399_SHR_KIND_R8,  &
   &            117.0673811_SHR_KIND_R8,  104.6907281_SHR_KIND_R8,   95.3227476_SHR_KIND_R8,  &
   &             86.7824524_SHR_KIND_R8,   86.0857729_SHR_KIND_R8,   70.5893698_SHR_KIND_R8,  &
   &            -69.9719343_SHR_KIND_R8,  -62.5817473_SHR_KIND_R8,   61.5450059_SHR_KIND_R8,  &
   &            -57.9364011_SHR_KIND_R8,   57.1899832_SHR_KIND_R8,  -57.0236109_SHR_KIND_R8,  &
   &            -54.2119253_SHR_KIND_R8,   53.2834147_SHR_KIND_R8,   52.1223575_SHR_KIND_R8,  &
   &            -49.0059908_SHR_KIND_R8,  -48.3118757_SHR_KIND_R8,  -45.4191685_SHR_KIND_R8,  &
   &            -42.2357920_SHR_KIND_R8,  -34.7971099_SHR_KIND_R8,   34.4623613_SHR_KIND_R8,  &
   &            -33.8356643_SHR_KIND_R8,   33.6689362_SHR_KIND_R8,  -31.2521586_SHR_KIND_R8,  &
   &            -30.8798701_SHR_KIND_R8,   28.4640769_SHR_KIND_R8,  -27.1960802_SHR_KIND_R8,  &
   &             27.0860736_SHR_KIND_R8,  -26.3437456_SHR_KIND_R8,   24.7253740_SHR_KIND_R8,  &
   &             24.6732126_SHR_KIND_R8,   24.4272733_SHR_KIND_R8,   24.0127327_SHR_KIND_R8,  &
   &             21.7150294_SHR_KIND_R8,  -21.5375347_SHR_KIND_R8,   18.1148363_SHR_KIND_R8,  &
   &            -16.9603104_SHR_KIND_R8,  -16.1765215_SHR_KIND_R8,   15.5567653_SHR_KIND_R8,  &
   &             15.4846529_SHR_KIND_R8,   15.2150632_SHR_KIND_R8,   14.5047426_SHR_KIND_R8,  &
   &            -14.3873316_SHR_KIND_R8,   13.1351419_SHR_KIND_R8,   12.8776311_SHR_KIND_R8,  &
   &             11.9867234_SHR_KIND_R8,   11.9385578_SHR_KIND_R8,   11.7030822_SHR_KIND_R8,  &
   &             11.6018181_SHR_KIND_R8,  -11.2617293_SHR_KIND_R8,  -10.4664199_SHR_KIND_R8,  &
   &             10.4333970_SHR_KIND_R8,  -10.2377466_SHR_KIND_R8,   10.1934446_SHR_KIND_R8,  &
   &            -10.1280191_SHR_KIND_R8,   10.0289441_SHR_KIND_R8,  -10.0034259_SHR_KIND_R8/)
 
   real   (SHR_KIND_R8), parameter :: mvrate(pmvelen) = & ! rates for mvelp sine series 
   &      (/    31.609974_SHR_KIND_R8, 32.620504_SHR_KIND_R8, 24.172203_SHR_KIND_R8,   &
   &             0.636717_SHR_KIND_R8, 31.983787_SHR_KIND_R8,  3.138886_SHR_KIND_R8,   &
   &            30.973257_SHR_KIND_R8, 44.828336_SHR_KIND_R8,  0.991874_SHR_KIND_R8,   &
   &             0.373813_SHR_KIND_R8, 43.668246_SHR_KIND_R8, 32.246691_SHR_KIND_R8,   &
   &            30.599444_SHR_KIND_R8,  2.147012_SHR_KIND_R8, 10.511172_SHR_KIND_R8,   &
   &            42.681324_SHR_KIND_R8, 13.650058_SHR_KIND_R8,  0.986922_SHR_KIND_R8,   &
   &             9.874455_SHR_KIND_R8, 13.013341_SHR_KIND_R8,  0.262904_SHR_KIND_R8,   &
   &             0.004952_SHR_KIND_R8,  1.142024_SHR_KIND_R8, 63.219948_SHR_KIND_R8,   &
   &             0.205021_SHR_KIND_R8,  2.151964_SHR_KIND_R8, 64.230478_SHR_KIND_R8,   &
   &            43.836462_SHR_KIND_R8, 47.439436_SHR_KIND_R8,  1.384343_SHR_KIND_R8,   &
   &             7.437771_SHR_KIND_R8, 18.829299_SHR_KIND_R8,  9.500642_SHR_KIND_R8,   &
   &             0.431696_SHR_KIND_R8,  1.160090_SHR_KIND_R8, 55.782177_SHR_KIND_R8,   &
   &            12.639528_SHR_KIND_R8,  1.155138_SHR_KIND_R8,  0.168216_SHR_KIND_R8,   &
   &             1.647247_SHR_KIND_R8, 10.884985_SHR_KIND_R8,  5.610937_SHR_KIND_R8,   &
   &            12.658184_SHR_KIND_R8,  1.010530_SHR_KIND_R8,  1.983748_SHR_KIND_R8,   &
   &            14.023871_SHR_KIND_R8,  0.560178_SHR_KIND_R8,  1.273434_SHR_KIND_R8,   &
   &            12.021467_SHR_KIND_R8, 62.583231_SHR_KIND_R8, 63.593761_SHR_KIND_R8,   &
   &            76.438310_SHR_KIND_R8,  4.280910_SHR_KIND_R8, 13.218362_SHR_KIND_R8,   &
   &            17.818769_SHR_KIND_R8,  8.359495_SHR_KIND_R8, 56.792707_SHR_KIND_R8,   &
   &            8.448301_SHR_KIND_R8,  1.978796_SHR_KIND_R8,  8.863925_SHR_KIND_R8,   &
   &             0.186365_SHR_KIND_R8,  8.996212_SHR_KIND_R8,  6.771027_SHR_KIND_R8,   &
   &            45.815258_SHR_KIND_R8, 12.002811_SHR_KIND_R8, 75.278220_SHR_KIND_R8,   &
   &            65.241008_SHR_KIND_R8, 18.870667_SHR_KIND_R8, 22.009553_SHR_KIND_R8,   &
   &            64.604291_SHR_KIND_R8, 11.498094_SHR_KIND_R8,  0.578834_SHR_KIND_R8,   &
   &             9.237738_SHR_KIND_R8, 49.747842_SHR_KIND_R8,  2.147012_SHR_KIND_R8,   &
   &             1.196895_SHR_KIND_R8,  2.133898_SHR_KIND_R8,  0.173168_SHR_KIND_R8/)

   real   (SHR_KIND_R8), parameter :: mvphas(pmvelen) = & ! phases for mvelp sine series
   &      (/    251.9025_SHR_KIND_R8, 280.8325_SHR_KIND_R8, 128.3057_SHR_KIND_R8,   &
   &            348.1074_SHR_KIND_R8, 292.7252_SHR_KIND_R8, 165.1686_SHR_KIND_R8,   &
   &            263.7951_SHR_KIND_R8,  15.3747_SHR_KIND_R8,  58.5749_SHR_KIND_R8,   &
   &             40.8226_SHR_KIND_R8, 308.4258_SHR_KIND_R8, 240.0099_SHR_KIND_R8,   &
   &            222.9725_SHR_KIND_R8, 106.5937_SHR_KIND_R8, 114.5182_SHR_KIND_R8,   &
   &            268.7809_SHR_KIND_R8, 279.6869_SHR_KIND_R8,  39.6448_SHR_KIND_R8,   &
   &            126.4108_SHR_KIND_R8, 291.5795_SHR_KIND_R8, 307.2848_SHR_KIND_R8,   &
   &             18.9300_SHR_KIND_R8, 273.7596_SHR_KIND_R8, 143.8050_SHR_KIND_R8,   &
   &            191.8927_SHR_KIND_R8, 125.5237_SHR_KIND_R8, 172.7351_SHR_KIND_R8,   &
   &            316.7998_SHR_KIND_R8, 319.6024_SHR_KIND_R8,  69.7526_SHR_KIND_R8,   &
   &            123.5968_SHR_KIND_R8, 217.6432_SHR_KIND_R8,  85.5882_SHR_KIND_R8,   &
   &            156.2147_SHR_KIND_R8,  66.9489_SHR_KIND_R8,  20.2082_SHR_KIND_R8,   &
   &            250.7568_SHR_KIND_R8,  48.0188_SHR_KIND_R8,   8.3739_SHR_KIND_R8,   &
   &             17.0374_SHR_KIND_R8, 155.3409_SHR_KIND_R8,  94.1709_SHR_KIND_R8,   &
   &            221.1120_SHR_KIND_R8,  28.9300_SHR_KIND_R8, 117.1498_SHR_KIND_R8,   &
   &            320.5095_SHR_KIND_R8, 262.3602_SHR_KIND_R8, 336.2148_SHR_KIND_R8,   &
   &            233.0046_SHR_KIND_R8, 155.6977_SHR_KIND_R8, 184.6277_SHR_KIND_R8,   &
   &            267.2772_SHR_KIND_R8,  78.9281_SHR_KIND_R8, 123.4722_SHR_KIND_R8,   &
   &            188.7132_SHR_KIND_R8, 180.1364_SHR_KIND_R8,  49.1382_SHR_KIND_R8,   &
   &            152.5268_SHR_KIND_R8,  98.2198_SHR_KIND_R8,  97.4808_SHR_KIND_R8,   &
   &            221.5376_SHR_KIND_R8, 168.2438_SHR_KIND_R8, 161.1199_SHR_KIND_R8,   &
   &             55.0196_SHR_KIND_R8, 262.6495_SHR_KIND_R8, 200.3284_SHR_KIND_R8,   &
   &            201.6651_SHR_KIND_R8, 294.6547_SHR_KIND_R8,  99.8233_SHR_KIND_R8,   &
   &            213.5577_SHR_KIND_R8, 154.1631_SHR_KIND_R8, 232.7153_SHR_KIND_R8,   &
   &            138.3034_SHR_KIND_R8, 204.6609_SHR_KIND_R8, 106.5938_SHR_KIND_R8,   &
   &            250.4676_SHR_KIND_R8, 332.3345_SHR_KIND_R8,  27.3039_SHR_KIND_R8/)
 
   !---------------------------Local variables----------------------------------
   integer :: i       ! Index for series summations
   real   (SHR_KIND_R8) :: obsum   ! Obliquity series summation
   real   (SHR_KIND_R8) :: cossum  ! Cos series summation for eccentricity/fvelp
   real   (SHR_KIND_R8) :: sinsum  ! Sin series summation for eccentricity/fvelp
   real   (SHR_KIND_R8) :: fvelp   ! Fixed vernal equinox long of perihelion
   real   (SHR_KIND_R8) :: mvsum   ! mvelp series summation
   real   (SHR_KIND_R8) :: beta    ! Intermediate argument for lambm0
   real   (SHR_KIND_R8) :: years   ! Years to time of interest ( pos <=> future)
   real   (SHR_KIND_R8) :: eccen2  ! eccentricity squared
   real   (SHR_KIND_R8) :: eccen3  ! eccentricity cubed

   !-------------------------- Formats -----------------------------------------
!   character(*),parameter :: svnID  = "SVN " // &
!   "$Id: shr_orb_mod.F90 6752 2007-10-04 21:02:15Z jwolfe $"
!   character(*),parameter :: svnURL = "SVN <unknown URL>" 
!  character(*),parameter :: svnURL = "SVN " // &
!  "$URL: https://svn-ccsm-models.cgd.ucar.edu/csm_share/trunk_tags/share3_090706b/shr/shr_orb_mod.F90 $"
!   character(len=*),parameter :: F00 = "('(shr_orb_params) ',4a)"
!   character(len=*),parameter :: F01 = "('(shr_orb_params) ',a,i9)"
!   character(len=*),parameter :: F02 = "('(shr_orb_params) ',a,f6.3)"
!   character(len=*),parameter :: F03 = "('(shr_orb_params) ',a,es14.6)"

   !----------------------------------------------------------------------------
   ! radinp and algorithms below will need a degree to radian conversion factor
 
!   if ( log_print .and. s_loglev > 0 ) then
!     write(s_logunit,F00) 'Calculate characteristics of the orbit:'
!     write(s_logunit,F00) svnID
!     write(s_logunit,F00) svnURL
!   end if
 
   ! Check for flag to use input orbit parameters
 
!   IF ( iyear_AD == SHR_ORB_UNDEF_INT ) THEN

      ! Check input obliq, eccen, and mvelp to ensure reasonable
 
!      if( obliq == SHR_ORB_UNDEF_REAL )then
!         write(s_logunit,F00) 'Have to specify orbital parameters:'
!         write(s_logunit,F00) 'Either set: iyear_AD, OR [obliq, eccen, and mvelp]:'
!         write(s_logunit,F00) 'iyear_AD is the year to simulate orbit for (ie. 1950): '
!         write(s_logunit,F00) 'obliq, eccen, mvelp specify the orbit directly:'
!         write(s_logunit,F00) 'The AMIP II settings (for a 1995 orbit) are: '
!         write(s_logunit,F00) ' obliq =  23.4441'
!         write(s_logunit,F00) ' eccen =   0.016715'
!         write(s_logunit,F00) ' mvelp = 102.7'
!         call shr_sys_abort()
!      else if ( log_print ) then
!         write(s_logunit,F00) 'Use input orbital parameters: '
!      end if
!      if( (obliq < SHR_ORB_OBLIQ_MIN).or.(obliq > SHR_ORB_OBLIQ_MAX) ) then
!         write(s_logunit,F03) 'Input obliquity unreasonable: ', obliq
!         call shr_sys_abort()
!      end if
!      if( (eccen < SHR_ORB_ECCEN_MIN).or.(eccen > SHR_ORB_ECCEN_MAX) ) then
!         write(s_logunit,F03) 'Input eccentricity unreasonable: ', eccen
!         call shr_sys_abort()
!      end if
!      if( (mvelp < SHR_ORB_MVELP_MIN).or.(mvelp > SHR_ORB_MVELP_MAX) ) then
!         write(s_logunit,F03) 'Input mvelp unreasonable: ' , mvelp
!         call shr_sys_abort()
!      end if
!      eccen2 = eccen*eccen
!      eccen3 = eccen2*eccen

!   ELSE  ! Otherwise calculate based on years before present
 
!      if ( log_print .and. s_loglev > 0) then
!         write(s_logunit,F01) 'Calculate orbit for year: ' , iyear_AD
!      end if

      yb4_1950AD = 1950.0_SHR_KIND_R8 - real(iyear_AD,SHR_KIND_R8)
      if ( abs(yb4_1950AD) .gt. 1000000.0_SHR_KIND_R8 )then
!         write(s_logunit,F00) 'orbit only valid for years+-1000000'
!         write(s_logunit,F00) 'Relative to 1950 AD'
!         write(s_logunit,F03) '# of years before 1950: ',yb4_1950AD
!         write(s_logunit,F01) 'Year to simulate was  : ',iyear_AD
!         call shr_sys_abort()
          write(6,*) 'Error in shr_orb,  abs(yb4_1950AD) .gt. 1000000.0_SHR_KIND_R8'
          call endrun()
      end if
 
      ! The following calculates the earths obliquity, orbital eccentricity
      ! (and various powers of it) and vernal equinox mean longitude of
      ! perihelion for years in the past (future = negative of years past),
      ! using constants (see parameter section) given in the program of:
      !
      ! Berger, Andre.  1978  A Simple Algorithm to Compute Long-Term Variations
      ! of Daily Insolation.  Contribution 18, Institute of Astronomy and
      ! Geophysics, Universite Catholique de Louvain, Louvain-la-Neuve, Belgium.
      !
      ! and formulas given in the paper (where less precise constants are also
      ! given):
      !
      ! Berger, Andre.  1978.  Long-Term Variations of Daily Insolation and
      ! Quaternary Climatic Changes.  J. of the Atmo. Sci. 35:2362-2367
      !
      ! The algorithm is valid only to 1,000,000 years past or hence.
      ! For a solution valid to 5-10 million years past see the above author.
      ! Algorithm below is better for years closer to present than is the
      ! 5-10 million year solution.
      !
      ! Years to time of interest must be negative of years before present
      ! (1950) in formulas that follow. 
 
      years = - yb4_1950AD
 
      ! In the summations below, cosine or sine arguments, which end up in
      ! degrees, must be converted to radians via multiplication by degrad.
      !
      ! Summation of cosine series for obliquity (epsilon in Berger 1978) in
      ! degrees. Convert the amplitudes and rates, which are in arc secs, into
      ! degrees via multiplication by psecdeg (arc seconds to degrees conversion
      ! factor).  For obliq, first term is Berger 1978 epsilon star; second
      ! term is series summation in degrees.
  
      obsum = 0.0_SHR_KIND_R8
      do i = 1, poblen
         obsum = obsum + obamp(i)*psecdeg*cos((obrate(i)*psecdeg*years + &
         &       obphas(i))*degrad)
      end do
      obliq = 23.320556_SHR_KIND_R8 + obsum
 
      ! Summation of cosine and sine series for computation of eccentricity 
      ! (eccen; e in Berger 1978) and fixed vernal equinox longitude of 
      ! perihelion (fvelp; pi in Berger 1978), which is used for computation 
      ! of moving vernal equinox longitude of perihelion.  Convert the rates, 
      ! which are in arc seconds, into degrees via multiplication by psecdeg.
 
      cossum = 0.0_SHR_KIND_R8
      do i = 1, pecclen
        cossum = cossum+ecamp(i)*cos((ecrate(i)*psecdeg*years+ecphas(i))*degrad)
      end do
 
      sinsum = 0.0_SHR_KIND_R8
      do i = 1, pecclen
        sinsum = sinsum+ecamp(i)*sin((ecrate(i)*psecdeg*years+ecphas(i))*degrad)
      end do
 
      ! Use summations to calculate eccentricity
 
      eccen2 = cossum*cossum + sinsum*sinsum
      eccen  = sqrt(eccen2)
      eccen3 = eccen2*eccen
 
      ! A series of cases for fvelp, which is in radians.
         
      if (abs(cossum) .le. 1.0E-8_SHR_KIND_R8) then
        if (sinsum .eq. 0.0_SHR_KIND_R8) then
          fvelp = 0.0_SHR_KIND_R8
        else if (sinsum .lt. 0.0_SHR_KIND_R8) then
          fvelp = 1.5_SHR_KIND_R8*pi
        else if (sinsum .gt. 0.0_SHR_KIND_R8) then
          fvelp = .5_SHR_KIND_R8*pi
        endif
      else if (cossum .lt. 0.0_SHR_KIND_R8) then
        fvelp = atan(sinsum/cossum) + pi
      else if (cossum .gt. 0.0_SHR_KIND_R8) then
        if (sinsum .lt. 0.0_SHR_KIND_R8) then
          fvelp = atan(sinsum/cossum) + 2.0_SHR_KIND_R8*pi
        else
          fvelp = atan(sinsum/cossum)
        endif
      endif
 
      ! Summation of sin series for computation of moving vernal equinox long
      ! of perihelion (mvelp; omega bar in Berger 1978) in degrees.  For mvelp,
      ! first term is fvelp in degrees; second term is Berger 1978 psi bar 
      ! times years and in degrees; third term is Berger 1978 zeta; fourth 
      ! term is series summation in degrees.  Convert the amplitudes and rates,
      ! which are in arc seconds, into degrees via multiplication by psecdeg.  
      ! Series summation plus second and third terms constitute Berger 1978
      ! psi, which is the general precession.
 
      mvsum = 0.0_SHR_KIND_R8
      do i = 1, pmvelen
        mvsum = mvsum + mvamp(i)*psecdeg*sin((mvrate(i)*psecdeg*years + &
        &       mvphas(i))*degrad)
      end do
      mvelp = fvelp/degrad + 50.439273_SHR_KIND_R8*psecdeg*years + 3.392506_SHR_KIND_R8 + mvsum
 
      ! Cases to make sure mvelp is between 0 and 360.
 
      do while (mvelp .lt. 0.0_SHR_KIND_R8)
        mvelp = mvelp + 360.0_SHR_KIND_R8
      end do
      do while (mvelp .ge. 360.0_SHR_KIND_R8)
        mvelp = mvelp - 360.0_SHR_KIND_R8
      end do

!   END IF  ! end of test on whether to calculate or use input orbital params
 
   ! Orbit needs the obliquity in radians
 
   obliqr = obliq*degrad
 
   ! 180 degrees must be added to mvelp since observations are made from the
   ! earth and the sun is considered (wrongly for the algorithm) to go around
   ! the earth. For a more graphic explanation see Appendix B in:
   !
   ! A. Berger, M. Loutre and C. Tricot. 1993.  Insolation and Earth Orbital
   ! Periods.  J. of Geophysical Research 98:10,341-10,362.
   !
   ! Additionally, orbit will need this value in radians. So mvelp becomes
   ! mvelpp (mvelp plus pi)
 
   mvelpp = (mvelp + 180._SHR_KIND_R8)*degrad
 
   ! Set up an argument used several times in lambm0 calculation ahead.
 
   beta = sqrt(1._SHR_KIND_R8 - eccen2)
 
   ! The mean longitude at the vernal equinox (lambda m nought in Berger
   ! 1978; in radians) is calculated from the following formula given in 
   ! Berger 1978.  At the vernal equinox the true longitude (lambda in Berger
   ! 1978) is 0.

   lambm0 = 2._SHR_KIND_R8*((.5_SHR_KIND_R8*eccen + .125_SHR_KIND_R8*eccen3)*(1._SHR_KIND_R8 + beta)*sin(mvelpp)  &
   &      - .250_SHR_KIND_R8*eccen2*(.5_SHR_KIND_R8    + beta)*sin(2._SHR_KIND_R8*mvelpp)            &
   &      + .125_SHR_KIND_R8*eccen3*(1._SHR_KIND_R8/3._SHR_KIND_R8 + beta)*sin(3._SHR_KIND_R8*mvelpp))
 
!   if ( log_print ) then
!     write(s_logunit,F03) '------ Computed Orbital Parameters ------'
!     write(s_logunit,F03) 'Eccentricity      = ',eccen
!     write(s_logunit,F03) 'Obliquity (deg)   = ',obliq
!     write(s_logunit,F03) 'Obliquity (rad)   = ',obliqr
!     write(s_logunit,F03) 'Long of perh(deg) = ',mvelp
!     write(s_logunit,F03) 'Long of perh(rad) = ',mvelpp
!     write(s_logunit,F03) 'Long at v.e.(rad) = ',lambm0
!     write(s_logunit,F03) '-----------------------------------------'
!   end if
 
END SUBROUTINE shr_orb_params

!===============================================================================

SUBROUTINE shr_orb_decl(calday ,eccen ,mvelpp ,lambm0 ,obliqr ,delta ,eccf)

!-------------------------------------------------------------------------------
!
! Compute earth/orbit parameters using formula suggested by
! Duane Thresher.
!
!---------------------------Code history----------------------------------------
!
! Original version:  Erik Kluzek
! Date:              Oct/1997
!
!-------------------------------------------------------------------------------

   !------------------------------Arguments--------------------------------
   real   (SHR_KIND_R8),intent(in)  :: calday ! Calendar day, including fraction
   real   (SHR_KIND_R8),intent(in)  :: eccen  ! Eccentricity
   real   (SHR_KIND_R8),intent(in)  :: obliqr ! Earths obliquity in radians
   real   (SHR_KIND_R8),intent(in)  :: lambm0 ! Mean long of perihelion at the 
                                              ! vernal equinox (radians)
   real   (SHR_KIND_R8),intent(in)  :: mvelpp ! moving vernal equinox longitude
                                              ! of perihelion plus pi (radians)
   real   (SHR_KIND_R8),intent(out) :: delta  ! Solar declination angle in rad
   real   (SHR_KIND_R8),intent(out) :: eccf   ! Earth-sun distance factor (ie. (1/r)**2)
 
   !---------------------------Local variables-----------------------------
   real   (SHR_KIND_R8),parameter :: dayspy = 365.0_SHR_KIND_R8  ! days per year
   real   (SHR_KIND_R8),parameter :: ve     = 80.5_SHR_KIND_R8   ! Calday of vernal equinox
                                                     ! assumes Jan 1 = calday 1
 
   real   (SHR_KIND_R8) ::   lambm  ! Lambda m, mean long of perihelion (rad)
   real   (SHR_KIND_R8) ::   lmm    ! Intermediate argument involving lambm
   real   (SHR_KIND_R8) ::   lamb   ! Lambda, the earths long of perihelion
   real   (SHR_KIND_R8) ::   invrho ! Inverse normalized sun/earth distance
   real   (SHR_KIND_R8) ::   sinl   ! Sine of lmm
 
   ! Compute eccentricity factor and solar declination using
   ! day value where a round day (such as 213.0) refers to 0z at
   ! Greenwich longitude.
   !
   ! Use formulas from Berger, Andre 1978: Long-Term Variations of Daily
   ! Insolation and Quaternary Climatic Changes. J. of the Atmo. Sci.
   ! 35:2362-2367.
   !
   ! To get the earths true longitude (position in orbit; lambda in Berger 
   ! 1978) which is necessary to find the eccentricity factor and declination,
   ! must first calculate the mean longitude (lambda m in Berger 1978) at
   ! the present day.  This is done by adding to lambm0 (the mean longitude
   ! at the vernal equinox, set as March 21 at noon, when lambda=0; in radians)
   ! an increment (delta lambda m in Berger 1978) that is the number of
   ! days past or before (a negative increment) the vernal equinox divided by
   ! the days in a model year times the 2*pi radians in a complete orbit.
 
   lambm = lambm0 + (calday - ve)*2._SHR_KIND_R8*pi/dayspy
   lmm   = lambm  - mvelpp
 
   ! The earths true longitude, in radians, is then found from
   ! the formula in Berger 1978:
 
   sinl  = sin(lmm)
   lamb  = lambm  + eccen*(2._SHR_KIND_R8*sinl + eccen*(1.25_SHR_KIND_R8*sin(2._SHR_KIND_R8*lmm)  &
   &     + eccen*((13.0_SHR_KIND_R8/12.0_SHR_KIND_R8)*sin(3._SHR_KIND_R8*lmm) - 0.25_SHR_KIND_R8*sinl)))
 
   ! Using the obliquity, eccentricity, moving vernal equinox longitude of
   ! perihelion (plus), and earths true longitude, the declination (delta)
   ! and the normalized earth/sun distance (rho in Berger 1978; actually inverse
   ! rho will be used), and thus the eccentricity factor (eccf), can be 
   ! calculated from formulas given in Berger 1978.
 
   invrho = (1._SHR_KIND_R8 + eccen*cos(lamb - mvelpp)) / (1._SHR_KIND_R8 - eccen*eccen)
 
   ! Set solar declination and eccentricity factor
 
   delta  = asin(sin(obliqr)*sin(lamb))
   eccf   = invrho*invrho
 
   return
 
END SUBROUTINE shr_orb_decl

!===============================================================================


END MODULE shr_orb_mod
module surfFileMod

!----------------------------------------------------------------------- 
!BOP
!
! !MODULE: surfFileMod
! 
! !DESCRIPTION: 
! Contains methods for reading in surface data file and determining
! two-dimensional subgrid weights as well as writing out new surface
! dataset. When reading in the surface dataset, determines array 
! which sets the PFT for each of the [maxpatch] patches and 
! array which sets the relative abundance of the PFT. 
! Also fills in the PFTs for vegetated portion of each grid cell. 
! Fractional areas for these points pertain to "vegetated" 
! area not to total grid area. Need to adjust them for fraction of grid 
! that is vegetated. Also fills in urban, lake, wetland, and glacier patches.
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: surfrd  !Read surface dataset and determine subgrid weights
!  public :: surfwrt !Write surface dataset
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!----------------------------------------------------------------------- 

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: surfrd
!
! !INTERFACE:
!Yaqiong Lu 01/13/2009
  subroutine surfrd(organicxy,efisopxy,gtixy,ilx,jlx,iveg,isl,lndmsk)
!
! !DESCRIPTION: 
! Read the surface dataset and create subgrid weights.
! The model's surface dataset recognizes 5 basic land cover types within 
! a grid cell: lake, wetland, urban, glacier, and vegetated. The vegetated 
! portion of the grid cell is comprised of up to [maxpatch_pft] PFTs. These
! subgrid patches are read in explicitly for each grid cell. This is in
! contrast to LSMv1, where the PFTs were built implicitly from biome types.
! Read surface boundary data with the exception of 
! monthly lai,sai,and heights at top and bottom of canopy 
! on [lsmlon] x [lsmlat] grid. 
!    o real edges of grid
!    o integer  number of longitudes per latitude
!    o real latitude  of grid cell (degrees)
!    o real longitude of grid cell (degrees)
!    o integer surface type: 0 = ocean or 1 = land
!    o integer soil color (1 to 9) for use with soil albedos
!    o real soil texture, %sand, for thermal and hydraulic properties
!    o real soil texture, %clay, for thermal and hydraulic properties
!    o real % of cell covered by lake    for use as subgrid patch
!    o real % of cell covered by wetland for use as subgrid patch
!    o real % of cell that is urban      for use as subgrid patch
!    o real % of cell that is glacier    for use as subgrid patch
!    o integer PFTs
!    o real % abundance PFTs (as a percent of vegetated area)
!
! OFFLINE MODE ONLY:
! Surface grid edges -- Grids do not have to be global. 
! If grid is read in from dataset, grid is assumed to be global 
! (does not have to be regular, however)
! If grid is generated by model, grid does not have to be global but must then
! define the north, east, south, and west edges:
!
!    o lsmedge(1)    = northern edge of grid (degrees): >  -90 and <= 90
!    o lsmedge(2)    = eastern edge of grid (degrees) : see following notes
!    o lsmedge(3)    = southern edge of grid (degrees): >= -90 and <  90
!    o lsmedge(4)    = western edge of grid (degrees) : see following notes
!
!      For partial grids, northern and southern edges are any latitude
!      between 90 (North Pole) and -90 (South Pole). Western and eastern
!      edges are any longitude between -180 and 180, with longitudes
!      west of Greenwich negative. That is, western edge >= -180 and < 180;
!      eastern edge > western edge and <= 180.
!
!      For global grids, northern and southern edges are 90 (North Pole)
!      and -90 (South Pole). The western and eastern edges depend on
!      whether the grid starts at Dateline or Greenwich. Regardless,
!      these edges must span 360 degrees. Examples:
!
!                              West edge    East edge
!                            ---------------------------------------------------
!  (1) Dateline            :        -180 to 180        (negative W of Greenwich)
!  (2) Greenwich (centered):    0 - dx/2 to 360 - dx/2
!
!    Grid 1 is the grid for offline mode
!    Grid 2 is the grid for cam and csm mode since the NCAR CAM 
!    starts at Greenwich, centered on Greenwich 
!
! !USES:
    use shr_kind_mod, only: r8 => shr_kind_r8
    use clm_varpar                      !parameters
    use clm_varsur                      !surface data 
    use pftvarcon, only : noveg, crop  !vegetation type (PFT) 
    use clm_varcon,only : sand,clay,soic,plant,cover,num_landcover_types
    use clm_varsur      , only :gti, wtxy,vegxy,soic2d,sand3d,clay3d,organic3d,efisop2d &
                                 ,pctgla,pctlak,pctwet,pcturb
    use decompMod , only: get_proc_bounds
!
! !ARGUMENTS:
    implicit none
!!ylu add new variables:
    integer :: ilx,jlx
    real(r8) :: organicxy(maxpatch)
    real(r8) :: efisopxy(6)
    real(r8) :: gtixy
!
! !CALLED FROM:
! subroutine initialize in module initializeMod
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein, Sam Levis and Gordon Bonan
!
!EOP
!   variables from MM5 11/25/2003 Jiming Jin

    integer  :: iveg,isl,lndmsk
!
! !LOCAL VARIABLES:
    integer  :: g,k,m,k1,k2,begg,endg                            ! indices
    integer  :: ncid,dimid,varid                         ! netCDF id's
    integer  :: ier                                      ! error status
    integer ,allocatable :: pft(:,:)            ! PFT
    integer ,allocatable :: cft(:,:)            ! CFT
    real(r8),allocatable :: pctcft_lunit(:,:)   ! % of crop lunit area for CFTs
    real(r8),allocatable :: pctpft_lunit(:,:)   ! % of vegetated lunit area PFTs
    real(r8), allocatable :: pctpft(:,:)               ! percent of vegetated gridcell area for PFTs
!ylu change pctspec to pctspec1, pctspec now to be a vector in clm_varsur,so pctspec can be used in pftdynMod.F
    real(r8) :: pctspec1                                  ! percent of gridcell made up of special landunits
    integer  :: cropcount                                ! temporary counter
    real(r8) :: sumscl                                   ! temporory scalar sum
    real(r8),allocatable :: sumvec(:)                                   ! temporary vector sum
    logical  :: found                                    ! temporary for error check
    integer  :: iindx, jindx                             ! temporary for error check
    integer  :: miss = 99999                             ! missing data indicator
    real(r8) :: wst(0:numpft)                            ! as[signed?] pft at specific i, j
    integer  :: wsti(maxpatch_pft)                       ! ranked indices of largest values in wst
    real(r8) :: wst_sum                                  ! sum of %pft
    real(r8) :: sumpct                                   ! sum of %pft over maxpatch_pft
    real(r8) :: diff                                     ! the difference (wst_sum - sumpct)
    real(r8) :: rmax                                     ! maximum patch cover
!ylu 01/16/2009
    integer  :: pftid
!!-------------------------------------------------------------------------
       ! Initialize surface data to fill value
  call CLMDebug('surfrd-mark1')
    call get_proc_bounds(begg=begg, endg=endg)

   call CLMDebug('get begg,endg')

       soic2d(:)   = -999
       sand3d(:,:) = -999.
       clay3d(:,:) = -999.
       pctlak(:)   = 0.0
       pctwet(:)   = 0.0
       pcturb(:)   = 0.0
       pctgla(:)   = 0.0
 !ylu
      pftid        = 0
    call CLMDebug('allocate sumvec')
    allocate(sumvec(begg:endg))
    call CLMDebug('allocate cft')
    allocate(cft(begg:endg,numcft))
    call CLMDebug('allocate pft')
    allocate(pft(begg:endg,maxpatch_pft))
    call CLMDebug('allocate pctpft_lunit')
    allocate(pctcft_lunit(begg:endg,numcft))
    call CLMDebug('allocate pctpft_lunit')
    allocate(pctpft_lunit(begg:endg,maxpatch_pft))
    call CLMDebug('allocate pctpft')  
    allocate(pctpft(begg:endg,0:numpft))
       pctpft(:,:) = 0.0
      pft(:,:)    = 0 
      call CLMDebug('surfrd-mark')

       ! Obtain netcdf file and read surface data

       do g=begg,endg

           soic2d(g) = soic(isl)
           efisop2d(:,g) = efisopxy(:)
          gti(g) = gtixy
           do k=1,nlevsoi
             sand3d(g,k)  = sand(isl)
             clay3d(g,k)  = clay(isl)
             organic3d(g,k) = organicxy(k) 
          end do

          call CLMDebug('surfrd-mark2')

!---------------------------------------------------
! in current versions of CLM, the lake scheme has problems 
! in simulating deep lakes. Esepcially, observed lake depth data should
! be used in the lake scheme. Wetland and glacier schemes haven't been developed
! within CLM. Thus, the following lines are temporirially commented out
! -- Jiming Jin 10/18/2012   

!           if(iveg == 17 .or. iveg == 18) then
!              pctwet(g)      = 100.0 
!           elseif(iveg.eq.16.and.lndmsk.eq.1) then
!              pctlak(g)      = 100.0
!           elseif(iveg.eq.24) then 
!              pctgla(g)      = 100.0
!           end if
!----------------------------------------------------
           do m=1,maxpatch_pft
               pft(g,m) = plant(iveg,m)
               if(cover(iveg,m).ne.0.0) then
                pctpft(g,pft(g,m)) = cover(iveg,m)
               end if
           end do

!CLM caculates urban separately. See module_sf_clm
!             pcturb(g) = 0
!             if(pctgla(g) .eq. 100 .or. pctlak(g) .eq. 100 .or.pctwet(g) .eq. 100) then
!                pftid = 1    !no pft distribution for this grid cell 
!                pctpft(g,0) = 100.
!                pctpft(g,1:numpft) = 0.
!                pft(g,:) = 0.
!              end if
!---------------------------------------------------------------------
         end do
           call CLMDebug('surfrd--mark3')
       ! Error check: valid PFTs and sum of cover must equal 100

#ifndef DGVM
       sumvec(:) = abs(sum(pctpft,dim=2)-100.)
          do g=begg,endg
             do m = 1, maxpatch_pft
              if (pft(g,m)<0 .or. pft(g,m)>numpft) then
                   write(6,*)'SURFRD error: invalid PFT for g,m=',ilx,jlx,m,pft(g,m)
                   call endrun
                end if
             end do
!ylu
              if (sumvec(g)>1.e-04 .and. pftid == 0) then 
               write(6,*)'SURFRD error: PFT cover ne 100 for g=',ilx,jlx
                do m=1,maxpatch_pft
                   write(6,*)'m= ',m,' pft= ',pft(g,m)
                end do
                write(6,*)'sumvec= ',sumvec(g)
                call endrun
             end if
          end do
#endif
          call CLMDebug('surfrd--mark4')
          ! 1. pctpft must go back to %vegetated landunit instead of %gridcell
          ! 2. pctpft bare = 100 when landmask = 1 and 100% special landunit
          ! NB: (1) and (2) do not apply to crops.
          ! For now keep all cfts (< 4 anyway) instead of 4 most dominant cfts

           do g=begg,endg 
               cft(g,:) = 0
                pctcft_lunit(g,:) = 0.
                cropcount = 0
                pctspec1 = pcturb(g) + pctgla(g) + pctlak(g) + pctwet(g)
                if (pctspec1 < 100.) then
                   do m = 0, numpft
                      if (crop(m) == 1. .and. pctpft(g,m) > 0.) then
                         cropcount = cropcount + 1
                         if (cropcount > maxpatch_cft) then
                            write(6,*) 'ERROR surfFileMod: cropcount>maxpatch_cft'
                            call endrun()
                         end if
                         cft(g,cropcount) = m
                         pctcft_lunit(g,cropcount) = pctpft(g,m) !* 100./(100.-pctspec)
                         pctpft(g,m) = 0.0
                      else if (crop(m) == 0.) then
                         pctpft(g,m) = pctpft(g,m) !* 100./(100.-pctspec)
                       end if
                   end do
                else if (pctspec1 == 100.) then
                   pctpft(g,0)        = 100.
                   pctpft(g,1:numpft) =   0.
                end if
             end do
          
          call CLMDebug('surfrd-mark5')
          ! Find pft and pct arrays
          ! Save percent cover by PFT [wst] and total percent cover [wst_sum]

             do g=begg,endg

                wst_sum = 0.
                sumpct = 0
                do m = 0, numpft
                   wst(m) = pctpft(g,m)
                   wst_sum = wst_sum + pctpft(g,m)
                end do

                ! Rank [wst] in ascendg order to obtain the top [maxpatch_pft] PFTs
!ylu  for lake, glacier,wetland, pft = noveg ,pftid = 1
                if (pftid .eq. 0) call mkrank (numpft, wst, miss, wsti, maxpatch_pft)

                ! Fill in [pft] and [pctpft] with data for top [maxpatch_pft] PFTs.
                ! If land model grid cell is ocean, set to no PFTs.
                ! If land model grid cell is land then:
                !  1. If [pctlnd_o] = 0, there is no PFT data from the input grid.
                !     Since need land data, use bare ground.
                !  2. If [pctlnd_o] > 0, there is PFT data from the input grid but:
                !     a. use the chosen PFT so long as it is not a missing value
                !     b. missing value means no more PFTs with cover > 0

                 if (pftid .eq. 0) then   ! vegetated grid
                   do m = 1, maxpatch_pft
                      if(wsti(m) /= miss) then
                          pft(g,m) = wsti(m)
                          pctpft_lunit(g,m) = wst(wsti(m))
                       else
                         pft(g,m) = noveg
                         pctpft_lunit(g,m) = 0.
                       end if
                      sumpct = sumpct + pctpft_lunit(g,m)
                   end do
                else                       ! grid with other type =100        ! model grid wants ocean
                   do m = 1, maxpatch_pft
                      pft(g,m) = 0
                      pctpft_lunit(g,m) = 0.
                   end do
                end if
          call CLMDebug('surfrd--mark6')

                ! Correct for the case of more than [maxpatch_pft] PFTs present

                if (sumpct < wst_sum) then
                   diff  = wst_sum - sumpct
                   sumpct = 0.
                   do m = 1, maxpatch_pft
                      pctpft_lunit(g,m) = pctpft_lunit(g,m) + diff/maxpatch_pft
                      sumpct = sumpct + pctpft_lunit(g,m)
                   end do
                end if

                ! Error check: make sure have a valid PFT

                do m = 1,maxpatch_pft
                   if (pft(g,m) < 0 .or. pft(g,m) > numpft) then
                      write (6,*)'surfrd error: invalid PFT at gridcell g=',ilx,jlx,pft(g,m)
                      call endrun()
                   end if
                end do

                ! As done in mksrfdatMod.F90 for other percentages, truncate pctpft to
                ! ensure that weight relative to landunit is not nonzero
                ! (i.e. a very small number such as 1e-16) where it really should be zero

                do m=1,maxpatch_pft
                   pctpft_lunit(g,m) = float(nint(pctpft_lunit(g,m)))
                end do
                do m=1,maxpatch_cft
                   pctcft_lunit(g,m) = float(nint(pctcft_lunit(g,m)))
                end do

                ! Make sure sum of PFT cover == 100 for land points. If not,
                ! subtract excess from most dominant PFT.
          call CLMDebug('surfrd--mark7')
                rmax = -9999.
                k1 = -9999
                k2 = -9999
                sumpct = 0.
                do m = 1, maxpatch_pft
                   sumpct = sumpct + pctpft_lunit(g,m)
                   if (pctpft_lunit(g,m) > rmax) then
                      k1 = m
                      rmax = pctpft_lunit(g,m)
                   end if
                end do
                do m = 1, maxpatch_cft
                   sumpct = sumpct + pctcft_lunit(g,m)
                   if (pctcft_lunit(g,m) > rmax) then
                      k2 = m
                      rmax = pctcft_lunit(g,m)
                   end if
                end do
                if (k1 == -9999 .and. k2 == -9999) then
                   write(6,*)'surfrd error: largest PFT patch not found'
                   call endrun()
                 else if(pftid /=1) then  
                   if (sumpct < 95 .or. sumpct > 105.) then
                      write(6,*)'surfrd error: sum of PFT cover =',sumpct,' at g=',ilx,jlx
                      call endrun()
                   else if (sumpct /= 100. .and. k2 /= -9999) then
                      pctcft_lunit(g,k2) = pctcft_lunit(g,k2) - (sumpct-100.)
                   else if (sumpct /= 100.) then
                      pctpft_lunit(g,k1) = pctpft_lunit(g,k1) - (sumpct-100.)
                   end if
                end if

                ! Error check: make sure PFTs sum to 100% cover

                sumpct = 0.
                do m = 1, maxpatch_pft
                   sumpct = sumpct + pctpft_lunit(g,m)
                end do
                do m = 1, maxpatch_cft
                   sumpct = sumpct + pctcft_lunit(g,m)
                end do
              if (pftid == 0) then    
                   if (abs(sumpct - 100.) > 0.000001) then
                      write(6,*)'surfFileMod error: sum(pct) over maxpatch_pft is not = 100.'
                      write(6,*)sumpct, g
                      call endrun()
                   end if
                   if (sumpct < -0.000001) then
                      write(6,*)'surfFileMod error: sum(pct) over maxpatch_pft is < 0.'
                      write(6,*)sumpct, g
                      call endrun()
                   end if
                end if

             end do   ! end of g loop

          call CLMDebug('surfrd--mark8')

       ! Error check: glacier, lake, wetland, urban sum must be less than 100

       found = .false.
          do g=begg,endg
             sumscl = pctlak(g)+pctwet(g)+pcturb(g)+pctgla(g)
             if (sumscl > 100.+1.e-04) then
                found = .true.
                iindx = ilx
                jindx = jlx
                exit
             end if
          if (found) exit
       end do
       if ( found ) then
          write(6,*)'surfrd error: PFT cover>100 for g=',ilx,jlx
          call endrun()
       end if

       ! Error check that urban parameterization is not yet finished

#ifndef TESTZACK
       found = .false.
          do g=begg,endg
             if (pcturb(g) /= 0.) then
                found = .true.
                iindx = ilx
                jindx = jlx
                exit
             end if
          if (found) exit
       end do
       if ( found ) then
          write (6,*)'surfrd error: urban parameterization not implemented at g= ',ilx,jlx
          call endrun()
       end if
#endif

    ! Determine array [veg], which sets the PFT for each of the [maxpatch]
    ! patches and array [wt], which sets the relative abundance of the PFT.
    ! Fill in PFTs for vegetated portion of grid cell. Fractional areas for
    ! these points [pctpft] pertain to "vegetated" area not to total grid area.
    ! So need to adjust them for fraction of grid that is vegetated.
    ! Next, fill in urban, lake, wetland, and glacier patches.

    vegxy(:,:) = 0
    wtxy(:,:)  = 0.
       do g=begg,endg
          if (lndmsk == 1) then
             sumscl = pcturb(g)+pctlak(g)+pctwet(g)+pctgla(g)
             do m = 1, maxpatch_pft
!ylu changed according to CLM4
#ifdef CNDV 
             if (create_crop_landunit) then ! been through surfrd_wtxy_veg_all
                if (crop(m-1) == 0) then    ! so update natural vegetation only
                   wtxy(g,m) = 0._r8       ! crops should have values >= 0.
               end if
            else                   ! not been through surfrd_wtxy_veg_all
               wtxy(g,m) = 0._r8  ! so update all vegetation
               vegxy(g,m) = m - 1 ! 0 (bare ground) to maxpatch_pft-1 (= 16)
            end if
!!!       
   call CLMDebug('surfrd--mark8')

!                vegxy(g,m) = noveg !spinup initialization
!                wtxy(g,m) = 1.0/maxpatch_pft * (100.-sumscl)/100.
#else
                vegxy(g,m) = pft(g,m)
                wtxy(g,m) = pctpft_lunit(g,m) * (100.-sumscl)/10000.
#endif
             end do
             vegxy(g,npatch_urban) = noveg
             wtxy(g,npatch_urban) = pcturb(g)/100.
             vegxy(g,npatch_lake)  = noveg
             wtxy(g,npatch_lake)  = pctlak(g)/100.
             vegxy(g,npatch_wet)   = noveg
             wtxy(g,npatch_wet)   = pctwet(g)/100.
             vegxy(g,npatch_glacier) = noveg
             wtxy(g,npatch_glacier) = pctgla(g)/100.

             do m = 1,maxpatch_cft
#ifdef DGVM
                   vegxy(g,npatch_glacier+m) = noveg ! currently assume crop=0 if DGVM mode
                   wtxy(g,npatch_glacier+m)= 0.
#else
                   vegxy(g,npatch_glacier+m) = cft(g,m)
                   wtxy(g,npatch_glacier+m)= pctcft_lunit(g,m) * (100.-sumscl)/10000.
#endif
            end do
          end if
    end do

    found = .false.
    sumvec(:) = abs(sum(wtxy,dim=2)-1.)
      do g=begg,endg
          if (sumvec(g) > 1.e-06 .and. lndmsk==1) then
             found = .true.
             iindx = ilx
             jindx = jlx
             exit
          endif
       if (found) exit
    end do
    if ( found ) then
       write (6,*)'surfrd error: WT > 1 or <1  occurs at g= ',iindx,jindx
       call endrun()
    end if
          call CLMDebug('surfrd done')


    deallocate(sumvec)
    deallocate(cft)
    deallocate(pft)
    deallocate(pctcft_lunit)
    deallocate(pctpft_lunit)
    deallocate(pctpft)


  end subroutine surfrd

end module surfFileMod

module SNICARMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: SNICARMod
!
! !DESCRIPTION:
! Calculate albedo of snow containing impurities 
! and the evolution of snow effective radius
!
! !USES:
  use shr_kind_mod  , only : r8 => shr_kind_r8
  use shr_const_mod , only : SHR_CONST_RHOICE
  use clm_varcon, only: ss_alb_bc1,asm_prm_bc1,ext_cff_mss_bc1,ss_alb_bc2,asm_prm_bc2,ext_cff_mss_bc2&
                        ,ss_alb_oc1,asm_prm_oc1,ext_cff_mss_oc1,ss_alb_oc2,asm_prm_oc2,ext_cff_mss_oc2&
                        ,ss_alb_dst1,asm_prm_dst1,ext_cff_mss_dst1,ss_alb_dst2,asm_prm_dst2,ext_cff_mss_dst2 &
                        ,ss_alb_dst3,asm_prm_dst3,ext_cff_mss_dst3,ss_alb_dst4,asm_prm_dst4,ext_cff_mss_dst4 & 
                        ,ss_alb_snw_drc,asm_prm_snw_drc,ext_cff_mss_snw_drc,ss_alb_snw_dfs,asm_prm_snw_dfs &
                        ,ext_cff_mss_snw_dfs,snowage_tau,snowage_kappa,snowage_drdt0 &
                        ,xx_ss_alb_snw_drc        &
                        ,xx_asm_prm_snw_drc       &
                        ,xx_ext_cff_mss_snw_drc   &
                        ,xx_ss_alb_snw_dfs        &
                        ,xx_asm_prm_snw_dfs       &
                        ,xx_ext_cff_mss_snw_dfs   &
                        ,xx_snowage_tau           &
                        ,xx_snowage_kappa         &
                        ,xx_snowage_drdt0         &
                        ,idx_Mie_snw_mx           &
                        ,idx_T_max                &
                        ,idx_Tgrd_max             &
                        ,idx_rhos_max             &
                        ,numrad_snw

  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: SNICAR_RT        ! Snow albedo and vertically-resolved solar absorption
  public :: SnowAge_grain    ! Snow effective grain size evolution
!
! !PUBLIC DATA MEMBERS:

  real(r8), public, parameter :: snw_rds_min = 54.526_r8  ! minimum allowed snow effective radius (also "fresh snow" value) [microns]
  integer,  public, parameter :: sno_nbr_aer =   8        ! number of aerosol species in snowpack (indices described above) [nbr]
  logical,  public, parameter :: DO_SNO_OC =    .false.   ! parameter to include organic carbon (OC) in snowpack radiative calculations
  logical,  public, parameter :: DO_SNO_AER =   .true.    ! parameter to include aerosols in snowpack radiative calculations

  real(r8), public, parameter :: scvng_fct_mlt_bcphi = 0.20_r8   ! scavenging factor for hydrophillic BC inclusion in meltwater [frc]
  real(r8), public, parameter :: scvng_fct_mlt_bcpho = 0.03_r8   ! scavenging factor for hydrophobic BC inclusion in meltwater [frc]
  real(r8), public, parameter :: scvng_fct_mlt_ocphi = 0.20_r8   ! scavenging factor for hydrophillic OC inclusion in meltwater [frc]
  real(r8), public, parameter :: scvng_fct_mlt_ocpho = 0.03_r8   ! scavenging factor for hydrophobic OC inclusion in meltwater [frc]
  real(r8), public, parameter :: scvng_fct_mlt_dst1  = 0.02_r8   ! scavenging factor for dust species 1 inclusion in meltwater [frc]
  real(r8), public, parameter :: scvng_fct_mlt_dst2  = 0.02_r8   ! scavenging factor for dust species 2 inclusion in meltwater [frc]
  real(r8), public, parameter :: scvng_fct_mlt_dst3  = 0.01_r8   ! scavenging factor for dust species 3 inclusion in meltwater [frc]
  real(r8), public, parameter :: scvng_fct_mlt_dst4  = 0.01_r8   ! scavenging factor for dust species 4 inclusion in meltwater [frc]

! !PRIVATE MEMBER FUNCTIONS:

!
! !PRIVATE DATA MEMBERS:
  ! Aerosol species indices:
  !  1= hydrophillic black carbon 
  !  2= hydrophobic black carbon
  !  3= hydrophilic organic carbon
  !  4= hydrophobic organic carbon
  !  5= dust species 1
  !  6= dust species 2
  !  7= dust species 3
  !  8= dust species 4
  integer,  parameter :: nir_bnd_bgn =   2               ! first band index in near-IR spectrum [idx]
  integer,  parameter :: nir_bnd_end =   5               ! ending near-IR band index [idx]

  integer,  parameter :: idx_T_min      = 1              ! minimum temperature index used in aging lookup table [idx]
  integer,  parameter :: idx_Tgrd_min   = 1              ! minimum temperature gradient index used in aging lookup table [idx]
  integer,  parameter :: idx_rhos_min   = 1              ! minimum snow density index used in aging lookup table [idx]

  integer,  parameter :: snw_rds_max_tbl = 1500          ! maximum effective radius defined in Mie lookup table [microns]
  integer,  parameter :: snw_rds_min_tbl = 30            ! minimium effective radius defined in Mie lookup table [microns]
  real(r8), parameter :: snw_rds_max     = 1500._r8      ! maximum allowed snow effective radius [microns]
  real(r8), parameter :: snw_rds_refrz   = 1000._r8      ! effective radius of re-frozen snow [microns]

  real(r8), parameter :: min_snw = 1.0E-30_r8            ! minimum snow mass required for SNICAR RT calculation [kg m-2]

  !real(r8), parameter :: C1_liq_Brun89 = 1.28E-17_r8    ! constant for liquid water grain growth [m3 s-1], from Brun89
  real(r8), parameter :: C1_liq_Brun89 = 0._r8           ! constant for liquid water grain growth [m3 s-1], from Brun89: zeroed to accomodate dry snow aging
  real(r8), parameter :: C2_liq_Brun89 = 4.22E-13_r8     ! constant for liquid water grain growth [m3 s-1], from Brun89: corrected for LWC in units of percent

  real(r8), parameter :: tim_cns_bc_rmv  = 2.2E-8_r8     ! time constant for removal of BC in snow on sea-ice [s-1] (50% mass removal/year)
  real(r8), parameter :: tim_cns_oc_rmv  = 2.2E-8_r8     ! time constant for removal of OC in snow on sea-ice [s-1] (50% mass removal/year)
  real(r8), parameter :: tim_cns_dst_rmv = 2.2E-8_r8     ! time constant for removal of dust in snow on sea-ice [s-1] (50% mass removal/year)

  ! scaling of the snow aging rate (tuning option):
  logical :: flg_snoage_scl    = .false.                 ! flag for scaling the snow aging rate by some arbitrary factor
  real(r8), parameter :: xdrdt = 1.0_r8                  ! arbitrary factor applied to snow aging rate

 
!
! !REVISION HISTORY:
! Created by Mark Flanner
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: SNICAR_RT
!
!
! !CALLED FROM:
! subroutine SurfaceAlbedo in module SurfaceAlbedoMod (CLM)
! subroutine albice (CSIM)
!
! !REVISION HISTORY:
! Author: Mark Flanner
!
! !INTERFACE:
  
  subroutine SNICAR_RT (flg_snw_ice, lbc, ubc, num_nourbanc, filter_nourbanc,  &
                        coszen, flg_slr_in, h2osno_liq, h2osno_ice, snw_rds,   &
                        mss_cnc_aer_in, albsfc, albout, flx_abs)

    !
    ! !DESCRIPTION:
    ! Determine reflectance of, and vertically-resolved solar absorption in, 
    ! snow with impurities.
    !
    ! Original references on physical models of snow reflectance include: 
    ! Wiscombe and Warren [1980] and Warren and Wiscombe [1980],
    ! Journal of Atmospheric Sciences, 37,
    !
    ! The multi-layer solution for multiple-scattering used here is from:
    ! Toon et al. [1989], Rapid calculation of radiative heating rates 
    ! and photodissociation rates in inhomogeneous multiple scattering atmospheres, 
    ! J. Geophys. Res., 94, D13, 16287-16301
    !
    ! The implementation of the SNICAR model in CLM/CSIM is described in:
    ! Flanner, M., C. Zender, J. Randerson, and P. Rasch [2007], 
    ! Present-day climate forcing and response from black carbon in snow,
    ! J. Geophys. Res., 112, D11202, doi: 10.1029/2006JD008003

    
    ! !USES:
    use clmtype
    use clm_varpar       , only : nlevsno, numrad
    use shr_const_mod    , only : SHR_CONST_PI
    use globals          , only : nstep

    !
    ! !ARGUMENTS:
    implicit none
    integer , intent(in)  :: flg_snw_ice                          ! flag: =1 when called from CLM, =2 when called from CSIM
    integer , intent(in)  :: lbc, ubc                             ! column index bounds [unitless]
    integer , intent(in)  :: num_nourbanc                         ! number of columns in non-urban filter
    integer , intent(in)  :: filter_nourbanc(ubc-lbc+1)           ! column filter for non-urban points
    real(r8), intent(in)  :: coszen(lbc:ubc)                      ! cosine of solar zenith angle for next time step (col) [unitless]
    integer , intent(in)  :: flg_slr_in                           ! flag: =1 for direct-beam incident flux, =2 for diffuse incident flux
    real(r8), intent(in)  :: h2osno_liq(lbc:ubc,-nlevsno+1:0)     ! liquid water content (col,lyr) [kg/m2]
    real(r8), intent(in)  :: h2osno_ice(lbc:ubc,-nlevsno+1:0)     ! ice content (col,lyr) [kg/m2]
    integer,  intent(in)  :: snw_rds(lbc:ubc,-nlevsno+1:0)        ! snow effective radius (col,lyr) [microns, m^-6]
    real(r8), intent(in)  :: mss_cnc_aer_in(lbc:ubc,-nlevsno+1:0,sno_nbr_aer)  ! mass concentration of all aerosol species (col,lyr,aer) [kg/kg]
    real(r8), intent(in)  :: albsfc(lbc:ubc,numrad)               ! albedo of surface underlying snow (col,bnd) [frc]
    real(r8), intent(out) :: albout(lbc:ubc,numrad)               ! snow albedo, averaged into 2 bands (=0 if no sun or no snow) (col,bnd) [frc]
    real(r8), intent(out) :: flx_abs(lbc:ubc,-nlevsno+1:1,numrad) ! absorbed flux in each layer per unit flux incident on top of snowpack (col,lyr,bnd) [frc]

    !
    ! !LOCAL VARIABLES:
    !
    ! local pointers to implicit in arguments
    !
    integer,  pointer :: snl(:)              ! negative number of snow layers (col) [nbr]
    real(r8), pointer :: h2osno(:)           ! snow liquid water equivalent (col) [kg/m2]   
    integer,  pointer :: clandunit(:)        ! corresponding landunit of column (col) [idx] (debugging only)
    integer,  pointer :: cgridcell(:)        ! columns's gridcell index (col) [idx] (debugging only)
    integer,  pointer :: ltype(:)            ! landunit type (lnd) (debugging only)
    real(r8), pointer :: londeg(:)           ! longitude (degrees) (debugging only)
    real(r8), pointer :: latdeg(:)           ! latitude (degrees) (debugging only)
!
! !OTHER LOCAL VARIABLES:
!EOP
!-----------------------------------------------------------------------
    !
    ! variables for snow radiative transfer calculations

    ! Local variables representing single-column values of arrays:
    integer :: snl_lcl                            ! negative number of snow layers [nbr]
    integer :: snw_rds_lcl(-nlevsno+1:0)          ! snow effective radius [m^-6]
    real(r8):: flx_slrd_lcl(1:numrad_snw)         ! direct beam incident irradiance [W/m2] (set to 1)
    real(r8):: flx_slri_lcl(1:numrad_snw)         ! diffuse incident irradiance [W/m2] (set to 1)
    real(r8):: mss_cnc_aer_lcl(-nlevsno+1:0,1:sno_nbr_aer) ! aerosol mass concentration (lyr,aer_nbr) [kg/kg]
    real(r8):: h2osno_lcl                         ! total column snow mass [kg/m2]
    real(r8):: h2osno_liq_lcl(-nlevsno+1:0)       ! liquid water mass [kg/m2]
    real(r8):: h2osno_ice_lcl(-nlevsno+1:0)       ! ice mass [kg/m2]
    real(r8):: albsfc_lcl(1:numrad_snw)           ! albedo of underlying surface [frc]
    real(r8):: ss_alb_snw_lcl(-nlevsno+1:0)       ! single-scatter albedo of ice grains (lyr) [frc]
    real(r8):: asm_prm_snw_lcl(-nlevsno+1:0)      ! asymmetry parameter of ice grains (lyr) [frc]
    real(r8):: ext_cff_mss_snw_lcl(-nlevsno+1:0)  ! mass extinction coefficient of ice grains (lyr) [m2/kg]
    real(r8):: ss_alb_aer_lcl(sno_nbr_aer)        ! single-scatter albedo of aerosol species (aer_nbr) [frc] 
    real(r8):: asm_prm_aer_lcl(sno_nbr_aer)       ! asymmetry parameter of aerosol species (aer_nbr) [frc]
    real(r8):: ext_cff_mss_aer_lcl(sno_nbr_aer)   ! mass extinction coefficient of aerosol species (aer_nbr) [m2/kg]


    ! Other local variables
    integer :: APRX_TYP                           ! two-stream approximation type (1=Eddington, 2=Quadrature, 3=Hemispheric Mean) [nbr]
    integer :: DELTA                              ! flag to use Delta approximation (Joseph, 1976) (1= use, 0= don't use)
    real(r8):: flx_wgt(1:numrad_snw)              ! weights applied to spectral bands, specific to direct and diffuse cases (bnd) [frc]
   
    integer :: flg_nosnl                          ! flag: =1 if there is snow, but zero snow layers, =0 if at least 1 snow layer [flg]   
    integer :: trip                               ! flag: =1 to redo RT calculation if result is unrealistic
    integer :: flg_dover                          ! defines conditions for RT redo (explained below)

    real(r8):: albedo                             ! temporary snow albedo [frc]
    real(r8):: flx_sum                            ! temporary summation variable for NIR weighting
    real(r8):: albout_lcl(numrad_snw)             ! snow albedo by band [frc]
    real(r8):: flx_abs_lcl(-nlevsno+1:1,numrad_snw)! absorbed flux per unit incident flux at top of snowpack (lyr,bnd) [frc]
 
    real(r8):: L_snw(-nlevsno+1:0)                ! h2o mass (liquid+solid) in snow layer (lyr) [kg/m2]
    real(r8):: tau_snw(-nlevsno+1:0)              ! snow optical depth (lyr) [unitless]
    real(r8):: L_aer(-nlevsno+1:0,sno_nbr_aer)    ! aerosol mass in snow layer (lyr,nbr_aer) [kg/m2] 
    real(r8):: tau_aer(-nlevsno+1:0,sno_nbr_aer)  ! aerosol optical depth (lyr,nbr_aer) [unitless]
    real(r8):: tau_sum                            ! cumulative (snow+aerosol) optical depth [unitless]
    real(r8):: tau_clm(-nlevsno+1:0)              ! column optical depth from layer bottom to snowpack top (lyr) [unitless] 
    real(r8):: omega_sum                          ! temporary summation of single-scatter albedo of all aerosols [frc]
    real(r8):: g_sum                              ! temporary summation of asymmetry parameter of all aerosols [frc]

    real(r8):: tau(-nlevsno+1:0)                  ! weighted optical depth of snow+aerosol layer (lyr) [unitless]
    real(r8):: omega(-nlevsno+1:0)                ! weighted single-scatter albedo of snow+aerosol layer (lyr) [frc]
    real(r8):: g(-nlevsno+1:0)                    ! weighted asymmetry parameter of snow+aerosol layer (lyr) [frc]
    real(r8):: tau_star(-nlevsno+1:0)             ! transformed (i.e. Delta-Eddington) optical depth of snow+aerosol layer (lyr) [unitless]
    real(r8):: omega_star(-nlevsno+1:0)           ! transformed (i.e. Delta-Eddington) SSA of snow+aerosol layer (lyr) [frc]
    real(r8):: g_star(-nlevsno+1:0)               ! transformed (i.e. Delta-Eddington) asymmetry paramater of snow+aerosol layer (lyr) [frc]
   
    integer :: g_idx, c_idx, l_idx                ! gridcell, column, and landunit indices [idx]
    integer :: bnd_idx                            ! spectral band index (1 <= bnd_idx <= numrad_snw) [idx]
    integer :: rds_idx                            ! snow effective radius index for retrieving Mie parameters from lookup table [idx]
    integer :: snl_btm                            ! index of bottom snow layer (0) [idx]
    integer :: snl_top                            ! index of top snow layer (-4 to 0) [idx]
    integer :: fc                                 ! column filter index
    integer :: i                                  ! layer index [idx]
    integer :: j                                  ! aerosol number index [idx]
    integer :: n                                  ! tridiagonal matrix index [idx]
    integer :: m                                  ! secondary layer index [idx]
    integer :: ix,k                               ! an index
   
    real(r8):: F_direct(-nlevsno+1:0)             ! direct-beam radiation at bottom of layer interface (lyr) [W/m^2]
    real(r8):: F_net(-nlevsno+1:0)                ! net radiative flux at bottom of layer interface (lyr) [W/m^2]
    real(r8):: F_abs(-nlevsno+1:0)                ! net absorbed radiative energy (lyr) [W/m^2]
    real(r8):: F_abs_sum                          ! total absorbed energy in column [W/m^2]
    real(r8):: F_sfc_pls                          ! upward radiative flux at snowpack top [W/m^2]
    real(r8):: F_btm_net                          ! net flux at bottom of snowpack [W/m^2]                    
    real(r8):: F_sfc_net                          ! net flux at top of snowpack [W/m^2]
    real(r8):: energy_sum                         ! sum of all energy terms; should be 0.0 [W/m^2]
    real(r8):: F_direct_btm                       ! direct-beam radiation at bottom of snowpack [W/m^2]
    real(r8):: mu_not                             ! cosine of solar zenith angle (used locally) [frc]

    integer :: err_idx                            ! counter for number of times through error loop [nbr]
    real(r8):: lat_coord                          ! gridcell latitude (debugging only)
    real(r8):: lon_coord                          ! gridcell longitude (debugging only)
    integer :: sfctype                            ! underlying surface type (debugging only)
    real(r8):: pi                                 ! 3.1415...


    ! intermediate variables for radiative transfer approximation:
    real(r8):: gamma1(-nlevsno+1:0)               ! two-stream coefficient from Toon et al. (lyr) [unitless]
    real(r8):: gamma2(-nlevsno+1:0)               ! two-stream coefficient from Toon et al. (lyr) [unitless]
    real(r8):: gamma3(-nlevsno+1:0)               ! two-stream coefficient from Toon et al. (lyr) [unitless]
    real(r8):: gamma4(-nlevsno+1:0)               ! two-stream coefficient from Toon et al. (lyr) [unitless]
    real(r8):: lambda(-nlevsno+1:0)               ! two-stream coefficient from Toon et al. (lyr) [unitless]
    real(r8):: GAMMA(-nlevsno+1:0)                ! two-stream coefficient from Toon et al. (lyr) [unitless]
    real(r8):: mu_one                             ! two-stream coefficient from Toon et al. (lyr) [unitless]
    real(r8):: e1(-nlevsno+1:0)                   ! tri-diag intermediate variable from Toon et al. (lyr) 
    real(r8):: e2(-nlevsno+1:0)                   ! tri-diag intermediate variable from Toon et al. (lyr) 
    real(r8):: e3(-nlevsno+1:0)                   ! tri-diag intermediate variable from Toon et al. (lyr) 
    real(r8):: e4(-nlevsno+1:0)                   ! tri-diag intermediate variable from Toon et al. (lyr) 
    real(r8):: C_pls_btm(-nlevsno+1:0)            ! intermediate variable: upward flux at bottom interface (lyr) [W/m2]
    real(r8):: C_mns_btm(-nlevsno+1:0)            ! intermediate variable: downward flux at bottom interface (lyr) [W/m2]
    real(r8):: C_pls_top(-nlevsno+1:0)            ! intermediate variable: upward flux at top interface (lyr) [W/m2]
    real(r8):: C_mns_top(-nlevsno+1:0)            ! intermediate variable: downward flux at top interface (lyr) [W/m2]
    real(r8):: A(-2*nlevsno+1:0)                  ! tri-diag intermediate variable from Toon et al. (2*lyr)
    real(r8):: B(-2*nlevsno+1:0)                  ! tri-diag intermediate variable from Toon et al. (2*lyr)
    real(r8):: D(-2*nlevsno+1:0)                  ! tri-diag intermediate variable from Toon et al. (2*lyr)
    real(r8):: E(-2*nlevsno+1:0)                  ! tri-diag intermediate variable from Toon et al. (2*lyr)
    real(r8):: AS(-2*nlevsno+1:0)                 ! tri-diag intermediate variable from Toon et al. (2*lyr)
    real(r8):: DS(-2*nlevsno+1:0)                 ! tri-diag intermediate variable from Toon et al. (2*lyr)
    real(r8):: X(-2*nlevsno+1:0)                  ! tri-diag intermediate variable from Toon et al. (2*lyr)
    real(r8):: Y(-2*nlevsno+1:0)                  ! tri-diag intermediate variable from Toon et al. (2*lyr)


    ! Assign local pointers to derived subtypes components (column-level)
    ! (CLM-specific)
    if (flg_snw_ice == 1) then
       snl            => clm3%g%l%c%cps%snl
       h2osno         => clm3%g%l%c%cws%h2osno
       clandunit      => clm3%g%l%c%landunit  ! (debug only)
       cgridcell      => clm3%g%l%c%gridcell  ! (debug only)
       ltype          => clm3%g%l%itype       ! (debug only)
       londeg         => clm3%g%londeg        ! (debug only)
       latdeg         => clm3%g%latdeg        ! (debug only)
    endif

  ix = 0
  do i=1, idx_Mie_snw_mx
  do j=1, numrad_snw
    ix = ix+1
    ss_alb_snw_drc(i,j)        = xx_ss_alb_snw_drc(ix)
    asm_prm_snw_drc(i,j)       = xx_asm_prm_snw_drc(ix)
    ext_cff_mss_snw_drc(i,j)   = xx_ext_cff_mss_snw_drc(ix)
    ss_alb_snw_dfs(i,j)        = xx_ss_alb_snw_dfs(ix)
    asm_prm_snw_dfs(i,j)       = xx_asm_prm_snw_dfs(ix)
    ext_cff_mss_snw_dfs(i,j)   = xx_ext_cff_mss_snw_dfs(ix)
  end do
  end do


 ix = 0
 do i=1,idx_T_max
 do j=1,idx_Tgrd_max
 do k=1,idx_rhos_max
    ix = ix + 1
    snowage_tau(i,j,k)   = xx_snowage_tau(ix)
    snowage_kappa(i,j,k) = xx_snowage_kappa(ix)
    snowage_drdt0(i,j,k) = xx_snowage_drdt0(ix)
 end do
 end do
 end do

 
    ! Define constants
    pi = SHR_CONST_PI

    ! always use Delta approximation for snow
    DELTA = 1


    ! Loop over all non-urban columns
    ! (when called from CSIM, there is only one column)
    do fc = 1,num_nourbanc
       c_idx = filter_nourbanc(fc)


       ! Zero absorbed radiative fluxes:
       do i=-nlevsno+1,1,1
          flx_abs_lcl(:,:)   = 0._r8
          flx_abs(c_idx,i,:) = 0._r8
       enddo
       
       ! set snow/ice mass to be used for RT:
       if (flg_snw_ice == 1) then
          h2osno_lcl = h2osno(c_idx)
       else
          h2osno_lcl = h2osno_ice(c_idx,0)
       endif


       ! Qualifier for computing snow RT: 
       !  1) sunlight from atmosphere model 
       !  2) minimum amount of snow on ground. 
       !     Otherwise, set snow albedo to zero
       if ((coszen(c_idx) > 0._r8) .and. (h2osno_lcl > min_snw)) then     

          ! Set variables specific to CLM
          if (flg_snw_ice == 1) then
             ! Assign local (single-column) variables to global values
             ! If there is snow, but zero snow layers, we must create a layer locally.
             ! This layer is presumed to have the fresh snow effective radius.
             if (snl(c_idx) > -1) then
                flg_nosnl         =  1
                snl_lcl           =  -1
                h2osno_ice_lcl(0) =  h2osno_lcl
                h2osno_liq_lcl(0) =  0._r8
                snw_rds_lcl(0)    =  nint(snw_rds_min)
             else
                flg_nosnl         =  0
                snl_lcl           =  snl(c_idx)
                h2osno_liq_lcl(:) =  h2osno_liq(c_idx,:)
                h2osno_ice_lcl(:) =  h2osno_ice(c_idx,:)
                snw_rds_lcl(:)    =  snw_rds(c_idx,:)
             endif
            
             snl_btm   = 0
             snl_top   = snl_lcl+1


             ! for debugging only
             l_idx     = clandunit(c_idx)
             g_idx     = cgridcell(c_idx)
             sfctype   = ltype(l_idx)
             lat_coord = latdeg(g_idx)
             lon_coord = londeg(g_idx)


          ! Set variables specific to CSIM
          else
             flg_nosnl         = 0
             snl_lcl           = -1
             h2osno_liq_lcl(:) = h2osno_liq(c_idx,:)
             h2osno_ice_lcl(:) = h2osno_ice(c_idx,:)
             snw_rds_lcl(:)    = snw_rds(c_idx,:)
             snl_btm           = 0
             snl_top           = 0
             sfctype           = -1
             lat_coord         = -90
             lon_coord         = 0
          endif

          ! Set local aerosol array
          do j=1,sno_nbr_aer
             mss_cnc_aer_lcl(:,j) = mss_cnc_aer_in(c_idx,:,j)
          enddo


          ! Set spectral underlying surface albedos to their corresponding VIS or NIR albedos
          albsfc_lcl(1)                       = albsfc(c_idx,1)
          albsfc_lcl(nir_bnd_bgn:nir_bnd_end) = albsfc(c_idx,2)


          ! Error check for snow grain size:
          do i=snl_top,snl_btm,1
             if ((snw_rds_lcl(i) < snw_rds_min_tbl) .or. (snw_rds_lcl(i) > snw_rds_max_tbl)) then
                write (6,*) "SNICAR ERROR: snow grain radius of ", snw_rds_lcl(i), " out of bounds."
                write (6,*) "NSTEP= ", nstep
                write (6,*) "flg_snw_ice= ", flg_snw_ice
                write (6,*) "column: ", c_idx, " level: ", i, " snl(c)= ", snl_lcl
                write (6,*) "lat= ", lat_coord, " lon= ", lon_coord
                write (6,*) "h2osno(c)= ", h2osno_lcl
                call endrun()
             endif
          enddo

          ! Incident flux weighting parameters
          !  - sum of all VIS bands must equal 1
          !  - sum of all NIR bands must equal 1
          !
          ! Spectral bands (5-band case)
          !  Band 1: 0.3-0.7um (VIS)
          !  Band 2: 0.7-1.0um (NIR)
          !  Band 3: 1.0-1.2um (NIR)
          !  Band 4: 1.2-1.5um (NIR)
          !  Band 5: 1.5-5.0um (NIR)
          !
          ! The following weights are appropriate for surface-incident flux in a mid-latitude winter atmosphere
          !
          ! 3-band weights
          if (numrad_snw==3) then
             ! Direct:
             if (flg_slr_in == 1) then
                flx_wgt(1) = 1._r8
                flx_wgt(2) = 0.66628670195247_r8
                flx_wgt(3) = 0.33371329804753_r8
             ! Diffuse:
             elseif (flg_slr_in == 2) then
                flx_wgt(1) = 1._r8
                flx_wgt(2) = 0.77887652162877_r8
                flx_wgt(3) = 0.22112347837123_r8
             endif

          ! 5-band weights
          elseif(numrad_snw==5) then
             ! Direct:
             if (flg_slr_in == 1) then
                flx_wgt(1) = 1._r8
                flx_wgt(2) = 0.493