!WRF:MODEL_LAYER:PHYSICS
!

MODULE module_mp_gsfcgce

   USE     module_wrf_error
   USE module_utility, ONLY: WRFU_Clock, WRFU_Alarm
   USE module_domain, ONLY : HISTORY_ALARM, Is_alarm_tstep
   USE module_mp_radar

!JJS 1/3/2008     vvvvv

!  common /bt/
   REAL,    PRIVATE ::          rd1,  rd2,   al,   cp

!  common /cont/
   REAL,    PRIVATE ::          c38, c358, c610, c149, &
                               c879, c172, c409,  c76, &
                               c218, c580, c141
!  common /b3cs/
   REAL,    PRIVATE ::           ag,   bg,   as,   bs, &
                                 aw,   bw,  bgh,  bgq, &
                                bsh,  bsq,  bwh,  bwq

!  common /size/
   REAL,    PRIVATE ::          tnw,  tns,  tng,       &
                               roqs, roqg, roqr

!  common /bterv/
   REAL,    PRIVATE ::          zrc,  zgc,  zsc,       &
                                vrc,  vgc,  vsc

!  common /bsnw/
   REAL,    PRIVATE ::          alv,  alf,  als,   t0,   t00,     &
                                avc,  afc,  asc,  rn1,  bnd1,     &
                                rn2, bnd2,  rn3,  rn4,   rn5,     &
                                rn6,  rn7,  rn8,  rn9,  rn10,     &
                              rn101,rn10a, rn11,rn11a,  rn12

   REAL,    PRIVATE ::         rn14, rn15,rn15a, rn16,  rn17,     &
                              rn17a,rn17b,rn17c, rn18, rn18a,     &
                               rn19,rn19a,rn19b, rn20, rn20a,     &
                              rn20b, bnd3, rn21, rn22,  rn23,     &
                              rn23a,rn23b, rn25,rn30a, rn30b,     &
                              rn30c, rn31, beta, rn32

   REAL,    PRIVATE, DIMENSION( 31 ) ::    rn12a, rn12b, rn13, rn25a

!  common /rsnw1/
   REAL,    PRIVATE ::         rn10b, rn10c, rnn191, rnn192,  rn30,     &
                             rnn30a,  rn33,  rn331,  rn332

!
   REAL,    PRIVATE, DIMENSION( 31 )  ::      aa1,  aa2
   DATA aa1/.7939e-7, .7841e-6, .3369e-5, .4336e-5, .5285e-5,     &
           .3728e-5, .1852e-5, .2991e-6, .4248e-6, .7434e-6,     &
           .1812e-5, .4394e-5, .9145e-5, .1725e-4, .3348e-4,     &
           .1725e-4, .9175e-5, .4412e-5, .2252e-5, .9115e-6,     &
           .4876e-6, .3473e-6, .4758e-6, .6306e-6, .8573e-6,     &
           .7868e-6, .7192e-6, .6513e-6, .5956e-6, .5333e-6,     &
           .4834e-6/
   DATA aa2/.4006, .4831, .5320, .5307, .5319,      &
           .5249, .4888, .3894, .4047, .4318,      &
           .4771, .5183, .5463, .5651, .5813,      &
           .5655, .5478, .5203, .4906, .4447,      &
           .4126, .3960, .4149, .4320, .4506,      &
           .4483, .4460, .4433, .4413, .4382,      &
           .4361/

!+---+-----------------------------------------------------------------+
!..The following 6 variables moved here to facilitate reflectivity
!.. calculation similar to other MP schemes, because when they get
!.. declared later in the code (now commented out), it makes things
!.. more difficult to integreate with the radar code.
      REAL    , PARAMETER ::     xnor = 8.0e6
      REAL    , PARAMETER ::     xnos = 1.6e7
      REAL    , PARAMETER ::     xnoh = 2.0e5
      REAL    , PARAMETER ::     xnog = 4.0e6
      REAL    , PARAMETER ::     rhohail = 917.
      REAL    , PARAMETER ::     rhograul = 400.
!+---+-----------------------------------------------------------------+

!JJS 1/3/2008     ^^^^^

CONTAINS

!-------------------------------------------------------------------
!  NASA/GSFC GCE
!  Tao et al, 2001, Meteo. & Atmos. Phy., 97-137
!-------------------------------------------------------------------
!  SUBROUTINE gsfcgce(  th, th_old,                                 &
  SUBROUTINE gsfcgce(  th,                                         &
                       qv, ql,                                     &
                       qr, qi,                                     &
                       qs,                                         &
!                       qvold, qlold,                               &
!                       qrold, qiold,                               &
!                       qsold,                                      &
                       rho, pii, p, dt_in, z,                      &
                       ht, dz8w, grav,                             &
                       rhowater, rhosnow,                          &
                       itimestep,                                  &
                       ids,ide, jds,jde, kds,kde,                  & ! domain dims
                       ims,ime, jms,jme, kms,kme,                  & ! memory dims
                       its,ite, jts,jte, kts,kte,                  & ! tile   dims
                       rainnc, rainncv,                            &
                       snownc, snowncv, sr,                        &
                       graupelnc, graupelncv,                      &
                       refl_10cm, diagflag, do_radar_ref,          &
!                       f_qg, qg, pgold,                            &
                       f_qg, qg,                                   &
                       ihail, ice2                                 &
                                                                   )

!-------------------------------------------------------------------
  IMPLICIT NONE
!-------------------------------------------------------------------
!
! JJS 2/15/2005
!
  INTEGER,      INTENT(IN   )    ::   ids,ide, jds,jde, kds,kde , &
                                      ims,ime, jms,jme, kms,kme , &
                                      its,ite, jts,jte, kts,kte 
  INTEGER,      INTENT(IN   )    ::   itimestep, ihail, ice2 

  REAL, DIMENSION( ims:ime , kms:kme , jms:jme ),                 &
        INTENT(INOUT) ::                                          &
                                                              th, &
                                                              qv, &
                                                              ql, &
                                                              qr, &
                                                              qi, &
                                                              qs, &
                                                              qg
!
  REAL, DIMENSION( ims:ime , kms:kme , jms:jme ),                 &
        INTENT(IN   ) ::                                          &
!                                                         th_old, &
!                                                          qvold, &
!                                                          qlold, &
!                                                          qrold, &
!                                                          qiold, &
!                                                          qsold, &
!                                                          qgold, &
                                                             rho, &
                                                             pii, &
                                                               p, &
                                                            dz8w, &
                                                               z

  REAL, DIMENSION( ims:ime , jms:jme ),                           &
        INTENT(INOUT) ::                               rainnc,    &
                                                       rainncv,   &
                                                       snownc,    &   
                                                       snowncv,   &
                                                       sr,        &
                                                       graupelnc, &
                                                       graupelncv 

!+---+-----------------------------------------------------------------+
  REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT)::           &  ! GT
                                                       refl_10cm
  LOGICAL, OPTIONAL, INTENT(IN) :: diagflag
  INTEGER, OPTIONAL, INTENT(IN) :: do_radar_ref
!+---+-----------------------------------------------------------------+

  REAL , DIMENSION( ims:ime , jms:jme ) , INTENT(IN) ::       ht

  REAL, INTENT(IN   ) ::                                   dt_in, &
                                                            grav, &
                                                        rhowater, &
                                                         rhosnow

  LOGICAL, INTENT(IN), OPTIONAL :: F_QG

!  LOCAL VAR


!jjs  INTEGER             ::                            min_q, max_q

!jjs  REAL, DIMENSION( its:ite , jts:jte )                            &
!jjs                               ::        rain, snow, graupel,ice

!
!  INTEGER :: IHAIL, itaobraun, ice2, istatmin, new_ice_sat, id
  INTEGER ::  itaobraun, istatmin, new_ice_sat, id

  INTEGER :: i, j, k
  INTEGER :: iskip, ih, icount, ibud, i24h 
  REAL    :: hour
  REAL    , PARAMETER :: cmin=1.e-20
  REAL    :: dth, dqv, dqrest, dqall, dqall1, rhotot, a1, a2 
!  REAL, DIMENSION( its:ite , kts:kte , jts:jte ) ::                   &
!                         th1, qv1, ql1, qr1, qi1, qs1, qg1
 
  LOGICAL :: flag_qg

!+---+-----------------------------------------------------------------+

      INTEGER:: NCALL=0

      IF (NCALL .EQ. 0) THEN
!..Set these variables needed for computing radar reflectivity.  These
!.. get used within radar_init to create other variables used in the
!.. radar module.
         xam_r = 3.14159*rhowater/6.
         xbm_r = 3.
         xmu_r = 0.
         xam_s = 3.14159*rhosnow/6.
         xbm_s = 3.
         xmu_s = 0.
         if (ihail .eq. 1) then
            xam_g = 3.14159*rhohail/6.
         else
            xam_g = 3.14159*rhograul/6.
         endif
         xbm_g = 3.
         xmu_g = 0.

         call radar_init
         NCALL = 1
      ENDIF
!+---+-----------------------------------------------------------------+

!
!c  ihail = 0    for graupel, for tropical region
!c  ihail = 1    for hail, for mid-lat region

! itaobraun: 0 for Tao's constantis, 1 for Braun's constants
!c        if ( itaobraun.eq.1 ) --> betah=0.5*beta=-.46*0.5=-0.23;   cn0=1.e-6
!c        if ( itaobraun.eq.0 ) --> betah=0.5*beta=-.6*0.5=-0.30;    cn0=1.e-8
   itaobraun = 1

!c  ice2 = 0    for 3 ice --- ice, snow and graupel/hail
!c  ice2 = 1    for 2 ice --- ice and snow only
!c  ice2 = 2    for 2 ice --- ice and graupel only, use ihail = 0 only
!c  ice2 = 3    for 0 ice --- no ice, warm only

!  if (ice2 .eq. 2) ihail = 0

  i24h=nint(86400./dt_in)
  if (mod(itimestep,i24h).eq.1) then
     write(6,*) 'ihail=',ihail,'  ice2=',ice2
     if (ice2.eq.0) then
        write(6,*) 'Running 3-ice scheme in GSFCGCE with'
        if (ihail.eq.0) then 
           write(6,*) '     ice, snow and graupel'
        else if (ihail.eq.1) then
                write(6,*) '     ice, snow and hail'
        else
             write(6,*) 'ihail has to be either 1 or 0'
             stop
        endif !ihail
     else if (ice2.eq.1) then
             write(6,*) 'Running 2-ice scheme in GSFCGCE with'
             write(6,*) '     ice and snow'
     else if (ice2.eq.2) then
             write(6,*) 'Running 2-ice scheme in GSFCGCE with'
             write(6,*) '     ice and graupel'
     else if (ice2.eq.3) then
             write(6,*) 'Running warm rain only scheme in GSFCGCE without any ice'
     else
             write(6,*) 'gsfcgce_2ice in namelist.input has to be 0, 1, 2, or 3'
             stop
     endif !ice2
  endif !itimestep

!c  new_ice_sat = 0, 1 or 2 
    new_ice_sat = 2 

!c istatmin
    istatmin = 180

!c id = 0  without in-line staticstics
!c id = 1  with in-line staticstics
    id = 0

!c ibud = 0 no calculation of dth, dqv, dqrest and dqall
!c ibud = 1 yes
    ibud = 0

!jjs   dt=dt_in
!jjs   rhoe_s=1.29
!
!   IF (P_QI .lt. P_FIRST_SCALAR .or. P_QS .lt. P_FIRST_SCALAR) THEN
!      CALL wrf_error_fatal3 ( "module_mp_lin.b" , 130 ,  'module_mp_lin: Improper use of Lin et al scheme; no ice phase. Please chose another one.')
!   ENDIF

! calculte fallflux and precipiation in MKS system

   call fall_flux(dt_in, qr, qi, qs, qg, p,                   &
                      rho, z, dz8w, ht, rainnc,               &
                      rainncv, grav,itimestep,                &
                      rhowater, rhosnow,                      &
                      snownc, snowncv, sr,                    &
                      graupelnc, graupelncv,                  &
                      ihail, ice2,                            &
                      ims,ime, jms,jme, kms,kme,              & ! memory dims
                      its,ite, jts,jte, kts,kte               ) ! tile   dims
!-----------------------------------------------------------------------

!c  set up constants used internally in GCE

   call consat_s (ihail, itaobraun)


!c Negative values correction

   iskip = 1
 
   if (iskip.eq.0) then
      call negcor(qv,rho,dz8w,ims,ime,jms,jme,kms,kme, &
                           itimestep,1,             &
                           its,ite,jts,jte,kts,kte)
      call negcor(ql,rho,dz8w,ims,ime,jms,jme,kms,kme, &
                           itimestep,2,             &
                           its,ite,jts,jte,kts,kte)
      call negcor(qr,rho,dz8w,ims,ime,jms,jme,kms,kme, &
                           itimestep,3,             &
                           its,ite,jts,jte,kts,kte)
      call negcor(qi,rho,dz8w,ims,ime,jms,jme,kms,kme, &
                           itimestep,4,             &
                           its,ite,jts,jte,kts,kte)
      call negcor(qs,rho,dz8w,ims,ime,jms,jme,kms,kme, &
                           itimestep,5,             &
                           its,ite,jts,jte,kts,kte)
      call negcor(qg,rho,dz8w,ims,ime,jms,jme,kms,kme, &
                           itimestep,6,             &
                           its,ite,jts,jte,kts,kte)
!   else if (mod(itimestep,i24h).eq.1) then
!      print *,'no neg correction in mp at timestep=',itimestep
   endif ! iskip

!c microphysics in GCE

   call SATICEL_S( dt_in, IHAIL, itaobraun, ICE2, istatmin,     &
                   new_ice_sat, id,                             &
!                   th, th_old, qv, ql, qr,                      &
                   th, qv, ql, qr,                      &
                   qi, qs, qg,                                  &
!                   qvold, qlold, qrold,                         &
!                   qiold, qsold, qgold,                         &
                   rho, pii, p, itimestep,                      & 
                   refl_10cm, diagflag, do_radar_ref,           & ! GT added for reflectivity calcs
!                  refl_10cm, grid_clock, grid_alarms,          & ! GT added for reflectivity calcs
                   ids,ide, jds,jde, kds,kde,                   & ! domain dims
                   ims,ime, jms,jme, kms,kme,                   & ! memory dims
                   its,ite, jts,jte, kts,kte                    & ! tile   dims
                                                                ) 


   END SUBROUTINE gsfcgce

!-----------------------------------------------------------------------
   SUBROUTINE fall_flux ( dt, qr, qi, qs, qg, p,              &
                      rho, z, dz8w, topo, rainnc,             &
                      rainncv, grav, itimestep,               &
                      rhowater, rhosnow,                      &
                      snownc, snowncv, sr,                    &
                      graupelnc, graupelncv,                  &
                      ihail, ice2,                            &
                      ims,ime, jms,jme, kms,kme,              & ! memory dims
                      its,ite, jts,jte, kts,kte               ) ! tile   dims
!-----------------------------------------------------------------------
! adopted from Jiun-Dar Chern's codes for Purdue Regional Model
! adopted by Jainn J. Shi, 6/10/2005
!-----------------------------------------------------------------------

  IMPLICIT NONE
  INTEGER, INTENT(IN   )               :: ihail, ice2,                &
                                          ims,ime, jms,jme, kms,kme,  &
                                          its,ite, jts,jte, kts,kte 
  INTEGER, INTENT(IN   )               :: itimestep
  REAL,    DIMENSION( ims:ime , kms:kme , jms:jme ),                  &
           INTENT(INOUT)               :: qr, qi, qs, qg       
  REAL,    DIMENSION( ims:ime , jms:jme ),                            &
           INTENT(INOUT)               :: rainnc, rainncv,            &
                                          snownc, snowncv, sr,        &
                                          graupelnc, graupelncv
  REAL,    DIMENSION( ims:ime , kms:kme , jms:jme ),                  &
           INTENT(IN   )               :: rho, z, dz8w, p

  REAL,    INTENT(IN   )               :: dt, grav, rhowater, rhosnow


  REAL,    DIMENSION( ims:ime , jms:jme ),                            &
           INTENT(IN   )               :: topo   

! temperary vars

  REAL,    DIMENSION( kts:kte )           :: sqrhoz
  REAL                                    :: tmp1, term0
  REAL                                :: pptrain, pptsnow,        &
                                         pptgraul, pptice
  REAL,    DIMENSION( kts:kte )       :: qrz, qiz, qsz, qgz,      &
                                         zz, dzw, prez, rhoz,     &
                                         orhoz


   INTEGER                    :: k, i, j
!

  REAL, DIMENSION( kts:kte )    :: vtr, vts, vtg, vti

  REAL                          :: dtb, pi, consta, constc, gambp4,    &
                                   gamdp4, gam4pt5, gam4bbar

!  Lin
!-GT  REAL    , PARAMETER ::     xnor = 8.0e6
!   REAL    , PARAMETER ::     xnos = 3.0e6
!-GT   REAL    , PARAMETER ::     xnos = 1.6e7   ! Tao's value
   REAL    , PARAMETER ::                              &
!             constb = 0.8, constd = 0.25, o6 = 1./6.,           &
             constb = 0.8, constd = 0.11, o6 = 1./6.,           &
             cdrag = 0.6
!  Lin
!  REAL    , PARAMETER ::     xnoh = 4.0e4
!-GT  REAL    , PARAMETER ::     xnoh = 2.0e5    ! Tao's value
!-GT  REAL    , PARAMETER ::     rhohail = 917.

! Hobbs
!-GT  REAL    , PARAMETER ::     xnog = 4.0e6
!-GT  REAL    , PARAMETER ::     rhograul = 400.
  REAL    , PARAMETER ::     abar = 19.3, bbar = 0.37,      &
                                      p0 = 1.0e5

  REAL    , PARAMETER ::     rhoe_s = 1.29

! for terminal velocity flux
  INTEGER                       :: min_q, max_q
  REAL                          :: t_del_tv, del_tv, flux, fluxin, fluxout ,tmpqrz
  LOGICAL                       :: notlast

!  if (itimestep.eq.1) then
!     write(6, *) 'in fall_flux'
!     write(6, *) 'ims=', ims, '  ime=', ime
!     write(6, *) 'jms=', jms, '  jme=', jme
!     write(6, *) 'kms=', kms, '  kme=', kme
!     write(6, *) 'its=', its, '  ite=', ite
!     write(6, *) 'jts=', jts, '  jte=', jte
!     write(6, *) 'kts=', kts, '  kte=', kte
!     write(6, *) 'dt=', dt
!     write(6, *) 'ihail=', ihail
!     write(6, *) 'ICE2=', ICE2
!     write(6, *) 'dt=', dt
!   endif 

!-----------------------------------------------------------------------
!  This program calculates precipitation fluxes due to terminal velocities.
!-----------------------------------------------------------------------

   dtb=dt
   pi=acos(-1.)
   consta=2115.0*0.01**(1-constb)
!   constc=152.93*0.01**(1-constd)
   constc=78.63*0.01**(1-constd)

!  Gamma function
   gambp4=ggamma(constb+4.)
   gamdp4=ggamma(constd+4.)
   gam4pt5=ggamma(4.5)
   gam4bbar=ggamma(4.+bbar)

!***********************************************************************
! Calculate precipitation fluxes due to terminal velocities.
!***********************************************************************
!
!- Calculate termianl velocity (vt?)  of precipitation q?z
!- Find maximum vt? to determine the small delta t

 j_loop:  do j = jts, jte
 i_loop:  do i = its, ite

   pptrain = 0.
   pptsnow = 0.
   pptgraul = 0.
   pptice  = 0.

   do k = kts, kte
      qrz(k)=qr(i,k,j)
      rhoz(k)=rho(i,k,j)
      orhoz(k)=1./rhoz(k)
      prez(k)=p(i,k,j)
      sqrhoz(k)=sqrt(rhoe_s/rhoz(k))
      zz(k)=z(i,k,j)
      dzw(k)=dz8w(i,k,j)
   enddo !k

      DO k = kts, kte
         qiz(k)=qi(i,k,j)
      ENDDO

      DO k = kts, kte
         qsz(k)=qs(i,k,j)
      ENDDO

   IF (ice2 .eq. 0) THEN
      DO k = kts, kte
         qgz(k)=qg(i,k,j)
      ENDDO
   ELSE
      DO k = kts, kte
         qgz(k)=0.
      ENDDO
   ENDIF


!
!-- rain
!
    t_del_tv=0.
    del_tv=dtb
    notlast=.true.
    DO while (notlast)
!
      min_q=kte
      max_q=kts-1
!
      do k=kts,kte-1
         if (qrz(k) .gt. 1.0e-8) then
            min_q=min0(min_q,k)
            max_q=max0(max_q,k)
            tmp1=sqrt(pi*rhowater*xnor/rhoz(k)/qrz(k))
            tmp1=sqrt(tmp1)
            vtr(k)=consta*gambp4*sqrhoz(k)/tmp1**constb
            vtr(k)=vtr(k)/6.
            if (k .eq. 1) then
               del_tv=amin1(del_tv,0.9*(zz(k)-topo(i,j))/vtr(k))
            else
               del_tv=amin1(del_tv,0.9*(zz(k)-zz(k-1))/vtr(k))
            endif
         else
            vtr(k)=0.
         endif
      enddo

      if (max_q .ge. min_q) then
!
!- Check if the summation of the small delta t >=  big delta t
!             (t_del_tv)          (del_tv)             (dtb)

         t_del_tv=t_del_tv+del_tv
!
         if ( t_del_tv .ge. dtb ) then
              notlast=.false.
              del_tv=dtb+del_tv-t_del_tv
         endif

! use small delta t to calculate the qrz flux
! termi is the qrz flux pass in the grid box through the upper boundary
! termo is the qrz flux pass out the grid box through the lower boundary
!
         fluxin=0.
         do k=max_q,min_q,-1
            fluxout=rhoz(k)*vtr(k)*qrz(k)
            flux=(fluxin-fluxout)/rhoz(k)/dzw(k)
!            tmpqrz=qrz(k)
            qrz(k)=qrz(k)+del_tv*flux
            qrz(k)=amax1(0.,qrz(k))
            qr(i,k,j)=qrz(k)
            fluxin=fluxout
         enddo
         if (min_q .eq. 1) then
            pptrain=pptrain+fluxin*del_tv
         else
            qrz(min_q-1)=qrz(min_q-1)+del_tv*  &
                          fluxin/rhoz(min_q-1)/dzw(min_q-1)
            qr(i,min_q-1,j)=qrz(min_q-1)
         endif
!
      else
         notlast=.false.
      endif
    ENDDO

!
!-- snow
!
    t_del_tv=0.
    del_tv=dtb
    notlast=.true.

    DO while (notlast)
!
      min_q=kte
      max_q=kts-1
!
      do k=kts,kte-1
         if (qsz(k) .gt. 1.0e-8) then
            min_q=min0(min_q,k)
            max_q=max0(max_q,k)
            tmp1=sqrt(pi*rhosnow*xnos/rhoz(k)/qsz(k))
            tmp1=sqrt(tmp1)
            vts(k)=constc*gamdp4*sqrhoz(k)/tmp1**constd
            vts(k)=vts(k)/6.
            if (k .eq. 1) then
               del_tv=amin1(del_tv,0.9*(zz(k)-topo(i,j))/vts(k))
            else
               del_tv=amin1(del_tv,0.9*(zz(k)-zz(k-1))/vts(k))
            endif
         else
            vts(k)=0.
         endif
      enddo

      if (max_q .ge. min_q) then
!
!
!- Check if the summation of the small delta t >=  big delta t
!             (t_del_tv)          (del_tv)             (dtb)

         t_del_tv=t_del_tv+del_tv

         if ( t_del_tv .ge. dtb ) then
              notlast=.false.
              del_tv=dtb+del_tv-t_del_tv
         endif

! use small delta t to calculate the qsz flux
! termi is the qsz flux pass in the grid box through the upper boundary
! termo is the qsz flux pass out the grid box through the lower boundary
!
         fluxin=0.
         do k=max_q,min_q,-1
            fluxout=rhoz(k)*vts(k)*qsz(k)
            flux=(fluxin-fluxout)/rhoz(k)/dzw(k)
            qsz(k)=qsz(k)+del_tv*flux
            qsz(k)=amax1(0.,qsz(k))
            qs(i,k,j)=qsz(k)
            fluxin=fluxout
         enddo
         if (min_q .eq. 1) then
            pptsnow=pptsnow+fluxin*del_tv
         else
            qsz(min_q-1)=qsz(min_q-1)+del_tv*  &
                         fluxin/rhoz(min_q-1)/dzw(min_q-1)
            qs(i,min_q-1,j)=qsz(min_q-1)
         endif
!
      else
         notlast=.false.
      endif

    ENDDO

!
!   ice2=0 --- with hail/graupel 
!   ice2=1 --- without hail/graupel 
!
  if (ice2.eq.0) then 
!
!-- If IHAIL=1, use hail.
!-- If IHAIL=0, use graupel.
!
!    if (ihail .eq. 1) then
!       xnog = xnoh
!       rhograul = rhohail
!    endif

    t_del_tv=0.
    del_tv=dtb
    notlast=.true.
!
    DO while (notlast)
!
      min_q=kte
      max_q=kts-1
!
      do k=kts,kte-1
         if (qgz(k) .gt. 1.0e-8) then
            if (ihail .eq. 1) then
!  for hail, based on Lin et al (1983)
              min_q=min0(min_q,k)
              max_q=max0(max_q,k)
              tmp1=sqrt(pi*rhohail*xnoh/rhoz(k)/qgz(k))
              tmp1=sqrt(tmp1)
              term0=sqrt(4.*grav*rhohail/3./rhoz(k)/cdrag)
              vtg(k)=gam4pt5*term0*sqrt(1./tmp1)
              vtg(k)=vtg(k)/6.
              if (k .eq. 1) then
                 del_tv=amin1(del_tv,0.9*(zz(k)-topo(i,j))/vtg(k))
              else
                 del_tv=amin1(del_tv,0.9*(zz(k)-zz(k-1))/vtg(k))
              endif !k
            else
! added by JJS
! for graupel, based on RH (1984)
              min_q=min0(min_q,k)
              max_q=max0(max_q,k)
              tmp1=sqrt(pi*rhograul*xnog/rhoz(k)/qgz(k))
              tmp1=sqrt(tmp1)
              tmp1=tmp1**bbar
              tmp1=1./tmp1
              term0=abar*gam4bbar/6.
              vtg(k)=term0*tmp1*(p0/prez(k))**0.4
              if (k .eq. 1) then
                 del_tv=amin1(del_tv,0.9*(zz(k)-topo(i,j))/vtg(k))
              else
                 del_tv=amin1(del_tv,0.9*(zz(k)-zz(k-1))/vtg(k))
              endif !k
            endif !ihail
         else
            vtg(k)=0.
         endif !qgz
      enddo !k

      if (max_q .ge. min_q) then
!
!
!- Check if the summation of the small delta t >=  big delta t
!             (t_del_tv)          (del_tv)             (dtb)

         t_del_tv=t_del_tv+del_tv

         if ( t_del_tv .ge. dtb ) then
              notlast=.false.
              del_tv=dtb+del_tv-t_del_tv
         endif

! use small delta t to calculate the qgz flux
! termi is the qgz flux pass in the grid box through the upper boundary
! termo is the qgz flux pass out the grid box through the lower boundary
!
         fluxin=0.
         do k=max_q,min_q,-1
            fluxout=rhoz(k)*vtg(k)*qgz(k)
            flux=(fluxin-fluxout)/rhoz(k)/dzw(k)
            qgz(k)=qgz(k)+del_tv*flux
            qgz(k)=amax1(0.,qgz(k))
            qg(i,k,j)=qgz(k)
            fluxin=fluxout
         enddo
         if (min_q .eq. 1) then
            pptgraul=pptgraul+fluxin*del_tv
         else
            qgz(min_q-1)=qgz(min_q-1)+del_tv*  &
                         fluxin/rhoz(min_q-1)/dzw(min_q-1)
            qg(i,min_q-1,j)=qgz(min_q-1)
         endif
!
      else
         notlast=.false.
      endif
!
    ENDDO
 ENDIF !ice2
!
!-- cloud ice  (03/21/02) follow Vaughan T.J. Phillips at GFDL
!

    t_del_tv=0.
    del_tv=dtb
    notlast=.true.
!
    DO while (notlast)
!
      min_q=kte
      max_q=kts-1
!
      do k=kts,kte-1
         if (qiz(k) .gt. 1.0e-8) then
            min_q=min0(min_q,k)
            max_q=max0(max_q,k)
            vti(k)= 3.29 * (rhoz(k)* qiz(k))** 0.16  ! Heymsfield and Donner
            if (k .eq. 1) then
               del_tv=amin1(del_tv,0.9*(zz(k)-topo(i,j))/vti(k))
            else
               del_tv=amin1(del_tv,0.9*(zz(k)-zz(k-1))/vti(k))
            endif
         else
            vti(k)=0.
         endif
      enddo

      if (max_q .ge. min_q) then
!
!
!- Check if the summation of the small delta t >=  big delta t
!             (t_del_tv)          (del_tv)             (dtb)

         t_del_tv=t_del_tv+del_tv

         if ( t_del_tv .ge. dtb ) then
              notlast=.false.
              del_tv=dtb+del_tv-t_del_tv
         endif

! use small delta t to calculate the qiz flux
! termi is the qiz flux pass in the grid box through the upper boundary
! termo is the qiz flux pass out the grid box through the lower boundary
!

         fluxin=0.
         do k=max_q,min_q,-1
            fluxout=rhoz(k)*vti(k)*qiz(k)
            flux=(fluxin-fluxout)/rhoz(k)/dzw(k)
            qiz(k)=qiz(k)+del_tv*flux
            qiz(k)=amax1(0.,qiz(k))
            qi(i,k,j)=qiz(k)
            fluxin=fluxout
         enddo
         if (min_q .eq. 1) then
            pptice=pptice+fluxin*del_tv
         else
            qiz(min_q-1)=qiz(min_q-1)+del_tv*  &
                         fluxin/rhoz(min_q-1)/dzw(min_q-1)
            qi(i,min_q-1,j)=qiz(min_q-1)
         endif
!
      else
         notlast=.false.
      endif
!
   ENDDO !notlast

!   prnc(i,j)=prnc(i,j)+pptrain
!   psnowc(i,j)=psnowc(i,j)+pptsnow
!   pgrauc(i,j)=pgrauc(i,j)+pptgraul
!   picec(i,j)=picec(i,j)+pptice
!                     

!   write(6,*) 'i=',i,' j=',j,'   ', pptrain, pptsnow, pptgraul, pptice
!   flush(6)

   snowncv(i,j) = pptsnow
   snownc(i,j) = snownc(i,j) + pptsnow
   graupelncv(i,j) = pptgraul
   graupelnc(i,j) = graupelnc(i,j) + pptgraul 
   RAINNCV(i,j) = pptrain + pptsnow + pptgraul + pptice                 
   RAINNC(i,j)  = RAINNC(i,j) + pptrain + pptsnow + pptgraul + pptice
   sr(i,j) = 0.
   if (RAINNCV(i,j) .gt. 0.) sr(i,j) = (pptsnow + pptgraul + pptice) / RAINNCV(i,j) 

  ENDDO i_loop
  ENDDO j_loop

!  if (itimestep.eq.6480) then
!     write(51,*) 'in the end of fallflux, itimestep=',itimestep
!     do j=jts,jte
!        do i=its,ite 
!           if (rainnc(i,j).gt.400.) then
!              write(50,*) 'i=',i,' j=',j,' rainnc=',rainnc
!           endif
!        enddo
!     enddo
!  endif

  END SUBROUTINE fall_flux

!----------------------------------------------------------------
   REAL FUNCTION ggamma(X)
!----------------------------------------------------------------
   IMPLICIT NONE
!----------------------------------------------------------------
      REAL, INTENT(IN   ) :: x
      REAL, DIMENSION(8)  :: B
      INTEGER             ::j, K1
      REAL                ::PF, G1TO2 ,TEMP

      DATA B/-.577191652,.988205891,-.897056937,.918206857,  &
             -.756704078,.482199394,-.193527818,.035868343/

      PF=1.
      TEMP=X
      DO 10 J=1,200
      IF (TEMP .LE. 2) GO TO 20
      TEMP=TEMP-1.
   10 PF=PF*TEMP
  100 FORMAT(//,5X,'module_gsfcgce: INPUT TO GAMMA FUNCTION TOO LARGE, X=',E12.5)
      WRITE(wrf_err_message,100)X
      CALL wrf_error_fatal(wrf_err_message)
   20 G1TO2=1.
      TEMP=TEMP - 1.
      DO 30 K1=1,8
   30 G1TO2=G1TO2 + B(K1)*TEMP**K1
      ggamma=PF*G1TO2

      END FUNCTION ggamma

!-----------------------------------------------------------------------
!c Correction of negative values  
   SUBROUTINE negcor ( X, rho, dz8w,                         &
                      ims,ime, jms,jme, kms,kme,              & ! memory dims
                      itimestep, ics,                         &
                      its,ite, jts,jte, kts,kte               ) ! tile   dims
!-----------------------------------------------------------------------
  REAL, DIMENSION( ims:ime , kms:kme , jms:jme ),                 &
        INTENT(INOUT) ::                                     X   
  REAL, DIMENSION( ims:ime , kms:kme , jms:jme ),                 &
        INTENT(IN   ) ::                              rho, dz8w  
  integer, INTENT(IN   ) ::                           itimestep, ics 

!c Local variables
!  REAL, DIMENSION( kts:kte ) ::  Y1, Y2
  REAL   ::   A0, A1, A2

  A1=0.
  A2=0.
  do k=kts,kte
     do j=jts,jte
        do i=its,ite
        A1=A1+max(X(i,k,j), 0.)*rho(i,k,j)*dz8w(i,k,j)
        A2=A2+max(-X(i,k,j), 0.)*rho(i,k,j)*dz8w(i,k,j)
        enddo
     enddo
  enddo

!  A1=0.0
!  A2=0.0
!  do k=kts,kte
!     A1=A1+Y1(k)
!     A2=A2+Y2(k)
!  enddo

  A0=0.0

  if (A1.NE.0.0.and.A1.GT.A2) then 
     A0=(A1-A2)/A1

  if (mod(itimestep,540).eq.0) then
     if (ics.eq.1) then
        write(61,*) 'kms=',kms,'  kme=',kme,'  kts=',kts,'  kte=',kte
        write(61,*) 'jms=',jms,'  jme=',jme,'  jts=',jts,'  jte=',jte 
        write(61,*) 'ims=',ims,'  ime=',ime,'  its=',its,'  ite=',ite 
     endif 
     if (ics.eq.1) then
         write(61,*) 'qv timestep=',itimestep
         write(61,*) '  A1=',A1,'   A2=',A2,'   A0=',A0
     else if (ics.eq.2) then
             write(61,*) 'ql timestep=',itimestep
             write(61,*) '  A1=',A1,'   A2=',A2,'   A0=',A0
     else if (ics.eq.3) then
             write(61,*) 'qr timestep=',itimestep
             write(61,*) '  A1=',A1,'   A2=',A2,'   A0=',A0
     else if (ics.eq.4) then
             write(61,*) 'qi timestep=',itimestep
             write(61,*) '  A1=',A1,'   A2=',A2,'   A0=',A0
     else if (ics.eq.5) then
             write(61,*) 'qs timestep=',itimestep
             write(61,*) '  A1=',A1,'   A2=',A2,'   A0=',A0
     else if (ics.eq.6) then
             write(61,*) 'qg timestep=',itimestep
             write(61,*) '  A1=',A1,'   A2=',A2,'   A0=',A0
     else
             write(61,*) 'wrong cloud specieis number'
     endif 
  endif 

     do k=kts,kte
        do j=jts,jte
           do i=its,ite
           X(i,k,j)=A0*AMAX1(X(i,k,j), 0.0)
           enddo
        enddo
     enddo
  endif

  END SUBROUTINE negcor

 SUBROUTINE consat_s (ihail,itaobraun)

!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!                                                                      c
!   Tao, W.-K., and J. Simpson, 1989: Modeling study of a tropical     c
!   squall-type convective line. J. Atmos. Sci., 46, 177-202.          c
!                                                                      c
!   Tao, W.-K., J. Simpson and M. McCumber, 1989: An ice-water         c
!   saturation adjustment. Mon. Wea. Rev., 117, 231-235.               c
!                                                                      c
!   Tao, W.-K., and J. Simpson, 1993: The Goddard Cumulus Ensemble     c
!   Model. Part I: Model description. Terrestrial, Atmospheric and     c
!   Oceanic Sciences, 4, 35-72.                                        c
!                                                                      c
!   Tao, W.-K., J. Simpson, D. Baker, S. Braun, M.-D. Chou, B.         c
!   Ferrier,D. Johnson, A. Khain, S. Lang,  B. Lynn, C.-L. Shie,       c
!   D. Starr, C.-H. Sui, Y. Wang and P. Wetzel, 2003: Microphysics,    c
!   radiation and surface processes in the Goddard Cumulus Ensemble    c
!   (GCE) model, A Special Issue on Non-hydrostatic Mesoscale          c
!   Modeling, Meteorology and Atmospheric Physics, 82, 97-137.         c
!                                                                      c
!   Lang, S., W.-K. Tao, R. Cifelli, W. Olson, J. Halverson, S.        c
!   Rutledge, and J. Simpson, 2007: Improving simulations of           c
!   convective system from TRMM LBA: Easterly and Westerly regimes.    c
!   J. Atmos. Sci., 64, 1141-1164.                                     c
!                                                                      c
!   Coded by Tao (1989-2003), modified by S. Lang (2006/07)            c
!                                                                      c
!   Implemented into WRF  by Roger Shi 2006/2007                       c
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

!        itaobraun=0   ! see Tao and Simpson (1993)
!        itaobraun=1   ! see Tao et al. (2003)

 integer :: itaobraun
 real    :: cn0

!JJS 1/3/2008  vvvvv
!JJS   the following common blocks have been moved to the top of
!JJS   module_mp_gsfcgce_driver_instat.F
!
! real,   dimension (1:31) ::  a1, a2
! data a1/.7939e-7,.7841e-6,.3369e-5,.4336e-5,.5285e-5,.3728e-5, &
!         .1852e-5,.2991e-6,.4248e-6,.7434e-6,.1812e-5,.4394e-5,.9145e-5, &
!         .1725e-4,.3348e-4,.1725e-4,.9175e-5,.4412e-5,.2252e-5,.9115e-6, &
!         .4876e-6,.3473e-6,.4758e-6,.6306e-6,.8573e-6,.7868e-6,.7192e-6, &
!         .6513e-6,.5956e-6,.5333e-6,.4834e-6/
! data a2/.4006,.4831,.5320,.5307,.5319,.5249,.4888,.3894,.4047, &
!         .4318,.4771,.5183,.5463,.5651,.5813,.5655,.5478,.5203,.4906, &
!         .4447,.4126,.3960,.4149,.4320,.4506,.4483,.4460,.4433,.4413, &
!         .4382,.4361/
!JJS 1/3/2008  ^^^^^


!     ******************************************************************
!JJS
      al = 2.5e10
      cp = 1.004e7
      rd1 = 1.e-3
      rd2 = 2.2
!JJS
      cpi=4.*atan(1.)
      cpi2=cpi*cpi
      grvt=980.
      cd1=6.e-1
      cd2=4.*grvt/(3.*cd1)
      tca=2.43e3
      dwv=.226
      dva=1.718e-4
      amw=18.016
      ars=8.314e7
      scv=2.2904487
      t0=273.16
      t00=238.16
      alv=2.5e10
      alf=3.336e9
      als=2.8336e10
      avc=alv/cp
      afc=alf/cp
      asc=als/cp
      rw=4.615e6
      cw=4.187e7
      ci=2.093e7
      c76=7.66
      c358=35.86
      c172=17.26939
      c409=4098.026
      c218=21.87456
      c580=5807.695
      c610=6.1078e3
      c149=1.496286e-5
      c879=8.794142
      c141=1.4144354e7
!***   DEFINE THE COEFFICIENTS USED IN TERMINAL VELOCITY
!***   DEFINE THE DENSITY AND SIZE DISTRIBUTION OF PRECIPITATION
!**********   HAIL OR GRAUPEL PARAMETERS   **********
      if(ihail .eq. 1) then
         roqg=.9
         ag=sqrt(cd2*roqg)
         bg=.5
         tng=.002
      else
         roqg=.4
         ag=351.2
!        AG=372.3 ! if ice913=1 6/15/02 tao's
         bg=.37
         tng=.04
      endif
!**********         SNOW PARAMETERS        **********
!ccshie 6/15/02 tao's
!      TNS=1.
!      TNS=.08 ! if ice913=1, tao's
      tns=.16 ! if ice913=0, tao's
      roqs=.1
!      AS=152.93
      as=78.63
!      BS=.25
      bs=.11
!**********         RAIN PARAMETERS        **********
      aw=2115.
      bw=.8
      roqr=1.
      tnw=.08
!*****************************************************************
      bgh=.5*bg
      bsh=.5*bs
      bwh=.5*bw
      bgq=.25*bg
      bsq=.25*bs
      bwq=.25*bw
!**********GAMMA FUNCTION CALCULATIONS*************
      ga3b  = gammagce(3.+bw)
      ga4b  = gammagce(4.+bw)
      ga6b  = gammagce(6.+bw)
      ga5bh = gammagce((5.+bw)/2.)
      ga3g  = gammagce(3.+bg)
      ga4g  = gammagce(4.+bg)
      ga5gh = gammagce((5.+bg)/2.)
      ga3d  = gammagce(3.+bs)
      ga4d  = gammagce(4.+bs)
      ga5dh = gammagce((5.+bs)/2.)
!CCCCC        LIN ET AL., 1983 OR LORD ET AL., 1984   CCCCCCCCCCCCCCCCC
      ac1=aw
!JJS
      ac2=ag
      ac3=as
!JJS
      bc1=bw
      cc1=as
      dc1=bs
      zrc=(cpi*roqr*tnw)**0.25
      zsc=(cpi*roqs*tns)**0.25
      zgc=(cpi*roqg*tng)**0.25
      vrc=aw*ga4b/(6.*zrc**bw)
      vsc=as*ga4d/(6.*zsc**bs)
      vgc=ag*ga4g/(6.*zgc**bg)
!     ****************************
!     RN1=1.E-3
      rn1=9.4e-15 ! 6/15/02 tao's
      bnd1=6.e-4
      rn2=1.e-3
!     BND2=1.25E-3
!     BND2=1.5E-3 ! if ice913=1 6/15/02 tao's
      bnd2=2.0e-3 ! if ice913=0 6/15/02 tao's
      rn3=.25*cpi*tns*cc1*ga3d
      esw=1.
      rn4=.25*cpi*esw*tns*cc1*ga3d
!     ERI=1.
      eri=.1  ! 6/17/02 tao's ice913=0 (not 1)
      rn5=.25*cpi*eri*tnw*ac1*ga3b
!     AMI=1./(24.*4.19E-10)
      ami=1./(24.*6.e-9) ! 6/15/02 tao's
      rn6=cpi2*eri*tnw*ac1*roqr*ga6b*ami
!     ESR=1. ! also if ice913=1 for tao's
      esr=.5 ! 6/15/02 for ice913=0 tao's
      rn7=cpi2*esr*tnw*tns*roqs
      esr=1.
      rn8=cpi2*esr*tnw*tns*roqr
      rn9=cpi2*tns*tng*roqs
      rn10=2.*cpi*tns
      rn101=.31*ga5dh*sqrt(cc1)
      rn10a=als*als/rw
!JJS
       rn10b=alv/tca
       rn10c=ars/(dwv*amw)
!JJS
      rn11=2.*cpi*tns/alf
      rn11a=cw/alf
!     AMI50=1.51e-7
      ami50=3.84e-6 ! 6/15/02 tao's
!     AMI40=2.41e-8
      ami40=3.08e-8 ! 6/15/02 tao's
      eiw=1.
!     UI50=20.
      ui50=100. ! 6/15/02 tao's
      ri50=2.*5.e-3
      cmn=1.05e-15
      rn12=cpi*eiw*ui50*ri50**2

      do 10 k=1,31
         y1=1.-aa2(k)
         rn13(k)=aa1(k)*y1/(ami50**y1-ami40**y1)
         rn12a(k)=rn13(k)/ami50
         rn12b(k)=aa1(k)*ami50**aa2(k)
         rn25a(k)=aa1(k)*cmn**aa2(k)
   10 continue

      egw=1.
      rn14=.25*cpi*egw*tng*ga3g*ag
      egi=.1
      rn15=.25*cpi*egi*tng*ga3g*ag
      egi=1.
      rn15a=.25*cpi*egi*tng*ga3g*ag
      egr=1.
      rn16=cpi2*egr*tng*tnw*roqr
      rn17=2.*cpi*tng
      rn17a=.31*ga5gh*sqrt(ag)
      rn17b=cw-ci
      rn17c=cw
      apri=.66
      bpri=1.e-4
      bpri=0.5*bpri ! 6/17/02 tao's
      rn18=20.*cpi2*bpri*tnw*roqr
      rn18a=apri
      rn19=2.*cpi*tng/alf
      rn19a=.31*ga5gh*sqrt(ag)
      rn19b=cw/alf
!
       rnn191=.78
       rnn192=.31*ga5gh*sqrt(ac2/dva)
!
      rn20=2.*cpi*tng
      rn20a=als*als/rw
      rn20b=.31*ga5gh*sqrt(ag)
      bnd3=2.e-3
      rn21=1.e3*1.569e-12/0.15
      erw=1.
      rn22=.25*cpi*erw*ac1*tnw*ga3b
      rn23=2.*cpi*tnw
      rn23a=.31*ga5bh*sqrt(ac1)
      rn23b=alv*alv/rw


!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!cc
!cc        "c0" in routine      "consat" (2d), "consatrh" (3d)
!cc        if ( itaobraun.eq.1 ) --> betah=0.5*beta=-.46*0.5=-0.23;   cn0=1.e-6
!cc        if ( itaobraun.eq.0 ) --> betah=0.5*beta=-.6*0.5=-0.30;    cn0=1.e-8

       if (itaobraun .eq. 0) then
         cn0=1.e-8
         beta=-.6
       elseif (itaobraun .eq. 1) then
         cn0=1.e-6
         beta=-.46
       endif
!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!      CN0=1.E-6
!      CN0=1.E-8 ! 6/15/02 tao's
!      BETA=-.46
!      BETA=-.6  ! 6/15/02 tao's

      rn25=cn0
      rn30a=alv*als*amw/(tca*ars)
      rn30b=alv/tca
      rn30c=ars/(dwv*amw)
      rn31=1.e-17

      rn32=4.*51.545e-4
!
      rn30=2.*cpi*tng
      rnn30a=alv*alv*amw/(tca*ars)
!
      rn33=4.*tns
       rn331=.65
       rn332=.44*sqrt(ac3/dva)*ga5dh
!

    return
 END SUBROUTINE consat_s

 SUBROUTINE saticel_s (dt, ihail, itaobraun, ice2, istatmin, &
                       new_ice_sat, id, &
                       ptwrf, qvwrf, qlwrf, qrwrf, &
                       qiwrf, qswrf, qgwrf, &
                       rho_mks, pi_mks, p0_mks,itimestep, &
                       refl_10cm, diagflag, do_radar_ref,           & ! GT added for reflectivity calcs
!                      refl_10cm, grid_clock, grid_alarms,          & ! GT added for reflectivity calcs
                       ids,ide, jds,jde, kds,kde, &
                       ims,ime, jms,jme, kms,kme, &
                       its,ite, jts,jte, kts,kte  &
                           )
    IMPLICIT NONE
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!                                                                      c
!   Tao, W.-K., and J. Simpson, 1989: Modeling study of a tropical     c
!   squall-type convective line. J. Atmos. Sci., 46, 177-202.          c
!                                                                      c
!   Tao, W.-K., J. Simpson and M. McCumber, 1989: An ice-water         c
!   saturation adjustment. Mon. Wea. Rev., 117, 231-235.               c
!                                                                      c
!   Tao, W.-K., and J. Simpson, 1993: The Goddard Cumulus Ensemble     c
!   Model. Part I: Model description. Terrestrial, Atmospheric and     c
!   Oceanic Sciences, 4, 35-72.                                        c
!                                                                      c
!   Tao, W.-K., J. Simpson, D. Baker, S. Braun, M.-D. Chou, B.         c
!   Ferrier,D. Johnson, A. Khain, S. Lang,  B. Lynn, C.-L. Shie,       c
!   D. Starr, C.-H. Sui, Y. Wang and P. Wetzel, 2003: Microphysics,    c
!   radiation and surface processes in the Goddard Cumulus Ensemble    c
!   (GCE) model, A Special Issue on Non-hydrostatic Mesoscale          c
!   Modeling, Meteorology and Atmospheric Physics, 82, 97-137.         c
!                                                                      c
!   Lang, S., W.-K. Tao, R. Cifelli, W. Olson, J. Halverson, S.        c
!   Rutledge, and J. Simpson, 2007: Improving simulations of           c
!   convective system from TRMM LBA: Easterly and Westerly regimes.    c
!   J. Atmos. Sci., 64, 1141-1164.                                     c
!                                                                      c
!   Tao, W.-K., J. J. Shi,  S. Lang, C. Peters-Lidard, A. Hou, S.      c
!   Braun, and J. Simpson, 2007: New, improved bulk-microphysical      c
!   schemes for studying precipitation processes in WRF. Part I:       c
!   Comparisons with other schemes. to appear on Mon. Wea. Rev.        C
!                                                                      c
!   Coded by Tao (1989-2003), modified by S. Lang (2006/07)            c
!                                                                      c
!   Implemented into WRF  by Roger Shi 2006/2007                       c
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!
!      COMPUTE ICE PHASE MICROPHYSICS AND SATURATION PROCESSES
!
  integer,    parameter ::  nt=2880, nt2=2*nt 

!cc   using scott braun's way for pint, pidep computations
  integer  ::   itaobraun,ice2,ihail,new_ice_sat,id,istatmin
  integer  ::   itimestep
  real     ::   tairccri, cn0, dt
!cc

!JJS      common/bxyz/ n,isec,nran,kt1,kt2
!JJS      common/option/ lipps,ijkadv,istatmin,iwater,itoga,imlifting,lin,
!JJS     1   irf,iadvh,irfg,ismg,id

!JJS      common/timestat/ ndt_stat,idq
!JJS      common/iice/ new_ice_sat
!JJS      common/bt/ dt,d2t,rijl2,dts,f5,rd1,rd2,bound,al,cp,ra,ck,ce,eps,
!JJS     1    psfc,fcor,sec,aminut,rdt

!JJS   the following common blocks have been moved to the top of 
!JJS   module_mp_gsfcgce_driver_instat.F

!      common/bt/ rd1,rd2,al,cp
!
!
!      common/bterv/ zrc,zgc,zsc,vrc,vgc,vsc
!      common/size/ tnw,tns,tng,roqs,roqg,roqr
!      common/cont/ c38,c358,c610,c149,c879,c172,c409,c76,c218,c580,c141
!        common/b3cs/ ag,bg,as,bs,aw,bw,bgh,bgq,bsh,bsq,bwh,bwq
!      common/bsnw/ alv,alf,als,t0,t00,avc,afc,asc,rn1,bnd1,rn2,bnd2, &
!         rn3,rn4,rn5,rn6,rn7,rn8,rn9,rn10,rn101,rn10a,rn11,rn11a, &
!         rn12,rn12a(31),rn12b(31),rn13(31),rn14,rn15,rn15a,rn16,rn17, &
!         rn17a,rn17b,rn17c,rn18,rn18a,rn19,rn19a,rn19b,rn20,rn20a,rn20b, &
!         bnd3,rn21,rn22,rn23,rn23a,rn23b,rn25,rn25a(31),rn30a,rn30b, &
!         rn30c,rn31,beta,rn32
!      common/rsnw1/ rn10b,rn10c,rnn191,rnn192,rn30,rnn30a,rn33,rn331, &
!                    rn332
!JJS

  integer ids,ide,jds,jde,kds,kde
  integer ims,ime,jms,jme,kms,kme
  integer its,ite,jts,jte,kts,kte
  integer i,j,k, kp

!fj>
!fj  real :: a0 ,a1 ,a2 ,afcp ,alvr ,ami100 ,ami40 ,ami50 ,ascp ,avcp ,betah &
  real :: a0 ,a1 ,a2 ,ami100 ,ami40 ,ami50 ,betah &
!fj   ,bg3 ,bgh5 ,bs3 ,bs6 ,bsh5 ,bw3 ,bw6 ,bwh5 ,cmin ,cmin1 ,cmin2 ,cp409 &
   ,bg3 ,bgh5 ,bs3 ,bs6 ,bsh5 ,bw3 ,bw6 ,bwh5 ,cmin ,cmin1 ,cmin2  &
!fj   ,cp580 ,cs580 ,cv409 ,d2t ,del ,dwvp ,ee1 ,ee2 ,f00 ,f2 ,f3 ,ft ,fv0 ,fvs &
   ,d2t ,del ,ee1 ,ee2 ,f00 ,f2 ,f3 ,ft ,fv0 ,fvs &
!fj   ,pi0 ,pir ,pr0 ,qb0 ,r00 ,r0s ,r101f ,r10ar ,r10t ,r11at ,r11rt ,r12r ,r14f &
    ,pir ,pr0 ,qb0 ,r0s ,r10t ,r11at  &
!fj   ,r14r ,r15af ,r15ar ,r15f ,r15r ,r16r ,r17aq ,r17as ,r17r ,r18r ,r19aq ,r19as &
!fj   ,r19bt ,r19rt ,r20bq ,r20bs ,r20t ,r22f ,r23af ,r23br ,r23t ,r25a ,r25rt ,r2ice &
   ,r19bt ,r20t ,r23t ,r25a ,r2ice &
!fj   ,r31r ,r32rt ,r3f ,r4f ,r5f ,r6f ,r7r ,r8r ,r9r ,r_nci ,rft ,rijl2 ,rp0 ,rr0 &
   ,r_nci ,rft ,rijl2 ,rr0 &
   ,rrq ,rrs ,rt0 ,scc ,sccc ,sddd ,see ,seee ,sfff ,smmm ,ssss ,tb0 ,temp ,ucog &
!fj   ,ucor ,ucos ,uwet ,vgcf ,vgcr ,vrcf ,vscf ,zgr ,zrr ,zsr
   ,ucor ,ucos ,uwet ,zgr ,zrr ,zsr &
   ,dd,cnd,dd1,dep,ern,rsub1,y1,y2,y3,y4,y5,qsi,esi,ssi,egs,dlt1,dlt2,dlt3,dm,pgaut &
   ,pgfr,pracs,psacr,psmlt,psdep,pgmlt,qvs,pssub,pgsub &
   ,pidw,pimlt,ssw,pihom,qsw,pr,pint,wgacr,prn,psn,ps,pg,qsacr
!fj<


!fj> 
!fj  real, dimension (its:ite,jts:jte,kts:kte) ::  fv
  real, dimension (its:ite) :: rp0,pi0,r00,cp409,cv409,cp580,cs580,alvr,afcp,avcp,ascp &
                           ,vrcf,vscf,vgcf,vgcr,dwvp,r3f,r4f,r5f,r6f,r7r,r8r,r9r,r101f &
                           ,r10ar,r11rt,r12r,r14r,r14f,r15r,r15ar,r15f,r15af,r16r,r17r &
                           ,r17aq,r17as,r18r,r19rt,r19aq,r19as,r20bq,r20bs,r22f,r23af  &
                           ,r23br,r25rt,r31r,r32rt
!fj  real, dimension (its:ite,jts:jte,kts:kte) ::  dpt, dqv
!fj  real, dimension (its:ite,jts:jte,kts:kte) ::  qcl, qrn,      &
!fj                                                qci, qcs, qcg
 !fj< 
!JJS 10/16/06    vvvv
!      real dpt1(ims:ime,jms:jme,kms:kme)
!      real dqv1(ims:ime,jms:jme,kms:kme),
!     1             qcl1(ims:ime,jms:jme,kms:kme)
!      real qrn1(ims:ime,jms:jme,kms:kme),
!     1             qci1(ims:ime,jms:jme,kms:kme)
!      real qcs1(ims:ime,jms:jme,kms:kme),
!     1             qcg1(ims:ime,jms:jme,kms:kme)
!JJS 10/16/06    ^^^^

!JJS

  real, dimension (ims:ime, kms:kme, jms:jme) ::  ptwrf, qvwrf 
  real, dimension (ims:ime, kms:kme, jms:jme) ::  qlwrf, qrwrf,        &
                                                  qiwrf, qswrf, qgwrf
!JJS 10/16/06    vvvv
!      real ptwrfold(ims:ime, kms:kme, jms:jme)
!      real qvwrfold(ims:ime, kms:kme, jms:jme),
!     1             qlwrfold(ims:ime, kms:kme, jms:jme)
!      real qrwrfold(ims:ime, kms:kme, jms:jme),
!     1             qiwrfold(ims:ime, kms:kme, jms:jme)
!      real qswrfold(ims:ime, kms:kme, jms:jme),
!     1             qgwrfold(ims:ime, kms:kme, jms:jme)
!JJS 10/16/06    ^^^^

!JJS in MKS
  real, dimension (ims:ime, kms:kme, jms:jme) ::  rho_mks
  real, dimension (ims:ime, kms:kme, jms:jme) ::  pi_mks
  real, dimension (ims:ime, kms:kme, jms:jme) ::  p0_mks
!JJS
!  real, dimension (its:ite,jts:jte,kts:kte) ::  ww1
!  real, dimension (its:ite,jts:jte,kts:kte) ::  rsw
!  real, dimension (its:ite,jts:jte,kts:kte) ::  rlw

!JJS      COMMON /BADV/
!fj>
!fj  real, dimension (its:ite,jts:jte) ::        &
  real, dimension (its:ite) ::        &
!fj           vg,      zg,       &
!fj<
!fj           ps,      pg,       &
!fj          prn,     psn,       &
!fj        pwacs,   wgacr,       &
!fj        pidep,    pint,       &
        pidep,           &
!fj          qsi,     ssi,       &
!fj          esi,     esw,       &
!fj          qsw,      pr,       &
!fj          ssw,   pihom,       &
!fj         pidw,   pimlt,       &
!fj        psaut,   qracs,       &
        psaut,          &
        psaci,   psacw,       &
        qsacw,   praci       
!fj        pmlts,   pmltg,       &
!fj        asss,       y1,    y2
!JJS       Y2(its:ite,jts:jte),    DDE(NB)

!JJS      COMMON/BSAT/
!fj>
!fj  real, dimension (its:ite,jts:jte) ::        &
  real, dimension (its:ite) ::        &
!fj<
     praut,   pracw,       &
      psfw,    psfi,       &
        dgacs,   dgacw,       &
        dgaci,   dgacr,       &
        pgacs,   wgacs,       &
        qgacw,   wgaci,       &
        qgacr,   pgwet,       &
!fj        pgaut,   pracs,       &
!fj        psacr,   qsacr,       &
!fj         pgfr,   psmlt,       &
!fj        pgmlt,   psdep,       &
!fj        pgdep,   piacr,       &
           piacr,       &
!fj           y5,     scv,       &
          scv,       &
          tca,     dwv
!fj          egs,      y3,       &
!fj           y4,     ddb

!JJS      COMMON/BSAT1/
!fj  real, dimension (its:ite,jts:jte) ::        &
  real, dimension (its:ite) ::        &
           pt,      qv,       &
           qc,      qr,       &
           qi,      qs,       &
!fj           qg,    tair,       &
!fj        tairc,   rtair,       &
           qg

!fj          dep,      dd,       &
!fj          dd1,     qvs,       &
!fj           dm,      rq,       &
!fj        rsub1,     col,       &
!fj          cnd,     ern,       &
!fj         dlt1,    dlt2,       &
!fj         dlt3,    dlt4,       &
!fj           zr,      vr,       &
!fj           zs,      vs,       &
!fj                 pssub,       &
!fj        pgsub,     dda
!fj>
  real, dimension (its:ite) ::        &
           tair, tairc, rtair,        &
           zr, zs, vr, vs, zg, vg
!fj<
!JJS      COMMON/B5/
!fj>
!fj  real, dimension (its:ite,jts:jte,kts:kte) ::  rho
!fj<
  real, dimension (kts:kte) ::                 & 
           tb,      qb,    rho1,              &
           ta,      qa,     ta1,     qa1,     &
         coef,      z1,      z2,      z3,     &
           am,     am1,      ub,      vb,     &
           wb,     ub1,     vb1,    rrho,     &
        rrho1,     wbx

!JJS      COMMON/B6/
!fj>
  !fjreal, dimension (its:ite,jts:jte,kts:kte) ::  p0, pi, f0
  real, dimension (its:ite) ::  p0
  real, dimension (kts:kte) ::    & 
           fd,      fe,        &
           st,      sv,        &
           sq,      sc,        &
           se,     sqa

!JJS      COMMON/BRH1/
  real, dimension (kts:kte) ::    & 
         srro,    qrro,    sqc,    sqr,    &
          sqi,     sqs,    sqg,   stqc,    &
         stqr,    stqi,   stqs,   stqg
  real, dimension (nt) ::    & 
          tqc,     tqr,    tqi,    tqs,    tqg

!JJS     common/bls/ y0(nx,ny),ts0new(nx,ny),qss0new(nx,ny)

!JJS      COMMON/BLS/
!fj>
!fj  real, dimension (ims:ime,jms:jme) ::     &
!fj           y0,     ts0,   qss0
!fj<

!JJS      COMMON/BI/ IT(its:ite,jts:jte), ICS(its:ite,jts:jte,4)
!fj>
!fj  integer, dimension (its:ite,jts:jte) ::        it  
  integer  ::        it  
!fj   integer, dimension (its:ite,jts:jte, 4) ::    ics 
!fj<

  integer :: i24h
  integer :: iwarm
  real :: r2is, r2ig
  

!JJS      COMMON/MICRO/
!  real, dimension (ims:ime,kms:kme,jms:jme)  ::    dbz 

!23456789012345678901234567890123456789012345678901234567890123456789012

!
!JJS 1/3/2008  vvvvv
!JJS   the following common blocks have been moved to the top of
!JJS   module_mp_gsfcgce_driver.F

!  real, dimension (31)   ::      aa1,  aa2
!  data aa1/.7939e-7, .7841e-6, .3369e-5, .4336e-5, .5285e-5,     &
!           .3728e-5, .1852e-5, .2991e-6, .4248e-6, .7434e-6,     &
!           .1812e-5, .4394e-5, .9145e-5, .1725e-4, .3348e-4,     &
!           .1725e-4, .9175e-5, .4412e-5, .2252e-5, .9115e-6,     &
!           .4876e-6, .3473e-6, .4758e-6, .6306e-6, .8573e-6,     &
!           .7868e-6, .7192e-6, .6513e-6, .5956e-6, .5333e-6,     &
!           .4834e-6/
!  data aa2/.4006, .4831, .5320, .5307, .5319,      &
!           .5249, .4888, .3894, .4047, .4318,      &
!           .4771, .5183, .5463, .5651, .5813,      &
!           .5655, .5478, .5203, .4906, .4447,      &
!           .4126, .3960, .4149, .4320, .4506,      &
!           .4483, .4460, .4433, .4413, .4382,      &
!           .4361/

!JJS 1/3/2008  ^^^^^

!+---+-----------------------------------------------------------------+
  REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT):: refl_10cm  ! GT
! TYPE (WRFU_Clock):: grid_clock
! TYPE (WRFU_Alarm), POINTER:: grid_alarms(:)


      REAL, DIMENSION(kts:kte):: qv1d, t1d, p1d, qr1d, qs1d, qg1d, dBZ
      LOGICAL, OPTIONAL, INTENT(IN) :: diagflag
      INTEGER, OPTIONAL, INTENT(IN) :: do_radar_ref
!+---+-----------------------------------------------------------------+
!
!jm 20090220      save

!    i24h=nint(86400./dt) 
!    if (mod(itimestep,i24h).eq.1) then
!       write(6, *) 'ims=', ims, '  ime=', ime
!       write(6, *) 'jms=', jms, '  jme=', jme
!       write(6, *) 'kms=', kms, '  kme=', kme
!       write(6, *) 'its=', its, '  ite=', ite
!       write(6, *) 'jts=', jts, '  jte=', jte
!       write(6, *) 'kts=', kts, '  kte=', kte
!       write(6, *) '    ihail=', ihail
!       write(6, *) 'itaobraun=',itaobraun
!       write(6, *) '    ice2=', ICE2
!       write(6, *) 'istatmin=',istatmin
!       write(6, *) 'new_ice_sat=', new_ice_sat
!       write(6, *) 'id=', id
!       write(6, *) 'dt=', dt
!    endif

!JJS  convert from mks to cgs, and move from WRF grid to GCE grid
!fj>
!fj      do k=kts,kte
!fj         do j=jts,jte
!fj         do i=its,ite
!fj         rho(i,j,k)=rho_mks(i,k,j)*0.001
!fj         p0(i,j,k)=p0_mks(i,k,j)*10.0
!fj         pi(i,j,k)=pi_mks(i,k,j)
!fj         dpt(i,j,k)=ptwrf(i,k,j)
!fj         dqv(i,j,k)=qvwrf(i,k,j)
!fj         qcl(i,j,k)=qlwrf(i,k,j)
!fj         qrn(i,j,k)=qrwrf(i,k,j)
!fj         qci(i,j,k)=qiwrf(i,k,j)
!fj         qcs(i,j,k)=qswrf(i,k,j)
!fj         qcg(i,j,k)=qgwrf(i,k,j)
!JJS 10/16/06    vvvv
!         dpt1(i,j,k)=ptwrfold(i,k,j)
!         dqv1(i,j,k)=qvwrfold(i,k,j)
!         qcl1(i,j,k)=qlwrfold(i,k,j)
!         qrn1(i,j,k)=qrwrfold(i,k,j)
!         qci1(i,j,k)=qiwrfold(i,k,j)
!         qcs1(i,j,k)=qswrfold(i,k,j)
!         qcg1(i,j,k)=qgwrfold(i,k,j)
!JJS 10/16/06     ^^^^
!fj         enddo !i
!fj         enddo !j
!fj      enddo !k
!fj
!fj      do k=kts,kte
!fj         do j=jts,jte
!fj         do i=its,ite
!fj         fv(i,j,k)=sqrt(rho(i,j,2)/rho(i,j,k))
!fj         enddo !i
!fj         enddo !j
!fj      enddo !k
!fj<
!JJS

!
!     ******   THREE CLASSES OF ICE-PHASE   (LIN ET AL, 1983)  *********

!JJS       D22T=D2T
!JJS       IF(IJKADV .EQ. 0) THEN
!JJS         D2T=D2T
!JJS       ELSE
         d2t=dt
!JJS       ENDIF
!
!       itaobraun=0 ! original pint and pidep & see Tao and Simpson 1993
        itaobraun=1 ! see Tao et al. (2003)
!
       if ( itaobraun.eq.0 ) then
          cn0=1.e-8
!c        beta=-.6
       elseif ( itaobraun.eq.1 ) then
          cn0=1.e-6
!         cn0=1.e-8  ! special
!c        beta=-.46
       endif
!C  TAO 2007 START
!   ICE2=0 ! default, 3ice with loud ice, snow and graupel
!              r2is=1., r2ig=1.
!   ICE2=1 ! 2ice with cloud ice and snow (no graupel) - r2iceg=1, r2ice=0.
!              r2is=1., r2ig=0.
!   ICE2=2 ! 2ice with cloud ice and graupel (no snow) - r2ice=1, r2iceg=0.
!              r2is=0., r2ig=1.
!c
!        r2ice=1.
!        r2iceg=1.
         r2ig=1.
         r2is=1.
          if (ice2 .eq. 1) then
!              r2ice=0.
!              r2iceg=1.
              r2ig=0.
              r2is=1.
          endif
          if (ice2 .eq. 2) then
!              r2ice=1.
!              r2iceg=0.
              r2ig=1.
              r2is=0.
          endif
!C  TAO 2007 END
     
!JJS  10/7/2008
!   ICE2=3 ! no ice, warm rain only
    iwarm = 0
    if (ice2 .eq. 3 ) iwarm = 1



      cmin=1.e-19
      cmin1=1.e-20
      cmin2=1.e-12
!hcl  ucor=3071.29/tnw**0.75
!hcl  ucos=687.97*roqs**0.25/tns**0.75
!hcl  ucog=687.97*roqg**0.25/tng**0.75
!hcl  uwet=4.464**0.95
      ucor=3071.29/(tnw/sqrt(sqrt(tnw)))
      ucos=687.97*sqrt(sqrt(roqs))/(tns/sqrt(sqrt(tns)))
      ucog=687.97*sqrt(sqrt(roqg))/(tng/sqrt(sqrt(tng)))
      uwet=exp(0.95*log(4.464))

      rijl2 = 1. / (ide-ids) / (jde-jds)

!JJScap $doacross local(j,i)

!JJS      DO 1 J=1,JMAX
!JJS      DO 1 I=1,IMAX
!fj>
!fj       do j=jts,jte
!fj          do i=its,ite
!fj          it(i,j)=1
!fj          enddo
!fj       enddo
!fj<

      f2=rd1*d2t
      f3=rd2*d2t

      ft=dt/d2t
      rft=rijl2*ft
      a0=.5*istatmin*rijl2
      rt0=1./(t0-t00)
      bw3=bw+3.
      bs3=bs+3.
      bg3=bg+3.
      bsh5=2.5+bsh
      bgh5=2.5+bgh
      bwh5=2.5+bwh
      bw6=bw+6.
      bs6=bs+6.
      betah=.5*beta
      r10t=rn10*d2t
      r11at=rn11a*d2t
      r19bt=rn19b*d2t
      r20t=-rn20*d2t
      r23t=-rn23*d2t
      r25a=rn25

!     ami50 for use in PINT
       ami50=3.76e-8
       ami100=1.51e-7
       ami40=2.41e-8

!C    ******************************************************************

!JJS      DO 1000 K=2,kles
      do 1000 k=kts,kte
         kp=k+1
!JJS         tb0=ta1(k)
!JJS         qb0=qa1(k)
         tb0=0.
         qb0=0.

      do 2000 j=jts,jte
!fj>
!fj         do 2000 i=its,ite
!OCL NOPREFETCH
         do 2011 i=its,ite
         r00(i)=rho_mks(i,k,j)*0.001
         p0(i)=p0_mks(i,k,j)*10.0
         pi0(i)=pi_mks(i,k,j)
         pt(i)=ptwrf(i,k,j)
         qv(i)=qvwrf(i,k,j)
         qc(i)=qlwrf(i,k,j)
         qr(i)=qrwrf(i,k,j)
         qi(i)=qiwrf(i,k,j)
         qs(i)=qswrf(i,k,j)
         qg(i)=qgwrf(i,k,j)
         fv0=sqrt(rho_mks(i,2,j)*0.001/r00(i))

!fj         rp0=3.799052e3/p0(i,j,k)
         rp0(i)=3.799052e3/p0(i)
!fj         pi0=pi(i,j,k)
!fj         pir=1./(pi(i,j,k))
!fj         pr0=1./p0(i,j,k)
         pir=1./pi0(i)
         pr0=1./p0(i)
!fj         r00=rho(i,j,k)
!fj         r0s=sqrt(rho(i,j,k))
!JJS         RR0=RRHO(K)
!fj         rr0=1./rho(i,j,k)
         rr0=1./r00(i)
!JJS         RRS=SRRO(K)
         rrs=sqrt(rr0)
!JJS         RRQ=QRRO(K)
         rrq=sqrt(rrs)
!fj>
!fj         f0(i,j,k)=al/cp/pi(i,j,k)
!fj         f00=f0(i,j,k)
!         f00=al/cp/pi0
!fj         fv0=fv(i,j,k)
!fj         fvs=sqrt(fv(i,j,k))
         fvs=sqrt(fv0)
!fj<
         zrr=1.e5*zrc*rrq
         zsr=1.e5*zsc*rrq
         zgr=1.e5*zgc*rrq
!fj>
!fj         cp409=c409*pi0
!fj         cv409=c409*avc
!fj         cp580=c580*pi0
!fj         cs580=c580*asc
!fj         alvr=r00*alv
!fj         afcp=afc*pir
!fj         avcp=avc*pir
!fj         ascp=asc*pir
!fj         vrcf=vrc*fv0
!fj         vscf=vsc*fv0
!fj         vgcf=vgc*fv0
!fj         vgcr=vgc*rrs
!fj         dwvp=c879*pr0
!fj         r3f=rn3*fv0
!fj         r4f=rn4*fv0
!fj         r5f=rn5*fv0
!fj         r6f=rn6*fv0
!fj         r7r=rn7*rr0
!fj         r8r=rn8*rr0
!fj         r9r=rn9*rr0
!fj         r101f=rn101*fvs
!fj         r10ar=rn10a*r00
!fj         r11rt=rn11*rr0*d2t
!fj         r12r=rn12*r00
!fj         r14r=rn14*rrs
!fj         r14f=rn14*fv0
!fj         r15r=rn15*rrs
!fj         r15ar=rn15a*rrs
!fj         r15f=rn15*fv0
!fj         r15af=rn15a*fv0
!fj         r16r=rn16*rr0
!fj         r17r=rn17*rr0
!fj         r17aq=rn17a*rrq
!fj         r17as=rn17a*fvs
!fj         r18r=rn18*rr0
!fj         r19rt=rn19*rr0*d2t
!fj         r19aq=rn19a*rrq
!fj         r19as=rn19a*fvs
!fj         r20bq=rn20b*rrq
!fj         r20bs=rn20b*fvs
!fj         r22f=rn22*fv0
!fj         r23af=rn23a*fvs
!fj         r23br=rn23b*r00
!fj         r25rt=rn25*rr0*d2t
!fj         r31r=rn31*rr0
!fj         r32rt=rn32*d2t*rrs
         cp409(i)=c409*pi0(i)
         cv409(i)=c409*avc
         cp580(i)=c580*pi0(i)
         cs580(i)=c580*asc
         alvr(i)=r00(i)*alv
         afcp(i)=afc*pir
         avcp(i)=avc*pir
         ascp(i)=asc*pir
         vrcf(i)=vrc*fv0
         vscf(i)=vsc*fv0
         vgcf(i)=vgc*fv0
         vgcr(i)=vgc*rrs
         dwvp(i)=c879*pr0
         r3f(i)=rn3*fv0
         r4f(i)=rn4*fv0
         r5f(i)=rn5*fv0
         r6f(i)=rn6*fv0
         r7r(i)=rn7*rr0
         r8r(i)=rn8*rr0
         r9r(i)=rn9*rr0
         r101f(i)=rn101*fvs
         r10ar(i)=rn10a*r00(i)
         r11rt(i)=rn11*rr0*d2t
         r12r(i)=rn12*r00(i)
         r14r(i)=rn14*rrs
         r14f(i)=rn14*fv0
         r15r(i)=rn15*rrs
         r15ar(i)=rn15a*rrs
         r15f(i)=rn15*fv0
         r15af(i)=rn15a*fv0
         r16r(i)=rn16*rr0
         r17r(i)=rn17*rr0
         r17aq(i)=rn17a*rrq
         r17as(i)=rn17a*fvs
         r18r(i)=rn18*rr0
         r19rt(i)=rn19*rr0*d2t
         r19aq(i)=rn19a*rrq
         r19as(i)=rn19a*fvs
         r20bq(i)=rn20b*rrq
         r20bs(i)=rn20b*fvs
         r22f(i)=rn22*fv0
         r23af(i)=rn23a*fvs
         r23br(i)=rn23b*r00(i)
         r25rt(i)=rn25*rr0*d2t
         r31r(i)=rn31*rr0
         r32rt(i)=rn32*d2t*rrs
!fj<

!JJS       DO 100 J=3,JLES
!JJS       DO 100 I=3,ILES
!fj>
!fj        pt(i,j)=dpt(i,j,k)
!fj        qv(i,j)=dqv(i,j,k)
!fj        qc(i,j)=qcl(i,j,k)
!fj        qr(i,j)=qrn(i,j,k)
!fj        qi(i,j)=qci(i,j,k)
!fj        qs(i,j)=qcs(i,j,k)
!fj        qg(i,j)=qcg(i,j,k)
!        IF (QV(I,J)+QB0 .LE. 0.) QV(I,J)=-QB0
!fj         if (qc(i,j) .le.  cmin1) qc(i,j)=0.0
!fj         if (qr(i,j) .le.  cmin1) qr(i,j)=0.0
!fj         if (qi(i,j) .le.  cmin1) qi(i,j)=0.0
!fj         if (qs(i,j) .le.  cmin1) qs(i,j)=0.0
!fj         if (qg(i,j) .le.  cmin1) qg(i,j)=0.0
         if (qc(i) .le.  cmin1) qc(i)=0.0
         if (qr(i) .le.  cmin1) qr(i)=0.0
         if (qi(i) .le.  cmin1) qi(i)=0.0
         if (qs(i) .le.  cmin1) qs(i)=0.0
         if (qg(i) .le.  cmin1) qg(i)=0.0
!fj        tair(i,j)=(pt(i,j)+tb0)*pi0
!fj        tairc(i,j)=tair(i,j)-t0
!fj         zr(i,j)=zrr
!fj         zs(i,j)=zsr
!fj         zg(i,j)=zgr
!fj         vr(i,j)=0.0
!fj         vs(i,j)=0.0
!fj         vg(i,j)=0.0
        tair(i)=(pt(i)+tb0)*pi0(i)
        tairc(i)=tair(i)-t0
         zr(i)=zrr
         zs(i)=zsr
         zg(i)=zgr
         vr(i)=0.0
         vs(i)=0.0
         vg(i)=0.0
 2011 continue
!fj<

!JJS 10/7/2008     vvvvv
    IF (IWARM .EQ. 1) THEN
!JJS   for calculating processes related to warm rain only
!fj>
         do 2020 i=its,ite
!fj                qi(i,j)=0.0
!fj                qs(i,j)=0.0
!fj                qg(i,j)=0.0
!fj                dep(i,j)=0.
!fj                pint(i,j)=0.
!fj                psdep(i,j)=0.
!fj                pgdep(i,j)=0.
!fj                dd1(i,j)=0.
!fj                pgsub(i,j)=0.
!fj                psmlt(i,j)=0.
!fj                pgmlt(i,j)=0.
!fj                pimlt(i,j)=0.
!fj                psacw(i,j)=0.
!fj                piacr(i,j)=0.
!fj                psfw(i,j)=0.
!fj                pgfr(i,j)=0.
!fj                dgacw(i,j)=0.
!fj                dgacr(i,j)=0.
!fj                psacr(i,j)=0.
!fj                wgacr(i,j)=0.
!fj                pihom(i,j)=0.
!fj                pidw(i,j)=0.
                qi(i)=0.0
                qs(i)=0.0
                qg(i)=0.0
                dep=0.
                pint=0.
                psdep=0.
!fj                pgdep(i)=0.
                dd1=0.
                pgsub=0.
                psmlt=0.
                pgmlt=0.
                pimlt=0.
                psacw(i)=0.
                piacr(i)=0.
                psfw(i)=0.
                pgfr=0.
                dgacw(i)=0.
                dgacr(i)=0.
                psacr=0.
                wgacr=0.
                pihom=0.
                pidw=0.

!fj                if (qr(i,j) .gt. cmin1) then
                if (qr(i) .gt. cmin1) then
!fj                   dd(i,j)=r00*qr(i,j)
                   dd=r00(i)*qr(i)
!fj                   y1(i,j)=dd(i,j)**.25
                   y1=sqrt(sqrt(dd))    !hcl
!fj                   zr(i,j)=zrc/y1(i,j)
                   zr(i)=zrc/y1
!fj                   vr(i,j)=max(vrcf*dd(i,j)**bwq, 0.)
                   vr(i)=max(vrcf(i)*exp(bwq*log(dd)), 0.)  !hcl
                endif

!* 21 * PRAUT   AUTOCONVERSION OF QC TO QR                        **21**
!* 22 * PRACW : ACCRETION OF QC BY QR                             **22**
!fj                pracw(i,j)=0.
!fj                praut(i,j)=0.0
                pracw(i)=0.
                praut(i)=0.0
!fj                pracw(i,j)=r22f*qc(i,j)/zr(i,j)**bw3
                pracw(i)=r22f(i)*qc(i)/exp(bw3*log(zr(i)))   !hcl
!fj                y1(i,j)=qc(i,j)-bnd3
                y1=qc(i)-bnd3
!fj                if (y1(i,j).gt.0.0) then
                if (y1.gt.0.0) then
!fj                    praut(i,j)=r00*y1(i,j)*y1(i,j)/(1.2e-4+rn21/y1(i,j))
                    praut(i)=r00(i)*y1*y1/(1.2e-4+rn21/y1)
                 endif

!C********   HANDLING THE NEGATIVE CLOUD WATER (QC)    ******************
!fj>
!fj                 Y1(I,J)=QC(I,J)/D2T
!fj                 PRAUT(I,J)=MIN(Y1(I,J), PRAUT(I,J))
!fj                 PRACW(I,J)=MIN(Y1(I,J), PRACW(I,J))
!fj                 Y1(I,J)=(PRAUT(I,J)+PRACW(I,J))*D2T
                 Y1=QC(I)/D2T
                 PRAUT(I)=MIN(Y1, PRAUT(I))
                 PRACW(I)=MIN(Y1, PRACW(I))
                 Y1=(PRAUT(I)+PRACW(I))*D2T
!fj<
               
!fj>
!fj               if (qc(i,j) .lt. y1(i,j) .and. y1(i,j) .ge. cmin2) then
!fj                   y2(i,j)=qc(i,j)/(y1(i,j)+cmin2)
!fj                   praut(i,j)=praut(i,j)*y2(i,j)
!fj                   pracw(i,j)=pracw(i,j)*y2(i,j)
!fj                   qc(i,j)=0.0
!fj               else
!fj                  qc(i,j)=qc(i,j)-y1(i,j)
!fj               endif
!fj               
!fj               PR(I,J)=(PRAUT(I,J)+PRACW(I,J))*D2T
!fj               QR(I,J)=QR(I,J)+PR(I,J)
               if (qc(i) .lt. y1 .and. y1 .ge. cmin2) then
                   y2=qc(i)/(y1+cmin2)
                   praut(i)=praut(i)*y2
                   pracw(i)=pracw(i)*y2
                   qc(i)=0.0
               else
                  qc(i)=qc(i)-y1
               endif
               
               PR=(PRAUT(I)+PRACW(I))*D2T
               QR(I)=QR(I)+PR
!fj<
                        
!*****   TAO ET AL (1989) SATURATION TECHNIQUE  ***********************
           
!fj           cnd(i,j)=0.0
           cnd=0.0
!fj           tair(i,j)=(pt(i,j)+tb0)*pi0
           tair(i)=(pt(i)+tb0)*pi0(i)
!fj              y1(i,j)=1./(tair(i,j)-c358)
              y1=1./(tair(i)-c358)
!fj              qsw(i,j)=rp0*exp(c172-c409*y1(i,j))
              qsw=rp0(i)*exp(c172-c409*y1)
!fj              dd(i,j)=cp409*y1(i,j)*y1(i,j)
              dd=cp409(i)*y1*y1
!fj              dm(i,j)=qv(i,j)+qb0-qsw(i,j)
              dm=qv(i)+qb0-qsw
!fj              cnd(i,j)=dm(i,j)/(1.+avcp*dd(i,j)*qsw(i,j))
              cnd=dm/(1.+avcp(i)*dd*qsw)
!c    ******   condensation or evaporation of qc  ******
!fj              cnd(i,j)=max(-qc(i,j), cnd(i,j))
              cnd=max(-qc(i), cnd)
!fj                         pt(i,j)=pt(i,j)+avcp*cnd(i,j)
                         pt(i)=pt(i)+avcp(i)*cnd
!fj             qv(i,j)=qv(i,j)-cnd(i,j)
!fj                         qc(i,j)=qc(i,j)+cnd(i,j)
             qv(i)=qv(i)-cnd
                         qc(i)=qc(i)+cnd

!C     ******   EVAPORATION   ******
!* 23 * ERN : EVAPORATION OF QR (SUBSATURATION)                   **23**
!fj            ern(i,j)=0.0
            ern=0.0

!fj            if(qr(i,j).gt.0.0) then
            if(qr(i).gt.0.0) then
!fj               tair(i,j)=(pt(i,j)+tb0)*pi0
               tair(i)=(pt(i)+tb0)*pi0(i)
!fj               rtair(i,j)=1./(tair(i,j)-c358)
               rtair(i)=1./(tair(i)-c358)
!fj               qsw(i,j)=rp0*exp(c172-c409*rtair(i,j))
               qsw=rp0(i)*exp(c172-c409*rtair(i))
!fj               ssw(i,j)=(qv(i,j)+qb0)/qsw(i,j)-1.0
!fj               dm(i,j)=qv(i,j)+qb0-qsw(i,j)
               ssw=(qv(i)+qb0)/qsw-1.0
               dm=qv(i)+qb0-qsw
!fj               rsub1(i,j)=cv409*qsw(i,j)*rtair(i,j)*rtair(i,j)
               rsub1=cv409(i)*qsw*rtair(i)*rtair(i)
!fj               dd1(i,j)=max(-dm(i,j)/(1.+rsub1(i,j)), 0.0)
               dd1=max(-dm/(1.+rsub1), 0.0)
!fj               y1(i,j)=.78/zr(i,j)**2+r23af*scv(i,j)/zr(i,j)**bwh5
               y1=.78/zr(i)*zr(i)+r23af(i)*scv(i)/exp(bwh5*log(zr(i)))   !hcl
!fj               y2(i,j)=r23br/(tca(i,j)*tair(i,j)**2)+1./(dwv(i,j) &
               y2=r23br(i)/(tca(i)*tair(i)*tair(i))+1./(dwv(i) &   !hcl
!fj                       *qsw(i,j))
                       *qsw)
!cccc
!fj               ern(i,j)=r23t*ssw(i,j)*y1(i,j)/y2(i,j)
!fj               ern(i,j)=min(dd1(i,j),qr(i,j),max(ern(i,j),0.))
               ern=r23t*ssw*y1/y2
               ern=min(dd1,qr(i),max(ern,0.))
!fj               pt(i,j)=pt(i,j)-avcp*ern(i,j)
               pt(i)=pt(i)-avcp(i)*ern
!fj               qv(i,j)=qv(i,j)+ern(i,j)
!fj               qr(i,j)=qr(i,j)-ern(i,j)
               qv(i)=qv(i)+ern
               qr(i)=qr(i)-ern
            endif

!fj>
 2020 continue
!fj<
       ELSE       ! part of if (iwarm.eq.1) then
!JJS 10/7/2008     ^^^^^

!JJS   for calculating processes related to both ice and warm rain

!     ***   COMPUTE ZR,ZS,ZG,VR,VS,VG      *****************************
!fj>
         do 2030 i=its,ite
!fj<

!fj            if (qr(i,j) .gt. cmin1) then
            if (qr(i) .gt. cmin1) then
!fj               dd(i,j)=r00*qr(i,j)
               dd=r00(i)*qr(i)
!fj               y1(i,j)=dd(i,j)**.25
               y1=sqrt(sqrt(dd))   !hcl
!fj               zr(i,j)=zrc/y1(i,j)
               zr(i)=zrc/y1
!fj               vr(i,j)=max(vrcf*dd(i,j)**bwq, 0.)
               vr(i)=max(vrcf(i)*exp(bwq*log(dd)), 0.)   !hcl
            endif

!fj            if (qs(i,j) .gt. cmin1) then
            if (qs(i) .gt. cmin1) then
!fj               dd(i,j)=r00*qs(i,j)
               dd=r00(i)*qs(i)
!fj               y1(i,j)=dd(i,j)**.25
               y1=sqrt(sqrt(dd))   !hcl
!fj               zs(i,j)=zsc/y1(i,j)
               zs(i)=zsc/y1
!fj               vs(i,j)=max(vscf*dd(i,j)**bsq, 0.)
               vs(i)=max(vscf(i)*exp(bsq*log(dd)), 0.)   !hcl
            endif

!fj            if (qg(i,j) .gt. cmin1) then
            if (qg(i) .gt. cmin1) then
!fj               dd(i,j)=r00*qg(i,j)
               dd=r00(i)*qg(i)
!fj               y1(i,j)=dd(i,j)**.25
               y1=sqrt(sqrt(dd))   !hcl
!fj               zg(i,j)=zgc/y1(i,j)
               zg(i)=zgc/y1
               if(ihail .eq. 1) then
!fj                  vg(i,j)=max(vgcr*dd(i,j)**bgq, 0.)
                  vg(i)=max(vgcr(i)*exp(bgq*log(dd)), 0.)   !hcl
               else
!fj                  vg(i,j)=max(vgcf*dd(i,j)**bgq, 0.)
                  vg(i)=max(vgcf(i)*exp(bgq*log(dd)), 0.)   !hcl
               endif
            endif

!fj            if (qr(i,j) .le. cmin2) vr(i,j)=0.0
            if (qr(i) .le. cmin2) vr(i)=0.0
!fj            if (qs(i,j) .le. cmin2) vs(i,j)=0.0
            if (qs(i) .le. cmin2) vs(i)=0.0
!fj            if (qg(i,j) .le. cmin2) vg(i,j)=0.0
            if (qg(i) .le. cmin2) vg(i)=0.0

!     ******************************************************************
!     ***   Y1 : DYNAMIC VISCOSITY OF AIR (U)
!     ***   DWV : DIFFUSIVITY OF WATER VAPOR IN AIR (PI)
!     ***   TCA : THERMAL CONDUCTIVITY OF AIR (KA)
!     ***   Y2 : KINETIC VISCOSITY (V)

!fj            y1(i,j)=c149*tair(i,j)**1.5/(tair(i,j)+120.)
            y1=c149*tair(i)*sqrt(tair(i))/(tair(i)+120.)   !hcl
!fj            dwv(i,j)=dwvp*tair(i,j)**1.81
            dwv(i)=dwvp(i)*exp(1.81*log(tair(i)))   !hcl
!fj            tca(i,j)=c141*y1(i,j)
            tca(i)=c141*y1
!fj            scv(i,j)=1./((rr0*y1(i,j))**.1666667*dwv(i,j)**.3333333)
            scv(i)=1./(exp(.1666667*log(1./r00(i)*y1))*exp(.3333333*log(dwv(i))))   !hcl
!JJS         RRS=SRRO(K)
 2030 continue
!JJS  100    CONTINUE

!*  1 * PSAUT : AUTOCONVERSION OF QI TO QS                        ***1**
!*  3 * PSACI : ACCRETION OF QI TO QS                             ***3**
!*  4 * PSACW : ACCRETION OF QC BY QS (RIMING) (QSACW FOR PSMLT)  ***4**
!*  5 * PRACI : ACCRETION OF QI BY QR                             ***5**
!*  6 * PIACR : ACCRETION OF QR OR QG BY QI                       ***6**

!JJS         DO 125 J=3,JLES
!JJS         DO 125 I=3,ILES
!fj>
!OCL SIMD
         do 2041 i=its,ite
!fj<
            psaut(i)=0.0
            psaci(i)=0.0
            praci(i)=0.0
            piacr(i)=0.0
            psacw(i)=0.0
            qsacw(i)=0.0
!fj            dd(i,j)=1./zs(i,j)**bs3
            dd=1./exp(bs3*log(zs(i)))   !hcl

!fj            if (tair(i,j).lt.t0) then
            if (tair(i).lt.t0) then
!fj               esi(i,j)=exp(.025*tairc(i,j))
               esi=exp(.025*tairc(i))
!fj               psaut(i,j)=r2is*max(rn1*esi(i,j)*(qi(i,j)-bnd1) ,0.0)
               psaut(i)=r2is*max(rn1*esi*(qi(i)-bnd1) ,0.0)
!fj               psaci(i,j)=r2is*r3f*esi(i,j)*qi(i,j)*dd(i,j)
               psaci(i)=r2is*r3f(i)*esi*qi(i)*dd
!JJS 3/30/06
!    to cut water to snow accretion by half
!               PSACW(I,J)=R4F*QC(I,J)*DD(I,J)
!fj               psacw(i,j)=r2is*0.5*r4f*qc(i,j)*dd(i,j)
               psacw(i)=r2is*0.5*r4f(i)*qc(i)*dd
!JJS 3/30/06
!fj               praci(i,j)=r2is*r5f*qi(i,j)/zr(i,j)**bw3
               praci(i)=r2is*r5f(i)*qi(i)/exp(bw3*log(zr(i)))   !hcl
!fj               piacr(i,j)=r2is*r6f*qi(i,j)*(zr(i,j)**(-bw6))
               piacr(i)=r2is*r6f(i)*qi(i)*exp((-bw6)*log(zr(i)))   !hcl
!JJS               PIACR(I,J)=R6F*QI(I,J)/ZR(I,J)**BW6
            else
               qsacw(i)=r2is*r4f(i)*qc(i)*dd
            endif

!* 21 * PRAUT   AUTOCONVERSION OF QC TO QR                        **21**
!* 22 * PRACW : ACCRETION OF QC BY QR                             **22**

!fj            pracw(i,j)=r22f*qc(i,j)/zr(i,j)**bw3
            pracw(i)=r22f(i)*qc(i)/exp(bw3*log(zr(i)))   !hcl
!fj            praut(i,j)=0.0
!fj            y1(i,j)=qc(i,j)-bnd3
            praut(i)=0.0
            y1=qc(i)-bnd3
!fj            if (y1(i,j).gt.0.0) then
            if (y1.gt.0.0) then
!fj               praut(i,j)=r00*y1(i,j)*y1(i,j)/(1.2e-4+rn21/y1(i,j))
               praut(i)=r00(i)*y1*y1/(1.2e-4+rn21/y1)
            endif
!fj>
 2041 continue
!fj<

!* 12 * PSFW : BERGERON PROCESSES FOR QS (KOENING, 1971)          **12**
!* 13 * PSFI : BERGERON PROCESSES FOR QS                          **13**

!fj>
         do 2042 i=its,ite
!fj            psfw(i,j)=0.0
!fj            psfi(i,j)=0.0
!fj            pidep(i,j)=0.0
            psfw(i)=0.0
            psfi(i)=0.0
            pidep(i)=0.0
!fj<

!fj            if(tair(i,j).lt.t0.and.qi(i,j).gt.cmin) then
            if(tair(i).lt.t0.and.qi(i).gt.cmin) then
!fj               y1(i,j)=max( min(tairc(i,j), -1.), -31.)
               y1=max( min(tairc(i), -1.), -31.)
!fj>
!fj               it(i,j)=int(abs(y1(i,j)))
!fj               y1(i,j)=rn12a(it(i,j))
!fj               y2(i,j)=rn12b(it(i,j))
               it=int(abs(y1))
               y1=rn12a(it)
               y2=rn12b(it)
!fj<
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!fj          psfw(i,j)=r2is* &
          psfw(i)=r2is* &
!fj                    max(d2t*y1(i,j)*(y2(i,j)+r12r*qc(i,j))*qi(i,j),0.0)
                    max(d2t*y1*(y2+r12r(i)*qc(i))*qi(i),0.0)
!fj               rtair(i,j)=1./(tair(i,j)-c76)
               rtair(i)=1./(tair(i)-c76)
!fj               y2(i,j)=exp(c218-c580*rtair(i,j))
               y2=exp(c218-c580*rtair(i))
!fj               qsi(i,j)=rp0*y2(i,j)
               qsi=rp0(i)*y2
!fj               esi(i,j)=c610*y2(i,j)
!fj               ssi(i,j)=(qv(i,j)+qb0)/qsi(i,j)-1.
               esi=c610*y2
               ssi=(qv(i)+qb0)/qsi-1.
!fj               r_nci=min(1.e-6*exp(-.46*tairc(i,j)),1.)
               r_nci=min(1.e-6*exp(-.46*tairc(i)),1.)
!              R_NCI=min(1.e-8*EXP(-.6*TAIRC(I,J)),1.) ! use Tao's
!fj               dm(i,j)=max( (qv(i,j)+qb0-qsi(i,j)), 0.)
               dm=max( (qv(i)+qb0-qsi), 0.)
!fj               rsub1(i,j)=cs580*qsi(i,j)*rtair(i,j)*rtair(i,j)
               rsub1=cs580(i)*qsi*rtair(i)*rtair(i)
!fj               y3(i,j)=1./tair(i,j)
               y3=1./tair(i)
!fj          dd(i,j)=y3(i,j)*(rn30a*y3(i,j)-rn30b)+rn30c*tair(i,j)/esi(i,j)
          dd=y3*(rn30a*y3-rn30b)+rn30c*tair(i)/esi
!fj               y1(i,j)=206.18*ssi(i,j)/dd(i,j)
               y1=206.18*ssi/dd
!fj               pidep(i,j)=y1(i,j)*sqrt(r_nci*qi(i,j)/r00)
               pidep(i)=y1*sqrt(r_nci*qi(i)/r00(i))
!fj               dep(i,j)=dm(i,j)/(1.+rsub1(i,j))/d2t
               dep=dm/(1.+rsub1)/d2t
!fj               if(dm(i,j).gt.cmin2) then
               if(dm.gt.cmin2) then
                  a2=1.
!fj                if(pidep(i,j).gt.dep(i,j).and.pidep(i,j).gt.cmin2) then
!fj                     a2=dep(i,j)/pidep(i,j)
!fj                     pidep(i,j)=dep(i,j)
!fj                endif
!fj                  psfi(i,j)=r2is*a2*.5*qi(i,j)*y1(i,j)/(sqrt(ami100) &
!fj                          -sqrt(ami40))
!fj                  elseif(dm(i,j).lt.-cmin2) then
                if(pidep(i).gt.dep.and.pidep(i).gt.cmin2) then
                     a2=dep/pidep(i)
                     pidep(i)=dep
                endif
                  psfi(i)=r2is*a2*.5*qi(i)*y1/(sqrt(ami100) &
                          -sqrt(ami40))
!fj>
!fj                  elseif(dm.lt.-cmin2) then
!
!        SUBLIMATION TERMS USED ONLY WHEN SATURATION ADJUSTMENT FOR ICE
!        IS TURNED OFF
!
!fj                  pidep(i,j)=0.
!fj                  psfi(i,j)=0.
!fj<
               else
!fj>
!fj                  pidep(i,j)=0.
!fj                  psfi(i,j)=0.
                  pidep(i)=0.
                  psfi(i)=0.
               endif
            endif
!fj>
 2042 continue
!fj<

!TTT***** QG=QG+MIN(PGDRY,PGWET)
!*  9 * PGACS : ACCRETION OF QS BY QG (DGACS,WGACS: DRY AND WET)  ***9**
!* 14 * DGACW : ACCRETION OF QC BY QG (QGACW FOR PGMLT)           **14**
!* 16 * DGACR : ACCRETION OF QR TO QG (QGACR FOR PGMLT)           **16**

!fj>
         if (ihail.eq.1) then
         do 2043 i=its,ite

!fj            if(qc(i,j)+qr(i,j).lt.1.e-4) then
            if(qc(i)+qr(i).lt.1.e-4) then
               ee1=.01
              else
                 ee1=1.
              endif
            ee2=0.09
!fj            egs(i,j)=ee1*exp(ee2*tairc(i,j))
            egs=ee1*exp(ee2*tairc(i))
!            EGS(I,J)=0.1 ! 6/15/02 tao's
!fj            if (tair(i,j).ge.t0) egs(i,j)=1.0
            if (tair(i).ge.t0) egs=1.0
!fj            y1(i,j)=abs(vg(i,j)-vs(i,j))
            y1=abs(vg(i)-vs(i))
!fj            y2(i,j)=zs(i,j)*zg(i,j)
            y2=zs(i)*zg(i)
!fj            y3(i,j)=5./y2(i,j)
!fj            y4(i,j)=.08*y3(i,j)*y3(i,j)
!fj            y5(i,j)=.05*y3(i,j)*y4(i,j)
            y3=5./y2
            y4=.08*y3*y3
            y5=.05*y3*y4
!fj            dd(i,j)=y1(i,j)*(y3(i,j)/zs(i,j)**5+y4(i,j)/zs(i,j)**3 &
            dd=y1*(y3/(zs(i)*zs(i)*zs(i)*zs(i)*zs(i))+y4/(zs(i)*zs(i)*zs(i)) &   !hcl
!fj                    +y5(i,j)/zs(i,j))
                    +y5/zs(i))
!fj            pgacs(i,j)=r2ig*r2is*r9r*egs(i,j)*dd(i,j)
            pgacs(i)=r2ig*r2is*r9r(i)*egs*dd
!JJS 1/3/06 from Steve and Chunglin
!fj            if (ihail.eq.1) then
!fj               dgacs(i,j)=pgacs(i,j)
               dgacs(i)=pgacs(i)
!fj            else
!fj               dgacs(i,j)=0.
!fj               dgacs(i)=0.
!fj            endif
!JJS 1/3/06 from Steve and Chunglin
!fj            wgacs(i,j)=r2ig*r2is*r9r*dd(i,j)
            wgacs(i)=r2ig*r2is*r9r(i)*dd
!            WGACS(I,J)=0.  ! 6/15/02 tao's
!fj            y1(i,j)=1./zg(i,j)**bg3
            y1=1./exp(bg3*log(zg(i)))   !hcl

!fj            if(ihail .eq. 1) then
!fj               dgacw(i,j)=r2ig*max(r14r*qc(i,j)*y1(i,j), 0.0)
               dgacw(i)=r2ig*max(r14r(i)*qc(i)*y1, 0.0)
!fj            else
!fj               dgacw(i,j)=r2ig*max(r14f*qc(i,j)*y1(i,j), 0.0)
!fj               dgacw(i)=r2ig*max(r14f(i)*qc(i)*y1, 0.0)
!fj            endif

!fj            qgacw(i,j)=dgacw(i,j)
            qgacw(i)=dgacw(i)
!fj            y1(i,j)=abs(vg(i,j)-vr(i,j))
            y1=abs(vg(i)-vr(i))
!fj            y2(i,j)=zr(i,j)*zg(i,j)
            y2=zr(i)*zg(i)
!fj            y3(i,j)=5./y2(i,j)
!fj            y4(i,j)=.08*y3(i,j)*y3(i,j)
!fj            y5(i,j)=.05*y3(i,j)*y4(i,j)
            y3=5./y2
            y4=.08*y3*y3
            y5=.05*y3*y4
!fj            dd(i,j)=r16r*y1(i,j)*(y3(i,j)/zr(i,j)**5+y4(i,j)/zr(i,j)**3 &
            dd=r16r(i)*y1*(y3/(zr(i)*zr(i)*zr(i)*zr(i)*zr(i))+y4/(zr(i)*zr(i)*zr(i)) &   !hcl
!fj                    +y5(i,j)/zr(i,j))
                    +y5/zr(i))
!fj            dgacr(i,j)=r2ig*max(dd(i,j), 0.0)
!fj            qgacr(i,j)=dgacr(i,j)
            dgacr(i)=r2ig*max(dd, 0.0)
            qgacr(i)=dgacr(i)

!fj            if (tair(i,j).ge.t0) then
            if (tair(i).ge.t0) then
!fj               dgacs(i,j)=0.0
!fj               wgacs(i,j)=0.0
!fj               dgacw(i,j)=0.0
!fj               dgacr(i,j)=0.0
               dgacs(i)=0.0
               wgacs(i)=0.0
               dgacw(i)=0.0
               dgacr(i)=0.0
            else
!fj               pgacs(i,j)=0.0
!fj               qgacw(i,j)=0.0
!fj               qgacr(i,j)=0.0
               pgacs(i)=0.0
               qgacw(i)=0.0
               qgacr(i)=0.0
            endif
!fj>
 2043 continue
      else 
!OCL SIMD
         do 2044 i=its,ite
            if(qc(i)+qr(i).lt.1.e-4) then
               ee1=.01
              else
                 ee1=1.
              endif
            ee2=0.09
            egs=ee1*exp(ee2*tairc(i))
            if (tair(i).ge.t0) egs=1.0
            y1=abs(vg(i)-vs(i))
            y2=zs(i)*zg(i)
            y3=5./y2
            y4=.08*y3*y3
            y5=.05*y3*y4
!hcl        dd=y1*(y3/zs(i)**5+y4/zs(i)**3 &
            dd=y1*(y3/(zs(i)*zs(i)*zs(i)*zs(i)*zs(i))+y4/(zs(i)*zs(i)*zs(i)) &
                    +y5/zs(i))
            pgacs(i)=r2ig*r2is*r9r(i)*egs*dd
               dgacs(i)=0.
            wgacs(i)=r2ig*r2is*r9r(i)*dd
!hcl        y1=1./zg(i)**bg3
            y1=1./exp(bg3*log(zg(i)))
            dgacw(i)=r2ig*max(r14f(i)*qc(i)*y1, 0.0)
            qgacw(i)=dgacw(i)
            y1=abs(vg(i)-vr(i))
            y2=zr(i)*zg(i)
            y3=5./y2
            y4=.08*y3*y3
            y5=.05*y3*y4
!hcl        dd=r16r(i)*y1*(y3/zr(i)**5+y4/zr(i)**3 &
            dd=r16r(i)*y1*(y3/(zr(i)*zr(i)*zr(i)*zr(i)*zr(i))+y4/(zr(i)*zr(i)*zr(i)) &
                    +y5/zr(i))
            dgacr(i)=r2ig*max(dd, 0.0)
            qgacr(i)=dgacr(i)
            if (tair(i).ge.t0) then
               wgacs(i)=0.0
               dgacw(i)=0.0
               dgacr(i)=0.0
            else
               pgacs(i)=0.0
               qgacw(i)=0.0
               qgacr(i)=0.0
            endif
 2044 continue
      endif
!fj<

!*******PGDRY : DGACW+DGACI+DGACR+DGACS                           ******
!* 15 * DGACI : ACCRETION OF QI BY QG (WGACI FOR WET GROWTH)      **15**
!* 17 * PGWET : WET GROWTH OF QG                                  **17**
!fj>
         if (ihail.eq.1) then
!fj<
         do 2037 i=its,ite

!fj            dgaci(i,j)=0.0
!fj            wgaci(i,j)=0.0
!fj            pgwet(i,j)=0.0
            dgaci(i)=0.0
            wgaci(i)=0.0
            pgwet(i)=0.0

!fj            if (tair(i,j).lt.t0) then
            if (tair(i).lt.t0) then
!fj               y1(i,j)=qi(i,j)/zg(i,j)**bg3
               y1=qi(i)/exp(bg3*log(zg(i)))   !hcl
!fj               if (ihail.eq.1) then
!fj                  dgaci(i,j)=r2ig*r15r*y1(i,j)
                  dgaci(i)=r2ig*r15r(i)*y1
!fj                  wgaci(i,j)=r2ig*r15ar*y1(i,j)
                  wgaci(i)=r2ig*r15ar(i)*y1
!                  WGACI(I,J)=0.  ! 6/15/02 tao's
!fj               else

!JJS                  DGACI(I,J)=r2ig*R15F*Y1(I,J)
!fj                   dgaci(i,j)=0.
!fj                   dgaci(i)=0.
!fj                  wgaci(i,j)=r2ig*r15af*y1(i,j)
!fj                  wgaci(i)=r2ig*r15af(i)*y1
!                  WGACI(I,J)=0.  ! 6/15/02 tao's
!fj               endif
!
!fj               if (tairc(i,j).ge.-50.) then
               if (tairc(i).ge.-50.) then
!fj                if (alf+rn17c*tairc(i,j) .eq. 0.) then
!fj                if (alf+rn17c*tairc(i) .eq. 0.) then
!fj                   write(91,*) itimestep, i,j,k, alf, rn17c, tairc(i,j)
!fj                   write(91,*) itimestep, i,j,k, alf, rn17c, tairc(i)
!fj                endif
!fj                y1(i,j)=1./(alf+rn17c*tairc(i,j))
                y1=1./(alf+rn17c*tairc(i))
!fj                if (ihail.eq.1) then
!fj                   y3(i,j)=.78/zg(i,j)**2+r17aq*scv(i,j)/zg(i,j)**bgh5
                   y3=.78/(zg(i)*zg(i))+r17aq(i)*scv(i)/exp(bgh5*log(zg(i)))   !hcl
!fj                else
!fj                   y3(i,j)=.78/zg(i,j)**2+r17as*scv(i,j)/zg(i,j)**bgh5
!fj                   y3=.78/zg(i)**2+r17as(i)*scv(i)/zg(i)**bgh5
!fj                endif
!fj                y4(i,j)=alvr*dwv(i,j)*(rp0-(qv(i,j)+qb0)) &
                y4=alvr(i)*dwv(i)*(rp0(i)-(qv(i)+qb0)) &
!fj                        -tca(i,j)*tairc(i,j)
                        -tca(i)*tairc(i)
!fj                dd(i,j)=y1(i,j)*(r17r*y4(i,j)*y3(i,j) &
                dd=y1*(r17r(i)*y4*y3 &
!fj                       +(wgaci(i,j)+wgacs(i,j))*(alf+rn17b*tairc(i,j)))
                       +(wgaci(i)+wgacs(i))*(alf+rn17b*tairc(i)))
!fj                pgwet(i,j)=r2ig*max(dd(i,j), 0.0)
                pgwet(i)=r2ig*max(dd, 0.0)
               endif
            endif
 2037 continue
         else
!OCL SIMD
         do 2045 i=its,ite

            dgaci(i)=0.0
            wgaci(i)=0.0
            pgwet(i)=0.0

            if (tair(i).lt.t0) then
!hcl           y1=qi(i)/zg(i)**bg3
               y1=qi(i)/exp(bg3*log(zg(i)))
               wgaci(i)=r2ig*r15af(i)*y1
               if (tairc(i).ge.-50.) then
                y1=1./(alf+rn17c*tairc(i))
!hcl            y3=.78/zg(i)**2+r17as(i)*scv(i)/zg(i)**bgh5
                y3=.78/(zg(i)*zg(i))+r17as(i)*scv(i)/exp(bgh5*log(zg(i)))
                y4=alvr(i)*dwv(i)*(rp0(i)-(qv(i)+qb0)) &
                        -tca(i)*tairc(i)
                dd=y1*(r17r(i)*y4*y3 &
                       +(wgaci(i)+wgacs(i))*(alf+rn17b*tairc(i)))
                pgwet(i)=r2ig*max(dd, 0.0)
               endif
            endif
 2045 continue
      endif
         do 2046 i=its,ite
               if (tairc(i).ge.-50.) then
                if (alf+rn17c*tairc(i) .eq. 0.) then
                   write(91,*) itimestep, i,j,k, alf, rn17c, tairc(i)
                endif
               endif
 2046 continue
!JJS  125    CONTINUE

!********   HANDLING THE NEGATIVE CLOUD WATER (QC)    ******************
!********   HANDLING THE NEGATIVE CLOUD ICE (QI)      ******************

!JJS         DO 150 J=3,JLES
!JJS         DO 150 I=3,ILES
!OCL NOPREFETCH
         do 2038 i=its,ite

!fj>
!fj            y1(i,j)=qc(i,j)/d2t
!fj            psacw(i,j)=min(y1(i,j), psacw(i,j))
!fj            praut(i,j)=min(y1(i,j), praut(i,j))
!fj            pracw(i,j)=min(y1(i,j), pracw(i,j))
!fj            psfw(i,j)= min(y1(i,j), psfw(i,j))
!fj            dgacw(i,j)=min(y1(i,j), dgacw(i,j))
!fj            qsacw(i,j)=min(y1(i,j), qsacw(i,j))
!fj            qgacw(i,j)=min(y1(i,j), qgacw(i,j))
            y1=qc(i)/d2t
            psacw(i)=min(y1, psacw(i))
            praut(i)=min(y1, praut(i))
            pracw(i)=min(y1, pracw(i))
            psfw(i)= min(y1, psfw(i))
            dgacw(i)=min(y1, dgacw(i))
            qsacw(i)=min(y1, qsacw(i))
            qgacw(i)=min(y1, qgacw(i))
!fj<

!fj>
!fj            y1(i,j)=(psacw(i,j)+praut(i,j)+pracw(i,j)+psfw(i,j) &
!fj                    +dgacw(i,j)+qsacw(i,j)+qgacw(i,j))*d2t
!fj            qc(i,j)=qc(i,j)-y1(i,j)
            y1=(psacw(i)+praut(i)+pracw(i)+psfw(i) &
                    +dgacw(i)+qsacw(i)+qgacw(i))*d2t
            qc(i)=qc(i)-y1
!fj<

!fj>
!fj            if (qc(i,j) .lt. 0.0) then
!fj               a1=1.
!fj               if (y1(i,j) .ne. 0.0) a1=qc(i,j)/y1(i,j)+1.
!fj               psacw(i,j)=psacw(i,j)*a1
!fj               praut(i,j)=praut(i,j)*a1
!fj               pracw(i,j)=pracw(i,j)*a1
!fj               psfw(i,j)=psfw(i,j)*a1
!fj               dgacw(i,j)=dgacw(i,j)*a1
!fj               qsacw(i,j)=qsacw(i,j)*a1
!fj               qgacw(i,j)=qgacw(i,j)*a1
!fj               qc(i,j)=0.0
            if (qc(i) .lt. 0.0) then
               a1=1.
               if (y1 .ne. 0.0) a1=qc(i)/y1+1.
               psacw(i)=psacw(i)*a1
               praut(i)=praut(i)*a1
               pracw(i)=pracw(i)*a1
               psfw(i)=psfw(i)*a1
               dgacw(i)=dgacw(i)*a1
               qsacw(i)=qsacw(i)*a1
               qgacw(i)=qgacw(i)*a1
               qc(i)=0.0
!fj<
            endif
!c
!
!******** SHED PROCESS (WGACR=PGWET-DGACW-WGACI-WGACS)
!c
!fj>
!fj            wgacr(i,j)=pgwet(i,j)-dgacw(i,j)-wgaci(i,j)-wgacs(i,j)
!fj            y2(i,j)=dgacw(i,j)+dgaci(i,j)+dgacr(i,j)+dgacs(i,j)
!fj            if (pgwet(i,j).ge.y2(i,j)) then
!fj               wgacr(i,j)=0.0
!fj               wgaci(i,j)=0.0
!fj               wgacs(i,j)=0.0
!fj            else
!fj               dgacr(i,j)=0.0
!fj               dgaci(i,j)=0.0
!fj               dgacs(i,j)=0.0
!fj            endif
            wgacr=pgwet(i)-dgacw(i)-wgaci(i)-wgacs(i)
            y2=dgacw(i)+dgaci(i)+dgacr(i)+dgacs(i)
            if (pgwet(i).ge.y2) then
               wgacr=0.0
               wgaci(i)=0.0
               wgacs(i)=0.0
            else
               dgacr(i)=0.0
               dgaci(i)=0.0
               dgacs(i)=0.0
            endif
!fj<
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!c
!fj>
!fj            y1(i,j)=qi(i,j)/d2t
!fj            psaut(i,j)=min(y1(i,j), psaut(i,j))
!fj            psaci(i,j)=min(y1(i,j), psaci(i,j))
!fj            praci(i,j)=min(y1(i,j), praci(i,j))
!fj            psfi(i,j)= min(y1(i,j), psfi(i,j))
!fj            dgaci(i,j)=min(y1(i,j), dgaci(i,j))
!fj            wgaci(i,j)=min(y1(i,j), wgaci(i,j))
!fj!
!fj            y2(i,j)=(psaut(i,j)+psaci(i,j)+praci(i,j)+psfi(i,j) &
!fj                   +dgaci(i,j)+wgaci(i,j))*d2t
!fj            qi(i,j)=qi(i,j)-y2(i,j)+pidep(i,j)*d2t
!fj
!fj            if (qi(i,j).lt.0.0) then
!fj               a2=1.
!fj               if (y2(i,j) .ne. 0.0) a2=qi(i,j)/y2(i,j)+1.
!fj               psaut(i,j)=psaut(i,j)*a2
!fj               psaci(i,j)=psaci(i,j)*a2
!fj               praci(i,j)=praci(i,j)*a2
!fj               psfi(i,j)=psfi(i,j)*a2
!fj               dgaci(i,j)=dgaci(i,j)*a2
!fj               wgaci(i,j)=wgaci(i,j)*a2
!fj               qi(i,j)=0.0
!fj            endif
!fj!
!fj            dlt3(i,j)=0.0
!fj            dlt2(i,j)=0.0
            y1=qi(i)/d2t
            psaut(i)=min(y1, psaut(i))
            psaci(i)=min(y1, psaci(i))
            praci(i)=min(y1, praci(i))
            psfi(i)= min(y1, psfi(i))
            dgaci(i)=min(y1, dgaci(i))
            wgaci(i)=min(y1, wgaci(i))
!
            y2=(psaut(i)+psaci(i)+praci(i)+psfi(i) &
                  +dgaci(i)+wgaci(i))*d2t
            qi(i)=qi(i)-y2+pidep(i)*d2t

            if (qi(i).lt.0.0) then
               a2=1.
               if (y2 .ne. 0.0) a2=qi(i)/y2+1.
               psaut(i)=psaut(i)*a2
               psaci(i)=psaci(i)*a2
               praci(i)=praci(i)*a2
               psfi(i)=psfi(i)*a2
               dgaci(i)=dgaci(i)*a2
               wgaci(i)=wgaci(i)*a2
               qi(i)=0.0
            endif
!
            dlt3=0.0
            dlt2=0.0
!

!            DLT4(I,J)=1.0
!            if(qc(i,j) .gt. 5.e-4) dlt4(i,j)=0.0
!            if(qs(i,j) .le. 1.e-4) dlt4(i,j)=1.0
!
!            IF (TAIR(I,J).ge.T0) THEN
!               DLT4(I,J)=0.0
!            ENDIF

!fj            if (tair(i,j).lt.t0) then
            if (tair(i).lt.t0) then
!fj               if (qr(i,j).lt.1.e-4) then
!fj                  dlt3(i,j)=1.0
!fj                  dlt2(i,j)=1.0
!fj               endif
!fj               if (qs(i,j).ge.1.e-4) then
!fj                  dlt2(i,j)=0.0
               if (qr(i).lt.1.e-4) then
                  dlt3=1.0
                  dlt2=1.0
               endif
               if (qs(i).ge.1.e-4) then
                  dlt2=0.0
               endif
            endif

            if (ice2 .eq. 1) then
!fj                  dlt3(i,j)=1.0
!fj                  dlt2(i,j)=1.0
                  dlt3=1.0
                  dlt2=1.0
            endif
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!fj            pr(i,j)=(qsacw(i,j)+praut(i,j)+pracw(i,j)+qgacw(i,j))*d2t
!fj            ps(i,j)=(psaut(i,j)+psaci(i,j)+psacw(i,j)+psfw(i,j) &
!fj                    +psfi(i,j)+dlt3(i,j)*praci(i,j))*d2t
            pr=(qsacw(i)+praut(i)+pracw(i)+qgacw(i))*d2t
            ps=(psaut(i)+psaci(i)+psacw(i)+psfw(i) &
                    +psfi(i)+dlt3*praci(i))*d2t
!           PS(I,J)=(PSAUT(I,J)+PSACI(I,J)+dlt4(i,j)*PSACW(I,J)
!    1              +PSFW(I,J)+PSFI(I,J)+DLT3(I,J)*PRACI(I,J))*D2T
!fj            pg(i,j)=((1.-dlt3(i,j))*praci(i,j)+dgaci(i,j)+wgaci(i,j) &
!fj                    +dgacw(i,j))*d2t
            pg=((1.-dlt3)*praci(i)+dgaci(i)+wgaci(i) &
                    +dgacw(i))*d2t
!           PG(I,J)=((1.-DLT3(I,J))*PRACI(I,J)+DGACI(I,J)+WGACI(I,J)
!    1              +DGACW(I,J)+(1.-dlt4(i,j))*PSACW(I,J))*D2T

!JJS  150    CONTINUE

!*  7 * PRACS : ACCRETION OF QS BY QR                             ***7**
!*  8 * PSACR : ACCRETION OF QR BY QS (QSACR FOR PSMLT)           ***8**

!JJS         DO 175 J=3,JLES
!JJS         DO 175 I=3,ILES

!fj           y1(i,j)=abs(vr(i,j)-vs(i,j))
            y1=abs(vr(i)-vs(i))
!fj            y2(i,j)=zr(i,j)*zs(i,j)
            y2=zr(i)*zs(i)
!fj            y3(i,j)=5./y2(i,j)
!fj            y4(i,j)=.08*y3(i,j)*y3(i,j)
!fj            y5(i,j)=.05*y3(i,j)*y4(i,j)
            y3=5./y2
            y4=.08*y3*y3
            y5=.05*y3*y4
!fj            pracs(i,j)=r2ig*r2is*r7r*y1(i,j)*(y3(i,j)/zs(i,j)**5 &
            pracs=r2ig*r2is*r7r(i)*y1*(y3/(zs(i)*zs(i)*zs(i)*zs(i)*zs(i)) &   !hcl
!fj                      +y4(i,j)/zs(i,j)**3+y5(i,j)/zs(i,j))
                      +y4/(zs(i)*zs(i)*zs(i))+y5/zs(i))   !hcl
!fj            psacr(i,j)=r2is*r8r*y1(i,j)*(y3(i,j)/zr(i,j)**5 &
            psacr=r2is*r8r(i)*y1*(y3/(zr(i)*zr(i)*zr(i)*zr(i)*zr(i)) &   !hcl
!fj                      +y4(i,j)/zr(i,j)**3+y5(i,j)/zr(i,j))
                      +y4/(zr(i)*zr(i)*zr(i))+y5/zr(i))   !hcl
!fj            qsacr(i,j)=psacr(i,j)
            qsacr=psacr

!fj            if (tair(i,j).ge.t0) then
            if (tair(i).ge.t0) then
!fj               pracs(i,j)=0.0
!fj               psacr(i,j)=0.0
               pracs=0.0
               psacr=0.0
            else
!fj               qsacr(i,j)=0.0
               qsacr=0.0
            endif
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!*  2 * PGAUT : AUTOCONVERSION OF QS TO QG                        ***2**
!* 18 * PGFR : FREEZING OF QR TO QG                               **18**

!fj            pgaut(i,j)=0.0
!fj            pgfr(i,j)=0.0
            pgaut=0.0
            pgfr=0.0

!fj            if (tair(i,j) .lt. t0) then
            if (tair(i) .lt. t0) then
!               Y1(I,J)=EXP(.09*TAIRC(I,J))
!               PGAUT(I,J)=r2is*max(RN2*Y1(I,J)*(QS(I,J)-BND2), 0.0)
!         IF(IHAIL.EQ.1) PGAUT(I,J)=max(RN2*Y1(I,J)*(QS(I,J)-BND2),0.0)
!fj               y2(i,j)=exp(rn18a*(t0-tair(i,j)))
               y2=exp(rn18a*(t0-tair(i)))
!JJS              PGFR(I,J)=r2ig*max(R18R*(Y2(I,J)-1.)/ZR(I,J)**7., 0.0)
!              pgfr(i,j)=r2ice*max(r18r*(y2(i,j)-1.)* &
!                                    (zr(i,j)**(-7.)), 0.0)
!        modify to prevent underflow on some computers (JD)
!fj               temp = 1./zr(i,j)
               temp = 1./zr(i)
               temp = temp*temp*temp*temp*temp*temp*temp
!fj               pgfr(i,j)=r2ig*max(r18r*(y2(i,j)-1.)* &
               pgfr=r2ig*max(r18r(i)*(y2-1.)* &
                                    temp, 0.0)
            endif

!JJS  175    CONTINUE

!********   HANDLING THE NEGATIVE RAIN WATER (QR)    *******************
!********   HANDLING THE NEGATIVE SNOW (QS)          *******************

!JJS         DO 200 J=3,JLES
!JJS         DO 200 I=3,ILES

!fj>
!fj            y1(i,j)=qr(i,j)/d2t
!fj            y2(i,j)=-qg(i,j)/d2t
!fj            piacr(i,j)=min(y1(i,j), piacr(i,j))
!fj            dgacr(i,j)=min(y1(i,j), dgacr(i,j))
!fj            wgacr(i,j)=min(y1(i,j), wgacr(i,j))
!fj            wgacr(i,j)=max(y2(i,j), wgacr(i,j))
!fj            psacr(i,j)=min(y1(i,j), psacr(i,j))
!fj            pgfr(i,j)= min(y1(i,j), pgfr(i,j))
!fj            del=0.
!fj            if(wgacr(i,j) .lt. 0.) del=1.
!fj            y1(i,j)=(piacr(i,j)+dgacr(i,j)+(1.-del)*wgacr(i,j) &
!fj                    +psacr(i,j)+pgfr(i,j))*d2t
!fj            qr(i,j)=qr(i,j)+pr(i,j)-y1(i,j)-del*wgacr(i,j)*d2t
!fj            if (qr(i,j) .lt. 0.0) then
!fj               a1=1.
!fj               if(y1(i,j) .ne. 0.) a1=qr(i,j)/y1(i,j)+1.
!fj               piacr(i,j)=piacr(i,j)*a1
!fj               dgacr(i,j)=dgacr(i,j)*a1
!fj               if (wgacr(i,j).gt.0.) wgacr(i,j)=wgacr(i,j)*a1
!fj               pgfr(i,j)=pgfr(i,j)*a1
!fj               psacr(i,j)=psacr(i,j)*a1
!fj               qr(i,j)=0.0
!fj            endif
!fj!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!fj            prn(i,j)=d2t*((1.-dlt3(i,j))*piacr(i,j)+dgacr(i,j) &
!fj                     +wgacr(i,j)+(1.-dlt2(i,j))*psacr(i,j)+pgfr(i,j))
!fj            ps(i,j)=ps(i,j)+d2t*(dlt3(i,j)*piacr(i,j) &
!fj                    +dlt2(i,j)*psacr(i,j))
!fj            pracs(i,j)=(1.-dlt2(i,j))*pracs(i,j)
!fj            y1(i,j)=qs(i,j)/d2t
!fj            pgacs(i,j)=min(y1(i,j), pgacs(i,j))
!fj            dgacs(i,j)=min(y1(i,j), dgacs(i,j))
!fj            wgacs(i,j)=min(y1(i,j), wgacs(i,j))
!fj            pgaut(i,j)=min(y1(i,j), pgaut(i,j))
!fj            pracs(i,j)=min(y1(i,j), pracs(i,j))
!fj            psn(i,j)=d2t*(pgacs(i,j)+dgacs(i,j)+wgacs(i,j) &
!fj                     +pgaut(i,j)+pracs(i,j))
!fj            qs(i,j)=qs(i,j)+ps(i,j)-psn(i,j)
!fj
!fj            if (qs(i,j).lt.0.0) then
!fj               a2=1.
!fj               if (psn(i,j) .ne. 0.0) a2=qs(i,j)/psn(i,j)+1.
!fj               pgacs(i,j)=pgacs(i,j)*a2
!fj               dgacs(i,j)=dgacs(i,j)*a2
!fj               wgacs(i,j)=wgacs(i,j)*a2
!fj               pgaut(i,j)=pgaut(i,j)*a2
!fj               pracs(i,j)=pracs(i,j)*a2
!fj               psn(i,j)=psn(i,j)*a2
!fj               qs(i,j)=0.0
!fj            endif
            y1=qr(i)/d2t
            y2=-qg(i)/d2t
            piacr(i)=min(y1, piacr(i))
            dgacr(i)=min(y1, dgacr(i))
            wgacr=min(y1, wgacr)
            wgacr=max(y2, wgacr)
            psacr=min(y1, psacr)
            pgfr= min(y1, pgfr)
            del=0.
            if(wgacr .lt. 0.) del=1.
            y1=(piacr(i)+dgacr(i)+(1.-del)*wgacr &
                  +psacr+pgfr)*d2t
            qr(i)=qr(i)+pr-y1-del*wgacr*d2t
            if (qr(i) .lt. 0.0) then
               a1=1.
               if(y1 .ne. 0.) a1=qr(i)/y1+1.
               piacr(i)=piacr(i)*a1
               dgacr(i)=dgacr(i)*a1
               if (wgacr.gt.0.) wgacr=wgacr*a1
               pgfr=pgfr*a1
               psacr=psacr*a1
               qr(i)=0.0
            endif
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
            prn=d2t*((1.-dlt3)*piacr(i)+dgacr(i) &
                     +wgacr+(1.-dlt2)*psacr+pgfr)
            ps=ps+d2t*(dlt3*piacr(i) &
                    +dlt2*psacr)
            pracs=(1.-dlt2)*pracs
            y1=qs(i)/d2t
            pgacs(i)=min(y1, pgacs(i))
            dgacs(i)=min(y1, dgacs(i))
            wgacs(i)=min(y1, wgacs(i))
            pgaut=min(y1, pgaut)
            pracs=min(y1, pracs)
            psn=d2t*(pgacs(i)+dgacs(i)+wgacs(i) &
                     +pgaut+pracs)
            qs(i)=qs(i)+ps-psn

            if (qs(i).lt.0.0) then
               a2=1.
               if (psn .ne. 0.0) a2=qs(i)/psn+1.
               pgacs(i)=pgacs(i)*a2
               dgacs(i)=dgacs(i)*a2
               wgacs(i)=wgacs(i)*a2
               pgaut=pgaut*a2
               pracs=pracs*a2
               psn=psn*a2
               qs(i)=0.0
            endif
!fj<
!
!C           PSN(I,J)=D2T*(PGACS(I,J)+DGACS(I,J)+WGACS(I,J)
!c                    +PGAUT(I,J)+PRACS(I,J))
!fj            y2(i,j)=d2t*(psacw(i,j)+psfw(i,j)+dgacw(i,j)+piacr(i,j) &
!fj                    +dgacr(i,j)+wgacr(i,j)+psacr(i,j)+pgfr(i,j))
            y2=d2t*(psacw(i)+psfw(i)+dgacw(i)+piacr(i) &
                    +dgacr(i)+wgacr+psacr+pgfr)
!fj            pt(i,j)=pt(i,j)+afcp*y2(i,j)
            pt(i)=pt(i)+afcp(i)*y2
!fj            qg(i,j)=qg(i,j)+pg(i,j)+prn(i,j)+psn(i,j)
            qg(i)=qg(i)+pg+prn+psn

!JJS  200    CONTINUE

!* 11 * PSMLT : MELTING OF QS                                     **11**
!* 19 * PGMLT : MELTING OF QG TO QR                               **19**

!JJS         DO 225 J=3,JLES
!JJS         DO 225 I=3,ILES

!fj            psmlt(i,j)=0.0
!fj            pgmlt(i,j)=0.0
            psmlt=0.0
            pgmlt=0.0
!fj            tair(i,j)=(pt(i,j)+tb0)*pi0
            tair(i)=(pt(i)+tb0)*pi0(i)

!fj            if (tair(i,j).ge.t0) then
            if (tair(i).ge.t0) then
!fj               tairc(i,j)=tair(i,j)-t0
               tairc(i)=tair(i)-t0
!fj               y1(i,j)=tca(i,j)*tairc(i,j)-alvr*dwv(i,j) &
               y1=tca(i)*tairc(i)-alvr(i)*dwv(i) &
!fj                               *(rp0-(qv(i,j)+qb0))
                               *(rp0(i)-(qv(i)+qb0))
!fj               y2(i,j)=.78/zs(i,j)**2+r101f*scv(i,j)/zs(i,j)**bsh5
               y2=.78/(zs(i)*zs(i))+r101f(i)*scv(i)/exp(bsh5*log(zs(i)))   !hcl
!fj               dd(i,j)=r11rt*y1(i,j)*y2(i,j)+r11at*tairc(i,j) &
               dd=r11rt(i)*y1*y2+r11at*tairc(i) &
!fj                       *(qsacw(i,j)+qsacr(i,j))
                       *(qsacw(i)+qsacr)
!fj               psmlt(i,j)=r2is*max(0.0, min(dd(i,j), qs(i,j)))
               psmlt=r2is*max(0.0, min(dd, qs(i)))

               if(ihail.eq.1) then
!fj                  y3(i,j)=.78/zg(i,j)**2+r19aq*scv(i,j)/zg(i,j)**bgh5
                  y3=.78/(zg(i)*zg(i))+r19aq(i)*scv(i)/exp(bgh5*log(zg(i)))   !hcl
               else
!fj                  y3(i,j)=.78/zg(i,j)**2+r19as*scv(i,j)/zg(i,j)**bgh5
                  y3=.78/(zg(i)*zg(i))+r19as(i)*scv(i)/exp(bgh5*log(zg(i)))   !hcl
               endif

!fj               dd1(i,j)=r19rt*y1(i,j)*y3(i,j)+r19bt*tairc(i,j) &
               dd1=r19rt(i)*y1*y3+r19bt*tairc(i) &
!fj                        *(qgacw(i,j)+qgacr(i,j))
                        *(qgacw(i)+qgacr(i))
!fj               pgmlt(i,j)=r2ig*max(0.0, min(dd1(i,j), qg(i,j)))
               pgmlt=r2ig*max(0.0, min(dd1, qg(i)))
!fj               pt(i,j)=pt(i,j)-afcp*(psmlt(i,j)+pgmlt(i,j))
               pt(i)=pt(i)-afcp(i)*(psmlt+pgmlt)
!fj               qr(i,j)=qr(i,j)+psmlt(i,j)+pgmlt(i,j)
!fj               qs(i,j)=qs(i,j)-psmlt(i,j)
!fj               qg(i,j)=qg(i,j)-pgmlt(i,j)
               qr(i)=qr(i)+psmlt+pgmlt
               qs(i)=qs(i)-psmlt
               qg(i)=qg(i)-pgmlt
            endif
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!* 24 * PIHOM : HOMOGENEOUS FREEZING OF QC TO QI (T < T00)        **24**
!* 25 * PIDW : DEPOSITION GROWTH OF QC TO QI ( T0 < T <= T00)     **25**
!* 26 * PIMLT : MELTING OF QI TO QC (T >= T0)                     **26**

!fj            if (qc(i,j).le.cmin1) qc(i,j)=0.0
!fj            if (qi(i,j).le.cmin1) qi(i,j)=0.0
            if (qc(i).le.cmin1) qc(i)=0.0
            if (qi(i).le.cmin1) qi(i)=0.0
!fj            tair(i,j)=(pt(i,j)+tb0)*pi0
            tair(i)=(pt(i)+tb0)*pi0(i)

!fj            if(tair(i,j).le.t00) then
            if(tair(i).le.t00) then
!fj               pihom(i,j)=qc(i,j)
               pihom=qc(i)
            else
!fj               pihom(i,j)=0.0
               pihom=0.0
            endif
!fj            if(tair(i,j).ge.t0) then
            if(tair(i).ge.t0) then
!fj               pimlt(i,j)=qi(i,j)
               pimlt=qi(i)
            else
!fj               pimlt(i,j)=0.0
               pimlt=0.0
            endif
!fj            pidw(i,j)=0.0
            pidw=0.0

!fj            if (tair(i,j).lt.t0 .and. tair(i,j).gt.t00) then
!fj               tairc(i,j)=tair(i,j)-t0
!fj               y1(i,j)=max( min(tairc(i,j), -1.), -31.)
            if (tair(i).lt.t0 .and. tair(i).gt.t00) then
               tairc(i)=tair(i)-t0
               y1=max( min(tairc(i), -1.), -31.)
!fj>
!fj               it(i,j)=int(abs(y1(i,j)))
!fj               y2(i,j)=aa1(it(i,j))
!fj               y3(i,j)=aa2(it(i,j))
               it=int(abs(y1))
               y2=aa1(it)
               y3=aa2(it)
!fj<
!fj               y4(i,j)=exp(abs(beta*tairc(i,j)))
               y4=exp(abs(beta*tairc(i)))
!fj               y5(i,j)=(r00*qi(i,j)/(r25a*y4(i,j)))**y3(i,j)
               y5=exp(y3*log(r00(i)*qi(i)/(r25a*y4)))   !hcl
!fj               pidw(i,j)=min(r25rt*y2(i,j)*y4(i,j)*y5(i,j), qc(i,j))
               pidw=min(r25rt(i)*y2*y4*y5, qc(i))
            endif

!fj            y1(i,j)=pihom(i,j)-pimlt(i,j)+pidw(i,j)
            y1=pihom-pimlt+pidw
!fj            pt(i,j)=pt(i,j)+afcp*y1(i,j)+ascp*(pidep(i,j))*d2t
            pt(i)=pt(i)+afcp(i)*y1+ascp(i)*(pidep(i))*d2t
!fj            qv(i,j)=qv(i,j)-(pidep(i,j))*d2t
!fj            qc(i,j)=qc(i,j)-y1(i,j)
!fj            qi(i,j)=qi(i,j)+y1(i,j)
            qv(i)=qv(i)-(pidep(i))*d2t
            qc(i)=qc(i)-y1
            qi(i)=qi(i)+y1

!* 31 * PINT  : INITIATION OF QI                                  **31**
!* 32 * PIDEP : DEPOSITION OF QI                                  **32**
!
!     CALCULATION OF PINT USES DIFFERENT VALUES OF THE INTERCEPT AND SLOPE FOR
!     THE FLETCHER EQUATION. ALSO, ONLY INITIATE MORE ICE IF THE NEW NUMBER
!     CONCENTRATION EXCEEDS THAT ALREADY PRESENT.
!* 31 * pint  : initiation of qi                                  **31**
!* 32 * pidep : deposition of qi                                  **32**
!fj           pint(i,j)=0.0
 2038 continue
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
        if ( itaobraun.eq.1 ) then
         do 2031 i=its,ite
!fj            tair(i,j)=(pt(i,j)+tb0)*pi0
            tair(i)=(pt(i)+tb0)*pi0(i)
!fj            if (tair(i,j) .lt. t0) then
            if (tair(i) .lt. t0) then
!fj             if (qi(i,j) .le. cmin) qi(i,j)=0.
              if (qi(i) .le. cmin2) qi(i)=0.
!fj               tairc(i,j)=tair(i,j)-t0
!fj               rtair(i,j)=1./(tair(i,j)-c76)
!fj               y2(i,j)=exp(c218-c580*rtair(i,j))
               tairc(i)=tair(i)-t0
               rtair(i)=1./(tair(i)-c76)
               y2=exp(c218-c580*rtair(i))
!fj              qsi(i,j)=rp0*y2(i,j)
              qsi=rp0(i)*y2
!fj               esi(i,j)=c610*y2(i,j)
!fj              ssi(i,j)=(qv(i,j)+qb0)/qsi(i,j)-1.
               esi=c610*y2
              ssi=(qv(i)+qb0)/qsi-1.
                        ami50=3.76e-8

!ccif ( itaobraun.eq.1 ) --> betah=0.5*beta=-.46*0.5=-0.23;   cn0=1.e-6
!ccif ( itaobraun.eq.0 ) --> betah=0.5*beta=-.6*0.5=-0.30;    cn0=1.e-8

!fj             y1(i,j)=1./tair(i,j)
             y1=1./tair(i)

!cc insert a restriction on ice collection that ice collection
!cc should be stopped at -30 c (with cn0=1.e-6, beta=-.46)

!fj             tairccri=tairc(i,j)          ! in degree c
             tairccri=tairc(i)          ! in degree c
             if(tairccri.le.-30.) tairccri=-30.

!            y2(i,j)=exp(betah*tairc(i,j))
             y2=exp(betah*tairccri)
!fj             y3(i,j)=sqrt(qi(i,j))
             y3=sqrt(qi(i))
!fj             dd(i,j)=y1(i,j)*(rn10a*y1(i,j)-rn10b)+rn10c*tair(i,j) &
             dd=y1*(rn10a*y1-rn10b)+rn10c*tair(i) &
!fj                                                /esi(i,j)
                                                /esi
!fj          pidep(i,j)=max(r32rt*ssi(i,j)*y2(i,j)*y3(i,j)/dd(i,j), 0.e0)
          pidep(i)=max(r32rt(i)*ssi*y2*y3/dd, 0.e0)

!fj           r_nci=min(cn0*exp(beta*tairc(i,j)),1.)
           r_nci=min(cn0*exp(beta*tairc(i)),1.)
!          r_nci=min(1.e-6*exp(-.46*tairc(i,j)),1.)

!fj           dd(i,j)=max(1.e-9*r_nci/r00-qi(i,j)*1.e-9/ami50, 0.)
           dd=max(1.e-9*r_nci/r00(i)-qi(i)*1.e-9/ami50, 0.)
!fj                dm(i,j)=max( (qv(i,j)+qb0-qsi(i,j)), 0.0)
                dm=max( (qv(i)+qb0-qsi), 0.0)
!fj                rsub1(i,j)=cs580*qsi(i,j)*rtair(i,j)*rtair(i,j)
                rsub1=cs580(i)*qsi*rtair(i)*rtair(i)
!fj              dep(i,j)=dm(i,j)/(1.+rsub1(i,j))
              dep=dm/(1.+rsub1)
!fj              pint(i,j)=max(min(dd(i,j), dm(i,j)), 0.)
              pint=max(min(dd, dm), 0.)

!             pint(i,j)=min(pint(i,j), dep(i,j))
!fj              pint(i,j)=min(pint(i,j)+pidep(i,j), dep(i,j))
              pint=min(pint+pidep(i), dep)

!              if (pint(i,j) .le. cmin) pint(i,j)=0.
!fj               if (pint(i,j) .le. cmin2) pint(i,j)=0.
               if (pint .le. cmin2) pint=0.
!fj              pt(i,j)=pt(i,j)+ascp*pint(i,j)
              pt(i)=pt(i)+ascp(i)*pint
!fj              qv(i,j)=qv(i,j)-pint(i,j)
!fj              qi(i,j)=qi(i,j)+pint(i,j)
              qv(i)=qv(i)-pint
              qi(i)=qi(i)+pint
           endif
 2031 continue
        endif  ! if ( itaobraun.eq.1 )
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
        if ( itaobraun.eq.0 ) then
         do 2032 i=its,ite
!fj             tair(i,j)=(pt(i,j)+tb0)*pi0
             tair(i)=(pt(i)+tb0)*pi0(i)
!fj             if (tair(i,j) .lt. t0) then
             if (tair(i) .lt. t0) then
!fj               if (qi(i,j) .le. cmin1) qi(i,j)=0.
               if (qi(i) .le. cmin1) qi(i)=0.
!fj               tairc(i,j)=tair(i,j)-t0
               tairc(i)=tair(i)-t0
!fj               dd(i,j)=r31r*exp(beta*tairc(i,j))
               dd=r31r(i)*exp(beta*tairc(i))
!fj               rtair(i,j)=1./(tair(i,j)-c76)
               rtair(i)=1./(tair(i)-c76)
!fj               y2(i,j)=exp(c218-c580*rtair(i,j))
               y2=exp(c218-c580*rtair(i))
!fj               qsi(i,j)=rp0*y2(i,j)
               qsi=rp0(i)*y2
!fj               esi(i,j)=c610*y2(i,j)
               esi=c610*y2
!fj               ssi(i,j)=(qv(i,j)+qb0)/qsi(i,j)-1.
               ssi=(qv(i)+qb0)/qsi-1.
!fj               dm(i,j)=max( (qv(i,j)+qb0-qsi(i,j)), 0.)
               dm=max( (qv(i)+qb0-qsi), 0.)
!fj               rsub1(i,j)=cs580*qsi(i,j)*rtair(i,j)*rtair(i,j)
               rsub1=cs580(i)*qsi*rtair(i)*rtair(i)
!fj               dep(i,j)=dm(i,j)/(1.+rsub1(i,j))
               dep=dm/(1.+rsub1)
!fj              pint(i,j)=max(min(dd(i,j), dm(i,j)), 0.)
              pint=max(min(dd, dm), 0.)
!fj               y1(i,j)=1./tair(i,j)
               y1=1./tair(i)
!fj               y2(i,j)=exp(betah*tairc(i,j))
               y2=exp(betah*tairc(i))
!fj               y3(i,j)=sqrt(qi(i,j))
!fj               dd(i,j)=y1(i,j)*(rn10a*y1(i,j)-rn10b) &
               y3=sqrt(qi(i))
               dd=y1*(rn10a*y1-rn10b) &
!fj                     +rn10c*tair(i,j)/esi(i,j)
                     +rn10c*tair(i)/esi
!fj             pidep(i,j)=max(r32rt*ssi(i,j)*y2(i,j)*y3(i,j)/dd(i,j), 0.)
             pidep(i)=max(r32rt(i)*ssi*y2*y3/dd, 0.)
!fj              pint(i,j)=pint(i,j)+pidep(i,j)
!fj              pint(i,j)=min(pint(i,j),dep(i,j))
              pint=pint+pidep(i)
              pint=min(pint,dep)
!c          if (pint(i,j) .le. cmin2) pint(i,j)=0.
!fj             pt(i,j)=pt(i,j)+ascp*pint(i,j)
             pt(i)=pt(i)+ascp(i)*pint
!fj             qv(i,j)=qv(i,j)-pint(i,j)
!fj             qi(i,j)=qi(i,j)+pint(i,j)
             qv(i)=qv(i)-pint
             qi(i)=qi(i)+pint
            endif
 2032 continue
        endif  ! if ( itaobraun.eq.0 )
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

!JJS  225    CONTINUE

!*****   TAO ET AL (1989) SATURATION TECHNIQUE  ***********************

         if (new_ice_sat .eq. 0) then
         do 2033 i=its,ite

!JJS            DO 250 J=3,JLES
!JJS            DO 250 I=3,ILES
!fj               tair(i,j)=(pt(i,j)+tb0)*pi0
               tair(i)=(pt(i)+tb0)*pi0(i)
!fj               cnd(i,j)=rt0*(tair(i,j)-t00)
               cnd=rt0*(tair(i)-t00)
!fj               dep(i,j)=rt0*(t0-tair(i,j))
               dep=rt0*(t0-tair(i))
!fj               y1(i,j)=1./(tair(i,j)-c358)
!fj               y2(i,j)=1./(tair(i,j)-c76)
               y1=1./(tair(i)-c358)
               y2=1./(tair(i)-c76)
!fj               qsw(i,j)=rp0*exp(c172-c409*y1(i,j))
!fj               qsi(i,j)=rp0*exp(c218-c580*y2(i,j))
               qsw=rp0(i)*exp(c172-c409*y1)
               qsi=rp0(i)*exp(c218-c580*y2)
!fj               dd(i,j)=cp409*y1(i,j)*y1(i,j)
               dd=cp409(i)*y1*y1
!fj               dd1(i,j)=cp580*y2(i,j)*y2(i,j)
               dd1=cp580(i)*y2*y2
!fj               if (qc(i,j).le.cmin) qc(i,j)=cmin
!fj               if (qi(i,j).le.cmin) qi(i,j)=cmin
               if (qc(i).le.cmin) qc(i)=cmin
               if (qi(i).le.cmin) qi(i)=cmin
!fj               if (tair(i,j).ge.t0) then
               if (tair(i).ge.t0) then
!fj                  dep(i,j)=0.0
!fj                  cnd(i,j)=1.
!fj                  qi(i,j)=0.0
                  dep=0.0
                  cnd=1.
                  qi(i)=0.0
               endif

!fj               if (tair(i,j).lt.t00) then
               if (tair(i).lt.t00) then
!fj                  cnd(i,j)=0.0
!fj                  dep(i,j)=1.
!fj                  qc(i,j)=0.0
                  cnd=0.0
                  dep=1.
                  qc(i)=0.0
               endif

!fj               y5(i,j)=avcp*cnd(i,j)+ascp*dep(i,j)
               y5=avcp(i)*cnd+ascp(i)*dep
!               if (qc(i,j) .ge. cmin .or. qi(i,j) .ge. cmin) then
!fj               y1(i,j)=qc(i,j)*qsw(i,j)/(qc(i,j)+qi(i,j))
!fj               y2(i,j)=qi(i,j)*qsi(i,j)/(qc(i,j)+qi(i,j))
!fj               y4(i,j)=dd(i,j)*y1(i,j)+dd1(i,j)*y2(i,j)
!fj               qvs(i,j)=y1(i,j)+y2(i,j)
!fj               rsub1(i,j)=(qv(i,j)+qb0-qvs(i,j))/(1.+y4(i,j)*y5(i,j))
!fj               cnd(i,j)=cnd(i,j)*rsub1(i,j)
!fj               dep(i,j)=dep(i,j)*rsub1(i,j)
!fj               if (qc(i,j).le.cmin) qc(i,j)=0.
!fj               if (qi(i,j).le.cmin) qi(i,j)=0.
               y1=qc(i)*qsw/(qc(i)+qi(i))
               y2=qi(i)*qsi/(qc(i)+qi(i))
               y4=dd*y1+dd1*y2
               qvs=y1+y2
               rsub1=(qv(i)+qb0-qvs)/(1.+y4*y5)
               cnd=cnd*rsub1
               dep=dep*rsub1
               if (qc(i).le.cmin) qc(i)=0.
               if (qi(i).le.cmin) qi(i)=0.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!c    ******   condensation or evaporation of qc  ******

!fj               cnd(i,j)=max(-qc(i,j),cnd(i,j))
               cnd=max(-qc(i),cnd)

!c    ******   deposition or sublimation of qi    ******

!fj               dep(i,j)=max(-qi(i,j),dep(i,j))
               dep=max(-qi(i),dep)

!fj               pt(i,j)=pt(i,j)+avcp*cnd(i,j)+ascp*dep(i,j)
               pt(i)=pt(i)+avcp(i)*cnd+ascp(i)*dep
!fj               qv(i,j)=qv(i,j)-cnd(i,j)-dep(i,j)
!fj               qc(i,j)=qc(i,j)+cnd(i,j)
!fj               qi(i,j)=qi(i,j)+dep(i,j)
               qv(i)=qv(i)-cnd-dep
               qc(i)=qc(i)+cnd
               qi(i)=qi(i)+dep
!JJS  250       continue
 2033 continue
         endif

         if (new_ice_sat .eq. 1) then
         do 2034 i=its,ite

!JJS            DO J=3,JLES
!JJS            DO I=3,ILES

!fj               tair(i,j)=(pt(i,j)+tb0)*pi0
               tair(i)=(pt(i)+tb0)*pi0(i)
!fj               cnd(i,j)=rt0*(tair(i,j)-t00)
!fj               dep(i,j)=rt0*(t0-tair(i,j))
!fj               y1(i,j)=1./(tair(i,j)-c358)
!fj               y2(i,j)=1./(tair(i,j)-c76)
               cnd=rt0*(tair(i)-t00)
               dep=rt0*(t0-tair(i))
               y1=1./(tair(i)-c358)
               y2=1./(tair(i)-c76)
!fj               qsw(i,j)=rp0*exp(c172-c409*y1(i,j))
!fj               qsi(i,j)=rp0*exp(c218-c580*y2(i,j))
               qsw=rp0(i)*exp(c172-c409*y1)
               qsi=rp0(i)*exp(c218-c580*y2)
!fj               dd(i,j)=cp409*y1(i,j)*y1(i,j)
               dd=cp409(i)*y1*y1
!fj               dd1(i,j)=cp580*y2(i,j)*y2(i,j)
               dd1=cp580(i)*y2*y2
!fj               y5(i,j)=avcp*cnd(i,j)+ascp*dep(i,j)
               y5=avcp(i)*cnd+ascp(i)*dep
!fj               y1(i,j)=rt0*(tair(i,j)-t00)*qsw(i,j)
!fj               y2(i,j)=rt0*(t0-tair(i,j))*qsi(i,j)
               y1=rt0*(tair(i)-t00)*qsw
               y2=rt0*(t0-tair(i))*qsi
!               IF (QC(I,J).LE.CMIN) QC(I,J)=CMIN
!               IF (QI(I,J).LE.CMIN) QI(I,J)=CMIN

!fj               if (tair(i,j).ge.t0) then
               if (tair(i).ge.t0) then
!                 QI(I,J)=0.0
!fj                  dep(i,j)=0.0
!fj                  cnd(i,j)=1.
!fj                  y2(i,j)=0.
!fj                  y1(i,j)=qsw(i,j)
                  dep=0.0
                  cnd=1.
                  y2=0.
                  y1=qsw
               endif
!fj               if (tair(i,j).lt.t00) then
               if (tair(i).lt.t00) then
!fj                  cnd(i,j)=0.0
!fj                  dep(i,j)=1.
!fj                  y2(i,j)=qsi(i,j)
!fj                  y1(i,j)=0.
                  cnd=0.0
                  dep=1.
                  y2=qsi
                  y1=0.
!                 QC(I,J)=0.0
               endif

!            Y1(I,J)=QC(I,J)*QSW(I,J)/(QC(I,J)+QI(I,J))
!            Y2(I,J)=QI(I,J)*QSI(I,J)/(QC(I,J)+QI(I,J))

!fj               y4(i,j)=dd(i,j)*y1(i,j)+dd1(i,j)*y2(i,j)
!fj               qvs(i,j)=y1(i,j)+y2(i,j)
!fj               rsub1(i,j)=(qv(i,j)+qb0-qvs(i,j))/(1.+y4(i,j)*y5(i,j))
!fj               cnd(i,j)=cnd(i,j)*rsub1(i,j)
!fj               dep(i,j)=dep(i,j)*rsub1(i,j)

               y4=dd*y1+dd1*y2
               qvs=y1+y2
               rsub1=(qv(i)+qb0-qvs)/(1.+y4*y5)
               cnd=cnd*rsub1
               dep=dep*rsub1
!               IF (QC(I,J).LE.CMIN) QC(I,J)=0.
!               IF (QI(I,J).LE.CMIN) QI(I,J)=0.

!C    ******   CONDENSATION OR EVAPORATION OF QC  ******

!fj               cnd(i,j)=max(-qc(i,j),cnd(i,j))
               cnd=max(-qc(i),cnd)

!C    ******   DEPOSITION OR SUBLIMATION OF QI    ******

!fj               dep(i,j)=max(-qi(i,j),dep(i,j))
               dep=max(-qi(i),dep)

!fj               pt(i,j)=pt(i,j)+avcp*cnd(i,j)+ascp*dep(i,j)
               pt(i)=pt(i)+avcp(i)*cnd+ascp(i)*dep
!fj               qv(i,j)=qv(i,j)-cnd(i,j)-dep(i,j)
!fj               qc(i,j)=qc(i,j)+cnd(i,j)
!fj               qi(i,j)=qi(i,j)+dep(i,j)
               qv(i)=qv(i)-cnd-dep
               qc(i)=qc(i)+cnd
               qi(i)=qi(i)+dep
!JJS            ENDDO
!JJS            ENDDO
 2034 continue
         endif

!c
!
          if (new_ice_sat .eq. 2) then
!OCL SIMD
         do 2035 i=its,ite
!JJS          do j=3,jles
!JJS             do i=3,iles
!fj          dep(i,j)=0.0
!fj          cnd(i,j)=0.0
          dep=0.0
          cnd=0.0
!fj          tair(i,j)=(pt(i,j)+tb0)*pi0
          tair(i)=(pt(i)+tb0)*pi0(i)
!fj          if (tair(i,j) .ge. 253.16) then
!fj              y1(i,j)=1./(tair(i,j)-c358)
          if (tair(i) .ge. 253.16) then
!fj              y1(i,j)=1./(tair(i)-c358)
              y1=1./(tair(i)-c358)
!fj              qsw(i,j)=rp0*exp(c172-c409*y1(i,j))
              qsw=rp0(i)*exp(c172-c409*y1)
!fj              dd(i,j)=cp409*y1(i,j)*y1(i,j)
              dd=cp409(i)*y1*y1
              dm=qv(i)+qb0-qsw
!fj              cnd(i,j)=dm(i,j)/(1.+avcp*dd(i,j)*qsw(i,j))
              cnd=dm/(1.+avcp(i)*dd*qsw)
!c    ******   condensation or evaporation of qc  ******
!fj              cnd(i,j)=max(-qc(i,j), cnd(i,j))
              cnd=max(-qc(i), cnd)
!fj             pt(i,j)=pt(i,j)+avcp*cnd(i,j)
             pt(i)=pt(i)+avcp(i)*cnd
!fj             qv(i,j)=qv(i,j)-cnd(i,j)
!fj             qc(i,j)=qc(i,j)+cnd(i,j)
             qv(i)=qv(i)-cnd
             qc(i)=qc(i)+cnd
         endif
 2035 continue
!fj          if (tair(i,j) .le. 258.16) then
!OCL SIMD
         do 2052 i=its,ite
          if (tair(i) .le. 258.16) then
!c             cnd(i,j)=0.0
!fj           y2(i,j)=1./(tair(i,j)-c76)
           y2=1./(tair(i)-c76)
!fj           qsi(i,j)=rp0*exp(c218-c580*y2(i,j))
           qsi=rp0(i)*exp(c218-c580*y2)
!fj          dd1(i,j)=cp580*y2(i,j)*y2(i,j)
          dd1=cp580(i)*y2*y2
!fj         dep(i,j)=(qv(i,j)+qb0-qsi(i,j))/(1.+ascp*dd1(i,j)*qsi(i,j))
         dep=(qv(i)+qb0-qsi)/(1.+ascp(i)*dd1*qsi)
!c    ******   deposition or sublimation of qi    ******
!fj             dep(i,j)=max(-qi(i,j),dep(i,j))
             dep=max(-qi(i),dep)
!fj             pt(i,j)=pt(i,j)+ascp*dep(i,j)
             pt(i)=pt(i)+ascp(i)*dep
!fj             qv(i,j)=qv(i,j)-dep(i,j)
!fj             qi(i,j)=qi(i,j)+dep(i,j)
             qv(i)=qv(i)-dep
             qi(i)=qi(i)+dep
         endif
!JJS       enddo
!JJS       enddo
 2052 continue
      endif

!c
!
!* 10 * PSDEP : DEPOSITION OR SUBLIMATION OF QS                   **10**
!* 20 * PGSUB : SUBLIMATION OF QG                                 **20**

!JJS         DO 275 J=3,JLES
!JJS         DO 275 I=3,ILES
!fj>
         if(ihail.eq.1) then
         do 2036 i=its,ite

!fj            psdep(i,j)=0.0
!fj            pgdep(i,j)=0.0
!fj            pssub(i,j)=0.0
!fj            pgsub(i,j)=0.0
            psdep=0.0
!fj            pgdep(i)=0.0
            pssub=0.0
            pgsub=0.0
!fj            tair(i,j)=(pt(i,j)+tb0)*pi0
            tair(i)=(pt(i)+tb0)*pi0(i)

!fj            if(tair(i,j).lt.t0) then
            if(tair(i).lt.t0) then
!fj               if(qs(i,j).lt.cmin1) qs(i,j)=0.0
!fj               if(qg(i,j).lt.cmin1) qg(i,j)=0.0
               if(qs(i).lt.cmin1) qs(i)=0.0
               if(qg(i).lt.cmin1) qg(i)=0.0
!fj               rtair(i,j)=1./(tair(i,j)-c76)
               rtair(i)=1./(tair(i)-c76)
!fj               qsi(i,j)=rp0*exp(c218-c580*rtair(i,j))
               qsi=rp0(i)*exp(c218-c580*rtair(i))
!fj               ssi(i,j)=(qv(i,j)+qb0)/qsi(i,j)-1.
               ssi=(qv(i)+qb0)/qsi-1.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!fj               y1(i,j)=r10ar/(tca(i,j)*tair(i,j)**2)+1./(dwv(i,j) &
               y1=r10ar(i)/(tca(i)*tair(i)*tair(i))+1./(dwv(i) &   !hcl
                      *qsi)
!fj               y2(i,j)=.78/zs(i,j)**2+r101f*scv(i,j)/zs(i,j)**bsh5
               y2=.78/(zs(i)*zs(i))+r101f(i)*scv(i)/exp(bsh5*log(zs(i)))   !hcl
!fj               psdep(i,j)=r10t*ssi(i,j)*y2(i,j)/y1(i,j)
!fj               pssub(i,j)=psdep(i,j)
!fj               psdep(i,j)=r2is*max(psdep(i,j), 0.)
!fj               pssub(i,j)=r2is*max(-qs(i,j), min(pssub(i,j), 0.))
               psdep=r10t*ssi*y2/y1
               pssub=psdep
               psdep=r2is*max(psdep, 0.)
               pssub=r2is*max(-qs(i), min(pssub, 0.))

!fj               if(ihail.eq.1) then
!fj                  y2(i,j)=.78/zg(i,j)**2+r20bq*scv(i,j)/zg(i,j)**bgh5
                  y2=.78/(zg(i)*zg(i))+r20bq(i)*scv(i)/exp(bgh5*log(zg(i)))   !hcl
!fj               else
!fj                  y2(i,j)=.78/zg(i,j)**2+r20bs*scv(i,j)/zg(i,j)**bgh5
!fj                  y2=.78/zg(i)**2+r20bs(i)*scv(i)/zg(i)**bgh5
!fj               endif

!fj               pgsub(i,j)=r2ig*r20t*ssi(i,j)*y2(i,j)/y1(i,j)
!fj               dm(i,j)=qv(i,j)+qb0-qsi(i,j)
               pgsub=r2ig*r20t*ssi*y2/y1
               dm=qv(i)+qb0-qsi
!fj               rsub1(i,j)=cs580*qsi(i,j)*rtair(i,j)*rtair(i,j)
               rsub1=cs580(i)*qsi*rtair(i)*rtair(i)

!     ********   DEPOSITION OR SUBLIMATION OF QS  **********************

!fj               y1(i,j)=dm(i,j)/(1.+rsub1(i,j))
!fj               psdep(i,j)=r2is*min(psdep(i,j),max(y1(i,j),0.))
!fj               y2(i,j)=min(y1(i,j),0.)
!fj               pssub(i,j)=r2is*max(pssub(i,j),y2(i,j))
               y1=dm/(1.+rsub1)
               psdep=r2is*min(psdep,max(y1,0.))
               y2=min(y1,0.)
               pssub=r2is*max(pssub,y2)

!     ********   SUBLIMATION OF QG   ***********************************

!fj               dd(i,j)=max((-y2(i,j)-qs(i,j)), 0.)
!fj              pgsub(i,j)=r2ig*min(dd(i,j), qg(i,j), max(pgsub(i,j),0.))
               dd=max((-y2-qs(i)), 0.)
              pgsub=r2ig*min(dd, qg(i), max(pgsub,0.))

!fj               if(qc(i,j)+qi(i,j).gt.1.e-5) then
!fj                  dlt1(i,j)=1.
!fj               else
!fj                  dlt1(i,j)=0.
!fj               endif
               if(qc(i)+qi(i).gt.1.e-5) then
                  dlt1=1.
               else
                  dlt1=0.
               endif

!fj               psdep(i,j)=dlt1(i,j)*psdep(i,j)
!fj               pssub(i,j)=(1.-dlt1(i,j))*pssub(i,j)
!fj               pgsub(i,j)=(1.-dlt1(i,j))*pgsub(i,j)
               psdep=dlt1*psdep
               pssub=(1.-dlt1)*pssub
               pgsub=(1.-dlt1)*pgsub

!fj               pt(i,j)=pt(i,j)+ascp*(psdep(i,j)+pssub(i,j)-pgsub(i,j))
               pt(i)=pt(i)+ascp(i)*(psdep+pssub-pgsub)
!fj               qv(i,j)=qv(i,j)+pgsub(i,j)-pssub(i,j)-psdep(i,j)
!fj               qs(i,j)=qs(i,j)+psdep(i,j)+pssub(i,j)
!fj               qg(i,j)=qg(i,j)-pgsub(i,j)
               qv(i)=qv(i)+pgsub-pssub-psdep
               qs(i)=qs(i)+psdep+pssub
               qg(i)=qg(i)-pgsub
            endif
 2036 continue
      else
!OCL SIMD
         do 2053 i=its,ite
            psdep=0.0
            pssub=0.0
            pgsub=0.0
            tair(i)=(pt(i)+tb0)*pi0(i)

            if(tair(i).lt.t0) then
               if(qs(i).lt.cmin1) qs(i)=0.0
               if(qg(i).lt.cmin1) qg(i)=0.0
               rtair(i)=1./(tair(i)-c76)
               qsi=rp0(i)*exp(c218-c580*rtair(i))
               ssi=(qv(i)+qb0)/qsi-1.
!hcl           y1=r10ar(i)/(tca(i)*tair(i)**2)+1./(dwv(i) &
               y1=r10ar(i)/(tca(i)*tair(i)*tair(i))+1./(dwv(i) &
                      *qsi)
!hcl           y2=.78/zs(i)**2+r101f(i)*scv(i)/zs(i)**bsh5
               y2=.78/(zs(i)*zs(i))+r101f(i)*scv(i)/exp(bsh5*log(zs(i)))
               psdep=r10t*ssi*y2/y1
               pssub=psdep
               psdep=r2is*max(psdep, 0.)
               pssub=r2is*max(-qs(i), min(pssub, 0.))
!hcl            y2=.78/zg(i)**2+r20bs(i)*scv(i)/zg(i)**bgh5
                y2=.78/(zg(i)*zg(i))+r20bs(i)*scv(i)/exp(bgh5*log(zg(i)))
               pgsub=r2ig*r20t*ssi*y2/y1
               dm=qv(i)+qb0-qsi
               rsub1=cs580(i)*qsi*rtair(i)*rtair(i)
               y1=dm/(1.+rsub1)
               psdep=r2is*min(psdep,max(y1,0.))
               y2=min(y1,0.)
               pssub=r2is*max(pssub,y2)
               dd=max((-y2-qs(i)), 0.)
              pgsub=r2ig*min(dd, qg(i), max(pgsub,0.))

               if(qc(i)+qi(i).gt.1.e-5) then
                  dlt1=1.
               else
                  dlt1=0.
               endif
               psdep=dlt1*psdep
               pssub=(1.-dlt1)*pssub
               pgsub=(1.-dlt1)*pgsub
               pt(i)=pt(i)+ascp(i)*(psdep+pssub-pgsub)
               qv(i)=qv(i)+pgsub-pssub-psdep
               qs(i)=qs(i)+psdep+pssub
               qg(i)=qg(i)-pgsub
            endif
 2053 continue
      endif

!* 23 * ERN : EVAPORATION OF QR (SUBSATURATION)                   **23**

!fj            ern(i,j)=0.0
         do 2054 i=its,ite
            ern=0.0

!fj            if(qr(i,j).gt.0.0) then
            if(qr(i).gt.0.0) then
!fj               tair(i,j)=(pt(i,j)+tb0)*pi0
               tair(i)=(pt(i)+tb0)*pi0(i)
!fj               rtair(i,j)=1./(tair(i,j)-c358)
               rtair(i)=1./(tair(i)-c358)
!fj               qsw(i,j)=rp0*exp(c172-c409*rtair(i,j))
               qsw=rp0(i)*exp(c172-c409*rtair(i))
!fj               ssw(i,j)=(qv(i,j)+qb0)/qsw(i,j)-1.0
               ssw=(qv(i)+qb0)/qsw-1.0
               dm=qv(i)+qb0-qsw
!fj               rsub1(i,j)=cv409*qsw(i,j)*rtair(i,j)*rtair(i,j)
               rsub1=cv409(i)*qsw*rtair(i)*rtair(i)
!fj               dd1(i,j)=max(-dm(i,j)/(1.+rsub1(i,j)), 0.0)
               dd1=max(-dm/(1.+rsub1), 0.0)
!fj               y1(i,j)=.78/zr(i,j)**2+r23af*scv(i,j)/zr(i,j)**bwh5
               y1=.78/(zr(i)*zr(i))+r23af(i)*scv(i)/exp(bwh5*log(zr(i)))   !hcl
!fj               y2(i,j)=r23br/(tca(i,j)*tair(i,j)**2)+1./(dwv(i,j) &
               y2=r23br(i)/(tca(i)*tair(i)*tair(i))+1./(dwv(i) &   !hcl
!fj                       *qsw(i,j))
                       *qsw)
!cccc
!fj               ern(i,j)=r23t*ssw(i,j)*y1(i,j)/y2(i,j)
!fj               ern(i,j)=min(dd1(i,j),qr(i,j),max(ern(i,j),0.))
               ern=r23t*ssw*y1/y2
               ern=min(dd1,qr(i),max(ern,0.))
!fj               pt(i,j)=pt(i,j)-avcp*ern(i,j)
               pt(i)=pt(i)-avcp(i)*ern
!fj               qv(i,j)=qv(i,j)+ern(i,j)
!fj               qr(i,j)=qr(i,j)-ern(i,j)
               qv(i)=qv(i)+ern
               qr(i)=qr(i)-ern
            endif

!JJS 10/7/2008     vvvvv
!fj>
 2054 continue
!fj<
    ENDIF    ! part of if (iwarm.eq.1) then
!JJS 10/7/2008     ^^^^^

!            IF (QV(I,J)+QB0 .LE. 0.) QV(I,J)=-QB0
!fj>
         do 2040 i=its,ite
!fj<
!fj            if (qc(i,j) .le. cmin1) qc(i,j)=0.
!fj            if (qr(i,j) .le. cmin1) qr(i,j)=0.
!fj            if (qi(i,j) .le. cmin1) qi(i,j)=0.
!fj            if (qs(i,j) .le. cmin1) qs(i,j)=0.
!fj            if (qg(i,j) .le. cmin1) qg(i,j)=0.
            if (qc(i) .le. cmin1) qc(i)=0.
            if (qr(i) .le. cmin1) qr(i)=0.
            if (qi(i) .le. cmin1) qi(i)=0.
            if (qs(i) .le. cmin1) qs(i)=0.
            if (qg(i) .le. cmin1) qg(i)=0.
!fj>
!fj            dpt(i,j,k)=pt(i,j)
!fj            dqv(i,j,k)=qv(i,j)
!fj            qcl(i,j,k)=qc(i,j)
!fj            qrn(i,j,k)=qr(i,j)
!fj            qci(i,j,k)=qi(i,j)
!fj            qcs(i,j,k)=qs(i,j)
!fj            qcg(i,j,k)=qg(i,j)

         ptwrf(i,k,j) = pt(i)
         qvwrf(i,k,j) = qv(i)
         qlwrf(i,k,j) = qc(i)
         qrwrf(i,k,j) = qr(i)
         qiwrf(i,k,j) = qi(i)
         qswrf(i,k,j) = qs(i)
         qgwrf(i,k,j) = qg(i)
 2040 continue
!JJS  275    CONTINUE

!fj         scc=0.
!fj         see=0.
!fj<

!         DO 110 J=3,JLES
!         DO 110 I=3,ILES
!            DD(I,J)=MAX(-CND(I,J), 0.)
!            CND(I,J)=MAX(CND(I,J), 0.)
!            DD1(I,J)=MAX(-DEP(I,J), 0.)

!ccshie 2/21/02 shie follow tao
!CC for reference    QI(I,J)=QI(I,J)-Y2(I,J)+PIDEP(I,J)*D2T
!CC for reference    QV(I,J)=QV(I,J)-(PIDEP(I,J))*D2T

!c            DEP(I,J)=MAX(DEP(I,J), 0.)
!            DEP(I,J)=MAX(DEP(I,J), 0.)+PIDEP(I,J)*D2T
!            SCC=SCC+CND(I,J)
!            SEE=SEE+DD(I,J)+ERN(I,J)

!  110    CONTINUE

!         SC(K)=SCC+SC(K)
!         SE(K)=SEE+SE(K)

!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!c     henry:  please take a look  (start)
!JJS modified by JJS on 5/1/2007  vvvvv

!JJS       do 305 j=3,jles
!JJS       do 305 i=3,iles
!fj>
!fj            dd(i,j)=max(-cnd(i,j), 0.)
!fj            cnd(i,j)=max(cnd(i,j), 0.)
!fj            dd1(i,j)=max(-dep(i,j), 0.)+pidep(i,j)*d2t
!fj            dep(i,j)=max(dep(i,j), 0.)
!fj<
!JJS  305  continue

!JJS       do 306 j=3,jles
!JJS       do 306 i=3,iles
!JJS              scc=scc+cnd(i,j)
!JJS              see=see+(dd(i,j)+ern(i,j))
!
!JJS            sddd=sddd+(dep(i,j)+pint(i,j)+psdep(i,j)+pgdep(i,j))
!JJS          ssss=ssss+dd1(i,j)
!JJS
!            shhh=shhh+rsw(i,j,k)*d2t
!            sccc=sccc+rlw(i,j,k)*d2t
!jjs
!JJS              smmm=smmm+(psmlt(i,j)+pgmlt(i,j)+pimlt(i,j))
!JJS              sfff=sfff+d2t*(psacw(i,j)+piacr(i,j)+psfw(i,j)
!JJS     1         +pgfr(i,j)+dgacw(i,j)+dgacr(i,j)+psacr(i,j))
!JJS     2        -qracs(i,j)+pihom(i,j)+pidw(i,j)

!fj>
!fj              sccc=cnd(i,j)
!fj              seee=dd(i,j)+ern(i,j)
!fj              sddd=dep(i,j)+pint(i,j)+psdep(i,j)+pgdep(i,j)
!fj              ssss=dd1(i,j) + pgsub(i,j)
!fj              smmm=psmlt(i,j)+pgmlt(i,j)+pimlt(i,j)
!fj              sfff=d2t*(psacw(i,j)+piacr(i,j)+psfw(i,j) &
!fj               +pgfr(i,j)+dgacw(i,j)+dgacr(i,j)+psacr(i,j) &
!fj               +wgacr(i,j))+pihom(i,j)+pidw(i,j)
!fj<

!           physc(i,k,j) = avcp * sccc / d2t
!           physe(i,k,j) = avcp * seee / d2t
!           physd(i,k,j) = ascp * sddd / d2t
!           physs(i,k,j) = ascp * ssss / d2t
!           physf(i,k,j) = afcp * sfff / d2t
!           physm(i,k,j) = afcp * smmm / d2t
!           physc(i,k,j) = physc(i,k,j) + avcp * sccc 
!           physe(i,k,j) = physc(i,k,j) + avcp * seee 
!           physd(i,k,j) = physd(i,k,j) + ascp * sddd 
!           physs(i,k,j) = physs(i,k,j) + ascp * ssss 
!           physf(i,k,j) = physf(i,k,j) + afcp * sfff 
!           physm(i,k,j) = physm(i,k,j) + afcp * smmm 

!JJS modified by JJS on 5/1/2007  ^^^^^

 2000 continue

 1000 continue

!JJS  ****************************************************************
!JJS  convert from GCE grid back to WRF grid
!fj>
!fj      do k=kts,kte
!fj         do j=jts,jte
!fj         do i=its,ite
!fj         ptwrf(i,k,j) = dpt(i,j,k)
!fj         qvwrf(i,k,j) = dqv(i,j,k)
!fj         qlwrf(i,k,j) = qcl(i,j,k)
!fj         qrwrf(i,k,j) = qrn(i,j,k)
!fj         qiwrf(i,k,j) = qci(i,j,k)
!fj         qswrf(i,k,j) = qcs(i,j,k)
!fj         qgwrf(i,k,j) = qcg(i,j,k)
!fj         enddo !i
!fj         enddo !j
!fj      enddo !k
!fj<

!     ****************************************************************

!+---+-----------------------------------------------------------------+
         IF ( PRESENT (diagflag) ) THEN
         if (diagflag .and. do_radar_ref == 1) then
            do j=jts,jte
            do i=its,ite
               DO K=kts,kte
                  t1d(k)=ptwrf(i,k,j)*pi_mks(i,k,j)
                  p1d(k)=p0_mks(i,k,j)
                  qv1d(k)=qvwrf(i,k,j)
                  qr1d(k)=qrwrf(i,k,j)
               ENDDO
               if (ice2.eq.0) then
                  DO K=kts,kte
                     qs1d(k)=qswrf(i,k,j)
                     qg1d(k)=qgwrf(i,k,j)
                  ENDDO
               elseif (ice2.eq.1) then
                  DO K=kts,kte
                     qs1d(k)=qswrf(i,k,j)
                  ENDDO
               elseif (ice2.eq.2) then
                  DO K=kts,kte
                     qs1d(k)=0.
                     qg1d(k)=qgwrf(i,k,j)
                  ENDDO
               elseif (ice2.eq.3) then
                  DO K=kts,kte
                     qs1d(k)=0.
                     qg1d(k)=0.
                  ENDDO
               endif
               call refl10cm_gsfc (qv1d, qr1d, qs1d, qg1d,             &
                       t1d, p1d, dBZ, kts, kte, i, j, ihail)
               do k = kts, kte
                  refl_10cm(i,k,j) = MAX(-35., dBZ(k))
               enddo
            enddo
            enddo
         endif
         ENDIF
!+---+-----------------------------------------------------------------+

      return
 END SUBROUTINE saticel_s

!JJS
!JJS      REAL FUNCTION GAMMA(X)
!JJS        Y=GAMMLN(X)
!JJS        GAMMA=EXP(Y)
!JJS      RETURN
!JJS      END
!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!JJS      real function GAMMLN (xx)
  real function gammagce (xx)
!**********************************************************************
  real*8 cof(6),stp,half,one,fpf,x,tmp,ser
  data cof,stp /  76.18009173,-86.50532033,24.01409822, &
     -1.231739516,.120858003e-2,-.536382e-5, 2.50662827465 /
  data half,one,fpf / .5, 1., 5.5 /
!
      x=xx-one
      tmp=x+fpf
      tmp=(x+half)*log(tmp)-tmp
      ser=one
      do  j=1,6
         x=x+one
        ser=ser+cof(j)/x
      enddo !j
      gammln=tmp+log(stp*ser)
!JJS
      gammagce=exp(gammln)
!JJS
      return
 END FUNCTION gammagce

!+---+-----------------------------------------------------------------+

      subroutine refl10cm_gsfc (qv1d, qr1d, qs1d, qg1d,                 &
                       t1d, p1d, dBZ, kts, kte, ii, jj, ihail)

      IMPLICIT NONE

!..Sub arguments
      INTEGER, INTENT(IN):: kts, kte, ii, jj, ihail
      REAL, DIMENSION(kts:kte), INTENT(IN)::                            &
                      qv1d, qr1d, qs1d, qg1d, t1d, p1d
      REAL, DIMENSION(kts:kte), INTENT(INOUT):: dBZ

!..Local variables
      REAL, DIMENSION(kts:kte):: temp, pres, qv, rho
      REAL, DIMENSION(kts:kte):: rr, rs, rg

      DOUBLE PRECISION, DIMENSION(kts:kte):: ilamr, ilams, ilamg
      DOUBLE PRECISION, DIMENSION(kts:kte):: N0_r, N0_s, N0_g
      DOUBLE PRECISION:: lamr, lams, lamg
      LOGICAL, DIMENSION(kts:kte):: L_qr, L_qs, L_qg

      REAL, DIMENSION(kts:kte):: ze_rain, ze_snow, ze_graupel
      DOUBLE PRECISION:: fmelt_s, fmelt_g

      INTEGER:: i, k, k_0, kbot, n
      LOGICAL:: melti

      DOUBLE PRECISION:: cback, x, eta, f_d
      REAL, PARAMETER:: R=287.
      REAL, PARAMETER:: PIx=3.1415926536

!+---+

      do k = kts, kte
         dBZ(k) = -35.0
      enddo

!+---+-----------------------------------------------------------------+
!..Put column of data into local arrays.
!+---+-----------------------------------------------------------------+
      do k = kts, kte
         temp(k) = t1d(k)
         qv(k) = MAX(1.E-10, qv1d(k))
         pres(k) = p1d(k)
         rho(k) = 0.622*pres(k)/(R*temp(k)*(qv(k)+0.622))

         if (qr1d(k) .gt. 1.E-9) then
            rr(k) = qr1d(k)*rho(k)
            N0_r(k) = xnor
            lamr = (xam_r*xcrg(3)*N0_r(k)/rr(k))**(1./xcre(1))
            ilamr(k) = 1./lamr
            L_qr(k) = .true.
         else
            rr(k) = 1.E-12
            L_qr(k) = .false.
         endif

         if (qs1d(k) .gt. 1.E-9) then
            rs(k) = qs1d(k)*rho(k)
            N0_s(k) = xnos
            lams = (xam_s*xcsg(3)*N0_s(k)/rs(k))**(1./xcse(1))
            ilams(k) = 1./lams
            L_qs(k) = .true.
         else
            rs(k) = 1.E-12
            L_qs(k) = .false.
         endif

         if (qg1d(k) .gt. 1.E-9) then
            rg(k) = qg1d(k)*rho(k)
            if (ihail.eq.1) then
               N0_g(k) = xnoh
            else
               N0_g(k) = xnog
            endif
            lamg = (xam_g*xcgg(3)*N0_g(k)/rg(k))**(1./xcge(1))
            ilamg(k) = 1./lamg
            L_qg(k) = .true.
         else
            rg(k) = 1.E-12
            L_qg(k) = .false.
         endif
      enddo

!+---+-----------------------------------------------------------------+
!..Locate K-level of start of melting (k_0 is level above).
!+---+-----------------------------------------------------------------+
      melti = .false.
      k_0 = kts
      do k = kte-1, kts, -1
         if ( (temp(k).gt.273.15) .and. L_qr(k)                         &
                                  .and. (L_qs(k+1).or.L_qg(k+1)) ) then
            k_0 = MAX(k+1, k_0)
            melti=.true.
            goto 195
         endif
      enddo
 195  continue

!+---+-----------------------------------------------------------------+
!..Assume Rayleigh approximation at 10 cm wavelength. Rain (all temps)
!.. and non-water-coated snow and graupel when below freezing are
!.. simple. Integrations of m(D)*m(D)*N(D)*dD.
!+---+-----------------------------------------------------------------+

      do k = kts, kte
         ze_rain(k) = 1.e-22
         ze_snow(k) = 1.e-22
         ze_graupel(k) = 1.e-22
         if (L_qr(k)) ze_rain(k) = N0_r(k)*xcrg(4)*ilamr(k)**xcre(4)
         if (L_qs(k)) ze_snow(k) = (0.176/0.93) * (6.0/PIx)*(6.0/PIx)     &
                                 * (xam_s/900.0)*(xam_s/900.0)          &
                                 * N0_s(k)*xcsg(4)*ilams(k)**xcse(4)
         if (L_qg(k)) ze_graupel(k) = (0.176/0.93) * (6.0/PIx)*(6.0/PIx)  &
                                    * (xam_g/900.0)*(xam_g/900.0)       &
                                    * N0_g(k)*xcgg(4)*ilamg(k)**xcge(4)
      enddo


!+---+-----------------------------------------------------------------+
!..Special case of melting ice (snow/graupel) particles.  Assume the
!.. ice is surrounded by the liquid water.  Fraction of meltwater is
!.. extremely simple based on amount found above the melting level.
!.. Uses code from Uli Blahak (rayleigh_soak_wetgraupel and supporting
!.. routines).
!+---+-----------------------------------------------------------------+

      if (melti .and. k_0.ge.kts+1) then
       do k = k_0-1, kts, -1

!..Reflectivity contributed by melting snow
          if (L_qs(k) .and. L_qs(k_0) ) then
           fmelt_s = MAX(0.005d0, MIN(1.0d0-rs(k)/rs(k_0), 0.99d0))
           eta = 0.d0
           lams = 1./ilams(k)
           do n = 1, nrbins
              x = xam_s * xxDs(n)**xbm_s
              call rayleigh_soak_wetgraupel (x,DBLE(xocms),DBLE(xobms), &
                    fmelt_s, melt_outside_s, m_w_0, m_i_0, lamda_radar, &
                    CBACK, mixingrulestring_s, matrixstring_s,          &
                    inclusionstring_s, hoststring_s,                    &
                    hostmatrixstring_s, hostinclusionstring_s)
              f_d = N0_s(k)*xxDs(n)**xmu_s * DEXP(-lams*xxDs(n))
              eta = eta + f_d * CBACK * simpson(n) * xdts(n)
           enddo
           ze_snow(k) = SNGL(lamda4 / (pi5 * K_w) * eta)
          endif

!..Reflectivity contributed by melting graupel

          if (L_qg(k) .and. L_qg(k_0) ) then
           fmelt_g = MAX(0.005d0, MIN(1.0d0-rg(k)/rg(k_0), 0.99d0))
           eta = 0.d0
           lamg = 1./ilamg(k)
           do n = 1, nrbins
              x = xam_g * xxDg(n)**xbm_g
              call rayleigh_soak_wetgraupel (x,DBLE(xocmg),DBLE(xobmg), &
                    fmelt_g, melt_outside_g, m_w_0, m_i_0, lamda_radar, &
                    CBACK, mixingrulestring_g, matrixstring_g,          &
                    inclusionstring_g, hoststring_g,                    &
                    hostmatrixstring_g, hostinclusionstring_g)
              f_d = N0_g(k)*xxDg(n)**xmu_g * DEXP(-lamg*xxDg(n))
              eta = eta + f_d * CBACK * simpson(n) * xdtg(n)
           enddo
           ze_graupel(k) = SNGL(lamda4 / (pi5 * K_w) * eta)
          endif

       enddo
      endif

      do k = kte, kts, -1
         dBZ(k) = 10.*log10((ze_rain(k)+ze_snow(k)+ze_graupel(k))*1.d18)
      enddo

      end subroutine refl10cm_gsfc

!+---+-----------------------------------------------------------------+

END MODULE  module_mp_gsfcgce
