!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  ADVMET           ADVection step returns local METeorology
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   ADVECTION STEP RETURNS LOCAL METEOROLOGY AT END POINT AFTER
!   INTEPOLATION OF METEOROLOGICAL INFORMATION FROM GRID TO POSITION
!   IN BOTH SPACE AND TIME.  NOT USED IN ADVECTION BUT BY OTHER
!   ROUTINES AS DISPERSION AND DEPOSITION.  THIS ROUTINE PROVIDES TH
!   ONLY INTERFACE OF METEOROLOGICAL INFORMATION TO NON-ADVECTION
!   SUBROUTINES.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 07 Jan 2014 (FN) - add variables column mass & layer height 
!                 28 Mar 2014 (FN) - clean up and rename variables
!                 15 May 2014 (FN) - remove JET
!                 01 Jul 2015 (FN) - clean up
!
!$$$

SUBROUTINE ADVMET(METZ,METO,VMIX,CDEP,RDEP,TRAJ,DSWF,XP,YP,     &
                  KCYCLE,NLVL,ICHEM,XGX,XGY,XZ0,XLU,XZT,        &
                  XXA,XXT,XXQ,XXP,XLVLZZ,XXE,XXX,XXH,           &
                  XCOLMS,XU0,XV0,XSS,XT0,XP0,XH0,XUF,XVF,XSF,   &
                  XZI,XRT,XCF,XDS,                              &
                  ids, ide, jds, jde, kds, kde,    &
                  ims, ime, jms, jme, kms, kme,    &
                  ips, ipe, jps, jpe, kps, kpe     )

  IMPLICIT NONE

  INCLUDE 'DEFMETO.INC'                ! meteo summary at last advection point

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  TYPE(bset),INTENT(OUT)   :: metz (:)   ! profile advection variables
  TYPE(aset),INTENT(INOUT) :: meto       ! surface advection variables
  LOGICAL,   INTENT(IN)    :: vmix,cdep,rdep,traj  
  LOGICAL,   INTENT(IN)    :: dswf          ! downward shortwave flag 

  REAL,      INTENT(IN)    :: xp,yp 
  INTEGER,   INTENT(IN)    :: kcycle,nlvl
  INTEGER,   INTENT(IN)    :: ichem         ! special conversion options

  REAL, DIMENSION(ims:ime,jms:jme), INTENT(IN)   :: xgx,xgy,  &
                                                    xz0,xlu,xzt

  REAL, DIMENSION(ims:ime,kms:kme,jms:jme), INTENT(IN) :: xxa,xxt,xxq,xxp,   &
                                                          xlvlzz,xxe,xxx,xxh

  REAL, DIMENSION(ims:ime,jms:jme), INTENT(IN)   :: xcolms,       &
                                                    xu0,xv0,xss,  &
                                                    xt0,xp0,xh0,  &
                                                    xuf,xvf,xsf,  &
                                                    xzi,xrt,xcf,xds

  INTEGER, INTENT(IN)    ::  ids , ide , jds , jde , kds , kde , &
                             ims , ime , jms , jme , kms , kme , &
                             ips , ipe , jps , jpe , kps , kpe

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  REAL,          PARAMETER :: grav  = 9.80616 ! gravity (m/s2)
  REAL,          PARAMETER :: rdry  = 287.04  ! dry air (J/Kg-K)
  REAL,          PARAMETER :: p2jm  = 100.0   ! mb to j/m3

  INTEGER                  :: kl,ii,jj
  REAL                     :: esat,evap,tvir,sphu,mixr,pres,temp,relh
  REAL                     :: xc,yc,tf,tr,ea,crf,sea,rfhr,zx,zk,var1,var2

!                             variables for diagnostic testing
! REAL                     :: DELP,DELZ,ZLVL,PBOT

!-------------------------------------------------------------------------------
  INTERFACE
!-------------------------------------------------------------------------------
  SUBROUTINE ADV2NT(S,X1,Y1,SS,ims,ime,jms,jme,kms,kme)
  IMPLICIT NONE
  REAL,      INTENT(IN)    :: x1,y1         ! position of interpolated value
  REAL,      INTENT(OUT)   :: ss            ! value of S at x1,y1,z1
  INTEGER,   INTENT(IN)    :: ims , ime , jms , jme , kms , kme
  REAL, DIMENSION(ims:ime,jms:jme), INTENT(IN) :: s
  END SUBROUTINE adv2nt
!-------------------------------------------------------------------------------
  SUBROUTINE ADV2PT(S,X1,Y1,SS,ims,ime,jms,jme,kms,kme)
  IMPLICIT NONE
  REAL,      INTENT(IN)    :: x1,y1         ! position of interpolated value
  REAL,      INTENT(OUT)   :: ss            ! value of S at x1,y1,z1
  INTEGER,   INTENT(IN)    :: ims , ime , jms , jme , kms , kme
  REAL, DIMENSION(ims:ime,jms:jme), INTENT(IN) :: s
  END SUBROUTINE adv2pt
!-------------------------------------------------------------------------------
  SUBROUTINE ADV3NT(S,X1,Y1,ZX,SS,ims,ime,jms,jme,kms,kme)
  IMPLICIT NONE
  REAL,      INTENT(IN)    :: x1,y1         ! position of interpolated value
  REAL,      INTENT(IN)    :: zx            ! vertical interpolation fraction
  REAL,      INTENT(OUT)   :: ss            ! value of S at x1,y1,z1
  INTEGER,   INTENT(IN)    :: ims , ime , jms , jme , kms , kme
  REAL, DIMENSION(ims:ime,kms:kme,jms:jme), INTENT(IN) :: s
  END SUBROUTINE adv3nt
!-------------------------------------------------------------------------------
  END INTERFACE
!-------------------------------------------------------------------------------

! save position indicies within meteo subgrid
  II=NINT(XP)
  JJ=NINT(YP)
  ZX=METO%ZNDX

! grid distance at point
  METO%GDISX=XGX(II,JJ)
  METO%GDISY=XGY(II,JJ)

! terrain surface elevation
  CALL ADV2NT(XZT,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
  METO%ZTER=VAR1

! roughness length 
  METO%AERO=XZ0(II,JJ)

! land-use category 
  METO%LAND=XLU(II,JJ)

!-------------------------------------------------------------------------------
! trajectory option only returns simple marker variable
!-------------------------------------------------------------------------------

  IF(TRAJ)THEN
     IF(METO%FLAG%PRES.GT.0.OR.METO%FLAG%SPHU.GT.0.OR.METO%FLAG%MIXR.GT.0)THEN
!       all trajectories return pressure
        CALL ADV3NT(XXP,XP,YP,ZX,VAR1,ims,ime,jms,jme,kms,kme)
        PRES=VAR1
        IF(METO%FLAG%PRES.GT.0) METO%TMRK(METO%FLAG%PRES)=PRES
     END IF

     IF(METO%FLAG%TPOT.GT.0)THEN
!       isentropic -> save potential temperature
        CALL ADV3NT(XXT,XP,YP,ZX,VAR1,ims,ime,jms,jme,kms,kme)
        METO%TMRK(METO%FLAG%TPOT)=VAR1
     END IF

     IF(METO%FLAG%TAMB.GT.0.OR.METO%FLAG%SPHU.GT.0.OR.METO%FLAG%MIXR.GT.0)THEN
!       ambient temperature
        CALL ADV3NT(XXA,XP,YP,ZX,VAR1,ims,ime,jms,jme,kms,kme)
        TEMP=VAR1
        IF(METO%FLAG%TAMB.GT.0) METO%TMRK(METO%FLAG%TAMB)=TEMP
     END IF

     IF(METO%FLAG%RELH.GT.0.OR.METO%FLAG%SPHU.GT.0.OR.METO%FLAG%MIXR.GT.0)THEN
!       relative humidity  
        CALL ADV3NT(XXQ,XP,YP,ZX,VAR1,ims,ime,jms,jme,kms,kme)
        RELH=VAR1
        IF(METO%FLAG%RELH.GT.0) METO%TMRK(METO%FLAG%RELH)=RELH*100.0
     END IF

     IF(METO%FLAG%SPHU.GT.0.OR.METO%FLAG%MIXR.GT.0)THEN
!       specific humidity (g/kg)
        ESAT=EXP(21.4-(5351.0/TEMP)) ! saturation vapor pressure
        EVAP=RELH*ESAT               ! vapor pressure
        SPHU=0.622*EVAP/PRES         ! specific humidity
        IF(METO%FLAG%SPHU.GT.0) METO%TMRK(METO%FLAG%SPHU)=SPHU*1000.0
     END IF

     IF(METO%FLAG%MIXR.GT.0)THEN
!       water vapor mixing ratio (g/kg)
        MIXR=0.622*EVAP/(PRES-EVAP)
        IF(METO%FLAG%MIXR.GT.0) METO%TMRK(METO%FLAG%MIXR)=MIXR*1000.0
     END IF

     IF(METO%FLAG%TERR.GT.0)THEN 
!       terrain height
        METO%TMRK(METO%FLAG%TERR)=METO%ZTER
     END IF
  END IF

!-------------------------------------------------------------------------------
! parameter profiles used in deposition and mixing calculations
!-------------------------------------------------------------------------------

  !FN-20140107, column mass
   CALL ADV2NT(XCOLMS,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
   METO%TOMU=VAR1

  IF(VMIX.AND.METO%FLAG%MIXD.GT.0)THEN
!    mixing depth                   
     CALL ADV2NT(XZI,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%MIXD=VAR1
     METO%TMRK(METO%FLAG%MIXD)=METO%MIXD 
  END IF

  IF(.NOT.TRAJ.AND.VMIX)THEN

!    to obtain the u-component turbulence (m2/s2)
     CALL ADV3NT(XXH,XP,YP,ZX,VAR1,ims,ime,jms,jme,kms,kme)
     METO%UMIX=VAR1

!    to obtain the v-component turbulence (m2/s2)
     CALL ADV3NT(XXE,XP,YP,ZX,VAR1,ims,ime,jms,jme,kms,kme)
     METO%VMIX=VAR1

     DO KL=1,NLVL
!       set vertical interpolation point to index position
        ZK=KL

!       to obtain the w-component turbulence (m2/s2)
        CALL ADV3NT(XXX,XP,YP,ZK,VAR1,ims,ime,jms,jme,kms,kme)
        METZ(KL)%WMIX=VAR1

!       interpolate pressure to position
        CALL ADV3NT(XXP,XP,YP,ZK,VAR1,ims,ime,jms,jme,kms,kme)
        METZ(KL)%PRES=VAR1

!       interpolate layer height to position
        CALL ADV3NT(XLVLZZ,XP,YP,ZK,VAR1,ims,ime,jms,jme,kms,kme)
        METZ(KL)%ZHGT=VAR1

        IF(CDEP)THEN

!          interpolate humidity to position
           CALL ADV3NT(XXQ,XP,YP,ZK,VAR1,ims,ime,jms,jme,kms,kme)
           METZ(KL)%RHFR=VAR1

!          ambient temperature
           CALL ADV3NT(XXA,XP,YP,ZK,VAR1,ims,ime,jms,jme,kms,kme)
           METZ(KL)%TEMP=VAR1
!          convert to dry air density
           ESAT=EXP(21.4-(5351.0/METZ(KL)%TEMP)) ! saturation vapor pressure
           EVAP=METZ(KL)%RHFR*ESAT               ! vapor pressure
           SPHU=0.622*EVAP/METZ(KL)%PRES         ! specific humidity
           TVIR=METZ(KL)%TEMP*(1.0+0.61*SPHU)    ! virtual temperature

           IF(ICHEM.EQ.8)THEN
!             stilt emulation use dry air density
              METZ(KL)%DENS=(P2JM*(METZ(KL)%PRES-EVAP))/(METZ(KL)%TEMP*RDRY)
           ELSE
!             use pressure and virtual temperature to find density
              METZ(KL)%DENS=(P2JM*METZ(KL)%PRES)/(TVIR*RDRY)
           END IF

        END IF

     END DO
  END IF

!-------------------------------------------------------------------------------
! surface level parameters for resistance and deposition calculations
!-------------------------------------------------------------------------------

  IF(RDEP)THEN
!    interpolate friction velocity to point and time
     CALL ADV2NT(XUF,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
!    convert from gp/min to m/sec
     XC=VAR1*XGX(II,JJ)/60.0

     CALL ADV2NT(XVF,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
!    convert from gp/min to m/sec
     YC=VAR1*XGY(II,JJ)/60.0
!    scalar friction velocity
     METO%USTR=SQRT(XC*XC+YC*YC)

!    interpolate stability function to point and time
     CALL ADV2NT(XSF,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%PSI=VAR1

!    static stability parameter 
     CALL ADV2NT(XSS,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%SSP=VAR1

!    2 meter relative humidity
     CALL ADV2NT(XH0,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%SFCH=VAR1

!    surface pressure
     CALL ADV2NT(XP0,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%SFCP=VAR1

!    low-level ambient temperature
     CALL ADV2NT(XT0,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%SFCT=VAR1

!    low-level scalar wind      
     CALL ADV2NT(XU0,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     XC=VAR1

     CALL ADV2NT(XV0,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     YC=VAR1
     METO%UBAR=SQRT(XC*XC+YC*YC)

!    mixing depth                   
     CALL ADV2NT(XZI,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%MIXD=VAR1
  END IF

  IF(RDEP.OR.(TRAJ.AND.METO%FLAG%DSWF.GT.0))THEN
!    interpolate downward shortwave flux
     IF(DSWF)THEN
        CALL ADV2NT(XDS,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
        METO%DSWF=VAR1
     ELSE
        CONTINUE
!FN-20140515, DSWF=.T.
!!       computation of solar angle only required for gaseous dry deposition
!!       or certain specialized chemistry applications
!        CALL SUNANG(JET,METO%PLAT,METO%PLON,EA,SEA)
!
!!       compute solar flux from solar angle and humidity
!!       required for resistance gaseous dry deposition
!        CALL SUNFLX(NLVL,SEA,METZ%RHFR,METO%DSWF,TR)
     END IF
     IF(METO%FLAG%DSWF.GT.0) METO%TMRK(METO%FLAG%DSWF)=METO%DSWF 
  END IF

  IF(CDEP.OR.(TRAJ.AND.METO%FLAG%RAIN.GT.0))THEN
!    determine cloud fraction at position to weight deposition
!    default=100% which means the rain covers the entire grid cell
     CALL ADV2PT(XCF,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%TCLD=VAR1                   ! time interpolated cloud
     CRF=VAR1                         ! cloud removal fraction

     !FN-20150520, time-step rainfall (m/sec) in metdum.F
     CALL ADV2PT(XRT,XP,YP,VAR1,ims,ime,jms,jme,kms,kme)
     METO%RAIN=MAX(0.0,VAR1)*CRF

   END IF

END SUBROUTINE advmet
