!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  ADVPNT           ADVection of one PoiNT in space
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   ADVECTION OF ONE POINT IN SPACE IS THE PRIMARY ROUTINE THAT IS USED BY
!   THE TRAJECTORY AND DISPERSION SIMULATIONS.  IT IS CALLED EACH TIME STEP.
!   THE ROUTINE CHECKS METEO DATA IN ARRAY, IF POINT FITS WITHIN THE TIME AND
!   SPACE LIMITS CALCULATION CONTINUES, OTHERWISE NEW DATA ARE INPUT.
!   IN ADDITION,  METEO VARIABLES AT THE END OF THE STEP ARE PLACED IN THE
!   METO STRUCTURE.  THOSE VALUES ARE USED IN SUBSEQUENT DISPERSION AND
!   TRAJECTORY OUTPUT ROUTINES.  NOTE THAT ARRAY DIMENSIONS ARE AT THE
!   COMPILED MAXIMUM IN THIS UPPER LEVEL ROUTINE.  ALL ROUTINES CALL
!   HERE REQUIRE SUB-GRID DIMENSIONS, EXCEPT IN THE VERTICAL DIMENSION.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 24 Sep 2012 (FN) - WRF-HYSPLIT coupling initial implementation
!                 07 Jan 2014 (FN) - modify for vertical coordinate
!                 28 Mar 2014 (FN) - clean up and rename variables
!                 28 Apr 2014 (FN) - modify for tight couping
!                 15 May 2014 (FN) - change min => sec
!                 26 Sep 2014 (FN) - move conversion of end-pts (xp,yp) to hyconc
!                 01 Jul 2015 (FN) - clean up 
!
! USAGE:  CALL ADVPNT(METZ,METO,VMIX,CDEP,RDEP,TRAJ,TKERD,TKERN,HDWP,
!                     XP,YP,ZP,DT,TRATIO,KG,ZSG,NLVL,
!                     UBAR,IFHR,ICHEM,KRET)
!     
!   INPUT ARGUMENT LIST:    see below
!   OUTPUT ARGUMENT LIST:   see below
!   INPUT FILES:            none 
!   OUTPUT FILES:           none
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE ADVPNT(METZ,METO,VMIX,CDEP,RDEP,TRAJ,TKERD,TKERN,HDWP,   &
                  XP,YP,ZP,DT,KG,ZSG,NLVL,  &
                  UBAR,IFHR,ICHEM,KRET)

  USE funits
  USE metval

  IMPLICIT NONE

  INCLUDE 'DEFARG2.INC' ! subroutine interfaces
  INCLUDE 'DEFGRID.INC' ! meteorology grid and file
  INCLUDE 'DEFMETO.INC' ! meteo variables returned at advection point

!-------------------------------------------------------------------------------
! argument list definitions
!-------------------------------------------------------------------------------

  TYPE(bset),INTENT(OUT)   :: metz (:)     ! profile advection variables
  TYPE(aset),INTENT(OUT)   :: meto         ! surface advection variables
  LOGICAL,   INTENT(IN)    :: vmix         ! return mixing profile flag
  LOGICAL,   INTENT(IN)    :: cdep         ! return deposition variable flag
  LOGICAL,   INTENT(IN)    :: rdep         ! resistance deposition flag
  LOGICAL,   INTENT(IN)    :: traj         ! return trajectory variables flag
  REAL,      INTENT(IN)    :: tkerd        ! day turbulent kinetic eneregy ratio  
  REAL,      INTENT(IN)    :: tkern        ! night turbulent kinetic eneregy ratio  
  INTEGER,   INTENT(IN)    :: hdwp         ! puff/particle distribution type  
  REAL,      INTENT(INOUT) :: xp,yp,zp     ! particle position for advection
  REAL,      INTENT(IN)    :: dt           ! advection time step (min)
  INTEGER,   INTENT(INOUT) :: kg           ! grid index for calculation
  REAL,      INTENT(IN)    :: zsg (:)      ! vertical sigma levels
  INTEGER,   INTENT(IN)    :: nlvl         ! number of vertical levels
  REAL,      INTENT(OUT)   :: ubar         ! advection velocity for component
  INTEGER,   INTENT(OUT)   :: ifhr         ! current forecast hour
  INTEGER,   INTENT(IN)    :: ichem        ! special conversion options
  INTEGER,   INTENT(OUT)   :: kret         ! return for point off grid

!-------------------------------------------------------------------------------
! internal variable definitions
!-------------------------------------------------------------------------------

  INTEGER :: iip,jjp   ! integer number of particles' grid index
  LOGICAL :: offg      ! off-grid, sub-grid

  INTEGER :: lx1       ! subgrid corner point
  INTEGER :: ly1
  INTEGER :: nxs       ! subgrid size
  INTEGER :: nys

! INTEGER                  :: nxt,nyt,nzs
  INTEGER                  :: jtime,hdwpx
  REAL                     :: xx,yy,zz
  CHARACTER(80)            :: ecode

!-------------------------------------------------------------------------------
! external variable definitons
!-------------------------------------------------------------------------------

  COMMON /GBLGRD/ HYGD, DREC, HYFL

!-------------------------------------------------------------------------------

  KG=1
  OFFG=.FALSE.

!-------------------------------------------------------------------------------
! compute new position
!-------------------------------------------------------------------------------

! map position from meteo grid to sub-grid
  XX=XP !XP-LX1+1 !FN-20141010
  YY=YP !YP-LY1+1
  ZZ=ZP

! advection for one time step
  HDWPX=HDWP
  IF(HDWPX.GE.100) HDWPX=MOD(HDWP/10,10) ! complex mode

!FN-20140421, HDWPX=HDWP=INITD=0
  IF(HDWPX.LE.4)THEN
!    standard 3D atmpospheric advection
     CALL ADVIEC(HY_U,HY_V,HY_W,XX,YY,ZZ,METO%ZNDX,DT,                   &
                 ddims(1),ddims(2),ddims(3),ddims(4),ddims(5),ddims(6),  &
                 mdims(1),mdims(2),mdims(3),mdims(4),mdims(5),mdims(6),  &
                 pdims(1),pdims(2),pdims(3),pdims(4),pdims(5),pdims(6)   )

  ELSEIF(HDWPX.EQ.5)THEN
!!    test for special surface advecting particles
!     CALL ADVSFC(UF(:,:),VF(:,:),                                    &
!          K1,K2,NLVL,MTIME,JET,XX,YY,DT,DREC%TAVRG,BACK,               &
!          HYGD%GLOBAL,HYGD%NX,HYGD%NY)

  ELSEIF(HDWPX.EQ.6)THEN
!!    lagrangian isobaric sampling
!     CALL ADVISO(U(:,:,:),V(:,:,:),P(:,:,:),ZSG,                &
!          K1,K2,NLVL,MTIME,JET,ZMDL,XX,YY,ZZ,METO%ZNDX,DT,TRATIO,              &
!          DREC%TAVRG,BACK,                                             &
!          HYGD%GLOBAL,HYGD%NX,HYGD%NY,                 &
!          METO%UVEL,METO%VVEL)

  ELSE
     KG=0
     KRET=1
     WRITE(*,*)' NOTICE advpnt: invalid HDWP/INITD option - ',HDWP
     RETURN
  END IF

! save advection distance as a wind speed (grid pts / min)
  UBAR = MAX(ABS(XX-XP),ABS(YY-YP))/ABS(DT)

! map position back to meteo grid
  XP=XX    !XX+LX1-1 !FN-20141010
  YP=YY    !YY+LY1-1
  ZP=MIN(1.0,ZZ)

! meteo variables interpolated to last advection point
  CALL ADVMET(METZ,METO,VMIX,CDEP,RDEP,TRAJ,DREC%DSWF,XP,YP,       &
              DREC%ACYCLE,NLVL,ICHEM,HY_GX,HY_GY,HY_Z0,HY_LU,HY_ZT,        &
              HY_A,HY_T,HY_Q,HY_P,LVLZZ,HY_E,HY_X,HY_H,           &
              COLMS,UU0,VV0,HY_SS,TT0,PP0,RH0,HY_UF,HY_VF,HY_SF,   &
              HY_ZI,RT0,CF0,HY_DS,                              &
              ddims(1),ddims(2),ddims(3),ddims(4),ddims(5),ddims(6),  &
              mdims(1),mdims(2),mdims(3),mdims(4),mdims(5),mdims(6),  &
              pdims(1),pdims(2),pdims(3),pdims(4),pdims(5),pdims(6)   )


! test if particle above the model top                  
  IF(TRAJ.AND.ZP.LT.ZSG(NLVL))THEN
!    terminate trajectories
     KG=0
     KRET=1
     WRITE(*,*)' NOTICE advpnt: trajectory above data domain'
     WRITE(*,*)' Trajectory sigma pt: ',ZP 
     WRITE(*,*)' Top of model domain: ',ZSG(NLVL) 
     RETURN
  ELSE
!    maintain particles (full reflection turbulence assumed)
     ZP=MAX(ZSG(NLVL),ZP)
  END IF

  IFHR=0 !FN-20140421, check
  KRET=0
  RETURN

!-------------------------------------------------------------------------------
! memory allocation errors
!-------------------------------------------------------------------------------

9000 WRITE(*,*)'*ERROR* advpnt: memory allocation - ',KRET,ECODE 
     STOP 900

END SUBROUTINE advpnt
