!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  CONSET           CONcentration grid SET data entry
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   CONCENTRATION GRID SET IS THE DATA ENTRY FOR SAMPLING GRID LOCATION
!   LEVELS, AND TIME INFORMATION FOR START, STOP, AND DURATION.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 15 May 2014 (FN) - add accumulated seconds
!                 15 Jul 2014 (FN) - delete reading CONTROL
!                 26 Sep 2014 (FN) - clean up
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE CONSET(CONC,NUMGRD,CGSIZE,CPACK,OLAT,OLON, &
                  XCNLAT,XCNLON,XDELLAT,XDELLON,XSPLAT,XSPLON)

  IMPLICIT NONE

  INCLUDE 'DEFCONC.INC'         ! pollutant and concentration grid

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------
 
  TYPE(cset), INTENT(OUT)   :: conc(:)              ! each concentration grid 
  INTEGER,    INTENT(IN)    :: numgrd               ! number concen grids
  REAL,       INTENT(INOUT) :: cgsize               ! minimum grid spacing (km) 
  INTEGER,    INTENT(IN)    :: cpack                ! concentration packing
  REAL,       INTENT(IN)    :: olat, olon           ! starting point

  REAL,DIMENSION(NUMGRD), INTENT(IN)  :: xcnlat,xcnlon,xdellat,xdellon,xsplat,xsplon

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  REAL,    PARAMETER :: ympd = 111.198323        ! km/deg-lat 
  REAL,    PARAMETER :: dgpr = 57.295828         ! deg/radian
  INTEGER, PARAMETER :: maxxp = 360
  INTEGER, PARAMETER :: maxyp = 181

  INTEGER           :: ii,jj,kl,kk,knum
  REAL              :: clat2,clon2,xmpd,cnlat,cnlon,splonm,splatm,splat,splon

!-------------------------------------------------------------------------------
! primary loop for the number of concentration grids
!-------------------------------------------------------------------------------

  DO KK=1,NUMGRD

     print *,'www conset',xcnlat(kk),xcnlon(kk),xdellat(kk),xdellon(kk)

     CONC(KK)%LEVELS=1

     CONC(KK)%HEIGHT(:)=50  !FN-0921

!FN-20140715, delete

!-------------------------------------------------------------------------------
! horizontal grid definition
!-------------------------------------------------------------------------------

!FN-20140715

   CONC(KK)%DELT_LAT=xdellat(KK)
   CONC(KK)%DELT_LON=xdellon(KK)

   CNLAT=XCNLAT(KK)
   CNLON=XCNLON(KK)
   SPLAT=XSPLAT(KK)
   SPLON=XSPLON(KK)

!FN-20140715
     IF(CNLAT.EQ.0.0.AND.CNLON.EQ.0.0)THEN
        CNLAT=OLAT
        CNLON=OLON
     END IF

! maximum concentration grid span
  SPLATM=MAXYP*CONC(KK)%DELT_LAT
  SPLONM=MAXXP*CONC(KK)%DELT_LON

!FN-20140715
  IF(SPLAT.EQ.0.0) SPLAT=SPLATM
  IF(SPLON.EQ.0.0) SPLON=SPLONM

! determine minimum grid spacing for all grids
! km/degree-longitude
  XMPD=YMPD*COS(CNLAT/DGPR)
  CONC(KK)%SIZE=MIN(XMPD*CONC(KK)%DELT_LON,YMPD*CONC(KK)%DELT_LAT)
! return minimum spacing over all grids
  CGSIZE=MIN(CONC(KK)%SIZE,CGSIZE)

! compute lower left corner
  CONC(KK)%X1Y1_LAT=MAX(-90.0,CNLAT-SPLAT/2.0)
  CONC(KK)%X1Y1_LON=CNLON-SPLON/2.0

! compute upper right corner
  CLAT2=MIN(90.0,CNLAT+SPLAT/2.0)
  CLON2=CNLON+SPLON/2.0

! special case for global grid
  IF(SPLON.GE.360.0)THEN
     CONC(KK)%X1Y1_LON=-180.0
     CLON2=180.0-CONC(KK)%DELT_LON
     CNLON=0.0
!    recompute span
     SPLON=CLON2-CONC(KK)%X1Y1_LON+1.0
  END IF

  IF(SPLAT.GE.180.0)THEN
     CONC(KK)%X1Y1_LAT=-90.0
     CLAT2=90.0
     CNLAT=0.0
!    recompute span
     SPLAT=CLAT2-CONC(KK)%X1Y1_LAT+1.0
  END IF

! determine number of grid points (not to exceed dimension)
  CONC(KK)%NUMB_LAT=1+NINT(SPLAT/CONC(KK)%DELT_LAT)
  CONC(KK)%NUMB_LON=1+NINT(SPLON/CONC(KK)%DELT_LON)

! dateline correction (RRD - 20/10/99)
  IF(CONC(KK)%X1Y1_LON.LT.-180.0) CONC(KK)%X1Y1_LON=CONC(KK)%X1Y1_LON+360.0

  print *,'www cgsize=',CGSIZE,CONC(KK)%SIZE
  print *,'www splat=',splat,splon,cnlat,cnlon
  print *,'www numb_lat=',CONC(KK)%NUMB_LAT,CONC(KK)%NUMB_LON,CONC(KK)%X1Y1_LAT,CONC(KK)%X1Y1_LON

!-------------------------------------------------------------------------------
! Redefine concentration grid for non-regular grids (cpack=2). The span will
! be set such that the number of grid points equals one and the grid center
! will be moved to corresond with the corner point.
!-------------------------------------------------------------------------------

  IF(CPACK.EQ.2)THEN
     CONC(KK)%X1Y1_LAT=CNLAT
     CONC(KK)%X1Y1_LON=CNLON
     CONC(KK)%NUMB_LAT=1
     CONC(KK)%NUMB_LON=1
  END IF

!-------------------------------------------------------------------------------
! output file names and directory
!-------------------------------------------------------------------------------

!FN-20140715, delete

!-------------------------------------------------------------------------------
! vertical grid levels or spacing
!-------------------------------------------------------------------------------

!FN-20140715, delete

!-------------------------------------------------------------------------------
! set sampling time intervals
!-------------------------------------------------------------------------------

!FN-20140715, delete

!-------------------------------------------------------------------------------
! check for consistency on snapshot and maximum value grids
!-------------------------------------------------------------------------------

! special case when SNAP<0 it specifies the averaging time in hours
! and DELTA becomes the interval at which that average is output
! modified 1/10/2007 to remain negative
  IF(CONC(KK)%SNAP.LT.0) CONC(KK)%SNAP=60*CONC(KK)%SNAP

! defining a maximum concentration grid requires a snapshot or average grid
  IF(CONC(KK)%SNAP.EQ.2)THEN 
     KNUM=0
     DO KL=1,NUMGRD 
        IF(CONC(KL)%SNAP.EQ.1)KNUM=KNUM+1
        IF(CONC(KL)%SNAP.EQ.2)KNUM=KNUM+1
     END DO
     IF(KNUM.EQ.0)THEN
        WRITE(*,*)'*ERROR* conset: maximum concentration grid defined for #',KK
        WRITE(*,*)'Requires definition of an identical snapshot or average grid!'
        STOP
     END IF
     IF(KNUM.GT.2)THEN
        WRITE(*,*)'*ERROR* conset: maximum concentration grid defined for #',KK
        WRITE(*,*)'Requires one snapshot or average grid ... not multiple!'
        STOP
     END IF
  END IF

!-------------------------------------------------------------------------------
! check if concentration grid covers source location (not required)
!-------------------------------------------------------------------------------

  II=NINT(1.0+(OLON-CONC(KK)%X1Y1_LON)/CONC(KK)%DELT_LON)
     IF(OLON.LT.0.0.AND.CONC(KK)%X1Y1_LON.GT.0.0)              &
     II=NINT(1.0+(OLON-CONC(KK)%X1Y1_LON+360)/CONC(KK)%DELT_LON)
  JJ=NINT(1.0+(OLAT-CONC(KK)%X1Y1_LAT)/CONC(KK)%DELT_LAT)
  IF(JJ.GT.CONC(KK)%NUMB_LAT.OR.JJ.LT.1.OR.      &
     II.GT.CONC(KK)%NUMB_LON.OR.II.LT.1) THEN
     WRITE(*,*)'WARNING conset: source point not on concentration grid #',KK
  END IF

END DO

END SUBROUTINE conset
