!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  EMSPNT           EMiSsion puff/particle at a PoiNT
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   EMISSION INITIALIZATION STARTS A NEW PUFF OR PARTICLE IF THE
!   CURRENT MODEL TIME IS WITHIN THE TIME LIMITS SPECIFIED FOR START
!   A POLLUTANT RELEASE.  EMISSIONS CAN BE STARTED AGAIN AT QCYCLE
!   INTERVALS.  IF MULTIPLE RELEASE LOCATIONS ARE DEFINED THEN THE
!   EMISSIONS ARE UNIFORMLY DISTRIBUTED WITHIN A LAYER FROM THE LAST
!   STARTING HEIGHT TO THE CURRENT STARTING HEIGHT WHEN TWO RELEASE
!   ARE IN THE SAME LOCATION.  OTHERWISE IT IS A POINT SOURCE.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 15 May 2014 (FN) - change min => sec
!
! USAGE:  CALL EMSPNT(SPOT,DIRT,NLOC,NUMTYP,NBPTYP,KPM,INITD,DT,JET,NSORT,
!              MASS,XPOS,YPOS,ZPOS,SIGH,SIGU,SIGV,SIGW,HDWP,PAGE,PTYP,PGRD,
!              QCYCLE,NUMPAR,MAXPAR,job_id,num_job,ichem,kret,GDISX,GDISY)
!
!   INPUT ARGUMENT LIST:    see below
!   OUTPUT ARGUMENT LIST:   see below
!   INPUT FILES:            none
!   OUTPUT FILES:           none
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE EMSPNT(SPOT,DIRT,NLOC,NUMTYP,NBPTYP,KPM,INITD,DT,JET,NSORT,MASS, &
                  XPOS,YPOS,ZPOS,SIGH,SIGU,SIGV,SIGW,HDWP,PAGE,PTYP,PGRD,   &
                  QCYCLE,NUMPAR,MAXPAR,job_id,num_job,ichem,kret,GDISX,GDISY)

  USE funits

  IMPLICIT NONE

  INCLUDE 'DEFCONC.INC' ! pollutant and concentration grid
  INCLUDE 'DEFSPOT.INC' ! multiple source information

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  TYPE(rset),INTENT(INOUT) :: spot(:) ! source location characteristics
  TYPE(pset),INTENT(INOUT) :: dirt(:) ! for each pollutant type 
  INTEGER,   INTENT(IN)    :: nloc    ! total number of source locations
  INTEGER,   INTENT(IN)    :: numtyp  ! number of pollutant types
  INTEGER,   INTENT(IN)    :: nbptyp  ! number of bins per type   
  INTEGER,   INTENT(INOUT) :: kpm     ! number of puffs or particles
  INTEGER,   INTENT(IN)    :: initd   ! initial distribution type
  REAL,      INTENT(IN)    :: dt      ! time step (min)
  REAL,      INTENT(IN)    :: qcycle  ! optional emission cycle time in hours
  INTEGER,   INTENT(IN)    :: numpar  ! maximum number of particles permitted

  INTEGER,   INTENT(INOUT) :: nsort (:)   ! index of sorted elements
  REAL,      INTENT(INOUT) :: mass  (:,:) ! mass of pollutant (arbitrary units)
  REAL,      INTENT(INOUT) :: xpos  (:)   ! horizontal position (grid units)
  REAL,      INTENT(INOUT) :: ypos  (:)   ! horizontal position (grid units)
  REAL,      INTENT(INOUT) :: zpos  (:)   ! puff center height (sigma)
  REAL,      INTENT(INOUT) :: sigh (:)    ! horizontal puff sigma 
  REAL,      INTENT(INOUT) :: sigu (:)    ! turbulence u'2    
  REAL,      INTENT(INOUT) :: sigv (:)    ! turbulence v'2
  REAL,      INTENT(INOUT) :: sigw (:)    ! turbulence w'2 or vertical puff sigma
  INTEGER,   INTENT(INOUT) :: hdwp  (:)   ! Horizontal distribution pollutant
  INTEGER,   INTENT(INOUT) :: page  (:)   ! pollutant age since release (min)
  INTEGER,   INTENT(INOUT) :: ptyp  (:)   ! pollutant type index number
  INTEGER,   INTENT(INOUT) :: pgrd  (:)   ! meteorological grid of puff position
  INTEGER,   INTENT(IN)    :: maxpar      ! maximum particle number
  INTEGER,   INTENT(IN)    :: job_id      ! mpi implementation 
  INTEGER,   INTENT(IN)    :: num_job     ! mpi implementation
  INTEGER,   INTENT(IN)    :: ichem       ! chemistry options index
  INTEGER,   INTENT(OUT)   :: kret        ! emit denial return code
  REAL,      INTENT(IN)    :: gdisx,gdisy ! meteorological grid spacing

  INTEGER(KIND=8),   INTENT(IN)    :: jet     ! current elapsed time (sec) !FN-20140515

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  REAL, PARAMETER  :: sigr   = 1.54      ! top-hat radius
  LOGICAL          :: emit               ! flag current emissions
  LOGICAL          :: pmit   = .FALSE.   ! previous emission flag
  REAL, PARAMETER  :: PI     = 3.14159265358979

  INTEGER          :: ksb,ksd,initk
  REAL             :: qhrs,qsum,qtot,zpl,delz,qval,qstep,rvalue
  INTEGER          :: np,np1,np2,kcycl,n,nrpts,ii,jj,kk,npar,ktp,kp,nphr,maxdim

  REAL,ALLOCATABLE :: tfact(:)

  SAVE   PMIT

!-------------------------------------------------------------------------------
! check to determine if emissions required
!-------------------------------------------------------------------------------

  IF(.NOT.ALLOCATED(tfact)) ALLOCATE (tfact(numtyp))

  KRET=0
  TFACT=1.0
  EMIT=.FALSE.
  DO KK=1,NUMTYP
     IF(INT(DT).GT.0)THEN
        KSB=JET-DIRT(KK)%START%SACC                       !FN-20140515
        KSD=MAX(INT(DIRT(KK)%QHRS*3600.0),INT(DT))        !FN-20140515, min=>sec
       !print *,'sss emspnt ksb=',ksb,ksd
     ELSE
        KSB=DIRT(KK)%START%SACC-JET                       !FN-20140515
        KSD=INT(MAX(ABS(DIRT(KK)%QHRS*3600.0),ABS(DT)))   !FN-20140515, min=>sec
     END IF
     IF(KSB.GE.0.AND.KSB.LT.KSD) THEN 
        EMIT=.TRUE.
!       adjust for short emissions relative the the time step
        IF(KSD-KSB.LT.ABS(INT(DT))) TFACT(KK)=(KSD-KSB)/ABS(DT)
     END IF
  END DO

  IF(EMIT.AND..NOT.PMIT) print *,'sss emspnt emissions started',JET,EMIT
  IF(.NOT.EMIT.AND.PMIT) print *,'sss emspnt emissions terminated',JET,EMIT
  PMIT=EMIT
  IF(.NOT.EMIT)RETURN

!-------------------------------------------------------------------------------
! determine type of emission and number of locations
!-------------------------------------------------------------------------------

! multiple starting locations can either be point sources
! if each starting point is in a different location or vertical
! line sources if two starting points are in the same location.
! The line source is distributed between the given heights

  IF(NLOC.GT.1)THEN
     SPOT(1)%ZV=SPOT(1)%ZP
     DO N=2,NLOC
        IF(INT(SPOT(N)%XP*10000.0).EQ.                                     &
           INT(SPOT(N-1)%XP*10000.0).AND.                                  &
           INT(SPOT(N)%YP*10000.0).EQ.                                     &
           INT(SPOT(N-1)%YP*10000.0))THEN

!          when position the same move previous point release
!          height into ZV (line source defined as ZV->ZP)
!          then only emit at locations with ZV<>0
           SPOT(N)%ZV=SPOT(N-1)%ZP
           SPOT(N-1)%ZV=0.0
        ELSE
!          point source bottom equals release height
           SPOT(N)%ZV=SPOT(N)%ZP
        END IF
     END DO

!    count up the number of different x,y release locations with
     NRPTS=0
     DO N=1,NLOC
!       those set to zero are used to determine line source heights
        IF(SPOT(N)%ZV.NE.0.0)NRPTS=NRPTS+1
     END DO

  ELSE
     NRPTS=1
     SPOT(1)%ZV=SPOT(1)%ZP
  END IF

!-------------------------------------------------------------------------------
! set the number of units to emit
!-------------------------------------------------------------------------------

  INITK=INITD
  IF(INITK.GE.100) INITK=MOD(INITD/10,10)
  MAXDIM = SIZE(mass,1)  ! number of pollutants on single particle

  QSUM=0.0
  QHRS=0.0
  NP=NUMTYP*NBPTYP
  IF(ICHEM.EQ.1)NP=1
  DO KK=1,NP
     IF(MAXDIM.EQ.1)THEN
!       pollutant hours is summed because each emitted as an independent particle
        QSUM=QSUM+MAX(ABS(DT/3600.0),ABS(DIRT(KK)%QHRS))  !FN-20140515, min=>
     ELSE
        QSUM=MAX(QSUM,ABS(DT/3600.0),ABS(DIRT(KK)%QHRS))  !FN-20140515, min=>
     END IF
     QHRS=MAX(QHRS,ABS(DIRT(KK)%QHRS))
  END DO
! no emission duration defined then exit
  IF(QHRS.EQ.0.0)RETURN

  IF(INITK.EQ.1.OR.INITK.EQ.2)THEN
!    gaussian or top-hat emissions
     NPAR=1
  ELSE
!    particle emission rate: number particles per hour per source
     IF(NUMPAR.GE.0)THEN
        NPHR=CEILING(FLOAT(NUMPAR/NRPTS)/QSUM)
     ELSE
        NPHR=ABS(NUMPAR)
     END IF
!FN-20141010
!!    particle emissions per time step (requires at least one per job)
!     NPAR=MAX(num_job, CEILING(FLOAT(NPHR)*ABS(DT/3600.0)))  !FN-20140515, min=>
      NPAR=CEILING(FLOAT(NPHR)*ABS(DT/3600.0))
  END IF

!-------------------------------------------------------------------------------
! loop through the number of independent source locations
!-------------------------------------------------------------------------------

  NP=0                   ! internal particle counter

nloop : DO N=1,NLOC

! check for source skip (vertical line source)
  IF(SPOT(N)%ZV.NE.0.0)THEN

! Each pollutant type will start its own trajectory unless this routine is 
! modified accordingly (MAXDIM >1). 

  IF(ICHEM.EQ.1)THEN
!    matrix option each source (n) is a unique pollutant
     NP1=N   
     NP2=N 
  ELSEIF(MAXDIM.GT.1)THEN
!    multiple species on one particle, the pollutant ID=1
     NP1=1   
     NP2=1 
  ELSE
!    default case releases new particle with each pollutant 
     NP1=1
     NP2=NUMTYP                
  END IF

  II=0
! main pollutant loop
  DO KK=NP1,NP2 
! bin size redistibution loop
  DO JJ=1,NBPTYP
! redistribution index
  II=II+1

! all type emissions go into index 1 of particle mass array unless maxdim>1
  KTP=MIN(KK,MAXDIM)

! check if this pollutant requires a start
  IF(MAXDIM.EQ.1)THEN
     EMIT=.FALSE.
     IF(INT(DT).GT.0)THEN
        KSB=JET-DIRT(KK)%START%SACC                      !FN-20140515
        KSD=MAX(INT(DIRT(KK)%QHRS*3600.0),INT(DT))       !FN-20140515, min=>sec
     ELSE
        KSB=DIRT(KK)%START%SACC-JET                      !FN-20140515
        KSD=INT(MAX(ABS(DIRT(KK)%QHRS*3600.0),ABS(DT)))  !FN-20140515, min=>sec
     END IF
     IF(KSB.GE.0.AND.KSB.LT.KSD) EMIT=.TRUE.
  ELSE
     EMIT=.TRUE.
  END IF

  print *,'sss emspnt EMIT=',EMIT,NPAR

  IF(EMIT)THEN
!    multiple emissions only for particles
ploop : DO KP=1,NPAR

!       Use for simulations with multiple processors to only emit particles 
!       when process id even multiple of particle numbe. In a single process
!       environment num_job=1 and job_id=0.

        NP=NP+1

!FN-20141010
!        IF(MOD(NP,num_job).NE.job_id)CYCLE ploop

        KPM=KPM+1
        IF(KPM.GT.MAXPAR)THEN
           KPM=MAXPAR
           WRITE(*,*)'Warning: emspnt - exceeding puff limit'
           KRET=1
           EXIT nloop
        END IF
        NSORT(KPM)=KPM

!       initial position from main program
        XPOS(KPM)=SPOT(N)%XP
        YPOS(KPM)=SPOT(N)%YP
        ZPOS(KPM)=SPOT(N)%ZP

!       horizontal variances all start at zero
        SIGU(KPM)=0.0
        SIGV(KPM)=0.0
        SIGW(KPM)=0.0

        IF(SPOT(N)%AREA.LE.0.0)THEN
!          points source defined for all species
           SIGH(KPM)=0.0
        ELSEIF(ICHEM.EQ.3)THEN
!          special case of dust emissions
           IF(SPOT(N)%QTRM.LT.0.0)THEN
!             old dust emission algorithm contains area
              SIGH(KPM)=SQRT(SPOT(N)%AREA/PI)/SIGR
           ELSE
!             new algorithm contains emission factor x area
              SIGH(KPM)=0.0
           END IF
        ELSE
!          normal point sources ... compute sigma for uniform radius
           SIGH(KPM)=SQRT(SPOT(N)%AREA/PI)/SIGR
           IF(INITK.EQ.0)THEN
!             for 3D particle simulation random adjust to positions
              CALL RANDOM_NUMBER(rvalue)
              XPOS(KPM)=SPOT(N)%XP+SIGR*(RVALUE-0.5)*SIGH(KPM)/GDISX
              CALL RANDOM_NUMBER(rvalue)
              YPOS(KPM)=SPOT(N)%YP+SIGR*(RVALUE-0.5)*SIGH(KPM)/GDISY
           END IF
        END IF

!       multiple locations get initial vertical distribution
!       when the first two starting points at the same x,y position
        DELZ=ABS(SPOT(N)%ZV-SPOT(N)%ZP)
        IF(DELZ.GT.0.0)THEN
           ZPL=MAX(SPOT(N)%ZV,SPOT(N)%ZP)
           IF(INITK.EQ.1.OR.INITK.EQ.2)THEN
!              puff variance set to layer depth
               SIGW(KPM)=ABS(DELZ/SIGR/2.0)
               ZPOS(KPM)=ZPL-DELZ/2.0
            ELSE
!              vertical variance set to zero
               SIGW(KPM)=0.0
! original->   ZPOS(KPM)=ZPL-FLOAT(KP)*DELZ/FLOAT(NPAR)
!              particles get randomly distributed in the 1/npar layer
               CALL RANDOM_NUMBER(rvalue)
               RVALUE=(KP-RVALUE)/NPAR
               ZPOS(KPM)=ZPL-RVALUE*DELZ
            END IF
         END IF

!        initial distribution (see main for definitions)
         HDWP(KPM)=INITD
!        initial age at zero
         PAGE(KPM)=0
!        pollutant type always defined from pollutant index
         IF(NBPTYP.GT.1)THEN
!           complex for representing sub-bin 
            PTYP(KPM)=(KK*1000)+II
         ELSE
!           simple unredistributed
            PTYP(KPM)=KK
            II=KK
         END IF
!        initial grid is the default startup grid from main
         PGRD(KPM)=SPOT(N)%KG

         MASS(:,KPM)=0.0
dloop :  DO KTP=1,MAXDIM
            IF(MAXDIM.GT.1)THEN
               II=KTP
               EMIT=.FALSE.
               IF(INT(DT).GT.0)THEN
                  KSB=JET-DIRT(II)%START%SACC                      !FN-20150515
                  KSD=MAX(INT(DIRT(II)%QHRS*3600.0),INT(DT))       !FN-20140515, min=>sec
               ELSE
                  KSB=DIRT(II)%START%SACC-JET                      !FN-20150515
                  KSD=INT(MAX(ABS(DIRT(II)%QHRS*3600.0),ABS(DT)))  !FN-20140515, min=>sec
               END IF
               IF(KSB.GE.0.AND.KSB.LT.KSD) EMIT=.TRUE.
               IF(.NOT.EMIT) CYCLE dloop
            END IF

         IF(SPOT(N)%QTRM.EQ.0.0)THEN
!           emission defined by species
            QTOT=ABS(DIRT(II)%QHRS)*DIRT(II)%QRATE
         ELSE
!           emission defined by source location
            QTOT=ABS(DIRT(II)%QHRS)*SPOT(N)%QTRM
         END IF

!--------------------------------------------------------
!        Section to configure PM10 dust emissions
!--------------------------------------------------------
!                                             %QTRM %AREA
!        OLD format control file: lat lon hgt -rate  area 
!        NEW format control file: lat lon hgt +ustr  soil 
!        A negative value in the QTRM field forces the model to use
!        the old method (a spatially constant threshold velocity)

         IF(ICHEM.EQ.3)THEN
            SIGU(KPM)=SPOT(N)%QTRM
            SIGV(KPM)=SPOT(N)%AREA

!           Regardless of the dust emission method, emitted dust 
!           particles always get a negative mass, which is made 
!           positive if the threshold is exceeded. Negative mass 
!           dust particles are deleted after the initial advection.

!           Mass represents the time fraction per particle.
            QTOT=-ABS(DIRT(KK)%QHRS)
         END IF
!--------------------------------------------------------

         IF(MAXDIM.EQ.1)THEN
!           number of time steps in emission period
            QSTEP=MAX(1.0,3600.0*ABS(DIRT(KK)%QHRS)/ABS(DT))    !FN-20140515, min=>sec
!           emission per time step
            QVAL=QTOT/QSTEP
!           divide amount over the number of units emitted
            MASS(KTP,KPM)=TFACT(KK)*QVAL/NPAR
         ELSE
            QSTEP=MAX(1.0,3600.0*ABS(DIRT(KTP)%QHRS)/ABS(DT))    !FN-20140515, min=>sec
            QVAL=QTOT/QSTEP
            MASS(KTP,KPM)=TFACT(KTP)*QVAL/NPAR
         END IF

!        internal pollutant dimension loop
         END DO dloop

!     particle loop
      END DO ploop

!  start time test
   END IF

! particle redistribution loop
  END DO

! pollutant type loop
  END DO 

! vertical line source skip duplicate start point
  END IF

! number of sources loop
  END DO nloop

!-------------------------------------------------------------------------------
! check for emission cycling
!-------------------------------------------------------------------------------

  DO KK=1,(NUMTYP*NBPTYP)
!    test for end of emission cycle
     IF(INT(DT).GT.0)THEN
        KSB=JET+INT(DT)-DIRT(KK)%START%SACC                !FN-20140515
        KSD=MAX(INT(DIRT(KK)%QHRS*3600.0),INT(DT))         !FN-20140515, min=>sec
     ELSE
        KSB=DIRT(KK)%START%SACC-JET-INT(DT)                !FN-20140515
        KSD=INT(MAX(ABS(DIRT(KK)%QHRS*3600.0),ABS(DT)))    !FN-20140515, min=>sec
     END IF

     IF(KSB.GE.KSD)THEN        
!       optional restart of emissions at some later time
        KCYCL=NINT(SIGN(QCYCLE*3600.0,DIRT(KK)%QHRS))      !FN-20140515, min=>sec
        DIRT(KK)%START%SACC=DIRT(KK)%START%SACC+KCYCL
     END IF
  END DO

  print *,'sss emspnt KPM=',KPM

END SUBROUTINE emspnt
