!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  METDUM           METeorolgy DUMmy call
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 03 Aug 2012 (FN) - WRF-HYSPLIT coupling initial implementation
!                 28 Oct 2013 (FN) - modify for eta coordinate
!                 28 Mar 2014 (FN) - clean up
!                 15 Apr 2014 (FN) - bug fix on 1st layer hy_t and hy_a
!                 21 Apr 2014 (FN) - remove metpos & metsub
!                                    modify for tight coupling
!                 28 May 2014 (FN) - add KSFC calculation
!                 05 Jun 2014 (FN) - use u- & v-wind at ZNT for 1st layer (eta=0)
!                 10 Jun 2014 (FN) - bring in config_flags
!                 09 Sep 2014 (FN) - add cella (cell area)
!                 26 Sep 2014 (FN) - clean up
!                 20 May 2015 (FN) - replace accumulated rainfall (rainc & rainnc)
!                                    with time-step rainfall (raincv & rainncv, m/sec)
!                 01 Jul 2015 (FN) - clean up
!
!$$$

SUBROUTINE METDUM(grid,config_flags)

  USE metval

  USE module_domain
  USE module_configure
  USE module_dm

  IMPLICIT NONE

  INCLUDE 'DEFARG2.INC' ! subroutine interfaces
  INCLUDE 'DEFGRID.INC' ! meteorology grid and file
  INCLUDE 'DEFMETO.INC' ! meteo variables returned at advection point

!-------------------------------------------------------------------------------
! argument list definitions
!-------------------------------------------------------------------------------

    ! input data
    TYPE(domain)            :: grid

    !  Structure that contains run-time configuration (namelist) data for domain
    TYPE (grid_config_rec_type) , INTENT(IN)          :: config_flags

!-------------------------------------------------------------------------------
! internal variable definitions
!-------------------------------------------------------------------------------

  LOGICAL                  :: vmix

  INTEGER                  :: kret

  INTEGER                  :: ii,jj,kk
  REAL                     :: xi,yj
  CHARACTER(80)            :: ecode

  REAL                     :: dtop,dbot,dtmp
  REAL                     :: dxmm,dymm       ! grid spacing in meter

  INTEGER ids , ide , jds , jde , kds , kde , &
          ims , ime , jms , jme , kms , kme , &
          ips , ipe , jps , jpe , kps , kpe

!-------------------------------------------------------------------------------
! external variable definitons
!-------------------------------------------------------------------------------

  COMMON /GBLGRD/ HYGD, DREC, HYFL

  real, external             :: cgszxy

!-------------------------------------------------------------------------------
! WRF related variables
!-------------------------------------------------------------------------------

  integer                    :: i,j,k

  real, parameter            :: grav   =     9.81  ! m/s**2; gravity


!-------------------------------------------------------------------------------
! get vertical levels infomation
!-------------------------------------------------------------------------------
   CALL get_ijk_from_grid (  grid ,                           &
                             ids, ide, jds, jde, kds, kde,    &
                             ims, ime, jms, jme, kms, kme,    &
                             ips, ipe, jps, jpe, kps, kpe)

   print *,'ccc hysp/metdum'
   print *,ids, ide, jds, jde, kds, kde
   print *,ims, ime, jms, jme, kms, kme
   print *,ips, ipe, jps, jpe, kps, kpe

   ddims(1)=ids   ; ddims(2)=ide
   ddims(3)=jds   ; ddims(4)=jde
   ddims(5)=kds   ; ddims(6)=kde

   mdims(1)=ims   ; mdims(2)=ime
   mdims(3)=jms   ; mdims(4)=jme
   mdims(5)=kms   ; mdims(6)=kme

   pdims(1)=ips   ; pdims(2)=ipe
   pdims(3)=jps   ; pdims(4)=jpe
   pdims(5)=kps   ; pdims(6)=kpe

   if (.not. allocated(etah))  allocate(etah(kms:kme))
   if (.not. allocated(etaf))  allocate(etaf(kms:kme))
   if (.not. allocated(hyeta)) allocate(hyeta(kms:kme))

   etah = grid%znu
   etaf = grid%znw

   hyeta(1)     = etaf(1)
   hyeta(2:kme) = etah(1:kme-1)

   neta = kme-kms+1

   print *,'ccc hyeta=',neta,hyeta

!-------------------------------------------------------------------------------
! Variable allocation
!-------------------------------------------------------------------------------
! check if array size has changed
  IF(.NOT.ALLOCATED(hy_a))THEN

     IF(ALLOCATED(hy_a))THEN
!       if already allocated deallocate before redefining
        ECODE='3d variable deallocation'
        DEALLOCATE( hy_u,hy_v,hy_w,hy_a,hy_t,hy_q,hy_e,hy_p,hy_x,hy_h,STAT=kret)
        IF(kret.NE.0)GOTO 9000

        ECODE='height variable'
        DEALLOCATE( lvlzz,STAT=kret)
        IF(kret.NE.0)GOTO 9000

        ECODE='2d variable deallocation'
        DEALLOCATE( pp0,rt0,cf0,uu0,vv0,tt0,rh0,colms,cella,STAT=kret)
        IF(kret.NE.0)GOTO 9000

        ECODE='Flux variable deallocation'
        DEALLOCATE( hy_uf,hy_vf,hy_hf,hy_sf,hy_zi,hy_ds,hy_ss,STAT=kret)
        IF(kret.NE.0)GOTO 9000

        ECODE='Fixed variable deallocation'
        DEALLOCATE( hy_gx,hy_gy,hy_z0,hy_zt,hy_lu,STAT=kret)
        IF(kret.NE.0)GOTO 9000
     END IF

! 3D varilables, memory order = XZY (ikj)
     ECODE='Velocity variables'
     ALLOCATE(hy_u(ims:ime,kms:kme,jms:jme),hy_v(ims:ime,kms:kme,jms:jme), &
              hy_w(ims:ime,kms:kme,jms:jme),STAT=kret)
     IF(kret.NE.0)GOTO 9000

     ECODE='Temperature variables'
     ALLOCATE(hy_t(ims:ime,kms:kme,jms:jme),hy_a(ims:ime,kms:kme,jms:jme),STAT=kret)
     IF(kret.NE.0)GOTO 9000

     ECODE='Moisture & TKE variables'
     ALLOCATE(hy_q(ims:ime,kms:kme,jms:jme),hy_e(ims:ime,kms:kme,jms:jme),STAT=kret)
     IF(kret.NE.0)GOTO 9000

     ECODE='Diagnostic variables'
     ALLOCATE(hy_p(ims:ime,kms:kme,jms:jme),hy_x(ims:ime,kms:kme,jms:jme), &
              hy_h(ims:ime,kms:kme,jms:jme),STAT=kret)
     IF(kret.NE.0)GOTO 9000

     ECODE='Height variables'
     ALLOCATE(lvlzz(ims:ime,kms:kme,jms:jme),STAT=kret)
     IF(kret.NE.0)GOTO 9000

! 2D varilables
     ECODE='Surface variables'
     ALLOCATE(pp0(ims:ime,jms:jme),rt0(ims:ime,jms:jme),cf0(ims:ime,jms:jme), &
              uu0(ims:ime,jms:jme),vv0(ims:ime,jms:jme),rh0(ims:ime,jms:jme), &
              tt0(ims:ime,jms:jme),colms(ims:ime,jms:jme),cella(ims:ime,jms:jme),&
              STAT=kret)
     IF(kret.NE.0)GOTO 9000
     RT0=0.0    ! default rain field none
     CF0=1.0    ! default field all cloud 

     ECODE='Flux variables'
     ALLOCATE(hy_uf(ims:ime,jms:jme),hy_vf(ims:ime,jms:jme),hy_hf(ims:ime,jms:jme), &
              hy_sf(ims:ime,jms:jme),hy_ss(ims:ime,jms:jme),hy_zi(ims:ime,jms:jme), &
              STAT=kret)
     IF(kret.NE.0)GOTO 9000
     HY_UF=0.0
     HY_VF=0.0
     HY_HF=0.0

     ECODE='Chemistry variables'
     ALLOCATE(hy_ds(ims:ime,jms:jme),STAT=kret)
     IF(kret.NE.0)GOTO 9000

     ECODE='Grid size variables'
     ALLOCATE(hy_gx(ims:ime,jms:jme),hy_gy(ims:ime,jms:jme),STAT=kret)
     IF(kret.NE.0)GOTO 9000


     ECODE='Fixed variables'
     ALLOCATE(hy_z0(ims:ime,jms:jme),hy_zt(ims:ime,jms:jme),hy_lu(ims:ime,jms:jme),STAT=kret)
     IF(kret.NE.0)GOTO 9000

  END IF

!-------------------------------------------------------------------------------
! Compute grid size
!-------------------------------------------------------------------------------

   CALL nl_get_dx ( grid%id , dxmm )
   CALL nl_get_dx ( grid%id , dymm )
   hy_gx(ims:ime,jms:jme) = grid%msfux(ims:ime,jms:jme)*dxmm
   hy_gy(ims:ime,jms:jme) = grid%msfvy(ims:ime,jms:jme)*dymm

   do jj=jms,jme-1
   do ii=ims,ime-1
      dtop=dxmm*grid%msfux(ii+1,jj)
      dbot=dxmm*grid%msfvy(ii,jj)
      dtmp=dymm*grid%msftx(ii,jj)
      cella(ii,jj)=(dtmp+dbot)*dtmp*0.5
   enddo
   enddo

!-------------------------------------------------------------------------------
! Load surface static inputs 
!-------------------------------------------------------------------------------

   hy_lu(ims:ime,jms:jme) = grid%lu_index(ims:ime,jms:jme) ! land use category
   hy_z0(ims:ime,jms:jme) = grid%znt(ims:ime,jms:jme)      ! roughness length (m)
   hy_zt(ims:ime,jms:jme) = grid%ht(ims:ime,jms:jme)       ! terrain elevation (m)

!-------------------------------------------------------------------------------
! Load surface data
!-------------------------------------------------------------------------------

   !FN-20140605
   tt0(ims:ime,jms:jme) = grid%th2(ims:ime,jms:jme)       ! use potential temp at 2m (deg-K)
   pp0(ims:ime,jms:jme) = grid%psfc(ims:ime,jms:jme)
       pp0 = pp0/100.                                     ! model sfc press (mb)
   rh0(ims:ime,jms:jme) = grid%q2(ims:ime,jms:jme)        ! use qv at 2m (kg/kg)
   uu0(ims:ime,jms:jme) = grid%uz0(ims:ime,jms:jme)       ! use u-component wind at ZNT (m/s)
   vv0(ims:ime,jms:jme) = grid%vz0(ims:ime,jms:jme)       ! use v-component wind at ZNT (m/s)
   hy_zi(ims:ime,jms:jme) = grid%pblh(ims:ime,jms:jme)    ! mixed layer depth (m)
   hy_uf(ims:ime,jms:jme) = grid%ust(ims:ime,jms:jme)     ! u* in similarity theory (m/s)
   hy_hf(ims:ime,jms:jme) = grid%hfx(ims:ime,jms:jme)     ! upward heat flux at sfc (w/m2)
   hy_ds(ims:ime,jms:jme) = grid%swdown(ims:ime,jms:jme)  ! downward shortwave flux at sfc (w/2)

   !FN-20150520
   rt0(ims:ime,jms:jme) = grid%raincv(ims:ime,jms:jme)+grid%rainncv(ims:ime,jms:jme)
       rt0 = (rt0/1000.)/grid%dt                          ! time-step rainfall (m/sec)

   colms(ims:ime,jms:jme) = grid%mu_1(ims:ime,jms:jme)+grid%mub(ims:ime,jms:jme)
                                                          ! "MU" perturbation dry air mass in column (Pa)
                                                          ! "MUB" base state dry air mass in column (Pa)

!-------------------------------------------------------------------------------
! Load 3D data
!-------------------------------------------------------------------------------
 
   ! local air pressure (mb)
   hy_p(:,2:kme,:) = (grid%p(:,kms:kme-1,:)+grid%pb(:,kms:kme-1,:))/100.
   hy_p(:,1,:)     = pp0(:,:)

   ! potential temp (deg-K)
   do jj=jms,jme
   do ii=ims,ime
    do kk=2,kme
       if (grid%t_2(ii,kk-1,jj) .ne. 0) then 
          hy_t(ii,kk,jj) = grid%t_2(ii,kk-1,jj)+300.
       endif
    enddo
    hy_t(ii,1,jj) = tt0(ii,jj)
   enddo
   enddo

   ! ambient temp (deg-K)
   do jj=jms,jme
   do ii=ims,ime
    do kk=1,kme
       if (hy_t(ii,kk,jj) .gt. 0 .and. hy_p(ii,kk,jj) .gt. 0) then
          hy_a(ii,kk,jj) = hy_t(ii,kk,jj)*(hy_p(ii,kk,jj)/1000.0)**0.286
       endif
    enddo
   enddo
   enddo

   ! u-component of wind
   hy_u(:,2:kme,:) = grid%u_2(:,kms:kme-1,:)
   hy_u(:,1,:) = uu0(:,:)

   ! v-component of wind
   hy_v(:,2:kme,:) = grid%v_2(:,kms:kme-1,:)
   hy_v(:,1,:) = vv0(:,:)

   ! compute height using (PH+PHB)/g +HGT
   do jj=jms,jme
   do ii=ims,ime
    do kk=2,kme
       dbot=(grid%ph_2(ii,kk-1,jj)+grid%phb(ii,kk-1,jj))/grav
       dtop=(grid%ph_2(ii,kk,jj)+grid%phb(ii,kk,jj))/grav
       lvlzz(ii,kk,jj)=((dbot+dtop)*0.5)-hy_zt(ii,jj)
    enddo
   enddo
   enddo
   lvlzz(:,1,:) = 0.0

   ! z-component of wind and convert to eta-dot
   do jj=jms,jme
   do ii=ims,ime
    do kk=2,kme
      dbot=grid%w_2(ii,kk-1,jj)
      dtop=grid%w_2(ii,kk,jj)
      hy_w(ii,kk,jj) = (dbot+dtop)*0.5

      dtmp=lvlzz(ii,kk,jj)-lvlzz(ii,kk-1,jj)
      if (dtmp .gt. 0.) then
         hy_w(ii,kk,jj) = hy_w(ii,kk,jj)*(hyeta(kk)-hyeta(kk-1))/dtmp
      endif
    enddo
   enddo
   enddo
   hy_w(:,1,:) = 0.0

   ! moisture
   hy_q(:,2:kme,:) = grid%moist(:,2:kme,:,p_qv)
   hy_q(:,1,:) = rh0(:,:)

   print *,'ccc size=',size(hy_t,1),size(hy_t,2),size(hy_t,3)

!-------------------------------------------------------------------------------
! Determine the top of surface layer index
!-------------------------------------------------------------------------------
  !FN-20141010
  do kk=1,kme
     dtmp=0.0
     do ii=ips,ipe
     do jj=jps,jpe
        if (lvlzz(ii,kk,jj) .ge. dtmp) dtmp=lvlzz(ii,kk,jj)
     enddo
     enddo
     if (dtmp .ge. 75.) goto 2999
     isfc=kk
  enddo
  2999 continue

  print *,'ccc isfc=',isfc,kk,dtmp

  dtmp=isfc
  dtmp=wrf_dm_max_real(dtmp)
  isfc=int(dtmp)

  print *,'ccc isfc=',isfc

!-------------------------------------------------------------------------------
! Get dianogsed variables including stabiblity and mixing variables
!-------------------------------------------------------------------------------

  VMIX = .TRUE.

      CALL PRFCOM(config_flags%vscale,config_flags%hscale, &
                  config_flags%tkerd,config_flags%tkern, &
                  ISFC,HY_GX,HY_GY,HY_Z0,HY_ZT,VMIX,     &
                  config_flags%kmixd,config_flags%kmix0, &
                  HY_ZI,pp0,tt0,uu0,vv0,                 &
                  HY_UF,HY_VF,HY_HF,HY_SF,HY_SS,         &
                  HY_U,HY_V,HY_W,HY_A,                   &
                  HY_T,HY_Q,HY_P,LVLZZ,                  &
                  HY_E,HY_H,HY_X,                        &
                  ids, ide, jds, jde, kds, kde,    &
                  ims, ime, jms, jme, kms, kme,    &
                  ips, ipe, jps, jpe, kps, kpe    )

print *,'ccc end of metdum => return back to hymain!'
RETURN
!-------------------------------------------------------------------------------
! memory allocation errors
!-------------------------------------------------------------------------------

9000 print *,'*ERROR* metdum: memory allocation - ',KRET,ECODE 
     STOP 900

END SUBROUTINE metdum 
