!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  ADVRNG           ADVection RaNGe to determine subgrid
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:99-03-03
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   ADVECTION RANGE COMPUTES THE RANGE OF PARTICLE POSITIONS ON THE
!   COMPUTATIONAL GRID (METEO GRID) WHICH IS USED TO DETERMINE THE
!   OPTIMUM LOCATION AND SIZE OF THE METEOROLOGICAL SUBGRID. THE
!   SUBGRID SIZE IS ALSO DETERMINED BY THE FREQUENCY OF THE METEO DATA
!   IN THAT IT IS NECESSARY TO CONSIDER HOW LONG A PARTICLE REMAINS ON
!   THE SUBGRID BEFORE MORE DATA IS REQUIRED TO BE LOADED.  IT IS BETTER
!   TO AVOID LOADING DATA MORE THAN ONCE FOR ANY TIME PERIOD.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: 16 Mar 1999 (RRD) - Initial version of the code
!                 01 Dec 1999 (RRD) - added minimum grid size from namelist
!                 03 Sep 2000 (RRD) - fortran90 upgrade
!                 05 Mar 2001 (RRD) - full grid option adjustment
!                 15 Mar 2001 (RRD) - global cyclic boundary conditions
!                 02 Oct 2001 (RRD) - simultaneous multiple meteorology
!                 17 Jan 2002 (RRD) - umax units & index change in kg loop
!                 09 Feb 2002 (RRD) - global when either at 75%
!                 13 Aug 2002 (RRD) - max subgrid limited to min fullgrid
!                 09 Sep 2002 (RRD) - fortran coding standards
!                 29 May 2003 (RRD) - test for various grid combinations
!                 02 Apr 2004 (RRD) - generic file unit numbers
!                 30 Apr 2006 (RRD) - removed global test (conflict with metsub)
!
! USAGE:  CALL ADVRNG(NGRD,MGMIN,UMAX,KPM,XPOS,YPOS,PGRD)
!
!   INPUT ARGUMENT LIST:     see below
!   OUTPUT ARGUMENT LIST:    see below
!   INPUT FILES:             none
!   OUTPUT FILES:            none
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE ADVRNG(NGRD,MGMIN,UMAX,KPM,XPOS,YPOS,PGRD)

  USE funits

  IMPLICIT NONE

  INCLUDE 'DEFGRID.INC' ! meteorology grid and file

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  INTEGER,    INTENT(IN)    :: ngrd     ! minimum subgrid size (from namelist)
  INTEGER,    INTENT(IN)    :: mgmin    ! minimum subgrid size (from namelist)
  REAL,       INTENT(IN)    :: umax     ! maximum wind speed (km / min)
  INTEGER,    INTENT(IN)    :: kpm      ! number of particles
  REAL,       INTENT(IN)    :: xpos (:) ! particle center positions (grid units)
  REAL,       INTENT(IN)    :: ypos (:) ! particle center positions (grid units)
  INTEGER,    INTENT(IN)    :: pgrd (:) ! particle meteorological grid 

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  INTEGER, ALLOCATABLE  :: lymax(:),lymin(:),lxmax(:),lxmin(:)

  INTEGER               :: kt = 1          ! analysis the same for all times
                                           ! such that the spatial extent of 
                                           ! all grids under kt index identical

  INTEGER               :: k2,kg,kp,kret,nxmin,nymin
  REAL                  :: xpt,ypt,dist,tlat,tlon

!-------------------------------------------------------------------------------
  COMMON /GBLGRD/ HYGD, DREC, HYFL
  SAVE lymax,lymin,lxmax,lxmin
!-------------------------------------------------------------------------------

! check for sufficient number
  IF(KPM.LE.0)RETURN

  IF(.NOT.ALLOCATED(lymax))THEN
     ALLOCATE(lymax(ngrd), STAT=kret)
     ALLOCATE(lymin(ngrd), STAT=kret)
     ALLOCATE(lxmax(ngrd), STAT=kret)
     ALLOCATE(lxmin(ngrd), STAT=kret)

!    initialize grid domain range
     HYGD%LXR=0 
     HYGD%LYR=0
  END IF

  NXMIN=HYGD(1,1)%NX
  NYMIN=HYGD(1,1)%NY
! set maximimum and minimum limits for each grid
  DO KG=1,NGRD
     LXMAX(KG)=1
     LXMIN(KG)=HYGD(KG,KT)%NX
     LYMAX(KG)=1
     LYMIN(KG)=HYGD(KG,KT)%NY

!    maximum subgrid size is the minimum full grid size
     NXMIN=MIN(NXMIN,HYGD(KG,KT)%NX)
     NYMIN=MIN(NYMIN,HYGD(KG,KT)%NY)
  END DO

! For all positions deterime max/min limits, but for multiple grids       
! convert limits from one grid to all other grids to determine if particles
! should influence subgrid limits on other grids besides their current 
! computational grid.

  DO KP=1,KPM
     KG=PGRD(KP)

     DO K2=1,NGRD
        IF(KG.NE.K2)THEN
!          for multiple grids convert position from current grid to true
!          then convert true back to new grid coordinates (mod 05/29/2003)
           IF(HYGD(KG,1)%LATLON.AND.HYGD(K2,1)%LATLON)THEN
              CALL GBL2LL(KG,KT,XPOS(KP),YPOS(KP),TLAT,TLON)
              CALL GBL2XY(K2,KT,TLAT,TLON,XPT,YPT)
           ELSEIF(HYGD(KG,1)%LATLON.AND.(.NOT.HYGD(K2,1)%LATLON))THEN
              CALL GBL2LL(KG,KT,XPOS(KP),YPOS(KP),TLAT,TLON)
              CALL CLL2XY(HYGD(K2,KT)%GBASE,TLAT,TLON,XPT,YPT)
           ELSEIF(.NOT.HYGD(KG,1)%LATLON.AND.HYGD(K2,1)%LATLON )THEN
              CALL CXY2LL(HYGD(KG,KT)%GBASE,XPOS(KP),YPOS(KP),TLAT,TLON)
              CALL GBL2XY(K2,KT,TLAT,TLON,XPT,YPT)
           ELSE
              CALL CXY2LL(HYGD(KG,KT)%GBASE,XPOS(KP),YPOS(KP),TLAT,TLON)
              CALL CLL2XY(HYGD(K2,KT)%GBASE,TLAT,TLON,XPT,YPT)
           END IF

        ELSE
           XPT=XPOS(KP)
           YPT=YPOS(KP)
        END IF

!       particle must be within the main grid to be valid (1/16/2002)
        IF(INT(XPT).GE.1.AND.INT(XPT).LT.HYGD(K2,KT)%NX.AND.   &
           INT(YPT).GE.1.AND.INT(YPT).LT.HYGD(K2,KT)%NY)THEN     
           LXMAX(K2)=MAX(LXMAX(K2),INT(XPT)+1)
           LXMIN(K2)=MIN(LXMIN(K2),INT(XPT))
           LYMAX(K2)=MAX(LYMAX(K2),INT(YPT)+1)
           LYMIN(K2)=MIN(LYMIN(K2),INT(YPT))
        END IF
     END DO
  END DO

!-------------------------------------------------------------------------------
! determine subgrid corner, range, and center position for each grid using kt=1
! as the test grid and writing the same result for that grid to all kt indicies
!-------------------------------------------------------------------------------

  DO KG=1,NGRD

!    Max transport distance = (grid_units/min)*(meteo time interval)
!    subgrid should be "dist" larger to contain particles on edge

     DIST=MAX(1.0,UMAX*FLOAT(DREC(KG,KT)%DELTA)/HYGD(KG,KT)%SIZE)

!    The subgrid range should be at least MGMIN grid units or the size
!    determined from the advection distance plus particle distribution. 
!    Subgrid is not permitted to shrink during a simulation

     HYGD(KG,:)%LXR=MIN(MAX(MGMIN,(NINT(2.0*DIST)+LXMAX(KG)-LXMIN(KG)+3),&
                     HYGD(KG,KT)%LXR),NXMIN)
     HYGD(KG,:)%LYR=MIN(MAX(MGMIN,(NINT(2.0*DIST)+LYMAX(KG)-LYMIN(KG)+3),&
                     HYGD(KG,KT)%LYR),NYMIN)
 
!    when subgrid in either direction reaches 75% of the full grid dimension
!    then set limits to maximum or previously set subgrid to global
     IF((HYGD(KG,KT)%LXR.GT.NINT(0.75*FLOAT(HYGD(KG,KT)%NX))).OR. & 
        (HYGD(KG,KT)%LYR.GT.NINT(0.75*FLOAT(HYGD(KG,KT)%NY))))THEN
        
!        .OR.HYGD(KG,KT)%GLOBAL)THEN -- removed from above if test (30 Apr 2006)

        HYGD(KG,:)%LXR=HYGD(KG,KT)%NX
        HYGD(KG,:)%LXC=FLOAT(HYGD(KG,KT)%NX+1)/2.0
        HYGD(KG,:)%LX1=1

        HYGD(KG,:)%LYR=HYGD(KG,KT)%NY
        HYGD(KG,:)%LYC=(HYGD(KG,KT)%NY+1.0)/2.0
        HYGD(KG,:)%LY1=1

!       input grid is global then set subgrid to global
!       comment following line due to conflict with metsub global test (30 Apr 2006)
!       IF(HYGD(KG,KT)%GBLDAT) HYGD(KG,KT)%GLOBAL=.TRUE.

     ELSE
!       compute grid center and corner position
        HYGD(KG,:)%LXC=FLOAT(LXMAX(KG)+LXMIN(KG))/2.0
        HYGD(KG,:)%LX1=MAX(1,NINT(HYGD(KG,KT)%LXC-FLOAT(HYGD(KG,KT)%LXR)/2.0))
!       check upper end to avoid subgrid compression
        IF(HYGD(KG,KT)%LX1+HYGD(KG,KT)%LXR.GT.HYGD(KG,KT)%NX) &
           HYGD(KG,: )%LX1=HYGD(KG,KT)%NX-HYGD(KG,KT)%LXR

!       compute grid center and corner position
        HYGD(KG,:)%LYC=(LYMAX(KG)+LYMIN(KG))/2.0
        HYGD(KG,:)%LY1=MAX(1,NINT(HYGD(KG,KT)%LYC-HYGD(KG,KT)%LYR/2.0))
!       check upper end to avoid subgrid compression
        IF(HYGD(KG,KT)%LY1+HYGD(KG,KT)%LYR.GT.HYGD(KG,KT)%NY) &
           HYGD(KG,: )%LY1=HYGD(KG,KT)%NY-HYGD(KG,KT)%LYR

     END IF

!    Subgrid data load flag turned off. When particle moves off the subgrid as
!    determined in metsub, the flag is set to true. Subsequent offgrid particles
!    result in an automatic expansion of the subgrid until the flag is reset
!    in this routine resulting in the definition of a new optimal subgrid.

     HYGD(KG,:)%DATLOAD=.FALSE.

!    optional diagnostic message
!    WRITE(KF21,*)' NOTICE advrng: (kg ,xyr,xy1) - ',KG,  &
!       HYGD(KG,KT)%LXR,HYGD(KG,KT)%LYR,HYGD(KG,KT)%LX1,HYGD(KG,KT)%LY1 

  END DO

END SUBROUTINE advrng
