!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  METGRD           computes METeorological GRiD size
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   METEOROLOGICAL GRID COMPUTES THE GRID SIZE FOR THE CURRENT GRID
!   IN ADDITION LOADS OTHER GRID SENSITIVE PARAMETERS SUCH AS LANDUSE
!   AND ROUGHNESS LENGTH.  VARIABLES PASSED THROUGH AS REQUIRED.
!   THE ROUTINE MUST BE CALLED EACH TIME THE SUBGRID CHANGES
!
! PROGRAM HISTORY LOG:
!   LAST REVISION: ...
!                  06 Sep 2012 (FN)  - WRF-HYSPLIT coupling initial implementation
!
! USAGE:  CALL METGRD(KG,KT,LX1,LY1,NXS,NYS,GX,GY)
!
!   INPUT ARGUMENT LIST:      see below
!   OUTPUT ARGUMENT LIST:     see below
!   INPUT FILES:              none
!   OUTPUT FILES:             unit KF21 diagnostic MESSAGE file
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$
   
SUBROUTINE METGRD(KG,KT,LX1,LY1,NXS,NYS,GX,GY)

  USE funits

  IMPLICIT NONE

  INCLUDE 'DEFGRID.INC' ! meteorology grid and file

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  INTEGER,    INTENT(IN)     :: kg            ! grid selection index
  INTEGER,    INTENT(IN)     :: kt            ! time selection index
  INTEGER,    INTENT(IN)     :: lx1,ly1       ! subgrid lower left position
  INTEGER,    INTENT(IN)     :: nxs,nys       ! subgrid dimensions

  REAL,       INTENT(OUT)    :: gx (:,:)      ! grid size array (m)
  REAL,       INTENT(OUT)    :: gy (:,:)      ! grid size array (m)

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  LOGICAL                    :: hset
  INTEGER                    :: ii,jj
  REAL                       :: xi,yj,clat,clon 
  REAL                       :: gdx,gdy

!-------------------------------------------------------------------------------
! external variables
!-------------------------------------------------------------------------------

  COMMON /GBLGRD/ HYGD, DREC, HYFL

  real, external             :: cgszxy

!-------------------------------------------------------------------------------
  INTERFACE
  SUBROUTINE GBL2LL(KG,KT,X,Y,CLAT,CLON)
  IMPLICIT NONE
  INTEGER, INTENT(IN)  :: kg          ! active grid number    
  INTEGER, INTENT(IN)  :: kt          ! active time number    
  REAL,    INTENT(IN)  :: x,y         ! grid position         
  REAL,    INTENT(OUT) :: clat,clon   ! latlon location       
  END SUBROUTINE GBL2LL
!-------------------------------------------------------------------------------
  SUBROUTINE GBLDXY(KG,KT,X,Y,GSX,GSY)
  IMPLICIT NONE
  INTEGER, INTENT(IN)  :: kg          ! active grid number    
  INTEGER, INTENT(IN)  :: kt          ! active time number    
  REAL,    INTENT(IN)  :: x,y         ! grid position         
  REAL,    INTENT(OUT) :: gsx,gsy     ! grid size at that location
  END SUBROUTINE GBLDXY
  END INTERFACE
!-------------------------------------------------------------------------------

  HSET=.FALSE.
! only read terrain file for pressure-sigma file with no terrain height
  IF(.NOT.DREC(KG,KT)%SHGT.AND.DREC(KG,KT)%Z_FLAG.EQ.1)HSET=.TRUE.

  DO JJ=1,NYS
  DO II=1,NXS

!    convert index from subgrid to full grid
     XI=FLOAT(II+LX1-1)
     YJ=FLOAT(JJ+LY1-1)

     IF(HYGD(KG,KT)%LATLON)THEN
!       find position of grid node
        CALL GBL2LL(KG,KT,XI,YJ,CLAT,CLON)

!       meters per grid cell
        CALL GBLDXY(KG,KT,XI,YJ,GDX,GDY)

!       at pole set spacing to 0.25 grid point below pole (2/8/02)
        IF(CLAT.EQ. 90.0) CALL GBLDXY(KG,KT,XI,(YJ-0.25),GDX,GDY)
        IF(CLAT.EQ.-90.0) CALL GBLDXY(KG,KT,XI,(YJ+0.25),GDX,GDY)

!       convert to meters
        GX(II,JJ)=GDX*1000.0
        GY(II,JJ)=GDY*1000.0

     ELSE
!       meters per grid cell
        GX(II,JJ)=CGSZXY(HYGD(KG,KT)%GBASE,XI,YJ)*1000.0

!       accounts for error in grid conversion at pole points
        IF(GX(II,JJ).LT.1.0)GX(II,JJ)=CGSZXY(HYGD(KG,KT)%GBASE,XI+0.5,YJ+0.5)*1000.0
!       conformal projection
        GY(II,JJ)=GX(II,JJ)

!       find position of grid node
!       CALL CXY2LL(HYGD(KG,KT)%GBASE,XI,YJ,CLAT,CLON)
     END IF

  END DO
  END DO

! WRITE(*,*)' NOTICE metgrd: (kg, xyr,xy1) - ',KG,NXS,NYS,LX1,LY1

END SUBROUTINE metgrd  
