!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  ADVIEC           ADVection via Improved Euler-Cauchy
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   ADVECTION VIA IMPROVED EULER-CAUCHY COMPUTES THE 3D ADVECTION OF
!   POINT IN SPACE AND TIME USING A TWO-STEP PROCESS.  THE ADVECTION
!   THE RESULTS OF AN AVERAGE VELOCITY AT THE INITIAL POINT IN SPACE
!   TIME AND THE VELOCITY AT THE FIRST GUESS POSITION (USING THE INITIAL
!   VELOCITY).  ALL INTERPOLATION IS LINEAR.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 19 Sep 2012 (FN) - compute fraction for temporal interpolation
!                 20 Apr 2014 (FN) - call eta2zx to compute fractional index
!                 28 Mar 2014 (FN) - clean up and rename variables
!                 15 May 2014 (FN) - change unit grid/min => grid/sec
!                 11 May 2015 (FN) - consider half grid shift for u- & v-components
!                 01 Jul 2015 (FN) - clean up
!
! Results will differ from standard integration due to internal precision is
! maintained while it appears in the standard approach precision is lost when
! the results are passed back to the main routine. The internal time step
! adjustment is triggered automatically when the advection distance exceeds
! 0.75 of the grid distance. The 0.75 is set in the namelist through
! the tratio variable. The automated integration feature should usually be
! invoked when the integration time step is forced to a value larger than the
! stability parameter and grid resolution permits.
!
!$$$

SUBROUTINE ADVIEC(XU,XV,XW,XX,YY,ZZ,ZX,DT,              &
                    ids , ide , jds , jde , kds , kde , &
                    ims , ime , jms , jme , kms , kme , &
                    ips , ipe , jps , jpe , kps , kpe   )

  IMPLICIT NONE

!-------------------------------------------------------------------------------
! argument list definitions
!-------------------------------------------------------------------------------

  REAL,    INTENT(IN)    :: dt              ! integration step (minutes)
  REAL,    INTENT(INOUT) :: xx,yy,zz        ! old (t) and new (t+dt) position
  REAL,    INTENT(OUT)   :: zx              ! last estimate of vertical index
  INTEGER, INTENT(IN)    ::  ids , ide , jds , jde , kds , kde , &
                             ims , ime , jms , jme , kms , kme , &
                             ips , ipe , jps , jpe , kps , kpe

  REAL, DIMENSION(ims:ime,kms:kme,jms:jme), INTENT(IN) :: xu,xv,xw

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  LOGICAL               :: step,global
  INTEGER               :: ipass,nxs,nys,delt,tsum
  REAL                  :: delx,xt,yt,zt
  REAL                  :: utint,uu1,vtint,vv1,wtint,ww1
  REAL                  :: tratio
  REAL                  :: SUX,SUY,SVX,SVY

!-------------------------------------------------------------------------------
  INTERFACE
  SUBROUTINE ADV3NT(S,X1,Y1,ZX,SS,ims,ime,jms,jme,kms,kme)
  IMPLICIT NONE
  REAL,      INTENT(IN)    :: x1,y1         ! position of interpolated value
  REAL,      INTENT(IN)    :: zx            ! vertical interpolation fraction
  REAL,      INTENT(OUT)   :: ss            ! value of S at x1,y1,z1
  INTEGER,   INTENT(IN)    :: ims , ime , jms , jme , kms , kme
  REAL, DIMENSION(ims:ime,kms:kme,jms:jme), INTENT(IN) :: s
  END SUBROUTINE adv3nt
  END INTERFACE
!-------------------------------------------------------------------------------

!FN-20150511
! defines u,v points relative to mass points
  SUX=0.5
  SUY=0.0
  SVX=0.0
  SVY=0.5

! initial time step and total integration time within this routine
  DELT=INT(DT)
  TSUM=0
  STEP=.FALSE.
  GLOBAL=.FALSE.  !FN-20141010
  TRATIO=0.75     !FN-20141010

! internal integration loop (usually only one interation required)
  DO WHILE (ABS(TSUM).LT.ABS(INT(DT)))

! need to save initial position in two-pass integration
  XT=XX
  YT=YY
  ZT=ZZ

! two pass to obtain average from velocity at old and new
  DO IPASS=1,2

     !FN-20140220
     CALL eta2zx(ZT,ZX)

     !FN-20150511
     CALL ADV3NT(XU,XT+SUX,YT+SUY,ZX,UTINT,ims,ime,jms,jme,kms,kme)
     CALL ADV3NT(XV,XT+SVX,YT+SVY,ZX,VTINT,ims,ime,jms,jme,kms,kme)
     CALL ADV3NT(XW,XT    ,YT    ,ZX,WTINT,ims,ime,jms,jme,kms,kme)

     !FN-20140515, grid/min => grid/sec
     UTINT=UTINT/60.0
     VTINT=VTINT/60.0
     WTINT=WTINT/60.0

     IF(IPASS.EQ.1)THEN

        IF(TSUM.EQ.0)THEN
           DELX=MAX(ABS(UTINT*DELT),ABS(VTINT*DELT))
!          test if maximum step exceeds 3/4 grid cell
           IF(DELX.GT.TRATIO)THEN
              print *,'adviec, delx=',delx
!             compute required internal time step
              DELT=MAX(1,INT(TRATIO*ABS(DELT)/DELX))
              DO WHILE (MOD(INT(ABS(DT)),DELT).NE.0.AND.DELT.GT.1)
                 DELT=DELT-1
              END DO
              DELT=SIGN(DELT,INT(DT))
              STEP=.TRUE.
           END IF
        END IF

!       first pass position simple integration
        XT=XX+UTINT*DELT
        YT=YY+VTINT*DELT
        ZT=ZZ+WTINT*DELT

!       off grid test for particles that may approach the limits of the subgrid
!       most are terminated outside of this routine if within 2 grid pts of edge
        IF(STEP.AND..NOT.GLOBAL)THEN
           print *,'adviec, step=',step,global
           if (xt.lt.float(ips) .or. xt.gt.float(ipe) .or. &
               yt.lt.float(jps) .or. xt.gt.float(jpe) ) then 
!             advect distance for remaining time and then terminate
              XT=XX+UTINT*(DT-TSUM)
              YT=YY+VTINT*(DT-TSUM)
              ZT=ZZ+WTINT*(DT-TSUM)
              RETURN
           END IF
        END IF

!       code modified to include [delt/tsum] 13 Jan 2003
        TSUM=TSUM+DELT

     ELSE
!       final pass average of first and second positions
        XX=0.5*(XT+XX+UTINT*DELT)
        YY=0.5*(YT+YY+VTINT*DELT)
        ZZ=0.5*(ZT+ZZ+WTINT*DELT)
     END IF

  END DO
  END DO

END SUBROUTINE adviec
