!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  DEPELM           DEPosition of a pollutant ELeMent
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   DEPOSITION OF A POLLUTANT ELEMENT COMPUTES GRAVITATIONAL SETTLING,
!   DRY DEPOSITION EITHER EXPLICIT OR VIA THE RESISTANCE METHOD,
!   WET REMOVAL, AND RADIOACTIVE DECAY AS APPLIED TO ONE POLLUTANT
!   PARTICLE OR PUFF EACH TIME STEP.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: 17 Nov 1997 (RRD)
!                 ...
!                 25 Sep 2014 (RRD) - option to use Sportisse-NAME wet removal
!                 05 Dec 2014 (RRD) - option to vary coefficients for rain vs snow
!                 10 Jun 2015 (FN)  - WRF-HYSPLIT coupling
!                 01 Jul 2015 (FN)  - clean up
!
!$$$

SUBROUTINE DEPELM(DIRT,OLAT,IBMO,NLVL,DT,ZSFC,MASS,DEPT,XPOS,YPOS,ZPOS,PAGL,  &
                  SIGW,ICHEM,KTYP,LAND,ROUG,USTR,PSI,SFLX,HDWP,RAIN,     &
                  RHB,RHT,DD,TT,QQ,KSFC,ZHGT,DRYD)

  IMPLICIT NONE

  INCLUDE 'DEFCONC.INC'         ! concentration and pollutant structure

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  TYPE(pset), INTENT(IN)   :: dirt (:) ! for each pollutant type
  REAL,     INTENT(IN)     :: olat     ! origin location
  INTEGER,  INTENT(IN)     :: ibmo     ! computational month
  INTEGER,  INTENT(IN)     :: nlvl     ! number of vertical levels
  REAL,     INTENT(IN)     :: dt       ! time step (sec)                  !FN-20150511
  REAL,     INTENT(IN)     :: zsfc     ! height of terrain surface (m)
  REAL,     INTENT(INOUT)  :: mass (:) ! mass (arbitrary units)
  REAL,     INTENT(IN)     :: xpos
  REAL,     INTENT(IN)     :: ypos
  REAL,     INTENT(INOUT)  :: zpos     ! praticle vertical position (eta) !FN-20150511
  REAL,     INTENT(IN)     :: pagl     ! particle position at height (m)  !FN-20150511
  REAL,     INTENT(INOUT)  :: sigw     ! vert sigma (sigma)
  INTEGER,  INTENT(IN)     :: ichem    ! special depositon options
  INTEGER,  INTENT(IN)     :: ktyp     ! pollutant type index number
  INTEGER,  INTENT(IN)     :: land     ! land use category (1-11)
  REAL,     INTENT(IN)     :: roug     ! aerodynamic roughness length (m)
  REAL,     INTENT(IN)     :: ustr     ! friction velocity (m/s)
  REAL,     INTENT(IN)     :: psi      ! integrated stability function for heat
  REAL,     INTENT(IN)     :: sflx     ! incident short wave flux (w/m2)
  INTEGER,  INTENT(INOUT)  :: hdwp     ! pollutant distribution type (index)
  REAL,     INTENT(IN)     :: rain     ! precipitation value (m/min)
  INTEGER,  INTENT(IN)     :: rhb,rht  ! rh limits for cloud bottom and top (%)
  INTEGER,  INTENT(IN)     :: ksfc     ! top of surface layer index
  REAL,     INTENT(IN)     :: dd   (:) ! air density profile (kg/m3)
  REAL,     INTENT(IN)     :: tt   (:) ! temperature profile
  REAL,     INTENT(IN)     :: qq   (:) ! humidity profile (fraction 0-1)
  REAL,     INTENT(IN)     :: zhgt (:) ! height profile (m)                !FN-20150511
  REAL,     INTENT(OUT)    :: dept (:) ! deposition total (mass units)
  REAL,     INTENT(OUT)    :: dryd (:) ! deposition velocity as computed

!-------------------------------------------------------------------------------
! internally defined variables
!-------------------------------------------------------------------------------

  REAL,  PARAMETER :: grav  = 9.801        ! GRAVITY (m/s2) 
  REAL,  PARAMETER :: dmvc  = 1.789E-02    ! DYNAMIC VISCOSITY (g m-1 s-1)    
  REAL,  PARAMETER :: frep  = 6.53E-08     ! MEAN FREE PATH (m at stp)
  REAL,  PARAMETER :: dstp  = 1.2E+03      ! STP DENSITY (g/m3)     
  REAL,  PARAMETER :: rgas  = 0.082057     ! gas constant (atm-liter / deg-mole)
  REAL,  PARAMETER :: sigr  = 1.54         ! vertical puff scan factor
 
  INTEGER          :: k,kt,kk,kp,krh,klvl,kbot,ktop,maxdim
  REAL             :: sc,vb,vd,vg,fracb,frbct,rtc,depv,zlvl,beta,frea,drop,rate
  REAL             :: aird,dens,pdia,pbot,ptop,cdepth,pdepth,cbot,ctop,rvalue

  REAL             :: zx,sfcl

!-------------------------------------------------------------------------------

  INTERFACE
  SUBROUTINE DEPDRY(DIRT,OLAT,IBMO,KPOL,LAND,ROUG,SFCL,USTR,PSI,SFLX,AIRD,   &
                    TEMP,PDIA,VG,VD)
  IMPLICIT NONE
  INCLUDE 'DEFCONC.INC'         ! concentration and pollutant structure
  TYPE(pset), INTENT(IN)  :: dirt(:)    ! for each pollutant type 
  REAL,     INTENT(IN)    :: olat       ! origin latitude
  INTEGER,  INTENT(IN)    :: ibmo       ! computational month
  INTEGER,  INTENT(IN)    :: kpol       ! polluant index number
  INTEGER,  INTENT(IN)    :: land       ! land-use category
  REAL,     INTENT(IN)    :: roug       ! aerodynamic roughness length (m)
  REAL,     INTENT(IN)    :: sfcl       ! height of constant flux layer (m)
  REAL,     INTENT(IN)    :: ustr       ! friction velocity (m/s)
  REAL,     INTENT(IN)    :: psi        ! integrated stability function heat
  REAL,     INTENT(IN)    :: sflx       ! solar irradiation at sfc (watts/m2)
  REAL,     INTENT(IN)    :: aird       ! ambient air density (g/m3)
  REAL,     INTENT(IN)    :: temp       ! canopy air temperature (deg K)
  REAL,     INTENT(IN)    :: pdia       ! particle diameter (meters)
  REAL,     INTENT(IN)    :: vg         ! gravitational settling velocity (m/s)
  REAL,     INTENT(OUT)   :: vd         ! total deposition velocity (m/s)
  END SUBROUTINE depdry
  END INTERFACE

!-------------------------------------------------------------------------------

  IF(HDWP.EQ.5) RETURN   ! deposited particles cannot deposit again
  IF(HDWP.EQ.6) RETURN   ! 10/12/05 lagrangian sampling option uses CONPAR

  MAXDIM = SIZE(mass,1)  ! number of pollutants on single particle

  CALL eta2zx(ZPOS,ZX)   ! vertical index !FN-20150511

  SFCL=ZHGT(KSFC)        ! height of the surface layer (m) !FN-20150511
 
! rounded vertical index position for meteorology profiles
  KLVL=NINT(ZX)

!FN-20150511
  PBOT=PAGL
  PTOP=PBOT
  KK=MIN(INT(ZX)+1,NLVL)
  PDEPTH=ZHGT(KK)-ZHGT(KK-1)
  PDEPTH=MAX(SFCL,PDEPTH)

! set default pollutant type (over-ride if MAXDIM>1)
  KT=KTYP
  KP=KTYP

! extract redistribution index
  IF(KTYP.GT.1000)THEN
     KP=KTYP/1000
     KT=KTYP-KP*1000
  END IF

! determine rain cloud depth (kbot<>0)
  KBOT=0
  IF(DIRT(KT)%DOWET.AND.ABS(RAIN).GT.0.0)THEN
     CBOT=0.0
     CTOP=0.0

!    determine bottom and top of the precip layer 
     KTOP=NLVL
     DO K=1,NLVL
        KRH=INT(QQ(K)*100.0+0.5)
!       rhb,rht (defaults 80,60) defined in namelist (27 May 2011)
        IF(KBOT.EQ.0.AND.KRH.GE.RHB)KBOT=K
        IF(KBOT.NE.0.AND.KTOP.EQ.NLVL.AND.KRH.LE.RHT)KTOP=K
     END DO 

     IF(KBOT.GT.0)THEN
        CTOP=ZHGT(KTOP)        !FN-20150520
        CBOT=ZHGT(KBOT)        !FN-20150520
     END IF
  END IF

!-------------------------------------------------------------------------------
! check for simultaneous species at position
!-------------------------------------------------------------------------------

  polnum : DO KK=1,MAXDIM

!    use default if multiple species not defined
     IF(MAXDIM.GT.1)KT=KK

     IF(DIRT(KT)%DODRY)THEN
!       explicit definition of the dry deposition velocity
        VD=DIRT(KT)%DRYVL
!       set gravitational settling if defined as particle
        IF(DIRT(KT)%DOGAS)THEN
           VG=0.0
        ELSE
           VG=VD
        END IF

     ELSEIF(DIRT(KT)%DRYVL.LT.0.0)THEN
!       dodry=false, special case for settling with no mass removal
        VG=ABS(DIRT(KT)%DRYVL)
        VD=0.0

     ELSEIF(DIRT(KT)%DOGRV.OR.DIRT(KT)%DORES)THEN
!       local air density
        AIRD=DD(KLVL)
!       convert (kg/m3) --> (g/m3)
        AIRD=AIRD*1000.0

!       compute gravitational settling for particles
        IF(DIRT(KT)%DOGRV)THEN
!          particle density from g/cc g/m3
           DENS=DIRT(KT)%PDENS*1.0E+06
!          particle diameter (um) to (m)
           PDIA=DIRT(KT)%PDIAM*1.0E-06
!          base settling velocity
           VB=PDIA*PDIA*GRAV*(DENS-AIRD)/(18.0*DMVC)
!          slip correction (mean free path = particle diameter)
           FREA=FREP*(DSTP/AIRD)
           SC=1.0+(2.0*FREA/PDIA)*(1.26+0.4*EXP(-0.55*PDIA/FREA))
!          final value apply shape correction factor
           VG=VB*SC/DIRT(KT)%SHAPE
        ELSE
           VG=0.0
        END IF

        IF(DIRT(KT)%DORES)THEN
!          compute resistance based VD
           CALL DEPDRY(DIRT,OLAT,IBMO,KT,LAND,ROUG,SFCL,USTR,PSI,SFLX, &
                       AIRD,TT(1),PDIA,VG,VD)
        ELSE
!          without resistance computation assume Vd = settling
           VD=VG
        END IF

     ELSE
        VD=0.0
        VG=0.0
     END IF
     DRYD(KP)=VD

!    change in vertical position due to settling
     IF(VG.GT.0.0)THEN
!    only do this once per particle (4 Apr 2011)
     IF(KK.EQ.MAXDIM)THEN
        DROP=VG*ABS(DT)         !FN-20150511, deleted "60.0" becuase dt (sec)
        PTOP=MAX(0.0,PTOP-DROP)
        PBOT=MAX(0.0,PBOT-DROP)
        CALL ht2eta(XPOS,YPOS,0.5*(PTOP+PBOT),ZPOS) !FN-20150511
!FN-20150511, check
!!       adjust values for puffs - don't overwrite for particles
!        IF(HDWP.EQ.1.OR.HDWP.EQ.2)THEN
!           SIGW=(PTOP-PBOT)/SIGR/2.0/(ZMDL-ZSFC)
!           PDEPTH=MAX(PTOP-PBOT,1.0)
!        END IF
     END IF
     END IF

!    zero out removal rate constant (1/min) !FN-20150511, (1/sec)
     BETA=0.0

!    if puff within first layer compute dry removal time constant
     IF(VD.GT.0.0.AND.PBOT.LT.SFCL)THEN
        BETA=VD/PDEPTH          !FN-20150511, deleted "60.0"

        IF((ICHEM.EQ.5.OR.ICHEM.EQ.7).AND.(BETA.GT.0.0))THEN
!          The probability deposition option implies that the particle
!          stays on the surface and drops all its mass rather than
!          just losing a fraction of its mass. The probability that
!          the particle will deposit on the surface is determined to
!          be true if a random number (0-1) is less than beta*dt. This
!          can only be used for single pollutant particles (maxdim=1)

           CALL RANDOM_NUMBER(RVALUE)

!          apply exponential for removal > 1%
           IF(ABS(DT)*BETA.GE.0.01) BETA=(1.0-EXP(-ABS(DT)*BETA))/ABS(DT)

           IF(RVALUE.LT.BETA*ABS(DT))THEN
!             random value within dry deposit range
              IF(ICHEM.EQ.7.AND.LAND.EQ.7)THEN
!                convert to deposited particle over water surfaces
!                and hold on to mass for future transport
                 HDWP=5 
              ELSE
!                over solid surfaces particles drop mass
                 DEPT(KK)=MASS(KK)
                 MASS(KK)=0.0                  
              END IF
              EXIT polnum
           ELSE
!             if no dry deposit then it may still wet deposit
              BETA=0.0
           END IF
        END IF

     END IF

!    test for wet removal processes
     IF(KBOT.GT.0)THEN

!       check if any part of pollutant within cloud layer
        IF(HDWP.EQ.1.OR.HDWP.EQ.2)THEN
!          for puffs use the puff depth
           CDEPTH=PDEPTH
        ELSE
!          for particles use the rain layer depth
           CDEPTH=MAX(SFCL,CTOP-CBOT)
        END IF

!       for particles PBOT=PTOP hence FRBCT=FRACB=1.0
        IF(PBOT.LT.CTOP)THEN
!          fraction of pollutant below cloud top
           FRBCT=1.0
           IF(PTOP.GT.CTOP)FRBCT=1.0-(PTOP-CTOP)/CDEPTH

!          fraction of pollutant above cloud bottom
           FRACB=1.0
           IF(PBOT.LT.CBOT)FRACB=1.0-(CBOT-PBOT)/CDEPTH
           IF(PTOP.LT.CBOT)FRACB=0.0

           IF(RAIN.LT.0.0)THEN
!             CMC method using RH because no precip field
!             cloud may be larger (RH can be as low as RHT)
!             but removal only occurs when RH>=RHB
              KRH=INT(QQ(KLVL)*100.0+0.5)
              RATE=MAX(0.0,(KRH-RHB)/(100.0-RHB))
              IF(PBOT.LT.CBOT) BETA=BETA+DIRT(KT)%WETLO*RATE  !FN-20150610, deleted "60.0" becuase dt (sec)
              IF(PTOP.GE.CBOT) BETA=BETA+DIRT(KT)%WETIN*RATE  !FN-20150610, deleted "60.0" becuase dt (sec)

           ELSE
              IF(DIRT(KT)%DOGAS)THEN
!                equilibrium concentration (mass units) for gases
!                applies as long as material is below cloud top
                 IF(PBOT.LT.CTOP)THEN
!                   deposition velocity
                    DEPV=DIRT(KT)%WETGAS*RGAS*TT(KLVL)*ABS(RAIN)
!                   rate constant
                    BETA=BETA+DEPV*FRBCT/CDEPTH
                 END IF

              ELSE
!                for particles
                 IF(DIRT(KT)%WETIN.LT.1.0)THEN
!                   input less than one is defined as scavenging coefficient
!                   where the rainfall rate in mm/hr (from m/min), adapted from NAME, see
!                   (Leadbetter et al., 2014, doi:10.1016/j.jenvrad.2014.03.018; and
!                   Sportisse 2007, doi:10.1016/j.atmosenv.2006.11.057) 

!#                  snow/rain variation not activated in this version
!#                  IF(TT(KLVL).LT.273.0) THEN
!#                     snow scavenging
!#                     RATE=ABS(RAIN*60000.0)**0.31      
!#                  ELSE
!#                     rain scavenging
                       RATE=ABS(RAIN*60000.0)**0.79      
!#                  END IF

                    IF(PBOT.LT.CBOT) BETA=BETA+DIRT(KT)%WETLO*RATE  !FN-20150610, deleted "60.0" becuase dt (sec)
                    IF(PTOP.GE.CBOT) BETA=BETA+DIRT(KT)%WETIN*RATE  !FN-20150610, deleted "60.0" becuase dt (sec)

                 ELSE
!                   input defined as scavenging ratio
!                   only fraction of mass below cloud removed: corrected 10/27/98
                    IF(PBOT.LT.CBOT) BETA=BETA+DIRT(KT)%WETLO*(1.0-FRACB) !FN-20150520, deleted "60.0" becuase dt (sec)

!                   for particles within cloud 
                    IF(PTOP.GT.CBOT)THEN
!                      deposition velocity
                       DEPV=DIRT(KT)%WETIN*ABS(RAIN)
!                      rate constant
                       BETA=BETA+DEPV*FRBCT*FRACB/CDEPTH
                    END IF
!                coefficient or ratio method
                 END IF

!          gas or particle
           END IF

!          precip field available
           END IF

!       pollutant layer test
        END IF

!    cloud layer exists test
     END IF

!    test for radioactive decay
     IF(DIRT(KT)%DORAD)THEN
!       convert half-life (days) to time constant (1/min) !FN-20150610, time constant (1/sec)
        RTC=LOG(0.5)/(DIRT(KT)%RHALF*1440.0*60.0)
!       apply immediately to mass since it doesn't deposit
        MASS(KK)=MASS(KK)*EXP(ABS(DT)*RTC)
     END IF

     IF(BETA.EQ.0.0)THEN
!       no deposition
        DEPT(KK)=0.0
     ELSEIF(ABS(DT)*BETA.LT.0.01)THEN
!       small removal values assume linear approximation
        DEPT(KK)=MASS(KK)*ABS(DT)*BETA
        MASS(KK)=MASS(KK)-DEPT(KK)
     ELSE
!       apply exponential for removal > 1%
        DEPT(KK)=MASS(KK)*(1.0-EXP(-ABS(DT)*BETA))
!       can't remove more than exists
        DEPT(KK)=MIN(MASS(KK),DEPT(KK))
        MASS(KK)=MASS(KK)-DEPT(KK)
     END IF

! species loop
  END DO polnum

END SUBROUTINE depelm
