!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  DEPSET           DEPosition parameters SET from input data
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   DEPOSITION PARAMETERS SET IS THE DATA ENTRY FOR POLLUTANT SPECIES
!   REQUIRED FOR GRAVITATIONAL SETTLING, DRY DEPOSITION, WET REMOVAL
!   AND RADIOACTIVE DECAY COMPUTATIONS. ONE SET OF ENTIRES REQUIRED
!   EACH DEFINED POLLUTANT TYPE.
! 
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 15 Jul 2014 (FN) - delete reading CONTROL
!                 26 Sep 2014 (FN) - clean up
!
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE DEPSET(DIRT,NUMPOL,CDEP,RDEP,SDEP,PDIAM,PDENS,SHAPE, & 
                  DRYVL,GPMOL,ACVTY,DIFTY,HENRY,WETGS,WETIN,WETLO,RHALF,SRATE)

  IMPLICIT NONE

  INCLUDE 'DEFCONC.INC'        ! pollutant and concentration grid

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  TYPE(pset), INTENT(INOUT) :: dirt(:)   ! for each pollutant type 
  INTEGER,    INTENT(IN)    :: numpol    ! number of pollutant types
  LOGICAL,    INTENT(OUT)   :: cdep      ! indicate wet or dry deposition
  LOGICAL,    INTENT(OUT)   :: rdep      ! resistance deposition
  LOGICAL,    INTENT(OUT)   :: sdep      ! resuspension option
  REAL,    DIMENSION(NUMPOL), INTENT(IN)  :: pdiam,pdens,shape
  REAL,    DIMENSION(NUMPOL), INTENT(IN)  :: dryvl,gpmol,acvty,difty,henry
  REAL,    DIMENSION(NUMPOL), INTENT(IN)  :: wetgs,wetin,wetlo,rhalf,srate

!-------------------------------------------------------------------------------
! internally defined variables
!-------------------------------------------------------------------------------

  INTEGER    :: kk

  character(20) :: label

!-------------------------------------------------------------------------------

  CDEP=.FALSE.
  RDEP=.FALSE.
  SDEP=.FALSE.

  print *,'www numpol=',numpol

!-------------------------------------------------------------------------------
! number of pollutants already defined
!-------------------------------------------------------------------------------

  DO KK=1,NUMPOL

!-------------------------------------------------------------------------------
! define as gas or particle
!-------------------------------------------------------------------------------

  DIRT(KK)%DOGAS=.FALSE.
  DIRT(KK)%PDIAM=PDIAM(KK)
  DIRT(KK)%PDENS=PDENS(KK)
  DIRT(KK)%SHAPE=SHAPE(KK)

! all three options required to define a particle
  IF(DIRT(KK)%PDIAM.EQ.0.0 .OR. DIRT(KK)%PDENS.EQ.0.0 .OR.              &
     DIRT(KK)%SHAPE.EQ.0.0)     DIRT(KK)%DOGAS=.TRUE.

!-------------------------------------------------------------------------------
! define dry deposition velocity (over-rides grav settling)
!-------------------------------------------------------------------------------

  DIRT(KK)%DORES=.FALSE.
  DIRT(KK)%DODRY=.FALSE.
  DIRT(KK)%DOVOL=.FALSE.
  DIRT(KK)%DRYVL=DRYVL(KK)
  DIRT(KK)%GPMOL=GPMOL(KK)
  DIRT(KK)%ACVTY=ACVTY(KK)
  DIRT(KK)%DIFTY=DIFTY(KK)
  DIRT(KK)%HENRY=HENRY(KK)

  IF(DIRT(KK)%DRYVL.GT.0.0)DIRT(KK)%DODRY=.TRUE.

! negative molecular weight indicates that emitted mass (kg) will be
! converted to volume units (ppm) only in subroutine emsgrd
  IF(DIRT(KK)%GPMOL.LT.0.0)THEN
     DIRT(KK)%DOVOL=.TRUE.
     DIRT(KK)%GPMOL=ABS(DIRT(KK)%GPMOL)
  END IF

! resistence method requires molecular weight for gases
! and only diameter for particles, however will use molecular
! weight as the turn-on resistence flag for both particles and gases
  IF(DIRT(KK)%GPMOL.GT.0.0)THEN
     DIRT(KK)%DORES=.TRUE.
     IF(DIRT(KK)%DOGAS)THEN   
        IF((DIRT(KK)%DRYVL+DIRT(KK)%ACVTY+DIRT(KK)%DIFTY+               &
            DIRT(KK)%HENRY).EQ.0.0) DIRT(KK)%DORES=.FALSE.
     END IF
  END IF

! check for consistent inputs for gravitational settling
  DIRT(KK)%DOGRV=.FALSE.
  IF(DIRT(KK)%PDIAM.GT.0.0.AND. DIRT(KK)%PDENS.GT.0.0.AND.              &
     DIRT(KK)%SHAPE.GT.0.0)     DIRT(KK)%DOGRV=.TRUE.

! specified dry deposition over-rides gravitational settling
! and resistance method specification
  IF(DIRT(KK)%DRYVL.GT.0.0)THEN
     DIRT(KK)%DOGRV=.FALSE.
     DIRT(KK)%DORES=.FALSE.
  END IF

!-------------------------------------------------------------------------------
! wet removal constants
!-------------------------------------------------------------------------------

  DIRT(KK)%DOWET=.FALSE.
  DIRT(KK)%WETGAS=WETGS(KK)
  DIRT(KK)%WETIN=WETIN(KK)
  DIRT(KK)%WETLO=WETLO(KK)

! check for consistency of wet removal definitions
  IF(DIRT(KK)%DOGAS)THEN
!    wet removal of gasses only if they are soluable
     DIRT(KK)%WETIN=0.0
     DIRT(KK)%WETLO=0.0
  ELSE
     DIRT(KK)%WETGAS=0.0
  END IF

  IF(DIRT(KK)%WETGAS.GT.0.0 .OR. DIRT(KK)%WETIN.GT.0.0 .OR.             &
     DIRT(KK)%WETLO .GT.0.0)     DIRT(KK)%DOWET=.TRUE.

!-------------------------------------------------------------------------------
! radioactive decay options (one year default)
!-------------------------------------------------------------------------------

  DIRT(KK)%DORAD=.FALSE.
  DIRT(KK)%RHALF=RHALF(KK)

  IF(DIRT(KK)%RHALF.GT.0.0)DIRT(KK)%DORAD=.TRUE.

!-------------------------------------------------------------------------------
! deposition resuspension option
!-------------------------------------------------------------------------------

  DIRT(KK)%DOSUS=.FALSE.
  DIRT(KK)%SRATE=SRATE(KK)

  IF(DIRT(KK)%SRATE.GT.0.0)DIRT(KK)%DOSUS=.TRUE.

!-------------------------------------------------------------------------------
! set species independent flags to see if any deposition options enabled
!-------------------------------------------------------------------------------

  IF(DIRT(KK)%DODRY.OR.DIRT(KK)%DORES.OR.                               &
     DIRT(KK)%DOGRV.OR.DIRT(KK)%DOWET.OR.DIRT(KK)%DORAD)                &
     CDEP=.TRUE.

  IF(DIRT(KK)%DRYVL.LT.0.0) CDEP=.TRUE.

  IF(DIRT(KK)%DORES)RDEP=.TRUE.
  IF(DIRT(KK)%DOSUS)SDEP=.TRUE.

  END DO 

END SUBROUTINE depset
