!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  DEPSUS           DEPosition reSUSpension of a pollutant
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   DEPOSITION RESUSPENSION OF A POLLUTANT - ASSUME THAT RATIO (K)
!   OF POLLUTANT IN AIR (C=MASS/M3) TO SURFACE VALUE (S=MASS/M2) =
!   10^-6 DIVIDED BY DURATION OF DEPOSITION IN DAYS.  FOR SIMPLICITY
!   WE ASSUME DAYS ALWAYS = 1.  K ALSO DEFINED AS R/S DS/DT, WHERE R
!   IS RELATED TO THE ATMOSPHERIC RESISTENCE = 1/KU*   THEREFORE
!   THE RESUSPENSION FLUX = S K / R.  NOTE THAT MULTIPLE SPECIES
!   CONCENTRATION FILES WILL RESULTS IN INDEPENDENT PARTICLES FOR EACH
!   DEFINED POLLUTANT FROM ONLY THE FIRST CONCENTRATION GRID.  MULTI
!   GRID DEFINITIONS ARE NOT SUPPORTED IN THIS APPLICATION. NOTE THAT
!   THE METEO VARIABLE FRICTION VELOCITY IS PASSED THROUGH AS THE
!   VALUE COMPUTED IN ADVPNT, THEREFORE ONLY VALID IN A SMALL DOMAIN
!   STANDARD MODEL CONFIGURATION DOES NOT SUPPORT FULL-GRID METEO.
! Revised (05 Aug 2003) such that subgrid meteorology is not supported when
! running this subroutine. The meteo subgrid must be set to full grid by
! setting the MGMIN parameter in the namelist to a large value. Required
! to determine U* at each emission grid point.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: 11 Jun 1997 (RRD) ......
!                 03 Jun 2014 (RRD) - added special warning message file (KF35)
!                 26 Mar 2015 (RRD) - corrected friction velocity definition
!                 29 May 2015 (FN)  - WRF-HYSPLIT inline coupling
!
! USAGE:  CALL DEPSUS(CONC,DIRT,INITD,KGM,KTM,NUMGRD,NUMTYP,DT,ICHEM,KPM,
!                     CSUM,MASS,XPOS,YPOS,ZPOS,SIGH,SIGU,SIGV,SIGW,HDWP,
!                     PAGE,PTYP,PGRD,NSORT,MAXPAR)
!
!   INPUT ARGUMENT LIST:    see below
!   OUTPUT ARGUMENT LIST:   see below
!   INPUT FILES:            none
!   OUTPUT FILES:           none
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE DEPSUS(CONC,DIRT,INITD,KGM,KTM,NUMGRD,NUMTYP,DT,ICHEM,KPM,  &
                  CSUM,MASS,XPOS,YPOS,ZPOS,SIGH,SIGU,SIGV,SIGW,HDWP,   &
                  PAGE,PTYP,PGRD,NSORT,MAXPAR,CGRDXP,CGRDYP)

  USE metval

  IMPLICIT NONE

  INCLUDE 'DEFGRID.INC' ! meteorological grid and file definitions
  INCLUDE 'DEFCONC.INC' ! concentration grid and pollutant definitions

!-------------------------------------------------------------------------------
! argument list definitions
!-------------------------------------------------------------------------------

  TYPE(cset), INTENT(IN)    :: conc(:)      ! for each concentration grid 
  TYPE(pset), INTENT(IN)    :: dirt(:)      ! for each pollutant type 
  INTEGER,    INTENT(IN)    :: initd        ! initial puff/particle distribution
  INTEGER,    INTENT(IN)    :: kgm          ! current meteorological grid number
  INTEGER,    INTENT(IN)    :: ktm          ! current meteorological time number
  INTEGER,    INTENT(IN)    :: numgrd       ! number of concentration grids
  INTEGER,    INTENT(IN)    :: numtyp       ! number of pollutants
  REAL,       INTENT(IN)    :: dt           ! time step (sec)  !FN-20150529
  INTEGER,    INTENT(IN)    :: ichem        ! chemistry option parameter
  INTEGER,    INTENT(INOUT) :: kpm          ! total number of puffs or particles
  REAL,       INTENT(INOUT) :: csum  (:,:,:,:,:)                
  REAL,       INTENT(INOUT) :: mass  (:,:)  ! (species, particles)
  REAL,       INTENT(INOUT) :: xpos  (:)    ! x position (numb part)
  REAL,       INTENT(INOUT) :: ypos  (:)    ! y position (numb part)
  REAL,       INTENT(INOUT) :: zpos  (:)    ! z position (numb part)
  REAL,       INTENT(INOUT) :: sigh  (:)    ! horiz sigma (meters)
  REAL,       INTENT(INOUT) :: sigu  (:)    ! u'2 turbulence  
  REAL,       INTENT(INOUT) :: sigv  (:)    ! v'2 turbulence  
  REAL,       INTENT(INOUT) :: sigw  (:)    ! vertical sigma
  INTEGER,    INTENT(INOUT) :: hdwp  (:)    ! Horizontal distribution
  INTEGER,    INTENT(INOUT) :: page  (:)    ! pollutant age (min)     
  INTEGER,    INTENT(INOUT) :: ptyp  (:)    ! pollutant type index numb 
  INTEGER,    INTENT(INOUT) :: pgrd  (:)    ! particle on this grid numb
  INTEGER,    INTENT(INOUT) :: nsort (:)    ! sorted array index values
  INTEGER,    INTENT(IN)    :: maxpar       ! maximum number of particles
  REAL,       INTENT(IN)    :: cgrdxp(:,:,:)
  REAL,       INTENT(IN)    :: cgrdyp(:,:,:)

!-------------------------------------------------------------------------------
! local variables definitions
!-------------------------------------------------------------------------------

  REAL,    PARAMETER   :: vonk    = 0.40      ! Von Karman's
  REAL,    PARAMETER   :: sigr    = 1.54      ! Radius definition
  REAL,    PARAMETER   :: PIE     = 3.14159265358979
  REAL,    PARAMETER   :: DEGPRD  = 180.0/PIE ! deg per radian
 
  INTEGER              :: ii,jj,kt,kl,kg,kgc,kgl,nxp,nyp
  REAL                 :: xp,yp,depamt,qtot,qrate,area,plat,plon,ustr,lulc

!-------------------------------------------------------------------------------
! external variables definitions
!-------------------------------------------------------------------------------

  COMMON /GBLGRD/ HYGD, DREC, HYFL

!-------------------------------------------------------------------------------
  INTERFACE
!-------------------------------------------------------------------------------
  SUBROUTINE ADV2NT(S,X1,Y1,SS,ims,ime,jms,jme,kms,kme)
  IMPLICIT NONE
  REAL,      INTENT(IN)    :: x1,y1         ! position of interpolated value
  REAL,      INTENT(OUT)   :: ss            ! value of S at x1,y1,z1
  INTEGER,   INTENT(IN)    :: ims , ime , jms , jme , kms , kme
  REAL, DIMENSION(ims:ime,jms:jme), INTENT(IN) :: s
  END SUBROUTINE adv2nt
!-------------------------------------------------------------------------------
  SUBROUTINE ADV2PT(S,X1,Y1,SS,ims,ime,jms,jme,kms,kme)
  IMPLICIT NONE
  REAL,      INTENT(IN)    :: x1,y1         ! position of interpolated value
  REAL,      INTENT(OUT)   :: ss            ! value of S at x1,y1,z1
  INTEGER,   INTENT(IN)    :: ims , ime , jms , jme , kms , kme
  REAL, DIMENSION(ims:ime,jms:jme), INTENT(IN) :: s
  END SUBROUTINE adv2pt
!-------------------------------------------------------------------------------
  END INTERFACE
!-------------------------------------------------------------------------------

  KGC=0
! find the first grid and level which has deposition defined
  DO KG=1,NUMGRD
     IF(KGC.EQ.0)THEN
        DO KL=1,CONC(KG)%LEVELS
           IF(CONC(KG)%HEIGHT(KL).EQ.0)THEN
              KGC=KG
              KGL=KL
           END IF
        END DO
     END IF
  END DO
  IF(KGC.EQ.0)THEN
     WRITE(*,*)'*ERROR* depsus: requires a deposition grid'
     WRITE(*,*)   '*ERROR* depsus: see message file for more information'
     STOP 900
  END IF

  print *,'sss depsus KGC=',KGC,KGL,cgrdxp(1,1,1),cgrdyp(1,1,1)

!FN-20150529
!! determine the concentration grid size
!  IF(ICHEM.EQ.4)THEN
!     NXP=GRID(KGM,KTM)%NX
!     NYP=GRID(KGM,KTM)%NY
!  ELSE
     NXP=CONC(KGC)%NUMB_LON
     NYP=CONC(KGC)%NUMB_LAT
!  END IF

  DO KT=1,NUMTYP
!    resuspension must be defined for this pollutant
     IF(DIRT(KT)%DOSUS)THEN

     DO JJ=1,NYP
     DO II=1,NXP

        !DEPAMT=CSUM(II,JJ,KGL,KT,KGC)
        IF(DEPAMT.GT.0.0)THEN

!FN-20150529
!           IF(ICHEM.EQ.4)THEN
!!             internal concentration grid matches meteorology grid
!              XP=FLOAT(II)
!              YP=FLOAT(JJ)
!           ELSE
!!             convert position to meteorological grid units
!              PLON=FLOAT(II-1)*CONC(KGC)%DELT_LON+CONC(KGC)%X1Y1_LON
!              PLAT=FLOAT(JJ-1)*CONC(KGC)%DELT_LAT+CONC(KGC)%X1Y1_LAT
!              IF(GRID(KGM,KTM)%LATLON)THEN
!                 CALL GBL2XY(KGM,KTM,PLAT,PLON,XP,YP)
!              ELSE
!                 CALL CLL2XY(GRID(KGM,KTM)%GBASE,PLAT,PLON, XP, YP)
!              END IF
!           END IF

            !FN-20150601
            XP=CGRDXP(II,JJ,KGC)
            YP=CGRDYP(II,JJ,KGC)

            CALL ADV2PT(HY_LU,XP,YP,lulc,mdims(1),mdims(2),mdims(3),mdims(4),mdims(5),mdims(6))

            CALL ADV2NT(HY_UF,XP,YP,ustr,mdims(1),mdims(2),mdims(3),mdims(4),mdims(5),mdims(6))

!FN-20150529
!!          no resuspension over water surfaces
!           IF(LU(NINT(XP),NINT(YP),KGM).NE.7)THEN 
!
!!          convert from gp/min to m/sec
!           UVEL=UF(NINT(XP),NINT(YP),POINT(2),KGM)*GX(NINT(XP),NINT(YP),KGM)/60.0
!           VVEL=VF(NINT(XP),NINT(YP),POINT(2),KGM)*GY(NINT(XP),NINT(YP),KGM)/60.0
!!
!!          pollutant flux - mass/m2-s
!           USTR=SQRT(UVEL*UVEL+VVEL*VVEL)                                
!           QRATE=DEPAMT*VONK*USTR*DIRT(KT)%SRATE

           IF (lulc .ne. 16.0) THEN

           QRATE=DEPAMT*VONK*USTR*DIRT(KT)%SRATE

!          determine mass lost from surface
           QTOT=MIN(DEPAMT, ABS(DT)*QRATE)    !FN-20150529, dt (sec)
           IF(QTOT.GT.1.0E-20)THEN

           CSUM(II,JJ,KGL,KT,KGC)=DEPAMT-QTOT
           MASS(KT,KPM)=QTOT*AREA

!          increment particle/puff counter
           KPM=KPM+1
           IF(KPM.GT.MAXPAR)THEN
              KPM=MAXPAR
              WRITE(*,*)'WARNING depsus: exceeding puff limit'
              WRITE(*,*)'WARNING depsus: exceeding puff limit'
              RETURN
           END IF

!           NSORT(KPM)=KPM  !FN-20150529
!          initial position always at ground
           XPOS(KPM)=XP
           YPOS(KPM)=YP
           ZPOS(KPM)=1.0

!          alongwind and vertical variances start at zero
           SIGU(KPM)=0.0
           SIGV(KPM)=0.0
           SIGW(KPM)=0.0

!          initial distribution (see main for definitions)
           HDWP(KPM)=INITD
!          initial age at zero
           PAGE(KPM)=0
!          pollutant type definition
           PTYP(KPM)=KT
!          initial grid is the default startup grid from main
           PGRD(KPM)=KGM

!          assume grid-cell area source (111000 m / deg - squared)
           AREA=1.2E+10*CONC(KGC)%DELT_LAT                              &
                       *CONC(KGC)%DELT_LON*COS(PLAT/DEGPRD)
!          compute sigma for uniform radius
           SIGH(KPM)=SQRT(AREA/PIE)/SIGR

           END IF ! mass test
           END IF ! over water test

        END IF

!    horizontal grid loop
     END DO
     END DO

! pollutant types
  END IF
  END DO

END SUBROUTINE depsus
