!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  EMSTMP           EMiSsion TeMPoral from a point
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   EMISSION INITIALIZATION STARTS A NEW PUFF OR PARTICLE IF THE
!   CURRENT MODEL TIME IS WITHIN THE TIME LIMITS SPECIFIED FOR START
!   A POLLUTANT RELEASE. EMISSION SPECIFICATIONS ARE DEFINED BY
!   AN INPUT FILE, EACH SOURCE CAN EMIT DIFFERENT POLLUTANTS EACH
!   AT DIFFERNT RATES STARTING AND ENDING AT UNIQUE TIMES.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ... 
!                    20 May 2015 (FN) - WRF-HYSPLIT inline coupling
!                    01 Jul 2015 (FN) - clean up
!
!$$$

SUBROUTINE EMSTMP(DIRT,SPRT,KG,NLOC,NUMTYP,NBPTYP,KPM,INITD,DT,JET,NUMPAR,  &
                  MAXPAR,NSORT,MASS,XPOS,YPOS,ZPOS,SIGH,SIGU,SIGV,SIGW,     &
                  HDWP,PAGE,PTYP,PGRD,job_id,num_job,kret,GDISX,GDISY)

  IMPLICIT NONE

  INCLUDE 'DEFSPRT.INC'     ! source emissions matrix 
  INCLUDE 'DEFCONC.INC'     ! pollutant and concentration grid

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  TYPE(pset),INTENT(IN)    :: dirt(:)   ! for each pollutant type 
  TYPE(qset),INTENT(INOUT) :: sprt(:,:) ! source location characteristics
  INTEGER,   INTENT(IN)    :: kg        ! active grid number
  INTEGER,   INTENT(IN)    :: nloc      ! total number of source locations
  INTEGER,   INTENT(IN)    :: numtyp    ! number of pollutant types
  INTEGER,   INTENT(IN)    :: nbptyp    ! number bins per type 
  INTEGER,   INTENT(INOUT) :: kpm       ! number of puffs or particles
  INTEGER,   INTENT(IN)    :: initd     ! initial distribution type
  REAL,      INTENT(IN)    :: dt        ! time step (sec) !FN-20150520
  INTEGER,   INTENT(IN)    :: numpar    ! maximum number of particles permitted
  INTEGER,   INTENT(IN)    :: maxpar    ! maximum particle number

  INTEGER,   INTENT(INOUT) :: nsort (:)   ! index of sorted elements
  REAL,      INTENT(INOUT) :: mass  (:,:) ! mass of pollutant (arbitrary units)
  REAL,      INTENT(INOUT) :: xpos  (:)   ! horizontal position (grid units)
  REAL,      INTENT(INOUT) :: ypos  (:)   ! horizontal position (grid units)
  REAL,      INTENT(INOUT) :: zpos  (:)   ! puff center height (sigma)
  REAL,      INTENT(INOUT) :: sigh  (:)   ! horizontal puff sigma 
  REAL,      INTENT(INOUT) :: sigu  (:)   ! turbulence u'2    
  REAL,      INTENT(INOUT) :: sigv  (:)   ! turbulence v'2
  REAL,      INTENT(INOUT) :: sigw  (:)   ! turbulence w'2 or vertical puff sigma
  INTEGER,   INTENT(INOUT) :: hdwp  (:)   ! Horizontal distribution pollutant
  INTEGER,   INTENT(INOUT) :: page  (:)   ! pollutant age since release (min)
  INTEGER,   INTENT(INOUT) :: ptyp  (:)   ! pollutant type index number
  INTEGER,   INTENT(INOUT) :: pgrd  (:)   ! meteorological grid of puff position
  INTEGER,   INTENT(IN)    :: job_id      ! mpi implementation 
  INTEGER,   INTENT(IN)    :: num_job     ! mpi implementation
  INTEGER,   INTENT(OUT)   :: kret        ! exceeding array return code  
  REAL,      INTENT(IN)    :: gdisx,gdisy ! meteorology grid spacing (m)

  INTEGER(KIND=8),   INTENT(IN)    :: jet     ! current elapsed time (sec) !FN-20150520

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  REAL, PARAMETER  :: sigr   = 1.54      ! top-hat radius
  REAL, PARAMETER  :: PI     = 3.14159265358979

  LOGICAL          :: ETEST
  REAL             :: zpl,delz,qsum,qhrs,qval,qstep,rvalue
  INTEGER          :: i,j,n,m,np,npar,ktp,kp,nphr,nrpts,maxdim,numpol,initk

  INTEGER(KIND=8)  :: jemst,jemen         ! starting, ending time (sec) for temporal emissions !FN-20150520

!-------------------------------------------------------------------------------
! Pollutants are emitted on different particles or all on the same particle.
! Currently this is the only non-chemistry associated emission routine that
! permits multiple species to be emitted on the same particle by increasing the
! size of the namelist variable MAXDIM=NUMTYP. Other combinations are not 
! permitted. The EMITIMES emission file requires NUMTYP entries for each
! emission location. Other emission characteristics are assigned from the first
! species in the list as defined in the CONTROL file.
!-------------------------------------------------------------------------------

  MAXDIM = SIZE(mass,1)  ! number of pollutants on single particle

  IF(MAXDIM.NE.1.AND.MAXDIM.NE.NUMTYP)THEN
     WRITE(*,*)'*ERROR*: emstmp - max mass dimension other than 1 or numtyp'
     WRITE(*,*)' maxdim = ',maxdim,'   numtyp = ',numtyp
     WRITE(*,*)'Abnormal termination (emstmp) - see MESSAGE file'
     STOP
  END IF
  KRET=0

  IF(MAXDIM.EQ.1)THEN
!    one pollutant per particle
     NUMPOL=NUMTYP
  ELSE
!    all pollutants on one particle, then position and other characteristics
!    of species #1 is used for all the other species on the same particle
     NUMPOL=1
  END IF

!-------------------------------------------------------------------------------
! check for line source configuration
!-------------------------------------------------------------------------------

! Multiple starting locations can either be point sources if each starting 
! point is in a different location or vertical line sources if two consecutive 
! starting points are in the same location. The line source mass is distributed
! between the given heights using the lower (index=n-1) level emission rate,
! which means that the top height emission value is always ignored.

  NRPTS=NLOC
  IF(NLOC.GT.1)THEN
     DO N=2,NLOC
     DO M=1,NUMTYP
        IF(INT(SPRT(N  ,M)%XP*10000.0).EQ.INT(SPRT(N-1,M)%XP*10000.0).AND.  &
           INT(SPRT(N  ,M)%YP*10000.0).EQ.INT(SPRT(N-1,M)%YP*10000.0).AND.  &
               SPRT(N  ,M)%START      .EQ.    SPRT(N-1,M)%START      .AND.  &
               SPRT(N  ,M)%STOP       .EQ.    SPRT(N-1,M)%STOP) THEN

!          when two positions are the same, flag the pair as a line
!          source by setting the heat value (at N) to a negative number
           SPRT(N,M)%HEAT=-1.0

!          for each line source pair, reduce the total number  
           IF(M.EQ.1)NRPTS=NRPTS-1
        END IF
     END DO
     END DO
  END IF

!-------------------------------------------------------------------------------
! set the number of units to emit
!-------------------------------------------------------------------------------

  INITK=INITD
  IF(INITK.GE.100) INITK=MOD(INITD/10,10)

! check if any emissions defined during this period
  QSUM=0.0
  QHRS=0.0
  DO N=1,NLOC
  DO M=1,NUMTYP
     !FN-20150520, min=>sec
     jemst=SPRT(N,M)%START*60
     jemen=SPRT(N,M)%STOP*60
!    maximum number of emission hours over all sources and pollutants  
     QSTEP=MIN(ABS((jemen-jemst)/3600.0),ABS(DT/3600.0))   !FN-20150520
     QSUM=MAX(QSUM,ABS((jemen-jemst)/3600.0),QSTEP)        !FN-20150520
     QHRS=MAX(QHRS,ABS((jemen-jemst)/3600.0))              !FN-20150520
  END DO
  END DO
  IF(QHRS.EQ.0.0)RETURN

  IF(INITK.EQ.1.OR.INITK.EQ.2)THEN
!    gaussian or top-hat emissions
     NPAR=1

  ELSE
!    particle emission rate: number particles per hour per source
     IF(NUMPAR.GE.0)THEN
        IF(MAXDIM.EQ.1)THEN
           NPHR=CEILING(FLOAT(NUMPAR/NRPTS/(NUMTYP*NBPTYP))/QSUM)
        ELSE
           NPHR=CEILING(FLOAT(NUMPAR/NRPTS)/QSUM)
        END IF
     ELSE
        NPHR=ABS(NUMPAR)
     END IF

!    particle emissions per time step (need at least one per job)
     NPAR=MAX(num_job,CEILING(NPHR*ABS(DT/3600.0)))         !FN-20150520
  END IF

!-------------------------------------------------------------------------------
! loop through the number of independent source locations
!-------------------------------------------------------------------------------

  NP=0 ! internal particle counter

  DO N=1,NLOC
     I=0  ! particle redistribution index

tloop : DO M=1,NUMPOL

     !FN-20150520, min=>sec
     jemst=SPRT(N,M)%START*60
     jemen=SPRT(N,M)%STOP*60

!    check for vertical line source ... always skip the first record of a pair
     IF(N.EQ.1.AND.NLOC.GT.1)THEN
        IF(SPRT(N+1,M)%HEAT.LT.0.0) CYCLE tloop
     END IF

     IF(MAXDIM.EQ.1)THEN
!       check if current pollutant requires emission 
        IF(DT.GT.0)THEN
!          forward/backward integration option (03 May 2007)
           ETEST=(JET.GE.jemst.AND.JET.LT.jemen).OR.   &
                 (JET.LT.jemst.AND.JET+INT(ABS(DT)).GT.jemst)        !FN-20150520
        ELSE
           ETEST=(JET.LE.jemst.AND.JET.GT.jemen).OR.   &
                 (JET.GT.jemst.AND.JET-INT(ABS(DT)).LT.jemst)        !FN-20150520
        END IF
     ELSE
!       check all pollutants, if any require emissions then continue    
        ETEST=.FALSE.
        DO KTP=1,NUMTYP
        IF(DT.GT.0)THEN
           IF((JET.GE.jemst.AND.JET.LT.jemen).OR.   &
              (JET.LT.jemst.AND.JET+INT(ABS(DT)).GT.jemst)) ETEST=.TRUE.     !FN-20150520
        ELSE
           IF((JET.LE.jemst.AND.JET.GT.jemen).OR.   &
              (JET.GT.jemst.AND.JET-INT(ABS(DT)).LT.jemst)) ETEST=.TRUE.     !FN-20150520
        END IF
        END DO
     END IF

     print *,'sss emstmp ETEST=',ETEST,NPAR

!    check if this pollutant requires a start
     IF(ETEST)THEN

!       particle redistribution loop
        DO J=1,NBPTYP
        I=I+1  ! index within dirt array

!       multiple emissions only for particles
        ploop : DO KP=1,NPAR

!          Use for simulations with multiple processors to only emit particles 
!          when process id even multiple of particle number. In a single process
!          environment num_job=1 and job_id=0.

           NP=NP+1

!FN-20150520
!           IF(MOD(NP,num_job).NE.job_id)CYCLE ploop

           KPM=KPM+1
           IF(KPM.GT.MAXPAR)THEN
              KPM=MAXPAR
              WRITE(*,*)'Warning: emstmp - exceeding puff limit'
              KRET=1
              RETURN    
           END IF
           NSORT(KPM)=KPM

!          initial position from main program
           XPOS(KPM)=SPRT(N,M)%XP
           YPOS(KPM)=SPRT(N,M)%YP
           ZPOS(KPM)=SPRT(N,M)%ZP     !FN-20150520

!          horizontal variances all start at zero
           SIGU(KPM)=0.0
           SIGV(KPM)=0.0
           SIGW(KPM)=0.0 

           IF(SPRT(N,M)%AREA.LE.0.0)THEN
!             points source defined for all species
              SIGH(KPM)=0.0
           ELSE
!             defined by source compute sigma for uniform radius
              SIGH(KPM)=SQRT(SPRT(N,M)%AREA/PI)/SIGR
!             for 3D particle adjust positions
              IF(INITK.EQ.0)THEN
                 CALL RANDOM_NUMBER(rvalue)
                 XPOS(KPM)=SPRT(N,M)%XP+SIGR*(RVALUE-0.5)*SIGH(KPM)/GDISX
                 CALL RANDOM_NUMBER(rvalue)
                 YPOS(KPM)=SPRT(N,M)%YP+SIGR*(RVALUE-0.5)*SIGH(KPM)/GDISY
              END IF       
           END IF

!          multiple locations get initial vertical distribution
!          when the first two starting points at the same x,y position
!          using negative heat as a flag 
           IF(SPRT(N,M)%HEAT.LT.0.0)THEN
              DELZ=ABS(SPRT(N,M)%QLVL-SPRT(N-1,M)%QLVL)
              IF(DELZ.GT.0.0)THEN
                 ZPL=MAX(SPRT(N,M)%QLVL,SPRT(N-1,M)%QLVL)
                 IF(INITK.EQ.1.OR.INITK.EQ.2)THEN
!                   puff variance set to layer depth
                    SIGW(KPM)=ABS(DELZ/SIGR/2.0)
                    ZPOS(KPM)=ZPL-DELZ/2.0
                 ELSE
!                   vertical variance set to zero
                    SIGW(KPM)=0.0
!                   particles get distributed in the layer
! original ->       ZPOS(KPM)=ZPL-FLOAT(KP)*DELZ/FLOAT(NPAR)
!                   particles get randomly distributed in the 1/npar layer
                    CALL RANDOM_NUMBER(rvalue)
                    RVALUE=(KP-RVALUE)/NPAR
                    ZPOS(KPM)=ZPL-RVALUE*DELZ
                 END IF
              END IF
           END IF

!          initial distribution (see main for definitions)
           HDWP(KPM)=INITD
!          initial age at zero
           PAGE(KPM)=0
!          pollutant type always defined from pollutant index
           IF(NBPTYP.GT.1)THEN
!             complex for representing sub-bin 
              PTYP(KPM)=(M*1000)+I
           ELSE
!             simple unredistributed
!             pollutant type=1 for multiple species on one particle  
              PTYP(KPM)=M
           END IF

!          initial grid is the default startup grid from main
           PGRD(KPM)=SPRT(N,M)%KG
!          number of time steps per hour
           QSTEP=MAX(1.0,ABS(3600.0/DT),1.0/QSUM)           !FN-20150520

           DO KTP=1,MAXDIM
              IF(MAXDIM.EQ.1)THEN
!                one species per particle emit per time step
                 QVAL=SPRT(N,M)%RATE/QSTEP
!                line source flag
                 IF(SPRT(N,M)%HEAT.LT.0.0) QVAL=SPRT(N-1,M)%RATE/QSTEP   
!                qrate represents a fraction when redistribution is selected
                 IF(NBPTYP.GT.1) QVAL=DIRT(I)%QRATE*QVAL
              ELSE
!                multiple species on one particle emit per time step
                 QVAL=SPRT(N,KTP)%RATE/QSTEP       !FN-20150520, mass units per sec
                 IF(SPRT(N,M)%HEAT.LT.0.0) QVAL=SPRT(N-1,KTP)%RATE/QSTEP   
              END IF

              IF(DT.GT.0)THEN
                 ETEST=(JET.GE.jemst.AND.JET.LT.jemen)      !FN-20150520
              ELSE
                 ETEST=(JET.LE.jemst.AND.JET.GT.jemen)      !FN-20150520
              END IF

              IF(ETEST)THEN
!                emit per particle
                 MASS(KTP,KPM)=QVAL/NPAR 
              ELSE
                 MASS(KTP,KPM)=0.0
              END IF
           END DO

!          special case where positive value indicates a plume rise calculation
!          sigw and page used as temporary variables which are then set to zero
!          in the main program after the particles are emitted before advection
           IF(SPRT(N,M)%HEAT.GT.0.0)THEN
              SIGW(KPM)=SPRT(N,M)%HEAT
              PAGE(KPM)=ABS(jemen-jemst)                     !FN-20150520
           END IF

!      particle loop
       END DO ploop

!      redistribution loop
       END DO

!  start time test
   END IF

!  pollutant type loop
   END DO tloop

!  number of sources loop
   END DO 

   print *,'sss emstmp KPM=',KPM

END SUBROUTINE emstmp
