!$$$  MAIN PROGRAM DOCUMENTATION BLOCK
!
! MAIN PROGRAM: HYCONC     MAIN HYSPLIT PROGRAM FOR AIR CONCENTRATIONS
!   PRGMMR: DRAXLER        ORG: R/ARL      DATE: 1998-08-26
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   IS THE MAIN PROGRAM FOR THE TRANSPORT AND DISPERSION MODEL HYSPLIT. GIVEN
!   A TIME, SOURCE LOCATION, AND RELEASE AMOUNT, THE MODEL COMPUTES AIR
!   CONCENTRATIONS OVER PRESPECIFIED SAMPLING PERIODS, ON PRESELECTED
!   LATITUDE-LONGITUDE GRIDS AT VARIOUS HEIGHTS ABOVE GROUND.  REQUIRED
!   METEOROLOGICAL DATA ARE PRESUMED TO HAVE ALREADY BEEN PREPARED FOR MODEL
!   INPUT. SEE ROUTINES AVN2ARL FOR A DISCUSSION OF METEO DATA PREPARATION.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 18 Sep 2012 (FN) - WRF-HYSPLIT coupling
!                 25 Nov 2013 (FN) - modifty for vertical coupling
!                 07 Jan 2014 (FN) - add new variable "pagl"
!                 28 Mar 2014 (FN) - clean up
!                 28 Apr 2014 (FN) - modify for tight couping
!                 15 May 2014 (FN) - change JET min => sec
!                 10 Jun 2014 (FN) - merge SETUP to WRF namelist.input
!                 15 Jul 2014 (FN) - read CONTROL from WRF IO level
!                 09 Sep 2014 (FN) - output particle positions and conc. in WRF format
!                 26 Sep 2014 (FN) - clean up
!                 10 Oct 2014 (FN) - running inline with WRF decomposed domains
!                 11 May 2015 (FN) - active DEPELM
!                 20 May 2015 (FN) - active EMSTMP with EFILE
!                 01 Jul 2015 (FN) - clean up
!
!$$$

SUBROUTINE HYCONC(ZSG,NLVL,HYCSUM,                  &
                  XPOS,YPOS,ZPOS,XPRT,YPRT,PAGL,    & 
                  ids,ide,jds,jde,                                              &
                  ims,ime,jms,jme,                                              &
                  ips,ipe,jps,jpe,proj,ksfc,runtj,                              & 
                  WRFDT,INITD,KPUFF,KHMAX,NUMPAR,MAXPAR,NBPTYP,QCYCLE,          &
                  KDEF,KZMIX,KBLS,KBLT,VSCALE,HSCALE,TVMIX,TKERD,TKERN,         &
                  KMIX0,KMIXD,NINIT,NDUMP,NCYCL,                                &
                  CONAGE,KMSL,ICHEM,CPACK,CMASS,KSPL,KRND,                      &
                  FRHMAX,SPLITF,FRHS,FRVS,FRTS,DXF,DYF,DZF,                     &
                  NUMLOC,NUMTYP,NUMGRD,NUMHGT,NUMPOL,NUMEMS,EFILE,              &
                  HYMACC,HYLAT,HYLON,HYLVL,                                     &
                  EMMACC,IDENT,QRATE,QHRS,   &
                  CNSTMA,CNENMA,CENTLAT,CENTLON,DELLAT,DELLON,SPLAT,SPLON, &
                  CNHGT,SNAP,CNDHR,CNDMN,           &
                  PDIAM,PDENS,SHAPE,DRYVL,GPMOL,ACVTY,DIFTY,HENRY,     &
                  WETGS,WETIN,WETLO,RHALF,SRATE,                       &
                  NMSST,NMSEN,EMLAT,EMLON,EMLVL,ERATE,EAREA,EHEAT      )                     

 USE funits
 USE module_llxy
 USE module_dm
 USE module_timing

 IMPLICIT NONE

!-------------------------------------------------------------------------------

  INTEGER :: job_id   = 0      ! root job number always zero
  INTEGER :: num_job  = 1      ! defaults to single processor

!-------------------------------------------------------------------------------
! variable definitions 
!-------------------------------------------------------------------------------

 INCLUDE 'mpif.h'
 INCLUDE 'DEFARG1.INC' ! main program subroutine interface
 INCLUDE 'DEFGRID.INC' ! meteorology grid and file
 INCLUDE 'DEFCONC.INC' ! pollutants and concentration grid
 INCLUDE 'DEFMETO.INC' ! meteo variables returned after advection
 INCLUDE 'DEFSPOT.INC' ! multiple source structure
 INCLUDE 'DEFSPRT.INC' ! source-pollutant definition matrix

!-------------------------------------------------------------------------------
! argument list
!-------------------------------------------------------------------------------

 REAL,      INTENT(IN)             :: wrfdt      ! WRF time step in second
 INTEGER,   INTENT(IN)             :: nlvl
 REAL, DIMENSION(nlvl), INTENT(IN) :: zsg        ! vertical levels

 INTEGER,   INTENT(IN)             :: initd,kpuff,khmax,numpar,maxpar,nbptyp
 INTEGER,   INTENT(IN)             :: kdef,kzmix,kbls,kblt
 INTEGER,   INTENT(IN)             :: kmix0,kmixd,ninit,ndump,ncycl
 INTEGER,   INTENT(IN)             :: conage,kmsl,ichem,cpack,cmass,kspl,krnd

 REAL,      INTENT(IN)             :: qcycle
 REAL,      INTENT(IN)             :: vscale,hscale,tvmix,tkerd,tkern
 REAL,      INTENT(IN)             :: frhmax,splitf,frhs,frvs,frts,dxf,dyf,dzf

 INTEGER,   INTENT(IN)             :: numloc,numtyp,numgrd,numhgt,numpol,numems
 INTEGER,   INTENT(IN)             :: hymacc

 CHARACTER*256,   INTENT(IN)       :: efile

 TYPE(proj_info), INTENT(IN)       :: proj

 INTEGER,   INTENT(IN)             :: ksfc
 INTEGER,   INTENT(IN)             :: runtj

 INTEGER,   INTENT(IN)                           :: ids , ide , jds , jde, &
                                                    ims , ime , jms , jme, &
                                                    ips , ipe , jps , jpe
 REAL, DIMENSION(ims:ime,jms:jme), INTENT(INOUT) :: hycsum

 REAL,    DIMENSION(maxpar), INTENT(INOUT) :: xpos,ypos,zpos  !FN-20140811
 REAL,    DIMENSION(maxpar), INTENT(INOUT) :: xprt,yprt,pagl  !FN-20140811

 REAL,    DIMENSION(NUMLOC), INTENT(IN)  :: hylat,hylon,hylvl
 REAL,    DIMENSION(NUMTYP), INTENT(IN)  :: qrate,qhrs
 REAL,    DIMENSION(NUMGRD), INTENT(IN)  :: centlat,centlon,dellat,dellon,splat,splon
 REAL,    DIMENSION(NUMHGT), INTENT(IN)  :: cnhgt
 INTEGER, DIMENSION(NUMTYP), INTENT(IN)  :: emmacc,ident
 INTEGER, DIMENSION(NUMGRD), INTENT(IN)  :: cnstma,cnenma
 INTEGER, DIMENSION(NUMGRD), INTENT(IN)  :: snap,cndhr,cndmn
 REAL,    DIMENSION(NUMPOL), INTENT(IN)  :: pdiam,pdens,shape
 REAL,    DIMENSION(NUMPOL), INTENT(IN)  :: dryvl,gpmol,acvty,difty,henry
 REAL,    DIMENSION(NUMPOL), INTENT(IN)  :: wetgs,wetin,wetlo,rhalf,srate
 INTEGER, DIMENSION(NUMEMS), INTENT(IN)  :: nmsst,nmsen
 REAL,    DIMENSION(NUMEMS), INTENT(IN)  :: emlat,emlon,emlvl,erate,earea,eheat

!-------------------------------------------------------------------------------
! variable allocations
!-------------------------------------------------------------------------------

! date information
  INTEGER              :: ibyr,ibmo,ibda,ibhr,ibmn

  INTEGER              :: nloc,mloc

  REAL,    ALLOCATABLE :: sigh (:)     !                Sigma_h (m) Sigma_h (m) 
  REAL,    ALLOCATABLE :: sigu (:)     !                u'2
  REAL,    ALLOCATABLE :: sigv (:)     !                v'2
  REAL,    ALLOCATABLE :: sigw (:)     !                w'2         Sigma_z (m)
  REAL,    ALLOCATABLE :: mass (:,:)   ! source mass
  REAL,    ALLOCATABLE :: msum (:)     ! mass used for summation
  REAL,    ALLOCATABLE :: pedg (:)     ! close to grid edge
  INTEGER, ALLOCATABLE :: page (:)     ! age
  INTEGER, ALLOCATABLE :: hdwp (:)     ! distribution
  INTEGER, ALLOCATABLE :: ptyp (:)     ! pollutant
  INTEGER, ALLOCATABLE :: pgrd (:)     ! meteo grid
  INTEGER, ALLOCATABLE :: nsort(:)     ! sort array

! gridded emissions variables

  REAL,         ALLOCATABLE :: qarea (:,:,:,:)          
  CHARACTER(4), ALLOCATABLE :: polid (:)  
  REAL                      :: qdlon    ! grid spacing
  REAL                      :: qdlat    ! grid spacing
  INTEGER                   :: nqlon    ! number of long in subgrid
  INTEGER                   :: nqlat    ! number of lats in subgrid
  INTEGER                   :: npval    ! number of pollutants in file
  INTEGER                   :: nqval    ! number of time periods per day

! concentration output array

  REAL, ALLOCATABLE :: csum     (:,:,:,:,:) ! conc (x,y,z,species,grids)

  REAL, ALLOCATABLE :: cgrdxp   (:,:,:) ! conc grid's corresponding xp (WRF grid) 
  REAL, ALLOCATABLE :: cgrdyp   (:,:,:) ! conc grid's corresponding yp (WRF grid)

  REAL, ALLOCATABLE :: dept (:)         ! deposition accumulation 
  REAL, ALLOCATABLE :: dryd (:)         ! deposition velocity 
  REAL, ALLOCATABLE :: zmass(:)         ! vertical mass diagnostic
  INTEGER           :: maxdim           ! pollutants per particle
  TYPE(cset), ALLOCATABLE :: conc(:)    ! for each concentration grid 
  TYPE(pset), ALLOCATABLE :: dirt(:)    ! for each pollutant type 


! meteorological data file and arrays

  TYPE(qset), ALLOCATABLE :: sprt (:,:) ! source-pollutant definition matrix
  TYPE(rset), ALLOCATABLE :: spot (:)   ! source location characteristics
  TYPE(bset), ALLOCATABLE :: metz (:)   ! profile advection variables
  TYPE(aset)              :: meto       ! surface advection variables

! simulation flags

  LOGICAL           :: back             ! integration direction
  LOGICAL           :: cdep             ! deposition process
  LOGICAL           :: rdep             ! resistance method
  LOGICAL           :: sdep             ! resuspension
  LOGICAL           :: ptest            ! particle.dat file
  LOGICAL           :: ftest1,ftest2    ! generic file test
  LOGICAL           :: qtemp = .false.  ! temporal emission file
  LOGICAL           :: qfile            ! gridded emission file
  LOGICAL           :: lagsam           ! lagrangian sampling file
  LOGICAL           :: spliton = .true. ! puff split status
  LOGICAL           :: ltraj            ! trajectory flag
  LOGICAL           :: tjset

! rounding and merging parameters
  REAL     :: frhe,frve,frte

  REAL     :: density
  REAL     :: plat0,plon0,plat1,plon1
  REAL     :: ubar,tmass,hydt,pmass,frac,conmix
  REAL     :: pcnt,rmass,height,fmass,sgb,zsfc,cgsize,sfcl,zdata 
  REAL     :: zx,zz,sgt,umax,tratio
  REAL     :: dist, xtmp, ytmp, ztmp
  REAL     :: cntr, avgrise, avgmixd

  INTEGER  :: p10f,frme,frmr
  INTEGER  :: rhb,rht,outdt,decay

  INTEGER  :: kret,kspk,kemit,initk
  INTEGER  :: hdwpx,numlag,kavrg,kmaxc,ksnap
  INTEGER  :: ktime,nstep,kz,ks,kt,kh,mm,maxdt,ifhr,ida,ihr,imo,numb
  INTEGER  :: iyr,imn,kp,kpt,i,j,k,k1,k2,iunit
  INTEGER  :: numspl,jtyp,ktyp
  INTEGER  :: n,ii,jj,kk,kg,kpm,kgrid

  INTEGER  :: mc0 

  INTEGER  :: nstr,mhrs,nver,kagl
  INTEGER  :: TOUT,TM_PRES,TM_TPOT,TM_TAMB,TM_RAIN,TM_MIXD,TM_RELH 
  INTEGER  :: TM_TERR,TM_DSWF,TM_SPHU,TM_MIXR 

  CHARACTER(80) :: ecode   ! memory allocation error message
  CHARACTER(80) :: pinpf   ! particle input file
  CHARACTER(80) :: poutf   ! particle output file

  INTEGER  :: nxp,nyp,nzp  !FN-20141101
  INTEGER  :: iip,jjp      !FN-20141010
  INTEGER  :: hycount = 0  !FN-0918
  REAL     :: eta      

  INTEGER(KIND=8) :: jet           ! jet (sec) !FN-20140515
  INTEGER(KIND=8) :: jemst,jemen   ! starting, ending time (sec) for temporal emissions !FN-20150520

  INTEGER  :: mpi_comm_here, crrtask, ierr  !FN-20141101
  INTEGER  :: mpi_inited
  INTEGER  :: seed_id

  INTEGER  :: ksb,ksd,mtime

  LOGICAL , EXTERNAL              :: wrf_dm_on_monitor

!-------------------------------------------------------------------------------
! external definitions
!-------------------------------------------------------------------------------

! meteorolgical grid, record, and file information
  COMMON /GBLGRD/ HYGD, DREC, HYFL

!FN-0924
  SAVE

!-------------------------------------------------------------------------------

 HYCOUNT=HYCOUNT+1
 print *,'www hysp/hyconc hycount=',hycount,wrfdt

  mpi_comm_here = MPI_COMM_WORLD
  CALL MPI_Comm_rank ( mpi_comm_here, crrtask, ierr )
  print *,'www mpi=',mpi_comm_here,crrtask,ierr

IF (HYCOUNT.EQ.1) THEN  ! initialization at 1st call of hymain (WRF start clock)

  seed_id=crrtask

  print *,'www starting HYSPLIT initialization!',seed_id,runtj

  print *,'www proj=',proj%truelat1,proj%truelat2,proj%stdlon,proj%dx

!-------------------------------------------------------------------------------
! concentration model default namelist parameters read from file: SETUP.CFG  
!-------------------------------------------------------------------------------

  MAXDIM=1     ! maximum number of pollutants to carry on one mass particle
  P10F=1.0     ! dust threshold velocity sensitivity factor

  FRMR=0.0     ! mass removal fraction during enhanced merging
  FRME=0.10    ! mass rounding fraction for enhanced merging

  RHB=80       ! cloud bottom RH used for wet deposition calculation
  RHT=60       ! cloud top RH 

  OUTDT=0      ! stilt emulation particle.dat output interval (min)

  DECAY=1      ! 0 = deposited material does NOT decay (only mass on particles)    
               ! 1 = deposited material on concentration grid decays 

  TRATIO=0.75  ! advection stability ratio

  PINPF='PARINIT'   ! particle dump input file
  POUTF='PARDUMP'   ! particle dump output file


  FRHE=1.5*FRHS
  FRVE=1.5*FRVS
  FRTE=1.5*FRTS

  if (runtj .eq. 1) then 
     ltraj=.true.
     if (ltraj) tjset=.true.
  endif

!-------------------------------------------------------------------------------
! allocate particle arrays
!-------------------------------------------------------------------------------

  ALLOCATE (sigh(maxpar),sigu(maxpar),sigv(maxpar),sigw(maxpar),STAT=kret)
  IF(kret.ne.0)THEN
     ECODE='Particle sigma'
     WRITE(*,*)'ERROR hyconc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  ALLOCATE (mass(maxdim,maxpar), msum(maxdim), STAT=kret)
  IF(kret.ne.0)THEN
     ECODE='Particle mass'
     WRITE(*,*)'ERROR hyconc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  ALLOCATE (page(maxpar),hdwp(maxpar),ptyp(maxpar),STAT=kret)
  IF(kret.ne.0)THEN
     ECODE='Particle characteristics'
     WRITE(*,*)'ERROR hyconc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  ALLOCATE (pgrd(maxpar),pedg(maxpar),nsort(maxpar),STAT=kret)
  IF(kret.ne.0)THEN
     ECODE='Particle grid and sort'
     WRITE(*,*)'ERROR hyconc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

!-------------------------------------------------------------------------------
! standard meteorological model initialization
!-------------------------------------------------------------------------------

  NLOC=NUMLOC
  CALL TM2DAY(HYMACC,IBYR,IBMO,IBDA,IBHR,IBMN)

  ALLOCATE (spot(nloc), STAT=kret)    ! defined for each source location
  IF(kret.ne.0)THEN     
     ECODE='Source location array'
     WRITE(*,*)'ERROR hyconc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  SPOT(1)%IBYR=IBYR
  SPOT(1)%IBMO=IBMO
  SPOT(1)%IBDA=IBDA
  SPOT(1)%IBHR=IBHR
  SPOT(1)%IBMN=IBMN

  print *,'www ibyr=',SPOT(1)%IBYR,SPOT(1)%IBMO,SPOT(1)%IBDA,SPOT(1)%IBHR,SPOT(1)%IBMN

  BACK=.FALSE.
  DO N=1,NLOC
     SPOT(N)%AREA=0.0
     SPOT(N)%QTRM=0.0
     SPOT(N)%OLAT=HYLAT(N)
     SPOT(N)%OLON=HYLON(N)
     SPOT(N)%OLVL=HYLVL(N)
  END DO

  print *,'www SPOT%=',SPOT(1)%OLAT,SPOT(1)%OLON,SPOT(1)%OLVL
  print *,'www nhrs=',ZDATA,BACK

!-------------------------------------------------------------------------------
! variable allocation
!-------------------------------------------------------------------------------

  ALLOCATE (metz(nlvl),STAT=kret)  
  IF(kret.ne.0)THEN      
     ECODE='Meteo return profile'
     WRITE(*,*)'ERROR hyconc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  ALLOCATE (zmass(nlvl),STAT=kret) 
  IF(kret.ne.0)THEN     
     ECODE='Level diagnostic information'
     WRITE(*,*)'ERROR hyconc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

!-------------------------------------------------------------------------------
! configure the starting location, grid, and time
!-------------------------------------------------------------------------------

  MC0=HYMACC
 
! elapsed time
  JET=MC0*60

  print *,'www starting JET=',JET,MC0

! same time applies to all source locations
  IF(NLOC.GT.1)THEN
     DO N=2,NLOC
        spot(n)%ibyr=spot(1)%ibyr
        spot(n)%ibmo=spot(1)%ibmo
        spot(n)%ibda=spot(1)%ibda
        spot(n)%ibhr=spot(1)%ibhr 
        spot(n)%ibmn=spot(1)%ibmn 
     END DO
  END IF

! set initial meteorological grid (time & grid index)
  KT=1
  KG=1

! default initial meteo grid (km) for time-step calculation
! the first grid defined always has the finest resolution
  CGSIZE=HYGD%SIZE

  print *,'www cgsize=',CGSIZE,MC0

!FN-20141010
! set initial meteorological grid index for each starting location
  DO N=1,NLOC
         
     call latlon_to_ij(proj,SPOT(N)%OLAT,SPOT(N)%OLON,SPOT(N)%XP,SPOT(N)%YP)

     iip=nint(SPOT(N)%XP)
     jjp=nint(SPOT(N)%YP)

     if ((iip .ge. ips .and. iip .le. ipe .and. iip .lt. ide-2) .and. &
         (jjp .ge. jps .and. jjp .le. jpe .and. jjp .lt. jde-2)) then
        call ht2eta(SPOT(N)%XP,SPOT(N)%YP,SPOT(N)%OLVL,eta)
        SPOT(N)%ZP=eta
     else 
        SPOT(N)%ZP=-9999.0
     end if

     !FN-20141010, broadcast info to all sub-domains 
     spot(n)%zp = wrf_dm_max_real(spot(n)%zp)

     print *,'www spot=',SPOT(N)%OLAT,SPOT(N)%OLON,SPOT(N)%OLVL
     print *,'www spot=',SPOT(N)%XP,SPOT(N)%YP,SPOT(N)%ZP

  END DO

  if (ltraj) return   !FN-20150610

!-------------------------------------------------------------------------------
! emission (source term) and pollutant initializaton
!-------------------------------------------------------------------------------

!FN-20140715, delete reading NUMTYP

! allocate additional pollutant array space if redistribution required
  IF(NBPTYP.GT.1)THEN
!    save room for the original distribution
     ALLOCATE (dirt(numtyp+nbptyp*numtyp),STAT=kret)
     WRITE(*,*)' NOTICE   main: particle size redistribution to: ',(NBPTYP*NUMTYP)
  ELSE
     ALLOCATE (dirt(numpol),STAT=kret)
  END IF

  IF(kret.ne.0)THEN
     ECODE='Pollutant description'
     WRITE(*,*)'ERROR hymodelc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  ALLOCATE (dept(numpol),dryd(numpol),STAT=kret)
  IF(kret.ne.0)THEN
     ECODE='Deposition description'
     WRITE(*,*)'ERROR hymodelc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  !FN-20140715, delete EMSSET
  DO N=1,NUMTYP
     WRITE(DIRT(N)%IDENT,'(a3,i1)') 'EXP',IDENT(N)
     DIRT(N)%QRATE=QRATE(N)
     DIRT(N)%QHRS=QHRS(N)
     CALL TM2DAY(EMMACC(N),DIRT(N)%START%YR,DIRT(N)%START%MO,DIRT(N)%START%DA,&
                           DIRT(N)%START%HR,DIRT(N)%START%MN)
     DIRT(N)%START%MACC=EMMACC(N)
     DIRT(N)%START%SACC=DIRT(N)%START%MACC*60
  END DO
  print *,'www ident=',ident,qrate,qhrs

  print *,'www dirt%=',DIRT(1)%IDENT,DIRT(1)%QRATE,DIRT(1)%QHRS
  print *,'www dirt%start=',DIRT(1)%START%YR,DIRT(1)%START%MO,DIRT(1)%START%DA,DIRT(1)%START%MN
  print *,'www dirt%macc=',DIRT(1)%START%MACC,DIRT(1)%START%SACC

  print *,'www efile=',trim(efile)  !FN-20150520, being read by hyrems.F

!FN-20150520
! input file for point source emissions
  IF(TRIM(EFILE).NE."NONE_SPECIFIED")THEN
     MLOC=SIZE(SPOT,1) ! use the same array size as control file 05/17/2013
     ALLOCATE (sprt(mloc,numpol),STAT=kret)
     IF(kret.ne.0)THEN
        ECODE='Source-Pollutant Matrix (from EFILE)'
        WRITE(*,*)'ERROR hymodelc: memory allocation - ',ECODE,KRET
        STOP 900
     END IF

     print *,'www mloc=',mloc,numpol

!    first guess default fill in from CONTROL file information !FN-20150520, check
     DO II=1,NUMPOL
     DO N=1,NLOC
        SPRT(N,II)%START=DIRT(II)%START%MACC
        SPRT(N,II)%STOP =DIRT(II)%START%MACC+DIRT(II)%QHRS*60
        SPRT(N,II)%KG   =SPOT(N)%KG
        SPRT(N,II)%XP   =SPOT(N)%XP
        SPRT(N,II)%YP   =SPOT(N)%YP
        SPRT(N,II)%QLVL =SPOT(N)%OLVL
        IF(SPOT(N)%QTRM.GT.0.0)THEN
           SPRT(N,II)%RATE=SPOT(N)%QTRM   ! from source record
        ELSE
           SPRT(N,II)%RATE=DIRT(II)%QRATE  ! from pollutant record
        END IF
        SPRT(N,II)%AREA =SPOT(N)%AREA
        SPRT(N,II)%HEAT =0.0
     END DO
     END DO

     print *,'www sprt%start=',SPRT(1,1)%START,SPRT(1,1)%STOP
     print *,'www sprt%xp=',SPRT(1,1)%XP,SPRT(1,1)%YP,SPRT(1,1)%QLVL

!    when doing particle redistribution, set emission values to one
!    which will result in emsize computing bin fractions
     IF(NBPTYP.GT.1) THEN
        SPRT(:,:)%RATE=1.0
        DIRT(:)%QRATE=1.0
     END IF
  END IF

!-------------------------------------------------------------------------------
! sampling (concentration) grid initialization (update CGSIZE if required)
!-------------------------------------------------------------------------------

  ALLOCATE (conc(numgrd),STAT=kret)
  IF(kret.ne.0)THEN
     ECODE='Concentration grid structure'
     WRITE(*,*)'ERROR hymodelc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  !FN-20140715
  CALL CONSET(CONC,NUMGRD,CGSIZE,CPACK,SPOT(1)%OLAT,SPOT(1)%OLON, &
              CENTLAT,CENTLON,DELLAT,DELLON,SPLAT,SPLON)

  DO N=1,NUMGRD

     CONC(N)%HEIGHT=50.

   ! vertical levels for concentration grid
     CONC(N)%LEVELS=NUMHGT
     DO KK=1,NUMHGT
        CONC(N)%HEIGHT(KK)=CNHGT(KK)
     END DO

     CONC(N)%DIR='./'
     CONC(N)%FILE='cdump'
     print *,'www conc%file=',CONC(N)%DIR,CONC(N)%FILE

     CALL TM2DAY(CNSTMA(N),CONC(N)%START%YR,CONC(N)%START%MO,CONC(N)%START%DA,&
                           CONC(N)%START%HR,CONC(N)%START%MN)
     CONC(N)%START%MACC=CNSTMA(N)
     CONC(N)%START%SACC=CONC(N)%START%MACC*60
     print *,'www conc%start=',CONC(N)%START%YR,CONC(N)%START%MO,CONC(N)%START%DA,CONC(N)%START%MN
     print *,'www conc%start=',CONC(N)%START%MACC,CONC(N)%START%SACC

     CALL TM2DAY(CNENMA(N),CONC(N)%STOP%YR,CONC(N)%STOP%MO,CONC(N)%STOP%DA,&
                           CONC(N)%STOP%HR,CONC(N)%STOP%MN)
     CONC(N)%STOP%MACC=CNENMA(N)
     CONC(N)%STOP%SACC=CONC(N)%STOP%MACC*60
     print *,'www conc%stop=',CONC(N)%STOP%YR,CONC(N)%STOP%MO,CONC(N)%STOP%DA,CONC(N)%STOP%MN
     print *,'www conc%stop=',CONC(N)%STOP%MACC,CONC(N)%STOP%SACC

     CONC(N)%SNAP=SNAP(N)
     CONC(N)%DELTA%HR=CNDHR(N)
     CONC(N)%DELTA%MN=CNDMN(N)
     print *,'www conc%snap= ',CONC(1)%SNAP,CONC(1)%DELTA%HR,CONC(1)%DELTA%MN

   ! sampling intervals in years, months, days not used
     CONC(N)%DELTA%YR=0
     CONC(N)%DELTA%MO=0
     CONC(N)%DELTA%DA=0

   ! convert sampling interval hours-minutes to minutes
     IF(CONC(N)%DELTA%HR.GE.1.AND.CONC(N)%DELTA%MN.NE.0)THEN
        WRITE(*,*)'WARNING conset: averaging >1 hr requires zero minutes field'
        WRITE(*,*)'Resetting averaging minutes to zero on grid #',N
        CONC(N)%DELTA%MN=0
     END IF
     CONC(N)%DELTA%MACC=60*CONC(N)%DELTA%HR+CONC(N)%DELTA%MN
     CONC(N)%DELTA%SACC=CONC(N)%DELTA%MACC*60

   ! save sampling start time in conc output file marker variable
     CONC(N)%NOW=CONC(N)%START

  END DO

  print *,'www numgrd=',numgrd,cgsize

  print *,'www conc%delt_lat=',CONC(1)%DELT_LAT,CONC(1)%DELT_LON,CONC(1)%NUMB_LAT,CONC(1)%NUMB_LON
  print *,'www conc%height=',CONC(1)%LEVELS,CONC(1)%HEIGHT(:)

! set the maximum and snapshot grid numbers
! definition of a maximum concentration grid requires a snapshot grid defined
  KMAXC=0
  KSNAP=0
  KAVRG=0
  DO K=1,NUMGRD
     IF(CONC(K)%SNAP.EQ.0) KAVRG=K
     IF(CONC(K)%SNAP.EQ.1) KSNAP=K
     IF(CONC(K)%SNAP.EQ.2) KMAXC=K
  END DO

 !FN-20141101
 ! set loop indicies and allocate array space
   NXP=MAXVAL(CONC(:)%NUMB_LON)
   NYP=MAXVAL(CONC(:)%NUMB_LAT)
   NZP=MAXVAL(CONC(:)%LEVELS)

  ECODE='STD Concentration array'
  ALLOCATE (csum (nxp,nyp,nzp,numpol,numgrd),STAT=kret)
  IF(kret.ne.0)THEN
     WRITE(*,*)'ERROR hymodelc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF
  csum(:,:,:,:,:) = 0.0

  print *,'www nxp=',nxp,nyp,nzp
  print *,'www csum=',size(csum,1),size(csum,2),size(csum,3)

!FN-20150601
  ALLOCATE (cgrdxp(nxp,nyp,numgrd),cgrdyp(nxp,nyp,numgrd),STAT=kret)
  IF(kret.ne.0)THEN
     WRITE(*,*)'ERROR hymodelc: memory allocation - ',ECODE,KRET
     STOP 900
  END IF

  DO N=1,NUMGRD 
     DO I=1,NXP
     DO J=1,NYP
        PLON0=FLOAT(I-1)*CONC(N)%DELT_LON+CONC(N)%X1Y1_LON
        PLAT0=FLOAT(J-1)*CONC(N)%DELT_LAT+CONC(N)%X1Y1_LAT

        call latlon_to_ij(proj,PLAT0,PLON0,cgrdxp(i,j,n),cgrdyp(i,j,n))
     END DO
     END DO
  END DO

  print *,'www cgrdxp=',size(cgrdxp,1),size(cgrdxp,2),size(cgrdxp,3),cgrdxp(1,1,1)
  print *,'www cgrdyp=',size(cgrdyp,1),size(cgrdyp,2),size(cgrdyp,3),cgrdyp(1,1,1)

!FN-20140715
! set the deposition parameters for each pollutant type
  CALL DEPSET(DIRT,NUMPOL,CDEP,RDEP,SDEP,PDIAM,PDENS,SHAPE, &
              DRYVL,GPMOL,ACVTY,DIFTY,HENRY,WETGS,WETIN,WETLO,RHALF,SRATE)

! confirm meteorological data set contains precip when computing wet removal
  DO KTYP=1,NUMTYP
     IF(DIRT(KTYP)%DOWET.AND.DREC%ACYCLE.EQ.0.0)THEN
        WRITE(*,*)'ERROR: pollutant wet removal defined but ...'
        WRITE(*,*)'meteorological data does not contain the precipitation field'
        WRITE(*,*)'Turn off wet removal for ',DIRT(KTYP)%IDENT,' and restart'
        STOP 900
     END IF
  END DO

  print *,'www cdep=',CDEP,RDEP,SDEP
  print *,'www dirt%=',DIRT(1)%DOGAS,DIRT(1)%DOWET,DIRT(1)%DORAD,DIRT(1)%DOSUS
  print *,'www dirt%=',DIRT(1)%DORES,DIRT(1)%DODRY,DIRT(1)%DOVOL,DIRT(1)%DOGRV

!FN-20140321, NBPTYP=1, particle size redistribution (EMSIZE)
!!!skip!!!

!FN-20140512, ICHEM=0, matrix modification
!!!skip!!!

!FN-20141101, opening cdump file
if (crrtask .eq. 0) CALL CONINI(SPOT,CONC,DIRT,NLOC,NUMGRD,NUMTYP,NBPTYP,CPACK)

!-------------------------------------------------------------------------------
! concentration grid sensitive switches
!-------------------------------------------------------------------------------

!FN-20140512, INITD=INITK=0, special limit for splitting puffs
!!!skip!!!

!-------------------------------------------------------------------------------
! diagnostic file initialization
!-------------------------------------------------------------------------------

!FN-20140515, delete opening CONC.CFG, print out to WRF's log file

  WRITE(*,'(A)')' &SETUP'
  WRITE(*,'(A,F4.2,A)')' tratio = ',tratio,','

  IF(initd.LT.10)THEN
     WRITE(*,'(A,I1,A)')' initd = ',initd,','
  ELSE
     WRITE(*,'(A,I3,A)')' initd = ',initd,','
  END IF

  WRITE(*,'(A,I1,A)')' kpuff = ',kpuff,','

  IF(khmax.LT.10)THEN
     WRITE(*,'(A,I1,A)')' khmax = ',khmax,','
  ELSEIF(khmax.LT.100)THEN
     WRITE(*,'(A,I2,A)')' khmax = ',khmax,','
  ELSEIF(khmax.LT.1000)THEN
     WRITE(*,'(A,I3,A)')' khmax = ',khmax,','
  ELSE
     WRITE(*,'(A,I4,A)')' khmax = ',khmax,','
  END IF

  IF(numpar.LT.0)THEN
     WRITE(*,'(A,I5,A)')' numpar = ',numpar,','
  ELSEIF(numpar.LT.10)THEN
     WRITE(*,'(A,I1,A)')' numpar = ',numpar,','
  ELSEIF(numpar.LT.100)THEN
     WRITE(*,'(A,I2,A)')' numpar = ',numpar,','
  ELSEIF(numpar.LT.1000)THEN
     WRITE(*,'(A,I3,A)')' numpar = ',numpar,','
  ELSEIF(numpar.LT.10000)THEN
     WRITE(*,'(A,I4,A)')' numpar = ',numpar,','
  ELSEIF(numpar.LT.100000)THEN
     WRITE(*,'(A,I5,A)')' numpar = ',numpar,','
  ELSEIF(numpar.LT.1000000)THEN
     WRITE(*,'(A,I6,A)')' numpar = ',numpar,','
  ELSE
     WRITE(*,'(A,I7,A)')' numpar = ',numpar,','
  END IF

  IF(maxpar.LT.10)THEN
     WRITE(*,'(A,I1,A)')' maxpar = ',maxpar,','
  ELSEIF(maxpar.LT.100)THEN
     WRITE(*,'(A,I2,A)')' maxpar = ',maxpar,','
  ELSEIF(maxpar.LT.1000)THEN
     WRITE(*,'(A,I3,A)')' maxpar = ',maxpar,','
  ELSEIF(maxpar.LT.10000)THEN
     WRITE(*,'(A,I4,A)')' maxpar = ',maxpar,','
  ELSEIF(maxpar.LT.100000)THEN
     WRITE(*,'(A,I5,A)')' maxpar = ',maxpar,','
  ELSEIF(maxpar.LT.1000000)THEN
     WRITE(*,'(A,I6,A)')' maxpar = ',maxpar,','
  ELSE
     WRITE(*,'(A,I7,A)')' maxpar = ',maxpar,','
  END IF

  IF(nbptyp.LT.10)THEN
     WRITE(*,'(A,I1,A)')' nbptyp = ',nbptyp,','
  ELSEIF(nbptyp.LT.100)THEN
     WRITE(*,'(A,I2,A)')' nbptyp = ',nbptyp,','
  ELSE
     WRITE(*,'(A,I3,A)')' nbptyp = ',nbptyp,','
  END IF

  IF(qcycle.LT.10.0)THEN
     WRITE(*,'(A,F3.1,A)')' qcycle = ',qcycle,','
  ELSEIF(qcycle.LT.100.0)THEN
     WRITE(*,'(A,F4.1,A)')' qcycle = ',qcycle,','
  ELSE
     WRITE(*,'(A,F5.1,A)')' qcycle = ',qcycle,','
  END IF

  K=INDEX(EFILE,' ')-1
  WRITE(*,'(3A)')' efile = ',''''//efile(:k),''','

  WRITE(*,'(A,I1,A)')' kdef = ',kdef,','
  WRITE(*,'(A,I1,A)')' kzmix = ',kzmix,','
  WRITE(*,'(A,I1,A)')' kbls = ',kbls,','
  WRITE(*,'(A,I1,A)')' kblt = ',kblt,','

  WRITE(*,'(A,F5.1,A)')' vscale = ',vscale,','
  WRITE(*,'(A,F7.1,A)')' hscale = ',hscale,','

  WRITE(*,'(A,F4.2,A)')' tvmix = ',tvmix,','
  WRITE(*,'(A,F4.2,A)')' tkerd = ',tkerd,','
  WRITE(*,'(A,F4.2,A)')' tkern = ',tkern,','

  IF(kmix0.LT.10)THEN
     WRITE(*,'(A,I1,A)')' kmix0 = ',kmix0,','
  ELSEIF(kmix0.LT.100)THEN
     WRITE(*,'(A,I2,A)')' kmix0 = ',kmix0,','
  ELSE
     WRITE(*,'(A,I3,A)')' kmix0 = ',kmix0,','
  END IF

  IF(kmixd.LT.10)THEN
     WRITE(*,'(A,I1,A)')' kmixd = ',kmixd,','
  ELSEIF(kmixd.LT.100)THEN
     WRITE(*,'(A,I2,A)')' kmixd = ',kmixd,','
  ELSEIF(kmixd.LT.1000)THEN
     WRITE(*,'(A,I3,A)')' kmixd = ',kmixd,','
  ELSE
     WRITE(*,'(A,I4,A)')' kmixd = ',kmixd,','
  END IF

  WRITE(*,'(A,I1,A)')' ninit = ',ninit,','
  IF(ndump.LT.10)THEN
     WRITE(*,'(A,I1,A)')' ndump = ',ndump,','
  ELSEIF(ndump.LT.100)THEN
     WRITE(*,'(A,I2,A)')' ndump = ',ndump,','
  ELSEIF(ndump.LT.1000)THEN
     WRITE(*,'(A,I3,A)')' ndump = ',ndump,','
  ELSE
     WRITE(*,'(A,I4,A)')' ndump = ',ndump,','
  END IF

  IF(ncycl.LT.10)THEN
     WRITE(*,'(A,I1,A)')' ncycl = ',ncycl,','
  ELSEIF(ncycl.LT.100)THEN
     WRITE(*,'(A,I2,A)')' ncycl = ',ncycl,','
  ELSEIF(ncycl.LT.1000)THEN
     WRITE(*,'(A,I3,A)')' ncycl = ',ncycl,','
  ELSE
     WRITE(*,'(A,I4,A)')' ncycl = ',ncycl,','
  END IF

  K=INDEX(PINPF,' ')-1
  WRITE(*,'(3A      )')' pinpf = ',''''//pinpf(:k),''','
  K=INDEX(POUTF,' ')-1
  WRITE(*,'(3A)')' poutf = ',''''//poutf(:k),''','

  IF(conage.LT.10)THEN
     WRITE(*,'(A,I1,A)')' conage = ',conage,','
  ELSEIF(conage.LT.100)THEN
     WRITE(*,'(A,I2,A)')' conage = ',conage,','
  ELSE
     WRITE(*,'(A,I3,A)')' conage = ',conage,','
  END IF

  WRITE(*,'(A,I1,A)')' kmsl = ',kmsl,','
  WRITE(*,'(A,I1,A)')' ichem = ',ichem,','
  WRITE(*,'(A,I1,A)')' cpack = ',cpack,','
  WRITE(*,'(A,I1,A)')' cmass = ',cmass,','
  WRITE(*,'(A,I1,A)')' kspl = ',kspl,','
  WRITE(*,'(A,I1,A)')' krnd = ',krnd,','
  WRITE(*,'(A,F4.2,A)')' frhmax = ',frhmax,','
  WRITE(*,'(A,F4.2,A)')' splitf = ',splitf,','
  WRITE(*,'(A,F4.2,A)')' frhs = ',frhs,','
  WRITE(*,'(A,F4.2,A)')' frvs = ',frvs,','
  WRITE(*,'(A,F4.2,A)')' frts = ',frts,','
  WRITE(*,'(A,F4.2,A)')' dxf = ',dxf,','
  WRITE(*,'(A,F4.2,A)')' dyf = ',dyf,','
  WRITE(*,'(A,F4.2,A)')' dzf = ',dzf,','
  WRITE(*,'(A)')' /'

!FN-20140715, delete closing CONTROL and SETUP

!FN-20140610, delete writing out configuration

! diagnostic message on status of deposition flags
  WRITE(*,*)' NOTICE   main: pollutant initialization flags'
  IF(DIRT(1)%DOGAS)WRITE(*,*)'      Gas pollutant - ',DIRT(1)%DOGAS
  IF(DIRT(1)%DODRY)WRITE(*,*)'     Dry deposition - ',DIRT(1)%DODRY
  IF(DIRT(1)%DOWET)WRITE(*,*)'     Wet deposition - ',DIRT(1)%DOWET
  IF(DIRT(1)%DORES)WRITE(*,*)'        Dynamic dry - ',DIRT(1)%DORES
  IF(DIRT(1)%DOGRV)WRITE(*,*)'      Grav settling - ',DIRT(1)%DOGRV
  IF(DIRT(1)%DOSUS)WRITE(*,*)'       Resuspension - ',DIRT(1)%DOSUS
  IF(DIRT(1)%DORAD)WRITE(*,*)'        Radioactive - ',DIRT(1)%DORAD
  IF(DIRT(1)%DOVOL)WRITE(*,*)'Volume Unit Convert - ',DIRT(1)%DOVOL

  WRITE(*,*)' NOTICE   main: meteorological data flags'
  WRITE(*,*)'    Ten meter wind - ',DREC%UFLG
  WRITE(*,*)'    Two meter temp - ',DREC%TFLG
  WRITE(*,*)'    Exchange coeff - ',DREC%EFLX
  WRITE(*,*)'         Heat flux - ',DREC%HFLX
  WRITE(*,*)'     Momentum flux - ',DREC%UFLX
  WRITE(*,*)' Velocity variance - ',DREC%VELV
  WRITE(*,*)'      3D TKE field - ',DREC%TKEN
  WRITE(*,*)'    Down shortwave - ',DREC%DSWF
  WRITE(*,*)' Friction velocity - ',DREC%USTR
  WRITE(*,*)'     Friction temp - ',DREC%TSTR
  WRITE(*,*)'    Terrain height - ',DREC%SHGT
  WRITE(*,*)'  Surface pressure - ',DREC%PRSS
  WRITE(*,*)' Mixed layer depth - ',DREC%MIXD
  WRITE(*,*)' Profile averaging - ',DREC%KZMIX
  WRITE(*,*)'  Stability method - ',DREC%KBLS
  WRITE(*,*)' Horizontal mixing - ',DREC%KDEF
  WRITE(*,*)' PBL mixing scheme - ',DREC%KBLT
  WRITE(*,*)'  Free Trop Mixing - ',DREC%TVMIX
  WRITE(*,*)' Precip accumulate - ',DREC%ACYCLE
  WRITE(*,*)'------------- Start computation messages -------------------'

!-------------------------------------------------------------------------------
! Initialize gridded emissions array, which enables gridded emissions from
! an ascii file (emission.txt) used in conjunction with call emsgrd.
!-------------------------------------------------------------------------------

  CALL EMSINI(NLOC,SPOT(:)%OLAT,SPOT(:)%OLON,QFILE,NQLON,NQLAT,NPVAL)

  print *,'www QFILE=',QFILE,NQLON,NQLAT,NPVAL

!FN-20140421, QFILE=F
!!!skip!!!

!-------------------------------------------------------------------------------
! optional initialization from previous simulation dump file
!-------------------------------------------------------------------------------

! initial value of particle/puff counter
  KPM=0

  print *,'www FTEST1=',FTEST1

!FN-20140328, delete IF(FTEST1)THEN 
!!!skip!!!

!-------------------------------------------------------------------------------
! starting sigma height initialization
!-------------------------------------------------------------------------------

  print *,'www KPM=',kpm,SPOT(1)%KG,KT

!FN-20141010
  HYGD%LX1=ips
  HYGD%LXR=ipe
  HYGD%LXC=FLOAT(ipe+1)/2.0

  HYGD%LY1=jps
  HYGD%LYR=jpe
  HYGD%LYC=FLOAT(jpe+1)/2.0

!-------------------------------------------------------------------------------
! lagrangian sampling configuration
!-------------------------------------------------------------------------------

  NUMLAG=0

  print *,'www LAGSAM=',LAGSAM,NUMLAG

!FN-20140328, delete IF(LAGSAM)THEN
!!!skip!!!

!-------------------------------------------------------------------------------
! final check on consistency of certain options
!-------------------------------------------------------------------------------

! mass removal fraction set to zero for particle simulations
! now (13 May 2011) permitted although default should be zero
! IF(INITD.EQ.0)FRMR=0.0

!FN-20140610, ICHEM=0 MAXDIM=1 NBPTYP=1 QFILE=F LAGSAM=F, check consistency
!!!skip!!!

!-------------------------------------------------------------------------------

  print *,'www ending HYSPLIT initialization!'

END IF ! HYCOUNT=1

!-------------------------------------------------------------------------------
! HYSPLIT integration
!-------------------------------------------------------------------------------

IF ( JET .EQ. HYFL%FIRST%SACC ) THEN    ! JET=current met time

print *,'www starting HYSPLIT computation!',seed_id
print *,'www jet=',JET,HYFL%FIRST%SACC

!-------------------------------------------------------------------------------
! main loop over number of simulation hours
!-------------------------------------------------------------------------------

      !FN-20140505
       HYDT=WRFDT

!    set integration direction dependent time step
     IF(BACK)HYDT=-HYDT

     print *,'ddd HYDT=',HYDT,WRFDT,ksfc

!    reset time step variables
     UMAX=0.0
     CGSIZE=HYGD%SIZE

!FN-20150520
  IF(TRIM(EFILE).EQ."NONE_SPECIFIED")THEN  ! no EFILE specified in namelist.input
   do n=1,nloc

      call latlon_to_ij(proj,spot(n)%olat,spot(n)%olon,spot(n)%xp,spot(n)%yp)

      iip=nint(spot(n)%xp)
      jjp=nint(spot(n)%yp)

      if ((iip .ge. ips .and. iip .le. ipe .and. iip .lt. ide-2) .and. &
          (jjp .ge. jps .and. jjp .le. jpe .and. jjp .lt. jde-2)) then
         call ht2eta(spot(n)%xp,spot(n)%yp,spot(n)%olvl,eta)
         spot(n)%zp=eta
      else
         spot(n)%zp=-9999.0
      end if

      spot(n)%zp = wrf_dm_max_real(spot(n)%zp)

      print *,'www spot%xp=',spot(n)%xp,spot(n)%yp,spot(n)%zp,spot(n)%olvl

   enddo
  ELSE                                     ! using EFILE
   do kk=1,numems
     jemst=nmsst(kk)*60
     jemen=nmsen(kk)*60

     if (jet .ge. jemst .and. jet .lt. jemen) then 

        do n=1,nloc
        do ii=1,numpol
           call latlon_to_ij(proj,emlat(kk),emlon(kk),sprt(n,ii)%xp,sprt(n,ii)%yp)

           iip=nint(sprt(n,ii)%xp)
           jjp=nint(sprt(n,ii)%yp)

           if ((iip .ge. ips .and. iip .le. ipe .and. iip .lt. ide-2) .and. &
               (jjp .ge. jps .and. jjp .le. jpe .and. jjp .lt. jde-2)) then
              call ht2eta(sprt(n,ii)%xp,sprt(n,ii)%yp,emlvl(kk),eta)
              sprt(n,ii)%zp=eta
           else
              sprt(n,ii)%zp=-9999.0
           end if

           sprt(n,ii)%zp = wrf_dm_max_real(sprt(n,ii)%zp)
           sprt(n,ii)%qlvl = emlvl(kk)
           print *,'www jemst=',jemst,jemen,n,ii
           print *,'www sprt%xp=',sprt(n,ii)%xp,sprt(n,ii)%yp,sprt(n,ii)%zp,sprt(n,ii)%qlvl

           sprt(n,ii)%start=nmsst(kk)
           sprt(n,ii)%stop=nmsen(kk)
           sprt(n,ii)%rate=erate(kk)
           sprt(n,ii)%area=earea(kk)
           sprt(n,ii)%heat=eheat(kk)

           print *,'www sprt%rate=',sprt(n,ii)%rate,sprt(n,ii)%area,sprt(n,ii)%heat

           QTEMP=.TRUE.

        enddo
        enddo

     endif

   enddo
  ENDIF

     print *,'ddd HYGD%LX1=',HYGD%LX1,HYGD%LY1,HYGD%LXR,HYGD%LYR

!-------------------------------------------------------------------------------
!    sub-loop for number of time steps per hour
!-------------------------------------------------------------------------------

        print *,'ddd QFILE=',QFILE,QTEMP

!FN-20150520
!       emissions each time step are from all grid cells or from a point
        IF(QFILE)THEN
           !FN-20140421, QFILE=F
           !!!skip!!!
        ELSEIF(QTEMP)THEN
!          temporal emission variations from input file (emsmat)
!          file emission option not valid with ensemble or backward mode
           KPT=KPM
           CALL EMSTMP(DIRT,SPRT,KGRID,NLOC,NUMTYP,NBPTYP,KPM,INITD,HYDT,JET,  &
                       NUMPAR,MAXPAR,NSORT,MASS,XPOS,YPOS,ZPOS,SIGH,SIGU,      &
                       SIGV,SIGW,HDWP,PAGE,PTYP,PGRD,job_id,num_job,KEMIT,     &
                       METO%GDISX,METO%GDISY)


           IF(KPM.GT.KPT)THEN
              CNTR=0.0
              AVGMIXD=0.0
              AVGRISE=0.0

!             new points have been added when KPM has increased
              DO KP=KPT+1,KPM
                 METO%ZTER=0.0
                 XTMP=XPOS(KP)
                 YTMP=YPOS(KP)
                 ZTMP=ZPOS(KP)            !FN-20150520
!!                dummy call to get terrain and other meteo variables
!!                rdep is set to true to return flux variables
!                 CALL ADVPNT(                                                  &
!                 METZ,METO,BACK,.TRUE.,CDEP,.TRUE.,.FALSE.,KSFC,TKERD,TKERN,   &
!                 INITD,XTMP,YTMP,ZTMP,JET,DT,TRATIO,KGRID,KTIME,NGRD,NTIM,     &
!                 ZSG,NLVL,ZMDL,KVEL,KMIXD,KMIX0,UBAR,IFHR,ICHEM,KRET)
                 IF(KRET.EQ.0)THEN
!                   meteorological variables are available
                    IF(SIGW(KP).GT.0.0)THEN
!!                      non-zero value is btu/hr heat release for plume rise
!                       CALL EMRISE(PAGE(KP),SIGW(KP),METO%SSP,METO%UBAR,       &
!                                   METO%USTR,METO%MIXD,HEIGHT)
!                       ZPOS(KP)=MAX(0.0,1.0-HEIGHT/(ZMDL-METO%ZTER))
!                       SIGW(KP)=0.0
!                       PAGE(KP)=0
!
!                       CNTR=CNTR+1.0
!                       AVGMIXD=AVGMIXD+METO%MIXD
!                       AVGRISE=AVGRISE+HEIGHT

                    ELSE
!!                      normally all turbulence values set to zero at start
!                       ZPOS(KP)=1.0-ZPOS(KP)/(ZMDL-METO%ZTER)
                    END IF
                 ELSE
                    ZPOS(KP)=ZTMP
                 END IF
              END DO
              IF(CNTR.GT.0.0.AND.KS.EQ.1)                                      &
              WRITE(KF21,*)' NOTICE emrise: (mixd,rise) - ',                   &
              NINT(AVGMIXD/CNTR),NINT(AVGRISE/CNTR)
           END IF

        ELSEIF(ltraj)then  !FN-20160601
           if (tjset) then
              print *,'ddd setup trajectories starting points!'
              nloop: do n=1,nloc
                 kpm=kpm+1
                 if (kpm.gt.maxpar)then
                     kpm=maxpar
                     exit nloop
                 endif 
                 xpos(kpm)=spot(n)%xp
                 ypos(kpm)=spot(n)%yp
                 zpos(kpm)=spot(n)%zp
                 hdwp(kpm)=initd
                 page(kpm)=0
                 print *,'tj=',kpm,xpos(kpm),ypos(kpm),zpos(kpm)
              enddo nloop
              tjset=.false.
              print *,'ddd number of trajectories...',kpm
           endif
        ELSE
!          start any new particles from a point (from emsset)
           CALL EMSPNT(SPOT,DIRT,NLOC,NUMTYP,NBPTYP,KPM,INITD,HYDT,JET,NSORT,    &
                       MASS,XPOS,YPOS,ZPOS,SIGH,SIGU,SIGV,SIGW,HDWP,PAGE,PTYP, &
                       PGRD,QCYCLE,NUMPAR,MAXPAR,job_id,num_job,ichem,KEMIT,   &
                       METO%GDISX,METO%GDISY)
        END IF

        print *,'ddd kpt=',kpt,kpm,mass(1,5)

!       diagnostic variables initialization
        TMASS=0.0
        DO KZ=1,NLVL
           ZMASS(KZ)=0.0
        END DO

!FN-20140421, delete LAGEMS, LAGSAM=F, emission of lagrangian samplers
!!!skip!!!

!FN-20150529
!       particle resuspension
        IF(SDEP)                                                                   &
           CALL DEPSUS(CONC,DIRT,INITD,KGRID,KTIME,NUMGRD,NUMTYP,HYDT,ICHEM,       &
                       KPM,CSUM,MASS,XPOS,YPOS,ZPOS,SIGH,SIGU,SIGV,SIGW,HDWP,      &
                       PAGE,PTYP,PGRD,NSORT,MAXPAR,CGRDXP,CGRDYP)

!FN-20150529
!       decay of ground-level deposition amounts
        IF(CDEP.AND.DECAY.EQ.1) CALL DEPRAD(CONC,DIRT,NUMGRD,NUMTYP,HYDT,CSUM)

!-------------------------------------------------------------------------------
!       loop through all particles/puffs
!-------------------------------------------------------------------------------

 IF ( wrf_dm_on_monitor() ) CALL start_timing

 ploop : DO KP=1,MAXPAR

        !FN-20141010
        iip=nint(xpos(kp))
        jjp=nint(ypos(kp))
        if ((iip .ge. ips .and. iip .le. ipe .and. iip .lt. ide-2) .and. &
            (jjp .ge. jps .and. jjp .le. jpe .and. jjp .lt. jde-2)) then
           pgrd(kp) = 1    !inside  the patch domain
        else
           pgrd(kp) = 0    !outside the patch domain
        endif

        if (pgrd(kp) .eq. 0) then
           xpos(kp) = -9999.0
           ypos(kp) = -9999.0
           zpos(kp) = -9999.0
           xprt(kp) = -9999.0
           yprt(kp) = -9999.0
           pagl(kp) = -9999.0

           sigu(kp) = -9999.0
           sigv(kp) = -9999.0
           sigw(kp) = -9999.0

           pedg(kp) = -9999.0
        end if

        if (kp .eq. 1) print *,'ddd XPOS=',xpos(kp),ypos(kp),zpos(kp),seed_id

!       skip terminated particles
        IF(PGRD(KP).EQ.0) CYCLE ploop


!       pollutant index
        JTYP=PTYP(KP)
        IF(NBPTYP.GT.1) JTYP=JTYP/1000

!       set current horizontal distribtion mode
        HDWPX=HDWP(KP)                            ! array in complex mode
        IF(HDWPX.GE.100)HDWPX=MOD(HDWP(KP)/10,10) ! final value in simple mode

        !FN-20140926
        call ij_to_latlon(proj,XPOS(KP),YPOS(KP),PLAT0,PLON0)

!       advects a single point for one time step
        CALL ADVPNT(METZ,METO,.TRUE.,CDEP,RDEP,.FALSE.,TKERD,TKERN,HDWPX,  &
             XPOS(KP),YPOS(KP),ZPOS(KP),HYDT,PGRD(KP),ZSG,NLVL,            &
             UBAR,IFHR,ICHEM,KRET)

        !FN-20141010
        !compute particle height according to ZPOS (eta value)
        call eta2ht(XPOS(KP),YPOS(KP),ZPOS(KP),PAGL(KP))

        !FN-20140926
        !convert end-point grid position to true coordinates
        call ij_to_latlon(proj,XPOS(KP),YPOS(KP),XPRT(KP),YPRT(KP))
        METO%PLAT=XPRT(KP)
        METO%PLON=YPRT(KP)

        !FN-20160601
        if (ltraj) cycle ploop

!       convert to (km/min) = (gp/min) (km/gp)
        UBAR=UBAR*HYGD%SIZE

!FN-20140421, ICHEM=0, pm10 dust emission algorithm
!!!skip!!!

!FN-20150511
!!       skip terminated particles
!        IF(PGRD(KP).EQ.0) CYCLE ploop
        KGRID=PGRD(KP)

!       surface terrain height for this position
        ZSFC=METO%ZTER

!       save maximim advection speed (0.06 km/min = 1 m/s)
        UMAX=MAX(0.060, UMAX, UBAR)

!FN-20141010
!!FN-20140421, ICHEM=0, increment particle age after advection
!!!!skip!!!

!       adjusts advection to simulate particle dispersion
        CALL PARDSP(METO%UMIX,METO%VMIX,METO%GDISX,METO%GDISY,METO%TOMU,HYDT,ZSFC,NLVL,METZ%WMIX,ZSG, &
                    METZ%PRES,METZ%ZHGT,VSCALE,HSCALE,XPOS(KP),YPOS(KP),ZPOS(KP),                     &
                    SIGV(KP),SIGW(KP),SIGU(KP),HDWPX,METO%ZNDX,seed_id)
        if (kp .eq. 1) print *,'ddd dsp=',xpos(kp),ypos(kp),zpos(kp),seed_id
        if (kp .eq. 1) print *,'ddd dsp=',sigu(kp),sigv(kp),sigw(kp)

        if ((xpos(kp) .ge. float(ips+1) .and. xpos(kp) .le. float(ipe-1)) .and. &
            (ypos(kp) .ge. float(jps+1) .and. ypos(kp) .le. float(jpe-1))) then
           pedg(kp) = 0.0   ! inside sub-grid
        else
           pedg(kp) = 1.0   ! close to sub-grid edge
        endif

!FN-20140107, delete PUFDSP, HDWP=INITD=0 (3D particle mode), vertical & horizontal puff mixing
!!!skip!!!

        !FN-20140107
        !compute particle height according to ZPOS (eta value)
         call eta2ht(XPOS(KP),YPOS(KP),ZPOS(KP),PAGL(KP))
         
        !FN-20140926
        !convert end-point grid position to true coordinates
         call ij_to_latlon(proj,XPOS(KP),YPOS(KP),XPRT(KP),YPRT(KP))

!FN-20140421, delete CHEM02, ICHEM=0, simple chemistry conversion option
!!!skip!!!

!FN-20150511
!       optional dry, wet, and decay routines
        IF(CDEP) CALL DEPELM(DIRT,SPOT(1)%OLAT,SPOT(1)%IBMO,NLVL,HYDT,ZSFC,MASS(:,KP),DEPT, &
                             XPOS(KP),YPOS(KP),ZPOS(KP),PAGL(KP),SIGW(KP),ICHEM,PTYP(KP), &
                             METO%LAND,METO%AERO,METO%USTR,METO%PSI,METO%DSWF,  &
                             HDWPX,METO%RAIN,RHB,RHT,METZ%DENS,METZ%TEMP,METZ%RHFR,  &
                             KSFC,METZ%ZHGT,DRYD)

!FN-20140421, ICHEM=0, surface water transport option
!!!skip!!!

!FN-20140107, ICHEM=0
!        IF(ICHEM.EQ.4)THEN
!          !!!skip!!!
!        ELSE
!
!           IF(ICHEM.EQ.6)THEN
!              !!!skip!!!
!
!           ELSEIF(ICHEM.EQ.8)THEN
!              !!!skip!!!
!
!           ELSEIF(ICHEM.EQ.9)THEN
!              !!!skip!!!
!
!           ELSE
              MSUM=MASS(:,KP)
!           END IF

!FN-20141010
!!FN-20140909, CMASS=0
!           IF(CMASS.EQ.0)THEN
!
!              IF(ABS(HYDT).GT.1.0)THEN
                 !FN-20140107
                 CALL CONSUM(CONC,NUMGRD,XPRT(KP),YPRT(KP),HYDT,JET,ZSFC,KMSL,         &
                             CGSIZE,MSUM,DEPT,PAGL(KP),SIGH(KP),SIGW(KP),              &
                             HDWPX,JTYP,CSUM)
!              ELSE
!!                first time entry, do not use meteorology grid size
!                 IF(CGSIZE.EQ.HYGD%SIZE) CGSIZE=MINVAL(CONC(:)%SIZE)
!
!!                compute the number of interpolations required when the advection
!!                distance is much greater than the concentration grid cell size
!!                use 200 km per deg rather than 111 to slightly over sample
!   
!!                test for dateline issues and adjust accordingly
!                 YTMP=METO%PLAT-PLAT0
!                 XTMP=METO%PLON-PLON0
!                 IF(ABS(XTMP).GT.180.0) XTMP=SIGN(360.0-ABS(XTMP),METO%PLON)
!   
!                 DIST=YTMP**2+(XTMP*COS(PLAT0/57.3))**2
!                 NUMB=MIN(15,INT(1+200.0*SQRT(DIST)/CGSIZE))
!                 DO MM=1,NUMB
!                    FRAC=FLOAT(MM)/NUMB
!                    PLAT1=YTMP*FRAC+PLAT0
!                    PLON1=XTMP*FRAC+PLON0
!                    FRAC=ABS(HYDT)/FLOAT(NUMB)
!                    !FN-20140107
!                    CALL CONSUM(CONC,NUMGRD,PLAT1,PLON1,HYDT,JET,ZSFC,KMSL,       &
!                         CGSIZE,MSUM,DEPT,PAGL(KP),SIGH(KP),SIGW(KP),             &
!                         HDWPX,JTYP,CSUM)
!                 END DO
!              END IF  ! fine grid interpolated summation (hydt=1)
!
!           ELSE
!              !FN-20140107, delete MASSUM, CMASS=0, sum mass to grid
!
!           END IF  ! mass sum section (cmass<>0)

!        END IF     ! standard concentration summation section (ichem<>4)

!FN-20140421, delete LAGSUM, LAGSAM=F
!!!skip!!!

!FN-20141010
!!FN-20140421, INITK=INITD=0
!!       sum mass for diagnostic analysis
!        IF(INITK.EQ.1.OR.INITK.EQ.2)THEN
!          !!!skip!!!
!
!        ELSE
!!          particles summed into nearest meteo index
!           K1=MAX(1, MIN(NLVL, NINT(METO%ZNDX) ))
!           K2=K1
!        END IF
!
!        FRAC=FLOAT(K2-K1+1)
!        DO KZ=K1,K2
!           PMASS=MASS(1,KP)
!           ZMASS(KZ)=ZMASS(KZ)+MASS(1,KP)/FRAC
!           MM=MAXDIM
!           DO WHILE(MM.GT.1)
!              PMASS=PMASS+MASS(MM,KP)
!              ZMASS(KZ)=ZMASS(KZ)+MASS(MM,KP)/FRAC
!              MM=MM-1
!           END DO
!           TMASS=TMASS+PMASS/FRAC
!        END DO
!!       mark zero mass for removal (except lagrangian samplers)
!        IF(PMASS.EQ.0.0.AND.HDWP(KP).NE.6)PGRD(KP)=0

!    particle/puff loop
     END DO ploop

    IF ( wrf_dm_on_monitor() ) CALL end_timing ('processing ploop' )

    !FN-20141010
    print *,'www saving particle information!'
    IF ( wrf_dm_on_monitor() ) CALL start_timing

    do kp=1,maxpar

     if (ltraj) then 
        xpos(kp) = wrf_dm_max_real(xpos(kp))
        ypos(kp) = wrf_dm_max_real(ypos(kp))
        zpos(kp) = wrf_dm_max_real(zpos(kp))

        xprt(kp) = wrf_dm_max_real(xprt(kp))
        yprt(kp) = wrf_dm_max_real(yprt(kp))
        pagl(kp) = wrf_dm_max_real(pagl(kp))

        if (kp.lt.50) print *,'www traj=',kp,xpos(kp),ypos(kp),zpos(kp),xprt(kp),yprt(kp),pagl(kp)
     else
       pedg(kp) = wrf_dm_max_real(pedg(kp))

       if (pedg(kp) .eq. 1.0) then 
          sigu(kp) = wrf_dm_max_real(sigu(kp))
          sigv(kp) = wrf_dm_max_real(sigv(kp))
          sigw(kp) = wrf_dm_max_real(sigw(kp))
          zpos(kp) = wrf_dm_max_real(zpos(kp))

          ypos(kp) = wrf_dm_max_real(ypos(kp))
          xpos(kp) = wrf_dm_max_real(xpos(kp))
          pagl(kp) = wrf_dm_max_real(pagl(kp))

       endif

           MSUM=MASS(:,KP)
           JTYP=PTYP(KP)
           IF(NBPTYP.GT.1) JTYP=JTYP/1000

           !FN-20141010
           iip=nint(xpos(kp))
           jjp=nint(ypos(kp))
           if ((iip .ge. ips .and. iip .le. ipe .and. iip .lt. ide-2) .and. &
               (jjp .ge. jps .and. jjp .le. jpe .and. jjp .lt. jde-2)) then
              !FN-20140909
              CALL AVGCON(CONC,NUMGRD,XPOS(KP),YPOS(KP),PAGL(KP),HYDT,JET,ZSFC,KMSL,  &
                          CGSIZE,MSUM,DEPT,SIGH(KP),SIGW(KP),HDWPX,JTYP,              &
                          HYCSUM,ims,ime,jms,jme)
           endif
     endif   !ltraj

    end do

    print *,'ddd csum=',maxval(csum),maxval(hycsum)

    IF ( wrf_dm_on_monitor() ) CALL end_timing ('processing saving')

!-------------------------------------------------------------------------------
!    end of particle loop ... now call routines requiring global data
!-------------------------------------------------------------------------------

!FN-20141010
!!    update maximum concentration array if requested
!     IF(KMAXC.NE.0.AND.KSNAP.NE.0)  &
!        CSUM(:,:,:,:,KMAXC)=MAX(CSUM(:,:,:,:,KMAXC),CSUM(:,:,:,:,KSNAP))
!     IF(KMAXC.NE.0.AND.KAVRG.NE.0)  &
!        CSUM(:,:,:,:,KMAXC)=MAX(CSUM(:,:,:,:,KMAXC),CSUM(:,:,:,:,KAVRG))

  if (.not.ltraj) then  !FN-20160610

     IF ( wrf_dm_on_monitor() ) CALL start_timing

!    check each time step for output to disk and zero-out array
     CALL CONDSK(CONC,DIRT,ICHEM,KGRID,KTIME,NUMGRD,NUMTYP,NBPTYP,         &
                 HYDT,JET,IFHR,CPACK,CSUM,crrtask)

     IF ( wrf_dm_on_monitor() ) CALL end_timing ('processing condsk')
 
!!FN-20140421, delete LAGOUT, LAGSAM=F
!!!!skip!!!
 
!    zero-out array if required
     CALL CONZRO(CONC,NUMGRD,HYDT,JET,IFHR,CSUM)

     print *,'ddd tmass=',JET,KPM,TMASS

  endif !ltraj

!-------------------------------------------------------------------------------
! after end of time steps per hour loop call more infrequent routines
!-------------------------------------------------------------------------------

!FN-20140421, no KH loop, NHRS=1, special diagnostics dump
!!!skip!!!

!FN-20131125, same diagnostics dump, but separate file, hourly
!!!skip!!!

!-------------------------------------------------------------------------------
! puff/particle conversion routines (call before merge)
!-------------------------------------------------------------------------------

!FN-20140421, delete PARPUF & PUFPAR, INITK=INITD=0, optional conversion of puffs to particles
!!!skip!!!

!-------------------------------------------------------------------------------
! puff splitting routines called each hour (default KSPL=1)
!-------------------------------------------------------------------------------

!FN-20140421, INITK=INITD=0
!!!skip!!!

!-------------------------------------------------------------------------------
! puff management routines called each hour
!-------------------------------------------------------------------------------

!FN-20140421, INITK=INITD=0
!!!skip!!!

!-------------------------------------------------------------------------------
! 3D particle management routines called at KRND interval (new 16 May 2011)
!-------------------------------------------------------------------------------

!FN-20141010
!!FN-20140421, INITK=INITD=0 & FRMR=0
!!  IF(INITD.EQ.0.AND.FRMR.NE.0.0.AND.MOD(KH,KRND).EQ.0)THEN
!!     !!!skip!!!
!!  ELSE
!     RMASS=0.0
!!  END IF
!
!! eliminate unused puffs or particles (pgrd=0)
!  !FN-20140303
!  CALL PUFDEL(KPM,MASS,XPOS,YPOS,ZPOS,XPRT,YPRT,PAGL,SIGH,SIGU,SIGV,SIGW,HDWP,PAGE,PTYP,      &
!              PGRD,NSORT,KHMAX,RMASS)
!

!-------------------------------------------------------------------------------
! less frequent but less restictive merge using enhanced merge parameters
! and the option to revove low mass particles if mass revoval turned on
!-------------------------------------------------------------------------------

!FN-20140421, INITK=INITD=0
!!!skip!!!

!-------------------------------------------------------------------------------
! particle input and output for model reinitialization
!-------------------------------------------------------------------------------

!FN-20141010
!! dump particle positions
!    !FN-0214,parout is same as cdump output interval
!    IF(NDUMP.GT.0.AND.MOD((JET-CONC(1)%START%SACC),CONC(KG)%DELTA%SACC).EQ.0)THEN
!
!      !print *,'www call PAROUT',JET-CONC(1)%START%SACC,CONC(1)%DELTA%SACC
!      INQUIRE(FILE=POUTF,OPENED=FTEST2)
!
!      IF(.NOT.FTEST2)OPEN(KF24,FILE=POUTF,FORM='UNFORMATTED',ACCESS='SEQUENTIAL')
!
!      !FN-20140107
!      CALL PAROUT(JET,KPM,MASS,XPRT,YPRT,PAGL,SIGH,SIGU,    &
!                  SIGV,SIGW,HDWP,PAGE,PTYP,PGRD,NSORT)
!
!    END IF

!FN-20140421, delete PARINP, FTEST1=F, load more particles if time matches
!!!skip!!!

     !FN-20140909
     JET=JET+INT(HYDT)        ! increment elapsed time

ELSE

  print *,'www skip HYSPLIT jet=',JET,HYFL%FIRST%SACC

END IF ! JET.NE.HYFL%FIRST%SACC

!-------------------------------------------------------------------------------

IF (HYFL%FIRST%SACC.EQ.HYFL%ENWRF%SACC) THEN 

  print *,'www closing and deallocation!'

!FN-20141101 
  if (crrtask .eq. 0 .and. .not.ltraj) then
     DO K=1,NUMGRD
        CLOSE(CONC(K)%UNIT)
     END DO
  endif

! special termination messages
  IF(.NOT.SPLITON)THEN
     WRITE(*,*)   'WARNING main: puff splitting turned off ... check MESSAGE'
     WRITE(*,*)'WARNING main: puff splitting turned off ... expand MAXPAR'
     WRITE(*,*)'        or increase merge parameters in CFG namelist file'
  END IF
  IF(KEMIT.EQ.1)THEN
     WRITE(*,*)   'WARNING particle array limit reached; emissions turned off'
     WRITE(*,*)'WARNING particle array limit reached; expand MAXPAR'
  END IF
  WRITE(*,*)'Complete Hysplit'
  IF(ICHEM.EQ.8) CLOSE(KF20)    ! STILT emulation PARTICLE.DAT file

  IF(ALLOCATED(sigh))     DEALLOCATE (sigh)
  IF(ALLOCATED(sigu))     DEALLOCATE (sigu)
  IF(ALLOCATED(sigv))     DEALLOCATE (sigv)
  IF(ALLOCATED(sigw))     DEALLOCATE (sigw)
  IF(ALLOCATED(msum))     DEALLOCATE (msum)
  IF(ALLOCATED(mass))     DEALLOCATE (mass)
  IF(ALLOCATED(zmass))    DEALLOCATE (zmass)
  IF(ALLOCATED(page))     DEALLOCATE (page)
  IF(ALLOCATED(hdwp))     DEALLOCATE (hdwp)
  IF(ALLOCATED(ptyp))     DEALLOCATE (ptyp)
  IF(ALLOCATED(pgrd))     DEALLOCATE (pgrd)
  IF(ALLOCATED(nsort))    DEALLOCATE (nsort)
  IF(ALLOCATED(conc))     DEALLOCATE (conc)
  IF(ALLOCATED(dirt))     DEALLOCATE (dirt)
  IF(ALLOCATED(spot))     DEALLOCATE (spot)
  IF(ALLOCATED(metz))     DEALLOCATE (metz)
  IF(ALLOCATED(cgrdxp))   DEALLOCATE (cgrdxp)
  IF(ALLOCATED(cgrdyp))   DEALLOCATE (cgrdyp)
  IF(ALLOCATED(dept))     DEALLOCATE (dept)
  IF(ALLOCATED(dryd))     DEALLOCATE (dryd)
  IF(ALLOCATED(csum))     DEALLOCATE (csum)
  IF(ALLOCATED(sprt))     DEALLOCATE (sprt)  
  IF(QFILE)               DEALLOCATE (polid,qarea)

  CALL DALLOC

END IF

!-------------------------------------------------------------------------------

print *,'www end of hyconc => return back to hymain!'

END  SUBROUTINE hyconc
