!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  METINI           METeorological INItialization
!   PRGMMR:    Fantine Ngan     ORG: R/ARL       DATE:2012-08-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   METEOROLOGICAL INITIALIZATION OPENS THE METEO FILES THE FIRST TIME
!   ON EACH DEFINED GRID SYSTEM; DEFINES DEFAULT GRID STRUCTURE WHICH
!   CANNOT CHANGE WITH TIME FOR A DEFINED GRID NUMBER. MULTIPLE FILE
!   TIME ARE FOR THE SAME GRID NUMBER AND REQUIRE THE SAME STRUCTURE
!   DEFINITIONS.
!
! PROGRAM HISTORY LOG:
!   LAST REVISION: 01 Aug 2012 (FN) - WRF-HYSPLIT coupling initial implementation
!                  25 Oct 2013 (FN) - modify for eta coordinate
!                  28 Mar 2014 (FN) - clean up
!                  21 Apr 2014 (FN) - remove NGRD (KG) and NTIM (KT)
!                  15 Apr 2014 (FN) - modify for tight coupling
!                  15 May 2014 (FN) - add accumulated seconds %SACC
!                  26 Sep 2014 (FN) - clean up
!
!$$$

SUBROUTINE METINI(grid,config_flags)

    USE metval

    USE module_domain
    USE module_configure

  IMPLICIT NONE

  INCLUDE 'DEFGRID.INC' ! meteorology grid and file

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

    ! input data
    TYPE(domain)            :: grid

    !  Structure that contains run-time configuration (namelist) data for domain
    TYPE (grid_config_rec_type) , INTENT(IN)          :: config_flags

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  LOGICAL    :: fields
  INTEGER    :: i,j,nvar,kg,kt,k2,nlvl,macc,mjet,itda,ithr,itmn

  integer                 :: nrec                             ! place holder for file record
  integer                 :: ibyr,ibmo,ibda,ibhr,ibmn,ibse    ! wrf model current date & time
  integer                 :: isyr,ismo,isda,ishr,ismn,isse    ! wrf model starting date & time
  integer                 :: ieyr,iemo,ieda,iehr,iemn,iese    ! wrf model ending date & time
  integer                 :: tmac
  integer                 :: fhr                         ! forecast hour
  character(20)           :: ccdate                      ! model current time YYYY-MM-DD_HH:MN:SE
  character(20)           :: stdate                      ! model start time YYYY-MM-DD_HH:MN:SE
  character(20)           :: endate                      ! model end time YYYY-MM-DD_HH:MN:SE

  integer                 :: wrfdt                       ! WRF time step
  integer                 :: nxs,nys,nzs                 ! stagger dimension
  integer                 :: map_proj
  real                    :: cen_lat,cen_lon,stand_lon
  real                    :: truelat1,truelat2,dxkm,dykm
  real                    :: ssac

!-------------------------------------------------------------------------------
! external variables
!-------------------------------------------------------------------------------

  COMMON /GBLGRD/ HYGD, DREC, HYFL

!-------------------------------------------------------------------------------
! WRF related variables
!-------------------------------------------------------------------------------

    TYPE(WRFU_Time)            :: currentTime, startTime, stopTime

    REAL, EXTERNAL             :: eqvlat                   ! function to compute reference lat

!-------------------------------------------------------------------------------
  INTERFACE
!-------------------------------------------------------------------------------
  SUBROUTINE TMINIT
  IMPLICIT NONE
  END SUBROUTINE tminit
!-------------------------------------------------------------------------------
  SUBROUTINE TM2DAY(MACM,IY,IM,ID,IH,MN)
  IMPLICIT NONE
  INTEGER,   INTENT(IN)    :: macm            ! accumulate minutes
  INTEGER,   INTENT(OUT)   :: iy,im,id,ih,mn  ! current date/time
  END SUBROUTINE tm2day
!-------------------------------------------------------------------------------
  SUBROUTINE TM2MIN(IY,IM,ID,IH,MN,MACC)
  IMPLICIT NONE
  INTEGER,  INTENT(IN)   :: iy,im,id,ih,mn       ! date and time
  INTEGER,  INTENT(OUT)  :: macc                 ! minutes since 1 Jan 1970
  END SUBROUTINE tm2min
!-------------------------------------------------------------------------------
  END INTERFACE
!-------------------------------------------------------------------------------

    print *,'bbb in hysp/metini grid%id=',grid%id

    CALL domain_clock_get( grid, current_time=currentTime, &
                                 start_time=startTime,     &
                                 start_timestr=stdate,     &
                                 stop_time=stopTime,       &
                                 stop_timestr=endate,      &
                                 current_timestr=ccdate)

    read (ccdate,'(i4,5(x,i2))') IBYR,IBMO,IBDA,IBHR,IBMN,IBSE
    read (stdate,'(i4,5(x,i2))') ISYR,ISMO,ISDA,ISHR,ISMN,ISSE
    read (endate,'(i4,5(x,i2))') IEYR,IEMO,IEDA,IEHR,IEMN,IESE

    print *,'bbb ccdate=',ccdate
    print *,'bbb stdate=',stdate
    print *,'bbb endate=',endate

    print *,'bbb sec=',IBSE,ISSE,IESE

! set all grid numbers to missing
  HYGD%NUMBER=1         !FN-0801, set to dummy value

!-------------------------------------------------------------------------------
! time initialization
!-------------------------------------------------------------------------------
  CALL TMINIT

!-------------------------------------------------------------------------------
! get date & time information
!-------------------------------------------------------------------------------

!FN-20140512, current HYSPLIT time = current WRF clock + WRFDT(min)
  CALL TM2MIN(IBYR,IBMO,IBDA,IBHR,IBMN,tmac)

  !FN-20141215
  wrfdt = int(grid%dt)

  HYFL%FIRST%MACC = tmac+int(wrfdt/60)
  print *,'bbb FIRST%MACC=',HYFL%FIRST%MACC,tmac,wrfdt

  CALL TM2DAY(HYFL%FIRST%MACC,                 &
              HYFL%FIRST%YR, HYFL%FIRST%MO,    &
              HYFL%FIRST%DA, HYFL%FIRST%HR,    &
              HYFL%FIRST%MN)

  print *,'bbb HYFL%FIRST=',HYFL%FIRST%YR, HYFL%FIRST%MO,HYFL%FIRST%DA, HYFL%FIRST%HR,HYFL%FIRST%MN

    HYFL%STWRF%YR = ISYR       ! WRF model starting time
    HYFL%STWRF%MO = ISMO
    HYFL%STWRF%DA = ISDA
    HYFL%STWRF%HR = ISHR
    HYFL%STWRF%MN = ISMN

    HYFL%ENWRF%YR = IEYR       ! WRF model ending time
    HYFL%ENWRF%MO = IEMO
    HYFL%ENWRF%DA = IEDA
    HYFL%ENWRF%HR = IEHR
    HYFL%ENWRF%MN = IEMN

    nrec = 1                   !FN-0801, set to dummy value

  CALL TM2MIN(HYFL%STWRF%YR, HYFL%STWRF%MO,                  &
              HYFL%STWRF%DA, HYFL%STWRF%HR,                  &
              HYFL%STWRF%MN, HYFL%STWRF%MACC)

  CALL TM2MIN(HYFL%ENWRF%YR, HYFL%ENWRF%MO,                  &
              HYFL%ENWRF%DA, HYFL%ENWRF%HR,                  &
              HYFL%ENWRF%MN, HYFL%ENWRF%MACC)

  fhr = INT((HYFL%FIRST%MACC-HYFL%STWRF%MACC)/60.0)
  HYFL%FIRST%IC = fhr          ! forecast hour

  print *,'bbb macc=',HYFL%FIRST%MACC,HYFL%STWRF%MACC,HYFL%ENWRF%MACC,fhr

!FN-20140515, accumulated seconds
  ssac = (float(tmac)*60.0)+float(ibse)+float(wrfdt)
  print *,'bbb ssac=',ssac
  HYFL%FIRST%SACC = HYFL%FIRST%MACC * 60 + ibse !(tmac*60)+ibse+wrfdt
  HYFL%STWRF%SACC = HYFL%STWRF%MACC * 60
  HYFL%ENWRF%SACC = HYFL%ENWRF%MACC * 60

  print *,'bbb ssac=',HYFL%FIRST%SACC,HYFL%STWRF%SACC,HYFL%ENWRF%SACC

! revelant to reading file, should not be used 
    DREC%REC_PER=NREC
    HYFL%ENDREC=NREC-1

!   set to dummy value to avoid division by zero
    DREC%DELTA=1

!-------------------------------------------------------------------------------
! get horizontal grid infomation
!-------------------------------------------------------------------------------

  CALL nl_get_map_proj  ( grid%id , map_proj )
  CALL nl_get_cen_lat   ( grid%id , cen_lat )
  CALL nl_get_cen_lon   ( grid%id , cen_lon )
  CALL nl_get_stand_lon ( grid%id , stand_lon )
  CALL nl_get_truelat1  ( grid%id , truelat1 )
  CALL nl_get_truelat2  ( grid%id , truelat2 )
  CALL nl_get_dx        ( grid%id , dxkm )
  CALL nl_get_dx        ( grid%id , dykm )
  CALL nl_get_e_we      ( grid%id , nxs  )
  CALL nl_get_e_sn      ( grid%id , nys  )
  CALL nl_get_e_vert    ( grid%id , nzs  )

  dxkm = dxkm/1000.
  dykm = dykm/1000.

  HYGD%NX = nxs - 1
  HYGD%NY = nys - 1
  HYGD%NZ = nzs - 1

  HYGD%POLE_LAT = eqvlat(truelat1,truelat2)
  HYGD%POLE_LON = stand_lon
  HYGD%REF_LAT  = eqvlat(truelat1,truelat2)
  HYGD%REF_LON  = stand_lon
  HYGD%SIZE     = 0.5*(dxkm+dykm)
  HYGD%ORIENT   = 0.0
  HYGD%TANG_LAT = eqvlat(truelat1,truelat2)
  HYGD%SYNC_XP  = 0.5*(HYGD%NX+1)
  HYGD%SYNC_YP  = 0.5*(HYGD%NY+1)
  HYGD%SYNC_LAT = cen_lat
  HYGD%SYNC_LON = cen_lon

  print *,'bbb pole_lat=',HYGD%POLE_LAT,HYGD%POLE_LON,HYGD%REF_LAT
  print *,'bbb  ref_lon=',HYGD%REF_LON,HYGD%SIZE,HYGD%ORIENT
  print *,'bbb tang_lat=',HYGD%TANG_LAT,HYGD%SYNC_XP,HYGD%SYNC_YP
  print *,'bbb sync_lat=',HYGD%SYNC_LAT,HYGD%SYNC_LON

  HYGD%MODEL_ID = 'AWRF'
  HYGD%DUMMY    = 0.0

  DREC%Z_FLAG   = 1

!-------------------------------------------------------------------------------
! surface parameters
! set flags for the type of surface data available
!-------------------------------------------------------------------------------

! ten meter wind and two meter temp
  DREC%TFLG=.TRUE.
  DREC%UFLG=.TRUE.

! surface exchange coefficient, heat, momentum, shortwave flux
  DREC%EFLX=.FALSE.
  DREC%UFLX=.FALSE.
  DREC%DSWF=.TRUE.
  DREC%HFLX=.TRUE.

! friction velocity and temperature
  DREC%USTR=.TRUE.
  DREC%TSTR=.FALSE.

! surface terrain height and pressure
  DREC%SHGT=.TRUE.
  DREC%PRSS=.TRUE.

! set default precipitation accumulation time (default = none)
  DREC%ACYCLE=99999   !FN-0816, using accumulated total precip

! input vertical velocity units m/s rather than hPa/s
  DREC%DZDT=.FALSE.

! mixed layer depth available 
  DREC%MIXD=.TRUE.

! vertical index above with data are extrapolated
  DREC%KTOP=0

! all WRF versions have u,v grids staggered with respect to mass points
! WRF=1 - all versions assume instantaneous velocities (from arw2arl)
! WRF=2 - not used; nearest neighbor interpolation (NMM-WRF from wrf2arl) 
! WRF=3 - associated with AER-WRF for STILT model (averaged fields)

  DREC%WRF=1  

! by default assume all model fields are snapshot (not time averaged)
  DREC%TAVRG=.FALSE.   !FN-0801, only use snapshot

! some 3D variables may not be available at all levels
  DREC%TKEN=.FALSE.  
  DREC%VELV=.FALSE.

! set moisture as specific humidity ... false means relative humidity
  DREC%QFLG=.TRUE.

! upper level pressure data indicates non-hydrostatic
  DREC%ZFLG=.TRUE.

! set moisture availability by level
  DREC%RFLG(:)=.TRUE.

! set vertical motion flag if field found
  DREC%WFLG(:)=.TRUE.

!-------------------------------------------------------------------------------
! initialize grid conversion subroutines
!-------------------------------------------------------------------------------

  HYGD%LATLON=.FALSE.
  HYGD%GLOBAL=.FALSE.
  HYGD%GBLDAT=.FALSE.
  HYGD%PRIME=.FALSE.

  print *,'bbb latlon=',HYGD%LATLON,HYGD%GBLDAT,HYGD%SIZE,HYGD%PRIME

!-------------------------------------------------------------------------------
! check for consistency of meteorological data with turbulence namelist options
! then set internal values for each input meteorological data set
!-------------------------------------------------------------------------------

  DREC%KDEF=MAX(0,MIN(1,config_flags%kdef))
  DREC%KBLS=MAX(1,MIN(2,config_flags%kbls))
  DREC%KBLT=MAX(1,MIN(4,config_flags%kblt))
  DREC%KZMIX=MAX(0,MIN(1,config_flags%kzmix))
  DREC%TVMIX=MAX(0.01,MIN(100.0,config_flags%tvmix))

  IF(config_flags%kbls.EQ.1)THEN
!    needs flux field
     FIELDS=(DREC%EFLX.OR.DREC%USTR.OR.DREC%UFLX).AND.  &
            (DREC%TSTR.OR.DREC%HFLX)
     IF(.NOT.FIELDS)THEN
        WRITE(*,*)'WARNING: metini - FLUXES not found in data'
        WRITE(*,*)'Setting KBLS=2 to use the wind/temp profiles!'
        DREC%KBLS=2
     END IF
  ELSEIF(config_flags%kbls.EQ.2)THEN
!    wind and temperature profile
     FIELDS=(DREC%UFLG.AND.DREC%TFLG)
     IF(.NOT.FIELDS)THEN
        WRITE(*,*)'WARNING: metini - SFC wind/temp not found in data'
        WRITE(*,*)'KBLS setting unchanged but compuatation may not be as accurate!'
     END IF
  ELSE
     CONTINUE
  END IF

  IF(config_flags%kblt.EQ.3)THEN
!    tke input field
     IF(.NOT.DREC%TKEN)THEN
        WRITE(*,*)'WARNING: metini - TKE not found in data'
        WRITE(*,*)'Setting KBLT=2 to use Kantha AND Clayson parameterization!'
        DREC%KBLT=2
     END IF
  ELSEIF(config_flags%kblt.EQ.4)THEN
!    velocity variance input field
     IF(.NOT.DREC%VELV)THEN
        WRITE(*,*)'WARNING: metini - VARIANCE not found in data'
        WRITE(*,*)'Setting KBLT=2 to use Kantha AND Clayson parameterization!'
        DREC%KBLT=2
     END IF
  ELSE
     CONTINUE
  END IF

  print *,'bbb KDEF=',DREC%KDEF,DREC%KBLS,DREC%KBLT
  print *,'bbb KZMIX=',DREC%KZMIX,DREC%TVMIX

print *,'bbb end of metini => return back to hymain!'
RETURN

END SUBROUTINE metini

!------------------------------------------------------------------------------
!from HYSPLIT
!*  WRITTEN ON 3/31/94 BY Dr. Albion Taylor  NOAA / OAR / ARL
!------------------------------------------------------------------------------
      REAL FUNCTION EQVLAT (XLAT1,XLAT2)
      PARAMETER (PI=3.14159265358979,RADPDG=PI/180,DGPRAD=180/PI)
      SIND(X) = SIN (RADPDG*X)
      SINL1 = SIND (XLAT1)
      SINL2 = SIND (XLAT2)
      IF (ABS(SINL1 - SINL2) .GT. .001) THEN
        AL1 = LOG((1. - SINL1)/(1. - SINL2))
        AL2 = LOG((1. + SINL1)/(1. + SINL2))
      ELSE
!  CASE LAT1 NEAR OR EQUAL TO LAT2
        TAU = - (SINL1 - SINL2)/(2. - SINL1 - SINL2)
        TAU = TAU*TAU
        AL1  = 2. / (2. - SINL1 - SINL2) * (1.    + TAU *                      &
                                          (1./3. + TAU *                       &
                                         (1./5. + TAU *                        &
                                         (1./7.))))
        TAU =   (SINL1 - SINL2)/(2. + SINL1 + SINL2)
        TAU = TAU*TAU
        AL2  = -2. / (2. + SINL1 + SINL2) * (1.    + TAU *                     &
                                           (1./3. + TAU *                      &
                                          (1./5. + TAU *                       &
                                          (1./7.))))
      ENDIF
      EQVLAT = ASIN((AL1 + AL2) / (AL1 - AL2))/RADPDG
      RETURN
      END FUNCTION EQVLAT
