!$$$  SUBPROGRAM DOCUMENTATION BLOCKMIN1MIN1
!
! SUBPROGRAM:  PARDSP           PARticle DISpersion turbulent component
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   PARTICLE DISPERSION ADDS THE TURBULENT VELOCITY COMPONENT TO THE
!   PREVIOUSLY CALCULATED POSITION.  BASED UPON DIFFUSIVITY PROVIDING
!   THE BASIS FOR SIGMA U,V,W.  LOW TURBULENCE CORRECTION ONLY APPLIES
!   TO THE VERTICAL COMPONENT.
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: ...
!                 09 Dec 2013 (FN) - add varialbes (zhgt, pres) for coupling
!                 07 Jan 2014 (FN) - modify for eta coordinate
!                 28 Mar 2014 (FN) - clean up
!                 15 May 2014 (FN) - change min => sec
!                 10 Jun 2014 (FN) - bring in vscale & hscale
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE pardsp(umix,vmix,gdisx,gdisy,tomu,dtm,zsfc,nlvl,wmix,zsg,       &
                  pres,zhgt,vscale,hscale,                                 &
                  xpos,ypos,zpos,vprime,wratio,uprime,hdwp,zndx,iseed)

  IMPLICIT NONE

!-------------------------------------------------------------------------------
! argument list variables 
!-------------------------------------------------------------------------------

  REAL,         INTENT(IN)     :: umix        ! u-component turbulence (m2/s2)
  REAL,         INTENT(IN)     :: vmix        ! v-component turbulence (m2/s2)
  REAL,         INTENT(IN)     :: gdisx       ! horizontal grid distance (m) 
  REAL,         INTENT(IN)     :: gdisy       ! horizontal grid distance (m) 
  REAL,         INTENT(IN)     :: tomu        ! column mass (mu, Pa), !FN-20140107
  REAL,         INTENT(IN)     :: dtm         ! advection time step (sec) !FN-20140515  
  REAL,         INTENT(IN)     :: zsfc        ! terrain height (m)         
  INTEGER,      INTENT(IN)     :: nlvl        ! number of levels in subgrid
  REAL,         INTENT(IN)     :: wmix (:)    ! vertical turbulence (m2/s2)
  REAL,         INTENT(IN)     :: zsg  (:)    ! WRF eta levels,   !FN-20140107
  REAL,         INTENT(IN)     :: pres (:)    ! pressure profile, !FN-20131209
  REAL,         INTENT(IN)     :: zhgt (:)    ! height profile,   !FN-20131209
  REAL,         INTENT(IN)     :: vscale      ! vert time scale (sec)  !FN-20140610
  REAL,         INTENT(IN)     :: hscale      ! horz time scale (sec)  !FN-20140610
  REAL,         INTENT(INOUT)  :: xpos        ! EW horizontal particle position
  REAL,         INTENT(INOUT)  :: ypos        ! NS horizontal particle position
  REAL,         INTENT(INOUT)  :: zpos        ! vertical particle position (eta), !FN-20140107
  REAL,         INTENT(INOUT)  :: vprime      ! last v turbulent velocity (y)
  REAL,         INTENT(INOUT)  :: wratio      ! last w turbulent velocity (z)
  REAL,         INTENT(INOUT)  :: uprime      ! last u turbulent velocity (x)
  INTEGER,      INTENT(IN)     :: hdwp        ! horizontal distribution index
  REAL,         INTENT(IN)     :: zndx        ! fractional vertical index  
  INTEGER,      INTENT(INOUT)  :: iseed       ! seed number for dispersion

!-------------------------------------------------------------------------------
! internally defined variables
!-------------------------------------------------------------------------------

  REAL                         :: dt                ! internal time step (sec)
  REAL                         :: zx                ! vertical index
  REAL                         :: sigu,sigv,sigw    ! velocity sigma
  INTEGER                      :: kz,kb,kt          ! vertical indicies
  REAL                         :: unew,vnew,wnew    ! new velocity components
  REAL                         :: rauto             ! velocity autocorrelation
  REAL                         :: wprime            ! vertical velocity turb  
  REAL                         :: delz              ! min vertical layer depth
  REAL                         :: delt              ! vertical time step 
  INTEGER                      :: knum              ! numb integration steps
  REAL                         :: fact              ! interp factor
  REAL                         :: vm                ! interp vertical mixing
  REAL                         :: sig2t,sig2b,sigdz ! velocity variances
  REAL                         :: zz,dist           ! temporary height variables
  integer       :: i,j

  !FN-20140107
  real                     :: detadt

!-------------------------------------------------------------------------------
! Configure subroutine interface argumment lists
!-------------------------------------------------------------------------------

  INTERFACE
    SUBROUTINE parvar (sigma,veloc,iseed) 
      IMPLICIT NONE
      REAL,         INTENT(IN )    :: sigma
      REAL,         INTENT(OUT)    :: veloc
      INTEGER,      INTENT(INOUT)  :: iseed
    END SUBROUTINE parvar
  END INTERFACE

!-------------------------------------------------------------------------------
! only compute dispersion for appropriate particle distributions (9/15/03)

  IF(HDWP.EQ.1.OR.HDWP.EQ.2.OR.HDWP.EQ.5.OR.HDWP.EQ.6) RETURN 

  !FN-20140515
  !DT=ABS(DTM*60.0)                    ! advection time step from min to sec
  DT=DTM                              ! advection time step in sec
  ZX=ZNDX                             ! initial particle position index

!-------------------------------------------------------------------------------
! horizontal turbulence (only with 3D particle when hdwp=0)
!-------------------------------------------------------------------------------

  IF(HDWP.EQ.0)THEN 
!    horizontal velocity standard deviations
     SIGU=SQRT(UMIX)
     SIGV=SQRT(VMIX)

!    call random number generator for each component
!    where prime defines the velocity standard deviation
     CALL PARVAR(SIGU,UNEW,ISEED)
     CALL PARVAR(SIGV,VNEW,ISEED)

!    autocorrelation function based upon time step and TL
     RAUTO=EXP(-DT/HSCALE)

!    compute new turbulent velocity component
     UPRIME=RAUTO*UPRIME+SQRT(1.0-RAUTO*RAUTO)*UNEW
     VPRIME=RAUTO*VPRIME+SQRT(1.0-RAUTO*RAUTO)*VNEW

!    adjust horizontal positions (m/s)(s)(grid/m)
     XPOS=XPOS+UPRIME*DT/GDISX
     YPOS=YPOS+VPRIME*DT/GDISY
  END IF

!-------------------------------------------------------------------------------
! vertical turbulence (only vertical particles hdwp = 0,3,4)
!-------------------------------------------------------------------------------


! determine the required vertical dispersion time step (sec)
! by computing the minimum time step for vertical stability
!FN-20131209
  DELZ=ABS(ZHGT(1)-ZHGT(2))
  DELT=MIN(DT,0.125*DELZ*DELZ/MAXVAL(WMIX)/VSCALE)
  DELT=MAX(1,NINT(DELT))

! round down time step to even multiple of DT
  DO WHILE (MOD(INT(DT),INT(DELT)).NE.0.AND.INT(DELT).GT.1)
     DELT=DELT-1.0
  END DO

! autocorrelation function based upon time step and TL
! delt replaced dt - RRD:1/3/00
  RAUTO=EXP(-DELT/VSCALE)

! go through iterations to match external DT
tstep : DO KNUM=1,NINT(DT/DELT) !FN-20140107

!    layer indicies
     KZ=MAX(1, MIN(NLVL, INT(ZX)))
     KT=MIN(KZ+1, NLVL)
     KB=KT-1

!    mixing at particle position
     FACT=MAX(0.0, MIN(1.0,(ZSG(KB)-ZPOS)/(ZSG(KB)-ZSG(KT))))
     VM=(WMIX(KT)-WMIX(KB))*FACT+WMIX(KB)

!    vertical velocity standard deviation (m/s)
     SIGW=SQRT(VM)
     CALL PARVAR(SIGW,WNEW,ISEED)

!    gradient of velocity variance
     SIG2T=SQRT(WMIX(KT))
     SIG2B=SQRT(WMIX(KB))
     !FN-20131209
     SIGDZ=(SIG2T-SIG2B) / (ZHGT(KB)-ZHGT(KT))
!    new turbulent vertical velocity
!    WPRIME=RAUTO*WPRIME+SQRT(1.0-RAUTO*RAUTO)*WNEW
!    low turbulence convergence correction
!    WPRIME=WPRIME+VSCALE*(1.0-RAUTO)*SIGDZ

!    correction for non-homogeneous turbulence: Chock and Winkler (1994)
!    J. Geophys. Res., Vol 99, No D1, Pages 1033-1041

     WPRIME=WRATIO*SIGW
     WRATIO=RAUTO*WRATIO+SQRT(1.0-RAUTO*RAUTO)*WNEW/SIGW  &
           +VSCALE*(1.0-RAUTO)*SIGDZ
     WPRIME=WRATIO*(SIGW+(WPRIME*DELT*SIGDZ))

    !FN-20140107
    !Compute d(eta)/dt = d(eta)/dz * dz/dt
    !                  = (1/mu) * ((P(z2)-P(k1))/(z2-z1)) * WPRIME
    !change pressure unit from mb to Pa
     DETADT = (1.0/TOMU)*(((PRES(KT)-PRES(KB))*100.)/(ZHGT(KT)-ZHGT(KB)))*WPRIME

!FN-20140107, vertical displacement => eta
     ZPOS=ZPOS-(DETADT*DELT)

!    full reflection at the bottom
     IF(ZPOS.GT.1.0)THEN
        ZPOS=2.0-ZPOS
!       reverse turbulent component
        WRATIO=-WRATIO
     END IF

!    full reflection at the top
     IF(ZPOS.LT.ZSG(NLVL))THEN
        ZPOS=2.0*ZSG(NLVL)-ZPOS
        WRATIO=-WRATIO
     END IF

!FN-20140107
     ZPOS = MIN(1.0,MAX(0.0,ZPOS))

!FN-20140318
     CALL eta2zx(ZPOS,ZX)

  END DO tstep

END SUBROUTINE pardsp
