!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  PRFCOM           PRoFile COMmon driver for input meteo
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   PROFILE COMMON IS THE COMMON DRIVER FOR THE FOUR METEOROLGICAL
!   PROFILE ANALYSIS PROGRAMS.  ROUTINE EXAMINES THE PROFILE AT EACH
!   SUB-GRID NODE AND CONVERTS THE INPUT DATA TO COMMON UNITS AND
!   INTERPOLATES THE SOUNDING TO THE INTERNAL MODEL SIGMA SYSTEM.
!   THE DIFFERENT INPUT DATA COORDINATE SYSTEMS ARE SUPPORTED FOR
!   CONVERSION: ABSOLUTE PRESSURE (PRFPRS), SIGMA PRESSURE (PRFSIG),
!   TERRAIN FOLLOWING SIGMA (PRFTER), AND ECMWF HYBRID.  AFTER UNITS
!   CONVERSION THE STABILITY ANALYSIS ROUTINES ARE CALLED AT EACH POINT.
!
! PROGRAM HISTORY LOG:
!   LAST REVISION: ...
!                  19 Nov 2013 (FN) - modify for eta coordinate
!                  28 Mar 2014 (FN) - clean up and rename variables
!                  10 Jun 2014 (FN) - bring in vscale & hscale
!
!$$$

SUBROUTINE PRFCOM(VSCALE,HSCALE,TKERD,TKERN,KSFC,XGX,XGY,XZ0,XZT,  &
                  VMIX,KMIXD,KMIX0,XZI,XP0,XT0,XU0,XV0,       &
                  XUF,XVF,XHF,XSF,XSS,XXU,XXV,XXW,            &
                  XXA,XXT,XXQ,XXP,XLVLZZ,XXE,XXH,XXX,         &
                  ids, ide, jds, jde, kds, kde,    &
                  ims, ime, jms, jme, kms, kme,    &
                  ips, ipe, jps, jpe, kps, kpe    )

  IMPLICIT NONE

  INCLUDE 'DEFGRID.INC' ! meteorological array 
  INCLUDE 'DEFARG4.INC' ! interface statements for local subroutines

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  REAL,    INTENT(IN)    :: vscale     ! vertical Lagrangian time scale (sec)  !FN-20140610
  REAL,    INTENT(IN)    :: hscale     ! horizontal Lagrangian time scale (sec)!FN-20140610
  REAL,    INTENT(IN)    :: tkerd      ! day turbulent kinetic eneregy ratio
  REAL,    INTENT(IN)    :: tkern      ! night turbulent kinetic eneregy ratio
  INTEGER, INTENT(IN)    :: ksfc       ! index top of the sfc layer
  LOGICAL, INTENT(IN)    :: vmix       ! indicator for mixing computation
  INTEGER, INTENT(IN)    :: kmixd      ! mixed layer depth options
  INTEGER, INTENT(IN)    :: kmix0      ! minimum mixing depth

  REAL, DIMENSION(ims:ime,jms:jme), INTENT(IN)   :: xgx,  &
                                                    xgy,  &
                                                    xz0,  &
                                                    xzt

  REAL, DIMENSION(ims:ime,jms:jme), INTENT(INOUT) :: xzi,               &
                                                     xp0,xt0,xu0,xv0,   &
                                                     xuf,xvf,xsf,xss
  REAL, DIMENSION(ims:ime,jms:jme), INTENT(IN)    :: xhf

  REAL, DIMENSION(ims:ime,kms:kme,jms:jme), INTENT(INOUT) :: xxu,xxv,xxw,       &
                                                             xxa,xxt,xxq,xxp,   &
                                                             xxe,xxh,xxx,xlvlzz

  INTEGER, INTENT(IN)    ::  ids , ide , jds , jde , kds , kde , &
                             ims , ime , jms , jme , kms , kme , &
                             ips , ipe , jps , jpe , kps , kpe


!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  REAL,    PARAMETER :: grav  = 9.80616    ! gravity (m/s2)
  REAL,    PARAMETER :: rdry  = 287.04     ! dry air (J/Kg-K)
  REAL,    PARAMETER :: p2jm  = 100.0      ! mb to j/m3
  REAL               :: qtmp,atmp,ttmp,ptmp,tvtmp,esat

  REAL, ALLOCATABLE :: uvar(:),vvar(:),wvar(:),tvar(:),qvar(:),pvar(:)
  REAL, ALLOCATABLE :: avar(:),zvar(:),dvar(:),xvar(:),hvar(:),evar(:)

  INTEGER           :: i,j,k,kret,kzmix,kbls,ktop

  REAL              :: offset      ! pressure offset for NMM
  REAL              :: zmdlt,gspdx,gspdy,hbot,xbot  
  REAL              :: pblh,ebot,tvmix

  LOGICAL           :: mixd,hflx,eflx,uflx,ustx,tstx,tken,velv
  LOGICAL           :: zflg,qflg,uflg,tflg,prss,shgt,dzdt

!-------------------------------------------------------------------------------
! external variables
!-------------------------------------------------------------------------------

  COMMON /GBLGRD/ HYGD, DREC, HYFL

!-------------------------------------------------------------------------------

! place structure in temporary variables   

  KTOP = DREC%KTOP
  MIXD = DREC%MIXD
  ZFLG = DREC%ZFLG
  QFLG = DREC%QFLG
  UFLG = DREC%UFLG
  TFLG = DREC%TFLG
  PRSS = DREC%PRSS
  SHGT = DREC%SHGT
  EFLX = DREC%EFLX
  UFLX = DREC%UFLX
  HFLX = DREC%HFLX
  USTX = DREC%USTR
  TSTX = DREC%TSTR
  TKEN = DREC%TKEN
  VELV = DREC%VELV
  KBLS = DREC%KBLS
  DZDT = DREC%DZDT
  ZMDLT= DREC%ZMDLT
  KZMIX= DREC%KZMIX
  TVMIX= DREC%TVMIX

  print *,'yyy hysp/prfcom tkerd=',TKERD,TKERN,KSFC
  print *,'yyy mixd=',MIXD,KMIXD,KMIX0,KBLS
  print *,'yyy uflx=',HFLX,EFLX,UFLX,USTX,TSTX
  print *,'yyy size=',size(xxt,1),size(xxt,2),size(xxt,3)
  print *,'yyy jms=',jms,jme,jps,jpe

!-------------------------------------------------------------------------------
! allocate working arrays
!-------------------------------------------------------------------------------

  IF(.NOT.ALLOCATED(UVAR))THEN

     ALLOCATE (UVAR(kps:kme),VVAR(kps:kme),WVAR(kps:kme),TVAR(kps:kme),QVAR(kps:kme),PVAR(kps:kme),STAT=kret)
     IF(kret.NE.0)THEN
        WRITE(*,*)'*ERROR* prfcom: memory allocation - output' 
        STOP 900
     END IF

     ALLOCATE (EVAR(kps:kme),HVAR(kps:kme),XVAR(kps:kme),STAT=kret)
     IF(kret.NE.0)THEN
        WRITE(*,*)'*ERROR* prfcom: memory allocation - mixing' 
        STOP 900
     END IF
     EVAR=0.0
     HVAR=0.0
     XVAR=0.0

     ALLOCATE (AVAR(kps:kme),ZVAR(kps:kme),DVAR(kps:kme),STAT=kret)
     IF(kret.NE.0)THEN
        WRITE(*,*)'*ERROR* prfcom: memory allocation - diagnostic' 
        STOP 900
     END IF
  END IF

  OFFSET=HYGD%DUMMY

!-------------------------------------------------------------------------------
! process each node on subgrid
!-------------------------------------------------------------------------------

!FN-20141010, using memory domain instead of patch domain
  DO J=jms,jme
  DO I=ims,ime

!    wind speed conversion (m/s) to (grid/min)
     GSPDX=60.0/XGX(I,J)
     GSPDY=60.0/XGY(I,J)

!-------------------------------------------------------------------------------
!    get profiles for met variables
!-------------------------------------------------------------------------------

     DO K=kms,kme                      !FN-20140505, memory dimension

        QTMP=XXQ(I,K,J)
        PTMP=XXP(I,K,J)
        ATMP=XXA(I,K,J)
        TTMP=XXT(I,K,J)

        ESAT=EXP(21.4-(5351.0/ATMP))   !convert to rh fraction
        QVAR(K)=QTMP*PTMP/(0.622*ESAT)

        TVTMP=(1.0+0.61*QTMP)*ATMP     !FN-20140227, virtual ambient temperature
        DVAR(K)=P2JM*PTMP/(TVTMP*RDRY)

        PVAR(K)=XXP(I,K,J)
        TVAR(K)=XXT(I,K,J)
        AVAR(K)=XXA(I,K,J)

        UVAR(K)=XXU(I,K,J)
        VVAR(K)=XXV(I,K,J)
        WVAR(K)=XXW(I,K,J)
        ZVAR(K)=XLVLZZ(I,K,J)          !FN-20140122, lvlzz is at AGL (m)

     END DO

!-------------------------------------------------------------------------------
!    stability analysis for concentration simulations
!-------------------------------------------------------------------------------
     IF(VMIX)THEN  !FN-20140505, VMIX=TRUE

!       analyze surface stability (z/L) from fluxes or wind/temp profile
        CALL STBANL(KBLS,I,J,KSFC,KMIX0,KMIXD,MIXD,TKEN,HFLX,EFLX,UFLX,USTX,   &
                    TSTX,XZ0(I,J),XUF(I,J),XVF(I,J),XHF(I,J),kme,              &
                    UVAR,VVAR,TVAR,ZVAR,EVAR,DVAR,                             &
                    XZI(I,J),XSF(I,J),XSS(I,J))

        IF(DREC%KBLT.EQ.1)THEN
!          compute mixing Beljaars-Holtslag 
!          CALL STBSND(TKERD,TKERN,KZMIX,TVMIX,KSFC,NLVL,U2,V2,T2,Z2,E2,H2,X2)

        ELSEIF(DREC%KBLT.EQ.2)THEN    !FN-20140505, KBLT=2
!          compute mixing Kantha-Clayson 
           CALL STBVAR(VSCALE,HSCALE,TKERD,TKERN,KZMIX,TVMIX,KSFC,kme,          &
                       UVAR,VVAR,TVAR,ZVAR,EVAR,HVAR,XVAR)

        ELSEIF(DREC%KBLT.EQ.3)THEN
!          velocity variances from input TKE
!          CALL STBTKE(TKERD,TKERN,KZMIX,TVMIX,KSFC,NLVL,U2,V2,   Z2,E2,H2,X2)

        ELSEIF(DREC%KBLT.EQ.4)THEN
!          velocity variances from input meteorological data file
!          no processing is required
           CONTINUE

        ELSE
           WRITE(*,*)'*ERROR* prfcom: turbulence method undefined'
           WRITE(*,*)' Unknown selection: ',DREC%KBLT
           STOP 900
        END IF

!       component friction velocities (reverse of 10m wind vectors)
        XUF(I,J)=-XUF(I,J)*GSPDX
        XVF(I,J)=-XVF(I,J)*GSPDY

     END IF

!-------------------------------------------------------------------------------
!    put adjusted data back into 3d array now interpolated to internal grid
!-------------------------------------------------------------------------------
     dataout : DO K=kps,kme                      !FN-20141010, memory dimension
        XXU(I,K,J)=UVAR(K)*GSPDX                 ! wind speed to grid / min
        XXV(I,K,J)=VVAR(K)*GSPDY
        XXW(I,K,J)=WVAR(K)*60.0                  ! convert dp/dt to per minute
        XXT(I,K,J)=TVAR(K)                       ! converted to potential
        XXA(I,K,J)=AVAR(K)                       ! local ambient temperature
        XXQ(I,K,J)=MAX(0.0,MIN(1.0,QVAR(K)))     ! humidity is relative fraction
        XXP(I,K,J)=PVAR(K)                       ! local pressure (mb)

        IF(.NOT.VMIX) CYCLE dataout
        XXH(I,K,J)=MAX(0.0,HVAR(K))              ! U component turbulence (m2/s2)
        XXE(I,K,J)=MAX(0.0,EVAR(K))              ! V component turbulence (m2/s2) 
        XXX(I,K,J)=MAX(0.0,XVAR(K))              ! W component turbulence (m2/s2)
     END DO dataout

     !FN-20150520, print out profilers
     if (i.eq.ide/2 .and. j.eq.jde/2) then 
        print *,'yyy ii=',i,j,gspdx,gspdy
        print *,'yyy pblh=',xzi(i,j),ksfc,xuf(i,j),xhf(i,j)
        print *,'yyy x2=',xvar
        print *,'yyy e2=',evar
        print *,'yyy h2=',hvar
     endif
 
  END DO     
  END DO     

!-------------------------------------------------------------------------------
! Vertical index at which the data extrapolation region starts. Zero indicates 
! that no extrapolation region was found and the top index will be set to NLVL.
!-------------------------------------------------------------------------------
  IF(KTOP.EQ.0)KTOP=kme
  DREC%KTOP=KTOP

  print *,'yyy ktop=',ktop,vmix,DREC%KDEF

!-------------------------------------------------------------------------------
! Deformation method can be used to compute the horizontal mixing 
!-------------------------------------------------------------------------------

! IF(VMIX.AND.(DREC(KG,KT)%KDEF.EQ.1)) CALL STBHOR(NXS,NYS,NLVL,GX,GY,U,V,H,E)

print *,'yyy end of prfcom => return back to metdum!'

END SUBROUTINE prfcom
