!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  STBANL           STaBility ANaLysis from meteo profiles
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   STABILITY ANALYSIS FROM MODEL SURFACE FLUX PARAMETERS
!   OR FROM PROFILE IF THOSE OUTPUTS ARE NOT AVAILABLE.
!   FOR COMPUTATIONAL PURPOSES LEVEL 2 IS THE TOP OF SURFACE
!   SURFACE LAYER.  ROUTINE RETURNS THE HEIGHT NORMALIZED
!   MONIN-OBUKHOV LENGTH, FRICTION TERMS, AND MIXED LAYER DEPTH.
!
! PROGRAM HISTORY LOG:
!   Last Revision: ...
!                  03 Oct 2012 (FN) - WRF-HYSPLIT coupling initial implementation
!                  28 Mar 2014 (FN) - clean up and rename variables
!
! USAGE:  CALL STBANL(KBLS,I,J,KS,KMIX0,KMIXD,MIXD,TKEN,HFLX,EFLX,UFLX,USTAR,
!                     TSTAR,Z0T,FMU,FMV,FHS,NL,UVAR,VVAR,TVAR,ZVAR,EVAR,DVAR,
!                     MIXH,PSIR,SSP)
!
!   INPUT ARGUMENT LIST:      see below
!   OUTPUT ARGUMENT LIST:     see below
!   INPUT FILES:              none
!   OUTPUT FILES:             none
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE STBANL(KBLS,I,J,KS,KMIX0,KMIXD,MIXD,TKEN,HFLX,EFLX,UFLX,USTAR, &
                  TSTAR,Z0T,FMU,FMV,FHS,NL,UVAR,VVAR,TVAR,ZVAR,EVAR,DVAR, &
                  MIXH,PSIR,SSP)

  USE funits
  USE stbcon

  IMPLICIT NONE

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  INTEGER,   INTENT(IN)    :: kbls         ! stability method  
  INTEGER,   INTENT(IN)    :: i,j          ! horizontal index of processed pt
  INTEGER,   INTENT(IN)    :: ks           ! index top of surface layer
  INTEGER,   INTENT(IN)    :: kmix0        ! minimum mixing depth
  INTEGER,   INTENT(IN)    :: kmixd        ! mixed layer depth options
  LOGICAL,   INTENT(IN)    :: mixd         ! mixed layer depth in data file
  LOGICAL,   INTENT(IN)    :: tken         ! tke field avaialble   
  LOGICAL,   INTENT(IN)    :: eflx         ! momentum flux as scalar exchange
  LOGICAL,   INTENT(IN)    :: uflx         ! fluxes available (momentum)
  LOGICAL,   INTENT(IN)    :: hflx         ! fluxes available (heat)      
  LOGICAL,   INTENT(IN)    :: ustar,tstar  ! friction veloc and temp available
  REAL,      INTENT(IN)    :: z0t          ! roughness length (m)

                           ! Variable contains momentum flux:
                           ! exchange coefficient (kg/m2-s) when EFLX=TRUE
                           ! friction velocity (m/s)        when USTAR=TRUE
  REAL,      INTENT(INOUT) :: fmu          ! u-component momentum flux (N/m2) 
  REAL,      INTENT(INOUT) :: fmv          ! v-component momentum flux (N/m2) 

                           ! Variable contains sensible heat flux:
                           ! friction temperature           when TSTAR=TRUE
  REAL,      INTENT(IN)    :: fhs          ! sensible heat flux (W/m2)

  INTEGER,   INTENT(IN)    :: nl           ! number of output sigma levels
  REAL,      INTENT(IN)    :: uvar(:)      ! horizontal wind component (m/s)
  REAL,      INTENT(IN)    :: vvar(:)      ! horizontal wind component (m/s)
  REAL,      INTENT(IN)    :: tvar(:)      ! virtual potential temp (deg K)
  REAL,      INTENT(INOUT) :: zvar(:)      ! height at levels (m)
  REAL,      INTENT(IN)    :: evar(:)      ! turbulent kinetic energy (J/kg)
  REAL,      INTENT(IN)    :: dvar(:)      ! air density (kg/m3)
  REAL,      INTENT(INOUT) :: mixh         ! mixed layer depth (m)
  REAL,      INTENT(OUT)   :: psir         ! integrated stability function heat
  REAL,      INTENT(OUT)   :: ssp          ! static stability parameter (1/s2) 

  LOGICAL    :: zflg
  REAL       :: tfhs,phim,phih,etrm,delt,delu,delz,tbar,ubar
  REAL       :: s,z0h,ri,dz,t,v,zl 
  INTEGER    :: k,kdat 

!                             stability analysis results each grid point
  REAL       :: fvel         ! scalar friction velocity (m/s)
  REAL       :: ustr         ! u- friction velocity (m/s)
  REAL       :: vstr         ! v- friction velocity (m/s)
  REAL       :: tstr         ! friction temperature (deg K)
  REAL       :: wstr         ! convective velocity scale (m/s)
  REAL       :: slen         ! Obukhov stability length (m)
  REAL       :: zmix         ! mixed layer depth (m)
  REAL       :: psi          ! integrated stability function heat

  COMMON /stbcom/ fvel,ustr,vstr,tstr,wstr,slen,zmix,psi

!-------------------------------------------------------------------------------
! analyze sounding to exclude extrapolated data
! go up the sounding to find the first level of real data
! prf??? routines set height to -height if internal model levels
! are below the first data level.  Prevents using interpolated
! levels for stability analysis
!-------------------------------------------------------------------------------

  KDAT=1
  DO WHILE (ZVAR(KDAT).LT.0.0)
!    convert back to positive value and save index
     ZVAR(KDAT)=-ZVAR(KDAT)
     KDAT=KDAT+1
  END DO

  IF(KDAT.GE.NL)THEN
     print *,'Error: KDAT > NL'
    !FN-1003, fail when doing parallel 
    !WRITE(*,*)   'Error: stbanl - no observed data, see MESSAGE file'
    !WRITE(*,*)'Error: stbanl - no observed data to match internal levels'
    !WRITE(*,*)'Number of vertical levels (internal, data): ',NL,KDAT
    !WRITE(*,*)'At horizontal grid position (i,j): ',I,J 
    !WRITE(*,*)'Level   Height  Temperature'
    !DO K=NL,1,-1
    !   WRITE(*,*)K,ZZ(K),TT(K)
    !END DO
    !STOP 900
  END IF

! level used as top point for sfc layer stability calculations
  KDAT=MAX(2,KS,KDAT)

!-------------------------------------------------------------------------------
! determine mixed layer depth
! pblh - input and output, initially comes from meteo data file if available     
! zmix - internally computed value depends upon kmixd, saved in common block
!-------------------------------------------------------------------------------

  IF(MIXD.AND.KMIXD.EQ.0)THEN
!    mixed layer depth from meteo file
     ZMIX=MAX(MIXH,FLOAT(KMIX0))

  ELSEIF(TKEN.AND.KMIXD.EQ.2)THEN
!    use tke profile to determine mixed layer depth
     K=1
     ZMIX=ZVAR(NL)
     DO WHILE (K.LT.NL)
!       mixed layer when tke drops a factor of two or reaches 0.21
        IF((2.0*EVAR(K+1).LT.EVAR(K)).OR.(EVAR(K+1).LE.0.21))THEN
           ZMIX=ZVAR(K)
           K=NL
        END IF
        K=K+1
     END DO

  ELSEIF(KMIXD.GE.10)THEN
!    special option where mixing depth defined in namelist file
     ZMIX=KMIXD

  ELSE
!    for none of the above or kmixd=1
!    find mixed layer depth as the height T(Zi) > 2 + Tmin
     ZMIX=ZVAR(NL)
     ZFLG=.FALSE.
     TBAR=MINVAL(TVAR)
     DO K=NL,1,-1
        IF(.NOT.ZFLG)ZMIX=ZVAR(K)
        IF(.NOT.ZFLG.AND.TVAR(K).LE.TBAR+2.0)ZFLG=.TRUE.
     END DO
  END IF

! minimum mixed layer height (day or night)
  ZMIX=MAX(ZMIX,FLOAT(KMIX0))

!-------------------------------------------------------------------------------
! determine z/L from surface flux fields
!-------------------------------------------------------------------------------

  IF((KBLS.EQ.1).AND.(USTAR.OR.UFLX.OR.EFLX).AND.(TSTAR.OR.HFLX))THEN
     IF(USTAR)THEN
!       wind speed
        UBAR=SQRT(UVAR(KDAT)*UVAR(KDAT)+VVAR(KDAT)*VVAR(KDAT))
!       friction velocity available as input field
        FVEL=MAX(FMU,0.0001)
!       define vector components to give the same scalar
        USTR=FVEL*UVAR(KDAT)/UBAR
        VSTR=FVEL*VVAR(KDAT)/UBAR

     ELSEIF(UFLX)THEN
!       both u and v momentum fluxes are available
        USTR=SIGN(MAX(0.01,SQRT(ABS(FMU)/DVAR(KDAT))),FMU)
        VSTR=SIGN(MAX(0.01,SQRT(ABS(FMV)/DVAR(KDAT))),FMV)
!       scalar friction velocity
        FVEL=SQRT(USTR*USTR+VSTR*VSTR)

     ELSE
!       only scalar exchange coefficient remains available
        USTR=MAX(0.01,SQRT(ABS(FMU)*ABS(UVAR(KDAT))/DVAR(KDAT)))
        VSTR=MAX(0.01,SQRT(ABS(FMU)*ABS(VVAR(KDAT))/DVAR(KDAT)))
        USTR=SIGN(USTR,UVAR(KDAT))
        VSTR=SIGN(VSTR,VVAR(KDAT))
!       scalar friction velocity
        FVEL=SQRT(USTR*USTR+VSTR*VSTR)
     END IF

     IF(TSTAR)THEN
!       friction temperature available
        TSTR=SIGN(MAX(0.0001,ABS(FHS)),FHS)
     ELSE
!       friction temperature from sensible heat flux
        TFHS=MAX(-25.0,FHS)
        TSTR=SIGN(MAX(0.0001,ABS(TFHS/(DVAR(KS)*CP*FVEL))),-TFHS)
     END IF

!    normalized Monin-Obukhov length (for top of sfc layer)
     ZL=ZVAR(KS)*VONK*GRAV*TSTR/(FVEL*FVEL*TVAR(KS))

!-------------------------------------------------------------------------------
! determine z/L from wind and temperature sounding
!-------------------------------------------------------------------------------

  ELSE
!    Bulk Richardson number uses only non-extrapolated levels
     TBAR=(TVAR(KDAT)+TVAR(1))/2.0
     DELZ=ZVAR(KDAT)-ZVAR(1)
     DELT=TVAR(KDAT)-TVAR(1)
     DELU=(UVAR(KDAT)-UVAR(1))**2+(VVAR(KDAT)-VVAR(1))**2
     DELU=MAX(0.0001,DELU)
     RI=GRAV*DELT*DELZ/TBAR/DELU

!    correction for excessive height (see Golder Ri~Z^2)
!    when data level above assumed surface layer top
     DZ=ZVAR(KDAT)*ZVAR(KDAT)/ZVAR(KS)/ZVAR(KS)
     RI=MAX(-1.0, MIN(RI/DZ,1.0))

!------------------------------------------------------------------------------
! convert bulk Ri to z/L at level KS using Hess formula
!-------------------------------------------------------------------------------

     S=LOG(ZVAR(KS)/Z0T+1.0)

!    roughness length for heat
     Z0H=0.1*Z0T
     T=LOG(ZVAR(KS)/Z0H +1.0)
     V=LOG(Z0T/Z0H)

     IF(RI.GT.0.0.AND.RI.LT.0.08)THEN
        ZL=(-T+10.0*S*RI+SQRT(T*T-20.0*S*T*RI+20.0*S*S*RI))/(10.0*(1.0-5.0*RI))
     ELSEIF(RI.GE.0.08)THEN
        ZL=(B1*S+B2)*RI*RI+(B3*S-B4*V-B5)*RI
     ELSE
        ZL=RI*(S*S/T-B6)
     END IF
     ZL=SIGN(ZL,RI)

     IF(ZL.GE.0.0)THEN
!       stable surface layer Beljaars-Holtslag
        ETRM=B*EXP(-D*ZL)*(1.0-D*ZL+C)
        PHIM=1.0+(A+ETRM)*ZL
        PHIH=PRN*(1.0+(A*SQRT(1.0+A*B*ZL)+ETRM)*ZL)
     ELSE
!       unstable surface layer Betchov-Yaglom / Kadar-Perepelkin
        PHIM=((1.0+0.625*ZL*ZL)/(1.0-7.5*ZL))**0.3333
        PHIH=0.64*((3.0-2.5*ZL)/(1.0-10.0*ZL+50.0*ZL*ZL))**0.3333
     END IF

!    compute friction terms
     FVEL=VONK*ZVAR(KS)*SQRT(DELU)/PHIM/DELZ
     TSTR=VONK*ZVAR(KS)*DELT/PHIH/DELZ

!    recompute Z/L from friction terms to be consistent
     ZL=ZVAR(KS)*VONK*GRAV*TSTR/(FVEL*FVEL*TVAR(KS))

!    define vector components to give the same scalar
     USTR=0.707107*FVEL
     VSTR=USTR
  END IF

!-------------------------------------------------------------------------------
! check limits on Z/L (-2 <= z/L <= 15 )
!-------------------------------------------------------------------------------

  ZL=MAX(-2.0, MIN(15.0, ZL))
  IF(ZL.EQ.0.0)ZL=0.001
  SLEN=ZVAR(KS)/ZL

!-------------------------------------------------------------------------------
! compute integral (psi) required for deposition calculations
!-------------------------------------------------------------------------------

  IF(ZL.LT.-0.001)THEN
!    unstable
     PSI=P1+ZL*(P2+ZL*(P3+ZL*(P4+P5*ZL)))
  ELSEIF(ZL.GE.-0.001.AND.ZL.LT.0.0)THEN
!    neutral
     PSI=-2.7283*ZL
  ELSE
!    stable
     PSI=-(1.0+A*B*ZL)**1.5-B*(ZL-C/D)*EXP(-D*ZL)-B*C/D+1.0
  END IF

!-------------------------------------------------------------------------------
! compute convective velocity scale
!-------------------------------------------------------------------------------

  IF(ZL.GE.0.0)THEN
     WSTR=0.0
  ELSE
     WSTR=ABS(GRAV*FVEL*TSTR*ZMIX/TVAR(KS))**0.3333
  END IF

!-------------------------------------------------------------------------------
! low-level static stability parameter
!-------------------------------------------------------------------------------

  TBAR=(TVAR(KDAT)+TVAR(1))/2.0
  DELZ=ZVAR(KDAT)-ZVAR(1)
  DELT=TVAR(KDAT)-TVAR(1)
  SSP=GRAV*DELT/TBAR/DELZ

!-------------------------------------------------------------------------------
! remaining variables to return
!-------------------------------------------------------------------------------

  FMU=USTR
  FMV=VSTR
  PSIR=PSI
  MIXH=ZMIX

END SUBROUTINE stbanl
