!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  EMSIZE           EMission SIZE distribution
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:10-09-08
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   Creates the mass distribution for a range of particle sizes given 
!   just a few points within the distribution. We assume that dV/d(log R)
!   is linear between the defined points for an increasing cumulative 
!   mass distribution with respect to particle radius. The input points
!   should be sorted by increasing particle size within the array. The
!   routine is invoked by setting NBPTYP (number of bins per type)
!   to a value greater than one. The polluant type index is then stored
!   as the previous (index * 1000) + (the redistributed index). 
!
! PROGRAM HISTORY LOG:
!   LAST REVISED: 08 Sep 2010 (RRD) - initial version
!
! USAGE:  CALL EMSIZE(DIRT,NUMTYP,MAXTYP)
!
!   INPUT ARGUMENT LIST:     see below
!   OUTPUT ARGUMENT LIST:    see below
!   INPUT FILES:             unit 5 or unit KF21 if input from file CONTROL
!   OUTPUT FILES:            unit KF22 if input from unit 5
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

SUBROUTINE EMSIZE(DIRT,NUMTYP,NBPTYP)

  USE funits

  IMPLICIT NONE

  INCLUDE 'DEFCONC.INC'         ! pollutant and concentration grid

!-------------------------------------------------------------------------------
! argument list variables
!-------------------------------------------------------------------------------

  TYPE(pset),    INTENT(INOUT) :: dirt(:)   ! for each pollutant type 
  INTEGER,       INTENT(IN)    :: numtyp    ! number of pollutant types
  INTEGER,       INTENT(IN)    :: nbptyp    ! number bins per type      

!-------------------------------------------------------------------------------
! internal variables
!-------------------------------------------------------------------------------

  REAL(4), ALLOCATABLE :: cmass(:)          ! cumulative mass distribution 

  REAL(4)    :: delr,delv,prad,vsum,diam
  INTEGER(4) :: j,k,n,m

! the number of defined bins in the control file
  IF(.NOT.ALLOCATED(cmass)) ALLOCATE (cmass(numtyp))

! compute the cumulative mass distribution
  vsum=0.0
  diam=0.0

  j=0
  DO n=1,numtyp
     cmass(n)=0.0
     IF(DIRT(N)%PDIAM.GT.0.0)THEN
        IF(j.EQ.0)j=n
        vsum=vsum+DIRT(N)%QRATE
        cmass(n)=vsum
!       particles should be ordered by increasing size
!       in the orignal definition in the CONTROL file
        IF(DIRT(N)%PDIAM.LE.DIAM)THEN
           WRITE(KF21,*)'*ERROR* emsize: particle size should increase with index'
           WRITE(*,*)   '*ERROR* emsize: particle size should increase with index'
           STOP 900
        END IF
        DIAM=DIRT(N)%PDIAM
     END IF
  END DO
  WRITE(KF21,*)'----------------- Particle Redistribution ---------------------'
  WRITE(KF21,*)'  pollutant         bin       index    emission            size'

! create the new distribution starting at array space numtyp+1
  k=numtyp

! go through each old bin
  DO n=1,numtyp

     IF(dirt(n)%pdiam.GT.0.0)THEN
!       compute change in mass and radius between adjacent bins
        IF(n.EQ.j)THEN
           delr=ALOG(dirt(n+1)%pdiam)-ALOG(dirt(n)%pdiam)
           delv=cmass(n)
        ELSE
           delr=ALOG(dirt(n)%pdiam)-ALOG(dirt(n-1)%pdiam)
           delv=cmass(n)-cmass(n-1)
        END IF

!       delta mass of new bins
        delv=delv/nbptyp
!       delta radius of new bins 
        delr=delr/nbptyp

!       compute radius below the first sub-divisional bin
        prad=ALOG(dirt(n)%pdiam)-delr*(nbptyp+1.0)/2.0
        diam=EXP(prad)

     ELSE
!       when gases also defined 
        delv=DIRT(N)%QRATE/nbptyp
        diam=0.0
     END IF

!    add mass for each new sub-divisional bin
     DO m=1,nbptyp  
        k=k+1

!       fill new array properties from original bin
        dirt(k)=dirt(n) 

!       replace values specific to new bins 
        dirt(k)%qrate=delv   
        IF(diam.EQ.0.0)THEN
           dirt(k)%pdiam=0.0
        ELSE
           dirt(k)%pdiam=EXP(prad)
        END IF

!       increment size for next new bin
        prad=prad+delr

!       diagnostic output to message file
        WRITE(KF21,*)n,m,(k-numtyp),dirt(k)%qrate,dirt(k)%pdiam
     END DO

  END DO
  WRITE(KF21,*)'---------------------------------------------------------------'

! reorder the array so that the new distribution starts at index=1

  DO n=1,(nbptyp*numtyp)
     dirt(n)=dirt(numtyp+n)
  END DO
 
  DEALLOCATE (cmass)

END SUBROUTINE emsize 
