!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:  METPOS           METeorological POSitioning finds record
!   PRGMMR:    ROLAND DRAXLER   ORG: R/ARL       DATE:96-06-01
!
! ABSTRACT:  THIS CODE WRITTEN AT THE AIR RESOURCES LABORATORY ...
!   METEOROLOGICAL POSITIONING ROUTINE USES THE CURRENT CALCULATION
!   POSITION AND TIME TO CHECK IF THE POSITION FALLS WITHIN THE METEO
!   DATA ALREADY LOADED INTO MEMORY. IF NOT IS IS DETERMINED IF DATA
!   ON A NEW GRID OR TIME ARE REQUIRED. THE ROUTINE RETURNS A POSITIVE 
!   RECORD NUMBER IF DATA ARE TO BE READ FROM THE INPUT FILE.
!
! PROGRAM HISTORY LOG:
!   LAST REVISION: ...
!                  06 Aug 2012 (FN) - WRF-HYSPLIT coupling initial implementation
!                  28 Aug 2013 (FN) - set up mtime
!
! USAGE:  CALL METPOS(BACK,XP,YP,JET,NGRD,NTIM,FTIME,MTIME,POINT,OFFG,
!                     KGC,KGX,KT1,KT2)
!
!   INPUT ARGUMENT LIST:     see below
!   OUTPUT ARGUMENT LIST:    see below
!   INPUT FILES:             none
!   OUTPUT FILES:            none
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!   MACHINE:  IBM RS6000
!
!$$$

 SUBROUTINE METPOS(BACK,XP,YP,JET,NGRD,NTIM,FTIME,MTIME,POINT,OFFG,   &
                   KGC,KGX,KT1,KT2)

  USE funits

  IMPLICIT NONE

  INCLUDE 'DEFGRID.INC' ! meteorology grid and file

!-------------------------------------------------------------------------------
! argument list definitions
!-------------------------------------------------------------------------------

  LOGICAL,   INTENT(IN)    :: back           ! defines backward integration
  REAL,      INTENT(INOUT) :: xp,yp          ! x,y grid particle position
  INTEGER,   INTENT(IN)    :: jet            ! elapsed time (minutes)
  INTEGER,   INTENT(IN)    :: ngrd           ! number of meteo grids   
  INTEGER,   INTENT(IN)    :: ntim           ! number of meteo times   
  INTEGER,   INTENT(IN)    :: ftime(:,:)     ! time of current array data

  INTEGER,   INTENT(OUT)   :: mtime (2)      ! time of requested input data
  INTEGER,   INTENT(OUT)   :: point (2)      ! index pointer to array 
  LOGICAL,   INTENT(OUT)   :: offg           ! off entire grid flag
  INTEGER,   INTENT(IN)    :: kgc            ! current position grid number 
  INTEGER,   INTENT(OUT)   :: kgx            ! new position grid number 
  INTEGER,   INTENT(OUT)   :: kt1            ! grid number time last    
  INTEGER,   INTENT(OUT)   :: kt2            ! grid number time next    

!-------------------------------------------------------------------------------
! internal definitions
!-------------------------------------------------------------------------------

  LOGICAL :: diag = .true.
  LOGICAL :: met1, met2
  REAL    :: tlat,tlon,xpt(mgrd),ypt(mgrd)
  INTEGER :: ii,jj,kg,kt,k1,k2

!-------------------------------------------------------------------------------
! external definitions
!-------------------------------------------------------------------------------

  COMMON /GBLGRD/ HYGD, DREC, HYFL
  SAVE DIAG

!-------------------------------------------------------------------------------
! First determine which meteorological grid is the optimum advection grid.
! Assume that the finest grid is always defined as #1 and particles will be 
! advected on that grid if possible. The search is required each entry because 
! a particle may move from a coarser grid to a finer grid.  Time is defined
! such that the particle moves away from index #1 toward index #2.
!-------------------------------------------------------------------------------

  MET1  = .FALSE. ! time flag if outside of temporal domain
  MET2  = .FALSE.
  OFFG  = .TRUE.  ! start with the assumption that particle within domain
  KGX=0           ! grid selection

  gloop : DO KG=NGRD,1,-1

!    initialize time pointers for each grid
     K1=0  
     K2=0

     tloop : DO KT=1,NTIM

!       check if this time period contains a valid grid
        IF(HYGD(KG,KT)%NUMBER.LT.0) CYCLE tloop              !FN-0806,NUMBER was set to 1

!       if test grid not equal to current then remap position
        IF(KG.NE.KGC)THEN  

!          for multiple grids convert position from current grid to true
!          then convert true back to new grid coordinates
           IF(HYGD(KGC,1)%LATLON.AND.HYGD(KG,1)%LATLON)THEN
              CALL GBL2LL(KGC,1,XP,YP,TLAT,TLON)
              CALL GBL2XY(KG,KT,TLAT,TLON,XPT(KG),YPT(KG))
           ELSEIF(     HYGD(KGC,1)%LATLON.AND.(.NOT.HYGD(KG,1)%LATLON))THEN
              CALL GBL2LL(KGC,1,XP,YP,TLAT,TLON)
              CALL CLL2XY(HYGD(KG,KT)%GBASE,TLAT,TLON,XPT(KG),YPT(KG))
           ELSEIF(.NOT.HYGD(KGC,1)%LATLON.AND.      HYGD(KG,1)%LATLON )THEN
              CALL CXY2LL(HYGD(KGC,1)%GBASE,XP,YP,TLAT,TLON)
              CALL GBL2XY(KG,KT,TLAT,TLON,XPT(KG),YPT(KG))
           ELSEIF(.NOT.HYGD(KGC,1)%LATLON.AND.(.NOT.HYGD(KG,1)%LATLON))THEN
              CALL CXY2LL(HYGD(KGC,1)%GBASE,XP,YP,TLAT,TLON)
              CALL CLL2XY(HYGD(KG,KT)%GBASE,TLAT,TLON,XPT(KG),YPT(KG))
           END IF

!          convert particle position to index units
           II=INT(XPT(KG))
           JJ=INT(YPT(KG))

        ELSE
!          convert particle position to index units
           II=INT(XP)
           JJ=INT(YP)
        END IF

!       particle must be within the spatial domain of one of the files
!       within a 2 grid cell external band

        IF(.NOT.HYGD(KG,KT)%GBLDAT)THEN
           IF(II.LT.2.OR.II.GE.HYGD(KG,KT)%NX-2.OR.             &
              JJ.LT.2.OR.JJ.GE.HYGD(KG,KT)%NY-2)    CYCLE tloop
        END IF
        OFFG=.FALSE.

!       particle must be within temporal domain of one of the files
 
         !FN-20130828
         MTIME(1)=HYFL(KG,KT)%LAST%MACC   !previous met time (time step of K1 array)
         MTIME(2)=HYFL(KG,KT)%FIRST%MACC  !current met time (time step of K2 array)
         IF (MTIME(1).EQ.0) MTIME(1)=MTIME(2)

         !FN-0806, only one snapshot will be used, set K1 & K2 to KT(=1)
         K1=KT
         K2=KT

     END DO tloop

!    when both times are defined then the grid is valid for computations
     IF(K1.NE.0.AND.K2.NE.0)THEN
        KGX=KG
        KT1=K1
        KT2=K2
     END IF

  END DO gloop

!-------------------------------------------------------------------------------
! The computational point must have been on one of the meteorological grids
! such that the grid number KGX is not equal to zero.
!-------------------------------------------------------------------------------

  IF(KGX.EQ.0)THEN
!    terminate calculation if position no on any computational grid 
     IF(SUM(FTIME).EQ.0) THEN   
!       internal arrays still empty if data had not been previously loaded
        OFFG=.TRUE.
        RETURN
     ELSE
        OFFG=.TRUE.
        RETURN
     END IF

  ELSEIF(KGX.NE.KGC)THEN
!    position remapped to the new grid's coordinate system
     XP=XPT(KGX)
     YP=YPT(KGX)
  END IF

! advection point is on a meteorological grid
  OFFG=.FALSE.

  point(1)=1
  point(2)=2

RETURN

END SUBROUTINE metpos
