SUBROUTINE P10ADJ(P10F,ZWND,UBAR,RAIN,USTR,RATE,AREA,MASS,PGRD)

!---------------------------------------------------------------
! Original method algorithms based upon Draxler et al.,2001,
! Atm Environ,35:4315-4330, where control file defines dust
! with diameter (3.0 um), density (2.5 g/cc), shape (1.0)
! Control file can be configured with .\bdyfiles\dustbdy to 
! find all desert land-use points. 
!---------------------------------------------------------------
! Updated method using spatially varying threshold and soil
! texture coefficients which were created independently through
! special processing of MODIS monthly AOD values:
! 1) old format control file: lat lon hgt -rate area
! 2) new format control file: lat lon hgt +ustr (soil*area)
!---------------------------------------------------------------
! Last Revised: 01 Nov 1999
!               10 Apr 2000 (RRD) - emission adjustment factors
!               22 Mar 2002 (RRD) - updated links with hysplit
!               25 Oct 2005 (RRD) - threshold velocity factor
!               21 May 2009 (RRD) - new spatial varying method 
!               12 Jun 2009 (RRD) - added alternate models    
!               02 Sep 2009 (RRD) - simplified linear equation
!---------------------------------------------------------------

  IMPLICIT NONE

  REAL,   INTENT(IN)    :: P10F ! threshold emission sensitivity factor
  REAL,   INTENT(IN)    :: ZWND ! release height or ref meteo height
  REAL,   INTENT(IN)    :: UBAR ! wind speed in km/min
  REAL,   INTENT(IN)    :: RAIN ! precipitation accumulation
  REAL,   INTENT(IN)    :: USTR ! friction velocity in m/s
  REAL,   INTENT(INOUT) :: RATE ! emission rate or threshold velocity 
  REAL,   INTENT(INOUT) :: AREA ! emission area or soil texture coeff
  REAL,   INTENT(INOUT) :: MASS ! particle/puff mass
  INTEGER,INTENT(INOUT) :: PGRD ! grid identification index

  REAL    :: UT,FRIC,UVEL,ZLOG,RCD,FLUX,ROW,GRAV,VONK,TRSH
  INTEGER :: LT = 3 ! set all cells to one soil type (active sand sheet)

  REAL :: Z0NS(7)   ! flux equation roughess length
  REAL :: USTV(7)   ! flux equation friction velocity
  REAL :: TKON(7)   ! flux equation emission factor

! assigned roughness length (m) by land use class
  DATA Z0NS/.0002,.0004,.00002,.0005,.003,.00002,.0002/

! assigned threshold friction velocity (m/s) by land use class
! 1 = gravel lag 
! 2 = deflated sand sheet 
! 3 = active sand sheet 
! 4 = smooth sand sheet
! 5 = coastal plain deposits
! 6 = Playa deposits
! 7 = Covered desert floor
  DATA USTV/1.,0.622,0.28,0.685,3.49,3.,0.75/

! soil texture relates soil type to pm10 emission flux
  DATA TKON/2*5.6E-04,  5.6E-04,  4*5.6E-04/

! constants: density(g/m^3), gravity (m/s^2), VonKarman
  DATA       ROW/1225.0/,    GRAV/9.8/,       VONK/0.4/

! convert wind from km/min to m/sec
  UVEL=1000.0*UBAR/60.0
 
!---------------------------------------------------------------------
! negative rate  defines the old method of fixed threshold ...

  IF(RATE.LT.0.0)THEN

!    compute threshold wind speed from threshold friction velocity
     ZLOG=ALOG(ZWND/Z0NS(LT))/VONK
     UT=P10F*USTV(LT)*ZLOG

!    drag coefficient
     IF(UVEL.LE.UT)THEN  
        RCD=1.0/ZLOG 
     ELSE
        RCD=1.0/ZLOG+0.003*(1.0-UT/UVEL)
     END IF

!### friction velocity from meteorological model
!### FRIC=USTR

!    local friction velocity computed from windspeed
     FRIC=RCD*UVEL

!    local velocity (m/s) exceeds threshold velocity
     IF(UVEL.GT.UT)THEN
!       pm10 emission flux equations
!       Gillette equation: vertical mass flux (g m-2 sec-1)
!       FLUX=AMAX1(0.0,TKON(LT)*ROW*FRIC*(FRIC*FRIC-USTV(LT)*USTV(LT))/GRAV)

!       Eq. 14 - Westphal, Toon, & Carlson, 1987, J. Geophys Res, 92:3027-3049
!       FLUX[g/cm2-sec]=1.0E-14*FRIC[cm/s]**4.0
!       FLUX[g/m2-sec] =1.0E-14*(FRIC[m/s]*100.0)**4.0*1.0E+04

        FLUX=0.01*FRIC**4.0
!       max limit at 1 mg/m2/s
        FLUX=AMIN1(0.001,FLUX)

     ELSE
        FLUX=0.0
     END IF

!-----------------------------------------------------------------------
! new method rate defines varying threshold friction velocity 

  ELSE
     FRIC=USTR
     TRSH=RATE*P10F/100.0  ! control file cm/s -> m/s

!    friction velocity (cm/s) exceeds threshold friction velocity 
     IF(FRIC.GE.TRSH)THEN
!       pm10 emission flux equations
!       FLUX[g/m2-sec] = [soil emission factor]  US[m/s]
!###    FLUX=AREA*FRIC**4.0

!       alternate cubic equation
!###    FLUX=AREA*FRIC*(FRIC**2.0 - TRSH**2.0)

!       simplified linear equation (from dimensional analysis)
!       [g/m2-sec] = [g/m3]           [m/sec]
!       [FLUX    ] = [SOIL DENSITY]   [FRICTION VELOCITY] 
         FLUX      =  AREA          * (FRIC - TRSH)
     ELSE
        FLUX=0.0
     END IF

!    soil texture coefficient already multiplied by the area
     AREA=1.0
  END IF

!---------------------------------------------------------------------
! Precipitation adjustment

  IF(RAIN.GT.0.0)FLUX=0.0

! mass is set in emspnt to equal (dt/60)*(qtrm/npar)
! negative mass is the flag for a recent particle emission
! emission duration is not a factor
  IF(FLUX.GT.0.0)THEN
!    convert to mass = (s/h) (g/m2-s) (m2) (h/part)
     MASS=3600.0*FLUX*AREA*ABS(MASS)

  ELSE
!    zero emissions and index, particle removed in main
     MASS=0.0
!    particle removed in main when grid# is zero
     PGRD=0
  END IF

! zero turbulence variables used as temporary array space 
  RATE=0.0
  AREA=0.0
  
END SUBROUTINE p10adj
