import numpy as np

def val_to_index(cnLevels, nColors, val):
    """
    Converts a value to a color table index (0-based) according
    to the specified set of levels.

    If (val < cnLevels(0)) returns 0
    If (val >= cnLevels(nLevels-1)) returns nColors-1
    If (val == NaN) returns 0
    """

    from math import floor

    nLevels = len(cnLevels)
    nIntervals = nLevels - 1
    intervalSpan = float(nColors) / float(nIntervals+1) # +1 for end intervals

    if val < cnLevels[0]:
        return 0

    if val >= cnLevels[nLevels-1]:
        return nColors-1

    j = 0

    for i in range(nIntervals):
        if val >= cnLevels[i] and val < cnLevels[i+1]:
            d = cnLevels[i+1] - cnLevels[i]
            w1 = (cnLevels[i+1] - val) / d
            w2 = (val - cnLevels[i]) / d
            jfloat = w2 * float(i+1) + w1 * float(i)
            j = int(floor(intervalSpan * float(i+1)))   # +1 for left end iterval
            break

    return j


def fill_voronoi_cells(wks, map, cnLevels, nColors, nEdgesOnCell, lonVertex, latVertex, verticesOnCell, h, color_func=None):
    """
    Fills Voronoi cells in an MPAS mesh according to the value of a field

    """

    import Ngl
    import numpy as np

    maxEdges = verticesOnCell.shape[1]
    nCells = nEdgesOnCell.size

    xpoly = np.zeros((maxEdges))
    ypoly = np.zeros((maxEdges))

    pres = Ngl.Resources()
    for iCell in range(nCells):
        for i in range(nEdgesOnCell[iCell]):
            xpoly[i] = lonVertex[verticesOnCell[iCell,i]-1]
            ypoly[i] = latVertex[verticesOnCell[iCell,i]-1]
            if i > 0:
                if abs(xpoly[i] - xpoly[0]) > 180.0:
                   if xpoly[i] > xpoly[0]:
                       xpoly[i] = xpoly[i] - 360.0
                   else:
                       xpoly[i] = xpoly[i] + 360.0

        if color_func:
            pres.gsFillColor = color_func(iCell)
        else:
            pres.gsFillColor = val_to_index(cnLevels, nColors, h[iCell]) + 2

        Ngl.add_polygon(wks,map,xpoly[0:nEdgesOnCell[iCell]],ypoly[0:nEdgesOnCell[iCell]],pres)


def outline_voronoi_cells(wks, map, nEdgesOnCell, lonVertex, latVertex, verticesOnCell, line_thickness=1.0, line_color='black'):
    """
    Draws outines of Voronoi cells of an MPAS mesh

    """

    import Ngl
    import numpy as np

    maxEdges = verticesOnCell.shape[1]
    nCells = nEdgesOnCell.size

    xpoly = np.zeros((maxEdges))
    ypoly = np.zeros((maxEdges))

    pres = Ngl.Resources()
    pres.gsEdgesOn = True
    pres.gsEdgeColor = line_color
    pres.gsEdgeThicknessF = line_thickness
    pres.gsFillColor = 1
    pres.gsFillOpacityF = 0.0
    for iCell in range(nCells):
        for i in range(nEdgesOnCell[iCell]):
            xpoly[i] = lonVertex[verticesOnCell[iCell,i]-1]
            ypoly[i] = latVertex[verticesOnCell[iCell,i]-1]
            if i > 0:
                if abs(xpoly[i] - xpoly[0]) > 180.0:
                   if xpoly[i] > xpoly[0]:
                       xpoly[i] = xpoly[i] - 360.0
                   else:
                       xpoly[i] = xpoly[i] + 360.0

        Ngl.add_polygon(wks,map,xpoly[0:nEdgesOnCell[iCell]],ypoly[0:nEdgesOnCell[iCell]],pres)


def label_voronoi_cells(wks, map, latCell, lonCell, values, format_string=None, text_color='white', text_size=0.0025):
    """
    Draws outines of Voronoi cells of an MPAS mesh

    """

    import Ngl

    cres = Ngl.Resources()

    cres.txFontColor = text_color
    cres.txFontHeightF = text_size

    if format_string:
        for iCell in range(len(latCell)):
            Ngl.add_text(wks, map, format_string.format(values[iCell]), lonCell[iCell], latCell[iCell], cres)
    else:
        for iCell in range(len(latCell)):
            Ngl.add_text(wks, map, repr(values[iCell]), lonCell[iCell], latCell[iCell], cres)


def draw_label_bar(wks, cnLevels, nColors):
    """
    Draws a label bar

    """

    import Ngl
    import numpy as np

    nLevels = len(cnLevels)
    nIntervals = nLevels - 1

    xcb = np.zeros(4)
    ycb = np.zeros(4)

    tres = Ngl.Resources()
    pres = Ngl.Resources()

    tres.txAngleF = 90.0
    tres.txFontHeightF = 0.010
    tres.txJust = 'CenterRight'
    xoffset = 0.125
    barwidth = 0.75
    yoffset = 0.13
    barheight = 0.05
    intervalwidth = barwidth / float(nIntervals)
    labelInterval = 2

    xcb[0] = xoffset
    ycb[0] = yoffset
    xcb[1] = xoffset - intervalwidth
    ycb[1] = yoffset + barheight/2.0
    xcb[2] = xoffset
    ycb[2] = yoffset + barheight

    pres.gsFillColor = 2

    Ngl.polygon_ndc(wks,xcb[0:3],ycb[0:3],pres)

    for i in range(nIntervals):
        xcb[0] = xoffset + i*intervalwidth
        ycb[0] = yoffset

        xcb[1] = xoffset + (i+1)*intervalwidth
        ycb[1] = yoffset

        xcb[2] = xoffset + (i+1)*intervalwidth
        ycb[2] = yoffset + barheight

        xcb[3] = xoffset + i*intervalwidth
        ycb[3] = yoffset + barheight

        pres.gsFillColor = val_to_index(cnLevels, nColors, cnLevels[i]) + 2

        Ngl.polygon_ndc(wks,xcb,ycb,pres)

#        if i % labelInterval == 0 or i == (nIntervals-1):
        if i % labelInterval == 0:
            label = f'{cnLevels[i] : 5.3g}'
            Ngl.text_ndc(wks, label, xcb[0], 0.98*yoffset, tres)


    xcb[0] = xoffset + barwidth
    ycb[0] = yoffset
    xcb[1] = xoffset + barwidth
    ycb[1] = yoffset + barheight
    xcb[2] = xoffset + barwidth + intervalwidth
    ycb[2] = yoffset + barheight/2.0
    pres.gsFillColor = (nColors-1) + 2
    Ngl.polygon_ndc(wks,xcb[0:3],ycb[0:3],pres)

    label = f'{cnLevels[nLevels-1] : 5.3g}'
    Ngl.text_ndc(wks, label, xcb[0], 0.98*yoffset, tres)


terrain_colormap = np.array([
[1.0,1.0,1.0],
[0.0,0.0,0.0],
[0.687500,0.882813,1.000000],  # map background water
[1.000000,0.890625,0.765625],  # map background land
[0.105882,0.388235,0.788235],  # water
[0.000000,0.770588,0.488235],
[0.000000,0.794118,0.417647],
[0.019608,0.803922,0.403922],
[0.050980,0.810196,0.410196],
[0.082353,0.816471,0.416471],
[0.113725,0.822745,0.422745],
[0.145098,0.829020,0.429020],
[0.160784,0.832157,0.432157],
[0.207843,0.841569,0.441569],
[0.239216,0.847843,0.447843],
[0.270588,0.854118,0.454118],
[0.301961,0.860392,0.460392],
[0.333333,0.866667,0.466667],
[0.364706,0.872941,0.472941],
[0.396078,0.879216,0.479216],
[0.411765,0.882353,0.482353],
[0.458824,0.891765,0.491765],
[0.490196,0.898039,0.498039],
[0.521569,0.904314,0.504314],
[0.552941,0.910588,0.510588],
[0.584314,0.916863,0.516863],
[0.615686,0.923137,0.523137],
[0.647059,0.929412,0.529412],
[0.662745,0.932549,0.532549],
[0.709804,0.941961,0.541961],
[0.741176,0.948235,0.548235],
[0.772549,0.954510,0.554510],
[0.803922,0.960784,0.560784],
[0.835294,0.967059,0.567059],
[0.866667,0.973333,0.573333],
[0.898039,0.979608,0.579608],
[0.913725,0.982745,0.582745],
[0.960784,0.992157,0.592157],
[0.992157,0.998431,0.598431],
[0.988235,0.984941,0.593647],
[0.972549,0.964863,0.585176],
[0.956863,0.944784,0.576706],
[0.941176,0.924706,0.568235],
[0.925490,0.904627,0.559765],
[0.909804,0.884549,0.551294],
[0.894118,0.864471,0.542824],
[0.878431,0.844392,0.534353],
[0.862745,0.824314,0.525882],
[0.847059,0.804235,0.517412],
[0.831373,0.784157,0.508941],
[0.823529,0.774118,0.504706],
[0.800000,0.744000,0.492000],
[0.784314,0.723922,0.483529],
[0.768627,0.703843,0.475059],
[0.752941,0.683765,0.466588],
[0.737255,0.663686,0.458118],
[0.721569,0.643608,0.449647],
[0.705882,0.623529,0.441176],
[0.690196,0.603451,0.432706],
[0.674510,0.583373,0.424235],
[0.658824,0.563294,0.415765],
[0.643137,0.543216,0.407294],
[0.627451,0.523137,0.398824],
[0.611765,0.503059,0.390353],
[0.596078,0.482980,0.381882],
[0.580392,0.462902,0.373412],
[0.572549,0.452863,0.369176],
[0.549020,0.422745,0.356471],
[0.533333,0.402667,0.348000],
[0.517647,0.382588,0.339529],
[0.501961,0.362510,0.331059],
[0.513725,0.377569,0.348392],
[0.529412,0.397647,0.369412],
[0.545098,0.417725,0.390431],
[0.560784,0.437804,0.411451],
[0.576471,0.457882,0.432471],
[0.592157,0.477961,0.453490],
[0.607843,0.498039,0.474510],
[0.623529,0.518118,0.495529],
[0.639216,0.538196,0.516549],
[0.654902,0.558275,0.537569],
[0.670588,0.578353,0.558588],
[0.678431,0.588392,0.569098],
[0.701961,0.618510,0.600627],
[0.717647,0.638588,0.621647],
[0.733333,0.658667,0.642667],
[0.749020,0.678745,0.663686],
[0.764706,0.698824,0.684706],
[0.780392,0.718902,0.705725],
[0.796078,0.738980,0.726745],
[0.811765,0.759059,0.747765],
[0.827451,0.779137,0.768784],
[0.843137,0.799216,0.789804],
[0.858824,0.819294,0.810824],
[0.874510,0.839373,0.831843],
[0.890196,0.859451,0.852863],
[0.905882,0.879529,0.873882],
[0.921569,0.899608,0.894902],
[0.929412,0.909647,0.905412],
[0.952941,0.939765,0.936941],
[0.968627,0.959843,0.957961],
[0.984314,0.979922,0.978980],
[1.000000,1.000000,1.000000]
])


map_colormap = np.array([
    [1.0       ,  1.0       ,  1.0       ],
    [0.0       ,  0.0       ,  0.0       ],
    [0.69019608,  0.88627451,  1.0       ],
    [1.0       ,  0.89411765,  0.76862745]
])
