program decode_bufr_all

!#################################################################################
!#  Decode all information in a bufr file and write to little_r file
!#  Input:  NCEP prepbufr observation
!#  Output: prepbufr2littleR.txt which has prepbufr obs and QC in little_r format 
!#          pbufr.table which is the table of prepbufr obs MNEMONIC
!#  Usage:  convert_prepbufr2littleR.exe  
!#          It looks for prepbufr obs with name 'bufrfile' and write to little_r
!#
!#  Ruifang Li 02/01/2011
!#  Ruifang Li 08/30/2011 Added PW for GPSIPW type
!#################################################################################

 
 implicit none

 character(80) bufrfile,bufrtablefile

!  for inventory

 integer, parameter :: max_message_type=30
 character(8) :: message_type_all(max_message_type) ! message type in this file
 integer :: num_message_all(max_message_type)  ! number of messages in each type
 integer :: num_subset_all(max_message_type)   ! number of subset in each message type
 integer :: num_message_type  ! number of message in this bufr file
 integer :: i,index,ndate,idate

!  for read in

 integer :: ireadmg,ireadsb
 character(8) message_type
 integer :: unit_in,unit_out  

! define MNEMONIC and data array
  
  character(80) hdstr,obstr,qcstr
  data hdstr  /'SID XOB YOB DHR ELV TYP T29'/
  data obstr  /'POB QOB TDO TOB ZOB UOB VOB PMO CAT PWO'/
  data qcstr  /'PQM QQM NUL TQM ZQM WQM WQM PMQ NUL PWQ'/ 
  
  real(8),dimension(7):: hdr
  real(8),dimension(10,255):: obsdat,qcmark
  integer :: iret, ntab, levs


 bufrfile='bufrfile'            ! NCEP prepbufr obs name
 bufrtablefile='pbufr.table'    ! NCEP prepbufr MNEMONIC table

! ####################################################################
! Read in information from bufr file and output inventory information
! ####################################################################

 write(*,*) ""
 write(*,*) "Reading bufr file and get inventory information, please wait for 1-2 minutes......."
 write(*,*) ""

 call bufr_inventory_message(bufrfile,bufrtablefile,max_message_type, &
                message_type_all,num_message_all,num_subset_all,           &
                num_message_type,ndate)
 
 write(*,*) 'The bufr/prepbufr file is : ',trim(bufrfile)
 write(*,*) 'The bufr table in this file is saved in : ',trim(bufrtablefile)
 write(*,*) 'The file date is : ',ndate
 write(*,*) 'The total number of message types : ',num_message_type
 write(*,*) ""
 
 write(*,'(2x,a10,a10,a11)') 'mssg_type','mssg_num','subset_num'
 do i=1,num_message_type
   write(*,'(i2,a10,i10,i11)') i,message_type_all(i),num_message_all(i),num_subset_all(i)
 enddo

 
! #######################################################################
! Start to read in information from bufr file and write to little_r file
! #######################################################################

 write(*,*) ""
 write(*,*) "Writting bufr data to little_r format, please wait for 2-3 minutes......."
 write(*,*) ""


 unit_out=20
 open(unit_out,file='prepbufr2littleR.txt')

 do i=1,num_message_type
    open(unit_out+i,file='prepbufr2littleR_'//trim(message_type_all(i))//'.txt')
 enddo

 unit_in=10
 open(unit_in,file=trim(bufrfile),form='unformatted')
 call openbf(unit_in,'IN',unit_in)
 call datelen(10)   ! Specify the length of output date: YYYYMMDDHH  or datelen(8): YYMMDDHH

 msg_report: do while (ireadmg(unit_in,message_type,idate) == 0) ! idate here is YYYYMMDDHH

    index=-10
    do  i=1,num_message_type
       if(message_type == message_type_all(i) ) index=i
    enddo 

! There are 20 messages types in NCEP prepbufr file. This do loop
! write out all subsets of current message type.
! Message types are defined in Table A from  http://www.emc.ncep.noaa.gov/mmb/data_processing/prepbufr.doc/table_1.htm

    sb_report: do while (ireadsb(unit_in) == 0)
       call ufbint(unit_in,hdr,7,1,iret,hdstr)        ! Read header info
       call ufbint(unit_in,obsdat,10,255,levs,obstr)   ! Read obs info
       call ufbint(unit_in,qcmark,10,255,levs,qcstr)   ! Read obs qc info

       ! For GPSIPW type, the PW is written to header record
       ! and so there is no data record in little_r. 
       ! Here returned levs=1, let levs=levs-1,
       ! it does not write data record when call write_little_R. 
       if (trim(message_type)=="GPSIPW") levs=levs-1 
    !   if ( trim(message_type) =="AIRCAR") then 
       call write_little_R(unit_out,hdr,obsdat,qcmark,levs,idate,message_type)
       call write_little_R(unit_out+index,hdr,obsdat,qcmark,levs,idate,message_type)
    enddo sb_report
 enddo msg_report
 
 call closbf(unit_in)
 close(unit_out)
 do i=1,num_message_type
    close(unit_out+i)
 enddo

end program


subroutine bufr_inventory_message(bufrfile,bufrtablefile,max_message_type, &
                message_type_all,num_message_all,num_subset_all,           &
                num_message_type,ndate)

! read buft table and inventory message type from a bufr file
!
! INPUT:
!   bufrfile:        bufr file name
!   bufrtablefile  : bufe table file name
!   max_message_type :  max number for message type
!
! OUTPUT:
!   message_type_all : all message types in this file
!   num_message_all  : number of messages in each type
!   num_subset_all   : number of subset in each message type
!   num_message_type : number of message in this bufr file

!
 implicit none

 character(80),intent(in) :: bufrfile,bufrtablefile
 character(8),intent(out) :: message_type_all(max_message_type)
 integer,intent(in) :: max_message_type
 integer,intent(out) :: num_message_all(max_message_type),num_subset_all(max_message_type),num_message_type,ndate

 integer  :: ireadmg,ireadsb
 integer :: unit_in=10,outlun=20           
 integer :: idate,iret,nmsg,ntb,i,ii
 character(8) message_type

 open(outlun,file=trim(bufrtablefile))
 open(unit_in,file=trim(bufrfile),form='unformatted')
 call openbf(unit_in,'IN',unit_in)
 call datelen(10)

! dump bufr MNEMONIC table
 call dxdump(unit_in,outlun)
 close(outlun)

 num_message_type=0
 num_subset_all(:)=0
 num_message_all(:)=0
 message_type_all(:)=''
 nmsg=0

 msg_report: do while (ireadmg(unit_in,message_type,idate) == 0)
    nmsg=nmsg+1
     ntb = 0
     sb_report: do while (ireadsb(unit_in) == 0)
       ntb = ntb+1
     enddo sb_report

! message inventory
     if(num_message_type == 0 ) then
       num_message_type=1
       message_type_all(num_message_type)=message_type
       num_message_all(num_message_type)= 1
       num_subset_all(num_message_type)= ntb
       ndate=idate
     else
       ii=0
       do i=1,num_message_type
          if(message_type_all(i) == message_type) ii=i
       enddo
       if( ii > 0 .and. ii <=num_message_type) then
          num_message_all(ii)=num_message_all(ii) + 1
          num_subset_all(ii)=num_subset_all(ii) + ntb
       else
          num_message_type=num_message_type+1
          if(num_message_type > max_message_type) then
             write(*,*) 'Error: too many message types'
             write(*,*) 'Need to increase :max_message_type'
             stop 1234
          endif
          message_type_all(num_message_type)=message_type
          num_message_all(num_message_type)=1
          num_subset_all(num_message_type)=ntb
       endif
     endif
 enddo msg_report

 call closbf(unit_in)

end subroutine bufr_inventory_message


subroutine write_little_R(file_num,hdr,obsdat,qcmark,levs,idate,message_type)

  USE module_type, only : report, meas_data
  implicit none

  integer,intent(in) :: file_num,idate,levs
  real(8),dimension(7), intent(in) :: hdr
  real(8),dimension(10,255), intent(in) :: obsdat,qcmark
  character(8),intent(in) :: message_type

  real, parameter :: bufr_missing = 10.0E9
  real :: temperature, mixing_ratio,sp_humidity
  REAL, PARAMETER :: eps     =   0.622  
  integer :: io_error,k,typ,t29
  
  type (report)     :: obs
  type (meas_data)  :: meas,meas_end
  
  character(len=14) :: cdate, dmn, obs_date
  character(len=8)  :: c_station_id,platform
  real(8)        :: rstation_id
  equivalence(rstation_id,c_station_id)

  CHARACTER ( LEN = 120 ) , PARAMETER :: rpt_format =  &
                ' (2F20.5 , 2A40 , '              & ! format for location_type
             // ' 2A40 , 1F20.5 , 5I10 , 3L10 , ' & ! format for source_info
             // ' 2I10 , A20 , '                  & ! fmt for valid_time
             // ' 13(F13.5 , I7),'                & ! fmt for 'terrestrial'
             // '1(:,F13.5 , I7))'                  ! fmt for PW other than GPS 
             !// '7(:,F13.5 , I7))'                  ! fmt for Brightness Temp

  CHARACTER ( LEN = 120 ) , PARAMETER :: meas_format = &
                ' ( 10( F13.5 , I7 ) ) '            ! fmt for measurement rcd

  CHARACTER ( LEN = 120 ) , PARAMETER :: end_format = &
                ' ( 3 ( I7 ) ) '                    ! fmt for end record

! Initialize header and end info for each of little_r dataset
  call initial_head_end(obs,meas_end)

  if(hdr(3) < bufr_missing)    obs%location % latitude =hdr(3)
  if(hdr(2) < bufr_missing) then
     if (hdr(2) <= 180) then
        obs%location % longitude=hdr(2)
     else
        obs%location % longitude=hdr(2) - 360
     end if
  end if 
  
  if(hdr(1) < bufr_missing)    then
     rstation_id=hdr(1)
     obs%location % id(1:8) = c_station_id   
  end if
  obs%location % name='NCEP PREPBUFR'
  if(hdr(4) < bufr_missing)    obs%info % elevation = hdr(5)

! Convert prepbufr report type to WMO code. 
! Refer: http://www.emc.ncep.noaa.gov/mmb/data_processing/prepbufr.doc/table_6.htm
!        http://www.emc.ncep.noaa.gov/mmb/data_processing/prepbufr.doc/table_2.htm
  typ=nint(hdr(6))
  t29=nint(hdr(7))
  select case(t29)
     case (11, 12, 22, 23, 31)
        select case (typ)
           case (120, 122, 132, 220, 222, 232) ;         ! Sound
              platform='FM-35'
           case (221) ;                   ! Pilot
              platform='FM-32'
           case default
              platform='FM-999'
              write (*,*) 'No WMO code matches MESSAGE  T29  TYP: ',message_type,'  ',t29,'  ',typ
        end select
     case (41) ! Aircraft flight-level (all types)
        platform='FM-97'
     case (522, 523);        ! Ships
        platform='FM-13'
     case (531, 532, 561, 562) ;          ! Buoy 
        platform='FM-18'
     case (511, 514)  ! Synoptic
        platform='FM-12'
     case (512)                           ! Metar
        platform='FM-15'
     case (61)                           ! Satellite soundings/retrievals/radiances
        platform='FM-131' 
     case (63)                           ! Geo. CMVs
        platform='FM-88' 
     case (582,583,584)      ! QuikSCAT 582 ,WindSat 583, and ASCAT 584
        platform='FM-281'
     case (74)       ! GPS PW
        platform='FM-111'
     case (71, 72, 73, 75, 76, 77)    ! Profiler
        platform='FM-132'
     case (571, 65) ! ssmi wind speed & total precipitable water
        platform='FM-125'
     case default
        select case (typ)
           case (111 , 210)        !  Tropical Cyclone Bogus
              platform='FM-135'
           case default
              platform='FM-999'
              write (*,*) 'No WMO code matches MESSAGE  T29  TYP: ',message_type,'  ', t29,'  ',typ 
        end select
  end select   
  obs%info % platform = platform

! convert obs time from prepbufr format to little_r format 
  write(cdate,'(i10)') idate
  write(dmn,'(i4,a1)') int(hdr(4)*60.0), 'm'
  call da_advance_time (cdate(1:10), trim(dmn), obs%valid_time % date_char)
 
! Sea level pressure is saved on lowest level in prepbufr data 
  if(obsdat(8,1) < bufr_missing)   then
     obs%ground%slp%data = obsdat(8,1)*100 ! Convert unit from hpa to pa
     obs%ground%slp%qc = qcmark(8,1)
  end if

 if ( obsdat(10,1) < bufr_missing ) then ! GPSIPW type have pw 
    obs%ground%pw%data = obsdat(10,1)/10  ! convert unit from mm to cm
    obs%ground%pw%qc   = 0                ! use 0 instead of qcmark(10,1)
 end if

 obs%info % num_vld_fld = levs
 
! Write hearder info for each record
  write ( file_num , IOSTAT = io_error , FMT = rpt_format ) &
      obs%location % latitude,     obs%location % longitude, &
      obs%location % id,           obs%location % name,      &
      obs%info % platform,         obs%info % source,        &
      obs%info % elevation,        obs%info % num_vld_fld,   &
      obs%info % num_error,        obs%info % num_warning,   &
      obs%info % seq_num,          obs%info % num_dups,      &
      obs%info % is_sound,         obs%info % bogus,         &
      obs%info % discard,                                    &
      obs%valid_time % sut,        obs%valid_time % julian,  &
      obs%valid_time % date_char,                            &
      obs%ground%slp%data,         obs%ground%slp%qc,        &
      obs%ground%ref_pres%data,    obs%ground%ref_pres%qc,   &
      obs%ground%ground_t%data,    obs%ground%ground_t%qc,   &
      obs%ground%sst%data,         obs%ground%sst%qc,        &
      obs%ground%psfc%data,        obs%ground%psfc%qc,       &
      obs%ground%precip%data,      obs%ground%precip%qc,     &
      obs%ground%t_max%data,       obs%ground%t_max%qc,      &
      obs%ground%t_min%data,       obs%ground%t_min%qc,      &
      obs%ground%t_min_night%data, obs%ground%t_min_night%qc,&
      obs%ground%p_tend03%data,    obs%ground%p_tend03%qc,   &
      obs%ground%p_tend24%data,    obs%ground%p_tend24%qc,   &
      obs%ground%cloud_cvr%data,   obs%ground%cloud_cvr%qc,  &
      obs%ground%ceiling%data,     obs%ground%ceiling%qc,    &
      obs%ground%pw     %data,     obs%ground%pw%qc

      ! The following hearder info is for SSMI Tb(brightness temperature)
      ! Is looks like that SSMI Tb is seldom used now, 
      ! So I comment following infomation.

      !obs%ground%tb19v  %data,     obs%ground%tb19v%qc,      &
      !obs%ground%tb19h  %data,     obs%ground%tb19h%qc,      &
      !obs%ground%tb22v  %data,     obs%ground%tb22v%qc,      &
      !obs%ground%tb37v  %data,     obs%ground%tb37v%qc,      &
      !obs%ground%tb37h  %data,     obs%ground%tb37h%qc,      &
      !obs%ground%tb85v  %data,     obs%ground%tb85v%qc,      &
      !obs%ground%tb85h  %data,     obs%ground%tb85h%qc


! Write obs info for each record 
  do k=1,levs,1

    call initial_data(meas)

    if(obsdat(1,k) < bufr_missing)    then
       meas % pressure    % data = obsdat(1,k)*100
       meas % pressure    % qc = qcmark(1,k)
    end if

    if(obsdat(5,k) < bufr_missing)    then
       meas % height      % data = obsdat(5,k)
       meas % height      % qc = qcmark(5,k)
    end if

    if(obsdat(4,k) < bufr_missing .and. obsdat(2,k) < bufr_missing ) then
       sp_humidity=obsdat(2,k)/1e6 ! convert from mg/kg to g/kg 
       mixing_ratio = sp_humidity/(1-sp_humidity)
       temperature = (obsdat(4,k)+273.15)*(1 + mixing_ratio)/(1 + mixing_ratio/eps) ! The unit of t and vt is degree K in this formula
       meas % temperature % data = temperature
       meas % temperature % qc = qcmark(4,k)

       if(obsdat(3,k) < bufr_missing ) then
          meas % dew_point   % data = obsdat(3,k)+273.15  ! Convert unit from degree C to degree K
          meas % rh    % data = 100. * exp (5418.12 * (1./meas%temperature%data - 1./meas%dew_point%data))
          meas % rh          % qc = qcmark(4,k)
       end if

    end if
    
    if(obsdat(6,k) < bufr_missing .and. obsdat(7,k) < bufr_missing)   then
       call uv2sd(obsdat(6,k),obsdat(7,k),meas %speed%data,meas%direction%data )
       meas % speed       % qc = qcmark(6,k)
       meas % direction   % qc = qcmark(6,k)
    end if

!    if(hdr(6) < bufr_missing) then
!       meas % thickness   % data=hdr(6) ! TYP
!    end if
!    if(obsdat(9,k) < bufr_missing) then
!       meas % thickness   % qc =obsdat(9,k) ! CAT 
!    end if
 
    write ( file_num , IOSTAT = io_error , FMT = meas_format )  &
       meas % pressure    % data,  meas % pressure    % qc, &
       meas % height      % data,  meas % height      % qc, &
       meas % temperature % data,  meas % temperature % qc, &
       meas % dew_point   % data,  meas % dew_point   % qc, &
       meas % speed       % data,  meas % speed       % qc, &
       meas % direction   % data,  meas % direction   % qc, &
       meas % u           % data,  meas % u           % qc, &
       meas % v           % data,  meas % v           % qc, &
       meas % rh          % data,  meas % rh          % qc, &
       meas % thickness   % data,  meas % thickness   % qc
  enddo
 

! Write end info for each record
  write ( file_num , IOSTAT = io_error , FMT = meas_format )  &
       meas_end % pressure    % data,  meas_end % pressure    % qc, &
       meas_end % height      % data,  meas_end % height      % qc, &
       meas_end % temperature % data,  meas_end % temperature % qc, &
       meas_end % dew_point   % data,  meas_end % dew_point   % qc, &
       meas_end % speed       % data,  meas_end % speed       % qc, &
       meas_end % direction   % data,  meas_end % direction   % qc, &
       meas_end % u           % data,  meas_end % u           % qc, &
       meas_end % v           % data,  meas_end % v           % qc, &
       meas_end % rh          % data,  meas_end % rh          % qc, &
       meas_end % thickness   % data,  meas_end % thickness   % qc
 
  
  write (file_num , IOSTAT = io_error , FMT = end_format ) &
            obs%info%num_vld_fld , &
            obs%info%num_error , &
            obs%info%num_warning

end subroutine write_little_R


subroutine initial_head_end(obs,meas_end)

  USE module_type
  implicit none
 
  TYPE (report),   intent(out)  :: obs
  TYPE (meas_data),intent(out)  :: meas_end
  real :: missing_value=-888888.00000
  integer:: qc_flag=0

! initialize report header info
      obs%location%latitude         = missing_value
      obs%location%longitude        = missing_value  
      obs%location%id               = ''
      obs%location%name             = ''      
      obs%info%platform             = ''
      obs%info%source               = ''        
      obs%info%elevation            = missing_value
      obs%info%num_vld_fld          = 0  
      obs%info%num_error            = 0
      obs%info%num_warning          = 0  
      obs%info%seq_num              = 0
      obs%info%num_dups             = 0      
      obs%info%is_sound             = .false.
      obs%info%bogus                = .false.        
      obs%info%discard              = .false.                                   
      obs%valid_time%sut            = missing_value
      obs%valid_time%julian         = missing_value 
      obs%valid_time%date_char      = ''                           

      obs%ground%slp%data           = missing_value 
      obs%ground%slp%qc             = qc_flag       
      obs%ground%ref_pres%data      = missing_value
      obs%ground%ref_pres%qc        = qc_flag  
      obs%ground%ground_t%data      = missing_value
      obs%ground%ground_t%qc        = qc_flag  
      obs%ground%sst%data           = missing_value
      obs%ground%sst%qc             = qc_flag       
      obs%ground%psfc%data          = missing_value  
      obs%ground%psfc%qc            = qc_flag      
      obs%ground%precip%data        = missing_value
      obs%ground%precip%qc          = qc_flag    
      obs%ground%t_max%data         = missing_value
      obs%ground%t_max%qc           = qc_flag     
      obs%ground%t_min%data         = missing_value 
      obs%ground%t_min%qc           = qc_flag     
      obs%ground%t_min_night%data   = missing_value 
      obs%ground%t_min_night%qc     = qc_flag
      obs%ground%p_tend03%data      = missing_value 
      obs%ground%p_tend03%qc        = qc_flag  
      obs%ground%p_tend24%data      = missing_value
      obs%ground%p_tend24%qc        = qc_flag  
      obs%ground%cloud_cvr%data     = missing_value
      obs%ground%cloud_cvr%qc       = qc_flag 
      obs%ground%ceiling%data       = missing_value 
      obs%ground%ceiling%qc         = qc_flag   
      obs%ground%pw%data            = missing_value    
      obs%ground%pw%qc              = qc_flag        
      obs%ground%tb19v%data         = missing_value
      obs%ground%tb19v%qc           = qc_flag     
      obs%ground%tb19h%data         = missing_value
      obs%ground%tb19h%qc           = qc_flag     
      obs%ground%tb22v%data         = missing_value
      obs%ground%tb22v%qc           = qc_flag     
      obs%ground%tb37v%data         = missing_value 
      obs%ground%tb37v%qc           = qc_flag     
      obs%ground%tb37h%data         = missing_value
      obs%ground%tb37h%qc           = qc_flag     
      obs%ground%tb85v%data         = missing_value
      obs%ground%tb85v%qc           = qc_flag     
      obs%ground%tb85h%data         = missing_value
      obs%ground%tb85h%qc           = qc_flag

! initialize report end info
       meas_end%pressure%data       = -777777.00000
       meas_end%pressure%qc         = qc_flag 
       meas_end%height%data         = -777777.00000
       meas_end%height%qc           = qc_flag
       meas_end%temperature%data    = missing_value
       meas_end%temperature%qc      = qc_flag
       meas_end%dew_point%data      = missing_value
       meas_end%dew_point%qc        = qc_flag
       meas_end%speed%data          = missing_value
       meas_end%speed%qc            = qc_flag
       meas_end%direction%data      = missing_value
       meas_end%direction%qc        = qc_flag
       meas_end%u%data              = missing_value
       meas_end%u%qc                = qc_flag
       meas_end%v%data              = missing_value
       meas_end%v%qc                = qc_flag
       meas_end%rh%data             = missing_value
       meas_end%rh%qc               = qc_flag
       meas_end%thickness%data      = missing_value
       meas_end%thickness%qc        = qc_flag

       obs%info%num_vld_fld         = 0
       obs%info%num_error           = 0
       obs%info%num_warning         = 0

end subroutine initial_head_end

subroutine initial_data(meas)

  USE module_type
  implicit none
 
  TYPE (meas_data),intent(out)  :: meas
  real :: missing_value=-888888.00000
  integer:: qc_flag=0

! initialize report data info
       meas%pressure%data    = missing_value
       meas%pressure%qc      = qc_flag
       meas%height%data      = missing_value
       meas%height%qc        = qc_flag
       meas%temperature%data = missing_value
       meas%temperature%qc   = qc_flag
       meas%dew_point%data   = missing_value
       meas%dew_point%qc     = qc_flag
       meas%speed%data       = missing_value
       meas%speed%qc         = qc_flag
       meas%direction%data   = missing_value
       meas%direction%qc     = qc_flag
       meas%u%data           = missing_value
       meas%u%qc             = qc_flag
       meas%v%data           = missing_value
       meas%v%qc             = qc_flag
       meas%rh%data          = missing_value
       meas%rh%qc            = qc_flag
       meas%thickness%data   = missing_value
       meas%thickness%qc     = qc_flag

end subroutine initial_data


subroutine uv2sd(u, v, s, d)
  real(8), intent(in)  :: u, v
  real, intent(out) :: s, d
  real :: a, conv
 
  conv = 180. / 3.1415926535897932346

  s = sqrt(u*u + v*v)

  if (s == 0.0) then
      d = 0.
      return
  endif
  if (v == 0.) then
      if (u > 0.) d = 270.
      if (u < 0.) d =  90.
  else
     a = atan (u/v)*conv
     if (u <= 0.0 .and. v <= 0.0 ) d = a
     if (u <= 0.0 .and. v >= 0.0 ) d = a + 180.0
     if (u >= 0.0 .and. v >= 0.0 ) d = a + 180.0
     if (u >= 0.0 .and. v <= 0.0 ) d = a + 360.0
  endif
end subroutine uv2sd

subroutine da_advance_time (date_in, dtime, date_out)

   ! HISTORY: 11/17/2008 modified and simplified from da_util/da_advance_time.f90
   !
   !   modified from da_advance_cymdh,
   !   - has accuracy down to second,
   !   - can use day/hour/minute/second (with/without +/- sign) to advance time,
   !   - can digest various input date format if it still has the right order (ie. cc yy mm dd hh nn ss)
   !   - can digest flexible time increment
   !
   !   eg.: da_advance_time 20070730      12             # advance 12 h
   !        da_advance_time 2007073012   -1d2h30m30s     # back 1 day 2 hours 30 minutes and 30 seconds
   !        da_advance_time 2007073012    1s-3h30m       # back 3 hours 30 minutes less 1 second
   !

   implicit none

   character(len=*),  intent(in)            :: date_in, dtime
   character(len=14), intent(out)           :: date_out

   integer :: ccyy, mm, dd, hh, nn, ss, dday, dh, dn, ds, gday, gsec
   integer :: i, n
   character(len=14) :: ccyymmddhhnnss
   integer :: datelen

   ccyymmddhhnnss = parsedate(date_in)
   datelen = len_trim(ccyymmddhhnnss)

   if (datelen == 8) then
      read(ccyymmddhhnnss(1:10), fmt='(i4, 2i2)')  ccyy, mm, dd
      hh = 0
      nn = 0
      ss = 0
   else if (datelen == 10) then
      read(ccyymmddhhnnss(1:10), fmt='(i4, 3i2)')  ccyy, mm, dd, hh
      nn = 0
      ss = 0
   else if (datelen == 12) then
      read(ccyymmddhhnnss(1:12), fmt='(i4, 4i2)')  ccyy, mm, dd, hh, nn
      ss = 0
   else if (datelen == 14) then
      read(ccyymmddhhnnss(1:14), fmt='(i4, 5i2)')  ccyy, mm, dd, hh, nn, ss
   else
      stop 'wrong input date'
   endif
   if (.not. validdate(ccyy,mm,dd,hh,nn,ss)) then
      write(0,*) trim(ccyymmddhhnnss)
      stop 'Start date is not valid, or has wrong format'
   endif

   call parsedt(dtime,dday,dh,dn,ds)

   hh = hh + dh
   nn = nn + dn
   ss = ss + ds

   ! advance minute according to second
   do while (ss < 0)
      ss = ss + 60
      nn = nn - 1
   end do
   do while (ss > 59)
      ss = ss - 60
      nn = nn + 1
   end do

   ! advance hour according to minute
   do while (nn < 0)
      nn = nn + 60
      hh = hh - 1
   end do
   do while (nn > 59)
      nn = nn - 60
      hh = hh + 1
   end do

   ! advance day according to hour
   do while (hh < 0)
      hh = hh + 24
      dday = dday - 1
   end do

   do while (hh > 23)
      hh = hh - 24
      dday = dday + 1
   end do

   ! advance day if dday /= 0
   if (dday /= 0) call change_date ( ccyy, mm, dd, dday)
  write(ccyymmddhhnnss(1:14), fmt='(i4, 5i2.2)')  ccyy, mm, dd, hh, nn, ss
   !if (datelen<14) then
   !   if(nn /= 0) datelen=12
   !   if(ss /= 0) datelen=14
   !endif
   date_out = ccyymmddhhnnss

contains

subroutine change_date( ccyy, mm, dd, delta )

   implicit none

   integer, intent(inout) :: ccyy, mm, dd
   integer, intent(in)    :: delta

   integer, dimension(12) :: mmday
   integer                :: dday, direction

   mmday = (/31,28,31,30,31,30,31,31,30,31,30,31/)

   mmday(2) = 28

   if (mod(ccyy,4) == 0) then
      mmday(2) = 29

      if (mod(ccyy,100) == 0) then
         mmday(2) = 28
      end if

      if (mod(ccyy,400) == 0) then
         mmday(2) = 29
      end if
   end if

   dday = abs(delta)
   direction = sign(1,delta)

   do while (dday > 0)

      dd = dd + direction

      if (dd == 0) then
         mm = mm - 1
         if (mm == 0) then
            mm = 12
            ccyy = ccyy - 1
         end if

         dd = mmday(mm)
      elseif ( dd > mmday(mm)) then
         dd = 1
         mm = mm + 1
         if(mm > 12 ) then
            mm = 1
            ccyy = ccyy + 1
         end if
      end if

      dday = dday - 1

   end do
   return
end subroutine change_date

function parsedate(datein)
   character(len=*), intent(in) :: datein

   character(len=14) :: parsedate
   character(len=1 ) :: ch
   integer :: n, i
   parsedate = '00000000000000'
   i=0
   do n = 1, len_trim(datein)
      ch = datein(n:n)
      if (ch >= '0' .and. ch <= '9') then
         i=i+1
         parsedate(i:i)=ch
      end if
   end do
   if (parsedate(11:14) == '0000') then
      parsedate(11:14) = ''
   else if(parsedate(13:14) == '00') then
      parsedate(13:14) = ''
   end if
   return
end function parsedate

subroutine parsedt(dt,dday,dh,dn,ds)
   character(len=*),  intent(in)    :: dt
   integer,           intent(inout) :: dday, dh, dn, ds

   character(len=1 ) :: ch
   integer :: n,i,d,s,nounit
   ! initialize time and sign
   nounit=1
   dday=0
   dh=0
   dn=0
   ds=0
   d=0
   s=1
   do n = 1, len_trim(dt)
      ch = dt(n:n)
      select case (ch)
         case ('0':'9')
           read(ch,fmt='(i1)') i
           d=d*10+i
         case ('-')
           s=-1
         case ('+')
           s=1
         case ('d')
           nounit=0
           dday=dday+d*s
           d=0
         case ('h')
           nounit=0
           dh=dh+d*s
           d=0
         case ('n','m')
           nounit=0
           dn=dn+d*s
           d=0
         case ('s')
           nounit=0
           ds=ds+d*s
           d=0
         case default
      end select
   end do
   if (nounit==1) dh=d*s
end subroutine parsedt

function isleapyear(year)
   ! check if year is leapyear
   integer,intent(in) :: year
   logical :: isleapyear
   if( mod(year,4) .ne. 0 ) then
     isleapyear=.FALSE.
   else
     isleapyear=.TRUE.
     if ( mod(year,100) == 0 .and. mod(year,400) .ne. 0 ) isleapyear=.FALSE.
   endif
end function isleapyear

function validdate(ccyy,mm,dd,hh,nn,ss)
   integer, intent(in) :: ccyy,mm,dd,hh,nn,ss

   logical :: validdate

   validdate = .true.

   if(ss > 59 .or. ss < 0 .or. &
      nn > 59 .or. nn < 0 .or. &
      hh > 23 .or. hh < 0 .or. &
                   dd < 1 .or. &
      mm > 12 .or. mm < 1 ) validdate = .false.

   if (mm == 2 .and. ( dd > 29 .or. &
                     ((.not. isleapyear(ccyy)) .and. dd > 28))) &
      validdate = .false.
end function validdate

end subroutine da_advance_time


