subroutine da_setup_obs_structures_radar( grid, ob, iv )

   !-------------------------------------------------------------------------
   ! Purpose: Define, allocate and read radar observation structure.
   !-------------------------------------------------------------------------

   implicit none

   type (y_type),  intent(out)   :: ob  ! Observation structure.
   type (iv_type), intent(inout) :: iv  ! O-B structure.
   type (domain),  intent(inout) :: grid  ! First guess structure

   character(len=filename_len)  :: filename
   integer                      :: n, i, j, k
   logical                      :: thin_3d
   integer  :: istart,iend,jstart,jend
   real     :: rlonlat(4)

   if (trace_use) call da_trace_entry("da_setup_obs_structures_radar")
   !-------------------------------
   ! 0.0 Make thinning grids
   !------------------------------
   call init_constants_derived
   thin_3d=.false.
   if ( thin_conv_ascii) then
      rlat_min =  r999
      rlat_max = -r999
      rlon_min =  r999
      rlon_max = -r999

      istart=MINVAL( grid%i_start(1:grid%num_tiles) )
      iend  =MAXVAL( grid%i_end  (1:grid%num_tiles) )
      jstart=MINVAL( grid%j_start(1:grid%num_tiles) )
      jend  =MAXVAL( grid%j_end  (1:grid%num_tiles) )

      do i = istart, iend
         do j = jstart, jend
            rlat_min=min(rlat_min, grid%xb%lat(i,j))
            rlat_max=max(rlat_max, grid%xb%lat(i,j))
            if( grid%xb%lon(i,j) < 0.0) then
               rlon_min=min(rlon_min, (r360+grid%xb%lon(i,j)))
               rlon_max=max(rlon_max, (r360+grid%xb%lon(i,j)))
            else
               rlon_min=min(rlon_min, grid%xb%lon(i,j))
               rlon_max=max(rlon_max, grid%xb%lon(i,j))
            endif
         enddo
      enddo

#ifdef DM_PARALLEL
      call mpi_reduce(rlat_min, rlonlat(1), 1, true_mpi_real, mpi_min, root, comm, ierr)
      call mpi_reduce(rlon_min, rlonlat(2), 1, true_mpi_real, mpi_min, root, comm, ierr)
      call mpi_reduce(rlat_max, rlonlat(3), 1, true_mpi_real, mpi_max, root, comm, ierr)
      call mpi_reduce(rlon_max, rlonlat(4), 1, true_mpi_real, mpi_max, root, comm, ierr)

      CALL mpi_bcast( rlonlat, 4 , true_mpi_real , root , comm, ierr )

      rlat_min = rlonlat(1)
      rlon_min = rlonlat(2)
      rlat_max = rlonlat(3)
      rlon_max = rlonlat(4)
#endif

      dlat_grid = rlat_max - rlat_min
      dlon_grid = rlon_max - rlon_min

      allocate(thinning_grid_conv(num_ob_indexes))
      thin_3d=.true.
      call make3grids (radar,thin_mesh_conv(n), thin_3d)
   end if


   !--------------------------------------------------------------------------
   ! [1.0] Scan RADAR observation header and get number of obs:
   !--------------------------------------------------------------------------
   if (num_fgat_time > 1) then
      do n=1, num_fgat_time

         iv%time = n
         filename = ' '

         ! scan radar observation file
         write(filename(1:10), fmt='(a, i2.2, a)') 'ob', n,'.radar'
         call da_scan_obs_radar(iv, filename, grid)

         iv%info(radar)%plocal(n) = iv%info(radar)%nlocal
         iv%info(radar)%ptotal(n) = iv%info(radar)%ntotal
      end do
   else
      iv%time = 1
      ! scan main body of radar observation file
      call da_scan_obs_radar(iv, 'ob.radar', grid)
      iv%info(radar)%plocal(iv%time) = iv%info(radar)%nlocal
      iv%info(radar)%ptotal(iv%time) = iv%info(radar)%ntotal
   end if

   !--------------------------------------------------------------------------
   ! Allocate based on input number of obs:
   !--------------------------------------------------------------------------
   ! This logic was originally found in da_allocate_observations; moved here
   if (iv%info(radar)%nlocal > 0) allocate(iv%radar (1:iv%info(radar)%nlocal))
   if (iv%info(radar)%nlocal > 0) then
      allocate (iv%info(radar)%name(iv%info(radar)%nlocal))
      allocate (iv%info(radar)%platform(iv%info(radar)%nlocal))
      allocate (iv%info(radar)%id(iv%info(radar)%nlocal))
      allocate (iv%info(radar)%date_char(iv%info(radar)%nlocal))
      allocate (iv%info(radar)%levels(iv%info(radar)%nlocal))
      allocate (iv%info(radar)%lat(iv%info(radar)%max_lev,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%lon(iv%info(radar)%max_lev,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%elv(iv%info(radar)%nlocal))
      allocate (iv%info(radar)%pstar(iv%info(radar)%nlocal))

      allocate (iv%info(radar)%slp(iv%info(radar)%nlocal))
      allocate (iv%info(radar)%pw(iv%info(radar)%nlocal))

      allocate (iv%info(radar)%x  (kms:kme,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%y  (kms:kme,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%i  (kms:kme,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%j  (kms:kme,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%dx (kms:kme,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%dxm(kms:kme,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%dy (kms:kme,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%dym(kms:kme,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%k  (iv%info(radar)%max_lev,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%dz (iv%info(radar)%max_lev,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%dzm(iv%info(radar)%max_lev,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%zk (iv%info(radar)%max_lev,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%proc_domain(iv%info(radar)%max_lev,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%thinned(iv%info(radar)%max_lev,iv%info(radar)%nlocal))
      allocate (iv%info(radar)%obs_global_index(iv%info(radar)%nlocal))

      iv%info(radar)%proc_domain(:,:)  = .false.
      iv%info(radar)%thinned(:,:)      = .false.
      iv%info(radar)%zk(:,:)           = missing_r
   end if

   if (num_fgat_time > 1) then

      do n=1, num_fgat_time
         iv%time = n
         filename = ' '  

         ! read radar observation file
         write(filename(1:10), fmt='(a, i2.2, a)') 'ob', n,'.radar'
         call da_read_obs_radar(iv, filename, grid)

         iv%info(radar)%thin_ptotal(n) = iv%info(radar)%thin_ntotal
         iv%info(radar)%thin_plocal(n) = iv%info(radar)%thin_nlocal
      end do
   else
      iv%time = 1

      ! read radar observation file
      call da_read_obs_radar(iv, 'ob.radar', grid)
      iv%info(radar)%thin_ptotal(iv%time) = iv%info(radar)%thin_ntotal
      iv%info(radar)%thin_plocal(iv%time) = iv%info(radar)%thin_nlocal
   end if


   ! Calculate DT for RF DA

   if (use_radar_rf) then
      if (.not. DT_cloud_model) then
         do j = jts,jte
            do i = its, ite
               do k = kts, kte
                   grid%xb%delt(i,j,k) = 0.0
               end do
            end do
         end do

         do n = 1, iv%info(radar)%nlocal
            do i=int(iv%info(radar)%i(1,n)), int(iv%info(radar)%i(1,n))+1
               do j=int(iv%info(radar)%j(1,n)), int(iv%info(radar)%j(1,n))+1
                  do k=kts, kte
                     grid%xb%delt(i,j,k) = 1800.0
                     grid%xb%qrn(i,j,k) = amax1(5.0E-5, grid%xb%qrn(i,j,k))
                     grid%xb%qcw(i,j,k) = amax1(5.0E-12, grid%xb%qcw(i,j,k))
                  end do
               end do
            end do
         end do
      end if
   end if

   !--------------------------------------------------------------------------
   ! [2.5] Set all thinned obs missing  
   !--------------------------------------------------------------------------
    if ( thin_conv_ascii ) then
       if ( iv%info(radar)%ntotal > 0 ) then
          if ( iv%info(radar)%nlocal > 0 ) then
             if ( ANY(iv%info(radar)%thinned(:,:)) ) then
                call da_set_3d_obs_missing(iv,radar)  ! assign missing values level-by-level if thinned=true data
             end if
          end if
       end if
    end if 
  
   !--------------------------------------------------------------------------
   ! [3.0] Calculate innovation vector (O-B) and create (smaller) ob structure:
   !--------------------------------------------------------------------------

   call da_fill_obs_structures_radar(iv, ob)

   iv%time = 1

   if ( thin_conv_ascii ) then
      thin_3d=.true.
      call destroygrids_conv (radar, thin_3d)
      deallocate(thinning_grid_conv)
   end if

   if (trace_use) call da_trace_exit("da_setup_obs_structures_radar")
end subroutine da_setup_obs_structures_radar

