#!/bin/csh

set nonomatch

#=============================================================================;
# Purpose     : Script for running WRFVAR with Multi-incremental
#
# Assuming    : 1. All the necessary files required by 4DVAR run are 
#                  already under $RUN_DIR, such as be.dat, namelist.input, *.tbl, 
#                  fg, fg02, wrfbdy_d01, da_wrfvar.exe, da_update_bc, etc.
#
#               2. da_bilin.exe, da_bdy.exe, da_decimation.exe located under
#                  ${MULTI_INC_TOOLS}
#
# How to run  : When everything is ready to go, call this script instead of 
#               da_wrfvar.exe for a Multi-incremental run
#
#               run_4dvar_multi_inc_wrapper.csh
#
# Limitation  : Grids need to match
#                         ( n - 1 ) mod m = 0
#               where n is the x/y grid number of high resolution, m is the
#               x/y grid number of low resolution. Default ratio is 1:3.
#
# Platform    : All the commands involved by this script are GNU/Linux 
#               commands on CentOS box. If involved this script other than 
#               CentOS, commands may not run as your expect, double check 
#               it before using.
#
#               Not fullly test with all platforms, use it at your own risk
#
# jliu@ucar.edu, MMM/NCAR, 01/13/2012
#=============================================================================;


#----------------------------User settings------------------------------------;
#
# Use these environment variables to override the default settings
#
# Variable Names     Default Value                 Description
#
# RUN_CMD            mpirun -np 16   Job submit command, "" for serial and OpenMP
#
# TIME_STEP_STAGE2   auto detect     Stage2 Time step for integration in integer
#                                    seconds As large as 6*DX (in km) and must be 
#                                    exactly divisible by VAR4D_bin exactly.
#
# RADT_STAGE2        auto detect     Minutes between radiation physics calls for 
#                                    Multi_inc stage2. 1 minute per km of dx.
#
# STAGE1_ONLY        FALSE           TRUE/FALSE,  Only run STAGE1
#
# STAGE2_ONLY        FALSE           TRUE/FALSE,  Only run STAGE2
#
# MULTI_INC          TRUE            TRUE/FALSE - Multi-incremental/full resolution
#

if ( ! $?RUN_CMD     ) set     RUN_CMD="mpirun -np 16" # "" -Serial/OpenMP
if ( ! $?MULTI_INC   ) set   MULTI_INC=TRUE            # TRUE / FALSE
if ( ! $?STAGE1_ONLY ) set STAGE1_ONLY=FALSE           # Only run stage1
if ( ! $?STAGE2_ONLY ) set STAGE2_ONLY=FALSE           # Only run stage2

if ( ! $?DEBUG            ) set DEBUG=FALSE            # TRUE /FALSE
if ( ! $?DECIMATION_FACTOR) set DECIMATION_FACTOR=3    # default decimation factor

#----------------------------End of User settings-----------------------------;

if ( ! $?MULTI_INC_TOOLS ) then
  if ( ${#argv} > 0 ) then
    set MULTI_INC_TOOLS=$1
  else
    set appname=${0:t}
    set  MULTI_INC_TOOLS=${0:h}
    if ( "$MULTI_INC_TOOLS" == "$appname" ) set MULTI_INC_TOOLS="."
  endif
endif

if ( $MULTI_INC == TRUE ) then

  foreach f (da_decimation.exe da_bilin.exe da_bdy.exe)
    if ( -e ${MULTI_INC_TOOLS}/$f ) then 
      if ( "$MULTI_INC_TOOLS" != "." ) then
        ln -sf ${MULTI_INC_TOOLS}/$f .
      endif
    else
      echo "$f NOT exists" > FAIL
      exit -1
    endif
  end

  if ( $STAGE2_ONLY == FALSE ) then

    set FILES_TO_CLEAN=(fg_hires fg_thin fg02_hires fg02_thin \
                        wrfbdy_stage1 wrfbdy_stage2 \
                        wrfvar_output wrfvar_output_hires wrfvar_output_lores \
                        ana02 ana02_hires ana02_lores\
                        wrfinput_hires FAIL \
                        namelist.input.stage2 namelist.input.stage1)

    foreach f ($FILES_TO_CLEAN)
      if ( -e $f ) rm -f $f
    end

    if ("`ls -l rsl* | wc -l`" != "0" ) then
      rm -rf rsl*
    endif

    if ("`ls -l gts* | wc -l`" != "0" ) then
      rm -f gts*
    endif

    set fname=(fg fg02 wrfinput_d01 wrfbdy_d01 namelist.input)
    set orig=()
    foreach f ( $fname )
      set orig=($orig ${f}_orig)
      if ( -e ${f}_orig ) then
        set buff=(`ls -l ${f}_orig | sed 's/->//g'`)
        ln -sf $buff[$#buff] $f
      else
        set buff=(`ls -l $f | sed 's/->//g'`)
        switch ($#buff)
          case 10 :
            set cmd="ln -sf "
            breaksw
          case 9  :
            set cmd="cp "
            breaksw
          default :
            echo "$f NOT exist" > FAIL
            exit -1
        endsw
        $cmd $buff[$#buff]  ${f}_orig
      endif
    end

cat > footprint.bat << EOF
# 0. Keep a reference for the original" 
cp fg_orig              fg_hires
cp fg02_orig            fg02_hires
cp namelist.input_orig  namelist.input.stage1

EOF

    cp fg_orig               fg_hires
    cp fg02_orig             fg02_hires
    cp namelist.input_orig   namelist.input.stage1

  # 1. Thinning domain #

    #  Input file : wrfinput_hires --> fg_hires (link)
    # Output file : wrfinput_lores --> fg_thin  (move)

cat >> footprint.bat << EOF
# 1.1 Thinning fg
ln -sf fg_hires wrfinput_hires
./da_decimation.exe
mv wrfinput_lores fg_thin

EOF

    ln -sf  fg_hires         wrfinput_hires
    ./da_decimation.exe
    if ( ! -e wrfinput_lores ) then
      echo "thinning fg failed" > FAIL
      exit -1
    endif
    mv wrfinput_lores        fg_thin

    #  Input file : wrfinput_hires --> fg02_hires (link)
    # Output file : wrfinput_lores --> fg02_thin  (move)

cat >> footprint.bat << EOF
# 1.2 Thinning fg02
ln -sf fg02_hires   wrfinput_hires
./da_decimation.exe
mv wrfinput_lores   fg02_thin
rm wrfinput_hires

EOF

    ln -sf  fg02_hires       wrfinput_hires
    ./da_decimation.exe
    if ( ! -e wrfinput_lores ) then
      echo "thinning fg02 failed" > FAIL
      exit -1
    endif
    mv wrfinput_lores        fg02_thin
    rm wrfinput_hires

  # 2. Generating boundary file #
  #  2.1 stage1 bdy
  #  Input file : fg          --> fg_hires         (link)
  #               fg02        --> fg02_hires       (link)
  #               wrfbdy_ref  --> wrfbdy_d01_orig  (link)
  # Output file : wrfbdy_out  --> wrfbdy_stage1    (move)

cat >> footprint.bat << EOF
# 2.1 Generating boundary file for stage1
ln -sf fg_hires          fg
ln -sf fg02_hires        fg02
ln -sf wrfbdy_d01_orig   wrfbdy_ref
./da_bdy.exe
mv wrfbdy_out            wrfbdy_stage1

EOF

    ln -sf fg_hires          fg
    ln -sf fg02_hires        fg02
    ln -sf wrfbdy_d01_orig   wrfbdy_ref
    da_bdy.exe
    if ( ! -e wrfbdy_out ) then
      echo "generating bdy file for stage1 failed" > FAIL
      exit -1
    endif
    mv wrfbdy_out            wrfbdy_stage1

  # 2.2 stage2 bdy
  #  Input file : fg          --> fg_thin          (link)
  #               fg02        --> fg02_thin        (link)
  #               wrfbdy_ref  --> wrfbdy_d01_orig  (link)
  # Output file : wrfbdy_out  --> wrfbdy_stage2    (move)

cat >> footprint.bat << EOF
# 2.2 Generating boundary file for stage2
ln -sf fg_thin           fg
ln -sf fg02_thin         fg02
ln -sf wrfbdy_d01_orig   wrfbdy_ref
da_bdy.exe
mv wrfbdy_out            wrfbdy_stage2
rm wrfbdy_ref

EOF

    ln -sf fg_thin           fg
    ln -sf fg02_thin         fg02
    ln -sf wrfbdy_d01_orig   wrfbdy_ref
    da_bdy.exe
    if ( ! -e wrfbdy_out ) then
      echo "generating bdy file for stage2 failed" > FAIL
      exit -1
    endif
    mv wrfbdy_out            wrfbdy_stage2
    rm wrfbdy_ref

  # 3. Stage 1 #

    sed -e '/[Mm][Uu][Ll][Tt][Ii]_[Ii][Nn][Cc]/d' \
        -e '/&[Ww][Rr][Ff][Vv][Aa][Rr]1$/ a\multi_inc=1,' \
         namelist.input.stage1 > namelist.input.stage1.edit

    mv namelist.input.stage1.edit namelist.input.stage1

cat >> footprint.bat << EOF
# 3. Run Stage1
ln -sf namelist.input.stage1 namelist.input
ln -sf fg_hires              fg
ln -sf fg_hires              wrfinput_d01
ln -sf fg02_hires            fg02
ln -sf wrfbdy_stage1         wrfbdy_d01

$RUN_CMD da_wrfvar.exe < /dev/null

EOF

    ln -sf namelist.input.stage1 namelist.input
    ln -sf fg_hires              fg
    ln -sf fg_hires              wrfinput_d01
    ln -sf fg02_hires            fg02
    ln -sf wrfbdy_stage1         wrfbdy_d01

    echo "stage1" > .current_stage
    echo "stage2" > .final_stage
    echo "stage1" > .last_stage

    if ( $DEBUG == FALSE ) then

      $RUN_CMD da_wrfvar.exe < /dev/null

      if ( "`tail  rsl.out.0000 | grep -o -m 1 successfully`" != "successfully" ) then
        echo "da_wrfvar stage1 failed " > FAIL
        exit -1
      endif
      if ( $STAGE1_ONLY == FALSE ) then
        set rsl_out_storage=rsl_out_stage1
        if ( ! -d $rsl_out_storage ) mkdir $rsl_out_storage
        foreach f ( rsl.*.[0-9][0-9][0-9][0-9])
          mv $f $rsl_out_storage
        end
      endif
    endif

    if ( $STAGE1_ONLY == TRUE ) exit

  endif

  # 4. Stage 2 #

  if ( "cat .current_stage" != "stage2" ) then 

    if ( ! $?NL_DX )   set   NL_DX=`grep -i -E "dx[ \t]*="   namelist.input | sed -e 's/\t/ /g' -e 's/  *//g' -e 's/,//' | cut -c4-`
    if ( ! $?NL_DY )   set   NL_DY=`grep -i -E "dy[ \t]*="   namelist.input | sed -e 's/\t/ /g' -e 's/  *//g' -e 's/,//' | cut -c4-`
    if ( ! $?NL_E_WE ) set NL_E_WE=`grep -i -E "e_we[ \t]*=" namelist.input | sed -e 's/\t/ /g' -e 's/  *//g' -e 's/,//' | cut -c6-`
    if ( ! $?NL_E_SN ) set NL_E_SN=`grep -i -E "e_sn[ \t]*=" namelist.input | sed -e 's/\t/ /g' -e 's/  *//g' -e 's/,//' | cut -c6-`

    set   NL_DX=`echo $NL_DX | sed 's/\.[0-9]*//g'`
    set   NL_DY=`echo $NL_DY | sed 's/\.[0-9]*//g'`
    set NL_E_WE=`echo $NL_E_WE | sed 's/\.[0-9]*//g'`
    set NL_E_SN=`echo $NL_E_SN | sed 's/\.[0-9]*//g'`

    @   NL_DX = $NL_DX * $DECIMATION_FACTOR
    @   NL_DY = $NL_DY * $DECIMATION_FACTOR
    @ NL_E_WE = ($NL_E_WE - 1) / $DECIMATION_FACTOR + 1
    @ NL_E_SN = ($NL_E_SN - 1) / $DECIMATION_FACTOR + 1

    if ( ! $?RADT_STAGE2 ) then
      @   RADT_STAGE2 = $NL_DX / 1000
    endif

    if ( ! $?TIME_STEP_STAGE2 ) then
      # must be divisible exactly by var4d_bin
      # begin from 6*DX
      set NL_VAR4D_BIN=`grep -i -E "var4d_bin[ \t]*=" namelist.input | \
                        sed -e 's/\t/ /g' -e 's/  *//g' \
                            -e 's/\.[0-9]*//g' -e 's/,//' | \
                        cut -c11-`
      @ TIME_STEP_STAGE2 = ( $NL_DX / 1000 ) * 6
      set i=$TIME_STEP_STAGE2
      while ( $i != 0 )
        @ i = $NL_VAR4D_BIN % $TIME_STEP_STAGE2
        @ TIME_STEP_STAGE2--
      end
      @ TIME_STEP_STAGE2++
    endif

    sed -e '/[Mm][Uu][Ll][Tt][Ii]_[Ii][Nn][Cc]/d' \
        -e '/&[Ww][Rr][Ff][Vv][Aa][Rr]1$/ a\multi_inc=2,' \
        -e "/[Dd][Xx][ \t]*=/ c\dx=${NL_DX}.0," \
        -e "/[Dd][Yy][ \t]*=/ c\dy=${NL_DY}.0," \
        -e "/[Ee]_[Ww][Ee][ \t]*=/ c\e_we=$NL_E_WE,"\
        -e "/[Ee]_[Ss][Nn][ \t]*=/ c\e_sn=$NL_E_SN," \
        -e "/[Tt][Ii][Mm][Ee]_[Ss][Tt][Ee][Pp][ \t]*=/ c\time_step=$TIME_STEP_STAGE2," \
        -e "/[Rr][Aa][Dd][Tt][ \t]*=/ c\radt=$RADT_STAGE2," \
        namelist.input > namelist.input.stage2
  endif 

cat >> footprint.bat << EOF
# 4. Run stage2
ln -sf namelist.input.stage2 namelist.input
ln -sf fg_thin               fg
ln -sf fg_thin               wrfinput_d01
ln -sf fg02_thin             fg02
ln -sf wrfbdy_stage2         wrfbdy_d01

$RUN_CMD da_wrfvar.exe < /dev/null

EOF

  if ( "`cat .last_stage`" == "stage1"  ) then
    ln -sf namelist.input.stage2 namelist.input
    ln -sf fg_thin               fg
    ln -sf fg_thin               wrfinput_d01
    ln -sf fg02_thin             fg02
    ln -sf wrfbdy_stage2         wrfbdy_d01
  endif

  echo "stage2" > .current_stage
  echo "stage2" > .last_stage

  if ( $DEBUG == FALSE ) then

    $RUN_CMD da_wrfvar.exe < /dev/null

    if ( "`tail  rsl.out.0000 | grep -o -m 1 successfully`" != "successfully" ) then
      echo "da_wrfvar stage2 failed " > FAIL
      exit -1
    endif

    mv wrfvar_output wrfvar_output_lores
  endif

  echo "mv wrfvar_output wrfvar_output_lores" >> footprint.bat

  # 5. Increment regridding #

  set VAR4D_LBC=`grep -i -E 'var4d_lbc[ \t]*=' namelist.input | \
                 sed -e 's/,//g' -e 's/[ \t]\+//' | \
                 cut -d'=' -f2 | cut -c1`
  echo "# 5.1 regridding for 2nd time level (ana02)"  >> footprint.bat

  if ( "$VAR4D_LBC" == "t" || "$VAR4D_LBC" == "T" ) then

    #  5.1 regridding for 2nd time level (ana02)
    #  Input file : fg                  --> fg02_thin   (link)
    #               wrfvar_output       --> ana02       (link)
    #               wrfinput_hires      --> fg02_hires  (link)
    # Output file : wrfvar_output_hires --> ana02_hires (move)

cat >> footprint.bat << EOF

ln -sf fg02_thin            fg
ln -sf fg02_hires           wrfinput_hires
ln -sf ana02                wrfvar_output
./da_bilin.exe

mv wrfvar_output_hires      ana02_hires
mv ana02                    ana02_lores
ln -sf ana02_hires          ana02

EOF

    if ( $DEBUG == FALSE ) then
      ln -sf fg02_thin            fg
      ln -sf fg02_hires           wrfinput_hires
      ln -sf ana02                wrfvar_output
      ./da_bilin.exe
      if ( ! -e wrfvar_output_hires ) then
        echo "regridding increment failed" > FAIL
        exit -1
      endif
      mv wrfvar_output_hires      ana02_hires

      # Link ana02 to ana02_hires
      mv ana02                    ana02_lores
      ln -sf ana02_hires          ana02
    endif

  else

    echo "#Skipped, VAR4D_LBC=$VAR4D_LBC \n" >> footprint.bat

  endif

  #  5.2 regridding for 1st time level (fg)
  #  Input file : fg                  --> fg_thin             (link)
  #               wrfvar_output       --> wrfvar_output_lores (link)
  #               wrfinput_hires      --> fg_hires            (link)
  # Output file : wrfvar_output_hires --> wrfvar_output       (link)

cat >> footprint.bat << EOF
# 5.2 regridding for 1st time level (fg)
ln -sf fg_thin              fg
ln -sf fg_hires             wrfinput_hires
ln -sf wrfvar_output_lores  wrfvar_output
./da_bilin.exe

ln -sf wrfvar_output_hires  wrfvar_output

EOF

  if ( $DEBUG == FALSE ) then
    ln -sf fg_thin              fg
    ln -sf fg_hires             wrfinput_hires
    ln -sf wrfvar_output_lores  wrfvar_output
    ./da_bilin.exe
    if ( ! -e wrfvar_output_hires ) then
      echo "regridding increment failed" > FAIL
      exit -1
    endif
    rm wrfinput_hires

    # the final analysis for WRF
    ln -sf wrfvar_output_hires  wrfvar_output
  endif

  if ( $STAGE2_ONLY == TRUE ) exit

  # restore all original links
  echo "6. restore all original links" >> footprint.bat

  if ( $?orig ) then
    set cmd=""
    foreach f_orig ( $orig )
      set f=`echo $f_orig | sed 's/_orig//'`
      set buff=(`ls -l $f_orig | sed 's/->//g'`)
      switch ($#buff)
        case 10:
          set cmd="ln -sf "
          breaksw
        case 9:
          set cmd="cp -f "
          breaksw
        default:
          set cmd="" 
          breaksw
      endsw
      if ( "$cmd" != "" ) then
        echo "rm $f" >> footprint.bat
        echo "$cmd $buff[$#buff] $f" >> footprint.bat
        rm $f
        $cmd $buff[$#buff] $f
      endif
    end
    echo "stage1" > .last_stage
    echo "stage1" > .current_stage
  endif

else

  $RUN_CMD da_wrfvar.exe < /dev/null

endif
